// MIT License
//
// Copyright (c) 2017-2019 MessageKit
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

import Foundation
import MessageKit
import UIKit

// MARK: - CustomMessagesFlowLayout

open class CustomMessagesFlowLayout: MessagesCollectionViewFlowLayout {
  open lazy var customMessageSizeCalculator = CustomMessageSizeCalculator(layout: self)

  open override func cellSizeCalculatorForItem(at indexPath: IndexPath) -> CellSizeCalculator {
    if isSectionReservedForTypingIndicator(indexPath.section) {
      return typingIndicatorSizeCalculator
    }
    let message = messagesDataSource.messageForItem(at: indexPath, in: messagesCollectionView)
    if case .custom = message.kind {
      return customMessageSizeCalculator
    }
    return super.cellSizeCalculatorForItem(at: indexPath)
  }

  open override func messageSizeCalculators() -> [MessageSizeCalculator] {
    var superCalculators = super.messageSizeCalculators()
    // Append any of your custom `MessageSizeCalculator` if you wish for the convenience
    // functions to work such as `setMessageIncoming...` or `setMessageOutgoing...`
    superCalculators.append(customMessageSizeCalculator)
    return superCalculators
  }
}

// MARK: - CustomMessageSizeCalculator

open class CustomMessageSizeCalculator: MessageSizeCalculator {
  // MARK: Lifecycle

  public override init(layout: MessagesCollectionViewFlowLayout? = nil) {
    super.init()
    self.layout = layout
  }

  // MARK: Open

  open override func sizeForItem(at _: IndexPath) -> CGSize {
    guard let layout = layout else { return .zero }
    let collectionViewWidth = layout.collectionView?.bounds.width ?? 0
    let contentInset = layout.collectionView?.contentInset ?? .zero
    let inset = layout.sectionInset.left + layout.sectionInset.right + contentInset.left + contentInset.right
    return CGSize(width: collectionViewWidth - inset, height: 44)
  }
}
