<?php

/**
 * Maho
 *
 * @category   Mage
 * @package    Mage_Adminhtml
 * @copyright  Copyright (c) 2006-2020 Magento, Inc. (https://magento.com)
 * @copyright  Copyright (c) 2022-2024 The OpenMage Contributors (https://openmage.org)
 * @copyright  Copyright (c) 2024 Maho (https://mahocommerce.com)
 * @license    https://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

use Mage_Adminhtml_Block_Widget_Grid_Massaction_Abstract as MassAction;

/**
 * Adminhtml customer grid block
 *
 * @category   Mage
 * @package    Mage_Adminhtml
 */
class Mage_Adminhtml_Block_Customer_Grid extends Mage_Adminhtml_Block_Widget_Grid
{
    /**
     * Mage_Adminhtml_Block_Customer_Grid constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->setId('customerGrid');
        $this->setUseAjax(true);
        $this->setDefaultSort('entity_id');
        $this->setSaveParametersInSession(true);
    }

    /**
     * @throws Mage_Core_Exception
     * @throws Mage_Eav_Exception
     */
    #[\Override]
    protected function _prepareCollection()
    {
        $collection = Mage::getResourceModel('customer/customer_collection')
            ->addNameToSelect()
            ->addAttributeToSelect('email')
            ->addAttributeToSelect('created_at')
            ->addAttributeToSelect('group_id')
            ->joinAttribute('billing_postcode', 'customer_address/postcode', 'default_billing', null, 'left')
            ->joinAttribute('billing_city', 'customer_address/city', 'default_billing', null, 'left')
            ->joinAttribute('billing_telephone', 'customer_address/telephone', 'default_billing', null, 'left')
            ->joinAttribute('billing_region', 'customer_address/region', 'default_billing', null, 'left')
            ->joinAttribute('billing_country_id', 'customer_address/country_id', 'default_billing', null, 'left');

        $this->setCollection($collection);

        return parent::_prepareCollection();
    }

    #[\Override]
    protected function _prepareColumns()
    {
        $this->addColumn('entity_id', [
            'header'    => Mage::helper('customer')->__('ID'),
            'index'     => 'entity_id',
        ]);

        $this->addColumn('name', [
            'header'    => Mage::helper('customer')->__('Name'),
            'index'     => 'name',
        ]);

        $this->addColumn('email', [
            'header'    => Mage::helper('customer')->__('Email'),
            'width'     => '150',
            'index'     => 'email',
        ]);

        $groups = Mage::getResourceModel('customer/group_collection')
            ->addFieldToFilter('customer_group_id', ['gt' => 0])
            ->load()
            ->toOptionHash();

        $this->addColumn('group', [
            'header'    =>  Mage::helper('customer')->__('Group'),
            'width'     =>  '150',
            'index'     =>  'group_id',
            'type'      =>  'options',
            'options'   =>  $groups,
        ]);

        $this->addColumn('Telephone', [
            'header'    => Mage::helper('customer')->__('Telephone'),
            'width'     => '100',
            'index'     => 'billing_telephone',
        ]);

        $this->addColumn('billing_postcode', [
            'header'    => Mage::helper('customer')->__('ZIP'),
            'width'     => '90',
            'index'     => 'billing_postcode',
        ]);

        $this->addColumn('billing_country_id', [
            'header'    => Mage::helper('customer')->__('Country'),
            'width'     => '100',
            'type'      => 'country',
            'index'     => 'billing_country_id',
        ]);

        $this->addColumn('billing_region', [
            'header'    => Mage::helper('customer')->__('State/Province'),
            'width'     => '100',
            'index'     => 'billing_region',
        ]);

        $this->addColumn('customer_since', [
            'header'    => Mage::helper('customer')->__('Customer Since'),
            'type'      => 'datetime',
            'align'     => 'center',
            'index'     => 'created_at',
            'gmtoffset' => true,
        ]);

        if (!Mage::app()->isSingleStoreMode()) {
            $this->addColumn('website_id', [
                'header'    => Mage::helper('customer')->__('Website'),
                'width'     => 120,
                'type'      => 'options',
                'options'   => Mage::getSingleton('adminhtml/system_store')->getWebsiteOptionHash(true),
                'index'     => 'website_id',
            ]);
        }

        $this->addColumn(
            'action',
            [
                'type'      => 'action',
                'getter'    => 'getId',
                'actions'   => [
                    [
                        'caption'   => Mage::helper('customer')->__('Edit'),
                        'url'       => ['base' => '*/*/edit'],
                        'field'     => 'id',
                    ],
                ],
                'index'     => 'stores',
                'is_system' => true,
            ],
        );

        $this->addExportType('*/*/exportCsv', Mage::helper('customer')->__('CSV'));
        $this->addExportType('*/*/exportXml', Mage::helper('customer')->__('Excel XML'));
        return parent::_prepareColumns();
    }

    /**
     * @return $this
     */
    #[\Override]
    protected function _prepareMassaction()
    {
        $this->setMassactionIdField('entity_id');
        $this->getMassactionBlock()->setFormFieldName('customer');

        $this->getMassactionBlock()->addItem(MassAction::DELETE, [
            'label'    => Mage::helper('customer')->__('Delete'),
            'url'      => $this->getUrl('*/*/massDelete'),
        ]);

        $this->getMassactionBlock()->addItem(MassAction::NEWSLETTER_SUBSCRIBE, [
            'label'    => Mage::helper('customer')->__('Subscribe to Newsletter'),
            'url'      => $this->getUrl('*/*/massSubscribe'),
        ]);

        $this->getMassactionBlock()->addItem(MassAction::NEWSLETTER_UNSUBSCRIBE, [
            'label'    => Mage::helper('customer')->__('Unsubscribe from Newsletter'),
            'url'      => $this->getUrl('*/*/massUnsubscribe'),
        ]);

        /** @var Mage_Customer_Helper_Data $helper */
        $helper = $this->helper('customer');
        $groups = $helper->getGroups()->toOptionArray();

        array_unshift($groups, ['label' => '', 'value' => '']);
        $this->getMassactionBlock()->addItem(MassAction::ASSIGN_GROUP, [
            'label'        => Mage::helper('customer')->__('Assign a Customer Group'),
            'url'          => $this->getUrl('*/*/massAssignGroup'),
            'additional'   => [
                'visibility'    => [
                    'name'     => 'group',
                    'type'     => 'select',
                    'class'    => 'required-entry',
                    'label'    => Mage::helper('customer')->__('Group'),
                    'values'   => $groups,
                ],
            ],
        ]);

        return $this;
    }

    /**
     * @return string
     */
    #[\Override]
    public function getGridUrl()
    {
        return $this->getUrl('*/*/grid', ['_current' => true]);
    }

    /**
     * @param Mage_Customer_Model_Customer $row
     * @return string
     */
    #[\Override]
    public function getRowUrl($row)
    {
        return $this->getUrl('*/*/edit', ['id' => $row->getId()]);
    }
}
