# Copyright (c) 2017-2024, Lawrence Livermore National Security, LLC and
# other Shroud Project Developers.
# See the top-level COPYRIGHT file for details.
#
# SPDX-License-Identifier: (BSD-3-Clause)

"""
Generate Python module for C or C++ code.

Entire library in a single header.
One Extension module per class


Variables prefixes used by generated code:
SH_     C or C++ version of argument
SH_     From statements.local  fmt.PY_local
SHPy_   Python object which corresponds to the argument {py_var}
SHCPy_  Python API variable which corresponds to the argument's py_ctype {ctype_var}
        Used with Py_complex.
SHTPy_  A temporary object, usually from PyArg_Parse
        to be converted to SHPy_ object. {pytmp_var}
SHData_ Data of NumPy object (fmt.data_var} - intermediate variable
        of PyArray_DATA cast to correct type.
SHDPy_  PyArray_Descr object  {pydescr_var}
SHD_    npy_intp array for shape, {npy_dims_var}
SHC_    PyCapsule owner of memory of NumPy array. {py_capsule}
        Used to deallocate memory.
SHSize_ Size of dimension argument {fmt.size_var}
SHValue PY_typedef_converter variable {fmt.value_var}
SHPyResult Return Python object.
        Necessary when a return object is combined with others by Py_BuildValue.
"""

from __future__ import print_function
from __future__ import absolute_import

import collections
import os
import re

from . import error
from . import declast
from .declstr import gen_decl, gen_decl_noparams, gen_arg_as_c, gen_arg_as_cxx
from . import fcfmt
from . import todict
from . import statements
from . import typemap
from . import util
from . import whelpers
from .util import wformat, append_format, append_format_lst

from collections import OrderedDict

# The dictionary of Python Scope statements.
py_dict = OrderedDict() # dictionary of Scope of all expanded py_statements.
default_scope = None  # for statements

# If multiple values are returned, save up to build a tuple to return.
# else build value from ctor, then return ctorvar.
# The value may be built earlier (bool, array), if so blk0 will be None.
# format   - Format arg to PyBuild_Tuple.
# vargs    - Variable for PyBuild_Tuple.
# blk0     - How to build object when not using Py_BuildValue.
#            i.e. there is a single object to return.
# blk      - PyStmts when there are more than one return value.
# ctorvar  - Variable created by blk0.
#            This may not be the function return value but may be
#            from a single intent(out) argument.
BuildTuple = collections.namedtuple("BuildTuple", "format vargs blk0 blk ctorvar incref")

# type_object_creation - code to add variables to module.
ModuleTuple = collections.namedtuple(
    "ModuleTuple",
    "type_object_creation "
    "decl_arraydescr "
    "define_arraydescr "
    "call_arraydescr "
    "code_arraydescr "
)

# Info used per file.  Each library, namespace and class create a file.
FileTuple = collections.namedtuple(
    "FileTuple", "MethodBody MethodDef GetSetBody GetSetDef")


# The C type associated with PY_format fields used with PyArg_ParseTupleAndKeywords.
PY_format_to_type = dict(
    d="double",
    D="float complex",
    f="float",
    H="unsigned short",
    h="short",
    I="unsigned int",
    i="int",
    K="unsigned long long",
    k="unsigned long",
    l="long",
    L="long long",
    n="size_t",
    s="char *",
)

class Wrapp(util.WrapperMixin):
    """Generate Python bindings.
    """

    def __init__(self, newlibrary, config, splicers):
        """
        Args:
            newlibrary - ast.LibraryNode.
            config -
            splicers -
        """
        self.newlibrary = newlibrary
        self.language = newlibrary.language
        self.patterns = self.newlibrary.patterns
        self.config = config
        self.log = config.log
        self._init_splicer(splicers)
        self.comment = "//"
        self.cont = ""
        self.linelen = newlibrary.options.C_line_length
        self.doxygen_begin = "/**"
        self.doxygen_cont = " *"
        self.doxygen_end = " */"
        self.need_numpy = False
        self.enum_impl = []
        self.module_init_decls = []
        self.need_blah = False
        self.header_type_include = util.Header(newlibrary)  # header files in module header
        self.shared_helper = {} # All accumulated helpers
        self.cursor = error.get_cursor()
        update_statements_for_language(self.language)

    def XXX_begin_output_file(self):
        """Start a new class for output"""
        pass

    def XXX_end_output_file(self):
        pass

    def XXX_begin_class(self):
        pass

    def reset_file(self):
        """Start a new output file"""
        self.header_impl = util.Header(self.newlibrary)
        self.c_helper = {}
#        self.c_helper_include = {}  # include files in generated C header

    def wrap_library(self):
        newlibrary = self.newlibrary
        fmt_library = newlibrary.fmtdict

        if self.language == "c":
            fmt_library.PY_header_filename_suffix = "h"
            fmt_library.PY_impl_filename_suffix = "c"
            fmt_library.PY_extern_C_begin = ""
        else:
            fmt_library.PY_header_filename_suffix = "hpp"
            fmt_library.PY_impl_filename_suffix = "cpp"
            fmt_library.PY_extern_C_begin = 'extern "C" '
        fmt_library.PY_cleanup_decref = "Py_DECREF"

        # Format variables
        newlibrary.eval_template("PY_header_filename")
        newlibrary.eval_template("PY_utility_filename")
        fmt_library.PY_PyObject = "PyObject"
        fmt_library.PyObject = "PyObject"
        fmt_library.PY_param_self = "self"
        fmt_library.PY_param_args = "args"
        fmt_library.PY_param_kwds = "kwds"
        fmt_library.PY_used_param_self = False
        fmt_library.PY_used_param_args = False
        fmt_library.PY_used_param_kwds = False
        fmt_library.PY_member_object = "XXXPY_member_object"
        fmt_library.PY_member_data = "XXXPY_member_data"
        fmt_library.py_ctype = None

        fmt_library.npy_rank = "0"   # number of dimensions
        fmt_library.npy_dims_var = fmt_library.nullptr # shape variable
        fmt_library.npy_intp_decl = ""     # shape array definition
        fmt_library.npy_intp_asgn = ""     # shape array assignment

        # Variables to accumulate output lines
        self.py_class_decl = []  # code for header file
        self.py_utility_definition = []
        self.py_utility_declaration = []
        self.py_utility_functions = []
        append_format(self.module_init_decls,
                      "PyObject *{PY_prefix}error_obj;", fmt_library)

        # reserved the 0 slot of capsule_order
        self.add_capsule_code('--none--', ['// Do not release'])
        fmt_library.capsule_order = "0"
        self.need_blah = False  # Not needed if no there gc routines are added.

        self.wrap_namespace(newlibrary.wrap_namespace, top=True)
        self.write_utility_file()
        self.write_module_header(newlibrary)
        self.write_setup()

    def wrap_namespace(self, node, top=False):
        """Wrap a library or namespace.

        Each class is written into its own file.

        Args:
            node - ast.LibraryNode, ast.NamespaceNode
            top  - True if top level module, else submodule.
        """
        node.eval_template("PY_module_filename")
        modinfo = ModuleTuple([], [], [], [], [])
        fileinfo = FileTuple([], [], [], [])

        if top:
            # have one namespace level, then replace name each time
            self._push_splicer("namespace")
            self._push_splicer("XXX") # placer holder
        for ns in node.namespaces:
            if ns.wrap.python:
                self.wrap_namespace(ns)
                self.register_submodule(ns, modinfo)
        if top:
            self._pop_splicer("XXX")  # This name will not match since it is replaced.
            self._pop_splicer("namespace")
        else:
            # Skip file component in scope_file for splicer name.
            self._update_splicer_top("::".join(node.scope_file[1:]))

        # preprocess all classes first to allow them to reference each other
        for cls in node.classes:
            if not cls.wrap.python:
                continue

            # XXX - classes and structs as classes
            ntypemap = cls.typemap
            fmt = cls.fmtdict
            ntypemap.PY_format = "O"

            # PyTypeObject for class
            cls.eval_template("PY_PyTypeObject")

            # PyObject for class
            cls.eval_template("PY_PyObject")

            fmt.PY_to_object_idtor_func = wformat("PP_{cxx_class}_to_Object_idtor", fmt)
            fmt.PY_to_object_func = wformat("PP_{cxx_class}_to_Object", fmt)
            fmt.PY_from_object_func = wformat("PP_{cxx_class}_from_Object", fmt)

            ntypemap.PY_PyTypeObject = fmt.PY_PyTypeObject
            ntypemap.PY_PyObject = fmt.PY_PyObject
            ntypemap.PY_to_object_idtor = fmt.PY_to_object_idtor_func
            ntypemap.PY_to_object = fmt.PY_to_object_func
            ntypemap.PY_from_object = fmt.PY_from_object_func

        self._push_splicer("class")
        for cls in node.classes:
            if not cls.wrap.python:
                continue
            name = cls.name
            self.reset_file()
            self._push_splicer(name)
            if cls.parse_keyword == "struct" and cls.options.PY_struct_arg != "class":
                self.create_arraydescr(cls, modinfo)
            else:
                self.need_blah = True
                self.wrap_class(cls, modinfo)
            self._pop_splicer(name)
        self._pop_splicer("class")

        self.reset_file()
        self.wrap_enums(node)

        if node.functions:
            self._push_splicer("function")
            #            self._begin_class()
            self.wrap_functions(None, node.functions, fileinfo)
            self._pop_splicer("function")

        self.write_module(node, modinfo, fileinfo, top)

    def register_submodule(self, ns, modinfo):
        """Create code to add submodule to a module.

        Args:
            ns - ast.NamespaceNode
            modinfo - ModuleTuple
        """
        fmt_ns = ns.fmtdict

        self.module_init_decls.append(
            wformat("PyObject *{PY_prefix}init_{PY_module_init}(void);", fmt_ns))

        output = modinfo.type_object_creation
        output.append(
            wformat("""
{{+
PyObject *submodule = {PY_prefix}init_{PY_module_init}();
if (submodule == {nullptr})
+INITERROR;-
Py_INCREF(submodule);
PyModule_AddObject(m, (char *) "{PY_module_name}", submodule);
-}}""",
                    fmt_ns,
                )
        )

    def wrap_enums(self, node):
        """Wrap enums for library, namespace or class.

        Args:
            node - ast.LibraryNode, ast.NamespaceNode, ast.ClassNode
        """
        enums = node.enums
        if not enums:
            return
        self._push_splicer("enums")
        for enum in enums:
            self.wrap_enum(enum)
        self._pop_splicer("enums")

    def wrap_enum(self, node):
        """Wrap an enumeration.
        If module, use PyModule_AddIntConstant.
        If class, create a descriptor.
        Without a setter, it will be read-only.

        Args:
            node -
        """
        fmtmembers = node._fmtmembers

        ast = node.ast
        output = self.enum_impl
        if node.parent.nodename != "class":
            # library/namespace enumerations
            # m is module pointer from module_middle
            output.append("")
            append_format(output, "// enum {namespace_scope}{enum_name}",
                          node.fmtdict)
            for member in ast.members:
                fmt_id = fmtmembers[member.name]
                append_format(
                    output,
                    'PyModule_AddIntConstant(m, "{enum_member_name}",'
                    " {namespace_scope}{enum_member_name});",
                    fmt_id,
                )
        else:
            output.append("\n{+")
            append_format(output, "// enumeration {enum_name}", node.fmtdict)
            output.append("PyObject *tmp_value;")
            for member in ast.members:
                fmt_id = fmtmembers[member.name]
                append_format(
                    output,
                    "tmp_value = PyLong_FromLong("
                    "{namespace_scope}{enum_member_name});\n"
                    "PyDict_SetItemString("
                    "(PyObject*) {PY_PyTypeObject}.tp_dict,"
                    ' "{enum_member_name}", tmp_value);\n'
                    "Py_DECREF(tmp_value);",
                    fmt_id,
                )
            output.append("-}")

    def wrap_class(self, node, modinfo):
        """Create an extension type for a C++ class.

        Wrapper code added to py_type_object_creation.

        Args:
            node - ast.ClassNode.
            modinfo - ModuleTuple
        """
        cursor = self.cursor
        cursor.push_phase("Wrapp.wrap_class")
        cursor.push_node(node)

        self.log.write("class {1.name}\n".format(self, node))
        fileinfo = FileTuple([], [], [], [])
        options = node.options
        fmt_class = node.fmtdict
        node.create_node_map()

        node.eval_template("PY_type_filename")
        fmt_class.PY_this_call = wformat("self->{PY_type_obj}->", fmt_class)

        # Create code for module to add type to module
        output = modinfo.type_object_creation
        output.append("")
        if node.cpp_if:
            output.append("#" + node.cpp_if)
        append_format(output, "// {cxx_class}", fmt_class)
        if node.baseclass:
            # Only single inheritance supported.
            fmt_class.PY_PyTypeObject_base  = node.baseclass[0][2].typemap.PY_PyTypeObject
            append_format(output, "{PY_PyTypeObject}.tp_base = &{PY_PyTypeObject_base};", fmt_class)
        else:
            append_format(output,
"""{PY_PyTypeObject}.tp_new   = PyType_GenericNew;
{PY_PyTypeObject}.tp_alloc = PyType_GenericAlloc;""", fmt_class)
        append_format(output,
"""if (PyType_Ready(&{PY_PyTypeObject}) < 0)
+return RETVAL;-
Py_INCREF(&{PY_PyTypeObject});
PyModule_AddObject(m, "{cxx_class}", (PyObject *)&{PY_PyTypeObject});""",
                fmt_class
        )
        if node.cpp_if:
            output.append("#endif // " + node.cpp_if)

        self.header_type_include.add_cxx_header(node)
            
        # header declarations
        output = self.py_class_decl
        output.append("")
        output.append("// ------------------------------")
        if node.cpp_if:
            output.append("#" + node.cpp_if)

        output.append(wformat("extern PyTypeObject {PY_PyTypeObject};", fmt_class))

        self._create_splicer("C_declaration", output)
        output.append("")
        if options.literalinclude:
            output.append("// start object " + fmt_class.PY_PyObject)
        append_format(
            output,
            "typedef struct {{\n"
            "PyObject_HEAD\n"
            "+{namespace_scope}{cxx_type} * {PY_type_obj};\n"
            "int {PY_type_dtor};",
            fmt_class,
        )

        # Create a PyObject pointer for each pointer member
        # to contain the actual data.
        self.init_member_obj(node)
        # object which holds data - NumPy array
        # Returned by getter
        self.process_member_obj(
            node, "PyObject *{PY_member_object};", output)
        # object which hold data - PyCapsule
        # Used to release memory
        self.process_member_obj(
            node, "PyObject *{PY_member_data};", output)

        self._create_splicer("C_object", output)
        append_format(output, "-}} {PY_PyObject};", fmt_class)
        if options.literalinclude:
            output.append("// end object " + fmt_class.PY_PyObject)
        output.append("")

        self.create_class_utility_functions(node)
        if node.cpp_if:
            output.append("#endif // " + node.cpp_if)

        self.wrap_enums(node)

        for var in node.variables:
            if var.wrap.python:
                self.wrap_class_variable(node, var, fileinfo)

        # wrap methods
        self.tp_init_default = "0"
        self._push_splicer("method")
        self.wrap_functions(node, node.functions, fileinfo)
        self._pop_splicer("method")

        self.write_extension_type(node, fileinfo)
        cursor.pop_phase("Wrapp.wrap_class")
        cursor.pop_node(node)

    def create_class_utility_functions(self, node):
        """Create some utility functions to convert to and from a PyObject.
        These functions are used by PyArg_ParseTupleAndKeywords
        and Py_BuildValue node is a C++ class.
        """
        fmt = node.fmtdict

        fmt.PY_capsule_name = wformat("PY_{cxx_class}_capsule_name", fmt)

        if node.cpp_if:
            cpp_if = "#" + node.cpp_if + "\n"
            cpp_endif = "\n#endif  // " + node.cpp_if
        else:
            cpp_if = ""
            cpp_endif = ""

        self._push_splicer("utility")
        append_format(
            self.py_utility_definition,
            cpp_if + 'const char *{PY_capsule_name} = "{cxx_class}";' + cpp_endif,
            fmt,
        )
        append_format(
            self.py_class_decl,
            "extern const char *{PY_capsule_name};",
            fmt,
        )
        output = self.py_utility_functions

        ########################################
        # To object helper with idtor argument.
        to_object = wformat(
            """{PY_PyObject} *obj =\t PyObject_New({PY_PyObject}, &{PY_PyTypeObject});
if (obj == {nullptr})+
return {nullptr};
-obj->{PY_type_obj} = addr;
obj->{PY_type_dtor} = idtor;""",
            fmt,
        )
        to_object = to_object.split("\n")
        self.process_member_obj(
            node, "obj->{PY_member_object} = {nullptr};", to_object)
        self.process_member_obj(
            node, "obj->{PY_member_data} = {nullptr};", to_object)
        append_format(
            to_object,
            "return {cast_reinterpret}PyObject *{cast1}obj{cast2};",
            fmt)

        proto = wformat(
            "PyObject *{PY_to_object_idtor_func}({namespace_scope}{cxx_type} *addr,\t int idtor)",
            fmt,
        )
        self.py_class_decl.append(proto + ";")

        output.append("")
        if node.cpp_if:
            output.append("#" + node.cpp_if)
        output.append("// Wrap pointer to struct/class.")
        output.append(proto)
        output.append("{+")
        self._create_splicer("to_object", output, to_object)
        output.append("-}")

        ########################################
        # To object.
        to_object = wformat(
            """PyObject *voidobj;
PyObject *args;
PyObject *rv;

voidobj = PyCapsule_New(addr, {PY_capsule_name}, {nullptr});
args = PyTuple_New(1);
PyTuple_SET_ITEM(args, 0, voidobj);
rv = PyObject_Call((PyObject *) &{PY_PyTypeObject}, args, {nullptr});
Py_DECREF(args);
return rv;""",
            fmt,
        )
        to_object = to_object.split("\n")

        proto = wformat(
            "PyObject *{PY_to_object_func}({namespace_scope}{cxx_type} *addr)",
            fmt,
        )
        self.py_class_decl.append(proto + ";")

        output.append("")
        if node.cpp_if:
            output.append("#" + node.cpp_if)
        output.append("// converter which may be used with PyBuild.")
        output.append(proto)
        output.append("{+")
        self._create_splicer("to_object", output, to_object)
        output.append("-}")

        ########################################
        # From object.
        from_object = wformat(
            """if (obj->ob_type != &{PY_PyTypeObject}) {{
    // raise exception
    return 0;
}}
{PY_PyObject} * self = ({PY_PyObject} *) obj;
*addr = self->{PY_type_obj};
return 1;""",
            fmt,
        )
        from_object = from_object.split("\n")

        proto = wformat(
            "int {PY_from_object_func}(PyObject *obj, void **addr)", fmt
        )
        self.py_class_decl.append(proto + ";")

        output.append("")
        output.append("// converter which may be used with PyArg_Parse.")
        output.append(proto)
        output.append("{+")
        self._create_splicer(
            "from_object", self.py_utility_functions, from_object
        )
        output.append("-}")
        if node.cpp_if:
            output.append("#endif  // " + node.cpp_if)

        self._pop_splicer("utility")

    def create_arraydescr(self, node, modinfo):
        """Create a NumPy PyArray_Descr for a struct.
        Install into module.

        struct {
          int ifield;
          double dfield;
        };

        numpy.dtype(
          {'names': ['ifield', 'dfield'],
           'formats': [np.int32, np.float64]},
#           'offsets':[0,8],
#           'itemsize':12},
          align=True)

        Args:
            node - ast.ClassNode
            modinfo - ModuleTuple
        """
        options = node.options
        fmt = node.fmtdict

        self.need_numpy = True

        modinfo.decl_arraydescr.append(
            wformat(
                "extern PyArray_Descr *{PY_struct_array_descr_variable};", fmt
            )
        )
        modinfo.define_arraydescr.append(
            wformat("PyArray_Descr *{PY_struct_array_descr_variable};", fmt)
        )
        append_format(
            modinfo.call_arraydescr,
            "{PY_struct_array_descr_variable} = {PY_struct_array_descr_create}();\n"
            'PyModule_AddObject(m, "{PY_struct_array_descr_name}",'
            " \t(PyObject *) {PY_struct_array_descr_variable});",
            fmt,
        )
        output = modinfo.code_arraydescr
        output.append("")
        if options.literalinclude:
            output.append("// start " + fmt.PY_struct_array_descr_create)
        append_format(
            output,
            "// Create PyArray_Descr for {cxx_class}\n"
            "static PyArray_Descr *{PY_struct_array_descr_create}({void_proto})",
            fmt,
        )
        output.append("{")
        output.append(1)

        tmpfmt = util.Scope(fmt)
        tmpfmt.nvars = len(node.variables)
        append_format(
            output,
            "int ierr;\n"
            "PyObject *obj = {nullptr};\n"
            "PyObject * lnames = {nullptr};\n"
            "PyObject * ldescr = {nullptr};\n"
            "PyObject * dict = {nullptr};\n"
            "PyArray_Descr *dtype = {nullptr};\n"
            "\n"
            "lnames = PyList_New({nvars});\n"
            "if (lnames == {nullptr}) goto fail;\n"
            "ldescr = PyList_New({nvars});\n"
            "if (ldescr == {nullptr}) goto fail;",
            tmpfmt
        )

        for i, var in enumerate(node.variables):
            ast = var.ast
            declarator = ast.declarator
            name = declarator.user_name
            output.extend(
                [
                    "",
                    "// " + name,
                    'obj = PyString_FromString("{}");'.format(name),
                    "if (obj == {}) goto fail;".format(fmt.nullptr),
                    "PyList_SET_ITEM(lnames, {}, obj);".format(i),
                ]
            )

            arg_typemap = ast.typemap
            if ast.declarator.is_pointer():
                PYN_typenum = "NPY_INTP"
            else:
                PYN_typenum = arg_typemap.PYN_typenum
            output.extend(
                [
                    "obj = (PyObject *) PyArray_DescrFromType({});".format(
                        PYN_typenum
                    ),
                    "if (obj == {}) goto fail;".format(fmt.nullptr),
                    "PyList_SET_ITEM(ldescr, {}, obj);".format(i),
                ]
            )

            # XXX - add offset and itemsize to be explicit?

        append_format(
            output,
            "obj = {nullptr};\n"
            "\n"
            "dict = PyDict_New();\n"
            "if (dict == {nullptr}) goto fail;\n"
            'ierr = PyDict_SetItemString(dict, "names", lnames);\n'
            "if (ierr == -1) goto fail;\n"
            "lnames = {nullptr};\n"
            'ierr = PyDict_SetItemString(dict, "formats", ldescr);\n'
            "if (ierr == -1) goto fail;\n"
            "ldescr = {nullptr};\n"
            # 'Py_INCREF(Py_True);\n'
            # 'ierr = PyDict_SetItemString(descr, "aligned", Py_True);\n'
            # 'if (ierr == -1) goto fail;\n'
            "ierr = PyArray_DescrAlignConverter(dict, &dtype);\n"
            "if (ierr == 0) goto fail;\n"
            "return dtype;",
            fmt
        )
        append_format(
            output,
            "^fail:\n"
            "Py_XDECREF(obj);\n"
            "if (lnames != {nullptr}) {{+\n"
            "for (int i=0; i < {nvars}; i++) {{+\n"
            "Py_XDECREF(PyList_GET_ITEM(lnames, i));\n"
            "-}}\n"
            "Py_DECREF(lnames);\n"
            "-}}\n"
            "if (ldescr != {nullptr}) {{+\n"
            "for (int i=0; i < {nvars}; i++) {{+\n"
            "Py_XDECREF(PyList_GET_ITEM(ldescr, i));\n"
            "-}}\n"
            "Py_DECREF(ldescr);\n"
            "-}}\n"
            "Py_XDECREF(dict);\n"
            "Py_XDECREF(dtype);\n"
            "return {nullptr};",
            tmpfmt
        )
        #    int PyArray_RegisterDataType(descr)

        output.append(-1)
        output.append("}")
        if options.literalinclude:
            output.append("// end " + fmt.PY_struct_array_descr_create)

    def wrap_class_variable(self, parent, node, fileinfo):
        """Wrap a VariableNode in a class/struct with descriptors.

        Args:
            node - ast.VariableNode.
            fileinfo - FileTuple
        """
        options = node.options
        ast = node.ast
        declarator = ast.declarator
        arg_typemap = ast.typemap
        
        func_cursor = self.cursor.push_node(node)
        fmt_var = node.fmtdict
        fmt_var.PY_getter = wformat(options.PY_member_getter_template, fmt_var)
        fmt_var.PY_setter = fmt_var.nullptr  # readonly
        # How to find other fields in struct.
        fmt_var.PY_struct_context = wformat("{PY_param_self}->{PY_type_obj}->", fmt_var)

        fmt = util.Scope(fmt_var)
        # c_var is used with PY_ctor
        fmt.c_var_raw = wformat("{PY_struct_context}{field_name}", fmt_var)
        fmt.c_var = fmt.c_var_raw
        fmt.ctor_expr = fmt.c_var_raw
        fmt.cxx_var = fmt.c_var
        fmt.c_var_non_const = fmt.c_var
        fmt.c_var_obj = wformat("{PY_param_self}->{PY_member_object}", fmt)
        fmt.c_var_data = wformat("{PY_param_self}->{PY_member_data}", fmt)
        fmt.cxx_var_obj = fmt.c_var_obj
        fmt.cxx_var_data = fmt.c_var_data
        fmt.py_var = "value"  # Used with PY_get
        fmt.PY_array_arg = options.PY_array_arg
        fmt.c_type = arg_typemap.c_type

        py_struct_dimension(parent, node, fmt)
        indirect_stmt = declarator.get_indirect_stmt()
        abstract = statements.find_abstract_declarator(ast)

        if arg_typemap.PY_get:
            fmt.PY_get = wformat(arg_typemap.PY_get, fmt)
        
        if arg_typemap.PYN_descr:
            # class
            fmt.PYN_descr = arg_typemap.PYN_descr
        else:
            fmt.PYN_typenum = arg_typemap.PYN_typenum

        stmts = ['py', 'descr', abstract]
        if indirect_stmt != "scalar":
            # Pointers and static arrays.
            stmts.append(options.PY_array_arg)
            if ast.const:
                # get a non-const pointer for NumPy
                fmt.c_var_non_const = wformat(
                    "{cast_const}{c_type} *{cast1}{c_var}{cast2}", fmt)

        intent_blk = lookup_stmts(stmts)
        output = fileinfo.GetSetBody
        func_cursor.stmt = intent_blk
        ########################################
        # getter
        output.append("")
        if options.debug:
            self.document_stmts(output, ast, intent_blk.name)
        append_format(
            output,
            "static PyObject *{PY_getter}("
            "{PY_PyObject} *{PY_param_self},"
            "\t void *SHROUD_UNUSED(closure))\n"
            "{{+",
            fmt,
        )
        fmt.cxx_decl = gen_arg_as_cxx(ast, name="rv")
        if arg_typemap.PY_ctor:
            fmt.ctor = wformat(arg_typemap.PY_ctor, fmt)
        else:
            fmt.ctor = "UUUctor"

        if intent_blk.name == 'py_default':
            intent_blk = None
        if intent_blk:
            self.update_descr_code_blocks(
                "getter", intent_blk, fmt, output)
        else:
#            linenumber = options.get("__line__", "?")
            output.append("#error no py_statements getter for {}"
                          .format(stmts0))
        output.append("-}")

        ########################################
        # setter
        if not declarator.attrs.get("readonly"):
            fmt_var.PY_setter = wformat(
                options.PY_member_setter_template, fmt_var
            )

            output.append("")
            if options.debug:
                self.document_stmts(output, ast, intent_blk.name)
            append_format(
                output,
                "static int {PY_setter}("
                "{PY_PyObject} *{PY_param_self}, PyObject *{py_var},"
                "\t void *SHROUD_UNUSED(closure))\n{{+",
                fmt
            )

            if intent_blk:
                fmt.cast_type = ast.as_cast(
                    language=self.language)
                self.update_descr_code_blocks(
                    "setter", intent_blk, fmt, output)
            else:
                output.append("#error no py_statements setter for {}"
                              .format(stmts0))
            # XXX - allow user to add error checks on value
            output.append("return 0;\n-}")

        self.cursor.pop_node(node)
        # Set pointers to functions
        fileinfo.GetSetDef.append(
            # XXX - the (char *) only needed for C++
            wformat(
                '{{(char *)"{variable_name}",\t '
                "(getter){PY_getter},\t "
                "(setter){PY_setter},\t "
                "{nullptr}, "  # doc
                "{nullptr}}},",
                fmt_var,
            )
        )  # closure

    def update_descr_code_blocks(self, name, stmts, fmt, output):
        """Format descr code.
        Process stmt.getter_helper and stmt.setter_helper.

        Args:
            name   - "getter" "setter"
            stmts  - util.Scope
            fmt    - util.Scope
            output - descr code/
        """
        if stmts.need_numpy:
            self.need_numpy = True
        helpers = getattr(stmts, name + "_helper")
        self.set_fmt_hnamefunc(helpers, fmt)
        # update_code_blocks
        append_format_lst(output, getattr(stmts, name), fmt)

    def set_fmt_fields(self, cls, fcn, ast, bind, is_result=False):
        """
        Set format fields for ast.
        Used with arguments and results.

        Args:
            cls - ast.ClassNode or None
            fcn   - ast.FunctionNode of calling function.
            ast - declast.Declaration
                  Abstract Syntax Tree of argument or result
            fmt - format dictionary
        """
        fmt = bind.fmtdict
        declarator = ast.declarator
        typemap = ast.typemap
        if typemap.PY_PyObject:
            fmt.PyObject = typemap.PY_PyObject
        if typemap.PY_PyTypeObject:
            fmt.PyTypeObject = typemap.PY_PyTypeObject
        if typemap.PYN_descr:
            fmt.PYN_descr = typemap.PYN_descr

        if ast.template_arguments:
            # XXX - need to move numpy_type to TemplateFormat then use
            #     targs[0].numpy_type in statements.
            vtypemap = ast.template_arguments[0].typemap
            fmt.numpy_type = vtypemap.PYN_typenum
            statements.set_template_fields(ast, fmt)
            fmt.npy_rank = "1"
            if is_result:
                fmt.npy_dims_var = "SHD_" + fmt.C_result
            else:
                fmt.npy_dims_var = "SHD_" + ast.declarator.user_name
            # Dimensions must be in npy_intp type array.
            # XXX - assumes 1-d
            fmt.npy_intp_decl = wformat("npy_intp {npy_dims_var}[1];\n", fmt)
            # XXX - cxx_var may not have prefix yet.
            fmt.npy_intp_asgn = wformat("{npy_dims_var}[0] = {cxx_var}->size();\n", fmt)

        dimension = bind.meta["dim_ast"]
        rank = bind.meta["rank"]
        if rank is not None:
            fmt.rank = str(rank)
        elif dimension:
            class_context = "self->{}->".format(fmt.PY_type_obj)
            visitor = ToDimension(cls, fcn, fmt, class_context)
            visitor.visit(dimension)
            fmt.rank = str(visitor.rank)

            fmt.npy_rank = str(visitor.rank)
            if is_result:
                fmt.npy_dims_var = "SHD_" + fmt.C_result
            else:
                fmt.npy_dims_var = "SHD_" + declarator.user_name
            # Dimensions must be in npy_intp type array.
            fmt.npy_intp_decl = wformat(
                "npy_intp {npy_dims_var}[{npy_rank}];\n", fmt)

            # Assign each rank of dimension.
            fmtdim = []
            fmtsize = []
            for i, dim in enumerate(visitor.shape):
                fmtdim.append("{}[{}] = {};\n".format(
                    fmt.npy_dims_var, i, dim))
                fmtsize.append("({})".format(dim))
            fmt.npy_intp_asgn = "\n".join(fmtdim)
            if len(fmtsize) > 1:
                fmt.array_size = "*\t".join(fmtsize)
            else:
                fmt.array_size = visitor.shape[0]
        elif declarator.is_indirect():
            fmt.array_size = "1"  # assume scalar

#        fmt.c_type = typemap.c_type
        fmt.cxx_type = wformat(typemap.cxx_type, fmt) # expand cxx_T

    def set_cxx_nonconst_ptr(self, ast, fmt):
        """Set fmt.cxx_nonconst_ptr.
        A non-const pointer to cxx_var (which may be same as c_var).
        cxx_addr is used with references.
        """
        if self.language == "c":
            if ast.const:
                fmt.cxx_nonconst_ptr = wformat(
                    "({cxx_type} *) {cxx_addr}{cxx_var}", fmt)
            else:
                fmt.cxx_nonconst_ptr = wformat(
                    "{cxx_addr}{cxx_var}", fmt)
        elif ast.const:
            # cast away constness
            fmt.cxx_nonconst_ptr = wformat(
                "const_cast<{cxx_type} *>\t({cxx_addr}{cxx_var})",
                fmt
            )
        else:
            fmt.cxx_nonconst_ptr = wformat("{cxx_addr}{cxx_var}", fmt)

    def set_fmt_hnamefunc(self, helper_lst, fmt):
        """process helper functions from py_statements.c_helper

        helper is of the form:  helper_name:alias

        Alias is needed when the helper_name is expanded,
        ex contains {cxx_type}
        """
        for i, helper in enumerate(helper_lst):
            c_helper = wformat(helper, fmt)
            parts = c_helper.split(":")
            name = self.add_helper(parts[0])
            if len(parts) == 1:
                setattr(fmt, "c_helper_" + parts[0], name)
            else:
                setattr(fmt, "c_helper_" + parts[1], name)

    def implied_blk(self, node, bind, pre_call):
        """Add the implied attribute to the call block.

        Called after all input arguments have their fmtpy dictionary
        updated.
        Added into wrapper after post_parse code is inserted --
        i.e. all intent in,inout arguments have been evaluated
        and PyArrayObjects created.

        Args:
            node -
            bind - statements.BindArg
            pre_call - list of strings
        """
        implied = bind.meta["implied"]
        fmt = bind.fmtdict
        fmt.c_implied = py_implied(implied, node)
        append_format_lst(pre_call, bind.stmt.call, fmt)

    def intent_out(self, typemap, intent_blk, fmt):
        """Add code for post-call.
        Create PyObject from C++ value to return.
        Used with function results, intent(OUT) and intent(INOUT) arguments.

        Args:
            typemap - typemap of C++ variable.
            intent_blk -
            fmt - format dictionary

        Create code to return out argument as a PyObject.
        If there are multiple out arguments, then Py_BuildValue
        will be used to create a tuple of values, so the
        code in blk may not be used in the wrapper.

        Return a BuildTuple instance.
        """
        blk = None
        incref = False
        
        if intent_blk.object_created:
            # Explicit code exists to create object.
            # For example, NumPy intent(OUT) arguments as part of pre-call.
            # If post_call is None, the Object has already been created
            build_format = "N"  # "O" increments reference; "N" does not.
            vargs = fmt.py_var
            blk0 = None
            if intent_blk.incref_on_return:
                # Used when there is only one return value and the object
                # was not created by the function.
                incref = wformat("Py_INCREF({py_var});", fmt)
        else:
            # Decide values for Py_BuildValue
            build_format = typemap.PY_build_format or typemap.PY_format
            vargs = typemap.PY_build_arg
            if not vargs:
                vargs = "{cxx_var}"

            if typemap.PY_ctor:
                if typemap.py_ctype:
                    fmt.ctor_expr = typemap.pytype_to_pyctor.format(ctor_expr=fmt.ctor_expr)
                    if fmt.py_ctype is None:
                        # Declare variable unless already declared by intent(inout)
                        fmt.py_ctype = typemap.py_ctype
                        fmt.ctype_var = "SHCPY_" + fmt.c_var
                        declare = [wformat("{py_ctype} {ctype_var};", fmt)]
                    else:
                        declare = []
                    blk = util.Scope(PyStmts,
                        declare=declare,
                        post_call=[
                            wformat(typemap.cxx_to_pytype, fmt),
                        ]
                    )
                vargs = wformat(vargs, fmt)
                ctor = typemap.PY_ctor.format(ctor_expr=fmt.ctor_expr)
                declare0 = "{PyObject} * {py_var} = {nullptr};"
                post_call0 = "{py_var} = " + ctor + ";"
            else:
                # ex. long long does not define PY_ctor.
                vargs = wformat(vargs, fmt)
                fmt.PY_build_format = build_format
                fmt.vargs = vargs
                declare0 = "{PyObject} * {py_var} = {nullptr};"
                post_call0 = '{py_var} = Py_BuildValue("{PY_build_format}", {vargs});'
            blk0 = util.Scope(PyStmts,
                declare=[wformat(declare0, fmt)],
                post_call=[wformat(post_call0, fmt)],
            )

        return BuildTuple(build_format, vargs, blk0, blk, fmt.py_var, incref)

    def wrap_functions(self, cls, functions, fileinfo):
        """Wrap functions for a library or class.
        Compute overloading map.

        Args:
            cls - ast.ClassNode
            functions -
            fileinfo - FileTuple
        """
        self.cursor.push_phase("Wrapp.wrap_functions")
        overloaded_methods = {}
        for function in functions:
            flist = overloaded_methods.setdefault(function.name, [])
            if not function.wrap.python:
                continue
            if not function.options.PY_create_generic:
                continue
            flist.append(function)
        self.overloaded_methods = overloaded_methods

        for function in functions:
            self.wrap_function(cls, function, fileinfo)

        self.multi_dispatch(functions, fileinfo)
        self.cursor.pop_phase("Wrapp.wrap_functions")

    def wrap_function(self, cls, node, fileinfo):
        """Write a Python wrapper for a C or C++ function.

        Args:
            cls  - ast.ClassNode or None for functions
            node - ast.FunctionNode.
            fileinfo - FileTuple

        fmt.c_var   - name of variable in PyArg_ParseTupleAndKeywords
        fmt.cxx_var - name of variable in c++ call.
        fmt.py_var  - name of PyObject variable

        # Used to prevent compiler warnings about unused variables.
        fmt.PY_used_param_args - True/False if parameter args is used
        fmt.PY_used_param_kwds - True/False if parameter kwds is used

        fmt.PY_error_return - 'NULL' for all wrappers except constructors
                              which are called via tp_init and require -1.
        """

        # need_rv_decl - need Return Value declaration.
        #  The simplest case is to assign to rv as part of calling function.
        #  When default arguments are present, a switch statement is create
        #  so set need_rv_decl = True to declare variable once,
        #  then call wrapped function several times.
        #  If goto_fail, set to True to avoid "crosses initialization" error.
        options = node.options
        if not node.wrap.python:
            return

        cursor = self.cursor
        func_cursor = cursor.push_node(node)

        if options.PY_array_arg not in ["numpy", "list"]:
            linenumber = options.get("__line__", "?")
            cursor.ast(linenumber,
                       "Illegal value for PY_array_arg '{}'".
                       format(options.PY_array_arg))
        if options.PY_struct_arg not in ["numpy", "list", "class"]:
            linenumber = options.get("__line__", "?")
            cursor.ast(linenumber,
                       "Illegal value for PY_struct_arg '{}'".
                       format(options.PY_struct_arg))

        if cls:
            cls_function = "method"
        else:
            cls_function = "function"
        self.log.write("Python {0} {1.declgen}\n".format(cls_function, node))

        fmt_func = node.fmtdict
        fmt = node._fmtlang.setdefault("py", util.Scope(fmt_func))
        fmt.PY_doc_string = "documentation"
        fmt.PY_array_arg = options.PY_array_arg

        ast = node.ast
        declarator = ast.declarator
        CXX_subprogram = declarator.get_subprogram()
        result_typemap = ast.typemap
        is_ctor = declarator.is_ctor
        is_dtor = declarator.is_dtor
        #        is_const = ast.const
        ml_flags = []

        if cls:
            if "static" in ast.storage:
                ml_flags.append("METH_STATIC")
                fmt_func.PY_this_call = (
                    fmt_func.namespace_scope + fmt_func.class_scope
                )
            else:
                fmt.PY_used_param_self = True

        if is_dtor:
            # Added in tp_del from write_tp_func.
            cursor.pop_node(node)
            return
        elif is_ctor:
            fmt_func.PY_type_method = "tp_init"
            node.eval_template("PY_type_impl")
            fmt_func.PY_name_impl = fmt_func.PY_type_impl
            self.tp_init_default = fmt_func.PY_type_impl
            fmt.PY_error_return = "-1"
        else:
            node.eval_template("PY_name_impl")
            fmt.PY_error_return = fmt_func.nullptr

        # XXX if a class, then knock off const since the PyObject
        # is not const, otherwise, use const from result.
        # This has been replaced by gen_arg methods, but not sure about const.
        #        if result_typemap.base == 'shadow':
        #            is_const = False
        #        else:
        #            is_const = None
        bind_result = self.process_function_result(cls, node, fmt)
        fmt_result = bind_result.fmtdict
        result_blk = bind_result.stmt
        func_cursor.stmt = result_blk
        stmts_comments = []
        if options.debug:
            stmts_comments.append(
                "// ----------------------------------------")
            stmts_comments.append(
                "// Function:  " + gen_decl_noparams(ast))
            self.document_stmts(stmts_comments, ast, result_blk.name)

        PY_code = []

        # arguments to PyArg_ParseTupleAndKeywords
        parse_format = []
        parse_vargs = []

        # arguments to Py_BuildValue
        build_tuples = []

        # Code blocks
        # Accumulate code from statements.
        declare_code = []  # variables for function
        post_declare_code = []
        post_parse_code = []
        pre_call_code = []
        post_call_code = []  # Create objects passed to PyBuildValue
        cleanup_code = []
        fail_code = []

        cxx_call_list = []

        if result_blk.arg_declare is not None:
            append_format_lst(declare_code, result_blk.arg_declare, fmt_result)

        # parse arguments
        # call function based on number of default arguments provided
        default_calls = []  # each possible default call
        found_default = False
        found_optional = False  # Optional added to parse_format
        set_optional = []
        if node._has_default_arg:
            declare_code.append("Py_ssize_t SH_nargs = 0;")
            append_format(
                PY_code,
                "if (args != {nullptr}) SH_nargs += PyTuple_Size(args);\n"
                "if (kwds != {nullptr}) SH_nargs += PyDict_Size(args);",
                fmt
            )

        goto_fail = False
        args = ast.declarator.params
        arg_names = []    # Arguments to function, intent in or inout.
        arg_offsets = []
        arg_implied = []  # Collect implied arguments
        offset = 0
        npyargs = 0       # Number of intent in or inout arguments.
        for arg in args:
            func_cursor.stmt = None
            func_cursor.arg = arg
            declarator = arg.declarator
            arg_name = declarator.user_name
            bind_arg = statements.get_arg_bind(node, arg, "py")
            fmt_arg = statements.set_bind_fmtdict(bind_arg, fmt)
            fmt_arg.c_var = arg_name
            fmt_arg.cxx_var = arg_name
            fmt_arg.py_var = "SHPy_" + arg_name
            fmt_arg.data_var = "SHData_" + arg_name
            fmt_arg.size_var = "SHSize_" + arg_name
            fmt_arg.value_var = "SHValue_" + arg_name
            fmt_arg.gen = fcfmt.FormatGen(node, arg, fmt_arg, self.language)

            arg_typemap = arg.typemap
            fmt_arg.numpy_type = arg_typemap.PYN_typenum
            # Add formats used by py_statements
            fmt_arg.c_type = arg_typemap.c_type
            if arg.const:
                fmt_arg.c_const = "const "
            else:
                fmt_arg.c_const = ""
            if declarator.is_pointer():
                fmt_arg.cxx_addr = ""
                fmt_arg.cxx_member = "->"
                fmt_arg.ctor_expr = "*" + fmt_arg.c_var
            else:
                fmt_arg.cxx_addr = "&"
                fmt_arg.cxx_member = "."
                fmt_arg.ctor_expr = fmt_arg.c_var
            update_fmt_from_typemap(fmt_arg, arg_typemap)
            attrs = declarator.attrs
            meta = bind_arg.meta

            self.set_fmt_fields(cls, node, arg, bind_arg)
            self.set_cxx_nonconst_ptr(arg, fmt_arg)
            pass_var = fmt_arg.c_var  # The variable to pass to the function
            as_object = False
            rank = meta["rank"]
            dimension = meta["dimension"]
            hidden = meta["hidden"]
            implied = attrs.get("implied", None)
            intent = meta["intent"]
            sgroup = arg_typemap.sgroup
            abstract = statements.find_abstract_declarator(arg)
            stmts = None

            intent_blk = None
            if node._generated == "struct_as_class_ctor":
                stmts = ["py", "ctor", abstract, options.PY_array_arg]
                intent_blk = lookup_stmts(stmts)
                if intent_blk.name == "py_default":
                    intent_blk = None
                struct_member = node.struct_members[arg_name]
                struct_fmt = struct_member.fmtdict
                fmt_arg.field_name = struct_fmt.field_name
                fmt_arg.PY_member_object = struct_fmt.PY_member_object
                field_size = struct_member.ast.declarator.get_array_size()
                if field_size is not None:
                    fmt_arg.field_size = field_size
                if not found_optional:
                    parse_format.append("|")  # add once
                    found_optional = True
            deref = meta["deref"]
            if deref not in ["scalar", "raw"]:
                deref = None
            if intent_blk is not None:
                pass
            elif declarator.is_function_pointer():
                stmts = ["py", intent, "procedure"]
            elif implied:
                stmts = ["py", "implied", abstract]
                meta["implied"] = implied
                arg_implied.append(bind_arg)
            elif sgroup == "char":
                stmts = ["py", intent, abstract]
                charlen = attrs.get("charlen")
                if charlen:
                    fmt_arg.charlen = charlen
                    stmts.append("charlen")
            elif arg_typemap.base == "struct":
                stmts = ["py", intent, abstract, arg_typemap.PY_struct_as]
            elif arg_typemap.base == "vector":
                stmts = ["py", intent, abstract, options.PY_array_arg]
            elif rank or dimension:
                # ex. (int * arg1 +intent(in) +rank(1))
                stmts = ["py", intent, abstract, options.PY_array_arg]
            elif deref == "raw":
                # A single pointer.
                stmts = ["py", intent, abstract, deref]
            else:
                # Scalar argument
                # ex. (int * arg1 +intent(in))
                stmts = ["py", intent, abstract]
            if options.debug:
                stmts_comments.append(
                    "// ----------------------------------------")
                stmts_comments.append("// Argument:  " + gen_decl(arg))
            if stmts is not None:
                if intent_blk is None:
                    intent_blk = lookup_stmts(stmts)
                # Useful for debugging.  Requested and found path.
                fmt_arg.stmt = intent_blk.name
                # Add some debug comments to function.
                if options.debug:
                    self.document_stmts(stmts_comments, arg, intent_blk.name)
            elif options.debug:
                stmts_comments.append(
                    self.comment + " Exact:     " + intent_blk.name)

            func_cursor.stmt = intent_blk
            bind_arg.stmt = intent_blk
            self.name_temp_vars(arg_name, bind_arg)
            self.set_fmt_hnamefunc(intent_blk.c_helper, fmt_arg)
            
            converter, lang = fcfmt.find_arg_converter("f", self.language, arg_typemap)
            if intent_blk.post_declare:
                # Any c to cxx conversion is explicit in statements.
                pass
            elif intent != "out" and converter:
                # XXX - Used by MPI and c_to_cxx
                # Make intermediate C++ variable
                # Needed to pass address of variable.
                # Convert type like with enums or MPI_Comm.
                # Helpful with debugging.
                fmt_arg.c_abstract_decl = gen_arg_as_c(
                    arg, name=False, add_params=False)
                fmt_arg.cxx_abstract_decl = gen_arg_as_cxx(
                    arg, name=False, add_params=False, as_ptr=True)
                fmt_arg.cxx_var = "SH_" + fmt_arg.c_var
                fmt_arg.cxx_decl = gen_arg_as_cxx(arg,
                    name=fmt_arg.cxx_var, add_params=False
                )
                fmt_arg.cxx_val = wformat(converter, fmt_arg)
                append_format(post_declare_code,
                              "{cxx_decl} =\t {cxx_val};", fmt_arg)
                pass_var = fmt_arg.cxx_var

            statements.apply_fmtdict_from_stmts(bind_arg)

            # Declare argument variable.
            if intent_blk.arg_declare is not None:
                # Explicit declarations from py_statements.
                append_format_lst(declare_code, intent_blk.arg_declare, fmt_arg)
            elif hasattr(arg_typemap, "is_typedef"):
                # XXX - this helps typedefs to define PyArg type instead C wrapper type
                declare_code.append("{} {};".format(
                    PY_format_to_type[arg_typemap.PY_format], arg_name))
            else:
                # Since all declarations are at the top, remove const
                # since it will be assigned later.
                # The type must be usable with ParseTupleAndKeywords.
                # enums will use ci_type.
                junk = gen_arg_as_c(arg, remove_const=True, lang=lang)
                declare_code.append(junk + ";")
            
            if implied or hidden:
                # Argument is implied from other arguments.
                pass
            elif intent in ["inout", "in"]:
                # names to PyArg_ParseTupleAndKeywords
                arg_names.append(arg_name)
                arg_offsets.append("(char *) SH_kwcpp+%d" % offset)
                offset += len(arg_name) + 1

                # XXX default should be handled differently
                if declarator.init is not None:
                    # Default value argument.
                    if not found_optional:
                        parse_format.append("|")  # add once
                        found_optional = True
                    found_default = True
                    # Since this argument is optional, save current state
                    # so we can process without this argument.
                    # Cleanup should always do Py_XDECREF instead of
                    # Py_DECREF since PyObject pointers may be NULL due
                    # to different paths of execution in switch statement.
                    fmt_func.PY_cleanup_decref = "Py_XDECREF"
                    # call for default arguments  (num args, arg string)
                    default_calls.append(
                        (
                            npyargs,
                            len(post_declare_code),
                            len(post_parse_code),
                            len(pre_call_code),
                            ",\t ".join(cxx_call_list),
                        )
                    )
                npyargs = npyargs + 1

                # Declare C variable - may be PyObject.
                # add argument to call to PyArg_ParseTypleAndKeywords
                if intent_blk.parse_format:
                    # Explicitly specified parse_format
                    # Must also define parse_args.
                    fmt_arg.pytmp_var = "SHTPy_" + fmt_arg.c_var
                    parse_format.append(intent_blk.parse_format)
                    append_format_lst(parse_vargs, intent_blk.parse_args, fmt_arg)
                elif arg_typemap.PY_PyTypeObject:
                    # Expect object of given type
                    # cxx_var is declared by py_statements.intent_out.post_parse.
                    fmt_arg.py_object = arg_typemap.PY_PyObject or "PyObject"
                    append_format(declare_code, "{py_object} * {py_var};", fmt_arg)
                    pass_var = fmt_arg.cxx_var
                    parse_format.append(arg_typemap.PY_format)
                    parse_format.append("!")
                    parse_vargs.append("&" + arg_typemap.PY_PyTypeObject)
                    parse_vargs.append("&" + fmt_arg.py_var)
                elif arg_typemap.PY_from_object:
                    # Use function to convert object
                    # cxx_var created directly (no c_var)
#                    append_format(declare_code, "{cxx_decl};", fmt_arg)
                    print("XXXX unused")
                    # XXX this code is not hit by the testsuite.
                    # XXX not sure how if cxx_decl is still needed.
                    pass_var = fmt_arg.cxx_var
                    parse_format.append(arg_typemap.PY_format)
                    parse_format.append("&")
                    parse_vargs.append(arg_typemap.PY_from_object)
                    parse_vargs.append("&" + fmt_arg.cxx_var)
                elif arg_typemap.py_ctype:
                    # Python object uses a API type for contents (ex. Py_complex)
                    fmt_arg.py_ctype = arg_typemap.py_ctype
                    fmt_arg.ctype_var = "SHCPY_" + fmt_arg.c_var
                    append_format(declare_code, "{py_ctype} {ctype_var};", fmt_arg)
                    parse_format.append(arg_typemap.PY_format)
                    parse_vargs.append("&" + fmt_arg.ctype_var)
                    fmt_arg.ctype_expr = arg_typemap.pytype_to_cxx.format(
                        work_var=fmt_arg.ctype_var)
                    append_format(post_parse_code, "{c_var} = {ctype_expr};", fmt_arg)
                else:
                    parse_format.append(arg_typemap.PY_format)
                    parse_vargs.append("&" + fmt_arg.c_var)

            if intent in ["inout", "out"]:
                if not hidden:
                    # output variable must be a pointer
                    build_tuples.append(
                        self.intent_out(arg_typemap, intent_blk, fmt_arg)
                    )

            # Code to convert parsed values (C or Python) to C++.
            self.add_stmt_capsule(intent_blk, fmt_arg)
            goto_fail = goto_fail or intent_blk.goto_fail
            self.need_numpy = self.need_numpy or intent_blk.need_numpy
            update_code_blocks(locals(), intent_blk, fmt_arg)
            self.header_impl.add_statements_headers_PY(intent_blk)

            # Pass correct value to wrapped function.
            if intent_blk.arg_call:
                append_format_lst(cxx_call_list, intent_blk.arg_call, fmt_arg)
            else:
                cxx_call_list.append(pass_var)
        # end for arg in args:
        func_cursor.arg = None
        func_cursor.stmt = result_blk

        # Add implied argument initialization to pre_call_code
        for bind_arg in arg_implied:
            intent_blk = self.implied_blk(node, bind_arg, pre_call_code)

        need_blank = False  # needed before next debug header
        if not arg_names:
            # no input arguments
            ml_flags.append("METH_NOARGS")
        else:
            ml_flags.append("METH_VARARGS")
            ml_flags.append("METH_KEYWORDS")
            fmt.PY_used_param_args = True
            fmt.PY_used_param_kwds = True
            need_blank = True

            if self.language == "cxx":
                kw_const = "const "
                fmt.PyArg_kwlist = "const_cast<char **>(SHT_kwlist)"
            else:
                kw_const = ""
                fmt.PyArg_kwlist = "SHT_kwlist"
            declare_code.append(
                kw_const
                + 'char *SHT_kwlist[] = {\f"'
                + '",\f"'.join(arg_names)
                + '",\f'
                + fmt.nullptr
                + ' };'
            )
            parse_format.extend([":", fmt.function_name])
            fmt.PyArg_format = "".join(parse_format)
            fmt.PyArg_vargs = ",\t ".join(parse_vargs)
            PY_code.extend(set_optional)
            append_format(
                PY_code,
                "if (!PyArg_ParseTupleAndKeywords"
                "({PY_param_args}, {PY_param_kwds},\t "
                '"{PyArg_format}",\t {PyArg_kwlist}, '
                "\t{PyArg_vargs}))\n"
                "+return {PY_error_return};-",
                fmt,
            )

        # call with all arguments
        default_calls.append(
            (
                npyargs,
                len(post_declare_code),
                len(post_parse_code),
                len(pre_call_code),
                ",\t ".join(cxx_call_list),
            )
        )

        # Add function pre_call code.
        need_blank0 = True
        if CXX_subprogram == "function":
            self.add_stmt_capsule(result_blk, fmt_result)
        # Result pre_call is added once before all default argument cases.
        if result_blk.pre_call:
            if need_blank0:
                PY_code.extend(["", "// result pre_call"])
            append_format_lst(PY_code, result_blk.pre_call, fmt_result)

        if found_default:
            PY_code.append("switch (SH_nargs) {")

        # Build up code for a function.
        # Call once for each default argument.
        added_fail_declare = False
        for npyargs, post_declare_len, post_parse_len, pre_call_len, call_list in default_calls:
            if found_default:
                PY_code.append("case %d:" % npyargs)
                PY_code.append(1)
                need_blank = False
                if post_declare_len or post_parse_len or pre_call_len:
                    # Only add scope if necessary.
                    # There may be declarations in these code blocks.
                    # Need to avoid error:
                    # jump to label 'fail' crosses initialization of ...
                    PY_code.append("{")
                    PY_code.append(1)
                    extra_scope = True
                else:
                    extra_scope = False

            if post_declare_len:
                if options.debug:
                    if need_blank:
                        PY_code.append("")
                    PY_code.append("// post_declare")
                PY_code.extend(post_declare_code[:post_declare_len])
                need_blank = True

            if post_parse_len:
                if options.debug:
                    if need_blank:
                        PY_code.append("")
                    PY_code.append("// post_parse")
                PY_code.extend(post_parse_code[:post_parse_len])
                need_blank = True

            if pre_call_len:
                if options.debug:
                    if need_blank:
                        PY_code.append("")
                    PY_code.append("// pre_call")
                PY_code.extend(pre_call_code[:pre_call_len])
                need_blank = True
            fmt.PY_call_list = call_list

            if options.debug and need_blank:
                PY_code.append("")

            capsule_order = None
            fmt.C_call_function = wformat(
                "{PY_this_call}{function_name}"
                "{CXX_template}({PY_call_list})", fmt)
            if (found_default or goto_fail) and result_blk.fail_call:
                if not added_fail_declare:
                    added_fail_declare = True
                    append_format_lst(declare_code, result_blk.fail_declare, fmt_result)
                append_format_lst(PY_code, result_blk.fail_call, fmt_result)
                need_rv_decl = False # The call block needs to declare the variable.
            elif result_blk.call:
                append_format_lst(PY_code, result_blk.call, fmt_result)
                need_rv_decl = False # The call block needs to declare the variable.

            if node.PY_error_pattern:
                lfmt = util.Scope(fmt)
                lfmt.c_var = fmt.C_result
                lfmt.cxx_var = fmt.C_result
                append_format(
                    PY_code, self.patterns[node.PY_error_pattern], lfmt
                )

            if found_default:
                PY_code.append("break;")
                PY_code.append(-1)
                if extra_scope:
                    PY_code.append("}")
                    PY_code.append(-1)
                    need_blank = False
        # End of loop over default arguments.
        if found_default:
            append_format(
                PY_code,
                "default:+\n"
                "PyErr_SetString(PyExc_ValueError,"
                "\t \"Wrong number of arguments\");\n"
                "return {nullptr};\n"
#                "goto fail;\n"
                "-}}",
                fmt)
# XXX - need to add a extra scope to deal with goto in C++
#            goto_fail = True;

        # Compute return value
        if CXX_subprogram == "function":
            ttt0 = self.intent_out(result_typemap, result_blk, fmt_result)
            # Add result to front of return tuple.
            build_tuples.insert(0, ttt0)
            if ttt0.format in ["O", "N"]:
                # If an object has already been created,
                # use another variable for the result.
                fmt.PY_result = "SHPyResult"
            update_code_blocks(locals(), result_blk, fmt_result)
            goto_fail = goto_fail or result_blk.goto_fail
            self.need_numpy = self.need_numpy or result_blk.need_numpy

        # If only one return value, return the ctor
        # else create a tuple with Py_BuildValue.
        return_code = []
        if result_blk.c_return:
            append_format_lst(return_code, result_blk.c_return, fmt_result)
        elif not build_tuples:
            return_code.append("Py_RETURN_NONE;")
        elif len(build_tuples) == 1:
            # Return a single object already created in build_tuples
            blk0 = build_tuples[0].blk0
            incref = build_tuples[0].incref
            if blk0 is not None:
                # Format variables are already expanded in intent_out.
                declare_code.extend(blk0.declare)
                post_call_code.extend(blk0.post_call)
            elif incref:
                post_call_code.append(incref)
            fmt.py_var = build_tuples[0].ctorvar
            append_format(return_code, "return (PyObject *) {py_var};", fmt)
        else:
            # fmt=format for function. Do not use fmt_result here.
            # There may be no return value, only intent(OUT) arguments.
            # create tuple object
            for blk in build_tuples:
                if blk.blk is None:
                    continue
                # Format variables are already expanded in intent_out.
                declare_code.extend(blk.blk.declare)
                post_call_code.extend(blk.blk.post_call)
            fmt.PyBuild_format = "".join([ttt.format for ttt in build_tuples])
            fmt.PyBuild_vargs = ",\t ".join([ttt.vargs for ttt in build_tuples])
            rv_blk = util.Scope(PyStmts,
                declare=["PyObject *{PY_result} = {nullptr};  // return value object"],
                post_call=["{PY_result} = "
                           'Py_BuildValue("{PyBuild_format}",\t {PyBuild_vargs});'],
                # Since this is the last statement before the Return,
                # no need to check for error. Just return NULL.
                # fail=["Py_XDECREF(SHPyResult);"],
            )
            update_code_blocks(locals(), rv_blk, fmt)
            append_format(return_code, "return {PY_result};", fmt)

        need_blank = False  # put return right after call
        if node._generated == "struct_as_class_ctor":
            if options.debug:
                PY_code.append("")
                PY_code.append("// post_call - initialize fields")
            # Create a convience variable to access struct.
            append_format(
                PY_code,
                "{namespace_scope}{cxx_type} *SH_obj = self->{PY_type_obj};",
                fmt)
            PY_code.extend(post_call_code)
            need_blank = True
        elif post_call_code and not is_ctor:
            # ctor does not need to build return values.
            # Called as the __init__ method.
            if options.debug:
                PY_code.append("")
                PY_code.append("// post_call")
            PY_code.extend(post_call_code)
            need_blank = True

        if cleanup_code:
            if options.debug:
                PY_code.append("")
                PY_code.append("// cleanup")
            PY_code.extend(cleanup_code)
            need_blank = True

        if options.debug and need_blank:
            PY_code.append("")
        PY_code.extend(return_code)

        if goto_fail:
            PY_code.extend(["", "^fail:"])
            PY_code.extend(fail_code)
            append_format(PY_code, "return {PY_error_return};", fmt)

        if len(declare_code):
            # Add blank line after declarations.
            declare_code.append("")
        if "py" in node.splicer:
            PY_force = node.splicer["py"]
            PY_impl = None
        else:
            PY_force = None
            PY_impl = [1] + declare_code + PY_code + [-1]

        expose = True
        if is_ctor:
            expose = False
        if len(self.overloaded_methods[ast.declarator.user_name]) > 1:
            # Only expose a multi-dispatch name, not each overload
            expose = False
        elif found_default:
            # Only one wrapper to deal with default arugments.
            # [C creates a wrapper per default argument]
            fmt = util.Scope(fmt)
            fmt.function_suffix = ""

        fmt.PY_ml_flags = "|".join(ml_flags)
        self.create_method(node, expose, is_ctor, fmt,
                           PY_force, PY_impl, stmts_comments,
                           fileinfo)
        cursor.pop_node(node)

    def create_method(self, node, expose, is_ctor, fmt,
                      PY_force, PY_impl, stmts_comments,
                      fileinfo):
        """Format the function.

        Args:
            node    - function node to wrap
                      or None when called from multi_dispatch.
            expose  - True if exposed to user.
            is_ctor - True if this is a constructor.
            fmt     - dictionary of format values.
            PY_force - list of inline splicer code.
            PY_impl - list of implementation code.
            stmts_comments -
            fileinfo - FileTuple
        """
        if node:
            cpp_if = node.cpp_if
        else:
            cpp_if = False

        body = fileinfo.MethodBody
        body.append("")
        if node and node.options.debug:
            body.extend(stmts_comments)
        if cpp_if:
            body.append("#" + node.cpp_if)
        if expose:
            append_format(
                body,
                "static char {PY_name_impl}__doc__[] =\n"
                '"{PY_doc_string}"\n'
                ";\n",
                fmt,
            )

        if node and node.options.doxygen and node.doxygen:
            self.write_doxygen(body, node.doxygen)
        if is_ctor:
            body.append("static int")
        else:
            body.append("static PyObject *")
        append_format(body, "{PY_name_impl}(", fmt)

        if fmt.PY_used_param_self:
            append_format(body, "  {PY_PyObject} *{PY_param_self},", fmt)
        else:
            append_format(
                body, "  PyObject *SHROUD_UNUSED({PY_param_self}),", fmt
            )
        if fmt.PY_used_param_args:
            append_format(body, "  PyObject *{PY_param_args},", fmt)
        else:
            append_format(
                body, "  PyObject *SHROUD_UNUSED({PY_param_args}),", fmt
            )
        if fmt.PY_used_param_args:
            append_format(body, "  PyObject *{PY_param_kwds})", fmt)
        else:
            append_format(
                body, "  PyObject *SHROUD_UNUSED({PY_param_kwds}))", fmt
            )

        body.append("{")
        # use function_suffix in splicer name since a single C++ function may
        # produce several methods.
        # XXX - make splicer name customizable?
        #        self._create_splicer(fmt.function_name, body, PY_impl)
        self._create_splicer(
            fmt.function_name +
            fmt.function_suffix +
            fmt.template_suffix,
            body,
            PY_impl,
            PY_force,
        )
        body.append("}")
        if cpp_if:
            body.append("#endif // " + cpp_if)

        if expose is True:
            if cpp_if:
                fileinfo.MethodDef.append("#" + cpp_if)
            # default name
            append_format(
                fileinfo.MethodDef,
                '{{"{function_name}{function_suffix}{template_suffix}",\t '
                "(PyCFunction){PY_name_impl},\t "
                "{PY_ml_flags},\t "
                "{PY_name_impl}__doc__}},",
                fmt,
            )
            if cpp_if:
                fileinfo.MethodDef.append("#endif // " + cpp_if)

    #        elif expose is not False:
    #            # override name
    #            fmt = util.Scope(fmt)
    #            fmt.expose = expose
    #            fileinfo.MethodDef.append( wformat('{{"{expose}", (PyCFunction){PY_name_impl}, {PY_ml_flags}, {PY_name_impl}__doc__}},', fmt))

    def name_temp_vars(self, rootname, bind):
        """Compute names of temporary wrapper variables.

        Create stmts.temps and stmts.local variables.
        """
        stmts = bind.stmt
        fmt = bind.fmtdict

        names = stmts.get("local", None)
        if names is not None:
            for name in names:
                setattr(fmt,
                        "py_local_{}".format(name),
                        "{}{}_{}".format(fmt.PY_local, rootname, name))
                # Enable older code to continue to work
                # using the old name
                if name == "cxx":
                    if bind.stmt["intent"] == "function":
                        fmt.py_local_cxx = fmt.CXX_local + rootname
                    else:
                        fmt.py_local_cxx = "SH_" + rootname
                    fmt.cxx_var = fmt.py_local_cxx
                elif name == "capsule":
                    if bind.stmt["intent"] == "function":
                        fmt.py_local_capsule = "SHC_" + fmt.CXX_local + rootname
                    else:
                        fmt.py_local_capsule = "SHC_" + rootname
                    fmt.py_capsule = fmt.py_local_capsule
        
    def process_function_result(self, cls, node, fmt):
        """Work on formatting for function result values.

        Return fmt_result
        Args:
            node    - FunctionNode to wrap.
            fmt     - dictionary of format values.
        """
        options = node.options
        ast = node.ast
        declarator = ast.declarator
        attrs = declarator.attrs
        is_ctor = declarator.is_ctor
        result_typemap = ast.typemap
        bind_result = statements.get_func_bind(node, "py")
        fmt_result = statements.set_bind_fmtdict(bind_result, fmt)  # fmt_func

        meta = bind_result.meta

        CXX_subprogram = declarator.get_subprogram()
        if CXX_subprogram != "subroutine":
            CXX_result = node.ast

            # Mangle result variable name to avoid possible conflict with arguments.
            fmt_result.cxx_var = wformat(
                "{CXX_local}{C_result}", fmt_result
            )

            if CXX_result.declarator.is_pointer():
                fmt_result.cxx_addr = ""
                fmt_result.cxx_member = "->"
                fmt_result.ctor_expr = "*" + fmt_result.cxx_var
            else:
                fmt_result.cxx_addr = "&"
                fmt_result.cxx_member = "."
                fmt_result.ctor_expr = fmt_result.cxx_var
            fmt_result.c_var = fmt_result.cxx_var
            fmt_result.py_var = fmt.PY_result
            fmt_result.data_var = "SHData_" + fmt_result.C_result
            fmt_result.size_var = "SHSize_" + fmt_result.C_result
            fmt_result.value_var = "SHValue_" + fmt_result.C_result
            fmt_result.numpy_type = result_typemap.PYN_typenum
            fmt_result.gen = fcfmt.FormatGen(node, node.ast, fmt_result, self.language)
            update_fmt_from_typemap(fmt_result, result_typemap)

            self.set_fmt_fields(cls, node, ast, bind_result, True)
            self.set_cxx_nonconst_ptr(ast, fmt_result)
        
        sgroup = result_typemap.sgroup
        abstract = statements.find_abstract_declarator(ast)
        stmts = None
        if CXX_subprogram == "subroutine":
            stmts = ["py", "subroutine"]
        elif is_ctor:
            stmts = ["py", meta["intent"], abstract]
        elif result_typemap.base == "struct":
            stmts = ["py", "function", abstract, options.PY_struct_arg]
        elif result_typemap.base == "vector":
            stmts = ["py", "function", abstract, options.PY_array_arg]
        elif sgroup == "native":
            stmts = ["py", "function", abstract]
            spointer = declarator.get_indirect_stmt()
            if spointer != "scalar":
                deref = meta["deref"]
                if deref == "scalar":
                    stmts.append(deref)
                else:
                    stmts.append(options.PY_array_arg)
        else:
            stmts = ["py", "function", abstract]
        result_blk = lookup_stmts(stmts)

        fmt_result.stmt = result_blk.name
        bind_result.stmt = result_blk
        self.name_temp_vars(fmt.C_result, bind_result)
        self.set_fmt_hnamefunc(result_blk.c_helper, fmt_result)
        statements.apply_fmtdict_from_stmts(bind_result)
                
        return bind_result

    def add_stmt_capsule(self, stmts, fmt):
        """Use code to release memory.
        Processes "destructor_name" and "destructor".

        For example, std::vector intent(out) must eventually release
        the vector via a capsule owned by the NumPy array.
        """
        # Create capsule destructor
        destructor_name = stmts["destructor_name"]
        if destructor_name:
            destructor_name = wformat(destructor_name, fmt)

            destructor = stmts["destructor"]
            if destructor:
                # Expand things like {cxx_T}
                del_work = []
                append_format_lst(del_work, destructor, fmt)
                destructor = del_work
            
            capsule_order = self.add_capsule_code(
                self.language + " " + destructor_name, destructor)
            fmt.capsule_order = capsule_order
                
    def write_tp_func(self, node, fmt_type, output):
        """Create functions for tp_init et.al.

        Args:
            node -
            fmt_type - dictionary used with PyTypeObject_template
                       to fill in type function names.
            output - list for generated functions.

        python:
          type: [ repr, str ]

        """
        # Type bodies must be filled in by user, no real way to guess
        # how to implement.
        # Except tp_init (constructor) and tp_del (destructor).
        fmt_func = node.fmtdict
        fmt = util.Scope(fmt_func)
        template = node.options.PY_type_impl_template
        PyObj = fmt_func.PY_PyObject
        if "type" in node.python:
            selected = node.python["type"][:]
            for auto in ["del"]:
                # Make some methods are there
                if auto not in selected:
                    selected.append(auto)
        else:
            selected = ["del"]

        # Dictionary of methods for bodies
        default_body = dict(richcompare=self.not_implemented)
        default_body["del"] = self.tp_del

        self._push_splicer("type")
        for typename in typenames:
            tp_name = "tp_" + typename
            if typename == "init":
                # The constructor method is used for tp_init
                fmt_type[tp_name] = self.tp_init_default
                continue
            if typename not in selected:
                fmt_type[tp_name] = fmt_func.nullptr
                continue
            fmt.PY_type_method = tp_name
            func_name = wformat(template, fmt)
            fmt_type[tp_name] = func_name
            tup = typefuncs[typename]
            output.append("static " + tup[0])
            output.append(
                ("{name} " + tup[1]).format(  # object used by tup[1]
                    name=func_name, object=PyObj
                )
            )
            output.append("{")
            default = default_body.get(typename, self.not_implemented_error)
            ret = fmt_func.nullptr if tup[2] == "NULL" else tup[2]
            default = default(node, typename, ret)

            # format and indent default bodies
            fmted = [1]
            append_format_lst(fmted, default, fmt_func)
            fmted.append(-1)

            self._create_splicer(typename, output, fmted)
            output.append("}")
        self._pop_splicer("type")


######
    def add_helper(self, name):
        """Use a helper function.
        Return the name of the function associated with helper.
        """
        self.c_helper[name] = True
        # Adjust for alias like with type char.
        return whelpers.PYHelpers[name]["c_fmtname"]
        
    def _gather_helper_code(self, name, done):
        """Add code from helpers.

        First recursively process dependent_helpers
        to add code in order.

        Args:
            name - Name of helper.
            done - Dictionary of previously processed helpers.
        """
        if name in done:
            return  # avoid recursion
        done[name] = True

        helper_info = whelpers.PYHelpers[name]
        if "dependent_helpers" in helper_info:
            for dep in helper_info["dependent_helpers"]:
                # check for recursion
                self._gather_helper_code(dep, done)

        scope = helper_info.get("scope", "file")
        # assert scope in ["file", "utility"]

        self.helper_need_numpy = (
            helper_info.get("need_numpy", False) or self.helper_need_numpy)

        lang_key = self.language + "_include"
        if lang_key in helper_info:
            for include in helper_info[lang_key]:
                self.helper_summary["include"][scope][include] = True
        elif "include" in helper_info:
            for include in helper_info["include"]:
                self.helper_summary["include"][scope][include] = True

        for key in ["proto", "source"]:
            lang_key = self.language + "_" + key 
            if lang_key in helper_info:
                self.helper_summary[key][scope].append(helper_info[lang_key])
            elif key in helper_info:
                self.helper_summary[key][scope].append(helper_info[key])

    def gather_helper_code(self, helpers):
        """Gather up all helpers requested and insert code into output.

        helpers should be self.c_helper or self.shared_helper

        Args:
            helpers - dictionary of helper names.
        """
        self.helper_summary = dict(
            include=dict(file={}, pwrap_impl={}),
            proto=dict(file=[], pwrap_impl=[]),
            source=dict(file=[], pwrap_impl=[]),
        )
        self.helper_need_numpy = False

        done = {}  # Avoid duplicates by keeping track of what's been gathered.
        for name in sorted(helpers.keys()):
            self._gather_helper_code(name, done)

    def find_file_helper_code(self):
        """Get "file" helper code.
        Add to shared_helper, then reset.

        Return dictionary of headers and list of source files.
        """
        if self.newlibrary.options.PY_write_helper_in_util:
            self.shared_helper.update(self.c_helper)
            self.c_helper = {}
            return {}, [], False
        self.gather_helper_code(self.c_helper)
        self.shared_helper.update(self.c_helper)
        self.c_helper = {}
        return (
            self.helper_summary["include"]["file"],
            self.helper_summary["source"]["file"],
            self.helper_need_numpy
        )

    def find_utility_helper_code(self):
        """Get "pwrap_impl" helper code.
        Added to PY_utility_filename and shared among files.

        Return list of code with typedefs.
        """
        self.gather_helper_code(self.shared_helper)
        return (
            self.helper_summary["include"]["pwrap_impl"],
            self.helper_summary["source"]["pwrap_impl"]
        )

    def find_shared_file_helper_code(self):
        """Get "file" helper code when added to utility file.
        """
        if self.newlibrary.options.PY_write_helper_in_util:
            self.gather_helper_code(self.shared_helper)
            return (
                self.helper_summary["include"]["file"],
                self.helper_summary["source"]["file"],
                self.helper_need_numpy,
            )
        return {}, [], False

######

    def write_extension_type(self, node, fileinfo):
        """
        Args:
            node - ast.ClassNode
            fileinfo - FileTuple
        """
        fmt = node.fmtdict
        fname = fmt.PY_type_filename

        hinclude, hsource, helper_need_numpy = self.find_file_helper_code()
        if helper_need_numpy:
            self.need_numpy = True
        # always include helper header
#        self.c_helper_include[library.fmtdict.C_header_utility] = True
#        self.shared_helper.update(self.c_helper)  # accumulate all helpers

        output = []
        if node.cpp_if:
            output.append("#" + node.cpp_if)

        append_format(output, '#include "{PY_header_filename}"', fmt)
        if self.need_numpy:
            self.add_numpy_includes(output)
        self._push_splicer("impl")

        # Use headers from implementation
        self.header_impl.add_shroud_dict(hinclude)
        self.header_impl.write_headers(output)

        self._create_splicer("include", output, blank=True)
        output.append(cpp_boilerplate)
        output.extend(hsource)
        self._create_splicer("C_definition", output, blank=True)
        self._create_splicer("additional_methods", output, blank=True)
        self._pop_splicer("impl")

        fmt_type = dict(
            PY_module_scope=fmt.PY_module_scope,
            PY_PyObject=fmt.PY_PyObject,
            PY_PyTypeObject=fmt.PY_PyTypeObject,
            cxx_class=fmt.cxx_class,
            nullptr=fmt.nullptr,
        )
        self.write_tp_func(node, fmt_type, output)

        output.extend(fileinfo.MethodBody)

        self._push_splicer("impl")
        self._create_splicer("after_methods", output)
        self._pop_splicer("impl")

        output.extend(fileinfo.GetSetBody)
        if fileinfo.GetSetDef:
            fmt_type["tp_getset"] = wformat(
                "{PY_prefix}{cxx_class}_getset", fmt
            )
            append_format(
                output, "\nstatic PyGetSetDef {tp_getset}[] = {{+", fmt_type
            )
            output.extend(fileinfo.GetSetDef)
            self._create_splicer("PyGetSetDef", output)
            append_format(output, "{{{nullptr}}}            /* sentinel */", fmt)
            output.append("-};")
        else:
            fmt_type["tp_getset"] = fmt.nullptr

        fmt_type["tp_methods"] = wformat("{PY_prefix}{cxx_class}_methods", fmt)
        append_format(
            output, "static PyMethodDef {tp_methods}[] = {{+", fmt_type
        )
        output.extend(fileinfo.MethodDef)
        self._create_splicer("PyMethodDef", output)
        append_format(
            output,
            "{{{nullptr},   (PyCFunction){nullptr}, 0, {nullptr}}}"
            "            /* sentinel */",
            fmt
        )
        output.append("-};")

        append_format(output, PyTypeObject_template, fmt_type)
        if node.cpp_if:
            output.append("#endif // " + node.cpp_if)

        self.config.pyfiles.append(os.path.join(self.config.python_dir, fname))
        self.write_output_file(fname, self.config.python_dir, output)

    def multi_dispatch(self, functions, fileinfo):
        """Look for overloaded methods.
        When found, create a method which will call each of the
        overloaded methods looking for the one which will accept
        the given arguments.

        Args:
            functions - list of ast.FunctionNode
            fileinfo - FileTuple
        """
        mdone = {}
        for function in functions:
            # preserve order of multi-dispatch functions
            mname = function.name
            if mname in mdone:
                continue
            mdone[mname] = True
            methods = self.overloaded_methods[mname]
            if len(methods) < 2:
                continue  # not overloaded

            node = methods[0]
            fmt_func = node.fmtdict
            fmt = util.Scope(fmt_func)
            fmt.function_suffix = ""
            fmt.template_suffix = ""
            fmt.PY_doc_string = "documentation"
            fmt.PY_ml_flags = "METH_VARARGS|METH_KEYWORDS"
            fmt.PY_used_param_self = True
            fmt.PY_used_param_args = True
            fmt.PY_used_param_kwds = True

            is_ctor = node.ast.declarator.is_ctor

            body = []
            body.append(1)
            body.append("Py_ssize_t SHT_nargs = 0;")
            append_format(
                body,
                "if (args != {nullptr}) SHT_nargs += PyTuple_Size(args);\n"
                "if (kwds != {nullptr}) SHT_nargs += PyDict_Size(args);",
                fmt
            )
            if is_ctor:
                fmt.PY_type_method = "tp_init"
                fmt.PY_name_impl = wformat(
                    node.options.PY_type_impl_template, fmt
                )
                fmt.PY_type_impl = fmt.PY_name_impl
                self.tp_init_default = fmt.PY_type_impl
                return_code = "return rv;"
                return_arg = "rv"
                fmt.PY_error_return = "-1"
                self.tp_init_default = fmt.PY_name_impl
                body.append("int rv;")
                expose = False
            else:
                fmt.PY_name_impl = wformat(
                    node.options.PY_name_impl_template, fmt
                )
                return_code = "return rvobj;"
                return_arg = "rvobj"
                fmt.PY_error_return = fmt.nullptr
                body.append("PyObject *rvobj;")
                expose = True

            for overload in methods:
                if overload.cpp_if:
                    body.append("#" + overload.cpp_if)
                if overload._nargs:
                    body.append(
                        "if (SHT_nargs >= %d && SHT_nargs <= %d) {+"
                        % overload._nargs
                    )
                else:
                    body.append(
                        "if (SHT_nargs == %d) {+" % len(overload.ast.declarator.params)
                    )
                append_format(
                    body,
                    return_arg + " = {PY_name_impl}(self, args, kwds);",
                    overload.fmtdict,
                )
                body.append("if (!PyErr_Occurred()) {+")
                body.append(return_code)
                body.append(
                    "-} else if (! PyErr_ExceptionMatches"
                    "(PyExc_TypeError)) {+"
                )
                body.append(return_code)
                body.append("-}\nPyErr_Clear();\n-}")
                if overload.cpp_if:
                    body.append("#endif // " + overload.cpp_if)

            body.append(
                "PyErr_SetString(PyExc_TypeError, "
                '"wrong arguments multi-dispatch");'
            )
            append_format(body, "return {PY_error_return};", fmt)
            body.append(-1)

            self.create_method(None, expose, is_ctor, fmt,
                               None, body, [], fileinfo)

    def write_module_header(self, node):
        """Write the header for the module.
        Args:
            node - ast.LibraryNode.
        """
        fmt = node.fmtdict
        fname = fmt.PY_header_filename
        self.header_type_include.add_cxx_header(node)
        hinclude, hsource = self.find_utility_helper_code()
        output = []

        # add guard
        guard = fname.replace(".", "_").upper()
        output.extend(["#ifndef %s" % guard, "#define %s" % guard])

        output.append("")
        output.append("#include <Python.h>")
        self.header_type_include.write_headers(output)

        output.append("")
        self._push_splicer("header")
        self._create_splicer("include", output)

        output.extend(hsource)
        if self.newlibrary.options.PY_write_helper_in_util:
            if self.helper_summary["proto"]["file"]:
                output.append("")
                output.append("// Helper functions.")
                output.extend(self.helper_summary["proto"]["file"])

        if self.py_utility_declaration:
            output.append("")
            output.append("// utility functions")
            output.extend(self.py_utility_declaration)

        if self.py_class_decl:
            output.extend(self.py_class_decl)
            output.append("// ------------------------------")

        #        output.extend(self.define_arraydescr)

        self._create_splicer("C_declaration", output, blank=True)
        self._pop_splicer("header")

        append_format(
            output,
            """
extern PyObject *{PY_prefix}error_obj;

#if PY_MAJOR_VERSION >= 3
{PY_extern_C_begin}PyMODINIT_FUNC PyInit_{PY_module_init}(void);
#else
{PY_extern_C_begin}PyMODINIT_FUNC init{PY_module_init}(void);
#endif
""",
            fmt,
        )
        output.append("#endif  /* %s */" % guard)
        #        self.config.pyfiles.append(
        #            os.path.join(self.config.python_dir, fname))
        self.write_output_file(fname, self.config.python_dir, output)

    def write_module(self, node, modinfo, fileinfo, top):
        """
        Write the Python extension module.
        Used with a Library or Namespace node

        Args:
            node - ast.LibraryNode
            modinfo - ModuleTuple
            fileinfo - FileTuple
            top - True = top module, else submodule.
        """
        fmt = node.fmtdict
        fname = fmt.PY_module_filename

        fmt.PY_library_doc = "library documentation"

        hinclude, hsource, helper_need_numpy = self.find_file_helper_code()
        if helper_need_numpy:
            self.need_numpy = True
        # always include helper header
#        self.c_helper_include[library.fmtdict.C_header_utility] = True
#        self.shared_helper.update(self.c_helper)  # accumulate all helpers

        output = []

        append_format(output, '#include "{PY_header_filename}"', fmt)
        if self.need_numpy:
            self.add_numpy_includes(output, top)

        self.header_impl.add_shroud_dict(hinclude)
        self.header_impl.write_headers(output)
        self._create_splicer("include", output, blank=True)
        output.append(cpp_boilerplate)
        output.extend(hsource)
        self._create_splicer("C_definition", output, blank=True)

        if top:
            output.extend(self.module_init_decls)
        output.extend(modinfo.define_arraydescr)

        self._create_splicer("additional_functions", output, blank=True)
        output.extend(fileinfo.MethodBody)

        append_format(
            output, "static PyMethodDef {PY_prefix}methods[] = {{", fmt
        )
        output.extend(fileinfo.MethodDef)
        append_format(
            output,
            "{{{nullptr},   (PyCFunction){nullptr}, 0, {nullptr}}}"
            "            /* sentinel */",
            fmt
        )
        output.append("};")

        output.extend(modinfo.code_arraydescr)

        if top:
            self.write_init_module(fmt, output, modinfo)
        else:
            self.write_init_submodule(fmt, output, modinfo)

        self.config.pyfiles.append(os.path.join(self.config.python_dir, fname))
        self.write_output_file(fname, self.config.python_dir, output)

    def write_init_module(self, fmt, output, modinfo):
        """Initialize the top level module.

        Uses Python's API for importing a module from a shared library.
        Deal with numpy initialization.

        Args:
            fmt - 
            output -
            modinfo - ModuleTuple
        """
        append_format(output, module_begin, fmt)
        self._create_splicer("C_init_locals", output)
        append_format(output, module_middle, fmt)
        if self.need_numpy:
            output.append("")
            output.append("import_array();")
        output.extend(modinfo.type_object_creation)
        output.extend(self.enum_impl)
        if modinfo.call_arraydescr:
            output.append("")
            output.append("// Define PyArray_Descr for structs")
            output.extend(modinfo.call_arraydescr)
        append_format(output, module_middle2, fmt)
        self._create_splicer("C_init_body", output)
        append_format(output, module_end, fmt)

    def write_init_submodule(self, fmt, output, modinfo):
        """Initialize namespace module.

        Always return a PyObject.
        """
        append_format(output, submodule_begin, fmt)
        output.extend(modinfo.type_object_creation)
#        output.extend(self.enum_impl)
        if modinfo.call_arraydescr:
            output.append("")
            output.append("// Define PyArray_Descr for structs")
            output.extend(modinfo.call_arraydescr)
        append_format(output, submodule_end, fmt)

    def add_numpy_includes(self, output, top=False):
        """Import numpy
        If top is True, then import_array will be called
        in the file being written.
        """
        if not top:
            output.append('#define NO_IMPORT_ARRAY')
        append_format(output,
                      '#define PY_ARRAY_UNIQUE_SYMBOL {PY_ARRAY_UNIQUE_SYMBOL}',
                      self.newlibrary.fmtdict)
        output.append("#define NPY_NO_DEPRECATED_API NPY_1_7_API_VERSION")
        output.append('#include "numpy/arrayobject.h"')
        
    def write_utility_file(self):
        """
        Do not write the file unless it has contents.
        """
        node = self.newlibrary
        fmt = node.fmtdict
        need_file = False
        hinclude, hsource, need_numpy = self.find_shared_file_helper_code()

        headers = util.Header(self.newlibrary)
        headers.add_shroud_file(fmt.PY_header_filename)
        headers.add_shroud_dict(hinclude)
        
        output = []
        if need_numpy:
            self.add_numpy_includes(output)
        headers.write_headers(output)
        output.append(cpp_boilerplate)

        if hsource:
            output.extend(hsource)
            need_file = True
        if self.py_utility_definition:
            output.append("")
            output.extend(self.py_utility_definition)
            need_file = True
        if self.py_utility_functions:
            output.append("")
            output.extend(self.py_utility_functions)
            need_file = True
        if self.need_blah:
            self.write_capsule_code(output, fmt)
            need_file = True
        if need_file:
            self.config.pyfiles.append(
                os.path.join(self.config.python_dir, fmt.PY_utility_filename)
            )
            self.write_output_file(
                fmt.PY_utility_filename, self.config.python_dir, output
            )

    def write_capsule_code(self, output, fmt):
        """Write a function used to delete memory when a
        NumPy array is deleted.

        Create a global variable of of context pointer used
        to switch to case used to release memory.

        Args:
            output -
            fmt -
        """
        append_format(
            output,
            "\n// ----------------------------------------\n"
            "typedef struct {{+\n"
            "const char *name;\n"
            "void (*dtor)(void *ptr);\n"
            "-}} {PY_dtor_context_typedef};",
            fmt
        )

        # Create variable with as array of {PY_dtor_context_typedef}
        # to contain function pointers to routines to release memory.
        fcnnames = []
        for i, name in enumerate(self.capsule_order):
            fcnname = fmt.PY_capsule_destructor_function + "_" + str(i)
            fcnnames.append((name, fcnname))
            output.append("\n// {} - {}".format(i, name))
            output.append("static void {}(void *ptr)".format(fcnname))
            output.append("{+")
            for line in self.capsule_code[name][1]:
                output.append(line)
            output.append("-}")

        output.append(
            "\n"
            "// Code used to release arrays for NumPy objects\n"
            "// via a Capsule base object with a destructor.\n"
            "// Context strings"
        )
        append_format(
            output, "static {PY_dtor_context_typedef} {PY_dtor_context_array}[] = {{+", fmt
        )
        for name in fcnnames:
            output.append('{{"{}", {}}},'.format(name[0], name[1]))
        output.append('{{{}, {}}},'.format(fmt.nullptr, fmt.nullptr))
        output.append("-};")

        # Write function to release from extension type.
        proto = wformat("void {PY_release_memory_function}(int icontext, void *ptr)", fmt)
        self.py_utility_declaration.append("extern " + proto + ";")
        output.append("\n// Release memory based on icontext.")
        output.append(proto)
        append_format(
            output,
            "{{+\n"
            "{PY_dtor_context_array}[icontext].dtor(ptr);\n"
            "-}}", fmt)

        # Write function to release NumPy capsule base object.
        proto = wformat("void *{PY_fetch_context_function}(int icontext)", fmt)
        self.py_utility_declaration.append("extern " + proto + ";")
        output.append("\n//Fetch garbage collection context.")
        output.append(proto)
        append_format(
            output,
            "{{+\n"
            "return {PY_dtor_context_array} + icontext;\n"
#            "return {cast_static}void *{cast1}({PY_dtor_context_array} + icontext){cast2};\n"
            "-}}",
            fmt)

        proto = wformat("void {PY_capsule_destructor_function}(PyObject *cap)", fmt)
        self.py_utility_declaration.append("extern " + proto + ";")
        output.append("\n// destructor function for PyCapsule")
        output.append(proto)
        append_format(
            output,
            "{{+\n"
            # 'const char* name = PyCapsule_GetName(cap);\n'
            'void *ptr = PyCapsule_GetPointer(cap, "{PY_numpy_array_capsule_name}");',
            fmt,
        )
        if self.language == "c":
            append_format(
                output,
                "{PY_dtor_context_typedef} * context = PyCapsule_GetContext(cap);", fmt)
        else:
            append_format(
                output,
                "{PY_dtor_context_typedef} * context = "
                "static_cast<{PY_dtor_context_typedef} *>\t("
                "PyCapsule_GetContext(cap));", fmt)
        output.append("context->dtor(ptr);")
        output.append("-}")

    capsule_code = {}
    capsule_order = []

    def add_capsule_code(self, name, lines):
        """Add unique names to capsule_code.
        Return index of name.

        Args:
            name -
            lines -
        """
        self.need_blah = True
        if name not in self.capsule_code:
            self.capsule_code[name] = (str(len(self.capsule_code)), lines)
            self.capsule_order.append(name)
        return self.capsule_code[name][0]

    def write_setup(self):
        """Write a setup.py file for the module"""
        library = self.newlibrary
        options = library.options
        fmt = library.fmtdict
        fname = "setup.py"

        if options.debug_testsuite:
            srcs = [ "'" + os.path.basename(name) + "'"
                     for name in self.config.pyfiles]
        else:
            srcs = [ "'" + name + "'" for name in self.config.pyfiles]
        fmt = dict(
            language="c++" if self.language == "cxx" else self.language,
            name=fmt.library_lower,
            source=",\n         ".join(srcs),
            include_dirs="None",
        )

        output = ["from setuptools import setup, Extension"]
        if self.need_numpy:
            output.append("import numpy")
            fmt["include_dirs"] = "[numpy.get_include()]"

        append_format(
            output, """
module = Extension(
    '{name}',
    sources=[
         {source}
    ],
    language='{language}',
    include_dirs = {include_dirs},
#    libraries = ['tcl83'],
#    library_dirs = ['/usr/local/lib'],      
#    extra_compile_args = [ '-O0', '-g' ],
#    extra_link_args =
)

setup(
    name='{name}',
    ext_modules = [module],""", fmt)
        setup = library.setup
        for key in [
                "author",
                "author_email",
                "description",
#                "long_description",
                "license",
                "url",
                "test_suite",
        ]:
            if key in setup:
                output.append("    {} = '{}',".format(key, setup[key]))
        output.append(")")
        self.comment = '#'
        self.write_output_file(fname, self.config.out_dir, output)

    def not_implemented_error(self, node, msg, ret):
        """A standard splicer for unimplemented code
        ret is the return value (NULL or -1 or '')

        Args:
            node - ast.ClassNode
            msg -
            ret -
        """
        lines = ['PyErr_SetString(PyExc_NotImplementedError, "%s");' % msg]
        if ret:
            lines.append("return %s;" % ret)
        else:
            lines.append("return;")
        return lines

    def not_implemented(self, node, msg, ret):
        """A standard splicer for rich comparison

        Args:
            node - ast.ClassNode
            msg -
            ret -
        """
        return [
            "Py_INCREF(Py_NotImplemented);",
            "return Py_NotImplemented;"
        ]

    def tp_del(self, node, msg, ret):
        """default method for tp_del.

        Args:
            node - ast.ClassNode
            msg  - 'del'
            ret  - ''
        """
        output = [
            "{PY_release_memory_function}(self->{PY_type_dtor}, self->{PY_type_obj});",
            "self->{PY_type_obj} = {nullptr};",
        ]
        self.process_member_obj(
            node, "Py_XDECREF(self->{PY_member_object});", output)
        self.process_member_obj(
            node, "Py_XDECREF(self->{PY_member_data});", output)
        return output

    def init_member_obj(self, node):
        """Update fmt for members of struct-as-class.
        """
        for var in node.variables:
            fmt = var.fmtdict
            if var.ast.declarator.is_array():
                fmt.py_var = "SHPy_" + fmt.variable_name
                var.eval_template("PY_member_object")
                var.eval_template("PY_member_data")

    def process_member_obj(self, node, text, output):
        """Loop over variables in the struct-as-class and add
        a line of formatted text for each pointer variable.
        """
        if node.parse_keyword != "struct":
            return
        print_header = True
        for var in node.variables:
            # var is VariableNode
            if not var.ast.declarator.is_array():
                continue
            if print_header:
                output.append("// Python objects for members.")
                print_header = False
            append_format(output, text, var.fmtdict)

# --- Python boiler plate

# Avoid warning errors about unused parameters
# Include in each source file and not the header file because
# we don't want to pollute the user's files.
cpp_boilerplate = """
#ifdef __cplusplus
#define SHROUD_UNUSED(param)
#else
#define SHROUD_UNUSED(param) param
#endif

#if PY_MAJOR_VERSION >= 3
#define PyInt_AsLong PyLong_AsLong
#define PyInt_FromLong PyLong_FromLong
#define PyInt_FromSize_t PyLong_FromSize_t
#define PyString_FromString PyUnicode_FromString
#define PyString_FromStringAndSize PyUnicode_FromStringAndSize
#endif"""

typenames = [
    "dealloc",
    "print",
    "compare",
    "getattr",
    "setattr",  # deprecated
    "getattro",
    "setattro",
    "repr",
    "hash",
    "call",
    "str",
    "init",
    "alloc",
    "new",
    "free",
    "del",
    "richcompare",
]


# return type, prototype, default return value
# [2] will be converted to fmt.nullptr if 'NULL'.
typefuncs = {
    "dealloc": ("void", "({object} *self)", ""),
    "print": ("int", "({object} *self, FILE *fp, int flags)", "-1"),
    "getattr": ("PyObject *", "({object} *self, char *name)", "NULL"),
    "setattr": ("int", "({object} *self, char *name, PyObject *value)", "-1"),
    "compare": ("int", "({object} *self, PyObject *)", "-1"),
    "repr": ("PyObject *", "({object} *self)", "NULL"),
    "hash": ("long", "({object} *self)", "-1"),
    "call": (
        "PyObject *",
        "({object} *self, PyObject *args, PyObject *kwds)",
        "NULL",
    ),
    "str": ("PyObject *", "({object} *self)", "NULL"),
    "getattro": ("PyObject *", "({object} *self, PyObject *name)", "NULL"),
    "setattro": (
        "int",
        "({object} *self, PyObject *name, PyObject *value)",
        "-1",
    ),
    "init": ("int", "({object} *self, PyObject *args, PyObject *kwds)", "-1"),
    "alloc": ("PyObject *", "(PyTypeObject *type, Py_ssize_t nitems)", "NULL"),
    "new": (
        "PyObject *",
        "(PyTypeObject *type, PyObject *args, PyObject *kwds)",
        "NULL",
    ),
    "free": ("void", "(void *op)", ""),
    "del": ("void", "({object} *self)", ""),
    "richcompare": (
        "PyObject *",
        "({object} *self, PyObject *other, int opid)",
        "",
    ),
}

# Note: that these strings have some format character to control indenting
#  + indent
#  - deindent
#  0 noindention

PyTypeObject_template = """
static char {cxx_class}__doc__[] =
"virtual class"
;

/* static */
PyTypeObject {PY_PyTypeObject} = {{+
PyVarObject_HEAD_INIT({nullptr}, 0)
"{PY_module_scope}.{cxx_class}",                       /* tp_name */
sizeof({PY_PyObject}),         /* tp_basicsize */
0,                              /* tp_itemsize */
/* Methods to implement standard operations */
(destructor){tp_dealloc},                 /* tp_dealloc */
(printfunc){tp_print},                   /* tp_print */
(getattrfunc){tp_getattr},                 /* tp_getattr */
(setattrfunc){tp_setattr},                 /* tp_setattr */
#if PY_MAJOR_VERSION >= 3
{nullptr},                               /* tp_reserved */
#else
(cmpfunc){tp_compare},                     /* tp_compare */
#endif
(reprfunc){tp_repr},                    /* tp_repr */
/* Method suites for standard classes */
{nullptr},                              /* tp_as_number */
{nullptr},                              /* tp_as_sequence */
{nullptr},                              /* tp_as_mapping */
/* More standard operations (here for binary compatibility) */
(hashfunc){tp_hash},                    /* tp_hash */
(ternaryfunc){tp_call},                 /* tp_call */
(reprfunc){tp_str},                    /* tp_str */
(getattrofunc){tp_getattro},                /* tp_getattro */
(setattrofunc){tp_setattro},                /* tp_setattro */
/* Functions to access object as input/output buffer */
{nullptr},                              /* tp_as_buffer */
/* Flags to define presence of optional/expanded features */
Py_TPFLAGS_DEFAULT,             /* tp_flags */
{cxx_class}__doc__,         /* tp_doc */
/* Assigned meaning in release 2.0 */
/* call function for all accessible objects */
(traverseproc){nullptr},                /* tp_traverse */
/* delete references to contained objects */
(inquiry){nullptr},                     /* tp_clear */
/* Assigned meaning in release 2.1 */
/* rich comparisons */
(richcmpfunc){tp_richcompare},                 /* tp_richcompare */
/* weak reference enabler */
0,                              /* tp_weaklistoffset */
/* Added in release 2.2 */
/* Iterators */
(getiterfunc){nullptr},                 /* tp_iter */
(iternextfunc){nullptr},                /* tp_iternext */
/* Attribute descriptor and subclassing stuff */
{tp_methods},                             /* tp_methods */
{nullptr},                              /* tp_members */
{tp_getset},                             /* tp_getset */
{nullptr},                              /* tp_base */
{nullptr},                              /* tp_dict */
(descrgetfunc){nullptr},                /* tp_descr_get */
(descrsetfunc){nullptr},                /* tp_descr_set */
0,                              /* tp_dictoffset */
(initproc){tp_init},                   /* tp_init */
(allocfunc){tp_alloc},                  /* tp_alloc */
(newfunc){tp_new},                    /* tp_new */
(freefunc){tp_free},                   /* tp_free */
(inquiry){nullptr},                     /* tp_is_gc */
{nullptr},                              /* tp_bases */
{nullptr},                              /* tp_mro */
{nullptr},                              /* tp_cache */
{nullptr},                              /* tp_subclasses */
{nullptr},                              /* tp_weaklist */
(destructor){tp_del},                 /* tp_del */
0,                              /* tp_version_tag */
#if PY_MAJOR_VERSION >= 3
(destructor){nullptr},                  /* tp_finalize */
#endif
-}};"""


module_begin = """
/*
 * init{library_lower} - Initialization function for the module
 * *must* be called init{library_lower}
 */
static char {PY_prefix}_doc__[] =
"{PY_library_doc}"
;

struct module_state {{
    PyObject *error;
}};

#if PY_MAJOR_VERSION >= 3
#define GETSTATE(m) ((struct module_state*)PyModule_GetState(m))
#else
#define GETSTATE(m) (&_state)
static struct module_state _state;
#endif

#if PY_MAJOR_VERSION >= 3
static int {library_lower}_traverse(PyObject *m, visitproc visit, void *arg) {{
    Py_VISIT(GETSTATE(m)->error);
    return 0;
}}

static int {library_lower}_clear(PyObject *m) {{
    Py_CLEAR(GETSTATE(m)->error);
    return 0;
}}

static struct PyModuleDef moduledef = {{
    PyModuleDef_HEAD_INIT,
    "{library_lower}", /* m_name */
    {PY_prefix}_doc__, /* m_doc */
    sizeof(struct module_state), /* m_size */
    {PY_prefix}methods, /* m_methods */
    {nullptr}, /* m_reload */
    {library_lower}_traverse, /* m_traverse */
    {library_lower}_clear, /* m_clear */
    NULL  /* m_free */
}};

#define RETVAL m
#define INITERROR return {nullptr}
#else
#define RETVAL
#define INITERROR return
#endif

{PY_extern_C_begin}PyMODINIT_FUNC
#if PY_MAJOR_VERSION >= 3
PyInit_{PY_module_init}(void)
#else
init{PY_module_init}(void)
#endif
{{+
PyObject *m = {nullptr};
const char * error_name = "{library_lower}.Error";
"""

module_middle = """

/* Create the module and add the functions */
#if PY_MAJOR_VERSION >= 3
m = PyModule_Create(&moduledef);
#else
m = Py_InitModule4("{PY_module_name}", {PY_prefix}methods,\t
+{PY_prefix}_doc__,
(PyObject*){nullptr},PYTHON_API_VERSION);
#endif
-if (m == {nullptr})
+return RETVAL;-
struct module_state *st = GETSTATE(m);"""

# XXX - +INITERROR;-
module_middle2 = """
{PY_prefix}error_obj = PyErr_NewException((char *) error_name, {nullptr}, {nullptr});
if ({PY_prefix}error_obj == {nullptr})
+return RETVAL;-
st->error = {PY_prefix}error_obj;
PyModule_AddObject(m, "Error", st->error);
"""

module_end = """
/* Check for errors */
if (PyErr_Occurred())
+Py_FatalError("can't initialize module {PY_module_name}");-
return RETVAL;
-}}
"""

# A submodule always returns a PyObject.
submodule_begin = """
#if PY_MAJOR_VERSION >= 3
static char {PY_prefix}_doc__[] =
"XXX submodule doc"  //"{PY_library_doc}"
;

struct module_state {{
    PyObject *error;
}};

static struct PyModuleDef moduledef = {{
    PyModuleDef_HEAD_INIT,
    "{PY_module_scope}", /* m_name */
    {PY_prefix}_doc__, /* m_doc */
    sizeof(struct module_state), /* m_size */
    {PY_prefix}methods, /* m_methods */
    {nullptr}, /* m_reload */
//    {library_lower}_traverse, /* m_traverse */
//    {library_lower}_clear, /* m_clear */
    {nullptr}, /* m_traverse */
    {nullptr}, /* m_clear */
    {nullptr}  /* m_free */
}};
#endif
#define RETVAL {nullptr}

PyObject *{PY_prefix}init_{PY_module_init}(void)
{{+
PyObject *m;
#if PY_MAJOR_VERSION >= 3
m = PyModule_Create(&moduledef);
#else
m = Py_InitModule3((char *) "{PY_module_name}", {PY_prefix}methods, {nullptr});
#endif
if (m == {nullptr})
+return {nullptr};-
"""
submodule_end = """
return m;
-}}
"""

def update_fmt_from_typemap(fmt, ntypemap):
    """Copy fields from typemap to use with creating output"""
    # XXX maybe use in wrap_namespace
    if ntypemap.PY_to_object_idtor:
        fmt.PY_to_object_idtor_func = ntypemap.PY_to_object_idtor
        # XXX - not sure if needed, avoid clutter for now.
#        fmt.PY_to_object_func = ntypemap.PY_to_object
#        fmt.PY_from_object_func = ntypemap.PY_from_object

######################################################################

def py_struct_dimension(parent, var, fmt):
    """
    Process ast.array or the dimension attribute.

    Set format fields.
    npy_rank        = rank of NumPy array.  Scalars are 1.
    npy_intp_values = comma separated list of dimensions
    npy_intp_size   = size of array, multiplied ranks.

    ex. npy_intp {npy_dims_var}[{npy_rank}] = {{ {npy_intp_values} }};

    Args:
        parent - ast.ClassNode.
        var    - ast.VariableNode.
        fmt    - util.Scope.
    """
    fmt.npy_dims_var = "dims"  # Name of local variables
    # fmt.npy_intp_asgn     # assign to
    #    fmt.npy_intp_values     # comma separated list of values
    ast = var.ast
    declarator = ast.declarator
    meta = statements.get_var_bind(var, "share").meta
    if declarator.array: # Fixed size array.
        metadim = declarator.array
    elif meta["dimension"] is not None:
        metadim = meta["dim_ast"]
    else:
        metadim = None
    if metadim:
        visitor = ToDimension(parent, None, var.fmtdict,
                              var.fmtdict.PY_struct_context)
        visitor.visit(metadim)
        fmt.rank = str(visitor.rank)
        fmt.npy_rank = fmt.rank
        fmt.npy_intp_values = ", ".join(visitor.shape)
        if visitor.rank == 1:
            fmt.npy_intp_size = visitor.shape[0]
        else:
            fmt.npy_intp_size = "*".join(
                ["(" + dim + ")" for dim in visitor.shape])
    else:
        # Scalar
        fmt.rank = "0"
        fmt.npy_rank = "1"
        fmt.npy_intp_values = "1"     # comma separated list of values
        fmt.npy_intp_size   = "1"

######################################################################

class ToDimension(todict.PrintNode):
    """Visit dimension expression.
    Convert cls references to correct scope.  obj->{argname}

    If a name is a member of the struct, prefix with PY_struct_context.

    struct Cstruct_list {
        int nitems;
        int *ivalue     +dimension(nitems+nitems);  # case 1
        double *dvalue  +dimension(nitems*TWO);     # case 2
    }

    case 1:  {context}nitems+{context}nitems
    case 2:  {context}nitems*TWO

    """

    def __init__(self, cls, fcn, fmt, context):
        """
        Args:
            cls  - ast.ClassNode or None
            fcn   - ast.FunctionNode of calling function
                    or None for struct.
            fmt  - util.Scope
            context - how to access Identifiers in cls.
                      Different for function arguments and
                      class/struct members.
        """
        super(ToDimension, self).__init__()
        self.cls = cls
        self.fcn = fcn
        self.fmt = fmt
        self.context = context

        self.rank = 0
        self.shape = []

    def visit_list(self, node):
        # list of dimension expressions
        self.rank = len(node)
        for dim in node:
            sh = self.visit(dim)
            self.shape.append(sh)

    def visit_Identifier(self, node):
        argname = node.name
        if self.fcn and argname == "size" and node.args:
            # size(in)
            argname = node.args[0].name
            #            arg = self.func.ast.find_arg_by_name(argname)
            fmt = self.fcn._bind["py"][argname].fmtdict
            if self.fcn.options.PY_array_arg == "numpy":
                return wformat("PyArray_SIZE({py_var})", fmt)
            else:
                return fmt.size_var
        # Look for members of class/struct.
        elif self.cls is not None and argname in self.cls.map_name_to_node:
            # This name is in the same class as the dimension.
            # Make name relative to the class.
            member = self.cls.map_name_to_node[argname]
            obj = self.fmt.PY_type_obj
            if member.may_have_args():
                if node.args is None:
                    print("{} must have arguments".format(argname))
                else:
                    return "{}{}({})".format(
                        self.context, argname, self.comma_list(node.args))
            else:
                if node.args is not None:
                    print("{} must not have arguments".format(argname))
                else:
                    return "{}{}".format(self.context, argname)
        elif node.args is None:
            return argname  # variable
        else:
            return self.param_list(node) # function
        return "--??--"

    def visit_AssumedRank(self, node):
        error.get_cursor().warning("Detected assumed-rank dimension")
        self.shape.append("===assumed-rank===")
        return "===assumed-rank==="

######################################################################

class ToImplied(todict.PrintNode):
    """Convert implied expression to Python wrapper code.

    expression has already been checked for errors by generate.check_implied.
    Convert functions:
      size     -  PyArray_SIZE
      len      -  strlen
      len_trim -
    """

    def __init__(self, expr, func):
        super(ToImplied, self).__init__()
        self.expr = expr
        self.func = func

    def visit_Identifier(self, node):
        """
        Args:
            node - declast.Identifier
        """
        argname = node.name
        # Look for functions
        if node.args is None:
            return argname
        ### functions
        elif argname == "size":
            # size(arg)
            argname = node.args[0].name
            #            arg = self.func.ast.find_arg_by_name(argname)
            fmt = self.func._bind["py"][argname].fmtdict
            if self.func.options.PY_array_arg == "numpy":
                if len(node.args) > 1:
                    return "PyArray_DIM({}, ({})-1)".format(
                        fmt.py_var, todict.print_node(node.args[1]))
                else:
                    return wformat("PyArray_SIZE({py_var})", fmt)
            else:
                return fmt.size_var
        elif argname == "len":
            # len(arg)
            argname = node.args[0].name
            #            arg = self.func.ast.find_arg_by_name(argname)
            fmt = self.func._bind["py"][argname].fmtdict

            # find argname in function parameters
            bind = statements.fetch_name_bind(self.func._bind, "py", argname)
            if bind.meta["intent"] == "out":
                arg = self.func.ast.declarator.find_arg_by_name(argname)
                #   char *text+intent(out)+charlen(XXX), 
                #   int ltext+implied(len(text)))
                # len(text) in this case is the value of "charlen"
                # since no variable is actually passed in as an argument.
                return arg.declarator.attrs["charlen"]
            # XXX - need code for len_trim?
            return wformat("strlen({cxx_var})", fmt)
        elif argname == "len_trim":
            # len_trim(arg)
            argname = node.args[0].name
            #            arg = self.func.ast.find_arg_by_name(argname)
            fmt = self.func._bind["py"][argname].fmtdict
            return wformat("strlen({cxx_var})", fmt)
            # XXX - need code for len_trim
            return wformat("ShroudLenTrim({cxx_var}, strlen({cxx_var}))", fmt)
            #c_helper="ShroudLenTrim"
        else:
            return self.param_list(node)


def py_implied(expr, func):
    """Convert string to Python code.

    Args:
        expr - string expression
        func - declast.FunctionNode
    """
    node = declast.ExprParser(expr).expression()
    visitor = ToImplied(expr, func)
    return visitor.visit(node)

######################################################################

def update_code_blocks(symtab, stmts, fmt):
    """ Accumulate info from statements.
    Append to lists in symtab.

    Args:
        symtab - result of locals() of caller
        stmts  - PyStmts
        fmt    - format dictionary (Scope)
    """
    for clause in ["declare", "post_declare", "post_parse", "pre_call",
                   "post_call", "cleanup", "fail"]:
        lstout = symtab[clause + "_code"]
        for cmd in getattr(stmts, clause):
            lstout.append(wformat(cmd, fmt))

def XXXdo_cast(lang, kind, typ, var):
    """Do cast based on language.

    Args:
        lang - c, c++
        kind - reinterpret, static
        typ -
        var -
    """
    if lang == "c":
        return "(%s) %s" % (typ, var)
    else:
        return "%s_cast<%s>\t(%s)" % (kind, typ, var)

def update_statements_for_language(language):
    """Preprocess statements for lookup.

    Update statements for c or c++.

    Parameters
    ----------
    language : str
        "c" or "c++"
    """
    statements.update_for_language(py_statements, language)
    statements.process_mixin(py_statements, default_stmts, py_dict)
    global default_scope
    default_scope = default_stmts["py"]


def write_stmts_tree(fp):
    """Write out statements tree.

    Parameters
    ----------
    fp : file
    """
    statements.print_tree_statements(fp, py_dict, default_stmts)
    tree = statements.update_stmt_tree(py_dict)
    lines = []
    statements.print_tree_index(tree, lines)
    fp.writelines(lines)


def lookup_stmts(path):
    name = statements.compute_name(path)
    stmt = py_dict.get(name, None)
    if stmt is None:
        stmt = py_dict["py_mixin_unknown"]
        error.cursor.warning("Unknown statement: {}".format(name))
    return stmt

PyStmts = util.Scope(
    None,
    name="py_default",
    intent=None,
    arg_declare=None,   # Empty list indicates no declaration.
    post_declare=[],
    fmtdict=None,
            
    arg_call=None,
    c_header=[], c_helper=[],
    cxx_header=[],
    need_numpy=False,
    object_created=False,
    incref_on_return=False,
    parse_format=None, parse_args=[],
    declare=[], post_parse=[], pre_call=[],
    call=[],
    post_call=[],
    c_return=[],
    destructor_name=None,
    destructor=[],
    cleanup=[], fail=[],
    fail_declare=[], fail_call=[],
    goto_fail=False,
    getter=[], getter_helper=[],
    setter=[], setter_helper=[],
    local=None,
)

default_stmts = dict(
    py=PyStmts,
)

# language   "py"
# intent     "in", "out", "inout", "function", "subroutine", "descr", "ctor", "dtor"
# sgroup     "native", "string", "char"
# spointer   "scalar" "*" "**", "&"
# generated  "buf"
# deref      "numpy", "list", "scalar", "raw"
#      PY_array_arg
#      typemap.PY_struct_as   class, list
# owner      "caller"


# Code clauses are used for C and C++.
# Differences are dealt with by format entries stdlib and cast.
# Language specific clauses are used in update_for_language.
# Function calls which return 'void *', do not require casts in C.
# It doesn't hurt to add them, but I dislike the clutter.
py_statements = [

########################################
#
    dict(
        name="py_mixin_unknown",
        comments=[
            "Default returned by lookup_fc_stmts when group is not found.",
        ],
    ),
    dict(
        name="py_mixin_cxx-as-scalar",
        fmtdict=dict(
            cxx_addr="&",
            cxx_member=".",
            cxx_deref="",
        ),
    ),
    dict(
        name="py_mixin_cxx-as-pointer",
        fmtdict=dict(
            cxx_addr="",
            cxx_member="->",
            cxx_deref="*",
        ),
    ),
    dict(
        name="py_mixin_function-void",
        comments=[
            "Call void function."
        ],
        call=[
            "{C_call_function};",
        ],
    ),
    dict(
        name="py_mixin_function-declare",
        comments=[
            "Declare variable and assign function result."
        ],
        notes=[
            "The default call will declare the variable as part of the call.",
            "If goto_fail is True or default parameters are defined,then split"
            "the statement to avoid declaring the variable after the goto."
        ],
        call=[
            "{gen.cxxdecl.cxx_var} =\t {C_call_function};",
        ],
        fail_declare=[
            "{gen.cxxdecl.cxx_var};",
        ],
        fail_call=[
            "{cxx_var} =\t {C_call_function};",
        ],
    ),
    dict(
        name="py_mixin_function-assign-pointee",
        comments=[
            "Call function and assign to pointee."
        ],
        call=[
            "*{cxx_var} = {C_call_function};",
        ],
    ),

    dict(name="py_mixin_array-parse",
         comments=[
             "Parse a Python Object to be used as PyArrayObject.",
         ],
         need_numpy=True,
         declare=[
             "PyObject * {pytmp_var};",    # Object set by ParseTupleAndKeywords.
             "PyArrayObject * {py_var} = {nullptr};",
         ],
         parse_format="O",
         parse_args=["&{pytmp_var}"],
         fail=[
             "Py_XDECREF({py_var});",
         ],
    ),

    dict(
        name="py_mixin_string-fmtdict",
        notes=[
            "ctor_expr is arguments to PyString_FromStringAndSize."
        ],
        fmtdict=dict(
            ctor_expr="{cxx_var}{cxx_member}data(),\t {cxx_var}{cxx_member}size()",
        ),
    ),
    dict(
        # WIP - replace cxx_local_var="scalar" cxx_member = .
        # cxx_member used type typemap.PY_build_arg
        name="py_mixin_string-fmtdict-scalar",
        notes=[
            "ctor_expr is arguments to PyString_FromStringAndSize."
        ],
        fmtdict=dict(
            cxx_member=".",
            ctor_expr="{cxx_var}.data(),\t {cxx_var}.size()",
        ),
    ),

    dict(
        name="py_mixin_array-ContiguousFromObject",
        notes=[
            "intent(IN)",
        ],
        post_parse=[
            "{py_var} = {cast_reinterpret}PyArrayObject *{cast1}PyArray_ContiguousFromObject("
            "\t{pytmp_var},\t {numpy_type},\t {rank},\t {rank}){cast2};",
            # NPY_ARRAY_DEFAULT | NPY_ARRAY_ENSUREARRAY
            # NPY_ARRAY_CARRAY | NPY_ARRAY_ENSUREARRAY
            # NPY_ARRAY_C_CONTIGUOUS | NPY_ARRAY_BEHAVED | NPY_ARRAY_ENSUREARRAY
            # NPY_ARRAY_C_CONTIGUOUS | NPY_ARRAY_ALIGNED | NPY_ARRAY_WRITEABLE | NPY_ARRAY_ENSUREARRAY
        ],
    ),
    dict(
        name="py_mixin_array-FROM-OTF",
        notes=[
            "intent(INOUT)",
        ],
        post_parse=[
            "{py_var} = {cast_reinterpret}PyArrayObject *{cast1}PyArray_FROM_OTF("
            "\t{pytmp_var},\t {numpy_type},\t NPY_ARRAY_INOUT_ARRAY){cast2};",
        ],
    ),
    dict(
        name="py_mixin_array-FROM-OFT-in",
        notes=[
            "intent(INOUT)",
        ],
        post_parse=[
            "{py_var} = {cast_reinterpret}PyArrayObject *{cast1}PyArray_FROM_OTF("
            "\t{pytmp_var},\t {numpy_type},\t NPY_ARRAY_IN_ARRAY){cast2};",
        ],
    ),
    dict(
        name="py_mixin_array-SimpleNew",
        notes=[
            "intent(OUT)",
        ],
        post_parse=[
            "{npy_intp_asgn}"  # Must contain a newline if non-blank.
            "{py_var} = {cast_reinterpret}PyArrayObject *{cast1}PyArray_SimpleNew("
            "{npy_rank}, {npy_dims_var}, {numpy_type}){cast2};",
        ],
    ),

    dict(
        name="py_mixin_array-SimpleNewFromData",
        notes=[
            "function result, intent(OUT)",
        ],
        need_numpy=True,
        declare=[
            "{npy_intp_decl}"
            "PyObject * {py_var} = {nullptr};",
        ],
        post_call=[
#            "{npy_intp_asgn}"
            "{npy_dims_var}[0] = {cxx_var}->size();",
            "{py_var} = "
            "PyArray_SimpleNewFromData({npy_rank},\t {npy_dims_var},"
            "\t {numpy_type},\t {cxx_var}->data());",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_mixin_array-SimpleNewFromData2",
        notes=[
            "function result, intent(OUT)",
        ],
        need_numpy=True,
        declare=[
            "{npy_intp_decl}"
            "PyObject * {py_var} = {nullptr};",
        ],
        post_call=[
            "{npy_intp_asgn}"
            "{py_var} = "
            "PyArray_SimpleNewFromData({npy_rank},\t {npy_dims_var},"
            "\t {numpy_type},\t {cxx_nonconst_ptr});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    
    dict(
        name="py_mixin_array-FromAny",
        notes=[
            "PyArray_FromAny steals a reference from PYN_descr",
            "and will decref it if an error occurs.",
        ],
        post_parse=[
            "Py_INCREF({PYN_descr});",
            "{py_var} = {cast_reinterpret}PyArrayObject *{cast1}"
            "PyArray_FromAny(\t{pytmp_var},\t {PYN_descr},"
            "\t 0,\t 1,\t NPY_ARRAY_IN_ARRAY,\t {nullptr}){cast2};",
        ],
    ),
    dict(
        name="py_mixin_array-NewFromDescr",
        post_parse=[
#            "{npy_intp_asgn}"
            "Py_INCREF({PYN_descr});",
            "{py_var} = {cast_reinterpret}PyArrayObject *{cast1}"
            "PyArray_NewFromDescr(\t&PyArray_Type,\t {PYN_descr},"
            "\t 0,\t {nullptr},\t {nullptr},\t {nullptr},\t 0,\t {nullptr}){cast2};",
        ],
    ),
    dict(
        name="py_mixin_array-NewFromDescr2",
        need_numpy=True,
        declare=[
            "{npy_intp_decl}"
            "PyObject * {py_var} = {nullptr};",
        ],
        post_call=[
            "{npy_intp_asgn}"
            "Py_INCREF({PYN_descr});",
            "{py_var} = "
            "PyArray_NewFromDescr(&PyArray_Type, \t{PYN_descr},\t"
            " {npy_rank}, {npy_dims_var}, \t{nullptr}, {cxx_var}, 0, {nullptr});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_mixin_array_error",
        post_parse=[
            "if ({py_var} == {nullptr}) {{+",
            "PyErr_SetString(PyExc_ValueError,"
            '\t "{c_var} must be a {rank}-D array of {c_type}");',
            "goto fail;",
            "-}}",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_mixin_template_array_error",
        notes=[
            "Use cxx_T instead of c_type for vector.",
        ],
        post_parse=[
            "if ({py_var} == {nullptr}) {{+",
            "PyErr_SetString(PyExc_ValueError,"
            '\t "{c_var} must be a 1-D array of {cxx_T}");',
            "goto fail;",
            "-}}",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_mixin_array-get-data",
        lang_c=dict(
            pre_call=[
                "{c_var} = PyArray_DATA({py_var});",
            ],
        ),
        lang_cxx=dict(
            pre_call=[
                "{cxx_var} = static_cast<{cxx_type} *>\t(PyArray_DATA({py_var}));",
            ],
        ),
    ),
    dict(
        name="py_mixin_array-capsule",
        notes=[
            "Create a PyCapsule and associate with a PyArray.",
            "This capsule contains a pointer to data and a destructor",
            "which will be used when the PyArray's reference count drops to zeo."
        ],
        local=[
            "capsule",
        ],
        declare=[
            "PyObject *{py_capsule} = {nullptr};",
        ],
        post_call=[
            "{py_capsule} = "
            'PyCapsule_New({cxx_var}, "{PY_numpy_array_capsule_name}", '
            "\t{PY_capsule_destructor_function});",
            "if ({py_capsule} == {nullptr}) goto fail;",
            "PyCapsule_SetContext({py_capsule},"
            "\t {PY_fetch_context_function}({capsule_order}));",
            "if (PyArray_SetBaseObject(\t"
            "{cast_reinterpret}PyArrayObject *{cast1}{py_var}{cast2},"
            "\t {py_capsule}) < 0)\t goto fail;",
        ],
        fail=[
            "Py_XDECREF({py_capsule});",
        ],
        goto_fail=True,
    ),

    dict(
        name="py_mixin_alloc-cxx-type",
        notes=[
            "Allocate memory",
            "Used with C++ objects, structs",
            "intent(out)",
            "Use with py_mixin_alloc_error2",
            "Initialize the variable to allow it be released in fail",
        ],
        local=[
            "cxx",
        ],
        arg_declare=[
            "{cxx_type} *{py_local_cxx} = {nullptr};"
        ],
        fail=[
            "if ({cxx_var} != {nullptr}) {{+",
            "{PY_release_memory_function}({capsule_order}, {cxx_var});",
            "-}}",
        ],
        destructor_name="{cxx_type} *",
        lang_c=dict(
            pre_call=[
                "{py_local_cxx} = malloc(sizeof({cxx_type}));",
            ],
            destructor=[
                "free(ptr);",
            ],
        ),
        lang_cxx=dict(
            pre_call=[
                "{py_local_cxx} = new {cxx_type};",
            ],
            destructor=[
                "{cxx_type} * cxx_ptr =\t static_cast<{cxx_type} *>(ptr);",
                "delete cxx_ptr;",
            ],
        ),
    ),
    dict(
        name="py_mixin_malloc_error2",
        # Uses py_local_cxx instead of cxx_var
        pre_call=[
            "if ({py_local_cxx} == {nullptr}) {{+",
            "PyErr_NoMemory();",
            "goto fail;",
            "-}}",
        ],
        goto_fail=True,
    ),

    dict(
        name="py_mixin_malloc_error",
        pre_call=[
            "if ({cxx_var} == {nullptr}) {{+",
            "PyErr_NoMemory();",
            "goto fail;",
            "-}}",
        ],
    ),
    dict(
        name="py_mixin_malloc",
        lang_c=dict(
            pre_call=[
                "{c_var} = malloc(\tsizeof({c_type}) * ({array_size}));",
                "if ({c_var} == {nullptr}) {{+",
                "PyErr_NoMemory();",
                "goto fail;",
                "-}}",
            ]
        ),
        lang_cxx=dict(
            pre_call=[
                "{cxx_var} = static_cast<{cxx_type} *>\t(std::malloc(\tsizeof({cxx_type}) * ({array_size})));",
                "if ({cxx_var} == {nullptr}) {{+",
                "PyErr_NoMemory();",
                "goto fail;",
                "-}}",
            ]
        ),
        goto_fail=True,
    ),

########################################
    dict(
        alias=[
            "py_in_native",
            "py_in_unknown",
        ],
    ),

########################################
# subroutine

    dict(
        name="py_subroutine",
        mixin=[
            "py_mixin_function-void",
        ],
    ),

########################################
# void
    dict(
        name="py_in_void*",
        comments=[
            "Accept a capsule and extract address."
        ],
        declare=[
            "PyObject *{py_var};",
        ],
        parse_format="O",
        parse_args=["&{py_var}"],
        post_parse=[
            "{c_var} = PyCapsule_GetPointer({py_var}, NULL);",
            "if (PyErr_Occurred())",
            "+goto fail;-",
        ],
        arg_call=[
            "{c_var}",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_out_void**",
        arg_declare=[
            "void *{c_var};",
        ],
        arg_call=[
            "&{c_var}",
        ],
        fmtdict=dict(
            ctor_expr="{cxx_var}",
        ),
    ),
    dict(
        name="py_out_void*&",
        base="py_out_void**",
        arg_call=[
            "{c_var}",
        ]
    ),
    dict(
        name="py_function_void*",
        mixin=[
            "py_mixin_function-declare",
        ],
        fmtdict=dict(
            ctor_expr="{cxx_var}",
        ),
    ),

########################################
# bool
    dict(
        name="py_in_bool",
        pre_call=["{cxx_var} = PyObject_IsTrue({py_var});"]
    ),
    dict(
        name="py_inout_bool",
        arg_declare=[
            "bool {cxx_var};",
        ],
        pre_call=["{cxx_var} = PyObject_IsTrue({py_var});"],
        # py_var is already declared for inout
        post_call=[
            "{py_var} = PyBool_FromLong({c_var});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_out_bool",
        arg_declare=[
            "bool {cxx_var};",
        ],
        declare=[
            "{PyObject} * {py_var} = {nullptr};",
        ],
        post_call=[
            "{py_var} = PyBool_FromLong({c_var});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_function_bool",
        mixin=[
            "py_mixin_function-declare",
        ],
        declare=[
            "{PyObject} * {py_var} = {nullptr};",
        ],
        post_call=[
            "{py_var} = PyBool_FromLong({c_var});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_out_bool*",
        base="py_out_bool",
        arg_call=["&{cxx_var}"],
    ),
    dict(
        name="py_inout_bool*",
        base="py_inout_bool",
        arg_call=["&{cxx_var}"],
    ),
    
####################
    dict(
        alias=[
            "py_function_native",
        ],
        notes=[
            "Call a function which returns a native scalar.",
        ],
        mixin=[
            "py_mixin_function-declare",
        ],
    ),
    dict(
        name="py_in_native*",
        arg_declare=["{c_type} {c_var};"],
        arg_call=["&{c_var}"],
    ),
    dict(
        name="py_inout_native*",
        arg_declare=["{c_type} {c_var};"],
        arg_call=["&{c_var}"],
        fmtdict=dict(
            ctor_expr="{c_var}",
        ),
    ),
    dict(
        name="py_out_native*",
        arg_declare=["{c_type} {c_var};"],
        arg_call=["&{c_var}"],
        fmtdict=dict(
            ctor_expr="{c_var}",
        ),
    ),
    dict(
        name="py_in_native&",
        arg_declare=["{c_type} {c_var};"],
    ),
    dict(
        name="py_inout_native&",
        arg_declare=["{c_type} {c_var};"],
    ),
    dict(
        name="py_out_native&",
        arg_declare=["{c_const}{c_type} {c_var};"],
    ),
    dict(
        alias=[
            "py_implied_native",
            "py_implied_bool",
        ],
        notes=[
            "Adding the argument's code to the call entry will cause the code",
            "to be inserted after non-implied arguments pre_call has been",
            "done. This allows c_implied to use other argument's values."
        ],
        arg_declare=[
            "{cxx_type} {c_var};",
        ],
        call=[
            "{c_var} = {c_implied};",
        ],
        arg_call=["{c_var}"],
    ),
    
####################
## numpy
    dict(
        name="py_in_native*_numpy",
        mixin=[
            "py_mixin_array-parse",
            "py_mixin_array-ContiguousFromObject",
            "py_mixin_array_error",
            "py_mixin_array-get-data",
        ],
        arg_call=["{c_var}"],
        cleanup=[
            "{PY_cleanup_decref}({py_var});",
        ],
    ),

    dict(
        name="py_inout_native*_numpy",
        mixin=[
            "py_mixin_array-parse",
            "py_mixin_array-FROM-OTF",
            "py_mixin_array_error",
            "py_mixin_array-get-data",
        ],
        arg_call=["{c_var}"],
        object_created=True,
    ),

    dict(
        name="py_out_native*_numpy",
        mixin=[
            "py_mixin_array-SimpleNew",
            "py_mixin_array_error",
            "py_mixin_array-get-data",
        ],
        need_numpy=True,
        declare=[
            "{npy_intp_decl}"  # Must contain a newline if non-blank.
            "PyArrayObject * {py_var} = {nullptr};",
        ],
        arg_call=["{c_var}"],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
    ),

    dict(
        name="py_function_native*_scalar",
        mixin=[
            "py_mixin_function-declare",
        ],
    ),
    dict(
        name="py_function_native*_list",
        mixin=[
            "py_mixin_function-declare",
        ],
        c_helper=["to_PyList_{cxx_type}:to_PyList"],
        declare=[
            "PyObject *{py_var} = {nullptr};",
        ],
        post_call=[
            "{py_var} = {c_helper_to_PyList}\t({cxx_var},\t {array_size});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        # XXX - library owns memory, test on +owner attribute
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        alias=[
            "py_function_native*_numpy",
            "py_function_native&_numpy",
        ],
        # XXX - need capsule if owner=caller
        mixin=[
            "py_mixin_function-declare",
            "py_mixin_array-SimpleNewFromData2",
        ],
    ),

    dict(
        name="py_out_native**_numpy",
        base="py_function_native*_numpy",
        # Declare a local variable for the argument.
        arg_declare=[
            "{c_const}{c_type} *{c_var};",
        ],
        declare=[
            "{npy_intp_decl}"
            "PyObject *{py_var} = {nullptr};"
        ],
        arg_call=["&{cxx_var}"],
    ),
    dict(
        name="py_out_native*&_numpy",
        base="py_out_native**_numpy",
        arg_call=["{cxx_var}"],
    ),

########################################
## list
    dict(
        name="py_in_native*_list",
        c_helper=["get_from_object_{cxx_type}_list:get_from_object"],
        parse_format="O",
        parse_args=["&{pytmp_var}"],
        arg_declare=[ # initialize
            "{cxx_type} * {cxx_var} = {nullptr};",
        ],
        declare=[
            "PyObject *{pytmp_var} = {nullptr};",
            "{PY_typedef_converter} {value_var} = {PY_value_init};",
            "{value_var}.name = \"{c_var}\";",
            "Py_ssize_t {size_var};",
        ],
        post_parse=[
            "if ({c_helper_get_from_object}\t({pytmp_var}, &{value_var}) == 0)",
            "+goto fail;-",
            "{cxx_var} = {cast_static}{cxx_type} *{cast1}{value_var}.data{cast2};",
            "{size_var} = {value_var}.size;",
        ],
        arg_call=["{cxx_var}"],
        cleanup=[
            "Py_XDECREF({value_var}.dataobj);",
        ],
        fail=[
            "Py_XDECREF({value_var}.dataobj);",
        ],
        goto_fail=True,
    ),

    dict(
        name="py_inout_native*_list",
#        c_helper=["update_PyList_{cxx_type}"],
        c_helper=[
            "get_from_object_{cxx_type}_list:get_from_object",
            "to_PyList_{cxx_type}:to_PyList"
        ],
        parse_format="O",
        parse_args=["&{pytmp_var}"],
        arg_declare=[
            "{cxx_type} * {cxx_var} = {nullptr};",
        ],
        declare=[
            "PyObject *{py_var};",
            "PyObject *{pytmp_var} = {nullptr};",
            "{PY_typedef_converter} {value_var} = {PY_value_init};",
            "{value_var}.name = \"{c_var}\";",
            "Py_ssize_t {size_var};",
        ],
        post_parse=[
            "if ({c_helper_get_from_object}\t({pytmp_var}, &{value_var}) == 0)",
            "+goto fail;-",
            "{cxx_var} = {cast_static}{cxx_type} *{cast1}{value_var}.data{cast2};",
            "{size_var} = {value_var}.size;",
        ],
        arg_call=["{cxx_var}"],
        post_call=[
#            "SHROUD_update_PyList_{cxx_type}({pytmp_var}, {cxx_var}, {size_var});",
            "{py_var} = {c_helper_to_PyList}\t({cxx_var},\t {size_var});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        cleanup=[
            "Py_XDECREF({value_var}.dataobj);",
        ],
        fail=[
            "Py_XDECREF({value_var}.dataobj);",
        ],
        goto_fail=True,
    ),

    dict(
        name="py_out_native*_list",
        mixin=[
            "py_mixin_malloc",
        ],
        c_helper=["to_PyList_{cxx_type}:to_PyList"],
        c_header=["<stdlib.h>"],  # malloc/free
        cxx_header=["<cstdlib>"],  # malloc/free
        arg_declare=[
            "{cxx_type} * {cxx_var} = {nullptr};",
        ],
        declare=[
            "PyObject *{py_var} = {nullptr};",
        ],
        arg_call=["{c_var}"],
        post_call=[
            "{py_var} = {c_helper_to_PyList}\t({cxx_var},\t {array_size});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        cleanup=[
            "{stdlib}free({cxx_var});",
            "{cxx_var} = {nullptr};",
        ],
        fail=[
            "Py_XDECREF({py_var});",
            "if ({cxx_var} != {nullptr})\t {stdlib}free({cxx_var});",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_out_native**_list",
        base="py_function_native*_list",
        # Declare a local variable for the argument.
        arg_declare=[
            "{c_const}{c_type} *{c_var};",
        ],
        declare=[
            "PyObject *{py_var} = {nullptr};"
        ],
        arg_call=["&{cxx_var}"],
    ),
    
########################################
## raw
    dict(
        # Declare a local pointer, pass address to library, convert to capsule.
        name="py_out_native**_raw",
        arg_declare=[
            "{c_type} *{c_var};",
        ],
        declare=[
            "PyObject *{py_var} = {nullptr};"
        ],
        arg_call=["&{cxx_var}"],
        post_call=[
            "{py_var} = PyCapsule_New({cxx_var}, NULL, NULL);",
        ],
        object_created=True,
    ),
########################################
# char
    dict(
        # Get a string from argument but only pass first character to C++.
        # Since char is a scalar, need to actually get a char * - parse_format, parse_args.
        # XXX - make sure c_var is only 1 long?
        name="py_in_char",
        arg_declare=[
            "char *{c_var};",
        ],
        parse_format="s",
        parse_args=["&{c_var}"],
        arg_call=["{c_var}[0]"],
    ),
    dict(
        name="py_function_char",
        mixin=[
            "py_mixin_function-declare",
        ],
        declare=[
            "{PyObject} * {py_var} = {nullptr};",
        ],
        post_call=[
            "{py_var} = PyString_FromStringAndSize(&{cxx_var}, 1);",
        ],
        object_created=True,
    ),
    dict(
        name="py_function_char*",
        mixin=[
            "py_mixin_function-declare",
        ],
        fmtdict=dict(
            ctor_expr="{c_var}",
        ),
    ),
    dict(
        name="py_in_char*",
        arg_call=["{c_var}"],
    ),
    dict(
        name="py_out_char*_charlen",
        arg_declare=[
            "{c_const}char {c_var}[{charlen}];  // intent(out)",
        ],
        arg_call=["{c_var}"],
        fmtdict=dict(
            ctor_expr="{c_var}",
        ),
    ),
    dict(
        name="py_inout_char*",
        arg_call=["{c_var}"],
        fmtdict=dict(
            ctor_expr="{c_var}",
        ),
    ),

    dict(
        name="py_in_char**",
        c_helper=["get_from_object_charptr"],
        parse_format="O",
        parse_args=["&{pytmp_var}"],
        arg_declare=[
            "{c_const}char ** {cxx_var} = {nullptr};",
        ],
        declare=[
            "PyObject * {pytmp_var};", # set by PyArg_Parse
            "{PY_typedef_converter} {value_var} = {PY_value_init};",
            "{value_var}.name = \"{c_var}\";",
            "Py_ssize_t {size_var};",
        ],
        pre_call=[
            "if ({c_helper_get_from_object_charptr}\t({pytmp_var}, &{value_var}) == 0)",
            "+goto fail;-",
            "{cxx_var} = {cast_static}char **{cast1}{value_var}.data{cast2};",
        ],
        arg_call=["{cxx_var}"],
        post_call=[
            "Py_XDECREF({value_var}.dataobj);",
        ],
        fail=[
            "Py_XDECREF({value_var}.dataobj);",
        ],
        goto_fail=True,
    ),
    
########################################
# string
# ctor_expr is arguments to PyString_FromStringAndSize.
    dict(
        alias=[
            "py_in_string",
            "py_in_string&",
        ],
        mixin=[
            "py_mixin_string-fmtdict-scalar",
        ],
        local=[
            "cxx",
        ],
        arg_declare=[
            # std::string defaults to making this scalar. Make sure it is pointer.
            "char *{c_var};",
        ],
        post_declare=[
            "{c_const}std::string {cxx_var}({c_var});"
        ],
        arg_call=[
            "{cxx_var}",
        ],
    ),
    dict(
        name="py_in_string*",
        base="py_in_string",
        arg_call=["&{cxx_var}"],
    ),
    dict(
        alias=[
            "py_inout_string",
            "py_inout_string&",
        ],
        mixin=[
            "py_mixin_string-fmtdict-scalar",
        ],
        local=[
            "cxx",
        ],
        arg_declare=[
            "char *{c_var};",
        ],
        post_declare=[
            "{c_const}std::string {cxx_var}({c_var});"
        ],
        arg_call=[
            "{cxx_var}",
        ],
    ),
    dict(
        alias=[
            "py_out_string",
            "py_out_string&",
        ],
        mixin=[
            "py_mixin_string-fmtdict-scalar",
        ],
        local=[
            "cxx",
        ],
        arg_declare=[],
        post_declare=[
            "{c_const}std::string {cxx_var};"
        ],
        arg_call=[
            "{cxx_var}",
        ],
    ),
    dict(
        name="py_function_string",
        mixin=[
            "py_mixin_function-declare",
            "py_mixin_string-fmtdict",
        ],
    ),
    dict(
        alias=[
            "py_function_string*",
            "py_function_string&",
        ],
        mixin=[
            "py_mixin_function-declare",
            "py_mixin_string-fmtdict",
        ],
    ),
    dict(
        name="py_inout_string*",
        base="py_inout_string",
        arg_call=["&{cxx_var}"],
    ),
    dict(
        name="py_out_string*",
        base="py_out_string",
        arg_call=["&{cxx_var}"],
    ),

########################################
# enum

    dict(
        name="py_function_enum",
        mixin=[
            "py_mixin_function-declare",
        ],
    ),
    dict(
        name="py_in_enum",
        local=[
            "cxx",
        ],
        post_declare=[
            "{cxx_type} {cxx_var} =\t {cast_static}{cxx_type}{cast1}{c_var}{cast2};",
        ],
        arg_call=[
            "{cxx_var}",
        ],
    ),

    dict(
        name="py_out_enum*",
#        cxx_local_var="scalar",  # XXX - enum.yaml returnEnumOutArg
        local=[
            "cxx"
        ],
        fmtdict=dict(
            cxx_member=".",
        ),
        declare=[
            "{cxx_type} {py_var};",
        ],
        arg_call=["&{cxx_var}"],
        # XXX - This indirection is not correct
        post_call=[
            "*{c_var} = ({cxx_type}) {py_var};",
        ],
    ),
    
########################################
# struct
# "struct", intent, PY_struct_arg
# numpy
# Note that Typemap.c_type is a C wrapper over a C++ struct
# created in wrapc.py. Do not use here.
    
    dict(
        name="py_ctor_struct",
        notes=[
            "Called via tp_init.",
        ],
        lang_c=dict(
            call=[
                "self->{PY_type_obj} = malloc(sizeof({cxx_type}));",
                "if (self->{PY_type_obj} == {nullptr}) {{+",
                "PyErr_NoMemory();",
                "return -1;",
                "-}}",
                "self->{PY_type_dtor} = {capsule_order};",
            ],
            destructor=[
                "free(ptr);",
            ],
        ),
        lang_cxx=dict(
            call=[
                "self->{PY_type_obj} = new {cxx_type};",
                "if (self->{PY_type_obj} == {nullptr}) {{+",
                "PyErr_NoMemory();",
                "return -1;",
                "-}}",
                "self->{PY_type_dtor} = {capsule_order};",
            ],
            destructor=[
                "{cxx_type} * cxx_ptr =\t static_cast<{cxx_type} *>(ptr);",
                "delete cxx_ptr;",
            ],
        ),
        destructor_name="{cxx_type} *",
        c_return=[
            "return 0;",
        ],
    ),
    dict(
        alias=[
            "py_function_struct_list",
            "py_function_struct*_list",
        ],
        mixin=[
            "py_mixin_function-declare",
        ],
    ),
    
# and does not apply in Python.
    dict(
        name="py_in_struct_list",
        arg_declare=[],
    ),
    dict(
        name="py_inout_struct_list",
        arg_declare=[],
    ),
    dict(
        name="py_out_struct_list",
        arg_declare=[],
        post_declare=[
            "{cxx_type} {cxx_var};",
        ],
    ),

    # struct-list-cxx   (XXX - is not compiled)
    dict(
        name="py_in_struct*_list",
        base="py_in_struct_list",
        arg_call=["&{cxx_var}"],
    ),
    dict(
        name="py_inout_struct*_list",
        base="py_inout_struct_list",
        arg_call=["&{cxx_var}"],
    ),
    dict(
        name="py_out_struct*_list",
        base="py_out_struct_list",
        arg_call=["&{cxx_var}"],
    ),

    dict(
        name="py_in_struct*_numpy",
        mixin=[
            "py_mixin_array-parse",
            "py_mixin_array-FromAny",
            "py_mixin_array_error",
            "py_mixin_array-get-data",
        ],
        fmtdict=dict(
            cxx_member="->",
        ),
        arg_declare=[ # Must be a pointer of cxx_type.
            "{cxx_type} *{cxx_var};",
        ],
        cleanup=[
            "{PY_cleanup_decref}({py_var});",
        ],
    ),
    dict(
        name="py_inout_struct*_numpy",
        mixin=[
            "py_mixin_array-parse",
            "py_mixin_array-FromAny",
            "py_mixin_array_error",
            "py_mixin_array-get-data",
        ],
        arg_declare=[ # Must be a pointer.
            "{cxx_type} *{cxx_var};",
        ],
        object_created=True,
    ),
    dict(
        name="py_out_struct*_numpy",
        mixin=[
            "py_mixin_array-NewFromDescr",
            "py_mixin_array_error",
            "py_mixin_array-get-data",
        ],
        # XXX - expand to array of struct
        need_numpy=True,
#        allocate_local_var=True,  # needed to release memory
        arg_declare=[
            "{cxx_type} *{cxx_var};", 
        ],
        declare=[
#            "{npy_intp_decl}"
            "PyArrayObject * {py_var} = {nullptr};",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
    ),
    dict(
        alias=[
            "py_function_struct_numpy",
        ],
        mixin=[
            "py_mixin_alloc-cxx-type",
            "py_mixin_malloc_error2",
            "py_mixin_array-NewFromDescr2",
            "py_mixin_array-capsule",
            "py_mixin_function-assign-pointee",
        ],
        # XXX - expand to array of struct
    ),
    dict(
        alias=[
            "py_function_struct*_numpy",
        ],
        mixin=[
            "py_mixin_array-NewFromDescr2",
            "py_mixin_function-declare",
        ],
    ),

    dict(
        name="py_in_struct&_numpy",
        base="py_in_struct*_numpy",
        arg_call=["*{cxx_var}"],
    ),
    dict(
        name="py_inout_struct&_numpy",
        base="py_inout_struct*_numpy",
        arg_call=["*{cxx_var}"],
    ),
    dict(
        name="py_out_struct&_numpy",
        base="py_out_struct*_numpy",
        arg_call=["*{cxx_var}"],
    ),
    dict(
        name="py_in_struct_numpy",
        base="py_in_struct*_numpy",
        arg_call=["*{cxx_var}"],
    ),
# cannot support inout/out with call-by-value
#        name="py_inout_struct*_numpy",
#        name="py_out_struct*_numpy",

##########
# Since cxx_var is always a pointer, use that case as the base for
# pass by value.
    dict(
        name="py_in_struct*_class",
        arg_declare=[], # No C variable, the pointer is extracted from PyObject.
        fmtdict=dict(
            cxx_member="->",
        ),
        post_declare=[
            "{c_const}{cxx_type} *{cxx_var} ="
            "\t {py_var} ? {py_var}->{PY_type_obj} : {nullptr};",
        ],
    ),
    dict(
        name="py_inout_struct*_class",
        arg_declare=[], # No C variable, the pointer is extracted from PyObject.
        fmtdict=dict(
            cxx_member="->",
        ),
        post_declare=[
            "{c_const}{cxx_type} *{cxx_var} ="
            "\t {py_var} ? {py_var}->{PY_type_obj} : {nullptr};",
        ],
        object_created=True,
        incref_on_return=True,
    ),
    dict(
        alias=[
            "py_out_struct*_class",
            "py_function_shadow",
        ],
#        allocate_local_var=True,  # needed to release memory
        fmtdict=dict(
            cxx_member="->",
        ),
        arg_declare=[
            "{cxx_type} *{cxx_var} = {nullptr};", 
        ],
        declare=[
            "PyObject *{py_var} = {nullptr};",
        ],
        lang_c=dict(
            pre_call=[
                "{c_var} = malloc(sizeof({c_type}));",
            ],
# XXX - should this be used?
#            dealloc_capsule=[
#                "free(ptr);",
#            ],
        ),
        lang_cxx=dict(
            pre_call=[
                "{cxx_var} = new {cxx_type};",
            ],
#            dealloc_capsule=[
#                "delete cxx_ptr;",
#            ],
        ),
        post_call=[
            "{py_var} = {PY_to_object_idtor_func}({cxx_addr}{cxx_var},\t {capsule_order});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_mixin_function-struct-class",
        declare=[
            "PyObject *{py_var} = {nullptr};  // struct_class",
        ],
        post_call=[
            "{py_var} = {PY_to_object_idtor_func}({cxx_addr}{cxx_var},\t {capsule_order});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        alias=[
            "py_function_struct_class",
        ],
        mixin=[
            "py_mixin_alloc-cxx-type",
            "py_mixin_malloc_error2",
            "py_mixin_function-struct-class",
            "py_mixin_function-assign-pointee",
        ],
        fmtdict=dict(
            # Used with py_mixin_function-struct-class
            cxx_addr="",
        ),
    ),
    dict(
        alias=[
            "py_function_struct*_class",
        ],
        mixin=[
            "py_mixin_function-struct-class",
            "py_mixin_function-declare",
        ],
    ),

    dict(
        name="py_in_struct_class",
        base="py_in_struct*_class",
        arg_call=["*{cxx_var}"],
    ),
# cannot support inout/out with call-by-value
#        name="py_inout_struct_class",
#        name="py_out_struct_class",
    dict(
        name="py_in_struct&_class",
        base="py_in_struct*_class",
        arg_call=["*{cxx_var}"],
    ),
    dict(
        name="py_inout_struct&_class",
        base="py_inout_struct*_class",
        arg_call=["*{cxx_var}"],
    ),
    dict(
        # XXX - this memory will leak
        name="py_out_struct&_class",
        base="py_out_struct*_class",
        arg_call=["*{cxx_var}"],
        post_call=[
            "{py_var} = {PY_to_object_idtor_func}({cxx_var},\t {capsule_order});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
    ),

########################################
# shadow a.k.a class
    dict(
        name="py_ctor_shadow",
        notes=[
            "Called via tp_init.",
        ],
        call=[
            "self->{PY_type_obj} = new {cxx_type}({PY_call_list});",
            "if (self->{PY_type_obj} == {nullptr}) {{+",
            "PyErr_NoMemory();",
            "return -1;",
            "-}}",
            "self->{PY_type_dtor} = {capsule_order};",
        ],
        destructor_name="{cxx_type} *",
        destructor=[
            "{cxx_type} * cxx_ptr =\t static_cast<{cxx_type} *>(ptr);",
            "delete cxx_ptr;",
        ],
        c_return=[
            "return 0;",
        ],
    ),
    dict(
        name="py_in_shadow*",
        arg_declare=[], # No C variable, the pointer is extracted from PyObject.
        fmtdict=dict(
            cxx_member="->",
        ),
        post_declare=[
            "{c_const}{cxx_type} *{cxx_var} ="
            "\t {py_var} ? {py_var}->{PY_type_obj} : {nullptr};"
        ],
    ),
    dict(
        name="py_inout_shadow_*",
        # XXX - is this tested?
        arg_declare=[], # No C variable, the pointer is extracted from PyObject.
        post_declare=[
            "{c_const}{cxx_type} *{cxx_var} ="
            "\t {py_var} ? {py_var}->{PY_type_obj} : {nullptr};"
        ],
    ),
    dict(
        name="py_out_shadow_*",
        declare=[
            "{PyObject} *{py_var} = {nullptr};"
        ],
        post_call=[
            "{py_var} ="
            "\t PyObject_New({PyObject}, &{PyTypeObject});",
            "if ({py_var} == {nullptr}) goto fail;",
            "{py_var}->{PY_type_obj} = {cxx_addr}{cxx_var};",
        ],
        object_created=True,
#            post_call_capsule=[
#                "{py_var}->{PY_type_dtor} = {PY_numpy_array_dtor_context} + {capsule_order};",
#            ],
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    dict(
        name="py_mixin_shadow-create-object",
#            declare=[
#                "{PyObject} *{py_var} = {nullptr};"
#            ],
        post_call=[
            "{PyObject} * {py_var} ="
            "\t PyObject_New({PyObject}, &{PyTypeObject});",
#                "if ({py_var} == {nullptr}) goto fail;",
            "{py_var}->{PY_type_obj} = {cxx_addr}{cxx_var};",
        ],
        object_created=True,
#            post_call_capsule=[
#                "{py_var}->{PY_type_dtor} = {PY_numpy_array_dtor_context} + {capsule_order};",
#            ],
#            fail=[
#                "Py_XDECREF({py_var});",
#            ],
#            goto_fail=True,
    ),
    dict(
        alias=[
            "py_function_shadow*",
            "py_function_shadow&",
        ],
        mixin=[
            "py_mixin_function-declare",
            "py_mixin_shadow-create-object",
        ],
    ),
    dict(
        name="py_in_shadow",
        base="py_in_shadow*",
        arg_call=["*{cxx_var}"],
    ),
    dict(
        name="py_in_shadow&",
        base="py_in_shadow*",
        arg_call=["*{cxx_var}"],
    ),
    
########################################
# std::vector  only used with C++
# list
    dict(
        name="py_in_vector<native>&_list",
        # Convert input list argument into a C++ std::vector.
        # Pass to C++ function.
        # cxx_var is released by the compiler.
        c_helper=["create_from_PyObject_vector_{cxx_T}:create_from_PyObject"],
        parse_format="O",
        parse_args=["&{pytmp_var}"],
        arg_declare=[],
        declare=[
            "PyObject * {pytmp_var};",  # Object set by ParseTupleAndKeywords.
        ],
        local=[
            "cxx",
        ],
        post_declare=[
            "std::vector<{cxx_T}> {cxx_var};",
        ],
        pre_call=[
            "if ({c_helper_create_from_PyObject}\t({pytmp_var}"
            ",\t \"{c_var}\",\t {cxx_var}) == -1)",
            "+goto fail;-",
        ],
        goto_fail=True,
        arg_call=[
            "{cxx_var}",
        ],
    ),
    dict(
        name="py_out_vector<native>&_list",
        # Create a pointer a std::vector and pass to C++ function.
        # Create a Python list with the std::vector.
        # cxx_var is released by the compiler.
        c_helper=["to_PyList_vector_{cxx_T}:to_PyList"],
        arg_declare=[],
        declare=[
            "PyObject * {py_var} = {nullptr};",
        ],
        local=[
            "cxx",
        ],
        post_declare=[
            "std::vector<{cxx_T}> {cxx_var};",
        ],
        arg_call=[
            "{cxx_var}",
        ],
        post_call=[
            "{py_var} = {c_helper_to_PyList}\t({cxx_var});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),
    # XXX - must release after copying result.
    dict(
        alias=[
            "py_function_vector_list",
            "py_function_vector<native>_list",
        ],
        mixin=[
            "py_mixin_function-declare",
        ],
        declare=[
            "PyObject * {py_var} = {nullptr};",
        ],
        post_call=[
            "{py_var} = SHROUD_to_PyList_vector_{cxx_T}\t({cxx_var});",
            "if ({py_var} == {nullptr}) goto fail;",
        ],
        object_created=True,
        fail=[
            "Py_XDECREF({py_var});",
        ],
        goto_fail=True,
    ),

##########
# numpy
# cxx_var will always be a pointer since we must save it in a capsule.
# vectors have the dimension attribute added by generate.py
    dict(
        name="py_in_vector<native>&_numpy",
        # Convert input argument into a NumPy array to make sure it is contiguous,
        # create a local std::vector which will copy the values.
        # Pass to C++ function.
        mixin=[
            "py_mixin_array-parse",
            "py_mixin_array-FROM-OFT-in",
            "py_mixin_template_array_error",
        ],
        local=[
            "cxx",
        ],
        arg_declare=[],
        post_declare=[
            "std::vector<{cxx_T}> {cxx_var};",
            "{cxx_T} * {data_var};",
        ],
        pre_call=[
            "{data_var} = static_cast<{cxx_T} *>(PyArray_DATA({py_var}));",
            "{cxx_var}.assign(\t{data_var},\t {data_var}+PyArray_SIZE({py_var}));",
        ],
        arg_call=[
            "{cxx_var}",
        ],
    ),
    dict(
        name="py_out_vector<native>&_numpy",
        # Create a pointer a std::vector and pass to C++ function.
        # Create a NumPy array with the std::vector as the capsule object.
        mixin=[
            "py_mixin_alloc-cxx-type",
            "py_mixin_malloc_error2",
            "py_mixin_array-SimpleNewFromData",
            "py_mixin_array-capsule",
        ],
        arg_call=["*{cxx_var}"],
    ),
    dict(
        alias=[
            "py_function_vector_numpy",
            "py_function_vector<native>_numpy",
        ],
        mixin=[
            "py_mixin_alloc-cxx-type",
            "py_mixin_malloc_error2",
            "py_mixin_array-SimpleNewFromData",
            "py_mixin_array-capsule",
            "py_mixin_function-assign-pointee",
        ],
    ),


    ########################################
    # ctor
    dict(
        name="py_mixin_ctor_array",
        arg_declare=[],  # No local variable, filled into struct directly.
        declare=[
            "{PY_typedef_converter} {value_var} = {PY_value_init};",
            "{value_var}.name = \"{field_name}\";",
        ],
        parse_format="O&",
        parse_args=["{c_helper_get_from_object}", "&{value_var}"],
        post_call=[
            "SH_obj->{field_name} = "
            "{cast_static}{c_type} *{cast1}{value_var}.data{cast2};",
            "self->{PY_member_object} = {value_var}.obj;"
            "  // steal reference",
        ],
    ),
    dict(
        # Fill an array struct member.
        # helper is set by groups which use this as base.
        name="py_mixin_ctor_array_fill",
        arg_declare=[],  # No local variable, filled into struct directly.
        declare=[
            # Initialize to NULL since it is optional.
            "PyObject *{py_var} = {nullptr};",
        ],
        parse_format="O",
        parse_args=["&{py_var}"],
        post_call=[
            "if ({py_var} != {nullptr}) {{+",
            "if ({c_helper_fill_from_PyObject}(\t{py_var},\t \"{c_var}\","
            "\t SH_obj->{field_name},\t {field_size}) == -1)",
            "+goto fail;-",
            "self->{PY_member_object} = {nullptr};",
            "-}}",
        ],
        goto_fail=True,
    ),
    
    dict(
        alias=[
            "py_ctor_native",
            "py_ctor_native_list",
            "py_ctor_native_numpy",
        ],
        arg_declare=[],  # No local variable, assign to struct in post_call.
        declare=[
            "{c_type} {c_var} = 0;",
        ],
        post_call=[
            "SH_obj->{field_name} = {field_name};",
        ],
    ),
    dict(
        alias=[
            "py_ctor_native[]",
            "py_ctor_native[]_list",
            "py_ctor_native[]_numpy",
        ],
        mixin=[
            "py_mixin_ctor_array_fill",
        ],
        c_helper=["fill_from_PyObject_{c_type}_{PY_array_arg}:fill_from_PyObject"],
    ),
    dict(
        alias=[
            "py_ctor_native*",
            "py_ctor_native*_list",
            "py_ctor_native*_numpy",
        ],
        mixin=[
            "py_mixin_ctor_array",
        ],
        c_helper=["get_from_object_{c_type}_{PY_array_arg}:get_from_object"],
    ),
    
    dict(
        alias=[
            "py_ctor_char[]",
            "py_ctor_char[]_list",
            "py_ctor_char[]_numpy",
        ],
        mixin=[
            "py_mixin_ctor_array_fill",
        ],
        c_helper=["fill_from_PyObject_char:fill_from_PyObject"],
    ),
    dict(
        alias=[
            "py_ctor_char*",
            "py_ctor_char*_numpy",
        ],
        mixin=[
            "py_mixin_ctor_array",
        ],
        c_helper=["get_from_object_char:get_from_object"],
    ),
    dict(
        alias=[
            "py_ctor_char**",
            "py_ctor_char**_list",
        ],
        mixin=[
            "py_mixin_ctor_array",
        ],
        c_helper=["get_from_object_charptr:get_from_object"],
        # Need explicit post_call to change cast to char **.
        post_call=[
            "SH_obj->{field_name} = "
            "{cast_static}char **{cast1}{value_var}.data{cast2};",
            "self->{PY_member_object} = {value_var}.obj;"
            "  // steal reference",
        ],
    ),
    
    ########################################
    # descriptors
    dict(
        name="py_descr_bool",
        setter=[
            "int rv = {PY_get};",
            "if (PyErr_Occurred()) {{+",
            "return -1;",
            "-}}",
            "{c_var} = rv;",
        ],
        getter=[
            "return PyBool_FromLong({c_var});",
        ],
    ),
    dict(
        name="py_descr_native",
        setter=[
            "{cxx_decl} = {PY_get};",
            "if (PyErr_Occurred()) {{+",
            "return -1;",
            "-}}",
            "{c_var} = rv;",
        ],
        getter=[
            "PyObject * rv = {ctor};",
            "return rv;",
        ],
    ),

    dict(
        name="py_descr_native*_list",
        setter_helper=["get_from_object_{c_type}_list:get_from_object"],
        setter=[
            "{PY_typedef_converter} cvalue;",
            "Py_XDECREF({c_var_obj});",
            "if ({c_helper_get_from_object}({py_var}, &cvalue) == 0) {{+",
            "{c_var} = {nullptr};",
            "{c_var_obj} = {nullptr};",
            # Exception is set by hnamefunc0
            "return -1;",
            "-}}",
            "{c_var} = {cast_static}{cast_type}{cast1}cvalue.data{cast2};",
            "{c_var_obj} = cvalue.obj;  // steal reference",
        ],
        getter_helper=["to_PyList_{c_type}:to_PyList"],
        getter=[
            "if ({c_var} == {nullptr}) {{+",
            "Py_RETURN_NONE;",
            "-}}",
            "if ({c_var_obj} != {nullptr}) {{+",
            "Py_INCREF({c_var_obj});",
            "return {c_var_obj};",
            "-}}",
            "PyObject *rv = {c_helper_to_PyList}({c_var}, {npy_intp_size});",
            "return rv;",
        ],
    ),
    dict(
        alias=[
            "py_descr_char*",
            "py_descr_char*_numpy",
        ],
        setter_helper=["get_from_object_{c_type}_list:get_from_object"],
        setter=[
            "{PY_typedef_converter} cvalue;",
            "Py_XDECREF({c_var_data});",
            "if ({c_helper_get_from_object}({py_var}, &cvalue) == 0) {{+",
            "{c_var} = {nullptr};",
            "{c_var_data} = {nullptr};",
            # Exception is set by hnamefunc0
            "return -1;",
            "-}}",
            "{c_var} = {cast_static}{cast_type}{cast1}cvalue.data{cast2};",
            "{c_var_data} = cvalue.dataobj;  // steal reference",
        ],
#        getter_helper=["to_PyList_{c_type}"],
        getter=[
            "if ({c_var} == {nullptr}) {{+",
            "Py_RETURN_NONE;",
            "-}}",
            # Always create a new object since the struct may change value.
#            "if ({c_var_obj} != {nullptr}) {{+",
#            "Py_INCREF({c_var_obj});",
#            "return {c_var_obj};",
#            "-}}",
            "PyObject * rv = {ctor};", # difference from py_descr_native_*_list
            "return rv;",
        ],
    ),
    # XXX - only helper is different from py_descr_native_*_list
    dict(
        name="py_descr_char**_list",
        setter_helper=["get_from_object_charptr:get_from_object"],
        setter=[
            "{PY_typedef_converter} cvalue;",
            "Py_XDECREF({c_var_data});",
            "if ({c_helper_get_from_object}({py_var}, &cvalue) == 0) {{+",
            "{c_var} = {nullptr};",
            "{c_var_data} = {nullptr};",
            "// XXXX set error",
            "return -1;",
            "-}}",
            "{c_var} = {cast_static}{cast_type}{cast1}cvalue.data{cast2};",
            "{c_var_data} = cvalue.dataobj;  // steal reference",
        ],
        getter_helper=["to_PyList_char:to_PyList"],
        getter=[
            "if ({c_var} == {nullptr}) {{+",
            "Py_RETURN_NONE;",
            "-}}",
            # Always create a new object since the struct may change value.
#            "if ({c_var_obj} != {nullptr}) {{+",
#            "Py_INCREF({c_var_obj});",
#            "return {c_var_obj};",
#            "-}}",
            "PyObject *rv = {c_helper_to_PyList}({c_var}, {npy_intp_size});",
            "return rv;",
        ],
    ),

    dict(
        name="py_descr_native[]_list",
        need_numpy = True,
        setter_helper=["fill_from_PyObject_{c_type}_{PY_array_arg}:fill_from_PyObject"],
        setter=[
            "Py_XDECREF({c_var_obj});",
            "{c_var_obj} = {nullptr};",
            "if ({c_helper_fill_from_PyObject}(\t{py_var},\t \"{field_name}\","
            "\t {c_var},\t {npy_intp_size}) == -1) {{+",
            "return -1;",
            "-}}",
        ],
        getter_helper=["to_PyList_{c_type}:to_PyList"],
        getter=[
            "PyObject *rv = {c_helper_to_PyList}({c_var}, {npy_intp_size});",
            "return rv;",
        ]
    ),
    dict(
        name="py_descr_native[]_numpy",
        need_numpy = True,
        setter_helper=["fill_from_PyObject_{c_type}_{PY_array_arg}:fill_from_PyObject"],
        setter=[
            "Py_XDECREF({c_var_obj});",
            "{c_var_obj} = {nullptr};",
            "if ({c_helper_fill_from_PyObject}(\t{py_var},\t \"{field_name}\","
            "\t {c_var},\t {npy_intp_size}) == -1) {{+",
            "return -1;",
            "-}}",
        ],
        getter=[
            "if ({c_var_obj} == {nullptr}) {{+",
            "// Create Numpy object which points to struct member.",
            "npy_intp {npy_dims_var}[{rank}] = {{ {npy_intp_values} }};",
            "{c_var_obj} = PyArray_SimpleNewFromData("
            "\t{npy_rank},\t {npy_dims_var},\t {PYN_typenum},\t {c_var});",
            "-}}",
            "Py_INCREF({c_var_obj});",
            "return {c_var_obj};",
        ]
    ),
    dict(
        name="py_descr_native*_numpy",
        need_numpy = True,
        setter_helper=["get_from_object_{c_type}_numpy:get_from_object"],
        setter=[
            "{PY_typedef_converter} cvalue;",
            "Py_XDECREF({c_var_obj});",
            "if ({c_helper_get_from_object}({py_var}, &cvalue) == 0) {{+",
            "{c_var} = {nullptr};",
            "{c_var_obj} = {nullptr};",
            "// XXXX set error",
            "return -1;",
            "-}}",
            "{c_var} = {cast_static}{cast_type}{cast1}cvalue.data{cast2};",
            "{c_var_obj} = cvalue.obj;  // steal reference",
        ],
        getter=[
            "if ({c_var} == {nullptr}) {{+",
            "Py_RETURN_NONE;",
            "-}}",
            "if ({c_var_obj} != {nullptr}) {{+",
            "Py_INCREF({c_var_obj});",
            "return {c_var_obj};",
            "-}}",
            "npy_intp {npy_dims_var}[{npy_rank}] = {{ {npy_intp_values} }};",
            "PyObject *rv = PyArray_SimpleNewFromData"
            "(\t{npy_rank},\t {npy_dims_var},\t {PYN_typenum},\t {c_var_non_const});",
            "if (rv != {nullptr}) {{+",
            "Py_INCREF(rv);",
            "{c_var_obj} = rv;",
            "-}}",
            "return rv;",
        ]
    ),

    dict(
        alias=[
            "py_descr_char[]",
            "py_descr_char[]_list",
            "py_descr_char[]_numpy",
        ],
        setter_helper=["fill_from_PyObject_char:fill_from_PyObject"], #_{PY_array_arg}",
        setter=[
            "Py_XDECREF({c_var_obj});",
            "{c_var_obj} = {nullptr};",
            "if ({c_helper_fill_from_PyObject}(\t{py_var},\t \"{field_name}\","
            "\t {c_var},\t {npy_intp_size}) == -1) {{+",
            "return -1;",
            "-}}",
        ],
        # XXX - PyString_FromStringAndSize({c_var}, sizeof({c_var});
        # c_var_obj is not cached since if the struct changes the
        # object should be remade.
        getter=[
            "if ({c_var_obj} != {nullptr}) {{+",
            "Py_INCREF({c_var_obj});",
            "return {c_var_obj};",
            "-}}",
            "PyObject * rv = PyString_FromString({c_var});",
            "// XXX assumes is null terminated",
            "return rv;",
        ],
    ),

    dict(
        alias=[
            "py_in_procedure",
        ],
        notes=[
            "Function pointers are not supported by the wrapper."
        ],
    ),
    
]
