/**
 * Licensed to JumpMind Inc under one or more contributor
 * license agreements.  See the NOTICE file distributed
 * with this work for additional information regarding
 * copyright ownership.  JumpMind Inc licenses this file
 * to you under the GNU General Public License, version 3.0 (GPLv3)
 * (the "License"); you may not use this file except in compliance
 * with the License.
 *
 * You should have received a copy of the GNU General Public License,
 * version 3.0 (GPLv3) along with this library; if not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jumpmind.metl.core.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jumpmind.db.sql.Row;
import org.jumpmind.metl.core.runtime.EntityData;
import org.jumpmind.metl.core.util.LogUtils;

public class RelationalModel extends AbstractModel implements IModel, IAuditable {

    private static final long serialVersionUID = 1L;
    
    public static final String TYPE = "RELATIONAL";

    List<ModelEntity> modelEntities;
    
    List<ModelRelation> modelRelations;

    public RelationalModel() {
        initializeModel();
    }
    
    public RelationalModel(String id) {        
        super(id);
        initializeModel();
    }    
    
    public RelationalModel(Folder folder) {
        super(folder);
    }

    private void initializeModel() {
        this.modelEntities = new ArrayList<ModelEntity>();
        this.modelRelations = new ArrayList<ModelRelation>();
    }

	public ModelEntity getEntityById(String entityId) {
        for (ModelEntity entity : modelEntities) {
            if (entity.getId().equalsIgnoreCase(entityId)) {
                return entity;
            }
        }
        return null;
    }

    public ModelEntity getEntityByName(String entityName) {
        for (ModelEntity entity : modelEntities) {
            if (entity.getName().equalsIgnoreCase(entityName)) {
                return entity;
            }
        }
        return null;
    }

    public ModelAttrib getAttributeById(String attributeId) {
        for (ModelEntity entity : modelEntities) {
            for (ModelAttrib modelAttribute : entity.getModelAttributes()) {
                if (modelAttribute.getId().equalsIgnoreCase(attributeId)) {
                    return modelAttribute;
                }
            }
        }
        return null;
    }

    public ModelAttrib getAttributeByName(String entityName, String attributeName) {
        ModelEntity entity = getEntityByName(entityName);
        if (entity != null) {
            for (ModelAttrib modelAttribute : entity.getModelAttributes()) {
                if (modelAttribute.getName().equalsIgnoreCase(attributeName)) {
                    return modelAttribute;
                }
            }
        }
        return null;
    }

    public List<ModelAttrib> getAttributesByName(String attributeName) {
        List<ModelAttrib> attributes = new ArrayList<ModelAttrib>();
        for (ModelEntity entity : modelEntities) {
            for (ModelAttrib modelAttribute : entity.getModelAttributes()) {
                if (modelAttribute.getName().equalsIgnoreCase(attributeName)) {
                    attributes.add(modelAttribute);
                }
            }
        }
        return attributes;
    }

    public List<ModelEntity> getModelEntities() {
        return modelEntities;
    }

    public void setModelEntities(List<ModelEntity> modelEntities) {
        this.modelEntities = modelEntities;
    }
    
    public List<ModelRelation> getModelRelations() {
		return modelRelations;
	}

	public void setModelRelations(List<ModelRelation> modelRelations) {
		this.modelRelations = modelRelations;
	}

    public void sortAttributes() {
        for (ModelEntity modelEntity : modelEntities) {
            AbstractObjectNameBasedSorter.sort(modelEntity.getModelAttributes());
        }
    }

    public Row toRow(EntityData data, boolean qualifyWithEntityName) {
        Row row = new Row(data.size()) {
            private static final long serialVersionUID = 1L;

            @Override
            public String toString() {
                return LogUtils.toJson(data.getChangeType().name(), this);
            }
        };
        Set<String> attributeIds = data.keySet();
        for (String attributeId : attributeIds) {
            ModelAttrib attribute = getAttributeById(attributeId);
            if (attribute != null) {
                ModelEntity entity = getEntityById(attribute.getEntityId());
                if (qualifyWithEntityName) {
                    row.put(entity.getName() + "." + attribute.getName(), data.get(attributeId));
                } else {
                    row.put(attribute.getName(), data.get(attributeId));
                }
            }
        }
        return row;
    }
    
    public Map<String,String> toStringMap(EntityData data, boolean qualifyWithEntityName) {
        Map<String,String> row = new HashMap<>(data.size());
        Set<String> attributeIds = data.keySet();
        for (String attributeId : attributeIds) {
            ModelAttrib attribute = getAttributeById(attributeId);
            if (attribute != null) {
                ModelEntity entity = getEntityById(attribute.getEntityId());
                Object value = data.get(attributeId);
                if (qualifyWithEntityName) {                    
                    row.put(entity.getName() + "." + attribute.getName(), value != null ? value.toString() : null);
                } else {
                    row.put(attribute.getName(), value != null ? value.toString() : null);
                }
            }
        }
        return row;
    }

// TODO: move to hierarchical model class
//    public ModelEntity getRootElement() {
//    	
//    		if (this.type.equalsIgnoreCase(TYPE_HIERARCHICAL) && getModelEntities().size()>0) {
//    			return getEntityTree().first();
//    		}
//    		else {
//    			return null;    			
//    		}
//    }
//    
//    public TreeSet<ModelEntity> getEntityTree() {
//		return new TreeSet<ModelEntity>(this.getModelEntities());
//    }
    
    public ModelAttrib getModelAttribByTypeEntityId(String typeEntityId) {
        for (ModelEntity entity:this.getModelEntities()) {
            for (ModelAttrib attrib : entity.getModelAttributes()) {
                if (typeEntityId.equalsIgnoreCase(attrib.getTypeEntityId())) {
                    return attrib;
                }
            }
        }
        return null;
    }

    @Override
    public String getType() {
        return TYPE;
    }
    
}
