package org.jetbrains.ring

class Graph {
    val nodes: MutableList<Node> = mutableListOf()
    val size
        get() = nodes.size

    fun first(): Node = nodes.first()

    operator fun get(nodeId: Int): Node = nodes.getOrElse(nodeId) { nodes.add(it, Node(it)) ; nodes[it] }
}

class Node(val id: Int) {
    val neighbors: MutableMap<Node, Int> = mutableMapOf()
}

fun parseData(content: List<String>): Graph {
    val graph = Graph()

    content.filter(String::isNotBlank).forEachIndexed { lineNumber, line ->
        addNeighbors(line, graph, lineNumber)
    }

    return graph
}

private data class ClosestNode(val node: Node, val distance: Int)

class Greedy(private val graph: Graph) {
    private val visitedNodes: MutableSet<Node> = mutableSetOf()

    private fun getClosest(node: Node): ClosestNode? {
        val nodes = node.neighbors
                .filter { it.value != 0 && !visitedNodes.contains(it.key) }
        val clostest = nodes
                .minBy { it.value }!!
        return ClosestNode(clostest.key, clostest.value)
    }

    fun solve() {
        var previousNode = graph.first()
        visitedNodes.add(graph.first())
        var cost = 0

        while (visitedNodes.size != graph.size) {
            val closest = getClosest(previousNode)!!
            visitedNodes.add(closest.node)
            cost += closest.distance
            previousNode = closest.node
        }

        cost += visitedNodes.last().neighbors[graph.first()]!!
    }
}

fun addNeighbors(line: String, graph: Graph, lineNumber: Int): Node {
    val node = graph[lineNumber]
    line.split(',').forEachIndexed { index, value ->
        node.neighbors.put(graph[index], value.trim().toInt())//.trim().toInt())
    }
    return node
}


class GraphSolverBenchmark {
    val content = listOf(
    "0,604,802,57,278,84,33,758,908,420,610,484,195,474,254,969,172,463,130,335,704,507,358,852,356,444,396,95,446,86,397,893,56,812,967,40,229,33,112,913,548,902,793,563,318,311,579,58,145,304,133,440,159,219,300,855,443,385,157,770,29,97,229,734,829,324,917,380,817,597,461,386,536,942,533,774,71,996,752,6,741,644,701,782,870,947,159,464,412,807,765,174,334,77,930,536,885,533,202,191,825,342,578,76,394,432,786,313,693,587,659,55,491,467,33,955,883,353,452,60,588,87,109,473,66,965,973,950,952,492,624,810,87,62,218,435,559,283,562,798,349,201,383,249,69,263,415,961,56,755,616,27,186,63,247,800,503,767,869,13,252,746,6,436,793,189,926,442,465,225,573,379,875,620,294,452,832,739,354,781,348,264,790,110,369,393,678,873,8,708,268,177,722,49,448,849,381,720,166,208,178,634,557,595,969,968,384,682,558,298,726,899,444,420,481,714,524,633,897,279,760,755,101,37,769,460,252,869,700,182,507,389,263,88,291,885,106,638,375,661,375,941,487,58,890,700,121,852,700,888,814,546,374,771,683,730,478,446,456,447,369,727,258,307,794,669,438,604,549,921,848,918,694,512,390,913,118,776,441,438,488,187,813,668,373,522,411,772,943,183,6,84,984,619,453,729,744,166,880,800,860,869,805,873,296,706,901,155,111,286,958,119,652,2,140,248,809,475,752,368,924,108,276,472,469,704,92,680,374,473,372,607,251,454,718,228,252,116,812,561,49,248,826,682,598,900,239,652,367,293,196,868,18,555,746,142,170,75,788,675,661,697,632,185,283,838,509,843,222,690,783,963,550,621,211,276,133,303,695,211,957,685,686,98,902,725,707,758,194,461,411,177,912,818,92,512,480,795,586,161,962,546,891,223,476,937,669,938,516,247,668,751,195,678,848,764,888,40,291,456,710,685,129,637,876,293,194,102,153,903,414,766,368,31,561,537,962,873,902,529,362,225,936,743,782,307,217,300,921,699,664,638,862,170,916,806,721,100,395,800,244,50,746,500,962,753,547,889,199,743,611,357,952,781,899,548,507,875,434,702,109,32,835,788,108,469,171,993,886,130,871,752,217,896,446,66,223,648,352,497,782,182,463,557,20,849,685,956,834,676,28,533,256,904,606,334,472,206,942,166,38,203,953,201,566,409,64,833,958,784,852,817,835,829,911,254,917,706,681,35,456,795,615,349,597,198,643,18,955,531,431,49,80,699,35,345,830,687,406,605,872,293,697,348,296,472,951,117,464,843,96,359,945,388,226,338,41,429,367,940,474,477,978,867,975,632,388,319,999,409,145,748,136,226,853,21,620,89,319,557,602,206,759,638,977,499,723,381,223,299,298,981,642,100,95,828,794,689,778,983,201,195,905,670,313,555,762,962,212,311,572,852,657,622,461,101,771,644,925,761,286,275,297,246,904,24,327,474,455,395,148,330,919,710,485,709,341,871,669,27,5,643,840,999,940,489,803,164,806,187,989,178,257,835,516,172,594,661,77,28,961,222,953,604,611,111,296,784,967,175,448,130,611,998,669,350,234,272,149,714,942,501,378,192,28,891,694,37,788,495,961,770,736,207,903,664,21,345,480,604,451,271,99,81,131,902,314,436,979,634,63,164,849,736,283,210,5,131,117,978,379,488,392,649,951,927,757,835,567,472,530,589,959,213,74,904,270,677,787,884,287,569,182,11,604,384,440,914,286,110,760,639,284,3,523,275,968,342,759,765,907,624,468,674,891,81,205,226,907,158,473,622,67,981,858,790,268,176,998,811,987,130,307,735,414,822,625,147,210,875,778,175,388,671,762,352,550,496,82,985,340,915,946,803,441,822,142,297,409,773,356,890,170,201,815,931,486,626,223,290,722,178,39,873,601,741,88,690,295,272,189,397,631,546,178,718,124,397,234,251,503,64,990,295,733,948,449,540,728,801,351,2,553,75,595,907,299,469,258,40,524,845,281,892,64,50,994,630,380,785,60,341,278,468,828,213,711,17,190,952,370,464,291,918,454,239,399,914,451,281,720,215,62,962,8,208,617,285,391,544,521,254,215,559,184,805,737,774,661,784,206,345,994,238,979,394,685,131,822,717,397,715,607,502,400,937,653,653,352,619,733,275,753,257,766,839,814,308,402,528,912,260,754,367,8,301,917,351,988,696,570,416,444,784,581,829,870,378,368,224,956,237,86,648,448,10,856,196,146,484,229,344,870,673,738,250,949,264,995,873",
    "604,0,40,254,413,440,450,189,382,388,460,802,266,415,70,10,137,344,215,456,530,796,664,686,149,440,863,46,325,418,304,904,295,948,626,722,167,27,396,183,396,822,81,117,114,950,786,101,888,750,550,293,584,254,732,694,278,971,813,730,589,978,214,629,781,358,941,962,144,609,629,25,779,492,412,778,302,220,159,617,673,703,602,256,997,805,922,332,116,639,333,384,526,980,764,929,183,987,806,519,293,758,992,261,587,366,154,544,480,983,148,855,372,746,876,207,833,719,607,546,124,719,555,154,264,486,643,437,828,729,571,950,71,908,63,125,560,335,612,16,296,884,446,530,622,121,742,378,816,616,307,115,492,646,831,308,254,860,338,739,614,918,21,493,370,512,817,584,872,221,184,321,218,878,919,841,327,320,949,838,138,537,868,329,759,923,496,960,488,452,407,844,390,619,477,822,968,17,632,471,639,273,142,763,708,694,395,53,976,778,842,967,875,668,980,367,48,424,692,958,223,374,929,828,450,465,411,75,199,145,705,227,66,620,982,33,317,925,780,598,712,29,191,650,287,301,735,765,117,501,324,264,10,538,808,114,389,563,375,421,267,265,554,862,609,244,662,714,194,853,788,359,355,944,781,907,520,52,669,597,292,269,463,378,64,452,672,474,660,455,32,871,305,571,646,477,890,624,850,69,155,897,498,813,295,719,696,381,341,877,30,213,227,394,332,421,924,385,405,407,486,681,293,515,590,497,657,130,680,535,881,478,474,541,522,135,163,180,442,380,478,682,218,350,355,533,870,515,757,740,73,837,600,762,337,555,950,316,235,236,786,502,476,804,44,717,793,688,246,854,85,476,243,513,895,569,29,561,528,525,333,986,829,680,587,336,182,909,196,614,118,198,700,94,100,29,381,933,46,32,920,646,46,482,479,588,596,263,550,902,261,326,272,780,568,295,597,704,46,768,76,731,369,25,421,622,846,756,686,176,583,435,364,415,614,571,145,888,999,235,727,913,128,904,57,300,25,764,822,924,415,526,689,858,224,381,526,607,119,646,544,202,365,277,556,681,668,350,496,113,14,914,813,974,528,389,490,491,550,983,812,652,170,618,665,439,966,386,512,521,589,218,481,288,995,791,791,674,582,593,29,787,687,878,640,11,883,827,723,884,670,129,831,500,422,196,936,719,10,736,641,227,538,609,954,84,304,961,667,781,137,842,79,937,810,56,179,335,760,197,991,995,876,5,342,379,132,231,783,59,459,932,907,612,40,138,857,958,614,164,338,5,467,894,513,352,630,460,511,179,419,826,155,584,448,16,82,909,104,608,900,971,763,98,464,955,314,12,618,150,252,880,429,309,539,687,681,883,888,13,931,804,158,451,768,954,695,751,230,61,834,940,156,410,734,599,444,399,956,210,290,117,878,725,758,453,652,41,207,144,162,46,689,775,318,311,247,748,318,828,500,276,191,218,503,751,994,171,530,884,538,653,69,728,14,905,147,88,291,192,669,742,260,206,152,419,84,814,656,236,262,476,806,589,155,935,197,623,546,492,794,179,487,555,710,215,94,707,326,469,456,186,82,830,22,865,589,375,312,626,550,221,670,295,353,690,167,99,370,833,642,437,746,565,23,956,830,266,866,792,56,337,424,136,669,2,568,584,294,362,421,304,621,582,439,825,45,208,249,63,468,764,594,90,745,7,24,252,528,973,117,290,548,202,264,831,235,162,683,648,530,873,286,333,221,223,95,665,884,235,8,839,225,756,840,748,901,385,514,720,113,972,196,509,601,160,133,350,444,846,464,507,733,303,597,568,640,328,932,520,288,663,519,741,115,152,722,948,318,60,973,239,488,813,322,616,336,241,957,892,196,691,424,249,932,603,677,122,731,939,960,571,430,280,976,144,894,356,855,794,830,721,988,887,318,723,560,688,644,44,946,988,512,979,431,640,688,805,386,809,863,329,152,131,603,77,781,325,321,501,876,240,152,734,13,707,607,375,159,446,415,422,808,798,299,683,47,260,871,344,638,799,881,997,342,572,487,984,96,757,77,843,367,32,840,777,640,38,541,59,160,765,92,32,56,662,451,449,517,476,672,902,22,352,970,140,513,198,595,85,601,341,719,608,663,861,509,621,813,37,195,915,662,284,749,33,812,480,949,133,947,204,246,782,244,810,48,661,585,135,99,321,450,513,70,896,564,677,490,703,977,243,630,613,262,721,100,807,17,24,534,892,626,618,59,640,908,996,874,212,342,778,657,755,777,470,233,547,913,804",
    "802,40,0,321,721,986,467,509,781,544,43,803,544,354,937,252,687,373,142,628,327,167,769,538,459,857,820,971,750,652,904,223,132,909,607,679,461,628,781,698,361,121,216,819,304,475,74,324,59,349,120,555,62,299,840,451,353,402,539,565,562,424,973,187,553,859,381,51,696,343,79,63,286,499,302,842,682,896,576,942,68,454,786,497,517,255,417,518,417,523,896,784,55,673,417,665,893,870,4,795,396,418,483,566,132,985,566,605,837,448,985,334,36,409,227,428,625,275,207,176,190,881,447,530,458,382,466,65,16,85,21,223,921,353,134,593,511,430,516,646,640,294,623,203,717,762,715,496,442,11,113,751,888,803,934,210,870,900,656,780,679,147,201,135,932,785,923,240,390,636,534,846,415,993,977,910,132,949,7,624,684,720,651,556,534,661,852,457,259,348,990,365,284,260,652,788,918,689,104,967,837,653,656,465,154,321,722,553,402,768,262,257,941,118,51,170,229,865,45,982,109,575,320,879,331,990,887,921,368,294,743,78,415,23,434,618,22,574,692,953,154,656,121,543,160,848,724,366,765,598,880,5,389,660,481,221,771,398,304,816,918,372,353,586,487,698,386,823,849,784,78,850,855,107,868,269,162,197,338,314,996,294,133,874,638,314,711,534,132,564,482,430,564,128,996,185,345,714,848,217,648,244,67,94,26,492,982,336,300,246,275,61,888,197,978,191,448,747,975,182,972,469,986,770,12,700,977,216,324,839,723,259,101,875,823,242,809,249,681,417,257,492,881,511,86,105,262,641,977,257,182,126,672,693,864,252,424,351,634,190,90,507,122,398,763,207,252,209,561,354,206,829,537,510,279,277,316,143,32,948,772,901,538,930,299,181,886,162,473,446,210,8,431,161,747,529,206,568,758,752,346,738,131,601,827,22,448,216,892,617,622,39,447,898,759,749,103,341,435,900,113,994,760,96,410,175,731,591,318,731,431,338,476,193,735,106,716,479,344,571,294,498,719,921,571,881,136,929,603,541,608,192,118,36,921,715,831,958,774,933,473,588,207,107,864,34,556,490,890,874,642,256,945,701,214,45,937,929,415,221,6,666,263,1,314,643,299,857,539,210,559,670,477,766,400,17,869,915,361,882,934,471,397,92,714,569,886,133,47,370,801,455,849,326,32,329,750,458,445,665,420,613,596,755,870,406,727,935,530,366,822,203,704,449,665,158,322,293,927,98,165,409,238,94,442,877,445,266,706,312,712,321,65,914,962,71,172,795,973,430,355,969,510,610,957,696,505,533,884,754,968,998,994,214,388,509,980,622,353,233,496,178,988,344,445,890,50,376,620,766,552,98,392,700,723,753,83,245,967,253,563,736,880,519,76,3,624,102,254,8,574,804,798,832,883,137,694,608,28,318,326,928,330,987,173,553,181,402,156,163,331,587,771,204,607,979,404,137,688,572,967,206,595,5,895,787,47,844,458,451,269,716,260,849,592,462,850,840,359,296,830,724,101,730,795,997,795,415,475,851,274,953,899,29,442,830,374,754,622,73,622,190,271,412,275,427,398,206,953,138,198,947,234,745,669,47,270,799,300,67,102,566,502,796,11,186,127,426,656,576,398,723,266,895,899,439,509,1000,517,991,77,617,659,695,334,797,375,721,220,739,464,67,62,396,530,888,346,67,264,744,772,726,838,590,964,760,201,16,9,160,868,124,267,460,928,316,325,173,52,820,394,410,759,46,246,275,400,664,787,817,860,333,452,897,618,760,943,24,834,426,894,126,413,484,412,769,301,349,973,720,343,210,915,640,221,94,937,576,995,751,235,914,855,548,434,415,280,83,405,104,197,213,852,907,357,4,99,969,116,474,421,86,950,62,216,409,559,488,511,618,766,232,239,591,556,351,839,689,387,944,336,262,156,978,879,623,634,258,805,64,603,919,87,809,605,36,197,183,639,333,991,644,570,761,180,673,802,137,78,680,825,283,630,661,843,964,443,945,868,455,470,458,710,493,378,467,194,984,178,44,765,755,523,121,834,711,136,668,888,467,799,429,25,817,322,339,141,662,560,647,840,462,761,410,764,177,39,138,974,919,138,965,727,270,191,780,59,213,694,381,534,970,20,39,952,195,784,683,218,146,915,520,335,561,873,916,478,11,470,583,601,973,266,88,141,299,916,97,951,306,220,434,419,183,307,287,216,258,162,249,210,135,677,290,752,535,984,861,840,436,14,529,762,746,944,873,838,795,296,199,149,675,941,533,422,227,581,470,328,976",
    "57,254,321,0,726,976,877,679,555,918,259,522,153,491,650,419,377,449,295,813,575,548,550,571,135,708,85,803,595,15,973,375,674,439,832,866,244,950,993,821,818,906,58,492,863,955,503,775,396,520,652,572,404,747,304,949,222,7,383,658,89,188,122,919,600,45,473,37,105,517,619,545,30,163,866,785,54,219,573,636,635,557,930,80,595,94,675,781,143,384,363,838,109,41,294,160,950,754,224,659,377,26,517,74,503,784,953,482,526,469,13,183,732,180,775,752,408,597,866,425,684,790,475,48,839,592,379,950,990,980,716,332,189,908,819,65,555,158,438,681,78,447,931,721,183,377,333,839,570,636,296,301,947,552,920,443,592,793,781,277,326,971,673,208,83,916,373,413,642,157,212,366,680,482,876,431,961,307,10,286,188,982,970,237,614,274,50,808,973,66,918,456,370,723,118,301,412,789,988,80,886,457,333,603,737,985,546,479,222,881,247,954,128,478,249,511,378,363,108,577,932,170,241,733,500,357,200,713,503,588,513,833,440,189,241,10,210,56,331,982,572,675,525,801,546,827,99,741,130,337,869,743,702,585,401,725,478,889,796,244,310,251,484,862,94,179,486,991,626,301,855,974,793,135,183,768,746,162,3,73,810,559,822,836,74,250,739,982,336,284,604,411,974,808,349,808,688,280,743,824,396,664,467,751,156,576,183,930,471,667,564,731,900,938,249,300,542,745,24,119,41,486,869,298,149,310,350,146,568,159,209,243,164,575,305,881,528,618,755,484,103,720,541,355,467,478,787,937,241,812,835,148,453,881,774,126,782,884,518,356,981,248,862,177,472,490,194,179,788,663,855,673,728,479,937,890,328,961,91,311,494,25,587,780,924,897,532,749,460,8,857,632,316,271,872,797,99,729,76,585,832,837,838,4,777,300,30,78,656,867,583,57,407,584,442,766,895,420,238,492,552,379,572,399,353,828,763,588,735,331,837,551,855,570,15,700,400,897,800,53,800,860,105,1,616,56,840,279,360,881,322,51,34,810,143,908,482,932,521,535,174,560,208,277,693,95,928,461,747,762,718,635,225,458,617,128,272,463,260,74,531,726,822,622,831,442,410,74,411,459,89,577,969,74,672,930,663,387,973,674,275,791,723,360,238,270,588,401,704,258,575,407,823,406,50,451,566,20,547,237,307,430,331,176,121,682,696,25,379,450,458,28,899,169,627,879,97,84,702,720,820,766,328,198,794,41,970,978,175,847,324,706,885,931,724,124,423,731,15,553,949,79,979,134,926,103,578,97,905,866,443,885,11,984,698,613,689,931,623,390,532,752,456,497,438,363,701,198,900,928,344,688,842,506,946,19,933,650,97,166,439,746,837,697,444,208,346,711,674,198,784,184,159,845,876,87,817,726,301,895,295,810,239,528,531,733,988,941,654,334,9,294,441,194,902,117,969,201,451,589,106,322,482,542,746,391,529,74,858,355,46,530,200,344,955,789,221,131,555,217,275,964,137,204,388,96,121,329,156,488,645,398,23,214,709,875,129,491,668,139,276,678,673,94,764,731,778,196,10,87,3,644,914,500,643,579,508,878,52,830,363,921,839,113,129,710,623,992,97,807,906,592,985,349,872,383,83,351,431,323,391,359,930,967,986,306,796,625,65,379,201,693,135,814,498,358,847,895,906,570,936,826,397,333,62,296,431,960,424,916,47,52,69,164,479,380,496,455,615,750,957,397,293,645,50,469,67,923,552,198,698,698,741,676,831,92,846,775,142,839,268,538,803,695,765,125,263,627,877,119,195,596,480,138,858,596,439,930,922,316,917,751,203,557,26,735,95,549,702,424,479,674,86,978,189,763,34,950,927,895,947,759,83,330,348,651,773,914,494,387,644,66,219,333,87,427,163,279,443,445,189,148,665,866,855,267,417,553,94,423,992,97,737,177,97,858,890,591,23,84,945,621,755,459,536,719,128,992,591,861,91,261,411,717,702,901,488,350,620,526,733,350,364,31,590,971,381,803,419,488,319,131,741,671,756,970,751,675,855,933,133,489,195,268,776,634,871,454,976,605,380,868,472,410,549,277,392,907,893,987,408,739,799,175,752,288,861,39,40,346,963,861,481,800,907,122,455,8,770,153,338,773,13,374,709,839,956,600,429,600,92,276,650,489,804,157,259,655,193,882,169,220,309,887,270,963,648,658,221,396,15,324,725,825,479,418,240,353,110,219,78,229,348,141,223,37,50,312,367,307,885,215,565,658,738,191,136,994,706,299",
    "278,413,721,726,0,584,44,514,846,58,260,489,486,66,420,805,805,115,978,230,125,277,219,921,253,211,368,206,811,700,708,271,620,26,885,289,859,190,731,963,186,586,52,700,83,981,733,396,915,102,122,905,891,296,198,736,352,527,836,911,443,87,176,376,531,656,959,381,929,609,312,81,1000,453,659,936,930,252,904,894,70,352,793,354,862,234,644,180,361,702,792,312,448,312,410,924,337,720,907,222,319,153,584,855,512,917,630,916,484,362,926,942,653,37,484,416,381,553,441,433,689,677,393,767,8,904,387,965,540,463,362,112,154,430,25,850,374,206,544,833,198,844,658,692,135,85,377,546,260,736,21,40,309,74,164,250,183,886,464,540,586,85,769,967,781,41,271,412,854,961,528,724,947,269,7,530,363,253,495,230,683,634,851,906,983,610,966,563,165,217,116,528,285,413,431,736,366,235,90,514,739,357,806,963,754,213,427,819,981,661,158,123,21,946,140,526,149,85,254,710,100,317,239,471,217,966,557,263,433,940,597,706,38,330,856,790,509,460,143,434,493,332,536,989,647,96,247,695,888,276,684,916,343,55,767,754,289,190,185,906,463,10,235,374,696,742,176,98,799,394,334,620,956,45,119,527,444,102,62,169,396,806,906,889,931,706,303,468,620,208,428,374,89,572,288,456,506,667,725,392,37,641,545,137,495,653,317,90,734,473,599,566,508,383,727,957,237,607,595,683,954,325,619,544,66,607,614,236,324,201,96,373,275,738,567,516,698,89,387,14,354,780,507,305,462,897,623,397,247,241,442,888,513,754,614,277,764,716,100,162,636,592,792,827,181,620,758,118,311,525,676,765,933,298,257,668,939,971,661,4,853,257,935,245,552,766,465,922,650,824,706,553,75,185,901,342,892,859,356,312,803,982,221,565,799,161,557,185,453,164,931,35,304,724,451,305,491,572,297,716,348,606,789,268,887,224,574,308,86,348,942,839,86,52,523,396,839,841,707,919,348,140,44,526,636,479,5,507,903,582,708,988,190,983,55,243,621,141,25,64,537,171,714,685,820,43,323,800,535,249,92,178,146,125,77,374,432,388,758,158,121,868,961,959,641,370,164,925,454,333,458,739,15,856,35,467,465,404,29,956,215,265,533,307,424,501,166,501,87,268,363,134,415,175,483,103,983,327,45,527,311,248,973,862,683,951,91,681,381,923,869,307,21,469,992,196,617,598,60,70,57,62,893,315,76,635,477,538,938,757,911,414,700,648,266,682,578,597,447,734,570,236,559,417,419,971,269,858,28,151,504,236,19,941,499,760,963,163,332,910,877,116,41,128,146,582,11,573,432,919,207,846,584,704,118,44,731,780,416,806,885,332,663,999,915,174,549,545,430,738,797,127,576,100,980,867,204,620,287,178,803,153,696,80,791,526,923,824,658,602,86,332,104,689,959,217,466,184,313,774,735,829,644,115,935,807,283,48,38,796,924,559,160,206,490,866,432,511,231,445,841,657,468,723,124,940,212,387,855,65,382,504,586,70,704,459,831,492,220,308,770,832,239,828,764,273,149,191,607,382,895,187,513,554,997,630,52,938,264,872,260,187,911,918,541,328,517,743,975,485,967,608,24,647,760,861,226,362,600,681,528,416,971,781,46,230,837,541,235,391,594,274,263,372,648,593,792,626,755,535,235,739,8,808,65,759,13,59,364,50,485,393,960,972,970,967,711,876,786,423,412,254,107,395,785,784,220,876,21,6,904,304,305,809,359,276,533,921,150,496,606,147,111,541,85,646,710,749,708,357,354,33,80,28,356,181,864,889,143,916,455,731,655,620,704,544,462,593,295,507,389,530,650,938,349,42,308,910,587,124,492,893,574,341,652,447,701,258,565,510,804,90,290,221,926,273,147,725,871,507,777,462,209,996,989,491,232,39,221,962,152,667,200,311,195,652,81,95,307,739,685,403,955,525,289,474,172,634,987,30,171,900,638,276,703,265,664,944,563,128,263,453,400,748,515,777,114,893,20,299,530,472,598,381,288,224,610,115,941,795,565,101,837,655,613,991,940,305,438,9,902,863,58,951,266,135,131,44,133,411,77,961,778,84,857,877,378,987,612,313,8,855,543,461,724,511,154,900,464,356,42,972,42,463,353,417,733,999,854,307,510,421,702,158,19,960,930,522,360,459,142,19,826,580,833,483,779,982,61,449,490,271,884,416,9,14,178,92,142,576,113,56,15,400,212,935,668,332,465,78,957,826,559,501,195,588,558,952,887,90",
    "84,440,986,976,584,0,109,366,390,794,546,750,18,671,300,198,220,770,609,610,544,818,439,158,84,27,534,980,462,713,488,755,518,552,134,652,299,852,156,676,904,251,841,175,504,116,242,248,794,281,805,595,59,924,109,36,87,931,779,556,782,393,576,675,456,939,788,887,922,858,606,22,943,781,129,203,381,316,445,851,976,223,902,231,910,48,531,336,936,64,421,685,890,581,899,792,947,188,981,373,819,114,983,410,313,752,402,350,142,295,285,2,25,319,458,666,716,671,842,659,288,987,558,590,853,878,812,337,210,805,262,360,209,461,756,97,523,220,821,140,285,367,287,922,213,359,961,319,674,229,245,848,181,394,165,388,319,497,690,797,985,796,96,484,873,694,607,736,713,371,559,476,446,7,431,981,181,64,237,962,445,63,561,882,980,211,458,409,222,559,641,570,185,33,389,435,961,606,170,962,72,777,127,261,532,979,344,263,451,602,48,408,122,297,161,685,422,83,288,284,866,289,828,859,732,853,816,481,776,586,175,168,591,206,784,533,541,381,423,522,148,569,995,386,382,119,123,876,440,695,838,940,338,791,594,427,825,688,535,185,420,307,834,229,508,748,502,745,643,656,122,190,543,340,552,360,441,988,911,846,39,154,10,577,874,447,240,332,512,112,357,714,536,203,95,570,861,815,978,287,31,875,428,246,888,903,710,205,946,117,457,713,368,959,930,361,614,446,575,234,875,264,620,608,952,510,256,966,191,610,786,661,45,108,333,399,371,971,354,102,382,937,739,149,732,597,236,703,633,191,512,569,328,903,32,657,890,789,873,258,187,634,205,397,870,895,344,448,910,415,595,81,977,659,648,742,463,7,65,572,807,967,221,16,131,72,635,394,389,609,382,805,632,55,538,92,696,452,703,891,218,502,645,815,611,696,310,328,129,186,275,164,28,379,136,194,322,483,572,630,357,673,216,738,994,171,174,216,51,175,878,821,883,356,825,69,512,984,694,394,477,145,115,292,971,911,75,372,893,645,46,588,209,681,247,154,105,619,319,272,501,367,219,524,797,253,589,699,340,421,192,999,53,339,250,510,284,43,354,49,389,606,522,695,531,805,144,140,433,396,516,367,764,935,892,78,8,306,72,130,215,210,312,635,595,79,226,990,395,307,630,279,938,203,728,848,258,574,425,63,396,478,171,348,37,64,461,50,753,587,751,573,947,221,624,363,693,442,41,858,767,632,994,759,462,559,678,625,692,897,51,425,119,567,895,664,536,207,162,841,227,760,143,956,426,241,20,569,828,373,843,870,284,246,152,87,106,801,876,249,802,529,722,154,263,891,703,650,872,155,196,749,154,16,648,11,383,482,235,373,648,206,249,417,712,517,492,251,137,559,974,70,306,933,457,864,437,910,834,181,348,372,213,48,903,74,969,145,406,485,253,210,338,36,277,533,632,420,213,53,823,430,75,619,463,196,760,490,338,757,161,64,626,395,717,79,839,274,903,630,282,966,337,290,656,166,637,967,735,177,766,349,119,493,957,388,574,422,513,339,453,676,262,313,924,444,115,895,53,405,463,422,575,2,386,120,896,422,569,189,414,596,996,4,697,939,246,41,657,801,12,28,206,396,835,161,359,70,493,509,365,595,142,234,819,305,924,312,678,123,445,105,454,944,221,249,131,967,433,694,240,856,553,22,91,684,191,882,530,691,712,446,837,726,790,197,263,511,936,938,427,718,186,141,274,981,777,558,152,635,279,862,988,779,683,555,710,68,186,640,200,301,599,320,466,344,123,342,449,638,124,667,105,180,965,56,446,9,568,837,211,865,280,10,144,488,270,187,147,263,664,695,592,990,950,790,235,528,992,108,543,239,161,592,124,348,117,779,827,951,762,679,837,892,640,722,154,730,412,269,81,203,201,59,764,616,170,292,254,306,43,547,950,8,63,73,889,91,873,829,512,3,769,862,225,147,221,373,886,830,290,344,379,174,273,988,561,899,397,739,169,137,866,383,219,304,546,576,342,500,682,873,424,61,156,632,802,549,15,535,981,21,921,188,345,152,612,888,224,529,445,486,727,165,94,23,53,415,16,109,392,569,650,117,589,546,981,572,213,572,239,522,559,932,672,394,734,883,703,758,609,942,128,811,998,103,821,190,73,647,715,315,877,162,417,608,215,671,160,235,332,30,472,905,503,385,772,467,9,739,23,814,719,862,674,138,258,562,714,67,423,220,485,699,427,752,873,174,771,921,89,607,196,535,191,89,761,182,869,282",
    "33,450,467,877,44,109,0,352,21,21,261,739,777,16,839,512,876,59,263,709,919,213,354,765,858,223,556,268,898,611,385,74,864,503,490,891,805,551,77,475,499,367,648,899,319,175,223,988,808,296,3,216,197,192,807,627,976,46,264,139,974,339,715,810,819,945,897,607,144,862,835,899,562,391,157,933,691,918,516,169,380,484,378,977,227,914,681,70,970,789,910,187,914,381,999,749,341,828,857,735,438,782,840,517,151,305,449,120,631,278,314,880,642,84,408,693,358,960,799,544,434,865,975,997,6,484,873,206,860,36,171,814,737,834,185,148,472,479,969,463,751,379,202,58,855,233,826,245,408,345,736,303,16,576,212,586,909,532,443,1000,180,352,740,341,215,940,618,140,760,427,228,771,698,371,882,533,728,996,572,242,143,962,133,390,664,786,580,895,212,873,589,171,625,558,770,356,535,784,930,214,210,51,932,163,617,662,588,139,152,417,852,326,445,917,924,798,425,323,297,389,483,151,569,529,292,298,569,270,942,671,785,569,805,232,328,315,966,65,852,726,788,349,169,870,783,324,281,584,278,219,120,909,935,557,541,706,587,419,755,576,642,363,387,228,385,632,2,815,899,790,587,494,813,65,477,543,874,356,33,518,998,110,306,25,965,769,226,909,535,597,878,101,87,32,404,194,555,544,597,886,282,890,215,973,211,751,527,769,887,873,704,803,864,296,609,396,229,670,895,222,739,438,40,256,939,377,256,512,335,850,182,404,680,529,297,362,801,53,607,385,80,880,586,930,860,162,509,899,464,917,31,743,275,381,874,906,921,596,405,75,522,322,838,198,354,885,657,580,519,588,224,156,745,834,34,768,258,68,722,727,255,819,521,605,886,353,937,657,340,105,332,240,945,437,260,398,945,948,762,39,474,897,604,11,965,945,709,705,151,240,772,682,688,692,676,177,308,504,586,118,109,445,610,344,751,510,120,104,959,131,925,680,736,812,763,985,268,815,64,714,482,886,439,496,496,11,736,304,833,818,393,83,161,51,62,419,496,372,810,953,368,171,439,698,939,370,711,482,106,925,562,341,529,829,260,941,279,925,327,901,56,528,113,289,780,181,460,640,557,566,9,235,410,959,527,370,646,257,43,710,408,205,74,150,496,805,363,333,577,154,471,906,921,504,752,302,415,329,592,628,305,919,521,285,545,342,934,653,696,267,167,699,775,723,274,910,535,893,644,818,917,306,400,694,43,679,826,866,837,311,263,568,957,63,260,523,834,747,627,863,691,192,189,555,88,41,663,150,13,367,747,765,810,952,697,811,909,895,375,608,664,496,128,100,210,631,702,553,779,629,532,989,160,198,455,258,144,518,979,486,159,89,91,168,178,429,621,286,95,574,949,493,175,153,273,73,276,353,224,468,914,910,885,11,85,168,3,38,823,572,915,41,817,812,61,730,597,543,601,31,163,389,26,543,593,847,648,138,134,430,266,439,858,20,186,967,710,264,802,265,317,325,453,549,104,412,156,54,103,668,113,700,736,399,458,984,689,85,738,357,148,343,992,903,916,982,428,271,947,383,605,606,553,287,511,891,537,243,101,532,262,223,948,859,463,91,700,195,364,942,891,411,202,954,444,548,581,988,562,116,400,21,111,369,898,20,653,498,915,933,398,893,140,264,674,221,426,628,639,467,614,480,89,777,903,861,457,133,711,487,708,677,428,159,468,675,927,897,216,734,162,769,996,168,859,901,93,630,989,343,85,575,415,907,24,35,537,311,783,799,657,11,792,52,935,669,542,311,687,757,47,551,106,87,70,111,628,694,514,117,617,239,425,839,960,52,415,810,85,178,937,760,98,193,574,301,863,431,611,1,193,832,447,18,333,493,773,40,129,638,642,914,908,880,163,240,558,485,621,217,825,76,898,172,816,165,36,51,509,123,198,645,880,628,331,591,964,91,823,41,843,605,449,123,361,833,977,39,382,466,605,22,574,750,443,385,172,167,470,539,881,2,760,855,126,747,839,402,158,193,933,445,478,20,311,54,448,144,272,144,895,336,630,716,119,529,869,350,462,867,382,686,65,561,989,337,754,901,348,599,209,545,229,847,542,484,350,678,98,302,503,792,528,681,137,733,843,101,754,678,573,221,962,393,64,522,277,372,660,484,5,758,35,707,982,780,937,125,775,48,912,516,36,409,138,140,70,5,354,319,351,811,989,151,615,276,357,364,656,256,629,151,36,31,529,864,676,620,901,788,450,981,210,594,198,834,984,953,356,615",
    "758,189,509,679,514,366,352,0,16,948,324,345,508,350,958,671,191,112,473,406,520,758,86,333,694,407,370,771,674,426,505,434,727,207,989,692,965,677,716,680,907,936,958,437,991,17,26,520,62,551,559,991,725,857,542,434,948,381,285,492,768,785,292,660,971,74,410,654,894,494,772,171,721,50,30,345,332,991,807,406,79,962,361,385,671,639,703,192,930,605,515,398,40,15,193,759,461,212,870,888,208,809,105,879,722,693,840,129,647,159,759,668,390,542,938,186,921,931,876,323,98,161,612,366,943,400,117,683,948,673,784,636,61,936,794,170,167,509,986,13,375,779,478,393,292,552,119,960,292,334,973,949,25,417,447,821,398,782,166,274,576,392,929,16,621,938,983,853,129,116,868,436,86,47,806,981,197,160,320,323,721,619,998,763,152,4,591,731,795,633,419,666,491,146,905,454,658,97,210,591,875,959,158,278,218,106,749,847,63,667,580,959,648,234,62,275,765,225,714,273,324,532,228,819,588,784,99,727,574,839,948,234,389,326,594,396,467,750,366,713,4,265,870,415,442,696,720,732,153,578,286,636,400,68,607,755,853,301,494,359,193,11,664,49,972,444,933,753,543,631,801,419,164,442,419,916,736,57,504,204,868,692,404,680,681,544,553,215,835,276,581,167,624,183,953,552,245,760,305,58,657,230,43,759,770,252,321,126,333,939,937,281,864,383,720,204,137,198,413,911,21,306,600,683,155,219,280,959,67,655,926,355,280,663,80,404,474,647,558,732,377,821,462,46,327,46,1000,936,412,585,804,956,239,61,384,489,226,356,638,990,527,670,763,73,270,807,180,821,786,832,877,433,468,532,438,276,544,53,47,372,257,43,765,652,81,518,314,145,923,466,908,757,846,365,626,919,875,274,927,242,716,829,145,68,911,884,352,766,704,419,589,351,5,560,146,226,139,8,984,354,384,369,149,993,511,918,356,447,999,173,125,654,494,491,878,548,653,600,912,977,584,199,75,817,60,55,644,75,828,940,472,291,410,912,479,684,837,639,560,723,704,300,499,717,614,535,190,302,770,874,740,459,430,217,242,923,541,619,92,105,976,429,685,303,453,302,219,812,514,256,41,821,398,717,885,940,801,971,791,181,38,460,713,75,367,249,36,64,734,534,8,291,254,471,879,91,435,325,501,301,586,186,747,745,731,715,762,349,608,775,367,320,980,478,217,344,820,636,555,611,932,505,746,334,864,711,840,680,945,8,478,40,823,823,141,401,352,426,905,647,757,950,351,926,400,169,612,600,322,451,321,604,201,733,147,459,106,271,562,518,443,953,990,485,380,723,618,987,457,97,926,439,14,653,477,548,290,458,762,361,102,840,516,904,344,158,27,158,786,24,876,311,54,227,577,339,537,115,226,983,481,672,120,18,750,538,756,455,860,98,431,416,524,629,405,827,836,795,122,296,953,103,928,855,883,117,45,955,882,877,718,174,536,461,461,776,643,762,383,575,503,143,421,152,508,839,769,581,627,340,686,690,248,880,878,492,914,889,901,657,257,545,813,949,616,465,710,828,209,539,227,662,925,69,78,795,821,888,297,479,612,716,409,362,543,452,994,374,961,932,299,864,24,646,476,717,856,871,194,530,145,879,578,52,733,158,218,976,574,134,895,452,801,450,650,953,13,720,726,657,950,637,591,669,814,827,582,892,200,473,946,765,598,555,503,111,45,553,782,755,233,493,939,716,583,922,471,365,919,736,251,655,422,606,133,710,763,302,548,843,55,568,560,640,778,249,485,294,495,598,224,178,939,526,442,371,532,871,618,852,992,84,430,937,566,711,801,425,70,50,711,828,727,395,233,692,640,934,642,14,455,254,998,1,755,565,161,400,423,114,114,875,963,88,645,482,717,975,320,42,761,595,12,256,972,462,173,377,969,277,295,103,708,292,231,356,780,5,718,752,608,417,928,129,191,462,625,216,200,968,875,425,519,939,924,447,356,988,725,461,17,361,52,857,637,424,587,424,820,616,400,126,331,229,968,225,537,182,852,159,225,774,929,482,146,103,346,260,114,131,981,380,927,238,594,5,9,877,943,539,538,76,257,554,170,815,886,334,878,421,543,234,231,687,732,578,132,379,23,637,106,478,763,842,479,185,97,907,373,969,19,346,504,455,941,844,857,9,792,736,13,397,207,848,49,962,893,761,467,112,70,763,222,757,970,825,24,737,670,493,191,806,119,118,591,153,787,944,919,638,579,758,75,191,543,926,692,111,754,299",
    "908,382,781,555,846,390,21,16,0,495,554,294,295,629,687,230,555,311,18,358,529,427,183,32,412,279,915,63,722,811,624,517,195,57,654,14,815,500,464,742,842,641,923,117,654,955,690,491,962,786,155,695,926,374,383,406,957,187,738,930,945,404,533,218,244,681,924,640,849,880,78,395,920,72,272,622,583,278,571,10,211,329,173,393,793,509,839,152,287,549,202,741,557,931,904,155,418,527,881,546,216,551,185,918,485,259,73,953,402,701,729,299,354,681,251,359,99,179,420,502,390,842,589,332,80,446,572,19,331,830,440,585,514,601,150,845,943,520,438,947,309,441,43,680,725,146,201,407,364,388,855,747,933,882,213,828,769,181,777,96,443,927,478,364,290,121,831,447,54,76,45,195,375,94,57,589,889,119,143,613,371,209,57,89,326,539,207,631,658,639,993,433,715,489,956,860,812,165,634,688,682,306,567,447,804,698,943,273,255,287,147,195,163,166,594,671,903,873,660,948,695,589,81,54,718,408,135,697,926,581,527,309,884,887,512,715,381,960,181,236,39,358,579,427,377,734,814,621,370,279,218,107,442,525,533,416,212,222,225,349,213,352,213,784,397,824,381,38,642,764,616,286,278,43,471,258,732,57,651,810,559,307,586,80,625,815,379,492,101,148,593,329,630,646,343,509,647,892,134,194,988,370,770,145,941,910,526,70,937,608,253,730,227,983,527,889,952,220,890,762,86,788,359,82,499,954,120,843,213,731,440,107,909,448,105,167,971,462,604,806,910,349,365,911,363,539,768,415,102,508,501,509,624,812,288,262,73,464,930,953,798,736,960,740,835,891,268,438,611,174,520,824,452,884,179,255,894,19,553,936,616,104,673,92,234,454,653,442,939,476,470,249,130,772,156,414,695,314,152,662,621,132,970,341,732,562,314,607,845,84,235,645,375,135,284,242,533,184,334,573,70,659,922,577,634,401,917,341,552,404,197,67,713,260,556,943,617,164,336,723,740,579,975,13,488,468,882,55,957,676,962,686,734,491,108,990,355,658,204,597,733,88,994,188,879,432,637,23,527,699,195,214,941,756,230,726,559,17,113,670,354,685,345,20,78,248,12,455,171,23,770,940,972,368,53,785,928,700,666,684,803,738,861,263,3,893,121,402,932,644,444,352,652,525,215,90,757,863,604,762,915,527,478,960,288,649,899,269,961,876,862,100,417,548,423,462,298,834,146,935,674,536,660,640,843,880,180,466,994,894,12,388,51,106,569,90,572,66,622,101,335,319,528,878,751,825,756,957,821,500,192,755,582,653,834,552,272,661,312,470,488,109,516,770,281,224,746,734,822,298,854,620,471,312,524,830,339,453,510,86,936,575,882,18,572,901,252,131,135,146,616,445,725,318,562,571,801,724,916,513,758,766,894,541,793,542,433,208,479,441,256,947,998,357,184,771,130,663,498,536,855,230,680,931,594,105,617,359,169,649,80,20,285,548,725,454,985,321,248,40,661,717,626,203,709,679,367,492,964,544,852,961,608,115,242,783,664,917,134,698,672,173,629,359,506,410,86,648,22,560,793,857,260,703,849,295,308,739,534,559,298,623,816,200,793,514,228,687,745,575,547,772,980,640,105,778,623,946,247,967,406,206,370,568,80,790,262,773,330,839,508,660,484,209,615,649,202,330,118,847,346,913,649,780,750,454,705,780,998,761,392,96,468,305,384,698,777,635,548,783,382,713,672,75,73,597,599,951,499,649,928,252,127,872,913,713,304,195,235,708,176,495,552,809,188,428,171,804,703,346,613,160,124,98,222,934,211,423,177,423,250,523,750,186,537,209,209,175,10,997,640,328,917,698,909,565,837,908,967,83,800,125,901,958,632,998,959,632,45,673,814,924,838,324,994,837,758,915,236,970,350,420,309,790,497,496,291,830,948,226,136,903,924,464,66,249,132,882,320,215,351,191,265,690,28,363,463,671,973,836,330,595,127,270,968,972,718,188,214,425,585,437,249,164,304,867,500,212,69,839,938,689,883,837,411,608,866,638,464,297,72,375,203,869,838,115,343,492,122,14,815,258,606,127,39,868,750,36,308,603,933,296,878,683,909,928,555,662,809,410,121,478,514,791,436,918,397,907,156,422,988,279,705,775,980,820,673,505,679,351,31,667,103,362,311,27,906,601,928,266,228,148,260,199,11,219,609,973,647,418,155,998,535,751,757,991,418,474,396,202,326,726,45,190,658,609,912,157,875,272,354,480,128,192,279,878,552,249,407,823",
    "420,388,544,918,58,794,21,948,495,0,906,650,437,53,933,817,60,131,55,582,893,880,829,213,669,73,736,131,573,120,403,566,346,932,878,9,187,198,621,731,396,65,529,612,196,130,591,875,15,187,802,769,214,671,331,611,474,305,521,745,991,195,227,791,853,739,158,898,390,180,403,79,812,252,187,260,706,2,142,852,582,265,22,190,134,824,608,952,465,637,875,648,756,91,38,732,141,802,664,824,107,364,957,216,205,78,98,510,295,359,166,476,23,533,618,951,242,228,42,758,301,431,961,245,778,830,396,406,618,552,777,542,287,627,859,457,90,342,314,568,326,952,541,216,593,909,164,816,460,891,999,294,884,228,991,978,487,168,649,868,100,979,942,304,625,552,300,340,275,986,223,626,801,650,714,197,186,558,811,755,44,622,15,820,55,960,518,850,870,876,109,984,273,167,528,279,578,338,938,996,663,465,603,70,140,509,83,122,85,655,328,177,197,911,890,562,659,736,183,903,747,944,880,324,820,773,395,676,435,637,505,729,232,145,282,353,322,285,771,498,64,112,157,333,211,631,751,496,404,917,846,328,238,740,853,596,377,188,313,234,928,472,832,69,551,786,359,763,50,85,365,77,821,660,782,662,468,724,52,110,936,392,778,750,872,407,562,867,951,404,277,454,202,514,458,514,958,770,505,709,114,212,20,463,105,113,674,566,6,540,134,164,106,200,639,839,382,227,91,934,918,362,35,509,326,231,284,753,833,627,620,407,116,261,421,775,292,441,923,764,249,734,52,37,886,981,178,845,796,670,917,277,905,449,777,206,819,542,236,609,518,117,1,202,725,710,812,678,179,352,713,136,383,896,582,749,340,979,70,588,389,826,430,288,898,467,637,462,179,299,471,503,301,433,701,553,759,861,224,387,451,624,833,83,911,305,280,247,796,112,742,398,55,211,326,388,669,331,536,181,330,645,935,158,519,881,517,629,451,427,619,771,557,523,353,20,368,699,795,229,873,787,984,501,554,308,765,21,90,896,523,627,549,854,544,896,12,168,159,342,133,800,164,413,634,517,524,198,33,193,514,909,285,608,755,137,931,425,410,805,829,990,72,191,607,492,910,410,432,339,627,935,499,73,507,421,693,258,899,381,276,995,86,969,802,410,411,77,203,991,625,843,538,103,736,200,601,453,834,216,490,821,323,760,607,431,439,767,210,165,664,770,60,884,394,127,512,403,270,641,497,654,658,987,812,875,528,948,782,713,878,184,588,193,129,681,980,786,667,71,666,663,909,411,221,23,829,125,549,323,190,433,957,530,270,933,601,243,429,208,877,786,323,11,234,71,861,757,141,25,239,670,315,445,578,607,104,109,824,523,462,792,39,937,991,103,134,271,408,120,672,327,455,770,134,842,329,994,902,899,116,90,543,821,211,649,562,588,773,349,388,245,748,385,339,940,8,126,996,242,984,718,795,346,773,259,546,564,265,928,365,612,390,795,687,931,179,767,518,997,477,624,616,270,448,468,496,102,330,547,550,336,70,555,305,595,628,115,522,184,166,873,566,122,490,64,682,794,925,557,305,609,812,675,201,556,970,541,530,354,509,949,266,468,392,393,567,383,978,81,927,817,162,598,792,79,736,77,212,598,555,805,173,312,377,278,78,33,570,440,609,216,533,183,313,400,726,851,599,931,269,213,940,984,133,838,320,835,760,288,775,52,170,422,830,998,692,788,113,410,24,303,616,926,951,760,904,408,508,516,547,458,473,702,850,807,79,891,683,448,559,195,184,424,857,108,226,359,900,558,474,496,244,437,596,295,730,969,926,313,755,791,579,967,855,424,872,918,643,808,577,723,31,871,427,86,145,468,338,593,796,498,512,923,500,865,303,486,87,321,859,363,180,204,535,931,77,981,545,875,941,959,367,842,474,385,76,171,494,751,248,596,509,678,420,213,168,606,39,836,27,625,777,4,392,337,711,157,976,138,658,146,177,817,204,801,600,145,904,365,76,895,886,907,254,354,219,744,842,205,307,277,245,495,265,251,849,29,632,823,196,477,122,35,504,698,724,445,628,746,987,528,315,531,374,428,942,806,533,646,167,626,436,213,460,34,753,14,142,362,412,882,727,698,390,713,588,773,621,626,560,90,954,781,841,49,876,372,873,354,78,52,980,212,943,275,307,559,378,854,294,998,990,145,654,829,815,627,578,142,921,255,946,86,61,457,18,142,18,54,808,306,550,691,122,143,837,306,561,793,838,805,229,418,373,295,501,346,12,509",
    "610,460,43,259,260,546,261,324,554,906,0,354,773,953,227,795,438,900,794,598,528,234,86,872,523,742,913,352,841,518,392,982,264,47,325,816,178,620,786,202,218,529,921,751,224,543,129,72,529,848,167,509,840,471,851,544,945,328,797,300,43,899,845,307,420,634,460,995,662,779,939,880,64,655,936,602,814,436,545,729,248,617,917,301,305,250,368,491,354,760,119,984,940,302,476,19,145,863,675,876,400,435,56,906,480,543,342,408,906,639,211,571,731,354,484,964,466,734,815,262,678,553,703,270,455,894,302,697,499,126,746,585,828,929,102,782,766,478,295,512,484,542,210,58,727,292,704,596,371,828,238,415,730,183,660,372,459,530,857,541,450,302,879,922,274,836,203,712,257,866,581,518,280,188,924,223,425,619,149,349,797,664,629,295,188,39,786,204,897,506,84,563,739,84,710,906,691,848,873,689,547,480,769,287,418,185,62,9,606,582,451,450,475,717,851,373,989,311,731,911,300,882,402,442,805,102,764,89,91,847,808,110,311,977,927,702,260,517,71,656,487,158,191,280,797,416,64,286,23,26,751,700,179,956,955,732,322,865,592,89,131,242,656,763,26,494,927,990,85,958,282,810,146,859,375,634,189,387,557,56,324,285,579,696,111,693,55,214,439,339,329,789,270,472,485,373,67,4,98,868,871,926,70,757,489,240,180,719,210,32,484,403,29,679,450,658,80,891,54,654,653,725,385,85,119,952,38,817,466,633,222,661,480,560,181,364,52,213,503,431,117,206,295,637,815,756,216,965,396,748,843,896,953,608,7,311,690,903,664,116,340,956,819,17,26,846,453,259,905,635,185,368,174,505,573,423,922,923,425,339,481,426,487,26,374,839,319,795,63,663,21,433,777,702,414,112,381,396,791,10,780,563,851,835,224,150,552,634,864,437,954,164,398,629,624,388,530,579,497,945,746,770,922,945,364,507,932,164,869,183,839,943,467,271,948,753,334,585,97,238,555,486,813,567,603,219,116,700,81,797,552,261,428,759,522,373,608,847,401,199,675,723,134,382,672,915,721,762,488,497,103,69,325,282,550,907,718,851,888,424,482,735,637,656,43,281,529,839,90,670,442,842,65,520,320,513,663,396,949,767,15,366,975,497,685,160,766,906,475,725,906,948,145,128,281,19,434,451,809,35,439,684,251,242,84,503,513,511,728,502,794,618,915,205,675,682,134,82,307,695,390,385,848,126,265,341,492,686,760,525,118,32,35,990,752,752,690,13,746,626,492,320,526,666,551,158,72,20,631,751,43,417,60,954,53,106,133,261,812,379,808,600,452,899,75,474,766,606,792,879,499,599,627,991,450,83,236,868,375,52,19,495,503,458,400,841,679,147,416,610,890,648,28,474,394,51,926,197,56,960,330,228,556,994,211,800,573,260,853,513,520,746,711,548,552,964,395,132,444,154,203,567,907,724,602,150,442,336,278,813,709,766,238,895,704,728,335,867,393,99,942,926,591,929,344,523,601,363,284,636,445,850,798,516,465,159,259,551,656,849,903,467,175,883,697,274,306,415,728,344,806,398,342,750,525,24,147,834,174,439,238,478,142,983,25,864,757,810,295,91,113,812,506,526,282,817,237,607,367,984,399,973,458,931,500,355,119,801,781,571,800,623,681,54,818,365,675,20,981,762,557,244,362,998,915,243,509,381,237,662,818,814,156,923,657,900,533,32,46,190,617,775,959,627,376,440,289,173,646,976,201,856,567,104,40,914,983,394,939,358,665,474,512,946,936,313,897,450,881,979,117,506,954,861,94,395,893,280,764,231,276,637,655,275,789,636,585,570,294,174,692,54,25,298,54,905,895,926,348,189,611,442,995,338,133,900,990,133,261,770,834,318,217,435,235,185,571,798,510,469,926,77,100,214,518,657,476,484,39,39,172,880,209,135,762,351,375,856,300,833,588,591,563,882,661,74,875,788,86,753,760,749,549,657,207,400,129,268,58,512,281,953,539,603,697,982,271,78,908,673,214,78,727,962,605,789,421,430,856,663,139,334,548,278,401,840,641,195,596,242,530,557,196,480,301,528,438,492,381,352,737,160,374,154,921,531,412,878,387,444,129,307,921,414,131,509,360,626,436,947,928,717,749,468,259,366,427,458,908,297,67,492,983,411,6,276,275,818,268,916,845,821,164,599,798,484,124,467,240,778,232,867,289,135,410,958,52,975,168,642,60,578,705,41,560,54,774,940,501,457,773,153,171,405,94,105,113,50,632,738",
    "484,802,803,522,489,750,739,345,294,650,354,0,879,269,568,43,905,661,885,938,831,68,641,935,878,424,774,63,352,484,122,756,55,875,305,105,785,232,574,24,310,982,609,630,633,802,420,952,8,600,243,323,748,412,46,140,483,755,879,204,434,514,32,906,660,831,705,293,929,793,341,215,442,575,543,913,296,33,61,1,675,259,733,932,89,250,531,728,444,924,356,253,864,245,829,293,562,126,657,764,376,26,326,613,405,70,267,388,239,405,991,507,952,270,279,784,547,601,9,771,828,645,421,60,617,643,832,489,455,24,754,444,321,709,110,178,788,787,271,923,880,858,213,818,463,238,241,242,389,35,682,165,872,22,102,508,436,251,254,252,363,874,2,925,635,866,371,982,618,354,97,473,293,594,643,262,253,479,868,250,430,681,359,356,818,66,626,323,817,498,670,61,545,837,327,532,772,557,269,374,38,735,954,340,158,99,414,601,618,473,248,789,754,567,381,793,563,852,151,66,537,192,881,421,920,487,354,240,58,859,61,174,385,73,644,570,154,570,199,136,920,70,917,712,603,907,475,901,368,40,288,313,235,770,101,496,492,7,867,627,968,424,60,541,306,792,486,824,923,926,644,497,194,335,471,855,326,189,369,357,598,119,848,553,387,339,136,543,111,352,659,347,542,180,502,606,996,320,414,974,388,625,396,256,863,479,711,522,356,627,729,668,458,74,245,311,369,780,68,689,124,95,512,100,672,142,650,243,104,153,296,149,680,654,304,88,529,72,214,49,866,885,244,411,123,105,565,701,256,382,804,444,503,406,877,729,542,11,212,86,766,903,311,204,340,354,518,180,854,131,904,615,956,834,153,54,142,423,861,961,231,126,765,160,710,299,41,656,592,432,82,387,69,699,697,143,935,143,736,872,11,716,975,987,366,742,145,838,686,651,383,400,185,859,801,740,171,975,7,124,402,916,231,750,195,347,194,552,823,598,251,666,440,747,400,724,922,24,760,925,511,448,858,429,305,249,866,917,643,376,830,70,754,951,305,431,209,352,947,541,28,442,306,207,631,274,337,130,265,246,287,983,457,568,370,266,152,738,824,383,670,201,572,667,809,265,732,313,38,963,361,431,537,418,275,295,836,466,365,567,396,949,529,181,303,627,738,486,281,897,858,566,569,834,860,576,244,613,746,887,759,772,712,463,781,529,397,242,302,445,175,155,588,718,803,615,292,686,184,896,213,558,157,171,583,437,629,184,445,787,697,248,440,908,327,558,310,806,334,809,494,601,503,797,247,352,651,104,462,920,821,839,926,871,482,172,571,164,85,333,983,343,591,287,3,652,167,283,156,374,976,598,920,966,35,52,324,884,702,264,995,898,149,493,454,50,697,117,151,868,292,618,642,566,973,932,51,782,355,771,275,409,723,183,481,579,966,600,165,876,667,42,450,643,538,230,391,306,929,350,778,677,102,220,511,303,585,5,109,189,859,250,387,921,982,251,547,555,617,146,817,541,32,541,397,839,568,900,639,80,617,774,145,633,261,825,625,821,166,802,766,320,330,449,197,832,226,377,350,273,446,272,335,883,706,231,75,580,380,835,735,161,453,545,345,438,308,381,80,395,927,41,683,528,505,867,235,284,172,379,571,461,19,506,892,978,606,285,534,717,737,217,264,74,141,954,99,559,36,988,550,603,81,497,495,342,392,329,404,424,309,3,330,828,434,537,392,273,789,263,151,513,605,336,329,6,28,906,258,75,806,145,735,959,754,481,890,916,648,678,152,116,432,497,580,301,105,922,128,194,728,440,897,278,3,84,166,566,850,713,735,63,627,516,19,975,336,147,654,895,748,3,638,648,498,718,101,546,786,801,156,10,413,132,164,923,252,132,857,122,400,608,818,644,319,198,81,473,56,306,981,500,394,805,824,490,40,236,571,662,880,304,279,701,113,834,394,319,251,534,247,725,882,884,828,378,531,869,884,873,211,401,280,705,815,644,923,221,642,128,162,500,830,294,348,910,203,898,456,327,138,492,723,592,401,786,357,428,883,66,282,832,563,204,155,662,450,248,775,649,51,205,640,87,729,538,295,166,363,996,788,301,947,6,945,40,611,907,214,218,853,895,692,351,771,494,19,454,916,655,317,992,874,73,547,295,776,975,291,52,57,991,281,359,819,288,43,808,69,693,208,488,705,406,56,663,598,72,349,515,869,299,293,167,960,56,78,930,685,562,654,736,739,978,526,481,88,493,422,604,806,624,840,598,59,286,889,744,711,437",
    "195,266,544,153,486,18,777,508,295,437,773,879,0,171,471,291,787,106,859,102,784,760,146,473,131,720,555,110,194,684,590,158,346,780,667,911,951,198,340,54,504,549,491,280,220,347,268,486,9,299,597,386,352,938,893,767,400,262,637,281,979,87,554,476,847,195,19,3,745,892,732,862,901,991,990,135,976,17,242,402,856,4,260,205,751,300,780,861,242,36,508,58,527,966,612,88,946,542,398,439,223,254,44,520,883,533,793,287,467,423,118,227,758,547,842,612,906,609,671,520,52,952,506,658,506,736,465,73,739,56,52,258,590,651,441,968,711,451,283,906,890,172,416,632,677,774,433,108,677,398,220,984,888,47,459,716,683,93,360,919,697,416,372,851,248,911,958,927,471,818,648,640,460,869,473,588,539,484,126,75,351,749,262,897,271,671,168,21,927,892,512,908,721,508,346,279,119,263,7,105,905,122,454,331,189,635,672,219,199,360,362,368,711,736,232,33,812,149,232,900,183,916,943,257,953,108,750,326,454,652,382,618,136,870,184,920,194,204,774,47,642,203,113,199,514,789,31,26,708,46,854,492,577,582,701,357,398,928,755,604,890,723,30,404,154,321,978,583,520,677,503,325,244,18,482,447,694,887,529,930,618,981,790,196,925,714,441,208,772,568,673,629,299,960,54,528,520,413,911,314,543,205,128,227,287,795,386,817,471,779,151,867,632,514,667,356,151,152,320,631,943,801,687,314,692,872,406,647,143,726,114,183,905,195,554,923,626,181,558,300,429,354,760,809,272,102,948,52,325,171,426,986,253,884,52,651,725,460,707,654,316,922,996,342,585,843,895,8,291,654,58,99,536,353,44,999,944,470,197,564,525,137,683,450,148,161,608,698,61,758,209,726,580,698,668,37,820,215,276,651,161,914,587,155,479,670,998,468,167,486,40,974,795,968,514,310,426,439,872,339,781,979,282,376,438,168,75,439,314,507,184,117,90,153,956,392,710,193,713,359,76,713,275,946,214,615,806,995,342,454,80,419,980,771,985,972,855,710,948,213,896,515,692,364,527,415,155,801,586,939,529,88,612,731,246,649,829,917,937,302,998,954,290,304,473,492,733,921,274,280,796,541,402,456,110,256,983,678,392,730,542,569,286,795,596,11,713,601,402,589,23,810,440,126,430,20,858,964,379,2,471,786,759,810,104,222,127,521,897,426,222,366,49,177,177,907,649,653,881,894,930,890,128,196,55,870,167,669,509,715,482,891,536,934,483,749,466,521,745,358,538,815,396,147,807,230,875,996,189,557,149,112,359,521,701,214,503,807,865,752,386,397,293,268,339,351,891,977,536,253,610,620,748,379,66,570,141,53,681,855,506,410,781,464,463,979,729,968,492,669,538,190,97,262,598,657,916,147,669,256,318,105,214,174,381,730,542,115,406,813,47,270,677,964,549,884,412,750,109,136,21,896,200,46,535,366,258,858,980,761,407,951,425,622,532,50,64,974,735,311,448,227,920,685,194,467,603,838,672,776,937,989,62,850,262,788,364,290,783,528,473,497,432,853,146,909,192,965,357,529,630,68,436,958,242,54,873,539,740,864,938,301,983,294,118,223,174,333,213,338,419,732,897,416,890,754,151,616,581,561,259,653,951,485,469,195,814,112,512,280,94,470,748,787,839,117,404,231,260,465,128,517,464,783,577,739,675,748,729,364,625,568,575,473,671,924,279,950,482,725,747,700,732,1,511,369,346,444,695,742,365,121,332,343,245,609,918,317,355,222,572,437,188,745,371,737,506,32,54,73,780,712,560,532,510,773,695,695,391,891,325,714,929,520,219,296,231,919,943,860,999,258,119,88,845,319,63,235,23,648,965,903,155,799,844,513,388,187,724,928,953,555,453,604,205,397,864,356,718,444,102,396,154,855,947,907,725,980,23,955,200,875,95,449,806,349,129,921,428,881,201,718,895,656,742,29,921,30,4,724,250,425,192,926,488,301,306,145,527,132,952,584,433,735,348,497,743,258,797,644,52,94,398,708,364,646,258,15,791,33,461,487,94,376,402,68,678,199,26,560,946,370,722,527,133,821,546,213,957,708,726,74,441,833,296,69,896,595,33,880,3,625,933,433,430,876,921,391,252,588,244,160,152,358,759,727,755,850,676,4,250,921,413,590,278,862,529,983,315,457,644,403,974,667,413,189,503,755,875,903,779,213,598,981,11,937,993,523,679,479,753,19,248,616,758,112,687,903,380,224,259,908,843,909,500,311,890,142,346,999",
    "474,415,354,491,66,671,16,350,629,53,953,269,171,0,370,567,309,812,670,925,317,695,660,548,423,717,659,520,184,281,506,575,658,880,711,541,122,654,833,38,195,850,264,354,217,256,39,184,807,183,471,522,274,155,496,90,624,931,714,44,118,891,255,22,446,569,488,980,299,928,550,209,256,703,953,165,935,415,478,737,450,369,683,385,492,491,299,906,31,551,157,957,995,608,806,928,415,414,584,674,887,262,776,668,872,272,736,764,135,252,749,734,961,373,784,490,33,703,725,50,816,160,609,225,428,208,796,654,70,363,657,29,630,903,90,529,456,379,560,32,748,629,629,795,890,766,557,208,680,979,556,753,93,468,788,141,686,944,305,220,724,198,262,741,86,238,844,624,344,283,813,60,483,650,935,200,613,321,737,389,657,292,152,794,800,60,22,392,784,791,25,517,115,576,414,831,724,381,143,143,498,710,458,409,623,637,56,732,322,616,476,823,651,177,177,963,532,708,58,356,615,888,120,815,324,718,333,44,996,809,145,441,106,40,451,445,450,204,698,356,912,827,438,785,49,268,600,613,827,643,647,267,626,616,561,860,764,38,459,285,35,225,932,474,511,864,284,321,385,469,244,767,950,123,635,871,405,57,596,930,517,297,949,388,28,235,526,902,135,336,105,708,119,844,440,306,102,914,858,564,738,819,886,318,218,173,317,512,504,111,840,105,424,424,340,847,438,403,74,124,609,54,846,348,389,134,686,913,262,21,141,764,28,221,963,203,50,354,646,924,388,955,933,773,856,796,209,839,188,679,662,768,161,566,554,647,267,532,122,992,263,149,387,670,554,98,736,617,550,693,1000,768,159,673,292,163,117,157,893,890,558,844,217,152,464,967,55,64,537,840,224,791,530,709,18,274,298,227,504,799,684,824,204,114,723,863,320,934,53,455,505,933,643,159,360,731,251,782,210,339,645,921,179,314,363,959,641,231,604,131,973,433,927,515,741,195,680,902,547,98,616,616,622,750,982,754,830,347,307,907,272,5,844,213,812,192,958,659,136,665,156,522,97,160,951,395,629,62,957,688,78,299,668,803,426,934,877,743,344,918,895,12,409,851,271,295,650,790,368,859,246,106,15,694,233,331,590,181,694,329,414,959,801,128,170,390,227,191,157,208,708,670,742,219,755,156,252,403,367,968,97,18,933,516,203,475,112,211,518,484,331,633,920,226,766,981,422,821,494,408,776,625,24,845,930,893,992,331,267,642,104,444,89,46,416,589,278,230,113,281,107,962,560,815,936,460,313,990,157,398,678,602,814,881,427,237,741,270,825,790,74,222,12,29,505,311,333,503,347,230,480,188,347,631,44,521,785,376,694,912,179,624,802,145,808,991,427,645,400,702,111,132,856,149,449,658,151,542,523,243,382,144,975,109,919,467,435,826,730,142,796,991,32,413,552,785,121,952,932,600,147,945,380,128,334,866,16,356,25,366,431,587,677,11,138,258,135,354,388,167,186,929,55,408,256,194,869,16,734,12,131,794,153,315,454,770,441,791,347,6,958,627,293,17,672,771,159,788,111,342,820,704,353,995,512,82,226,686,79,511,76,183,115,514,844,964,230,914,592,671,196,794,297,911,208,496,483,695,399,6,127,936,767,509,459,894,986,87,578,674,972,675,589,526,831,542,226,561,954,499,269,969,163,133,937,939,488,37,929,259,598,775,991,797,321,247,332,302,927,556,960,264,756,769,472,270,248,216,402,197,460,468,144,404,77,729,585,36,546,154,983,58,329,522,833,822,184,255,433,301,573,341,731,379,997,865,153,161,43,573,924,692,17,737,563,943,830,587,55,334,762,122,419,574,719,479,422,194,395,369,591,510,564,542,66,18,62,904,989,475,495,47,885,878,650,296,490,435,620,841,921,297,155,590,925,870,788,161,549,951,204,545,205,199,95,833,448,773,327,973,853,399,179,252,271,181,589,671,295,600,152,838,929,369,942,140,231,550,846,582,279,830,942,927,767,138,775,731,616,459,752,168,978,489,891,601,525,991,862,112,577,958,338,19,132,554,279,710,565,339,286,429,350,486,594,73,25,987,161,99,680,773,809,901,116,93,356,155,936,168,191,450,467,490,32,609,968,247,891,331,897,939,414,548,918,936,159,70,312,721,883,62,997,28,187,817,778,28,694,819,677,115,499,498,969,38,709,409,327,974,977,568,402,983,478,510,68,490,140,441,314,698,339,19,152,925,76,681,694,119,340,936,567,700,346,353,938,415,732,899",
    "254,70,937,650,420,300,839,958,687,933,227,568,471,370,0,417,780,847,487,857,293,818,622,123,96,762,744,354,512,654,425,493,763,12,396,149,785,594,312,273,841,941,480,554,262,675,654,65,908,547,131,166,475,995,10,529,720,343,818,8,865,416,795,542,26,225,147,801,708,703,382,440,35,368,104,866,595,94,458,87,637,278,164,524,254,56,414,615,330,50,373,616,659,986,328,671,395,473,566,432,162,751,708,574,481,253,616,700,446,405,557,269,708,132,895,182,994,518,499,509,225,150,547,619,820,705,884,841,302,64,47,376,399,926,732,112,225,176,258,702,647,849,288,149,199,493,908,733,864,180,164,932,363,9,638,989,563,182,88,858,205,171,21,76,518,828,597,754,817,721,957,126,819,14,573,342,789,880,88,572,621,310,501,485,382,890,360,890,774,460,506,905,286,538,744,307,855,148,235,378,944,1,554,368,718,69,96,452,398,354,191,385,742,801,431,971,906,415,345,560,932,630,189,160,393,933,371,856,909,844,423,998,301,201,319,39,14,815,505,893,233,930,42,306,780,455,635,671,116,194,90,292,11,418,360,293,966,373,489,668,285,671,95,87,420,807,702,590,591,36,377,705,380,105,479,291,745,326,698,649,794,94,923,798,491,571,786,498,357,129,412,268,816,118,890,642,197,471,557,835,712,386,144,198,266,452,915,822,348,437,343,172,388,94,319,56,869,596,183,333,734,948,47,133,333,914,393,496,894,901,147,755,232,433,258,77,772,142,970,621,307,621,903,917,606,585,913,78,516,365,918,323,623,901,566,953,59,252,599,719,473,21,919,603,781,454,173,660,431,924,737,425,428,33,60,264,699,919,369,763,863,479,163,73,322,2,176,939,292,526,890,449,596,241,922,314,195,576,857,608,80,300,880,65,203,902,949,389,567,438,268,946,672,968,366,739,986,160,245,339,215,43,347,579,318,85,86,951,627,479,628,302,938,320,643,859,995,705,319,903,79,72,214,260,770,616,285,53,328,586,148,573,76,600,438,195,819,685,852,231,608,585,293,351,888,129,948,469,477,650,898,983,178,465,947,918,768,422,274,489,283,610,876,567,960,254,230,324,176,738,720,472,523,221,609,45,958,411,265,96,573,82,493,323,461,942,589,453,269,158,463,987,353,581,448,850,215,832,968,425,507,900,227,126,9,488,335,839,724,907,442,618,311,106,321,849,156,307,666,315,434,966,203,313,835,948,182,395,213,237,773,29,876,195,86,826,335,679,233,344,808,105,482,603,189,487,510,380,658,387,774,581,252,192,531,838,762,913,128,642,36,61,494,38,176,278,763,610,342,572,664,450,226,67,419,514,377,364,212,820,530,428,632,408,16,826,788,766,788,280,362,754,691,790,115,921,508,841,665,632,771,541,746,729,590,757,63,180,546,734,233,552,886,200,91,87,785,657,786,151,141,107,685,297,353,357,159,158,702,42,715,917,832,59,11,977,227,879,797,9,567,514,412,979,872,579,630,201,903,576,577,245,53,764,828,902,88,631,525,376,504,888,435,612,155,147,401,993,473,157,655,280,769,828,297,931,819,147,123,723,693,781,501,35,240,783,638,103,343,632,901,548,874,784,298,874,39,483,118,625,875,209,762,182,504,185,327,232,742,702,373,982,78,871,815,48,809,303,293,187,894,446,463,364,395,791,222,877,786,614,348,7,772,229,78,409,538,924,89,600,101,773,834,891,714,123,955,852,209,946,392,151,357,927,847,290,718,777,714,154,628,885,360,705,656,72,26,186,107,581,430,722,773,253,527,798,733,798,253,327,947,595,4,886,288,525,488,357,280,137,829,280,511,436,749,218,697,401,525,19,732,265,261,398,160,828,382,929,255,88,356,153,785,402,488,137,781,30,908,898,504,931,648,19,194,587,318,379,463,416,586,130,797,958,824,664,782,243,161,238,325,824,653,595,180,608,937,121,278,785,358,170,436,487,968,898,623,637,833,726,423,743,995,283,517,905,6,730,328,777,290,708,925,189,493,875,287,371,740,904,100,25,77,458,31,211,694,881,220,215,335,229,822,23,914,968,13,814,218,303,976,578,449,136,685,148,789,974,534,697,963,782,218,317,931,159,800,606,593,416,553,604,46,266,676,615,956,822,879,10,351,188,125,869,910,619,946,777,900,261,620,973,202,798,280,742,940,912,48,769,467,284,157,52,856,437,287,253,601,902,950,568,105,707,976,93,172,471,11,477,810,282,609,956,414,349,545,824,765,501",
    "969,10,252,419,805,198,512,671,230,817,795,43,291,567,417,0,669,407,187,539,62,456,304,977,900,967,804,163,336,640,427,233,478,804,284,994,373,134,567,694,759,761,88,361,216,196,886,340,312,440,30,811,652,51,758,529,324,113,509,431,992,697,80,283,626,159,963,640,734,917,633,332,458,82,736,793,446,626,233,69,310,933,325,244,45,196,69,374,264,613,748,66,299,484,672,515,531,836,420,883,621,429,297,175,171,239,897,489,48,262,455,486,490,358,491,917,621,68,359,325,958,922,438,864,471,318,771,14,724,71,932,259,356,383,296,389,651,275,178,549,624,102,291,911,218,471,841,909,306,185,587,609,154,477,303,657,544,821,302,270,303,546,429,549,515,122,199,867,262,296,754,34,214,761,210,141,746,334,404,32,807,695,306,393,134,310,137,511,494,786,913,467,697,889,413,105,49,631,639,959,939,628,169,732,275,248,427,593,256,150,117,205,323,794,59,600,32,445,436,272,209,371,361,625,559,62,193,825,813,83,955,814,429,679,570,439,716,83,493,864,879,374,159,429,730,78,782,852,374,237,901,587,499,225,998,418,113,211,156,283,224,672,408,665,343,858,85,890,339,371,674,685,664,738,867,168,819,811,819,817,882,715,941,429,410,653,937,428,188,822,501,96,964,18,315,523,954,520,502,992,290,659,91,113,885,624,310,512,751,453,873,930,748,432,905,132,849,674,454,874,793,50,454,780,958,343,247,313,567,850,609,254,788,711,811,438,183,294,950,273,254,976,671,937,89,567,348,251,360,823,160,509,618,345,147,788,767,506,486,833,365,206,471,271,831,279,297,116,335,507,778,646,887,364,977,784,836,656,129,558,169,931,228,133,632,785,648,519,330,413,286,932,351,476,455,481,608,152,674,53,567,217,375,929,469,709,335,187,460,52,500,895,602,94,252,165,947,205,187,787,463,994,443,71,760,65,523,667,99,818,974,661,774,29,45,711,372,518,700,104,741,120,558,335,835,795,476,438,863,968,418,596,620,669,157,331,781,552,628,693,639,350,713,400,112,318,692,181,95,30,620,46,124,840,961,822,321,576,598,797,292,816,757,995,525,728,907,334,722,565,18,895,851,479,225,102,368,492,405,536,839,180,677,462,2,494,456,835,454,190,697,473,499,987,704,618,420,879,746,517,86,426,169,861,106,57,412,535,479,928,227,802,604,479,222,758,618,20,385,207,402,862,935,250,621,260,655,196,941,112,815,79,973,423,852,983,101,62,49,336,613,505,617,348,339,469,458,602,340,538,551,351,450,417,108,863,226,67,386,654,78,181,231,587,892,95,604,387,518,277,467,788,297,7,126,65,339,430,420,553,219,722,125,239,953,777,569,394,324,784,23,116,738,505,463,857,663,202,665,70,843,314,103,308,67,913,96,326,854,224,542,859,351,914,725,144,44,919,60,467,384,475,122,722,690,676,805,149,491,613,56,281,49,419,962,152,866,508,937,226,226,943,745,85,239,422,607,799,400,724,628,969,509,761,278,940,591,482,571,362,908,292,860,577,532,685,844,409,482,860,143,128,649,239,795,536,76,20,447,182,367,39,144,135,150,544,40,411,766,20,687,867,920,155,763,843,472,498,518,382,471,984,876,446,32,556,752,39,247,742,931,51,616,870,371,140,900,490,456,753,896,874,504,445,969,529,623,351,773,635,906,910,521,365,356,389,748,268,585,119,907,598,879,821,316,835,275,625,484,251,450,982,547,253,425,538,733,775,857,98,12,11,688,482,43,583,935,601,797,113,41,416,209,25,804,569,553,726,341,378,690,231,168,568,683,906,121,716,168,708,933,669,338,236,216,679,913,311,307,651,499,901,787,251,197,337,179,465,196,127,906,654,850,426,710,33,556,174,795,326,536,748,925,835,61,343,9,526,377,305,356,404,774,260,39,121,707,487,572,524,689,289,220,44,789,186,946,114,382,657,442,611,45,519,755,587,2,265,28,969,453,430,379,442,516,330,652,752,60,402,719,75,173,270,218,221,931,769,886,647,196,838,211,867,370,318,460,593,874,253,924,360,119,417,946,660,446,484,41,893,401,187,757,294,968,252,581,759,997,495,21,887,366,252,86,256,672,967,195,93,523,400,7,763,869,740,15,227,741,790,5,759,122,168,139,602,131,241,390,286,112,676,560,539,693,885,809,652,326,924,995,699,540,176,520,4,774,92,945,139,221,825,104,518,534,59,404,795,700,100,803,313,202,319,137,796,278,7,696,303",
    "172,137,687,377,805,220,876,191,555,60,438,905,787,309,780,669,0,676,334,535,110,499,368,115,167,93,409,442,485,233,729,548,200,876,622,439,257,847,921,91,385,914,469,58,706,858,307,636,277,311,413,744,988,697,463,635,601,20,450,955,448,809,805,954,160,645,63,139,986,676,831,656,920,827,776,465,599,199,149,553,204,956,86,616,803,55,462,526,701,599,883,75,572,571,325,666,847,750,26,949,367,617,394,710,982,566,351,51,430,142,705,697,255,436,82,706,910,302,52,414,824,800,383,176,913,630,806,788,267,26,608,719,532,365,687,980,278,191,139,92,852,34,768,784,627,691,447,74,46,598,475,183,584,339,356,226,73,709,953,913,369,331,737,283,21,749,500,570,703,271,59,350,252,457,638,43,989,427,63,355,4,959,680,858,500,813,281,67,235,220,452,676,32,838,494,375,622,135,585,695,691,642,220,671,755,236,832,486,206,10,776,648,169,636,65,532,116,110,549,370,315,775,99,423,812,864,837,582,44,641,812,895,853,725,276,463,212,295,64,892,770,693,848,362,535,674,31,193,605,18,604,928,56,626,433,382,179,458,978,771,671,749,109,205,93,232,159,598,110,594,946,969,862,566,224,416,385,108,118,752,353,225,557,522,644,985,171,649,495,555,148,566,163,552,643,513,918,666,503,778,486,59,46,714,636,214,122,674,647,684,625,398,34,74,248,555,429,577,64,650,671,435,583,252,809,321,931,473,881,905,668,984,673,348,917,419,322,258,872,781,695,520,351,158,24,649,587,172,110,252,938,812,137,790,77,499,26,72,631,587,67,26,498,988,772,250,858,282,594,854,769,6,896,140,296,221,669,276,262,509,66,358,670,409,287,352,482,85,328,705,581,906,680,526,835,782,611,777,594,410,245,675,884,783,723,540,989,954,20,481,303,625,132,704,669,805,695,592,950,914,649,394,559,456,689,189,414,558,490,601,424,500,890,574,787,667,220,960,613,681,992,219,594,736,275,889,782,111,513,612,587,75,509,178,542,836,193,569,923,664,93,977,978,439,471,389,221,301,623,159,423,780,892,408,656,234,454,79,553,182,827,290,790,736,740,896,329,929,462,104,230,358,485,190,164,810,583,943,399,454,365,105,605,96,81,746,828,962,63,61,342,837,586,807,190,765,575,97,725,257,6,150,771,923,602,350,810,873,81,755,33,447,575,33,422,915,215,331,340,503,207,705,568,307,922,537,349,556,739,366,199,642,366,166,547,941,176,338,558,718,53,407,250,925,411,920,674,411,169,633,900,320,953,777,204,382,478,870,742,907,84,154,436,516,229,525,262,171,539,457,947,73,992,980,442,186,871,218,939,80,64,193,554,572,690,417,762,572,996,427,109,755,18,454,227,897,626,720,392,231,512,340,132,703,350,254,832,994,708,841,734,387,456,467,631,783,960,911,835,213,302,382,496,983,238,182,303,380,734,349,254,965,224,237,712,333,39,86,205,616,689,686,759,401,837,417,468,183,138,844,56,336,354,963,915,880,802,672,879,84,342,438,351,217,942,273,315,128,611,416,45,763,220,745,871,13,933,436,464,457,870,176,265,545,909,457,762,448,146,785,768,533,954,551,11,660,257,421,208,607,406,527,815,100,632,964,52,828,162,171,234,213,130,853,720,811,519,496,62,151,841,880,375,881,578,545,244,911,723,714,940,659,217,923,877,951,248,481,978,408,276,404,892,873,520,500,220,182,63,931,212,507,818,745,614,298,792,616,755,650,655,430,514,388,797,834,703,127,949,248,438,631,966,630,133,305,998,437,614,602,270,470,190,831,538,302,53,385,191,718,833,537,936,56,501,143,540,971,45,925,625,131,79,608,806,862,869,8,30,319,605,631,613,605,893,652,658,292,888,941,386,506,782,206,208,965,497,663,816,411,939,687,228,396,628,845,855,416,881,196,801,835,895,798,393,843,794,940,210,952,116,453,566,510,80,216,722,378,383,567,642,41,464,252,154,221,177,375,905,72,284,758,765,727,591,635,229,59,747,729,716,825,237,555,873,723,962,365,273,890,535,261,9,613,516,597,626,974,950,343,953,880,231,600,36,528,284,785,581,481,783,68,600,876,349,736,742,338,119,235,654,715,94,447,732,413,501,421,75,103,473,23,954,94,759,880,522,562,578,874,610,232,255,977,249,416,690,522,800,955,789,974,104,552,499,352,189,676,625,742,465,941,522,417,298,714,112,508,417,523,684,301,33,895,529,570,443,314,937,758",
    "463,344,373,449,115,770,59,112,311,131,900,661,106,812,847,407,676,0,40,23,996,231,587,141,284,596,718,189,487,11,264,597,427,921,151,214,109,947,617,842,393,142,840,240,51,446,53,749,512,341,949,115,55,309,951,611,910,245,274,827,124,507,726,978,615,539,989,338,870,1,883,789,220,534,777,976,672,700,106,711,306,160,359,687,576,905,598,694,529,391,686,503,55,934,24,774,331,257,348,383,315,442,239,828,341,228,829,684,218,38,943,866,564,80,686,965,898,820,698,4,113,969,362,489,124,794,684,376,808,848,527,102,939,359,344,226,613,390,189,327,413,267,669,63,543,301,230,373,437,17,381,273,792,586,494,596,997,68,843,946,55,174,651,469,430,871,815,958,754,652,816,936,572,906,641,438,940,267,449,284,739,370,539,909,249,310,235,493,166,574,972,223,418,517,407,721,677,76,534,959,726,319,11,963,27,20,694,15,137,457,66,239,552,777,226,687,799,490,316,72,102,762,762,16,11,639,957,36,326,603,107,428,845,444,836,578,744,769,237,688,796,70,910,357,685,665,519,584,16,574,502,38,975,705,388,729,701,76,538,956,56,207,950,623,31,584,751,272,818,480,390,257,661,934,137,990,647,710,786,629,702,650,685,460,982,231,600,617,994,915,672,468,206,222,748,787,30,146,664,249,610,727,197,610,501,682,327,862,285,694,497,177,359,192,336,568,63,293,73,776,317,127,6,260,476,86,503,831,271,203,679,271,356,340,756,652,323,849,49,539,976,420,244,243,847,36,477,578,211,681,279,34,303,268,165,867,503,318,75,296,866,245,490,768,948,718,541,112,619,276,988,324,596,822,397,586,381,438,67,776,289,111,329,895,621,944,125,324,370,208,149,966,437,910,113,345,63,259,757,86,357,325,149,884,459,802,915,917,191,113,208,488,579,420,545,219,237,325,390,4,28,597,484,268,562,740,571,176,631,377,683,280,988,889,589,718,338,350,753,457,942,343,322,257,564,20,282,239,124,132,854,215,617,730,98,61,831,264,702,416,678,830,635,441,927,38,809,540,66,72,236,558,893,515,978,280,746,663,999,39,629,427,306,848,669,223,323,842,920,200,735,751,438,435,788,290,714,782,725,167,718,890,116,664,303,521,848,809,334,87,579,245,2,243,306,466,455,116,189,281,989,455,791,37,968,551,392,28,34,857,853,663,936,943,410,480,407,296,118,985,68,42,497,97,430,780,207,142,574,680,518,382,873,826,157,915,51,582,79,432,84,182,444,530,614,208,823,908,547,116,922,556,811,866,321,130,867,937,667,311,218,88,18,880,154,335,718,720,988,429,26,677,525,852,421,493,425,713,923,568,946,776,602,832,73,659,425,254,712,563,111,482,264,429,200,835,815,874,389,619,125,732,902,704,344,935,736,895,59,258,216,48,619,588,653,159,621,909,474,376,66,369,967,253,717,587,582,774,382,154,261,897,383,639,777,451,497,329,780,515,313,218,127,269,238,508,661,13,101,146,208,962,513,566,770,64,314,836,24,242,552,778,487,22,288,773,870,339,916,56,861,207,234,994,367,473,960,562,393,705,816,133,774,347,762,381,350,170,126,647,114,986,774,826,42,856,505,368,73,33,899,101,790,551,769,38,771,890,432,513,368,587,190,15,531,971,729,121,769,307,964,473,790,746,751,484,128,566,761,637,251,734,798,929,729,348,231,842,926,261,440,145,354,500,551,933,376,235,593,560,60,281,808,676,741,169,701,869,274,120,37,958,970,195,1000,130,68,503,72,185,288,448,103,558,448,374,987,402,637,698,566,702,848,664,121,314,756,116,121,779,768,898,821,443,653,884,768,58,40,629,804,741,427,555,755,953,242,484,237,29,119,598,340,402,346,914,663,616,794,320,70,316,941,875,304,954,870,573,555,551,429,47,724,777,477,93,188,26,244,138,246,379,251,173,197,80,186,560,123,732,948,710,589,446,476,28,524,270,951,462,331,872,22,503,668,536,751,588,545,184,782,995,223,606,630,574,208,577,807,902,742,823,25,826,920,445,927,381,970,907,411,568,412,489,191,27,888,55,485,492,275,799,159,678,227,626,255,159,979,364,280,378,682,346,26,394,682,699,399,932,961,199,51,136,251,645,466,690,188,849,285,452,81,303,303,457,379,397,887,129,751,383,288,651,600,814,680,916,515,334,632,220,959,304,989,658,236,740,371,303,668,47,770,83,697,787,46,735,558,492,267,368,548,544,755,250,44,258,478,899",
    "130,215,142,295,978,609,263,473,18,55,794,885,859,670,487,187,334,40,0,961,451,126,268,474,960,830,869,969,534,5,142,574,324,240,649,478,379,779,978,288,265,48,517,758,431,775,101,841,59,565,609,600,483,562,234,912,588,224,640,155,46,208,705,846,944,668,919,230,828,211,293,675,498,622,456,910,498,713,451,566,903,793,250,285,666,433,188,760,297,268,318,403,663,268,45,149,105,440,737,112,30,34,631,609,949,101,804,103,980,490,400,985,795,955,738,549,602,81,464,431,306,133,576,436,121,85,1000,477,79,861,968,54,330,872,148,631,773,247,854,632,999,466,589,948,295,626,680,535,14,267,211,999,598,403,838,616,282,315,903,701,734,430,311,6,40,969,578,562,77,279,84,595,910,152,289,681,342,269,390,14,386,356,822,356,314,690,752,244,920,199,6,186,808,652,82,808,587,370,105,14,935,1,695,414,904,912,820,662,780,29,348,175,724,245,246,913,731,902,406,635,370,657,138,795,520,629,133,864,237,731,806,973,478,166,932,15,569,160,803,867,207,650,268,340,942,455,325,869,3,35,23,211,983,45,108,421,254,540,947,513,27,487,75,635,335,198,317,426,692,305,556,259,827,64,367,789,280,473,895,85,264,616,786,207,162,227,445,971,409,769,834,971,438,492,387,574,573,939,850,923,956,814,562,254,902,773,280,489,392,547,349,574,739,42,738,126,940,601,425,112,244,862,733,905,315,942,371,961,881,976,908,415,33,265,542,60,148,427,825,27,441,441,25,579,272,366,28,650,613,770,956,745,353,303,182,874,824,565,184,139,148,84,45,327,479,81,550,542,340,210,878,857,72,454,491,649,580,293,805,24,69,447,152,629,771,586,353,309,421,846,380,39,82,287,249,337,433,36,589,71,940,54,626,712,646,59,401,499,879,893,101,434,46,695,471,700,892,140,501,586,783,982,973,579,852,98,81,971,50,457,708,450,337,621,301,823,717,387,276,904,260,990,447,206,878,601,181,67,392,420,237,683,27,591,242,469,630,347,749,650,793,925,298,923,476,953,666,549,104,629,185,153,355,701,403,234,829,203,235,319,385,527,599,301,980,911,81,319,741,516,940,520,758,257,316,813,569,722,314,843,650,896,722,22,237,768,34,401,475,511,519,644,443,803,359,783,698,824,266,659,752,313,159,505,545,333,286,359,467,568,13,201,581,520,811,100,152,797,321,884,830,556,257,302,546,736,619,894,229,388,105,594,4,840,736,105,10,207,158,283,238,872,8,111,454,476,201,110,299,913,395,645,329,580,774,987,96,200,459,665,526,639,101,935,891,925,714,617,665,820,388,90,185,179,111,440,137,894,405,87,5,888,630,135,813,389,990,98,318,358,351,446,834,515,820,395,281,100,25,147,349,606,865,835,885,158,895,453,707,658,120,427,102,476,244,145,808,55,477,829,640,747,489,720,504,350,154,681,564,143,881,130,67,239,111,149,537,554,191,952,248,383,926,617,69,939,235,663,853,543,491,791,738,757,903,58,86,591,709,281,454,332,660,947,296,17,926,182,898,144,445,314,939,95,907,368,563,816,949,501,467,83,328,592,952,109,269,5,767,37,839,482,353,830,722,891,847,961,801,496,634,531,252,553,591,746,734,87,355,260,989,829,725,669,420,896,948,93,572,527,742,953,672,625,463,353,432,629,494,555,354,631,217,888,303,738,935,85,377,766,114,966,575,987,152,629,58,31,951,76,470,44,439,587,789,210,827,471,946,997,228,312,12,699,631,578,583,210,49,624,408,297,274,302,884,941,338,749,93,809,60,509,593,824,938,389,774,744,208,113,196,357,404,530,120,110,640,514,720,173,466,325,81,330,868,345,441,468,862,205,448,107,317,485,816,31,744,648,475,769,569,953,947,916,659,462,61,219,764,890,769,997,927,946,233,866,178,192,325,521,32,573,352,171,259,890,761,747,134,810,808,998,663,254,253,14,475,609,148,709,566,364,932,209,768,668,183,869,755,598,770,218,434,172,752,329,898,123,795,531,255,235,634,454,374,737,322,285,75,415,158,616,90,767,97,263,552,870,984,520,76,963,785,740,137,249,484,89,91,170,299,174,696,298,336,22,197,690,624,347,462,583,667,713,268,108,425,820,626,391,927,55,854,483,105,631,270,60,861,625,247,222,239,272,712,857,769,101,273,753,838,638,683,139,815,806,81,362,109,377,319,296,830,846,53,514,779,353,386,707,798,648,992,935,59,847,425,998,201,69",
    "335,456,628,813,230,610,709,406,358,582,598,938,102,925,857,539,535,23,961,0,564,541,18,691,671,992,980,200,957,281,761,131,916,499,866,525,447,991,102,120,116,344,457,957,671,801,139,906,787,718,463,818,851,42,296,9,4,335,558,419,182,611,5,695,103,414,217,81,545,573,987,608,545,100,559,274,257,452,785,833,387,649,451,627,727,694,916,473,544,67,930,143,551,906,222,144,697,334,527,215,677,180,299,982,127,552,921,74,629,258,512,679,852,100,210,700,642,531,578,329,172,337,815,560,18,175,173,602,66,193,368,223,775,357,305,854,682,582,549,693,373,326,337,706,361,85,130,39,581,475,889,658,193,73,80,97,446,415,668,284,208,499,376,498,299,111,476,811,985,132,844,395,73,166,162,661,414,234,473,509,500,98,256,952,626,722,544,862,238,666,842,448,591,822,228,692,835,867,440,485,352,407,319,955,808,426,552,722,389,252,153,299,914,703,564,48,939,849,414,573,193,871,940,468,404,567,675,711,464,42,905,689,580,213,387,684,704,814,223,366,284,118,126,721,741,663,19,32,475,4,193,288,265,37,444,108,696,496,664,960,531,80,420,642,507,711,837,52,15,664,798,220,379,419,846,269,463,25,785,620,293,112,807,637,913,600,820,998,126,760,172,100,318,874,152,667,380,796,628,94,2,968,870,940,270,36,594,595,685,321,732,387,601,750,10,410,432,792,493,43,51,654,700,57,255,775,735,355,200,527,490,123,766,537,109,504,947,537,775,572,527,193,962,295,794,1,396,118,371,114,856,463,70,707,533,213,964,350,480,877,284,421,485,657,180,953,889,249,772,28,361,139,31,494,86,739,838,182,684,460,166,550,841,386,440,750,517,242,716,532,605,532,825,732,399,897,261,660,805,283,238,32,710,603,914,629,698,252,635,852,137,352,812,100,153,431,882,246,920,407,127,4,752,995,210,52,831,492,514,213,777,265,848,874,445,503,944,837,139,350,39,764,721,410,100,184,760,948,417,32,105,936,276,696,482,292,325,93,623,278,289,401,34,932,152,822,406,720,785,98,476,542,830,807,146,430,907,207,495,629,331,877,737,577,184,214,119,277,717,845,197,513,892,527,20,410,938,793,992,262,931,344,805,778,701,373,377,837,362,468,704,59,136,190,843,623,580,747,412,426,127,725,339,922,900,127,435,957,681,354,604,166,286,724,206,173,169,759,735,694,199,604,829,468,174,289,854,691,903,562,116,241,284,430,271,119,206,64,116,937,663,149,612,297,273,733,132,468,710,80,408,313,768,384,383,360,536,586,393,51,65,280,341,691,733,290,78,775,556,797,144,632,973,558,287,703,276,686,588,71,402,836,374,416,332,741,797,650,736,152,986,757,50,857,325,227,254,59,820,502,594,278,793,702,357,689,959,138,231,585,754,622,141,126,459,112,620,474,9,770,149,237,289,836,116,171,757,111,272,479,695,7,550,860,253,96,744,875,816,12,249,545,823,994,570,767,916,754,291,31,805,537,766,543,61,622,366,820,265,821,566,988,460,84,859,345,117,890,236,166,939,943,515,688,860,341,742,318,71,675,11,843,184,611,707,256,148,906,882,459,186,408,342,254,405,72,423,732,286,234,505,920,668,324,959,410,851,471,629,352,873,895,277,112,52,805,200,937,568,523,945,941,932,612,88,599,273,398,280,265,785,678,158,509,186,762,565,457,961,107,822,19,773,62,756,270,601,71,117,220,756,333,791,715,433,860,994,143,741,320,625,567,574,55,133,861,485,631,292,685,612,725,476,860,810,238,279,799,870,420,652,678,119,323,360,400,792,899,372,925,721,496,820,173,352,605,513,804,463,83,459,282,596,35,961,343,395,356,439,286,341,563,338,65,381,971,797,886,834,388,224,923,763,152,854,424,248,888,804,174,148,103,548,155,88,371,424,563,334,992,908,948,421,350,474,862,405,924,29,312,78,830,91,74,292,105,690,887,907,212,802,519,99,392,633,124,632,702,190,206,920,421,718,24,365,665,489,734,858,653,921,868,796,469,602,735,178,595,41,76,895,311,531,617,537,737,877,823,379,804,994,642,374,593,445,470,920,356,843,110,647,753,381,31,566,578,995,352,396,772,241,879,362,464,590,828,182,291,143,8,866,611,620,643,793,455,631,569,284,638,124,296,381,607,440,765,937,969,606,655,653,934,547,934,976,425,715,104,803,933,838,562,506,802,515,512,798,349,396,9,485,43,934,861,212,148,11,943,838,148",
    "704,530,327,575,125,544,919,520,529,893,528,831,784,317,293,62,110,996,451,564,0,818,652,964,387,318,521,261,165,483,239,417,828,995,62,168,179,42,224,832,281,356,885,459,93,317,449,469,118,909,631,847,359,805,622,712,925,802,585,207,301,445,390,678,550,584,783,111,805,122,112,989,711,106,168,343,853,231,7,109,501,733,883,707,20,726,39,590,303,516,457,206,368,152,612,452,229,709,470,799,404,973,124,446,955,614,524,435,668,159,125,840,410,238,114,416,680,41,630,967,818,424,474,34,894,637,605,184,616,844,769,503,763,630,380,23,671,17,708,154,427,964,446,436,56,554,746,643,333,448,739,558,711,219,235,163,554,574,166,95,895,323,257,323,140,264,48,38,335,333,148,608,777,620,336,460,952,111,13,262,131,952,753,416,386,865,459,176,926,793,482,762,885,147,945,86,376,730,732,290,88,866,181,887,163,771,574,903,146,461,369,23,535,362,18,398,648,854,501,436,102,12,683,878,655,324,242,660,72,119,418,212,132,783,266,505,225,670,928,923,217,784,914,516,712,322,859,241,515,312,731,939,51,219,103,57,882,461,591,370,74,225,832,337,542,304,829,832,357,568,406,494,343,206,323,882,945,564,220,93,6,946,890,448,102,545,569,463,137,29,859,960,467,744,84,259,197,385,671,899,754,251,59,63,676,153,761,224,530,997,694,957,661,446,218,143,15,421,541,640,306,439,39,63,299,865,147,696,917,56,705,64,451,425,115,384,932,751,211,251,364,15,874,138,439,552,488,346,451,666,10,126,341,793,422,994,782,34,248,726,895,403,963,857,630,75,250,118,325,948,503,863,525,619,717,642,632,480,9,685,609,194,973,865,313,270,570,671,352,651,148,885,996,794,771,992,471,458,1000,273,784,982,153,933,356,460,483,509,61,969,992,315,455,260,997,245,368,792,772,50,53,381,288,243,347,157,767,88,316,639,341,445,608,363,933,267,474,42,84,271,142,766,715,160,759,557,90,236,31,662,587,896,571,647,653,620,746,728,673,865,853,320,435,144,404,987,606,591,322,985,74,387,939,660,84,252,665,329,479,684,394,966,887,605,298,197,995,408,118,931,894,293,65,706,99,577,355,79,293,420,334,883,200,478,663,469,426,838,928,57,460,408,403,748,715,545,504,928,192,597,886,251,211,101,277,492,552,689,951,537,567,632,690,213,675,76,9,169,886,274,147,656,891,616,664,267,653,628,209,656,910,701,618,126,561,466,650,141,910,235,548,512,707,329,597,724,591,157,445,470,648,505,820,454,796,277,332,172,134,905,896,906,747,130,141,852,329,685,567,600,524,726,678,743,404,830,856,201,401,356,546,127,483,55,898,529,166,98,493,305,438,741,697,642,631,728,294,861,684,190,688,735,517,850,457,879,614,570,631,197,974,618,656,536,870,96,200,422,355,759,936,358,741,460,348,520,947,474,371,60,291,60,809,822,12,450,485,872,100,147,414,315,823,591,571,391,846,264,556,952,693,847,663,171,576,825,111,908,587,49,696,710,250,303,396,40,10,933,92,405,865,31,860,673,552,471,726,619,730,263,242,862,640,25,181,737,397,970,523,155,562,814,736,18,597,210,443,473,873,960,62,246,3,986,749,66,552,351,802,515,580,556,593,767,678,268,834,860,70,466,267,523,233,760,456,652,567,264,477,608,772,522,506,359,954,63,216,103,460,335,784,440,390,525,609,734,337,213,37,393,691,438,954,309,294,957,279,714,886,892,9,911,423,489,773,399,648,940,959,129,401,660,755,101,675,875,35,865,50,127,356,400,376,627,383,142,262,782,112,316,36,148,319,266,461,32,2,235,128,896,102,493,609,611,396,437,23,978,884,587,494,71,176,748,618,787,555,440,720,765,594,47,65,469,69,363,19,65,986,56,562,276,153,132,201,932,660,843,232,134,654,308,458,899,808,931,734,609,841,927,87,804,522,281,60,719,263,54,54,789,667,869,986,572,347,642,393,987,368,400,243,788,177,204,492,574,693,846,196,534,839,524,781,282,632,431,100,825,988,928,171,173,841,69,963,552,649,203,792,416,581,285,476,714,724,721,543,342,15,430,254,232,886,449,269,765,591,554,246,541,351,252,167,781,737,587,62,487,328,908,772,813,174,709,796,248,49,216,161,101,127,508,102,694,98,530,562,1000,88,413,780,120,309,119,727,447,523,861,140,229,568,412,449,525,904,313,219,650,157,426,416,60,676,299,313,317,400,54,888,774",
    "507,796,167,548,277,818,213,758,427,880,234,68,760,695,818,456,499,231,126,541,818,0,253,24,452,95,64,826,727,77,988,67,57,877,89,355,338,517,755,736,371,820,301,912,686,86,206,212,1,443,810,940,799,698,861,924,335,630,434,265,90,699,838,363,740,377,333,222,86,171,794,984,907,268,979,16,26,111,298,878,277,858,826,659,272,260,606,627,532,654,119,843,918,462,552,310,954,895,890,622,179,331,414,32,683,795,420,210,161,62,833,199,351,334,92,657,106,126,249,429,978,814,474,243,653,979,263,732,864,214,950,763,61,276,848,868,499,396,871,87,15,366,261,986,992,650,515,2,397,983,215,159,491,289,799,952,668,60,676,676,566,918,86,951,934,716,990,924,530,124,358,928,739,599,903,259,842,515,932,947,918,333,279,153,918,394,37,590,598,622,648,440,791,620,963,244,582,319,536,912,419,73,73,53,50,140,490,319,836,328,843,948,441,685,479,100,797,614,294,137,108,551,411,678,37,51,808,828,19,424,178,638,438,332,149,13,665,3,195,341,472,993,371,12,636,621,985,962,316,226,67,381,510,554,881,75,432,219,443,575,559,892,880,425,374,49,602,24,923,444,39,970,803,54,340,746,187,292,102,418,228,334,962,968,161,308,532,659,323,208,942,342,223,991,389,329,957,554,279,596,664,741,502,924,115,378,745,98,827,996,101,984,715,456,974,452,902,862,267,214,44,637,422,334,296,980,238,467,290,974,362,48,93,861,158,562,223,443,87,8,969,946,29,147,449,396,146,584,740,449,562,383,940,506,583,362,837,81,998,472,418,220,725,491,982,506,288,586,546,798,622,180,132,741,453,857,684,436,214,309,774,34,349,945,265,95,235,216,447,777,709,757,665,788,420,820,47,960,87,163,299,948,654,264,884,225,529,154,97,880,129,543,842,911,886,498,721,572,584,523,411,226,441,645,489,219,409,80,519,904,597,486,789,569,502,558,395,605,413,935,914,814,413,443,263,107,810,726,401,701,386,504,103,281,230,803,994,9,457,973,572,462,854,197,925,338,777,366,624,299,708,523,754,986,346,908,264,308,124,293,418,839,427,602,819,391,810,814,550,72,76,829,69,311,31,66,298,973,440,391,233,157,336,778,712,450,229,728,922,360,584,107,425,45,243,254,905,322,499,583,61,578,979,660,28,65,230,600,825,864,347,937,378,754,475,130,76,148,341,385,734,306,312,7,966,73,245,607,959,143,335,436,931,885,532,295,774,634,218,358,387,407,298,161,306,831,386,128,163,471,525,559,730,179,274,196,531,787,859,307,357,164,406,964,408,43,433,146,706,160,752,76,815,707,965,141,247,352,118,203,364,424,151,709,888,267,429,157,911,944,810,722,530,277,660,23,756,697,357,456,121,222,964,431,503,518,651,699,779,531,588,550,560,114,414,201,562,813,602,978,273,877,967,903,770,491,579,350,880,406,991,568,158,141,327,226,517,806,584,825,624,504,630,866,846,347,824,633,946,837,725,391,38,712,21,443,872,988,14,702,745,309,975,6,275,706,165,461,644,560,204,34,260,719,53,480,686,904,266,979,607,601,898,808,121,721,539,685,380,533,285,947,521,400,913,816,502,176,251,342,449,121,160,488,494,126,142,598,420,614,302,228,869,555,613,712,531,333,938,631,967,989,416,700,352,193,620,401,775,954,167,63,3,572,222,471,468,968,719,352,895,859,733,323,530,981,543,671,420,904,182,196,381,711,152,580,406,149,731,308,71,206,995,956,808,645,526,828,141,754,902,775,853,976,314,915,660,355,350,805,507,200,507,45,362,470,254,261,507,265,475,246,879,306,104,424,186,240,278,648,692,804,932,568,3,811,216,137,950,957,397,43,313,902,367,537,310,717,867,387,102,148,638,454,968,220,606,752,6,144,601,392,546,631,852,790,426,95,772,433,84,700,435,516,521,909,170,786,995,892,437,480,516,516,863,965,660,450,131,771,281,339,360,381,327,290,321,571,635,331,846,186,398,98,134,398,806,901,302,326,922,371,177,510,718,347,235,322,874,775,928,284,366,436,622,998,763,257,305,384,640,329,504,75,306,69,466,719,323,401,657,386,40,173,11,767,787,64,856,476,391,540,692,699,20,313,273,219,212,47,932,586,14,309,643,367,945,853,582,572,745,866,104,532,265,554,200,3,86,246,443,432,257,597,209,55,451,316,935,971,693,16,745,623,62,357,476,133,689,824,227,858,549,415,53,366,208,865,341,947",
    "358,664,769,550,219,439,354,86,183,829,86,641,146,660,622,304,368,587,268,18,652,253,0,873,213,93,216,408,976,798,427,620,229,638,875,168,304,548,176,848,393,716,954,364,180,31,116,597,499,229,688,965,829,632,71,116,674,382,836,308,277,187,821,18,426,800,223,405,95,286,638,630,998,569,967,425,319,599,101,383,380,246,138,664,836,680,607,124,624,19,911,747,889,747,704,317,309,578,799,893,894,585,961,592,173,370,116,513,706,498,523,688,103,128,686,789,343,46,735,217,305,918,288,22,68,526,533,934,201,466,52,814,48,463,97,340,976,652,122,182,203,535,808,108,19,356,898,23,769,409,399,616,160,32,764,311,244,595,84,554,436,483,95,209,909,473,713,662,326,509,370,814,427,285,808,288,695,177,771,849,512,961,512,499,803,834,395,481,421,25,297,433,486,417,202,606,223,189,662,310,670,459,242,382,273,883,412,132,35,773,880,962,339,54,810,852,368,395,654,756,643,745,668,113,29,129,382,424,858,327,110,912,592,511,65,77,645,292,855,71,431,759,156,604,537,557,909,379,949,444,113,117,960,45,412,406,156,899,873,657,179,933,804,389,484,126,726,619,789,565,177,903,808,631,988,940,326,844,314,623,114,228,538,491,528,772,39,853,280,336,412,587,722,869,300,70,718,141,152,935,716,634,270,568,477,411,114,314,20,261,561,903,27,65,751,393,735,572,419,742,6,459,245,147,72,51,538,737,674,471,474,742,943,372,17,58,611,142,434,94,996,801,267,775,739,345,528,380,628,853,429,192,291,428,804,561,797,825,497,493,1,414,626,873,221,686,168,11,349,73,988,86,144,336,627,149,701,143,352,684,373,835,523,826,582,816,981,518,548,906,822,330,350,879,962,778,390,868,386,326,188,496,653,803,352,38,886,295,430,155,826,594,125,477,457,844,58,905,368,4,594,862,783,386,969,829,420,295,115,490,22,507,602,267,975,245,190,680,260,154,733,754,385,447,31,376,337,941,164,970,893,714,880,498,424,219,9,3,89,583,57,932,746,422,584,70,926,544,319,265,498,326,449,507,291,854,591,170,475,269,184,332,945,930,430,650,637,301,562,898,669,697,560,28,841,790,909,836,664,697,149,965,751,439,286,223,125,669,870,674,711,976,507,32,339,68,753,938,450,856,33,118,398,228,348,470,676,423,812,567,134,460,857,130,818,230,476,79,982,437,679,185,900,751,96,460,185,84,916,529,346,645,215,830,999,839,131,90,389,929,211,534,568,685,473,48,891,652,412,144,371,340,384,759,312,415,921,516,519,705,294,935,486,733,268,731,198,55,670,270,165,849,357,211,708,710,39,584,174,529,855,990,665,746,629,843,279,728,249,131,935,566,104,518,609,27,674,675,315,927,325,8,209,346,848,301,624,388,87,480,798,487,49,225,170,324,610,921,469,550,306,511,370,798,518,659,195,752,670,763,144,884,764,427,101,113,507,598,421,486,917,535,740,337,290,452,132,198,973,952,16,265,351,308,22,706,327,49,787,560,494,530,348,120,322,961,269,437,372,307,334,132,797,650,483,169,383,654,496,620,492,118,702,4,457,50,529,374,169,710,888,763,527,978,126,548,252,680,922,100,619,36,645,728,472,468,303,907,175,258,568,258,673,888,120,282,341,627,168,981,7,14,820,596,569,104,281,402,486,745,49,787,363,18,526,696,896,732,6,66,194,639,16,135,997,20,882,713,438,770,281,447,866,34,459,159,796,499,448,354,566,103,986,925,765,423,85,604,561,869,950,226,302,214,552,948,465,662,160,733,435,460,638,123,335,919,733,880,21,483,187,653,38,33,338,436,336,603,555,711,983,574,786,596,39,325,688,714,159,565,779,451,644,360,940,678,342,662,255,532,407,366,580,724,166,340,677,237,202,920,193,917,748,955,767,876,336,642,638,451,847,480,348,313,643,153,616,132,361,421,680,131,285,27,640,342,777,479,20,506,491,47,180,197,690,732,966,255,94,71,22,863,767,269,961,686,415,498,81,660,719,154,586,967,120,260,162,550,493,388,852,721,870,208,128,56,64,274,160,306,962,181,881,851,892,923,662,956,725,849,521,449,798,353,984,775,504,319,499,582,735,211,214,147,769,452,513,883,975,290,153,414,393,556,621,760,708,152,392,201,384,613,883,674,231,945,549,860,263,828,716,470,634,242,93,845,767,75,320,931,572,647,779,564,303,10,245,904,801,291,723,723,881,696,138,632,795,664,490,127",
    "852,686,538,571,921,158,765,333,32,213,872,935,473,548,123,977,115,141,474,691,964,24,873,0,8,930,962,128,274,555,24,852,567,553,386,709,481,436,99,42,651,318,186,953,242,868,322,603,543,476,937,625,980,128,910,71,368,12,513,915,810,455,175,3,589,602,871,700,319,567,456,140,54,679,382,793,213,730,283,920,779,726,107,369,208,116,113,877,36,63,136,148,977,282,753,58,490,62,174,879,995,452,956,528,23,778,609,878,237,651,935,523,846,732,84,937,118,925,814,228,708,664,236,756,535,434,794,929,188,760,676,505,949,981,609,19,147,647,6,2,568,709,661,704,54,775,311,963,388,678,497,446,71,675,468,462,233,29,798,779,555,662,992,172,122,388,677,157,318,482,881,164,395,114,98,131,921,93,823,385,789,82,631,112,82,320,959,237,653,41,151,685,240,260,169,823,62,171,504,950,750,577,281,887,713,454,760,113,141,72,774,525,400,15,344,845,779,755,564,36,162,224,805,381,34,873,417,500,482,831,537,226,159,489,219,211,732,328,651,275,976,432,472,972,812,483,217,171,78,440,452,939,483,679,563,544,981,381,809,543,455,305,935,766,596,417,48,97,336,381,834,729,820,333,304,352,445,444,963,145,873,557,27,583,843,142,817,457,577,406,648,230,927,74,395,558,264,490,650,814,270,745,453,584,911,973,510,612,270,278,646,26,391,83,194,310,299,164,541,328,277,228,688,665,145,746,991,984,629,65,2,123,973,465,916,141,534,376,825,296,978,102,151,264,224,627,538,151,169,783,754,731,63,423,816,616,963,90,137,660,926,864,880,921,448,990,895,449,703,640,907,155,247,549,118,15,156,338,818,49,85,165,314,695,13,785,294,734,579,908,461,736,307,792,509,567,241,563,798,176,841,32,949,567,643,87,133,811,412,330,525,137,51,459,631,72,37,800,213,707,88,9,968,338,931,610,825,864,69,283,837,852,966,48,238,562,898,597,170,337,176,750,749,86,969,54,190,765,333,29,140,194,802,274,811,687,601,804,403,686,112,509,554,764,914,372,842,654,824,427,625,808,231,168,792,158,875,661,720,976,822,942,816,84,901,609,883,274,111,661,778,222,674,328,499,811,618,703,420,295,632,263,770,148,99,702,681,995,343,66,125,131,263,228,148,299,239,303,396,792,523,842,385,418,317,93,724,122,869,8,244,960,367,870,755,146,727,867,587,835,303,288,782,750,964,881,654,379,187,646,792,66,290,933,270,559,293,19,232,353,796,554,170,784,238,248,108,228,498,337,609,561,604,176,639,268,583,479,671,203,764,883,690,606,247,555,499,216,508,77,81,198,184,304,339,856,275,981,403,470,250,869,161,989,597,583,30,587,306,484,503,607,837,197,691,201,192,893,247,728,420,15,594,457,678,445,206,503,360,989,695,127,403,11,522,595,268,296,976,145,925,364,323,753,805,614,138,576,778,897,621,796,791,617,834,475,710,291,364,632,56,846,518,300,559,421,736,786,918,589,389,591,714,825,471,477,304,448,276,291,32,980,667,485,377,884,40,896,359,528,173,813,863,20,88,333,966,588,174,506,425,737,349,200,79,728,823,582,666,464,97,982,665,655,237,656,53,943,390,893,141,911,652,681,910,822,387,959,983,278,129,694,70,903,169,849,217,418,665,824,759,65,316,262,605,541,765,876,919,583,692,171,139,110,574,792,100,992,431,787,528,321,122,731,643,138,393,130,112,185,644,997,213,225,70,877,882,172,942,934,288,446,848,314,409,169,754,894,152,860,577,771,874,936,653,276,203,638,152,515,799,359,134,254,20,83,823,29,744,68,444,575,442,25,706,515,126,46,489,881,636,14,951,497,918,391,52,940,383,248,742,203,828,72,32,949,174,961,739,809,154,107,490,815,509,969,74,485,810,831,146,744,538,263,34,332,62,826,456,643,675,894,462,764,984,188,482,304,244,317,438,348,41,171,665,969,106,950,497,837,548,776,322,61,746,802,352,152,843,306,793,161,109,353,407,380,791,541,382,775,504,787,367,950,128,990,209,857,427,649,330,392,655,213,636,51,212,605,337,490,393,837,52,307,353,110,187,617,668,570,69,977,818,717,68,883,516,934,109,490,444,592,790,42,310,149,5,692,752,644,612,34,831,516,959,4,948,621,300,559,845,41,744,411,593,314,470,428,77,137,4,203,587,574,613,5,592,591,815,330,478,889,900,231,136,995,668,843,366,931,664,51,467,952,259,852,559,243,17,83",
    "356,149,459,135,253,84,858,694,412,669,523,878,131,423,96,900,167,284,960,671,387,452,213,8,0,135,511,388,733,119,454,770,527,977,384,286,345,139,774,3,896,861,342,433,219,528,849,703,763,327,234,107,613,928,57,591,91,323,727,210,537,641,54,493,515,412,568,605,121,540,805,771,506,967,376,534,779,19,8,495,936,27,229,799,819,946,537,504,789,534,434,818,323,600,182,193,719,736,367,835,176,517,84,236,831,126,565,864,898,985,975,186,91,450,315,995,974,872,463,308,855,342,277,978,960,916,283,517,545,291,856,770,160,811,270,899,145,116,410,246,271,696,231,914,436,140,805,679,173,582,29,752,740,130,624,667,115,69,303,349,284,384,126,456,103,371,936,329,666,348,667,346,848,341,194,702,168,29,316,395,869,855,391,87,92,690,874,361,395,182,810,423,345,389,604,306,727,668,367,900,215,240,561,30,661,119,906,202,830,862,140,681,790,155,263,22,715,792,177,732,551,995,604,640,771,232,766,649,596,519,26,948,255,430,80,960,514,462,88,753,597,557,985,61,60,184,783,787,861,768,820,732,91,471,868,790,220,395,550,580,625,11,229,740,251,439,745,200,514,197,993,280,880,859,749,532,433,763,844,410,978,814,686,329,611,663,5,57,794,946,351,13,704,290,732,718,496,906,200,850,739,759,294,16,357,37,544,56,88,44,165,120,204,315,881,333,577,663,840,574,941,564,626,76,40,338,251,501,410,200,510,348,923,437,939,335,672,379,563,764,857,880,333,905,985,62,971,214,530,717,558,764,955,940,735,337,205,157,218,594,254,264,501,781,168,893,866,40,89,218,624,239,8,802,824,688,329,157,250,762,789,919,304,919,373,326,845,422,864,699,946,743,841,422,109,454,953,397,48,791,507,766,85,402,262,535,412,31,684,982,673,78,211,747,396,835,216,882,734,419,699,384,210,673,584,803,205,110,506,419,679,253,635,378,995,983,831,986,674,68,130,311,437,172,266,117,569,848,29,912,235,107,968,266,17,420,108,286,410,995,510,260,59,912,629,209,943,654,154,52,351,411,660,642,739,892,10,390,819,950,899,374,841,140,303,867,625,776,350,165,393,431,915,840,351,424,951,739,504,522,364,977,108,337,92,80,819,639,455,413,218,132,13,178,815,739,375,222,36,247,612,77,461,459,911,4,487,793,994,900,677,503,399,296,255,806,411,991,371,578,768,436,444,457,592,427,446,462,991,666,920,509,57,595,823,487,308,453,201,176,27,962,981,265,223,598,249,949,436,759,498,687,749,51,626,564,236,400,802,577,910,762,318,115,786,401,410,538,951,734,53,439,61,532,667,623,651,635,385,680,128,34,806,148,243,714,788,634,274,265,983,358,716,920,922,286,982,699,593,978,655,245,919,338,270,997,186,719,170,593,500,701,155,846,177,883,559,525,60,706,852,120,330,999,576,938,56,459,822,817,165,798,116,718,51,735,168,712,903,668,910,621,184,692,714,624,66,746,648,409,108,285,472,470,730,368,195,679,15,295,95,820,90,68,112,246,731,598,769,969,979,807,346,820,886,630,358,905,463,948,586,816,523,122,876,235,68,23,784,900,479,592,690,240,512,335,84,222,70,847,333,932,359,508,786,982,463,380,168,1000,658,418,241,507,577,112,343,613,827,19,37,222,735,199,106,17,928,910,754,496,257,638,328,461,681,569,49,31,828,438,395,484,641,306,97,115,951,937,422,97,429,148,368,576,256,992,258,448,66,102,100,864,696,336,78,755,986,251,328,279,646,658,811,20,294,742,67,786,35,265,75,926,927,854,338,930,45,373,16,645,934,330,796,250,306,788,313,161,252,925,463,390,942,214,7,928,5,653,77,644,394,352,443,669,793,217,100,476,574,354,615,492,987,667,385,912,778,183,24,219,836,480,291,931,190,319,679,121,856,763,68,682,923,435,164,35,628,76,805,797,899,104,566,646,922,402,776,735,559,724,92,558,162,312,82,888,886,523,591,129,863,712,986,412,383,333,557,966,242,14,378,135,767,483,612,372,827,224,967,151,459,675,319,342,988,434,857,569,531,983,270,103,401,740,332,17,707,675,569,589,501,854,667,803,118,355,806,404,415,841,265,815,155,946,692,457,543,276,414,859,743,364,554,874,623,614,553,308,59,343,539,294,412,101,976,387,238,354,94,494,13,204,468,599,84,629,133,734,560,684,39,730,563,97,218,512,525,915,52,172,689,511,394,472,79,412,556,623,78,684",
    "444,440,857,708,211,27,223,407,279,73,742,424,720,717,762,967,93,596,830,992,318,95,93,930,135,0,392,530,608,328,271,341,90,140,472,212,167,781,718,147,197,993,337,764,233,377,70,102,425,306,970,951,945,636,924,752,167,867,901,265,581,739,891,25,734,46,598,521,348,95,142,667,357,879,30,609,946,53,263,766,314,539,625,877,111,59,637,929,336,844,653,503,732,866,832,657,829,413,245,973,781,739,187,50,40,589,140,390,967,787,800,551,817,906,388,381,582,245,543,981,2,372,612,111,950,837,566,659,23,679,344,654,850,491,870,237,203,850,107,327,586,731,452,550,810,451,167,72,907,497,670,26,554,396,308,720,307,199,654,721,418,409,826,962,787,729,702,906,600,679,643,976,756,804,197,361,504,450,758,133,998,168,881,753,17,798,608,441,296,783,735,702,991,787,460,720,392,433,678,824,238,91,982,872,56,832,729,903,445,113,486,578,287,474,338,133,933,400,619,809,694,670,818,30,184,136,955,632,996,141,434,292,69,542,565,883,925,46,207,663,611,381,394,875,119,517,327,416,460,766,296,287,571,461,265,655,752,716,377,244,938,404,47,952,460,338,771,787,815,733,453,111,895,484,523,582,692,390,219,526,830,227,893,881,246,236,167,68,566,194,687,866,439,767,373,513,85,303,665,58,885,796,209,188,54,839,363,554,585,238,769,424,271,195,687,854,255,490,906,524,669,865,231,988,607,930,36,384,247,965,283,422,542,124,88,406,255,926,407,586,87,906,390,971,359,313,318,155,102,517,98,238,888,284,628,212,860,206,461,598,320,147,383,90,516,660,2,934,640,238,315,187,586,603,642,511,85,794,58,958,520,936,712,345,763,234,345,798,194,447,194,323,920,932,988,26,122,923,856,128,908,793,394,593,536,591,182,682,562,769,886,953,798,464,492,840,344,838,450,499,475,874,874,231,285,874,90,359,560,382,907,266,918,93,70,854,406,70,277,635,294,313,363,313,846,943,782,4,9,250,699,235,357,954,275,797,5,933,542,117,687,790,854,895,686,482,794,835,999,120,18,160,274,855,119,84,519,800,573,207,481,68,789,927,670,741,803,47,279,216,493,222,238,140,447,447,995,200,253,188,964,296,598,924,849,303,846,943,845,701,542,670,765,545,716,807,434,707,247,326,772,49,679,785,893,685,261,373,960,156,325,215,958,101,385,530,164,236,696,178,634,12,466,918,606,280,887,148,189,416,142,418,748,619,638,907,911,366,288,658,528,939,243,450,766,23,793,838,862,322,454,516,90,960,743,797,841,407,370,468,9,176,160,568,809,752,471,968,735,158,368,695,703,570,420,957,512,854,819,167,894,215,527,374,337,283,539,52,133,715,821,661,384,187,914,386,484,738,995,869,166,604,662,483,416,198,971,719,311,972,166,428,115,931,431,339,55,839,933,191,378,693,673,67,25,618,81,673,297,243,678,272,330,958,617,289,912,775,550,526,187,915,106,983,604,768,750,633,136,389,887,301,689,586,705,317,729,656,611,845,454,294,12,393,562,117,90,258,392,96,114,403,222,3,428,621,353,150,496,940,59,730,576,45,981,546,332,110,471,998,623,187,487,296,26,683,280,474,983,125,455,566,966,371,502,835,19,706,454,835,588,201,324,586,504,173,426,579,620,481,969,526,65,958,14,582,145,282,343,744,570,299,529,989,449,709,245,807,665,279,830,450,319,46,428,840,695,710,496,445,129,343,749,926,31,484,676,432,961,205,547,314,171,109,189,766,969,851,952,492,373,843,163,493,370,341,461,210,862,332,941,613,200,409,973,902,51,541,100,147,307,10,494,651,825,399,93,124,450,417,165,866,108,190,189,808,394,23,399,883,87,883,189,432,74,703,634,512,156,799,1,1,131,725,697,738,984,166,24,520,562,21,535,878,40,86,554,255,233,932,478,758,448,21,152,665,734,509,499,963,373,626,801,74,688,836,185,803,185,884,788,88,17,710,111,92,518,172,798,28,474,983,458,207,890,541,252,319,710,151,140,522,601,241,74,426,295,371,474,674,130,691,29,180,156,875,778,247,771,358,569,742,638,112,511,979,819,569,869,719,344,531,783,752,430,843,360,89,852,475,223,907,521,133,77,229,548,279,961,680,612,221,782,715,904,7,620,850,639,244,136,448,186,747,885,560,414,906,228,667,510,228,197,563,650,833,672,333,763,548,872,400,699,948,917,61,814,222,49,558,944,67,664,857,71,722,894,213,9,438",
    "396,863,820,85,368,534,556,370,915,736,913,774,555,659,744,804,409,718,869,980,521,64,216,962,511,392,0,900,77,745,968,52,97,341,18,418,53,305,301,718,621,641,566,324,468,154,792,470,65,824,547,889,90,460,677,734,40,640,368,168,156,661,579,260,273,700,600,204,844,449,907,905,134,58,955,818,578,195,675,417,236,192,418,659,238,928,360,824,133,127,584,722,992,108,382,987,160,340,478,396,849,486,533,802,608,827,71,385,471,370,912,284,30,306,928,144,773,328,959,458,356,211,153,743,702,658,432,64,251,741,105,54,36,969,972,388,874,833,433,970,263,582,365,131,300,222,15,167,315,733,156,181,757,221,494,933,32,537,61,53,293,784,234,365,155,388,439,16,15,217,974,171,282,547,884,937,902,652,173,597,564,569,493,735,47,144,474,291,454,24,619,846,257,878,116,512,131,299,776,783,783,307,225,187,724,827,310,341,21,852,941,864,447,417,410,152,194,575,265,747,529,788,563,913,614,94,80,148,629,93,72,289,694,813,811,150,393,13,467,993,102,885,908,184,688,615,357,145,102,64,361,316,468,408,862,975,211,117,918,847,34,210,776,431,121,598,541,364,928,668,483,327,704,527,337,57,400,3,868,725,537,687,969,355,141,434,198,301,843,918,288,405,275,264,674,468,166,327,900,83,614,861,534,166,528,914,528,334,515,753,812,684,275,526,677,279,248,853,877,341,419,651,363,705,170,792,887,621,743,119,701,275,52,517,618,729,777,228,495,751,616,786,920,129,966,949,560,98,81,883,815,103,493,53,764,188,935,859,758,539,659,38,220,825,865,466,212,892,322,992,334,648,727,828,481,556,54,259,461,276,273,708,517,69,224,193,625,951,54,204,476,743,687,55,507,567,222,155,372,298,592,352,767,383,351,102,679,635,841,199,677,882,262,530,412,428,664,233,391,490,620,626,354,625,997,105,326,479,811,465,92,526,601,79,249,399,11,505,886,573,919,564,295,911,852,934,587,181,150,108,327,810,440,692,178,707,908,999,69,39,340,662,45,64,526,967,446,564,530,4,552,170,169,408,482,71,641,988,276,463,460,428,958,86,100,220,915,797,447,827,681,707,923,745,897,663,645,538,208,70,124,715,535,95,408,578,985,557,689,219,977,765,189,480,811,682,579,611,200,450,144,762,238,125,861,232,753,930,35,504,572,440,733,812,832,11,913,675,306,101,848,978,263,157,693,829,505,492,301,633,362,810,857,272,764,392,741,18,591,618,809,270,950,881,789,820,165,14,198,405,968,908,921,869,509,33,793,384,899,810,238,77,612,223,176,442,33,703,119,591,255,67,997,154,129,409,492,252,300,205,729,341,663,368,606,572,807,801,518,135,646,154,767,572,64,297,346,238,997,940,369,351,655,8,21,981,280,888,266,338,320,585,635,990,571,944,378,448,768,616,77,337,287,838,695,521,479,623,694,716,14,607,185,675,491,480,809,942,228,322,67,65,266,323,530,38,889,703,859,178,491,439,347,168,876,362,751,831,296,73,994,485,859,654,312,414,386,226,544,462,644,667,704,346,875,741,167,813,184,549,628,714,589,785,270,778,212,606,495,886,500,63,224,736,940,197,650,339,922,964,160,480,824,347,158,643,632,528,287,892,133,501,264,641,961,187,836,427,287,694,390,810,111,781,677,867,14,56,431,599,115,341,786,990,884,630,896,20,201,295,655,253,281,370,281,988,567,629,648,430,591,624,613,117,479,587,876,554,219,607,348,761,396,963,682,184,108,742,890,634,231,954,418,138,19,290,481,460,485,107,564,719,932,463,890,331,622,241,155,105,789,319,144,690,217,271,180,732,786,886,879,575,375,502,953,937,593,782,38,665,927,110,375,422,698,291,279,27,779,411,350,603,895,811,894,434,234,597,850,784,722,913,518,495,862,942,451,44,246,22,526,121,854,836,954,749,678,195,972,201,976,794,660,691,203,642,301,457,923,453,826,233,701,734,331,862,923,164,189,316,844,214,499,867,822,856,489,768,723,55,776,767,928,609,543,175,265,754,856,883,110,489,352,266,747,11,293,273,286,757,820,592,563,534,304,416,810,471,547,572,932,837,427,859,18,626,766,859,809,655,639,109,15,57,592,991,176,745,930,929,320,756,634,807,115,738,847,748,29,941,712,723,31,88,214,64,274,716,392,486,948,419,575,482,861,467,149,106,468,562,566,798,31,373,325,492,30,338,519,296,755,175,875,740,570,945,753,250,919,756",
    "95,46,971,803,206,980,268,771,63,131,352,63,110,520,354,163,442,189,969,200,261,826,408,128,388,530,900,0,492,882,490,660,480,216,925,575,165,974,995,391,559,338,861,311,617,37,53,659,525,153,516,88,746,365,255,772,67,175,677,690,544,857,607,617,767,644,882,454,293,696,937,892,183,611,575,839,23,150,472,123,987,735,918,18,369,51,589,281,357,637,359,294,996,953,117,664,39,792,786,627,720,259,522,994,337,159,302,746,35,642,305,488,191,534,177,709,7,718,443,21,734,858,543,945,661,111,486,209,496,426,686,140,549,197,736,551,308,368,133,666,48,113,317,917,496,11,736,489,173,399,499,643,448,62,673,496,205,352,949,209,555,154,958,761,314,477,278,213,435,600,339,173,84,322,623,22,708,240,570,381,892,397,127,687,584,674,259,204,785,203,529,807,618,869,230,337,705,647,484,289,555,867,940,676,333,552,727,17,424,79,377,13,374,75,873,358,152,631,587,875,691,446,764,388,396,712,211,23,187,582,774,730,916,623,721,784,787,705,880,493,544,768,619,755,299,299,833,76,906,372,596,289,59,627,265,909,552,131,62,540,978,697,958,102,875,71,377,344,237,672,953,551,983,122,382,77,573,473,626,768,462,32,419,639,417,809,203,647,945,716,292,634,646,628,631,97,774,556,581,204,742,479,448,944,514,288,385,399,787,179,611,258,481,368,756,381,364,425,546,885,894,869,959,473,499,683,729,884,902,523,494,482,241,552,410,42,645,10,736,7,921,939,248,141,909,754,830,708,795,862,698,602,616,514,528,624,480,988,433,250,540,804,185,223,884,42,294,104,983,979,69,32,907,237,890,613,203,25,463,166,48,860,297,714,799,871,621,451,937,556,761,381,206,338,939,912,675,659,989,312,106,979,125,399,541,887,802,77,322,601,550,81,474,310,345,120,156,917,37,238,933,108,485,351,985,113,804,109,599,901,844,540,883,807,934,193,515,741,515,2,624,781,138,883,355,373,63,426,759,41,484,671,616,187,969,408,160,790,356,29,650,525,52,529,7,196,461,174,612,561,1,415,182,129,876,583,697,194,610,346,764,14,935,692,867,513,710,726,610,520,78,530,684,515,761,378,859,19,856,589,143,306,733,419,348,39,635,824,383,234,450,382,684,309,622,671,68,859,39,41,348,971,251,57,109,956,127,985,560,773,651,444,748,881,704,951,940,30,765,88,588,720,310,242,896,607,367,894,887,569,440,333,239,825,794,211,569,774,986,818,262,832,466,910,144,655,415,601,118,119,470,559,500,161,967,259,34,141,594,903,379,529,277,407,604,198,594,561,129,455,809,887,939,705,998,915,839,794,280,819,927,343,172,191,881,68,578,850,966,399,302,194,815,534,968,956,607,134,557,383,979,105,522,814,551,699,979,252,737,42,395,798,587,554,71,529,797,761,684,66,520,996,302,87,762,501,892,149,11,2,200,266,868,568,497,407,387,95,954,625,620,340,151,704,671,724,237,463,600,503,627,851,258,496,856,143,625,414,749,504,320,229,335,234,1,850,518,286,730,803,2,40,458,457,638,78,670,404,461,242,881,20,345,199,777,834,284,863,420,531,163,985,197,414,58,850,897,837,462,893,713,120,516,454,284,721,715,137,119,448,193,349,544,162,501,501,511,827,837,350,347,427,400,230,451,678,279,716,821,879,908,773,30,268,855,754,596,701,557,713,952,932,386,11,682,919,178,535,102,827,908,988,138,203,267,766,994,547,699,702,266,424,206,263,785,316,782,335,834,521,898,442,371,437,163,509,638,858,895,463,55,813,913,645,887,860,897,275,370,990,775,824,239,775,726,979,914,890,822,883,616,135,524,613,241,370,519,876,662,192,123,841,483,165,478,945,711,146,16,372,508,878,427,758,932,537,627,331,246,575,262,496,471,201,503,824,756,350,958,540,146,422,175,775,634,789,969,944,847,261,358,806,629,666,96,982,270,63,939,758,515,653,840,520,440,999,584,768,26,34,767,601,623,570,995,8,888,211,775,147,437,691,633,49,407,716,49,387,220,855,789,258,821,148,684,797,444,629,63,776,894,255,445,887,51,819,129,476,890,540,732,357,526,670,463,42,624,137,727,269,646,300,945,168,122,28,826,477,523,11,800,448,181,764,871,178,758,49,901,233,96,213,878,306,2,537,899,393,861,533,572,504,923,713,393,113,420,767,832,181,5,833,995,657,845,565,615,398,787,288,659,179,626,723,142,972,722,844,469,264",
    "446,325,750,595,811,462,898,674,722,573,841,352,194,184,512,336,485,487,534,957,165,727,976,274,733,608,77,492,0,383,129,57,115,794,896,829,523,476,567,959,492,596,239,16,265,809,732,470,156,464,66,480,332,55,412,873,600,123,658,25,157,550,202,615,371,574,461,425,659,412,548,161,161,287,160,969,617,516,450,605,718,571,725,322,222,250,787,600,606,51,898,594,843,917,789,174,486,914,698,218,315,589,757,961,663,875,573,665,556,212,323,736,456,566,37,723,662,953,613,751,162,227,431,233,863,595,353,53,908,645,710,241,28,681,337,661,356,348,815,438,444,860,570,20,33,638,322,717,812,351,276,920,717,746,603,79,274,43,3,201,210,758,436,141,899,708,979,627,280,645,784,743,492,285,95,268,990,740,721,465,292,738,621,329,389,437,837,223,178,899,708,708,511,295,101,580,504,562,168,815,471,841,151,337,879,442,650,949,271,211,593,522,37,229,765,675,26,313,895,293,458,235,414,405,501,528,377,257,952,887,908,769,206,239,883,974,383,497,907,824,384,966,477,630,958,57,62,87,274,550,963,993,71,970,655,724,769,247,628,541,48,275,357,831,135,797,827,426,334,291,621,78,635,411,988,590,445,392,431,364,987,338,468,173,237,125,557,769,291,231,866,20,808,473,726,177,556,349,801,770,712,611,274,353,560,550,98,240,177,362,209,682,305,461,495,921,581,61,427,770,238,243,719,619,597,3,330,825,232,424,107,799,332,867,424,907,274,805,929,534,631,159,841,762,688,780,131,725,798,287,766,115,46,664,509,882,310,55,670,512,464,951,4,409,320,742,843,919,13,896,646,958,260,256,341,772,61,846,186,528,391,983,291,16,776,805,49,715,639,965,310,344,358,732,190,185,75,798,907,697,577,924,460,446,130,107,874,360,912,84,286,33,726,102,919,910,306,815,860,942,789,710,545,369,34,659,893,797,879,517,967,13,946,361,652,296,738,975,512,394,88,192,61,655,382,885,355,856,482,935,310,228,455,288,92,721,14,329,587,928,78,162,9,117,948,526,348,979,107,932,406,107,59,861,793,165,811,539,344,377,394,439,812,441,992,969,160,562,24,386,911,550,299,547,189,499,412,351,921,282,497,357,173,548,815,778,657,720,956,57,526,264,925,647,612,882,470,248,628,239,133,729,281,530,143,388,433,628,768,294,724,213,945,145,885,239,150,305,315,932,644,726,97,671,381,431,354,17,278,940,498,460,56,919,455,491,1000,744,32,349,529,121,238,690,184,611,386,817,209,114,104,750,810,599,631,394,774,44,240,351,380,493,806,897,197,708,325,597,474,279,924,319,871,57,773,296,888,438,760,860,264,889,384,675,286,282,783,781,33,237,827,125,660,28,539,843,415,282,834,826,660,621,577,39,283,439,380,511,756,856,374,616,368,496,888,884,584,80,610,221,899,604,25,416,476,775,613,505,133,40,758,363,632,109,2,292,431,754,994,885,65,794,328,787,960,780,892,834,209,677,431,359,374,694,352,559,9,246,780,33,119,444,710,610,28,366,156,391,251,259,18,901,114,83,85,383,960,90,577,910,441,575,255,901,991,254,247,577,97,563,76,854,554,976,863,769,157,954,104,553,485,842,334,808,371,726,625,465,30,674,538,774,872,391,950,53,127,407,734,258,913,817,117,969,761,935,685,398,111,566,172,873,591,407,814,980,469,239,10,635,339,245,892,900,654,313,910,602,511,483,326,778,487,623,448,160,229,898,343,94,474,734,916,155,372,978,840,684,257,295,141,964,712,26,872,851,829,553,363,572,801,972,203,124,247,993,781,536,759,538,994,249,653,353,485,575,367,268,620,777,647,851,802,378,970,830,753,731,622,228,321,579,761,637,854,928,627,722,270,784,693,50,607,34,757,899,232,415,784,796,299,80,193,627,478,402,982,990,95,308,149,461,576,434,609,146,687,2,308,209,958,440,61,808,141,687,385,290,735,564,629,840,852,698,981,739,230,195,953,456,81,637,275,714,951,67,764,445,783,143,972,428,120,599,789,271,626,253,170,551,699,608,152,430,693,509,188,319,198,847,536,337,575,803,375,944,722,101,228,161,746,235,123,652,971,857,247,971,947,742,142,598,248,958,679,254,77,723,999,176,589,429,523,748,398,618,914,814,749,114,539,955,564,511,994,59,859,441,546,549,342,401,475,836,151,182,360,999,676,750,297,154,374,558,206,425,315,817,474,999,375,765,354,615,824,510,200,475,860,851",
    "86,418,652,15,700,713,611,426,811,120,518,484,684,281,654,640,233,11,5,281,483,77,798,555,119,328,745,882,383,0,806,714,266,69,179,807,425,846,564,334,148,810,112,360,273,168,723,779,511,273,761,417,813,184,955,183,592,912,48,14,777,968,274,799,314,56,563,488,650,887,162,697,487,726,431,332,510,412,92,215,347,858,459,169,208,899,615,522,322,9,663,351,174,547,524,774,66,935,563,511,532,816,238,297,570,455,180,596,541,629,180,169,986,749,471,423,564,10,117,149,858,617,681,82,534,286,183,396,404,986,517,112,147,874,64,542,323,661,149,88,520,814,621,134,855,29,429,127,938,920,758,980,985,236,44,748,730,783,915,654,30,800,768,27,407,184,641,90,637,653,291,579,372,187,731,514,201,526,254,418,458,601,31,30,194,582,539,345,759,95,563,765,416,849,512,864,221,351,194,91,375,228,424,983,388,807,279,521,318,697,625,710,937,959,58,686,387,750,582,532,433,342,269,592,612,84,80,683,745,654,253,855,427,841,527,573,818,1,122,276,141,377,225,911,804,568,952,355,844,937,441,678,129,128,813,688,697,438,601,187,154,819,718,366,604,189,44,912,956,30,661,152,346,431,420,275,670,708,977,86,375,749,344,544,452,75,394,977,705,329,881,856,495,955,744,462,172,34,12,296,139,189,92,336,531,951,485,362,436,356,513,420,285,890,268,947,852,555,176,593,793,777,636,815,936,427,309,218,149,695,874,585,731,441,458,924,606,592,763,653,504,318,313,438,859,956,904,258,917,192,118,809,524,157,912,268,627,79,49,449,840,727,781,514,349,25,282,764,971,740,351,307,315,782,236,742,138,362,516,306,588,40,996,835,153,746,713,705,855,624,34,877,251,350,261,816,486,686,657,940,466,222,255,419,823,944,226,74,916,916,343,118,692,965,501,533,737,32,156,671,220,666,551,984,613,308,306,308,873,274,321,532,919,266,311,596,171,68,757,949,483,22,446,251,860,519,992,431,249,153,479,16,948,542,301,326,166,685,132,858,635,789,789,112,884,340,637,958,394,433,184,443,901,821,699,174,596,648,487,977,845,781,865,32,343,177,823,115,461,780,548,268,83,458,468,349,591,705,689,713,603,37,764,913,18,698,282,10,557,932,44,763,326,892,557,42,863,136,680,592,292,232,65,788,146,283,331,512,291,640,613,593,845,729,272,870,953,298,223,351,493,762,841,306,925,16,753,271,323,13,435,622,392,477,835,367,20,293,912,857,761,294,302,35,706,520,574,896,259,367,527,423,781,387,233,260,727,224,255,422,655,126,666,465,617,554,420,53,670,178,210,186,597,186,983,772,474,690,583,802,290,49,558,232,260,962,195,615,885,799,575,582,190,581,556,703,52,758,967,416,386,742,801,307,936,49,33,20,842,935,570,14,420,572,254,307,99,730,561,220,204,814,793,234,293,377,643,148,207,415,471,398,400,963,484,288,4,295,477,330,160,403,776,537,813,569,762,77,985,586,341,884,371,381,300,351,510,902,829,183,374,364,527,80,503,938,970,149,977,453,576,432,414,756,543,424,217,275,460,280,122,227,140,64,691,835,478,117,943,286,600,458,491,904,843,480,893,922,44,55,516,356,307,463,196,364,36,291,965,457,22,479,248,229,742,79,351,396,522,335,716,509,712,648,912,428,601,710,210,531,2,992,289,246,230,759,98,438,59,113,778,760,622,298,308,996,302,17,899,541,618,43,675,83,191,949,946,365,756,599,479,931,682,667,366,644,543,858,362,487,569,935,620,692,486,703,322,566,458,470,938,748,35,25,874,654,983,489,168,602,462,732,656,401,204,93,466,14,59,339,542,108,439,839,525,775,737,366,40,827,629,435,257,140,176,426,224,378,330,440,760,931,75,584,806,382,769,534,155,38,185,49,970,13,683,918,328,343,891,219,92,822,179,553,554,431,249,405,888,284,490,901,388,963,838,655,618,439,282,95,206,749,276,761,876,92,32,604,543,498,63,887,907,415,534,587,896,714,884,837,215,975,848,159,298,662,273,314,909,767,885,314,347,442,72,959,793,316,160,735,735,302,838,618,43,619,562,961,578,225,948,743,235,874,4,952,463,240,532,577,128,276,66,328,68,351,768,810,970,917,157,670,385,290,920,446,614,928,532,80,789,604,952,584,94,449,683,470,582,84,81,603,62,940,967,256,36,965,204,487,667,842,431,552,601,768,332,27,563,642,996,198,319,425,99,541,396,347,216,423",
    "397,304,904,973,708,488,385,505,624,403,392,122,590,506,425,427,729,264,142,761,239,988,427,24,454,271,968,490,129,806,0,500,19,949,953,345,971,962,839,732,325,290,11,445,78,904,992,30,398,35,59,817,89,224,157,955,876,736,232,567,492,30,241,888,678,871,753,546,449,666,408,803,937,332,314,213,694,594,295,294,765,436,143,496,495,683,463,49,574,516,610,266,751,672,385,367,781,267,425,123,737,548,957,24,705,750,685,267,550,933,173,285,953,269,406,218,625,702,726,258,854,738,383,193,523,879,350,895,550,279,448,194,89,391,527,118,51,644,296,715,523,535,731,861,878,356,171,899,782,944,520,879,739,626,23,656,643,121,318,457,92,152,794,422,424,712,858,906,65,813,283,773,790,542,531,670,555,125,76,297,277,112,630,676,411,281,869,973,548,108,549,226,185,560,344,760,491,433,776,630,574,287,447,674,903,381,747,187,811,185,985,302,70,765,261,838,133,993,275,286,461,694,562,430,632,848,334,163,262,156,146,764,631,153,146,554,860,110,128,941,660,109,676,609,187,217,796,455,115,707,514,199,383,769,488,440,440,5,680,79,338,161,809,153,940,125,290,841,505,220,913,299,349,416,545,5,3,494,628,469,948,741,860,170,126,792,383,897,27,497,152,572,86,36,631,342,905,536,217,474,603,597,371,777,399,997,70,813,869,680,408,454,58,481,810,735,377,813,644,760,275,212,943,673,167,417,936,424,391,169,502,230,727,813,745,855,109,563,711,33,555,651,599,866,879,586,61,269,926,260,468,713,845,394,651,806,552,948,899,307,698,718,595,312,807,849,124,860,788,911,742,703,828,607,200,870,662,955,93,194,559,271,332,303,320,323,813,550,234,961,331,56,594,260,658,232,856,806,898,31,394,445,462,491,705,113,857,518,862,349,328,550,841,860,618,330,998,414,400,939,353,994,123,992,983,906,631,971,188,875,34,717,77,693,559,790,725,225,499,34,400,662,845,734,752,537,269,534,749,690,798,523,877,712,568,635,197,789,294,809,710,752,404,784,461,443,894,53,696,550,835,310,291,704,560,895,36,98,569,542,92,399,986,379,596,586,114,946,758,613,303,879,753,437,631,702,779,788,712,537,997,917,716,684,159,748,176,151,496,90,700,437,723,272,943,188,670,239,294,195,410,105,784,402,149,737,29,904,593,240,679,428,335,862,126,921,511,293,556,191,526,215,769,255,421,337,521,368,526,496,81,747,457,488,912,557,597,277,234,600,390,994,438,872,960,168,349,230,946,945,874,999,22,914,194,503,808,755,761,543,384,530,247,225,685,674,321,593,175,481,556,451,955,570,264,747,98,470,56,829,957,678,76,648,627,82,654,721,171,46,133,702,955,816,294,734,952,584,106,564,221,915,760,635,35,148,672,95,851,807,579,833,648,14,541,58,665,80,636,613,115,517,48,146,524,794,185,695,220,391,157,808,322,717,734,701,893,740,834,906,48,619,56,279,556,161,426,81,90,909,816,802,29,514,883,476,391,233,329,597,3,721,279,397,262,438,181,218,914,4,275,123,384,950,97,762,198,98,557,284,927,205,87,432,981,924,137,14,33,623,844,471,271,482,749,701,464,225,859,538,750,417,963,459,643,104,183,459,563,397,452,800,574,16,647,611,125,759,656,204,433,823,183,237,143,530,799,241,584,952,591,989,148,955,36,259,831,896,981,436,858,117,941,209,31,904,388,782,982,365,254,394,936,381,581,179,105,101,270,582,995,180,44,249,439,833,307,495,434,812,253,868,546,429,965,613,399,411,240,646,850,586,173,503,686,150,443,83,749,424,905,532,507,174,679,699,592,630,255,722,397,200,343,968,553,751,558,859,495,848,607,402,700,615,126,825,336,422,486,708,838,43,66,287,84,766,138,840,339,184,638,366,885,705,599,782,950,618,666,542,873,437,215,450,566,87,155,710,676,644,820,535,678,43,583,786,337,94,32,148,777,958,409,157,412,138,668,773,812,522,98,732,265,902,937,204,777,21,874,829,171,609,331,828,325,635,567,611,988,661,453,93,708,603,629,904,964,759,278,252,832,809,776,632,860,822,995,173,817,952,519,352,764,852,516,481,691,429,471,684,968,767,384,207,286,208,950,537,133,371,573,704,585,146,745,949,959,598,345,702,747,32,955,55,544,25,422,102,287,842,554,108,807,905,163,706,804,144,76,463,641,642,591,980,922,764,875,968,154,768,925,205,156,984,474,227,499,518,946,653",
    "893,904,223,375,271,755,74,434,517,566,982,756,158,575,493,233,548,597,574,131,417,67,620,852,770,341,52,660,57,714,500,0,126,276,671,125,556,179,628,311,665,594,340,115,195,834,240,860,740,164,145,473,502,670,332,922,645,981,296,286,786,838,389,175,673,803,246,587,725,687,82,935,486,615,778,216,402,237,530,570,759,474,338,513,217,949,914,682,736,813,467,541,248,29,14,794,895,66,34,549,523,545,32,3,669,157,824,271,192,210,283,324,375,724,339,96,833,252,15,311,587,663,328,258,236,328,991,407,19,514,730,871,415,631,167,986,812,450,235,902,804,563,620,192,481,528,470,403,605,897,850,649,213,694,683,350,630,385,311,220,486,31,477,535,398,285,688,98,407,710,949,305,879,395,559,648,780,797,323,524,261,853,233,312,896,960,693,450,144,74,739,614,318,663,82,781,452,648,304,701,798,610,89,347,821,538,182,102,544,857,634,983,967,53,108,983,512,284,795,128,880,443,334,37,172,197,561,663,26,196,911,121,768,930,856,379,87,866,831,517,489,332,301,291,340,474,355,541,59,974,164,991,851,599,230,285,992,158,858,32,248,793,91,559,294,126,893,791,74,972,379,812,328,682,815,995,484,538,292,334,681,381,938,460,728,373,547,395,134,300,560,997,110,932,611,464,675,117,396,171,176,320,52,40,854,144,792,410,122,247,91,289,824,921,579,532,971,379,729,941,48,115,548,905,568,768,651,582,808,505,641,205,399,607,439,138,354,836,276,6,914,995,468,721,97,884,853,603,295,519,958,344,625,252,297,619,412,211,532,287,829,847,452,612,50,729,683,995,875,365,882,886,919,93,23,141,94,345,428,741,530,619,490,12,545,165,42,174,67,232,249,203,33,936,489,596,139,435,677,83,612,186,188,450,533,194,711,371,341,715,927,767,217,566,750,46,632,113,729,983,283,559,192,28,674,955,315,285,919,980,398,518,159,894,358,595,539,967,828,329,980,831,117,945,789,194,17,71,346,433,920,377,243,644,26,38,538,546,601,682,986,643,700,396,510,524,601,48,269,598,445,496,37,279,210,15,446,698,133,4,745,457,31,462,146,459,425,962,251,452,910,986,451,919,429,387,151,655,323,106,833,276,145,251,950,884,156,25,798,302,432,640,914,458,172,671,459,682,767,144,217,440,397,185,581,733,15,204,253,552,543,773,718,918,713,122,936,35,406,931,614,884,774,195,94,854,607,840,22,834,396,917,315,536,371,943,43,496,967,71,971,101,271,55,126,426,82,616,61,263,498,640,987,630,145,162,811,472,101,365,688,931,91,195,442,994,219,319,202,524,141,161,45,862,162,977,999,941,153,420,203,192,820,505,792,698,548,17,616,647,477,696,168,709,949,679,681,948,248,585,895,238,743,590,822,394,644,154,348,454,173,635,992,876,91,676,951,880,617,399,763,653,848,460,527,641,529,585,963,347,474,747,577,501,376,70,787,383,675,874,131,402,865,480,919,258,951,136,729,557,550,127,415,282,501,526,907,466,108,897,731,206,148,448,849,507,26,656,219,115,673,119,814,693,726,290,653,419,58,823,593,241,36,293,724,690,77,438,283,353,746,70,485,625,825,241,968,123,962,854,146,80,200,55,978,830,794,143,402,518,451,96,689,541,834,978,520,223,554,858,632,484,794,152,392,80,389,578,397,119,397,490,514,535,556,569,826,208,741,276,790,291,78,793,606,72,445,200,158,997,623,249,992,491,967,223,994,909,303,833,868,381,336,230,186,289,918,304,288,847,444,7,264,228,397,309,473,394,911,780,468,887,787,286,658,537,881,956,980,380,845,988,406,494,170,556,479,482,437,773,252,901,490,890,415,697,760,652,101,219,108,773,705,283,946,813,974,631,995,901,267,734,44,173,962,479,398,967,669,543,115,871,114,434,357,797,28,549,258,558,223,139,155,787,641,820,594,429,678,97,272,861,648,264,829,845,696,837,715,58,981,607,807,4,812,622,355,856,651,512,659,291,766,450,474,745,994,638,853,829,803,482,86,523,26,856,8,819,995,463,305,510,625,111,306,331,75,851,143,346,704,640,129,968,489,998,757,573,673,339,137,110,667,939,115,653,248,2,666,78,787,541,236,364,85,793,266,182,156,738,581,445,523,969,587,647,493,346,214,793,675,447,782,431,105,351,363,571,456,832,496,509,736,440,982,292,960,319,444,772,16,922,851,405,502,940,149,252,529,402,960,323,739,542,466,972,390,668,38,22",
    "56,295,132,674,620,518,864,727,195,346,264,55,346,658,763,478,200,427,324,916,828,57,229,567,527,90,97,480,115,266,19,126,0,897,271,659,573,383,611,885,405,848,699,379,648,93,218,473,398,771,379,679,342,853,453,121,443,371,767,882,808,573,42,862,473,453,891,955,227,958,102,81,139,887,666,6,782,837,780,548,663,337,610,557,69,129,488,336,920,66,463,93,195,681,866,349,259,754,100,334,815,969,537,934,528,28,721,574,947,459,657,232,663,672,221,44,226,805,953,195,672,191,493,975,262,485,958,683,395,507,85,477,188,455,579,602,732,388,620,822,788,93,593,277,721,878,758,237,217,361,815,825,378,109,243,874,804,190,232,467,561,715,234,339,185,150,739,861,406,895,234,275,341,559,265,618,189,70,857,294,457,427,921,629,460,704,621,794,186,78,331,572,794,123,941,728,555,335,474,234,409,339,81,620,450,674,217,751,90,609,888,437,440,994,817,971,261,676,25,251,985,956,366,830,679,913,478,410,878,313,532,132,142,96,359,410,593,747,716,268,932,303,727,851,891,422,139,674,911,312,502,898,551,629,994,328,708,347,324,160,462,870,910,491,994,372,999,286,529,234,26,32,47,483,958,302,12,580,956,370,809,329,835,167,227,396,419,999,552,202,474,260,975,809,852,540,737,561,573,989,23,506,676,237,246,23,62,104,774,893,180,808,96,395,261,602,508,643,909,185,854,944,532,121,406,922,77,571,113,530,624,868,520,285,478,980,276,447,439,918,247,131,690,564,334,511,416,825,150,551,554,205,891,258,140,545,402,334,556,171,330,222,781,926,112,473,547,816,166,341,746,495,168,403,816,395,441,411,122,71,955,796,490,912,744,391,248,535,864,612,358,178,613,651,646,435,479,798,597,6,638,941,559,775,933,469,391,388,950,924,763,283,930,39,847,898,416,37,802,971,346,896,662,68,425,714,855,698,946,564,763,435,102,386,94,475,586,20,660,310,985,637,470,124,127,534,111,140,94,763,234,412,319,61,622,516,212,393,663,305,113,168,797,420,358,764,23,404,17,872,874,459,499,624,718,238,994,353,194,454,142,6,716,114,710,947,6,215,791,151,428,919,539,207,910,184,366,818,445,123,640,486,369,684,601,394,636,807,633,189,813,588,842,580,332,117,169,773,477,244,476,961,866,768,510,121,701,218,405,686,573,583,657,485,886,271,884,669,423,756,749,38,900,163,268,741,795,236,772,489,536,216,596,810,941,945,122,662,55,800,918,179,611,854,791,102,504,718,392,651,838,406,764,28,204,364,614,430,904,895,795,753,956,643,393,904,795,544,836,163,892,240,870,803,533,528,811,122,635,617,274,742,306,242,319,474,404,746,381,853,964,223,154,474,480,411,356,544,562,899,117,354,626,4,898,109,943,940,707,610,780,89,110,14,953,126,136,940,3,181,536,654,42,784,753,583,438,48,512,186,455,980,972,756,737,513,8,331,137,866,979,449,273,817,509,219,174,844,76,974,566,636,22,345,828,577,731,164,602,632,597,654,128,820,185,737,273,726,213,286,138,571,125,195,459,171,27,193,6,603,179,198,360,909,631,478,646,604,826,747,932,156,728,289,743,838,193,758,80,359,967,283,699,962,664,684,782,598,671,878,777,524,15,207,323,137,127,722,63,532,139,989,34,862,489,777,855,737,494,895,63,833,438,985,321,307,491,1000,369,419,280,522,496,930,923,144,496,365,130,878,812,261,709,687,766,3,777,711,382,671,344,845,529,443,211,249,699,772,577,656,417,192,514,633,244,641,121,434,429,144,357,441,273,723,201,817,535,913,910,801,296,166,188,178,237,722,895,15,954,397,24,932,191,254,391,804,905,75,194,356,594,220,600,153,194,892,540,445,220,428,159,781,930,895,176,167,189,700,670,941,488,14,760,793,628,186,736,124,356,80,109,312,978,105,623,314,596,936,856,427,896,627,125,813,585,60,133,702,921,621,532,212,849,945,631,758,263,895,989,513,203,840,726,72,272,226,539,268,557,256,870,340,461,947,165,49,298,731,970,533,253,680,816,224,690,659,596,217,874,58,410,930,272,59,142,909,247,527,78,151,766,337,805,748,708,154,408,991,996,298,908,514,973,994,107,989,630,537,958,437,409,854,564,992,421,90,945,116,585,798,514,854,774,256,151,402,104,400,677,434,884,181,35,623,708,392,797,981,671,944,571,703,630,802,539,529,313,635,315,969,182,143,404,646,542,102,929,488,507,979",
    "812,948,909,439,26,552,503,207,57,932,47,875,780,880,12,804,876,921,240,499,995,877,638,553,977,140,341,216,794,69,949,276,897,0,406,545,822,554,704,814,512,285,329,674,956,121,655,534,857,147,130,932,814,413,613,609,533,574,517,993,519,46,217,328,427,435,133,32,973,925,796,521,262,915,817,572,427,768,632,986,9,979,986,351,4,264,190,12,482,571,183,857,739,527,56,326,970,857,851,762,200,813,694,950,52,199,140,882,539,825,457,959,687,168,838,505,78,651,860,995,598,567,167,307,42,810,306,720,862,97,952,543,915,349,328,260,888,448,75,824,971,592,958,582,430,383,770,18,751,115,540,947,373,96,307,40,324,99,655,68,883,489,571,954,172,58,523,476,507,247,765,617,360,750,390,580,999,220,503,944,673,521,927,883,108,658,895,336,935,994,680,386,325,247,744,880,404,736,279,931,757,677,95,139,758,125,471,719,153,411,955,886,476,712,687,240,348,753,664,919,163,903,775,804,697,603,107,962,218,140,59,109,473,528,990,515,108,188,410,911,601,722,542,256,423,259,140,355,918,41,463,354,283,634,215,675,934,700,572,571,374,382,15,29,615,839,873,873,451,515,515,747,394,334,660,601,775,21,33,691,437,111,989,314,447,361,126,660,930,782,98,341,209,266,911,589,670,163,146,714,786,541,674,607,478,411,889,774,414,81,225,473,947,316,587,191,139,512,728,112,608,201,205,716,371,448,893,117,704,411,212,199,337,741,942,10,63,613,229,445,164,317,653,137,952,697,641,955,221,760,90,678,421,426,471,996,570,502,912,16,899,404,663,509,589,684,600,975,62,465,780,881,407,864,162,8,680,639,350,961,597,694,421,100,196,561,36,928,462,345,929,670,627,787,918,246,765,792,497,951,110,712,312,276,593,616,264,7,105,844,959,106,258,184,763,869,341,793,262,912,869,817,816,801,443,909,601,538,462,42,192,64,288,941,736,266,124,523,187,905,237,254,207,278,134,83,776,666,238,476,108,940,181,651,858,330,418,882,256,491,466,401,183,178,897,885,803,772,334,154,355,42,231,463,912,600,439,411,150,565,153,76,796,807,287,65,722,403,420,204,795,296,447,948,562,238,121,461,40,726,368,828,944,256,604,438,552,296,302,690,570,927,586,202,28,798,346,962,602,177,814,935,621,123,735,108,333,269,602,887,586,4,130,454,403,260,980,7,102,398,218,323,826,295,341,980,541,378,905,532,535,877,839,849,1000,677,920,946,491,34,73,680,618,78,936,480,436,932,323,306,117,771,400,187,485,762,900,182,335,468,811,991,825,438,554,298,770,878,347,230,28,77,182,54,871,676,647,50,762,312,233,777,858,434,337,34,635,577,545,469,949,357,401,215,850,881,942,937,865,315,861,456,816,577,934,703,220,63,955,91,610,751,355,109,203,272,73,466,395,572,701,829,823,127,524,892,536,484,235,460,708,991,69,633,71,983,657,685,973,116,802,539,127,943,577,53,21,738,327,352,703,250,261,93,538,48,304,6,929,462,462,354,741,621,481,902,942,941,177,36,972,727,575,893,970,326,459,734,215,936,510,724,233,307,885,698,85,577,40,543,594,779,369,671,446,254,222,665,137,608,388,923,958,280,608,763,336,376,415,982,218,381,400,146,410,727,359,218,920,378,716,527,566,387,218,935,315,253,741,378,411,765,862,474,606,340,256,907,767,644,760,238,278,68,388,156,383,529,114,439,561,673,342,211,989,540,985,379,248,3,824,731,685,407,187,622,425,269,774,379,364,937,820,770,330,312,991,895,391,716,736,661,710,45,229,625,459,301,533,612,695,962,34,566,205,44,402,828,502,165,20,996,57,89,404,492,694,136,369,827,740,119,646,155,543,817,557,699,512,542,190,666,878,988,138,135,727,734,68,370,2,720,492,456,38,432,848,244,457,796,877,67,78,854,645,926,370,177,339,752,286,593,61,55,90,787,739,313,411,502,434,129,499,719,655,830,29,183,599,297,903,537,579,531,393,112,177,257,503,710,435,24,394,935,321,742,978,880,663,33,257,873,42,293,337,377,708,49,706,691,143,228,979,972,809,818,657,471,780,137,805,418,987,365,193,755,136,24,624,686,866,302,947,395,93,897,84,676,324,519,375,137,58,94,996,785,880,99,923,489,468,802,711,836,571,762,760,593,534,681,187,190,842,686,941,191,51,679,751,157,260,232,952,60,612,590,607,646,92,752,103,914,842,553,30,317,756,635,69,697",
    "967,626,607,832,885,134,490,989,654,878,325,305,667,711,396,284,622,151,649,866,62,89,875,386,384,472,18,925,896,179,953,671,271,406,0,352,114,710,187,521,922,379,983,956,176,705,159,602,29,494,890,905,937,977,346,35,73,12,633,805,541,909,534,304,818,730,355,644,826,153,261,98,855,894,863,419,717,379,257,814,582,557,870,364,557,881,950,382,499,578,939,193,433,187,997,673,217,108,396,427,46,8,926,627,366,462,601,621,455,269,878,474,183,315,709,942,767,277,967,502,698,262,332,135,385,23,718,503,671,562,45,712,745,470,95,706,550,386,798,286,634,580,584,596,589,421,815,548,727,276,442,868,760,993,937,296,929,126,256,275,847,6,247,85,180,320,545,200,90,84,161,275,425,553,718,317,700,461,339,554,759,15,972,424,693,571,53,362,988,98,130,926,766,817,865,148,418,993,309,190,382,445,750,295,760,32,915,148,367,862,658,824,364,235,446,384,118,813,456,993,347,536,61,685,558,930,793,500,269,715,327,10,649,618,597,410,785,703,433,832,943,281,518,129,707,744,391,927,489,711,761,698,98,798,263,843,124,800,41,973,341,396,189,146,460,324,176,707,985,410,358,385,237,141,939,658,145,803,781,811,495,613,837,713,981,4,987,645,121,584,164,261,356,423,247,643,234,150,519,627,800,612,998,124,505,640,974,360,675,998,396,516,118,918,546,511,493,64,917,442,979,240,860,632,620,52,827,569,388,514,238,678,977,989,499,574,336,78,199,673,325,728,466,684,48,929,662,425,96,953,768,50,600,806,449,974,935,895,105,568,894,613,653,396,334,226,308,364,240,107,331,696,44,561,153,757,905,382,908,983,488,808,496,215,446,204,532,127,642,182,721,834,31,300,590,974,107,297,976,478,346,809,368,320,29,921,721,470,810,240,944,232,517,351,754,450,900,519,407,847,610,755,710,559,502,796,846,708,589,582,225,864,681,624,8,177,770,924,524,52,190,303,787,78,93,119,559,730,577,305,271,3,842,960,873,171,618,981,177,344,361,963,800,255,569,360,467,194,798,769,957,986,759,257,31,595,52,145,289,391,795,891,686,365,206,223,921,540,689,87,933,764,262,436,585,30,37,711,347,442,608,80,3,919,35,161,375,253,510,732,489,298,120,885,481,831,283,78,575,290,47,287,829,332,559,637,308,413,585,567,324,943,239,976,121,263,217,283,269,471,654,658,559,393,600,558,466,825,413,150,355,575,33,599,909,440,753,400,468,759,824,491,926,499,831,942,994,702,441,811,71,596,95,586,173,47,945,996,901,560,496,682,853,622,672,660,928,841,77,407,878,751,696,236,423,927,398,710,28,540,812,589,532,540,914,260,21,177,99,811,630,229,83,624,684,661,912,429,830,552,875,947,272,329,571,169,894,386,921,16,469,187,321,922,161,335,358,306,209,499,285,55,602,298,499,188,831,457,598,440,62,621,857,915,536,989,868,506,323,875,484,175,297,409,364,10,764,953,888,32,502,334,990,382,293,144,243,411,840,953,176,73,13,74,31,911,893,521,828,306,648,37,722,471,496,323,797,447,149,846,83,586,643,476,978,213,744,342,435,596,643,789,82,196,167,882,900,840,106,334,354,273,37,382,632,365,365,70,627,865,949,567,325,433,848,737,276,284,809,674,690,392,519,654,697,978,96,725,232,551,387,221,331,559,466,249,280,759,843,142,504,265,397,671,319,925,688,141,94,829,49,630,657,953,417,788,630,446,319,555,31,517,717,894,734,757,705,835,858,604,264,694,77,218,303,857,80,878,14,728,922,881,149,455,772,762,324,819,845,435,483,302,754,82,989,574,1,811,933,326,310,742,995,141,418,688,623,187,786,496,470,629,150,582,345,915,708,43,690,871,707,462,521,490,474,749,743,603,264,834,555,480,586,412,44,269,461,912,293,545,982,986,328,484,182,112,546,495,377,661,496,173,667,486,447,117,997,841,753,276,943,82,714,596,436,344,718,609,336,535,915,694,278,504,685,818,719,476,769,513,854,331,41,401,98,707,844,215,909,639,839,883,876,1000,263,137,436,112,827,798,48,304,333,103,997,263,508,134,309,853,134,569,656,70,570,584,42,910,458,253,607,201,48,332,204,342,417,958,583,40,630,417,786,221,798,286,899,553,8,695,854,554,829,41,764,287,353,725,674,751,844,234,669,671,763,100,443,843,556,66,639,242,640,844,382,898,284,968,873,260,105,699,899,381,407,947,739,102,937,938",
    "40,722,679,866,289,652,891,692,14,9,816,105,911,541,149,994,439,214,478,525,168,355,168,709,286,212,418,575,829,807,345,125,659,545,352,0,310,336,656,507,41,260,375,99,406,593,307,816,240,12,690,244,765,408,961,959,699,532,274,333,241,510,709,631,550,340,107,628,666,492,999,435,386,337,618,992,376,845,652,713,979,824,679,210,388,107,151,436,977,379,701,759,568,457,440,516,479,236,520,525,221,683,254,436,348,60,833,140,116,243,943,938,773,370,564,869,318,236,340,946,633,363,712,745,39,94,701,698,464,121,611,694,872,815,948,577,110,168,382,121,790,590,779,492,871,51,506,48,218,753,295,134,349,94,682,100,707,856,993,481,498,351,245,383,326,46,522,660,303,37,222,413,70,667,818,330,349,643,400,967,963,289,201,245,748,22,177,191,699,726,296,313,526,674,299,123,133,183,321,573,301,709,258,469,762,976,309,719,195,828,762,327,783,926,230,220,474,528,833,292,546,877,742,305,86,445,870,917,75,430,796,204,165,767,745,839,513,495,204,801,823,969,666,255,978,347,578,644,951,918,584,75,702,125,561,408,916,843,515,923,773,262,683,873,162,14,29,340,835,709,59,364,850,643,463,66,584,612,37,496,687,319,736,337,87,656,262,376,186,64,708,836,877,501,101,36,980,733,606,332,140,457,484,998,182,716,479,279,719,751,548,869,911,886,72,497,764,572,907,124,971,598,352,309,1,638,190,449,534,81,9,506,810,628,304,816,563,957,424,408,691,302,990,253,950,167,122,302,487,797,307,634,310,558,64,621,113,664,804,348,518,842,327,119,226,971,623,173,11,423,61,428,402,15,546,569,780,232,425,465,387,2,459,88,943,291,774,639,914,785,659,465,519,393,763,641,893,447,73,615,484,922,41,962,48,447,918,786,928,451,630,483,387,285,280,76,319,769,902,47,631,969,35,68,894,669,527,248,281,199,251,344,457,65,439,888,801,478,293,622,447,929,855,54,440,136,573,551,512,121,319,114,682,131,84,880,872,85,623,859,882,929,135,394,333,397,963,370,311,254,271,885,520,301,240,697,162,920,105,14,113,25,345,466,803,571,931,104,202,946,168,940,466,648,199,6,878,887,219,433,242,693,131,938,278,511,661,246,642,527,296,886,414,44,649,562,614,652,291,583,792,61,960,102,516,947,991,161,560,858,923,219,118,641,852,753,808,303,272,836,830,479,769,557,331,832,256,255,228,668,871,417,370,441,161,32,892,672,294,746,106,524,246,578,783,511,699,267,596,914,227,977,995,513,464,540,274,595,482,685,697,873,89,742,490,511,663,347,110,659,108,99,536,757,180,876,245,310,696,405,359,871,853,994,737,650,431,243,903,964,273,22,505,114,314,676,247,761,759,372,58,511,302,895,104,814,50,435,781,30,831,824,930,716,762,462,358,755,29,420,411,884,573,921,914,813,408,647,646,687,834,968,110,531,46,668,810,866,234,190,602,633,287,437,347,958,183,776,464,914,928,931,772,932,339,822,865,206,633,828,895,7,57,957,203,321,944,720,362,974,381,199,34,825,75,854,792,153,103,827,151,544,580,395,777,3,245,723,911,935,540,219,956,610,264,410,607,253,896,534,64,456,533,112,823,833,423,779,136,593,385,794,855,464,697,626,268,814,595,500,594,40,189,182,813,239,928,684,490,896,128,287,16,233,858,298,532,824,88,252,142,327,132,959,220,758,649,68,480,953,87,846,589,399,430,76,785,223,305,893,44,840,324,399,742,891,256,424,384,225,136,936,763,166,746,705,222,258,406,884,729,486,177,78,622,282,96,605,440,597,961,959,625,706,963,969,827,488,450,701,882,807,782,242,876,277,521,443,395,814,293,419,936,599,5,283,112,2,937,131,544,34,304,318,873,633,107,664,107,977,294,604,751,987,520,906,767,631,520,744,494,753,187,645,934,553,777,872,850,314,264,188,214,466,967,22,9,303,24,369,671,229,255,250,558,306,953,913,75,59,843,523,374,902,739,414,547,597,529,646,370,432,743,805,973,796,224,527,343,621,573,498,975,660,204,194,245,854,974,44,633,380,834,395,655,90,854,307,94,758,246,17,790,630,137,552,487,653,763,929,29,172,482,986,318,224,503,942,580,444,827,369,17,186,373,839,57,954,200,157,377,554,730,795,770,981,128,861,839,917,493,952,295,87,611,880,951,50,908,162,989,851,474,304,986,525,211,353,128,824,599,220,164,802,441,142,826,696,229,476",
    "229,167,461,244,859,299,805,965,815,187,178,785,951,122,785,373,257,109,379,447,179,338,304,481,345,167,53,165,523,425,971,556,573,822,114,310,0,153,876,804,859,472,364,502,573,351,181,496,790,304,743,367,765,359,16,450,964,328,948,617,440,43,730,703,481,131,94,972,304,414,269,165,66,350,577,165,199,649,204,865,998,953,996,244,239,866,203,723,649,281,749,77,990,905,16,751,620,756,142,673,532,525,582,944,376,463,88,253,161,507,87,681,920,83,256,359,904,330,606,681,744,702,713,805,780,289,988,740,714,88,928,493,668,600,672,907,841,819,521,40,941,558,137,9,327,309,302,913,697,22,793,154,909,879,341,581,955,732,584,8,420,183,193,955,391,79,970,229,928,217,238,856,491,154,195,193,128,842,303,353,94,104,138,267,313,59,702,604,372,487,231,663,525,258,305,723,824,215,248,630,949,831,315,331,65,978,568,978,626,526,935,662,790,843,547,708,607,714,685,983,651,374,258,903,620,565,614,43,408,616,60,201,510,826,23,832,710,36,916,465,853,24,623,646,481,191,343,584,782,737,37,992,12,427,905,986,865,737,268,717,245,712,221,51,907,687,40,541,679,464,242,719,89,992,881,300,474,717,492,73,601,136,122,261,203,267,445,708,866,339,807,906,956,674,721,244,510,400,535,668,949,24,499,274,325,573,112,511,733,10,496,403,661,223,363,519,84,579,785,889,362,948,130,493,443,155,863,116,966,870,620,533,914,414,989,670,149,404,44,923,966,365,645,337,161,741,79,642,227,918,879,816,210,804,360,986,177,222,746,740,587,989,21,610,344,127,476,9,586,892,830,689,866,25,972,822,15,193,366,829,85,666,102,160,8,833,792,834,503,912,35,650,764,380,44,714,831,428,94,956,759,35,270,509,112,347,174,799,930,62,963,558,790,299,654,955,715,521,196,924,822,245,220,538,171,76,567,917,745,305,202,999,414,830,723,906,477,34,256,674,880,813,403,780,280,315,45,946,498,696,882,50,969,282,608,339,501,875,805,108,584,241,446,249,448,744,985,335,303,997,173,520,10,881,488,16,297,88,665,628,720,850,983,505,753,952,501,602,364,564,26,84,576,301,786,361,68,561,850,416,56,719,990,101,697,699,248,45,681,575,314,405,478,365,291,969,748,374,217,260,343,198,768,905,960,29,283,866,863,137,860,501,839,585,125,170,46,277,209,346,128,234,357,355,24,451,908,182,508,877,430,195,96,368,373,279,614,260,326,192,551,61,933,128,670,546,646,742,320,410,70,555,689,298,769,237,703,608,585,923,899,282,924,853,677,978,938,110,726,677,329,913,466,878,884,698,86,862,962,254,846,92,542,600,95,463,516,373,140,103,385,258,226,932,384,195,343,499,127,465,705,152,511,758,395,360,437,39,426,89,53,368,419,914,826,149,695,727,206,681,568,163,891,798,167,723,205,130,769,321,437,985,527,289,450,496,556,567,973,540,545,244,559,182,390,357,130,668,905,546,792,68,105,912,88,854,256,46,336,826,320,219,765,28,502,260,446,518,430,156,721,169,291,384,564,423,386,885,591,98,542,692,618,880,400,777,885,59,589,624,626,621,517,63,547,424,523,493,608,676,584,181,623,675,722,525,238,629,649,968,860,647,926,333,659,78,41,878,497,966,869,301,28,483,801,869,160,724,411,411,351,575,388,232,60,342,818,362,742,102,387,855,712,306,944,486,887,735,696,318,691,258,614,269,133,706,65,652,693,438,44,232,30,483,492,979,954,837,915,112,349,97,447,500,371,403,800,123,505,258,437,330,79,207,297,91,430,452,651,708,274,821,780,673,499,159,309,701,394,998,877,743,333,852,765,710,502,450,193,875,629,731,475,633,284,514,690,295,273,344,921,818,935,689,419,391,468,785,80,530,376,194,259,224,298,555,355,473,961,306,987,200,639,443,812,481,685,293,794,253,41,163,481,6,166,419,654,298,939,808,637,12,403,853,405,438,838,950,654,243,870,925,901,897,619,436,303,112,897,420,109,217,404,639,306,202,801,835,942,369,507,105,676,335,356,510,879,217,951,63,947,550,939,551,224,440,209,44,853,96,243,58,960,162,249,458,506,453,787,511,667,145,666,41,484,712,389,925,415,988,545,403,320,95,477,383,345,127,614,921,69,173,419,538,947,278,877,618,2,98,358,384,641,81,27,663,47,522,773,995,976,190,777,264,655,748,424,83,88,690,645,31,60,872,996,250,984,757,512,835",
    "33,27,628,950,190,852,551,677,500,198,620,232,198,654,594,134,847,947,779,991,42,517,548,436,139,781,305,974,476,846,962,179,383,554,710,336,153,0,636,824,690,230,321,896,269,479,951,740,713,401,891,908,530,467,603,945,713,693,486,424,981,809,67,431,167,780,782,171,630,686,553,631,529,221,534,823,274,171,891,556,815,211,629,863,411,924,11,314,622,566,267,49,538,922,572,198,25,996,101,4,709,793,475,259,993,442,605,516,305,513,77,733,808,155,501,836,573,553,460,908,702,707,778,893,600,15,228,458,223,735,211,401,958,285,322,850,236,91,809,63,12,129,170,51,416,486,27,919,494,798,244,93,879,520,994,997,507,324,867,274,897,852,965,614,129,719,958,315,57,675,583,153,511,70,350,320,730,965,498,993,232,544,915,394,226,323,943,210,974,650,309,539,615,992,825,645,251,49,769,381,678,640,927,612,933,908,489,341,153,435,877,349,573,388,530,125,784,871,603,334,853,195,440,692,68,268,496,947,373,907,826,270,771,292,219,935,158,992,418,345,42,884,1,331,944,318,903,757,249,797,847,74,408,107,397,120,151,254,307,418,832,407,533,549,560,418,525,393,280,101,842,108,366,91,974,265,982,857,234,926,560,188,850,770,777,827,609,240,25,740,167,622,655,658,241,419,147,441,83,229,352,645,83,919,474,696,668,278,882,662,475,123,326,151,595,620,305,241,322,4,917,295,791,105,674,189,971,434,930,896,850,358,927,934,105,218,673,794,874,580,507,211,284,759,737,241,86,482,99,505,816,212,143,949,549,42,611,385,303,779,930,105,329,376,98,29,258,248,600,901,109,510,351,601,313,49,874,910,39,881,576,475,451,130,314,923,657,839,580,286,516,329,688,352,606,10,358,913,564,463,574,154,977,833,977,973,255,786,425,946,432,169,782,678,383,177,868,988,270,776,486,848,110,79,279,221,83,848,590,3,546,972,871,697,387,190,819,684,864,867,15,161,927,222,973,851,618,170,822,343,784,44,664,491,418,37,207,932,881,812,274,727,735,600,288,708,606,387,200,244,484,923,732,80,613,176,458,50,278,198,209,949,966,534,198,21,849,161,343,474,373,807,225,685,310,117,701,77,950,555,973,990,639,185,490,990,886,377,228,255,489,42,709,648,267,39,706,265,988,788,791,149,613,690,224,701,954,22,795,443,741,369,188,978,576,577,433,906,848,81,226,19,564,38,9,726,881,152,201,219,187,337,250,854,948,990,181,622,753,756,267,603,892,731,658,875,259,629,757,78,873,97,37,817,938,718,840,464,638,666,772,299,650,349,282,743,808,130,236,472,881,973,130,565,232,887,452,667,389,980,84,799,406,183,297,444,380,404,141,330,268,379,374,963,204,58,486,292,337,530,748,737,44,812,65,849,196,233,107,810,690,35,855,921,951,821,483,893,768,187,350,985,52,657,507,274,686,955,823,903,883,63,234,398,43,281,819,756,992,353,263,36,599,155,646,865,452,797,680,561,649,500,318,667,269,513,742,963,729,852,169,385,398,120,595,408,619,471,301,862,846,900,930,242,574,216,425,110,112,144,851,180,377,300,132,595,497,955,758,991,164,649,354,869,309,461,629,743,652,35,22,75,809,199,307,298,386,479,265,511,911,890,393,857,991,155,770,405,690,842,49,599,339,674,805,810,716,621,329,974,130,257,598,807,857,629,835,118,972,277,654,904,497,707,377,434,844,886,548,495,352,754,990,567,696,267,522,254,874,612,228,775,364,101,213,241,799,938,284,669,922,988,361,992,469,342,435,515,394,795,508,599,740,924,437,601,4,407,468,304,399,650,889,228,325,44,917,416,690,652,761,400,502,801,797,140,582,799,153,679,664,38,260,597,595,664,947,784,830,323,929,35,112,992,527,265,640,944,926,118,317,662,333,40,953,731,833,273,834,872,927,465,908,668,611,310,461,274,36,509,701,36,381,76,945,605,813,924,170,923,52,767,773,893,700,730,156,401,439,581,874,871,579,894,857,9,739,980,757,543,811,348,389,291,730,999,767,201,872,857,469,307,773,111,556,999,128,667,208,543,319,602,266,312,103,294,759,721,936,940,495,214,258,531,221,2,179,460,949,512,599,539,790,487,635,770,650,92,512,655,754,228,441,728,259,468,105,447,432,674,171,267,686,745,947,541,615,411,306,251,506,798,165,817,302,145,652,351,260,790,493,261,774,830,60,787,359,121,885,237,91,440,95,355,215,188,750,537,763,349",
    "112,396,781,993,731,156,77,716,464,621,786,574,340,833,312,567,921,617,978,102,224,755,176,99,774,718,301,995,567,564,839,628,611,704,187,656,876,636,0,634,880,367,291,800,380,358,7,279,299,800,601,318,951,884,884,930,921,2,719,23,701,861,760,289,366,519,158,114,203,857,217,823,727,126,677,667,568,882,584,996,892,952,870,490,564,739,891,50,532,857,342,273,13,282,684,878,505,999,515,939,392,996,894,603,942,959,141,716,709,821,324,813,409,745,552,623,742,854,256,787,946,953,811,33,260,577,802,76,176,417,725,730,148,393,776,273,776,773,177,610,671,35,338,150,725,834,274,600,583,59,274,801,337,696,65,755,931,548,808,266,345,654,507,723,479,467,222,179,227,619,442,22,728,918,31,479,524,902,26,987,938,346,4,855,911,287,35,147,787,89,348,107,810,944,750,898,106,996,347,652,43,378,825,968,484,7,720,551,635,356,768,298,564,212,122,442,221,349,99,539,146,830,766,352,131,394,547,387,885,885,976,694,59,898,823,39,786,396,81,217,511,149,64,674,959,934,459,354,510,399,98,571,984,342,902,443,541,65,240,253,367,778,432,880,468,561,626,309,423,12,420,643,860,885,414,364,649,151,827,477,129,41,359,957,177,485,390,134,890,966,33,219,317,609,924,208,952,34,150,438,479,711,698,22,383,202,523,946,900,583,17,492,511,500,413,983,739,188,665,438,223,228,724,288,632,960,732,626,379,392,514,114,895,574,15,785,373,489,592,802,821,351,914,592,894,67,429,260,989,846,956,750,138,327,39,756,643,153,316,118,617,54,524,676,88,912,741,884,340,288,406,695,728,500,259,85,586,755,261,408,250,248,268,915,37,592,510,201,432,790,629,823,67,594,109,316,71,675,558,522,693,200,591,750,748,211,678,625,291,819,892,797,304,444,433,699,911,311,905,934,285,522,543,33,447,785,204,490,757,59,9,848,631,997,673,431,929,197,453,554,16,845,156,885,490,915,233,310,207,709,536,635,64,620,763,28,243,433,609,159,839,903,403,246,122,387,368,481,4,185,535,625,827,837,232,886,591,948,547,671,348,779,442,431,423,349,215,311,182,896,523,460,422,874,523,139,19,322,585,917,229,236,149,413,996,42,468,610,665,188,238,395,152,61,97,608,945,291,31,781,322,598,146,748,585,219,225,107,464,251,79,5,711,819,411,893,762,885,177,334,545,982,416,71,822,673,450,618,462,399,944,884,373,129,248,385,986,786,191,330,398,298,495,811,908,116,852,167,859,439,736,234,936,457,719,515,208,291,194,666,236,800,656,440,535,817,167,33,463,506,929,550,765,984,307,320,677,307,630,469,404,240,844,308,522,408,371,870,669,629,580,948,134,348,992,312,556,140,420,153,405,540,266,643,486,612,617,496,50,959,232,485,414,371,773,696,998,160,512,846,358,450,705,691,55,899,999,97,395,197,779,293,213,970,715,669,928,721,563,6,107,535,823,782,375,486,582,328,365,117,289,822,861,12,589,750,524,234,670,541,136,509,700,375,441,622,508,146,106,828,502,670,246,569,203,571,238,902,148,211,514,328,359,798,733,993,427,7,97,252,706,129,753,633,510,257,742,183,436,927,398,364,605,96,722,121,676,440,147,364,381,81,423,406,797,514,441,168,51,667,173,711,945,113,425,599,915,959,329,972,399,643,112,709,204,783,833,297,564,533,37,980,448,157,490,442,209,86,751,806,879,222,680,107,192,837,729,121,584,84,444,41,439,236,206,506,560,452,41,27,687,754,637,659,261,886,144,246,945,636,62,138,930,588,528,153,58,674,97,187,648,646,856,555,669,269,158,933,451,249,872,973,648,403,783,406,698,582,36,909,696,569,450,682,194,977,447,445,445,501,681,922,117,99,648,855,664,767,859,583,251,192,960,427,629,911,805,257,184,428,558,78,404,172,913,981,722,357,859,668,582,608,784,704,327,492,428,484,982,646,356,367,117,922,985,52,784,468,540,250,245,359,616,727,350,647,152,711,498,416,328,491,275,719,281,556,222,805,994,123,320,534,772,839,914,570,403,371,81,340,242,856,707,569,71,46,396,556,251,619,544,251,265,516,754,751,507,665,154,297,115,407,749,671,814,859,885,220,468,294,657,268,847,175,172,388,305,161,352,190,707,422,369,324,824,505,301,175,303,679,816,225,141,718,479,392,887,723,773,920,516,553,916,379,334,157,847,783,198,322,847,250,318,779,257,342,915,995,985,915",
    "913,183,698,821,963,676,475,680,742,731,202,24,54,38,273,694,91,842,288,120,832,736,848,42,3,147,718,391,959,334,732,311,885,814,521,507,804,824,634,0,825,126,977,599,271,612,352,534,649,208,203,99,308,108,1,202,765,370,611,451,992,42,846,387,568,305,771,189,985,806,458,44,833,585,442,455,143,507,274,717,127,365,655,510,463,481,928,350,52,800,134,649,619,981,365,151,906,251,184,675,609,74,208,997,359,620,706,473,770,871,918,164,952,170,969,549,341,1,695,160,119,462,297,646,396,812,944,339,6,665,240,906,994,598,714,433,448,652,864,113,383,317,977,628,332,611,740,167,875,997,60,217,332,319,501,609,557,665,68,53,600,486,656,772,336,431,27,552,865,545,106,766,862,266,806,575,285,560,899,512,614,280,499,888,820,875,224,866,540,461,815,613,391,678,88,782,656,607,300,996,520,831,400,749,815,615,93,382,778,406,231,761,30,582,168,732,608,655,539,264,286,42,645,706,207,58,506,606,177,501,176,421,539,636,35,968,200,91,27,890,359,658,974,842,33,910,526,122,487,237,501,742,576,322,503,444,165,695,466,683,473,104,920,30,265,624,864,135,181,248,141,170,962,10,85,5,531,850,301,200,296,590,358,713,898,103,844,66,519,272,229,948,315,943,39,13,602,338,14,919,208,52,311,653,622,71,392,939,593,889,826,197,89,746,125,365,958,4,252,362,627,158,363,263,399,776,951,324,775,212,427,480,528,584,79,338,425,874,430,342,861,441,731,87,886,421,254,77,631,331,535,364,285,598,120,612,782,593,834,29,287,71,255,345,973,98,15,82,4,963,931,818,616,960,441,303,642,211,91,77,320,862,975,238,397,47,422,735,51,489,170,340,623,94,652,683,901,414,244,196,292,611,735,939,365,474,677,730,108,639,944,833,101,916,326,754,777,431,61,223,337,91,66,553,123,325,7,603,143,434,378,740,668,780,347,680,845,451,193,517,743,716,105,129,321,88,411,666,622,516,927,196,441,735,660,169,96,792,898,118,53,479,778,872,178,641,577,525,846,293,37,19,351,7,229,390,397,857,898,210,184,272,551,792,276,288,578,913,392,608,320,978,695,44,130,283,524,844,415,30,500,781,137,973,422,338,36,406,597,48,61,925,839,641,108,198,271,578,863,40,135,222,962,807,500,544,390,40,391,303,620,129,459,516,879,752,776,647,932,764,254,918,709,190,241,827,818,366,170,847,20,26,831,820,351,116,143,459,645,824,562,229,301,194,746,14,236,380,892,395,785,622,200,19,260,970,841,482,389,767,576,318,396,421,31,31,44,913,882,331,742,856,208,466,456,818,14,873,371,319,348,421,398,13,235,330,402,561,237,608,332,385,342,557,693,517,287,377,79,208,236,6,811,282,893,427,297,309,292,552,986,373,450,81,88,755,520,21,208,914,617,550,517,907,830,79,529,386,768,739,999,214,905,576,351,863,234,616,952,925,91,356,656,131,194,848,243,828,175,772,786,528,82,576,959,588,325,740,986,516,381,373,702,551,60,167,20,279,483,245,219,103,865,950,584,914,118,481,210,664,151,158,538,986,920,462,201,496,638,306,806,638,950,826,690,489,283,569,662,81,555,571,45,244,245,208,958,111,802,80,415,476,535,585,637,767,41,595,837,799,931,997,304,693,687,802,671,33,963,768,610,228,932,762,727,844,285,731,449,470,578,644,799,133,300,159,767,954,540,446,970,902,349,185,615,362,43,635,150,266,398,405,371,669,192,50,841,27,25,45,301,361,216,335,58,712,768,722,253,116,346,217,523,500,702,808,991,13,110,740,440,991,448,584,173,437,533,180,951,695,266,494,740,393,637,437,898,117,778,779,363,635,720,222,817,637,63,315,583,245,925,549,294,765,884,768,983,854,170,119,292,915,505,158,792,725,178,581,895,915,534,393,971,922,229,612,307,634,859,619,37,11,955,6,663,288,879,560,137,570,82,605,103,781,648,202,869,124,139,916,811,556,553,297,344,387,85,998,581,821,440,952,885,110,188,661,388,650,197,453,999,515,145,507,765,815,832,248,128,192,668,126,210,737,217,211,544,382,568,409,502,162,16,184,248,822,343,19,620,625,828,298,599,35,813,55,435,579,369,89,438,845,578,643,217,522,991,335,987,80,9,979,415,565,531,201,336,98,870,809,810,234,619,95,197,308,731,204,17,367,260,903,317,302,318,912,792,338,801,523,585,388,179,511,281,220,154,887,587,966",
    "548,396,361,818,186,904,499,907,842,396,218,310,504,195,841,759,385,393,265,116,281,371,393,651,896,197,621,559,492,148,325,665,405,512,922,41,859,690,880,825,0,208,684,473,706,130,446,885,211,143,367,99,258,196,77,706,32,895,875,793,422,661,77,655,499,786,700,126,53,775,324,233,156,651,39,229,893,487,98,771,439,972,401,737,425,70,901,151,837,763,490,232,140,518,706,934,198,959,106,367,958,414,627,484,431,462,407,368,129,251,454,698,627,101,420,364,350,759,154,443,210,805,254,580,759,791,370,965,916,684,499,760,120,584,852,104,556,536,995,770,834,303,239,335,915,776,781,703,436,557,48,252,480,451,388,290,354,259,936,47,347,750,858,901,702,412,187,114,433,472,868,402,445,154,769,95,858,683,342,779,77,797,641,377,304,634,237,610,235,262,500,581,253,592,167,122,347,834,892,499,342,628,922,211,673,46,837,822,877,873,748,556,314,713,762,27,949,253,468,502,269,4,172,641,497,365,168,648,778,637,53,884,157,618,477,555,641,500,160,745,798,892,289,862,659,244,34,127,545,346,116,761,826,202,478,521,174,217,732,267,988,502,509,703,586,23,333,889,666,321,590,210,910,505,274,283,271,236,96,383,881,293,798,889,703,96,779,575,309,23,938,756,586,604,634,613,379,21,248,676,642,40,275,392,110,270,266,415,270,204,496,995,427,357,598,855,556,395,561,897,330,305,939,598,56,559,916,737,484,620,937,229,74,827,35,789,581,358,252,947,154,96,924,965,563,968,900,938,774,56,892,960,807,262,663,172,888,922,835,834,879,92,415,309,290,862,464,525,232,804,893,755,320,442,496,979,576,982,591,856,537,185,70,584,812,282,510,389,663,46,289,583,851,181,804,999,988,833,975,337,755,660,62,17,569,741,710,158,969,863,979,315,701,351,296,446,569,475,652,49,738,275,709,181,849,306,608,561,106,315,317,734,306,891,530,629,16,531,139,677,932,640,295,527,504,180,579,823,552,797,640,771,42,432,829,855,569,432,988,8,924,538,801,243,198,262,632,714,824,679,898,166,705,380,664,134,959,224,577,157,428,453,468,96,801,192,409,321,334,55,247,134,618,180,603,503,960,394,165,574,207,133,504,76,983,851,819,904,221,728,187,297,979,662,460,86,693,114,205,284,300,401,72,222,357,101,265,527,715,388,354,462,268,485,875,650,101,224,406,467,708,947,874,932,630,81,126,217,764,44,965,138,506,717,862,967,172,390,485,244,743,134,931,241,639,300,164,580,92,412,187,562,509,790,628,716,276,323,807,270,624,547,419,361,498,177,224,54,662,827,504,81,462,12,449,292,959,991,253,865,877,898,162,886,214,966,641,902,828,446,419,895,480,255,793,620,490,682,149,750,874,412,967,200,797,824,631,406,182,202,135,478,378,770,661,229,847,448,95,818,398,889,496,67,520,163,352,230,157,492,302,440,256,204,564,794,755,650,548,580,532,128,194,408,924,710,830,721,914,897,884,389,573,772,335,15,790,597,935,894,592,404,582,471,201,786,918,951,647,217,91,344,717,640,820,193,48,801,172,885,928,973,739,963,365,828,81,484,888,322,359,331,544,95,847,274,369,668,853,373,306,622,514,313,235,202,468,821,954,935,813,420,190,890,159,191,254,706,415,987,587,578,402,944,428,926,993,863,81,603,746,269,723,697,278,697,114,57,353,629,273,993,456,103,696,987,843,893,650,731,460,900,180,670,665,465,359,233,895,22,248,237,226,101,612,884,424,611,610,216,534,108,92,506,567,321,912,668,543,307,510,334,685,63,182,563,245,930,169,230,649,381,796,56,814,832,468,196,519,923,578,541,98,68,233,441,864,87,807,219,440,673,369,582,356,733,732,869,776,92,859,461,870,105,143,958,775,82,592,634,820,617,303,437,492,40,122,166,517,95,964,381,727,216,333,496,934,979,962,880,604,78,969,368,201,352,654,604,232,719,151,352,62,508,795,333,775,590,875,93,89,584,417,556,186,391,502,208,101,88,619,557,87,413,52,542,991,352,91,746,302,562,92,264,588,507,924,148,360,836,931,64,844,90,668,406,779,7,892,828,103,529,335,90,593,941,448,200,406,550,885,788,105,887,943,990,833,868,898,141,293,759,470,508,289,698,248,754,756,867,526,550,308,263,858,648,475,593,111,662,420,719,298,482,300,233,750,110,309,433,769,527,819,727,826,754,540,50,888,21,935,369,617,137,359,14,45,687,111,543",
    "902,822,121,906,586,251,367,936,641,65,529,982,549,850,941,761,914,142,48,344,356,820,716,318,861,993,641,338,596,810,290,594,848,285,379,260,472,230,367,126,208,0,641,88,268,757,539,283,405,356,594,265,179,605,753,735,959,145,736,685,678,902,215,835,136,218,592,401,512,660,310,625,936,828,67,433,37,868,126,352,51,923,546,496,508,709,892,44,666,464,693,191,79,536,635,232,248,99,996,562,90,971,393,687,743,951,744,70,989,380,456,44,219,607,604,7,362,876,613,569,155,380,962,718,84,406,524,90,82,832,16,679,527,474,46,422,122,735,105,411,810,444,357,720,536,495,546,589,468,965,937,975,150,609,880,862,813,554,642,556,693,362,889,247,961,61,195,6,895,414,687,987,847,627,323,243,986,523,495,36,515,480,256,490,731,879,192,813,356,780,266,472,345,339,742,463,639,232,418,656,991,99,735,228,384,443,668,337,33,842,437,319,535,264,930,119,61,395,576,639,403,58,632,873,865,112,499,835,491,314,629,246,74,315,464,540,832,907,515,58,561,790,423,855,127,717,71,408,410,144,14,464,73,191,728,570,883,699,746,116,177,647,121,611,326,743,94,933,207,216,860,295,748,556,305,673,961,952,913,697,491,397,376,702,256,13,788,386,467,299,699,794,796,964,361,597,763,173,105,757,802,42,773,968,619,796,485,291,259,26,106,53,289,585,811,219,968,572,22,275,873,128,57,739,201,995,176,752,863,354,981,954,188,488,144,303,599,328,128,192,194,149,920,741,864,414,829,384,535,149,112,790,466,975,898,855,179,785,150,412,84,461,96,960,682,936,278,326,932,561,136,628,823,898,354,564,399,693,678,512,509,992,105,250,78,375,811,706,200,69,968,49,559,731,110,526,472,824,38,596,103,933,219,648,321,415,95,210,303,552,215,991,562,775,991,722,998,361,902,495,539,766,798,754,986,556,715,911,88,953,981,769,487,993,44,315,765,701,478,819,595,114,568,874,485,460,646,646,736,920,528,84,867,598,775,950,389,936,565,891,382,799,217,27,248,521,254,144,88,586,544,609,552,938,205,97,579,796,158,576,967,68,611,64,938,769,658,410,187,28,156,573,746,346,42,343,800,597,94,998,347,573,56,121,283,795,212,25,581,678,179,858,362,168,504,503,127,643,329,791,743,749,417,609,59,694,637,365,973,685,509,24,929,890,335,472,775,455,721,908,530,438,27,120,603,397,528,615,848,576,752,353,145,41,201,754,366,233,190,133,704,154,839,993,81,670,918,152,955,464,108,32,60,243,793,608,162,377,238,957,7,986,286,862,280,791,146,670,857,569,525,780,753,772,117,575,576,398,429,34,661,756,890,90,540,922,132,731,181,142,688,884,225,254,106,215,718,257,403,710,210,206,182,140,422,626,937,465,273,127,136,816,717,648,662,650,284,30,410,689,147,714,914,775,347,855,130,141,684,149,56,808,930,24,19,745,409,625,326,335,779,522,144,554,405,551,454,698,408,79,654,73,317,478,519,829,887,441,670,212,24,157,969,20,993,29,690,553,664,989,634,360,547,500,373,860,616,734,949,944,717,64,977,642,616,119,449,655,490,619,396,737,585,304,446,45,489,644,595,785,712,854,661,428,625,218,540,891,252,988,244,76,563,391,456,582,644,25,626,858,392,253,759,512,258,333,590,279,507,427,830,883,10,138,779,917,441,412,809,611,601,720,401,84,525,994,54,977,709,347,701,765,765,217,798,331,895,585,877,818,663,727,719,386,367,913,2,326,50,682,167,404,226,723,95,760,504,935,632,663,733,768,531,392,274,490,377,217,76,385,559,789,673,377,957,517,866,544,749,783,215,36,163,14,508,374,519,932,866,181,294,752,121,143,348,546,135,536,320,669,141,986,947,587,886,959,213,41,180,585,959,239,413,424,293,509,825,886,66,312,580,621,963,737,992,814,521,297,169,451,892,156,581,694,527,714,307,341,344,120,486,623,298,687,809,488,66,413,295,109,889,545,624,199,40,59,12,243,884,681,786,649,804,565,470,130,94,611,746,558,885,40,365,655,471,755,518,466,489,27,766,529,912,139,560,603,663,380,568,23,589,470,52,934,597,19,785,693,706,603,812,986,104,713,600,912,328,52,46,960,55,967,143,802,530,591,613,369,584,115,651,887,600,677,655,705,823,414,643,758,369,530,975,451,665,298,232,592,216,707,451,754,634,224,302,356,205,159,665,688,517,235,805,526,420,151,476,415,224,921",
    "793,81,216,58,52,841,648,958,923,529,921,609,491,264,480,88,469,840,517,457,885,301,954,186,342,337,566,861,239,112,11,340,699,329,983,375,364,321,291,977,684,641,0,438,867,729,43,24,58,214,419,195,449,860,328,109,866,403,359,786,574,682,838,848,727,170,867,525,171,636,152,930,705,571,804,990,447,128,857,950,239,389,956,670,67,640,751,59,810,1000,851,592,670,790,230,549,991,369,500,465,173,35,596,698,308,651,823,119,288,453,856,318,902,452,726,929,920,15,6,886,947,37,423,766,88,400,867,422,458,686,588,502,536,635,927,12,252,241,448,926,884,965,307,455,197,511,132,303,647,83,686,152,979,611,650,642,609,75,834,721,980,235,694,456,901,677,505,281,109,478,337,629,322,600,4,364,117,989,904,902,532,790,490,441,876,128,761,219,220,214,636,560,410,288,828,733,576,928,454,107,106,424,160,158,355,273,835,850,824,852,53,4,701,166,777,119,801,238,284,687,345,309,125,643,413,320,278,208,20,165,356,221,339,812,817,525,439,265,795,646,172,696,572,108,177,557,317,428,310,509,232,672,3,349,767,589,829,671,994,453,876,312,481,565,270,28,863,710,861,779,944,106,322,853,120,648,675,346,779,179,388,934,904,754,499,527,603,775,644,335,934,937,726,68,723,156,794,32,388,63,458,300,228,720,566,979,770,351,696,292,589,104,136,10,843,833,118,679,32,212,789,501,326,374,934,678,84,943,38,311,59,190,78,224,544,179,984,988,625,953,31,987,876,239,244,111,103,281,858,659,426,331,234,693,218,873,341,559,913,494,923,213,280,241,636,308,693,417,653,837,685,18,16,723,623,74,891,62,614,630,300,735,638,883,917,144,515,948,643,730,535,790,308,632,114,897,148,355,198,476,148,445,483,273,693,331,486,615,490,508,607,254,260,930,169,462,545,711,889,291,244,104,790,824,38,663,464,676,3,91,93,263,263,895,335,960,35,326,877,98,494,574,801,713,285,332,652,723,774,525,275,310,23,322,346,194,41,635,190,968,597,10,132,406,116,220,845,217,729,181,825,315,937,518,539,130,436,38,583,428,430,173,865,208,967,169,321,814,262,266,33,667,89,934,864,29,870,801,96,197,407,577,919,865,455,185,864,747,434,923,155,642,153,614,316,925,266,626,12,600,644,441,198,37,268,662,779,618,537,110,77,297,889,830,980,191,263,42,498,21,120,148,273,316,427,725,830,937,127,241,966,96,153,509,951,171,268,188,386,223,150,318,747,838,638,790,164,560,476,69,121,756,53,385,207,830,533,250,992,478,210,947,516,808,392,911,139,911,157,972,274,191,185,265,387,182,151,531,290,424,700,116,410,162,42,622,557,979,74,705,891,791,978,108,728,145,795,389,602,21,748,25,410,306,600,419,919,445,905,900,526,100,606,970,860,918,709,928,105,526,138,361,181,843,11,927,22,26,972,326,268,10,34,262,963,173,312,422,521,477,888,787,973,265,382,519,820,656,765,353,214,91,983,739,990,520,55,543,829,418,323,677,96,516,856,46,416,335,881,911,814,247,82,128,110,704,954,43,846,709,228,792,227,337,373,756,7,605,311,622,649,863,789,131,756,500,988,26,985,944,154,344,94,586,917,522,588,498,791,928,550,664,354,345,668,734,718,176,800,233,610,313,777,332,358,650,109,609,257,607,48,557,520,54,85,153,345,170,350,908,304,874,699,849,929,223,998,896,725,101,392,887,661,737,740,391,764,190,75,266,720,258,916,395,634,347,403,504,297,517,939,569,244,813,655,831,303,81,496,202,320,522,131,242,57,919,885,56,231,693,565,974,852,603,87,573,419,750,797,624,421,364,301,530,120,729,804,787,836,417,535,278,346,445,222,613,864,784,309,403,439,986,643,854,944,2,522,274,717,834,215,821,936,860,553,845,606,556,891,167,10,208,924,153,839,346,884,361,201,476,271,37,992,942,918,140,895,502,306,137,983,527,553,639,672,515,628,652,903,63,896,150,313,620,603,616,312,289,343,840,986,399,161,615,795,640,821,985,465,232,397,698,544,350,426,972,644,3,850,506,293,60,684,21,463,598,217,782,640,138,295,387,690,209,81,937,321,706,556,54,824,220,838,328,432,910,628,164,98,986,239,110,1,386,501,829,622,722,195,955,9,154,776,111,819,857,395,766,976,655,25,224,83,735,689,461,587,180,840,197,636,537,653,361,796,225,415,27,427,97,866,650,824,478,142,20,675,715,595,590",
    "563,117,819,492,700,175,899,437,117,612,751,630,280,354,554,361,58,240,758,957,459,912,364,953,433,764,324,311,16,360,445,115,379,674,956,99,502,896,800,599,473,88,438,0,752,566,320,297,213,597,810,563,743,121,71,497,4,551,788,167,73,421,90,402,435,989,509,423,285,285,698,648,871,1000,896,534,492,148,342,385,816,986,443,449,219,69,990,606,903,788,753,132,756,582,918,980,205,220,484,395,473,318,27,914,783,912,51,620,538,16,579,327,626,946,406,223,898,412,360,848,981,544,83,625,335,423,636,969,945,865,187,890,794,538,57,90,255,919,455,310,539,738,787,954,994,851,803,469,285,307,790,998,121,535,249,38,180,235,899,757,636,955,498,757,535,215,330,428,608,139,653,119,126,830,688,377,479,262,138,243,536,853,73,139,711,431,760,357,900,742,761,540,240,852,239,91,821,351,558,85,74,195,682,873,597,107,172,561,908,721,865,87,193,946,977,32,375,605,205,739,714,179,339,818,714,334,678,294,320,638,901,430,331,852,4,493,743,435,390,237,298,202,941,215,422,883,444,173,456,828,248,750,343,810,538,273,442,502,130,458,568,779,188,502,920,786,224,370,469,619,603,780,73,897,528,406,579,400,871,756,354,412,728,608,484,824,606,292,902,47,463,433,526,348,563,88,714,465,462,91,435,135,565,232,769,320,913,80,196,642,796,247,176,21,446,610,156,8,22,654,28,225,632,259,478,189,240,705,768,958,9,929,172,99,163,411,619,28,492,818,327,342,303,241,205,440,614,55,713,517,262,398,974,607,928,755,628,308,396,408,723,264,883,84,699,231,684,975,765,110,513,53,399,395,535,712,386,27,977,203,694,629,286,772,740,899,676,382,495,300,870,554,734,919,98,669,457,539,609,416,62,236,627,168,188,246,791,697,283,945,285,472,95,749,97,423,665,321,446,497,692,893,428,728,613,490,75,627,23,991,267,474,83,75,637,390,130,512,133,846,223,72,448,84,957,401,497,954,711,560,608,849,432,352,271,916,682,810,805,761,330,912,313,456,894,583,725,875,878,412,54,145,826,114,574,338,389,712,913,208,840,159,346,359,892,476,689,264,537,986,249,947,947,17,634,974,225,540,441,289,202,519,356,974,643,253,989,422,141,621,362,707,940,356,929,499,193,751,506,427,255,792,283,414,319,82,446,302,671,293,346,881,557,386,961,933,292,22,619,313,819,231,293,710,47,311,453,678,459,935,191,230,220,464,354,753,987,689,362,632,867,174,919,355,702,499,140,405,273,426,337,661,551,794,218,658,973,800,185,105,592,708,625,201,198,600,19,3,940,856,311,288,895,749,30,68,83,404,604,359,699,603,991,841,613,415,671,841,970,851,846,516,694,602,454,538,534,279,433,446,285,405,898,257,846,776,535,602,567,306,551,406,216,198,972,643,591,549,242,978,82,516,679,543,849,121,300,520,33,19,343,964,876,503,283,494,768,742,876,21,284,882,872,639,460,527,219,84,600,640,592,200,345,35,174,912,517,298,996,370,740,662,706,736,312,728,967,567,767,855,946,178,266,518,30,742,217,877,535,270,566,739,428,113,418,183,341,123,232,449,508,859,255,803,813,94,815,443,297,602,913,106,941,268,440,680,14,982,215,560,649,489,230,136,837,970,738,896,559,460,140,549,346,762,735,165,361,495,130,756,876,508,980,765,382,146,249,968,936,821,657,492,436,746,641,418,38,168,977,575,271,288,20,261,526,801,364,228,229,104,132,680,833,457,918,939,335,342,709,609,743,370,15,601,50,710,685,267,399,332,779,594,844,26,756,878,122,10,297,62,255,747,425,824,611,14,309,961,572,226,526,745,59,172,850,286,934,769,858,809,988,861,763,982,616,919,847,988,897,764,404,743,75,531,951,739,303,493,89,416,583,336,155,599,103,455,242,55,840,881,326,717,839,600,785,121,947,211,28,993,366,715,513,7,476,362,108,742,906,595,810,154,512,889,683,428,932,930,996,41,661,711,67,562,851,269,692,400,613,337,524,858,378,27,765,702,96,696,801,917,761,814,286,416,742,690,498,840,487,911,92,225,484,452,895,244,689,370,817,260,175,183,845,167,133,58,316,855,926,95,303,426,151,862,493,87,845,155,876,980,350,314,667,798,899,973,82,948,833,288,811,491,288,198,727,141,973,384,270,793,379,9,59,286,252,744,450,231,313,674,374,308,253,974,56,568,602,79,1,5,141,815,371,459,23,997,387,259,91,967",
    "318,114,304,863,83,504,319,991,654,196,224,633,220,217,262,216,706,51,431,671,93,686,180,242,219,233,468,617,265,273,78,195,648,956,176,406,573,269,380,271,706,268,867,752,0,959,664,268,830,946,536,541,953,851,468,50,828,534,94,452,300,576,642,695,641,548,157,42,105,384,508,566,880,729,329,454,522,408,426,831,42,622,65,313,472,942,656,175,217,421,684,642,336,912,475,987,684,399,958,719,73,115,525,35,488,511,122,228,764,343,509,211,266,645,199,735,219,290,502,515,497,80,938,995,168,764,656,54,18,108,515,153,743,974,670,776,316,274,53,948,669,772,43,730,583,614,853,584,924,581,589,775,168,642,427,461,176,515,136,809,725,984,818,357,269,433,177,910,110,848,188,661,929,533,116,307,671,111,255,8,587,945,438,397,840,935,140,230,181,299,603,714,486,954,737,747,923,794,828,343,141,429,953,311,264,146,938,416,351,239,510,4,298,242,1,602,813,577,738,626,251,950,811,275,476,945,447,702,53,927,442,252,681,213,403,245,344,789,735,331,672,470,510,396,269,717,282,145,884,561,715,248,498,439,673,59,776,307,379,603,501,552,977,34,551,304,222,84,561,586,79,944,243,261,548,661,801,277,663,393,291,472,52,707,244,869,327,130,569,302,920,349,220,517,586,966,907,225,519,921,608,531,25,482,557,465,97,680,64,837,758,33,396,551,218,715,718,547,741,70,552,866,472,33,715,1000,899,816,371,490,251,208,34,327,176,212,578,242,211,100,44,562,236,635,55,703,600,533,699,791,952,800,945,377,615,740,849,476,444,750,445,425,66,2,878,495,665,638,383,402,73,420,451,153,280,489,892,842,973,579,625,204,228,170,611,683,510,364,637,890,607,502,3,767,646,161,260,909,403,641,342,303,641,63,765,167,790,488,276,414,873,389,500,227,746,74,166,966,541,331,686,447,207,69,152,396,958,3,190,17,386,205,744,746,517,132,381,750,412,908,183,945,813,902,34,357,341,915,969,325,606,24,257,858,721,145,776,449,35,513,65,34,261,453,571,922,1,828,775,464,68,332,874,174,474,273,18,835,218,562,993,267,329,294,940,976,875,499,783,101,809,497,703,437,539,554,605,879,992,124,978,553,679,66,338,463,321,494,190,209,841,413,269,769,158,50,825,458,281,507,207,610,504,120,975,448,193,826,978,134,687,283,740,768,835,810,531,652,380,383,172,171,818,676,578,207,447,469,787,326,649,348,138,116,591,271,564,397,416,882,938,798,491,609,869,21,483,175,944,591,765,301,737,252,541,226,442,286,932,425,570,575,188,493,112,806,869,637,968,335,874,691,377,610,134,91,784,830,45,714,563,664,277,429,89,322,207,605,405,307,162,77,452,411,995,181,386,648,983,896,684,670,559,375,156,986,463,642,143,433,13,883,475,67,384,932,96,36,884,308,929,402,889,211,474,579,910,285,499,419,445,996,286,897,910,804,90,764,962,724,31,131,134,741,701,291,864,372,662,585,625,70,224,24,693,83,555,878,303,475,827,613,825,564,193,60,601,644,592,796,830,601,969,960,571,470,507,917,631,613,234,814,666,174,704,456,198,162,196,187,311,984,778,627,245,309,346,636,439,484,188,541,655,444,568,935,696,464,849,446,408,27,571,305,801,96,546,911,474,592,32,840,716,427,302,472,600,908,619,846,420,77,68,430,280,591,108,46,76,993,272,107,868,188,675,704,894,291,117,485,781,145,781,124,287,505,981,928,29,949,835,5,16,598,706,735,234,205,48,794,346,7,283,217,783,358,559,289,584,472,182,139,333,764,74,993,633,764,96,478,980,92,493,262,128,635,429,517,299,924,120,822,317,730,907,120,33,873,982,137,834,544,128,550,59,567,481,140,499,972,297,922,415,392,355,922,318,381,753,763,371,334,506,3,154,805,34,826,92,109,503,618,315,24,388,175,628,772,988,790,237,131,679,122,274,491,134,51,812,213,887,894,841,231,384,350,779,181,355,325,386,130,864,870,954,983,919,937,474,191,156,344,82,625,489,173,351,244,445,895,84,580,856,256,805,436,915,703,873,447,41,345,386,100,703,603,376,121,657,701,383,165,716,527,532,693,187,379,883,187,22,742,565,385,823,379,35,654,338,702,831,283,814,914,520,2,352,394,299,542,487,282,921,587,94,907,134,336,453,755,488,731,909,77,387,756,974,803,259,715,72,43,65,399,404,226,342,146,191,576,723,305,125,831,447,465,811,474,714,536",
    "311,950,475,955,981,116,175,17,955,130,543,802,347,256,675,196,858,446,775,801,317,86,31,868,528,377,154,37,809,168,904,834,93,121,705,593,351,479,358,612,130,757,729,566,959,0,6,724,543,988,115,77,57,895,461,937,927,240,577,446,911,730,121,10,666,706,829,350,574,223,597,242,84,744,143,320,905,56,970,226,730,847,430,181,909,747,484,314,293,88,145,787,829,321,109,713,80,751,677,343,149,465,147,230,414,359,135,152,210,493,857,226,756,687,951,319,492,302,645,746,643,244,891,93,65,314,731,612,946,72,363,993,738,322,149,146,57,563,271,613,419,245,753,322,614,785,24,180,286,790,673,665,709,70,434,178,251,435,978,154,381,586,216,602,24,14,203,459,250,873,932,820,282,884,361,649,824,783,740,139,210,637,573,876,580,531,725,977,958,190,547,14,329,885,23,282,627,863,816,542,764,504,221,711,465,898,568,459,392,282,952,85,994,832,846,627,951,725,172,456,241,270,277,755,743,73,551,300,755,858,909,418,729,676,529,840,799,718,130,363,119,937,484,974,572,548,895,771,477,444,416,548,839,866,944,507,960,391,601,922,828,419,384,92,976,986,914,793,277,790,936,846,947,26,911,872,39,137,827,227,82,664,862,470,578,452,784,89,205,896,911,88,506,62,626,653,887,498,963,44,929,638,867,301,268,150,723,928,922,333,41,687,161,336,672,688,647,973,201,692,176,825,270,716,643,71,241,974,881,835,551,747,718,705,917,824,908,286,391,905,854,697,123,46,506,470,522,883,70,615,972,501,869,404,331,689,719,306,861,678,27,942,66,603,514,645,533,337,44,185,681,156,292,251,534,834,388,763,890,530,617,827,800,151,226,335,884,395,86,101,871,366,23,359,757,174,811,828,361,53,21,988,910,802,529,834,831,833,108,560,317,917,453,305,148,917,733,36,260,498,913,757,319,87,203,304,311,955,708,388,739,641,126,124,16,706,582,60,24,904,846,820,606,823,932,6,935,715,705,142,41,143,419,373,992,13,617,518,148,624,292,293,459,640,59,644,23,176,109,381,43,142,36,542,61,990,153,414,969,190,255,554,767,523,74,259,501,118,735,874,261,643,629,478,483,32,2,729,440,431,258,669,708,679,559,336,384,487,400,510,910,700,704,346,686,953,944,617,935,448,263,935,77,915,74,441,870,110,28,878,962,180,203,776,773,609,6,405,235,168,80,848,211,782,282,716,656,956,360,886,94,469,941,188,673,968,298,843,892,363,710,400,378,477,71,656,610,427,635,946,521,363,873,144,176,939,11,58,150,564,31,717,480,300,58,722,848,176,89,760,589,428,232,345,89,114,487,655,584,174,802,543,178,778,811,925,457,322,19,358,8,664,192,674,381,107,947,911,338,804,908,510,710,190,16,581,804,747,615,408,32,570,261,523,998,181,725,396,920,18,541,661,670,407,124,243,513,419,336,935,965,684,691,299,142,711,209,33,826,788,798,818,765,900,730,261,6,907,751,630,334,473,346,846,184,432,276,921,908,553,781,771,148,295,613,967,801,39,375,36,836,386,601,486,26,153,699,768,631,653,471,537,758,747,652,476,768,612,259,861,398,548,157,35,332,27,896,514,736,453,561,463,958,21,598,333,843,906,837,250,916,81,325,891,145,790,621,70,577,219,858,240,911,675,450,774,538,126,126,14,258,50,909,486,560,706,254,906,653,320,758,310,195,663,85,801,444,861,193,108,288,307,959,796,215,4,731,332,14,895,604,178,401,389,38,787,684,902,638,104,562,570,942,864,352,878,5,251,824,828,8,801,595,321,232,454,41,435,898,904,233,121,130,333,713,483,452,663,449,671,832,207,86,72,305,743,46,953,542,458,196,204,795,714,772,417,72,29,191,553,423,285,461,937,404,92,690,979,994,65,671,146,97,767,510,451,768,73,385,740,738,750,618,518,815,40,402,955,619,107,643,608,126,628,431,727,993,807,985,651,724,361,772,444,159,782,965,407,517,721,815,159,157,169,280,577,838,249,328,600,859,980,975,975,686,920,434,538,465,727,125,671,646,368,712,625,630,921,607,760,481,863,288,260,848,850,384,303,219,337,58,895,806,116,307,505,770,484,881,135,325,893,365,675,255,958,552,614,389,352,225,533,683,581,444,260,786,5,542,582,5,308,25,327,226,693,452,890,620,926,850,669,825,259,254,163,989,707,164,332,649,514,424,719,447,257,246,252,400,820,497,172,307,4,326,542,846,547,733,647",
    "579,786,74,503,733,242,223,26,690,591,129,420,268,39,654,886,307,53,101,139,449,206,116,322,849,70,792,53,732,723,992,240,218,655,159,307,181,951,7,352,446,539,43,320,664,6,0,878,543,871,960,864,898,314,371,263,859,100,427,343,591,541,42,786,982,982,776,774,430,83,332,530,357,385,245,895,887,445,716,433,990,542,205,931,693,618,140,524,813,808,204,704,685,599,528,963,262,337,700,766,965,673,34,159,145,681,634,789,656,554,669,349,760,823,766,99,71,353,800,267,616,256,134,286,153,300,546,78,996,117,304,54,884,808,617,47,932,767,571,391,768,694,86,639,392,76,397,674,603,216,723,812,155,343,514,993,596,976,120,306,190,680,666,3,435,541,678,888,803,649,94,961,20,760,979,643,888,384,917,802,35,346,274,146,663,913,105,838,348,152,543,956,811,419,879,481,197,514,765,525,665,134,514,295,493,686,731,354,987,857,485,475,966,706,21,841,382,665,533,184,432,345,677,493,111,129,89,920,917,515,346,711,552,85,390,375,715,829,995,600,237,803,558,395,736,861,198,477,843,470,255,945,964,888,883,977,64,326,999,979,499,268,457,551,482,113,23,556,759,87,524,100,114,793,221,555,525,846,442,998,160,841,322,986,807,877,398,542,815,244,300,747,49,110,11,703,193,537,161,655,223,437,928,978,494,199,340,31,45,592,92,899,330,962,889,10,409,517,48,994,932,850,507,244,356,695,70,62,397,723,40,977,731,652,768,698,485,725,92,317,532,165,133,481,807,137,592,103,263,820,618,84,799,557,194,527,872,459,100,968,916,110,209,18,793,777,503,242,297,588,671,948,464,638,590,629,577,494,680,560,681,393,182,551,57,125,933,753,268,456,126,557,891,444,401,659,158,24,392,551,523,437,550,796,244,289,355,556,906,173,208,782,801,563,173,254,515,57,487,603,234,524,196,770,505,716,240,974,827,309,678,598,165,188,60,389,180,167,842,111,310,631,451,231,969,164,150,917,757,801,354,808,319,590,560,27,349,676,314,960,774,562,790,139,564,939,992,803,684,149,474,404,159,63,228,43,879,334,771,82,970,595,872,870,932,481,165,100,426,482,383,579,803,147,310,193,881,372,840,159,820,472,2,18,810,762,624,731,287,293,642,366,485,718,836,135,584,154,729,142,987,601,281,54,42,891,614,815,3,144,31,700,75,618,281,424,450,624,602,829,752,235,428,331,660,512,841,376,789,999,979,452,767,733,243,742,479,149,598,694,774,13,582,325,570,976,19,950,172,454,831,487,146,646,625,869,909,378,772,806,807,815,923,903,511,361,295,363,888,546,645,428,270,618,386,128,185,287,223,21,652,174,69,770,119,642,467,381,430,551,513,367,956,989,594,106,559,257,170,794,808,328,691,889,775,781,90,283,322,78,315,228,539,503,195,117,525,670,203,576,967,886,454,51,35,348,832,965,291,942,969,78,222,42,565,503,400,516,517,283,564,813,840,677,259,273,930,958,691,129,964,957,349,243,783,862,360,453,996,844,882,319,639,635,568,911,520,260,964,959,729,160,900,104,649,779,908,159,389,887,959,857,352,560,490,354,584,848,749,292,442,852,85,329,875,435,893,135,364,259,886,710,152,452,562,755,664,654,635,100,575,610,44,723,451,593,853,904,580,555,141,299,21,616,686,292,503,932,779,807,317,4,27,837,437,95,446,811,86,935,125,111,763,561,924,336,540,715,133,596,806,884,950,51,598,617,332,684,736,970,699,867,428,175,377,309,161,257,84,818,607,460,482,198,70,936,548,846,17,818,373,318,192,893,13,79,821,455,624,106,901,432,793,302,933,787,807,627,998,668,175,860,130,918,729,529,216,280,148,142,445,143,686,437,40,885,938,414,544,558,381,286,977,671,754,957,687,776,272,957,480,732,150,847,238,233,902,301,897,728,238,493,999,171,810,826,840,815,623,183,950,109,15,322,552,861,432,713,615,319,281,543,175,606,388,863,98,886,20,253,996,449,890,800,700,52,193,714,409,756,726,980,331,4,429,773,30,157,996,717,37,317,600,754,933,927,62,185,561,215,845,974,24,591,875,381,445,508,617,632,217,801,471,373,315,314,905,71,969,92,425,985,481,242,904,422,589,243,527,29,882,491,930,853,797,628,225,942,214,269,326,816,272,249,174,440,396,9,6,834,82,945,820,333,749,124,866,606,866,493,244,743,888,469,988,976,342,508,320,339,746,317,426,878,748,885,942,9,126,303",
    "58,101,324,775,396,248,988,520,491,875,72,952,486,184,65,340,636,749,841,906,469,212,597,603,703,102,470,659,470,779,30,860,473,534,602,816,496,740,279,534,885,283,24,297,268,724,878,0,954,597,947,566,935,507,285,939,574,729,932,68,722,984,385,833,142,311,526,407,612,932,686,867,36,634,818,58,877,334,932,151,520,315,194,554,915,579,450,146,852,833,731,701,168,826,204,450,367,998,372,548,419,242,148,83,378,378,656,215,764,816,245,342,684,289,397,792,411,968,855,884,684,279,183,459,105,226,422,140,800,934,827,590,63,814,438,398,407,4,420,889,479,727,952,261,61,826,742,536,995,996,737,952,156,899,126,664,732,947,223,986,931,385,412,749,734,237,299,512,992,876,89,160,518,446,556,430,775,431,407,672,150,882,260,972,854,416,83,692,305,396,578,85,196,939,384,506,310,381,901,980,350,10,368,520,98,518,761,690,462,476,982,979,752,821,358,712,791,998,415,221,26,613,23,1000,725,531,400,954,365,585,107,853,92,395,825,543,949,569,110,542,982,566,611,466,716,348,873,542,602,502,972,305,354,116,149,421,917,464,233,256,264,888,304,608,895,950,443,431,381,144,817,604,239,817,797,299,534,334,810,917,815,858,861,671,474,845,305,773,808,902,596,330,38,826,777,804,651,592,521,556,942,75,137,93,416,651,277,53,340,217,862,466,497,470,178,612,451,304,114,277,435,96,955,947,5,303,832,622,944,670,415,497,426,319,736,684,599,569,248,991,899,702,611,825,310,224,630,215,725,233,84,84,18,252,399,304,524,987,237,262,982,679,360,377,99,478,999,919,5,674,936,326,426,361,76,232,250,560,350,983,338,66,992,149,237,16,183,475,41,974,886,424,647,864,203,437,361,608,123,834,137,925,273,100,6,815,282,175,30,839,512,806,845,375,303,197,354,339,544,949,34,917,361,179,605,55,644,475,550,632,618,768,203,116,111,148,622,261,173,190,695,771,287,429,45,489,758,882,410,610,677,631,367,803,94,898,894,472,374,950,308,795,799,177,218,194,523,432,823,138,370,868,806,193,943,39,212,975,87,423,579,189,440,516,983,690,568,740,717,177,430,624,494,611,494,771,651,894,642,685,668,974,877,904,590,979,478,469,255,635,552,525,221,963,45,915,476,326,962,298,177,935,537,909,928,651,221,251,367,393,652,750,105,48,863,611,883,706,620,545,790,138,374,162,47,23,787,999,458,561,675,454,994,5,11,939,559,773,703,448,180,205,23,653,183,749,672,185,586,411,661,408,93,62,795,16,794,867,319,682,266,181,931,752,398,53,724,602,922,847,683,586,370,57,86,634,88,86,906,908,151,19,480,807,142,825,71,83,24,50,153,540,603,848,459,654,870,986,248,3,870,598,727,325,353,153,19,45,669,281,493,761,579,390,632,128,72,667,344,525,661,164,832,790,549,340,690,805,572,490,184,516,196,62,242,784,939,893,687,968,782,109,971,506,544,790,865,242,847,235,954,802,595,289,654,576,788,539,785,916,287,490,991,100,229,83,814,844,161,237,114,406,333,982,756,164,112,885,303,545,157,618,726,994,271,660,444,826,970,746,584,443,120,254,780,726,684,36,988,671,904,318,555,714,295,484,553,386,666,424,590,782,864,341,461,186,149,792,489,376,808,452,379,469,554,292,597,348,210,759,283,993,56,152,937,675,46,966,590,374,247,293,556,625,877,525,300,664,152,737,165,410,689,643,665,817,701,928,981,421,44,256,369,741,427,397,978,543,564,144,656,693,30,207,511,340,365,268,258,940,235,127,676,31,439,886,563,929,472,646,866,15,355,568,42,723,171,750,666,46,866,769,194,530,933,828,91,436,665,196,504,250,429,948,233,337,794,355,879,286,605,791,404,906,297,773,23,599,76,912,384,747,575,642,444,937,771,404,658,485,165,363,161,234,310,721,630,955,935,571,884,685,136,893,469,380,13,236,244,337,798,67,73,231,534,98,389,558,603,774,781,870,89,105,679,945,313,494,937,703,44,524,125,828,78,541,982,949,618,381,404,66,507,169,402,761,250,975,789,611,764,928,624,285,742,740,917,630,970,14,863,932,745,536,32,783,419,798,174,153,436,786,424,865,423,447,520,821,867,776,885,886,124,924,959,304,184,451,339,638,986,798,592,849,144,719,328,693,193,955,678,663,214,380,599,842,61,486,979,21,820,944,498,370,192,352,388,78,165,133,590,927,905,281,352,941,920,953,292,665",
    "145,888,59,396,915,794,808,62,962,15,529,8,9,807,908,312,277,512,59,787,118,1,499,543,763,425,65,525,156,511,398,740,398,857,29,240,790,713,299,649,211,405,58,213,830,543,543,954,0,120,144,121,842,607,660,504,730,305,31,626,434,987,994,786,590,488,994,505,801,246,509,266,241,253,644,804,462,110,628,505,939,917,615,522,916,142,64,692,7,564,438,636,305,23,157,310,414,264,472,536,140,26,562,135,169,18,185,569,721,704,898,581,825,663,58,9,535,55,964,816,134,606,246,403,87,782,754,184,906,604,751,336,486,51,707,915,94,202,505,370,381,289,748,509,34,584,372,906,718,155,204,894,159,771,714,320,299,540,518,93,458,639,715,140,556,79,558,551,220,544,269,290,976,111,624,591,566,760,742,832,63,553,889,431,831,258,557,70,150,803,700,571,636,126,294,112,88,141,825,321,789,272,838,926,308,400,825,180,751,665,81,165,100,602,429,488,853,977,133,732,598,203,826,753,878,940,763,989,427,656,660,44,230,999,754,928,751,504,772,613,176,893,745,410,886,589,86,285,384,257,129,242,495,412,988,699,378,991,360,767,492,160,396,227,996,350,984,165,560,329,957,790,458,140,571,730,568,628,259,978,126,421,240,713,818,649,800,443,993,594,601,841,134,705,564,154,849,331,417,3,266,592,920,340,124,989,707,840,288,104,737,945,843,605,352,239,469,269,609,449,107,817,568,439,440,509,196,224,229,189,424,448,27,308,242,409,212,109,462,954,453,670,253,152,159,629,630,912,989,578,349,299,65,976,58,69,262,641,999,484,981,255,424,745,432,161,823,965,843,987,577,133,498,203,730,555,170,266,137,937,278,303,359,116,657,231,618,343,645,923,469,63,701,83,197,379,140,656,599,368,109,14,46,453,825,69,892,414,112,815,101,837,15,52,955,935,885,534,404,659,909,748,131,756,494,836,448,474,459,46,332,522,813,827,38,719,990,699,97,89,88,369,926,530,618,268,900,283,575,250,256,346,46,875,177,132,31,898,478,849,469,259,207,770,705,782,581,702,250,521,6,784,18,893,249,653,95,464,34,287,128,531,379,160,579,408,51,598,52,535,251,748,71,910,669,586,401,353,182,912,47,759,296,208,212,414,739,835,713,23,407,73,529,101,852,655,786,49,838,352,257,148,887,105,10,346,286,162,305,440,989,564,392,447,100,626,777,874,280,185,508,193,659,245,212,723,397,735,475,558,742,432,768,196,557,435,801,139,630,217,608,993,825,73,243,429,827,481,319,43,561,835,672,109,661,344,307,437,839,61,618,739,215,792,836,108,894,634,793,402,653,756,44,683,696,346,82,233,73,510,378,917,67,257,733,359,110,269,803,854,801,967,880,225,847,355,182,662,84,407,338,789,269,830,30,755,875,828,697,22,723,403,73,729,888,60,556,347,316,173,667,145,94,650,491,307,674,941,760,510,255,716,754,518,228,416,781,643,844,447,709,942,590,691,53,513,109,658,802,847,409,782,262,741,797,826,525,863,431,75,441,703,669,318,917,762,502,887,171,422,888,572,76,304,796,607,718,784,852,341,251,580,233,390,833,976,521,411,667,618,237,796,521,494,23,618,479,343,751,778,299,382,393,458,390,368,83,200,467,363,302,356,309,839,276,804,111,870,734,670,143,455,899,13,974,30,726,706,68,455,962,738,768,149,131,206,616,729,533,686,323,489,454,381,221,386,991,256,595,683,767,10,318,772,789,354,655,491,574,179,870,70,952,32,435,798,213,87,742,291,555,154,773,428,201,621,398,769,412,398,827,534,359,729,142,202,852,783,624,543,261,333,215,871,782,448,860,191,783,380,7,453,608,729,477,60,284,642,28,13,384,472,176,414,496,360,166,816,667,305,864,851,244,592,982,120,278,365,860,646,843,39,576,825,843,207,325,317,489,655,365,450,724,496,959,655,513,659,214,721,865,827,369,282,13,441,115,514,622,204,746,867,896,416,799,631,868,329,923,828,503,428,572,33,160,631,952,909,793,763,387,463,940,388,590,360,736,877,542,431,406,126,660,399,770,355,132,608,492,387,836,656,834,112,50,937,516,30,186,965,893,699,137,400,331,144,439,102,309,246,591,239,705,2,973,454,320,62,348,690,332,183,121,469,676,104,318,960,332,500,476,636,293,804,660,657,642,763,273,712,16,489,139,103,547,703,217,953,192,522,152,7,779,96,191,695,413,562,641,722,642,717,325,382,535,137,924,388,888,776",
    "304,750,349,520,102,281,296,551,786,187,848,600,299,183,547,440,311,341,565,718,909,443,229,476,327,306,824,153,464,273,35,164,771,147,494,12,304,401,800,208,143,356,214,597,946,988,871,597,120,0,676,502,739,122,757,469,633,597,41,405,468,844,833,882,61,303,311,633,845,962,279,929,861,321,730,239,620,591,602,637,908,466,279,737,671,976,754,190,725,576,536,503,441,352,670,428,548,302,619,309,10,575,299,458,834,823,101,191,568,444,920,335,545,426,436,481,814,122,73,153,324,801,623,257,410,506,310,232,569,213,115,760,311,452,267,585,247,765,765,436,775,745,639,254,760,726,697,517,785,256,652,395,425,136,977,476,578,360,284,576,977,693,578,37,529,316,369,319,3,254,767,798,511,921,902,641,84,477,72,985,107,594,931,776,219,570,402,912,86,371,138,484,350,953,500,393,719,414,481,903,234,983,665,641,354,314,461,384,678,319,658,566,896,539,551,545,464,928,908,964,81,527,250,529,739,648,126,785,787,483,607,22,896,764,778,574,188,453,589,4,445,477,536,686,14,709,858,624,656,28,661,59,456,376,917,643,524,6,219,789,485,813,513,120,622,994,11,538,477,303,877,192,686,505,385,606,983,160,773,506,796,884,883,297,984,298,131,557,58,991,133,298,236,369,73,367,980,391,937,87,155,677,719,276,513,495,732,614,960,988,185,3,105,677,718,699,87,721,999,948,427,170,5,253,268,328,151,349,164,43,246,111,753,969,566,774,757,762,441,330,262,625,702,419,168,311,198,724,516,774,36,444,272,943,157,391,454,109,107,251,936,801,796,547,295,456,200,992,40,768,948,877,942,706,327,424,856,22,21,807,80,575,927,770,619,671,733,304,623,340,282,725,375,71,441,631,891,157,309,783,267,952,396,963,806,553,247,724,14,59,549,942,110,880,558,165,415,626,222,91,943,507,397,427,856,183,238,392,330,766,552,68,766,558,211,994,491,755,494,583,848,805,124,379,487,369,879,180,385,757,762,958,347,124,477,706,673,710,142,131,146,260,921,918,457,55,657,810,269,104,533,883,851,944,671,296,821,648,625,648,530,673,465,474,693,38,887,508,395,940,197,678,443,755,363,672,740,683,307,607,152,472,691,793,167,994,352,575,693,153,526,444,954,287,358,206,204,151,209,972,687,246,262,29,737,508,894,813,244,731,960,127,194,899,943,924,599,490,542,646,27,691,165,950,802,675,390,680,827,631,981,301,656,778,299,902,65,307,131,106,249,53,544,319,228,527,964,731,251,893,671,283,241,719,608,497,244,256,268,661,462,639,846,112,337,195,563,707,592,712,935,631,11,559,509,502,363,230,211,56,966,319,35,463,489,3,227,837,186,939,418,312,432,995,268,116,871,344,922,996,430,999,737,827,866,986,452,818,646,432,519,411,747,603,228,424,570,749,837,607,335,929,205,12,628,580,921,177,998,359,22,565,718,842,563,748,353,631,772,273,376,806,561,538,218,8,756,93,322,964,304,874,790,495,918,831,615,380,428,353,467,1000,352,309,698,359,766,260,480,545,97,358,115,623,249,907,868,809,1000,409,26,968,687,876,495,233,21,552,429,254,473,927,853,967,976,665,306,923,34,884,724,416,296,899,681,543,708,657,572,249,99,117,784,352,564,804,718,890,781,835,316,563,897,312,274,436,953,677,611,191,463,485,814,143,628,54,801,735,307,302,808,720,190,751,484,8,984,773,960,966,579,515,797,897,376,106,321,985,2,376,167,212,317,28,670,400,180,567,548,542,21,951,435,937,954,155,906,244,86,319,151,357,470,538,670,747,607,349,867,59,665,491,544,428,813,582,653,926,53,29,575,970,253,886,899,813,649,899,500,940,922,184,222,425,445,285,593,409,591,773,601,918,39,757,521,935,513,549,466,346,148,493,622,312,644,990,680,769,612,73,86,979,608,431,988,850,905,597,779,67,657,82,107,77,186,941,706,61,156,241,674,195,493,115,41,836,974,342,432,275,251,394,572,873,983,146,641,312,404,407,74,951,898,857,831,30,847,776,340,528,337,9,755,469,985,444,711,340,966,811,750,791,858,531,595,813,677,718,389,273,272,288,338,251,904,200,98,28,854,565,705,292,46,140,776,701,943,153,760,187,238,750,247,701,849,896,228,963,625,320,112,376,124,592,207,550,674,930,269,791,897,19,694,326,430,852,912,594,321,611,629,954,332,602,139,566,402,164,995,963,961,568,305,390,22,101,101,970,958,407",
    "133,550,120,652,122,805,3,559,155,802,167,243,597,471,131,30,413,949,609,463,631,810,688,937,234,970,547,516,66,761,59,145,379,130,890,690,743,891,601,203,367,594,419,810,536,115,960,947,144,676,0,62,384,383,596,12,461,343,664,102,343,365,762,658,710,988,143,26,323,707,419,205,413,312,821,385,819,908,427,461,358,398,746,348,821,91,491,707,481,765,919,488,5,641,215,755,240,300,833,932,391,724,911,519,703,768,78,257,926,504,237,979,530,521,793,444,684,486,812,16,956,229,834,673,1000,519,894,195,648,694,439,448,916,436,870,954,182,595,385,203,827,294,940,692,671,807,19,112,328,276,290,835,365,933,889,536,435,444,670,872,934,231,941,402,650,790,718,948,194,636,354,817,614,390,967,624,459,756,521,339,160,248,53,564,413,965,148,499,851,521,930,284,251,487,706,61,650,211,89,255,786,500,858,630,214,953,471,53,603,737,598,729,135,990,208,624,489,393,311,375,584,134,240,84,825,531,411,204,406,819,214,236,569,638,904,55,349,555,950,540,823,844,862,844,498,532,564,831,228,753,327,564,782,212,826,229,566,543,373,198,825,709,706,848,361,355,506,875,787,604,584,373,83,258,512,936,197,639,22,753,104,298,491,507,337,626,475,809,641,18,876,734,530,398,668,921,840,243,365,219,487,594,767,398,157,384,344,338,546,89,683,42,419,100,854,202,457,218,412,680,200,102,123,112,161,395,126,584,800,453,434,411,513,847,332,677,105,413,912,641,376,379,434,458,937,179,258,429,692,192,446,894,527,182,691,728,752,826,328,773,481,106,792,128,89,505,7,842,223,58,592,372,287,962,454,654,185,710,580,926,274,656,93,383,121,318,980,832,441,435,236,987,548,723,535,713,554,435,557,277,957,498,709,154,550,824,166,89,983,431,911,522,208,389,912,341,120,192,611,16,50,678,119,131,847,571,673,422,576,681,437,710,320,583,826,539,54,888,191,97,83,989,327,372,613,710,184,734,656,368,38,996,268,444,67,647,598,22,323,441,161,223,742,200,303,956,564,698,819,794,698,173,583,358,133,128,254,161,737,942,424,221,65,625,46,148,218,744,703,802,739,60,695,421,561,684,276,570,324,919,943,728,38,639,341,104,597,696,338,686,474,232,439,790,658,458,9,561,88,976,244,278,45,993,939,700,968,57,573,740,756,948,524,452,575,917,606,54,246,652,744,343,334,439,649,950,920,962,800,985,885,189,400,5,608,85,814,221,198,395,725,877,864,271,768,666,986,947,272,697,125,863,889,439,396,699,999,454,191,686,109,998,403,618,511,278,345,603,171,647,157,975,251,252,2,203,523,425,431,661,854,961,639,471,837,991,289,665,235,777,461,691,595,212,642,710,769,719,587,357,563,830,500,928,18,765,130,224,371,420,3,54,858,233,713,333,196,216,675,778,150,613,530,755,484,624,49,30,204,523,176,165,580,897,352,972,259,938,311,305,73,610,518,95,444,494,918,825,795,862,910,438,505,516,450,214,125,184,607,526,945,476,977,741,629,2,366,623,956,503,685,306,959,362,398,32,804,568,482,999,945,782,67,504,598,377,837,357,542,110,309,186,45,536,913,200,427,322,682,356,915,283,195,431,87,801,545,784,452,932,143,856,745,267,250,421,733,40,854,186,123,106,933,809,383,264,588,234,82,557,451,218,721,873,646,738,205,343,975,175,904,80,749,257,199,338,244,554,513,615,312,784,360,739,599,367,849,380,786,229,80,204,341,820,624,806,44,482,494,304,419,477,648,697,750,317,615,507,495,631,703,837,637,313,952,700,498,946,295,780,378,898,23,128,946,296,526,540,248,501,945,829,478,564,595,940,363,723,226,594,403,811,565,435,783,418,5,873,363,20,571,281,705,952,260,161,149,510,588,690,967,467,795,272,276,452,234,963,474,515,874,142,851,81,758,624,547,966,455,604,2,157,615,758,638,846,966,339,954,187,605,530,102,802,91,684,571,586,624,706,405,859,406,930,261,430,19,653,33,473,987,357,820,205,431,717,949,449,713,703,151,392,264,183,477,11,968,893,337,172,151,541,586,641,27,432,587,178,670,348,729,829,964,845,620,91,789,467,807,728,196,11,764,633,598,718,374,792,639,532,562,219,273,420,895,408,706,731,446,579,517,700,477,26,286,433,386,20,856,449,674,383,836,720,700,584,947,560,582,321,45,600,916,199,838,964,298,784,714,418,892,972,553,534,845,378,960,213,273,861",
    "440,293,555,572,905,595,216,991,695,769,509,323,386,522,166,811,744,115,600,818,847,940,965,625,107,951,889,88,480,417,817,473,679,932,905,244,367,908,318,99,99,265,195,563,541,77,864,566,121,502,62,0,442,871,933,416,80,303,135,927,358,213,286,258,314,313,386,345,142,532,224,538,322,841,542,253,292,85,628,878,200,769,758,255,889,867,25,982,191,359,8,707,712,178,607,46,647,932,91,135,844,952,158,516,993,183,812,275,606,919,118,822,187,124,274,333,454,82,494,372,879,364,363,408,547,884,164,152,363,644,879,307,114,915,520,470,281,476,915,780,651,321,426,268,180,415,712,408,429,201,238,389,936,144,993,833,150,150,846,518,566,714,633,221,763,42,622,555,125,298,460,24,192,11,352,743,369,864,227,716,241,107,384,700,748,36,192,145,908,207,718,845,694,909,240,897,104,989,25,822,584,219,629,899,11,135,564,899,496,23,609,724,675,602,97,251,182,324,932,262,770,835,29,73,557,600,240,944,84,570,871,137,840,465,621,171,559,741,253,831,777,447,825,961,978,88,952,530,159,50,932,16,213,113,496,252,496,48,426,435,159,670,752,2,245,553,371,238,718,896,713,412,38,581,109,370,753,538,950,710,32,880,367,920,758,311,291,63,503,296,260,588,494,647,42,254,629,800,52,531,879,869,797,687,759,347,113,144,352,842,711,150,807,959,53,426,45,819,844,83,523,892,217,748,466,10,654,932,73,790,543,88,751,233,515,798,633,887,50,67,590,315,191,611,599,495,52,655,52,774,997,469,134,92,666,516,470,589,680,906,32,522,704,133,238,531,394,327,198,477,320,980,635,11,506,481,954,822,212,215,828,333,636,344,782,119,80,491,360,651,183,270,103,869,485,622,545,594,561,599,830,143,976,120,844,326,720,365,529,93,900,601,413,611,203,158,782,969,782,838,173,162,549,89,118,683,468,946,427,110,672,336,130,135,466,814,864,259,996,579,420,855,902,481,497,562,331,422,773,240,606,95,594,208,290,742,364,122,338,266,127,816,721,31,778,340,463,689,748,587,899,816,666,986,709,285,397,528,936,408,190,122,977,822,379,289,145,99,682,833,729,113,346,499,577,488,402,290,305,356,651,530,56,373,125,673,499,737,985,394,65,287,417,814,928,334,505,811,515,656,600,876,112,730,11,895,830,711,400,749,160,685,529,121,339,876,789,558,127,331,820,114,827,150,832,559,149,47,322,272,606,921,884,195,985,617,513,431,788,609,769,537,768,214,42,415,293,641,390,830,161,517,319,753,115,675,186,53,350,648,583,194,180,307,283,920,605,130,467,436,781,948,94,605,592,78,285,759,873,839,321,619,294,462,97,296,121,815,190,642,220,989,113,314,39,965,771,103,836,31,976,443,425,846,306,47,406,589,67,579,232,266,393,322,799,767,968,768,472,703,533,567,232,778,387,742,111,6,453,733,559,930,560,860,596,385,430,454,143,577,374,241,582,505,537,58,262,731,806,323,964,747,653,835,2,908,791,619,904,906,447,129,734,411,178,7,554,774,461,650,208,517,837,868,770,152,568,238,143,457,931,464,363,29,601,915,674,485,409,185,721,630,438,251,31,138,165,850,573,317,931,75,249,1000,693,811,400,982,674,139,261,752,148,178,211,878,120,681,47,741,284,389,23,997,646,869,280,651,524,380,377,638,647,958,134,591,165,629,420,472,355,915,417,281,756,615,408,249,360,919,131,852,75,743,428,629,915,242,397,955,495,533,853,322,30,696,46,301,57,501,331,33,205,871,220,727,522,633,660,544,598,292,292,916,232,914,982,548,934,52,300,277,135,737,78,691,249,291,639,495,992,462,747,337,911,907,257,282,440,620,870,430,906,166,85,373,143,96,986,207,391,712,576,887,673,537,11,448,633,781,838,603,639,456,758,451,882,347,928,375,641,77,912,614,329,961,48,754,120,920,443,205,873,366,146,264,568,826,159,945,736,132,373,444,302,402,486,274,547,556,249,120,638,355,579,437,410,173,700,133,990,724,575,189,74,6,899,292,679,267,176,898,216,688,781,920,63,455,660,205,987,614,352,484,366,935,551,872,127,307,342,301,494,913,980,595,960,391,563,775,358,894,809,694,416,627,675,119,32,362,49,200,822,550,692,271,4,721,481,568,140,676,817,711,296,362,169,794,815,579,632,435,313,129,875,888,292,443,104,941,339,624,115,899,179,894,972,122,56,12,778,566,91,124,87,84,738,30,799,136",
    "159,584,62,404,891,59,197,725,926,214,840,748,352,274,475,652,988,55,483,851,359,799,829,980,613,945,90,746,332,813,89,502,342,814,937,765,765,530,951,308,258,179,449,743,953,57,898,935,842,739,384,442,0,805,491,400,320,458,926,37,841,201,449,112,87,780,969,761,667,22,215,841,437,264,337,332,611,854,102,911,569,467,657,663,370,524,753,228,668,354,939,477,394,828,808,285,999,993,305,439,222,138,98,421,19,816,28,657,449,717,880,445,896,366,534,247,565,334,375,372,734,723,188,809,72,896,447,560,117,172,75,350,945,334,738,495,44,115,538,12,953,45,613,321,790,295,762,274,309,541,51,993,438,683,949,169,414,981,438,385,851,284,904,981,344,92,547,68,413,952,730,897,592,480,212,407,758,996,166,423,337,865,421,229,167,580,653,349,405,622,650,337,361,352,784,846,14,996,23,234,269,328,994,622,697,77,134,364,993,859,922,713,939,786,952,392,882,176,426,295,37,103,969,86,251,247,492,815,802,825,88,447,726,566,270,493,77,45,690,709,514,405,66,737,522,247,692,265,633,504,386,81,65,606,733,278,395,453,781,804,854,66,762,734,317,393,90,213,735,305,179,820,764,338,775,74,496,623,906,405,329,792,76,900,634,67,180,569,240,612,329,657,148,513,51,794,743,850,484,388,49,141,81,513,998,567,269,485,710,695,538,491,205,325,151,542,983,674,589,760,476,569,860,972,882,725,778,997,339,694,658,392,420,354,58,896,336,224,297,714,894,959,56,233,658,67,113,562,154,708,696,743,44,898,414,87,412,492,356,19,614,215,843,734,804,105,479,94,445,726,552,367,44,99,667,530,95,392,611,163,785,212,334,808,798,723,456,114,153,966,788,436,786,236,802,749,950,79,879,936,294,934,803,963,887,849,127,288,565,456,92,112,757,642,533,139,522,669,128,647,905,633,491,797,149,106,429,560,416,252,199,135,996,691,34,576,960,39,719,953,524,311,287,460,714,335,173,57,969,753,370,632,939,784,88,274,429,187,737,974,687,576,968,528,375,119,895,635,371,165,7,31,805,51,587,65,657,732,620,267,604,797,894,877,717,937,528,728,796,450,968,814,538,980,506,526,337,631,63,961,743,890,697,62,290,834,821,390,122,148,540,217,655,462,187,911,980,510,641,904,256,247,831,645,966,55,234,656,363,350,64,790,760,999,736,258,943,713,134,529,227,773,151,2,352,801,676,236,421,904,128,930,829,681,658,991,773,785,207,458,747,77,939,663,173,425,887,64,57,76,201,303,488,794,151,483,416,685,818,380,487,794,12,377,436,522,173,222,576,219,262,371,8,542,193,771,169,749,31,115,510,802,761,36,942,40,427,511,554,305,226,522,503,380,832,811,993,211,377,647,305,405,450,644,171,322,808,147,708,802,8,638,152,458,953,568,685,265,189,568,545,121,371,109,603,475,4,750,840,31,840,928,536,719,261,365,336,199,72,308,312,286,964,583,351,404,453,437,543,566,786,907,560,361,33,48,907,165,868,290,483,491,795,912,244,520,949,247,192,308,124,767,428,628,100,579,874,177,988,677,412,531,45,606,659,776,742,347,714,186,160,646,159,137,62,787,545,978,531,95,176,598,569,142,325,463,404,409,908,376,30,684,817,631,814,712,704,870,738,563,903,47,292,368,582,509,12,626,993,683,238,756,135,67,655,475,150,389,540,678,180,181,717,966,868,245,96,243,819,414,713,875,513,915,277,192,16,900,473,709,368,538,205,473,159,110,589,603,1,896,884,501,291,575,527,744,187,624,823,751,90,541,29,761,902,877,762,220,170,222,446,318,940,956,662,729,324,528,804,756,506,475,475,63,287,836,857,913,776,477,358,741,593,656,840,642,127,798,660,322,89,214,538,336,252,592,870,511,481,8,469,683,98,340,67,682,2,352,649,159,258,756,193,392,335,980,972,824,660,920,538,52,659,874,511,894,197,771,683,813,184,36,140,231,663,470,839,60,115,402,12,515,482,275,242,892,487,532,884,847,659,69,12,995,371,783,527,846,52,50,42,361,973,80,702,569,832,794,355,864,506,538,218,724,154,774,497,736,570,834,364,800,776,363,567,812,660,384,331,734,882,778,25,407,94,809,662,487,592,954,533,347,914,516,250,158,962,433,470,940,642,521,418,123,654,397,582,126,295,605,363,168,235,755,42,313,244,829,205,918,941,766,117,635,686,460,958,237,13,529,652,711,109,889,90,635,992,279,287,956",
    "219,254,299,747,296,924,192,857,374,671,471,412,938,155,995,51,697,309,562,42,805,698,632,128,928,636,460,365,55,184,224,670,853,413,977,408,359,467,884,108,196,605,860,121,851,895,314,507,607,122,383,871,805,0,163,331,865,503,804,944,606,503,596,459,776,297,349,623,522,408,110,605,863,62,433,646,421,750,220,541,746,185,989,259,256,280,732,46,377,620,978,557,766,803,629,444,916,625,52,511,181,689,100,703,894,812,741,741,15,551,426,500,432,35,24,912,323,144,814,918,173,825,940,300,552,167,83,102,234,572,23,285,771,529,386,469,539,672,892,398,978,343,171,585,413,276,425,432,548,741,799,695,227,197,291,36,747,523,340,172,476,688,964,976,693,794,11,869,107,94,525,732,226,811,34,78,222,381,923,707,62,263,603,346,559,956,558,11,60,390,521,144,941,899,572,266,757,944,127,541,288,420,479,519,851,362,735,333,82,101,683,697,2,696,486,578,241,201,89,146,616,67,482,468,251,873,266,671,650,348,21,180,635,40,431,813,807,383,728,290,3,660,642,958,280,73,337,244,431,5,651,214,644,421,472,538,861,918,703,355,222,708,650,29,867,90,203,767,243,434,434,534,811,378,511,91,477,34,697,726,534,495,987,650,320,581,218,586,964,841,929,207,178,973,71,318,972,818,539,971,524,708,201,465,312,865,230,646,31,693,581,551,52,865,224,114,224,62,586,674,363,171,808,202,77,543,889,934,419,887,541,391,736,346,887,511,340,513,904,800,452,2,417,114,338,276,466,968,91,283,827,445,324,676,274,885,448,746,685,220,709,627,587,697,39,667,521,560,603,208,224,8,578,519,930,647,919,153,723,989,648,819,132,183,550,843,888,928,237,596,59,921,993,880,472,860,107,560,408,359,464,104,410,562,301,679,840,271,18,386,559,861,532,891,565,879,755,389,382,474,465,783,530,728,164,457,420,712,769,54,74,206,2,812,936,388,220,558,779,813,552,602,35,306,451,366,617,742,763,653,669,563,247,14,303,393,453,188,404,190,505,566,477,688,31,62,137,835,178,106,618,901,515,890,934,320,883,484,857,657,597,648,995,849,898,143,832,165,663,538,972,830,514,363,390,34,324,795,733,911,572,272,3,292,474,990,769,183,854,123,537,549,913,97,763,496,236,939,643,412,148,508,200,930,365,930,435,108,545,399,219,63,732,698,465,566,360,68,63,164,351,855,209,646,480,256,55,283,737,978,935,979,377,574,546,196,833,272,6,508,203,524,288,658,984,944,179,611,565,208,303,644,461,985,271,684,477,196,209,149,127,353,176,618,710,817,743,564,850,829,874,105,606,845,387,914,137,181,690,964,928,824,262,3,862,528,135,559,547,451,607,367,614,826,436,539,649,106,139,178,868,306,909,982,633,266,881,288,560,338,452,947,111,259,304,761,238,894,528,869,821,853,878,851,582,732,431,922,434,46,805,196,72,13,321,693,876,899,678,754,149,420,451,626,34,397,563,678,645,727,914,919,88,612,299,54,223,532,84,628,469,94,566,821,919,875,588,825,189,462,530,956,16,864,820,315,134,942,127,810,937,744,47,894,267,377,729,40,482,391,529,918,807,224,181,977,802,970,19,60,961,332,363,36,246,757,886,506,249,267,69,582,630,409,410,971,501,952,780,162,850,244,622,974,958,122,380,508,971,172,933,59,777,169,689,222,643,220,35,453,106,923,180,294,867,669,802,72,866,804,845,249,761,766,435,492,415,384,303,80,401,842,306,595,817,127,329,85,528,479,866,611,515,179,875,535,842,592,313,335,811,873,270,472,674,152,641,521,958,491,669,351,833,353,979,205,731,245,44,316,510,407,123,590,268,801,807,355,354,136,250,656,562,184,456,342,686,853,565,413,931,710,713,800,975,615,217,380,496,492,777,20,357,446,149,12,546,796,768,611,461,454,416,968,54,42,94,91,596,944,512,92,86,970,664,326,515,417,223,316,799,421,408,728,345,266,176,835,679,467,926,38,521,808,433,987,563,6,618,773,825,252,871,809,451,877,925,523,27,703,793,259,848,784,259,391,669,254,997,8,7,385,751,855,682,113,78,967,225,51,923,911,37,957,334,50,83,120,941,566,960,351,191,850,205,777,846,355,787,632,764,239,360,649,170,370,696,88,870,66,735,568,821,285,515,967,832,169,951,799,607,671,829,279,418,799,679,869,153,748,905,112,612,957,469,613,770,972,265,308,295,595,459,524,763,790,218,177,816,895",
    "300,732,840,304,198,109,807,542,383,331,851,46,893,496,10,758,463,951,234,296,622,861,71,910,57,924,677,255,412,955,157,332,453,613,346,961,16,603,884,1,77,753,328,71,468,461,371,285,660,757,596,933,491,163,0,150,664,288,651,186,9,900,286,411,567,180,129,469,148,152,858,30,862,57,327,264,807,795,478,700,724,145,811,360,138,870,301,672,109,702,864,175,15,782,801,999,828,740,342,527,334,931,775,504,897,339,744,323,703,988,920,807,743,674,780,295,983,25,575,424,496,669,414,71,50,652,776,781,804,494,628,856,664,315,163,399,551,16,105,936,178,778,287,305,412,222,4,602,637,568,820,754,257,474,817,618,573,563,11,771,78,587,430,987,392,602,14,829,554,981,945,545,569,437,385,431,134,941,39,985,247,402,735,473,505,278,266,668,186,263,520,417,898,267,715,126,730,585,972,732,232,115,996,636,379,285,278,892,932,116,617,739,98,581,17,81,243,190,405,995,787,399,566,291,216,792,673,503,385,205,185,491,535,191,43,370,628,544,83,915,726,42,392,241,934,907,739,888,720,429,512,872,954,188,619,632,51,851,765,535,379,432,170,444,161,463,525,623,4,4,316,414,423,120,498,766,630,605,499,108,953,255,494,853,283,403,568,162,261,888,918,416,2,90,990,963,84,399,915,272,117,762,979,304,63,19,618,833,670,438,373,333,956,3,138,644,295,442,130,310,539,568,925,176,110,957,377,521,506,688,462,690,786,897,55,993,650,174,993,402,543,422,930,263,482,316,182,234,262,138,86,760,528,744,443,565,845,939,176,289,532,883,476,424,321,976,688,935,76,644,670,102,51,498,691,290,132,167,970,130,627,532,194,732,101,385,425,564,511,413,689,466,681,426,474,271,793,794,420,934,922,382,345,970,320,771,342,961,300,954,709,213,727,824,434,431,47,63,281,217,146,578,943,401,412,326,416,758,809,971,898,385,155,958,831,393,618,833,405,737,340,917,563,903,859,688,541,431,160,559,92,648,146,652,533,407,859,17,260,628,59,580,804,504,612,524,417,664,53,597,606,624,549,645,753,387,638,116,777,831,366,163,840,10,700,997,825,688,631,427,171,403,201,101,406,487,751,155,769,400,122,956,171,834,176,119,682,552,508,913,615,3,955,124,281,273,895,647,325,832,432,886,693,857,192,362,629,773,355,320,459,232,476,563,740,645,766,794,704,160,523,246,267,265,345,674,584,51,274,91,112,894,946,231,760,908,163,201,924,121,976,43,354,876,809,597,117,393,858,628,363,413,654,781,720,494,432,50,243,752,426,801,624,641,521,803,262,81,406,325,389,884,789,416,803,643,275,111,520,688,275,588,223,311,307,3,261,914,106,975,758,847,946,621,961,503,703,120,879,192,321,466,532,652,822,265,840,681,287,388,83,166,969,452,160,389,263,337,479,331,108,240,56,921,210,211,410,127,808,380,670,404,130,274,509,564,687,76,288,371,584,20,129,629,271,902,245,19,824,220,834,40,60,515,930,206,878,729,11,498,253,403,46,390,230,979,881,260,23,19,989,169,408,974,187,993,919,912,220,324,762,461,573,420,779,688,193,658,873,845,730,834,224,738,860,17,183,863,797,133,170,603,659,26,719,681,929,324,896,565,851,345,869,16,935,682,753,853,680,428,963,641,479,684,719,108,561,643,592,562,522,431,839,462,389,939,700,136,664,351,164,308,325,466,172,114,238,789,166,266,334,120,701,972,132,438,642,590,61,729,837,664,552,304,104,14,838,41,812,714,70,856,982,822,970,702,664,141,430,306,719,474,145,359,303,420,343,163,67,285,868,402,896,506,414,61,48,202,817,37,813,614,473,943,597,434,96,486,707,844,942,138,648,850,494,541,801,361,81,686,310,585,897,663,706,670,724,144,920,201,582,512,327,706,686,118,686,173,275,880,778,604,112,726,603,704,351,753,653,966,71,687,714,17,826,791,200,846,779,263,52,543,941,315,662,838,165,319,490,735,491,664,235,703,385,921,733,203,846,261,691,825,439,599,691,127,146,414,862,355,315,444,469,804,652,509,817,547,198,857,18,853,358,809,196,598,508,972,376,116,745,477,84,365,927,232,869,411,747,544,735,887,298,169,408,178,102,526,443,393,74,2,508,916,355,101,382,482,639,66,55,425,472,861,204,344,208,546,165,395,296,804,857,798,635,424,78,982,815,394,700,630,313,336,154,119,859,628,382,539,161,423,636,259,143,62,791,574,199,714",
    "855,694,451,949,736,36,627,434,406,611,544,140,767,90,529,529,635,611,912,9,712,924,116,71,591,752,734,772,873,183,955,922,121,609,35,959,450,945,930,202,706,735,109,497,50,937,263,939,504,469,12,416,400,331,150,0,645,237,340,118,864,315,648,709,634,385,641,168,699,3,683,300,596,454,693,514,294,442,14,178,55,462,110,910,160,374,113,768,315,491,228,697,168,431,511,321,34,878,703,858,635,320,913,668,439,904,625,105,99,31,570,868,628,182,513,231,699,199,348,438,318,999,24,584,245,895,739,641,503,469,160,448,214,846,391,894,550,629,26,351,426,935,519,426,551,602,129,131,817,634,212,145,381,957,726,789,470,880,752,8,962,831,300,411,72,419,857,28,198,297,203,109,795,960,305,689,842,760,658,103,810,317,494,859,828,176,9,88,288,419,617,24,500,68,502,852,211,78,912,539,640,857,372,153,545,59,71,29,398,614,767,681,338,731,309,755,181,256,837,750,299,864,136,904,358,982,867,437,681,764,299,248,57,538,65,441,803,644,756,978,606,812,319,308,163,774,496,945,715,468,335,353,765,423,463,167,632,144,871,698,423,509,133,251,86,177,295,551,644,24,611,644,642,786,284,190,90,974,14,276,147,353,413,252,730,111,742,833,657,67,356,965,854,295,502,97,926,775,403,681,128,518,522,844,48,914,204,127,98,994,660,434,844,58,744,620,381,715,365,116,438,332,158,37,918,397,712,651,306,342,72,932,267,563,795,917,702,965,885,223,170,10,667,729,142,361,850,716,516,129,930,624,77,113,768,254,550,770,641,591,178,934,391,944,756,916,250,781,786,704,464,16,575,434,35,317,279,845,764,925,210,61,797,272,519,38,509,110,845,960,858,254,798,760,164,613,130,594,161,865,829,696,660,590,61,196,722,718,572,116,665,216,186,901,911,256,483,36,349,711,520,114,773,848,355,550,43,174,257,847,518,298,80,615,106,213,60,407,55,597,559,410,8,228,462,880,350,167,230,370,11,147,620,757,795,938,9,754,60,931,422,730,991,96,713,286,498,142,794,979,653,273,978,806,931,258,127,722,814,508,533,846,961,851,560,716,321,951,572,233,35,874,472,568,987,94,26,258,18,68,79,715,665,183,513,671,307,923,20,16,655,213,851,193,724,305,195,139,115,971,58,481,920,980,231,910,593,928,698,724,922,973,173,998,206,603,195,823,714,593,92,327,410,1000,112,417,816,688,379,155,774,552,474,256,916,299,37,224,948,8,470,661,559,140,558,54,463,574,967,869,640,370,273,633,77,966,540,162,831,567,67,679,167,965,942,225,79,860,757,7,14,755,520,636,762,485,458,253,31,340,918,415,381,315,378,234,97,880,412,168,179,283,498,887,153,145,358,536,190,747,558,285,70,76,658,711,348,941,951,204,778,906,884,395,312,480,439,603,580,440,104,463,818,125,126,268,894,203,106,729,109,412,258,906,15,869,413,233,219,865,385,991,979,479,931,960,442,317,336,305,36,70,543,97,413,445,421,344,626,353,853,173,555,399,105,989,450,231,409,69,771,157,564,672,482,4,662,83,254,100,312,266,848,606,640,364,135,276,161,477,700,727,9,239,734,663,198,504,585,745,428,878,672,588,125,448,349,139,39,953,953,320,120,907,857,686,994,10,527,247,412,524,268,517,293,291,141,483,325,972,844,532,658,670,443,896,208,353,6,901,503,769,309,501,310,882,262,773,405,907,376,460,511,356,888,186,308,234,6,93,456,444,187,10,607,709,847,288,769,189,871,287,183,206,618,756,835,458,989,729,87,190,161,21,648,24,517,250,469,846,557,672,428,77,288,294,818,626,907,595,706,182,676,863,734,193,276,376,363,959,821,442,183,535,835,694,684,192,497,90,622,458,595,230,293,377,796,682,359,538,707,741,947,683,570,403,857,832,536,522,98,359,224,431,509,446,665,583,391,657,343,516,829,414,618,727,260,315,429,814,727,421,86,791,71,639,579,737,72,837,244,847,42,118,241,471,267,97,843,667,498,33,63,138,831,625,275,49,11,253,595,812,261,270,721,26,1,229,770,268,966,938,858,592,605,209,380,82,228,146,902,722,715,23,686,266,379,624,59,841,895,772,345,47,314,830,504,360,889,614,488,939,16,376,887,664,594,327,608,722,495,225,493,229,58,234,743,166,885,61,648,229,200,911,679,472,751,78,421,83,59,435,407,229,9,426,530,51,222,367,306,715,686,504,997,869,336,152,415,816",
    "443,278,353,222,352,87,976,948,957,474,945,483,400,624,720,324,601,910,588,4,925,335,674,368,91,167,40,67,600,592,876,645,443,533,73,699,964,713,921,765,32,959,866,4,828,927,859,574,730,633,461,80,320,865,664,645,0,389,702,97,475,182,555,966,340,865,99,86,926,35,777,555,161,257,28,43,681,25,204,407,577,467,146,514,568,390,177,4,636,26,780,462,759,33,295,625,361,83,653,508,76,220,603,451,277,358,292,857,580,134,886,291,236,354,361,932,290,403,920,85,919,591,32,778,886,204,484,677,123,15,6,986,741,529,421,975,9,187,918,40,362,377,173,47,735,883,418,93,293,501,496,73,657,924,144,870,545,219,462,201,26,319,296,617,331,33,897,838,457,177,686,933,504,114,179,342,640,608,362,182,763,231,768,465,654,392,531,685,534,234,114,333,200,953,489,921,641,381,570,452,852,308,752,187,321,647,721,677,93,878,991,235,664,927,396,998,359,120,443,112,34,379,29,127,14,207,241,775,982,796,548,148,56,453,810,361,631,265,803,936,187,880,168,371,89,89,116,215,625,32,828,195,597,208,638,600,383,865,487,936,915,584,136,720,978,309,775,87,347,169,947,658,120,642,534,179,362,401,266,550,426,616,126,724,905,951,416,990,43,260,466,385,406,464,963,747,630,687,460,351,28,675,690,755,328,90,994,87,526,926,138,291,477,280,796,947,87,432,801,403,831,802,221,313,133,806,492,522,543,644,627,117,391,870,938,546,99,11,540,68,164,359,919,357,816,329,303,151,744,260,118,100,929,643,639,428,726,403,989,181,129,287,727,617,287,549,331,757,742,929,978,895,52,79,627,665,408,621,290,568,761,372,785,999,536,702,871,560,13,994,995,784,751,356,22,192,643,554,454,115,553,96,594,166,104,28,692,365,398,943,116,103,962,970,553,446,952,635,615,183,324,911,806,831,647,185,867,457,399,810,17,925,532,117,730,409,405,359,548,483,975,43,337,685,884,920,645,439,144,684,776,768,612,871,896,790,37,6,941,874,45,681,335,224,527,780,451,952,945,620,158,654,749,643,425,937,827,44,227,333,277,648,429,940,716,345,347,157,296,109,230,641,371,358,800,569,909,92,843,154,991,731,653,928,491,244,247,974,884,790,407,294,750,488,989,524,240,442,778,849,865,374,102,205,177,847,278,329,685,454,623,290,150,798,468,342,493,407,198,479,549,809,928,537,672,795,265,194,880,51,146,136,915,897,88,388,657,313,350,137,242,513,557,153,768,254,228,793,756,341,83,427,172,137,159,308,280,503,570,30,32,812,253,341,24,556,480,987,31,279,26,245,27,70,580,965,261,968,53,836,172,376,808,187,910,804,544,304,209,421,462,736,792,68,613,698,591,420,515,203,948,105,290,787,741,489,27,637,94,416,308,343,68,788,743,449,713,457,180,824,404,50,697,511,459,899,58,868,43,696,895,31,869,602,696,355,322,142,539,460,273,526,927,832,443,274,494,953,915,310,297,703,145,680,476,196,672,399,989,425,824,258,317,71,458,914,327,24,714,330,154,659,543,274,404,20,210,286,435,694,395,237,233,936,516,150,786,235,225,647,684,296,913,452,365,805,467,276,328,659,826,846,746,910,407,473,550,958,253,225,521,511,262,41,853,246,510,524,219,167,179,337,106,347,232,846,371,134,371,708,600,75,160,952,748,335,116,716,93,440,546,630,973,105,197,701,365,855,705,140,939,847,483,395,621,450,123,242,107,139,639,658,609,414,150,222,359,741,192,628,426,693,789,271,102,193,529,241,717,108,985,352,550,98,366,601,119,977,286,211,694,765,346,819,346,442,36,62,234,100,261,782,694,677,220,690,858,753,156,287,984,562,590,279,775,775,750,31,130,783,66,248,891,858,379,780,663,868,891,842,299,619,788,904,395,757,447,764,682,154,572,287,292,573,672,757,573,266,822,490,961,63,794,15,829,794,240,733,391,86,980,469,711,169,388,302,291,575,658,70,334,304,936,987,643,547,924,504,862,326,528,985,687,138,708,365,455,912,461,30,616,643,719,585,222,770,498,360,247,529,736,751,85,234,259,605,858,410,591,829,700,611,586,484,657,359,474,523,404,58,691,984,655,895,689,988,583,280,358,667,112,303,734,520,966,546,395,32,648,75,565,817,24,482,813,821,319,762,307,253,651,254,856,87,684,137,503,663,806,860,992,293,593,435,603,655,705,541,242,659,976,702,474,245,62,741,734,255,216,95",
    "385,971,402,7,527,931,46,381,187,305,328,755,262,931,343,113,20,245,224,335,802,630,382,12,323,867,640,175,123,912,736,981,371,574,12,532,328,693,2,370,895,145,403,551,534,240,100,729,305,597,343,303,458,503,288,237,389,0,785,532,196,233,290,69,745,181,16,761,876,976,558,752,870,715,742,412,660,396,401,161,814,690,303,193,580,100,968,782,520,223,241,438,148,434,612,319,5,970,559,14,338,577,801,775,119,537,537,940,698,918,231,4,369,465,278,114,41,630,266,460,845,710,947,883,194,737,680,289,655,466,716,767,50,353,158,529,927,231,74,912,29,546,949,826,282,424,201,63,28,360,937,218,699,972,478,1000,307,956,988,982,251,114,969,563,977,939,435,850,323,102,762,987,653,104,14,310,711,565,201,144,570,386,319,237,197,544,755,652,992,187,478,541,39,464,976,774,805,421,374,385,29,244,420,734,297,451,422,453,327,941,864,641,673,726,928,910,411,97,571,229,482,349,565,222,366,126,166,515,403,795,319,307,39,261,680,175,420,318,186,991,586,835,800,152,346,276,314,40,11,495,183,886,536,5,232,4,731,706,868,432,881,587,141,262,258,336,788,378,975,375,756,277,940,320,768,459,812,837,269,760,55,654,683,812,510,443,841,381,891,575,451,689,563,218,588,39,477,60,728,664,489,501,532,560,321,206,523,773,541,787,970,271,770,530,225,1000,697,714,579,625,167,716,775,63,717,951,480,279,425,222,189,191,832,121,295,666,161,365,804,502,904,299,502,872,854,918,32,905,754,963,418,291,75,940,845,524,865,161,479,342,27,324,777,206,844,883,915,270,208,392,119,496,450,465,513,952,722,889,576,34,741,703,282,48,846,627,141,327,833,951,640,80,931,628,901,519,86,481,72,63,582,592,389,835,428,127,183,186,716,645,96,693,413,167,19,804,690,473,694,434,598,488,433,710,18,740,124,207,726,180,980,952,659,930,624,151,269,600,101,205,659,151,95,616,330,696,377,225,254,640,527,917,662,703,117,927,951,625,770,189,445,721,368,417,345,946,346,459,87,379,404,787,650,334,551,964,361,11,680,737,360,866,882,165,845,96,385,273,702,164,676,940,191,690,37,265,362,607,812,589,832,277,535,661,573,625,170,24,917,139,592,908,855,152,481,615,773,582,103,512,145,371,594,129,446,615,533,194,660,694,446,907,330,282,681,508,720,802,401,987,27,40,875,177,717,99,541,739,691,513,887,771,327,500,300,761,569,469,872,259,86,17,449,734,342,887,348,968,767,675,667,468,215,503,304,996,6,289,417,526,923,665,181,160,411,912,735,779,620,856,919,978,580,394,430,220,436,167,474,801,902,233,145,109,229,607,852,891,942,723,66,38,723,806,218,623,722,508,264,399,806,112,285,582,881,460,208,405,103,617,388,303,238,442,259,480,442,182,874,996,36,963,517,601,574,325,210,671,18,541,384,927,49,70,788,193,723,441,434,262,842,685,765,407,305,217,442,967,134,651,829,976,651,300,924,772,120,433,985,842,410,447,598,387,257,322,653,433,357,40,468,183,542,617,869,824,635,380,4,581,592,213,658,974,92,605,740,702,686,432,581,549,848,814,272,512,202,36,691,94,623,71,73,845,425,462,667,210,877,613,709,588,474,520,463,928,177,526,300,647,745,295,703,408,890,855,481,513,364,863,505,104,268,793,819,74,21,772,14,555,365,695,965,905,7,972,128,924,641,832,918,688,17,297,643,599,957,961,465,710,495,959,415,176,156,825,413,428,367,598,636,64,726,269,364,844,242,711,120,959,317,139,538,584,791,801,550,870,816,767,749,381,250,444,675,397,816,245,323,594,207,765,731,678,348,172,343,517,148,904,530,590,156,890,357,390,624,373,498,798,804,21,360,561,883,942,484,724,457,659,16,669,3,730,841,632,273,872,424,123,117,24,637,133,690,150,112,833,901,752,111,462,319,849,329,836,377,703,863,919,282,274,852,131,916,644,532,269,135,683,815,379,129,582,377,985,765,470,427,339,412,235,244,180,270,163,20,363,166,833,284,800,71,61,656,596,357,362,13,904,902,712,561,22,976,238,963,745,912,961,978,643,956,746,531,725,459,531,144,741,532,230,605,944,79,459,531,542,348,271,405,960,322,295,825,825,893,866,921,234,371,109,889,583,171,361,522,74,726,495,176,703,618,741,455,998,765,671,777,923,325,874,938,876,981,268,677,83,988,484,289,896,32,950,867,743,280,501,923,248",
    "157,813,539,383,836,779,264,285,738,521,797,879,637,714,818,509,450,274,640,558,585,434,836,513,727,901,368,677,658,48,232,296,767,517,633,274,948,486,719,611,875,736,359,788,94,577,427,932,31,41,664,135,926,804,651,340,702,785,0,143,458,78,679,687,713,746,498,680,280,15,985,569,53,289,244,846,563,150,528,69,31,618,387,557,799,67,926,332,918,585,473,673,690,673,873,632,310,31,326,811,128,758,878,279,608,676,90,920,35,667,980,345,523,166,903,746,456,369,438,949,202,442,676,163,157,722,949,623,425,479,669,902,277,866,356,214,996,704,649,933,669,605,814,492,403,37,985,213,796,699,547,279,897,451,613,305,914,744,681,30,451,633,358,173,1000,824,160,187,770,720,510,373,629,818,486,526,803,556,424,880,303,925,880,353,166,948,910,6,627,541,797,308,103,708,910,423,208,46,606,399,671,864,233,218,812,753,765,815,516,521,779,843,640,515,157,43,564,251,732,151,641,271,816,689,352,964,229,172,579,894,756,763,25,18,241,621,344,899,391,412,770,860,543,209,185,116,377,703,702,403,387,845,285,113,349,41,380,48,779,558,678,266,987,139,953,255,564,850,29,103,319,807,180,57,256,935,71,830,766,360,81,890,44,48,988,19,951,209,925,815,465,680,463,171,443,413,909,940,135,698,759,82,96,884,447,638,773,337,618,978,964,502,947,143,526,805,615,315,701,146,610,806,511,147,485,6,699,213,343,28,184,827,692,415,608,673,978,725,104,107,821,169,120,432,147,101,373,688,848,697,993,106,502,332,399,699,258,208,464,710,499,983,986,683,547,17,486,41,544,812,903,212,286,999,608,905,135,59,652,650,118,177,524,580,441,682,165,378,882,517,961,94,388,844,68,179,780,412,50,21,872,430,109,948,505,973,918,437,14,982,566,196,697,749,857,669,49,470,132,456,920,319,153,740,940,404,363,193,493,11,198,950,505,367,534,464,130,736,145,487,270,248,323,77,11,827,62,749,428,828,547,878,169,28,221,384,652,32,469,966,677,952,656,548,140,441,808,420,558,18,415,906,194,365,883,94,674,721,956,354,92,126,707,154,986,301,238,961,463,303,946,492,716,681,276,383,99,24,3,682,469,569,587,221,856,527,379,905,998,971,71,909,803,527,721,952,239,70,339,230,933,450,142,788,937,571,14,91,571,794,164,425,311,34,726,876,938,896,653,94,336,331,376,517,46,983,699,93,290,52,127,872,552,874,720,311,913,912,393,921,94,550,397,536,364,227,1,777,442,527,334,106,361,895,876,101,78,337,531,255,679,67,107,27,135,272,208,941,150,296,951,821,13,855,347,348,475,515,617,187,453,484,838,822,112,75,786,152,955,918,765,404,948,912,874,392,771,162,740,566,704,71,923,852,305,397,407,928,608,150,289,947,786,918,302,814,308,417,379,878,166,625,560,17,846,259,342,551,137,452,295,137,315,291,888,680,761,726,723,960,632,860,993,236,86,831,676,359,899,406,477,983,89,386,929,295,713,473,640,220,670,726,826,716,320,419,136,436,867,357,557,512,887,843,441,130,81,672,711,609,754,387,162,494,958,475,651,502,537,814,202,499,299,393,590,961,773,483,27,28,10,716,313,653,514,569,701,16,893,855,213,342,499,280,833,59,920,182,554,195,836,125,956,730,354,36,166,64,240,701,271,66,555,322,861,724,10,831,305,797,550,782,941,807,257,804,645,643,460,156,368,233,44,49,68,4,372,476,134,470,423,280,610,255,382,825,328,722,91,338,766,777,286,968,952,175,265,926,877,152,15,832,645,823,587,431,881,830,380,987,234,955,33,825,88,400,502,645,940,994,400,932,10,466,811,696,312,235,348,342,34,592,288,637,49,275,243,295,210,324,114,562,472,920,227,966,409,78,135,146,462,128,482,480,91,266,658,287,278,706,789,925,81,234,868,851,719,124,238,787,926,135,974,980,758,734,120,869,30,551,974,484,827,482,111,611,918,379,676,660,511,458,563,914,364,611,253,307,288,986,133,830,932,640,916,303,536,914,953,452,908,724,810,354,549,917,529,18,318,806,722,977,233,157,624,597,958,20,775,331,43,276,612,791,134,749,203,657,660,884,1,462,508,606,851,858,124,969,13,500,334,224,172,850,605,927,469,70,489,292,839,759,956,405,424,30,151,757,820,626,83,601,683,789,202,789,726,48,834,220,116,2,54,465,842,240,251,866,319,340,869,566,604,518,645,344,942,898,67,232",
    "770,730,565,658,911,556,139,492,930,745,300,204,281,44,8,431,955,827,155,419,207,265,308,915,210,265,168,690,25,14,567,286,882,993,805,333,617,424,23,451,793,685,786,167,452,446,343,68,626,405,102,927,37,944,186,118,97,532,143,0,474,892,778,882,708,395,346,463,227,841,694,961,10,49,672,399,236,928,71,180,473,892,306,831,264,29,999,763,361,632,201,213,153,805,974,866,539,707,391,398,151,192,917,7,480,743,833,550,700,76,330,951,566,957,872,530,749,306,570,98,706,85,929,659,440,155,281,849,927,539,547,255,419,187,494,54,144,326,924,665,858,450,500,945,652,65,639,161,24,853,336,434,858,166,199,835,978,404,247,17,708,443,595,916,410,122,444,446,408,586,901,837,837,345,101,284,373,148,382,192,489,690,979,340,702,940,449,585,550,879,770,639,867,840,264,376,6,235,726,376,32,470,633,269,996,550,587,183,730,856,18,149,361,11,718,303,840,219,20,933,653,790,541,869,197,544,58,305,79,289,223,576,572,944,536,478,594,922,233,344,512,724,78,639,170,540,884,84,778,79,743,144,453,898,690,241,409,738,836,389,549,893,216,675,498,452,438,919,594,886,711,391,831,884,18,407,585,236,688,215,299,652,25,739,890,939,66,742,720,541,513,735,782,370,170,511,54,683,771,313,861,67,1,81,293,507,515,683,133,148,497,599,688,950,561,194,587,297,526,847,506,452,104,132,972,934,438,501,628,694,744,366,299,12,306,770,507,679,860,349,525,895,677,47,158,61,91,261,674,175,629,166,459,459,580,621,990,31,665,645,920,240,297,166,610,219,784,882,184,6,993,640,18,294,126,442,286,169,597,458,935,440,156,34,650,205,982,338,947,679,348,251,642,100,658,725,286,775,645,46,677,473,186,569,98,590,143,912,198,15,382,543,125,994,166,95,206,887,823,373,321,455,571,759,265,161,507,476,448,655,145,956,634,92,636,303,461,723,689,43,836,28,443,921,286,664,68,429,398,218,907,171,22,292,367,48,706,293,861,618,679,503,309,443,788,639,720,866,62,748,583,505,851,376,266,621,542,590,629,389,957,101,666,724,61,666,437,685,833,44,738,728,83,930,28,550,622,657,135,26,335,570,985,702,69,854,395,709,202,841,260,606,442,290,536,589,310,494,617,837,986,914,282,679,391,80,125,699,935,406,408,574,313,848,227,949,587,986,597,409,626,789,174,124,390,808,373,177,158,979,631,530,264,253,567,880,804,911,544,649,288,59,454,228,641,738,592,964,861,445,806,951,880,819,526,150,539,448,245,669,111,490,925,52,65,797,608,396,247,37,839,525,579,409,739,962,766,409,355,200,462,947,678,954,833,605,531,537,227,440,706,108,395,555,881,556,747,28,263,536,652,867,231,120,658,333,517,336,906,272,318,904,127,651,256,108,453,951,745,980,486,246,138,770,701,614,14,686,593,503,410,994,304,984,535,648,84,50,703,528,639,155,400,378,864,862,836,654,891,997,112,21,541,360,701,239,749,170,74,327,171,672,374,711,640,776,875,832,918,418,903,209,503,930,274,569,300,422,684,244,165,505,901,102,998,72,547,504,309,67,157,964,822,184,879,513,596,769,728,38,739,162,782,575,777,159,73,226,408,527,717,731,112,32,137,199,129,522,978,505,159,496,366,795,442,369,750,57,382,271,197,777,223,749,672,133,411,151,682,843,675,514,163,676,872,988,236,751,244,703,998,765,736,613,561,100,817,81,34,35,469,819,953,62,865,888,833,73,245,31,381,428,173,599,698,633,202,658,244,551,779,98,425,368,16,861,818,688,914,898,937,908,473,571,59,205,401,940,866,620,518,510,65,828,691,941,802,66,985,475,792,833,49,993,892,630,369,234,987,896,546,733,236,374,873,473,388,439,963,967,835,420,728,898,81,304,148,924,607,848,700,314,965,228,351,837,974,672,351,509,248,932,497,659,22,690,504,437,491,956,145,110,995,50,265,720,391,158,662,131,520,431,691,307,170,859,402,420,614,621,845,237,148,943,207,656,748,329,189,61,29,841,369,265,358,645,948,138,498,555,588,311,936,253,293,352,928,111,343,887,250,814,440,194,285,770,268,213,498,327,348,22,916,820,110,330,821,361,723,16,743,338,776,319,132,482,779,879,891,184,798,256,256,184,109,562,986,385,781,216,828,895,374,174,91,570,830,490,15,951,154,121,508,769,97,795,583,799,502,649,530,360,652,332,788,288,468,55,571,469",
    "29,589,562,89,443,782,974,768,945,991,43,434,979,118,865,992,448,124,46,182,301,90,277,810,537,581,156,544,157,777,492,786,808,519,541,241,440,981,701,992,422,678,574,73,300,911,591,722,434,468,343,358,841,606,9,864,475,196,458,474,0,791,289,974,149,803,336,442,841,606,101,621,223,780,507,550,948,226,847,820,671,305,898,191,253,860,864,652,188,85,573,33,521,341,275,682,697,830,840,928,603,740,311,26,8,40,154,952,387,712,211,645,321,749,128,313,142,513,590,839,453,376,659,954,989,346,859,269,978,470,738,815,818,264,490,418,411,939,842,240,849,430,735,475,377,300,149,830,710,577,574,535,923,772,253,902,306,657,561,137,842,603,358,40,282,102,813,684,227,268,595,387,453,75,798,379,404,226,268,525,553,79,92,153,310,797,36,523,495,88,353,521,370,545,778,318,504,946,428,707,711,612,85,81,571,91,818,344,916,724,356,520,863,77,633,446,606,867,312,884,805,358,243,22,888,986,405,332,989,912,80,274,441,727,416,7,783,925,398,65,885,763,419,660,373,769,768,452,887,414,301,202,34,240,984,800,192,44,178,743,112,576,988,924,597,684,681,146,778,612,342,667,336,842,797,943,943,191,739,897,815,269,166,4,108,87,992,781,186,279,121,534,291,10,942,855,140,347,280,316,908,826,125,374,308,116,899,527,913,524,25,965,164,702,362,778,159,859,728,494,473,893,400,916,668,409,57,437,426,570,914,894,468,66,673,391,297,913,149,158,118,514,866,280,105,495,594,718,732,849,99,398,393,240,575,14,447,94,513,359,615,602,633,578,152,828,19,965,623,212,301,199,849,257,306,786,241,434,982,311,795,947,249,764,4,835,48,563,574,611,951,370,769,877,750,87,181,601,638,703,474,905,808,116,450,676,873,258,575,687,404,277,693,924,187,725,334,854,210,579,339,884,282,336,653,432,627,906,13,791,542,494,688,298,887,132,103,611,905,246,406,254,696,418,921,310,395,674,978,776,777,374,982,395,523,149,933,657,400,559,842,29,525,162,867,872,383,738,32,37,814,833,489,788,15,927,979,932,170,249,323,994,46,208,376,890,322,241,181,874,855,475,97,246,816,99,407,996,759,477,293,264,428,438,904,16,260,600,944,452,330,68,645,180,151,93,115,811,962,375,113,456,634,320,119,679,3,327,678,589,410,347,477,17,138,351,108,240,770,65,327,704,126,552,470,887,304,505,885,850,250,400,354,464,500,150,647,971,496,98,289,450,182,167,76,604,434,225,875,790,192,457,896,932,558,250,893,982,941,47,738,512,300,745,684,634,264,265,632,676,730,880,187,142,704,383,255,101,195,363,900,235,549,524,259,835,961,983,97,771,253,50,662,880,309,582,308,493,216,166,194,889,610,899,799,735,506,482,103,373,711,71,503,744,765,229,9,74,154,304,130,563,271,938,449,669,431,569,957,53,541,879,446,351,283,531,752,578,332,527,513,593,996,137,637,603,308,839,341,850,797,43,803,96,635,111,549,481,108,865,649,878,191,707,352,214,43,883,430,917,316,230,580,944,65,875,503,358,665,92,528,29,731,252,87,586,462,730,475,484,208,91,552,144,528,272,364,328,440,848,909,163,329,681,259,122,34,291,311,570,860,521,187,685,566,201,882,761,293,722,434,883,597,538,173,369,725,760,604,350,552,944,464,8,280,188,59,182,308,345,675,284,891,54,368,938,166,600,15,968,389,364,718,286,946,223,307,183,387,571,996,448,183,411,263,497,405,784,196,576,211,526,853,657,287,115,262,114,785,860,254,674,95,245,494,134,390,748,430,830,941,768,590,118,477,278,896,500,737,490,303,528,968,648,415,624,552,79,143,241,591,746,681,455,361,759,197,15,59,336,663,515,736,407,964,503,854,549,863,693,323,798,648,948,479,51,740,384,917,685,955,953,560,385,787,443,790,623,366,120,911,598,181,919,42,147,6,691,915,874,669,110,499,632,271,464,717,672,945,320,937,631,20,645,8,546,957,392,269,73,986,98,49,843,84,181,401,567,338,959,860,860,459,731,392,333,546,307,413,458,784,859,735,924,567,415,786,779,985,406,966,17,830,358,102,527,831,892,353,711,834,557,534,654,465,78,101,619,888,662,802,609,605,431,92,525,662,841,461,744,619,516,824,268,959,560,511,286,960,489,991,341,850,307,181,915,545,831,104,881,536,930,317,579,250,923,301,296,574,979,576,931,440,165,448,186,299,796,128,342",
    "97,978,424,188,87,393,339,785,404,195,899,514,87,891,416,697,809,507,208,611,445,699,187,455,641,739,661,857,550,968,30,838,573,46,909,510,43,809,861,42,661,902,682,421,576,730,541,984,987,844,365,213,201,503,900,315,182,233,78,892,791,0,234,465,805,969,292,980,982,742,792,352,781,175,682,857,549,758,270,740,76,280,471,374,415,259,524,635,97,241,677,866,824,905,667,952,341,224,66,229,902,333,837,99,49,565,893,817,58,819,769,1000,273,136,478,373,948,908,807,788,722,477,588,432,56,8,245,347,926,80,734,78,553,238,860,248,653,229,852,873,110,123,267,781,621,949,368,667,36,339,692,374,513,388,14,383,918,908,462,80,877,144,111,306,692,334,357,239,873,901,691,97,783,815,688,208,842,25,583,574,92,284,724,62,836,946,834,400,761,37,119,272,647,442,814,679,765,712,348,756,299,758,110,968,570,518,504,998,682,297,339,487,53,107,629,562,856,450,122,64,48,768,190,332,131,562,450,484,834,318,63,245,745,884,963,714,688,83,901,181,848,957,894,561,253,119,113,105,498,262,223,787,191,971,153,9,534,112,247,964,702,542,694,997,11,575,896,230,345,76,341,714,375,104,341,201,914,653,155,71,709,604,658,575,739,382,359,582,538,691,547,468,588,300,106,519,775,277,517,160,480,118,945,363,249,989,539,997,132,264,515,605,741,578,213,675,202,63,868,998,534,143,229,55,340,687,460,410,468,251,804,124,93,529,491,667,69,174,48,671,58,878,182,848,720,671,327,606,453,832,768,926,196,928,889,463,558,248,92,338,243,13,626,990,482,456,621,772,863,476,245,19,645,843,92,614,99,959,904,51,763,753,864,159,412,322,543,812,990,336,822,626,69,654,63,513,806,575,613,350,368,349,133,352,184,583,543,936,145,71,698,966,271,212,638,544,84,406,349,532,348,898,687,382,954,926,477,5,34,313,460,119,112,563,554,458,329,371,247,316,938,668,102,690,46,363,258,600,159,38,219,167,282,159,13,393,304,590,770,375,958,977,284,921,576,406,209,734,727,987,425,846,131,253,393,682,609,838,450,312,795,385,865,73,714,256,464,280,469,181,824,323,729,405,494,667,451,168,323,743,305,439,593,21,886,702,820,764,806,469,993,584,124,294,408,59,290,267,451,530,589,753,613,497,631,652,832,648,645,33,476,665,715,613,446,548,57,746,326,118,903,358,377,44,105,165,24,779,660,609,691,635,265,87,304,373,60,256,437,134,199,349,704,890,710,534,985,602,870,305,846,423,771,603,893,252,596,339,566,348,712,518,842,759,11,947,437,250,510,410,823,552,181,357,911,546,366,225,179,669,268,479,230,24,700,426,887,536,990,935,816,471,908,681,304,915,793,220,837,654,588,721,248,801,571,258,853,772,164,322,474,314,80,750,264,945,884,485,428,29,920,889,826,322,512,735,90,834,5,206,723,526,592,921,903,531,809,636,441,726,640,507,656,138,305,58,152,410,454,423,4,429,86,669,151,181,746,888,293,969,983,977,696,643,155,992,721,994,557,291,71,791,638,304,366,518,687,929,530,656,86,863,314,383,713,134,720,110,90,594,276,421,908,819,607,431,300,547,340,904,273,799,426,70,911,951,365,171,644,576,210,370,214,143,408,892,555,35,921,76,367,900,72,998,602,821,952,410,503,63,856,47,902,70,671,74,546,869,891,577,508,358,817,827,37,298,542,372,237,80,761,329,420,590,87,132,23,36,314,800,290,296,271,316,553,2,445,256,918,873,877,954,746,473,980,346,47,976,350,441,186,526,26,171,730,208,645,380,550,7,348,277,247,384,375,931,270,608,592,860,901,505,388,353,41,539,848,804,559,990,32,818,191,181,578,350,692,23,88,277,802,417,269,121,743,625,924,513,953,564,76,464,601,892,679,805,266,850,4,689,70,277,779,92,645,848,11,81,676,527,634,713,538,4,105,840,814,82,906,499,165,372,858,816,695,876,407,651,912,866,158,323,248,898,524,574,190,38,956,208,427,745,984,899,138,876,427,22,488,777,260,149,976,555,765,561,399,25,35,476,965,745,924,982,513,900,493,424,967,194,881,293,341,246,566,469,338,635,353,565,725,222,401,382,367,322,188,320,593,633,953,940,347,786,613,42,175,976,976,374,414,905,652,332,688,353,857,366,226,579,212,754,298,6,962,789,75,138,20,7,435,148,899,795,644,755,85,2,378,423,68,616,992,164,721,948,766,82,111,863",
    "229,214,973,122,176,576,715,292,533,227,845,32,554,255,795,80,805,726,705,5,390,838,821,175,54,891,579,607,202,274,241,389,42,217,534,709,730,67,760,846,77,215,838,90,642,121,42,385,994,833,762,286,449,596,286,648,555,290,679,778,289,234,0,712,545,46,26,566,549,393,476,824,79,281,471,847,614,977,742,537,463,94,242,423,60,657,177,566,584,830,849,879,659,451,334,493,529,234,837,970,913,312,195,637,18,156,11,913,15,907,844,825,672,180,969,879,609,894,246,662,317,630,920,798,343,492,957,707,194,475,680,990,450,338,374,768,96,649,610,597,612,898,859,502,420,39,144,290,302,797,837,442,90,54,514,250,932,951,500,882,13,497,994,568,194,252,687,298,174,552,78,419,117,431,227,306,399,139,538,694,598,816,877,166,953,738,334,381,613,428,794,415,381,401,743,645,904,468,427,141,16,995,878,439,707,944,721,880,906,771,682,864,136,648,674,27,817,901,892,501,797,267,34,964,589,289,842,359,60,980,671,881,330,882,916,688,807,613,83,444,24,325,969,230,853,747,350,758,724,664,287,900,474,91,235,879,557,499,601,891,494,387,977,513,122,78,725,800,52,266,621,87,527,37,375,635,333,268,656,727,262,708,642,611,600,136,600,408,658,962,627,992,660,158,987,90,493,234,737,866,61,193,847,916,813,145,697,458,924,968,331,872,174,330,485,403,856,130,397,245,894,808,629,270,660,884,365,718,304,176,996,657,156,611,586,862,813,484,226,629,508,870,450,454,117,138,468,691,264,155,895,593,464,747,414,481,389,880,833,46,305,496,858,834,561,127,158,95,636,183,776,553,858,548,635,957,653,123,352,824,545,252,58,460,996,697,329,250,906,632,606,844,337,277,399,488,726,456,81,316,139,377,99,549,864,838,69,104,830,409,376,681,991,582,347,422,380,928,307,268,231,980,490,886,453,300,454,273,403,634,697,504,221,668,576,644,43,556,275,696,46,848,366,357,416,504,122,282,580,714,893,108,776,799,734,807,998,319,18,748,544,812,557,409,141,984,209,434,234,955,788,916,137,739,596,642,597,147,55,266,198,253,295,427,134,369,196,867,470,179,705,173,874,199,692,380,575,307,44,925,286,134,997,111,80,759,819,218,857,805,196,644,442,55,877,386,329,770,377,986,39,385,320,162,695,87,102,343,483,905,280,24,646,516,460,744,244,399,78,781,202,818,216,148,421,229,29,926,22,633,801,942,970,510,890,182,103,490,559,46,719,442,773,767,804,194,297,81,760,490,953,522,614,843,68,416,286,150,687,719,361,746,23,967,86,150,25,709,304,522,559,946,821,526,548,640,763,206,833,542,467,278,231,544,536,746,258,750,122,840,622,471,18,481,657,925,297,111,263,501,688,129,980,990,824,564,339,12,232,828,649,86,9,317,890,566,944,543,84,517,322,73,329,50,504,942,594,227,14,658,284,51,738,391,717,893,870,217,616,583,759,581,46,302,71,149,271,992,446,726,327,471,417,291,432,102,726,937,646,602,847,873,564,75,414,289,759,930,913,91,136,220,946,641,751,345,256,349,564,887,761,653,42,422,260,819,583,790,756,884,464,687,505,442,562,776,634,399,491,147,561,844,219,905,294,345,283,383,533,929,456,62,950,225,86,821,435,895,426,599,563,712,926,22,640,812,266,589,529,69,552,857,580,894,132,859,495,418,189,324,356,269,901,842,499,565,818,77,575,563,38,87,602,66,36,96,581,204,28,582,538,120,322,884,760,933,211,549,144,61,802,470,867,873,755,715,419,9,509,351,731,44,134,448,754,829,402,255,137,284,638,816,399,256,314,62,609,621,608,491,35,450,137,23,413,354,519,304,25,213,175,952,774,972,115,846,356,637,295,236,546,109,240,545,205,636,501,122,688,601,979,861,27,530,927,560,701,737,760,379,806,211,673,991,932,870,528,710,369,856,953,999,61,742,695,158,552,856,163,936,382,505,955,974,99,872,480,551,832,951,496,736,909,592,912,256,594,122,884,298,425,312,34,819,785,960,205,127,600,54,566,142,70,567,690,165,51,69,984,988,331,338,101,658,270,327,596,375,747,220,434,758,613,269,940,552,828,967,929,84,342,475,171,418,457,768,705,131,728,835,397,670,777,606,692,770,153,576,603,349,857,419,799,486,132,704,309,193,174,369,881,567,647,475,82,151,218,64,610,386,111,92,521,615,530,961,748,826,171,625,692,331,638,118,382,20,923,53,273,439",
    "734,629,187,919,376,675,810,660,218,791,307,906,476,22,542,283,954,978,846,695,678,363,18,3,493,25,260,617,615,799,888,175,862,328,304,631,703,431,289,387,655,835,848,402,695,10,786,833,786,882,658,258,112,459,411,709,966,69,687,882,974,465,712,0,739,89,31,583,342,766,330,753,614,371,646,418,50,260,811,127,182,446,71,913,538,225,789,357,728,51,758,732,22,89,820,376,600,610,91,6,163,365,546,569,448,195,574,259,557,91,44,2,916,523,369,321,783,24,670,699,985,414,959,751,196,538,556,593,310,321,361,985,620,479,283,681,436,8,585,383,948,417,492,906,218,533,332,878,185,603,400,543,798,232,325,45,391,833,27,314,471,276,403,659,87,496,644,697,154,878,141,711,822,924,27,944,106,620,809,530,92,899,632,955,22,856,599,254,38,98,787,58,132,737,374,310,612,550,802,240,121,314,682,618,473,911,395,733,139,599,424,986,358,308,593,969,167,560,82,322,386,513,9,696,923,549,992,827,670,756,439,203,312,566,170,848,911,387,238,484,707,553,554,457,860,981,452,597,299,262,587,796,114,553,808,530,171,204,914,333,462,440,569,742,825,170,860,593,318,784,386,918,649,145,327,279,97,478,528,510,819,78,563,151,724,143,549,67,646,323,700,87,139,460,716,123,926,801,638,652,955,129,143,542,327,336,196,464,432,952,556,129,199,137,555,499,612,612,846,316,665,64,825,118,448,106,945,864,32,753,155,5,810,449,761,374,925,712,578,314,339,612,296,878,861,213,260,419,369,75,304,460,952,273,519,290,311,429,733,202,875,867,918,236,483,345,38,572,272,864,425,486,186,938,295,579,574,866,356,169,897,809,167,289,383,68,236,96,552,701,710,97,461,8,402,866,880,642,920,707,889,208,260,749,563,590,361,142,601,256,811,567,679,58,404,981,23,659,28,815,386,855,131,261,787,197,254,827,572,666,575,365,305,361,280,440,659,732,615,66,511,494,120,248,678,504,476,956,910,924,742,90,49,890,434,537,327,90,579,886,228,377,606,18,136,798,138,747,857,925,683,465,529,739,32,334,487,241,904,6,744,700,851,5,874,865,515,45,186,154,326,206,214,668,354,732,468,563,862,512,182,272,787,90,726,891,637,875,959,388,691,318,541,191,888,23,602,624,929,866,300,426,638,415,379,826,226,671,794,980,624,459,900,495,42,844,359,543,109,137,100,429,480,928,545,858,369,707,526,809,700,968,149,853,778,425,345,938,581,889,330,484,377,86,671,970,728,726,548,558,97,798,34,834,934,787,890,557,680,12,773,489,878,384,614,623,333,152,675,746,776,3,417,751,803,531,684,45,196,537,295,641,425,570,677,194,658,449,351,241,660,893,631,91,92,330,296,488,754,713,989,144,247,34,673,277,386,415,310,787,930,592,450,652,486,664,970,701,317,230,704,397,162,496,579,460,133,348,826,980,112,993,69,387,328,28,972,67,469,970,242,29,62,959,269,378,606,967,443,329,607,357,947,486,201,653,305,445,461,887,630,462,944,118,211,337,587,333,736,913,505,340,781,163,658,547,146,702,243,688,19,196,587,719,241,77,66,582,30,669,853,382,962,255,348,351,890,310,201,897,346,169,595,596,281,80,67,65,441,319,450,282,695,422,620,610,181,602,927,947,450,388,949,223,357,605,182,250,440,102,790,472,123,516,174,159,317,366,596,81,855,11,861,61,14,512,340,986,842,676,900,112,22,847,574,645,329,593,503,899,149,130,173,109,312,262,863,88,593,915,857,778,647,987,130,85,138,422,755,199,934,894,149,686,63,250,635,670,135,479,846,724,732,75,11,743,254,350,498,240,543,712,245,652,565,862,48,185,643,561,667,100,812,961,753,984,945,667,910,147,410,165,116,245,297,236,937,615,421,183,983,380,957,179,711,50,228,609,190,828,336,990,259,913,289,214,303,943,445,868,261,29,265,705,568,865,330,679,904,686,575,618,868,473,643,110,815,970,558,877,332,882,191,11,615,66,664,446,15,585,252,172,665,728,64,719,576,663,364,399,328,495,437,898,495,318,672,895,152,669,707,632,976,311,101,882,825,117,906,271,365,65,190,399,367,235,434,402,881,901,879,638,517,500,747,410,783,4,953,372,367,821,245,79,568,642,59,522,235,140,201,279,797,169,879,643,716,874,329,86,124,748,378,134,486,345,399,102,715,491,863,925,985,83,64,279,762,551,181,998,724,337,782,546,13,190,272,163,607,446",
    "829,781,553,600,531,456,819,971,244,853,420,660,847,446,26,626,160,615,944,103,550,740,426,589,515,734,273,767,371,314,678,673,473,427,818,550,481,167,366,568,499,136,727,435,641,666,982,142,590,61,710,314,87,776,567,634,340,745,713,708,149,805,545,739,0,808,797,627,19,445,180,328,617,876,310,901,870,922,11,738,770,914,728,195,986,869,457,253,698,536,710,600,594,966,967,381,729,5,209,745,93,86,155,492,143,117,864,902,479,842,215,399,249,326,697,410,58,578,467,450,275,955,520,18,365,803,861,240,133,74,51,872,617,317,385,231,996,279,751,857,635,994,255,386,83,124,545,585,979,27,122,245,84,376,727,389,169,939,26,9,113,917,1,27,132,472,939,246,334,184,928,343,179,406,330,844,574,798,373,9,777,505,355,907,641,714,13,397,742,600,190,941,820,336,19,319,607,576,349,254,923,947,590,175,551,106,509,440,69,131,297,569,198,601,680,191,209,371,41,847,104,945,329,971,872,467,374,37,760,197,114,343,404,101,814,223,505,7,419,792,954,971,779,759,331,24,579,251,713,321,972,583,690,706,373,937,1000,678,413,242,4,941,625,204,884,372,985,696,4,15,983,551,984,632,352,745,122,667,957,13,769,35,539,288,396,707,7,775,365,410,2,351,127,998,415,108,954,693,523,877,640,63,383,354,71,772,772,607,120,398,765,82,590,170,650,441,238,995,993,717,610,564,350,246,850,815,181,230,554,392,685,450,180,129,92,86,914,91,298,352,462,111,182,116,802,59,139,68,62,504,603,8,931,549,663,886,730,789,818,938,556,301,83,208,692,896,331,46,553,722,687,756,990,504,723,6,906,879,877,32,772,154,39,321,111,227,67,366,462,212,29,986,741,976,209,115,992,136,478,637,757,930,57,505,784,213,735,707,910,915,163,896,710,7,651,870,637,151,967,712,248,791,47,364,453,359,865,303,979,450,901,884,759,10,863,260,235,119,454,134,681,771,520,369,580,770,163,516,58,44,194,842,923,987,425,306,210,114,387,345,283,632,406,769,676,143,457,75,890,192,73,548,574,815,612,365,717,779,191,984,747,313,709,55,917,409,852,858,210,78,904,757,943,278,488,386,930,926,472,3,528,157,245,766,754,326,747,907,35,624,841,188,345,251,911,901,799,618,955,437,382,233,9,679,129,519,859,348,350,625,415,175,6,6,330,468,163,618,555,327,971,498,494,948,320,239,745,278,103,509,945,773,828,313,309,984,845,128,764,506,625,255,865,797,323,196,995,390,36,308,24,258,690,542,759,461,719,170,347,253,590,79,349,783,249,693,589,222,802,679,23,90,704,361,450,118,194,391,27,517,375,461,527,562,50,783,792,286,23,406,183,386,980,908,379,558,415,764,950,580,692,330,325,83,461,66,655,509,129,818,571,439,598,269,344,848,275,456,332,402,228,853,769,354,73,381,915,774,133,360,783,670,101,411,794,951,149,280,113,555,507,515,193,39,447,488,787,980,589,643,703,383,106,803,969,401,982,773,848,768,865,683,939,793,155,743,673,970,168,533,106,70,743,641,348,665,389,585,70,583,469,104,129,117,991,843,834,310,235,66,597,300,82,809,322,662,233,857,441,466,70,625,20,177,418,380,919,856,65,269,554,495,808,584,724,631,952,407,870,773,388,706,809,374,846,502,769,356,802,720,579,52,910,918,58,468,575,85,806,868,199,848,879,767,76,451,681,476,176,146,606,942,441,233,548,536,68,352,445,908,234,244,906,948,624,284,399,218,442,441,104,572,526,138,397,344,501,604,168,166,579,213,808,139,227,238,813,755,297,363,18,37,142,207,554,930,262,601,812,654,621,637,699,825,972,137,811,576,962,40,428,569,725,481,596,476,394,555,887,188,623,433,851,323,310,467,467,417,946,445,294,612,8,704,537,554,727,985,263,158,362,114,181,564,339,190,753,103,794,11,11,993,507,31,253,764,235,199,965,638,483,96,637,933,382,896,92,764,127,295,843,82,649,384,302,687,694,783,382,744,571,527,954,113,409,209,219,920,375,973,306,344,303,297,217,243,193,259,988,916,404,851,547,205,408,326,819,490,590,363,263,831,973,982,474,176,484,98,886,380,921,120,591,221,573,240,522,569,822,280,952,383,710,813,111,860,450,828,687,157,920,923,280,146,931,891,737,538,915,718,571,277,632,377,845,609,77,6,773,213,17,522,473,770,803,376,396,559,826,487,239,482,130,258,219,554,335,355,584,5",
    "324,358,859,45,656,939,945,74,681,739,634,831,195,569,225,159,645,539,668,414,584,377,800,602,412,46,700,644,574,56,871,803,453,435,730,340,131,780,519,305,786,218,170,989,548,706,982,311,488,303,988,313,780,297,180,385,865,181,746,395,803,969,46,89,808,0,405,196,872,53,810,342,341,40,31,677,390,974,42,361,27,624,334,893,32,603,38,414,588,33,126,437,142,337,40,161,737,188,374,215,817,33,532,37,221,738,37,77,498,519,129,809,396,904,92,642,819,199,920,900,47,848,2,564,416,123,103,955,907,745,929,297,902,920,694,47,703,285,905,226,977,449,73,28,173,403,620,459,175,227,52,173,37,67,190,54,751,944,813,23,128,458,988,340,853,317,456,55,538,541,309,465,466,107,16,847,675,551,109,906,119,971,551,615,943,450,617,175,900,639,549,386,39,350,197,552,322,14,383,904,173,889,443,312,163,53,129,636,289,395,362,485,934,96,135,393,50,827,660,460,414,88,253,796,501,437,67,662,968,482,479,285,954,587,479,312,983,667,366,38,291,333,818,158,201,658,546,915,427,737,744,612,694,482,303,540,84,740,442,330,812,979,976,12,574,887,934,135,526,349,149,957,399,891,429,496,178,488,619,91,965,669,294,561,267,13,414,419,551,784,315,767,34,591,284,263,489,228,619,684,729,697,97,324,696,715,386,271,251,17,29,186,249,642,241,320,836,624,809,629,538,552,731,675,693,76,831,734,680,821,468,916,307,743,142,465,551,784,80,302,546,665,408,804,738,134,350,429,600,988,527,827,279,197,543,7,223,548,828,714,251,567,578,413,272,600,901,333,993,429,995,750,936,341,261,546,66,726,995,682,755,225,422,306,106,600,603,199,808,329,961,187,403,725,300,329,635,825,676,133,545,938,295,438,652,458,891,900,38,702,895,748,758,719,769,248,32,216,498,972,701,783,610,753,757,734,426,707,475,568,119,869,525,455,652,695,228,381,409,878,218,895,427,363,825,791,655,454,874,790,791,84,711,391,242,294,921,650,549,908,774,846,56,69,633,334,787,22,610,814,682,504,312,581,16,988,318,331,571,909,124,650,880,201,554,297,439,285,978,550,723,636,692,188,617,949,742,594,49,826,354,616,365,662,862,488,241,895,857,436,268,813,55,751,89,415,446,579,626,871,787,862,625,162,54,324,141,211,420,728,104,695,182,865,9,871,645,439,561,321,752,452,103,269,836,1000,644,307,625,403,747,434,725,6,608,744,568,895,505,856,917,300,166,280,978,393,153,556,769,254,125,842,62,333,171,455,282,422,750,274,218,130,170,900,989,917,154,308,104,279,816,606,597,430,41,598,642,483,190,827,224,273,897,862,769,772,198,55,989,974,129,747,491,822,822,80,833,604,152,127,870,33,783,237,822,678,523,444,653,53,317,630,756,955,378,1,492,655,22,30,247,649,906,257,516,128,723,574,480,546,406,429,529,896,894,221,151,494,998,482,655,86,366,536,644,222,904,767,186,436,80,756,129,436,536,405,94,479,513,753,302,486,396,453,649,444,324,613,395,663,293,775,679,340,693,69,476,638,382,274,947,948,221,998,214,148,451,64,186,620,441,29,778,246,94,894,41,417,519,845,734,302,703,375,906,390,871,733,313,967,606,832,656,71,752,692,873,596,468,529,708,893,365,170,171,705,520,187,920,715,395,569,763,148,621,321,323,49,783,797,442,541,403,590,589,589,171,595,971,999,964,708,198,282,296,280,158,428,210,393,996,127,336,805,723,842,656,380,597,164,372,308,770,901,50,221,117,728,466,150,294,533,515,823,719,755,306,788,249,888,83,369,662,79,710,102,823,577,921,91,298,352,234,919,297,263,789,626,855,964,941,649,774,353,817,92,847,794,527,591,115,420,860,536,651,979,328,567,973,525,725,226,396,452,818,606,120,808,206,259,985,362,990,851,102,666,523,691,486,530,251,607,142,122,31,38,250,101,272,449,929,871,993,786,646,50,996,199,836,357,210,430,390,825,149,752,507,877,442,162,967,111,241,324,922,453,150,56,821,740,224,699,998,954,342,904,771,426,588,77,434,233,801,290,396,194,422,459,411,425,939,850,290,20,733,429,830,363,958,156,259,819,44,25,923,232,545,829,212,58,893,528,129,289,758,289,478,2,408,299,170,123,838,233,47,508,603,552,244,2,249,287,213,157,526,828,555,178,653,347,344,106,57,16,670,204,235,654,296,48,912,710,911,552,522,873,872,22,903,597",
    "917,941,381,473,959,788,897,410,924,158,460,705,19,488,147,963,63,989,919,217,783,333,223,871,568,598,600,882,461,563,753,246,891,133,355,107,94,782,158,771,700,592,867,509,157,829,776,526,994,311,143,386,969,349,129,641,99,16,498,346,336,292,26,31,797,405,0,901,681,982,797,778,461,609,445,24,668,1000,786,31,410,833,203,120,723,521,677,28,567,924,439,754,263,437,888,956,152,439,258,274,845,682,987,559,160,779,729,819,155,609,967,641,706,669,285,115,458,522,55,249,966,667,122,267,518,546,563,401,291,993,535,876,294,254,691,616,284,114,706,585,921,743,739,139,732,215,276,420,388,226,346,586,186,560,598,54,648,802,956,273,165,350,929,391,109,121,570,560,745,388,323,461,635,561,989,519,69,349,22,348,725,181,447,115,579,854,818,700,972,305,445,253,163,243,904,595,850,814,174,379,627,401,552,42,171,766,664,132,853,426,633,258,933,225,101,686,715,984,257,343,987,755,704,732,392,271,513,777,275,501,852,93,32,652,873,314,247,212,690,439,786,89,297,502,375,534,722,826,224,764,302,198,660,726,805,186,589,801,158,89,564,432,437,624,589,181,916,11,601,415,692,290,893,347,810,607,917,529,344,531,879,443,606,676,252,993,244,994,971,843,22,946,801,157,658,899,731,416,340,802,742,278,606,241,428,846,110,584,269,921,470,125,203,849,406,957,663,995,980,112,342,783,828,448,605,956,439,564,439,976,32,198,109,310,648,985,627,995,572,574,560,385,166,239,671,481,431,193,95,684,546,507,116,370,301,831,780,847,747,651,410,904,184,193,861,661,921,761,828,821,676,61,980,917,729,324,42,952,516,321,8,600,941,245,376,663,186,320,318,886,723,291,986,480,126,492,77,198,787,820,672,95,353,967,792,456,924,364,951,619,973,205,434,194,131,228,114,965,57,33,876,259,965,674,806,876,461,191,880,405,20,747,453,13,415,946,919,496,946,357,982,507,392,697,558,913,442,254,109,973,7,495,657,675,846,578,447,726,954,711,652,815,444,187,890,744,165,660,112,910,713,96,832,409,417,312,222,832,227,57,364,751,43,194,680,120,712,703,655,597,930,259,755,478,829,373,294,548,490,765,376,122,167,328,363,863,351,502,801,627,100,779,498,939,590,486,185,407,260,315,723,871,108,593,290,373,253,826,791,355,409,143,957,806,381,866,246,563,49,847,606,201,957,502,976,128,463,223,797,203,785,446,444,627,731,379,892,119,727,423,877,426,968,767,854,365,125,138,593,635,436,503,318,618,38,421,988,852,294,864,311,624,724,281,310,262,9,908,193,633,630,467,20,523,100,520,896,149,120,576,791,392,81,640,90,722,9,400,875,561,508,98,968,106,76,980,979,720,788,634,77,314,107,231,601,479,872,478,316,54,893,984,219,143,196,323,641,382,992,134,47,940,604,800,488,498,98,431,787,335,339,233,898,251,953,321,78,696,617,770,393,8,3,42,160,982,142,264,257,263,908,894,299,963,778,244,565,523,693,652,395,740,788,3,986,597,565,422,418,887,569,842,421,587,324,4,781,922,523,411,73,380,968,232,869,675,108,239,456,650,678,694,151,388,431,286,880,264,646,701,867,214,220,602,746,782,325,484,756,426,927,14,456,958,152,967,890,248,564,814,706,312,516,775,269,309,798,777,528,555,976,419,29,397,591,556,415,810,514,847,915,295,364,45,481,484,173,208,761,621,675,246,217,629,332,666,526,951,348,152,505,166,310,167,11,603,426,694,379,918,623,485,111,654,361,725,88,756,272,491,699,365,538,515,73,937,588,685,331,926,418,57,760,825,471,308,868,317,481,75,327,73,633,496,299,175,195,537,305,341,820,209,424,743,25,755,889,789,472,340,105,943,855,252,478,191,431,732,600,416,233,801,16,322,381,776,873,528,88,76,758,419,350,105,767,538,432,21,819,713,226,975,894,239,301,491,900,316,516,793,387,642,942,193,59,140,364,940,510,453,269,90,848,772,660,230,992,336,878,784,481,603,54,878,77,198,779,978,229,98,755,214,89,199,716,45,145,961,80,75,779,917,661,657,531,531,742,593,347,365,660,79,182,574,727,777,530,248,374,663,369,496,405,43,176,193,900,679,857,742,659,859,196,237,229,529,317,22,193,40,864,854,391,735,282,519,766,761,17,147,61,754,801,871,314,294,143,123,82,563,686,781,224,901,570,316,910,753,139,65,389,624,517,1,597,717,207,795,919,578",
    "380,962,51,37,381,887,607,654,640,898,995,293,3,980,801,640,139,338,230,81,111,222,405,700,605,521,204,454,425,488,546,587,955,32,644,628,972,171,114,189,126,401,525,423,42,350,774,407,505,633,26,345,761,623,469,168,86,761,680,463,442,980,566,583,627,196,901,0,99,973,768,99,49,130,285,432,649,634,199,608,892,124,598,771,776,991,346,21,817,628,631,866,240,494,698,524,84,656,669,941,532,383,606,291,487,676,276,622,680,360,192,60,296,860,907,891,980,683,410,870,635,749,279,427,186,250,216,397,365,393,108,219,223,446,794,78,197,723,490,374,139,234,850,634,662,85,882,951,22,596,268,18,999,466,681,165,172,212,507,740,123,109,302,141,28,201,973,885,993,241,813,322,290,181,111,738,68,426,727,147,411,881,9,551,109,370,873,168,576,551,91,423,241,605,399,321,110,14,691,849,588,917,350,115,605,282,137,950,314,872,955,958,463,467,306,686,25,824,150,24,875,858,251,128,789,637,171,521,763,165,52,556,847,362,586,730,447,690,595,195,264,896,802,257,955,378,152,551,448,594,422,753,177,500,916,379,49,780,904,397,52,836,432,857,960,793,362,881,975,144,27,736,848,248,819,554,537,36,258,15,33,876,745,301,29,207,824,803,717,782,285,579,830,71,512,815,667,637,415,805,90,852,569,440,114,466,878,897,912,932,636,416,568,702,174,640,533,189,317,702,883,126,45,927,977,525,240,499,145,44,99,746,266,662,669,372,790,939,701,497,209,697,569,86,350,794,614,848,166,206,866,403,805,638,753,983,800,513,288,229,346,548,854,771,434,87,948,641,870,564,356,600,307,329,175,214,652,548,900,231,428,657,537,165,809,7,599,872,453,760,354,785,483,245,591,489,242,209,48,582,476,718,462,626,947,942,785,921,147,932,621,596,641,171,302,879,308,803,94,504,147,461,347,279,450,491,321,160,116,181,798,13,796,325,356,732,458,106,71,479,995,267,961,177,205,599,546,228,461,448,611,970,204,628,522,468,957,437,741,476,760,720,995,78,725,939,228,23,82,379,685,637,264,57,726,585,860,848,544,594,812,901,187,600,855,884,153,334,288,483,745,29,949,103,550,637,552,828,181,438,115,259,88,539,425,868,77,636,415,91,714,58,822,440,889,600,917,634,843,668,122,759,449,271,531,84,965,701,816,572,917,93,154,695,580,49,236,216,411,69,296,494,884,983,45,422,586,418,205,718,546,591,90,70,445,275,354,520,804,962,567,378,398,681,889,83,50,201,948,884,477,847,886,607,740,869,441,954,738,801,417,604,202,758,215,959,261,176,700,625,670,920,636,440,549,131,939,277,834,526,238,568,629,701,274,373,818,119,326,51,360,745,135,143,368,393,849,63,288,986,40,717,367,258,14,711,870,954,309,712,216,677,653,53,53,819,460,905,222,766,255,531,343,350,223,708,499,692,851,415,810,878,753,852,158,628,284,897,88,224,400,148,948,914,27,853,627,543,706,492,299,241,794,165,216,445,674,600,354,120,251,396,742,137,397,909,545,760,483,196,823,492,980,700,691,139,497,958,239,740,141,2,208,217,721,745,830,123,181,36,186,922,48,232,704,206,744,807,327,157,440,603,999,156,6,440,409,888,70,485,548,184,933,545,597,205,247,656,583,650,519,343,847,893,410,591,968,710,98,287,446,604,392,834,768,328,72,567,480,977,642,510,390,826,226,399,126,565,7,618,369,677,337,156,69,986,65,664,848,540,530,413,955,705,845,232,659,202,827,474,504,777,156,232,247,740,596,333,553,759,635,263,165,655,136,980,740,253,290,216,882,566,600,21,953,933,613,108,847,615,130,250,428,201,616,418,736,706,171,43,973,912,869,149,576,673,319,394,31,762,917,918,760,405,42,929,290,387,354,736,598,316,283,448,677,837,561,822,916,933,101,355,963,509,549,872,591,854,501,421,242,783,32,981,426,415,662,385,455,499,291,104,168,128,165,344,544,305,547,705,647,387,692,699,871,70,960,642,929,713,942,827,859,631,942,484,970,904,325,148,456,851,139,420,10,439,756,308,885,960,679,171,817,534,774,156,761,131,524,787,239,630,609,301,293,872,661,36,522,206,852,443,715,33,831,176,706,646,761,890,981,808,541,771,93,878,158,11,423,86,983,861,566,501,687,829,730,810,864,876,981,809,261,441,827,66,397,813,211,624,745,921,762,168,605,64,878,523,946,351,922,154,527,818,35,245,232,514,272,554",
    "817,144,696,105,929,922,144,894,849,390,662,929,745,299,708,734,986,870,828,545,805,86,95,319,121,348,844,293,659,650,449,725,227,973,826,666,304,630,203,985,53,512,171,285,105,574,430,612,801,845,323,142,667,522,148,699,926,876,280,227,841,982,549,342,19,872,681,99,0,748,373,915,594,615,822,710,140,285,686,590,99,311,556,166,61,351,269,859,142,860,475,325,86,393,89,160,515,331,154,529,216,379,314,6,458,323,227,961,113,624,392,583,254,411,107,182,991,492,793,382,239,354,602,803,676,250,62,252,832,979,322,16,276,681,575,624,191,399,823,47,126,63,886,575,347,855,181,534,44,845,679,64,540,917,551,815,207,765,238,88,852,849,793,161,827,437,427,17,8,11,488,851,841,172,628,747,645,461,346,664,734,849,567,117,71,591,278,193,20,811,760,408,875,396,401,303,664,843,868,296,992,648,104,948,829,37,15,544,44,854,58,147,945,50,990,969,734,258,776,616,881,240,355,196,661,554,267,621,570,924,843,478,413,971,743,688,317,699,676,236,301,171,614,845,679,790,473,458,677,628,755,38,657,309,647,855,973,826,221,299,458,745,910,171,258,640,854,424,170,739,426,31,129,377,399,138,205,196,64,957,172,375,689,343,234,631,196,658,402,883,919,497,872,701,499,835,694,469,775,326,194,995,866,840,369,95,189,709,848,449,462,320,316,314,881,675,226,92,390,585,189,897,747,697,928,553,834,822,28,144,762,325,40,244,773,335,919,27,243,897,856,367,732,820,565,513,875,206,837,322,264,400,498,500,558,218,221,440,963,472,218,888,377,999,615,93,438,726,382,7,103,998,411,503,99,857,41,594,531,835,900,694,44,139,585,217,750,179,799,934,886,980,255,299,722,716,848,694,167,542,440,439,291,444,835,364,447,719,305,833,59,620,432,126,445,979,138,879,45,585,737,331,172,97,818,898,798,198,248,390,83,263,343,118,106,990,379,906,795,462,226,219,282,831,197,640,318,278,248,68,529,32,106,698,302,23,757,828,69,720,299,571,658,835,680,407,351,763,650,177,282,413,516,159,487,175,800,909,16,975,347,239,744,763,729,189,880,935,750,785,420,986,838,572,511,410,658,84,954,627,541,435,957,384,633,74,250,258,442,901,726,891,325,155,300,187,456,893,926,778,559,181,646,446,309,792,304,457,26,767,150,460,543,224,300,371,921,377,677,149,550,955,653,812,394,961,523,478,802,323,571,579,909,526,888,998,219,901,238,725,264,759,577,509,94,281,17,525,911,686,386,557,730,788,879,325,701,970,379,740,401,507,974,366,843,68,113,733,954,633,191,10,713,343,447,655,715,38,525,145,955,43,783,844,928,201,794,832,803,706,561,885,782,900,743,249,596,74,408,561,874,482,609,791,27,618,378,715,889,415,276,841,1000,220,146,513,83,131,991,101,20,808,276,270,335,800,720,715,291,523,833,972,267,937,743,767,629,77,737,380,767,762,84,813,549,538,815,238,219,118,965,122,866,672,34,564,275,922,829,54,976,89,949,55,276,197,79,717,77,206,494,221,16,729,299,497,418,567,432,722,154,851,572,703,498,865,308,729,966,790,336,653,46,970,744,509,602,569,350,201,786,749,356,249,551,918,474,306,126,720,142,245,160,283,390,126,630,9,321,268,981,167,928,219,634,547,485,537,390,796,338,538,800,834,830,984,223,978,512,822,71,729,902,603,928,391,975,205,504,402,427,471,105,955,653,686,778,352,58,448,591,923,453,237,254,454,348,995,806,647,873,788,96,779,583,982,926,41,138,168,252,496,293,550,847,571,94,2,523,284,207,392,545,961,353,889,599,557,533,521,521,178,247,786,561,50,171,140,136,655,219,40,179,228,623,893,651,325,308,146,502,253,53,970,87,821,304,16,879,891,764,494,996,217,767,931,48,72,595,823,302,743,791,89,357,550,745,632,162,587,931,137,697,909,712,301,204,61,797,520,386,470,395,845,388,862,612,800,593,963,831,70,807,100,822,310,164,367,772,946,106,464,145,209,686,779,86,795,375,649,723,367,752,499,400,38,877,935,822,7,358,905,386,296,795,746,103,664,102,92,635,86,25,261,413,730,788,894,195,788,201,717,566,629,337,424,554,701,959,780,143,147,485,918,125,382,313,45,793,69,725,31,156,992,878,873,869,309,781,925,928,52,603,459,824,709,975,876,441,275,400,390,980,118,122,473,129,214,268,496,179,481,171,467,601,586,131,661,67,817",
    "597,609,343,517,609,858,862,494,880,180,779,793,892,928,703,917,676,1,211,573,122,171,286,567,540,95,449,696,412,887,666,687,958,925,153,492,414,686,857,806,775,660,636,285,384,223,83,932,246,962,707,532,22,408,152,3,35,976,15,841,606,742,393,766,445,53,982,973,748,0,625,620,237,89,87,711,953,431,764,479,692,108,145,651,190,146,838,706,813,169,118,565,684,374,720,519,68,546,941,459,408,419,979,660,413,225,390,777,563,281,770,140,716,977,68,721,712,100,627,219,810,794,526,213,423,669,994,947,977,485,883,825,487,612,593,900,267,142,461,712,176,825,123,684,392,250,245,321,769,251,204,226,48,510,978,754,890,282,992,726,699,344,344,603,785,659,409,697,441,513,308,343,659,339,349,780,947,357,786,318,222,89,606,327,208,388,747,153,490,288,762,497,844,760,53,218,32,759,516,870,816,571,915,712,820,361,2,182,425,252,598,750,172,856,478,590,120,954,283,182,663,956,749,284,824,850,180,73,962,152,783,449,532,977,741,678,842,45,621,548,245,595,516,963,362,742,726,704,441,112,236,964,301,678,640,240,428,808,645,516,48,410,896,401,590,723,14,829,358,656,902,75,899,240,650,894,757,565,196,172,217,477,138,528,863,195,913,91,466,792,174,426,711,788,71,119,830,221,258,47,252,812,119,904,484,331,169,650,260,45,91,391,308,832,223,96,530,187,104,74,95,940,859,382,802,818,50,65,101,240,227,67,751,20,625,966,91,973,241,2,20,317,348,178,34,493,810,608,668,359,739,857,965,886,335,912,247,702,359,585,718,81,804,225,89,587,659,726,230,765,686,418,397,902,889,146,537,24,780,822,816,560,81,248,438,172,787,27,903,332,594,50,221,499,506,899,175,228,137,437,28,26,690,827,401,236,251,691,886,987,497,993,727,813,581,920,537,252,228,747,348,815,35,911,855,90,838,100,322,559,599,673,918,152,311,818,831,809,887,786,304,51,983,437,702,957,156,895,40,6,898,587,294,982,592,697,105,128,515,40,958,705,780,440,616,438,920,841,317,717,464,184,808,290,50,591,328,357,276,2,906,872,611,869,272,555,749,896,926,410,873,962,864,613,70,406,546,144,910,813,463,555,712,649,506,467,115,70,460,907,667,311,338,37,153,985,505,985,938,426,311,579,683,710,198,157,500,704,528,338,943,829,378,384,265,415,166,128,912,54,152,940,960,683,603,947,1,872,48,686,362,157,847,155,86,443,322,265,688,868,526,899,466,459,328,716,557,443,175,237,899,95,601,289,381,289,465,845,350,603,856,543,809,751,748,126,789,354,227,736,510,412,776,335,632,820,468,433,765,447,267,203,404,468,327,509,951,9,107,777,71,897,57,196,356,143,137,486,256,571,7,979,822,250,190,66,484,191,980,902,552,988,414,275,171,754,258,113,929,997,505,138,710,145,854,953,314,271,548,799,323,722,405,608,644,31,70,697,672,332,293,234,175,64,495,269,199,570,567,13,837,813,566,409,182,8,877,782,940,87,949,934,365,316,601,709,920,650,512,876,971,423,996,12,697,93,113,304,248,587,908,843,265,240,414,792,292,426,26,780,426,123,623,959,450,301,436,952,46,134,925,426,357,336,158,421,650,890,80,299,895,54,748,228,104,816,247,227,579,756,453,240,56,350,229,299,566,156,733,478,482,135,498,491,665,400,5,204,186,744,497,621,726,123,263,15,668,311,819,391,965,244,250,302,345,726,307,481,979,718,508,601,53,353,996,680,356,182,437,151,500,550,552,387,710,721,587,984,894,602,22,183,566,590,224,940,321,827,15,95,638,77,764,443,164,658,142,252,336,398,693,303,892,88,612,18,897,264,823,305,467,998,294,607,824,569,895,610,675,960,242,391,421,801,744,472,324,821,359,395,381,558,196,696,216,555,254,77,834,265,396,288,214,916,350,156,814,131,268,775,504,611,838,929,997,732,58,155,499,13,129,541,834,518,404,525,106,185,311,278,298,230,242,23,269,463,587,330,414,454,989,854,788,509,904,207,592,499,885,121,718,524,644,488,191,636,913,812,974,115,815,205,919,975,852,210,708,967,858,119,975,183,836,445,621,960,523,616,112,295,307,886,825,562,991,130,910,337,65,820,605,692,97,8,858,283,149,561,659,648,910,681,600,793,573,3,253,258,155,333,447,898,36,932,572,969,932,506,636,347,83,569,547,372,881,36,225,173,927,483,158,537,953,474,900,998,86,537,610,694",
    "461,629,79,619,312,606,835,772,78,403,939,341,732,550,382,633,831,883,293,987,112,794,638,456,805,142,907,937,548,162,408,82,102,796,261,999,269,553,217,458,324,310,152,698,508,597,332,686,509,279,419,224,215,110,858,683,777,558,985,694,101,792,476,330,180,810,797,768,373,625,0,189,957,393,629,141,569,908,187,922,20,449,125,277,949,957,658,827,116,78,577,901,428,540,851,310,89,651,102,488,979,680,364,780,938,197,182,970,670,965,184,287,939,778,731,597,105,481,569,10,958,238,811,705,670,228,714,442,13,590,889,931,338,940,575,473,400,350,679,564,17,197,81,709,60,449,729,921,834,698,735,555,51,223,567,261,791,264,231,762,817,74,702,967,248,448,423,172,923,94,129,625,414,766,300,587,537,767,38,133,247,424,574,848,156,281,389,266,2,101,140,557,367,893,823,448,82,106,297,706,751,44,500,176,751,546,452,340,996,129,564,342,68,628,453,995,274,288,689,356,288,586,713,676,938,30,357,832,760,4,245,459,958,371,493,978,948,190,201,219,335,916,546,748,963,464,914,871,381,203,698,161,481,657,927,677,207,385,110,651,807,986,989,67,537,519,840,730,511,603,131,55,785,57,335,2,395,636,216,180,884,53,276,6,400,704,433,888,288,698,552,699,253,352,172,522,991,278,665,793,102,582,520,676,707,698,995,47,283,537,481,568,487,576,224,881,34,492,912,741,936,971,911,955,107,59,728,760,986,940,736,43,220,232,534,749,855,756,564,458,119,923,995,571,4,342,90,527,364,518,226,626,682,864,737,31,682,845,639,424,664,911,924,659,228,509,509,458,499,633,232,930,597,311,804,967,508,466,992,266,548,160,964,728,687,948,205,208,745,374,102,734,653,848,687,638,815,930,42,955,843,192,476,315,116,273,22,876,264,492,690,525,702,528,217,263,801,527,275,59,1000,747,519,136,526,431,148,12,151,777,678,489,97,253,706,870,658,536,367,458,329,229,965,989,718,468,971,911,401,454,813,453,932,185,977,5,196,534,897,898,82,193,153,857,976,714,131,756,148,793,621,753,921,100,247,359,491,320,676,139,821,217,640,918,431,482,182,875,385,664,708,604,659,65,138,920,182,448,186,999,666,903,791,647,323,984,166,311,34,961,274,8,552,214,952,834,50,216,395,78,568,348,565,280,965,570,529,10,588,175,380,624,880,532,354,353,172,466,29,404,598,967,508,43,680,282,521,224,47,805,108,533,201,197,908,856,104,833,772,492,232,651,646,345,382,115,575,271,312,501,827,935,144,569,284,389,555,377,805,331,254,136,501,441,903,735,771,295,276,669,18,301,485,255,673,7,683,24,334,396,946,798,667,797,754,596,320,726,719,536,313,418,753,685,998,610,242,428,638,552,289,253,88,606,137,776,621,182,240,924,250,536,554,925,748,906,892,973,624,846,59,190,356,727,35,271,639,951,329,618,93,173,863,564,145,170,62,627,452,82,870,241,398,918,501,690,86,560,107,41,990,752,104,781,695,739,697,849,49,931,901,165,40,969,166,258,837,597,355,470,564,560,175,26,743,252,359,610,917,421,622,936,632,145,669,387,330,955,679,153,734,704,319,365,380,490,229,845,34,451,978,183,79,693,679,144,697,373,475,940,846,515,616,170,242,693,349,529,744,88,300,280,119,617,245,144,2,157,551,590,862,69,419,339,67,888,201,771,152,544,381,345,917,479,31,749,161,120,454,725,655,946,623,334,294,476,862,648,312,831,17,408,962,683,408,461,17,961,693,541,734,142,706,349,720,87,154,521,858,50,879,748,416,407,157,639,882,426,24,510,536,530,445,875,407,292,892,709,176,255,809,630,329,393,892,678,375,901,273,714,286,879,510,731,303,700,809,139,250,150,994,403,984,596,866,511,617,859,672,477,994,180,632,690,306,402,344,614,892,292,833,925,402,430,332,980,90,434,921,187,844,295,971,358,205,257,670,232,982,940,407,209,452,475,253,9,397,599,82,480,950,926,939,669,211,514,445,574,492,144,908,690,820,820,278,804,886,706,783,814,443,989,621,944,455,221,636,130,979,338,540,464,754,907,530,561,488,119,901,363,423,345,892,833,18,500,855,6,699,475,457,928,278,468,937,59,779,386,253,882,856,984,246,518,616,656,546,978,394,913,34,686,986,239,379,452,670,111,371,60,479,864,884,553,575,110,905,263,134,811,398,218,250,250,32,148,736,887,706,327,766,122,924,206,223,285,177,48,989,280",
    "386,25,63,545,81,22,899,171,395,79,880,215,862,209,440,332,656,789,675,608,989,984,630,140,771,667,905,892,161,697,803,935,81,521,98,435,165,631,823,44,233,625,930,648,566,242,530,867,266,929,205,538,841,605,30,300,555,752,569,961,621,352,824,753,328,342,778,99,915,620,189,0,229,571,325,263,461,650,96,642,619,352,729,871,940,605,502,310,203,332,39,767,137,432,284,274,560,549,908,3,471,55,162,499,440,221,702,518,713,407,62,649,410,272,768,106,373,924,600,111,154,596,911,854,126,476,11,181,115,749,811,482,754,512,371,580,835,796,327,583,524,799,367,8,643,346,56,795,6,286,766,47,120,418,367,730,937,150,625,915,850,906,478,736,15,746,790,90,345,252,969,326,761,986,823,492,643,179,13,371,195,351,538,380,110,637,258,731,610,198,845,467,840,335,860,846,961,165,774,813,416,301,538,962,518,710,545,536,963,450,830,655,934,155,727,393,618,85,295,796,455,197,200,180,865,524,498,666,425,572,846,78,644,441,27,482,751,746,390,79,448,713,116,400,624,247,627,368,594,573,516,138,424,4,353,92,766,114,333,596,670,985,680,582,210,59,391,773,912,238,904,941,842,626,775,753,318,88,971,162,583,836,803,961,424,588,764,678,457,166,926,348,676,431,782,902,940,241,913,301,505,615,661,57,5,19,305,406,445,10,627,405,845,306,767,994,321,6,600,110,174,812,278,93,258,439,645,806,12,845,488,748,862,483,639,265,673,918,650,131,660,926,651,295,620,572,386,760,456,532,689,773,450,627,302,991,857,29,160,849,929,189,680,413,918,78,545,622,372,239,586,305,907,527,404,289,960,41,507,70,759,755,539,705,691,328,670,720,460,626,648,782,954,133,790,700,662,870,354,660,942,724,844,504,488,817,336,959,5,10,892,351,356,89,70,932,280,20,137,839,879,592,161,715,161,61,334,388,12,192,579,310,246,922,570,491,652,49,778,176,550,83,910,312,823,581,768,770,537,413,596,224,401,196,467,20,56,100,943,252,319,675,655,52,435,219,71,195,995,455,153,77,288,447,349,510,622,679,171,791,192,648,566,123,528,158,724,698,889,418,475,249,665,232,194,494,646,994,673,679,782,883,577,660,793,798,282,480,438,771,971,569,983,112,498,729,712,40,812,900,78,381,726,125,633,215,735,312,667,431,464,224,476,52,917,157,787,660,19,451,325,27,492,996,494,410,425,398,366,544,997,578,507,37,538,882,636,520,423,661,344,494,613,207,132,980,222,453,209,346,288,208,688,881,222,467,741,312,22,754,949,434,684,525,258,860,126,967,762,41,608,473,194,384,13,752,709,931,599,81,514,920,708,157,546,841,805,911,677,13,547,257,637,368,62,145,416,951,435,692,619,840,77,36,376,517,10,89,425,938,431,228,339,821,284,84,611,596,944,37,780,629,642,686,742,622,587,408,235,19,575,199,285,153,516,516,63,958,885,541,790,61,960,138,976,289,390,684,885,288,66,964,40,974,252,420,264,885,852,868,807,409,529,934,251,645,749,97,688,863,518,319,738,816,2,49,963,653,996,324,211,518,513,820,956,781,857,300,100,150,132,503,81,934,950,858,2,424,443,549,349,180,463,959,562,96,209,873,540,484,997,485,711,800,418,425,599,645,744,593,240,199,806,185,69,546,933,714,106,953,411,753,473,297,316,508,354,894,368,5,15,65,385,583,410,293,189,1000,962,77,370,532,124,185,773,721,2,79,544,918,436,786,608,180,165,278,824,120,818,257,378,900,778,482,748,150,164,689,226,841,193,651,642,831,42,664,690,705,785,354,211,63,870,217,461,111,868,422,581,818,814,618,875,754,806,521,48,638,378,25,90,44,610,747,348,610,762,412,288,183,47,30,561,782,546,754,382,532,364,564,230,599,580,593,342,400,235,805,521,716,735,431,993,442,410,712,917,586,780,626,204,379,577,5,873,434,979,710,516,446,396,849,562,242,816,697,676,68,24,97,345,244,546,790,654,777,462,45,814,604,26,570,599,566,240,572,847,321,708,335,372,366,258,570,84,63,582,347,900,320,476,512,760,563,839,841,653,356,570,616,691,653,321,367,744,980,832,820,566,84,386,516,121,6,391,316,82,470,258,710,875,479,78,262,790,503,846,964,175,444,613,32,186,931,523,291,476,886,234,49,919,717,700,785,82,135,832,102,791,366,323,150,676,896,948,250,84,158,756,297,762,749,6,222,15,945,61,403,582,785",
    "536,779,286,30,1000,943,562,721,920,812,64,442,901,256,35,458,920,220,498,545,711,907,998,54,506,357,134,183,161,487,937,486,139,262,855,386,66,529,727,833,156,936,705,871,880,84,357,36,241,861,413,322,437,863,862,596,161,870,53,10,223,781,79,614,617,341,461,49,594,237,957,229,0,797,260,378,75,345,635,745,625,589,222,699,202,107,479,392,570,575,530,3,128,717,293,742,923,351,79,902,417,755,571,303,878,680,687,692,678,610,611,574,336,34,921,66,325,3,38,975,135,330,507,471,831,242,905,866,736,810,667,822,176,375,282,556,289,828,992,278,744,63,637,62,673,275,957,745,29,202,592,464,668,705,665,617,147,197,711,885,337,87,749,692,911,270,265,755,497,463,683,7,43,328,504,69,197,862,55,677,445,269,352,529,786,331,239,492,596,739,646,602,532,436,285,251,415,753,528,426,127,636,170,244,821,569,557,965,977,754,371,499,40,500,182,135,485,207,650,345,49,8,587,631,796,725,472,30,833,700,988,665,537,77,114,794,938,150,242,238,365,914,160,427,267,38,912,288,599,745,858,416,227,149,231,977,151,201,209,720,72,892,553,819,323,641,564,258,248,28,78,618,512,280,592,149,431,972,33,639,937,374,503,358,446,939,531,77,789,9,237,838,367,818,104,57,279,139,979,590,895,935,78,972,350,85,561,272,512,580,777,872,283,699,514,269,414,894,250,457,44,987,674,165,767,44,544,126,200,829,3,911,829,932,921,768,67,764,703,783,814,303,380,966,56,979,849,169,117,35,736,702,841,111,114,302,102,650,726,277,146,998,887,940,63,313,47,870,570,104,903,113,202,348,561,836,93,294,605,394,213,861,602,596,664,295,555,729,258,673,923,233,356,600,828,530,698,950,545,272,883,493,334,81,67,462,615,869,763,597,408,99,535,379,454,93,480,446,928,903,594,929,655,227,570,754,756,78,614,199,169,751,179,66,533,710,739,61,330,233,718,964,453,703,685,860,553,986,828,355,786,34,792,213,70,914,126,729,176,798,314,195,726,9,917,378,817,653,805,273,585,650,53,604,51,716,372,510,860,661,4,241,720,469,683,619,65,486,633,651,593,392,169,487,428,729,78,224,348,243,631,964,404,373,154,888,570,872,782,729,828,33,294,368,286,764,89,151,805,888,163,906,436,112,760,173,151,515,702,329,551,580,180,560,177,920,514,21,519,140,87,764,747,522,395,907,783,727,858,663,399,741,441,752,644,374,831,578,631,654,145,603,457,28,851,99,393,69,913,999,261,677,991,868,340,690,727,670,725,294,413,921,576,123,611,493,730,904,862,143,882,808,50,771,663,981,531,39,101,270,425,710,826,627,12,332,199,682,370,994,722,720,805,771,544,775,843,250,834,579,524,949,271,97,278,676,645,246,710,912,981,160,963,280,573,561,555,718,445,720,679,666,198,45,832,325,834,768,335,30,975,640,520,98,365,195,556,303,20,911,648,965,126,358,197,855,526,558,710,973,68,480,67,974,272,237,172,66,691,624,315,891,363,971,697,817,52,690,394,62,573,408,415,289,962,459,926,907,937,130,979,627,354,85,701,788,612,376,393,384,405,318,306,134,605,724,116,901,196,997,377,81,147,837,850,511,438,713,620,233,879,86,220,837,336,308,63,422,107,913,376,948,672,886,631,975,643,546,215,314,259,734,327,353,424,658,635,824,161,252,173,170,932,733,848,143,318,975,24,506,595,35,995,649,845,53,347,866,322,85,233,165,855,770,431,170,191,116,917,976,479,569,232,500,128,391,721,110,39,785,6,83,932,996,361,99,692,752,773,276,317,629,483,338,232,837,59,566,455,380,785,264,183,697,780,762,35,515,397,84,554,326,826,456,71,138,911,36,866,552,636,372,258,580,247,374,335,891,758,231,300,830,734,348,98,20,187,86,717,914,73,424,327,595,49,862,135,822,398,770,712,406,542,529,344,980,897,702,935,80,196,978,386,579,156,146,559,88,918,688,947,943,909,398,761,616,169,570,448,711,93,209,665,314,868,225,916,736,283,3,777,611,235,664,821,278,316,179,634,733,610,49,775,719,99,824,991,835,191,807,937,397,744,549,601,695,678,238,378,394,191,908,526,103,42,426,552,327,449,373,16,524,55,724,232,76,394,661,52,970,233,722,535,963,665,872,314,782,12,742,535,888,574,640,283,54,828,338,875,681,22,782,57,922,69,821,318,765,512,959,256,624,611,851,117,667,676,583,765,433",
    "942,492,499,163,453,781,391,50,72,252,655,575,991,703,368,82,827,534,622,100,106,268,569,679,967,879,58,611,287,726,332,615,887,915,894,337,350,221,126,585,651,828,571,1000,729,744,385,634,253,321,312,841,264,62,57,454,257,715,289,49,780,175,281,371,876,40,609,130,615,89,393,571,797,0,236,370,876,626,717,36,212,608,694,400,760,67,752,403,455,793,63,950,911,677,315,894,258,74,42,826,740,211,968,448,444,726,251,21,763,460,293,541,437,642,877,960,589,602,805,500,752,158,390,648,131,809,519,209,440,874,521,937,629,871,504,561,900,754,630,159,759,444,606,11,292,183,612,818,109,680,765,737,440,406,19,3,133,943,783,431,36,831,880,896,285,643,160,735,289,840,682,339,198,117,497,247,202,572,325,490,865,211,43,503,170,869,879,489,499,278,102,937,279,620,913,96,35,551,759,17,547,437,346,274,686,217,911,86,309,268,559,564,346,411,7,81,889,533,585,673,576,975,434,550,349,197,597,817,738,778,711,814,29,204,417,189,280,53,327,72,90,524,234,262,604,490,261,944,13,20,609,126,270,86,925,511,514,739,99,787,452,47,853,138,438,313,341,16,849,365,465,535,544,695,85,553,351,31,356,203,638,643,84,366,995,646,579,815,262,91,953,211,156,648,82,918,826,822,10,776,651,126,511,396,58,942,993,481,815,98,458,183,925,925,842,695,161,274,917,421,318,688,215,841,233,744,289,442,721,55,900,367,858,370,851,265,298,736,247,60,248,25,245,385,788,374,98,587,758,709,1000,402,586,598,831,895,976,493,409,610,290,604,486,950,169,273,487,231,756,150,327,722,113,791,189,559,630,497,10,51,249,247,427,192,820,272,771,396,246,275,109,974,268,231,457,761,642,801,400,317,481,578,781,990,638,607,458,322,824,862,727,373,874,942,23,239,674,605,828,598,398,736,551,340,485,780,157,830,481,57,99,326,222,663,520,698,29,37,320,1,166,151,883,121,884,670,750,744,406,511,242,971,377,76,930,612,545,969,719,484,91,495,234,623,866,409,971,517,157,808,399,58,910,741,369,76,990,591,731,749,274,98,335,352,518,803,986,126,985,751,330,853,399,496,25,929,274,370,506,56,751,189,15,244,42,173,934,446,67,82,630,501,665,848,860,794,476,574,314,880,753,264,251,692,62,883,322,66,304,144,939,330,984,752,374,989,128,253,262,960,932,627,902,126,551,410,260,280,923,21,814,605,690,722,411,386,597,932,837,940,994,397,32,88,825,366,804,127,875,280,495,4,869,228,433,368,460,399,974,8,893,598,824,514,608,638,391,21,868,890,764,218,237,393,87,149,692,687,678,452,591,174,890,845,316,639,811,388,100,901,403,640,882,184,413,99,106,51,321,824,67,350,79,267,145,975,128,793,817,614,380,88,271,223,844,264,732,528,119,772,370,114,547,276,753,301,52,330,975,467,242,457,288,226,243,771,966,786,116,926,211,901,474,842,937,558,361,490,72,604,30,844,634,163,507,683,377,900,687,388,123,128,207,994,150,161,783,405,435,773,244,537,711,250,756,38,311,57,432,93,126,53,273,87,485,162,795,530,416,483,811,592,466,940,712,285,152,255,543,428,596,215,202,351,725,26,472,783,446,49,616,656,404,822,658,165,437,245,377,783,279,724,900,255,310,46,598,86,55,567,399,128,932,761,556,753,669,958,132,889,758,204,516,539,733,713,63,469,712,920,992,225,909,75,47,748,863,929,146,541,265,442,503,355,737,771,625,760,156,908,286,261,211,262,55,790,604,176,216,815,832,483,672,198,395,154,43,148,619,200,683,944,394,418,201,855,549,452,191,479,298,717,424,562,75,353,561,150,942,199,253,247,528,498,923,842,157,402,355,803,468,299,494,817,295,713,955,161,365,234,727,688,239,267,657,205,497,433,268,113,30,722,457,879,751,299,633,265,778,601,469,566,961,622,111,457,100,444,799,355,584,415,488,717,537,782,941,291,158,249,502,680,115,160,165,287,966,744,237,132,539,169,704,720,20,196,756,175,856,991,861,825,943,201,136,106,297,98,161,875,873,302,897,746,376,418,790,800,645,21,430,490,194,58,721,54,415,957,470,927,123,226,662,835,254,329,118,961,704,402,541,28,304,316,733,688,60,999,655,35,983,63,186,442,319,693,585,128,408,83,483,315,855,790,522,277,415,522,525,328,491,450,939,729,255,381,938,939,489,880,586,703,456,729,521,705,756,234,601,895",
    "533,412,302,866,659,129,157,30,272,187,936,543,990,953,104,736,776,777,456,559,168,979,967,382,376,30,955,575,160,431,314,778,666,817,863,618,577,534,677,442,39,67,804,896,329,143,245,818,644,730,821,542,337,433,327,693,28,742,244,672,507,682,471,646,310,31,445,285,822,87,629,325,260,236,0,465,641,299,197,778,426,24,692,716,22,1000,783,283,753,638,813,38,171,438,266,354,284,79,230,379,366,136,133,393,253,757,357,65,324,708,711,356,585,424,107,171,813,203,666,937,857,429,295,427,140,718,152,763,75,227,912,275,435,18,184,5,807,889,409,223,854,175,787,165,654,719,518,691,837,800,928,104,933,564,713,801,142,43,158,306,266,496,234,186,821,167,658,419,910,599,29,999,634,231,39,226,130,2,751,28,237,535,216,466,820,190,846,928,660,428,924,117,222,948,778,195,493,584,608,766,635,457,436,641,551,643,164,320,233,899,277,580,535,365,767,456,389,796,637,814,328,200,626,558,626,489,876,728,132,539,652,579,783,133,986,527,895,572,960,564,490,9,284,197,806,737,774,461,315,419,250,654,856,60,19,957,308,651,254,538,368,469,743,326,429,503,326,171,752,298,538,425,875,221,130,660,517,139,53,283,511,531,107,709,570,814,10,449,389,218,216,856,464,272,916,251,592,428,299,82,36,720,862,540,462,798,390,156,676,700,949,925,480,786,599,728,98,992,978,456,837,687,202,162,132,861,54,957,215,660,135,417,576,744,389,991,379,161,674,513,630,726,511,454,869,973,947,818,517,620,756,526,446,16,421,346,150,466,685,201,819,270,859,754,505,827,811,690,293,592,968,693,803,931,915,560,129,990,381,411,96,603,643,185,105,295,870,68,600,395,187,773,891,616,719,240,496,941,780,674,653,861,622,977,809,99,323,678,106,609,895,119,608,806,869,271,975,30,554,820,351,666,168,729,831,410,773,931,383,657,184,507,188,914,379,243,438,215,561,252,813,717,19,964,641,716,337,274,440,584,437,183,701,630,551,175,870,64,827,908,74,905,202,154,859,582,925,681,481,530,75,277,767,19,835,918,249,232,453,387,839,331,967,13,198,689,802,605,258,203,341,882,31,658,689,98,371,356,127,856,864,882,443,926,526,343,366,849,699,564,381,999,628,188,165,88,568,698,719,828,977,443,804,482,209,428,310,485,368,467,238,520,593,688,651,801,449,746,77,691,281,859,240,395,28,346,556,586,790,216,714,255,344,114,723,898,530,519,464,252,208,159,102,262,586,247,369,961,125,260,953,32,581,322,818,23,898,541,443,505,534,458,78,323,659,729,779,938,955,526,163,628,910,161,629,870,602,742,683,776,528,349,622,402,719,425,806,238,833,598,161,487,908,402,384,102,377,725,569,862,690,862,307,895,401,985,263,720,190,773,518,862,487,328,292,926,261,551,956,312,241,896,460,73,212,802,263,944,130,836,671,736,534,664,791,822,130,280,792,519,788,360,877,864,611,470,169,382,297,208,179,451,52,552,419,739,549,687,134,268,334,951,48,962,781,140,265,681,208,314,347,119,16,578,447,558,720,70,375,738,57,634,163,923,666,296,620,20,444,247,521,557,941,672,824,292,183,350,661,288,408,645,437,719,868,623,855,693,834,260,759,546,444,981,490,946,875,216,767,306,610,114,594,443,119,953,378,801,147,792,332,822,263,535,491,580,992,470,423,644,889,800,480,235,537,181,885,471,629,218,302,514,772,854,166,272,948,95,542,165,446,715,442,740,159,718,206,63,740,837,282,495,400,309,860,465,185,611,137,693,887,770,357,897,591,146,268,893,877,40,747,454,362,366,249,376,933,51,208,972,994,51,514,710,950,778,183,378,141,724,383,800,358,813,368,953,348,620,215,585,90,729,766,902,456,524,326,21,260,118,864,753,827,898,227,603,572,371,721,306,736,926,610,325,261,439,281,14,398,846,474,468,338,314,433,916,267,992,943,868,449,860,660,679,261,914,956,323,962,779,308,958,169,493,822,858,254,539,132,118,339,778,686,441,716,788,967,373,776,718,699,902,567,9,283,185,83,897,869,4,573,991,945,418,51,975,757,572,617,295,824,717,479,797,917,682,71,500,921,560,613,77,340,297,348,590,683,297,268,130,191,863,659,441,349,497,699,24,43,560,710,895,5,829,759,364,293,357,751,939,887,424,141,58,505,551,951,35,258,389,506,167,281,959,325,412,404,267,239,990,953,757,252,76,662,93,88,720,367,943",
    "774,778,842,785,936,203,933,345,622,260,602,913,135,165,866,793,465,976,910,274,343,16,425,793,534,609,818,839,969,332,213,216,6,572,419,992,165,823,667,455,229,433,990,534,454,320,895,58,804,239,385,253,332,646,264,514,43,412,846,399,550,857,847,418,901,677,24,432,710,711,141,263,378,370,465,0,587,908,169,38,629,538,271,824,624,405,65,622,827,786,978,835,338,595,929,545,804,517,119,485,480,953,299,416,368,411,175,399,470,348,425,61,99,506,953,507,11,774,82,422,966,637,573,314,101,900,104,888,790,456,590,945,588,303,588,99,413,712,78,443,196,739,231,27,873,7,902,211,16,12,544,311,794,113,835,67,416,30,980,293,875,203,283,398,152,339,659,698,763,836,196,333,664,396,258,778,852,274,238,50,243,292,579,901,264,202,964,752,555,237,709,268,638,894,235,291,705,434,140,340,213,973,748,715,540,163,874,35,434,328,161,742,355,911,814,7,187,745,185,477,508,112,561,634,182,371,280,901,103,140,819,318,739,676,354,400,162,933,749,745,583,352,560,510,511,589,967,103,719,993,533,605,652,750,397,565,626,798,155,864,635,525,148,913,153,744,20,876,504,684,339,396,62,68,556,110,82,93,192,431,982,156,882,118,309,344,506,726,659,620,872,690,926,712,999,976,4,848,51,404,21,277,628,527,965,842,509,344,853,222,57,532,135,835,343,678,646,130,292,403,163,195,81,135,133,111,175,930,329,485,499,579,612,529,731,400,485,522,292,685,33,399,813,677,536,131,277,639,529,815,852,552,946,487,368,440,707,734,166,377,968,706,187,726,875,491,154,727,404,961,131,171,717,722,697,459,891,95,838,323,83,723,663,476,944,858,611,169,871,782,687,354,800,190,392,308,272,754,823,467,580,719,971,42,650,994,26,374,752,2,447,452,904,388,915,776,315,678,271,76,589,107,73,269,728,202,935,437,339,66,716,650,501,940,181,642,127,878,360,362,878,417,457,43,902,258,677,931,718,170,94,861,581,819,729,722,328,59,559,209,362,353,748,685,260,387,273,658,745,353,632,196,729,903,891,135,675,498,80,615,507,713,22,358,647,893,650,313,715,835,765,397,681,512,657,275,62,130,59,869,724,231,770,298,598,367,488,280,473,398,576,510,117,417,289,865,94,385,601,540,893,811,392,627,566,275,515,120,177,936,615,930,570,787,210,886,478,666,399,954,163,51,789,726,894,291,89,111,168,200,22,985,8,670,649,32,67,529,926,437,949,153,515,75,596,200,497,704,905,135,429,117,684,634,285,849,56,920,365,463,712,285,866,601,618,495,271,320,681,921,805,487,437,465,936,218,675,187,89,838,581,512,941,17,290,220,558,678,868,55,870,631,800,417,491,838,824,795,549,183,738,159,412,944,590,500,323,867,176,758,27,687,615,594,608,776,296,705,614,986,62,62,537,42,25,885,831,168,4,972,989,552,566,161,332,629,372,211,892,71,137,48,171,95,394,289,237,363,691,819,542,271,418,579,857,37,914,584,651,178,906,439,32,389,691,152,347,737,543,981,21,288,461,670,782,676,210,629,31,589,628,821,993,106,497,515,211,446,847,39,473,981,625,733,381,410,486,431,388,439,854,939,209,557,466,991,500,446,777,947,916,151,816,305,229,92,426,839,625,418,689,670,755,423,770,162,982,715,624,881,727,530,727,692,914,926,448,178,453,696,455,406,170,15,160,840,378,860,554,429,780,309,840,868,261,327,665,500,606,491,20,948,670,236,830,217,492,66,635,702,723,394,593,564,705,668,816,871,799,630,245,209,470,627,637,958,449,584,478,550,872,212,245,615,279,662,190,819,29,545,186,931,224,891,35,33,389,238,531,187,30,878,716,780,337,163,604,601,240,301,655,257,342,756,136,912,904,466,280,540,10,879,163,654,648,596,309,958,634,318,588,865,866,646,24,404,4,60,245,93,918,893,453,46,94,802,224,468,978,734,783,579,735,706,748,474,653,670,142,259,916,874,692,353,890,454,124,146,357,518,523,637,567,644,842,659,275,31,889,167,571,655,366,151,816,98,589,720,452,451,995,625,458,920,611,836,45,8,485,465,93,454,621,612,15,38,833,278,571,712,437,45,905,415,533,793,931,591,816,375,853,615,917,684,960,27,151,619,191,974,302,179,221,790,238,61,188,195,729,87,641,867,171,562,474,985,586,888,729,671,719,214,752,916,808,393,885,341,308,43,745,798,525,129,740,784,202,348,376,111",
    "71,302,682,54,930,381,691,332,583,706,814,296,976,935,595,446,599,672,498,257,853,26,319,213,779,946,578,23,617,510,694,402,782,427,717,376,199,274,568,143,893,37,447,492,522,905,887,877,462,620,819,292,611,421,807,294,681,660,563,236,948,549,614,50,870,390,668,649,140,953,569,461,75,876,641,587,0,439,88,841,930,402,918,735,185,215,159,60,716,475,886,664,422,425,744,926,310,881,325,361,149,379,323,86,927,564,878,480,839,14,160,771,635,793,834,695,693,121,988,832,931,887,399,544,122,653,990,302,450,827,832,159,293,198,685,417,133,114,666,882,835,743,865,801,123,755,882,336,656,927,933,610,245,696,460,869,31,187,636,693,602,665,97,544,313,157,900,602,903,845,693,622,630,504,753,430,666,325,101,89,954,729,905,724,335,347,158,488,260,824,324,954,791,715,552,233,616,578,487,461,613,30,679,853,630,785,908,566,262,55,915,556,742,919,438,210,963,496,613,234,88,288,682,817,608,437,354,800,812,209,876,303,924,184,761,837,258,46,776,94,268,314,34,230,318,246,994,187,752,812,46,414,247,8,630,521,66,991,42,952,684,392,460,16,233,698,124,451,37,490,205,278,187,335,951,165,141,442,573,288,412,582,929,38,573,16,108,476,882,972,731,238,630,548,463,706,133,651,167,990,568,465,700,463,376,302,50,454,591,462,659,507,283,367,985,137,842,473,879,518,240,508,660,31,597,656,955,746,853,253,783,987,732,618,220,136,129,826,877,587,185,465,891,208,288,427,212,485,904,689,931,886,462,752,492,268,945,837,813,724,188,99,186,297,451,751,704,236,266,594,494,928,371,513,127,462,786,68,366,799,14,269,812,912,601,417,787,436,7,53,353,110,93,668,251,422,778,827,483,348,924,195,827,806,374,645,343,812,551,613,48,161,325,205,585,846,29,893,988,153,659,835,563,512,805,961,287,84,767,834,81,112,344,582,77,524,897,880,433,60,148,16,506,811,342,307,719,216,404,667,436,403,287,816,521,168,404,663,981,317,611,968,869,632,667,489,46,830,321,212,796,808,797,308,747,558,418,14,287,578,300,394,538,30,138,573,345,69,545,809,829,887,648,870,882,854,169,852,546,666,159,721,271,894,555,571,689,451,742,456,702,730,743,229,64,843,842,881,451,937,410,442,230,483,589,223,713,222,70,441,799,27,843,636,832,766,44,104,382,881,443,199,532,903,514,652,287,263,500,922,238,632,408,846,723,870,894,577,596,148,16,98,432,76,274,908,567,186,947,970,38,609,467,602,557,346,962,560,78,536,355,826,655,66,288,407,364,840,925,596,76,948,843,249,389,726,486,58,33,67,856,558,30,297,87,419,687,13,599,785,832,850,88,768,154,699,803,564,250,695,780,539,603,978,702,180,90,522,795,627,921,333,950,528,872,576,845,499,290,610,955,144,659,756,358,473,535,960,566,986,108,335,146,625,631,651,655,137,231,743,405,400,890,337,85,979,528,10,824,535,77,274,365,66,64,71,198,412,820,449,589,795,608,916,113,321,992,919,981,234,617,53,827,11,797,550,857,520,152,946,121,875,451,479,111,593,953,551,792,602,420,859,435,70,115,970,67,362,260,538,897,563,757,794,960,907,506,509,581,747,237,801,689,102,862,779,814,560,666,886,143,964,676,732,625,492,568,566,342,26,351,590,733,984,221,387,412,37,162,80,833,14,340,193,443,903,811,732,226,537,873,508,304,588,627,584,231,836,426,60,131,561,520,315,861,803,710,298,127,421,354,109,781,757,849,380,705,573,900,844,12,432,605,213,646,513,697,562,109,707,11,547,272,401,410,11,275,299,685,68,397,303,517,379,629,63,329,504,3,984,561,931,703,616,890,469,574,318,463,475,689,411,765,473,21,333,231,244,175,479,818,153,563,34,10,610,818,962,887,662,33,138,375,953,354,166,142,730,845,931,295,955,986,513,758,194,538,70,307,230,742,227,349,756,356,762,350,861,959,632,958,755,789,932,574,26,499,651,201,790,381,229,231,303,445,526,118,632,277,600,365,500,828,64,972,572,250,507,345,345,962,764,913,352,111,135,497,770,716,843,19,903,471,985,398,315,474,501,35,458,109,771,24,463,742,41,457,550,838,700,682,529,542,830,371,825,816,386,610,960,887,876,654,135,434,328,188,335,64,468,282,186,927,325,873,656,349,159,707,253,440,29,320,731,707,545,575,532,699,665,550,271,984,129,929,83,904,402,10,838",
    "996,220,896,219,252,316,918,991,278,2,436,33,17,415,94,626,199,700,713,452,231,111,599,730,19,53,195,150,516,412,594,237,837,768,379,845,649,171,882,507,487,868,128,148,408,56,445,334,110,591,908,85,854,750,795,442,25,396,150,928,226,758,977,260,922,974,1000,634,285,431,908,650,345,626,299,908,439,0,663,93,765,747,893,926,934,768,303,683,174,686,133,486,78,892,504,527,668,587,817,859,969,191,795,525,180,886,735,650,607,241,510,53,305,639,865,681,351,265,84,924,55,218,37,66,929,995,123,185,976,19,405,169,775,772,741,670,390,700,602,560,38,791,751,947,354,620,651,664,802,575,681,567,865,644,862,59,833,559,854,468,731,199,864,971,370,90,787,383,820,60,437,423,552,707,847,782,742,574,89,992,311,25,43,424,808,728,356,640,123,34,727,104,483,21,261,957,308,622,104,743,938,813,874,800,641,794,530,595,950,744,298,564,397,72,448,81,927,995,944,959,27,949,257,235,8,127,193,398,154,254,434,672,209,580,652,314,887,387,43,25,349,345,249,826,408,176,472,59,877,647,61,121,3,503,140,346,434,902,424,43,812,107,325,960,413,568,335,185,68,684,367,480,559,931,528,282,675,940,825,920,927,946,487,71,228,885,94,150,795,386,594,305,310,327,155,137,189,442,459,807,900,944,84,292,725,686,299,791,595,897,709,397,704,608,417,521,8,605,711,519,48,58,190,905,676,946,779,540,977,681,928,86,38,858,73,625,924,873,768,407,894,23,247,667,697,834,385,297,929,998,79,824,442,912,322,283,459,514,66,311,988,477,997,62,137,475,543,561,193,827,819,931,246,696,262,493,733,283,510,898,150,472,599,453,236,231,801,921,159,741,627,377,174,305,807,679,845,402,415,692,88,826,861,576,718,375,149,773,555,44,965,665,439,355,144,113,857,78,633,238,579,670,200,952,780,246,698,203,196,271,288,144,524,45,611,313,801,431,715,571,626,956,583,138,781,94,985,312,269,365,22,372,600,533,707,257,25,791,205,283,55,573,710,620,34,540,875,260,217,304,549,224,467,864,153,822,561,702,124,29,696,710,206,999,580,716,855,777,93,74,872,207,841,44,723,145,45,595,487,679,411,418,188,123,773,333,33,508,676,435,410,612,173,504,560,952,19,700,626,350,372,595,753,213,962,171,22,921,82,831,465,328,988,967,714,190,674,272,539,305,175,618,251,645,753,858,933,697,391,169,34,18,146,947,46,900,125,877,360,351,459,517,948,732,439,635,278,345,380,781,239,336,192,559,229,175,281,231,264,838,128,818,147,670,307,508,537,764,247,274,932,854,64,42,691,553,184,23,420,358,877,513,219,783,495,440,679,601,468,414,629,914,718,934,485,138,378,966,302,543,211,904,256,880,881,999,300,611,654,49,643,393,558,778,54,24,278,138,439,22,211,422,775,909,407,927,30,107,495,127,838,699,35,277,496,832,851,968,620,893,886,271,966,892,647,791,979,645,986,79,849,559,174,35,385,521,597,970,925,93,159,31,552,697,553,454,943,243,432,889,989,467,159,921,527,628,316,914,551,1000,143,718,86,993,127,652,205,276,654,706,712,823,154,765,546,394,565,697,240,267,45,583,58,891,321,747,182,216,113,656,341,969,962,39,814,639,882,826,146,691,318,671,3,965,129,311,95,279,275,576,615,680,798,26,673,188,602,977,75,529,168,695,266,885,30,726,656,770,704,205,360,728,460,902,66,485,810,369,513,24,782,139,402,336,74,172,204,708,752,497,469,336,419,339,886,804,708,50,217,833,604,701,353,459,273,277,538,521,369,691,267,999,330,340,94,342,717,557,911,180,713,339,3,412,930,501,479,242,523,359,988,59,914,485,370,843,740,928,371,765,572,17,49,186,21,795,63,551,301,446,725,379,326,269,702,835,314,545,461,731,478,619,817,349,562,524,524,974,556,804,30,106,445,683,435,571,74,622,330,951,178,842,952,820,796,528,14,35,130,457,797,428,803,944,204,620,793,615,452,310,975,822,384,525,173,461,593,891,698,700,692,887,826,55,705,910,101,344,335,715,760,393,273,839,352,960,610,314,752,997,115,334,561,205,493,570,5,351,526,842,65,714,818,115,224,312,615,461,165,300,304,63,507,769,617,215,587,825,144,511,666,422,203,39,234,694,929,106,646,753,204,662,572,299,772,169,846,581,727,269,227,28,361,387,643,638,77,770,708,176,944,235,514,16,788,722,14,869,68,776",
    "752,159,576,573,904,445,516,807,571,142,545,61,242,478,458,233,149,106,451,785,7,298,101,283,8,263,675,472,450,92,295,530,780,632,257,652,204,891,584,274,98,126,857,342,426,970,716,932,628,602,427,628,102,220,478,14,204,401,528,71,847,270,742,811,11,42,786,199,686,764,187,96,635,717,197,169,88,663,0,13,628,963,947,870,529,561,911,739,137,529,353,40,435,322,780,494,10,753,960,518,587,324,852,670,353,690,46,848,904,487,233,683,519,613,978,944,558,441,854,547,784,202,637,276,389,409,570,620,522,321,197,755,263,396,747,650,956,361,618,160,570,772,963,919,952,519,618,697,227,238,738,910,690,474,698,71,825,320,982,645,416,734,79,521,297,166,957,652,88,120,920,123,876,969,514,764,541,654,710,877,190,206,648,798,680,537,416,997,695,788,620,432,565,725,746,317,507,929,55,669,117,114,217,427,127,25,589,809,614,325,94,265,500,591,368,857,643,721,660,650,155,332,138,336,57,871,386,141,963,131,794,668,561,651,132,193,652,116,676,791,830,180,840,537,587,421,327,380,953,723,977,289,139,576,292,2,842,65,363,651,838,669,760,900,862,141,209,940,413,269,690,871,164,639,920,567,92,606,851,919,862,547,293,544,254,327,995,759,749,905,464,9,421,51,269,123,23,892,626,555,982,145,420,141,539,540,727,64,869,104,681,56,2,126,513,728,689,816,874,110,11,507,437,262,762,682,82,777,548,120,34,933,657,562,945,48,366,935,973,628,631,657,709,295,456,615,223,784,458,148,487,990,641,50,391,349,640,385,368,315,116,180,648,500,146,353,653,28,605,197,89,552,695,856,309,886,901,913,716,417,176,932,729,475,89,568,599,539,855,734,650,939,360,243,739,550,816,402,519,520,404,936,996,595,349,119,762,671,232,54,411,234,630,937,778,772,466,639,213,548,669,391,693,914,567,818,330,140,513,677,632,387,302,753,10,920,349,267,893,381,343,201,151,313,546,361,430,533,791,337,97,577,534,330,923,271,462,852,504,347,754,24,634,522,550,85,154,215,317,143,523,816,395,802,580,236,767,670,460,208,191,493,731,340,584,921,43,727,635,871,858,239,734,754,443,21,88,657,287,712,64,277,749,189,687,276,744,446,734,851,611,709,696,125,494,528,789,682,732,431,948,379,501,991,677,262,982,889,65,179,942,771,168,738,740,992,711,239,701,851,51,856,749,822,525,964,854,123,100,830,965,489,267,934,314,172,49,511,762,685,795,548,432,509,560,326,525,99,69,57,370,956,952,153,209,588,283,931,204,689,484,723,383,753,558,622,165,673,115,190,909,165,230,373,283,673,889,409,824,62,418,352,669,971,49,378,554,747,777,874,758,95,639,418,804,886,313,407,458,21,578,953,55,811,894,777,563,264,770,310,863,641,725,824,668,265,767,974,833,451,904,467,220,274,153,892,138,425,879,49,201,394,528,660,59,879,42,283,628,430,850,653,451,729,168,845,662,565,990,24,897,629,343,7,32,909,717,118,553,989,7,443,184,416,783,199,94,632,209,976,795,531,121,343,830,39,260,655,306,761,792,660,377,215,309,975,686,769,371,613,697,817,324,155,743,357,846,692,844,267,255,664,110,882,704,172,450,276,773,701,221,350,813,491,238,63,147,56,230,818,113,606,292,800,158,175,416,888,749,666,67,95,716,936,530,96,140,634,178,506,795,735,274,575,999,471,786,503,624,369,129,491,829,228,253,362,918,399,510,841,318,851,304,965,623,561,293,643,342,885,650,221,711,698,263,606,842,946,469,827,275,150,388,166,944,527,63,497,624,423,150,156,144,435,820,123,819,356,466,827,164,824,901,850,876,987,464,38,166,218,926,769,142,15,21,981,719,107,862,56,919,474,754,868,343,66,786,842,762,25,97,305,999,449,582,993,68,991,189,11,142,914,255,878,585,250,933,58,774,390,329,97,260,161,35,774,619,175,939,267,568,594,120,810,831,285,858,267,506,47,122,708,339,135,204,69,653,347,954,559,84,626,152,591,940,462,57,47,163,112,90,50,743,547,435,141,761,727,934,456,414,51,462,852,943,695,750,968,59,194,376,760,889,343,944,236,624,525,484,342,82,945,150,206,12,799,112,140,744,814,640,136,707,651,442,795,742,695,903,443,227,714,682,220,124,961,287,921,210,631,711,214,907,432,997,625,987,35,972,350,54,415,417,823,135,696,113,223,630,48,252,294,922,885,418,517,638,343,857,945",
    "6,617,942,636,894,851,169,406,10,852,729,1,402,737,87,69,553,711,566,833,109,878,383,920,495,766,417,123,605,215,294,570,548,986,814,713,865,556,996,717,771,352,950,385,831,226,433,151,505,637,461,878,911,541,700,178,407,161,69,180,820,740,537,127,738,361,31,608,590,479,922,642,745,36,778,38,841,93,13,0,898,967,850,312,20,836,922,975,359,111,299,743,767,422,839,718,779,636,138,965,983,741,979,911,725,443,822,520,24,803,100,999,195,851,149,94,901,179,486,950,685,330,703,857,106,277,883,39,510,238,516,839,783,605,77,660,655,200,168,42,321,794,870,157,260,540,357,706,542,899,424,164,153,177,320,541,109,486,227,811,467,192,24,974,442,594,654,795,548,316,496,292,342,717,345,14,542,753,30,938,179,198,93,873,516,494,542,63,942,565,639,202,426,142,197,562,898,855,984,946,20,214,704,585,466,731,443,356,852,78,278,480,409,666,317,779,268,865,367,357,349,337,198,658,953,270,29,316,449,458,516,274,776,258,591,789,36,282,94,950,473,473,547,763,840,228,294,131,451,424,284,427,124,479,212,615,217,788,932,767,859,641,780,419,489,964,27,830,646,606,620,421,85,245,934,694,231,306,904,865,920,493,510,169,890,829,635,765,52,195,972,922,140,966,864,727,564,120,601,119,73,771,549,180,74,680,367,3,204,815,210,736,229,623,444,63,889,835,658,462,369,618,561,132,430,717,447,104,126,29,439,356,934,335,901,115,41,771,500,427,642,197,466,268,797,989,770,330,264,406,66,562,451,806,500,84,225,282,624,375,188,604,655,528,683,195,429,156,675,269,695,631,324,524,258,742,117,171,414,882,808,826,619,957,787,984,413,977,459,390,305,533,887,908,962,987,936,709,30,348,127,654,356,714,543,116,193,390,121,219,207,635,829,522,5,851,448,627,86,71,877,277,502,790,764,196,718,310,820,566,362,568,780,706,116,943,575,153,534,27,843,756,822,218,378,224,253,160,923,836,374,108,868,494,977,960,129,676,338,327,858,604,323,722,412,264,902,370,902,69,533,653,454,655,96,110,407,624,191,441,442,107,77,58,460,418,42,994,206,476,162,965,568,527,717,104,316,763,902,339,953,18,559,756,129,832,830,555,330,896,382,118,630,907,28,486,247,364,991,301,677,258,263,906,718,133,977,175,723,958,558,87,24,882,770,557,136,898,750,29,305,51,783,523,705,210,81,238,384,233,609,722,995,942,624,765,647,545,984,660,674,200,271,807,792,704,857,375,506,400,522,218,841,759,686,359,370,247,64,687,726,108,837,438,810,122,301,521,5,307,599,142,991,594,563,696,419,502,408,822,359,471,950,746,484,784,499,268,841,499,131,518,822,270,434,712,615,878,692,783,550,525,277,605,825,486,313,68,416,667,727,166,160,665,237,108,180,309,27,175,583,84,97,438,24,375,579,942,513,227,155,654,174,93,879,571,110,168,738,888,811,981,437,545,316,340,769,942,488,259,999,197,737,684,360,495,707,992,644,396,553,557,237,962,867,868,136,584,407,35,51,871,883,521,439,54,300,232,419,242,364,774,899,334,666,942,964,921,216,466,688,889,6,247,582,371,943,87,605,924,372,250,299,651,74,896,511,594,149,103,369,935,953,274,604,921,862,330,462,208,768,234,953,841,826,533,724,403,751,209,585,207,229,807,522,22,129,781,336,559,973,263,503,61,416,480,324,641,576,844,230,511,120,793,948,14,139,665,799,487,708,908,672,564,879,526,845,622,372,130,75,697,796,324,685,968,433,950,359,171,249,683,610,580,584,521,670,669,106,212,58,204,829,23,589,206,321,872,183,689,860,694,315,893,922,438,509,488,904,816,285,575,804,346,205,233,16,317,886,452,228,568,362,870,477,32,557,901,854,898,922,373,119,906,51,668,569,41,629,734,493,821,428,819,447,729,378,80,166,320,106,88,233,217,561,648,685,597,161,506,653,847,739,432,442,483,142,777,311,605,839,973,129,216,590,181,576,932,45,326,826,340,780,106,213,170,320,345,962,692,5,431,370,448,409,386,696,643,953,946,251,250,190,441,360,784,205,89,167,240,205,174,110,104,339,847,119,332,142,942,970,330,576,288,780,858,317,639,923,726,918,275,953,436,475,377,674,940,54,351,734,345,255,680,513,136,51,441,21,698,437,368,676,125,584,406,31,894,235,600,965,666,17,341,608,927,355,583,494,709,579,785,932,762,496,416,775,81,259,718",
    "741,673,68,635,70,976,380,79,211,582,248,675,856,450,637,310,204,306,903,387,501,277,380,779,936,314,236,987,718,347,765,759,663,9,582,979,998,815,892,127,439,51,239,816,42,730,990,520,939,908,358,200,569,746,724,55,577,814,31,473,671,76,463,182,770,27,410,892,99,692,20,619,625,212,426,629,930,765,628,898,0,230,448,214,436,205,404,69,52,140,293,818,174,54,57,223,373,970,450,853,38,515,587,435,154,407,435,472,945,592,934,763,968,250,311,317,685,633,732,178,980,278,645,722,326,65,37,686,906,973,31,31,337,138,638,162,446,652,508,976,636,758,716,69,206,319,609,975,313,667,904,919,297,126,29,554,473,787,930,700,112,922,787,77,59,501,981,917,647,614,41,903,831,335,842,920,557,174,962,871,920,924,287,268,836,430,549,540,215,992,360,387,131,500,813,397,395,421,229,601,646,267,163,832,590,520,609,251,628,387,629,695,118,685,117,11,836,655,266,525,858,814,465,564,435,395,760,173,307,762,524,701,331,796,76,623,285,228,12,213,877,320,142,920,643,237,816,662,606,316,641,101,974,865,926,616,807,26,138,154,429,155,929,213,232,856,723,377,221,360,808,890,371,268,462,916,992,34,597,552,617,530,487,205,785,725,586,416,405,471,267,340,424,299,55,3,69,448,912,922,287,442,676,454,316,28,680,824,988,437,568,863,78,226,307,516,272,556,753,486,727,877,454,606,181,215,914,158,133,402,601,854,490,439,491,88,712,907,946,518,1000,6,59,1000,899,828,381,892,889,120,897,711,718,843,633,424,566,485,35,169,586,391,97,432,861,544,416,137,717,359,96,454,598,159,709,366,78,937,916,776,473,623,626,623,265,23,318,797,61,380,798,594,374,950,545,212,439,861,407,138,94,667,546,686,63,899,803,730,729,47,351,842,145,369,304,915,374,600,210,105,138,860,490,476,335,469,53,634,849,676,867,51,637,789,941,180,869,478,808,997,535,251,159,676,583,413,450,684,446,247,900,536,588,301,267,780,432,150,252,725,564,219,893,229,33,553,619,37,872,846,282,971,7,309,113,374,556,336,979,624,563,349,916,761,379,556,785,786,311,908,497,923,257,913,993,242,977,827,350,935,648,667,766,754,578,985,143,578,876,190,546,941,578,274,604,505,785,269,613,434,161,395,294,625,519,19,822,97,421,85,169,792,832,888,390,296,122,258,999,144,672,931,687,106,700,254,603,752,578,989,64,276,651,288,207,225,302,862,927,23,596,630,719,666,858,866,440,798,226,299,736,675,715,677,472,340,684,214,932,237,383,570,767,739,800,957,716,124,747,992,763,861,624,788,420,882,624,201,850,739,570,873,302,726,400,490,918,316,358,385,879,280,538,78,872,881,778,224,599,410,176,804,656,445,157,939,399,106,439,774,768,322,586,137,66,730,56,743,689,606,143,861,716,713,525,733,38,20,597,296,515,170,70,229,781,453,140,985,108,500,23,574,210,412,242,662,448,608,396,144,462,823,634,309,310,177,58,854,202,672,425,279,530,502,210,231,620,215,223,426,335,597,868,630,434,499,118,473,862,519,410,713,987,760,814,680,919,897,858,320,591,181,812,104,401,884,289,687,599,608,60,963,216,697,594,627,116,865,550,476,131,218,806,641,410,762,834,612,828,245,55,730,52,714,955,99,2,384,818,229,743,699,940,700,281,777,514,470,430,570,822,60,371,306,471,354,114,138,86,878,303,858,445,624,710,253,456,669,804,859,637,409,360,63,165,494,458,998,231,343,107,350,720,64,486,953,328,187,759,598,338,855,669,197,203,88,737,489,73,766,199,532,249,855,314,780,451,994,537,762,594,658,652,88,280,888,77,398,384,497,64,883,835,177,981,735,688,463,978,140,966,694,348,409,284,907,890,939,184,646,912,69,777,414,716,429,63,620,611,456,994,16,740,366,802,832,699,688,126,845,540,649,1,426,953,561,75,592,271,444,326,811,852,400,675,813,221,623,36,523,68,891,930,158,219,948,564,947,190,97,453,784,704,713,188,78,370,2,608,494,595,53,519,274,858,402,862,300,932,423,214,662,607,175,829,319,919,161,688,537,41,885,894,608,130,831,35,272,718,896,591,470,457,55,676,965,26,445,869,166,512,172,395,665,171,966,821,807,762,146,876,102,85,282,242,761,399,671,897,484,46,637,304,316,571,541,770,876,683,675,878,368,66,27,926,973,110,193,129,797,170,92,641,379,710,469,216,279,993,38",
    "644,703,454,557,352,223,484,962,329,265,617,259,4,369,278,933,956,160,793,649,733,858,246,726,27,539,192,735,571,858,436,474,337,979,557,824,953,211,952,365,972,923,389,986,622,847,542,315,917,466,398,769,467,185,145,462,467,690,618,892,305,280,94,446,914,624,833,124,311,108,449,352,589,608,24,538,402,747,963,967,230,0,339,598,207,37,779,627,70,531,607,536,148,572,148,317,651,421,194,436,892,602,229,654,510,833,609,549,789,481,378,204,337,396,989,361,103,237,125,501,810,715,225,58,399,499,384,883,551,601,706,37,152,79,574,497,538,492,266,455,260,555,910,686,440,736,276,860,918,520,380,659,122,674,457,423,917,354,24,175,21,969,975,880,902,773,715,198,361,816,373,132,691,25,38,545,161,504,491,180,389,812,751,557,711,143,353,974,953,625,564,210,187,110,517,182,460,910,310,969,484,963,339,950,9,560,829,352,903,851,778,998,454,588,650,818,887,311,829,88,868,673,511,640,587,92,868,858,5,613,934,121,228,664,180,782,163,403,696,881,528,770,736,920,65,265,68,363,442,846,919,616,649,907,797,674,507,269,275,840,424,591,616,984,902,291,511,979,483,767,6,348,485,37,499,921,422,53,289,732,435,194,511,188,300,121,188,310,574,803,932,370,687,566,483,566,968,372,263,632,212,640,306,305,279,540,235,287,787,215,727,891,508,995,979,444,994,31,620,75,386,7,659,281,650,112,462,4,111,947,812,298,346,254,527,145,176,186,355,830,247,615,707,626,230,492,299,496,962,47,974,728,201,522,268,550,576,344,185,570,903,405,627,165,137,759,949,614,362,861,868,529,531,507,320,738,410,852,712,680,150,788,995,277,933,43,942,862,754,634,294,363,153,123,666,464,845,205,71,409,546,745,912,422,379,318,396,442,899,696,39,9,875,638,424,24,353,302,768,929,522,182,913,799,162,230,719,462,686,296,401,690,167,769,642,366,824,317,433,506,207,830,565,794,281,350,631,581,715,467,691,248,362,339,862,306,489,624,736,170,705,818,245,922,192,718,169,944,852,786,92,33,861,85,313,354,792,337,444,877,139,958,499,723,22,939,799,783,162,461,629,117,616,563,122,310,627,111,70,296,467,105,675,228,103,972,652,473,290,266,232,554,909,877,136,584,914,264,221,141,259,952,597,658,29,824,268,154,449,606,764,475,593,681,799,161,850,194,122,24,633,153,112,861,702,354,183,119,697,944,631,248,126,287,885,189,243,818,476,478,895,188,917,315,450,790,759,273,222,830,495,455,884,620,951,656,331,14,27,898,484,523,291,170,131,827,314,630,333,112,248,878,443,884,998,671,2,738,333,770,623,951,677,887,346,493,735,783,933,528,332,483,899,471,595,631,355,461,424,12,555,355,414,566,959,617,788,695,253,48,706,277,559,119,577,657,751,869,582,232,43,430,1,349,688,814,566,120,184,351,768,257,811,985,863,580,912,29,32,219,148,351,306,11,477,603,117,434,196,643,774,66,929,906,973,759,448,161,796,694,416,36,239,173,156,178,307,405,157,133,582,308,936,532,478,762,366,377,607,425,247,645,558,317,604,913,226,127,508,119,98,358,2,486,854,80,672,326,592,288,938,487,326,351,90,179,907,667,366,894,252,882,324,204,727,565,908,101,783,876,922,280,219,111,13,701,410,113,632,797,843,725,900,266,908,587,502,451,473,613,730,107,892,357,7,328,469,628,201,854,943,809,490,218,613,371,530,232,243,276,501,91,256,740,264,861,785,546,474,777,338,650,691,194,343,989,69,626,143,649,172,1,411,370,73,961,546,189,767,161,164,571,463,572,990,833,365,714,717,583,872,359,5,920,929,397,68,438,652,484,335,571,29,557,454,164,642,335,88,907,948,490,310,685,820,625,11,879,550,720,18,391,338,623,227,870,426,679,534,809,247,998,888,494,279,961,874,900,251,842,141,356,12,220,850,415,698,20,880,293,498,705,197,903,246,990,487,222,350,771,230,585,715,174,110,913,380,226,754,358,955,66,872,628,688,973,299,121,50,505,489,433,953,227,653,906,239,333,58,895,891,283,25,35,936,9,442,713,632,152,62,513,510,971,819,701,782,253,714,334,931,614,93,60,297,314,68,283,342,75,180,858,844,494,824,864,626,255,762,593,58,24,167,809,119,852,83,792,697,45,457,26,77,261,356,174,252,252,789,337,914,747,725,272,141,477,833,505,278,963,962,901,362,68,172,393,815,604,469,592",
    "701,602,786,930,793,902,378,361,173,22,917,733,260,683,164,325,86,359,250,451,883,826,138,107,229,625,418,918,725,459,143,338,610,986,870,679,996,629,870,655,401,546,956,443,65,430,205,194,615,279,746,758,657,989,811,110,146,303,387,306,898,471,242,71,728,334,203,598,556,145,125,729,222,694,692,271,918,893,947,850,448,339,0,487,742,764,983,413,296,375,897,564,693,134,263,198,246,269,4,692,966,259,992,408,547,162,521,954,46,199,415,591,79,575,719,549,453,243,542,107,414,204,484,251,413,999,953,372,757,982,354,706,492,409,245,820,780,812,55,474,611,589,679,643,929,404,269,780,824,109,720,457,216,414,341,312,413,318,668,523,806,887,268,19,434,447,870,973,550,144,260,255,889,698,886,101,288,805,568,698,849,983,632,539,888,850,691,745,534,667,138,121,791,821,27,352,374,630,321,568,928,438,15,818,94,8,526,802,404,573,42,935,45,80,884,798,282,540,871,180,621,217,310,33,788,107,663,805,983,4,991,908,176,411,387,890,828,605,188,545,552,687,203,937,238,715,305,338,480,116,385,306,430,98,522,603,649,652,875,966,442,530,665,953,366,236,899,211,514,276,798,267,168,852,923,270,920,565,102,481,539,484,240,869,82,534,297,443,414,152,496,181,685,476,398,257,238,48,329,679,610,551,519,506,593,165,141,631,49,311,508,879,165,616,651,402,181,47,259,143,638,821,581,684,451,273,989,63,649,484,443,301,595,329,263,430,671,990,556,749,966,212,651,641,305,583,559,589,309,909,43,643,66,561,789,550,873,12,327,522,65,187,751,178,137,719,965,932,921,931,608,539,580,523,175,242,683,833,604,685,503,593,428,207,887,292,401,88,258,923,475,660,440,734,20,832,972,300,705,886,317,317,14,43,209,859,502,220,89,634,210,698,665,897,56,678,43,921,552,646,48,769,455,685,491,721,990,557,325,29,808,923,290,137,216,530,614,831,178,588,283,597,60,328,67,889,67,676,376,308,82,964,243,144,484,23,874,876,430,986,158,484,99,418,501,924,220,980,219,983,895,137,173,470,684,771,984,23,826,482,387,458,923,591,182,545,28,738,994,645,620,885,169,21,942,723,58,315,531,287,27,856,310,166,742,591,537,486,792,400,607,970,45,488,178,808,149,241,215,153,412,861,94,372,111,809,119,302,809,402,735,463,692,643,307,102,648,549,897,201,897,209,390,483,606,66,445,348,145,975,672,164,644,840,527,602,683,139,26,883,723,680,124,281,57,281,812,915,376,612,684,504,419,406,310,709,292,780,992,28,197,613,300,432,76,475,618,511,526,460,914,894,74,756,836,338,555,337,293,317,413,658,273,876,777,37,957,947,804,86,731,644,10,604,110,989,780,649,507,54,69,772,281,346,685,754,694,619,729,829,518,642,248,586,994,863,962,747,108,777,246,773,516,138,222,166,490,600,484,111,72,55,566,250,476,749,177,239,623,627,857,479,969,303,660,316,659,415,710,175,33,907,723,304,644,512,264,854,633,244,304,803,472,63,84,5,181,98,433,645,750,319,797,132,340,601,43,610,815,836,387,539,191,963,862,327,683,244,232,532,956,250,278,60,377,33,756,736,647,325,897,100,604,352,992,52,207,925,530,446,712,762,855,490,64,427,504,524,124,969,406,152,698,211,410,675,279,939,273,353,607,216,252,330,892,960,898,2,748,829,30,952,959,743,255,26,876,98,461,29,429,906,757,88,512,351,520,259,409,944,928,597,116,297,342,896,825,836,467,337,212,240,657,445,179,806,294,494,53,250,202,948,66,351,742,546,196,61,123,908,346,798,89,854,947,877,985,268,457,763,324,363,209,536,460,454,87,765,554,260,214,705,673,697,560,517,576,548,251,154,560,185,820,51,486,222,117,699,832,387,491,134,308,300,931,435,581,505,367,757,374,257,916,62,893,607,146,609,437,393,299,398,895,462,431,120,86,470,80,835,627,753,777,307,595,485,717,552,17,489,811,311,780,508,112,984,627,656,5,992,453,133,754,806,396,398,998,238,627,688,230,53,27,287,925,269,314,27,934,573,177,398,856,241,905,297,157,197,951,916,778,437,747,465,37,917,321,945,432,547,691,630,573,727,654,671,847,853,982,629,77,39,677,362,501,293,802,443,548,909,467,884,818,644,647,992,882,632,772,166,383,615,27,801,284,744,296,22,262,582,939,500,614,884,716,77,95,623,294,410,148,485,995,978,348,839,956,911,334,387",
    "782,256,497,80,354,231,977,385,393,190,301,932,205,385,524,244,616,687,285,627,707,659,664,369,799,877,659,18,322,169,496,513,557,351,364,210,244,863,490,510,737,496,670,449,313,181,931,554,522,737,348,255,663,259,360,910,514,193,557,831,191,374,423,913,195,893,120,771,166,651,277,871,699,400,716,824,735,926,870,312,214,598,487,0,814,347,391,248,410,804,476,679,394,518,637,471,786,664,574,413,591,524,910,156,244,18,389,925,21,641,157,652,281,758,607,664,289,395,807,193,722,420,84,185,704,221,429,307,677,593,506,372,403,115,408,392,778,11,378,796,515,973,314,703,377,830,50,767,426,8,225,12,123,68,208,385,9,136,575,398,527,425,956,649,765,781,118,549,370,725,396,173,354,749,744,793,834,57,468,740,433,668,537,871,123,164,373,384,869,184,219,833,781,911,509,147,959,223,821,381,649,233,998,413,873,214,656,518,792,998,803,851,423,121,24,387,290,512,681,131,134,480,726,188,778,379,804,391,102,351,737,996,942,872,865,422,116,302,942,619,114,129,407,390,288,611,876,654,340,86,18,767,873,152,812,135,646,857,744,401,178,522,41,860,650,839,687,819,754,213,462,550,772,785,857,897,470,65,988,871,824,706,466,373,986,559,629,360,822,171,294,908,992,277,232,247,783,174,387,658,560,330,748,951,130,486,434,692,849,42,498,667,976,764,383,521,683,79,853,40,900,310,775,680,192,972,40,551,718,169,666,147,186,664,527,29,924,426,634,580,530,343,469,194,858,409,160,369,710,863,966,12,838,493,89,369,360,379,942,174,486,316,49,783,96,793,991,853,442,810,114,609,540,281,7,9,638,363,368,763,191,341,546,552,452,60,63,29,540,694,273,816,609,618,830,81,568,712,648,153,213,778,961,920,147,537,804,842,280,646,502,982,225,54,124,429,834,635,600,803,814,149,324,353,578,279,919,160,527,295,419,707,820,594,958,9,566,564,909,64,243,851,497,786,46,200,893,314,104,205,953,590,184,326,562,618,91,930,934,3,30,353,104,83,941,575,342,821,687,809,522,143,422,726,23,809,800,983,987,719,52,413,640,687,683,189,505,382,535,555,500,431,361,939,838,230,545,236,587,271,192,459,857,149,84,524,348,775,754,661,204,730,881,464,800,714,818,458,415,340,793,432,335,801,204,213,989,419,372,960,200,688,179,383,410,985,164,288,230,679,732,989,99,10,738,285,91,671,39,279,611,414,43,212,661,271,740,24,973,478,976,256,919,465,190,173,667,422,449,248,224,820,1000,638,67,6,117,17,387,712,833,254,794,872,345,546,155,210,404,336,456,999,631,764,613,818,72,735,938,818,46,321,574,143,268,646,243,109,936,988,17,153,642,351,299,518,979,430,643,522,9,663,858,477,397,360,787,391,178,454,328,682,138,490,790,874,621,620,834,330,636,259,123,798,787,491,895,154,135,568,15,488,579,387,232,598,145,335,495,805,862,285,580,2,240,119,142,749,21,585,284,598,3,974,206,267,518,838,844,308,739,792,925,263,312,938,367,98,143,23,660,103,509,133,999,592,399,774,673,846,961,544,947,290,437,458,16,360,539,956,348,780,229,87,6,69,819,654,987,360,804,651,239,572,225,940,179,25,987,911,953,410,215,54,598,225,235,422,754,689,293,382,334,461,18,269,967,511,662,165,895,240,119,328,125,688,691,450,811,979,391,197,799,189,56,512,575,76,491,636,355,532,972,432,492,1,131,664,666,929,527,136,988,16,891,342,632,486,428,628,474,804,707,256,207,621,15,494,649,78,53,614,463,510,643,695,972,745,627,147,3,362,422,401,281,739,928,70,601,504,485,927,798,633,421,45,576,626,437,565,916,856,804,721,260,25,163,291,491,911,389,707,312,97,143,678,367,816,267,902,462,963,434,949,673,464,548,172,673,522,770,36,871,114,484,736,473,26,808,940,625,104,935,705,218,963,247,269,286,703,75,825,259,709,459,96,883,38,316,895,99,419,46,9,169,968,988,993,449,888,139,325,179,510,302,545,3,567,625,813,377,70,490,333,457,126,174,894,878,768,850,399,535,419,704,280,598,840,897,291,566,66,79,688,768,124,982,51,837,7,554,590,694,566,621,405,462,573,22,515,451,441,321,829,222,277,256,680,899,759,918,187,669,979,900,858,604,408,919,104,982,570,231,611,132,33,691,764,875,780,722,516,560,829,559,608,189,920,949,826,644,952,575,944,710,98,995,848,191,416",
    "870,997,517,595,862,910,227,671,793,134,305,89,751,492,254,45,803,576,666,727,20,272,836,208,819,111,238,369,222,208,495,217,69,4,557,388,239,411,564,463,425,508,67,219,472,909,693,915,916,671,821,889,370,256,138,160,568,580,799,264,253,415,60,538,986,32,723,776,61,190,949,940,202,760,22,624,185,934,529,20,436,207,742,814,0,557,41,734,603,383,164,529,579,951,610,703,153,651,264,727,585,487,127,223,634,205,293,725,464,241,225,809,461,968,992,639,31,686,155,147,447,276,306,952,53,490,554,460,187,840,849,356,270,386,380,309,12,582,616,847,271,493,953,28,446,520,630,114,343,309,366,305,489,724,736,317,488,107,219,94,34,500,625,333,628,951,436,494,317,457,813,536,320,540,729,849,654,889,210,56,275,677,997,620,403,612,363,813,942,620,434,762,361,974,828,695,873,253,827,493,227,404,637,237,53,7,478,69,276,284,886,635,171,672,783,396,582,303,486,506,928,612,875,334,513,313,649,510,742,368,337,948,576,329,691,238,511,686,673,370,366,992,432,810,133,524,20,741,150,275,937,380,78,876,692,509,277,367,779,454,797,775,439,965,715,534,748,390,801,540,549,53,350,480,183,4,893,404,84,689,582,930,19,980,433,740,666,104,56,472,610,992,340,756,962,8,53,58,351,518,698,511,912,696,85,411,174,645,374,733,97,539,29,994,702,879,754,789,387,80,303,165,927,770,413,147,900,320,831,254,166,692,258,499,459,778,416,694,24,611,477,143,68,16,225,474,871,933,687,347,611,118,894,830,122,459,865,267,560,628,986,947,700,110,686,636,367,501,420,635,581,228,954,624,694,329,240,741,807,549,619,74,93,125,435,358,456,815,985,589,390,760,371,142,308,531,453,216,754,912,736,696,746,517,191,271,388,689,370,1000,998,597,78,209,972,833,997,874,825,856,64,730,800,937,53,268,881,880,268,135,98,752,324,846,521,591,538,533,714,892,450,839,740,256,241,360,23,338,420,650,671,88,359,399,217,571,756,569,438,88,350,852,560,91,864,650,105,475,355,566,369,51,427,71,730,354,846,168,948,240,392,555,391,308,30,266,293,516,517,681,508,358,105,475,140,720,327,953,926,830,181,867,695,123,811,497,850,915,275,794,306,151,267,100,827,519,78,138,676,842,522,960,941,73,162,682,382,372,63,345,591,78,39,722,805,997,504,92,937,77,610,976,65,88,287,359,53,777,796,987,987,292,733,414,346,442,901,161,714,310,554,667,547,384,316,202,594,788,135,453,317,509,182,585,50,223,955,25,624,500,700,824,801,889,826,731,327,90,876,253,181,51,987,481,322,656,700,868,104,336,930,925,571,518,335,227,178,777,761,851,216,689,294,910,661,305,24,514,653,338,412,624,844,456,670,70,179,805,134,581,295,131,677,262,647,107,515,751,949,395,868,479,187,122,950,179,771,452,685,110,313,342,431,21,173,490,267,754,694,57,460,121,261,571,807,166,648,236,356,4,233,297,545,626,815,720,519,463,514,259,198,510,294,601,64,582,893,7,378,79,273,548,986,625,646,128,586,920,184,439,431,753,552,701,522,896,806,677,760,743,327,150,746,114,392,937,309,626,260,912,468,587,420,532,87,274,564,433,194,509,295,816,765,630,605,360,437,754,668,396,845,716,66,632,41,809,522,725,340,127,467,280,610,797,306,781,87,604,553,546,78,425,488,621,16,932,997,240,550,793,175,806,912,372,1,330,967,424,80,256,83,28,535,140,929,924,34,14,562,702,751,19,143,394,100,976,586,991,861,90,321,991,215,775,328,438,351,835,908,58,189,129,559,704,149,279,882,430,484,475,275,727,909,252,158,702,541,227,184,160,976,119,276,693,953,391,680,593,965,865,970,747,185,620,978,876,167,705,997,699,792,961,223,61,563,442,282,485,487,176,369,884,319,589,180,440,498,342,563,603,931,597,636,755,953,814,967,68,858,49,578,101,288,326,959,228,104,635,486,25,365,727,905,628,557,273,314,302,51,452,766,344,633,454,819,376,449,935,4,751,786,720,137,112,363,256,349,261,103,118,489,399,72,30,16,101,85,572,929,583,32,437,853,300,146,41,489,526,122,267,146,213,485,47,860,424,752,396,770,634,614,30,696,894,394,806,402,387,782,200,855,177,373,761,254,416,973,49,556,856,696,345,990,638,611,949,602,237,83,510,397,850,934,3,922,952,833,903,301,83,601,115,547,648,636,431,444,89,419,150",
    "947,805,255,94,234,48,914,639,509,824,250,250,300,491,56,196,55,905,433,694,726,260,680,116,946,59,928,51,250,899,683,949,129,264,881,107,866,924,739,481,70,709,640,69,942,747,618,579,142,976,91,867,524,280,870,374,390,100,67,29,860,259,657,225,869,603,521,991,351,146,957,605,107,67,1000,405,215,768,561,836,205,37,764,347,557,0,746,13,869,856,879,815,324,563,728,771,503,888,338,275,133,422,436,385,586,533,188,959,762,684,174,234,348,109,7,327,315,801,718,25,312,9,853,623,171,317,12,502,355,690,561,125,990,936,689,226,377,380,77,936,609,597,356,715,890,961,831,91,344,685,913,515,485,125,390,706,85,246,604,99,938,800,85,692,737,208,988,445,54,307,806,277,761,218,97,300,8,194,254,494,753,902,734,945,16,450,642,528,39,998,181,19,271,401,808,762,123,802,70,675,497,409,285,194,707,518,490,156,777,885,438,136,852,277,732,106,854,573,990,578,481,628,672,479,971,388,273,238,84,972,358,310,413,70,260,861,622,805,317,808,47,5,637,715,730,347,36,183,231,759,911,195,149,232,881,75,429,101,664,266,51,275,324,324,666,414,743,376,518,347,562,634,19,25,707,467,833,928,661,310,260,681,833,868,809,563,23,206,748,299,544,511,783,444,103,400,519,410,229,416,559,732,910,685,463,710,871,519,888,260,158,958,889,686,239,368,517,804,792,650,590,837,594,478,423,628,485,85,290,301,159,429,247,215,786,696,380,271,656,518,730,430,79,543,19,531,233,262,805,280,843,179,702,869,951,204,171,732,20,729,776,313,369,550,650,744,667,925,4,992,356,850,590,886,353,859,258,607,279,121,607,113,852,66,358,831,600,392,793,800,834,686,281,396,233,104,582,579,952,874,913,168,393,722,434,693,149,103,455,328,149,596,821,842,231,494,375,174,122,533,858,754,107,921,637,20,315,741,162,969,309,793,373,986,390,404,77,161,25,578,146,8,361,253,234,279,15,883,486,460,172,52,917,240,554,725,484,278,546,935,666,914,889,201,686,738,29,575,601,65,447,604,513,300,279,372,230,620,705,817,869,694,668,956,612,832,444,77,730,750,875,605,568,173,463,62,431,967,560,85,671,685,225,509,234,474,301,316,171,756,942,51,829,715,940,53,289,924,115,503,678,913,171,488,992,102,394,544,174,186,585,913,6,776,479,225,65,586,900,985,326,2,150,712,822,697,968,448,188,970,594,927,257,915,548,622,115,846,352,959,256,828,854,880,551,899,542,715,146,17,373,231,523,947,560,693,595,330,683,345,488,942,289,587,380,730,347,818,871,168,984,515,752,142,572,304,30,213,361,427,991,583,831,993,84,70,235,678,373,497,960,268,669,131,785,204,958,654,271,965,467,509,371,923,690,751,477,233,396,380,453,701,25,182,618,737,204,415,451,922,110,387,227,825,838,817,221,404,22,642,142,427,959,829,445,492,735,619,684,435,33,73,482,925,756,735,687,71,189,971,110,218,744,247,483,881,477,552,548,26,812,264,708,406,998,418,968,859,70,744,807,903,593,98,632,462,474,754,552,859,658,207,533,460,691,295,603,534,118,341,822,57,217,547,726,704,988,800,528,933,316,916,64,764,211,619,594,939,467,818,441,815,545,926,962,810,724,960,933,818,444,79,816,323,414,536,933,795,820,31,790,672,183,357,782,208,643,350,198,904,750,925,612,152,860,846,951,148,188,612,259,237,777,859,244,351,447,834,615,685,572,661,250,282,109,840,872,453,151,644,349,117,658,355,937,545,741,28,630,804,170,389,293,169,722,720,634,507,813,270,457,788,734,187,957,229,457,721,749,161,667,663,262,960,141,36,21,801,585,631,661,446,364,72,395,613,167,25,280,490,845,488,567,172,745,109,27,361,995,474,378,32,117,234,941,636,643,110,493,583,55,411,619,590,83,713,695,52,28,866,383,421,162,494,674,119,684,601,799,637,382,440,106,46,169,993,630,479,845,273,687,589,955,865,393,347,735,433,283,879,483,125,951,899,60,729,85,31,79,790,414,263,21,125,758,915,566,893,982,895,369,823,844,992,597,610,951,631,756,242,523,562,638,74,171,410,456,114,494,842,434,572,769,430,527,884,940,975,473,358,548,250,917,237,924,396,706,225,513,655,289,887,993,25,816,303,258,589,8,130,892,865,792,990,763,559,146,507,392,287,932,407,906,351,376,89,984,657,179,763,572,405,978,537,329,186,896,999,993,191",
    "159,922,417,675,644,531,681,703,839,608,368,531,780,299,414,69,462,598,188,916,39,606,607,113,537,637,360,589,787,615,463,914,488,190,950,151,203,11,891,928,901,892,751,990,656,484,140,450,64,754,491,25,753,732,301,113,177,968,926,999,864,524,177,789,457,38,677,346,269,838,658,502,479,752,783,65,159,303,911,922,404,779,983,391,41,746,0,1000,640,445,26,335,454,850,944,211,283,244,609,194,655,887,828,551,279,622,111,852,340,964,322,579,779,725,948,409,498,832,706,134,612,181,94,721,570,928,195,712,471,697,55,926,110,169,628,681,495,608,35,191,497,600,658,194,349,512,813,908,911,978,870,556,733,655,310,799,786,671,6,525,891,612,597,214,344,102,791,603,693,949,118,283,425,262,900,834,376,665,708,316,259,215,426,890,466,665,141,140,782,311,974,65,499,257,615,311,487,991,248,734,751,694,756,811,606,398,925,280,424,465,314,678,785,28,586,607,757,783,988,128,523,468,568,41,978,440,685,804,862,682,739,61,842,23,675,722,34,316,831,444,967,919,240,190,677,775,423,175,693,676,319,890,798,429,374,502,486,241,997,760,9,739,880,338,754,928,204,724,572,440,355,70,148,523,747,439,749,623,642,883,940,910,322,957,29,915,350,413,249,118,802,529,851,710,299,775,885,94,236,79,242,954,472,354,248,163,212,815,930,396,347,59,120,121,57,561,389,599,618,970,49,172,730,951,456,274,784,905,915,28,570,197,732,258,946,706,925,16,77,865,48,428,519,278,671,282,740,714,68,246,732,330,339,241,93,700,733,938,840,559,770,801,81,634,514,758,628,318,103,524,815,484,186,493,444,102,380,473,650,178,641,662,898,737,577,22,378,848,110,766,261,454,532,655,634,667,482,582,485,959,659,44,791,2,528,738,197,438,457,350,178,663,105,973,878,702,668,402,317,234,37,874,73,749,582,357,938,584,741,887,977,172,892,844,58,598,396,506,507,530,446,291,467,370,539,839,899,956,77,974,451,308,162,438,819,738,652,901,414,398,587,210,654,794,323,619,486,441,105,164,64,99,652,745,19,520,68,283,161,431,367,107,380,258,969,395,196,98,175,252,224,852,627,367,963,978,732,556,979,702,826,394,475,851,925,142,800,75,324,45,686,819,999,742,948,250,180,382,40,758,571,832,766,567,754,655,537,767,399,68,975,212,40,599,898,345,878,3,538,883,56,31,742,999,600,243,888,206,738,150,109,687,294,114,707,990,218,819,277,535,187,289,305,857,784,149,221,834,737,799,328,752,99,947,576,332,672,899,59,411,443,104,299,17,336,401,222,320,170,166,528,957,449,99,106,230,878,356,515,990,589,22,318,199,816,91,591,777,155,698,4,699,644,384,621,893,334,673,489,919,675,493,886,886,173,33,538,316,116,446,32,90,48,601,60,687,939,216,635,423,984,688,490,622,657,918,394,432,85,545,700,884,877,149,55,246,761,543,19,307,427,62,433,419,130,501,367,330,627,902,714,821,682,929,731,479,213,642,618,87,879,325,724,271,905,512,885,812,316,830,927,148,685,991,59,706,533,168,31,420,24,311,803,836,857,946,363,521,929,587,741,770,568,442,636,103,440,831,416,34,890,402,252,137,524,778,502,65,621,263,381,993,385,220,413,909,200,630,466,964,613,516,227,619,893,688,183,407,586,903,32,747,16,738,684,72,520,204,863,89,878,886,368,614,346,557,65,318,292,214,153,691,473,443,846,875,726,561,483,466,179,894,909,450,781,374,293,841,23,870,915,593,813,53,80,377,452,797,321,928,772,434,309,742,810,231,839,200,578,686,141,80,996,615,38,463,183,166,433,663,26,656,970,59,726,423,974,9,568,707,633,488,209,213,159,246,233,114,810,250,632,593,658,209,490,508,488,462,551,88,321,424,527,811,931,216,250,82,281,3,902,722,870,804,650,560,757,883,266,724,488,141,339,701,669,518,545,448,92,908,194,580,411,617,454,677,492,86,697,709,950,29,456,920,33,433,522,582,667,683,25,808,804,146,321,254,349,644,797,32,649,735,686,244,569,454,188,813,505,899,978,17,435,68,424,75,152,199,584,19,682,527,322,9,525,387,630,139,419,517,440,272,934,201,627,71,52,717,888,622,406,616,512,933,932,797,300,23,985,581,341,120,680,417,752,674,563,112,178,460,336,654,6,48,800,149,84,830,656,800,943,28,88,638,532,761,79,873,298,573,172,708,333,582,774,767,28,996,880,159",
    "464,332,518,781,180,336,70,192,152,952,491,728,861,906,615,374,526,694,760,473,590,627,124,877,504,929,824,281,600,522,49,682,336,12,382,436,723,314,50,350,151,44,59,606,175,314,524,146,692,190,707,982,228,46,672,768,4,782,332,763,652,635,566,357,253,414,28,21,859,706,827,310,392,403,283,622,60,683,739,975,69,627,413,248,734,13,1000,0,865,273,347,460,77,449,519,11,291,187,46,455,742,425,387,426,912,666,274,793,580,830,847,214,342,769,820,661,156,321,473,79,25,995,376,585,491,481,95,255,202,438,433,642,501,295,217,810,834,466,380,490,658,706,242,241,891,391,7,947,405,461,225,927,37,385,126,54,643,546,280,611,681,160,191,480,682,520,436,293,321,431,646,962,40,18,32,155,471,867,129,117,46,60,326,585,627,752,368,436,965,379,620,896,118,40,974,908,963,389,800,952,939,515,864,895,667,669,559,187,458,953,19,581,264,983,984,349,959,805,353,613,64,291,283,18,632,257,859,595,600,678,274,570,331,147,825,627,218,704,478,851,686,345,42,398,134,80,230,680,382,515,904,584,844,821,723,890,147,802,239,668,680,255,34,562,148,147,521,435,611,355,775,683,708,140,535,934,778,699,274,722,638,622,683,274,622,850,203,94,344,946,334,542,714,887,451,356,513,787,78,442,667,373,333,480,944,848,640,471,632,228,48,348,480,585,418,768,892,278,380,856,580,360,928,893,342,994,718,861,3,913,33,617,355,722,692,262,749,390,11,510,78,869,171,327,867,238,344,260,853,548,573,288,886,184,105,321,206,675,742,630,944,378,796,799,402,867,771,546,625,119,110,315,3,914,654,880,742,999,866,75,899,357,645,680,108,47,534,678,531,173,76,602,372,978,444,112,802,242,768,321,361,287,105,737,445,459,314,824,357,291,130,11,900,237,929,465,926,119,123,735,954,449,756,290,595,112,421,160,792,887,396,285,811,24,494,470,387,56,200,585,422,893,974,432,692,287,879,343,33,1,212,204,975,94,353,806,903,625,289,2,974,366,139,870,586,880,625,874,835,597,717,198,480,324,664,21,616,388,307,85,862,917,807,958,168,844,836,817,441,291,714,594,972,293,516,919,137,205,652,601,742,798,691,784,864,3,850,219,767,326,568,784,718,856,213,280,167,639,817,136,734,355,34,736,380,915,915,508,749,40,252,489,670,954,458,148,592,581,795,676,211,410,481,850,538,242,407,950,944,938,539,365,476,706,447,775,681,853,728,618,280,634,792,515,112,624,389,55,782,602,73,74,343,699,389,326,575,550,776,94,149,829,949,173,648,682,956,27,603,797,352,147,850,82,980,44,977,248,343,879,558,989,263,105,189,752,190,550,901,999,496,80,340,867,999,215,617,969,47,672,709,654,99,822,721,527,548,766,326,988,376,853,422,526,950,812,74,739,804,262,328,131,866,751,325,33,878,736,314,942,562,386,585,59,640,151,981,177,469,81,308,660,187,647,190,49,936,984,91,208,155,892,103,351,387,494,823,508,408,590,531,696,261,647,364,579,662,619,419,516,422,697,960,700,547,466,644,931,910,295,961,447,510,76,109,513,9,20,678,476,863,574,829,740,135,510,599,650,448,562,727,840,559,470,470,350,934,692,896,499,926,528,280,996,736,507,707,510,113,150,68,207,288,444,626,239,572,574,72,972,327,126,541,722,724,501,948,352,26,881,739,837,698,674,716,920,365,930,413,124,752,857,249,945,928,543,267,913,287,958,860,824,46,672,109,921,79,526,827,696,762,832,326,130,472,802,72,342,980,616,621,261,114,710,347,6,957,724,234,62,139,150,973,975,207,733,765,843,223,369,351,685,797,729,847,501,971,661,13,880,916,974,758,728,761,107,922,943,546,35,283,588,14,777,144,664,369,483,94,464,214,182,257,446,34,16,809,20,281,812,737,505,500,47,347,504,237,687,903,1,559,541,748,552,21,6,673,399,515,610,43,207,726,934,502,276,808,630,587,691,132,429,424,984,581,865,748,812,270,838,208,662,855,443,545,85,406,197,799,997,681,575,525,717,832,914,689,744,164,92,440,785,553,888,373,16,483,847,88,628,177,259,134,581,347,151,630,453,427,110,42,41,296,92,718,715,613,696,278,920,461,533,333,474,454,655,784,760,783,20,180,345,15,465,758,293,822,716,4,487,898,83,339,398,565,740,265,119,802,442,203,731,767,395,680,774,32,253,786,974,664,650,783,251,183,250,593,112,459,679",
    "412,116,417,143,361,936,970,930,287,465,354,444,242,31,330,264,701,529,297,544,303,532,624,36,789,336,133,357,606,322,574,736,920,482,499,977,649,622,532,52,837,666,810,903,217,293,813,852,7,725,481,191,668,377,109,315,636,520,918,361,188,97,584,728,698,588,567,817,142,813,116,203,570,455,753,827,716,174,137,359,52,70,296,410,603,869,640,865,0,67,85,37,385,208,26,633,138,113,817,670,650,808,663,434,194,448,614,770,258,273,209,807,214,345,767,773,167,876,887,790,317,432,456,387,43,974,629,652,105,532,479,197,101,126,989,324,523,180,588,644,202,666,102,156,536,833,246,618,58,490,216,719,673,841,974,981,77,874,306,390,840,189,696,535,84,341,723,849,142,957,270,509,404,699,395,309,587,218,520,857,173,744,99,23,415,648,232,904,455,204,922,329,919,631,917,474,881,575,150,843,102,544,154,927,98,893,92,684,738,589,766,680,422,137,277,192,362,51,249,212,909,272,843,44,539,944,93,832,740,933,483,501,316,10,863,380,798,641,925,354,298,181,533,566,366,333,290,448,590,287,650,545,685,198,252,555,952,175,622,704,379,623,86,179,356,783,284,123,177,829,666,396,814,775,611,544,913,550,775,191,1,96,322,956,334,469,906,465,85,659,119,333,673,31,27,959,590,505,171,663,319,306,764,9,84,647,760,33,392,359,942,790,42,640,622,99,897,919,197,384,996,181,20,351,429,554,925,348,514,149,540,393,786,221,969,157,533,711,380,48,665,765,48,468,166,352,506,204,730,383,35,575,326,123,576,318,174,544,901,619,32,3,948,890,206,520,319,902,278,733,155,895,392,904,847,276,740,127,333,157,624,154,273,865,76,837,689,796,74,869,16,436,792,584,486,278,563,888,683,804,856,717,993,509,736,50,783,562,81,921,743,636,116,781,19,905,328,600,420,414,255,226,165,106,495,36,831,560,414,837,710,729,798,229,580,62,169,538,934,40,201,718,977,741,142,990,423,394,162,634,205,31,755,478,885,110,580,642,61,165,234,755,370,587,843,993,49,387,708,626,146,30,89,617,771,122,38,692,388,881,621,70,212,189,272,977,510,625,485,768,806,832,894,781,354,277,277,913,768,230,798,758,213,885,523,459,511,985,665,427,88,972,751,279,786,277,55,461,798,103,916,989,420,926,999,859,631,843,450,519,774,460,479,538,277,541,214,266,173,388,337,701,864,868,544,886,272,629,683,288,787,863,354,560,72,968,943,962,1000,326,375,246,96,385,423,14,178,39,811,409,329,496,319,482,529,503,375,48,488,41,591,516,778,628,571,563,50,726,280,633,413,878,59,30,891,753,975,442,645,914,939,562,700,273,124,269,2,730,494,727,261,167,202,403,510,148,263,55,334,813,346,588,154,562,720,935,668,107,535,965,19,682,417,280,298,17,721,384,826,479,239,770,76,120,412,138,306,99,262,244,529,184,78,783,250,836,734,178,939,582,618,223,15,178,606,836,506,653,392,519,388,912,400,714,694,649,343,228,210,298,366,343,554,868,925,324,480,352,592,203,688,519,116,57,607,279,364,556,420,187,734,413,168,560,190,260,134,517,453,495,195,483,496,475,99,151,615,460,127,150,679,466,983,349,135,498,180,61,13,836,907,552,399,908,777,708,142,353,390,500,409,358,556,441,710,923,839,363,484,68,788,766,442,875,77,334,347,372,156,738,903,957,151,37,385,170,292,31,297,488,328,161,356,827,511,980,997,910,815,895,385,345,216,942,709,950,424,726,683,214,868,813,799,729,86,852,110,115,436,50,500,617,371,345,254,487,800,590,628,723,302,118,865,37,200,913,377,119,935,846,670,819,976,206,214,721,374,979,775,746,217,685,757,180,453,662,546,623,749,299,159,320,52,533,956,848,927,467,283,758,286,803,725,588,715,336,564,567,926,109,457,495,93,508,133,392,970,769,546,339,468,241,109,824,439,855,996,32,853,33,164,191,904,881,157,742,714,922,733,233,553,94,173,490,853,364,567,199,120,599,36,461,341,877,834,353,384,546,279,25,885,141,369,586,892,825,737,343,632,84,630,803,488,503,313,596,977,54,509,596,977,426,752,109,136,649,478,693,940,290,471,654,598,878,515,51,744,5,211,948,248,327,185,446,393,291,449,169,492,732,846,621,717,241,814,320,864,616,766,597,708,874,573,516,530,276,221,50,913,285,322,496,192,182,64,673,594,319,745,933,901,715,537,522,729,226,416,218,284,468",
    "807,639,523,384,702,64,789,605,549,637,760,924,36,551,50,613,599,391,268,67,516,654,19,63,534,844,127,637,51,9,516,813,66,571,578,379,281,566,857,800,763,464,1000,788,421,88,808,833,564,576,765,359,354,620,702,491,26,223,585,632,85,241,830,51,536,33,924,628,860,169,78,332,575,793,638,786,475,686,529,111,140,531,375,804,383,856,445,273,67,0,442,147,977,649,230,187,505,727,302,991,432,986,946,397,83,839,598,202,723,824,448,88,490,375,9,4,804,6,914,77,465,983,909,333,830,101,987,999,79,324,44,38,56,473,966,221,374,909,682,734,681,739,985,249,418,224,440,217,274,977,287,830,977,352,372,293,662,30,735,479,234,113,933,712,689,147,709,515,194,925,961,725,271,227,231,834,157,684,795,812,328,943,911,114,303,641,893,740,539,508,700,526,805,459,696,827,987,470,647,141,985,609,644,455,372,540,148,732,38,300,76,647,758,713,977,984,482,809,664,495,423,653,644,935,939,889,228,991,835,688,679,345,124,205,803,81,262,921,642,997,168,427,717,110,99,12,12,481,64,10,417,345,840,50,186,627,18,667,628,245,424,898,894,606,767,823,804,980,551,936,29,110,60,187,985,818,727,206,496,926,615,515,41,261,616,273,374,916,994,124,151,635,285,749,733,487,639,654,591,294,232,204,379,639,427,135,23,251,719,185,840,691,686,693,504,387,253,933,997,842,157,943,66,140,644,824,744,102,750,438,431,362,503,806,663,706,783,577,813,703,921,511,714,414,156,393,84,430,520,610,238,67,752,958,633,9,178,425,993,711,719,970,513,858,721,390,133,180,908,52,782,815,343,328,17,412,922,806,614,157,855,779,474,88,65,385,387,696,191,737,997,758,848,441,942,579,525,333,778,325,101,339,106,197,673,413,69,149,327,745,710,152,987,854,214,200,942,186,866,959,776,339,499,782,284,121,107,107,591,68,28,385,483,766,245,539,590,180,768,811,779,699,909,295,727,31,31,272,424,28,281,59,730,763,642,838,152,845,874,476,216,339,640,94,579,356,576,315,686,679,841,819,758,568,25,766,595,882,802,421,18,939,719,939,485,705,133,463,574,650,503,719,514,767,859,973,214,416,319,922,285,516,557,410,657,867,233,839,859,638,735,492,851,583,103,182,604,417,615,758,871,622,898,278,303,171,713,209,255,594,320,55,258,512,814,80,282,603,977,320,147,372,629,639,607,15,75,901,35,861,813,718,343,850,400,207,848,665,110,677,326,700,192,546,998,678,418,596,337,944,584,159,68,580,642,925,545,998,565,289,503,202,589,853,669,556,728,880,631,851,295,974,997,857,51,918,390,200,214,653,931,520,222,554,39,357,704,275,149,435,851,696,305,86,789,238,372,306,287,408,260,986,879,336,157,754,968,601,636,579,850,971,72,239,43,649,412,588,365,818,459,911,409,267,910,690,657,793,980,349,577,792,257,449,825,154,943,901,18,745,367,57,738,933,768,67,383,723,282,66,71,687,342,208,251,548,401,107,147,582,901,115,128,829,908,207,688,518,385,732,684,742,847,218,480,875,929,190,254,389,365,763,222,23,243,115,669,782,192,317,96,828,258,47,227,386,257,256,523,66,997,108,923,860,447,217,750,433,629,461,59,248,140,314,734,801,513,111,217,326,486,428,90,923,776,379,997,481,24,192,351,950,374,982,232,153,199,118,983,418,69,360,881,374,697,856,358,365,608,223,504,683,723,578,81,823,318,490,183,954,565,319,468,951,565,855,429,887,629,718,581,106,480,717,394,898,358,419,297,542,729,998,16,648,17,747,967,65,575,376,695,836,669,602,682,570,834,198,134,37,712,953,180,413,75,399,535,756,690,191,501,784,105,671,439,827,511,35,637,994,635,612,836,664,750,14,897,170,949,878,79,621,594,280,329,350,778,202,637,518,375,301,341,728,896,876,271,762,758,760,486,519,730,106,350,974,790,761,728,864,659,846,983,434,732,340,976,686,157,221,781,219,96,287,381,748,841,519,338,557,748,989,240,129,781,239,861,146,97,598,313,294,437,962,809,971,538,582,683,445,804,630,491,722,76,871,808,509,728,670,441,274,812,891,498,440,143,511,97,350,644,887,937,821,523,217,30,208,34,358,619,346,411,282,188,195,114,451,938,555,987,347,908,836,227,954,193,298,479,813,766,109,313,490,39,877,732,454,738,391,124,599,364,641,269,443,323,787,868,972,288,207,241,798,276,420,727,40,628,640",
    "765,333,896,363,792,421,910,515,202,875,119,356,508,157,373,748,883,686,318,930,457,119,911,136,434,653,584,359,898,663,610,467,463,183,939,701,749,267,342,134,490,693,851,753,684,145,204,731,438,536,919,8,939,978,864,228,780,241,473,201,573,677,849,758,710,126,439,631,475,118,577,39,530,63,813,978,886,133,353,299,293,607,897,476,164,879,26,347,85,442,0,929,376,392,925,13,438,150,507,742,513,151,798,81,984,197,578,445,393,945,658,896,841,904,145,64,179,211,351,665,704,643,147,560,761,639,63,46,981,298,463,24,374,567,135,903,831,485,662,711,686,194,809,979,301,87,500,673,719,660,177,498,469,718,124,987,138,322,780,663,664,743,968,663,954,654,704,367,905,883,106,915,953,990,589,641,289,479,58,248,592,267,239,469,390,769,769,288,825,550,674,352,246,237,218,989,897,991,500,95,840,886,142,781,976,486,666,308,622,48,3,77,479,881,779,409,707,462,886,806,261,216,166,295,257,883,402,301,204,17,157,369,162,905,457,115,70,891,563,905,767,235,226,639,381,188,961,886,335,911,289,963,240,611,152,593,178,32,634,253,281,97,47,611,407,634,340,63,29,349,318,832,704,456,573,806,507,732,917,670,249,376,813,824,378,998,74,956,578,981,403,983,466,553,34,190,458,92,631,366,908,703,703,200,925,814,81,500,873,549,935,792,134,336,207,865,314,866,842,843,868,328,551,462,702,206,28,133,859,697,447,391,504,835,729,374,689,86,114,366,214,492,561,733,599,981,248,728,190,334,504,504,934,991,780,934,667,972,291,522,73,215,807,837,464,684,285,961,808,771,249,36,607,861,465,43,7,523,404,840,948,978,169,306,873,304,96,81,463,104,168,549,498,847,406,888,39,969,338,277,883,530,723,394,100,152,610,132,517,428,588,770,650,88,29,491,306,611,448,616,456,636,282,691,470,947,116,813,384,209,396,434,618,29,271,240,71,108,577,300,231,372,853,8,416,831,120,564,412,829,734,44,602,655,842,664,125,754,944,56,96,235,226,121,439,410,663,456,175,112,934,977,717,844,341,158,589,275,75,89,540,976,677,184,188,256,460,728,480,964,408,215,178,309,631,977,208,24,69,821,125,715,924,968,347,829,505,581,129,290,177,852,440,388,140,440,738,354,6,42,882,706,2,393,348,128,706,721,620,912,518,839,938,650,689,853,33,468,436,568,53,787,648,427,897,319,506,728,826,242,131,246,51,454,305,982,383,356,155,699,394,707,198,468,546,114,673,798,858,34,244,486,70,696,455,258,444,451,27,715,33,397,698,108,734,15,565,345,512,134,637,947,271,238,265,249,42,294,9,90,827,388,567,490,559,481,280,422,938,370,803,732,140,763,300,659,927,766,524,497,970,823,109,350,967,490,224,378,85,913,295,282,981,320,588,968,40,513,699,788,886,359,529,39,585,202,978,163,504,61,226,578,945,926,732,140,572,487,679,398,233,346,540,415,618,523,254,373,721,907,91,507,861,511,796,539,117,793,576,926,74,963,622,302,716,295,530,270,221,76,372,567,848,138,872,5,771,732,593,910,501,947,637,524,239,636,408,205,27,383,252,431,37,311,628,433,28,624,943,119,722,968,166,106,954,656,393,655,939,946,691,602,971,845,846,92,118,137,967,995,589,542,973,165,982,877,68,398,497,908,149,204,408,652,61,700,713,13,984,330,557,445,123,398,620,594,54,730,948,902,831,429,831,196,325,159,984,287,142,58,84,155,342,998,836,737,476,962,612,921,402,377,807,543,806,731,328,993,100,733,549,413,930,108,889,323,902,322,774,838,539,131,408,653,114,324,60,815,758,60,815,550,857,638,602,464,704,614,302,821,76,357,587,919,855,60,577,706,460,79,429,724,940,634,344,599,613,582,802,80,327,465,913,128,373,998,866,428,360,814,152,691,426,79,285,308,9,567,609,901,549,136,546,330,242,842,964,560,875,397,367,289,459,91,826,230,638,217,515,903,142,543,821,731,844,64,417,410,59,641,842,378,977,19,401,144,327,932,17,255,464,770,88,487,121,140,2,312,945,54,204,738,861,466,467,803,419,402,361,902,300,48,734,656,295,158,876,959,404,788,339,158,113,175,175,564,297,721,666,356,130,393,118,229,484,744,574,484,734,169,668,323,484,709,356,938,33,746,509,71,479,310,626,854,129,328,398,311,244,90,436,712,347,857,370,751,509,51,119,7,37,115,459,426,214,880,506,711,498,82",
    "174,384,784,838,312,685,187,398,741,648,984,253,58,957,616,66,75,503,403,143,206,843,747,148,818,503,722,294,594,351,266,541,93,857,193,759,77,49,273,649,232,191,592,132,642,787,704,701,636,503,488,707,477,557,175,697,462,438,673,213,33,866,879,732,600,437,754,866,325,565,901,767,3,950,38,835,664,486,40,743,818,536,564,679,529,815,335,460,37,147,929,0,457,538,364,395,890,820,290,466,187,17,448,822,359,638,867,266,606,175,88,392,655,899,659,516,503,959,560,697,835,697,760,916,841,691,315,676,17,830,722,918,683,610,298,560,609,924,647,556,932,528,923,472,66,739,87,363,794,210,586,706,560,315,605,9,144,13,44,328,492,546,334,57,601,670,468,252,234,918,816,680,363,365,636,376,175,979,432,906,273,525,197,933,675,506,149,770,636,905,908,259,818,820,750,270,343,547,380,621,574,304,472,709,287,521,380,870,756,969,679,460,999,133,234,169,975,211,762,851,326,770,413,770,308,751,423,196,881,810,394,571,495,930,780,499,216,648,124,209,702,216,433,276,330,765,944,68,598,763,248,839,596,627,249,311,519,367,966,471,284,228,619,313,713,741,334,586,654,886,712,276,467,763,628,191,600,53,61,582,397,653,615,638,591,408,463,539,13,335,918,11,895,448,486,134,382,123,183,976,658,59,573,509,963,362,292,164,581,39,207,791,741,308,144,157,705,770,261,646,113,82,555,163,600,198,802,137,661,947,927,799,976,310,664,162,243,281,948,362,569,403,887,402,705,43,357,223,278,683,609,814,608,256,815,496,545,612,369,526,791,804,256,888,119,936,157,133,829,11,201,872,378,381,716,561,393,270,531,51,813,711,946,940,458,241,131,148,164,521,980,892,319,492,591,65,797,582,293,659,501,63,75,752,355,70,713,902,417,147,520,556,547,898,830,370,525,735,85,382,827,215,616,268,822,206,702,573,500,686,360,607,258,323,708,914,207,857,595,95,417,787,277,726,367,56,323,678,658,282,777,428,45,889,353,262,699,680,685,858,186,394,769,156,457,883,578,161,148,196,93,971,524,805,687,539,575,828,809,797,430,408,122,614,635,57,336,315,545,169,346,780,45,391,521,902,442,475,307,111,300,688,4,646,164,935,763,583,372,914,143,939,616,880,390,488,401,158,601,860,116,795,508,337,308,850,329,551,322,162,410,824,255,395,299,368,838,172,47,191,426,127,159,205,329,481,181,130,354,815,298,174,175,962,410,116,286,398,463,540,294,646,470,224,934,312,884,509,456,32,713,574,820,676,907,553,716,866,901,297,945,642,882,245,303,567,559,247,159,690,939,23,219,975,641,703,587,618,84,728,304,230,995,549,613,324,346,877,364,135,487,230,155,83,206,480,787,499,716,559,936,64,798,117,63,199,113,914,863,296,254,616,67,266,194,737,619,135,192,429,38,801,462,553,216,693,388,627,84,760,473,492,552,100,384,830,675,448,488,956,690,117,796,35,510,428,978,193,765,640,441,780,52,528,282,563,62,377,400,85,776,12,85,69,222,525,762,74,371,462,929,724,915,83,552,828,903,36,824,555,472,84,439,766,404,30,23,774,947,945,598,8,477,840,489,573,769,832,791,679,977,381,711,367,124,16,284,162,51,362,404,378,894,587,690,844,555,7,7,30,873,768,501,572,802,715,496,325,169,418,534,131,919,915,866,719,154,101,124,652,746,138,595,663,573,947,727,281,30,859,159,862,878,503,668,553,225,710,758,410,301,785,236,601,329,274,758,325,795,63,317,635,502,326,813,333,21,46,58,109,303,270,130,708,460,291,900,803,770,349,1000,477,48,754,527,336,74,217,637,126,879,414,939,366,476,204,824,200,969,248,720,439,46,771,647,426,56,954,126,716,311,308,306,592,640,437,294,194,288,67,983,299,409,199,448,713,59,302,594,546,139,326,515,693,323,211,175,254,169,859,352,767,299,222,334,177,758,356,703,656,450,194,882,935,410,504,686,305,766,688,716,639,225,311,919,800,256,655,400,573,986,457,534,943,734,111,624,311,947,223,496,549,622,506,778,112,825,615,67,620,736,478,197,972,848,66,211,923,578,754,322,776,708,159,399,359,890,313,848,19,177,501,565,23,589,654,42,67,490,125,639,264,474,141,455,57,98,869,349,544,352,435,78,489,606,768,135,106,7,808,701,877,100,849,780,744,546,272,410,124,68,45,626,370,546,560,598,426,132,29,414,239,623,849,755,47,929,800,131,205",
    "334,526,55,109,448,890,914,40,557,756,940,864,527,995,659,299,572,55,663,551,368,918,889,977,323,732,992,996,843,174,751,248,195,739,433,568,990,538,13,619,140,79,670,756,336,829,685,168,305,441,5,712,394,766,15,168,759,148,690,153,521,824,659,22,594,142,263,240,86,684,428,137,128,911,171,338,422,78,435,767,174,148,693,394,579,324,454,77,385,977,376,457,0,693,190,634,39,976,180,34,605,814,358,905,154,120,907,631,663,201,729,514,224,750,265,447,995,366,420,993,193,878,528,800,336,635,871,241,462,731,636,777,63,918,841,433,559,352,698,474,774,159,911,323,607,427,670,801,832,487,893,679,538,31,557,739,658,147,995,555,985,15,701,250,226,548,861,677,300,215,64,505,759,541,220,321,786,112,48,696,8,199,442,159,809,964,535,212,464,972,708,136,986,390,683,112,939,2,947,942,654,329,542,316,998,240,48,907,32,542,175,391,37,267,965,918,179,806,417,754,725,478,679,43,866,664,404,219,862,233,988,548,200,18,917,787,918,825,650,235,865,842,215,479,13,72,503,595,687,927,114,302,559,597,849,842,356,690,160,755,545,154,192,799,258,532,675,604,174,293,983,397,56,299,134,499,893,661,998,287,201,557,894,558,603,438,216,512,79,607,63,478,948,767,746,724,445,471,474,514,139,737,215,125,507,923,787,250,681,177,108,556,606,218,945,961,621,30,736,278,882,448,186,242,526,790,50,267,215,714,415,609,705,269,282,551,818,298,839,857,516,949,703,112,358,763,380,160,959,110,951,736,456,948,851,433,756,157,285,706,839,294,159,313,556,470,732,715,785,146,49,913,218,721,318,594,384,144,571,422,136,966,733,594,204,714,187,993,755,340,989,541,430,465,890,561,210,383,214,738,740,508,404,990,155,144,900,225,301,894,642,328,424,889,970,95,47,256,408,716,348,295,310,183,17,995,283,106,603,890,695,178,358,365,434,244,579,688,379,698,877,977,145,470,31,337,938,747,232,164,198,868,909,960,941,313,901,895,790,429,507,518,290,354,77,530,706,89,992,813,852,18,976,336,722,276,897,728,867,973,77,643,300,682,615,749,175,982,111,437,17,587,165,645,668,86,507,992,524,473,317,999,466,607,749,475,522,186,560,383,442,443,325,571,89,438,137,533,426,57,357,434,697,95,859,488,184,885,889,486,52,900,730,232,502,995,994,628,431,515,946,290,540,919,195,929,966,921,143,181,118,427,817,277,694,680,801,463,889,94,412,760,335,104,56,905,614,822,376,460,668,937,761,751,450,908,361,341,415,929,503,994,391,196,11,44,467,574,568,256,405,878,876,518,309,562,488,428,409,993,281,747,713,531,277,169,137,903,618,206,806,837,955,796,200,826,328,475,89,814,110,542,773,145,318,541,224,261,225,922,506,876,2,355,831,482,156,252,696,442,993,933,652,670,896,314,131,11,758,91,704,607,60,836,86,482,18,131,555,557,226,47,582,986,874,814,660,502,605,497,214,114,472,472,207,256,673,876,60,973,683,677,159,456,94,657,486,449,478,587,363,471,303,370,827,811,108,878,589,440,420,22,263,116,301,25,379,594,427,178,579,739,972,540,934,51,117,756,980,419,939,65,4,424,552,123,49,381,280,434,351,425,671,679,867,217,337,691,326,922,559,976,742,341,952,712,307,637,689,109,487,620,898,46,707,58,388,69,779,570,318,328,898,691,888,538,583,478,207,428,959,823,777,771,657,565,891,366,891,959,615,804,243,627,626,178,184,226,735,3,543,391,704,750,54,569,303,88,58,689,831,141,91,569,278,648,94,530,19,118,632,931,921,908,16,995,664,27,555,989,559,765,161,856,388,280,543,95,303,644,54,266,196,353,571,764,549,332,902,241,436,172,93,568,733,558,324,549,157,49,630,857,93,947,771,991,634,94,518,771,340,701,757,428,608,846,694,459,880,275,760,853,709,671,592,874,985,452,268,187,944,520,419,578,123,277,294,965,771,732,671,757,169,700,142,922,852,406,486,874,956,729,570,788,392,116,165,90,856,653,605,843,800,234,646,2,148,400,372,733,813,654,499,53,771,259,443,969,638,575,878,5,274,81,264,483,3,557,447,991,113,388,232,172,313,217,575,90,346,124,511,902,264,394,445,892,843,395,602,274,360,923,619,225,27,183,90,803,169,348,514,519,485,993,175,106,317,436,869,797,926,81,692,800,196,815,884,522,888,964,181,585,819,969,886,101,569,8,204,520",
    "77,980,673,41,312,581,381,15,931,91,302,245,966,608,986,484,571,934,268,906,152,462,747,282,600,866,108,953,917,547,672,29,681,527,187,457,905,922,282,981,518,536,790,582,912,321,599,826,23,352,641,178,828,803,782,431,33,434,673,805,341,905,451,89,966,337,437,494,393,374,540,432,717,677,438,595,425,892,322,422,54,572,134,518,951,563,850,449,208,649,392,538,693,0,197,186,781,479,786,953,220,763,356,880,847,695,518,473,803,202,120,692,461,701,957,39,311,31,597,366,127,331,880,761,183,238,600,919,682,254,11,112,965,294,586,541,94,627,988,992,868,195,574,996,175,949,867,287,76,976,773,269,507,986,420,508,756,869,275,499,562,344,62,353,124,285,440,304,455,573,674,52,64,692,373,340,766,198,764,574,525,785,648,688,453,551,265,665,739,832,915,715,281,278,804,472,822,45,176,431,717,314,276,3,984,173,449,741,890,368,324,901,458,345,462,919,513,575,496,827,526,413,382,378,971,392,16,79,194,246,214,256,217,89,965,124,57,717,215,63,225,261,307,487,953,414,715,206,667,526,803,366,212,539,177,607,297,944,29,78,641,895,193,917,825,824,264,902,292,839,379,624,872,949,487,253,838,411,938,722,109,242,613,63,313,739,526,341,236,851,428,534,562,124,537,369,650,313,165,152,873,684,27,984,367,695,233,562,185,307,540,938,945,32,201,482,948,67,352,896,395,667,809,125,22,572,794,711,71,678,291,469,939,262,125,147,773,527,263,649,743,2,576,125,173,830,241,912,948,95,208,652,86,227,552,685,299,216,732,678,789,219,973,379,123,457,40,488,387,307,222,430,253,503,68,605,141,862,311,701,740,163,982,325,911,257,198,537,416,12,815,193,921,736,273,173,551,76,129,844,793,291,811,70,123,16,630,503,474,698,851,19,76,96,977,222,461,181,249,291,379,153,423,360,265,762,424,596,862,710,109,404,858,410,666,486,670,75,529,167,556,349,562,150,274,907,365,612,555,666,936,134,282,955,504,570,26,679,200,845,668,512,580,270,958,32,399,124,383,986,412,959,930,220,669,510,237,456,764,363,173,180,280,248,271,385,271,336,948,454,842,372,332,45,163,265,706,941,706,526,462,522,7,363,802,791,615,477,352,641,816,120,397,55,846,374,426,747,935,822,431,658,75,236,122,846,844,669,29,813,627,320,387,55,350,487,721,938,427,706,147,932,775,34,418,12,963,97,277,905,435,337,775,956,760,700,358,230,169,547,368,276,845,576,560,311,978,688,965,337,806,385,415,615,994,867,526,635,698,446,197,743,566,382,789,533,653,817,638,805,413,236,66,73,531,322,337,967,304,130,413,400,836,248,462,818,491,514,266,323,916,445,355,665,686,464,28,643,110,641,727,640,8,409,572,797,791,995,329,233,119,531,908,836,34,784,343,108,928,864,954,802,138,636,436,823,834,333,812,568,616,41,329,419,371,439,47,804,336,88,929,101,288,218,914,434,274,192,445,761,108,107,856,856,498,243,444,463,84,798,704,757,565,771,859,578,286,138,784,590,917,822,498,919,61,844,944,729,934,894,731,433,956,814,228,117,656,114,514,528,573,226,195,378,86,746,565,83,432,801,879,344,492,18,223,163,995,315,124,498,25,433,710,736,477,500,558,628,981,263,797,962,76,788,944,531,842,316,431,531,324,345,750,25,500,425,360,889,479,495,76,912,205,660,514,558,123,506,968,455,869,512,950,793,691,243,518,142,932,748,100,161,845,708,885,826,338,683,932,31,563,697,269,726,527,693,706,662,15,808,927,295,970,777,865,339,352,548,567,151,288,507,443,996,748,843,54,591,326,79,275,614,328,754,128,854,136,782,726,18,184,590,44,998,793,275,669,235,17,308,782,237,300,219,917,962,546,566,550,862,110,443,115,365,219,698,181,710,103,166,385,221,834,218,101,411,824,69,120,951,870,591,834,131,863,62,676,16,560,474,957,106,727,200,462,77,149,711,975,591,825,258,959,152,244,32,908,703,447,566,310,577,568,248,415,860,656,865,402,686,407,591,860,624,435,316,724,191,675,674,12,176,743,413,725,665,565,760,80,209,583,236,693,213,756,434,449,884,298,932,677,406,770,151,433,432,416,634,40,629,570,13,600,300,58,985,879,109,629,320,174,444,284,176,168,540,386,387,66,660,573,61,440,438,835,974,724,444,875,760,881,861,28,110,922,631,822,424,655,381,806,65,212,148,295,936,239,533,29,91",
    "930,764,417,294,410,899,999,193,904,38,476,829,612,806,328,672,325,24,45,222,612,552,704,753,182,832,382,117,789,524,385,14,866,56,997,440,16,572,684,365,706,635,230,918,475,109,528,204,157,670,215,607,808,629,801,511,295,612,873,974,275,667,334,820,967,40,888,698,89,720,851,284,293,315,266,929,744,504,780,839,57,148,263,637,610,728,944,519,26,230,925,364,190,197,0,421,934,583,210,353,616,31,760,879,525,329,272,499,416,569,94,224,253,512,102,264,865,705,837,607,591,828,100,516,323,166,448,137,130,777,403,14,450,143,61,550,235,818,26,745,742,115,443,4,697,703,369,109,733,707,527,868,632,724,443,662,311,277,413,398,195,232,988,646,687,834,749,593,680,592,556,510,381,855,322,424,728,147,3,394,669,788,491,345,360,729,677,258,677,431,624,847,821,514,150,915,881,541,597,7,124,839,197,17,856,98,256,66,563,647,593,835,840,598,167,546,99,244,936,464,769,640,126,205,82,869,16,987,318,787,671,572,286,994,322,30,444,158,598,142,135,575,391,548,470,767,118,195,564,656,661,587,4,253,219,753,976,49,584,742,155,463,48,573,201,981,416,322,343,252,114,393,64,86,175,997,95,16,974,330,797,740,307,544,312,844,259,316,197,81,464,38,170,909,53,666,633,19,832,654,168,355,154,952,130,233,742,834,248,85,888,488,615,305,464,486,146,129,70,133,487,621,842,828,966,553,268,308,311,300,120,924,384,697,514,191,798,720,202,761,813,68,891,784,695,15,552,363,789,516,554,669,369,787,407,161,241,272,606,385,883,773,883,821,370,94,290,374,659,10,292,273,701,863,784,820,100,370,409,709,110,248,393,135,536,473,117,78,361,78,504,636,599,417,852,960,890,714,897,539,879,17,46,313,892,140,614,466,138,576,883,520,120,582,772,106,706,326,782,595,502,79,659,289,202,36,303,519,719,125,645,697,811,224,998,533,926,462,887,324,497,958,706,976,666,12,390,838,570,692,301,791,860,747,925,370,77,776,13,200,16,108,13,150,714,687,688,356,720,281,692,747,947,37,296,886,339,466,74,417,738,797,573,389,349,614,200,691,140,890,693,376,886,778,329,768,6,840,387,590,460,698,854,351,842,970,493,647,447,770,798,253,790,719,608,171,327,421,771,399,604,265,328,856,706,941,503,358,825,612,505,450,524,17,289,5,816,136,648,186,840,864,192,29,45,168,383,103,772,639,359,981,430,24,217,238,381,244,393,460,976,953,690,987,499,625,841,838,62,639,511,303,641,778,862,662,253,607,678,361,59,190,387,120,907,224,818,140,895,115,660,208,815,781,480,434,584,981,46,709,441,843,565,394,197,929,455,205,619,458,58,489,991,365,973,522,204,195,71,954,855,693,549,561,608,93,559,897,406,774,757,580,810,611,670,475,841,829,951,400,769,271,366,895,225,659,57,507,938,362,351,300,700,229,498,890,470,27,631,278,223,236,478,804,121,910,125,776,273,108,854,454,376,539,781,804,669,166,522,604,619,785,985,402,41,278,132,8,607,925,388,453,786,457,699,961,968,877,884,975,686,324,976,883,353,919,935,668,362,326,310,786,82,206,329,62,53,998,906,760,680,484,599,795,243,622,489,424,88,103,468,160,77,476,977,554,418,355,3,514,347,827,333,56,176,851,477,683,598,599,859,473,457,140,233,966,422,82,129,523,343,558,913,410,298,164,516,338,726,302,810,668,184,904,580,51,378,266,775,214,712,599,2,554,15,585,335,222,206,321,419,920,249,225,919,422,240,861,571,306,230,689,846,975,83,868,587,670,779,99,747,115,870,169,505,246,610,216,831,862,534,371,79,866,19,650,231,24,95,296,377,382,693,661,35,790,446,576,863,75,496,432,37,739,79,402,505,539,529,373,557,428,646,8,736,420,812,697,685,274,662,357,748,726,378,378,782,919,767,990,374,712,818,984,46,382,850,946,561,427,946,364,856,757,560,860,175,352,268,698,504,588,243,933,812,168,252,459,683,238,802,296,558,314,869,277,190,95,169,856,498,857,977,911,981,624,651,737,366,218,684,877,752,946,402,968,853,36,838,579,83,410,410,517,541,357,535,454,645,418,814,112,416,386,362,834,989,320,61,235,158,299,359,172,868,560,54,143,648,616,917,277,222,656,462,983,317,172,868,667,47,36,956,883,341,750,707,796,217,439,266,317,317,633,284,295,890,722,507,820,479,701,700,499,327,131,509,268,936,777",
    "536,929,665,160,924,792,749,759,155,732,19,293,88,928,671,515,666,774,149,144,452,310,317,58,193,657,987,664,174,774,367,794,349,326,673,516,751,198,878,151,934,232,549,980,987,713,963,450,310,428,755,46,285,444,999,321,625,319,632,866,682,952,493,376,381,161,956,524,160,519,310,274,742,894,354,545,926,527,494,718,223,317,198,471,703,771,211,11,633,187,13,395,634,186,421,0,508,44,737,836,718,50,584,397,145,27,249,965,637,668,699,624,631,614,983,135,428,352,184,668,91,656,370,845,911,475,459,487,885,690,564,841,910,522,432,671,626,663,193,536,317,421,321,859,879,628,310,944,157,712,315,84,839,487,916,237,606,569,248,995,90,905,151,160,847,483,978,20,973,291,34,517,652,671,255,726,6,372,829,872,638,387,978,837,414,164,746,120,430,474,640,893,57,234,427,571,969,389,178,290,756,641,976,532,230,306,547,501,696,299,341,526,40,213,268,538,732,95,922,923,123,5,280,354,789,553,144,42,657,121,69,851,952,316,521,276,55,681,736,982,24,137,797,114,279,628,752,426,530,142,798,546,192,196,201,12,645,236,637,363,653,683,973,182,418,178,249,265,81,236,539,146,647,341,263,229,48,777,321,260,469,238,322,221,105,31,846,294,146,329,303,201,634,556,89,295,994,274,864,526,773,894,15,615,33,173,755,13,639,288,437,617,459,642,234,434,889,944,456,653,757,440,725,438,753,496,952,440,414,975,207,459,217,588,565,826,60,457,603,646,827,993,901,620,368,893,345,316,668,946,698,481,520,43,531,311,666,156,454,971,126,899,533,34,600,729,306,931,175,373,156,58,148,518,120,120,284,718,538,570,917,250,357,603,738,812,198,735,759,838,740,877,140,366,88,965,643,174,175,30,824,894,61,163,676,839,37,989,643,779,728,881,3,101,961,90,336,185,807,851,536,121,529,55,834,413,967,168,920,220,626,877,333,368,871,827,77,277,59,960,147,443,255,76,980,468,185,779,893,643,434,904,178,378,68,70,755,48,510,948,425,12,901,18,351,522,796,397,510,823,36,207,223,971,397,896,817,68,244,123,603,966,996,736,329,371,758,616,606,829,525,781,227,49,695,245,206,856,909,661,686,105,146,798,204,762,84,409,951,773,901,394,345,281,746,835,287,245,252,88,360,636,769,498,253,19,392,5,187,744,493,217,278,169,186,946,346,557,748,722,620,940,422,69,47,258,400,732,644,483,92,999,513,623,228,975,876,785,119,218,360,436,433,14,343,615,909,125,166,527,75,261,877,366,890,718,794,130,780,841,644,469,572,207,467,808,619,786,53,823,250,184,19,187,490,759,695,364,874,848,81,338,245,692,147,993,533,17,375,857,699,720,793,258,668,916,893,991,197,161,926,50,653,23,46,155,525,717,12,769,128,842,43,61,548,330,521,809,346,754,29,21,57,809,554,956,95,484,374,980,545,937,61,932,755,164,994,464,464,908,497,748,747,816,509,760,391,230,598,578,550,879,261,525,121,710,766,210,280,748,192,413,55,64,565,795,985,808,180,939,298,923,667,564,462,751,356,562,991,561,119,866,633,135,69,4,600,59,212,780,1,84,300,409,641,458,517,564,337,290,774,571,981,666,242,618,323,241,329,126,705,853,792,618,553,93,879,290,627,474,831,402,849,649,107,161,794,739,981,928,573,501,617,347,897,334,23,910,539,634,580,340,898,85,815,206,868,691,586,866,388,103,472,412,482,997,958,665,749,618,425,669,405,753,832,275,844,542,769,668,588,734,773,326,156,876,371,561,7,762,826,870,184,169,381,310,98,689,794,101,483,29,473,291,370,108,411,567,691,938,903,831,194,638,986,286,43,487,561,649,129,677,944,328,535,214,410,837,973,892,514,114,804,576,84,832,583,445,324,75,824,855,381,31,229,674,511,93,259,358,916,528,422,360,26,131,197,148,577,174,971,710,474,537,965,142,283,910,316,797,96,526,224,370,83,612,284,426,447,509,57,123,159,149,486,839,232,759,720,794,963,698,845,163,665,503,941,763,957,987,969,157,194,583,363,958,357,756,817,917,256,827,739,397,288,437,165,194,598,310,917,328,338,831,743,893,201,687,429,54,260,850,666,540,965,311,979,635,851,274,510,250,898,507,632,189,814,395,236,580,53,357,958,253,947,513,321,278,425,375,105,367,443,634,495,993,27,135,926,883,22,770,867,307,801,503,523,830,527,548,42,951,637,142,547,309,22,649,430,692",
    "885,183,893,950,337,947,341,461,418,141,145,562,946,415,395,531,847,331,105,697,229,954,309,490,719,829,160,39,486,66,781,895,259,970,217,479,620,25,505,906,198,248,991,205,684,80,262,367,414,548,240,647,999,916,828,34,361,5,310,539,697,341,529,600,729,737,152,84,515,68,89,560,923,258,284,804,310,668,10,779,373,651,246,786,153,503,283,291,138,505,438,890,39,781,934,508,0,577,911,843,570,583,560,856,498,383,91,218,213,238,396,83,109,172,284,72,470,50,670,32,76,62,624,864,10,438,780,2,571,268,41,91,828,811,144,569,678,377,244,727,918,296,404,890,590,924,407,125,221,149,642,465,261,49,494,910,525,252,614,629,485,665,857,143,948,607,681,45,971,849,437,250,923,612,610,378,897,390,337,196,195,80,408,640,7,267,207,443,727,266,934,870,837,807,972,802,437,435,824,545,76,385,444,324,711,83,292,125,92,981,481,499,705,961,177,892,992,101,298,460,213,929,671,560,882,689,94,445,42,429,199,317,741,608,429,305,13,219,152,230,851,381,376,664,935,749,891,622,819,359,891,570,66,656,171,977,786,654,127,90,162,506,637,775,959,226,415,316,441,318,469,963,6,114,76,448,995,873,954,389,94,876,439,320,489,120,708,328,559,729,431,642,211,468,46,19,13,583,686,567,269,994,117,932,205,124,400,57,684,175,778,288,264,985,361,692,315,887,935,375,337,76,555,348,44,933,55,108,125,369,949,105,540,43,277,313,128,341,216,305,96,853,214,435,172,476,163,594,292,261,484,391,883,775,654,827,606,461,753,839,890,634,809,475,127,977,363,241,298,178,455,827,974,290,239,530,900,230,603,776,180,169,132,212,423,300,307,623,287,386,632,222,299,135,221,889,635,657,899,887,301,330,254,113,474,14,67,326,69,334,218,157,835,111,196,103,292,390,278,277,181,261,473,384,906,868,74,903,696,966,87,906,136,249,1,799,545,323,123,481,888,176,574,515,655,99,245,167,257,982,530,495,720,219,799,963,841,682,498,617,701,465,369,617,31,49,34,440,325,377,651,530,174,162,137,771,979,655,788,480,731,893,27,996,499,511,451,554,401,432,720,19,938,350,200,166,184,997,361,68,554,908,530,511,222,295,5,272,208,597,690,792,628,326,480,853,268,786,499,312,95,144,125,636,695,112,576,79,28,210,210,590,496,874,92,675,701,280,643,916,454,961,449,960,88,700,992,366,744,248,765,7,386,33,623,48,457,339,602,96,293,670,311,173,705,867,613,991,525,700,868,602,563,875,532,858,423,916,117,447,1,676,822,896,966,125,761,101,255,930,215,886,205,7,54,345,64,301,67,270,220,475,495,515,640,400,931,100,138,819,379,809,581,930,200,106,446,556,461,732,293,775,247,958,277,157,20,761,60,375,124,980,101,576,71,919,301,5,547,372,235,98,283,721,74,815,478,205,235,632,724,327,444,559,938,286,990,617,599,522,400,312,420,502,507,180,413,596,901,143,339,849,298,273,182,90,279,191,556,558,935,872,160,956,94,865,827,183,239,435,631,384,683,400,579,53,881,395,460,884,679,246,685,651,883,548,446,555,160,102,193,43,222,477,30,311,707,141,125,228,120,914,91,668,865,607,639,530,412,608,885,699,987,414,744,910,474,777,813,609,585,619,246,800,336,888,383,445,847,988,649,338,808,794,621,390,949,629,124,852,983,442,359,616,142,77,835,545,875,870,919,254,569,937,683,840,472,409,79,471,356,449,659,421,832,767,500,210,278,362,56,372,214,602,611,491,274,325,292,655,760,725,340,235,79,391,59,612,655,139,552,384,941,742,380,584,523,822,145,741,480,781,603,152,521,266,968,968,825,436,684,795,318,707,860,195,324,910,935,319,357,301,878,410,797,87,695,724,712,910,565,856,746,326,879,980,20,672,162,960,158,326,25,770,777,864,170,722,507,754,354,107,209,606,361,326,153,550,474,243,116,367,658,287,307,121,784,849,71,790,607,522,101,480,884,951,556,172,179,365,30,328,264,388,694,836,173,448,492,915,807,721,122,587,266,535,711,727,500,918,578,587,995,316,239,815,96,174,852,584,688,415,693,576,812,360,638,281,187,1000,785,516,427,599,458,851,14,644,97,989,940,883,612,641,737,938,430,170,162,813,970,903,772,444,351,965,407,524,344,581,177,306,38,109,533,86,965,452,326,943,914,654,961,859,436,917,299,117,883,969,457,31,801,910,621,778,316,313,52,275",
    "533,987,870,754,720,188,828,212,527,802,863,126,542,414,473,836,750,257,440,334,709,895,578,62,736,413,340,792,914,935,267,66,754,857,108,236,756,996,999,251,959,99,369,220,399,751,337,998,264,302,300,932,993,625,740,878,83,970,31,707,830,224,234,610,5,188,439,656,331,546,651,549,351,74,79,517,881,587,753,636,970,421,269,664,651,888,244,187,113,727,150,820,976,479,583,44,577,0,993,710,382,712,174,507,466,271,779,54,912,942,366,637,829,240,993,577,450,486,356,812,48,528,589,534,466,224,599,96,830,866,233,594,573,102,621,468,44,469,242,577,686,202,767,507,311,772,867,584,973,464,554,781,236,610,403,634,875,543,215,980,484,599,133,290,15,287,716,449,725,119,917,430,578,755,914,469,87,507,399,279,468,787,24,639,998,391,74,967,166,430,535,401,429,72,836,859,425,654,256,486,682,277,134,694,488,671,677,85,736,27,391,62,557,465,516,833,323,505,477,237,474,581,451,235,472,34,691,191,260,362,637,43,117,692,579,729,755,439,887,997,93,892,317,251,236,294,116,882,641,515,540,18,821,713,946,550,112,459,303,120,188,473,152,832,896,531,254,306,397,16,202,479,133,7,737,469,697,626,72,830,470,141,279,455,282,704,865,763,358,499,777,219,970,715,916,298,328,920,495,632,66,255,19,708,890,856,967,575,28,829,361,173,410,114,917,555,678,358,355,70,623,433,407,219,408,658,815,229,76,635,754,54,50,925,651,829,302,719,13,113,378,252,943,901,897,218,732,117,538,423,563,261,877,910,908,349,263,459,196,196,235,922,961,802,790,348,718,998,166,957,432,878,453,486,904,605,258,477,955,830,588,267,459,166,993,648,109,442,5,619,214,402,235,798,469,69,187,356,286,409,995,181,660,300,39,572,71,43,603,322,970,653,334,950,564,84,37,865,318,925,31,175,956,921,222,956,491,284,27,382,976,144,684,477,607,14,965,590,307,458,159,289,369,606,707,294,924,421,679,339,890,602,947,942,402,545,875,533,313,789,260,515,667,978,322,998,952,667,269,429,86,214,52,805,368,156,301,526,77,336,497,328,147,114,763,519,726,171,288,893,615,253,588,538,457,139,523,340,950,439,689,299,225,433,252,120,706,439,316,356,128,529,573,263,361,337,484,450,259,66,561,730,395,152,493,760,686,386,539,292,478,206,228,268,325,578,81,896,304,752,900,881,265,782,431,463,661,1,148,991,336,522,144,486,79,55,870,533,545,834,119,869,271,253,47,145,839,311,598,106,267,773,667,731,658,613,509,687,883,457,924,707,835,343,625,206,625,920,68,539,684,351,475,104,849,598,82,419,389,38,118,776,442,699,856,146,558,531,872,119,61,468,529,933,293,782,276,639,838,940,66,597,128,955,218,866,509,604,24,89,614,515,632,946,308,337,167,209,116,477,810,958,43,684,846,736,796,619,475,408,257,300,255,463,767,219,889,619,981,3,601,552,18,413,201,752,645,424,703,173,740,87,115,891,640,933,702,286,889,247,422,607,931,18,93,112,939,66,792,450,309,882,177,679,501,653,343,838,131,974,890,521,837,723,240,490,984,787,606,25,912,133,690,755,358,103,374,91,703,128,154,274,724,152,378,630,257,348,177,628,547,380,927,374,90,71,321,714,130,478,452,559,306,833,719,574,459,341,727,873,433,19,829,206,669,875,216,926,201,251,205,434,69,918,489,962,652,432,716,260,935,503,104,106,872,507,106,833,374,281,115,697,952,147,76,406,374,412,896,133,245,907,821,845,906,682,70,227,607,8,552,840,545,139,286,829,257,275,995,30,387,552,991,551,983,694,923,805,185,982,320,761,281,518,657,145,821,402,988,479,331,448,343,226,533,820,679,175,251,941,588,392,407,305,692,749,59,325,543,718,284,261,576,308,224,36,983,816,829,761,246,653,467,29,737,428,90,518,603,528,186,709,897,880,13,673,537,902,37,761,109,757,669,859,791,563,338,989,987,406,997,382,584,793,463,163,191,200,385,163,236,368,241,168,400,905,330,372,177,428,423,619,271,660,382,946,717,579,261,794,283,680,782,189,966,731,725,755,929,482,41,171,948,499,117,786,385,861,773,521,617,961,985,87,884,534,623,456,790,591,531,938,924,439,946,908,767,830,676,774,817,445,976,746,330,108,175,65,170,735,816,64,963,792,50,739,304,30,694,177,125,941,946,730,380,190,598,337,79,845,250,6,99,96,867,419,324,229,569,944",
    "202,806,4,224,907,981,857,870,881,664,675,657,398,584,566,420,26,348,737,527,470,890,799,174,367,245,478,786,698,563,425,34,100,851,396,520,142,101,515,184,106,996,500,484,958,677,700,372,472,619,833,91,305,52,342,703,653,559,326,391,840,66,837,91,209,374,258,669,154,941,102,908,79,42,230,119,325,817,960,138,450,194,4,574,264,338,609,46,817,302,507,290,180,786,210,737,911,993,0,701,224,422,111,754,310,509,908,865,478,422,924,147,32,972,500,125,921,4,316,624,452,246,758,650,335,626,62,874,638,383,431,294,921,770,42,847,888,671,803,278,369,860,71,564,66,486,475,521,128,244,885,968,408,640,151,289,701,538,279,240,321,162,299,885,463,531,15,526,177,68,716,360,732,973,439,983,834,170,723,867,393,854,214,457,716,798,612,652,850,77,788,637,491,319,384,116,18,676,629,595,593,250,522,718,163,286,571,473,56,620,585,971,354,289,512,12,305,195,335,453,364,596,245,457,302,82,460,431,934,528,877,306,748,623,659,539,666,59,824,763,777,938,878,120,328,665,24,810,2,561,272,881,872,428,72,337,142,618,862,985,496,423,978,608,939,409,487,23,795,355,276,362,723,15,850,255,542,56,437,189,726,627,923,914,974,286,87,31,426,927,741,806,211,124,533,95,598,443,149,129,114,739,246,224,305,406,194,2,172,652,393,464,64,437,338,413,289,377,139,197,743,171,882,945,427,231,988,564,813,799,465,899,273,158,747,986,216,819,694,645,889,929,824,886,142,84,387,280,598,964,265,234,963,557,202,815,83,165,598,292,294,433,61,844,661,773,793,264,255,160,858,950,534,128,965,120,298,963,102,797,518,144,654,60,805,673,335,545,448,46,471,51,466,230,741,3,322,812,569,611,929,63,157,263,550,522,542,545,149,598,108,827,186,557,347,322,860,98,609,528,365,178,30,322,574,652,795,753,384,925,985,888,215,388,223,794,838,220,585,502,916,363,590,14,836,551,863,722,695,710,383,255,600,217,833,532,449,82,507,658,237,396,134,148,31,433,109,947,251,629,558,521,94,219,331,157,516,269,165,288,473,811,235,363,567,585,359,401,814,596,168,523,898,725,350,410,293,289,837,864,583,383,265,448,368,65,861,76,967,247,219,668,355,371,987,504,871,813,302,584,317,635,358,211,641,303,986,315,789,34,517,855,963,746,511,353,805,838,518,453,790,63,556,667,414,193,533,143,942,825,595,618,763,545,945,968,829,391,145,298,173,545,896,159,124,595,914,294,624,924,681,81,686,842,943,88,552,822,329,786,750,141,759,974,925,89,871,611,113,437,184,841,357,210,653,193,439,392,920,357,681,199,821,528,917,393,206,151,329,812,115,315,999,476,186,710,397,651,660,659,274,859,590,463,808,439,372,736,966,763,365,410,199,988,955,321,860,682,599,825,65,136,671,519,165,485,621,847,639,302,432,338,909,922,232,919,677,157,976,450,266,81,841,881,674,138,231,952,498,498,230,129,782,988,900,212,429,884,24,844,595,824,499,282,131,288,932,862,57,972,297,955,760,240,802,635,381,35,919,772,846,543,482,231,463,167,710,394,937,630,754,764,58,421,395,519,431,630,486,116,342,138,149,676,423,382,892,330,803,10,572,343,107,532,422,749,790,832,94,116,77,241,817,795,975,778,937,518,676,265,335,515,749,302,950,978,983,743,375,416,338,209,565,287,877,943,179,463,488,799,649,489,520,61,31,180,784,725,333,676,611,399,414,514,859,327,420,232,26,58,306,766,186,984,43,905,495,514,249,821,749,645,278,801,995,840,243,592,92,293,122,674,651,697,837,823,355,835,424,594,767,996,562,284,79,75,706,816,98,123,355,79,527,758,413,73,129,660,860,982,820,994,914,950,163,334,207,824,740,508,190,465,962,700,465,829,229,453,709,691,553,793,648,611,611,321,426,850,315,413,665,356,11,756,676,796,958,275,327,207,614,541,746,66,663,297,953,276,891,363,646,832,394,819,109,527,248,261,549,788,786,136,194,748,185,19,804,550,25,342,650,726,149,209,4,67,69,653,518,117,606,44,174,676,867,982,149,602,200,276,799,910,429,315,659,380,11,278,53,140,457,556,342,442,60,565,311,922,115,115,737,937,647,804,45,581,840,349,93,653,642,276,732,19,388,535,116,60,557,909,681,330,381,748,662,252,814,272,654,778,246,821,773,129,772,576,768,701,739,284,32,968,217,452,519,293,490,520,559,185",
    "191,519,795,659,222,373,735,888,546,824,876,764,439,674,432,883,949,383,112,215,799,622,893,879,835,973,396,627,218,511,123,549,334,762,427,525,673,4,939,675,367,562,465,395,719,343,766,548,536,309,932,135,439,511,527,858,508,14,811,398,928,229,970,6,745,215,274,941,529,459,488,3,902,826,379,485,361,859,518,965,853,436,692,413,727,275,194,455,670,991,742,466,34,953,353,836,843,710,701,0,887,566,273,185,811,311,502,841,5,179,459,929,203,991,358,869,573,671,895,684,340,257,896,774,637,763,547,976,338,595,170,57,361,675,483,656,285,832,96,447,108,508,330,842,886,127,545,189,28,556,849,834,54,187,523,197,317,646,372,562,675,865,928,556,921,188,57,966,153,708,346,127,813,633,553,509,629,714,302,690,133,453,786,485,727,61,968,312,846,286,250,838,455,214,982,757,423,473,199,714,51,998,635,266,891,831,237,280,247,572,156,32,459,647,700,987,739,456,256,503,478,848,44,297,138,747,64,268,450,161,549,898,643,245,759,946,252,360,479,894,729,875,782,835,392,124,628,776,405,435,413,236,120,230,288,101,537,51,103,519,240,360,841,914,513,910,535,790,484,322,441,708,537,81,595,123,491,414,878,429,802,447,281,792,39,674,159,460,3,350,817,689,180,751,795,411,8,364,470,8,93,318,64,443,235,650,545,168,311,983,113,762,335,388,905,746,929,93,528,129,907,100,698,669,417,314,70,839,429,503,926,423,879,803,302,506,440,966,330,572,892,498,756,435,42,831,389,47,963,304,320,284,343,966,219,311,409,628,184,253,832,368,92,354,197,43,439,262,616,350,896,865,220,133,971,307,497,771,946,909,737,755,58,299,380,580,822,656,917,885,974,263,699,662,407,292,539,824,245,736,623,25,33,592,789,394,326,160,151,214,879,546,49,60,459,573,40,252,934,493,887,190,942,698,809,620,686,767,926,274,134,862,178,308,940,767,633,839,366,828,602,92,60,406,241,551,4,401,199,760,341,735,405,739,314,96,8,307,835,193,76,805,231,266,953,54,947,888,758,100,37,282,149,220,978,557,256,828,176,41,659,294,397,907,654,791,340,531,392,567,283,977,276,758,70,672,543,218,590,9,899,433,576,367,948,303,585,92,126,91,700,461,698,114,994,11,760,171,428,575,605,148,597,123,218,518,225,277,694,958,650,977,45,470,709,812,411,772,413,420,673,864,308,366,389,171,200,214,441,314,218,626,376,433,733,217,933,729,978,835,593,36,955,583,8,420,808,712,404,793,504,209,420,617,406,848,958,441,793,616,319,343,789,287,542,59,554,287,336,370,892,234,848,67,848,335,297,166,781,232,801,991,992,491,547,660,132,352,3,108,304,341,943,72,242,980,908,733,289,642,108,462,83,272,140,145,433,164,407,839,12,389,659,942,269,972,439,5,685,81,666,936,864,596,775,710,819,418,332,396,255,338,561,480,244,932,464,372,67,819,17,945,727,810,254,475,640,280,761,672,134,885,718,339,439,142,629,964,512,25,887,415,303,650,366,807,368,19,752,903,195,461,107,799,559,6,51,450,377,104,395,858,465,633,239,33,369,274,235,58,676,375,798,370,768,654,562,633,29,742,475,311,386,580,943,711,665,363,103,560,104,149,495,591,951,499,982,370,883,914,602,644,810,954,80,905,65,480,197,826,579,181,654,611,485,621,308,352,862,272,258,864,55,977,274,128,112,943,59,246,142,312,264,21,876,618,63,102,799,976,206,189,663,673,624,973,939,926,251,869,759,493,796,248,577,90,508,60,909,453,462,212,927,385,801,66,701,515,767,361,34,232,568,35,615,284,654,94,979,872,93,556,45,939,489,163,708,375,649,941,74,943,491,258,14,171,976,398,736,872,238,25,3,569,74,429,34,262,45,552,403,453,386,383,819,95,863,336,250,599,957,513,41,486,633,73,877,949,53,926,160,137,138,698,568,608,697,557,712,44,933,303,618,366,765,553,561,589,798,569,326,923,607,53,410,684,526,20,221,908,282,893,412,779,187,38,605,657,945,403,286,760,766,536,623,708,1,733,644,222,575,912,149,837,990,251,486,167,135,301,105,782,267,898,446,323,526,790,92,52,521,294,33,785,116,120,380,611,798,51,170,168,617,632,180,607,57,45,978,593,482,774,200,605,195,326,484,349,372,962,766,61,464,432,55,230,570,109,701,900,967,283,941,943,125,860,837,263,521,993,607,435,740,84,949,421,533,160,3,906",
    "825,293,396,377,319,819,438,208,216,107,400,376,223,887,162,621,367,315,30,677,404,179,894,995,176,781,849,720,315,532,737,523,815,200,46,221,532,709,392,609,958,90,173,473,73,149,965,419,140,10,391,844,222,181,334,635,76,338,128,151,603,902,913,163,93,817,845,532,216,408,979,471,417,740,366,480,149,969,587,983,38,892,966,591,585,133,655,742,650,432,513,187,605,220,616,718,570,382,224,887,0,245,334,398,718,57,632,534,465,827,989,513,226,895,73,893,204,347,983,826,904,208,302,533,745,66,746,527,775,389,528,165,771,138,303,675,645,250,78,221,562,589,736,308,638,481,326,481,880,198,630,994,112,894,678,224,487,911,104,18,159,890,945,498,76,76,432,317,151,962,87,964,171,232,860,800,687,972,844,721,31,581,471,42,103,836,728,755,537,921,975,1,52,30,338,672,437,995,724,848,734,387,209,460,214,640,339,985,673,378,311,316,929,981,2,438,978,851,732,285,609,365,287,889,807,132,114,178,623,246,447,572,867,78,26,866,946,383,751,998,419,752,911,244,668,105,659,371,690,186,453,782,406,552,160,174,551,205,710,192,61,974,565,579,551,166,550,957,904,3,424,237,135,93,823,425,535,409,789,443,887,306,424,402,376,348,866,827,774,930,84,854,506,90,435,148,309,695,663,571,266,867,81,388,801,131,32,577,442,960,606,521,723,958,416,706,361,768,100,591,206,113,319,578,811,507,663,818,16,714,393,571,645,764,498,392,919,535,697,672,163,242,946,657,526,871,838,184,636,898,218,81,446,956,897,443,160,711,280,197,526,834,166,748,903,628,94,361,260,34,342,113,172,238,422,87,755,300,328,843,566,947,407,928,551,827,849,596,430,590,439,718,449,771,482,724,677,391,453,2,835,857,319,995,193,554,829,95,165,296,968,102,192,521,421,666,229,228,290,654,768,604,148,982,723,140,108,297,541,465,45,382,147,992,327,155,637,485,743,64,814,895,925,150,840,165,836,206,404,31,743,289,768,667,946,473,317,687,247,49,588,366,882,347,185,155,886,366,930,417,631,424,185,240,886,449,704,957,218,844,204,63,765,398,548,892,111,161,935,629,740,806,125,136,513,878,758,409,17,622,829,348,792,247,584,281,235,595,117,615,107,553,315,193,920,363,232,815,463,493,954,485,775,102,575,884,158,938,666,820,308,697,380,193,842,384,286,37,753,449,443,718,312,948,377,172,155,139,810,87,439,527,760,562,739,22,303,444,635,622,560,885,120,648,547,342,413,164,678,877,644,598,658,668,29,673,418,820,952,400,434,272,430,98,649,607,875,218,69,855,352,212,669,411,339,170,237,690,454,320,939,281,954,854,456,200,506,139,696,964,841,294,324,328,392,939,474,521,394,448,32,883,783,943,103,863,186,788,94,392,172,265,556,98,73,901,860,607,745,23,284,573,13,36,551,252,707,881,947,7,591,988,98,280,235,757,886,975,825,137,273,311,900,245,519,704,443,162,377,157,336,484,831,558,429,38,572,217,99,52,848,893,112,190,57,597,718,97,26,216,428,147,900,468,108,283,727,250,854,138,949,301,309,833,863,251,58,522,616,673,53,891,14,245,814,839,63,338,335,910,890,862,112,357,622,106,565,225,291,888,593,905,290,739,848,455,635,140,971,235,7,382,87,504,815,643,141,986,428,198,979,351,607,307,687,798,463,988,273,236,383,365,356,18,778,252,886,493,470,914,65,498,928,177,67,875,625,767,222,122,739,505,650,86,706,527,715,52,531,36,449,9,350,822,923,834,609,490,358,924,514,231,446,594,998,388,550,495,844,347,2,667,631,379,901,974,747,721,873,809,427,693,639,379,196,160,430,378,770,624,285,576,850,270,113,497,91,881,394,542,456,544,28,719,84,669,600,300,50,263,10,567,966,855,640,692,95,867,791,206,339,575,408,235,580,808,13,277,1,314,612,178,658,873,659,588,554,492,467,292,763,719,567,115,955,853,99,21,927,259,642,749,781,996,91,548,276,662,668,375,625,298,500,886,608,964,97,772,462,478,39,557,47,533,584,780,330,704,264,24,492,516,884,450,787,437,979,531,957,894,794,887,616,498,52,309,865,275,25,905,812,911,740,917,180,106,275,222,110,399,538,276,675,321,741,516,697,88,970,395,19,582,421,699,384,989,448,55,889,517,616,607,595,466,600,363,439,985,231,444,820,415,821,929,139,158,824,873,407,613,741,630,60,494,246,543,941,914,860,451",
    "342,758,418,26,153,114,782,809,551,364,435,26,254,262,751,429,617,442,34,180,973,331,585,452,517,739,486,259,589,816,548,545,969,813,8,683,525,793,996,74,414,971,35,318,115,465,673,242,26,575,724,952,138,689,931,320,220,577,758,192,740,333,312,365,86,33,682,383,379,419,680,55,755,211,136,953,379,191,324,741,515,602,259,524,487,422,887,425,808,986,151,17,814,763,31,50,583,712,422,566,245,0,628,271,783,867,605,919,650,657,864,802,711,26,893,945,559,916,630,417,38,274,383,66,553,423,42,539,797,358,469,925,916,277,836,211,18,594,763,712,610,146,809,388,316,668,204,839,915,125,365,38,128,978,923,192,812,787,564,543,994,745,706,643,946,682,417,757,437,659,301,8,996,602,776,655,536,414,329,18,69,215,876,64,284,75,325,177,848,563,657,642,476,742,187,454,324,547,687,589,990,996,501,490,591,676,294,143,53,162,838,967,837,33,855,650,83,648,760,457,944,564,539,175,116,243,419,756,706,17,859,706,473,764,112,580,456,131,150,616,884,613,998,580,111,589,348,866,768,476,792,166,352,949,886,49,947,991,404,49,460,494,973,982,79,332,612,523,812,215,324,937,393,466,293,958,235,842,207,572,28,227,784,2,640,13,468,66,942,786,885,933,682,506,868,126,371,739,578,439,947,828,887,802,80,733,754,20,590,790,177,484,64,643,923,401,741,220,806,558,23,589,358,227,788,150,292,887,572,665,964,577,255,311,603,413,953,644,509,355,548,203,136,289,457,680,545,652,581,346,428,210,846,878,940,457,76,866,820,84,731,696,259,871,602,155,498,981,578,524,195,676,15,912,295,970,826,409,936,152,975,285,808,477,466,93,157,634,619,44,87,421,15,916,746,383,604,414,301,333,394,354,346,311,404,268,564,638,751,141,226,672,688,790,298,426,720,861,594,698,134,250,550,157,735,850,309,807,805,75,916,999,804,907,184,602,340,331,388,40,29,587,325,121,501,431,117,927,426,777,494,746,913,596,190,749,267,658,547,483,696,912,477,821,982,280,578,857,321,239,503,885,651,336,453,847,877,49,816,333,653,111,887,985,874,368,51,344,448,978,490,970,296,260,568,733,88,719,428,614,1000,746,873,878,285,270,408,51,221,111,27,11,657,171,906,940,104,882,302,743,622,255,556,215,108,750,963,358,117,909,515,669,210,796,167,390,792,394,316,568,785,803,6,307,248,474,272,942,322,134,914,204,159,752,903,16,177,842,991,249,436,888,614,175,126,675,430,970,265,895,152,519,802,373,703,484,585,465,171,674,350,798,892,846,241,663,963,241,549,279,715,886,474,670,471,680,795,309,996,518,643,498,73,287,709,935,308,780,663,727,426,248,294,100,193,316,735,354,573,465,334,902,44,836,238,255,576,306,430,306,499,565,531,545,134,417,71,474,23,264,191,345,439,212,762,47,753,530,619,94,767,367,82,332,453,339,148,686,544,969,647,499,907,382,642,756,16,63,957,371,685,988,202,990,961,815,798,407,489,646,601,154,667,757,499,697,70,208,731,749,297,981,999,597,642,409,203,995,270,988,840,210,953,199,620,710,430,755,554,530,478,980,970,871,747,522,494,302,190,49,126,12,9,351,670,755,205,309,428,248,712,168,789,713,11,928,506,878,108,288,786,241,504,897,453,304,944,15,829,513,979,845,478,102,196,966,634,725,433,400,651,175,225,696,795,437,40,30,246,811,602,214,831,482,35,605,889,701,535,283,683,668,388,252,796,651,712,744,733,829,15,777,725,206,217,646,873,678,372,289,64,573,763,667,296,616,334,141,431,484,166,292,417,964,460,353,262,59,668,500,258,976,30,209,189,62,743,912,551,966,543,398,133,866,710,444,691,767,358,3,923,790,242,987,571,839,801,11,684,229,291,878,577,396,774,976,66,29,368,323,571,208,484,514,579,70,102,336,515,15,938,699,944,464,305,999,314,327,114,676,373,429,158,159,554,691,680,731,422,214,245,874,346,223,770,636,955,419,300,362,695,312,943,168,487,576,796,730,967,440,528,356,840,233,285,282,532,586,627,385,838,5,998,71,276,656,403,248,91,257,503,695,495,765,777,537,197,487,829,608,104,494,391,947,343,74,251,568,24,285,693,498,77,415,681,189,494,13,850,5,548,292,302,359,792,613,229,110,327,350,651,374,393,520,251,775,86,56,635,273,188,257,952,308,265,876,33,793,625,639,40,321,629,703,280,934,70,448,360,254",
    "578,992,483,517,584,983,840,105,185,957,56,326,44,776,708,297,394,239,631,299,124,414,961,956,84,187,533,522,757,238,957,32,537,694,926,254,582,475,894,208,627,393,596,27,525,147,34,148,562,299,911,158,98,100,775,913,603,801,878,917,311,837,195,546,155,532,987,606,314,979,364,162,571,968,133,299,323,795,852,979,587,229,992,910,127,436,828,387,663,946,798,448,358,356,760,584,560,174,111,273,334,628,0,656,370,912,110,602,245,666,524,459,77,584,709,706,490,438,358,742,16,448,74,761,258,206,455,809,448,598,238,507,77,449,669,782,821,656,921,4,755,428,39,522,764,223,78,822,880,237,124,668,38,97,296,968,227,650,204,664,387,897,858,141,979,528,205,685,112,117,119,669,321,97,803,322,38,852,70,416,308,79,750,280,293,782,74,622,147,425,495,133,488,894,325,209,785,185,803,331,781,535,336,19,4,449,567,711,913,370,210,913,32,811,756,141,83,346,158,342,294,914,565,822,877,226,230,511,876,716,641,444,375,603,418,419,333,368,520,930,898,127,586,345,178,27,912,561,297,686,572,31,23,872,887,906,56,463,813,145,978,815,41,268,966,660,821,362,305,659,780,887,922,346,799,929,136,919,759,902,211,947,651,190,797,416,58,111,315,490,138,531,849,788,705,409,677,894,315,815,919,68,485,6,981,22,394,963,835,896,738,36,247,387,784,684,819,541,9,779,255,287,903,913,19,23,39,537,676,160,801,774,362,780,856,29,513,759,11,661,890,696,23,630,688,438,419,347,143,429,445,19,915,28,83,417,657,636,518,829,226,592,397,588,240,442,941,76,508,316,739,501,743,697,451,13,146,837,430,724,796,243,134,776,387,74,419,180,204,526,941,522,931,351,100,404,597,126,738,708,549,880,421,402,515,30,261,279,407,876,740,626,333,805,771,776,184,409,638,463,729,167,469,154,245,622,228,561,519,766,972,71,392,867,713,925,83,44,922,993,365,735,91,446,408,2,711,860,923,38,803,215,126,387,369,516,696,26,728,868,728,550,173,589,816,938,448,832,767,606,641,415,679,449,9,204,579,600,886,56,126,816,416,549,187,526,743,564,370,99,439,724,954,746,593,170,400,441,915,354,603,624,885,63,459,445,57,96,615,580,743,487,970,528,690,729,158,844,772,415,314,391,987,477,240,812,882,611,450,659,936,290,512,32,794,100,547,268,309,744,693,606,226,694,60,699,117,229,648,538,260,519,622,316,265,431,752,486,115,650,889,929,7,136,606,137,37,163,612,524,309,756,457,805,108,877,478,379,920,187,320,909,812,210,794,849,814,911,869,949,645,766,538,880,694,741,272,530,414,798,288,965,987,124,912,555,237,310,880,228,711,544,740,245,498,668,70,985,872,100,749,681,780,653,670,734,303,354,993,937,135,853,12,23,161,326,754,790,554,465,775,832,692,899,733,80,459,70,438,962,26,106,291,75,202,315,379,397,703,375,806,124,586,365,312,256,853,378,300,982,599,507,205,441,851,270,263,313,749,70,837,297,539,208,363,714,115,974,224,864,848,863,980,941,668,514,524,836,134,656,785,584,867,691,468,353,924,697,877,990,821,465,595,907,955,131,984,26,382,418,352,550,146,296,953,354,947,894,994,626,595,41,164,240,297,612,488,804,326,119,461,457,789,500,302,321,727,528,436,725,906,130,485,556,629,680,630,143,578,782,148,800,430,328,989,514,596,90,371,635,10,502,795,508,90,385,520,569,802,832,887,516,124,645,151,383,528,887,854,289,866,302,322,557,73,619,313,315,17,197,522,343,760,261,303,292,64,621,461,792,872,786,67,6,302,491,951,77,321,94,384,361,141,214,99,773,104,887,801,699,115,636,304,171,759,451,568,255,214,743,429,816,719,421,680,693,64,376,901,804,328,225,910,213,133,989,242,76,648,153,233,86,122,789,844,46,447,40,360,185,881,437,773,454,304,549,862,756,359,242,359,112,615,495,361,39,810,139,25,616,643,459,546,813,151,394,119,210,4,285,920,643,372,703,359,321,852,124,427,200,38,91,470,798,377,508,691,456,599,388,478,392,219,250,694,846,407,43,67,587,860,156,155,173,710,783,191,789,428,598,683,130,836,872,444,737,931,956,573,510,756,452,683,866,673,621,473,911,404,141,238,979,379,434,240,902,732,855,278,326,479,188,613,994,519,807,636,680,414,211,914,45,255,335,361,669,572,287,403,528,449,87,625,81,78,230,24,983,715,461",
    "76,261,566,74,855,410,517,879,918,216,906,613,520,668,574,175,710,828,609,982,446,32,592,528,236,50,802,994,961,297,24,3,934,950,627,436,944,259,603,997,484,687,698,914,35,230,159,83,135,458,519,516,421,703,504,668,451,775,279,7,26,99,637,569,492,37,559,291,6,660,780,499,303,448,393,416,86,525,670,911,435,654,408,156,223,385,551,426,434,397,81,822,905,880,879,397,856,507,754,185,398,271,656,0,247,880,501,218,154,799,619,900,78,579,533,480,248,184,733,219,739,888,61,31,824,327,195,171,986,36,744,724,58,575,292,922,614,899,805,157,15,234,864,356,640,999,413,858,248,423,9,705,743,252,393,433,372,292,480,390,984,286,352,323,385,303,911,124,501,232,360,195,536,464,6,220,501,931,551,667,280,508,696,324,775,892,46,645,576,560,765,756,732,905,397,290,751,816,885,887,312,107,377,903,624,31,480,928,209,470,366,342,723,655,838,594,733,516,763,554,528,671,992,91,290,23,34,826,701,804,912,353,999,905,117,644,524,664,977,451,362,918,583,336,486,159,276,907,721,564,313,810,876,129,757,270,743,46,8,732,876,220,820,796,70,960,204,720,394,904,389,573,645,180,201,609,725,316,836,317,605,615,235,562,482,212,556,494,162,163,614,998,235,393,952,836,427,62,193,436,827,220,486,548,534,205,517,288,598,466,522,894,522,731,510,855,693,405,901,954,625,2,751,911,204,493,606,178,134,645,799,44,848,5,498,575,664,157,554,875,765,144,633,611,251,2,503,431,506,263,210,81,43,214,364,183,51,473,505,478,260,395,926,75,458,67,267,410,167,692,242,94,464,109,596,762,982,683,903,75,810,471,879,291,487,767,252,950,856,730,679,184,810,399,49,76,539,701,31,585,426,461,187,438,100,627,280,361,970,989,875,119,835,733,65,52,980,693,793,415,654,151,390,842,381,739,993,751,624,705,23,154,888,243,800,866,406,410,878,100,896,589,558,376,340,276,627,687,573,903,199,569,398,165,192,550,941,452,974,324,517,663,978,120,939,907,455,493,97,515,801,236,474,5,773,985,57,726,934,490,460,994,568,735,580,607,835,752,338,303,961,658,726,665,564,807,875,23,269,181,275,692,34,240,76,459,158,951,461,495,781,380,585,879,62,507,988,28,996,2,167,346,575,469,688,756,825,150,158,398,996,228,263,94,341,904,252,884,251,885,64,501,937,89,553,591,87,247,821,260,29,308,198,961,232,756,992,636,428,626,358,511,551,876,846,661,654,902,903,396,948,13,143,290,395,141,44,23,740,769,632,381,634,789,917,709,390,477,917,134,471,196,948,475,290,117,216,58,195,483,51,886,642,108,869,308,777,338,865,20,928,513,178,112,910,169,579,434,703,412,27,480,51,157,900,475,711,934,957,417,622,181,515,288,149,300,599,195,251,501,41,728,918,866,159,732,411,759,236,347,199,678,824,138,296,529,691,900,20,934,269,903,622,670,98,720,9,64,725,223,830,350,531,774,122,988,395,369,895,21,505,712,737,154,552,16,190,834,196,842,41,823,262,29,607,371,908,590,899,990,518,118,45,847,85,668,686,692,272,224,912,120,984,341,258,190,94,225,844,330,986,836,548,33,681,595,723,798,975,50,326,750,494,667,70,106,712,652,600,966,5,719,895,983,164,662,894,158,602,202,140,648,989,761,319,343,178,26,659,521,824,182,65,279,53,681,873,471,171,236,369,762,376,436,494,131,611,257,234,27,505,137,592,814,322,579,675,270,656,206,340,47,111,849,226,571,305,474,512,655,142,643,807,959,225,196,633,600,443,729,26,46,938,688,288,212,356,147,396,268,281,271,501,486,828,437,864,457,485,244,304,569,914,404,355,991,183,888,836,531,662,923,759,746,254,676,324,963,643,784,695,359,113,501,77,209,167,371,986,409,871,781,411,2,175,339,130,517,84,963,703,157,180,491,549,409,94,580,812,132,671,395,988,871,626,55,988,495,414,437,841,445,839,575,644,512,548,285,435,749,809,220,744,268,131,32,849,733,655,280,387,33,794,42,998,403,431,707,258,379,502,881,331,708,961,146,82,708,275,700,561,796,538,264,208,64,452,704,845,950,259,7,286,892,837,2,237,304,625,868,560,370,589,650,389,648,825,314,844,601,34,325,501,837,812,973,783,912,635,821,736,138,426,516,241,138,988,111,46,361,386,621,155,633,566,229,484,735,145,515,117,731,963,632,471,778,996,454,124,770",
    "394,587,132,503,512,313,151,722,485,205,480,405,883,872,481,171,982,341,949,127,955,683,173,23,831,40,608,337,663,570,705,669,528,52,366,348,376,993,942,359,431,743,308,783,488,414,145,378,169,834,703,993,19,894,897,439,277,119,608,480,8,49,18,448,143,221,160,487,458,413,938,440,878,444,253,368,927,180,353,725,154,510,547,244,634,586,279,912,194,83,984,359,154,847,525,145,498,466,310,811,718,783,370,247,0,358,576,342,814,284,213,550,526,175,346,721,353,374,409,547,745,739,311,236,698,452,747,342,375,839,784,641,273,962,571,227,806,289,310,843,518,43,846,195,536,71,787,324,116,796,371,997,633,600,17,144,202,388,28,792,313,324,883,38,389,882,215,398,309,755,575,454,811,739,893,866,504,658,804,839,625,98,62,216,899,343,972,384,791,929,723,87,261,489,20,452,867,368,209,78,966,307,561,325,189,441,37,995,445,582,856,476,82,111,837,890,802,74,740,635,684,951,416,662,993,334,74,840,902,507,462,579,396,107,302,259,646,614,657,444,131,473,253,914,902,710,224,575,198,150,103,802,872,843,73,124,647,83,404,813,948,235,151,455,607,509,693,532,263,348,696,877,339,757,861,276,639,966,55,829,326,381,10,320,638,735,710,300,192,129,774,32,929,46,882,379,916,232,115,681,339,795,833,604,641,291,18,715,40,86,664,180,791,361,933,700,478,631,399,935,339,300,326,394,6,418,316,655,660,757,13,321,450,258,828,748,68,494,473,682,110,344,469,700,375,609,10,669,686,956,849,939,765,202,171,68,856,776,904,717,114,482,473,43,351,411,640,775,993,250,42,655,636,628,691,928,904,895,903,584,255,766,131,671,899,776,84,450,183,951,980,514,240,42,10,958,978,507,905,319,280,20,189,838,63,523,594,44,923,30,277,363,787,751,903,39,946,638,262,83,218,243,624,466,56,308,678,885,593,96,455,52,146,342,700,617,287,300,930,890,490,847,236,352,150,368,558,74,592,132,945,326,41,642,206,19,864,932,987,321,558,315,143,558,724,267,567,70,31,829,710,144,341,198,954,947,471,231,232,956,740,514,772,962,297,226,637,828,877,475,790,871,697,850,912,445,522,994,164,746,389,593,185,660,9,568,268,934,67,207,468,895,36,141,502,796,287,515,986,538,995,127,754,110,561,793,624,304,47,893,725,755,9,254,925,585,113,480,486,550,629,587,516,963,113,900,73,638,303,802,148,945,599,770,681,809,808,139,650,454,984,476,214,353,8,689,515,436,991,386,716,483,783,508,627,621,734,949,651,202,337,299,83,816,881,440,339,839,884,384,85,712,695,249,975,867,889,346,550,652,517,141,850,228,446,407,936,966,4,363,97,39,810,780,763,996,371,281,757,248,813,67,835,216,572,801,288,148,201,910,397,38,926,555,831,69,183,394,17,817,933,654,239,768,748,559,701,622,4,583,753,449,685,480,854,608,47,396,115,569,933,336,646,604,465,1,659,266,788,866,114,278,552,96,251,49,72,250,266,115,819,377,477,985,426,796,643,457,171,481,778,678,220,631,845,48,469,908,388,943,858,770,673,983,27,373,761,281,708,413,457,408,43,57,718,668,906,829,913,773,647,625,196,276,83,119,492,267,867,339,82,943,681,766,174,126,205,480,572,623,942,495,954,249,143,316,933,959,109,59,684,687,760,950,80,25,468,614,156,954,31,434,722,129,465,654,571,489,470,266,197,86,686,233,903,793,146,863,903,633,298,901,465,693,534,948,396,215,789,209,840,337,287,884,319,981,827,711,251,252,866,839,661,994,983,808,578,555,615,538,934,684,129,294,490,545,96,567,40,890,950,897,669,7,430,476,734,749,674,124,722,663,583,873,757,479,50,889,901,112,761,339,497,7,551,825,345,155,326,933,535,693,936,13,653,208,244,540,319,202,670,455,843,646,68,930,661,485,202,741,599,586,293,569,232,311,266,579,363,885,628,118,858,245,512,812,354,496,707,915,164,504,156,507,823,780,768,221,148,646,905,699,147,337,564,186,299,409,959,781,164,722,135,226,123,19,898,420,659,468,4,953,808,699,422,83,109,922,14,847,216,838,571,550,186,576,146,717,876,712,533,767,369,598,524,977,140,313,104,144,77,749,745,479,333,876,468,665,846,345,985,219,613,388,683,61,240,407,940,816,991,494,722,481,989,544,395,451,787,673,939,433,53,546,454,693,733,236,219,684,964,216,17,164,251,934,798,947,93,778",
    "432,366,985,784,917,752,305,693,259,78,543,70,533,272,253,239,566,228,101,552,614,795,370,778,126,589,827,159,875,455,750,157,28,199,462,60,463,442,959,620,462,951,651,912,511,359,681,378,18,823,768,183,816,812,339,904,358,537,676,743,40,565,156,195,117,738,779,676,323,225,197,221,680,726,757,411,564,886,690,443,407,833,162,18,205,533,622,666,448,839,197,638,120,695,329,27,383,271,509,311,57,867,912,880,358,0,625,924,453,69,537,112,830,548,136,943,863,861,137,928,831,670,682,316,251,846,960,979,189,62,819,128,884,245,504,513,567,650,247,472,928,457,308,738,210,954,374,449,792,616,253,658,538,550,129,998,659,648,9,547,537,931,590,296,732,4,536,922,37,919,791,348,481,449,16,695,73,257,94,72,439,376,506,370,277,43,789,913,658,856,853,700,897,808,513,110,618,258,315,294,563,4,441,854,726,47,802,509,856,965,786,13,195,166,245,646,453,335,646,966,884,335,427,513,45,679,937,859,178,466,270,378,262,784,780,766,460,550,107,480,211,303,712,753,830,776,174,619,192,713,678,976,617,488,200,500,312,296,133,673,215,21,795,533,814,260,776,483,395,16,888,787,619,905,345,685,380,645,304,868,206,211,47,608,995,198,270,545,390,817,113,288,921,586,252,511,737,715,354,388,407,377,701,271,214,379,580,703,676,685,534,632,39,630,715,103,113,281,155,439,222,302,228,199,967,677,328,621,931,204,445,729,148,692,311,978,938,660,232,964,795,855,465,944,410,889,989,216,779,363,463,72,427,211,871,483,62,645,163,733,639,115,2,168,764,656,820,915,608,306,195,205,352,738,275,218,24,257,181,57,964,346,768,36,957,279,406,857,57,517,549,187,991,394,76,444,921,34,402,805,729,832,132,993,401,322,279,376,166,589,431,545,794,248,529,101,619,416,107,679,330,112,586,923,117,821,351,497,383,52,388,547,287,689,785,155,815,365,345,760,525,37,662,629,782,16,43,54,589,707,468,223,16,881,539,203,873,954,437,658,778,606,772,134,998,964,936,703,468,807,491,411,840,269,613,3,341,236,679,992,414,786,720,283,514,754,656,598,134,411,80,768,90,203,175,414,11,289,512,936,887,600,451,380,567,951,840,302,616,999,18,433,150,292,197,22,733,435,755,533,283,592,764,384,227,10,198,669,678,414,672,641,372,806,438,560,949,383,851,821,94,823,986,519,694,919,776,656,604,447,806,488,865,84,754,315,968,154,736,957,372,164,552,629,157,963,181,554,966,496,126,77,416,837,150,110,538,31,491,799,335,373,769,518,152,315,619,428,174,607,447,265,922,368,107,451,821,85,162,33,269,473,309,559,940,658,830,417,724,178,448,942,326,88,737,611,224,108,284,758,242,468,991,427,191,378,668,955,100,153,347,476,80,93,258,190,423,17,445,374,275,636,297,972,806,459,51,438,721,68,819,548,359,847,933,299,914,421,452,616,694,665,773,255,946,152,571,419,790,563,233,317,776,321,122,131,705,164,337,801,358,837,232,851,818,683,65,510,368,623,793,752,340,78,128,520,972,889,316,383,191,244,367,825,171,306,698,194,211,991,840,326,655,259,282,17,754,971,834,637,118,464,458,698,690,223,594,860,445,45,361,681,477,132,375,357,292,410,819,463,202,729,476,845,884,765,211,941,245,980,738,742,974,14,514,526,490,854,682,829,513,815,596,594,109,344,874,913,175,185,565,576,917,233,468,911,664,98,152,725,713,765,437,729,903,388,808,692,774,742,761,971,753,403,200,585,749,602,27,206,957,142,305,946,960,902,806,577,263,68,649,312,991,752,856,532,39,321,848,599,663,143,75,7,20,269,109,692,435,595,799,180,353,891,67,992,945,878,820,667,422,102,668,218,516,894,574,884,615,751,378,832,375,374,244,760,699,854,956,160,679,130,595,244,982,661,145,840,592,59,710,216,19,539,364,480,93,748,121,471,844,404,975,656,434,500,993,32,769,611,401,331,164,869,433,993,700,285,17,523,619,666,516,569,54,996,685,383,142,993,638,365,100,684,162,729,582,716,454,860,432,372,957,124,179,100,94,310,499,572,131,837,514,421,472,29,83,255,230,255,475,260,755,308,572,345,720,386,838,795,386,279,55,65,638,73,769,677,692,165,276,614,801,613,992,723,827,908,504,143,637,952,96,732,685,866,386,810,570,295,588,622,129,568,14,89,684,321,668,732,723,72,87,941,215,638,835,191,846,253",
    "786,154,566,953,630,402,449,840,73,98,342,267,793,736,616,897,351,829,804,921,524,420,116,609,565,140,71,302,573,180,685,824,721,140,601,833,88,605,141,706,407,744,823,51,122,135,634,656,185,101,78,812,28,741,744,625,292,537,90,833,154,893,11,574,864,37,729,276,227,390,182,702,687,251,357,175,878,735,46,822,435,609,521,389,293,188,111,274,614,598,578,867,907,518,272,249,91,779,908,502,632,605,110,501,576,625,0,568,574,410,589,852,919,700,547,889,519,697,308,863,901,346,17,599,718,962,884,850,403,279,379,730,69,411,554,693,443,780,623,197,511,410,483,10,197,535,883,553,125,445,826,672,609,887,898,661,405,689,391,381,392,479,619,623,523,334,132,137,330,283,581,615,196,921,304,46,451,515,503,174,260,690,410,462,181,487,252,532,653,240,601,624,215,785,433,32,146,190,392,244,549,459,275,598,291,90,717,323,892,670,916,20,148,556,535,484,956,862,14,93,520,777,990,477,335,292,896,765,122,569,423,398,357,266,751,568,893,487,232,483,357,708,497,417,145,205,17,877,743,61,747,643,72,746,340,782,935,932,792,898,907,183,21,225,163,817,476,125,891,280,369,27,136,692,451,256,93,237,284,760,627,171,597,386,512,737,612,508,257,718,504,896,83,443,895,150,347,181,157,917,998,983,617,983,755,571,340,758,982,110,414,672,306,219,881,150,287,927,667,241,256,576,994,417,478,843,880,905,728,464,37,225,13,591,856,671,473,260,553,640,70,969,889,787,480,318,362,697,680,619,673,550,266,115,781,366,105,116,696,992,509,465,441,763,375,882,477,72,298,636,902,15,350,180,164,514,335,150,353,906,233,442,493,573,857,6,536,958,902,808,578,641,460,939,416,759,685,389,718,458,218,751,527,628,935,292,927,430,5,793,517,698,471,327,618,274,131,675,988,336,850,84,668,853,721,951,487,182,204,277,931,12,529,350,801,275,566,42,297,582,122,200,483,395,779,49,760,728,705,572,165,377,409,481,8,57,491,864,72,78,199,394,981,925,772,887,789,319,648,372,371,345,468,373,347,576,731,484,945,575,767,406,212,706,560,724,907,35,975,278,425,35,175,26,762,53,977,761,453,434,663,539,688,785,784,482,569,679,16,419,953,501,725,7,220,132,589,4,358,319,320,651,698,88,335,227,622,350,936,234,663,303,970,604,163,339,488,245,259,882,556,264,725,407,725,314,850,804,801,518,826,788,777,746,288,952,736,804,291,550,8,214,814,509,226,235,522,215,304,793,709,794,924,92,584,956,816,180,383,600,133,388,77,398,852,619,598,799,222,96,603,16,889,563,665,299,69,513,47,898,965,310,918,681,670,212,590,409,33,694,518,786,722,823,737,849,568,680,615,920,918,929,962,986,67,13,665,433,629,635,630,398,665,402,436,820,138,955,129,941,911,813,908,164,716,240,98,522,947,751,689,388,893,318,513,791,973,683,121,281,321,202,833,505,143,595,407,6,744,536,901,414,666,441,342,289,709,283,760,522,789,759,853,932,871,950,533,622,789,63,276,59,821,815,399,770,247,2,764,242,291,115,329,113,356,806,409,543,399,670,404,855,444,49,779,400,257,317,375,179,405,904,802,485,598,703,918,679,269,900,692,404,137,693,816,110,486,229,249,881,666,296,21,123,977,46,802,865,715,555,129,358,869,976,450,211,702,323,853,16,840,692,198,705,173,912,804,971,74,125,224,410,868,977,246,755,385,144,836,22,14,433,788,100,78,758,521,274,334,295,816,520,693,705,546,81,298,395,54,773,259,656,877,911,417,404,540,71,927,986,506,789,832,795,923,455,781,360,788,270,816,718,169,159,727,371,688,973,49,727,760,623,682,824,69,482,128,869,630,7,279,18,84,696,124,958,757,461,821,392,822,159,682,99,709,539,143,315,533,233,190,990,148,568,583,996,938,147,661,382,147,582,816,805,423,341,850,296,476,841,985,437,479,30,157,771,984,476,335,974,324,309,393,126,751,845,985,586,603,508,237,379,253,948,675,63,389,734,299,832,389,248,914,423,727,596,528,970,247,659,460,56,803,531,359,54,143,29,119,125,171,758,516,630,977,188,45,661,4,800,258,656,858,154,517,587,397,128,440,818,224,678,393,249,394,200,541,532,714,45,992,440,191,958,111,254,29,827,871,312,390,165,326,69,629,418,495,758,757,817,352,309,322,773,104,274,787,782,593,797,420,722,64,407,46,70,913,611,211,469",
    "313,544,605,482,916,350,120,129,953,510,408,388,287,764,700,489,51,684,103,74,435,210,513,878,864,390,385,746,665,596,267,271,574,882,621,140,253,516,716,473,368,70,119,620,228,152,789,215,569,191,257,275,657,741,323,105,857,940,920,550,952,817,913,259,902,77,819,622,961,777,970,518,692,21,65,399,480,650,848,520,472,549,954,925,725,959,852,793,770,202,445,266,631,473,499,965,218,54,865,841,534,919,602,218,342,924,568,0,598,15,965,81,22,997,580,389,417,558,67,124,461,641,290,651,990,579,564,710,252,37,835,334,186,994,877,255,866,805,763,656,16,989,266,713,402,394,870,357,174,453,657,843,427,909,87,772,981,674,236,204,998,584,856,332,798,808,327,769,987,433,936,61,309,996,474,901,811,524,257,677,404,969,427,451,765,76,312,702,994,658,8,732,768,957,239,927,209,328,742,844,912,840,125,671,38,406,328,606,402,830,518,635,792,433,497,124,17,652,315,474,699,294,966,4,837,437,82,551,724,569,553,820,359,947,403,715,317,31,878,280,621,117,820,556,365,692,9,915,204,347,241,616,732,781,619,684,953,738,815,113,193,544,396,135,774,35,707,611,596,27,815,271,76,645,980,965,989,961,937,326,781,846,507,271,719,360,70,264,696,13,21,223,738,976,569,786,873,228,953,847,363,469,503,782,646,763,493,313,533,746,740,338,321,57,470,583,225,58,605,708,500,48,156,100,911,617,953,494,483,913,738,525,964,229,699,48,504,420,735,314,180,120,608,878,43,199,320,691,419,659,762,190,17,6,772,849,903,577,597,146,411,674,656,18,501,670,40,678,705,514,997,514,33,152,822,672,173,450,475,986,826,519,349,342,75,207,136,802,813,826,972,463,644,986,703,602,81,816,594,51,573,245,837,798,926,851,446,7,615,546,509,469,620,441,554,540,573,770,527,231,318,873,518,405,518,38,329,540,68,415,48,197,500,554,578,460,200,246,267,434,651,925,267,591,588,165,68,933,431,233,655,431,708,376,871,517,116,627,305,210,511,915,128,82,248,350,261,141,702,184,611,701,876,190,320,429,3,120,795,946,756,520,540,894,469,19,806,675,978,816,517,412,368,259,270,125,696,745,50,913,442,408,565,61,23,397,207,971,596,301,40,791,206,22,813,786,246,926,686,318,42,900,875,748,23,864,1,330,6,207,144,437,537,293,114,607,547,808,845,135,577,819,86,158,329,691,451,441,79,270,884,963,800,861,847,577,648,692,403,421,66,192,934,428,468,274,447,828,912,253,628,547,529,239,627,14,708,931,14,707,592,998,202,893,342,84,183,414,339,484,225,982,812,565,848,49,893,60,473,181,56,996,792,395,470,22,533,584,562,844,644,109,246,354,863,459,39,866,383,117,73,462,36,806,881,968,571,876,720,216,895,187,205,166,959,69,370,350,241,28,692,303,166,885,875,497,406,885,729,681,538,223,737,77,1,597,453,927,303,135,238,598,850,524,285,480,602,84,404,127,843,969,722,893,679,337,629,379,671,634,621,666,363,875,721,548,70,997,722,220,970,778,898,541,185,326,661,690,62,730,944,346,143,861,843,505,867,179,351,718,671,761,501,748,698,516,488,760,57,977,622,589,564,949,756,341,55,873,306,839,774,321,117,745,300,603,212,628,125,661,357,123,441,770,385,423,498,202,91,342,309,973,981,219,824,499,410,313,106,539,257,468,422,105,26,650,753,578,349,953,436,324,986,362,730,1000,793,656,334,770,394,767,449,291,163,100,66,423,402,839,171,254,672,674,30,876,973,724,410,972,236,760,983,712,385,251,104,946,386,538,65,471,621,947,232,210,718,586,489,654,881,124,635,159,50,221,547,848,342,235,689,292,221,707,197,609,463,744,287,852,710,905,194,123,999,501,924,779,35,81,175,815,318,742,45,388,905,710,217,659,115,211,49,786,845,444,955,23,820,386,83,746,473,12,153,799,702,562,687,794,856,96,505,216,935,642,598,125,191,689,94,876,781,55,74,241,152,971,948,453,107,70,456,464,497,701,817,440,582,171,321,809,899,747,414,174,433,975,154,947,488,695,810,950,850,496,565,364,893,30,353,367,910,738,886,772,524,579,962,732,414,382,430,725,892,563,150,648,263,78,398,856,954,324,672,666,552,271,280,892,242,474,754,465,772,114,595,289,452,302,988,272,89,215,155,836,667,351,268,767,541,217,21,436,835,816,755,485,660,591,843,155,673,105,984,103,329,54,291,419",
    "693,480,837,526,484,142,631,647,402,295,906,239,467,135,446,48,430,218,980,629,668,161,706,237,898,967,471,35,556,541,550,192,947,539,455,116,161,305,709,770,129,989,288,538,764,210,656,764,721,568,926,606,449,15,703,99,580,698,35,700,387,58,15,557,479,498,155,680,113,563,670,713,678,763,324,470,839,607,904,24,945,789,46,21,464,762,340,580,258,723,393,606,663,803,416,637,213,912,478,5,465,650,245,154,814,453,574,598,0,792,176,44,491,467,855,18,726,457,887,925,242,201,502,352,386,906,612,573,908,36,801,153,136,972,815,181,934,439,757,117,692,317,935,103,719,698,475,444,413,771,597,954,511,467,308,910,461,562,915,22,354,919,72,555,158,922,453,191,417,378,32,217,812,523,151,880,72,881,200,56,44,335,491,488,40,314,406,508,730,500,152,82,992,1000,651,785,802,32,184,813,900,286,295,703,923,793,517,676,510,250,728,932,753,225,603,133,794,237,407,505,61,53,94,604,89,482,989,20,378,715,138,695,283,964,541,331,214,459,397,677,252,86,918,238,505,63,122,27,818,616,458,127,724,281,301,570,366,354,589,797,694,206,127,307,206,26,839,912,426,174,910,818,816,823,717,304,601,996,29,133,739,243,676,10,404,628,124,338,831,803,215,132,945,196,388,738,772,263,255,24,93,555,488,904,846,402,490,13,129,785,386,75,798,285,588,389,679,598,782,360,634,27,307,69,792,960,665,632,426,250,317,828,16,794,306,285,653,623,981,4,971,206,564,422,802,759,705,394,242,393,30,614,309,701,999,862,71,454,429,153,832,985,787,873,910,398,635,948,231,616,788,215,600,871,356,449,199,770,707,586,842,429,437,883,719,51,660,915,795,436,47,24,525,222,573,619,408,153,224,650,570,152,368,328,259,932,364,381,572,522,906,274,78,911,966,578,926,978,762,47,654,146,114,533,697,258,453,773,660,616,74,295,963,572,660,47,528,300,649,706,459,617,288,324,118,40,692,72,678,623,365,751,177,880,232,318,986,26,858,976,489,121,531,857,168,301,30,681,858,513,290,785,814,547,835,419,16,827,954,34,163,717,984,680,787,229,807,239,201,515,260,40,531,466,956,369,52,702,307,72,853,123,373,652,589,525,876,433,633,39,639,690,814,597,19,348,262,413,860,497,690,803,672,677,343,970,658,417,777,181,998,287,9,638,326,623,225,824,17,340,822,544,225,908,745,80,739,258,381,711,206,11,460,748,813,821,798,37,83,357,958,599,692,311,194,528,834,497,137,518,117,133,57,203,406,481,663,96,305,957,162,142,957,973,780,913,996,203,391,815,892,175,322,673,168,125,203,105,72,540,372,105,286,276,169,787,584,887,914,8,903,266,92,881,367,969,600,737,139,377,494,781,995,518,499,475,195,473,140,487,596,985,115,371,665,259,995,668,869,647,632,460,3,902,938,645,157,963,714,239,488,518,457,218,106,556,357,760,667,290,71,809,720,366,365,671,195,684,764,510,356,79,246,801,569,815,490,601,136,345,683,612,530,536,557,35,621,609,877,702,675,657,331,161,288,84,546,733,84,979,174,94,410,503,36,645,673,280,771,246,835,230,839,822,128,668,29,476,623,862,20,536,708,64,376,984,18,733,987,766,170,823,345,583,600,587,33,182,918,323,569,773,996,468,745,11,600,308,607,337,531,100,12,962,829,138,735,915,970,967,177,156,818,977,531,850,778,587,515,460,924,791,457,997,54,137,356,997,948,102,341,275,793,604,230,997,716,601,322,952,996,601,376,752,997,903,678,564,854,975,912,589,80,879,559,408,732,804,225,431,788,835,225,831,795,846,105,106,827,670,81,872,434,126,333,90,708,25,748,353,569,234,135,572,386,2,842,752,873,780,579,607,297,925,145,146,399,585,716,697,657,530,373,803,499,726,436,705,582,675,677,142,480,741,959,373,248,581,493,115,905,893,535,183,257,344,219,466,753,275,775,856,355,988,362,989,624,600,610,714,82,574,897,600,369,303,312,798,101,473,111,607,827,395,665,821,436,679,101,461,482,341,437,789,241,314,143,880,432,483,691,36,833,450,848,928,415,949,775,106,60,927,350,881,519,21,202,429,38,467,956,698,196,337,720,751,977,638,642,306,951,455,296,606,982,594,982,743,664,487,620,265,294,261,245,526,38,960,476,121,38,371,133,136,24,466,713,613,847,418,23,376,304,192,475,751,228,434,769,14,641,367,440,545,185,444,61,193,989,250",
    "587,983,448,469,362,295,278,159,701,359,639,405,423,252,405,262,142,38,490,258,159,62,498,651,985,787,370,642,212,629,933,210,459,825,269,243,507,513,821,871,251,380,453,16,343,493,554,816,704,444,504,919,717,551,988,31,134,918,667,76,712,819,907,91,842,519,609,360,624,281,965,407,610,460,708,348,14,241,487,803,592,481,199,641,241,684,964,830,273,824,945,175,201,202,569,668,238,942,422,179,827,657,666,799,284,69,410,15,792,0,565,760,261,134,90,559,562,183,927,753,166,827,862,772,128,882,161,345,527,975,659,568,952,706,307,204,389,46,525,202,927,780,919,468,678,265,651,653,514,236,332,602,74,349,623,746,818,564,184,966,521,608,449,725,262,777,569,788,548,718,462,940,930,834,211,535,600,102,396,728,203,532,242,225,695,28,737,556,849,620,868,294,375,558,292,335,421,520,693,49,188,972,683,510,132,975,184,95,96,513,580,234,482,395,825,545,620,989,884,325,5,251,838,800,622,895,74,335,456,952,94,797,94,900,254,523,390,963,320,277,902,301,511,269,375,417,659,391,7,508,228,712,146,359,846,274,271,602,170,596,898,268,11,851,719,677,560,217,266,297,52,167,171,450,239,90,951,92,549,273,819,534,821,648,410,662,971,687,425,384,68,531,350,95,403,671,408,810,120,254,783,767,999,202,982,867,937,589,941,592,143,896,37,520,683,474,697,131,904,321,156,352,625,706,490,502,930,776,524,846,311,45,231,603,994,157,551,264,759,119,610,224,504,149,966,640,310,426,40,713,873,185,636,349,76,234,903,476,898,164,881,705,503,402,187,911,408,923,334,477,11,548,667,318,330,120,798,786,710,948,976,664,26,617,140,590,770,266,320,90,15,53,399,348,113,691,391,677,558,325,695,680,391,398,358,95,636,815,366,659,508,110,98,800,832,858,223,697,762,573,685,629,321,747,112,572,167,613,865,731,970,103,384,714,743,746,132,971,979,980,996,541,111,52,997,377,887,30,279,982,454,96,857,384,669,702,417,527,929,425,891,941,642,472,981,743,38,241,177,554,613,370,674,322,480,721,89,932,949,738,947,997,843,821,4,380,939,966,335,64,477,307,206,362,284,938,813,108,561,922,784,415,339,146,156,755,810,338,403,622,667,476,76,201,991,774,886,267,675,500,71,616,292,347,790,647,307,336,426,329,697,211,166,912,59,530,910,332,888,582,21,749,344,94,325,813,689,398,942,655,100,27,984,182,354,737,798,362,254,148,946,7,190,965,80,816,13,802,87,49,928,638,92,843,708,257,655,761,526,970,16,664,740,861,96,816,493,330,924,583,445,315,201,66,895,435,735,533,852,256,861,555,299,437,69,893,970,309,237,903,252,863,1,444,172,209,315,559,368,507,860,783,681,852,460,244,137,469,782,115,413,529,294,62,425,32,126,592,974,519,650,686,648,61,424,243,554,595,30,956,178,328,504,806,331,742,809,973,767,164,680,690,808,254,18,535,228,603,846,568,678,245,200,68,496,866,665,100,382,400,305,328,276,984,648,160,110,208,148,189,635,978,166,513,799,104,432,799,444,642,693,923,59,82,472,735,451,166,246,628,26,52,494,606,841,900,491,642,878,805,886,366,797,773,725,987,735,145,708,47,179,891,745,664,829,562,459,859,991,536,136,521,242,50,305,706,732,317,989,587,193,595,494,626,147,129,607,939,211,776,347,471,801,538,188,784,873,728,293,924,799,4,500,689,653,128,898,379,334,446,859,289,814,404,888,536,534,894,757,371,60,637,157,530,308,748,588,448,61,581,285,507,812,772,8,863,594,760,509,723,150,986,974,952,192,869,81,630,94,901,344,894,523,707,731,174,141,363,290,844,703,230,665,690,258,97,486,636,961,610,232,374,341,592,685,204,635,129,885,856,577,499,858,164,146,519,339,546,146,785,757,95,31,715,734,733,45,405,556,679,231,867,152,194,566,41,829,698,606,287,715,988,333,273,701,275,765,695,825,590,874,368,688,724,864,455,160,146,478,505,605,151,530,380,696,195,874,409,570,321,6,966,966,149,312,615,206,617,589,462,75,570,922,53,951,231,93,727,161,509,357,801,264,587,848,803,238,398,683,568,333,181,23,780,189,721,212,12,893,623,501,383,309,327,480,504,194,3,964,255,528,287,290,447,857,37,417,352,804,291,118,214,656,983,110,315,983,250,23,439,457,104,554,974,127,381,13,440,138,211,228,517,710,502,925,879,271,378,560,439",
    "659,148,985,13,926,285,314,759,729,166,211,991,118,749,557,455,705,943,400,512,125,833,523,935,975,800,912,305,323,180,173,283,657,457,878,943,87,77,324,918,454,456,856,579,509,857,669,245,898,920,237,118,880,426,920,570,886,231,980,330,211,769,844,44,215,129,967,192,392,770,184,62,611,293,711,425,160,510,233,100,934,378,415,157,225,174,322,847,209,448,658,88,729,120,94,699,396,366,924,459,989,864,524,619,213,537,589,965,176,565,0,161,186,563,798,336,970,379,551,52,182,973,956,741,474,423,392,501,610,286,866,775,636,440,51,517,529,710,81,174,202,973,683,378,102,866,591,719,252,973,199,687,688,180,830,364,755,255,452,26,452,389,982,746,894,962,379,931,373,570,160,245,297,117,937,274,522,346,517,334,103,433,695,20,985,737,475,269,354,199,989,622,213,583,332,242,934,957,253,430,391,211,210,811,432,532,284,562,686,145,677,17,910,917,258,275,284,279,40,727,865,271,349,908,77,21,203,324,599,899,860,775,408,858,919,688,578,232,491,61,695,413,637,554,443,956,89,927,865,397,481,12,121,129,698,273,889,436,91,206,191,377,460,1000,703,483,757,605,565,9,570,367,659,787,139,582,466,604,863,987,309,999,643,622,249,4,206,269,749,620,78,958,285,434,792,142,412,692,432,211,955,817,758,898,344,310,864,137,880,19,84,501,664,735,449,523,809,178,159,948,358,375,493,118,857,99,122,63,652,375,899,693,450,201,882,532,101,461,802,21,82,204,303,855,64,976,122,336,731,244,800,433,387,658,717,8,227,110,182,343,172,36,512,101,553,163,205,752,147,560,873,166,768,506,222,215,575,990,694,546,696,890,91,409,287,413,190,925,59,337,416,926,821,153,323,904,878,193,614,88,583,859,489,5,886,775,471,226,363,609,558,919,881,860,157,366,300,154,662,626,505,693,632,672,316,513,278,840,572,774,782,856,528,621,800,282,31,404,513,292,610,427,767,37,111,18,725,75,975,20,630,211,43,81,931,537,971,907,30,324,995,930,256,123,803,13,817,993,496,635,373,939,83,845,248,574,362,629,549,954,289,610,749,847,514,433,254,774,402,648,642,36,531,461,161,451,982,205,120,774,126,815,785,329,650,989,824,321,928,983,623,756,721,122,470,671,715,151,437,94,230,401,722,914,671,433,30,71,315,909,226,265,996,107,923,549,418,557,206,234,408,753,386,870,222,494,308,850,297,88,80,673,695,318,332,575,901,693,483,688,887,717,446,617,858,289,828,889,337,326,806,485,455,854,493,655,784,284,133,268,951,46,715,848,499,281,899,401,912,134,929,106,241,862,836,491,801,783,990,707,329,249,54,298,754,529,44,538,318,548,412,864,318,256,657,651,349,819,262,322,885,5,75,261,146,411,452,573,470,136,726,91,393,25,355,862,513,377,465,637,766,402,731,874,95,655,465,801,314,838,528,910,809,475,309,660,964,63,337,896,826,631,16,113,32,381,500,726,465,496,132,331,459,894,80,538,77,150,664,402,671,647,421,802,389,220,309,403,997,760,961,575,187,697,903,349,323,957,748,381,137,144,499,850,660,380,773,756,836,647,922,405,159,528,214,726,188,989,871,402,661,827,455,228,291,836,253,826,237,69,878,984,900,622,950,490,996,125,440,105,921,726,852,283,420,13,876,202,435,798,948,976,198,945,750,2,469,549,729,936,963,892,692,642,569,923,516,151,346,697,335,788,203,817,758,259,498,779,326,581,616,17,642,603,23,743,807,662,160,248,399,493,46,487,522,982,645,689,609,445,880,88,818,425,768,452,767,106,693,947,89,399,161,550,426,948,827,434,118,360,402,25,405,511,665,166,655,258,198,20,935,140,93,117,671,378,800,922,376,460,872,585,359,543,414,285,648,557,75,38,218,145,235,431,560,327,59,418,415,200,127,695,711,82,875,886,28,328,898,701,19,247,927,482,793,828,910,203,72,512,952,833,133,302,263,220,870,797,103,126,916,305,577,214,662,257,145,80,302,586,255,252,129,561,52,302,716,578,120,61,346,534,608,903,535,37,679,567,957,21,137,652,96,540,781,213,41,65,370,623,847,370,89,958,136,167,236,698,395,828,549,898,63,328,70,124,827,530,789,386,524,478,409,708,456,664,911,926,172,890,403,547,62,423,687,753,246,279,969,217,593,173,486,901,409,514,235,167,746,138,346,684,458,756,266,141,348,154,151,777,765,195,69,297,46,369,527,32,309,573",
    "55,855,334,183,942,2,880,668,299,476,571,507,227,734,269,486,697,866,985,679,840,199,688,523,186,551,284,488,736,169,285,324,232,959,474,938,681,733,813,164,698,44,318,327,211,226,349,342,581,335,979,822,445,500,807,868,291,4,345,951,645,1000,825,2,399,809,641,60,583,140,287,649,574,541,356,61,771,53,683,999,763,204,591,652,809,234,579,214,807,88,896,392,514,692,224,624,83,637,147,929,513,802,459,900,550,112,852,81,44,760,161,0,252,200,751,959,901,473,351,332,92,357,574,269,106,82,916,476,556,793,697,565,822,782,267,875,521,860,686,273,268,582,24,878,339,791,330,115,607,169,884,305,69,965,877,945,120,694,490,750,659,211,21,859,925,702,323,321,573,353,547,659,853,485,675,373,242,662,952,979,283,223,473,992,84,196,227,358,575,175,666,368,726,819,733,331,775,180,161,385,960,708,810,120,6,527,268,230,54,823,481,237,804,423,929,729,278,962,408,18,368,907,165,692,764,680,952,273,276,108,697,94,925,184,821,782,257,372,798,846,140,454,311,384,694,357,338,844,206,387,669,891,542,747,255,257,850,56,22,832,874,420,444,199,373,414,654,471,205,462,231,774,142,473,255,800,288,139,166,463,483,291,468,429,534,548,750,601,40,341,662,289,194,754,670,334,109,644,408,450,697,384,445,572,489,690,719,177,417,559,349,708,540,19,15,978,699,928,821,390,867,541,146,519,168,273,287,287,216,196,652,164,77,952,155,732,802,920,323,296,519,27,944,304,633,842,586,320,740,234,742,504,555,780,509,677,386,524,72,866,457,853,363,311,362,397,624,779,639,420,356,703,158,269,369,185,845,454,348,125,2,856,11,615,86,761,126,920,184,835,504,34,15,613,73,73,749,318,91,177,331,204,51,647,703,376,348,898,700,755,358,735,69,125,44,766,790,796,458,123,630,996,133,960,532,886,886,213,809,94,848,651,238,35,148,970,314,984,351,739,921,90,461,615,809,756,786,738,466,314,443,259,498,845,65,468,989,668,611,251,616,9,292,70,740,243,491,507,354,935,75,993,993,719,789,479,239,563,730,681,862,330,897,408,664,893,873,612,902,412,483,257,982,357,408,16,884,692,937,626,575,152,644,348,464,510,967,620,317,403,301,640,23,614,199,759,215,886,850,870,975,670,607,882,153,885,293,830,534,823,971,127,593,631,956,176,365,752,934,870,803,273,856,349,528,81,48,531,492,86,413,563,385,687,546,584,581,796,312,807,590,304,641,874,79,610,274,787,35,701,707,765,385,299,66,480,323,915,482,811,674,493,212,54,278,442,888,613,411,110,643,671,528,164,351,826,107,471,614,758,156,600,765,777,312,926,984,214,469,319,200,127,428,809,503,965,730,152,275,503,348,483,303,591,503,912,384,12,27,427,461,621,251,283,900,953,703,871,915,113,472,918,380,41,578,655,998,423,700,995,778,543,385,658,294,648,95,696,539,762,970,313,3,178,921,347,486,382,383,371,715,339,805,730,984,691,699,230,119,731,144,246,482,879,846,555,899,966,399,705,383,468,983,324,260,943,483,49,366,502,732,32,245,750,361,663,367,73,314,716,135,431,658,812,76,912,217,668,340,696,797,952,318,169,901,43,833,548,56,171,74,481,117,552,409,829,454,392,943,832,357,395,649,64,36,776,770,527,84,736,255,380,464,328,662,530,893,164,402,974,59,148,661,960,373,872,90,236,14,678,761,602,741,959,593,447,57,207,520,30,756,210,83,573,186,72,908,543,350,867,951,785,686,303,553,377,685,570,612,19,948,430,815,480,878,249,175,890,488,232,881,724,405,352,260,561,296,908,956,885,208,443,389,182,867,328,51,574,543,615,898,534,535,936,273,102,999,319,470,688,444,15,754,730,72,986,207,107,837,817,780,755,301,344,974,624,885,876,368,287,723,757,224,415,696,907,900,192,180,78,559,716,127,8,98,615,830,809,726,583,164,682,854,369,212,901,865,472,388,143,396,88,274,501,477,249,359,670,732,258,529,519,98,894,591,681,795,295,171,908,848,373,287,687,98,521,579,242,935,119,111,130,975,604,707,169,957,969,374,86,437,438,243,972,191,391,564,255,198,478,911,827,186,608,847,680,936,918,794,80,503,334,22,725,237,865,624,215,843,292,82,637,47,708,129,130,718,657,222,226,625,622,781,378,131,867,618,581,640,4,286,560,784,349,2,946,203,239,592,757,290,14,50,755,354,151,147,256,730,600",
    "491,372,36,732,653,25,642,390,354,23,731,952,758,961,708,490,255,564,795,852,410,351,103,846,91,817,30,191,456,986,953,375,663,687,183,773,920,808,409,952,627,219,902,626,266,756,760,684,825,545,530,187,896,432,743,628,236,369,523,566,321,273,672,916,249,396,706,296,254,716,939,410,336,437,585,99,635,305,519,195,968,337,79,281,461,348,779,342,214,490,841,655,224,461,253,631,109,829,32,203,226,711,77,78,526,830,919,22,491,261,186,252,0,7,816,506,181,193,957,399,236,695,841,854,30,983,56,244,560,164,300,750,374,201,798,141,287,237,205,326,784,504,899,2,82,132,790,328,326,935,810,37,723,929,542,323,600,252,14,898,514,857,365,157,684,119,71,72,848,524,275,910,687,665,639,851,904,377,231,24,811,175,837,611,299,900,477,738,717,853,248,33,150,524,206,759,597,945,948,98,261,468,708,992,376,881,90,115,337,165,954,185,489,526,15,902,993,218,368,997,413,443,13,570,435,527,727,757,507,109,531,439,857,530,69,961,900,792,198,219,623,438,578,355,269,459,681,127,21,327,155,295,674,930,191,9,557,761,547,562,169,971,393,616,889,676,502,364,12,593,869,851,836,561,420,666,111,358,717,555,699,77,34,801,155,905,767,881,545,502,5,114,120,695,924,797,86,579,389,140,350,552,298,687,782,319,306,838,880,167,906,25,76,533,671,487,291,773,573,952,338,987,97,910,124,979,130,785,758,474,850,612,322,335,312,237,639,833,939,727,926,346,952,459,269,86,174,578,286,180,339,751,153,456,878,389,927,934,9,599,66,157,430,725,78,720,551,798,584,291,177,778,203,212,511,451,385,420,710,975,968,910,109,343,618,728,623,87,941,509,525,384,822,859,821,211,326,514,459,118,375,61,155,621,358,188,331,61,403,318,460,680,872,504,790,277,337,436,105,716,389,526,996,552,809,906,631,59,378,906,687,590,646,878,498,663,818,603,65,339,320,354,952,902,392,516,99,25,527,29,3,32,20,873,429,400,151,860,314,931,52,624,235,321,150,547,543,109,834,9,11,660,763,156,428,403,168,717,4,686,340,896,814,190,393,805,607,173,351,298,536,2,648,921,790,186,639,423,43,86,729,519,678,891,401,147,989,738,861,174,468,959,283,203,370,147,874,777,144,625,19,669,429,900,328,837,875,376,53,936,431,372,710,385,447,155,621,481,216,242,727,46,124,767,481,972,545,15,678,124,339,464,232,181,395,134,178,364,929,486,200,234,944,618,264,821,507,710,746,905,678,97,440,933,455,215,431,83,781,426,821,136,649,600,966,853,427,715,436,170,765,723,694,145,780,405,702,304,888,454,588,556,346,540,350,626,225,298,622,56,821,25,52,974,961,164,642,384,581,95,613,34,458,101,219,291,431,107,661,964,164,970,300,456,411,91,931,123,975,419,627,830,188,201,374,895,637,647,772,720,696,906,374,10,362,396,147,994,287,531,914,782,972,754,44,325,909,948,69,225,39,883,209,623,440,826,541,952,549,807,896,943,830,631,246,700,863,612,845,829,958,177,89,705,653,547,108,798,489,410,364,869,479,185,586,899,365,753,612,731,753,370,849,810,129,685,965,676,893,644,847,127,802,572,855,848,441,48,283,140,412,939,971,556,944,930,264,799,270,932,41,309,714,911,749,436,564,87,980,388,153,180,141,284,71,222,384,632,480,823,152,168,892,665,109,324,9,761,986,334,518,98,509,300,676,794,350,144,122,113,801,89,111,623,135,534,202,691,316,430,258,159,277,545,844,13,316,20,397,618,542,881,86,946,40,146,518,409,555,287,433,376,644,374,311,919,93,783,564,187,919,440,609,822,600,763,754,276,351,207,709,995,301,641,778,537,695,42,18,686,711,29,931,583,725,146,344,533,953,746,154,420,980,579,452,188,773,604,639,570,335,947,764,381,12,646,557,293,424,752,60,822,985,75,361,325,849,430,178,874,714,395,209,789,699,201,330,768,153,191,462,468,578,581,887,790,279,618,299,869,247,814,242,637,686,965,562,813,507,878,455,873,891,253,423,336,532,689,903,636,513,525,434,810,588,218,139,340,565,581,421,349,651,443,267,189,661,113,646,413,42,542,522,849,507,849,557,832,197,736,78,818,594,568,61,31,265,235,733,881,484,282,78,394,520,972,378,792,237,350,115,28,477,960,30,736,732,597,869,551,900,772,381,594,685,60,999,772,234,718,602,246,956,13,679,898,180,498,223,15",
    "467,746,409,180,37,319,84,542,681,533,354,270,547,373,132,358,436,80,955,100,238,334,128,732,450,906,306,534,566,749,269,724,672,168,315,370,83,155,745,170,101,607,452,946,645,687,823,289,663,426,521,124,366,35,674,182,354,465,166,957,749,136,180,523,326,904,669,860,411,977,778,272,34,642,424,506,793,639,613,851,250,396,575,758,968,109,725,769,345,375,904,899,750,701,512,614,172,240,972,991,895,26,584,579,175,548,700,997,467,134,563,200,7,0,734,714,754,525,439,820,122,158,840,487,72,499,655,605,493,691,106,661,357,336,702,583,201,684,56,576,710,783,420,794,716,86,978,540,645,975,238,567,399,613,851,387,276,193,814,554,70,266,58,265,880,176,305,399,155,32,945,326,777,510,856,173,412,943,711,872,23,222,774,966,498,384,856,784,804,873,714,37,764,81,800,690,161,550,708,763,361,616,966,389,134,533,535,472,345,78,581,681,586,923,943,1,734,833,954,11,223,220,852,66,391,673,810,137,920,590,404,747,6,618,55,118,725,855,226,110,458,37,105,536,936,981,912,471,63,73,982,116,216,303,722,180,810,181,40,381,514,43,413,907,197,671,283,997,859,945,590,45,560,961,755,346,677,495,542,546,962,19,930,109,917,198,465,612,427,530,307,359,946,235,354,150,797,156,673,241,162,134,316,743,662,132,696,822,99,839,477,408,58,868,750,12,75,91,521,455,747,695,826,600,545,170,372,783,985,467,926,568,643,972,341,782,497,284,542,280,724,980,609,308,783,630,805,884,60,206,311,606,889,967,712,906,55,533,351,758,545,633,858,87,752,400,438,228,374,54,632,812,522,570,566,512,370,399,587,430,723,228,574,637,798,662,218,277,355,935,894,16,911,693,547,373,44,310,213,422,374,119,366,269,872,635,591,929,347,628,139,92,411,193,186,796,495,301,345,602,109,400,312,60,184,887,701,881,132,753,492,10,527,669,505,662,806,382,508,196,975,266,373,456,592,983,37,585,968,710,484,64,403,972,91,134,897,123,398,598,208,512,885,45,605,772,525,456,102,946,950,794,76,917,248,395,173,359,542,636,501,87,219,70,86,714,3,778,612,994,455,548,549,593,518,457,784,800,96,28,142,476,913,243,892,488,902,972,422,927,529,134,188,446,186,400,285,756,20,245,116,247,88,312,181,668,531,371,451,895,691,410,573,568,169,835,501,701,622,276,439,639,872,5,264,17,984,840,566,124,955,94,308,697,665,433,574,998,547,795,611,386,857,138,165,415,465,75,152,129,733,815,199,547,403,202,381,274,731,773,912,475,122,168,954,902,218,335,639,622,858,478,48,646,965,474,916,843,220,952,227,518,183,965,557,772,773,982,844,388,362,271,65,259,540,478,831,940,988,541,164,401,170,106,769,470,700,147,518,830,872,234,738,102,474,785,188,510,854,812,14,500,283,531,41,212,381,404,736,631,920,8,26,174,119,935,764,489,8,730,917,228,511,362,919,796,759,937,688,83,693,84,937,764,269,326,366,114,246,180,911,694,362,561,365,357,639,981,200,144,757,919,274,202,202,779,783,698,42,876,429,745,9,134,483,641,223,482,394,501,382,183,26,725,416,64,438,476,729,964,82,585,589,709,891,252,682,531,379,512,908,496,26,771,27,542,75,382,651,169,104,405,909,145,491,768,122,879,687,868,472,363,415,554,860,842,865,763,506,264,504,249,136,98,281,417,426,259,285,541,727,108,569,700,573,544,939,226,604,424,100,928,781,141,758,932,763,3,608,482,368,380,517,364,154,772,562,104,853,127,783,718,790,276,928,722,322,68,810,349,717,44,957,268,763,109,960,374,209,145,19,353,729,100,255,621,607,271,822,147,438,715,578,372,737,594,511,441,139,170,696,460,570,18,4,577,773,977,520,399,223,785,902,524,3,959,921,73,454,753,846,567,68,580,609,863,527,941,9,997,273,75,780,358,174,839,574,333,698,662,400,695,878,804,580,266,362,928,279,530,293,560,93,877,799,444,177,577,662,467,581,558,524,63,985,170,312,965,439,296,279,530,262,914,746,115,480,969,343,845,776,732,427,235,827,39,270,578,982,408,669,751,295,657,853,890,281,847,338,534,353,463,309,304,221,553,854,52,167,961,635,831,440,71,810,956,381,780,425,625,711,94,257,599,478,526,633,320,747,52,25,514,772,884,767,556,392,29,117,708,622,414,412,381,637,418,246,502,238,508,963,917,682,538,883,307,925,193,294,197",
    "33,876,227,775,484,458,408,938,251,618,484,279,842,784,895,491,82,686,738,210,114,92,686,84,315,388,928,177,37,471,406,339,221,838,709,564,256,501,552,969,420,604,726,406,199,951,766,397,58,436,793,274,534,24,780,513,361,278,903,872,128,478,969,369,697,92,285,907,107,68,731,768,921,877,107,953,834,865,978,149,311,989,719,607,992,7,948,820,767,9,145,659,265,957,102,983,284,993,500,358,73,893,709,533,346,136,547,580,855,90,798,751,816,734,0,680,950,959,533,686,870,199,372,646,206,769,275,364,912,771,859,85,364,783,580,899,639,5,575,483,866,113,885,193,268,544,686,581,970,535,195,122,107,455,570,614,774,744,131,820,246,770,608,515,784,930,245,458,547,872,53,987,543,495,660,280,781,63,720,597,799,966,938,465,981,794,888,640,187,668,818,932,440,11,256,585,672,865,598,569,827,84,753,530,68,403,940,296,836,973,633,271,787,776,195,229,347,85,103,866,697,740,779,464,407,899,443,408,79,506,739,898,27,174,223,751,584,6,448,203,481,739,503,267,80,157,435,139,805,493,304,52,146,435,880,372,971,987,702,695,668,974,145,231,994,587,163,234,275,123,986,459,939,780,593,523,537,177,226,260,65,705,586,64,302,990,196,44,639,354,617,383,421,802,588,111,231,267,107,558,103,54,9,642,356,993,566,556,934,789,562,540,336,91,56,661,435,874,849,766,619,314,626,27,664,545,227,6,456,513,825,122,984,586,644,513,45,953,539,987,199,206,95,818,997,400,459,125,976,702,631,947,847,188,45,206,718,880,968,52,753,47,986,333,275,988,883,966,629,960,596,677,760,638,47,802,845,694,937,326,895,915,692,819,588,75,917,611,819,304,517,744,847,967,743,118,831,568,285,744,532,973,743,264,609,144,579,396,97,238,324,920,987,20,172,100,79,865,950,55,323,77,933,366,19,901,131,462,612,296,648,548,446,270,622,732,21,3,485,854,591,114,349,33,545,585,133,53,257,396,644,84,654,325,901,542,920,457,548,22,709,737,636,716,824,430,434,224,97,570,86,150,976,300,337,709,439,791,994,782,378,963,99,625,884,577,924,580,373,197,119,864,484,579,155,52,614,299,823,139,912,524,898,93,379,28,68,831,175,718,47,351,489,75,362,704,594,11,862,440,269,971,150,439,6,151,689,868,266,169,285,857,785,913,413,530,256,401,28,939,783,797,673,702,342,415,257,786,580,716,5,258,176,731,343,973,512,351,653,588,803,898,661,880,323,364,743,553,969,153,648,4,630,205,21,531,17,376,227,283,670,933,510,591,168,874,32,934,493,483,297,390,676,385,846,404,717,783,255,467,993,461,23,957,859,697,613,75,196,112,424,166,850,978,408,449,286,764,897,248,290,474,177,691,51,744,643,943,144,316,867,924,245,274,203,779,561,594,269,341,782,387,811,26,371,199,175,982,790,9,5,204,294,283,315,478,820,989,135,701,943,561,301,329,19,375,690,45,146,423,734,510,406,216,335,624,917,278,996,270,745,426,662,87,333,464,139,516,693,303,117,284,811,514,678,548,470,293,572,767,389,30,431,455,771,935,382,406,524,184,733,995,706,291,487,108,576,542,873,908,515,770,775,790,720,954,931,528,440,755,546,831,357,555,474,507,803,997,277,715,6,228,857,143,566,583,330,364,144,406,836,739,446,896,414,883,44,124,476,677,959,992,832,295,798,144,756,665,83,404,31,594,52,322,106,922,831,845,376,347,855,455,434,473,488,289,967,144,769,992,788,583,890,351,558,506,456,107,256,499,47,970,942,741,297,763,235,809,420,941,911,88,925,630,882,997,27,594,169,640,262,11,984,614,145,330,885,924,907,306,210,356,264,889,508,38,80,880,360,212,475,664,461,734,236,708,56,174,661,381,106,71,729,395,431,623,627,854,349,147,563,544,427,530,630,386,492,914,867,671,784,356,567,397,470,924,36,147,577,412,839,167,511,228,849,737,590,758,659,543,623,34,963,476,748,552,427,956,597,718,416,161,500,664,412,353,324,50,591,118,255,551,371,30,489,201,208,210,520,546,491,392,272,570,747,664,122,66,817,913,51,754,571,975,199,130,802,217,525,766,386,529,891,513,336,238,310,849,726,76,488,612,389,156,360,327,343,776,285,362,980,532,136,710,873,447,577,750,499,784,255,276,118,546,903,213,832,512,483,243,154,250,702,275,2,820,355,291,532,425,258,243,592,100,914,331,358,893,241,465",
    "955,207,428,752,416,666,693,186,359,951,964,784,612,490,182,917,706,965,549,700,416,657,789,937,995,381,144,709,723,423,218,96,44,505,942,869,359,836,623,549,364,7,929,223,735,319,99,792,9,481,444,333,247,912,295,231,932,114,746,530,313,373,879,321,410,642,115,891,182,721,597,106,66,960,171,507,695,681,944,94,317,361,549,664,639,327,409,661,773,4,64,516,447,39,264,135,72,577,125,869,893,945,706,480,721,943,889,389,18,559,336,959,506,714,680,0,359,219,153,117,569,584,600,387,533,868,299,377,929,278,591,566,756,79,577,38,682,125,857,746,811,464,261,904,913,923,845,832,411,368,89,940,971,920,581,319,223,800,268,593,646,249,992,758,431,637,591,76,35,893,857,859,667,324,347,824,115,202,853,20,955,750,2,969,132,928,808,898,306,826,42,612,722,711,841,564,366,786,831,209,71,739,916,26,229,872,236,191,635,419,222,522,689,1,499,998,978,707,523,410,667,57,460,936,899,752,682,502,100,21,775,530,681,130,265,541,586,936,872,126,503,174,604,257,314,34,728,614,812,218,187,163,949,100,997,977,832,310,15,847,370,878,278,721,127,345,4,815,483,167,631,198,939,201,457,278,291,662,585,159,217,605,989,661,537,347,94,74,702,974,226,245,140,58,540,789,624,918,25,132,904,308,211,398,856,193,986,317,288,581,122,866,919,315,528,759,524,940,637,55,989,642,827,994,738,301,733,204,887,671,351,196,651,595,890,544,655,129,845,991,905,547,202,956,369,561,502,132,121,480,410,296,596,336,917,816,919,852,787,895,569,307,810,382,732,226,920,339,368,503,529,983,203,793,283,569,554,123,725,134,231,310,361,812,925,855,126,840,43,299,301,515,516,161,837,364,694,431,602,70,304,636,871,907,237,554,33,126,262,762,405,979,678,246,570,371,438,846,577,361,654,536,423,359,47,299,656,520,53,992,574,229,249,61,773,164,252,277,75,666,207,563,853,502,96,556,911,352,189,526,179,307,612,884,797,125,87,65,137,214,150,85,786,864,950,402,122,563,668,465,890,176,289,554,836,812,781,929,235,854,489,74,278,184,877,805,320,784,949,655,33,173,176,694,845,306,463,86,867,93,758,612,180,831,999,636,781,519,32,401,548,72,517,486,80,188,140,487,742,523,774,350,192,182,728,39,314,174,375,351,117,971,36,193,631,323,68,222,291,176,182,316,747,765,639,689,412,482,365,496,117,406,943,53,311,505,739,43,872,411,787,56,21,194,499,90,7,626,207,675,164,24,771,767,743,668,977,868,484,455,975,523,305,735,516,78,727,910,775,867,846,902,694,650,685,177,159,459,989,252,700,590,746,625,308,934,689,357,909,844,599,587,266,265,752,686,964,325,686,884,924,422,793,851,590,209,444,600,517,995,444,974,22,568,939,154,542,670,270,346,328,718,467,457,297,968,428,449,93,767,100,155,417,959,909,712,308,153,954,218,555,316,906,805,680,861,292,611,472,550,217,716,728,20,308,442,647,748,323,918,852,345,599,234,577,931,685,56,343,540,725,23,801,595,610,936,522,761,773,135,7,243,805,407,527,835,669,729,39,253,474,910,119,212,79,338,50,662,649,346,240,647,556,161,327,747,360,222,887,575,449,168,173,182,369,862,211,455,204,556,707,508,323,11,637,598,369,28,683,780,84,126,799,421,100,50,515,90,979,523,519,182,965,548,155,615,75,29,596,300,420,800,630,303,848,117,503,768,253,485,265,205,62,261,802,86,785,893,732,429,665,115,819,58,370,997,62,543,470,904,598,317,237,982,73,65,918,649,866,205,70,892,557,661,865,527,749,154,787,690,154,19,245,412,100,321,562,283,116,312,864,888,855,256,147,521,834,161,258,765,494,430,204,276,752,266,783,252,954,630,969,787,96,28,799,297,990,676,325,422,208,201,516,732,948,239,837,381,173,612,918,643,71,480,684,3,449,527,713,409,719,48,814,164,949,503,424,649,181,258,438,809,513,445,626,911,685,806,607,448,892,662,86,949,29,25,107,863,958,288,713,989,947,347,46,977,920,427,49,559,58,674,46,774,856,391,669,467,619,556,597,757,429,620,447,64,735,498,809,343,607,635,253,779,568,691,592,894,953,796,405,517,770,711,283,853,187,571,402,73,603,923,380,345,578,551,686,453,111,53,565,610,497,223,632,407,53,212,12,158,883,10,703,325,487,343,309,500,711,34,248,964,407,595,344,370,711,945,827,187",
    "883,833,625,408,381,716,358,921,99,242,466,547,906,33,994,621,910,898,602,642,680,106,343,118,974,582,773,7,662,564,625,833,226,78,767,318,904,573,742,341,350,362,920,898,219,492,71,411,535,814,684,454,565,323,983,699,290,41,456,749,142,948,609,783,58,819,458,980,991,712,105,373,325,589,813,11,693,351,558,901,685,103,453,289,31,315,498,156,167,804,179,503,995,311,865,428,470,450,921,573,204,559,490,248,353,863,519,417,726,562,970,901,181,754,950,359,0,787,956,232,426,379,408,185,693,517,181,863,170,443,486,90,660,840,405,432,148,888,950,899,67,423,847,913,923,832,457,561,236,544,400,734,744,26,91,582,887,654,693,449,423,117,510,326,794,742,502,193,452,107,473,839,561,159,890,947,371,802,756,218,291,188,686,71,504,450,776,719,290,16,819,795,892,67,82,374,318,614,410,980,391,635,911,197,75,250,177,960,604,837,677,793,8,252,169,411,92,535,452,860,520,95,24,290,59,416,829,119,49,734,96,367,301,872,360,347,101,681,167,886,29,156,123,148,840,132,357,12,719,828,963,618,76,680,350,69,975,581,147,249,744,340,442,900,857,334,110,378,394,850,636,207,718,554,298,722,827,692,837,852,840,786,603,896,153,850,314,10,60,80,671,226,507,713,822,673,646,278,268,641,234,145,518,819,100,428,96,338,810,659,61,423,711,96,190,964,96,757,862,321,426,518,220,428,786,898,984,179,221,992,23,811,713,519,869,775,959,693,728,62,60,452,820,735,579,878,443,598,387,938,486,619,269,674,505,256,705,6,619,395,836,515,192,5,591,640,724,490,502,821,742,630,802,476,150,446,725,270,812,99,7,653,247,186,326,896,45,386,439,566,743,261,177,633,431,250,1000,721,218,235,432,298,502,633,377,342,151,547,120,253,760,860,358,393,340,593,803,61,806,815,981,949,614,543,444,607,870,64,961,860,971,575,263,861,698,372,634,868,338,591,30,908,481,998,165,411,813,375,67,338,322,49,586,122,16,921,432,328,344,730,232,36,595,268,799,225,97,650,957,125,735,411,810,115,687,5,956,630,284,237,747,728,395,673,359,913,175,105,668,121,641,255,405,330,319,158,143,986,881,299,761,931,359,553,521,429,677,40,883,427,895,912,142,254,717,562,185,994,356,333,405,963,960,489,1,187,6,94,706,296,47,943,121,921,832,294,274,984,510,136,825,354,549,624,966,901,253,699,819,32,912,658,622,370,287,464,608,193,747,381,310,723,166,127,389,79,504,52,66,674,735,455,494,351,891,425,847,946,943,845,549,21,82,840,910,703,781,556,902,181,306,891,838,461,333,908,817,675,665,529,755,623,824,834,633,362,431,851,587,99,977,213,608,41,423,651,775,708,175,299,15,165,582,959,567,492,713,284,294,163,219,493,994,212,94,704,883,54,579,554,335,593,513,597,36,423,43,43,845,891,221,740,587,61,639,432,299,320,891,729,702,761,37,629,159,66,609,605,622,180,510,151,632,816,465,550,378,814,485,585,250,97,553,530,81,410,619,822,533,436,416,607,766,663,984,530,952,292,225,103,945,908,623,167,83,474,658,251,474,224,577,525,148,532,534,648,989,646,297,941,738,124,562,646,857,209,705,170,684,63,261,322,102,285,582,138,781,435,473,547,987,781,404,78,9,734,172,195,549,183,840,575,774,250,581,866,541,442,159,466,515,875,173,951,784,723,972,623,49,358,189,960,730,362,869,768,858,728,937,910,351,696,209,445,894,662,657,985,935,200,399,673,977,955,378,466,785,598,114,922,398,274,174,707,33,296,272,948,385,597,336,639,40,308,138,909,323,278,339,842,907,358,850,314,987,141,947,979,490,253,342,361,498,127,735,892,493,137,367,676,92,353,398,344,360,618,58,424,273,377,694,988,476,751,200,478,699,687,484,861,397,305,126,595,760,7,615,456,429,725,763,786,3,536,713,549,499,519,884,793,719,659,415,871,513,749,249,698,760,48,336,18,163,6,997,833,767,46,342,56,833,238,653,338,264,867,699,358,259,12,924,437,348,77,274,129,982,80,45,647,697,81,684,790,321,15,608,444,733,694,549,33,173,52,599,203,315,997,439,72,870,461,186,72,730,636,162,751,529,571,433,170,898,156,643,371,906,990,913,367,966,347,655,87,471,784,489,759,355,3,927,420,574,214,375,971,853,738,913,101,329,129,327,725,135,272,440,32,893,385,690,216,476,27,958,450,81,200,912,269",
    "353,719,275,597,553,671,960,931,179,228,734,601,609,703,518,68,302,820,81,531,41,126,46,925,872,245,328,718,953,10,702,252,805,651,277,236,330,553,854,1,759,876,15,412,290,302,353,968,55,122,486,82,334,144,25,199,403,630,369,306,513,908,894,24,578,199,522,683,492,100,481,924,3,602,203,774,121,265,441,179,633,237,243,395,686,801,832,321,876,6,211,959,366,31,705,352,50,486,4,671,347,916,438,184,374,861,697,558,457,183,379,473,193,525,959,219,787,0,183,781,719,562,779,743,353,546,274,952,546,371,160,283,630,698,5,277,682,368,973,705,276,377,533,644,688,70,844,952,171,278,988,919,695,119,909,26,844,993,184,177,712,651,289,131,184,364,773,255,411,215,512,31,580,906,791,956,508,770,757,838,776,417,138,482,341,333,101,30,805,389,488,758,437,161,731,392,694,183,46,395,244,766,806,228,680,79,380,316,845,649,272,597,622,449,758,765,447,355,663,210,868,49,262,475,237,832,812,266,500,465,530,447,919,776,426,323,38,816,726,514,577,267,143,765,741,240,123,781,402,793,298,977,432,19,33,947,937,699,725,976,216,693,56,346,853,113,275,863,7,494,273,962,61,497,170,666,541,234,867,687,449,1,275,752,895,547,263,37,979,417,149,263,381,41,702,697,89,16,647,214,281,542,712,194,149,760,976,14,604,75,581,200,268,586,62,274,249,948,146,946,856,837,124,810,416,450,133,317,71,135,553,712,762,118,198,828,311,830,602,294,515,242,801,296,243,894,193,383,994,494,888,899,954,856,18,293,476,663,909,234,574,182,881,359,3,175,508,75,259,994,110,36,515,260,556,416,536,449,541,924,376,116,268,917,900,690,217,360,37,307,665,314,708,909,341,504,52,990,254,41,979,463,624,969,678,750,867,357,80,32,209,57,49,474,392,12,374,841,789,132,500,966,824,363,127,137,373,664,273,659,923,932,902,158,52,469,606,640,587,6,675,138,321,795,315,324,735,804,175,660,231,908,53,979,646,508,688,222,169,454,49,482,826,307,472,787,749,131,372,794,189,47,93,760,698,651,734,222,312,119,722,566,522,288,486,311,614,713,637,793,547,679,420,478,754,714,139,501,5,83,375,186,258,544,917,119,220,939,268,833,560,800,131,503,855,998,273,48,313,969,845,928,169,330,990,291,838,387,635,712,181,857,224,271,256,235,189,599,84,841,182,675,227,375,266,947,261,471,614,928,584,88,940,118,567,193,856,15,507,198,7,239,659,773,628,71,529,233,562,124,209,142,446,525,238,301,477,821,604,924,719,288,873,959,426,957,61,770,768,856,423,568,521,145,258,136,776,726,138,512,274,807,470,52,469,651,674,494,657,991,615,939,617,643,425,362,636,530,284,446,441,966,557,348,656,655,251,242,916,449,414,833,675,659,320,787,60,914,891,989,402,295,956,740,807,414,160,925,186,78,754,35,392,520,214,220,263,233,72,655,480,645,208,700,265,618,426,563,422,69,275,667,521,91,4,970,452,154,548,972,314,374,199,719,520,901,549,141,863,309,11,682,169,803,544,541,99,956,266,356,864,59,331,34,394,197,789,953,671,901,60,874,153,668,909,865,574,248,180,385,934,589,584,215,468,989,982,326,461,64,621,880,683,920,385,723,159,461,875,908,63,315,133,458,182,440,254,289,852,714,504,617,995,581,76,598,562,141,340,427,582,284,88,129,397,552,657,313,473,241,980,89,728,31,985,322,440,911,360,157,950,960,981,714,603,960,410,18,857,726,911,405,590,333,480,782,644,880,601,353,992,335,108,93,179,832,844,410,177,564,336,546,385,5,454,679,283,542,309,934,225,381,928,373,132,566,740,185,37,380,795,401,347,293,354,782,316,613,499,697,989,158,949,420,347,779,858,191,116,381,499,394,487,76,620,759,278,729,481,23,717,196,767,979,225,471,838,634,23,634,362,298,592,229,892,537,892,425,861,698,244,161,607,220,660,998,478,323,775,461,501,336,92,636,588,381,656,855,992,603,755,738,958,994,219,260,991,14,412,339,380,3,226,12,589,649,623,689,806,851,670,649,445,151,196,815,752,913,311,388,225,888,577,366,111,498,334,911,893,67,169,428,808,859,916,285,760,591,693,12,986,210,551,141,959,662,899,264,851,485,754,524,794,238,62,380,655,490,480,219,390,756,944,926,63,596,408,529,980,70,189,14,750,229,540,605,503,604,501,340,581,313,627,81,402,307,619,289,23,872",
    "452,607,207,866,441,842,799,876,420,42,815,9,671,725,499,359,52,698,464,578,630,249,735,814,463,543,959,443,613,117,726,15,953,860,967,340,606,460,256,695,154,613,6,360,502,645,800,855,964,73,812,494,375,814,575,348,920,266,438,570,590,807,246,670,467,920,55,410,793,627,569,600,38,805,666,82,988,84,854,486,732,125,542,807,155,718,706,473,887,914,351,560,420,597,837,184,670,356,316,895,983,630,358,733,409,137,308,67,887,927,551,351,957,439,533,153,956,183,0,667,459,600,223,840,142,263,315,93,953,535,498,515,813,481,538,374,153,376,220,128,756,144,629,718,111,977,688,179,291,489,634,494,535,330,561,355,651,511,951,827,209,83,701,729,893,974,339,657,459,21,234,238,482,490,619,566,721,820,534,359,272,579,849,315,467,947,560,290,646,931,752,34,578,810,388,56,468,136,905,389,281,507,713,507,998,747,944,313,380,445,916,222,918,509,320,46,87,62,603,158,567,70,407,289,55,379,16,81,89,459,395,937,250,282,537,242,988,412,320,808,609,966,978,530,184,647,427,6,136,866,107,593,850,231,425,424,166,96,288,867,584,77,523,612,844,256,892,58,837,970,42,550,885,240,568,258,364,802,733,45,245,735,586,698,505,535,333,476,876,723,171,955,796,778,536,948,974,500,18,972,424,513,84,687,107,518,358,796,477,111,98,790,72,138,600,645,987,662,178,885,953,662,326,300,770,250,506,939,872,344,816,505,185,347,220,542,559,413,295,111,135,318,246,636,372,68,317,278,775,676,542,666,75,60,260,962,910,593,146,43,689,875,84,656,157,593,165,139,944,519,921,484,598,391,730,314,203,918,795,86,523,436,232,251,35,654,340,378,614,333,390,117,975,808,8,231,394,412,784,127,373,526,428,193,848,935,770,566,716,410,273,84,888,1,816,627,408,556,144,600,289,183,640,728,233,184,106,569,355,502,772,200,543,118,100,648,452,580,536,200,233,137,602,913,365,642,612,9,931,509,536,997,133,85,34,457,910,504,11,674,385,624,276,449,175,445,537,460,328,703,55,36,515,115,548,924,472,442,133,219,882,343,439,778,62,822,412,418,138,381,779,309,216,429,490,111,115,482,736,36,200,534,31,643,214,352,554,807,930,476,762,725,130,886,123,517,913,73,16,114,538,450,434,806,503,210,510,883,624,800,346,302,128,328,177,884,989,189,774,931,299,127,605,748,638,165,124,750,84,824,492,830,754,755,816,103,695,772,935,985,391,623,971,565,56,372,670,985,744,349,628,428,355,794,581,61,69,344,195,700,488,18,104,241,833,148,952,515,224,283,698,137,944,12,657,153,86,612,157,741,335,959,126,697,873,622,153,856,708,355,826,101,45,471,173,28,220,142,442,629,327,268,509,192,73,312,454,613,908,180,499,356,131,398,2,49,546,665,829,896,798,848,815,553,937,181,117,96,491,57,756,818,323,302,211,152,986,3,987,885,65,403,342,556,18,823,673,518,221,277,662,762,886,384,430,333,124,569,441,669,641,295,354,997,518,55,218,583,168,688,565,134,428,329,720,926,460,715,217,214,514,187,476,966,877,330,442,974,514,819,503,70,973,950,363,593,170,692,969,999,570,181,481,518,764,830,800,121,585,918,707,652,469,721,297,82,164,972,746,167,287,756,353,507,549,838,600,62,299,999,421,179,870,740,256,348,556,815,971,24,949,820,623,397,51,35,597,898,349,180,784,616,913,489,690,870,247,941,140,290,32,720,920,367,677,760,909,22,117,23,419,572,866,325,41,548,27,191,992,838,543,664,138,939,711,796,170,785,947,666,645,833,277,7,676,114,493,234,743,360,912,592,711,552,560,623,258,443,291,516,755,409,358,803,283,875,628,141,360,194,275,523,75,964,852,441,924,586,334,356,325,999,62,295,302,965,236,86,399,206,612,391,952,2,909,677,531,594,239,818,22,855,448,875,490,150,681,737,211,425,439,823,856,919,326,948,964,22,711,89,974,70,371,679,989,5,532,631,329,713,941,12,50,680,639,605,109,862,463,338,199,972,230,85,525,681,842,322,298,595,601,892,937,624,264,719,434,441,745,937,56,884,447,24,147,655,293,937,411,876,358,635,523,14,75,434,756,579,648,645,768,245,208,69,649,98,111,647,984,802,780,975,591,76,109,896,334,83,788,354,896,735,612,920,669,244,387,660,90,299,709,559,109,881,849,351,683,585,491,383,54,708,31,757,788,391,444,98,183,433",
    "60,546,176,425,433,659,544,323,502,758,262,771,520,50,509,325,414,4,431,329,967,429,217,228,308,981,458,21,751,149,258,311,195,995,502,946,681,908,787,160,443,569,886,848,515,746,267,884,816,153,16,372,372,918,424,438,85,460,949,98,839,788,662,699,450,900,249,870,382,219,10,111,975,500,937,422,832,924,547,950,178,501,107,193,147,25,134,79,790,77,665,697,993,366,607,668,32,812,624,684,826,417,742,219,547,928,863,124,925,753,52,332,399,820,686,117,232,781,667,0,312,597,977,269,38,42,77,513,358,508,510,497,482,390,911,58,245,336,933,175,585,986,509,648,83,254,332,271,223,400,457,387,570,38,327,560,237,489,426,98,43,584,389,573,111,257,642,859,933,630,384,283,771,431,281,360,764,240,112,976,466,404,89,20,316,923,876,808,554,596,461,533,393,710,885,684,55,597,717,961,786,545,521,966,386,901,996,783,226,169,905,366,768,460,531,969,477,284,603,396,994,520,375,667,473,601,329,94,674,802,972,314,976,437,170,631,501,614,973,830,845,579,120,448,746,928,21,318,361,522,48,853,899,545,660,91,12,374,905,67,682,834,273,233,493,143,834,293,169,133,708,201,694,565,939,257,401,45,624,136,367,566,296,957,34,890,259,311,301,927,387,786,610,571,104,597,463,517,104,557,693,577,387,691,686,395,722,550,620,130,726,415,469,371,934,695,958,867,84,937,393,829,994,869,939,62,464,186,465,558,796,262,802,29,518,15,785,805,838,813,484,995,893,195,513,608,362,843,267,102,680,197,674,46,662,249,166,360,873,247,272,106,842,197,916,479,338,87,19,576,477,481,338,99,67,917,548,652,249,613,686,660,335,382,124,408,889,265,123,815,475,613,32,783,316,950,338,547,399,485,675,696,488,393,903,731,547,564,821,292,226,925,256,51,536,637,582,191,968,352,997,961,324,458,915,575,422,880,57,636,36,927,536,757,665,682,917,367,658,753,658,332,158,516,608,186,884,735,822,273,559,356,393,581,735,13,234,138,860,750,507,120,456,141,790,682,843,343,603,810,116,535,412,438,697,342,264,555,758,24,693,472,62,801,19,830,292,948,810,798,42,325,621,520,240,541,246,638,332,906,102,384,811,380,462,215,455,945,666,511,662,254,730,801,950,86,155,906,238,589,503,471,493,392,482,324,64,58,352,39,37,44,623,987,251,65,168,714,431,165,668,973,987,796,500,731,485,90,159,210,478,840,623,541,726,310,846,545,670,973,641,58,391,525,285,154,107,557,391,937,340,398,751,135,161,161,56,79,68,289,899,494,665,611,474,542,436,997,326,457,952,553,622,152,418,17,545,847,220,996,74,283,78,697,116,533,142,765,553,340,903,850,736,667,543,893,642,420,193,619,592,342,451,579,496,508,19,327,687,659,191,221,993,683,47,316,91,814,980,542,873,127,443,748,363,838,409,112,591,56,248,200,847,73,561,263,180,912,320,651,327,194,427,308,331,741,658,918,985,571,213,34,984,198,392,806,329,547,299,158,728,773,10,262,625,687,260,187,706,537,679,777,69,63,914,784,845,531,171,831,372,297,383,925,722,117,793,657,364,455,120,344,907,299,225,326,299,886,908,277,99,712,895,684,261,644,854,151,566,652,620,3,758,555,454,867,69,66,384,214,252,288,812,703,745,929,599,420,431,839,324,784,576,673,398,208,425,477,285,500,565,941,929,890,783,969,621,150,940,256,554,752,440,379,636,597,838,134,269,770,307,162,263,466,617,291,378,148,323,889,357,779,153,919,197,871,908,746,124,405,205,162,209,3,84,509,626,463,254,444,272,323,580,26,213,895,37,77,240,216,350,835,179,341,577,787,219,810,304,993,575,712,978,528,21,768,322,746,48,979,553,798,183,946,545,767,719,931,977,930,904,344,487,690,845,666,151,875,873,237,232,409,743,944,24,788,155,55,332,928,250,682,9,833,482,140,314,71,11,146,480,682,376,317,37,428,384,811,846,524,29,47,980,797,430,455,322,845,446,911,683,998,547,982,99,627,745,228,97,809,979,648,35,548,217,103,264,991,172,272,819,638,351,898,187,357,393,534,948,64,319,677,830,927,319,74,720,397,496,732,763,671,72,664,652,937,36,3,763,106,197,453,601,35,923,617,694,230,500,379,62,661,343,875,924,678,46,41,135,156,806,426,26,197,205,775,285,475,692,293,125,237,253,905,281,93,448,289,973,341,705,261,980,617,421,324,362,307,403,387",
    "588,124,190,684,689,288,434,98,390,301,678,828,52,816,225,958,824,113,306,172,818,978,305,708,855,2,356,734,162,858,854,587,672,598,698,633,744,702,946,119,210,155,947,981,497,643,616,684,134,324,956,879,734,173,496,318,919,845,202,706,453,722,317,985,275,47,966,635,239,810,958,154,135,752,857,966,931,55,784,685,980,810,414,722,447,312,612,25,317,465,704,835,193,127,591,91,76,48,452,340,904,38,16,739,745,831,901,461,242,166,182,92,236,122,870,569,426,719,459,312,0,723,393,838,654,968,431,402,678,605,56,110,64,492,615,27,187,135,42,207,8,848,785,751,838,49,179,504,884,110,481,559,697,740,69,146,130,54,469,978,282,845,96,847,216,844,30,398,196,35,576,494,833,368,541,463,585,480,636,243,925,821,452,663,618,973,320,446,900,198,290,56,379,363,901,928,604,741,633,982,691,968,410,997,381,160,85,882,521,603,630,860,381,2,233,885,242,352,518,315,688,992,604,590,5,859,493,328,467,596,815,274,265,326,154,360,675,577,727,661,471,176,775,952,142,597,885,576,492,517,424,218,694,110,439,565,791,212,856,654,710,793,38,533,959,716,483,478,235,123,21,668,200,636,134,300,756,929,884,185,701,903,60,626,803,811,841,280,382,621,399,5,922,866,725,880,712,22,108,424,221,412,591,59,857,792,885,858,629,318,488,706,426,29,105,273,665,691,259,413,37,969,333,958,946,370,394,2,448,781,709,871,179,195,356,859,293,348,448,323,325,259,645,545,255,716,746,631,216,304,947,727,304,869,263,63,671,238,259,133,589,268,964,831,822,189,479,913,935,2,537,29,708,783,628,140,23,426,924,885,350,816,380,52,60,566,765,626,405,135,211,453,659,853,654,431,607,679,597,770,828,760,626,861,77,659,667,41,194,14,905,932,856,997,677,37,140,922,677,890,171,636,475,757,596,156,33,950,617,349,227,338,2,278,89,916,931,728,51,532,741,87,856,893,850,993,924,3,993,860,743,440,225,462,431,369,155,846,651,246,614,435,148,929,176,951,366,176,572,504,747,132,354,359,794,52,897,339,921,566,735,47,72,336,14,178,358,805,576,188,48,188,928,861,973,579,727,183,834,815,954,236,200,298,873,198,587,750,11,973,209,553,848,565,609,858,210,823,578,636,247,123,213,911,629,21,275,890,214,19,217,555,672,123,752,107,138,426,934,696,584,56,220,568,868,283,739,154,430,343,337,964,998,111,889,305,901,694,446,509,207,574,758,389,415,503,403,377,681,849,826,864,637,936,642,559,79,632,819,400,722,733,612,716,274,608,359,240,389,392,410,700,82,662,255,745,99,492,70,612,645,585,270,255,769,870,520,139,259,808,656,663,699,915,524,252,899,479,400,556,982,499,303,199,548,645,887,655,508,483,103,590,693,66,30,255,716,774,677,190,519,419,180,3,740,342,79,695,315,203,866,685,348,998,637,704,462,425,56,441,461,873,803,449,633,826,757,156,208,588,781,528,683,40,699,153,483,446,378,370,779,641,718,636,936,424,103,925,336,608,754,682,936,142,770,722,227,701,479,363,44,811,799,256,391,346,493,55,364,991,680,700,418,844,737,676,981,771,28,786,745,732,16,86,900,936,998,507,493,221,927,409,725,870,659,611,111,891,161,352,227,706,157,181,716,26,551,576,865,888,331,839,185,698,138,505,125,895,171,185,7,426,108,78,766,834,109,242,944,88,514,25,489,4,794,336,286,420,467,88,303,117,985,717,159,503,402,975,473,800,354,376,389,617,363,642,917,571,631,899,445,373,887,166,96,820,50,263,185,433,277,616,651,159,222,910,928,185,121,356,253,631,619,592,917,453,609,194,846,550,821,305,206,215,292,968,503,985,638,989,63,161,965,816,411,476,584,596,77,418,891,918,890,502,226,740,571,210,827,707,101,828,166,992,467,353,658,493,872,822,979,152,579,966,323,341,196,140,235,718,210,135,384,8,846,584,164,375,446,312,207,317,546,382,85,202,827,142,667,864,899,639,809,129,714,103,699,332,730,732,152,179,770,956,134,828,526,285,768,409,80,572,306,621,260,158,111,549,400,467,127,672,353,754,827,24,397,60,405,126,941,226,39,889,687,276,680,477,46,820,842,924,255,964,469,777,616,692,380,122,537,709,638,721,50,474,202,327,250,919,598,748,812,523,286,785,774,463,371,133,417,251,951,24,216,735,707,639,517,251,447,906,940,869,300,23,163,74,517,560",
    "87,719,881,790,677,987,865,161,842,431,553,645,952,160,150,922,800,969,133,337,424,814,918,664,342,372,211,858,227,617,738,663,191,567,262,363,702,707,953,462,805,380,37,544,80,244,256,279,606,801,229,364,723,825,669,999,591,710,442,85,376,477,630,414,955,848,667,749,354,794,238,596,330,158,429,637,887,218,202,330,278,715,204,420,276,9,181,995,432,983,643,697,878,331,828,656,62,528,246,257,208,274,448,888,739,670,346,641,201,827,973,357,695,158,199,584,379,562,600,597,723,0,297,730,890,770,180,605,699,195,299,301,38,756,516,733,610,432,590,883,318,471,770,820,952,545,418,663,525,399,490,972,261,605,106,913,457,675,97,236,594,116,700,208,959,178,144,167,393,638,964,890,248,915,746,971,441,876,11,218,770,410,850,386,502,938,263,92,413,271,179,252,656,230,650,913,609,136,618,970,904,792,80,622,722,257,906,434,625,858,71,363,615,437,461,953,941,119,618,70,139,64,844,537,729,538,852,14,531,489,518,635,179,941,5,217,150,250,819,957,693,333,423,209,385,375,820,310,770,851,409,102,743,535,611,958,330,59,75,99,614,547,660,225,577,487,247,820,372,682,687,207,583,378,44,744,287,78,543,983,170,138,265,486,381,485,107,324,655,589,744,294,163,689,910,919,338,387,21,613,926,297,412,116,516,481,947,220,78,619,226,56,693,555,570,673,158,350,568,775,154,476,789,946,437,204,465,329,706,334,100,930,488,223,534,204,843,31,888,789,691,455,721,895,741,248,546,307,88,572,69,336,951,638,432,773,299,664,146,204,104,405,243,652,866,350,466,416,590,317,652,646,288,422,84,780,609,237,177,992,583,959,416,994,630,112,657,807,894,945,76,362,600,787,842,847,793,816,940,890,278,625,811,64,450,279,936,223,229,201,7,305,674,765,791,42,945,439,466,346,733,285,310,232,959,270,260,392,863,206,754,991,468,157,828,546,370,801,635,82,50,735,411,133,653,412,931,814,447,991,87,907,41,132,503,45,836,691,439,528,796,750,702,844,853,36,253,699,367,483,316,655,647,764,882,933,768,775,776,975,721,863,141,677,484,725,470,127,471,773,531,512,546,587,699,561,195,845,261,594,617,85,105,592,948,723,948,643,13,313,362,656,107,35,58,975,87,872,101,344,38,346,98,589,34,917,450,837,334,166,461,80,357,948,611,930,778,116,484,984,737,776,567,22,525,237,102,210,370,521,169,504,141,932,327,194,49,94,164,833,529,810,822,983,95,727,304,707,146,167,48,733,411,48,863,438,802,315,657,6,243,430,556,644,285,184,813,438,17,561,491,832,983,898,947,862,947,404,592,339,787,526,854,836,11,621,405,61,301,151,531,355,328,153,465,466,684,847,261,453,531,572,110,546,932,497,682,381,689,990,359,67,381,665,668,970,793,937,884,780,542,445,222,175,547,509,130,693,394,845,411,555,565,936,590,857,981,999,377,365,814,202,244,507,289,668,372,927,809,571,301,106,742,691,785,629,370,39,100,413,816,685,318,642,237,294,434,384,66,742,185,256,510,934,178,390,177,317,639,24,731,537,733,845,923,714,228,111,103,843,529,986,13,278,384,171,177,89,284,70,720,253,282,466,652,434,422,359,128,813,286,664,150,175,687,992,661,300,609,497,161,826,538,336,522,62,510,695,523,589,342,585,636,610,602,742,240,100,61,814,472,541,394,65,833,214,764,793,302,868,131,478,973,331,611,29,850,296,894,92,157,718,617,975,913,473,881,102,916,67,880,138,390,625,237,634,935,334,827,298,403,516,812,157,582,829,650,881,402,595,844,64,2,623,68,783,699,160,154,203,484,598,634,423,811,661,202,125,865,368,886,663,9,466,250,958,132,889,242,726,323,296,346,666,327,50,134,567,470,282,717,908,115,716,383,679,645,773,54,768,158,345,524,145,381,494,739,905,967,870,874,394,960,947,543,272,849,553,86,333,674,324,43,269,846,266,840,231,6,41,19,82,490,160,851,253,890,304,182,133,216,158,41,430,397,678,235,950,675,8,226,338,408,212,238,271,77,650,392,394,89,914,433,235,807,944,789,382,442,969,933,859,724,244,505,562,71,48,670,129,78,892,748,834,242,957,273,424,368,777,709,318,61,443,337,764,187,32,370,565,475,823,557,630,822,864,132,879,662,798,11,322,285,479,839,270,10,483,28,173,430,336,485,443,252,177,73,365,490,791,55,648,175,528,198,644,139,141,838,262",
    "109,555,447,475,393,558,975,612,589,961,703,421,506,609,547,438,383,362,576,815,474,474,288,236,277,612,153,543,431,681,383,328,493,167,332,712,713,778,811,297,254,962,423,83,938,891,134,183,246,623,834,363,188,940,414,24,32,947,676,929,659,588,920,959,520,2,122,279,602,526,811,911,507,390,295,573,399,37,637,703,645,225,484,84,306,853,94,376,456,909,147,760,528,880,100,370,624,589,758,896,302,383,74,61,311,682,17,290,502,862,956,574,841,840,372,600,408,779,223,977,393,297,0,984,986,109,878,308,384,926,192,425,187,55,993,393,374,862,550,241,372,922,641,301,468,440,844,72,577,847,938,547,10,406,868,329,223,919,991,90,733,16,694,119,651,436,817,257,639,173,831,885,51,938,303,626,936,942,233,522,425,327,491,916,238,671,977,976,785,274,451,785,240,691,939,501,905,101,698,96,402,359,730,291,622,392,783,189,27,771,12,528,681,115,726,728,936,841,476,41,339,143,464,619,699,604,359,878,370,406,211,287,661,521,553,463,363,326,128,895,200,68,347,164,947,379,784,23,857,712,395,579,720,340,514,214,23,880,645,261,333,707,64,175,234,325,20,927,82,164,266,564,851,54,681,2,535,468,573,861,709,709,981,794,123,781,704,869,240,893,687,447,897,120,893,419,168,280,582,749,519,880,382,670,618,347,933,275,342,840,544,425,687,958,804,182,176,366,787,747,76,903,934,31,737,56,208,398,618,961,926,394,911,876,851,772,684,504,690,972,996,633,736,991,491,780,654,259,161,456,345,369,700,608,694,829,165,538,665,515,618,853,333,399,665,108,399,964,989,108,469,793,454,256,483,635,185,577,1,780,314,333,738,691,424,821,537,263,414,556,860,347,517,769,375,500,845,694,293,871,531,191,382,176,746,801,105,716,163,820,291,360,334,778,890,716,141,584,745,996,831,48,387,678,670,232,455,700,668,303,227,728,16,927,68,599,551,996,858,468,713,917,983,999,686,264,249,105,87,361,914,494,554,802,931,634,84,581,160,896,103,293,16,113,250,277,980,550,462,823,86,840,799,714,669,890,636,366,702,440,50,951,807,8,830,751,793,816,681,44,335,640,919,579,944,765,89,224,764,322,346,566,216,538,100,835,118,167,303,657,670,975,805,540,104,422,893,844,32,464,105,257,192,137,471,663,708,496,803,187,581,49,265,966,73,406,535,502,136,429,899,394,700,677,982,840,156,769,216,216,725,389,474,693,100,417,970,731,272,107,476,756,21,35,784,189,446,316,970,135,779,104,647,443,813,624,552,779,985,264,17,831,318,283,169,99,334,136,322,970,980,146,827,94,337,202,343,955,254,212,470,28,73,356,533,331,785,479,689,524,366,511,27,555,285,708,400,459,796,631,627,60,392,666,364,862,914,529,32,559,774,712,435,584,917,965,97,713,674,938,969,972,894,829,77,724,517,436,337,541,356,126,166,540,351,578,810,896,214,567,362,200,636,356,7,512,421,213,91,704,890,99,475,640,999,201,492,743,830,328,134,955,854,70,220,329,45,481,917,389,674,465,426,492,530,313,857,133,912,846,680,417,872,499,811,390,674,497,952,951,78,746,501,300,168,358,345,386,893,631,413,740,15,655,534,21,92,164,887,903,933,892,550,608,730,30,928,54,923,855,244,278,662,609,318,236,111,118,706,829,275,395,562,809,497,51,82,278,104,49,796,334,751,616,495,793,96,473,522,355,340,620,183,494,873,585,927,882,446,269,195,683,698,802,78,961,531,426,623,778,400,616,789,711,840,948,256,477,496,980,89,806,32,777,717,322,660,640,487,628,791,633,133,965,811,947,107,496,23,334,133,62,416,568,681,103,282,822,921,616,355,33,60,205,657,993,934,289,181,352,135,302,674,566,787,212,126,821,752,480,313,248,28,181,168,173,159,117,738,433,96,92,606,894,12,197,993,203,765,793,926,159,492,575,132,687,648,169,183,100,165,950,352,55,900,910,301,425,527,797,805,384,256,295,893,430,793,747,252,483,458,468,349,385,61,628,801,146,874,230,961,686,701,22,583,30,148,995,80,120,573,842,368,679,81,538,290,737,446,100,80,184,36,141,733,472,606,170,296,679,689,921,791,5,732,991,321,165,512,710,920,481,770,292,278,650,918,996,894,596,632,121,723,210,399,429,26,118,250,919,827,282,758,491,630,542,93,229,943,360,519,683,586,929,662,252,282,33,55,226,553,172,649,723,993,404,353,583,643,280",
    "473,154,530,48,767,590,997,366,332,245,270,60,658,225,619,864,176,489,436,560,34,243,22,756,978,111,743,945,233,82,193,258,975,307,135,745,805,893,33,646,580,718,766,625,995,93,286,459,403,257,673,408,809,300,71,584,778,883,163,659,954,432,798,751,18,564,267,427,803,213,705,854,471,648,427,314,544,66,276,857,722,58,251,185,952,623,721,585,387,333,560,916,800,761,516,845,864,534,650,774,533,66,761,31,236,316,599,651,352,772,741,269,854,487,646,387,185,743,840,269,838,730,984,0,820,71,615,595,871,917,404,610,181,698,794,213,924,499,125,967,222,858,255,284,569,88,423,611,237,35,622,210,741,204,564,677,514,945,65,949,687,108,926,184,889,458,617,596,812,187,606,992,903,95,511,926,151,854,834,288,732,770,810,400,110,583,777,133,502,752,715,494,435,492,237,25,910,736,420,7,802,145,788,97,965,75,559,886,348,159,836,802,626,146,650,55,593,727,148,623,12,156,518,753,958,489,7,819,31,836,642,499,768,368,703,185,6,457,382,648,954,921,379,818,846,206,469,283,868,424,778,397,240,765,830,865,168,319,311,245,348,548,478,447,813,631,629,412,887,494,408,618,338,246,584,118,258,657,749,259,489,281,991,146,488,931,316,353,28,671,727,58,414,620,238,662,560,955,351,874,344,579,121,26,413,697,723,676,676,169,940,549,907,416,567,207,917,294,18,510,719,424,32,676,256,453,993,230,800,787,209,377,36,241,926,609,579,623,108,322,601,524,854,872,466,579,950,192,405,915,579,282,100,350,543,140,835,205,479,188,614,443,927,19,760,988,217,92,23,432,361,284,668,126,643,716,967,667,655,612,328,936,606,63,386,49,50,238,609,577,521,938,921,982,973,674,483,566,58,973,587,396,111,556,237,403,545,765,649,172,240,569,592,968,955,481,869,375,774,790,620,795,658,941,805,41,425,541,317,893,735,843,603,869,920,884,912,633,308,52,460,894,731,718,738,571,146,182,866,160,748,708,119,14,644,306,99,264,98,973,175,690,968,7,39,906,921,316,227,413,184,799,550,311,491,23,667,469,227,910,128,861,413,907,827,701,733,558,162,729,69,66,516,974,362,937,8,779,312,898,622,594,188,430,158,159,140,681,952,948,953,159,640,244,808,490,673,506,351,565,6,566,142,972,640,876,471,541,876,471,687,680,766,895,896,684,21,936,204,443,53,115,411,771,996,340,423,216,806,29,76,565,37,688,731,361,224,594,290,531,776,370,533,705,772,927,873,963,458,118,122,939,691,613,537,668,841,483,593,639,231,335,796,816,367,324,928,85,950,168,655,416,837,898,187,782,980,724,167,619,44,676,202,484,751,10,508,952,683,633,904,56,675,923,858,599,70,297,961,109,865,569,979,989,347,987,451,955,749,855,450,367,44,934,37,741,518,238,9,355,415,557,264,829,193,78,984,360,442,903,593,967,230,713,620,38,856,809,837,643,660,391,765,762,427,962,32,621,897,973,382,749,290,997,833,448,823,643,321,290,906,919,584,963,247,507,309,365,517,712,520,579,640,258,816,599,488,43,683,296,294,796,774,917,525,781,110,172,753,533,240,939,753,766,979,329,125,618,642,334,718,469,295,381,342,798,750,665,536,433,633,681,56,361,626,478,931,459,276,203,994,316,171,793,203,733,556,517,674,508,170,960,677,314,155,378,189,394,946,339,245,420,590,355,982,961,786,836,451,779,976,16,754,171,865,207,327,348,673,756,458,769,908,226,205,674,473,275,726,684,451,3,310,449,968,245,583,106,439,818,149,825,336,927,168,195,415,340,406,772,751,525,892,938,289,944,342,450,207,355,155,334,915,121,298,599,284,726,984,224,281,495,879,270,810,882,459,557,231,766,469,668,265,818,51,29,920,823,845,373,273,221,518,452,262,825,389,954,180,647,46,987,184,852,805,884,37,470,885,44,587,643,783,500,279,418,716,50,863,239,792,989,158,983,26,721,186,545,222,668,262,471,974,638,571,233,993,63,39,956,767,585,593,404,285,429,795,38,605,347,866,113,333,245,732,283,398,783,33,601,635,210,467,282,960,611,79,161,391,608,708,721,767,765,386,730,686,559,856,457,590,976,183,278,101,64,594,253,938,122,236,608,201,999,297,332,550,785,251,357,739,363,150,694,878,973,144,444,264,337,253,645,711,875,307,933,816,102,392,335,133,381,168,297,915,135,519,640,649,129,400,302,561,55,886,937,925,483,101,665",
    "66,264,458,839,8,853,6,943,80,778,455,617,506,428,820,471,913,124,121,18,894,653,68,535,960,950,702,661,863,534,523,236,262,42,385,39,780,600,260,396,759,84,88,335,168,65,153,105,87,410,1000,547,72,552,50,245,886,194,157,440,989,56,343,196,365,416,518,186,676,423,670,126,831,131,140,101,122,929,389,106,326,399,413,704,53,171,570,491,43,830,761,841,336,183,323,911,10,466,335,637,745,553,258,824,698,251,718,990,386,128,474,106,30,72,206,533,693,353,142,38,654,890,986,820,0,496,923,457,750,587,494,552,415,281,131,76,601,27,584,308,641,686,563,504,331,570,807,108,902,239,444,302,582,713,297,771,254,358,844,752,708,262,425,661,158,630,313,868,382,594,472,226,718,533,557,27,515,68,455,513,461,431,741,761,283,477,841,235,475,478,700,89,273,595,257,879,181,598,379,927,12,577,718,78,275,950,983,18,246,553,966,612,237,105,639,926,901,480,740,697,18,985,627,934,971,339,672,768,104,388,179,155,437,784,362,384,502,764,762,316,786,982,142,162,41,258,233,468,300,326,208,893,74,815,536,562,452,876,626,853,188,98,924,923,981,187,247,403,408,933,780,706,532,628,48,66,368,982,745,154,512,346,907,120,577,436,806,765,71,100,526,540,50,757,752,410,570,820,61,21,498,766,773,386,514,32,728,4,67,3,743,103,268,132,714,554,48,856,120,422,509,483,843,763,68,957,735,545,153,820,257,769,774,693,41,863,261,57,956,641,892,722,106,878,484,948,594,132,177,832,109,37,312,402,480,696,858,958,559,515,673,422,77,22,541,24,407,226,902,498,628,630,308,365,925,668,643,870,383,745,477,146,754,97,641,751,199,675,491,277,515,94,214,119,648,291,647,308,573,461,904,724,959,608,450,973,7,300,998,959,281,610,348,272,534,918,385,350,618,976,333,303,941,160,418,241,857,877,258,428,626,859,795,681,700,247,108,223,822,943,831,667,915,102,425,756,480,981,522,275,73,861,195,434,911,415,194,24,816,875,145,309,148,292,270,33,744,234,437,184,759,555,573,628,575,237,73,697,42,220,565,397,215,40,270,922,367,737,430,848,486,528,941,739,726,864,271,246,417,315,307,230,783,48,589,844,294,592,540,954,566,163,190,416,860,181,969,588,678,675,95,289,866,686,193,265,763,669,885,859,305,319,553,991,59,544,492,117,665,107,356,833,211,956,472,635,132,579,934,944,689,972,633,84,16,489,808,950,951,429,699,497,36,708,35,323,549,640,730,251,497,208,994,753,689,103,5,247,751,291,602,493,477,586,822,415,29,972,651,150,989,528,967,761,789,174,972,421,165,485,269,114,751,2,625,495,725,870,685,581,553,182,110,565,140,992,566,738,880,213,946,578,343,502,538,578,380,401,244,387,765,771,993,681,37,914,80,1000,548,730,766,940,362,9,20,362,166,652,706,72,362,671,792,499,260,406,589,522,658,407,475,9,338,546,759,519,60,935,312,48,721,329,896,671,324,111,557,341,651,596,693,383,181,735,232,942,23,491,420,269,367,272,596,559,698,737,131,506,71,771,131,890,557,472,196,609,974,582,748,155,710,69,956,972,840,962,325,748,781,244,451,941,329,856,519,694,363,427,356,331,200,710,870,895,473,671,107,449,955,88,837,542,591,504,817,762,188,614,381,377,809,270,503,807,332,933,764,513,926,435,845,373,566,13,522,649,883,686,42,464,98,118,913,598,704,890,924,838,887,766,340,524,542,755,940,142,432,695,733,588,924,827,477,171,625,313,861,832,988,378,617,529,108,254,440,609,624,24,956,526,589,85,554,442,62,14,271,440,548,774,481,91,143,177,271,614,758,844,129,296,107,539,258,570,246,173,908,111,742,309,601,120,379,57,883,346,565,870,283,359,603,844,168,871,819,471,315,315,317,857,369,230,711,779,414,164,650,201,357,378,654,472,712,493,365,910,330,594,329,194,675,336,537,214,748,794,636,698,982,766,931,227,872,528,682,30,669,428,141,699,682,711,488,499,246,741,919,702,518,421,470,708,234,881,500,918,833,939,486,182,187,602,905,915,90,706,280,614,274,41,617,232,608,650,529,832,314,945,660,244,982,824,49,839,34,135,246,810,619,472,428,729,511,393,963,521,16,378,197,327,937,759,501,695,635,215,257,523,907,827,515,372,652,910,373,162,443,23,540,565,673,796,329,104,849,579,267,232,686,593,370,848,260,626,753,959,634,925,383,741",
    "965,486,382,592,904,878,484,400,446,830,894,643,736,208,705,318,630,794,85,175,637,979,526,434,916,837,658,111,595,286,879,328,485,810,23,94,289,15,577,812,791,406,400,423,764,314,300,226,782,506,519,884,896,167,652,895,204,737,722,155,346,8,492,538,803,123,546,250,250,669,228,476,242,809,718,900,653,995,409,277,65,499,999,221,490,317,928,481,974,101,639,691,635,238,166,475,438,224,626,763,66,423,206,327,452,846,962,579,906,882,423,82,983,499,769,868,517,546,263,42,968,770,109,71,496,0,188,882,148,202,857,516,659,943,349,928,944,876,69,211,210,949,64,127,117,701,289,364,164,257,423,309,899,608,525,462,571,892,909,153,933,50,639,518,299,564,957,511,775,769,915,10,230,810,227,358,5,289,670,590,171,301,885,50,794,639,505,956,980,212,932,628,64,876,270,111,733,58,27,854,781,532,994,727,984,641,953,826,7,391,331,221,908,230,515,100,159,443,207,745,535,747,415,467,186,940,799,234,667,947,154,400,891,664,266,717,631,490,826,423,814,298,115,181,804,149,251,938,92,25,41,897,795,945,919,789,283,484,568,848,446,431,723,812,150,646,673,589,760,248,200,653,690,958,131,404,406,113,327,204,472,847,668,503,992,682,609,349,432,665,649,18,17,68,936,778,164,633,384,851,743,937,263,356,228,668,900,55,757,881,419,422,789,408,344,328,330,577,896,850,774,423,927,644,201,946,436,328,305,627,664,721,855,143,819,720,469,808,673,773,864,90,170,485,241,10,528,735,72,330,87,560,812,345,334,889,831,396,972,211,933,813,530,168,741,646,296,260,160,107,187,707,67,390,265,458,310,384,183,137,670,246,658,663,602,83,120,367,324,686,145,994,336,653,412,376,51,860,410,540,54,172,368,661,634,590,820,453,344,160,646,402,144,950,899,627,91,263,686,893,665,6,261,19,312,428,703,803,666,546,312,490,368,591,577,458,129,598,127,846,977,55,684,687,47,917,450,141,430,365,402,424,288,739,579,517,162,987,559,199,780,582,783,480,951,642,884,626,624,449,690,289,895,707,320,74,646,540,736,645,50,76,155,646,6,976,635,962,787,717,669,308,933,451,212,764,110,350,605,759,927,507,148,706,267,790,189,618,307,644,604,788,3,518,149,799,358,855,765,867,909,557,115,730,559,804,372,650,478,287,547,297,360,491,910,916,439,630,699,99,664,487,88,273,241,692,798,416,604,518,865,193,202,852,883,442,885,823,799,371,442,626,289,286,182,522,209,235,778,281,852,425,980,848,734,323,411,444,11,661,613,411,926,147,987,206,805,698,807,60,545,599,921,469,530,955,616,545,284,10,602,820,894,921,947,510,948,341,886,556,803,882,366,774,164,174,405,663,966,263,263,139,930,200,21,339,87,95,444,84,835,626,155,671,840,963,300,849,943,59,617,863,594,369,319,506,294,383,401,277,265,680,911,913,556,168,348,39,282,647,496,318,225,475,63,840,764,156,783,167,89,749,925,945,75,16,653,622,791,723,949,445,960,34,168,822,231,42,54,163,413,780,77,358,486,486,391,606,420,623,379,735,920,153,587,826,523,404,802,32,685,515,727,656,21,398,613,232,623,56,22,903,231,618,147,499,144,136,842,181,637,35,304,918,890,75,184,728,683,754,512,997,920,96,855,75,962,885,730,812,817,798,290,400,920,589,213,11,384,116,395,210,154,735,758,781,705,3,219,590,102,761,543,474,922,172,702,661,569,179,167,363,402,28,191,864,80,491,888,963,925,683,569,713,843,944,382,205,173,553,112,915,905,688,620,949,980,814,563,888,538,57,99,592,412,519,120,706,217,371,366,237,631,296,665,696,939,769,346,524,225,899,230,83,945,354,918,614,246,959,952,670,501,721,652,643,683,8,838,419,950,247,653,852,388,385,108,250,347,500,550,366,576,937,733,856,95,352,506,152,44,194,131,129,434,992,689,357,460,171,8,214,229,231,510,842,984,163,468,960,395,556,468,606,978,316,509,634,27,803,988,369,672,350,358,623,339,490,99,56,727,959,666,124,451,352,823,984,998,844,653,426,395,614,397,114,319,633,903,721,313,295,922,545,129,506,262,199,497,707,831,553,598,357,538,260,256,448,389,513,330,435,835,65,38,149,124,238,173,192,128,466,796,581,74,537,61,384,577,552,916,707,491,610,437,758,881,625,152,602,195,966,257,16,797,299,379,830,906,41,483,204,569,340,654,779,237,978,292,90",
    "973,643,466,379,387,812,873,117,572,396,302,832,465,796,884,771,806,684,1000,173,605,263,533,794,283,566,432,486,353,183,350,991,958,306,718,701,988,228,802,944,370,524,867,636,656,731,546,422,754,310,894,164,447,83,776,739,484,680,949,281,859,245,957,556,861,103,563,216,62,994,714,11,905,519,152,104,990,123,570,883,37,384,953,429,554,12,195,95,629,987,63,315,871,600,448,459,780,599,62,547,746,42,455,195,747,960,884,564,612,161,392,916,56,655,275,299,181,274,315,77,431,180,878,615,923,188,0,673,46,622,240,176,919,904,570,481,732,645,523,543,208,104,978,565,903,847,288,641,129,580,47,781,15,577,98,421,229,464,942,612,229,784,319,992,163,242,874,248,193,242,810,247,578,795,214,789,426,946,186,616,651,605,524,796,285,61,44,138,229,636,897,574,794,178,560,574,89,114,629,494,927,823,314,48,746,717,843,604,737,489,96,674,940,979,913,839,631,248,939,220,146,569,936,849,139,252,912,645,449,856,750,801,356,761,487,411,178,897,263,884,954,430,868,237,254,43,150,189,893,676,426,657,997,601,63,523,575,438,445,429,142,421,904,865,54,772,767,38,838,977,328,812,139,108,284,803,469,589,132,675,324,501,152,911,707,567,286,531,833,805,155,117,259,459,624,166,671,3,106,35,88,89,277,97,505,286,480,181,338,469,738,145,356,438,267,811,989,612,888,933,716,996,117,867,822,816,609,67,52,122,839,880,8,707,471,723,174,373,232,666,597,532,152,838,881,875,289,86,521,923,145,599,200,275,743,270,833,58,496,790,483,814,741,216,62,755,918,207,600,616,771,867,307,130,830,441,377,177,765,781,473,167,935,503,953,470,166,316,510,304,549,965,267,684,93,44,432,545,57,228,193,830,48,893,252,76,134,255,463,175,203,816,701,398,728,754,243,52,569,764,81,291,714,24,51,786,496,228,708,50,32,8,157,278,720,377,378,169,859,619,905,214,166,479,61,995,865,166,624,11,334,154,295,45,152,158,1000,671,263,478,452,13,797,876,308,542,672,601,152,853,953,403,63,451,217,897,100,612,810,192,250,290,187,793,505,288,9,975,715,700,268,310,80,209,990,783,598,232,422,96,19,575,829,283,174,214,297,662,712,957,904,936,338,345,238,450,583,647,44,160,165,542,54,138,950,342,838,839,952,925,62,291,539,980,399,433,785,983,208,45,360,742,32,500,474,932,354,20,924,167,813,109,353,120,101,900,430,485,177,430,50,672,636,781,469,858,582,796,869,818,912,803,626,417,156,734,860,305,783,111,929,652,844,922,46,795,248,413,719,114,892,471,107,709,81,906,440,417,595,998,496,275,802,756,213,375,437,699,123,929,821,740,738,237,412,411,752,422,114,361,499,64,541,618,409,159,990,37,780,349,418,173,282,581,412,429,296,373,873,757,911,778,705,173,228,156,931,269,359,303,94,12,853,690,948,870,361,220,400,62,658,732,7,504,330,354,176,811,53,821,447,315,841,980,329,163,986,101,509,167,94,898,237,353,541,985,691,803,480,154,41,841,813,121,619,690,901,327,652,802,69,158,863,929,431,439,953,375,856,790,432,889,51,281,743,272,592,532,266,879,285,431,654,819,984,782,389,319,173,336,383,72,334,657,598,182,997,673,458,50,528,112,945,971,232,75,383,367,754,270,716,294,73,104,631,406,68,196,779,286,228,372,585,288,122,164,826,328,793,66,530,469,491,2,658,737,693,664,156,443,336,143,502,18,13,554,809,237,929,667,920,873,44,535,509,119,143,121,364,288,127,524,810,232,668,878,290,522,224,642,200,230,420,658,570,980,152,314,455,395,123,876,500,985,781,754,419,72,231,493,364,599,83,298,536,212,25,415,167,412,151,726,601,425,252,988,139,718,638,308,339,919,929,868,245,746,988,747,806,520,37,820,219,823,53,654,502,832,453,158,593,726,822,117,496,706,29,418,104,882,82,550,90,436,94,545,914,982,537,240,773,125,281,790,906,793,440,167,866,596,279,683,281,80,14,739,47,296,530,298,928,304,156,3,22,536,228,983,937,497,853,387,686,629,675,859,236,252,509,164,9,389,166,645,219,325,868,662,259,500,747,852,666,121,304,697,918,255,797,750,387,244,577,387,423,183,613,567,893,103,761,867,654,709,925,576,414,42,983,842,505,666,965,105,105,976,287,725,662,155,501,530,722,598,424,159,829,45,937,957,302,39,794,441,613,101,730,311,366,735",
    "950,437,65,950,965,337,206,683,19,406,697,489,73,654,841,14,788,376,477,602,184,732,934,929,517,659,64,209,53,396,895,407,683,720,503,698,740,458,76,339,965,90,422,969,54,612,78,140,184,232,195,152,560,102,781,641,677,289,623,849,269,347,707,593,240,955,401,397,252,947,442,181,866,209,763,888,302,185,620,39,686,883,372,307,460,502,712,255,652,999,46,676,241,919,137,487,2,96,874,976,527,539,809,171,342,979,850,710,573,345,501,476,244,605,364,377,863,952,93,513,402,605,308,595,457,882,673,0,668,449,456,198,853,327,505,808,227,164,468,273,931,342,625,285,616,668,9,595,209,470,701,915,234,747,74,649,839,414,913,814,331,518,162,931,854,853,7,661,6,199,269,628,269,174,481,429,962,549,821,916,309,857,341,305,937,231,948,276,84,161,309,620,94,129,725,161,419,210,787,753,609,681,691,199,186,220,466,544,242,256,709,220,676,416,905,47,62,72,838,724,743,434,930,355,584,553,277,95,738,828,322,977,552,285,495,815,987,586,606,751,635,115,822,915,791,984,713,469,72,269,740,611,588,724,19,798,251,641,219,170,17,130,467,797,497,909,70,262,139,976,767,833,744,778,894,496,870,393,826,324,824,623,666,18,393,100,459,198,103,620,765,891,799,419,456,566,602,664,334,611,82,450,583,312,497,719,602,462,480,714,767,135,596,23,109,255,2,893,113,294,500,967,640,867,470,101,439,947,604,27,998,356,941,129,14,259,68,444,656,241,829,919,679,382,34,904,934,332,117,275,837,249,534,51,109,208,572,454,298,576,293,413,492,500,210,762,782,460,67,485,337,977,426,442,385,5,833,834,119,102,408,124,28,948,601,158,711,382,853,802,259,433,368,478,287,933,109,450,438,636,159,461,554,703,813,819,533,670,716,145,467,406,927,889,641,230,817,355,110,901,871,193,346,5,716,856,107,55,83,941,344,438,148,59,282,366,130,983,906,182,799,729,15,880,650,106,396,179,185,829,126,708,437,397,151,151,664,847,220,96,913,267,949,9,59,790,482,193,104,72,456,796,152,872,712,280,685,194,911,418,699,535,309,36,536,250,77,521,263,85,671,458,949,945,914,219,662,274,340,851,18,429,950,26,533,295,712,527,559,138,636,768,508,642,760,60,194,912,135,104,662,613,935,793,21,563,325,996,38,413,15,624,788,598,58,971,50,792,942,463,700,945,987,628,324,7,202,63,457,173,563,939,459,561,263,155,803,338,715,10,627,771,11,846,377,235,501,689,16,522,249,10,902,172,836,874,613,273,662,38,950,168,942,329,340,321,47,513,15,673,216,767,293,64,212,27,312,260,348,978,944,718,599,779,334,330,272,513,651,251,605,282,781,960,181,978,129,435,568,436,886,958,60,268,304,560,202,532,151,97,962,539,320,866,673,346,43,488,115,177,896,598,196,902,719,261,642,580,52,168,762,807,362,103,442,547,47,543,472,423,492,298,610,429,122,463,360,749,887,910,33,314,927,404,873,4,905,965,945,401,336,229,337,181,259,329,65,59,977,728,334,58,123,883,409,847,462,343,453,128,892,19,483,718,470,60,936,414,543,509,898,70,284,345,395,280,923,901,339,323,338,875,972,736,974,41,757,275,761,884,136,473,427,620,340,565,815,502,325,113,387,342,519,460,481,30,2,151,89,929,597,86,449,455,274,26,399,785,303,182,703,99,131,784,332,108,62,140,467,883,161,222,902,574,83,332,834,427,241,222,513,559,871,351,526,430,562,615,713,364,856,981,778,115,407,145,690,759,892,559,916,469,101,875,253,952,444,327,407,669,313,361,954,706,429,680,88,124,467,498,629,183,809,270,699,60,478,17,874,33,73,689,25,973,168,127,199,193,729,365,651,220,6,487,63,909,720,222,224,944,143,247,432,457,700,147,412,682,466,548,622,947,302,419,24,299,641,922,733,714,175,327,806,662,63,68,119,367,143,861,915,367,73,162,316,782,534,509,743,631,993,207,843,440,536,473,120,925,720,684,486,691,627,239,899,353,138,475,96,694,691,354,874,542,609,180,777,100,192,909,105,674,266,635,745,189,555,259,151,389,965,126,911,560,326,862,457,109,728,866,578,361,458,794,872,69,428,856,393,479,865,61,589,989,170,153,523,544,233,761,35,227,464,788,717,273,591,159,810,220,4,835,965,514,268,590,220,938,279,994,304,11,72,963,609,633,669,884,607,139,171,54,619,833,919,215,135,473,712,233",
    "952,828,16,990,540,210,860,948,331,618,499,455,739,70,302,724,267,808,79,66,616,864,201,188,545,23,251,496,908,404,550,19,395,862,671,464,714,223,176,6,916,82,458,945,18,946,996,800,906,569,648,363,117,234,804,503,123,655,425,927,978,926,194,310,133,907,291,365,832,977,13,115,736,440,75,790,450,976,522,510,906,551,757,677,187,355,471,202,105,79,981,17,462,682,130,885,571,830,638,338,775,797,448,986,375,189,403,252,908,527,610,556,560,493,912,929,170,546,953,358,678,699,384,871,750,148,46,668,0,158,337,173,601,624,124,539,513,133,320,77,748,916,520,56,465,905,124,842,97,221,369,151,316,4,371,616,765,236,376,559,115,508,925,135,774,885,266,644,648,299,977,805,454,553,243,671,25,892,862,757,817,629,932,248,62,206,265,800,233,996,765,140,301,90,950,246,88,112,836,883,128,709,699,787,244,92,41,624,536,359,244,440,893,562,857,708,20,459,290,153,642,533,223,951,795,464,817,345,25,952,894,295,811,662,333,819,76,543,95,741,844,845,163,123,662,448,457,439,226,188,267,339,227,379,18,232,886,940,814,433,847,225,812,896,353,881,755,482,483,278,273,350,180,311,206,536,765,181,371,820,961,168,770,571,204,707,396,146,26,364,546,642,291,590,629,390,77,452,656,267,572,682,262,48,839,288,920,536,284,107,757,582,115,661,638,924,591,432,629,277,237,590,181,948,603,851,57,547,866,492,990,591,371,230,768,399,212,319,369,934,954,859,948,371,824,922,527,382,218,401,425,751,953,673,36,659,30,779,357,104,930,278,246,200,423,413,681,583,689,18,776,176,891,664,449,782,193,741,251,655,345,52,102,881,606,344,901,741,297,778,555,822,246,9,970,446,614,778,750,503,87,246,467,242,644,543,686,341,480,983,51,840,628,956,845,510,257,915,22,640,255,279,749,248,712,90,711,514,174,237,259,774,923,38,687,25,213,619,652,701,973,126,714,995,54,328,815,6,714,851,382,266,430,221,737,75,942,398,658,993,261,388,579,798,557,934,73,26,937,38,9,214,360,55,443,801,558,875,702,483,716,350,684,800,370,348,601,219,386,393,12,196,542,655,503,84,657,634,731,675,547,568,260,171,12,72,967,935,788,528,755,724,274,677,49,859,516,611,758,604,836,811,601,728,954,606,121,412,375,629,82,782,533,97,708,32,345,758,697,310,686,746,361,98,139,85,527,655,212,160,609,227,889,541,657,678,316,413,501,282,660,566,942,864,868,834,272,211,152,945,402,102,221,249,477,659,415,82,455,48,217,531,962,403,650,689,685,475,287,662,171,421,899,474,633,373,545,317,994,484,554,12,821,651,569,288,262,255,577,550,387,795,970,189,512,739,747,352,721,459,8,413,372,292,90,180,969,677,371,973,636,421,385,188,501,104,531,979,111,319,115,379,833,754,910,500,621,228,615,175,204,176,914,377,727,628,952,507,3,404,461,55,72,836,946,939,732,367,151,456,375,600,701,829,63,901,856,244,831,65,761,750,802,239,772,913,279,605,718,768,999,483,875,569,664,260,387,552,293,501,306,681,738,956,978,451,697,617,898,495,584,258,664,292,389,358,766,742,189,345,20,207,67,894,633,735,494,150,383,588,359,776,814,947,747,148,996,665,398,276,287,270,173,881,908,165,876,283,421,216,541,95,575,113,309,902,995,124,891,223,426,600,989,205,804,491,193,48,466,11,977,82,458,253,240,935,966,316,675,643,949,48,307,853,171,397,141,609,730,617,222,135,569,240,560,662,366,54,290,912,997,101,84,359,991,576,341,971,534,876,72,517,84,272,824,947,879,477,548,548,604,746,515,489,629,129,519,562,441,954,962,632,86,311,674,682,426,916,194,919,329,877,652,133,612,181,654,943,527,833,31,698,273,655,689,635,313,560,519,677,374,519,703,416,119,176,954,728,498,338,122,475,25,559,387,342,432,883,842,74,450,79,312,629,306,262,415,22,763,529,356,517,944,718,162,310,978,829,671,323,909,99,458,896,572,665,713,119,436,421,976,411,258,663,880,813,68,706,555,591,136,950,456,920,685,499,637,562,962,836,142,870,476,165,401,854,908,158,309,258,800,554,214,856,918,247,531,960,258,688,967,316,77,774,655,396,604,148,766,484,180,928,81,958,476,584,326,311,100,945,384,611,418,448,107,75,149,43,961,288,202,866,86,585,7,16,155,649,107,687,377,412,67,64,959,49,509,959,313,103",
    "492,729,85,980,463,805,36,673,830,552,126,24,56,363,64,71,26,848,861,193,844,214,466,760,291,679,741,426,645,986,279,514,507,97,562,121,88,735,417,665,684,832,686,865,108,72,117,934,604,213,694,644,172,572,494,469,15,466,479,539,470,80,475,321,74,745,993,393,979,485,590,749,810,874,227,456,827,19,321,238,973,601,982,593,840,690,697,438,532,324,298,830,731,254,777,690,268,866,383,595,389,358,598,36,839,62,279,37,36,975,286,793,164,691,771,278,443,371,535,508,605,195,926,917,587,202,622,449,158,0,13,146,288,260,842,352,144,636,590,893,654,669,967,99,851,342,578,178,628,81,484,353,972,451,695,557,250,962,228,222,161,115,812,288,467,414,65,815,470,739,834,894,627,929,588,74,559,881,227,739,163,849,609,825,343,733,585,489,916,101,344,899,159,210,976,482,574,233,865,149,330,257,659,510,262,222,203,594,96,776,100,405,397,861,480,342,41,177,995,42,676,68,413,856,814,522,745,835,394,416,986,685,753,828,375,991,153,776,972,424,512,247,642,277,341,116,24,789,240,963,319,273,560,942,502,181,568,189,5,815,430,192,224,182,617,204,10,560,885,758,833,848,774,885,15,76,630,92,963,850,356,100,939,506,699,426,304,167,907,213,650,23,969,757,330,434,596,830,389,173,27,482,394,684,363,875,835,326,113,855,840,692,2,844,299,640,532,834,296,54,106,906,899,434,429,521,374,810,474,239,226,574,463,892,844,743,251,927,592,660,883,560,115,926,943,748,925,922,508,26,413,389,867,12,536,561,29,266,605,656,941,842,751,264,500,820,106,325,959,9,706,432,16,583,795,827,575,561,351,467,5,121,477,355,635,540,326,340,865,809,530,480,743,490,383,600,447,984,604,129,717,243,695,14,889,392,804,663,795,156,877,675,777,649,98,460,418,305,83,686,238,647,679,664,226,869,839,159,419,585,812,525,221,141,186,230,288,304,784,751,322,810,679,27,952,368,336,764,768,66,413,191,501,390,757,132,895,914,941,130,737,123,112,648,629,817,191,458,378,110,98,255,452,744,364,153,598,93,353,184,677,397,845,891,581,169,647,646,173,238,282,331,740,310,22,453,826,35,51,583,32,622,684,155,372,699,42,150,795,556,206,935,317,879,935,640,188,44,49,205,113,266,957,63,32,444,928,17,286,845,980,788,239,325,134,331,702,144,198,134,410,690,104,641,533,354,46,614,81,756,254,208,771,44,949,532,235,228,700,853,721,698,468,150,477,19,249,730,734,744,271,266,754,568,121,896,435,121,446,283,833,599,448,273,375,343,233,807,317,33,350,90,222,731,265,900,852,140,937,842,629,826,151,642,940,479,912,801,444,838,779,829,721,678,897,631,60,217,823,253,716,519,178,180,325,388,33,915,359,793,899,204,398,222,306,948,229,81,753,61,423,401,748,43,194,389,530,702,28,90,330,537,285,570,783,119,546,547,758,844,547,604,470,274,434,967,744,86,155,783,635,111,847,386,12,186,646,183,190,888,784,38,696,303,940,513,589,791,681,916,154,955,308,751,192,834,282,622,993,967,152,14,919,184,165,303,521,170,617,175,500,93,67,405,603,417,49,692,140,75,403,878,950,555,399,466,699,532,859,991,474,893,819,203,844,40,661,114,444,321,898,1000,544,234,131,478,119,524,600,584,94,639,354,161,657,16,933,899,426,958,349,358,512,40,984,652,673,785,572,696,49,993,531,925,570,679,632,384,338,606,809,519,69,844,778,672,316,729,634,901,807,42,139,715,788,380,857,781,455,853,316,812,695,515,122,179,757,338,241,494,342,349,476,584,156,446,142,798,667,830,286,430,725,920,767,369,790,633,270,589,496,195,306,989,881,242,271,361,459,946,963,503,985,429,19,916,974,929,497,333,97,982,319,493,424,75,988,905,594,808,774,720,247,3,813,137,908,652,699,714,393,757,701,44,554,534,962,330,402,598,115,208,561,16,870,614,174,417,882,623,853,925,767,467,415,232,799,720,279,478,58,960,142,942,59,451,708,329,589,303,688,53,771,168,482,622,510,892,792,999,206,740,125,366,571,763,252,582,192,521,935,929,725,154,119,136,520,731,300,746,382,572,965,68,622,372,940,760,365,392,369,679,193,84,399,708,332,958,785,738,255,485,687,6,559,597,366,356,866,726,577,185,996,570,951,555,475,182,156,912,408,187,638,531,31,768,885,871,753,345,638,586,856,892,307,487,612,530,805,641",
    "624,571,21,716,362,262,171,784,440,777,746,754,52,657,47,932,608,527,968,368,769,950,52,676,856,344,105,686,710,517,448,730,85,952,45,611,928,211,725,240,499,16,588,187,515,363,304,827,751,115,439,879,75,23,628,160,6,716,669,547,738,734,680,361,51,929,535,108,322,883,889,811,667,521,912,590,832,405,197,516,31,706,354,506,849,561,55,433,479,44,463,722,636,11,403,564,41,233,431,170,528,469,238,744,784,819,379,835,801,659,866,697,300,106,859,591,486,160,498,510,56,299,192,404,494,857,240,456,337,13,0,980,246,430,524,61,345,710,992,762,667,181,65,117,239,768,964,297,649,838,340,572,122,939,84,232,807,368,162,602,512,336,37,422,585,896,410,562,153,696,205,341,827,15,690,698,189,419,65,210,704,677,102,655,909,187,128,438,255,569,929,928,93,965,543,262,587,331,814,365,607,110,80,271,182,723,845,635,62,25,853,997,710,73,114,701,779,360,149,839,613,298,386,425,369,67,475,739,794,330,627,49,173,780,726,654,688,157,969,150,833,804,556,851,202,709,521,639,866,556,616,835,70,282,609,988,44,207,201,126,519,874,116,947,78,999,367,985,733,432,102,170,303,36,899,328,633,374,427,89,686,224,315,215,944,269,818,384,19,179,236,750,938,102,569,304,185,31,290,275,260,345,906,368,983,784,629,318,937,976,449,649,605,110,869,369,513,591,290,889,640,512,759,946,211,92,329,216,148,886,862,87,353,456,599,621,220,705,161,403,782,238,178,950,610,144,797,980,384,406,470,299,115,857,770,758,31,511,775,152,294,219,579,872,981,91,27,899,923,118,87,259,355,824,994,360,235,997,824,377,65,374,755,486,731,213,511,522,913,118,966,138,573,246,747,504,899,164,270,241,100,624,296,307,942,106,552,513,753,910,100,810,584,987,885,88,87,301,479,818,78,551,978,318,435,401,959,535,129,419,640,80,981,810,665,215,330,457,86,609,549,491,677,271,725,937,279,709,55,377,416,738,345,919,390,37,929,26,209,229,435,292,50,929,694,418,11,664,699,651,805,720,787,846,279,82,595,197,688,766,864,294,946,436,978,503,223,504,819,202,376,285,136,880,145,703,518,1,74,5,150,343,234,613,964,819,528,376,584,387,444,56,593,216,660,890,845,605,513,915,4,308,916,787,246,976,2,201,359,259,961,378,436,836,247,461,580,455,59,116,864,138,275,159,325,705,594,698,364,385,142,799,349,263,457,827,501,48,501,221,383,581,853,197,872,395,653,838,806,190,322,406,169,578,346,343,340,503,348,672,64,296,818,340,961,630,547,922,725,722,966,92,329,545,239,46,775,774,998,238,822,943,602,500,821,493,245,263,462,471,971,295,335,548,779,920,11,467,262,170,23,26,302,821,142,792,284,510,447,903,928,790,918,761,376,592,325,967,856,778,616,172,421,900,48,691,903,219,498,334,213,921,628,592,257,739,781,226,508,460,849,933,234,326,922,588,666,228,433,905,357,1,927,724,960,281,280,834,644,24,417,48,455,407,221,419,74,547,921,391,939,326,640,736,453,976,941,406,814,290,186,171,282,617,833,810,215,481,369,104,59,924,823,601,491,405,246,320,156,301,55,583,6,480,563,288,773,862,160,369,364,851,269,254,549,744,438,948,986,979,505,609,930,427,841,472,442,168,835,43,316,859,520,150,838,92,37,805,314,592,734,357,610,339,410,284,624,904,85,263,644,792,478,832,981,799,305,6,671,860,655,991,774,695,889,180,362,230,239,831,809,572,344,771,297,951,381,377,215,361,852,892,389,477,494,304,781,757,432,442,134,296,320,501,983,304,768,551,856,450,664,99,785,748,3,725,483,555,347,176,90,584,973,840,701,913,260,761,907,404,968,382,461,498,780,570,97,788,421,136,329,839,468,645,945,473,244,895,746,660,941,446,867,997,263,512,468,588,808,835,435,233,308,984,854,791,496,76,662,137,637,882,520,968,425,482,616,713,248,858,318,712,411,829,432,801,748,814,684,150,394,994,268,530,980,268,194,158,411,343,272,459,693,808,400,956,685,239,189,885,759,995,410,907,90,891,947,349,516,562,598,90,689,233,610,593,913,125,688,784,957,296,952,951,140,721,340,341,889,855,881,418,741,772,765,990,293,548,758,667,390,481,679,100,435,144,643,552,487,235,89,816,178,241,569,570,635,691,861,879,23,603,111,444,509,867,951,635,511,950,564,14,632,230,533,296,86,404,495,808",
    "810,950,223,332,112,360,814,636,585,542,585,444,258,29,376,259,719,102,54,223,503,763,814,505,770,654,54,140,241,112,194,871,477,543,712,694,493,401,730,906,760,679,502,890,153,993,54,590,336,760,448,307,350,285,856,448,986,767,902,255,815,78,990,985,872,297,876,219,16,825,931,482,822,937,275,945,159,169,755,839,31,37,706,372,356,125,926,642,197,38,24,918,777,112,14,841,91,594,294,57,165,925,507,724,641,128,730,334,153,568,775,565,750,661,85,566,90,283,515,497,110,301,425,610,552,516,176,198,173,146,980,0,600,714,530,944,641,687,502,725,140,31,842,557,928,211,112,379,169,714,459,803,919,840,240,945,894,972,668,579,28,233,889,151,412,207,470,216,105,31,972,755,823,82,34,816,536,896,395,844,955,154,42,432,318,759,240,411,132,197,821,255,172,736,192,145,212,796,457,336,772,678,620,556,50,621,883,595,816,396,176,935,993,138,880,613,331,197,582,798,165,737,607,778,686,730,972,719,258,10,893,417,136,841,505,397,918,270,381,425,63,217,260,838,925,923,439,435,538,72,858,379,461,351,522,905,719,967,773,821,839,598,144,981,425,815,93,740,297,876,608,160,331,971,657,247,362,889,416,393,742,356,256,351,921,965,100,769,969,71,373,696,632,825,163,163,204,398,325,105,187,905,432,713,488,793,611,173,336,380,542,147,33,417,886,997,556,209,322,363,986,466,729,953,596,269,45,656,590,192,464,39,175,809,836,296,490,434,426,687,142,270,651,975,148,420,749,572,845,34,492,550,950,355,944,554,92,687,907,249,468,363,592,983,378,552,460,25,349,405,448,824,806,261,326,87,18,263,270,946,233,749,829,474,297,320,339,59,467,260,328,937,201,355,725,724,870,880,555,180,759,202,762,441,855,389,867,874,349,923,405,808,660,21,185,805,755,12,909,630,209,631,791,326,187,532,900,631,965,578,188,341,315,65,606,811,290,561,450,222,595,14,364,837,118,634,860,321,250,454,766,478,932,617,264,477,277,692,254,256,249,682,654,106,515,380,12,395,984,323,850,479,847,621,865,688,145,148,35,406,806,409,842,177,18,151,730,243,189,437,27,141,763,988,420,171,438,763,239,706,52,295,248,872,281,426,174,201,952,703,137,95,810,694,739,127,845,995,727,48,498,609,365,615,327,133,463,632,129,603,25,415,3,514,724,931,920,436,751,294,418,417,280,779,336,330,567,499,880,293,384,456,740,838,45,457,964,908,95,815,670,661,460,907,993,798,145,601,353,327,132,529,990,717,985,730,189,189,226,666,440,850,439,996,868,127,296,753,334,462,40,239,928,751,562,536,329,619,743,312,981,642,676,457,443,95,107,438,545,521,905,675,32,91,248,283,423,52,873,723,208,712,274,886,67,962,105,634,690,417,863,135,577,442,773,305,562,72,643,599,461,157,960,19,318,926,902,574,196,193,75,896,702,323,885,621,642,513,520,624,860,272,691,760,445,114,740,763,758,413,280,603,478,997,856,479,827,441,370,322,125,752,415,349,583,689,774,952,286,799,42,561,562,775,899,735,146,566,637,348,694,848,926,909,119,590,203,6,213,624,690,547,455,284,494,71,833,336,11,679,870,86,877,328,181,504,548,99,490,157,918,442,688,224,569,107,48,17,196,943,188,140,447,72,485,919,888,20,695,550,833,981,444,92,846,889,377,507,842,629,569,888,163,31,745,107,505,874,543,314,894,612,203,825,360,21,785,433,38,163,656,378,110,264,288,167,813,494,391,7,53,726,150,287,354,595,102,488,576,837,738,369,552,844,970,705,319,608,4,891,951,853,577,516,267,647,292,89,106,484,381,160,737,847,154,282,531,442,278,745,144,782,741,274,64,976,221,770,188,498,924,53,997,727,561,264,60,837,438,528,785,99,989,488,142,629,410,320,849,863,419,771,958,648,795,725,639,605,465,157,934,739,599,325,625,499,951,419,402,402,857,787,479,926,474,621,98,570,482,395,711,844,785,36,855,679,725,264,123,509,656,495,637,626,154,66,673,52,934,269,721,631,193,358,650,421,425,517,433,945,240,695,258,698,805,335,734,500,619,812,449,592,909,881,197,522,492,370,747,537,390,784,869,656,727,354,776,326,604,590,98,975,317,769,418,161,466,685,907,691,869,389,514,632,17,101,333,866,372,153,246,105,657,852,64,980,649,792,987,363,414,307,644,491,850,149,43,75,695,758,487,849,515,316,462,629,269,970,936,457",
    "87,71,921,189,154,209,737,61,514,287,828,321,590,630,399,356,532,939,330,775,763,61,48,949,160,850,36,549,28,147,89,415,188,915,745,872,668,958,148,994,120,527,536,794,743,738,884,63,486,311,916,114,945,771,664,214,741,50,277,419,818,553,450,620,617,902,294,223,276,487,338,754,176,629,435,588,293,775,263,783,337,152,492,403,270,990,110,501,101,56,374,683,63,965,450,910,828,573,921,361,771,916,77,58,273,884,69,186,136,952,636,822,374,357,364,756,660,630,813,482,64,38,187,181,415,659,919,853,601,288,246,600,0,929,65,734,786,629,769,732,319,334,464,874,977,491,832,411,961,574,551,988,255,101,56,869,938,696,444,266,186,442,479,586,989,718,967,764,277,836,310,892,114,40,554,443,73,770,487,105,485,414,103,583,511,333,415,582,397,900,625,73,745,713,983,661,754,70,779,713,545,723,381,968,965,409,564,135,849,899,191,776,805,715,19,968,92,607,937,231,291,221,963,334,981,612,994,292,630,321,656,861,284,131,282,384,904,627,644,1000,124,841,668,586,10,282,582,151,489,702,304,182,584,173,548,819,781,592,997,311,332,226,29,872,334,118,451,5,265,19,618,673,518,107,440,492,784,73,494,77,492,979,723,409,598,512,238,926,81,686,513,490,309,142,235,146,752,224,630,2,847,46,116,718,779,656,547,929,956,902,420,852,337,793,271,289,380,600,824,758,976,25,717,204,960,329,366,509,124,317,528,604,745,379,199,286,692,588,469,273,634,919,726,12,57,144,809,849,802,621,868,399,359,698,346,155,847,509,911,297,845,667,122,575,84,149,852,163,906,666,564,175,846,906,96,426,632,459,955,83,139,341,980,411,782,368,862,979,104,685,615,458,895,623,468,531,565,332,764,798,256,181,854,490,258,15,142,840,974,211,782,135,367,637,167,743,821,195,94,966,289,265,269,698,725,579,635,670,640,844,309,879,267,979,247,122,842,141,361,155,275,195,686,532,407,994,699,576,978,104,222,624,261,126,684,913,343,865,687,995,559,850,881,127,350,495,808,762,661,177,241,963,480,211,820,9,419,167,649,720,385,409,541,163,628,435,794,472,511,664,671,127,209,709,313,833,355,927,560,691,493,374,716,757,117,614,723,481,619,823,280,269,809,840,241,115,52,162,846,71,152,375,903,166,684,874,906,673,845,325,939,567,895,12,780,232,117,332,209,815,573,799,302,323,528,985,453,997,104,345,274,134,665,136,17,975,536,938,456,634,409,480,619,593,727,875,990,764,748,431,129,929,399,885,91,789,162,538,782,350,799,463,163,298,262,950,731,56,545,789,592,448,354,324,984,365,640,150,395,712,116,572,127,384,636,275,369,748,580,52,891,614,571,185,551,802,58,219,589,118,217,181,187,257,966,693,46,484,429,641,278,496,351,108,904,178,758,279,700,416,607,867,649,474,444,644,608,281,874,926,323,376,964,860,260,960,182,850,376,283,517,93,598,298,467,729,382,204,12,817,712,439,28,584,743,741,708,678,323,480,948,346,639,223,482,387,161,720,170,478,724,7,609,591,495,940,154,113,551,744,92,344,390,981,198,76,513,451,309,565,246,447,844,306,417,764,773,971,196,546,254,34,713,610,262,271,79,150,512,997,831,290,17,475,502,357,64,28,25,734,844,272,763,376,698,678,543,726,392,865,775,535,323,67,631,436,221,121,563,221,415,667,834,204,771,138,950,192,88,968,325,481,620,876,877,686,335,531,365,931,770,974,929,976,348,831,531,328,807,629,848,233,647,496,425,871,361,516,671,987,1000,118,565,435,574,603,771,197,104,769,786,262,259,947,785,614,11,978,375,213,64,194,786,764,891,778,593,406,689,845,572,775,241,72,703,128,736,508,336,775,936,99,537,335,100,980,205,9,7,88,817,1,294,30,355,706,947,687,781,650,522,669,275,858,389,22,622,360,181,380,551,55,536,354,978,799,523,150,479,376,650,816,140,43,558,386,522,288,207,368,285,928,692,582,760,189,600,167,769,943,827,862,923,786,628,987,24,860,327,923,484,197,425,502,620,742,522,238,926,629,938,157,930,34,595,454,897,746,563,433,4,836,125,989,652,800,746,952,11,634,793,283,485,879,454,759,122,933,513,577,292,239,781,884,849,816,345,592,149,468,534,408,279,704,690,112,767,675,82,415,239,429,794,591,720,423,509,740,313,122,710,608,72,156,965,656,302,663,367,865,495,576,275,607,902,172,794,237,516,944",
    "62,908,353,908,430,461,834,936,601,627,929,709,651,903,926,383,365,359,872,357,630,276,463,981,811,491,969,197,681,874,391,631,455,349,470,815,600,285,393,598,584,474,635,538,974,322,808,814,51,452,436,915,334,529,315,846,529,353,866,187,264,238,338,479,317,920,254,446,681,612,940,512,375,871,18,303,198,772,396,605,138,79,409,115,386,936,169,295,126,473,567,610,918,294,143,522,811,102,770,675,138,277,449,575,962,245,411,994,972,706,440,782,201,336,783,79,840,698,481,390,492,756,55,698,281,943,904,327,624,260,430,714,929,0,215,503,244,422,354,230,943,706,431,135,997,814,814,108,402,165,866,195,798,151,541,445,495,661,298,921,606,537,555,880,356,528,99,312,18,554,874,720,557,159,733,101,497,965,900,910,777,879,277,924,29,593,849,183,295,23,87,354,201,487,189,937,636,89,515,356,641,587,831,309,567,495,298,838,507,57,736,546,787,330,438,982,632,23,353,248,775,785,754,593,612,21,852,22,562,862,64,276,824,313,740,314,156,856,486,181,979,352,106,512,264,706,235,238,69,702,474,923,176,474,637,738,425,317,178,349,349,835,84,745,379,336,933,541,799,272,979,404,431,438,162,754,116,269,167,814,465,329,981,266,596,657,214,411,444,931,660,19,615,735,101,128,742,570,593,982,889,647,278,147,473,47,54,945,871,710,518,867,568,519,457,532,737,144,907,120,192,510,211,731,368,413,303,70,591,72,746,171,655,980,539,918,490,680,781,604,823,884,438,862,934,682,702,393,653,365,719,777,157,898,217,346,381,926,776,973,207,782,814,960,383,86,897,564,754,352,203,331,128,585,719,771,539,994,304,960,287,147,367,37,658,207,230,108,976,725,775,893,339,734,799,719,213,367,50,576,896,360,983,373,94,197,337,566,644,470,741,362,731,498,966,594,391,450,799,959,976,239,638,494,704,269,180,370,620,824,157,950,345,225,855,440,315,232,40,822,385,513,2,528,609,954,983,880,610,66,251,188,467,885,795,260,729,899,769,904,486,733,648,367,881,400,498,59,189,700,914,217,544,869,215,522,428,877,175,298,281,522,19,11,551,860,345,305,110,889,854,771,516,232,739,776,188,212,153,434,542,642,816,303,457,280,580,593,879,771,323,439,695,664,656,103,617,752,223,979,953,952,25,435,464,572,820,478,45,80,631,365,231,924,498,392,581,277,360,99,548,49,265,221,557,199,105,131,422,564,290,527,923,684,9,286,873,23,963,367,240,894,272,215,46,206,860,96,570,477,898,439,332,40,750,417,756,49,622,876,991,231,367,513,58,497,478,427,331,575,100,558,143,988,48,813,978,150,199,830,474,446,598,687,972,864,889,970,459,856,707,539,404,15,43,994,296,204,11,342,83,753,191,737,282,36,360,273,705,829,989,968,771,179,775,599,272,336,781,313,789,205,135,692,213,105,372,626,478,535,156,972,286,919,962,309,825,821,114,7,71,809,29,207,470,306,676,344,489,259,819,986,831,362,562,756,784,445,242,722,218,642,588,6,497,781,261,641,589,728,361,311,942,257,994,577,363,584,777,327,962,962,179,398,234,236,10,245,416,141,579,778,860,716,787,770,747,378,696,505,553,808,688,479,193,589,540,802,941,364,761,130,287,357,432,840,569,630,337,928,509,888,736,63,905,323,224,365,214,878,476,844,771,973,276,898,658,107,394,914,944,319,689,329,450,483,638,304,518,484,69,568,874,893,23,376,848,298,92,808,447,83,662,752,199,998,705,281,854,787,147,627,452,832,158,318,492,315,697,701,659,955,345,946,626,642,768,621,94,166,978,416,501,738,162,207,12,984,806,424,578,937,774,649,790,455,512,920,828,619,366,645,957,745,274,594,505,803,224,916,597,884,46,327,973,231,719,544,856,369,766,306,366,483,463,721,150,204,740,662,132,567,646,708,592,785,476,787,698,269,69,778,362,535,359,798,848,995,170,157,959,146,411,119,519,695,635,692,282,314,114,408,347,766,567,444,924,580,582,702,813,98,725,213,789,978,437,107,468,711,884,681,377,587,192,119,874,76,864,295,234,586,207,494,714,624,656,32,442,79,419,325,584,696,23,124,570,600,549,97,79,736,217,978,24,326,370,849,14,865,471,933,521,610,894,22,672,625,167,296,186,370,133,706,123,765,431,830,804,390,516,752,18,350,926,98,196,566,146,284,449,13,235,999,927,145,304,414,906,409,180,444,686,422,157,185,684,456,674,853",
    "218,63,134,819,25,756,185,794,150,859,102,110,441,90,732,296,687,344,148,305,380,848,97,609,270,870,972,736,337,64,527,167,579,328,95,948,672,322,776,714,852,46,927,57,670,149,617,438,707,267,870,520,738,386,163,391,421,158,356,494,490,860,374,283,385,694,691,794,575,593,575,371,282,504,184,588,685,741,747,77,638,574,245,408,380,689,628,217,989,966,135,298,841,586,61,432,144,621,42,483,303,836,669,292,571,504,554,877,815,307,51,267,798,702,580,577,405,5,538,911,615,516,993,794,131,349,570,505,124,842,524,530,65,215,0,931,351,515,330,549,44,913,645,370,954,741,328,342,958,951,631,134,278,865,556,728,408,493,205,690,960,84,491,751,766,855,468,555,954,703,598,30,179,880,804,293,659,713,555,521,314,580,812,375,901,153,980,103,440,206,570,687,396,412,164,175,246,581,642,219,718,519,596,223,894,34,398,747,816,679,222,116,297,538,689,649,112,238,501,726,810,212,396,531,824,532,354,818,191,423,667,460,147,165,550,41,994,986,793,555,929,290,631,259,935,998,102,53,806,880,970,912,868,152,276,849,599,158,618,150,483,587,950,614,173,297,866,816,880,210,314,268,511,321,407,390,558,221,785,219,426,663,928,410,523,686,443,23,500,707,549,194,125,623,67,820,324,793,523,928,505,879,566,28,850,764,656,778,419,899,378,261,335,396,984,143,348,264,373,291,166,367,687,355,246,140,340,337,157,576,59,522,252,562,610,820,656,897,424,796,699,529,371,807,417,861,764,209,831,117,921,33,137,613,706,584,111,575,421,605,198,207,189,518,89,93,232,66,949,534,110,345,771,326,155,62,579,243,550,744,57,343,690,483,927,225,757,479,682,988,63,828,428,267,918,172,363,333,381,292,278,407,892,899,653,632,728,758,855,778,311,181,912,733,515,313,151,886,825,961,195,328,160,529,582,305,697,427,827,62,771,138,705,165,351,85,59,466,191,180,14,192,585,279,816,204,124,761,916,457,829,92,927,978,779,324,324,641,998,515,820,672,409,850,923,108,159,759,107,964,123,579,277,104,624,738,775,195,692,203,267,328,825,664,327,789,966,876,794,649,346,437,496,861,541,350,77,609,722,258,11,32,645,471,936,264,125,55,351,44,799,250,788,546,943,115,405,989,659,469,760,273,845,601,268,810,596,684,505,735,86,26,544,809,657,169,764,897,948,674,408,843,576,363,967,392,750,348,801,822,481,934,311,469,93,506,212,169,643,809,642,28,560,657,616,902,887,753,775,334,68,245,66,650,501,382,584,887,736,432,81,633,9,422,626,884,905,338,70,725,691,76,985,23,461,414,918,246,698,804,966,948,341,426,542,903,551,118,823,766,885,312,234,26,237,207,596,380,519,802,268,563,304,127,776,336,250,992,813,646,175,769,206,952,18,828,226,225,194,475,468,631,603,611,382,743,770,240,869,588,34,593,655,866,69,756,553,85,198,156,575,506,102,455,675,584,560,743,644,585,214,413,19,173,823,995,484,319,201,943,860,27,42,249,657,968,846,437,72,675,582,2,229,441,450,298,449,128,725,117,246,443,190,153,894,337,530,310,348,193,627,971,2,903,673,534,805,331,11,150,184,21,584,792,483,672,967,61,431,985,949,664,890,415,425,713,518,538,22,903,85,621,275,100,979,271,982,323,37,209,161,945,926,826,433,223,937,23,647,10,113,563,186,273,233,953,175,850,976,832,594,110,294,402,750,1000,926,39,640,977,208,631,840,318,621,748,366,225,174,455,636,984,822,289,237,274,427,527,100,386,571,880,15,99,376,194,864,928,564,496,570,864,176,776,143,713,101,86,412,210,190,909,478,645,120,335,207,872,459,451,521,123,563,658,727,527,312,270,411,380,51,911,943,437,132,227,126,59,453,716,786,837,316,218,512,657,821,672,711,876,987,509,566,318,814,452,546,594,991,880,614,82,787,739,470,469,911,948,453,98,11,14,109,634,282,844,898,935,967,732,810,616,436,376,349,773,566,726,887,84,461,135,413,451,46,535,787,172,475,514,313,398,581,591,19,164,649,575,472,727,296,95,847,803,549,736,144,877,996,424,323,169,52,516,841,766,976,689,974,988,363,715,497,847,240,969,126,276,342,334,727,474,755,894,289,330,61,449,183,951,975,572,855,453,92,302,671,948,329,33,234,499,866,555,24,16,689,255,302,992,840,941,58,317,464,6,928,483,297,66,893,811,762,23,829,197,138,835,383,496",
    "435,125,593,65,850,97,148,170,845,457,782,178,968,529,112,389,980,226,631,854,23,868,340,19,899,237,388,551,661,542,118,986,602,260,706,577,907,850,273,433,104,422,12,90,776,146,47,398,915,585,954,470,495,469,399,894,975,529,214,54,418,248,768,681,231,47,616,78,624,900,473,580,556,561,5,99,417,670,650,660,162,497,820,392,309,226,681,810,324,221,903,560,433,541,550,671,569,468,847,656,675,211,782,922,227,513,693,255,181,204,517,875,141,583,899,38,432,277,374,58,27,733,393,213,76,928,481,808,539,352,61,944,734,503,931,0,290,797,581,57,306,783,69,761,741,173,731,570,221,121,834,29,658,994,987,6,974,858,663,447,67,280,327,536,769,125,405,859,437,871,230,967,990,8,107,620,132,761,877,610,358,2,569,171,23,217,199,180,602,38,452,235,820,538,174,110,151,111,653,603,8,255,417,922,128,532,903,983,91,699,796,462,55,838,118,229,143,221,696,426,4,256,479,258,940,299,170,889,287,479,418,909,549,424,739,701,780,920,733,124,637,496,914,84,143,542,448,20,686,964,436,802,516,765,330,365,982,432,445,921,718,23,845,243,598,332,223,473,980,746,859,23,728,350,253,590,414,202,442,549,537,850,600,252,385,865,660,349,339,631,373,899,523,902,171,86,64,338,7,882,467,434,815,901,335,459,292,507,730,900,955,175,957,255,462,303,481,824,284,232,273,953,498,225,254,320,968,254,79,834,698,102,660,479,246,40,268,511,276,730,899,633,731,301,709,90,36,524,213,864,636,131,602,702,517,256,444,683,957,668,540,385,147,322,778,802,731,249,576,320,818,917,68,93,312,160,968,794,837,75,423,826,671,467,117,609,777,509,341,302,122,563,371,24,880,283,590,37,413,776,999,846,382,618,870,566,904,378,54,934,762,368,619,715,981,238,467,560,737,307,24,688,741,388,548,290,559,62,219,889,42,959,440,394,506,441,296,661,404,577,354,652,216,661,365,861,881,975,663,803,574,244,25,80,776,742,718,842,712,311,448,116,846,443,805,518,992,476,919,172,299,434,374,397,508,41,758,527,426,423,632,684,462,750,765,405,896,18,141,298,847,317,794,3,846,892,119,917,172,882,876,939,789,652,67,104,269,498,915,824,704,390,971,172,3,713,853,183,21,841,608,445,162,419,538,897,490,559,885,163,885,166,921,899,257,185,765,611,110,409,437,762,124,432,781,738,274,409,309,608,782,695,146,193,217,463,901,885,228,272,584,730,317,21,221,257,731,641,925,479,815,94,541,312,799,111,825,45,426,417,515,567,61,386,259,494,752,58,200,269,137,751,600,576,188,444,585,716,34,953,80,300,847,452,403,159,908,719,114,937,333,617,806,895,458,615,462,734,703,4,864,192,361,679,951,697,247,553,812,575,939,22,255,618,367,876,41,906,247,839,114,821,803,917,202,441,443,973,476,115,484,304,51,431,144,947,476,777,476,814,194,893,441,340,919,550,837,310,567,613,108,99,359,774,33,19,224,450,960,94,147,958,72,174,420,635,556,468,292,864,562,413,731,108,4,594,975,201,852,95,841,982,932,155,702,632,521,809,292,684,482,472,120,573,319,70,42,918,995,822,24,215,762,450,166,790,277,207,1,148,52,82,792,67,533,818,452,68,663,30,249,216,879,784,497,712,372,117,217,517,506,686,747,533,296,166,923,516,438,126,343,309,944,234,792,890,408,153,71,457,803,511,219,453,96,544,863,722,807,666,166,380,45,952,877,668,235,225,793,853,126,566,489,323,717,895,652,185,435,137,203,155,878,971,849,396,998,673,663,935,504,41,308,625,158,714,804,897,392,825,289,188,341,376,168,60,709,660,501,273,546,797,143,120,272,120,597,630,499,352,904,186,672,889,32,40,911,108,170,115,998,871,117,658,838,638,733,323,398,656,295,683,293,175,445,14,140,858,845,354,694,407,642,791,669,838,610,846,800,27,879,79,178,855,136,335,231,28,221,304,354,808,242,647,807,245,152,163,487,680,318,500,397,627,140,858,599,51,755,983,672,998,175,580,734,571,128,258,558,126,922,381,502,686,219,8,12,907,410,161,397,485,745,523,551,148,788,174,626,685,126,560,92,264,476,482,384,443,448,918,194,862,416,142,684,900,464,373,604,70,136,173,940,791,715,257,523,180,829,442,720,683,146,391,690,420,256,598,939,98,212,153,40,282,539,894,870,331,804,63,403,985,155,386,563,129,753,201,564,38",
    "559,560,511,555,374,523,472,167,943,90,766,788,711,456,225,651,278,613,773,682,671,499,976,147,145,203,874,308,356,323,51,812,732,888,550,110,841,236,776,448,556,122,252,255,316,57,932,407,94,247,182,281,44,539,551,550,9,927,996,144,411,653,96,436,996,703,284,197,191,267,400,835,289,900,807,413,133,390,956,655,446,538,780,778,12,377,495,834,523,374,831,609,559,94,235,626,678,44,888,285,645,18,821,614,806,567,443,866,934,389,529,521,287,201,639,682,148,682,153,245,187,610,374,924,601,944,732,227,513,144,345,641,786,244,351,290,0,176,575,629,986,230,976,364,720,474,837,698,508,369,431,734,504,626,141,85,861,569,301,319,670,817,32,156,619,998,893,255,156,938,209,272,328,968,561,640,388,765,457,359,181,210,331,412,510,300,795,399,298,881,620,702,872,326,317,126,262,219,110,143,562,814,290,10,353,222,619,743,857,364,701,733,954,543,802,493,20,816,13,49,348,405,123,876,237,731,60,552,832,564,121,274,99,309,631,912,23,444,994,7,333,241,38,896,146,504,614,378,953,685,104,111,135,790,758,240,763,429,565,178,321,12,283,36,711,399,679,330,801,36,709,799,123,773,786,192,960,983,122,779,152,131,987,440,191,637,292,946,936,106,204,664,355,275,566,375,328,444,264,212,617,550,691,880,573,229,677,652,71,309,693,727,947,249,652,654,152,32,980,593,24,366,489,994,588,403,624,73,718,814,138,288,420,204,745,258,193,746,940,502,740,563,889,765,360,712,908,551,488,713,722,443,477,615,129,912,281,533,994,687,133,768,465,928,63,755,537,86,170,248,125,301,436,652,863,883,552,282,418,470,633,158,970,840,336,182,547,380,240,345,516,867,854,190,75,662,930,987,330,53,757,143,281,158,779,119,407,51,616,959,34,912,406,511,592,567,750,633,128,910,584,365,607,705,984,168,900,756,719,324,46,164,908,229,852,410,717,672,193,600,622,14,138,311,536,820,346,651,666,131,455,345,503,20,495,625,536,658,407,199,630,351,951,389,835,327,202,273,818,60,403,106,962,892,625,733,710,682,280,283,425,615,903,527,712,89,611,757,400,152,120,264,525,551,694,521,516,96,975,226,809,361,278,473,157,201,877,631,60,376,777,355,352,448,352,114,390,235,137,921,532,740,360,713,963,361,350,444,668,22,366,104,452,151,833,114,53,624,1000,741,200,884,788,541,204,706,988,567,78,634,351,200,431,548,939,463,416,701,360,450,682,248,50,536,24,357,510,8,933,842,793,533,279,589,337,540,660,53,919,58,455,795,12,753,634,215,89,560,849,168,739,560,911,487,295,128,493,161,517,357,925,700,58,790,175,466,762,690,324,823,958,428,677,517,620,536,782,195,590,423,282,964,725,3,896,94,36,243,483,393,190,534,848,763,85,927,200,347,771,130,722,905,217,463,365,843,780,752,851,303,633,511,530,817,564,382,288,101,427,548,579,805,175,559,758,971,919,295,825,33,322,990,693,935,462,861,338,36,455,151,279,832,856,21,555,513,707,925,891,855,335,808,658,383,115,414,814,779,521,302,250,23,648,82,145,546,301,496,549,476,185,283,345,941,152,778,295,879,72,939,343,30,967,765,70,574,199,963,810,680,220,351,693,185,781,771,399,166,851,139,617,258,887,127,569,821,106,552,445,402,578,546,92,112,605,597,373,216,588,742,42,317,788,392,310,213,252,309,93,532,388,558,658,782,734,472,824,422,706,395,579,567,856,40,452,242,718,310,405,895,735,623,614,112,143,616,244,314,661,495,743,262,434,954,853,655,517,851,451,205,312,504,920,498,416,440,101,976,934,200,849,923,388,522,695,737,402,575,57,560,420,161,437,24,16,691,906,165,279,476,801,324,955,669,613,778,144,437,236,743,320,169,112,245,619,783,341,775,921,763,950,849,381,909,151,211,389,962,287,298,564,493,396,273,12,686,440,273,427,911,724,733,949,561,772,408,425,743,221,584,903,665,521,302,820,776,160,648,118,560,174,926,919,245,750,970,134,856,671,882,102,608,253,349,222,604,24,948,843,456,252,145,400,747,607,926,520,621,497,550,696,833,123,203,719,76,858,860,485,226,517,525,705,28,819,361,675,364,781,658,164,724,335,421,61,803,234,337,865,809,47,725,276,981,256,349,802,434,255,72,252,575,861,133,323,671,133,316,293,888,687,600,303,195,622,588,874,676,19,303,485,295,867,6,973,293,514,826,974,190",
    "283,335,430,158,206,220,479,509,520,342,478,787,451,379,176,275,191,390,247,582,17,396,652,647,116,850,833,368,348,661,644,450,388,448,386,168,819,91,773,652,536,735,241,919,274,563,767,4,202,765,595,476,115,672,16,629,187,231,704,326,939,229,649,8,279,285,114,723,399,142,350,796,828,754,889,712,114,700,361,200,652,492,812,11,582,380,608,466,180,909,485,924,352,627,818,663,377,469,671,832,250,594,656,899,289,650,780,805,439,46,710,860,237,684,5,125,888,368,376,336,135,432,862,499,27,876,645,164,133,636,710,687,629,422,515,797,176,0,160,435,672,292,756,320,383,674,250,210,786,444,447,977,479,258,403,868,102,245,475,615,791,701,60,986,862,958,915,52,164,208,26,734,929,54,935,756,539,989,394,683,743,672,503,105,304,896,27,850,142,425,148,499,482,922,87,184,920,834,855,125,271,439,546,928,610,527,963,437,728,340,827,165,368,169,539,890,73,19,455,819,564,275,978,946,657,708,415,982,131,271,75,995,976,569,50,938,863,845,975,495,463,591,629,602,544,190,200,521,352,694,542,209,625,44,392,682,297,312,236,956,18,260,256,976,399,29,398,843,632,442,898,845,912,779,248,8,965,142,232,599,49,60,740,529,147,122,362,192,294,586,815,616,607,410,35,52,395,53,846,968,969,744,921,618,429,27,489,545,951,828,726,568,208,777,230,113,703,553,933,257,515,895,725,682,303,215,207,597,180,844,523,141,198,286,970,515,869,49,376,258,905,536,468,587,259,22,505,636,795,224,672,443,992,615,6,107,523,476,281,880,836,289,487,975,817,559,216,427,893,363,459,457,476,961,145,256,685,691,154,156,766,484,727,942,669,977,746,756,149,331,613,913,281,670,540,752,967,941,736,148,621,4,987,324,854,118,896,600,613,55,146,415,617,949,58,43,822,717,584,818,316,80,115,783,606,821,651,756,232,824,369,363,338,514,589,642,918,718,946,83,41,768,680,597,955,806,42,225,979,91,754,757,439,900,71,998,880,245,764,456,54,749,960,559,593,685,193,806,40,143,719,956,71,359,437,382,102,418,160,988,939,813,409,699,678,247,491,352,70,485,398,562,714,760,742,663,806,554,796,395,123,345,882,629,533,204,866,952,395,265,889,99,797,803,128,576,460,736,219,219,576,70,113,88,527,506,508,39,369,971,360,555,728,577,160,913,227,394,752,922,504,941,902,331,361,800,132,409,192,860,481,750,956,877,562,478,570,557,590,291,323,521,584,526,553,20,830,366,36,311,645,222,6,976,553,229,962,328,552,955,282,136,24,520,215,157,819,663,924,836,986,168,694,443,332,377,356,105,344,469,264,310,339,997,379,823,536,989,750,12,602,918,712,235,233,344,507,388,902,25,617,44,137,435,367,957,262,863,247,40,570,626,56,877,659,29,606,926,25,266,593,193,627,950,296,296,855,739,788,895,607,610,847,924,823,892,779,977,841,536,159,304,534,226,148,749,708,502,733,784,594,829,451,822,198,204,527,986,822,920,228,346,412,803,67,53,927,213,858,590,81,38,571,347,98,933,148,979,448,743,463,725,355,928,397,955,865,136,305,447,443,826,488,16,556,614,639,85,398,977,180,329,315,76,361,583,150,468,714,982,881,308,119,693,56,945,280,574,572,19,344,700,227,832,629,712,661,336,823,558,675,209,413,378,196,446,474,295,795,160,304,967,722,491,859,578,11,850,3,805,254,258,332,603,543,597,300,324,899,700,682,643,288,368,881,621,690,179,702,384,304,447,426,586,173,300,458,199,156,3,99,492,454,7,454,319,273,185,35,469,382,16,990,931,192,724,659,322,70,763,256,586,411,621,322,554,807,72,144,468,806,463,983,201,506,926,329,267,94,515,785,61,443,445,884,659,34,528,976,657,864,798,555,389,28,903,81,112,258,110,127,327,97,219,472,51,793,602,794,909,623,980,426,619,959,337,559,969,350,564,104,742,823,506,731,976,124,267,890,767,524,203,728,765,130,746,940,116,884,343,85,497,398,850,795,901,869,423,838,161,993,650,204,735,557,902,460,268,443,274,773,899,268,26,547,404,964,458,850,805,129,344,740,61,479,637,316,846,900,203,320,137,999,883,767,567,320,560,33,106,402,720,661,80,647,93,141,396,367,691,33,360,366,188,531,103,213,930,914,61,686,882,927,535,766,874,904,336,687,457,120,458,40,456,94,679,349,100,605,838,742,72,85,881,894,449,937,919,658,623",
    "562,612,516,438,544,821,969,986,438,314,295,271,283,560,258,178,139,189,854,549,708,871,122,6,410,107,433,133,815,149,296,235,620,75,798,382,521,809,177,864,995,105,448,455,53,271,571,420,505,765,385,915,538,892,105,26,918,74,649,924,842,852,610,585,751,905,706,490,823,461,679,327,992,630,409,78,666,602,618,168,508,266,55,378,616,77,35,380,588,682,662,647,698,988,26,193,244,242,803,96,78,763,921,805,310,247,623,763,757,525,81,686,205,56,575,857,950,973,220,933,42,590,550,125,584,69,523,468,320,590,992,502,769,354,330,581,575,160,0,317,140,590,46,38,784,651,461,70,834,674,322,907,358,748,1,654,571,313,592,410,527,571,638,391,585,308,279,616,556,759,164,716,584,916,381,598,716,36,957,973,629,867,407,696,554,505,144,881,290,586,866,471,884,270,629,895,649,697,143,58,540,150,403,487,540,62,816,747,41,926,509,938,412,17,907,618,227,163,434,774,884,884,881,978,67,116,210,791,902,636,377,730,744,700,713,726,303,885,455,803,400,749,881,982,887,280,337,788,207,997,644,822,132,934,332,753,47,265,236,589,995,428,265,770,23,937,544,208,728,952,834,295,272,23,98,825,705,751,25,903,883,768,879,85,563,314,272,895,71,298,250,207,259,293,71,289,289,372,23,371,166,885,750,144,433,53,671,12,11,964,295,529,116,253,707,683,653,929,729,894,479,749,94,855,114,458,993,988,350,843,752,735,409,242,857,790,339,627,341,943,816,399,278,321,778,559,52,807,863,315,814,200,799,643,769,392,879,448,207,65,847,200,45,122,30,608,800,574,656,220,263,677,854,568,950,716,556,806,523,497,130,702,278,663,222,298,997,51,608,656,862,481,549,510,129,177,791,983,472,371,582,807,156,126,171,732,342,245,467,381,874,632,87,950,74,52,755,265,911,872,281,845,66,986,707,878,565,627,710,550,865,78,880,601,307,661,1,881,948,473,335,600,428,208,408,109,27,67,745,443,57,314,672,345,486,415,731,106,846,203,224,986,344,913,812,182,806,470,824,244,282,661,583,372,612,422,398,876,59,961,126,462,11,611,874,166,100,545,39,717,96,362,987,455,877,753,225,136,428,555,781,119,568,258,291,597,79,306,62,718,102,334,208,709,323,860,796,439,297,714,730,910,743,39,447,887,962,451,104,791,999,811,213,703,190,457,334,888,962,85,332,335,180,372,801,921,774,803,382,834,46,85,458,218,265,872,937,383,855,393,19,44,963,655,570,568,267,342,164,239,403,669,874,781,229,729,572,858,613,235,31,606,769,809,125,319,474,181,770,412,268,215,944,540,273,681,864,284,182,345,723,856,479,872,283,443,183,74,57,477,446,704,592,633,683,204,505,823,790,677,523,622,385,800,949,21,508,748,955,216,347,382,42,662,672,999,661,561,814,721,274,85,156,966,290,384,925,412,395,273,585,571,823,954,372,228,395,746,411,47,784,649,481,247,922,592,933,311,455,374,787,162,279,110,834,803,185,226,859,919,402,631,662,612,40,362,281,697,636,970,371,822,52,534,17,273,193,461,274,707,207,821,411,740,959,144,319,256,562,623,595,314,944,155,413,559,690,119,314,122,843,251,217,262,428,337,373,844,949,438,265,435,783,457,515,170,514,954,260,561,432,795,138,698,755,636,761,865,463,397,675,371,621,455,582,358,48,745,789,83,8,925,609,835,488,900,81,458,701,792,315,977,121,92,383,236,945,846,419,67,333,718,692,543,56,562,103,436,183,965,831,391,281,467,492,322,157,189,619,101,53,280,742,800,545,566,192,581,491,158,995,776,392,762,343,910,890,501,439,707,131,470,696,829,237,327,556,624,275,873,616,519,181,241,148,540,649,229,147,192,748,845,481,795,486,651,976,645,997,77,543,837,451,586,404,415,391,901,222,36,699,818,969,710,629,102,999,206,149,560,314,556,943,870,83,269,402,905,306,981,118,639,665,868,397,949,880,334,85,537,3,518,519,118,516,716,521,721,142,654,754,22,690,52,278,969,980,693,112,895,953,11,665,72,802,859,978,384,573,219,993,341,762,288,994,380,369,541,702,805,206,591,600,18,998,35,960,281,214,901,658,458,723,327,399,610,409,694,364,222,202,955,2,936,970,271,392,248,784,499,533,639,198,91,320,740,311,392,733,224,133,788,490,408,1000,830,144,352,126,372,396,742,527,808,62,324,967,995,402,441,103,427,940,311,188,901,161,551,339,124",
    "798,16,646,681,833,140,463,13,947,568,512,923,906,32,702,549,92,327,632,693,154,87,182,2,246,327,970,666,438,88,715,902,822,824,286,121,40,63,610,113,770,411,926,310,948,613,391,889,370,436,203,780,12,398,936,351,40,912,933,665,240,873,597,383,857,226,585,374,47,712,564,583,278,159,223,443,882,560,160,42,976,455,474,796,847,936,191,490,644,734,711,556,474,992,745,536,727,577,278,447,221,712,4,157,843,472,197,656,117,202,174,273,326,576,483,746,899,705,128,175,207,883,241,967,308,211,543,273,77,893,762,725,732,230,549,57,629,435,317,0,251,532,19,465,810,839,90,742,567,459,333,780,36,397,781,763,780,818,769,728,391,641,230,32,999,699,843,476,985,734,829,350,86,65,481,673,325,649,623,710,328,331,450,158,288,605,519,890,809,122,198,16,225,849,886,667,496,800,75,692,794,347,864,528,947,734,35,143,568,912,94,302,599,270,256,942,251,797,189,89,668,232,429,663,50,511,293,19,762,588,305,889,158,886,12,701,326,993,892,704,301,392,292,306,68,405,6,545,803,352,489,330,127,7,898,624,995,73,814,508,40,777,414,561,352,898,401,282,867,960,959,688,437,515,548,520,279,994,88,296,121,878,734,722,678,754,225,764,940,511,192,12,105,192,568,912,785,700,242,98,292,84,848,149,53,946,820,60,284,301,138,579,347,302,500,237,294,564,715,437,424,775,833,536,265,54,687,200,972,775,992,438,801,363,713,476,602,524,172,488,593,722,746,130,751,352,161,944,506,213,35,909,556,174,873,743,277,842,243,220,349,264,884,25,792,803,239,568,352,856,368,251,660,740,790,816,836,843,304,516,741,218,631,660,142,442,161,575,27,480,54,547,18,981,346,402,738,322,858,294,702,812,652,807,652,624,118,686,44,816,847,732,549,562,323,775,869,422,805,651,845,249,209,956,9,633,765,582,6,98,59,429,334,887,879,174,826,236,531,703,669,872,518,255,326,313,585,668,212,307,77,998,946,575,530,250,631,926,219,6,496,15,520,25,325,809,528,480,264,251,729,787,733,484,517,825,226,960,872,921,596,622,128,929,317,41,209,303,345,104,19,19,831,954,248,61,979,848,886,965,126,395,379,842,201,698,595,337,597,378,951,554,746,88,23,103,564,715,459,130,584,645,563,746,645,890,89,905,718,59,700,513,540,668,739,703,630,93,630,686,766,173,961,437,418,58,726,865,429,236,130,187,237,507,545,604,371,448,70,861,914,392,117,352,150,481,334,843,557,462,127,206,745,395,181,136,552,486,451,900,451,358,413,695,621,401,826,108,586,251,833,691,774,676,437,12,763,431,640,956,403,656,413,523,653,872,977,21,638,948,433,492,519,704,832,245,643,563,456,518,718,694,924,301,98,686,927,265,644,407,295,703,619,132,796,966,247,853,693,63,446,895,716,556,872,331,832,130,736,199,196,631,827,783,832,863,658,184,87,821,476,228,745,929,446,868,881,513,347,851,788,428,987,739,73,592,530,702,52,291,880,992,350,897,232,213,64,762,696,543,608,501,280,140,118,860,398,223,547,256,166,567,226,760,68,833,728,989,677,334,503,906,129,682,199,574,583,177,638,53,407,794,152,690,39,522,738,491,781,427,684,340,276,513,184,575,94,903,172,469,127,695,858,15,427,473,563,106,296,894,168,480,55,435,419,443,156,980,786,249,278,266,483,884,763,958,581,299,237,551,694,141,93,301,454,198,82,544,197,553,98,4,278,601,981,409,296,311,431,506,601,707,562,306,635,179,974,620,34,956,13,234,852,314,599,491,665,78,217,23,243,318,256,346,518,879,461,854,138,605,681,212,307,791,28,441,95,418,713,601,706,81,355,35,395,644,192,871,279,182,571,228,600,953,115,5,685,907,92,717,216,992,710,374,662,366,337,912,856,840,389,730,348,504,878,401,863,106,648,258,462,950,445,925,103,879,861,159,411,304,419,811,505,356,287,65,191,993,458,88,249,412,919,519,264,474,589,823,346,853,945,961,909,116,234,682,892,884,193,823,112,596,579,45,984,442,334,544,455,808,883,244,332,270,353,866,874,902,317,997,792,653,185,689,317,776,891,903,506,128,955,854,594,778,971,254,894,36,767,13,514,575,555,265,639,19,77,145,234,794,668,37,742,872,72,191,121,254,836,828,86,949,413,653,620,442,863,970,881,487,513,806,81,66,744,78,448,837,774,836,58,508,220,322,498,919,716,890,935,160",
    "349,296,640,78,198,285,751,375,309,326,484,880,890,748,647,624,852,413,999,373,427,15,203,568,271,586,263,48,444,520,523,804,788,971,634,790,941,12,671,383,834,810,884,539,669,419,768,479,381,775,827,651,953,978,178,426,362,29,669,858,849,110,612,948,635,977,921,139,126,176,17,524,744,759,854,196,835,38,570,321,636,260,611,515,271,609,497,658,202,681,686,932,774,868,742,317,918,686,369,108,562,610,755,15,518,928,511,16,692,927,202,268,784,710,866,811,67,276,756,585,8,318,372,222,641,210,208,931,748,654,667,140,319,943,44,306,986,672,140,251,0,63,630,532,985,622,807,872,708,650,242,242,119,813,133,869,692,178,439,756,728,444,711,274,855,130,897,882,922,10,341,450,974,359,230,587,332,547,771,499,157,985,167,397,72,398,31,491,153,860,257,976,432,764,582,764,712,807,76,617,581,947,631,250,184,115,95,518,669,43,232,58,101,582,534,363,834,720,527,849,679,574,679,482,968,838,807,449,4,886,932,457,760,194,148,80,675,292,333,945,605,353,209,549,576,772,907,122,170,221,216,464,703,278,114,985,120,578,238,432,392,339,153,939,342,695,684,358,115,592,321,26,614,604,896,202,306,639,287,287,5,937,214,159,83,87,368,118,975,165,876,577,765,118,550,223,100,624,37,486,26,584,273,246,79,370,758,376,159,662,248,261,17,228,345,502,90,547,530,189,115,535,524,217,981,679,11,499,694,651,352,875,612,766,431,655,936,894,368,204,73,615,540,390,29,774,538,417,104,344,313,249,107,946,598,509,656,161,422,714,478,4,240,664,678,462,448,586,344,177,724,225,350,472,18,922,162,852,474,171,928,213,253,767,485,447,975,910,243,684,351,748,699,451,467,726,961,846,389,555,807,368,258,467,309,89,786,906,227,157,652,898,797,217,732,625,371,685,542,608,458,183,899,480,374,861,460,937,817,614,124,248,556,437,598,560,452,182,61,453,965,491,357,89,744,59,171,158,100,387,699,945,575,293,442,465,844,602,974,32,890,552,832,494,645,254,428,592,191,883,692,856,72,952,686,198,885,863,947,701,35,54,961,257,777,817,112,901,256,949,804,709,989,366,420,413,616,812,853,865,681,656,453,75,706,641,708,621,271,484,761,470,384,896,586,391,44,580,159,670,253,551,986,572,666,635,899,843,416,888,998,330,145,991,548,697,596,481,735,652,518,39,633,946,113,416,775,609,942,964,359,604,129,43,393,693,294,457,245,782,915,601,979,105,118,429,322,429,714,175,871,392,760,648,891,627,446,450,882,244,241,989,192,617,988,288,630,809,385,514,285,593,549,981,766,825,634,380,61,438,717,897,985,687,725,22,483,824,178,479,157,556,725,870,275,118,198,621,79,275,341,632,169,753,589,26,704,980,883,889,457,950,180,950,318,300,747,235,227,732,629,574,180,747,125,230,963,50,438,20,518,426,290,345,73,471,514,384,439,599,808,837,250,249,632,86,433,743,71,744,789,223,831,50,83,17,900,496,678,656,528,124,883,672,984,672,71,362,173,482,871,437,155,444,218,450,141,637,528,302,181,71,874,819,877,278,887,922,434,797,929,932,402,545,284,768,979,510,653,157,323,927,707,149,211,583,358,623,757,648,951,49,55,234,208,581,585,691,535,611,421,5,780,192,747,507,171,860,466,305,522,546,910,23,295,618,898,267,812,127,505,913,486,880,957,94,705,825,366,547,289,330,319,84,510,499,262,666,860,122,750,74,591,301,638,803,191,813,919,498,298,709,361,630,21,724,195,882,705,379,1,252,646,225,743,347,684,378,292,162,882,202,549,554,145,606,785,399,769,801,841,48,152,845,406,261,595,592,916,342,652,480,319,319,48,815,753,656,315,998,946,319,50,368,894,70,420,435,671,417,129,124,367,365,671,124,644,951,895,933,467,908,122,448,150,36,881,982,983,83,15,815,942,60,843,541,539,190,880,883,562,709,634,861,104,271,856,619,244,997,620,759,667,31,592,141,965,402,345,473,934,944,885,778,668,943,141,109,83,340,238,49,762,3,267,139,473,469,374,401,669,244,33,986,488,182,581,482,918,335,817,615,839,552,870,508,920,210,974,838,539,412,954,36,949,662,171,782,157,807,9,720,70,279,28,764,156,296,699,442,870,288,76,720,605,249,374,902,495,374,271,189,995,798,274,832,955,324,692,389,706,881,974,803,91,487,779,465,560,46,710,825,786,273,223,33,171,168,153,152",
    "201,884,294,447,844,367,379,779,441,952,542,858,172,629,849,102,34,267,466,326,964,366,535,709,696,731,582,113,860,814,535,563,93,592,580,590,558,129,35,317,303,444,965,738,772,245,694,727,289,745,294,321,45,343,778,935,377,546,605,450,430,123,898,417,994,449,743,234,63,825,197,799,63,444,175,739,743,791,772,794,758,555,589,973,493,597,600,706,666,739,194,528,159,195,115,421,296,202,860,508,589,146,428,234,43,457,410,989,317,780,973,582,504,783,113,464,423,377,144,986,848,471,922,858,686,949,104,342,916,669,181,31,334,706,913,783,230,292,590,532,63,0,856,39,725,224,18,943,809,208,35,528,659,158,264,408,951,665,958,42,641,782,53,177,952,184,147,488,272,324,57,474,369,747,643,448,612,623,683,628,936,373,997,298,217,770,666,540,247,665,62,105,425,14,449,93,252,169,546,188,702,505,497,551,379,115,224,217,871,173,176,36,571,942,576,503,388,210,808,22,802,585,512,787,500,604,638,404,163,171,652,639,105,818,381,459,421,118,731,391,178,772,144,210,907,254,744,376,182,58,60,891,809,482,495,129,37,260,192,682,732,10,413,101,224,433,940,425,536,967,891,206,870,768,263,389,640,100,833,643,999,560,212,311,96,359,720,846,852,198,581,319,508,53,707,288,888,387,788,505,489,459,212,149,664,963,361,713,340,516,459,234,852,112,755,157,949,317,919,943,437,876,291,28,533,715,393,456,129,323,388,478,799,83,935,953,422,410,604,754,481,11,408,173,21,74,739,229,418,254,456,989,46,966,600,74,376,281,841,510,232,872,980,580,303,337,684,208,144,418,338,183,308,249,121,632,17,168,299,177,836,103,834,459,973,865,524,604,795,993,770,554,126,626,472,825,364,657,200,292,326,357,462,428,788,912,570,545,935,304,245,951,546,434,848,172,339,40,791,890,397,713,205,309,179,910,308,386,9,403,286,173,734,740,971,97,451,834,829,84,176,179,703,611,108,321,621,288,234,409,711,543,563,64,296,313,161,532,724,902,618,213,102,100,945,386,305,907,606,108,945,991,881,934,458,523,414,201,659,199,546,971,422,582,647,720,964,740,386,172,772,110,605,590,754,491,701,139,674,277,583,192,519,259,617,442,424,736,184,49,260,355,652,627,856,128,856,670,129,862,990,125,679,113,468,299,393,688,202,67,610,924,694,602,347,171,797,447,568,439,713,404,532,945,754,112,279,492,794,786,320,477,43,448,350,467,702,529,529,463,135,171,323,761,693,631,580,465,870,299,831,412,865,467,945,664,577,339,729,101,429,723,169,121,624,593,944,458,588,796,55,642,381,402,973,259,759,869,259,628,644,57,110,476,841,25,44,158,300,597,610,409,145,954,557,17,967,377,289,20,522,487,285,711,323,401,914,390,784,437,386,797,198,433,486,436,56,438,280,717,279,96,723,132,644,750,396,312,607,726,629,57,168,664,589,107,358,402,28,406,393,788,379,953,565,41,357,485,868,901,437,132,69,97,312,196,110,549,880,804,300,491,929,150,762,100,73,337,681,135,593,918,633,43,449,497,406,429,863,446,316,245,691,547,351,535,770,604,103,523,25,299,621,674,955,849,587,449,578,222,48,268,180,21,287,687,642,863,392,996,737,657,835,359,305,584,154,357,830,388,718,120,481,472,462,759,739,995,305,894,736,81,911,534,221,200,922,181,156,335,427,519,926,181,101,847,493,923,564,81,32,987,363,853,189,37,238,881,785,770,47,645,405,644,846,106,744,147,279,769,935,973,991,538,603,318,200,254,709,996,348,320,734,523,872,640,204,910,454,154,430,474,58,144,572,21,426,448,581,315,23,102,361,545,678,567,770,287,480,685,212,410,996,305,433,512,129,910,198,939,727,203,43,620,91,951,701,825,403,283,482,139,871,919,650,798,908,448,494,504,625,131,407,897,26,778,842,909,642,100,134,51,741,735,721,855,685,290,206,463,961,784,37,607,696,496,531,234,138,16,219,696,884,966,962,545,584,568,380,580,654,96,983,336,711,580,432,716,242,42,369,471,502,929,767,521,359,347,220,376,835,157,246,515,310,672,592,791,623,331,400,665,753,88,497,134,239,509,926,186,358,706,220,474,984,856,449,179,673,568,381,722,404,62,628,570,917,692,968,448,166,234,344,118,173,468,775,370,235,886,859,32,471,664,963,474,277,905,985,513,25,102,85,906,421,813,139,729,683,250,605,774,397,359,51,880,627,461,458,872",
    "383,446,623,931,658,287,202,478,43,541,210,213,416,629,288,291,768,669,589,337,446,261,808,661,231,452,365,317,570,621,731,620,593,958,584,779,137,170,338,977,239,357,307,787,43,753,86,952,748,639,940,426,613,171,287,519,173,949,814,500,735,267,859,492,255,73,739,850,886,123,81,367,637,606,787,231,865,751,963,870,716,910,679,314,953,356,658,242,102,985,809,923,911,574,443,321,404,767,71,330,736,809,39,864,846,308,483,266,935,919,683,24,899,420,885,261,847,533,629,509,785,770,641,255,563,64,978,625,520,967,65,842,464,431,645,69,976,756,46,19,630,856,0,667,592,355,429,3,166,141,25,659,435,371,389,179,540,711,358,381,803,46,795,467,587,740,523,41,381,831,641,815,301,225,476,690,578,162,264,390,518,134,218,552,248,720,940,486,453,578,15,517,702,507,499,868,785,245,261,495,85,391,133,434,892,986,384,125,914,148,176,274,444,718,597,865,252,246,728,607,275,877,379,123,420,958,787,85,593,829,336,822,142,256,233,135,412,461,372,571,771,641,102,174,757,90,450,313,65,660,500,875,349,825,121,926,670,898,844,87,794,368,425,888,842,567,868,662,250,234,829,475,536,613,294,165,367,974,99,297,911,866,531,197,653,852,285,353,31,52,997,361,980,433,961,957,241,766,604,323,337,933,270,73,251,374,67,877,511,718,182,742,829,188,41,650,345,422,140,15,324,358,730,374,255,9,144,206,148,539,896,132,967,512,835,811,97,254,83,77,846,716,593,394,412,889,291,334,321,531,453,234,861,90,375,729,578,5,423,635,313,620,983,863,956,786,438,348,284,917,801,382,759,703,867,82,570,375,717,362,463,743,896,194,793,314,708,803,558,771,765,27,481,149,33,115,622,460,248,819,907,658,948,303,824,816,202,701,495,530,705,495,20,224,812,362,429,747,472,877,975,985,941,423,798,932,994,197,632,207,942,205,445,166,590,699,970,111,90,616,105,543,932,876,552,201,120,187,311,13,790,604,597,394,89,744,558,38,464,492,19,528,998,809,739,853,740,784,290,527,506,451,974,87,944,645,27,841,31,399,826,415,859,520,689,100,839,714,744,379,964,944,696,517,534,910,426,185,201,618,246,346,289,404,7,342,458,459,476,618,336,699,675,605,898,799,576,214,501,984,117,673,807,783,540,584,457,839,14,855,507,646,63,905,766,296,993,883,398,462,958,118,338,156,217,676,948,154,55,149,298,755,908,254,656,320,13,318,568,380,937,783,35,645,220,400,276,810,39,560,923,153,244,583,8,463,425,905,497,572,599,802,961,317,580,58,920,945,345,280,551,250,13,249,251,321,412,545,826,83,880,70,828,905,348,995,324,838,642,575,206,388,109,987,547,348,154,42,504,923,80,272,501,614,277,341,266,632,450,844,412,23,318,58,739,199,69,278,40,443,105,452,124,21,37,348,936,212,260,787,6,915,171,518,541,800,52,590,8,186,510,82,239,384,488,914,841,616,660,837,502,813,598,849,832,487,499,852,759,366,200,977,767,554,47,464,508,570,414,695,787,384,50,965,174,766,986,180,532,372,71,507,840,952,575,670,515,985,238,887,731,875,435,846,846,413,307,712,973,470,390,380,415,469,20,523,124,989,905,356,936,642,878,939,766,771,486,848,853,925,190,995,481,167,878,334,710,913,778,549,251,815,103,105,97,826,347,773,425,460,651,919,835,962,202,898,909,454,963,125,674,57,909,17,462,936,718,941,691,723,55,86,210,794,292,876,678,430,91,366,630,549,311,643,800,969,506,465,471,508,454,419,839,53,638,872,26,120,879,752,505,807,486,713,386,221,312,886,763,303,585,677,407,886,539,208,270,49,272,405,247,143,948,287,979,308,359,491,892,72,563,820,22,37,234,908,449,488,230,476,635,880,526,989,92,482,249,447,255,699,439,88,981,669,819,753,660,388,955,498,698,318,124,928,811,545,713,733,554,460,980,683,411,808,697,877,689,695,338,326,892,259,52,582,838,452,386,396,906,947,189,831,898,368,28,833,32,526,175,260,33,154,710,233,748,523,242,492,494,920,424,5,609,71,100,362,877,180,187,444,828,342,823,862,878,498,211,816,472,735,217,294,737,185,613,207,445,820,500,470,573,999,693,602,552,768,735,442,351,783,795,901,441,519,80,602,855,251,350,339,890,84,213,69,222,98,46,476,105,919,519,603,154,155,619,153,40,305,367,181,146,157,935,430,691,574,145,932,595,214",
    "249,530,203,721,692,922,58,393,680,216,58,818,632,795,149,911,784,63,948,706,436,986,108,704,914,550,131,917,20,134,861,192,277,582,596,492,9,51,150,628,335,720,455,954,730,322,639,261,509,254,692,268,321,585,305,426,47,826,492,945,475,781,502,906,386,28,139,634,575,684,709,8,62,11,165,27,801,947,919,157,69,686,643,703,28,715,194,241,156,249,979,472,323,996,4,859,890,507,564,842,308,388,522,356,195,738,10,713,103,468,378,878,2,794,193,904,913,644,718,648,751,820,301,284,504,127,565,285,56,99,117,557,874,135,370,761,364,320,38,465,532,39,667,0,192,766,458,641,400,831,864,772,332,761,540,735,643,978,580,409,489,299,746,720,982,241,336,979,566,331,809,437,979,917,320,112,970,359,613,338,448,841,518,710,318,50,168,452,496,950,385,205,707,150,516,533,205,790,966,250,529,893,679,20,559,500,13,93,448,935,339,795,967,127,184,74,966,262,74,318,174,528,354,810,813,575,641,1000,786,299,774,400,989,577,111,875,890,467,118,488,697,838,497,127,233,464,22,935,93,960,837,49,503,559,75,930,601,977,726,498,458,739,807,363,827,810,917,468,516,756,425,811,393,404,293,448,828,550,766,571,410,210,313,445,985,115,124,606,748,152,619,724,635,485,683,285,917,673,970,203,795,43,144,144,748,131,246,351,606,520,200,174,877,617,794,548,610,43,907,485,546,325,205,514,720,840,614,165,466,621,278,705,803,263,753,59,453,657,960,620,646,832,98,768,257,967,346,39,694,600,328,38,122,634,634,828,344,184,13,868,728,644,723,129,223,207,644,421,22,754,713,423,921,196,961,837,195,664,478,76,768,684,316,616,697,182,305,535,287,437,789,667,807,909,461,294,511,531,54,413,398,901,814,426,120,530,172,64,707,256,980,631,186,116,759,641,535,559,702,419,985,489,692,30,520,564,583,930,583,876,610,861,521,894,279,382,331,637,121,119,447,546,289,203,558,992,417,876,684,830,460,333,270,465,703,985,60,72,294,958,618,265,497,973,366,345,7,588,734,408,631,395,888,615,518,995,746,60,280,547,241,445,151,401,899,341,577,654,379,991,844,999,364,826,644,215,154,340,775,463,989,669,825,795,798,615,637,39,879,977,538,687,603,62,407,147,254,408,441,869,371,60,144,962,292,451,567,19,99,934,573,831,462,921,209,269,927,175,177,359,43,781,25,646,61,899,259,522,526,804,589,800,438,582,387,896,98,442,19,569,161,491,923,697,770,831,200,57,505,958,133,447,123,647,39,116,547,517,620,718,580,20,484,492,439,821,936,658,587,459,764,513,945,992,707,123,335,788,721,35,606,103,62,746,241,127,157,807,535,546,449,804,699,910,306,776,495,490,10,769,434,813,207,153,685,949,679,580,949,761,26,233,38,428,416,699,9,160,461,665,993,751,473,352,598,663,292,352,856,203,447,625,594,122,974,107,405,711,977,653,254,164,909,163,90,7,4,885,682,943,505,201,319,680,964,885,414,871,535,979,254,374,610,250,136,248,710,709,941,176,125,53,231,421,976,617,648,205,361,176,309,332,642,986,978,603,661,192,342,191,49,227,518,856,789,201,689,238,218,633,875,33,8,474,787,70,284,723,569,363,653,399,266,49,209,424,236,943,463,368,15,81,677,674,632,136,367,935,539,151,562,165,919,996,883,179,18,962,447,222,479,205,738,326,252,410,428,955,945,767,732,639,445,511,906,982,565,657,533,337,625,692,595,450,573,909,434,566,171,225,974,261,177,759,230,585,193,7,420,286,303,340,293,720,671,459,257,501,497,255,803,843,266,496,22,416,920,457,918,941,117,82,994,296,251,233,778,808,320,754,479,549,596,593,475,238,850,992,861,736,698,592,633,238,216,854,392,94,832,711,55,826,61,325,696,410,707,777,421,838,547,407,156,734,141,942,851,274,108,625,792,906,227,376,134,160,534,325,925,977,677,564,153,789,434,459,510,456,837,678,925,855,513,478,459,984,610,574,737,957,331,842,596,900,433,338,997,178,614,848,18,186,653,192,450,52,835,985,711,402,623,514,579,490,656,304,751,710,169,462,427,817,503,655,24,931,338,576,307,987,858,438,429,816,640,196,778,503,9,131,433,646,293,577,857,550,40,699,183,898,180,149,618,262,438,878,143,832,939,211,498,584,139,227,724,543,222,516,105,542,965,183,893,141,773,276,826,796,116,910,628,183,594,950,864,552,124,877,275,844",
    "69,622,717,183,135,213,855,292,725,593,727,463,677,890,199,218,627,543,295,361,56,992,19,54,436,810,300,496,33,855,878,481,721,430,589,871,327,416,725,332,915,536,197,994,583,614,392,61,34,760,671,180,790,413,412,551,735,282,403,652,377,621,420,218,83,173,732,662,347,392,60,643,673,292,654,873,123,354,952,260,206,440,929,377,446,890,349,891,536,418,301,66,607,175,697,879,590,311,66,886,638,316,764,640,536,210,197,402,719,678,102,339,82,716,268,913,923,688,111,83,838,952,468,569,331,117,903,616,465,851,239,928,977,997,954,741,720,383,784,810,985,725,592,192,0,280,349,302,448,648,272,151,764,487,946,675,765,581,765,455,987,180,685,753,774,333,990,896,65,35,733,562,859,614,875,665,834,294,533,117,450,885,455,695,637,60,97,714,737,576,651,839,626,688,829,133,850,801,748,411,410,388,695,107,207,481,276,919,915,389,202,390,234,653,994,578,558,120,852,514,45,692,464,860,48,801,557,857,671,804,603,61,445,141,862,6,897,990,910,294,96,786,713,765,722,977,258,823,968,348,393,581,151,876,443,94,746,111,615,794,914,151,791,709,271,442,877,4,683,982,191,173,725,995,964,597,491,975,887,169,525,8,100,912,547,498,129,484,282,178,386,961,209,852,73,931,529,183,43,211,38,145,306,999,859,668,933,631,476,672,470,419,211,191,755,111,164,662,506,20,565,882,947,978,93,883,25,375,441,800,683,673,191,353,984,221,228,159,13,395,740,663,659,941,114,414,511,580,998,857,537,343,681,826,544,176,696,328,214,157,804,385,950,99,71,744,654,576,230,106,815,891,489,412,587,579,605,299,718,505,458,167,154,899,886,721,986,206,47,987,798,326,981,445,482,277,644,615,414,136,815,69,447,730,284,440,849,529,547,712,414,96,32,587,164,705,738,893,661,270,316,71,392,488,92,371,697,447,993,242,17,54,169,642,81,10,292,591,312,574,631,548,856,912,650,888,318,838,383,203,465,625,418,7,863,8,536,299,94,810,487,944,738,773,528,327,479,965,207,815,415,656,991,732,285,125,858,7,232,5,706,595,513,413,222,26,37,587,26,336,214,678,915,260,950,24,925,870,780,184,930,630,84,615,626,44,349,436,527,89,336,273,524,697,584,50,130,842,561,267,592,487,237,493,383,762,75,917,386,132,233,487,107,258,975,730,300,399,741,650,393,237,935,530,801,755,141,593,614,20,503,271,886,553,333,522,188,635,836,752,731,412,42,946,585,333,126,758,645,719,473,201,957,801,423,567,852,840,765,564,389,186,517,11,550,354,214,161,20,832,388,987,434,625,326,544,631,76,91,651,29,21,801,726,400,27,39,272,230,912,466,46,499,642,545,141,818,378,422,152,614,714,362,634,597,651,778,148,387,208,142,796,853,363,374,721,689,796,263,52,463,925,162,558,921,946,959,646,728,750,822,183,334,581,458,791,912,73,202,788,851,647,454,577,921,567,945,383,374,497,593,932,576,152,676,378,376,13,653,166,139,918,594,448,24,863,365,666,926,749,141,184,605,92,693,80,738,246,414,480,82,721,875,906,636,421,589,502,165,641,211,705,48,982,830,270,933,827,901,810,663,614,356,488,499,435,898,456,966,815,848,247,677,442,125,417,70,604,619,633,252,774,799,44,8,157,24,866,297,382,203,866,851,477,740,52,317,247,518,753,87,121,152,904,841,899,445,570,864,460,361,126,307,720,613,493,726,781,583,90,156,860,222,829,53,972,70,214,367,970,742,451,469,137,619,929,19,394,836,904,657,102,808,870,771,525,238,285,41,634,33,700,63,439,413,564,441,514,10,527,500,315,711,635,418,126,187,105,324,406,554,240,217,832,324,31,299,643,867,252,921,824,638,245,942,736,467,262,70,789,406,656,118,321,711,632,945,282,315,89,203,131,663,753,380,837,10,867,309,35,623,666,755,166,969,204,292,921,866,215,749,579,971,960,225,275,83,702,990,403,955,648,505,340,217,518,857,566,648,618,955,790,938,557,442,603,351,691,685,40,389,224,612,166,613,51,9,969,202,160,834,651,852,849,476,436,193,836,572,966,848,507,948,868,820,434,398,828,151,334,700,346,708,873,729,218,640,131,300,964,491,615,812,551,19,167,979,748,258,436,638,805,502,332,45,335,410,591,937,195,202,985,671,415,294,635,25,624,463,540,997,622,448,758,870,120,233,960,793,213,949,965,536,596,257,118,767,700,286,868",
    "263,121,762,377,85,359,233,552,146,909,292,238,774,766,493,471,691,301,626,85,554,650,356,775,140,451,222,11,638,29,356,528,878,383,421,51,309,486,834,611,776,495,511,851,614,785,76,826,584,726,807,415,295,276,222,602,883,424,37,65,300,949,39,533,124,403,215,85,855,250,449,346,275,183,719,7,755,620,519,540,319,736,404,830,520,961,512,391,833,224,87,739,427,949,703,628,924,772,486,127,481,668,223,999,71,954,535,394,698,265,866,791,132,86,544,923,832,70,977,254,49,545,440,88,570,701,847,668,905,342,768,211,491,814,741,173,474,674,651,839,622,224,355,766,280,0,895,884,467,577,178,929,230,93,436,301,576,353,539,912,254,239,264,398,220,704,278,942,233,621,649,220,447,809,874,910,638,903,706,14,994,170,385,135,330,308,99,242,555,434,477,766,635,815,193,166,268,854,266,749,837,302,925,484,683,789,987,762,704,850,582,316,195,938,170,801,553,6,370,9,183,341,99,133,207,460,444,551,722,645,181,920,906,716,311,440,383,846,350,487,661,396,489,836,951,115,992,442,717,927,647,604,58,63,575,849,722,524,630,657,306,389,496,619,787,254,46,213,621,380,788,436,424,883,472,212,188,811,777,582,540,224,262,570,779,348,662,238,452,839,335,833,565,561,867,678,391,365,457,698,858,930,905,732,130,996,904,571,751,897,790,86,760,476,956,141,769,872,420,343,221,224,304,451,27,295,332,918,303,45,287,695,694,179,843,375,688,764,354,655,667,272,651,559,328,357,305,990,968,821,771,268,554,510,311,117,349,161,913,298,368,830,752,776,951,320,189,485,24,94,708,154,501,215,455,367,189,7,853,286,749,298,21,509,233,148,997,56,598,542,838,149,32,815,917,755,71,878,459,656,991,591,801,157,971,130,410,141,1000,99,661,960,782,1000,892,1000,247,636,507,338,890,346,132,658,713,126,503,523,40,363,633,988,985,759,869,534,399,227,162,377,322,203,404,51,676,657,64,942,358,442,53,196,927,962,90,215,790,696,144,292,747,840,222,992,798,368,261,367,80,385,725,305,554,123,677,838,785,521,894,647,401,975,12,484,466,909,282,341,752,737,730,291,99,350,732,664,834,641,393,335,473,904,625,287,110,66,930,58,865,760,46,992,620,942,252,530,800,354,703,768,569,636,946,402,369,903,273,375,633,807,362,622,392,225,549,599,774,929,427,959,386,380,401,884,814,513,318,380,671,776,754,552,381,652,268,671,606,524,748,627,585,802,616,926,413,404,405,298,661,253,452,90,775,563,947,856,367,206,695,638,633,992,588,706,669,21,247,760,437,5,720,932,952,589,911,22,597,296,453,56,727,87,622,928,179,260,805,510,994,90,233,821,654,18,910,177,194,674,78,83,96,377,643,536,755,462,676,966,895,329,545,106,504,575,916,689,825,179,981,850,69,136,788,484,56,735,853,879,137,515,725,86,725,59,698,94,506,446,986,787,466,637,754,179,203,176,63,890,897,980,888,530,923,738,718,831,830,166,779,351,906,310,994,503,409,232,78,738,230,747,668,108,129,240,841,416,403,376,841,10,450,67,182,112,346,898,667,695,484,314,9,994,616,54,662,902,260,565,360,809,122,430,196,782,472,99,491,952,247,23,344,22,57,856,917,905,513,769,12,989,418,284,430,78,641,903,514,356,392,713,463,641,266,33,405,369,979,157,592,404,790,740,267,669,739,666,303,194,506,875,483,323,33,774,356,207,7,563,652,756,237,487,91,470,607,244,833,314,905,238,372,704,414,409,878,107,576,166,739,325,161,310,776,37,624,719,508,690,72,891,229,231,359,305,716,673,121,645,846,546,305,407,297,750,921,999,433,372,744,543,630,183,604,790,643,973,388,472,333,446,237,715,625,267,165,713,524,250,503,763,664,735,645,444,441,500,92,673,295,474,167,474,618,111,865,37,638,412,247,960,320,662,752,11,663,164,208,712,950,399,700,893,672,564,175,511,156,969,563,632,594,128,46,467,493,305,987,681,752,140,80,922,117,34,402,845,573,557,236,122,158,173,627,220,816,889,398,140,199,939,2,719,60,355,430,319,877,705,256,858,130,703,812,469,114,503,858,858,320,698,413,372,934,661,444,990,224,571,117,221,171,431,561,294,728,377,290,323,948,983,801,826,624,230,879,162,337,482,906,339,32,87,583,263,365,488,494,275,944,538,250,832,541,59,992,936,20,641,662,534,930,136,765,117,984,150,282,832,97,835",
    "415,742,715,333,377,961,826,119,201,164,704,241,433,557,908,841,447,230,680,130,746,515,898,311,805,167,15,736,322,429,171,470,758,770,815,506,302,27,274,740,781,546,132,803,853,24,397,742,372,697,19,712,762,425,4,129,418,201,985,639,149,368,144,332,545,620,276,882,181,245,729,56,957,612,518,902,882,651,618,357,609,276,269,50,630,831,813,7,246,440,500,87,670,867,369,310,407,867,475,545,326,204,78,413,787,374,883,870,475,651,591,330,790,978,686,845,457,844,688,332,179,418,844,423,807,289,288,9,124,578,964,112,832,814,328,731,837,250,461,90,807,18,429,458,349,895,0,175,750,29,983,188,116,316,807,870,591,444,992,889,661,255,328,437,612,137,461,95,195,892,296,960,917,928,256,942,355,680,931,409,881,424,797,629,172,589,384,29,33,28,207,577,731,486,669,485,986,158,798,649,540,132,806,574,337,485,951,837,243,988,346,370,128,938,959,895,135,717,502,930,666,430,389,446,488,316,804,822,133,303,515,646,512,786,67,260,50,479,873,466,517,478,21,177,502,870,462,105,609,802,749,79,843,209,133,538,174,907,179,890,317,81,315,914,757,251,618,849,425,497,179,442,632,155,514,4,921,516,773,403,765,847,855,386,300,342,248,301,304,199,172,906,541,430,652,609,273,319,227,18,722,301,757,362,544,214,421,212,373,876,181,375,992,549,844,361,699,388,70,412,150,965,543,387,975,127,264,106,853,470,549,285,687,278,278,604,944,741,718,900,683,382,708,60,976,901,505,369,680,962,879,594,528,295,389,889,272,299,327,886,130,990,995,752,918,467,658,74,495,966,566,45,750,548,175,95,948,835,910,931,57,529,341,512,286,929,592,745,968,287,723,74,168,468,379,694,742,358,567,755,443,142,807,677,126,536,538,530,222,141,688,80,76,760,50,640,786,325,125,26,335,651,258,167,899,160,760,264,290,630,365,880,204,292,290,685,546,434,965,826,472,564,148,860,69,418,123,782,515,86,650,82,191,625,550,609,139,242,330,4,215,789,413,491,611,112,58,818,711,658,922,351,477,422,409,640,940,817,884,150,410,270,372,77,123,626,862,20,710,317,487,613,296,62,52,271,18,539,43,563,752,44,63,325,904,915,898,481,163,76,653,728,962,539,769,510,257,10,788,985,774,888,576,760,675,9,670,9,842,691,467,153,825,903,696,73,159,405,177,215,749,5,99,881,771,322,4,859,606,624,29,948,992,796,348,357,320,170,60,136,408,142,741,500,838,79,334,305,301,190,175,17,893,696,564,420,468,677,16,918,889,618,238,484,862,690,349,488,132,975,454,706,54,246,205,817,400,426,230,981,561,157,343,929,551,869,375,811,82,98,81,477,588,215,824,16,387,891,812,595,56,952,909,292,613,453,561,983,991,516,43,766,727,659,625,724,934,934,582,542,979,713,569,33,797,816,466,112,361,534,764,654,864,186,512,687,290,700,726,348,959,227,142,785,197,320,376,521,81,745,573,570,813,414,906,986,538,498,628,400,414,693,294,146,711,384,951,335,564,714,583,236,418,864,775,551,541,112,190,994,406,772,138,667,207,994,204,570,419,410,209,80,252,718,519,59,933,432,123,639,762,702,125,498,742,212,570,230,767,796,427,296,680,488,978,200,445,663,152,395,661,998,116,470,204,113,866,533,119,310,872,477,991,852,735,250,84,730,890,899,305,864,779,368,570,814,984,587,820,868,912,805,241,930,510,647,951,144,595,121,610,609,22,961,538,625,71,854,779,838,443,580,329,76,769,458,337,642,614,185,397,327,762,373,901,362,180,283,875,286,591,378,98,49,557,95,114,851,819,873,664,738,506,369,683,241,329,28,216,378,257,468,719,445,406,748,76,195,167,398,115,142,93,358,162,163,175,64,926,693,227,336,42,424,43,579,631,366,458,10,240,688,978,557,862,580,58,95,248,529,347,701,191,104,243,828,591,344,85,221,537,496,818,453,850,36,730,62,695,945,475,366,642,493,332,615,759,440,228,205,507,948,277,304,364,317,775,175,208,242,196,177,382,239,384,977,49,861,163,83,881,263,45,847,993,811,98,599,38,372,922,109,524,255,362,958,404,1,667,933,585,855,682,913,324,509,577,167,893,570,433,727,636,557,400,396,277,449,404,118,334,393,881,860,652,199,188,771,627,264,759,395,676,339,664,529,424,372,35,587,609,32,503,353,768,909,920,187,792,794,312,478,348,206,732,7,629,111,126,99",
    "961,378,496,839,546,319,245,960,407,816,596,242,108,208,733,909,74,373,535,39,643,2,23,963,679,72,167,489,717,127,899,403,237,18,548,48,913,919,600,167,703,589,303,469,584,180,674,536,906,517,112,408,274,432,602,131,93,63,213,161,830,667,290,878,585,459,420,951,534,321,921,795,745,818,691,211,336,664,697,706,975,860,780,767,114,91,908,947,618,217,673,363,801,287,109,944,125,584,521,189,481,839,822,858,324,449,553,357,444,653,719,115,328,540,581,832,561,952,179,271,504,663,72,611,108,364,641,595,842,178,297,379,411,108,342,570,698,210,70,742,872,943,3,641,302,884,175,0,394,34,634,173,564,256,603,255,767,381,672,971,116,994,782,715,168,710,459,29,719,469,894,808,590,477,140,281,477,452,327,602,228,927,942,746,922,815,928,695,394,57,670,103,610,282,228,503,263,961,738,443,814,234,556,544,86,390,40,734,394,635,624,888,966,967,203,67,473,795,901,888,695,525,949,677,257,21,387,322,446,753,727,249,43,699,886,33,916,179,210,416,521,390,708,916,194,106,379,361,146,943,226,302,280,188,212,729,613,170,697,328,230,266,411,288,607,17,230,613,350,357,988,666,49,958,940,157,330,772,782,900,535,462,460,214,418,993,961,932,707,427,757,326,232,26,639,275,567,586,301,221,918,573,809,316,928,421,832,223,871,860,620,297,285,652,817,194,631,9,810,197,865,495,172,107,274,250,143,357,361,36,332,495,347,63,94,409,244,64,759,55,420,728,451,333,855,216,253,346,955,730,19,833,865,635,380,501,488,691,759,910,702,355,277,105,281,382,932,710,821,901,343,936,381,898,515,394,448,736,448,818,30,59,53,86,423,425,820,469,494,342,206,841,6,269,257,291,240,471,808,391,91,612,206,893,217,399,976,391,159,619,394,17,303,337,931,697,948,870,648,136,238,126,36,153,625,985,205,893,227,265,332,952,158,696,835,83,919,920,214,291,79,441,697,502,414,45,892,207,156,778,145,142,187,640,529,541,293,210,715,698,673,840,187,15,658,802,313,729,261,830,838,278,230,4,227,870,605,889,581,347,390,749,136,530,38,630,195,115,691,279,971,767,759,910,894,582,903,571,866,279,478,290,537,944,667,466,432,528,571,436,119,68,987,107,844,307,17,598,658,909,884,10,981,159,883,139,994,462,401,103,102,136,616,520,505,483,150,969,979,736,742,40,225,80,807,922,38,982,523,780,833,95,77,900,617,813,995,865,551,698,317,657,181,680,213,821,341,314,591,738,582,565,787,157,337,615,933,409,620,553,648,867,491,679,814,575,706,358,789,694,194,190,744,257,347,676,855,392,450,90,491,438,260,959,483,107,792,459,46,739,291,115,948,915,553,305,614,663,978,414,776,467,661,788,902,774,887,854,975,54,964,937,291,286,84,41,669,151,717,199,688,989,221,662,280,85,249,203,639,429,137,206,66,536,143,438,761,633,697,16,705,279,313,799,326,157,961,988,527,540,274,967,900,615,613,945,665,97,62,404,454,578,436,675,88,474,354,291,722,933,20,186,853,717,906,883,177,146,676,532,94,846,933,910,529,634,431,1000,29,941,368,896,633,115,863,787,309,916,7,156,359,619,977,831,568,842,203,324,302,263,369,118,639,186,428,976,359,628,140,827,707,122,568,490,709,655,436,819,683,518,339,448,197,676,917,65,572,531,88,991,597,553,493,845,840,269,619,626,405,505,852,702,580,164,348,118,123,599,359,439,263,987,341,777,190,699,263,905,44,548,560,458,280,979,296,353,760,891,674,596,93,297,998,913,12,336,466,235,346,157,390,493,607,561,796,455,633,788,77,628,156,73,633,114,278,17,113,730,70,738,759,574,104,706,942,693,174,219,831,785,264,71,229,860,469,700,971,395,826,142,228,150,229,681,997,433,485,954,618,61,725,352,5,565,295,119,443,387,641,567,251,375,753,932,870,639,942,783,457,771,507,721,261,522,356,15,578,674,352,231,145,690,800,534,910,343,807,780,233,504,367,49,302,521,130,804,98,629,599,22,800,766,248,766,810,559,262,878,252,183,120,284,594,281,570,420,118,67,608,515,372,361,778,390,646,213,915,928,991,72,129,172,488,268,324,365,896,983,26,698,185,146,751,931,271,343,256,217,169,756,577,957,421,292,719,538,741,970,992,731,933,563,54,762,606,353,744,344,735,789,631,346,976,925,25,820,515,898,264,325,725,938,942,696,580,265,449,917,172,147",
    "56,816,442,570,260,674,408,292,364,460,371,389,677,680,864,306,46,437,14,581,333,397,769,388,173,907,315,173,812,938,782,605,217,751,727,218,697,494,583,875,436,468,647,285,924,286,603,995,718,785,328,429,309,548,637,817,293,28,796,24,710,36,302,185,979,175,388,22,44,769,834,6,29,109,837,16,656,802,227,542,313,918,824,426,343,344,911,405,58,274,719,794,832,76,733,157,221,973,128,28,880,915,880,248,116,792,125,174,413,514,252,607,326,645,970,411,236,171,291,223,884,525,577,237,902,164,129,209,97,628,649,169,961,402,958,221,508,786,834,567,708,809,166,400,448,467,750,394,0,58,373,558,146,141,899,379,805,868,404,349,749,813,976,903,799,301,741,994,915,33,743,909,423,786,362,897,587,768,937,271,421,88,878,260,218,665,514,118,847,333,418,617,962,571,537,168,719,941,103,432,886,724,929,713,74,376,732,561,81,31,741,143,266,799,953,605,967,39,888,478,767,511,150,337,92,656,446,469,266,657,798,788,922,942,772,596,637,110,143,741,914,508,737,642,396,945,91,637,684,468,800,974,767,789,969,794,349,102,387,366,816,632,763,767,381,331,152,246,74,111,948,200,495,748,867,411,900,617,138,243,822,733,933,100,737,968,795,76,648,657,989,892,639,451,714,583,31,874,343,596,918,450,749,458,390,198,255,774,105,182,458,729,747,661,990,606,936,182,770,24,694,24,781,776,407,609,677,544,275,885,657,19,146,62,779,929,289,161,986,41,617,693,997,452,168,921,131,760,867,189,171,872,675,164,293,33,115,682,327,566,592,332,760,923,249,372,906,787,196,320,856,875,280,727,806,550,974,360,108,92,178,332,420,548,217,137,955,605,697,968,999,732,442,693,515,950,933,919,412,451,69,248,453,667,737,294,241,54,799,816,736,767,34,256,414,71,941,756,773,329,201,124,654,903,349,14,315,102,386,807,483,280,839,348,705,130,6,518,776,123,265,637,736,987,203,42,175,707,949,935,827,375,858,996,426,192,162,278,943,241,676,763,916,979,532,616,291,813,854,918,200,631,4,437,236,677,490,297,233,442,978,822,158,108,74,72,411,25,373,463,100,619,826,103,714,424,420,930,503,347,251,595,157,640,985,630,804,675,934,900,781,770,874,422,787,550,210,537,604,306,229,598,385,635,672,131,800,382,285,994,710,30,368,240,553,320,809,956,734,833,517,176,974,97,491,956,9,669,313,632,923,37,511,441,453,131,91,779,164,299,506,5,786,993,775,768,169,588,738,930,83,82,129,399,126,665,274,739,540,979,584,110,552,826,103,205,137,927,758,771,656,872,745,504,649,968,395,679,447,775,572,616,301,223,764,253,986,415,109,46,595,548,675,679,74,949,457,308,457,36,439,231,897,793,968,894,528,63,900,406,920,994,815,155,916,296,416,91,937,325,240,465,180,315,606,869,442,818,725,889,157,569,974,8,860,21,670,503,94,147,527,565,960,968,284,660,678,438,524,720,862,5,754,811,991,655,323,184,637,401,349,644,367,14,50,885,87,285,299,684,685,107,830,283,179,530,662,390,712,696,975,505,373,69,601,74,912,286,52,160,272,364,813,302,68,758,168,77,611,399,958,271,409,454,539,377,765,503,927,840,45,35,195,303,514,232,810,700,367,248,592,592,162,302,227,95,584,958,468,615,375,234,105,542,530,842,640,128,553,307,778,95,605,18,527,133,623,806,890,622,764,949,95,65,652,901,680,231,549,948,902,143,763,207,299,941,690,770,416,206,288,266,843,624,222,360,256,202,780,788,261,823,55,67,600,792,167,649,772,380,373,954,38,635,772,609,798,58,231,469,311,928,159,641,2,630,764,567,256,217,8,447,887,908,746,444,463,549,504,672,387,762,353,413,499,56,926,980,766,94,660,903,479,92,305,990,567,291,596,508,115,939,615,300,538,50,54,675,337,371,91,923,269,634,450,288,401,184,943,420,897,839,335,613,655,936,199,82,122,407,496,550,909,683,45,671,245,385,158,245,791,779,281,52,202,599,629,236,678,69,388,552,158,149,161,694,336,721,196,908,673,220,84,259,399,233,488,833,178,761,375,260,443,247,815,988,843,701,887,989,375,128,997,516,337,566,498,909,822,815,264,59,243,947,222,921,325,471,454,434,170,417,725,308,639,147,705,105,993,795,610,476,841,733,367,431,458,810,325,161,13,906,549,476,841,509,345,549,983,880,141,91,474,957,905,711,276,351,170,224",
    "755,616,11,636,736,229,345,334,388,891,828,35,398,979,180,185,598,17,267,475,448,983,409,678,582,497,733,399,351,920,944,897,361,115,276,753,22,798,59,997,557,965,83,307,581,790,216,996,155,256,276,201,541,741,568,634,501,360,699,853,577,339,797,603,27,227,226,596,845,251,698,286,202,680,800,12,927,575,238,899,667,520,109,8,309,685,978,461,490,977,660,210,487,976,707,712,149,464,244,556,198,125,237,423,796,616,445,453,771,236,973,169,935,975,535,368,544,278,489,400,110,399,847,35,239,257,580,470,221,81,838,714,574,165,951,121,369,444,674,459,650,208,141,831,648,577,29,34,58,0,570,306,342,369,497,748,914,766,767,883,184,458,652,530,497,156,820,900,545,348,527,180,164,684,541,841,434,56,512,407,853,999,745,246,634,19,571,627,691,541,814,419,950,360,457,448,965,50,757,113,836,76,324,660,516,346,239,363,354,51,895,607,891,325,113,258,668,92,910,498,136,196,17,521,497,544,864,572,235,251,808,928,41,547,483,575,558,542,891,163,555,687,995,139,897,146,695,940,679,881,208,98,732,495,891,129,796,289,682,388,901,521,719,153,114,404,59,690,509,855,419,496,12,444,39,228,410,366,443,205,4,523,367,867,747,79,534,172,603,235,934,396,487,208,968,619,148,241,110,148,821,836,176,249,397,429,30,834,865,54,143,523,368,26,31,568,374,607,617,44,319,288,441,319,81,486,906,217,469,439,195,922,799,965,80,232,126,120,12,388,334,304,930,748,545,647,329,170,664,374,173,880,616,693,48,882,610,573,586,981,342,261,928,330,729,855,17,813,865,829,894,932,467,402,802,952,712,962,847,548,379,824,955,376,840,740,489,296,807,513,973,50,138,629,973,793,431,183,798,598,950,332,583,536,444,220,891,393,720,111,823,113,676,955,870,708,539,77,266,350,970,451,300,821,250,684,538,562,268,150,244,898,426,336,712,764,303,86,267,663,299,227,261,591,827,290,699,784,185,101,621,677,779,427,591,202,221,565,615,367,536,809,672,968,524,207,738,576,374,21,610,439,590,983,522,254,665,507,995,62,882,214,605,266,241,862,32,879,589,993,382,454,240,987,924,695,735,100,871,63,630,764,895,839,285,142,161,900,775,226,28,134,909,485,503,192,568,895,407,650,406,371,105,848,759,441,426,365,446,220,607,954,279,290,947,379,828,811,227,992,645,425,852,640,525,112,35,107,970,654,429,649,582,957,797,478,194,120,878,477,167,587,237,2,427,604,743,814,564,84,473,195,967,688,700,17,258,962,182,198,276,342,219,770,44,163,266,397,625,386,675,444,123,264,56,646,907,351,737,185,340,152,913,390,75,165,315,14,903,645,232,465,606,280,880,615,192,464,497,305,493,487,255,450,569,874,366,735,841,64,80,300,671,556,262,46,484,420,929,836,573,632,287,749,629,782,86,682,777,399,25,498,715,146,932,560,629,379,163,779,707,31,535,101,225,266,780,142,309,717,180,17,271,435,453,969,338,228,16,467,831,47,802,394,396,666,71,267,771,949,413,312,314,856,368,790,81,370,144,896,923,42,316,242,223,121,860,497,398,809,2,723,322,968,312,437,657,508,273,293,938,581,879,110,807,468,808,946,85,679,317,231,517,103,319,292,782,550,28,447,12,967,27,520,232,24,125,728,314,804,178,345,376,901,254,792,227,517,358,959,243,886,264,910,86,781,620,87,251,462,405,45,422,388,387,413,586,491,784,796,178,508,806,48,908,425,643,562,748,512,140,633,901,437,361,741,525,698,122,183,72,993,151,38,56,447,324,518,710,156,475,616,604,905,502,621,735,671,392,750,525,282,64,136,261,333,839,674,221,420,327,497,100,136,680,84,227,367,967,588,966,4,716,496,27,830,377,768,234,197,624,609,810,586,160,261,848,581,184,796,244,939,892,834,861,403,215,23,102,800,953,401,570,174,790,363,451,145,832,779,89,455,822,809,779,303,920,901,581,171,995,555,258,173,69,419,743,325,443,638,917,73,414,975,558,823,464,623,299,7,743,993,49,950,119,139,608,48,456,954,395,761,951,477,916,4,145,535,168,7,324,880,138,527,137,638,641,124,99,128,243,801,926,263,254,163,510,980,615,573,563,480,362,35,105,441,602,601,616,197,853,211,71,12,744,294,248,172,252,574,831,849,299,605,274,599,896,210,273,156,10,705,76,729,559,31,415,111,826,222,748,925,178,641,999,93,524,509,503,600",
    "616,307,113,296,21,245,736,973,855,999,238,682,220,556,164,587,475,381,211,889,739,215,399,497,29,670,156,499,276,758,520,850,815,540,442,295,793,244,274,60,48,937,686,790,589,673,723,737,204,652,290,238,51,799,820,212,496,937,547,336,574,692,837,400,122,52,346,268,679,204,735,766,592,765,928,544,933,681,738,424,904,380,720,225,366,913,870,225,216,287,177,586,893,773,527,315,642,554,885,849,630,365,124,9,371,253,826,657,597,332,199,884,810,238,195,89,400,988,634,457,481,490,938,622,444,423,47,701,369,484,340,459,551,866,631,834,431,447,322,333,242,35,25,864,272,178,983,634,373,570,0,383,751,268,898,485,281,371,874,492,739,512,372,100,918,266,184,422,710,796,183,578,107,21,290,868,934,987,978,305,366,629,305,615,791,949,79,988,215,727,245,574,373,544,95,651,32,83,355,42,200,538,527,493,222,325,560,159,33,850,283,464,865,364,734,502,275,643,251,482,639,759,663,539,85,71,875,490,969,81,875,731,246,598,548,958,879,132,994,145,452,404,991,939,50,144,500,162,527,957,384,268,647,106,810,257,740,527,624,860,115,320,95,951,56,849,589,690,557,696,684,579,607,747,510,810,375,272,627,378,20,978,953,240,178,691,199,534,408,767,265,642,339,638,620,341,747,349,682,72,441,313,236,478,698,347,969,770,220,18,509,992,515,688,372,103,336,683,856,312,3,851,570,684,874,998,948,597,240,944,542,886,248,67,707,638,273,848,261,952,405,346,983,272,339,848,621,933,288,945,515,806,423,343,460,147,76,400,505,138,953,903,734,559,351,685,730,340,122,207,851,953,661,803,620,743,990,601,165,236,360,253,668,691,851,53,174,66,719,611,217,245,455,171,898,318,300,983,873,797,207,931,155,588,738,674,876,678,845,548,256,388,106,279,295,417,703,740,259,530,995,154,628,867,265,941,967,51,322,842,824,845,842,100,218,426,469,592,600,560,129,221,907,78,979,52,739,193,398,87,919,621,226,45,399,763,604,408,719,684,902,563,862,423,753,995,516,360,713,947,257,228,402,633,866,685,264,44,600,327,712,176,872,356,425,104,636,545,204,236,286,537,307,941,156,923,397,214,948,492,488,869,503,791,293,219,859,895,129,145,79,990,272,892,624,466,892,686,968,716,766,156,355,477,740,150,243,932,900,804,502,136,165,466,212,147,123,276,715,619,764,862,339,757,127,836,546,329,351,389,231,148,385,634,492,611,94,338,637,966,14,326,362,810,471,68,720,186,154,75,895,477,28,723,841,907,197,367,410,184,17,527,660,715,166,251,962,459,303,807,893,984,391,59,208,58,216,458,87,441,570,78,592,840,600,616,269,20,763,313,110,642,485,674,121,370,598,937,498,262,303,126,971,799,31,902,575,43,550,366,462,945,324,232,852,819,304,101,653,237,859,287,667,140,498,455,255,646,173,545,389,923,213,393,69,475,949,288,375,285,945,622,625,580,904,77,422,198,907,780,85,879,857,965,762,590,381,551,65,213,958,732,248,120,196,930,409,983,674,92,282,838,570,61,20,318,47,927,138,456,393,14,883,79,469,196,98,647,81,762,455,857,203,496,359,172,636,425,727,781,33,594,231,787,594,855,59,287,151,747,279,132,486,705,113,294,355,469,996,498,648,30,731,577,266,57,834,923,647,680,540,203,803,860,623,625,295,230,223,229,742,369,181,690,815,941,190,756,599,102,563,205,351,678,13,734,322,964,829,853,103,851,539,498,864,765,610,872,40,144,325,949,493,457,574,522,544,714,342,648,319,9,802,184,939,814,867,942,800,479,427,153,727,136,132,914,659,871,407,400,733,569,469,707,276,252,943,76,420,845,259,490,617,292,71,862,47,530,523,22,821,487,295,236,792,661,126,686,510,522,917,76,40,877,699,558,455,211,884,781,227,515,519,642,988,178,620,463,134,364,854,280,162,745,803,395,565,984,335,523,68,460,722,347,898,18,874,727,592,568,226,189,904,974,472,838,587,903,768,247,610,890,33,261,388,433,284,937,633,81,311,947,947,774,575,143,654,606,744,284,997,863,53,485,199,172,982,411,98,833,402,986,81,946,103,171,646,195,19,811,18,189,599,722,312,429,808,228,173,622,65,187,114,418,53,809,191,765,418,62,444,709,706,671,706,52,778,522,2,152,926,594,824,957,485,476,468,641,528,605,235,687,882,103,219,770,476,170,936,341,618,768,655,806,835,59,665,136,486,328",
    "27,115,751,301,40,848,303,949,747,294,415,165,984,753,932,609,183,273,999,658,558,159,616,446,752,26,181,643,920,980,879,649,825,947,868,134,154,93,801,217,252,975,152,998,775,665,812,952,894,395,835,389,993,695,754,145,73,218,279,434,535,374,442,543,245,173,586,18,64,226,555,47,464,737,104,311,610,567,910,164,919,659,457,12,305,515,556,927,719,830,498,706,679,269,868,84,465,781,968,834,994,38,668,705,997,658,672,843,954,602,687,305,37,567,122,940,734,919,494,387,559,972,547,210,302,309,781,915,151,353,572,803,988,195,134,29,734,977,907,780,242,528,659,772,151,929,188,173,558,306,383,0,516,490,109,547,469,445,852,514,696,87,305,780,471,514,219,823,918,345,968,771,782,333,269,754,934,301,101,866,110,18,988,780,136,364,207,816,478,465,183,760,912,964,85,738,79,903,662,751,232,736,422,714,836,610,77,144,864,17,891,63,944,293,412,437,487,862,842,300,116,534,726,895,371,617,961,440,149,838,84,570,706,4,756,557,785,121,6,861,378,502,686,616,507,356,715,983,1,997,667,99,869,109,402,846,978,437,954,302,337,898,301,76,695,626,70,162,635,300,164,303,755,310,62,426,298,232,643,300,419,254,200,305,862,628,946,862,787,471,924,752,296,718,366,856,675,145,283,282,563,966,765,409,768,927,863,673,621,58,910,355,323,246,368,774,610,766,850,981,769,465,776,461,433,794,466,798,922,62,170,681,649,933,210,558,171,887,421,231,663,799,490,867,55,915,958,646,849,97,130,836,697,733,619,126,207,224,438,738,272,724,442,244,45,82,426,61,729,370,570,434,936,537,519,357,906,367,805,814,557,229,660,968,70,530,184,845,473,78,559,839,71,637,500,173,506,824,399,969,764,551,654,679,109,847,257,366,392,159,267,110,46,662,571,833,786,475,758,353,696,424,696,615,61,878,328,685,255,529,58,94,804,785,96,53,747,409,41,284,330,391,187,24,843,359,593,862,377,988,348,968,818,571,455,57,894,439,262,65,194,407,538,344,186,593,93,737,234,299,97,608,947,182,204,504,603,394,911,614,137,977,666,975,768,146,438,412,774,203,499,640,862,585,228,636,938,892,607,337,721,353,998,186,20,639,51,446,641,747,425,819,574,228,801,889,888,145,270,217,381,247,844,176,323,249,494,516,431,969,324,173,50,302,826,99,603,633,352,585,394,535,410,377,562,13,212,669,948,33,85,131,758,619,249,375,153,351,260,348,26,290,288,437,64,718,283,548,435,190,989,104,388,895,749,961,490,97,927,656,830,204,854,165,489,810,510,530,269,142,765,34,70,400,409,878,949,65,1000,290,747,921,59,555,339,974,739,765,159,811,227,141,500,431,336,779,145,224,888,621,954,254,188,603,555,903,180,70,551,482,559,500,843,520,866,791,66,330,743,909,232,34,116,227,132,832,549,698,676,368,194,526,830,171,16,613,983,92,410,607,316,962,7,129,454,918,602,551,504,99,702,634,899,788,549,450,732,801,463,737,326,211,331,300,565,370,383,205,819,925,145,851,34,410,280,432,319,827,229,970,102,567,700,853,683,82,608,101,392,351,862,628,133,95,877,822,252,334,240,312,627,103,298,996,344,693,152,218,228,784,760,967,54,891,57,392,318,51,736,219,573,890,88,974,137,983,542,433,606,728,745,187,84,977,239,67,71,124,246,106,219,871,868,529,817,186,121,654,922,602,989,33,643,721,35,158,543,483,717,744,113,86,604,490,615,9,4,632,281,96,467,354,302,785,37,55,158,705,921,29,861,433,776,531,17,675,278,394,157,262,389,170,227,505,794,743,679,195,48,447,996,136,438,14,858,683,235,862,385,320,418,915,161,401,77,763,294,998,757,342,199,308,980,482,247,9,859,191,735,933,732,756,31,477,520,374,288,400,772,63,243,208,777,856,680,166,626,326,344,342,68,454,652,742,867,964,620,75,163,366,720,876,840,885,103,324,201,139,378,313,794,986,728,893,485,487,290,251,647,504,490,494,180,328,280,37,699,985,952,778,901,285,575,337,752,426,702,486,969,169,232,151,558,886,40,296,887,733,182,600,715,753,972,560,895,272,592,412,863,194,678,164,95,409,246,21,333,408,209,140,859,384,612,716,714,827,848,989,299,935,748,894,994,239,542,805,490,588,839,207,888,165,417,473,13,92,792,614,659,39,438,673,379,669,977,125,853,250,941,51,311,605,940,620,868,941,65,950,665,457",
    "186,492,888,947,309,181,16,25,933,884,730,872,888,93,363,154,584,792,598,193,711,491,160,71,740,554,757,448,717,985,739,213,378,373,760,349,909,879,337,332,480,150,979,121,168,709,155,156,159,425,365,936,438,227,257,381,657,699,897,858,923,513,90,798,84,37,186,999,540,48,51,120,668,440,933,794,245,865,690,153,297,122,216,123,489,485,733,37,673,977,469,560,538,507,632,839,261,236,408,54,112,128,38,743,633,538,609,427,511,74,688,69,723,399,107,971,744,695,535,570,697,261,10,741,582,899,15,234,316,972,122,919,255,798,278,658,504,479,358,36,119,659,435,332,764,230,116,564,146,342,751,516,0,110,771,723,429,472,584,883,934,724,997,793,527,939,577,482,995,318,888,110,188,75,493,537,679,365,902,42,957,987,153,835,218,772,291,565,568,294,331,696,461,31,231,491,220,518,602,656,740,594,222,355,484,637,66,535,867,891,615,701,267,478,218,309,218,942,269,592,294,137,268,277,903,123,906,278,193,759,834,243,136,903,44,485,888,418,170,736,197,958,171,475,729,43,162,292,917,127,382,335,170,740,836,196,787,840,994,241,486,23,93,687,183,673,208,328,224,63,122,490,516,51,627,105,290,46,861,295,399,456,179,153,810,921,408,554,876,512,461,10,289,521,427,94,837,506,385,693,583,850,60,175,512,865,59,791,229,724,151,163,991,685,826,1000,992,541,244,361,952,165,875,207,386,639,972,759,246,355,714,92,191,917,104,244,921,834,91,496,764,429,850,711,672,551,252,819,786,46,410,605,791,590,696,326,302,700,130,699,55,174,477,401,53,731,551,285,869,31,201,359,666,419,126,553,955,189,535,476,435,558,931,103,300,586,837,138,109,553,594,494,312,764,226,31,27,453,444,225,135,413,796,185,20,750,656,235,186,380,777,868,960,614,342,534,896,766,454,172,565,648,757,796,986,361,871,692,582,467,663,521,232,429,477,679,273,454,903,219,812,559,848,117,865,963,899,238,644,769,777,113,739,45,95,692,461,713,867,930,263,47,62,524,784,86,495,998,571,144,44,611,907,815,993,89,370,205,70,731,310,659,556,128,676,215,381,62,983,418,937,82,521,124,654,677,713,149,184,6,540,176,728,993,591,897,318,7,918,524,657,768,772,176,626,123,79,295,502,811,739,552,281,338,664,728,345,579,671,975,95,245,559,920,90,187,358,330,272,883,927,532,506,797,2,758,786,210,46,343,137,967,989,484,190,825,627,428,524,476,498,455,71,946,546,428,966,292,649,973,953,220,635,669,257,764,404,243,577,960,807,511,883,333,623,51,89,673,314,446,23,367,361,784,102,893,358,888,905,886,912,881,87,133,288,479,972,259,700,756,576,262,963,945,886,901,505,786,666,915,699,430,38,965,499,284,86,45,980,330,652,366,474,978,402,975,196,365,768,825,584,147,31,355,196,369,119,122,142,419,884,283,357,744,477,542,686,492,529,188,457,665,134,592,413,492,675,5,39,76,173,842,791,532,451,840,329,860,394,9,537,128,836,327,727,677,912,714,656,273,80,513,53,690,614,123,267,756,802,998,47,890,907,196,750,739,491,610,469,100,520,405,719,384,9,208,360,139,810,133,314,766,132,786,97,58,580,183,973,455,428,4,884,325,944,331,729,573,965,394,629,129,455,861,320,626,261,854,943,508,462,717,686,37,38,150,860,768,382,437,170,813,263,399,693,585,584,221,116,857,670,711,388,281,341,827,674,46,223,156,170,685,8,517,880,237,48,504,57,485,530,620,758,239,499,371,811,910,150,731,15,104,52,731,364,205,681,608,474,469,354,202,770,667,348,728,901,492,69,147,194,428,92,135,853,111,325,505,69,557,664,935,356,145,511,269,831,444,882,842,888,955,222,872,526,148,398,407,835,150,146,653,967,382,396,600,369,661,530,166,440,210,21,11,558,909,114,694,235,125,271,257,774,416,612,182,595,251,298,689,868,529,382,774,183,808,775,440,135,952,42,994,456,604,823,49,856,546,388,107,199,26,585,632,772,349,232,807,508,289,477,611,927,477,332,199,706,187,964,229,11,508,159,609,772,692,123,35,877,714,900,751,63,985,493,406,854,915,776,669,42,541,595,365,291,448,28,447,659,259,445,570,127,182,272,650,646,783,878,380,871,477,133,165,643,409,734,898,604,770,331,217,168,591,422,958,232,779,64,650,214,996,68,296,892,556,317,326,701,786,955,893,601,77,694,353,866,205,578,211",
    "63,646,803,552,74,394,576,417,882,228,183,22,47,468,9,477,339,586,403,73,219,289,32,675,130,396,221,62,746,236,626,694,109,96,993,94,879,520,696,319,451,609,611,535,642,70,343,899,771,136,933,144,683,197,474,957,924,972,451,166,772,388,54,232,376,67,560,466,917,510,223,418,705,406,564,113,696,644,474,177,126,674,414,68,724,125,655,385,841,352,718,315,31,986,724,487,49,610,640,187,894,978,97,252,600,550,887,909,467,349,180,965,929,613,455,920,26,119,330,38,740,605,406,204,713,608,577,747,4,451,939,840,101,151,865,994,626,258,748,397,813,158,371,761,487,93,316,256,141,369,268,490,110,0,473,618,329,340,703,856,217,716,692,697,104,182,422,927,706,320,542,412,786,536,657,428,875,810,759,253,145,88,244,960,928,247,867,134,819,842,113,196,482,10,682,84,877,48,203,245,77,973,840,425,357,844,100,605,838,175,722,680,206,873,979,972,166,152,593,271,538,108,451,991,267,232,590,608,152,171,528,39,442,657,251,404,864,694,756,635,170,337,610,684,758,382,367,19,72,824,701,55,922,293,820,494,308,351,234,755,494,640,240,513,417,856,597,423,408,955,48,475,444,223,727,382,464,667,830,363,805,439,949,208,973,574,326,670,116,150,373,577,772,457,426,79,514,554,482,494,727,274,412,648,481,111,630,869,681,17,992,233,763,752,8,424,839,477,866,943,569,954,126,209,364,48,859,474,205,637,911,907,939,962,502,36,51,305,347,623,116,144,996,705,582,379,132,843,98,23,540,468,420,797,107,847,105,702,92,990,544,767,979,775,674,716,852,493,382,823,298,539,16,57,284,317,11,67,824,680,562,483,594,932,388,297,892,127,915,584,589,993,527,128,712,570,630,353,448,96,131,388,872,138,214,725,454,169,693,607,355,772,560,508,90,903,60,546,69,470,74,912,579,157,587,480,36,34,839,796,695,355,12,446,387,804,756,747,633,971,967,687,866,947,376,756,522,411,92,163,903,7,657,877,544,209,35,86,327,456,209,353,271,789,255,712,561,323,248,987,593,765,536,489,942,619,954,644,756,891,64,395,917,757,350,90,249,219,51,265,353,462,940,416,378,381,931,245,212,590,547,798,542,258,739,259,115,910,304,280,667,652,22,730,785,134,922,709,588,65,408,30,581,864,721,895,880,934,77,811,640,846,94,859,92,190,729,188,676,94,848,506,283,380,542,752,510,937,6,14,46,283,366,540,861,234,605,696,133,199,198,134,326,198,968,363,922,84,888,202,963,608,413,553,41,234,838,473,455,414,492,619,793,290,521,793,35,3,359,257,679,224,281,639,23,541,864,215,19,150,366,222,918,543,958,492,899,672,389,742,783,448,203,198,800,888,773,808,3,680,400,196,888,428,626,26,613,674,513,102,604,811,763,833,941,598,942,464,737,482,36,270,98,680,800,604,492,989,649,642,244,907,995,24,259,412,224,825,853,971,189,126,167,875,486,11,537,227,227,493,831,267,428,238,299,661,616,823,405,752,822,636,727,44,609,727,246,177,376,740,614,131,707,281,138,919,571,42,328,411,792,311,787,803,765,598,388,208,468,352,122,600,602,190,855,417,19,371,275,819,644,619,361,194,915,16,543,818,453,400,758,475,263,860,774,410,782,561,134,918,860,365,73,550,22,337,569,454,273,583,90,643,680,659,987,167,724,357,122,918,253,631,216,626,749,894,674,954,508,837,159,892,448,790,559,956,460,161,908,183,412,158,371,774,570,640,533,811,926,487,283,225,855,269,958,559,136,42,611,473,270,760,459,943,408,795,801,992,840,356,517,724,250,704,373,953,931,871,275,290,522,23,437,236,825,755,305,712,243,552,11,150,601,564,238,349,848,959,56,735,55,867,196,229,466,162,844,925,23,713,690,420,38,243,995,5,41,733,441,768,731,664,488,982,491,788,899,763,518,3,826,346,59,126,522,729,89,611,505,16,994,918,23,317,720,723,256,355,277,33,712,174,313,740,122,929,497,86,279,109,384,179,632,421,194,629,892,544,584,701,782,763,604,512,358,939,583,460,914,227,42,869,361,229,895,82,227,988,68,37,73,196,562,927,622,854,640,902,496,662,800,728,843,646,353,866,874,661,92,865,752,454,334,998,746,104,327,882,40,281,908,897,154,446,829,633,774,592,243,735,919,577,805,606,961,309,398,949,286,490,886,292,224,119,475,450,681,397,871,304,838,115,370,914,629,387,47,564,917,453",
    "247,831,934,920,164,165,212,447,213,991,660,102,459,788,638,303,356,494,838,80,235,799,764,468,624,308,494,673,603,44,23,683,243,307,937,682,341,994,65,501,388,880,650,249,427,434,514,126,714,977,889,993,949,291,817,726,144,478,613,199,253,14,514,325,727,190,598,681,551,978,567,367,665,19,713,835,460,862,698,320,29,457,341,208,736,390,310,126,974,372,124,605,557,420,443,916,494,403,151,523,678,923,296,393,17,129,898,87,308,623,830,877,542,851,570,581,91,909,561,327,69,106,868,564,297,525,98,74,371,695,84,240,56,541,556,987,141,403,1,781,133,264,389,540,946,436,807,603,899,497,898,109,771,473,0,302,599,779,287,837,470,117,732,655,447,728,951,630,957,815,939,334,339,716,558,882,524,697,49,163,257,584,100,28,700,975,535,512,649,622,314,89,291,859,185,118,19,701,299,804,25,475,586,744,241,11,6,342,32,728,56,649,60,833,429,350,846,607,392,198,602,166,541,588,210,710,597,920,5,300,809,265,674,347,162,443,575,791,65,746,666,540,829,39,716,136,605,246,386,924,763,822,295,592,751,51,169,168,405,722,357,411,3,226,355,424,233,61,980,671,753,321,986,904,203,124,524,904,518,244,86,781,894,29,502,128,932,717,683,26,24,100,191,140,264,949,133,804,861,925,967,745,435,988,286,67,670,614,86,986,545,730,268,881,195,385,880,120,879,857,182,801,943,795,720,205,155,774,925,895,505,789,465,382,842,243,609,255,163,779,184,860,612,944,110,158,638,102,512,878,617,745,967,332,110,704,801,722,870,803,78,204,930,172,742,239,746,829,655,146,946,836,139,441,928,156,252,36,660,9,992,964,890,565,826,946,967,328,585,437,961,714,354,360,311,811,283,272,298,657,267,815,651,688,951,555,881,120,6,796,505,724,313,123,913,362,179,488,706,300,448,468,80,575,190,507,558,36,281,787,51,455,947,837,980,401,337,206,947,714,511,496,228,500,664,262,536,412,970,185,944,825,55,883,60,753,74,311,86,398,430,653,897,97,791,121,349,500,798,524,370,672,502,870,380,942,891,970,307,808,611,879,554,880,196,843,521,982,347,89,613,499,74,501,720,614,258,993,957,901,315,17,824,331,141,771,507,792,524,577,524,898,461,544,738,804,143,376,727,300,311,457,64,618,899,802,853,501,769,829,793,181,361,157,211,508,891,835,479,103,326,644,216,663,722,11,307,533,21,857,969,619,254,793,456,945,530,267,547,866,926,673,610,15,96,882,943,848,240,743,68,77,400,121,50,194,987,512,13,524,741,703,285,245,260,222,844,100,977,996,528,669,577,739,960,290,489,27,552,362,502,487,242,930,388,406,399,119,889,999,135,893,907,773,182,461,984,568,530,445,165,589,762,776,70,744,316,367,490,340,971,854,623,120,580,100,506,760,648,311,858,372,938,116,461,154,110,121,970,530,345,864,395,674,797,495,172,239,580,674,60,334,643,526,617,981,627,835,869,89,620,502,862,673,556,486,82,666,151,571,744,570,553,799,2,171,714,581,93,7,104,491,481,169,722,49,674,149,573,150,148,162,693,559,436,84,97,869,129,213,128,526,650,84,352,874,825,333,552,540,306,179,342,673,736,47,171,834,261,233,705,516,399,833,415,579,929,659,648,900,793,474,223,157,399,416,554,341,685,926,226,867,284,547,115,720,298,873,230,271,181,772,774,998,668,424,341,248,298,187,143,916,337,31,415,164,196,164,317,991,374,908,103,222,147,911,921,346,762,652,244,365,439,84,43,990,432,114,405,803,332,690,790,897,405,937,935,836,398,688,709,723,931,508,115,266,434,627,777,481,376,459,934,775,456,348,257,532,657,556,128,899,502,577,383,526,301,954,163,667,245,396,742,288,347,669,957,691,7,918,899,104,581,969,381,10,541,45,303,752,83,939,639,934,751,801,888,919,124,159,90,27,328,322,679,388,306,515,227,207,940,623,474,171,380,626,865,596,757,996,701,642,911,248,9,696,29,389,718,458,327,478,828,942,502,668,868,113,550,493,836,934,898,530,290,69,654,657,257,404,798,525,408,518,734,663,590,488,2,760,607,227,287,305,619,350,647,681,289,154,24,171,62,60,261,944,772,808,328,137,76,810,657,841,674,198,524,727,278,221,265,224,406,864,471,451,803,531,911,54,161,352,219,924,954,392,149,894,846,107,776,715,798,850,973,480,962,90,915,536,911,65,320,322,574,950,953,284,648,287",
    "800,308,210,443,250,388,586,821,828,978,372,508,716,141,989,657,226,596,616,97,163,952,311,462,667,720,933,496,79,748,656,350,874,40,296,100,581,997,755,609,290,862,642,38,461,178,993,664,320,476,536,833,169,36,618,789,870,1000,305,835,902,383,250,45,389,54,54,165,815,754,261,730,617,3,801,67,869,59,71,541,554,423,312,385,317,706,799,54,981,293,987,9,739,508,662,237,910,634,289,197,224,192,968,433,144,998,661,772,910,746,364,945,323,387,614,319,582,26,355,560,146,913,329,677,771,462,421,649,616,557,232,945,869,445,728,6,85,868,654,763,869,408,179,735,675,301,870,255,379,748,485,547,723,618,302,0,430,246,653,638,447,638,33,636,750,172,207,292,758,825,767,590,292,457,650,989,357,91,306,686,486,501,326,737,896,792,884,356,238,669,768,383,705,304,731,980,258,567,556,45,716,98,922,456,843,200,856,538,43,603,866,854,314,713,345,704,467,174,496,403,927,234,229,383,487,919,637,160,384,857,182,985,563,307,749,863,355,768,167,718,773,984,99,640,777,614,738,993,343,649,746,986,302,168,235,1000,502,560,98,794,56,144,473,171,114,341,743,135,774,460,208,96,260,865,286,220,207,462,195,493,16,760,609,818,415,863,73,799,964,552,680,433,652,676,615,151,642,283,241,781,225,991,804,3,349,581,35,455,518,606,626,288,740,156,624,770,241,212,271,526,588,97,286,1000,593,776,484,605,298,890,312,553,862,70,607,147,835,872,363,496,583,215,970,554,735,669,56,383,375,554,723,214,834,959,6,765,947,364,768,751,160,674,370,591,294,115,670,196,313,985,573,575,251,248,59,327,50,282,966,533,5,417,881,448,381,755,246,598,523,980,894,607,692,325,830,959,321,629,102,48,400,232,424,977,343,877,609,358,475,895,17,398,425,863,578,29,485,457,303,963,863,312,308,40,832,768,849,635,550,480,880,526,582,124,657,231,503,710,918,766,340,584,69,876,625,748,896,387,206,150,756,461,457,677,9,940,958,987,638,777,746,876,914,555,289,264,875,585,200,331,849,969,257,776,330,326,640,387,384,316,123,432,58,612,961,454,7,223,360,862,686,838,725,364,703,100,249,397,695,546,827,945,239,878,618,764,80,751,546,667,820,732,964,828,529,214,622,846,493,640,862,491,579,327,340,521,442,982,420,822,656,740,50,391,24,347,560,438,807,618,391,742,174,307,215,770,700,491,481,763,482,136,115,210,416,838,973,584,389,386,820,845,673,918,61,477,582,729,553,493,994,703,116,915,319,916,541,202,608,310,669,93,856,466,188,213,995,299,824,89,465,311,367,870,606,161,524,843,8,880,943,246,739,665,431,934,188,540,744,669,952,138,606,791,326,921,891,746,148,769,574,491,378,600,240,911,408,668,907,661,961,380,638,567,45,435,435,446,883,432,753,181,810,512,436,43,614,950,723,320,106,43,56,472,860,886,770,517,832,307,541,860,912,373,954,599,249,93,75,895,739,383,929,490,637,242,124,78,554,989,607,848,638,840,579,360,470,554,381,201,8,220,449,739,798,159,716,551,597,543,775,690,955,410,231,915,394,227,230,844,360,899,100,583,820,586,568,616,849,766,367,948,259,985,740,841,170,825,251,777,820,625,907,602,779,678,969,969,952,700,542,732,404,30,620,181,940,299,53,958,806,382,467,969,564,683,687,417,145,982,219,865,761,569,516,304,772,149,791,468,93,232,849,408,122,397,944,71,543,62,292,491,295,606,265,260,85,540,248,521,231,691,14,347,193,675,849,106,815,527,608,24,644,392,993,194,510,926,804,417,72,229,893,635,783,713,344,901,958,457,400,646,878,986,95,74,764,565,794,872,148,543,705,28,899,142,203,688,755,906,138,470,219,877,964,256,634,715,544,476,801,497,197,342,738,31,11,974,927,373,113,453,673,373,377,135,164,808,562,560,28,846,666,692,71,848,351,747,57,580,837,414,718,963,866,537,767,961,730,865,646,448,881,323,303,97,15,710,454,879,748,25,819,379,22,210,522,363,954,837,654,38,600,975,993,477,881,966,533,713,518,775,696,730,564,714,974,493,10,299,563,796,511,807,866,597,533,467,909,567,181,175,880,807,274,815,283,885,98,232,742,453,721,572,406,194,516,818,589,988,668,615,951,165,465,308,907,497,484,12,229,700,682,716,293,388,492,660,307,955,604,227,687,889,745,239,381,642,167,672,520,213,735,669,96,269,512,101,123,716",
    "503,254,870,592,183,319,909,398,769,487,459,436,683,686,563,544,73,997,282,446,554,668,244,233,115,307,32,205,274,730,643,630,804,324,929,707,955,507,931,557,354,813,609,180,176,251,596,732,299,578,435,150,414,747,573,470,545,307,914,978,306,918,932,391,169,751,648,172,207,890,791,937,147,133,142,416,31,833,825,109,473,917,413,9,488,85,786,643,77,662,138,144,658,756,311,606,525,875,701,317,487,812,227,372,202,659,405,981,461,818,755,120,600,276,774,223,887,844,651,237,130,457,223,514,254,571,229,839,765,250,807,894,938,495,408,974,861,102,571,780,692,951,540,643,765,576,591,767,805,914,281,469,429,329,599,430,0,986,519,970,36,608,406,929,997,46,186,643,123,282,914,507,972,81,3,740,39,207,769,157,751,641,243,264,63,965,949,217,655,290,488,429,915,332,416,598,355,304,790,630,679,227,728,20,698,899,214,220,278,108,581,182,424,890,194,225,65,567,432,430,72,367,699,798,538,351,184,445,43,377,542,266,489,281,113,583,658,989,301,653,144,949,716,456,496,811,336,937,885,588,382,234,139,343,994,642,147,313,11,858,850,41,494,572,556,471,392,362,594,902,355,812,755,387,412,959,182,550,396,463,573,16,867,15,514,450,407,136,52,820,399,642,302,77,924,902,651,281,261,862,597,343,194,289,351,172,110,906,992,202,689,19,478,702,616,935,220,310,922,336,604,290,864,786,168,13,85,785,767,301,554,453,17,610,904,147,699,151,342,794,657,112,212,753,226,684,732,985,808,260,221,523,827,528,478,736,322,429,313,853,102,191,52,804,557,329,253,377,54,806,422,834,836,998,372,516,742,882,197,481,94,705,983,161,875,922,492,783,458,711,486,426,531,955,255,858,879,940,688,112,702,203,682,587,22,689,62,699,463,978,482,977,88,333,446,363,570,426,86,314,228,936,277,985,405,142,902,902,199,945,729,263,962,549,729,310,232,393,220,749,465,167,295,696,841,792,824,167,285,28,700,27,745,48,849,713,832,850,70,675,509,694,394,548,969,487,466,842,151,439,2,242,904,180,325,512,843,470,877,923,122,620,411,966,926,404,946,303,309,438,424,128,56,855,861,529,315,430,406,987,597,752,380,936,944,624,941,854,235,303,97,518,133,646,706,695,39,799,985,234,416,908,512,447,866,294,806,242,361,305,412,155,227,742,458,533,457,144,881,515,104,573,844,638,379,338,558,856,54,385,398,177,192,676,739,770,460,63,910,177,38,915,540,667,526,720,190,540,289,599,893,441,908,758,953,511,597,251,226,448,102,780,811,343,515,245,981,732,812,46,388,774,986,990,446,285,684,640,752,768,977,56,845,187,452,32,546,563,259,412,41,880,619,224,666,650,276,465,33,920,680,270,28,488,634,892,351,443,884,91,137,937,623,898,405,417,711,590,500,204,641,983,333,56,565,263,557,791,746,243,909,595,658,9,486,495,859,980,710,384,720,965,904,781,916,237,67,208,465,298,582,544,603,536,625,338,634,562,113,216,175,262,844,76,661,379,187,893,711,736,337,152,436,247,123,118,169,595,238,542,78,704,17,620,497,171,956,950,504,833,687,331,237,800,751,520,66,38,278,912,957,965,910,167,701,45,538,780,885,268,639,111,229,470,539,395,919,826,624,591,734,555,760,320,300,264,338,543,486,64,241,82,746,353,165,263,306,360,60,920,720,529,160,557,810,994,759,328,823,625,80,528,698,7,591,530,459,250,372,999,505,744,737,981,251,985,661,558,245,294,890,420,306,614,857,895,552,128,851,432,274,835,622,921,405,756,774,975,682,553,610,52,153,455,36,364,258,109,524,76,446,316,716,715,388,129,34,92,221,109,865,969,720,341,390,247,919,893,512,731,432,975,892,461,828,537,916,710,861,556,925,381,879,75,170,79,885,191,280,189,683,502,286,893,865,732,196,857,498,823,917,826,625,28,842,228,280,117,925,202,371,829,45,145,130,12,902,525,308,806,583,642,768,220,926,588,69,360,931,540,37,585,943,672,144,827,533,807,148,349,318,427,40,481,824,753,784,742,898,241,585,940,588,691,613,332,704,559,592,713,521,240,442,251,269,423,381,112,103,357,589,946,803,639,186,539,983,521,788,278,300,821,828,623,518,605,446,86,355,213,444,686,577,314,112,621,323,785,342,817,51,581,411,877,777,619,904,462,532,963,930,793,46,679,790,832,282,810,349,896,130,787,769,252,129,389,761,787,589,216",
    "767,860,900,793,886,497,532,782,181,168,530,251,93,944,182,821,709,68,315,415,574,60,595,29,69,199,537,352,43,783,121,385,190,99,126,856,732,324,548,665,259,554,75,235,515,435,976,947,540,360,444,150,981,523,563,880,219,956,744,404,657,908,951,833,939,944,802,212,765,282,264,150,197,943,43,30,187,559,320,486,787,354,318,136,107,246,671,546,874,30,322,13,147,869,277,569,252,543,538,646,911,787,650,292,388,648,689,674,562,564,255,694,252,193,744,800,654,993,511,489,54,675,919,945,358,892,464,414,236,962,368,972,696,661,493,858,569,245,313,818,178,665,711,978,581,353,444,381,868,766,371,445,472,340,779,246,986,0,660,672,169,523,8,989,516,631,811,243,572,668,371,541,301,163,9,868,591,224,437,338,869,778,726,440,615,39,965,954,964,494,60,58,680,644,55,291,123,892,326,936,251,706,989,941,828,347,384,734,226,192,440,26,334,375,104,606,502,80,833,262,71,985,703,45,937,298,935,221,706,484,886,761,931,32,322,559,305,107,172,73,827,850,135,890,432,19,978,843,953,479,879,928,619,942,424,874,722,580,691,612,894,280,296,500,247,757,953,600,796,965,387,655,474,268,86,975,948,871,220,36,787,922,910,687,629,428,901,541,24,398,924,189,216,737,621,360,538,921,482,602,930,59,791,652,29,35,697,771,36,171,264,176,986,555,402,305,869,685,658,770,528,788,517,860,284,679,960,430,2,81,297,4,697,989,613,612,638,531,716,798,561,355,49,342,447,678,619,148,26,905,579,739,370,543,64,224,479,852,88,730,121,824,424,246,670,701,350,484,748,436,437,241,440,774,971,700,59,600,938,913,33,656,578,857,405,536,344,110,883,517,482,405,367,270,523,845,653,54,432,124,335,533,379,288,501,178,352,112,82,385,152,503,862,68,971,584,647,752,421,586,945,871,826,634,758,700,655,94,382,604,405,696,160,860,629,419,539,661,966,245,1,74,520,685,925,162,783,431,415,828,688,908,411,567,933,599,617,109,385,249,410,680,656,63,622,807,448,889,735,959,956,886,117,897,584,925,858,585,950,211,68,410,68,679,32,143,781,703,368,725,179,282,620,516,8,268,35,561,828,666,802,398,145,284,470,29,132,222,272,161,219,599,296,772,951,401,102,854,631,33,500,738,460,752,921,770,82,517,317,930,543,728,17,103,717,35,31,870,650,438,746,733,527,790,134,583,387,302,721,135,351,966,961,212,346,767,204,180,937,458,3,676,620,721,451,34,155,809,635,326,919,21,993,152,482,259,390,897,779,168,107,711,28,248,495,332,546,838,311,16,381,106,171,18,926,182,969,54,352,34,384,705,478,600,382,430,441,476,362,826,372,454,23,341,336,126,737,414,577,9,254,424,421,345,920,794,716,559,651,265,969,981,418,756,716,340,295,534,109,688,43,487,512,691,552,778,263,670,173,732,821,632,886,498,188,578,60,3,919,189,347,524,503,388,890,33,991,110,378,533,46,457,294,68,467,655,120,42,351,810,716,599,370,342,855,199,733,569,752,702,92,294,263,126,891,799,963,250,637,384,563,961,423,67,598,356,711,653,721,643,739,185,465,14,916,32,518,885,589,198,400,673,617,923,396,472,692,626,87,216,317,674,18,680,458,617,988,964,302,461,440,763,759,107,482,560,287,338,912,250,582,876,883,289,446,311,970,745,225,993,513,985,958,384,808,916,518,156,328,907,827,621,123,360,977,878,952,255,315,24,306,618,343,535,7,52,397,312,273,790,401,580,98,287,258,985,860,293,494,333,470,71,974,391,262,475,810,648,57,442,838,922,900,215,392,962,877,564,924,925,723,60,302,772,139,514,188,313,139,920,198,330,727,421,977,511,836,663,827,755,899,372,209,419,612,644,873,430,610,374,326,108,183,220,977,545,581,31,920,459,20,498,939,673,298,194,856,343,503,93,787,777,221,346,291,887,590,810,604,689,769,517,344,290,852,564,274,168,373,989,281,964,538,368,657,219,220,168,541,712,432,316,640,943,155,245,975,893,411,653,180,105,938,367,826,284,299,170,314,634,637,899,350,538,39,969,375,519,424,726,66,257,322,775,745,841,873,288,951,614,566,242,248,782,373,484,287,396,553,458,158,658,714,211,740,899,858,931,658,789,342,381,438,372,407,343,450,281,209,160,986,371,454,602,47,128,345,478,906,14,869,636,158,747,140,811,686,912,299,197,131,372,804,481,213,943,384,308,861,642",
    "869,338,656,781,464,690,443,166,777,649,857,254,360,305,88,302,953,843,903,668,166,676,84,798,303,654,61,949,3,915,318,311,232,655,256,993,584,867,808,68,936,642,834,899,136,978,120,223,518,284,670,846,438,340,11,752,462,988,681,247,561,462,500,27,26,813,956,507,238,992,231,625,711,783,158,980,636,854,982,227,930,24,668,575,219,604,6,280,306,735,780,44,995,275,413,248,614,215,279,372,104,564,204,480,28,9,391,236,915,184,452,490,14,814,131,268,693,184,951,426,469,97,991,65,844,909,942,913,376,228,162,668,444,298,205,663,301,475,592,769,439,958,358,580,765,539,992,672,404,767,874,852,584,703,287,653,519,660,0,760,391,188,157,221,86,893,928,185,128,907,435,536,526,919,479,103,613,45,340,58,394,453,654,825,718,297,598,25,2,156,522,986,628,16,675,14,719,188,764,429,195,421,930,67,227,129,52,475,599,956,798,628,500,787,856,785,641,35,467,647,992,984,792,553,200,238,346,391,554,471,895,843,41,591,474,452,196,223,608,24,235,407,851,884,111,468,245,14,507,514,812,921,940,553,12,899,568,461,564,645,866,158,886,759,489,105,460,757,640,948,359,269,14,954,377,982,818,324,376,166,482,526,704,144,173,96,789,47,908,195,104,641,339,276,745,504,210,857,877,145,400,313,866,142,904,209,65,949,340,500,59,678,762,949,876,406,357,622,298,890,552,278,78,907,332,975,251,218,669,771,810,151,779,378,490,784,134,390,449,196,153,221,882,197,266,228,220,284,289,408,757,377,102,885,211,642,696,280,444,439,956,398,653,828,623,592,593,579,121,496,455,595,669,134,716,585,158,264,941,112,201,134,302,401,99,303,189,428,583,135,681,142,29,803,43,501,271,116,841,854,452,536,285,582,548,564,862,461,950,71,657,552,117,895,488,463,387,707,218,509,228,6,974,330,828,421,104,243,466,440,961,57,768,444,67,51,372,16,509,359,944,282,794,941,331,280,189,572,508,873,78,595,888,472,873,703,303,354,156,693,678,534,762,677,268,99,257,122,738,832,350,161,665,600,597,324,877,36,518,927,76,465,212,393,383,514,566,445,83,272,998,515,402,389,18,931,635,175,483,108,349,966,353,485,839,174,765,298,833,35,547,360,162,781,895,3,206,505,587,120,136,986,587,18,189,157,95,186,760,896,860,380,853,594,364,901,766,245,810,736,934,766,58,596,376,749,83,341,216,607,495,492,174,930,695,33,267,713,348,295,439,510,66,238,935,737,619,94,707,950,839,362,93,839,555,487,930,433,343,958,412,186,521,434,446,714,729,38,265,820,850,405,387,847,575,541,237,976,163,509,917,539,448,324,964,225,961,769,142,116,646,476,869,357,454,171,332,693,26,715,93,804,555,784,741,957,313,294,275,689,401,692,858,802,862,250,585,902,870,106,493,324,205,591,643,841,371,848,146,815,535,725,225,153,97,339,372,314,538,132,623,396,258,936,108,683,907,76,494,903,993,34,74,843,577,734,723,333,96,810,275,646,783,654,889,177,85,345,667,946,982,744,766,211,329,994,279,123,665,188,835,722,356,137,919,43,938,768,787,343,755,147,588,308,755,676,270,721,786,580,653,720,255,608,478,207,476,830,207,856,317,674,44,619,767,986,326,160,467,433,190,286,11,358,917,76,928,175,552,271,991,171,475,990,116,773,960,744,731,756,79,63,543,461,702,349,435,923,741,955,656,86,733,550,801,837,800,94,613,325,215,533,843,383,125,204,434,712,839,606,404,925,521,585,267,123,809,355,551,146,484,242,394,473,775,298,968,314,347,566,673,915,335,293,907,896,592,104,150,651,648,481,800,87,673,890,380,207,346,120,246,377,443,396,499,451,737,42,290,558,350,322,585,416,810,838,893,880,908,717,677,645,49,345,750,626,119,295,714,461,765,22,321,794,376,534,635,126,770,991,397,543,108,264,28,737,54,625,486,750,660,656,436,746,648,856,553,406,100,110,308,772,330,638,794,119,416,899,194,928,735,134,416,150,563,930,913,906,869,517,162,270,191,22,826,891,394,73,362,102,771,868,295,135,28,845,231,972,862,330,958,288,608,773,447,874,888,196,419,688,53,323,635,959,59,427,132,226,643,152,701,602,801,395,290,984,290,245,342,34,47,21,544,433,473,537,307,730,736,534,496,819,50,125,768,706,298,324,296,318,45,562,250,539,762,437,210,907,417,201,969,642,726,364,89,656,272,782,968,327",
    "13,739,780,277,540,797,1000,274,96,868,541,252,919,220,858,270,913,946,701,284,95,676,554,779,349,721,53,209,201,654,457,220,467,68,275,481,8,274,266,53,47,556,721,757,809,154,306,986,93,576,872,518,385,172,771,8,201,982,30,17,137,80,882,314,9,23,273,740,88,726,762,915,885,431,306,293,693,468,645,811,700,175,523,398,94,99,525,611,390,479,663,328,555,499,398,995,629,980,240,562,18,543,664,390,792,547,381,204,22,966,26,750,898,554,820,593,449,177,827,98,978,236,90,949,752,153,612,814,559,222,602,579,266,921,690,447,319,615,410,728,756,42,381,409,455,912,889,971,349,883,492,514,883,856,837,638,970,672,760,0,102,533,624,509,277,227,849,445,304,402,264,488,940,858,679,724,846,14,777,431,376,661,671,42,54,767,421,659,462,334,589,655,876,874,823,960,118,617,644,236,876,400,200,395,370,898,467,100,665,656,943,475,675,320,295,130,228,84,735,394,743,486,841,975,999,119,174,306,897,333,311,261,923,296,337,429,460,294,623,989,55,828,5,63,39,594,718,475,853,425,172,239,699,996,723,233,285,562,567,837,113,495,968,313,720,284,704,490,560,53,828,569,352,693,318,804,310,85,758,11,710,532,67,716,766,209,711,500,996,584,322,941,616,475,641,355,663,237,418,382,925,634,483,172,327,506,884,38,976,379,57,237,387,310,721,510,347,871,252,161,265,595,473,822,930,662,105,616,469,275,425,689,532,711,954,637,198,40,974,777,897,712,369,27,874,113,450,682,97,263,629,927,332,260,382,190,120,272,943,954,813,866,211,816,970,102,482,24,428,160,271,540,999,554,484,339,154,262,781,649,796,641,874,237,171,786,920,263,256,222,759,684,530,434,478,3,141,910,323,455,249,35,94,113,839,149,508,763,462,379,518,656,591,905,623,283,964,836,850,395,947,715,31,685,386,132,842,147,814,450,108,665,478,755,64,906,111,193,680,146,569,812,496,710,320,196,478,397,643,909,472,648,622,915,908,259,608,582,375,776,592,431,825,585,487,752,710,854,684,878,201,122,847,960,448,531,460,288,859,691,438,51,166,65,589,401,976,73,371,418,272,426,655,503,915,880,3,446,307,630,475,5,11,599,436,293,46,629,47,913,796,978,161,751,63,15,874,194,402,502,711,663,690,68,200,646,691,374,679,203,453,233,140,92,446,730,756,830,110,70,148,229,344,782,333,300,500,179,172,948,492,436,729,465,702,825,685,103,634,12,393,26,310,389,613,179,306,139,453,558,930,657,732,830,679,782,512,495,389,516,442,932,236,473,435,481,819,637,112,54,380,313,240,536,390,512,411,464,890,377,923,823,616,342,266,852,422,446,857,318,527,11,420,649,628,289,545,415,731,744,48,378,3,532,613,86,229,140,798,241,678,189,681,571,854,335,438,415,31,287,567,922,442,162,419,293,886,680,150,320,951,710,117,279,248,436,85,252,847,328,752,908,433,162,131,338,215,321,885,852,625,284,565,115,21,155,60,960,697,951,600,878,60,383,475,511,206,98,942,183,291,153,592,697,262,593,666,646,568,243,564,575,457,195,175,398,203,31,809,346,859,419,818,811,824,708,887,457,415,5,542,579,199,768,747,499,816,773,290,790,612,63,765,343,796,994,469,85,476,524,710,569,612,526,910,826,17,163,950,620,978,686,769,767,65,883,656,133,796,358,729,678,880,188,585,960,40,748,500,799,145,972,726,766,410,819,880,525,767,428,390,230,577,597,782,939,39,24,96,978,571,64,606,87,667,440,575,874,778,20,640,236,657,523,583,805,462,526,236,7,720,956,924,880,758,840,61,224,194,88,995,933,627,930,159,490,285,409,338,726,543,751,30,623,954,172,433,84,382,809,715,781,921,661,241,534,476,562,795,537,609,103,502,268,259,206,649,794,252,103,292,738,701,672,949,444,233,611,593,367,926,227,27,636,715,328,930,533,122,837,98,806,238,811,595,681,625,106,543,777,654,150,886,872,980,615,684,101,570,314,649,198,815,722,513,148,780,366,512,451,657,465,693,573,604,960,42,225,868,84,550,682,190,503,89,461,920,546,665,895,332,735,258,466,124,595,56,623,650,850,65,469,712,367,8,755,128,666,79,619,974,9,901,796,507,512,180,448,910,451,931,377,487,210,945,521,459,569,125,863,855,202,78,865,945,996,851,461,292,543,80,481,335,292,560,301,173,506,373,692,294,679,118,694,981,182,511,166,742,47",
    "252,614,679,326,586,985,180,576,443,100,450,363,697,724,205,303,369,55,734,208,895,566,436,555,284,418,293,555,210,30,92,486,561,883,847,498,420,897,345,600,347,693,980,636,725,381,190,931,458,977,934,566,851,476,78,962,26,251,451,708,842,877,13,471,113,128,165,123,852,699,817,850,337,36,266,875,602,731,416,467,112,21,806,527,34,938,891,681,840,234,664,492,985,562,195,90,485,484,321,675,159,994,387,984,313,537,392,998,354,521,452,659,514,70,246,646,423,712,209,43,282,594,733,687,708,933,229,331,115,161,512,28,186,606,960,67,670,791,527,391,728,641,803,489,987,254,661,116,749,184,739,696,934,217,470,447,36,169,391,102,0,34,635,310,15,380,402,85,475,646,818,171,542,993,272,9,163,64,128,160,295,658,75,300,918,126,960,339,916,300,504,415,33,512,244,521,72,247,121,423,211,677,578,314,177,511,486,45,262,548,878,244,978,828,301,333,336,856,267,447,599,682,686,64,825,187,339,710,919,554,898,777,868,699,770,547,975,204,623,864,869,74,862,333,668,908,855,198,325,181,908,361,751,613,231,136,615,705,872,365,380,32,898,905,158,673,275,912,914,3,742,609,371,667,386,799,453,561,252,262,1000,526,167,376,971,949,821,602,398,729,514,39,930,69,34,36,731,970,343,852,500,996,754,531,35,521,139,890,978,575,766,436,642,193,356,760,618,260,909,41,65,223,683,945,4,338,444,927,284,813,977,842,346,493,904,152,218,995,130,828,980,746,701,51,643,118,143,524,334,460,991,251,54,636,973,655,309,36,108,938,578,705,333,92,976,929,217,131,111,211,122,165,41,324,946,771,674,721,3,314,624,931,502,791,567,919,173,309,531,992,466,424,783,398,626,248,793,429,991,739,724,760,376,986,968,428,664,79,898,979,991,996,809,524,643,278,156,420,24,211,200,107,545,60,587,646,721,960,147,668,744,755,679,515,264,150,517,470,833,515,399,405,81,71,262,885,443,998,480,700,686,348,398,565,223,702,951,898,662,534,998,97,593,955,69,560,625,427,23,290,47,621,41,592,775,886,629,22,92,94,941,999,323,803,779,178,607,681,235,17,168,725,268,240,281,643,502,618,993,305,356,505,784,7,584,327,57,208,676,675,133,504,81,743,476,983,944,142,329,170,578,316,990,624,212,550,533,321,485,630,332,524,56,805,368,748,297,105,625,468,132,43,399,987,360,171,753,659,971,667,760,725,235,474,835,570,161,120,713,852,49,522,779,659,512,449,650,38,978,969,963,329,355,628,325,378,115,366,877,532,607,366,248,157,999,779,560,401,420,691,193,784,474,576,352,437,143,767,861,91,551,5,829,114,405,904,561,390,164,820,158,50,150,771,488,435,330,175,271,288,174,484,857,934,251,469,598,472,239,491,573,393,939,490,615,556,189,467,824,763,815,621,54,373,937,894,14,440,653,613,989,92,210,237,923,216,975,351,141,486,227,271,667,474,332,631,552,826,786,403,811,268,824,991,17,897,576,150,839,72,938,485,624,719,139,246,14,513,216,841,14,104,276,62,183,109,461,954,304,785,209,855,511,180,223,165,219,782,813,895,159,557,29,834,756,217,48,674,605,413,580,113,293,840,197,847,15,387,589,768,188,5,183,449,670,541,545,182,628,860,137,302,485,992,61,858,436,634,57,862,547,900,125,972,835,424,775,279,344,681,48,942,630,993,438,578,871,509,403,672,223,872,958,925,446,856,841,555,459,58,529,309,141,895,306,900,889,712,961,407,78,339,162,981,786,534,573,625,948,594,346,569,831,334,224,499,740,875,854,186,189,353,359,182,704,272,446,964,757,679,613,747,927,590,747,127,756,637,162,920,832,12,684,396,989,943,68,62,365,10,554,738,816,466,82,31,733,683,410,577,21,836,362,841,929,524,44,407,599,79,33,529,279,728,122,52,919,417,318,124,6,78,849,793,471,633,280,563,575,905,442,66,878,279,742,633,859,551,393,531,182,247,887,311,52,733,139,694,433,692,365,858,492,136,111,426,33,12,614,453,928,918,766,943,747,771,97,746,499,375,472,605,368,939,712,884,152,799,464,26,514,965,284,622,594,979,181,896,543,107,177,13,888,941,682,500,66,973,542,653,259,397,886,210,701,63,482,105,475,382,676,828,683,500,268,882,340,246,488,143,666,236,98,702,969,974,91,784,151,31,592,376,459,563,700,772,23,497,498,548,776,980,46,13,929,698,744,115,283,738",
    "746,918,147,971,85,796,352,392,927,979,302,874,416,198,171,546,331,174,430,499,323,918,483,662,384,409,784,154,758,800,152,31,715,489,6,351,183,852,654,486,750,362,235,955,984,586,680,385,639,693,231,714,284,688,587,831,319,114,633,443,603,144,497,276,917,458,350,109,849,344,74,906,87,831,496,203,665,199,734,192,922,969,887,425,500,800,612,160,189,113,743,546,15,344,232,905,665,599,162,865,890,745,897,286,324,931,479,584,919,608,389,211,857,266,770,249,117,651,83,584,845,116,16,108,262,50,784,518,508,115,336,233,442,537,84,280,817,701,571,641,444,782,46,299,180,239,255,994,813,458,512,87,724,716,117,638,608,523,188,533,34,0,302,3,519,341,267,34,787,923,527,155,245,340,768,390,30,670,290,743,524,607,682,903,696,5,594,810,640,292,650,522,628,333,530,849,288,662,527,426,923,185,27,530,543,794,647,727,22,408,112,52,417,956,825,477,589,398,868,270,707,285,798,761,320,959,274,406,353,772,464,574,394,385,224,292,295,937,256,561,6,439,190,654,241,646,423,380,944,515,495,847,719,480,756,611,669,65,140,198,431,347,450,958,438,349,378,971,136,909,750,296,7,753,660,719,189,57,347,350,394,763,398,585,177,21,606,104,552,904,861,641,266,197,46,464,957,827,761,715,770,694,988,999,802,23,372,357,278,510,576,224,72,421,18,443,926,879,748,775,760,768,304,104,86,314,195,589,133,213,421,949,655,948,277,823,5,279,473,630,279,681,7,25,718,192,261,147,275,446,134,785,213,532,55,660,745,454,321,622,835,128,515,29,646,126,485,973,896,246,530,964,421,429,791,546,557,566,731,50,17,823,229,25,709,768,266,700,590,560,944,569,182,372,24,802,200,19,94,59,427,713,400,167,175,595,622,450,755,816,753,343,644,581,596,299,267,634,344,624,608,706,459,127,224,152,232,787,690,775,804,267,43,118,732,670,98,101,453,588,914,410,787,374,349,176,113,918,35,480,816,482,553,280,841,409,413,794,358,458,335,786,163,962,812,874,328,850,469,594,693,674,532,549,913,957,624,511,548,391,783,620,923,874,816,848,937,67,722,150,852,123,546,676,539,791,366,223,252,409,813,487,617,222,515,718,339,506,62,756,211,729,941,603,296,560,757,605,20,951,771,558,761,106,124,901,140,608,286,366,578,17,660,906,216,159,221,265,432,298,714,237,993,906,328,308,502,855,633,259,606,799,15,74,640,786,320,755,923,965,205,39,283,756,853,719,459,236,758,407,114,515,636,320,170,365,765,202,164,359,839,216,239,593,956,135,35,536,915,562,980,267,668,617,835,282,780,419,860,892,715,159,655,63,480,246,174,189,246,731,939,740,242,194,322,986,289,481,500,739,875,599,867,5,457,413,692,244,552,358,671,751,668,550,501,124,268,654,173,429,551,958,551,721,243,920,442,622,940,27,415,782,783,212,234,513,390,783,923,279,894,451,878,201,822,64,8,383,80,709,595,698,484,672,383,334,324,763,429,278,878,657,253,905,767,174,529,177,863,265,838,226,673,503,986,580,872,996,130,510,518,470,887,177,980,41,444,201,978,722,173,292,797,999,867,587,343,907,392,969,480,732,503,778,720,796,811,937,383,141,702,986,237,269,701,925,914,928,958,904,864,470,917,897,869,232,115,607,123,794,851,873,909,884,818,726,965,191,266,974,327,21,327,53,877,921,94,56,662,901,809,764,165,830,150,38,485,16,894,99,800,215,273,983,481,809,442,446,489,112,782,901,930,537,284,427,722,289,766,382,614,421,240,674,240,808,510,750,833,339,331,580,493,958,815,200,226,471,56,55,823,485,326,812,241,987,135,196,505,921,806,304,120,139,70,328,732,74,197,346,263,416,106,965,711,49,545,819,624,837,93,819,746,888,135,167,410,181,706,790,292,146,131,224,968,375,808,736,615,495,998,144,484,531,358,547,128,721,12,140,284,269,656,491,425,22,831,856,252,768,889,532,650,998,128,895,415,624,944,65,97,937,644,889,850,468,348,442,33,302,569,528,276,493,978,509,3,20,170,830,728,573,901,891,768,971,5,927,865,149,948,700,841,978,52,758,448,679,386,218,497,735,399,639,803,782,82,93,66,162,444,628,335,804,179,138,917,256,244,286,850,178,652,247,485,145,161,424,733,502,744,740,324,952,154,727,19,946,183,975,180,976,244,534,558,464,583,953,296,748,47,61,670,534,304,124,491,134,460,983",
    "6,21,201,673,769,96,740,929,478,942,879,2,372,262,21,429,737,651,311,376,257,86,95,992,126,826,234,958,436,768,794,477,234,571,247,245,193,965,507,656,858,889,694,498,818,216,666,412,715,578,941,633,904,964,430,300,296,969,358,595,358,111,994,403,1,988,929,302,793,344,702,478,749,880,234,283,97,864,79,24,787,975,268,956,625,85,597,191,696,933,968,334,701,62,988,151,857,133,299,928,945,706,858,352,883,590,619,856,72,449,982,21,365,58,608,992,510,289,701,389,96,700,694,926,425,639,319,162,925,812,37,889,479,555,491,327,32,60,638,230,711,53,795,746,685,264,328,782,976,652,372,305,997,692,732,33,406,8,157,624,635,302,0,682,559,170,179,793,801,913,629,503,538,884,480,439,249,700,769,644,517,31,358,480,272,87,45,384,945,264,68,806,395,974,922,984,640,451,124,382,199,886,612,79,145,748,999,963,28,564,900,279,199,587,829,989,897,358,528,299,551,637,413,382,174,175,146,876,602,426,892,385,4,918,237,649,401,37,286,698,37,91,325,314,230,956,795,374,948,150,488,986,408,724,609,353,727,536,676,100,547,666,641,650,51,755,874,259,206,718,723,810,883,635,963,407,150,247,137,222,862,494,922,372,240,528,768,970,79,292,600,459,451,914,725,615,102,140,145,263,858,271,190,331,901,209,646,856,822,601,123,998,107,413,453,772,817,608,215,261,716,499,775,263,144,864,271,468,732,729,821,715,450,206,953,913,811,892,687,772,920,442,447,626,645,930,614,44,781,81,922,419,277,736,289,456,734,959,310,727,218,522,104,627,354,406,169,787,171,70,315,280,738,562,864,323,638,507,374,162,540,275,19,192,762,862,763,945,658,499,262,40,17,385,163,676,512,890,131,902,526,825,987,489,350,745,270,659,384,410,631,943,92,397,570,571,220,509,923,374,563,865,337,188,941,452,485,562,675,926,163,104,763,813,938,968,178,805,941,999,864,710,829,817,586,668,271,103,746,757,354,107,360,588,320,957,76,294,332,748,358,27,533,561,943,501,979,958,553,180,805,299,65,25,770,653,52,580,471,953,358,640,991,18,883,871,835,816,523,872,359,910,471,148,68,501,596,608,96,369,167,87,553,5,792,541,517,779,726,427,564,874,5,987,674,135,580,665,38,494,711,382,172,481,406,847,612,405,754,922,587,224,899,581,789,365,730,343,340,519,715,428,177,719,123,657,512,78,103,934,767,356,448,881,434,17,798,598,670,469,953,989,862,1000,330,650,119,404,174,707,451,156,904,700,15,223,457,136,91,627,339,163,523,67,674,967,281,695,94,815,50,390,548,4,343,886,346,558,984,66,407,754,345,750,508,90,839,693,705,255,444,207,428,15,4,105,580,459,959,963,727,201,104,665,758,985,838,617,572,14,22,133,191,663,85,180,372,203,765,649,922,885,78,701,65,465,986,243,74,657,725,534,59,748,680,51,343,418,143,456,9,632,980,93,330,13,118,763,454,293,395,14,6,187,942,640,106,950,480,671,370,878,507,231,537,514,395,795,743,139,880,722,735,211,439,371,125,930,232,388,156,763,303,536,914,114,434,597,341,791,510,382,828,593,536,364,208,273,836,979,100,909,976,540,351,120,364,506,980,943,379,208,5,956,641,403,402,7,535,452,237,245,974,341,820,267,239,184,824,26,959,911,676,108,621,916,312,947,924,345,720,328,281,223,278,353,401,311,782,273,430,956,869,295,907,787,111,95,813,919,217,715,612,995,849,845,949,707,84,683,701,90,903,664,752,286,98,167,625,817,777,535,59,93,257,330,110,164,578,664,562,352,570,707,170,859,289,582,28,621,9,353,54,436,618,556,759,743,144,611,270,806,117,280,756,599,607,871,237,441,235,38,787,359,477,451,409,711,54,382,438,855,771,565,570,748,52,994,811,799,462,754,459,803,136,507,376,259,963,972,992,324,845,459,749,583,855,756,356,43,512,850,736,707,361,878,531,551,797,514,746,144,43,852,759,796,229,422,380,575,363,212,979,717,925,818,615,130,435,825,386,736,751,991,641,191,596,341,632,655,539,826,303,656,435,183,287,273,115,630,469,680,237,583,3,711,434,660,49,967,553,475,361,377,976,452,357,28,339,792,945,549,703,205,767,935,303,338,411,528,697,426,911,310,784,975,187,988,47,306,85,650,373,904,923,70,616,161,460,742,685,138,686,614,613,420,170,397,555,789,477,697,961,202,250,904,66,48,64,570",
    "436,493,135,208,967,484,341,16,364,304,922,925,851,741,76,549,283,469,6,498,323,951,209,172,456,962,365,761,141,27,422,535,339,954,85,383,955,614,723,772,901,247,456,757,357,602,3,749,140,37,402,221,981,976,987,411,617,563,173,916,40,306,568,659,27,340,391,141,161,603,967,736,692,896,186,398,544,971,521,974,77,880,19,649,333,692,214,480,535,712,663,57,250,353,646,160,143,290,885,556,498,643,141,323,38,296,623,332,555,725,746,859,157,265,515,758,326,131,729,573,847,208,119,184,661,518,992,931,135,288,422,151,586,880,751,536,156,986,391,32,274,177,467,720,753,398,437,715,903,530,100,780,793,697,655,636,929,989,221,509,310,3,682,0,900,967,760,768,517,819,224,238,489,864,363,496,368,675,357,661,971,157,642,47,474,973,966,222,917,157,193,29,995,725,737,930,611,226,868,20,949,910,190,501,825,874,366,177,698,573,825,59,701,453,173,337,869,247,955,783,189,441,896,692,49,751,834,750,611,506,889,903,142,225,653,486,49,87,426,452,111,35,913,918,875,795,526,127,447,303,618,67,650,726,737,321,705,379,106,886,37,994,314,950,487,498,885,990,197,518,863,95,430,898,491,395,183,128,150,446,610,488,689,422,173,860,132,468,496,387,941,708,798,940,870,534,347,684,557,352,521,319,188,417,444,35,572,460,506,931,318,931,471,806,317,382,105,736,943,78,147,276,914,718,120,398,40,108,326,677,579,595,380,29,911,553,787,201,359,126,269,528,649,883,262,849,121,661,696,960,274,203,441,164,293,492,794,965,353,739,673,450,849,584,999,203,682,430,221,878,279,991,153,923,776,102,92,190,986,427,120,791,563,173,796,411,17,152,878,580,833,381,549,130,145,159,74,979,640,622,125,53,409,59,592,303,737,362,419,14,416,946,1,446,100,355,996,108,246,280,501,821,172,848,808,219,706,56,671,326,900,680,896,689,110,212,628,866,793,315,815,615,136,196,631,605,728,962,160,165,856,105,813,21,786,870,104,969,152,438,836,682,302,234,214,622,235,67,467,934,478,604,557,111,508,924,230,270,134,772,674,845,622,577,575,800,758,928,420,408,653,501,697,657,687,202,580,582,434,880,791,11,609,683,487,303,634,709,69,145,687,898,839,209,139,534,87,674,290,828,643,41,658,105,689,601,758,239,895,851,694,9,522,787,142,759,973,131,355,369,823,815,3,748,466,474,422,313,964,628,965,720,817,552,143,264,240,114,531,531,156,980,303,310,604,745,404,587,775,918,506,344,35,56,326,270,367,875,608,700,45,947,739,563,683,698,29,382,139,284,260,848,908,866,564,488,395,19,615,246,20,423,282,130,819,655,684,65,158,880,562,898,488,595,841,735,597,283,907,93,930,984,892,994,902,649,123,530,269,472,920,666,931,832,281,384,344,613,624,173,722,373,398,974,466,75,578,858,497,693,656,65,861,168,886,571,110,340,925,145,654,432,772,436,486,607,851,70,446,57,494,685,792,458,772,107,34,241,278,37,354,6,391,783,796,354,840,823,294,773,308,974,219,251,525,934,598,65,342,273,913,186,770,548,247,453,441,532,28,788,505,732,531,81,827,609,842,279,745,520,446,418,808,619,623,862,208,283,256,649,527,298,998,948,459,960,488,295,59,822,71,578,201,335,438,853,75,249,608,268,491,897,991,554,484,690,922,457,784,741,127,68,852,420,375,42,246,495,995,635,421,90,940,850,399,535,561,753,628,774,668,761,270,682,148,762,476,805,910,905,238,648,639,182,449,135,361,888,613,366,506,636,979,703,649,697,55,290,307,854,152,641,150,946,460,147,549,221,255,266,918,911,553,65,466,976,659,283,567,9,353,958,89,803,688,993,700,813,348,710,841,454,430,412,412,704,839,885,499,915,30,789,882,851,215,958,828,454,509,961,801,602,420,832,514,824,837,182,928,28,14,984,35,853,754,57,689,775,510,170,346,700,825,681,733,905,932,657,783,365,515,906,22,229,681,430,87,445,636,845,997,787,304,68,928,439,227,456,398,281,682,308,457,151,631,558,334,483,199,232,739,202,408,678,881,311,472,13,35,129,968,532,827,969,706,762,825,706,444,695,8,344,225,187,368,244,57,197,749,45,94,453,993,875,875,603,320,61,340,165,614,472,630,749,649,961,386,578,609,977,225,676,694,415,601,96,656,277,819,891,533,430,656,388,739,614,298,624,974,718,669,278,119,276,637,112,400,570,596,503",
    "793,370,932,83,781,873,215,621,290,625,274,635,248,86,518,515,21,430,40,299,140,934,909,122,103,787,155,314,899,407,424,398,185,172,180,326,391,129,479,336,702,961,901,535,269,24,435,734,556,529,650,763,344,693,392,72,331,977,1000,410,282,692,194,87,132,853,109,28,827,785,248,15,911,285,821,152,313,370,297,442,59,902,434,765,628,737,344,682,84,689,954,601,226,124,687,847,948,15,463,921,76,946,979,385,389,732,523,798,158,262,894,925,684,880,784,431,794,184,893,111,216,959,651,889,158,299,163,854,774,467,585,412,989,356,766,769,619,862,585,999,855,952,587,982,774,220,612,168,799,497,918,471,527,104,447,750,997,516,86,277,15,519,559,900,0,920,614,201,389,800,48,367,610,540,845,948,891,237,721,11,563,543,668,942,938,669,157,391,591,194,660,899,49,409,87,623,831,588,420,613,406,157,750,392,999,352,23,363,738,953,772,686,763,776,800,199,587,670,731,398,769,46,621,701,445,423,253,248,914,284,891,837,810,467,412,600,446,57,944,607,250,21,786,338,530,276,960,228,262,981,788,443,477,179,391,668,302,543,691,485,525,494,214,297,497,682,908,894,289,376,89,900,8,973,232,260,723,130,262,475,297,276,209,957,416,722,170,249,81,731,728,55,249,607,821,992,571,259,789,511,637,907,516,818,112,26,899,158,173,787,801,486,675,807,783,740,226,616,383,367,761,965,80,913,512,231,684,354,489,213,272,261,584,274,656,30,380,341,629,566,975,475,372,818,423,722,191,269,409,830,936,634,27,677,565,982,963,529,978,449,537,1000,739,140,983,278,128,225,225,200,894,716,950,97,408,200,631,132,64,801,234,7,145,166,502,980,485,360,469,509,74,322,355,846,110,163,686,213,468,401,634,104,435,517,299,304,861,254,569,436,915,371,467,579,370,724,702,315,174,271,425,912,364,720,338,326,178,518,690,905,743,236,445,212,648,552,994,206,711,967,890,683,203,202,74,578,130,533,113,147,938,667,708,258,951,778,860,587,879,820,245,141,919,859,844,597,559,154,584,760,856,811,963,227,963,860,652,204,508,877,669,542,750,904,650,967,269,116,633,499,272,386,985,975,506,371,763,889,759,556,164,185,222,472,93,88,666,32,406,620,121,573,355,256,125,318,610,505,733,736,379,91,290,877,151,955,320,320,882,227,833,235,760,529,340,604,863,522,211,446,979,987,135,240,58,945,575,548,945,616,525,448,182,587,265,126,878,123,51,508,959,167,372,859,42,810,404,756,272,68,325,751,44,460,944,95,119,2,181,321,711,253,618,237,519,9,88,145,324,526,128,644,810,869,515,473,879,330,629,947,54,56,785,655,463,391,696,287,303,156,878,742,511,812,746,479,368,415,560,817,515,861,12,779,630,889,937,999,320,491,67,833,447,656,560,705,829,224,922,408,879,427,242,424,968,129,210,323,264,298,920,704,846,213,60,130,952,489,141,652,284,985,947,145,626,955,583,628,719,875,455,372,987,573,827,233,559,290,197,404,25,248,683,308,876,54,104,87,397,737,60,465,791,20,484,830,912,249,265,710,756,529,135,367,624,278,55,712,192,63,519,471,701,367,763,322,273,643,882,62,986,666,772,77,336,752,9,591,831,652,891,514,979,688,114,609,551,7,290,919,922,456,738,845,258,676,153,243,209,856,943,720,294,166,450,918,15,661,118,447,746,881,927,690,358,347,951,466,810,484,899,252,420,217,55,472,158,142,265,582,531,556,691,979,841,22,373,950,627,518,259,290,102,389,264,700,875,501,505,355,214,958,139,130,313,413,687,949,647,192,610,357,873,87,293,886,932,467,540,209,672,393,372,649,475,26,530,43,970,398,36,443,453,77,298,839,938,769,111,724,55,288,253,860,147,101,429,597,408,980,481,364,679,469,724,644,831,592,517,109,818,956,524,776,407,353,483,242,46,704,311,840,425,192,795,490,407,820,273,54,135,505,259,881,197,369,564,277,196,752,845,145,956,154,571,661,224,859,507,456,408,172,546,684,622,868,194,333,545,77,764,865,150,33,487,672,649,621,408,979,278,558,545,454,499,85,625,90,542,853,141,709,246,119,446,848,194,57,467,364,279,925,673,183,596,373,20,526,755,816,418,459,976,992,243,555,457,957,779,271,917,467,135,531,659,31,933,701,392,651,902,588,151,109,416,861,966,402,827,706,872,143,770,567,387,669,547,318,632,752,808,1000,349,897,705,601,857,838,789,554",
    "189,512,785,916,41,694,940,938,121,552,836,866,911,238,828,122,749,871,969,111,264,716,473,388,371,729,388,477,708,184,712,285,150,58,320,46,79,719,467,431,412,61,677,215,433,14,541,237,79,316,790,42,92,794,602,419,33,939,824,122,102,334,252,496,472,317,121,201,437,659,448,746,270,643,167,339,157,90,166,594,501,773,447,781,951,208,102,520,341,147,654,670,548,285,834,483,607,287,531,188,76,682,528,303,882,4,334,808,922,777,962,702,119,176,930,637,742,364,974,257,844,178,436,458,630,564,242,853,885,414,896,207,718,528,855,125,998,958,308,699,130,184,740,241,333,704,137,710,301,156,266,514,939,182,728,172,46,631,893,227,380,341,170,967,920,0,566,463,969,959,505,25,843,751,884,228,681,395,365,984,64,530,538,355,762,602,357,739,601,93,133,676,364,625,218,891,269,627,145,256,638,990,297,210,841,255,42,144,58,531,642,127,813,162,690,152,516,342,457,396,647,564,99,325,332,501,143,546,429,79,280,592,819,675,246,943,368,558,148,85,416,876,142,489,878,752,127,18,463,16,333,60,763,936,892,407,239,605,460,448,936,110,522,107,85,276,779,968,773,128,639,887,680,619,173,678,620,899,565,168,39,670,573,110,44,806,27,878,790,387,6,750,403,629,323,570,432,664,323,6,424,687,720,600,992,657,435,695,957,585,513,94,14,70,119,104,934,810,35,721,97,685,587,820,995,287,399,956,350,639,310,675,824,780,268,591,369,641,374,141,971,984,570,193,748,663,373,993,469,603,890,804,434,351,575,398,839,929,510,724,773,859,863,617,689,355,585,306,756,193,706,580,197,780,678,937,716,837,457,459,61,639,828,687,766,760,691,135,135,250,828,831,736,898,896,534,898,100,54,664,44,142,848,250,659,775,931,79,865,287,295,800,333,911,920,619,152,415,851,152,382,735,110,671,941,437,104,13,80,379,720,701,755,221,103,742,473,836,644,896,428,948,393,6,627,247,218,124,171,979,185,188,521,385,46,201,482,689,166,150,195,499,258,968,353,60,277,658,190,755,821,353,328,679,710,633,690,601,138,592,461,102,420,624,542,931,386,117,115,588,220,661,242,114,153,939,163,439,11,629,980,498,571,760,532,898,432,465,646,696,343,661,578,979,665,194,190,199,919,801,16,158,689,337,788,567,385,188,709,825,43,69,115,430,322,698,533,578,45,567,66,439,57,470,949,525,768,893,246,121,53,152,659,807,206,241,522,12,613,150,464,823,275,815,103,478,365,402,573,601,957,505,125,67,382,981,91,839,704,736,856,870,271,611,533,673,160,640,896,606,968,882,377,233,62,773,405,247,156,387,817,708,804,780,547,287,137,256,498,387,717,837,416,312,295,353,469,903,938,29,942,337,441,346,692,445,590,832,394,226,410,948,664,737,308,954,714,215,391,563,831,365,523,445,995,624,884,600,135,179,279,339,189,318,540,521,4,271,243,17,231,222,953,619,510,750,933,780,145,237,306,399,810,603,823,766,522,110,107,544,837,19,33,478,225,763,508,233,811,106,234,941,465,893,14,401,723,689,49,382,385,795,679,70,627,239,658,944,673,881,218,459,615,166,586,851,551,47,944,256,806,978,336,569,668,394,748,797,805,705,66,952,281,133,910,171,777,533,533,174,560,251,835,145,276,837,773,411,334,547,474,770,570,106,407,454,260,23,252,63,897,613,148,440,767,177,692,552,877,177,601,316,184,158,530,732,280,689,771,43,513,268,71,514,457,795,581,538,743,243,377,148,727,50,81,709,764,843,570,738,782,470,69,378,118,410,370,833,648,772,613,207,279,144,930,27,808,52,395,430,507,142,718,855,180,193,541,531,999,751,562,177,297,253,730,68,65,306,837,824,471,95,155,397,664,393,454,392,883,513,648,535,716,33,579,375,752,984,983,123,700,635,996,604,915,687,50,543,216,663,70,891,370,768,797,174,955,473,44,808,78,410,163,685,525,75,119,224,688,257,662,210,127,215,834,629,829,143,390,108,994,368,83,177,848,156,588,757,363,46,543,747,77,825,280,418,460,851,175,258,962,161,444,491,105,5,422,142,138,683,925,295,607,205,550,56,356,785,599,72,403,217,796,710,691,305,266,174,511,548,128,875,363,443,885,248,153,577,574,143,311,33,517,258,242,479,152,288,941,130,8,331,305,356,515,726,430,103,742,460,606,737,435,492,810,772,811,423,355,272,856,37,234,273,467,927,366,544,116,235",
    "926,817,923,373,271,607,618,983,831,300,203,371,958,844,597,199,500,815,578,476,48,990,713,677,936,702,439,278,979,641,858,688,739,523,545,522,970,958,222,27,187,195,505,330,177,203,678,299,558,369,718,622,547,11,14,857,897,435,160,444,813,357,687,644,939,456,570,973,427,409,423,790,265,160,658,659,900,787,957,654,981,715,870,118,436,988,791,436,723,709,704,468,861,440,749,978,681,716,15,57,432,417,205,911,215,536,132,327,453,569,379,323,71,305,245,591,502,773,339,642,30,144,817,617,313,957,874,7,266,65,410,470,967,99,468,405,893,915,279,843,897,147,523,336,990,278,461,459,741,820,184,219,577,422,951,207,186,811,928,849,402,267,179,760,614,566,0,576,135,254,969,926,409,386,329,276,349,849,756,294,801,691,618,48,993,177,619,347,526,303,188,150,944,494,436,916,586,389,112,326,358,943,107,342,770,993,679,472,767,629,163,564,729,977,33,127,745,922,628,76,204,370,111,126,129,424,187,285,102,891,129,782,318,493,595,217,690,885,648,243,137,890,731,327,7,397,200,569,580,159,481,201,874,520,280,562,756,329,8,18,655,777,453,944,661,997,493,618,793,341,773,214,578,882,623,832,515,113,143,204,542,948,197,446,76,264,682,339,155,733,225,140,927,701,136,566,903,972,697,905,627,769,622,820,49,269,432,80,496,246,659,449,450,904,402,992,772,993,345,73,887,419,270,27,788,106,848,23,352,418,19,206,147,326,57,237,774,979,558,356,282,593,146,425,717,630,958,476,940,460,603,681,24,695,696,124,799,458,250,254,115,777,337,593,547,752,773,820,873,404,843,636,512,985,244,989,683,671,529,938,91,929,461,201,281,936,967,620,693,62,171,331,749,118,167,46,438,474,819,423,685,91,263,12,496,57,498,739,705,261,732,438,510,233,339,175,936,30,160,854,345,452,482,578,660,914,588,786,336,482,357,46,271,934,854,691,759,69,732,578,421,934,179,945,437,915,382,691,663,989,894,696,225,271,571,405,46,739,300,866,299,267,796,559,434,16,238,977,637,861,40,667,434,92,166,326,642,242,417,905,432,702,423,263,915,689,511,659,320,985,461,214,379,459,628,780,849,296,605,389,688,811,462,171,774,182,657,850,179,782,64,818,257,763,831,884,555,132,792,302,449,45,150,577,869,809,765,126,464,969,488,634,805,944,697,2,769,662,9,874,879,487,525,67,474,340,954,793,138,956,204,55,185,730,311,117,510,732,536,594,254,783,275,877,722,349,656,610,703,71,988,400,56,712,542,785,102,366,673,270,836,787,160,190,118,984,554,654,337,199,302,942,259,237,241,328,12,968,684,933,427,845,619,822,859,147,328,658,741,324,509,221,126,269,259,783,61,72,524,328,757,913,938,804,948,216,141,194,712,582,979,154,20,655,875,578,99,583,532,515,330,998,745,408,84,162,638,324,689,507,987,685,467,166,145,790,233,287,591,892,985,743,107,550,857,753,798,71,870,651,764,280,935,535,712,182,927,170,947,528,48,839,138,506,629,760,419,246,478,183,201,329,909,773,65,663,588,752,844,141,663,194,756,262,53,897,640,944,73,758,481,368,597,990,95,584,632,173,516,81,792,387,985,3,983,461,758,947,671,564,89,945,470,696,589,732,213,31,296,748,785,324,474,110,135,769,221,851,330,170,150,553,187,344,775,907,475,185,963,548,653,83,775,514,729,654,299,897,575,775,41,449,298,708,774,327,171,898,257,634,459,891,584,831,714,36,87,561,262,928,572,563,11,50,235,137,591,788,524,279,361,858,640,787,61,72,180,848,203,926,382,671,806,855,536,232,943,850,488,196,698,655,164,79,860,270,666,694,388,783,963,669,578,543,530,155,667,550,387,427,238,54,455,309,689,322,405,422,627,848,951,882,454,659,326,66,88,624,849,268,883,232,176,218,244,398,604,474,929,288,479,16,437,673,324,339,484,244,728,691,883,479,14,770,12,878,630,364,20,598,509,720,737,956,663,490,284,836,38,586,143,185,874,569,424,401,943,223,363,462,487,802,703,332,37,380,109,93,879,447,690,647,292,964,620,491,476,116,202,913,184,1,546,202,843,61,599,354,759,79,527,928,519,68,274,829,842,780,198,136,604,182,484,852,616,396,662,55,790,71,428,212,13,29,437,62,624,226,530,799,969,539,779,486,745,902,387,650,691,729,643,657,39,833,850,528,616,461,722,605,894,482,47,717,821,83,981,42,173,52",
    "442,584,240,413,412,736,140,853,447,340,712,982,927,624,754,867,570,958,562,811,38,924,662,157,329,906,16,213,627,90,906,98,861,476,200,660,229,315,179,552,114,6,281,428,910,459,888,512,551,319,948,555,68,869,829,28,838,850,187,446,684,239,298,697,246,55,560,885,17,697,172,90,755,735,419,698,602,383,652,795,917,198,973,549,494,445,603,293,849,515,367,252,677,304,593,20,45,449,526,966,317,757,685,124,398,922,137,769,191,788,931,321,72,399,458,76,193,255,657,859,398,167,257,596,868,511,248,661,644,815,562,216,764,312,555,859,255,52,616,476,882,488,41,979,896,942,95,29,994,900,422,823,482,927,630,292,643,243,185,445,85,34,793,768,201,463,576,0,198,972,269,468,73,915,300,856,200,559,529,524,205,598,208,347,178,66,750,579,729,763,307,223,351,860,945,340,831,765,218,519,570,316,762,428,531,74,877,915,781,290,891,237,538,854,688,545,777,796,548,88,812,410,533,751,473,589,908,719,188,79,179,827,398,330,7,597,400,231,683,422,517,307,512,778,743,148,22,233,749,795,917,754,693,854,933,3,443,809,623,627,581,532,93,674,722,872,476,368,913,265,45,162,946,541,238,972,405,356,701,176,915,278,912,644,301,374,821,147,820,241,856,926,517,477,162,258,200,832,975,388,926,861,702,818,97,330,628,617,462,131,247,576,457,460,229,145,949,411,406,340,215,803,177,253,821,544,827,81,878,917,121,923,187,913,510,627,209,8,300,193,51,673,828,823,873,190,118,341,770,16,932,464,62,754,254,521,501,602,424,587,116,572,889,350,213,799,590,868,580,470,489,465,807,260,700,772,518,207,572,424,298,4,670,998,466,911,976,777,498,958,791,524,13,566,953,483,282,383,775,291,927,383,12,845,678,887,161,586,502,695,29,624,540,768,162,739,438,913,28,407,161,183,60,120,413,872,874,314,164,530,599,852,7,626,833,101,635,230,322,800,936,945,116,351,365,43,652,149,287,286,927,461,921,265,553,240,908,328,640,417,993,317,136,698,926,195,810,381,363,158,502,682,725,203,657,73,168,320,343,885,881,326,604,184,873,291,825,235,114,810,666,952,713,13,752,131,64,36,145,224,872,408,120,953,699,611,661,105,152,505,93,430,69,950,832,677,515,891,747,922,297,356,1000,230,319,463,881,68,546,47,496,562,922,302,324,623,418,867,336,368,192,615,299,127,115,931,733,860,347,209,874,220,440,87,219,380,179,972,948,746,257,969,390,123,917,443,462,89,284,907,572,143,81,25,37,432,876,291,521,142,554,254,515,21,388,618,842,664,830,818,930,631,358,30,447,746,202,200,323,675,37,917,462,76,2,454,476,875,569,196,794,626,678,383,894,898,685,958,260,355,55,637,710,402,626,231,491,437,960,108,148,253,366,365,737,104,536,124,718,793,319,680,533,631,831,570,594,3,351,14,851,752,297,91,868,948,265,787,645,742,795,686,8,303,869,915,887,449,438,440,289,775,796,116,787,558,886,160,321,325,464,795,381,48,13,808,583,998,399,862,734,823,753,156,169,427,820,137,506,975,382,702,918,943,155,175,266,269,125,144,656,593,474,722,442,680,970,899,421,854,845,419,458,270,566,682,40,247,279,255,764,533,18,620,995,127,304,471,593,730,513,931,916,947,937,155,50,314,755,262,451,742,928,487,71,696,75,567,259,686,220,563,59,101,973,466,772,577,433,342,336,982,389,784,930,719,661,331,496,953,354,302,473,124,56,356,14,946,815,351,650,444,275,767,690,414,209,762,757,671,235,74,324,290,363,177,521,801,673,273,939,144,880,515,737,206,731,694,999,966,292,86,695,969,742,307,526,929,984,18,22,655,315,570,398,954,968,591,481,170,801,392,492,48,674,255,777,237,595,725,376,411,275,543,161,496,428,924,918,193,718,80,81,176,173,483,505,730,803,507,189,214,175,798,453,974,92,608,408,733,389,808,932,99,485,864,637,123,211,544,133,944,163,410,919,771,242,421,820,610,731,741,826,965,898,823,127,983,170,585,76,395,879,264,38,659,865,880,157,598,886,367,988,532,434,598,718,920,505,930,168,897,438,672,244,837,683,228,801,860,229,606,221,524,93,738,878,178,696,281,990,813,845,593,312,523,397,464,286,290,693,56,968,192,325,717,195,26,834,753,934,66,644,200,127,149,292,848,308,638,263,715,953,432,420,952,257,583,936,938,191,22,1000,445,975,866,58,831,392,392",
    "465,872,390,642,854,713,760,129,54,275,257,618,471,344,817,262,703,754,77,985,335,530,326,318,666,600,15,435,280,637,65,407,406,507,90,303,928,57,227,865,433,895,109,608,110,250,803,992,220,3,194,125,413,107,554,198,457,323,770,408,227,873,174,154,334,538,745,993,8,441,923,345,497,289,910,763,903,820,88,548,647,361,550,370,317,54,693,321,142,194,905,234,300,455,680,973,971,725,177,153,151,437,112,501,309,37,330,987,417,548,373,573,848,155,547,35,452,411,459,933,196,393,639,812,382,775,193,6,648,470,153,105,277,18,954,437,156,164,556,985,922,272,381,566,65,233,195,719,915,545,710,918,995,706,957,758,123,572,128,304,475,787,801,517,389,969,135,198,0,64,153,270,883,818,408,696,590,317,834,745,465,29,91,346,233,87,47,641,88,890,730,292,536,840,421,646,362,133,50,341,346,470,304,989,60,55,924,859,528,341,237,530,106,495,358,598,370,722,813,710,594,179,805,50,540,372,622,797,296,306,544,62,920,823,783,895,149,815,211,783,27,499,777,951,826,848,161,738,722,501,354,685,700,34,561,764,273,313,301,26,651,364,121,624,480,607,972,199,282,955,778,358,583,891,385,319,548,854,90,533,800,480,701,49,431,76,622,15,11,786,966,227,905,701,113,110,474,67,978,397,642,507,789,139,533,546,76,507,899,707,906,447,638,12,812,618,752,639,740,262,133,158,374,798,532,555,542,80,494,616,18,627,396,454,913,473,952,655,159,211,928,925,294,215,748,991,320,562,608,869,656,270,216,793,14,366,576,456,303,990,19,97,476,454,554,499,601,635,317,430,683,18,88,484,558,773,658,282,483,82,272,794,986,99,40,30,121,922,276,833,638,99,459,274,148,983,758,5,937,687,399,750,281,100,982,106,637,894,344,221,179,528,766,347,629,244,490,572,453,629,921,761,293,619,702,636,573,33,501,995,220,698,114,423,964,151,387,432,14,161,109,471,670,910,79,654,323,880,993,17,670,62,695,349,591,908,635,850,798,239,634,704,874,699,122,546,380,906,709,984,461,296,822,595,778,531,395,680,64,356,115,335,377,502,674,206,901,638,556,393,622,674,552,432,781,693,963,704,117,156,692,280,420,149,408,694,838,386,893,683,186,509,484,636,840,580,811,458,152,156,411,863,12,870,507,506,823,645,305,290,448,542,175,769,291,504,530,10,737,875,720,960,156,820,980,80,244,100,726,286,634,224,809,752,581,280,973,461,761,586,696,839,109,417,995,424,307,913,412,142,236,447,537,164,395,818,552,709,493,450,903,211,788,56,499,344,121,790,436,223,430,873,1000,352,637,883,134,408,544,466,76,364,23,895,104,106,719,226,88,984,920,920,235,149,934,344,793,370,887,580,263,8,328,295,885,767,453,949,233,547,544,663,474,973,656,450,845,150,897,366,496,132,429,747,265,258,22,899,786,242,190,357,458,582,58,63,634,957,831,59,523,564,287,390,889,277,145,616,45,443,131,393,622,913,942,355,348,687,501,139,318,347,830,203,505,163,182,585,275,608,260,281,924,304,880,146,977,694,72,895,642,794,216,298,855,156,351,118,683,458,175,281,187,855,126,614,645,394,300,254,69,74,340,955,423,204,457,694,448,345,400,688,90,972,18,967,800,884,638,562,229,230,615,279,180,24,994,485,372,636,999,825,320,253,482,868,216,410,423,770,551,723,899,962,545,971,43,630,520,425,207,865,641,888,790,26,479,857,808,516,130,870,500,103,631,98,33,535,698,737,807,904,531,180,630,440,5,573,282,14,355,564,83,264,186,828,907,842,970,491,273,403,950,394,263,896,889,312,985,275,825,250,508,298,878,938,927,219,727,971,916,964,111,618,111,257,789,715,600,324,351,448,477,67,103,997,490,45,630,552,662,742,812,179,353,458,61,339,798,678,505,109,447,785,661,124,286,66,318,253,84,120,977,605,628,428,146,850,153,476,740,320,230,236,146,225,551,558,107,850,538,589,215,246,23,804,575,337,864,445,956,728,245,744,495,745,396,527,105,443,206,250,176,6,299,519,997,82,177,15,998,318,590,226,610,471,405,632,258,701,916,926,321,545,441,14,633,871,448,286,423,279,881,463,173,833,958,673,98,512,245,155,633,850,602,917,920,367,248,335,448,869,345,117,253,918,331,407,498,514,920,656,368,99,820,218,443,653,847,443,634,484,733,338,882,880,387,400,622,588,684,226,939,804,635,577,643,983,788,409",
    "225,221,636,157,961,371,427,116,76,986,866,354,818,283,721,296,271,652,279,132,333,124,509,482,348,679,217,600,645,653,813,710,895,247,84,37,217,675,619,545,472,414,478,139,848,873,649,876,544,254,636,298,952,94,981,297,177,102,720,586,268,901,552,878,184,541,388,241,11,513,94,252,463,840,599,836,845,60,120,316,614,816,144,725,457,307,949,431,957,925,883,918,215,573,592,291,849,119,68,708,962,659,117,232,755,919,283,433,378,718,570,353,524,32,872,893,107,215,21,630,35,638,173,187,594,769,242,199,299,739,696,31,836,554,703,871,938,208,759,734,10,324,831,331,35,621,892,469,33,348,796,345,318,320,815,825,282,668,907,402,646,923,913,819,800,959,254,972,64,0,327,944,277,707,5,885,775,158,501,181,734,532,787,186,528,83,696,875,230,690,318,45,349,650,211,844,735,173,482,811,370,538,724,17,86,333,116,12,677,847,220,261,373,912,931,506,910,999,621,17,124,729,557,186,224,234,532,427,901,747,138,897,730,129,933,419,741,352,176,202,641,416,259,668,162,377,606,573,87,824,530,282,835,635,654,146,309,218,340,314,522,998,65,2,399,348,437,969,461,839,700,499,247,480,865,771,899,99,885,560,260,893,535,14,622,535,638,909,430,487,662,371,849,27,349,361,151,620,97,732,419,162,95,352,792,434,96,689,542,392,653,774,292,527,10,86,982,331,302,158,383,260,762,253,612,30,875,535,214,296,474,178,973,582,637,817,227,474,730,361,781,428,100,825,523,844,802,60,832,790,51,19,325,995,811,772,13,624,176,269,941,454,967,272,927,863,879,799,230,852,222,233,484,646,341,215,53,707,657,372,255,381,855,900,157,725,690,313,836,937,621,337,961,75,152,797,583,420,931,855,800,685,997,47,598,863,548,72,44,60,903,103,422,212,605,294,591,280,251,21,105,499,730,433,509,579,567,769,144,800,642,889,481,944,636,523,302,670,964,340,501,599,129,86,738,694,775,770,436,886,37,421,138,998,33,385,136,200,981,625,528,857,852,486,321,171,207,62,949,81,515,893,239,708,169,176,26,571,563,796,636,625,7,738,335,700,393,515,445,253,714,722,180,153,32,198,639,200,670,127,245,952,779,289,858,994,303,552,173,839,76,560,702,77,202,188,81,656,938,926,635,575,539,546,807,176,434,9,87,535,644,121,840,690,381,133,697,757,34,710,547,704,25,327,537,653,635,12,708,577,978,494,813,698,149,54,518,514,153,899,527,907,998,577,763,563,902,183,558,966,741,36,449,510,293,527,148,871,64,931,296,280,601,913,964,194,785,666,417,758,134,32,382,914,740,850,879,710,694,114,241,243,321,958,16,126,450,733,973,199,15,327,152,111,636,79,443,498,18,334,623,632,837,562,35,473,26,354,343,526,667,691,685,101,115,171,609,641,194,187,968,157,427,510,23,444,693,388,348,71,935,253,247,377,900,660,811,717,567,884,844,175,238,610,615,605,838,134,680,589,504,998,762,580,796,194,388,278,54,198,491,470,902,729,235,788,491,478,579,161,175,443,162,177,368,344,508,949,968,647,266,338,141,69,144,670,45,932,455,124,678,712,380,145,179,966,928,211,856,170,760,846,513,322,789,978,10,47,965,759,802,551,977,841,23,980,926,210,64,215,182,150,989,789,668,85,379,869,775,383,646,569,340,358,528,762,575,903,180,482,573,568,368,350,533,165,66,494,129,493,610,41,599,410,200,252,58,681,530,599,578,255,611,56,519,571,610,379,391,493,989,664,216,342,580,220,632,299,919,943,272,877,393,729,378,684,903,70,32,79,972,756,761,664,793,405,196,763,534,587,40,317,131,675,747,985,699,57,526,665,138,55,648,250,466,932,658,28,333,242,145,662,576,887,121,325,384,115,171,3,928,816,493,553,615,74,331,412,941,848,442,741,21,275,327,166,523,892,793,265,711,555,788,697,678,96,939,604,144,892,643,152,713,989,572,827,316,799,681,635,6,623,537,272,265,781,474,67,159,449,810,178,880,99,84,982,739,893,144,433,160,727,760,533,10,362,170,335,977,74,100,538,536,109,401,874,950,946,425,913,164,698,234,500,14,35,700,422,120,606,460,959,171,506,201,90,961,556,73,153,467,563,239,534,328,724,104,815,13,662,910,760,519,354,46,441,36,252,504,374,907,989,450,804,234,548,325,366,828,453,777,30,106,929,526,301,871,676,483,183,846,11,915,79,793,679,401,525,907,532",
    "573,184,534,212,528,559,228,868,45,223,581,97,648,813,957,754,59,816,84,844,148,358,370,881,667,643,974,339,784,291,283,949,234,765,161,222,238,583,442,106,868,687,337,653,188,932,94,89,269,767,354,460,730,525,945,203,686,762,510,901,595,691,78,141,928,309,323,813,488,308,129,969,683,682,29,196,693,437,920,496,41,373,260,396,813,806,118,646,270,961,106,816,64,674,556,34,437,917,716,346,87,301,119,360,575,791,581,936,32,462,160,547,275,945,53,857,473,512,234,384,576,964,831,606,472,915,810,269,977,834,205,972,310,874,598,230,209,26,164,829,341,57,641,809,733,649,296,894,743,527,183,968,888,542,939,767,914,371,435,264,818,527,629,224,48,505,969,269,153,327,0,66,476,45,857,582,690,849,521,462,606,892,259,528,798,798,616,896,539,468,777,173,756,839,328,633,276,50,788,441,277,215,288,947,849,443,643,360,867,305,348,239,527,508,548,553,399,319,719,805,20,766,835,536,395,55,101,776,98,926,23,64,704,624,350,265,659,546,608,975,181,255,461,775,549,694,427,65,475,162,844,666,802,468,793,214,449,119,588,174,562,835,158,660,688,73,187,489,114,216,832,184,637,195,706,989,987,61,60,204,666,201,589,273,592,569,598,570,57,680,891,618,324,993,972,574,689,200,762,143,859,552,456,53,168,258,677,28,884,716,157,716,350,708,539,68,683,561,476,576,146,427,617,722,572,873,155,892,464,328,33,162,700,591,967,872,554,830,776,168,655,350,88,84,941,869,825,803,56,757,656,364,132,229,345,764,155,57,76,635,95,744,676,290,166,18,527,938,405,702,140,522,818,679,34,342,354,539,377,13,443,201,582,883,865,736,729,638,930,791,109,447,227,580,134,807,555,272,264,95,968,294,697,426,224,344,376,31,349,371,568,590,375,713,197,486,827,393,748,82,500,431,58,922,240,706,78,856,903,50,630,102,987,132,744,418,528,718,381,99,12,398,750,357,274,484,376,101,431,498,215,155,933,700,517,785,9,792,536,124,761,722,173,904,194,805,438,495,695,176,617,783,141,334,493,850,901,798,953,817,44,831,782,492,430,319,878,61,529,390,457,186,34,659,989,290,123,828,425,729,521,348,989,512,953,566,151,701,727,505,417,787,526,515,453,325,788,756,586,58,773,843,339,100,248,489,914,589,190,715,64,707,334,612,958,856,134,398,651,962,658,332,723,631,605,10,896,66,826,410,177,791,244,746,766,265,945,686,449,362,503,450,939,117,592,32,88,320,498,461,523,588,26,789,648,518,279,655,965,243,641,537,153,935,151,48,529,798,446,219,968,195,609,702,569,74,542,627,1000,979,604,227,547,240,998,330,617,298,91,937,101,3,278,169,403,208,833,409,740,901,330,165,692,149,950,991,546,476,416,136,869,513,221,150,287,231,319,90,788,447,223,190,574,972,380,574,664,170,212,204,938,67,184,817,336,75,964,689,296,853,503,516,60,995,28,416,629,610,802,369,424,453,583,230,781,209,222,614,76,365,719,823,110,106,437,569,982,916,275,603,841,127,910,359,321,367,898,788,662,513,705,910,335,765,337,579,592,250,604,942,55,905,920,117,44,487,99,950,173,691,598,697,508,603,836,750,648,652,202,966,963,860,701,752,187,540,826,184,623,439,733,853,472,559,142,204,102,435,32,721,448,274,727,698,297,459,613,142,317,274,416,615,298,819,463,910,362,333,805,673,572,915,81,952,586,374,196,773,7,984,198,237,857,823,844,857,319,202,429,319,324,286,57,493,554,77,372,231,341,635,359,616,543,748,999,773,291,285,737,660,502,697,899,781,736,166,266,222,767,974,445,800,680,835,263,352,606,976,826,978,19,444,643,207,820,873,708,519,127,72,804,13,486,790,947,349,785,700,919,99,694,784,801,22,69,556,226,358,415,617,745,992,578,299,780,441,822,438,590,251,128,984,908,58,7,929,97,482,744,352,632,981,572,494,117,390,708,607,79,379,992,521,586,343,549,447,565,925,867,527,968,301,213,249,768,646,860,503,195,378,161,45,107,941,318,826,701,864,175,416,882,426,186,846,83,966,150,70,817,537,651,68,659,378,751,67,851,212,336,727,636,771,54,804,63,175,602,276,615,114,848,597,192,581,326,62,742,139,85,34,949,377,216,848,924,888,145,935,96,53,925,192,447,949,351,826,572,386,7,908,142,695,427,134,193,24,274,277,88,262,401,788,851,502,581,831,595,201,330,155",
    "379,321,846,366,724,476,771,436,195,626,518,473,640,60,126,34,350,936,595,395,608,928,814,164,346,976,171,173,743,579,773,305,275,617,275,413,856,153,22,766,402,987,629,119,661,820,961,160,290,798,817,24,897,732,545,109,933,987,373,837,387,97,419,711,343,465,461,322,851,343,625,326,7,339,999,333,622,423,123,292,903,132,255,173,536,277,283,962,509,725,915,680,505,52,510,517,250,430,360,127,964,8,669,195,454,348,615,61,217,940,245,659,910,326,987,859,839,31,238,283,494,890,885,992,226,10,247,628,805,894,341,755,892,720,30,967,272,734,716,350,450,474,815,437,562,220,960,808,909,180,578,771,110,412,334,590,507,541,536,488,171,155,503,238,367,25,926,468,270,944,66,0,868,730,795,927,8,596,499,172,956,394,154,879,320,684,928,300,392,588,431,717,342,392,317,217,540,79,953,285,552,834,662,402,705,317,563,923,428,441,817,474,838,927,401,513,938,845,500,833,559,785,675,783,964,446,456,283,960,615,507,463,157,694,255,514,393,708,764,771,541,873,89,154,649,345,892,38,745,20,730,433,928,455,390,212,151,57,237,739,614,369,200,186,255,549,186,481,93,83,831,48,888,95,330,286,326,288,10,134,489,813,454,52,132,544,551,164,507,326,678,964,114,816,214,345,856,205,281,368,536,643,416,366,692,427,679,199,521,348,503,243,936,572,432,586,497,712,10,724,169,426,3,183,374,169,74,75,559,941,580,375,66,377,166,954,190,954,82,918,167,640,893,232,604,95,737,751,695,143,29,841,298,309,355,84,264,386,748,764,281,310,112,58,134,252,30,275,328,198,449,113,995,821,266,272,419,754,899,815,59,911,502,595,79,392,139,933,730,627,780,375,233,760,3,263,772,891,628,346,658,695,542,993,656,385,844,880,642,601,115,945,701,494,467,41,760,663,980,627,509,923,440,593,492,485,731,317,34,739,993,573,408,302,80,46,304,229,156,638,462,282,308,943,300,831,162,720,142,937,603,985,462,871,144,759,134,815,103,819,867,52,590,87,727,437,226,476,479,83,956,366,684,474,207,255,211,986,889,811,692,621,667,505,358,370,859,772,192,531,76,406,781,404,824,849,59,645,8,327,776,926,417,912,373,754,318,401,887,615,26,799,193,782,253,553,839,991,158,504,861,419,198,427,359,909,118,833,496,857,59,176,699,112,293,187,572,752,813,925,579,949,816,882,42,831,720,994,319,325,797,713,605,13,868,360,575,481,176,750,506,661,973,108,984,287,778,615,168,391,412,683,848,594,902,68,392,606,419,705,554,279,739,515,916,720,531,86,426,568,161,295,77,960,840,417,722,239,670,863,55,812,317,626,174,382,249,910,518,488,299,778,523,982,142,416,999,879,418,406,186,211,97,670,492,987,879,373,261,181,435,849,129,870,778,60,221,641,3,914,930,308,679,234,827,760,886,203,348,676,896,848,82,586,894,386,327,704,846,897,266,900,43,221,172,386,134,250,713,234,522,223,671,696,109,559,506,554,180,189,494,233,622,578,725,781,594,754,79,531,87,25,118,251,463,592,541,916,589,715,333,88,981,692,46,121,529,620,759,746,57,199,831,44,711,35,587,853,68,556,131,206,770,780,276,500,15,202,843,603,343,429,852,236,244,638,705,430,573,537,561,985,433,899,425,913,136,619,589,401,558,374,777,10,483,937,21,58,223,505,719,758,964,640,449,946,432,344,285,566,626,424,102,881,211,427,191,29,970,23,688,200,28,402,900,351,932,678,397,855,618,268,385,883,41,854,151,930,907,186,900,611,248,757,723,487,181,844,445,883,695,826,30,217,486,307,396,556,293,18,36,114,306,302,727,896,778,271,232,379,918,154,545,392,759,799,861,177,998,105,424,471,835,898,303,306,205,363,925,211,195,826,900,681,247,764,819,313,885,47,677,929,163,649,433,304,482,74,540,955,654,187,911,883,452,764,244,349,924,874,612,910,423,929,740,996,336,632,588,227,902,813,178,571,800,184,279,658,335,945,421,346,400,345,939,563,870,904,112,558,312,596,673,154,943,970,248,334,945,766,65,760,871,328,319,597,21,43,816,896,267,356,846,392,215,105,453,743,160,12,61,189,849,207,56,399,179,430,597,332,397,294,580,845,715,629,235,909,789,406,947,272,788,856,209,43,657,384,87,348,373,367,89,111,393,593,122,359,845,273,685,241,186,408,387,915,66,387,812,510,386,113,103,370,775,163,188,678,667,563",
    "875,218,415,680,947,446,698,86,375,801,280,293,460,483,819,214,252,572,910,73,777,739,427,395,848,756,282,84,492,372,790,879,341,360,425,70,491,511,728,862,445,847,322,126,929,282,20,518,976,511,614,192,592,226,569,795,504,653,629,837,453,783,117,822,179,466,635,290,841,659,414,761,43,198,634,664,630,552,876,342,831,691,889,354,320,761,425,40,404,271,953,363,759,64,381,652,923,578,732,813,171,996,321,536,811,481,196,309,812,930,297,853,687,777,543,667,561,580,482,771,833,248,51,903,718,230,578,269,454,627,827,823,114,557,179,990,328,929,584,86,974,369,301,979,859,447,917,590,423,164,107,782,188,786,339,292,972,301,526,940,542,245,538,489,610,843,409,73,883,277,476,868,0,108,957,740,669,187,756,453,31,960,430,395,450,904,544,449,798,124,387,703,670,50,540,395,935,518,948,253,683,608,231,950,586,650,16,962,82,59,757,586,11,842,119,91,160,599,566,758,904,401,264,220,809,626,801,414,195,272,993,968,226,784,53,325,337,846,969,783,994,160,484,430,40,9,924,487,633,732,796,512,929,44,76,146,436,190,356,691,466,339,602,946,126,279,146,228,455,374,170,165,159,220,429,115,276,449,528,978,508,529,309,188,90,208,846,352,586,997,931,410,258,624,170,123,205,518,212,977,632,171,864,603,66,382,762,111,532,609,535,741,602,635,866,500,544,14,201,284,832,656,669,397,268,977,679,273,756,695,460,335,62,578,987,337,254,605,277,550,625,449,266,592,58,73,696,75,228,710,772,445,365,993,733,85,240,318,337,149,589,376,115,593,109,285,890,297,457,38,636,766,202,579,447,125,466,498,101,336,239,367,192,219,300,163,643,379,774,370,107,786,17,258,35,87,760,595,719,319,506,691,570,14,358,930,117,2,304,853,432,373,192,241,219,307,515,37,176,237,68,860,795,886,577,103,563,3,773,326,870,138,973,355,724,627,4,939,784,645,89,66,40,181,249,671,941,972,850,73,496,849,296,585,811,486,297,875,26,101,389,678,669,416,139,660,347,767,687,561,11,363,223,489,624,639,817,705,59,189,174,43,358,611,860,652,870,681,45,884,515,7,471,14,810,319,95,616,291,385,448,798,406,538,981,785,692,842,676,662,94,365,494,508,695,449,640,718,630,81,798,39,80,453,334,953,252,532,525,902,563,54,857,671,722,511,594,861,83,467,88,324,268,79,278,764,295,68,20,14,396,260,2,936,244,76,349,953,947,340,589,98,800,53,45,724,865,611,697,660,754,646,357,499,230,95,850,258,335,268,205,331,35,764,963,179,39,98,885,456,622,217,254,160,222,447,976,968,965,240,634,422,320,822,882,783,133,467,642,561,266,278,440,326,371,963,374,314,376,23,803,636,695,917,912,612,67,527,351,926,524,370,321,106,275,851,604,886,957,229,427,766,638,825,655,298,35,749,268,381,455,816,841,603,327,737,974,5,901,855,135,213,998,137,371,665,571,762,253,967,401,432,532,752,92,375,226,103,748,496,560,862,263,304,15,861,815,876,622,782,428,441,627,737,792,669,198,994,402,47,723,76,534,440,22,931,446,916,649,913,458,735,942,151,303,856,191,619,22,705,175,838,971,723,274,100,301,709,355,514,383,681,551,469,499,760,251,485,862,903,352,31,14,485,235,359,544,863,130,651,102,501,745,201,35,901,804,405,776,222,156,15,882,131,762,299,498,698,418,176,590,769,329,491,276,723,947,651,893,440,160,95,884,921,515,172,531,574,939,410,930,998,177,603,105,244,994,214,864,368,590,248,356,376,98,887,656,564,185,215,286,431,36,307,348,805,594,947,714,44,251,848,329,450,216,644,846,28,993,320,103,575,326,146,563,705,984,753,378,557,592,124,542,508,100,857,127,137,244,634,113,443,387,182,332,664,954,805,431,352,945,993,780,187,104,56,84,213,247,724,28,695,575,636,721,529,280,652,504,181,763,234,621,52,895,534,158,619,983,340,422,765,77,38,114,766,904,124,994,596,369,302,56,180,655,65,382,864,556,870,643,698,79,950,246,794,317,60,799,165,492,73,791,235,656,87,537,644,837,744,167,319,466,632,416,296,855,961,956,398,110,417,864,741,44,90,897,136,173,534,244,154,599,464,972,870,120,443,577,991,704,517,20,743,163,688,505,751,356,486,774,485,646,496,496,11,784,439,392,346,650,523,802,894,900,191,19,414,274,551,608,226,648,571,535,374,530,385,726,142",
    "620,878,993,482,269,7,371,47,94,650,188,594,869,650,14,761,457,906,152,166,620,599,285,114,341,804,547,322,285,187,542,395,559,750,553,667,154,70,918,266,154,627,600,830,533,884,760,446,111,921,390,11,480,811,437,960,114,104,818,345,75,815,431,924,406,107,561,181,172,339,766,986,328,117,231,396,504,707,969,717,335,25,698,749,540,218,262,18,699,227,990,365,541,692,855,671,612,755,973,633,232,602,97,464,739,449,921,996,523,834,117,485,665,510,495,324,159,906,490,431,368,915,938,95,533,810,795,174,553,929,15,82,40,159,880,8,968,54,916,65,359,747,225,917,614,809,928,477,786,684,21,333,75,536,716,457,81,163,919,858,993,340,884,864,540,751,386,915,818,707,45,730,108,0,606,86,267,114,287,422,638,408,943,716,720,780,503,332,618,67,726,134,612,462,671,213,702,996,636,963,75,131,666,455,255,600,753,663,332,306,912,768,979,444,446,948,818,119,158,186,400,301,354,196,146,12,697,178,154,681,937,261,296,179,372,522,643,597,67,246,529,775,727,180,760,958,57,486,103,986,64,930,650,277,824,609,731,62,675,273,274,791,750,710,431,721,572,791,865,14,963,481,349,271,139,752,974,737,385,254,422,227,40,969,140,592,374,28,259,656,857,428,202,112,423,441,127,756,38,944,722,7,98,468,716,643,262,567,991,810,739,339,773,980,150,557,677,208,62,253,168,724,361,356,195,536,897,191,931,41,321,702,764,851,804,870,823,408,82,451,689,957,779,114,929,917,559,192,404,734,472,35,521,79,938,254,974,302,454,683,682,468,546,760,423,186,248,401,155,857,987,47,161,527,773,164,700,323,550,829,107,925,521,7,603,530,204,467,994,509,566,979,159,34,684,493,922,154,636,570,712,784,105,779,223,44,945,528,205,921,94,867,113,491,185,697,554,376,712,826,136,13,940,143,52,267,34,321,565,729,173,800,406,956,697,786,155,894,841,381,446,536,791,713,842,805,311,671,953,418,780,720,901,79,64,177,253,140,251,739,89,89,344,96,334,828,625,839,725,43,65,438,229,147,291,928,465,33,212,794,623,969,812,908,15,28,904,931,891,13,384,470,751,223,431,33,833,745,347,153,313,292,321,975,548,996,192,513,123,633,986,506,67,560,865,832,723,240,350,739,654,285,661,204,216,155,37,884,210,671,576,636,503,69,815,377,739,866,625,998,578,965,875,831,871,139,352,499,373,671,301,973,546,168,323,445,417,65,769,570,324,204,696,388,845,418,878,608,672,29,410,776,524,491,570,359,258,333,618,635,825,477,472,75,505,599,306,97,923,331,465,556,448,400,508,515,100,888,110,876,15,682,428,961,572,901,894,589,390,438,212,4,878,746,52,399,192,777,434,59,815,708,210,417,783,733,443,661,989,244,938,847,944,647,908,665,236,534,370,871,943,222,941,340,994,109,903,60,903,639,386,280,417,838,684,527,210,956,534,339,428,934,432,394,710,509,51,708,893,62,882,564,668,57,571,653,566,249,146,536,251,142,98,234,128,443,766,448,389,383,481,594,983,308,524,107,703,147,308,369,97,150,152,382,739,724,249,177,232,445,590,951,514,786,688,8,415,419,283,798,885,678,736,849,485,474,494,825,271,163,534,874,547,181,822,155,929,238,622,243,295,567,329,185,615,380,751,973,633,385,279,316,510,818,987,483,60,322,376,152,977,528,582,579,100,146,148,426,864,23,659,239,301,479,528,470,503,417,479,745,721,432,744,880,809,604,594,628,21,548,149,991,234,984,129,208,578,568,214,998,72,394,610,78,476,57,688,261,984,438,525,423,807,773,390,123,107,779,1,734,639,669,402,652,893,705,342,154,261,667,737,708,371,355,509,753,291,262,165,613,201,186,284,857,976,588,786,623,431,386,157,128,625,474,763,212,168,73,246,132,506,828,293,627,401,634,42,49,622,206,934,698,918,894,20,620,60,565,384,624,746,225,601,524,616,395,613,307,360,842,891,189,112,697,622,142,594,974,354,282,793,959,816,894,23,988,24,353,981,391,391,556,309,196,80,700,414,136,547,794,446,565,469,625,409,94,749,112,183,956,121,126,25,503,809,309,961,747,599,909,853,854,653,255,196,981,168,584,694,319,542,421,922,654,255,514,969,432,439,551,312,983,487,14,276,820,954,2,185,104,767,892,460,479,914,852,97,234,854,509,753,460,337,385,816,900,329,295,40,683,684,980,129,275,306,228,273,599,302,996",
    "294,919,977,876,7,431,882,806,57,714,924,643,473,935,573,210,638,641,289,162,336,903,808,98,194,197,884,623,95,731,531,559,265,390,718,818,195,350,31,806,769,323,4,688,116,361,979,556,624,902,967,352,212,34,385,305,179,14,486,101,798,688,227,27,330,16,989,111,628,349,300,823,504,497,39,258,753,847,514,345,842,38,886,744,729,97,900,32,395,231,589,636,220,373,322,255,610,914,439,553,860,776,803,6,893,16,304,474,151,211,937,675,639,856,660,347,890,791,619,281,541,746,303,511,557,227,214,481,243,588,690,34,554,733,804,107,561,935,381,481,230,643,476,320,875,874,256,140,362,541,290,269,493,657,558,650,3,9,479,679,272,768,480,363,845,884,329,300,408,5,857,795,957,606,0,963,612,518,99,450,661,446,272,803,836,863,715,373,354,651,541,911,200,50,139,219,949,793,659,849,760,444,78,424,966,877,47,244,459,626,612,551,655,357,329,308,835,520,527,865,604,588,980,659,375,143,38,315,363,251,182,364,835,187,673,148,753,370,926,653,809,374,361,754,210,876,265,697,788,713,303,138,207,744,88,347,700,583,448,996,999,739,747,907,610,350,317,50,930,241,374,583,602,915,409,15,744,143,818,523,98,383,472,518,304,578,5,363,102,648,892,832,734,707,496,218,995,625,395,986,930,128,328,543,106,979,621,843,464,276,938,632,906,266,422,204,378,172,535,622,919,20,432,960,678,41,438,217,266,925,53,962,486,986,654,88,191,696,209,590,899,680,518,502,579,317,105,322,320,79,195,269,62,65,786,559,633,633,65,630,704,971,165,439,460,617,538,39,140,577,96,101,330,581,516,2,958,242,846,264,129,179,129,728,306,956,604,93,42,929,443,124,529,443,273,54,291,121,168,190,28,191,539,950,523,960,372,777,410,703,194,409,282,950,96,636,115,336,616,280,697,705,135,909,922,815,814,135,952,403,27,310,98,316,524,299,97,254,711,229,271,764,860,786,564,796,315,259,897,151,510,494,892,981,822,295,381,670,822,73,349,378,146,923,138,216,197,615,205,863,862,95,744,552,148,371,147,252,361,369,960,723,94,636,341,431,25,855,884,799,634,820,29,333,358,547,784,925,489,873,872,394,697,805,522,334,802,422,419,547,337,621,968,448,396,514,572,409,92,123,737,930,84,753,108,975,157,293,370,713,737,135,827,328,378,885,930,788,811,721,960,551,792,253,397,911,67,992,55,771,922,450,543,364,602,33,566,157,526,41,764,586,24,167,756,966,296,622,517,135,226,567,23,2,922,597,523,394,21,47,390,54,802,507,584,933,873,830,725,448,637,244,145,918,774,412,616,164,913,119,584,887,321,15,54,342,482,302,415,543,122,439,322,45,142,473,415,596,2,468,366,626,903,967,47,69,917,120,148,299,15,645,739,9,223,743,619,818,432,642,615,221,901,398,364,976,859,164,82,580,62,29,967,360,509,408,144,611,772,435,235,143,335,491,631,170,401,747,399,792,293,570,274,523,662,433,686,314,955,756,919,692,409,983,5,794,880,847,52,118,92,948,344,538,88,515,590,291,623,112,964,431,665,290,952,990,950,818,338,315,546,408,104,153,788,579,117,71,250,218,561,191,730,464,997,121,335,253,485,194,565,676,103,940,590,198,759,647,866,647,696,956,343,893,231,112,171,103,535,252,323,885,437,714,644,904,853,655,308,707,894,344,258,44,643,803,927,275,858,381,114,595,313,673,34,45,82,74,257,465,724,814,773,254,485,601,30,88,60,239,135,857,943,45,129,752,752,497,280,94,897,990,10,599,439,896,880,122,306,353,149,695,834,541,757,121,572,518,310,737,488,512,355,639,164,406,765,397,882,881,221,634,421,346,881,416,91,264,808,962,38,273,819,839,644,141,479,223,236,505,506,495,84,850,427,778,122,269,586,464,361,565,349,811,594,886,205,772,468,204,758,263,62,428,523,697,152,809,297,557,872,931,337,583,861,411,859,501,797,650,111,398,839,362,375,281,132,52,289,294,883,198,179,659,951,871,759,715,220,129,654,195,762,269,369,102,295,868,827,503,293,657,450,201,948,219,218,64,704,639,426,292,643,396,60,366,986,146,839,7,572,551,532,679,835,7,328,535,550,131,737,716,861,359,404,560,939,183,630,850,653,120,652,215,266,760,106,404,408,397,848,515,613,295,10,890,231,113,719,658,419,888,202,746,318,40,98,68,868,30,73,320,637,272,167,891,380,755,554,683",
    "452,841,910,431,530,981,533,981,589,197,223,262,588,200,342,141,43,438,681,661,460,259,288,131,702,361,937,22,268,514,670,648,618,580,317,330,193,320,479,575,95,243,364,377,307,649,643,430,591,641,624,743,407,78,431,689,342,310,526,284,379,208,306,944,844,847,519,738,747,780,587,492,69,247,226,778,430,782,764,14,920,545,101,793,849,300,834,155,309,834,641,376,321,340,424,726,378,469,983,509,800,655,322,220,866,695,46,901,880,535,274,373,851,173,280,824,947,956,566,360,463,971,626,926,27,358,789,429,671,74,698,816,443,101,293,620,640,756,598,673,587,448,690,112,665,910,942,281,897,841,868,754,537,428,882,989,740,868,103,724,9,390,439,496,948,228,276,856,696,885,582,927,740,86,963,0,961,630,883,31,969,86,135,935,702,387,391,546,350,216,393,539,77,940,899,825,922,791,339,39,323,829,548,333,79,690,366,906,537,783,386,530,169,910,178,615,199,658,645,118,881,422,912,614,60,341,899,518,992,625,195,514,887,275,85,704,749,58,67,555,734,650,44,674,768,145,993,287,886,261,671,649,793,356,726,890,73,358,693,431,45,635,631,422,188,544,597,189,250,615,755,734,623,538,947,244,279,330,164,359,71,845,338,39,846,300,611,639,712,2,477,328,989,145,95,370,250,95,984,804,952,83,415,960,683,917,729,63,284,586,427,517,606,303,106,251,409,935,721,170,520,868,472,250,867,776,664,107,581,415,572,622,981,891,714,314,826,916,801,967,34,145,362,677,448,617,68,882,738,384,145,403,992,347,96,571,775,215,182,655,349,639,379,371,496,779,65,405,402,448,678,59,728,777,657,122,203,728,711,597,884,874,471,975,964,171,776,929,636,319,246,963,507,65,606,198,718,228,352,672,11,645,348,695,660,846,88,702,715,777,900,829,849,117,636,160,943,704,92,30,25,812,591,798,636,184,486,184,479,900,47,635,770,606,377,105,824,181,696,560,332,901,716,507,540,98,110,36,373,150,674,498,397,669,650,477,748,175,573,949,203,823,752,363,239,193,229,581,271,860,208,856,530,170,756,604,485,25,262,818,303,550,918,172,484,838,263,126,993,596,77,713,805,150,256,341,36,508,640,865,886,950,481,514,431,866,552,552,729,662,771,641,821,811,206,479,650,136,123,660,848,470,661,323,148,442,207,152,391,100,402,186,322,268,54,158,734,571,206,588,196,388,461,690,678,994,753,940,890,928,300,311,31,863,800,39,388,615,747,99,938,133,754,101,544,583,248,939,643,607,767,294,156,118,233,580,870,221,420,113,472,424,782,384,393,903,914,527,350,998,986,684,943,173,294,783,640,725,133,711,174,423,861,635,859,757,667,625,4,253,354,31,363,696,467,289,195,882,512,979,603,674,860,198,204,645,668,985,90,384,671,73,78,13,865,485,928,71,414,36,563,932,662,521,256,658,223,89,271,526,585,685,113,675,222,571,201,61,852,866,358,274,122,229,471,869,546,578,340,394,432,377,334,317,371,606,830,84,218,565,886,842,496,920,400,417,842,334,655,587,395,969,345,415,70,442,752,84,533,612,817,83,992,925,756,722,369,769,363,817,807,850,866,288,520,943,498,180,496,726,96,149,606,180,401,49,98,740,218,646,338,85,183,232,815,209,705,145,923,529,953,401,324,804,698,438,412,870,202,498,131,737,791,622,680,461,744,704,791,44,193,56,301,258,612,272,338,632,85,93,508,978,359,184,486,787,524,957,446,136,86,639,818,898,751,630,269,855,92,616,906,878,741,495,339,32,151,577,10,792,246,962,237,226,550,932,436,873,934,811,958,116,147,565,909,661,245,128,708,467,297,71,458,298,409,201,915,76,918,762,669,129,667,749,196,753,332,296,538,938,357,541,29,623,299,782,993,67,408,695,608,771,415,278,931,323,351,535,894,225,110,944,897,147,771,746,764,274,966,154,853,177,595,39,748,505,99,74,655,322,906,445,178,135,800,988,182,627,115,276,857,719,250,240,774,214,139,584,221,866,806,596,543,909,541,260,622,413,759,76,975,66,505,813,593,42,743,743,798,860,800,704,599,611,241,747,176,67,152,211,847,416,346,964,686,717,866,230,896,634,729,490,864,957,38,769,785,616,21,804,24,479,839,25,788,696,546,741,210,456,750,637,302,908,280,768,172,233,515,120,973,284,60,393,325,454,572,790,515,73,423,23,370,741,959,278,409,844,629,741,960,27,71,441,152,63,603,155,92,596",
    "832,327,132,961,363,181,728,197,889,186,425,253,539,613,789,746,989,940,342,414,952,842,695,921,168,504,902,708,990,201,555,780,189,999,700,349,128,730,524,285,858,986,117,479,671,824,888,775,566,84,459,369,758,222,134,842,640,711,803,373,404,842,399,106,574,675,69,68,645,947,537,643,197,202,130,852,666,742,541,542,557,161,288,834,654,8,376,471,587,157,289,175,786,766,728,6,897,87,834,629,687,536,38,501,504,73,451,811,72,600,522,242,904,412,781,115,371,508,721,764,585,441,936,151,515,5,426,962,25,559,189,536,73,497,659,132,388,539,716,325,332,612,578,970,834,638,355,477,587,434,934,934,679,875,524,357,39,591,613,846,163,30,249,368,891,681,349,200,590,775,690,8,669,267,612,961,0,364,178,759,124,142,113,577,614,291,132,400,508,552,142,221,353,203,431,656,326,901,414,239,985,355,58,512,219,542,159,461,856,107,759,326,117,339,968,627,229,832,912,476,704,316,704,51,345,90,859,369,399,739,477,887,881,728,430,548,125,224,766,931,604,367,448,230,646,980,668,939,83,574,619,652,920,742,29,981,282,915,493,82,368,974,545,769,547,600,814,455,552,848,367,772,90,754,825,196,812,700,964,683,326,202,234,75,771,831,297,328,466,266,347,900,444,88,319,810,618,946,51,469,467,727,370,108,55,511,556,584,542,482,568,139,371,729,29,150,525,285,693,892,208,437,450,444,647,5,988,958,244,864,102,302,190,671,331,569,457,609,669,728,859,548,818,535,134,654,645,288,439,526,792,290,674,259,167,432,616,956,580,143,580,595,357,600,256,708,885,914,430,551,634,214,726,34,118,549,977,363,663,549,71,424,218,439,422,466,784,417,454,273,321,864,781,596,340,733,818,181,137,302,744,702,316,517,332,957,943,732,388,398,310,728,741,660,397,483,441,749,293,141,689,116,655,986,454,305,42,995,638,576,515,389,46,136,773,307,232,884,834,28,457,349,356,809,475,932,800,465,886,658,919,309,677,288,684,979,52,477,410,693,878,563,998,768,560,206,627,386,811,395,486,624,217,303,747,838,65,681,44,955,949,780,68,36,865,257,866,32,12,866,220,39,314,479,621,203,754,589,880,360,156,736,724,888,670,709,297,149,21,220,138,431,11,911,623,662,233,562,955,466,372,817,9,751,432,263,784,736,534,215,539,491,206,497,834,476,760,497,878,67,47,125,127,748,800,476,65,547,388,663,184,275,283,12,651,326,129,235,292,879,286,682,309,208,730,304,706,100,123,671,188,698,468,956,837,537,927,507,12,602,374,636,619,839,663,766,565,370,516,591,179,908,762,660,16,270,379,724,610,381,699,592,681,37,655,84,387,745,177,125,41,163,296,456,557,447,144,40,635,877,9,384,584,768,355,933,88,558,975,459,105,77,766,940,408,215,930,239,670,679,144,359,851,727,113,763,467,705,53,817,475,35,580,546,691,128,887,965,9,476,403,162,303,831,934,690,92,765,393,670,143,943,477,427,868,337,406,557,453,407,361,513,737,737,76,792,23,245,786,759,818,967,298,838,410,364,480,825,220,721,702,363,942,687,100,206,311,357,679,130,310,32,356,862,595,643,326,465,299,941,920,621,731,3,409,841,990,845,402,575,438,352,271,88,928,709,610,169,82,809,463,669,74,413,747,835,523,168,462,928,547,812,865,930,176,77,494,798,564,394,461,357,306,418,716,215,375,263,472,513,357,605,453,151,707,479,432,760,986,44,925,113,28,456,486,38,226,875,959,729,968,274,293,933,646,582,463,393,3,235,263,919,276,822,647,980,729,142,302,123,896,604,355,853,706,826,221,453,764,257,595,371,866,889,799,845,198,629,169,553,240,971,297,254,400,669,762,247,67,640,272,2,427,956,522,31,134,195,98,234,549,575,541,556,598,738,451,212,57,58,334,73,426,743,914,606,663,993,140,213,483,849,448,751,996,909,46,365,874,90,942,394,900,43,993,456,26,854,326,205,879,161,8,369,729,796,416,116,30,203,969,179,621,763,272,151,958,455,136,224,177,238,841,593,775,726,222,942,616,642,273,597,779,852,786,780,63,544,876,579,610,738,347,93,542,686,861,556,791,764,821,665,473,315,137,837,95,564,89,33,595,986,634,239,453,469,181,260,700,720,511,960,539,265,906,915,505,195,966,909,570,657,677,608,262,506,557,890,703,710,966,631,706,325,297,684,512,966,177,202,370,744,352,913,401,274,716,859,234,14",
    "739,320,949,307,253,64,996,160,119,558,619,479,484,321,880,334,427,267,269,234,111,515,177,93,29,450,652,240,740,526,125,797,70,220,461,643,842,965,902,560,683,523,989,262,111,783,384,431,760,477,756,864,996,381,941,760,608,565,556,148,226,25,139,620,798,551,349,426,461,357,767,179,862,572,2,274,325,574,654,753,174,504,805,57,889,194,665,867,218,684,479,979,112,198,147,372,390,507,170,714,972,414,852,931,658,257,515,524,881,102,346,662,377,943,63,202,802,770,820,240,480,876,942,854,68,289,946,549,892,881,419,896,770,965,713,761,765,989,36,649,547,623,162,359,294,903,680,452,768,56,987,301,365,810,697,91,207,224,45,14,64,670,700,675,237,395,849,559,317,158,849,596,187,114,518,630,364,0,303,988,558,36,348,464,798,298,772,692,284,625,615,545,523,889,279,255,61,805,213,250,364,123,756,719,884,476,820,225,192,944,229,126,46,210,895,124,781,163,868,967,161,28,639,371,53,551,187,658,666,953,167,53,984,724,22,856,916,573,997,187,373,525,984,745,680,509,165,408,362,542,643,433,292,612,452,292,665,491,762,919,265,777,250,690,740,855,399,349,593,170,380,142,861,836,676,492,40,42,167,338,176,427,298,627,598,805,635,699,109,153,260,209,53,586,400,917,23,280,130,841,3,332,768,232,290,688,679,388,573,382,465,118,445,786,108,210,297,303,227,55,833,275,452,104,400,824,372,934,482,367,82,41,29,587,617,748,661,719,130,220,946,237,174,120,485,837,541,510,994,407,863,16,567,582,92,728,905,626,813,138,680,930,5,279,565,968,99,162,152,204,210,840,852,342,780,283,39,866,125,641,306,583,674,584,602,715,260,540,576,131,436,515,211,44,539,410,750,883,911,753,92,244,9,263,860,670,701,133,953,565,228,810,805,636,540,488,590,589,475,963,915,430,845,574,325,554,8,782,5,84,760,746,510,675,555,286,730,428,361,581,453,971,917,545,901,930,103,239,995,955,66,327,165,241,730,609,834,753,91,772,85,479,883,910,84,801,827,12,714,832,855,42,379,663,44,580,603,911,454,340,995,996,812,749,684,882,130,131,536,515,862,495,294,684,914,524,942,34,543,68,667,421,815,625,696,62,232,156,177,126,178,716,10,474,664,889,595,432,408,511,256,446,469,573,157,574,235,528,201,608,766,911,948,203,210,593,156,507,433,406,150,737,593,372,894,689,735,459,621,315,243,313,903,913,874,980,961,575,826,308,600,467,380,242,704,42,665,264,301,989,532,678,339,158,719,30,357,575,496,406,839,808,274,560,639,87,393,762,657,638,464,599,890,102,598,154,947,132,246,678,309,357,74,72,555,559,25,569,521,557,259,551,76,176,281,142,195,828,208,401,240,716,5,300,739,754,233,202,307,471,550,93,764,914,381,989,413,624,987,437,96,611,371,601,479,70,884,129,427,402,605,61,656,750,98,35,442,550,325,341,445,408,32,944,798,170,780,707,245,234,721,629,822,347,282,853,625,543,672,125,213,587,928,223,105,452,412,937,90,54,49,411,835,247,365,823,842,394,313,985,386,824,278,331,806,774,659,730,176,421,956,857,460,410,646,324,857,650,931,881,699,517,255,323,849,758,948,345,839,172,766,430,64,992,234,239,173,584,958,577,229,797,175,800,38,94,832,970,224,47,179,722,384,227,355,627,544,105,977,98,789,663,602,612,580,374,696,962,288,879,580,725,816,801,979,320,671,905,851,307,425,966,853,532,516,679,107,277,637,776,722,325,485,592,87,458,69,174,661,651,380,302,8,441,428,305,774,650,410,370,794,500,369,841,632,50,357,82,420,40,268,936,512,2,598,364,890,79,355,15,117,297,837,684,868,88,50,491,357,472,208,104,445,400,169,948,230,312,315,205,509,749,230,686,426,103,28,60,612,147,42,405,206,831,554,84,406,511,307,512,396,495,924,145,384,155,767,26,180,921,413,327,907,650,785,890,268,72,275,877,754,445,76,420,11,474,911,296,357,893,927,760,846,298,973,828,481,595,114,721,59,477,659,92,485,101,475,425,13,589,91,974,130,308,436,906,427,392,376,509,691,986,792,328,636,135,102,141,489,164,959,813,575,317,720,49,827,275,892,32,161,774,631,737,427,607,119,210,837,196,661,645,806,367,882,144,127,8,401,341,422,398,671,765,682,854,315,401,82,781,413,166,408,269,205,165,266,883,234,742,525,89,386,313,918,211,651,179,553,351",
    "354,949,7,10,495,237,572,320,143,811,149,868,126,737,88,404,63,449,390,473,13,932,771,823,316,758,173,570,721,254,76,323,857,503,339,400,303,498,26,899,342,495,904,138,255,740,917,407,742,72,521,227,166,923,39,658,362,201,424,382,268,583,538,809,373,109,22,727,346,786,38,13,55,325,751,238,101,89,710,30,962,491,568,468,210,254,708,129,520,795,58,432,48,764,3,829,337,399,723,302,844,329,70,551,804,94,503,257,200,396,517,952,231,711,720,853,756,757,534,112,636,11,233,834,455,670,186,821,862,227,65,395,487,900,555,877,457,394,957,623,771,683,264,613,533,706,931,327,937,512,978,101,902,759,49,306,769,437,340,777,128,290,769,357,721,365,756,529,834,501,521,499,756,287,99,883,178,303,0,912,598,682,64,985,494,741,660,691,380,67,613,191,415,173,68,138,744,329,649,482,596,249,296,226,667,780,713,140,22,683,280,140,723,331,799,702,180,751,672,720,448,93,15,711,456,453,20,253,361,442,470,110,464,571,159,255,497,477,133,59,169,212,935,628,521,710,308,576,903,106,282,484,494,772,204,567,876,132,540,740,106,912,645,42,714,193,671,765,34,467,329,56,577,646,836,541,58,688,99,268,989,93,75,85,597,738,89,450,670,520,893,756,131,104,364,290,44,666,708,200,457,214,263,771,750,197,115,272,280,818,504,522,113,359,26,229,845,389,450,529,830,770,425,251,726,498,187,876,745,857,837,370,362,753,654,232,688,206,849,757,784,864,326,886,431,938,898,943,386,156,595,109,46,396,750,388,446,57,266,2,913,56,886,44,807,750,709,259,902,714,589,578,414,55,131,701,209,570,976,808,237,897,603,392,115,73,824,967,922,852,407,111,35,229,258,876,349,534,802,74,685,500,147,45,837,659,882,814,462,34,695,87,597,440,609,315,452,732,310,374,815,945,788,999,242,317,886,107,510,729,992,180,458,645,876,131,430,903,205,817,863,113,925,697,922,963,757,320,138,741,54,875,78,456,677,740,149,129,48,707,60,451,9,542,287,602,127,52,965,663,298,521,742,378,412,549,836,700,570,980,162,170,190,907,60,481,649,514,234,971,232,292,302,301,260,425,482,754,817,629,944,100,340,225,100,811,633,125,4,952,746,643,75,800,378,802,715,537,445,31,350,792,377,110,891,368,278,197,369,345,963,258,380,925,381,45,192,660,742,367,488,222,240,388,438,978,877,352,134,267,116,164,938,755,128,401,562,678,229,360,648,668,351,752,102,765,165,246,118,488,417,754,52,736,312,735,802,723,854,84,5,112,650,905,674,577,75,146,416,960,76,642,262,571,901,44,964,361,410,339,943,365,938,500,194,484,65,142,161,268,26,397,750,786,233,888,570,210,439,435,224,574,880,623,970,75,805,205,395,735,999,207,515,314,240,136,429,385,341,80,761,667,530,836,764,820,381,155,282,118,714,733,282,937,460,670,838,630,296,645,826,90,410,681,366,261,418,753,667,928,759,833,724,378,715,874,612,963,40,577,690,417,732,749,307,708,523,108,560,478,513,800,40,91,206,773,718,120,301,753,262,463,460,989,302,540,920,34,593,60,960,55,124,231,87,546,33,931,894,499,585,905,509,782,645,651,390,341,405,228,748,91,880,613,471,374,845,409,194,514,575,77,354,264,698,319,94,520,69,633,898,451,43,901,379,733,812,325,982,299,692,220,395,495,866,423,157,403,535,732,642,647,739,295,902,957,746,790,7,94,53,181,503,614,280,661,446,177,276,319,817,906,822,841,201,786,675,496,242,643,908,537,973,598,470,364,830,309,17,398,523,601,495,395,284,426,861,881,299,903,320,555,155,551,126,16,539,118,666,228,699,920,360,208,477,529,652,50,943,570,55,737,958,907,381,411,410,838,484,157,140,471,276,333,438,707,217,396,227,845,832,626,949,925,41,744,439,83,422,234,763,338,394,534,306,334,71,491,597,752,906,932,455,639,34,354,325,632,596,557,457,201,58,445,739,937,433,957,55,131,334,67,922,702,409,29,611,714,984,929,669,427,60,504,33,16,812,260,6,25,997,451,819,73,182,974,51,451,319,959,768,25,770,67,986,405,747,71,914,485,39,982,273,643,46,302,519,317,692,346,111,376,293,413,852,21,375,552,78,143,529,213,92,65,762,974,192,307,765,959,472,684,383,437,363,316,423,573,704,284,86,903,695,217,380,65,443,133,333,87,235,775,710,120,882,332,15,792,108,778",
    "781,838,624,286,230,962,242,323,613,755,349,250,75,389,572,32,355,284,14,509,262,947,849,385,395,133,597,381,465,418,297,524,294,944,554,967,353,993,987,512,779,36,902,243,8,139,802,672,832,985,339,716,423,707,985,103,182,144,880,192,525,574,694,530,9,906,348,147,664,318,133,371,677,490,28,50,89,992,877,938,871,180,698,740,56,494,316,117,857,812,248,906,696,574,394,872,196,279,867,690,721,18,416,667,839,72,174,677,56,728,334,979,24,872,597,20,218,838,359,976,243,218,522,288,513,590,616,916,757,739,210,844,105,910,521,610,359,683,973,710,499,628,390,338,117,14,409,602,271,407,305,866,42,253,163,686,157,338,58,431,160,743,644,661,11,984,294,524,745,181,462,172,453,422,450,31,759,988,912,0,106,562,127,217,657,146,14,522,270,969,432,851,424,743,145,844,944,421,79,399,506,216,259,799,739,540,260,630,585,563,679,614,757,545,128,285,301,43,414,501,767,353,883,698,652,653,511,150,137,527,288,479,367,810,116,678,634,673,48,84,651,236,516,722,771,824,745,998,45,82,681,787,876,949,311,719,451,902,22,586,744,58,37,954,604,645,32,919,96,367,152,116,228,624,221,369,504,166,615,737,473,857,816,345,370,761,543,916,64,392,225,553,631,773,998,298,722,523,267,32,957,499,961,20,239,39,942,925,14,481,129,914,139,262,260,920,433,376,456,622,79,465,15,55,543,214,647,408,128,692,746,234,475,347,409,727,65,455,765,713,863,100,580,897,855,925,971,866,322,316,372,586,761,908,127,262,603,19,310,139,599,148,681,338,967,612,594,92,728,826,919,822,376,396,613,617,74,805,338,245,344,621,742,725,51,627,583,416,97,575,810,299,135,667,507,807,274,409,897,295,796,836,1000,85,155,144,339,833,147,189,43,666,425,88,600,294,71,63,44,720,632,325,192,390,33,428,19,880,512,217,143,785,305,766,125,332,200,844,839,470,31,441,68,805,301,379,954,468,704,128,759,399,174,393,337,203,114,641,290,783,902,703,192,192,523,210,24,86,688,808,601,851,133,615,974,237,278,732,160,687,104,938,38,757,803,377,139,338,229,160,980,184,41,414,264,673,866,838,127,802,614,994,432,800,466,783,220,549,933,25,29,204,462,384,396,460,338,751,289,406,297,53,281,185,544,715,122,392,42,533,366,962,230,484,943,148,598,342,212,449,469,374,618,554,561,907,530,24,80,273,200,382,585,251,754,150,616,136,685,287,282,873,593,816,353,380,226,186,525,495,279,984,176,474,492,970,334,412,237,834,324,609,158,195,328,621,398,548,246,779,320,649,185,517,54,640,799,372,119,798,457,867,672,380,757,941,373,311,188,485,232,150,667,886,358,688,880,257,155,613,432,200,703,904,335,577,446,727,826,469,587,701,416,88,427,902,760,908,821,291,868,187,206,591,870,188,519,583,899,650,388,536,527,836,878,287,945,175,628,649,215,620,539,122,844,885,292,685,463,531,906,685,595,796,183,786,145,190,836,244,594,758,625,714,352,565,648,497,150,453,55,669,391,793,105,165,809,371,10,929,195,913,586,613,512,248,554,789,152,878,933,2,585,766,200,358,715,77,925,756,3,20,12,373,286,308,272,688,390,635,484,15,590,183,678,447,459,217,945,309,29,361,426,78,401,210,932,455,886,682,681,71,703,182,82,381,958,329,11,886,764,297,393,611,525,324,207,840,944,666,910,240,165,509,488,807,734,582,700,981,673,783,174,530,894,9,465,521,296,892,352,532,728,770,816,65,786,659,612,526,278,707,582,817,836,815,407,423,477,74,262,790,860,302,805,522,937,725,951,72,274,317,293,314,456,863,435,998,294,418,781,28,856,196,658,444,220,681,542,464,459,854,863,372,827,782,491,161,582,420,342,293,460,306,209,39,611,90,876,715,553,634,871,859,810,451,658,572,856,181,717,861,828,213,834,636,322,87,811,533,116,752,851,307,353,654,686,119,741,631,177,182,547,316,423,954,932,721,26,100,173,306,571,452,981,996,123,382,422,651,518,813,966,688,299,360,43,552,912,475,499,786,739,208,453,383,259,730,445,699,129,143,812,784,59,829,914,902,740,773,161,79,574,775,3,452,648,852,11,413,190,341,84,104,859,42,146,698,637,927,355,229,721,307,479,473,936,226,334,35,509,603,651,126,651,952,146,527,326,692,668,248,184,561,759,314,678,322,15,558,882,104,594,764,695,786,106,437,110,821",
    "348,138,684,188,683,445,143,721,371,44,797,430,351,657,621,807,4,739,386,500,131,918,512,789,869,998,564,892,292,458,277,261,457,673,759,963,94,232,938,614,77,515,532,536,587,210,35,150,63,107,160,241,337,62,247,810,763,570,303,489,553,92,598,92,777,119,725,411,734,222,247,195,445,865,237,243,954,311,190,179,920,389,849,433,275,753,259,46,173,328,592,273,8,525,669,638,195,468,393,133,31,69,308,280,625,439,260,404,44,203,103,283,811,23,799,955,291,776,272,466,925,770,425,732,461,171,651,309,817,163,704,955,485,777,314,358,181,743,629,328,157,936,518,448,450,994,881,228,421,853,366,110,957,145,257,486,751,869,394,376,295,524,517,971,563,64,801,205,465,734,606,956,31,638,661,969,124,558,598,106,0,5,861,288,818,753,486,750,353,196,39,471,428,85,877,502,609,986,413,746,32,18,925,483,750,382,760,305,764,578,593,852,54,934,642,715,287,958,684,165,161,601,222,607,823,132,484,656,378,31,848,43,42,300,466,259,953,595,775,337,169,355,651,826,77,430,663,293,392,952,525,450,560,595,664,872,495,833,667,130,111,936,869,403,402,771,431,7,481,76,894,691,712,226,533,58,793,447,193,227,229,514,879,659,357,770,45,100,500,311,745,804,507,793,721,162,779,663,186,162,981,642,289,445,227,617,634,106,409,651,150,87,193,961,138,352,227,542,958,855,382,606,29,502,709,619,530,66,350,172,965,580,187,708,459,745,680,74,482,770,442,590,992,873,767,950,116,140,562,773,750,700,746,746,280,300,986,707,732,115,392,408,276,535,292,218,388,331,273,923,570,69,299,122,511,583,491,919,46,47,814,66,94,357,319,261,140,498,3,655,44,829,461,208,426,380,749,270,220,386,687,88,34,762,170,389,481,394,234,633,275,8,793,519,85,502,810,549,20,448,47,300,945,652,34,790,211,610,763,437,572,779,268,608,422,594,56,66,940,951,955,377,873,443,762,824,768,429,62,621,308,411,414,380,307,144,611,681,638,904,725,516,503,590,354,94,140,434,790,528,917,212,304,12,319,328,557,28,377,622,620,648,605,106,146,1000,842,555,52,955,937,244,405,19,398,253,341,344,156,613,461,486,333,80,982,262,464,458,671,707,412,779,67,81,361,481,286,650,691,145,455,394,40,183,544,898,377,452,706,756,385,152,997,879,719,109,261,794,788,378,909,952,700,90,889,943,617,145,773,370,714,3,628,552,272,266,433,877,168,475,248,379,590,967,647,206,932,238,289,435,754,964,634,15,486,786,752,36,733,191,226,722,922,540,350,591,384,375,13,911,850,340,184,80,745,518,889,280,462,773,179,755,997,187,962,725,488,463,990,14,631,623,366,211,936,717,839,983,432,680,758,886,385,517,371,401,853,152,292,227,683,184,144,704,532,215,177,420,673,819,858,938,607,410,359,245,43,801,911,646,822,179,699,805,409,54,301,959,188,657,655,114,959,845,873,125,371,68,723,516,23,405,438,762,132,818,57,397,488,135,112,729,596,303,836,741,562,220,929,823,740,296,933,50,517,195,963,15,109,62,944,110,889,786,428,753,94,258,417,92,529,89,541,48,160,997,410,400,455,764,198,726,406,776,157,95,842,9,894,563,675,779,100,633,379,262,18,706,138,668,2,80,308,841,964,420,11,4,479,967,534,571,198,864,169,575,142,645,744,597,936,15,555,604,578,928,295,396,368,332,195,746,908,294,538,613,350,976,863,553,679,109,869,308,196,862,345,137,646,352,843,76,331,848,503,384,614,162,31,305,574,779,695,672,845,136,232,477,761,977,998,323,444,899,657,224,928,780,833,172,391,677,454,213,409,52,376,936,175,563,565,40,787,339,821,851,829,482,191,553,210,19,3,524,728,84,271,474,98,565,161,302,256,986,996,398,765,864,957,360,633,460,918,765,556,665,263,742,403,96,732,77,694,770,186,789,252,419,582,422,783,154,283,564,503,930,482,594,101,785,5,17,854,353,479,391,32,181,681,125,975,823,828,49,337,195,76,991,640,458,468,283,882,978,788,689,48,24,458,928,718,971,373,532,608,384,88,181,803,828,776,447,152,229,142,181,875,366,144,627,518,84,191,353,426,312,570,917,988,629,502,325,616,924,841,571,297,297,643,43,763,342,982,763,504,776,316,997,349,604,532,633,271,900,777,397,894,938,509,767,872,226,798,477,519,694,52,973,433,62,157,681,542,306,150,401,81,348,963,564",
    "264,537,720,982,634,63,962,619,209,622,664,681,749,292,310,695,959,370,356,98,952,333,961,82,855,168,569,397,738,601,112,853,427,521,15,289,104,544,346,280,797,480,790,853,945,637,346,882,553,594,248,107,865,263,402,317,231,386,925,690,79,284,816,899,505,971,181,881,849,89,424,351,269,211,535,292,729,25,206,198,924,812,983,668,677,902,215,60,744,943,267,525,199,785,788,387,80,787,854,453,581,215,79,508,98,376,690,969,335,532,433,223,175,222,966,750,188,417,579,404,821,410,327,770,431,301,605,857,629,849,677,154,414,879,580,2,210,672,867,331,985,373,134,841,885,170,424,927,88,999,629,18,987,88,584,501,641,778,453,661,658,607,31,157,543,530,691,598,29,532,892,394,960,408,446,86,142,36,682,562,5,0,538,268,402,686,635,959,180,317,426,496,519,111,239,939,455,273,396,95,178,835,436,228,788,549,655,404,116,486,953,306,301,286,948,954,486,678,584,602,446,338,379,560,402,301,224,831,196,57,686,511,436,511,523,22,268,115,243,928,610,440,488,999,192,639,600,836,283,291,10,913,161,232,864,155,284,680,810,873,676,949,578,129,628,131,726,950,309,693,992,646,936,346,41,617,808,26,559,750,205,292,15,459,938,89,133,965,946,395,726,503,789,628,829,316,661,334,201,507,533,344,690,405,943,846,18,779,105,307,298,408,710,790,247,93,261,392,984,533,171,259,583,781,236,338,327,806,903,169,376,514,690,193,589,784,345,57,522,265,888,763,828,686,209,135,428,922,481,957,939,913,711,3,842,965,975,357,769,597,388,294,492,944,834,465,658,861,928,698,175,145,94,538,433,277,850,246,271,35,26,838,442,272,884,322,404,894,845,267,592,471,795,438,60,248,326,263,769,658,241,872,808,219,339,985,240,335,812,552,383,501,182,460,321,680,725,693,850,438,458,768,732,608,626,693,930,510,22,858,444,979,567,786,125,898,506,258,688,149,412,336,631,746,906,855,228,987,312,515,562,369,623,859,172,351,747,379,263,152,594,643,116,816,840,806,320,328,254,497,600,501,379,589,774,343,319,733,595,125,120,483,980,797,997,713,125,744,574,255,766,598,226,926,244,705,704,790,406,905,761,380,340,850,654,880,309,972,965,166,748,883,347,616,547,60,980,760,304,748,364,541,581,445,339,475,756,32,140,989,733,185,352,600,10,574,565,858,225,40,542,448,114,565,846,997,329,76,555,418,561,916,754,139,577,823,55,261,518,176,698,67,596,583,623,356,232,816,539,926,922,449,999,59,814,356,316,792,397,830,717,969,482,48,906,284,289,169,261,776,878,223,354,893,719,90,401,350,13,32,700,521,421,161,817,869,496,101,649,395,71,723,874,29,573,995,304,704,201,475,745,678,688,138,771,37,440,341,522,438,152,682,903,707,266,907,924,866,94,352,464,643,773,65,43,608,189,487,632,24,530,888,142,220,579,38,128,839,405,256,810,729,211,332,246,952,969,318,762,647,923,34,11,55,4,978,638,317,224,555,443,933,56,757,825,735,192,872,404,199,405,429,689,510,694,83,85,21,676,498,1,17,987,354,474,503,588,298,648,747,546,553,593,571,531,949,788,303,322,361,658,186,647,597,430,750,163,212,453,249,593,35,555,569,588,609,891,530,481,819,460,564,883,806,792,561,554,54,786,779,246,594,493,628,306,287,281,497,696,845,442,164,90,206,879,733,377,670,688,429,282,499,485,279,42,498,184,112,563,977,201,905,597,306,594,62,738,919,192,265,358,430,835,620,845,21,671,945,357,850,349,812,412,306,494,789,977,291,710,947,310,381,74,202,944,300,259,885,617,149,69,277,874,762,160,265,576,24,590,233,318,574,551,866,658,206,378,685,515,432,907,740,870,87,16,148,820,958,12,818,912,991,5,482,357,408,555,908,590,424,129,392,587,822,575,116,366,133,605,221,770,78,292,879,96,821,558,893,854,804,285,41,540,450,226,38,591,386,945,808,946,959,132,767,57,310,489,774,855,88,673,477,455,231,854,713,353,510,991,469,313,161,990,425,97,612,743,159,52,685,839,288,501,564,785,381,475,199,452,437,173,382,53,279,961,373,873,35,389,482,720,549,105,13,690,830,352,801,307,195,611,395,969,891,191,966,497,236,131,605,964,837,430,401,607,985,641,921,913,788,258,828,615,113,364,501,966,302,415,548,601,883,61,924,646,982,918,87,142,258,520,547,865,317,87,157,365,816,129,492",
    "790,868,651,970,851,561,133,998,57,15,629,359,262,152,501,306,680,539,822,256,753,279,512,631,391,881,493,127,621,31,630,233,921,927,972,201,138,915,4,499,641,256,490,73,438,573,274,260,889,931,53,384,421,603,735,494,768,319,880,979,92,724,877,632,355,551,447,9,567,606,574,538,352,43,216,579,905,43,648,93,287,751,632,537,997,734,426,326,99,911,239,197,442,648,491,978,408,24,214,786,471,876,750,696,62,506,410,427,491,242,695,473,837,774,938,2,686,138,849,89,452,850,491,810,741,885,524,341,932,609,102,42,103,277,812,569,331,503,407,450,167,997,218,518,455,385,797,942,878,745,305,988,153,244,100,326,243,726,654,671,75,682,358,642,668,538,618,208,91,787,259,154,430,943,272,135,113,348,64,127,861,538,0,558,215,949,756,139,242,491,941,745,315,424,370,260,34,466,693,562,920,490,707,239,94,565,245,730,369,414,810,334,418,189,194,199,709,818,299,731,701,799,806,741,3,224,629,65,220,203,756,260,688,461,508,319,641,88,819,922,98,52,763,108,883,819,906,626,803,263,13,795,596,438,491,432,377,121,131,516,328,324,294,753,679,490,366,874,742,977,185,51,212,739,36,660,718,387,459,845,780,306,555,278,585,452,129,442,935,157,329,783,499,249,600,514,322,62,214,462,712,719,464,376,758,825,46,783,675,334,494,983,409,498,983,507,824,326,170,235,194,576,742,983,197,177,645,776,625,704,107,374,170,341,575,887,384,248,181,601,808,109,790,671,791,404,178,750,409,517,429,158,444,129,194,308,579,49,704,274,712,209,251,93,518,57,597,847,820,580,290,473,712,78,119,595,13,139,920,770,894,158,671,143,601,668,199,619,295,533,757,218,585,328,969,772,469,871,384,341,32,620,903,793,975,696,568,658,75,634,263,854,493,956,348,566,835,529,977,366,13,972,504,842,379,634,475,29,657,760,480,316,508,409,897,36,378,934,216,90,648,194,899,787,852,170,479,554,546,689,495,765,31,963,702,745,689,37,674,580,532,170,43,62,573,659,916,36,121,53,619,154,692,564,151,932,962,226,904,79,312,678,659,286,47,511,256,467,898,141,347,30,611,386,527,988,575,875,602,937,644,649,298,529,888,329,512,861,629,370,181,774,244,514,886,605,828,694,139,414,327,770,120,282,80,408,219,706,819,180,849,252,131,319,528,158,64,288,217,168,552,474,190,646,270,990,883,754,131,246,458,368,772,190,906,504,209,525,478,863,451,691,679,595,929,173,840,984,900,496,994,536,647,952,971,410,25,812,874,471,599,559,344,741,640,88,845,45,266,528,721,791,417,451,623,560,330,958,716,28,565,363,712,95,647,199,866,908,318,345,227,774,949,781,366,70,908,561,741,995,834,360,911,242,843,540,127,333,819,611,15,328,616,756,498,551,227,448,123,677,672,881,160,220,916,462,204,395,470,189,775,298,179,957,834,49,993,74,68,327,429,82,200,937,479,344,270,69,421,361,111,339,5,428,404,910,36,806,321,827,362,424,341,853,209,200,205,132,431,828,573,860,256,738,917,970,986,586,304,983,608,57,388,32,38,130,758,475,587,74,258,870,2,719,306,463,41,904,746,801,299,375,62,897,935,889,668,415,528,609,302,321,846,677,805,203,799,99,378,479,597,723,519,265,360,474,293,875,694,893,291,577,154,520,990,744,495,699,319,509,31,849,933,767,417,908,668,401,440,552,615,123,940,658,742,635,817,97,405,424,667,829,861,423,312,852,903,902,258,835,6,329,25,769,938,998,133,168,41,238,347,438,834,940,953,549,633,715,880,214,306,820,304,633,311,698,16,989,259,393,913,547,453,129,438,876,503,263,664,76,783,447,589,340,198,392,328,150,139,957,365,251,341,619,804,941,263,298,224,332,282,567,205,109,532,8,20,837,106,733,731,456,852,961,302,780,858,242,15,119,727,666,375,690,380,791,304,376,675,217,835,188,370,889,683,749,625,271,141,971,297,998,781,246,336,884,139,828,851,22,140,852,402,119,776,982,314,864,841,231,863,201,951,26,448,980,312,483,951,823,280,348,29,499,953,928,757,719,630,300,794,238,977,410,477,715,246,237,294,502,619,982,541,352,944,127,339,576,980,569,481,951,936,880,809,671,855,788,638,307,795,764,556,163,423,329,828,647,864,849,38,5,384,446,204,752,571,521,682,748,653,874,695,561,2,95,612,275,669,189,741,65,301,310,618,81,559,561,796,48",
    "110,329,556,237,906,882,390,763,89,820,295,356,897,794,485,393,858,909,356,952,416,153,499,112,87,753,735,687,329,30,676,312,629,883,424,245,267,394,855,888,377,490,441,139,397,876,146,972,431,776,564,700,229,346,473,859,465,237,353,340,153,62,166,955,907,615,115,551,117,327,848,380,529,503,466,901,724,424,798,873,268,557,539,871,620,945,890,585,23,114,469,933,159,688,345,837,640,639,457,485,42,64,280,324,216,370,462,451,488,225,20,992,611,966,465,969,71,482,315,20,663,386,916,400,761,50,796,305,248,825,655,432,583,924,375,171,412,105,696,158,397,298,552,710,695,135,629,746,260,246,615,780,835,960,28,737,264,440,825,42,300,903,480,47,942,355,48,347,346,186,528,879,395,716,803,935,577,464,985,217,288,268,558,0,562,271,962,559,52,410,774,598,786,929,25,953,543,779,920,983,91,859,615,317,308,839,491,758,67,527,464,700,994,865,935,448,875,625,777,88,369,559,213,771,229,947,837,303,229,161,440,665,755,114,433,752,146,395,902,468,969,967,166,428,747,303,223,648,155,681,150,598,252,541,395,758,394,770,659,561,617,958,127,228,96,732,881,48,63,975,862,938,643,379,24,617,814,285,657,909,757,566,77,528,888,849,126,225,522,101,732,946,514,126,653,921,388,704,820,887,366,567,226,690,81,270,894,732,130,589,482,931,431,661,930,692,205,520,367,397,35,984,972,99,589,706,65,894,7,53,455,171,819,488,478,875,825,648,65,664,526,943,118,666,133,255,495,175,611,430,279,333,318,286,410,138,912,773,744,434,555,862,579,112,163,438,426,13,778,598,727,459,263,687,345,152,592,550,666,551,996,694,802,551,616,477,305,537,681,790,354,793,718,58,457,312,192,919,890,327,840,770,842,826,959,282,44,963,931,845,192,792,479,891,286,32,506,486,565,832,32,161,590,388,854,67,220,535,254,721,968,514,961,893,886,598,4,795,34,665,243,866,457,517,70,630,651,281,868,237,628,46,209,324,976,400,213,833,496,942,80,340,26,550,566,834,782,502,99,629,75,402,69,156,837,575,600,648,778,349,430,981,80,129,176,490,337,870,556,956,725,675,307,549,334,877,470,307,740,281,252,494,175,31,434,507,191,139,633,56,367,257,398,940,35,169,953,399,305,937,919,132,379,890,720,310,711,465,932,34,407,992,576,13,689,18,596,784,89,214,704,377,553,53,159,573,945,123,159,10,929,190,300,554,984,893,30,803,279,908,604,391,691,577,594,438,849,121,375,539,315,395,848,979,715,490,96,828,876,80,475,102,432,900,224,719,309,538,52,11,304,459,279,62,101,323,740,470,369,951,585,61,852,707,355,601,752,249,418,26,467,855,725,872,342,716,785,997,897,508,471,359,736,660,923,354,355,779,275,332,144,390,430,285,798,720,609,646,456,597,366,939,944,58,387,870,124,50,367,495,296,262,495,738,262,621,109,697,563,48,12,319,110,147,389,281,392,988,386,490,219,237,6,70,80,951,181,390,999,401,522,842,261,957,488,802,721,827,106,289,160,192,710,388,465,744,918,908,31,167,393,775,531,820,428,238,477,819,999,147,967,171,898,566,537,259,981,610,837,351,434,6,436,914,729,518,983,505,581,844,191,397,168,20,345,813,567,416,260,287,235,634,882,977,392,268,416,87,445,271,221,655,321,156,814,177,356,762,114,443,417,63,73,14,183,397,958,776,236,175,361,920,972,378,193,259,242,125,982,900,952,798,155,9,269,539,677,609,456,586,715,396,581,70,799,931,164,317,406,8,879,681,682,157,32,784,220,974,758,60,390,553,286,341,361,121,727,920,460,662,266,756,769,326,961,678,584,477,44,623,814,725,587,976,198,935,393,6,553,454,33,941,751,385,202,95,954,49,752,106,550,390,835,387,923,677,224,524,826,899,39,430,397,486,814,139,887,736,145,966,746,581,661,107,267,536,443,127,958,245,856,689,778,549,616,45,13,248,658,760,386,625,98,551,71,78,327,46,193,402,493,981,662,579,556,935,149,158,86,896,313,572,263,13,446,869,654,98,119,874,54,965,366,979,334,494,887,932,33,686,310,291,378,426,144,281,233,692,473,966,44,878,765,254,344,487,990,560,28,386,514,566,91,613,299,940,790,761,382,524,891,615,832,825,485,923,398,989,429,880,460,696,843,664,749,303,980,466,26,836,131,783,901,49,182,676,444,431,864,715,599,188,517,445,580,181,667,885,201,344",
    "369,759,534,614,983,980,664,152,326,55,188,818,271,800,382,134,500,249,314,626,386,918,803,82,92,17,47,584,389,194,411,896,460,108,693,748,313,226,911,820,304,731,876,711,840,580,663,854,831,219,413,748,167,559,505,828,654,197,166,702,310,836,953,22,641,943,579,109,71,208,156,110,786,170,820,264,335,808,680,516,836,711,888,123,403,16,466,627,415,303,390,675,809,453,360,414,7,998,716,727,103,284,293,775,899,277,181,765,40,695,985,84,299,498,981,132,504,341,467,316,618,502,238,110,283,794,285,937,62,343,909,318,511,29,901,23,510,304,554,288,72,217,248,318,637,330,172,922,218,634,791,136,218,928,700,896,63,615,718,54,918,696,272,474,938,762,993,178,233,528,798,320,450,720,836,702,614,798,494,657,818,402,215,562,0,535,598,471,828,895,121,62,944,171,388,87,996,263,755,459,175,987,144,911,536,876,58,208,549,372,840,188,912,729,160,986,904,70,632,345,393,549,911,224,7,830,704,437,609,489,221,120,24,355,92,482,177,82,544,7,396,692,732,686,96,963,85,104,228,720,55,519,34,891,887,213,822,804,494,137,786,768,292,639,27,42,261,295,772,195,183,991,951,486,792,392,957,397,253,448,239,859,915,176,5,177,766,404,551,337,455,86,201,986,948,631,63,934,631,940,408,467,986,565,726,622,633,737,107,91,783,236,166,632,28,899,965,710,198,638,67,144,686,58,631,584,821,805,147,114,400,471,338,430,536,122,868,491,332,298,234,231,193,769,514,462,352,28,550,675,216,776,763,795,982,762,637,677,890,29,380,813,820,470,89,988,296,61,719,750,288,920,426,53,270,213,604,335,978,760,918,187,523,19,269,817,556,716,640,834,952,931,498,200,258,146,674,49,839,181,130,894,944,942,200,490,282,791,447,594,143,219,270,562,337,821,794,618,902,828,117,899,212,858,530,664,766,96,440,810,521,828,167,787,434,12,420,538,99,952,437,468,991,391,206,346,917,3,724,22,475,195,31,602,423,51,846,249,333,447,779,332,625,372,33,168,189,932,383,782,523,728,517,774,274,529,540,636,861,87,121,73,59,864,19,758,307,385,903,845,360,228,14,262,450,332,945,921,527,406,895,379,641,711,403,741,500,716,422,800,619,207,411,602,823,814,960,579,239,709,515,198,947,218,447,350,585,700,516,734,967,100,558,437,410,989,120,322,968,902,406,553,644,400,842,19,274,77,539,104,430,327,831,313,203,131,733,604,124,433,866,131,499,318,629,746,595,271,602,14,963,759,481,270,708,657,766,142,701,764,379,81,39,348,155,2,23,494,663,673,377,597,175,42,89,543,69,537,609,146,462,788,591,224,909,37,482,606,199,816,129,542,907,716,166,790,19,584,504,793,938,797,298,475,309,762,988,688,752,317,180,377,952,100,692,89,549,261,370,309,926,721,908,271,842,571,780,853,785,263,55,624,334,205,819,419,365,363,135,487,799,200,908,658,840,483,540,433,56,22,698,93,925,477,354,180,295,197,668,788,988,966,140,759,932,539,843,124,258,863,195,459,592,950,752,137,805,699,678,307,730,304,119,955,682,455,971,239,898,276,947,447,726,818,922,27,42,918,588,415,359,267,576,827,968,889,849,609,778,926,290,139,681,494,544,198,385,775,331,740,12,257,217,242,635,984,162,304,63,594,848,154,803,579,708,102,133,858,737,87,365,856,157,220,317,129,884,672,931,863,243,858,144,887,954,585,40,669,521,520,289,447,292,137,122,337,308,492,256,923,43,395,827,85,995,762,288,649,681,252,798,264,186,95,394,54,536,203,397,144,326,64,421,15,196,772,709,171,477,590,675,764,112,334,215,843,927,608,538,901,199,752,413,99,236,55,277,319,930,953,571,755,69,56,249,71,311,712,789,549,78,66,655,797,816,219,789,895,272,900,736,284,265,93,595,969,589,860,68,29,626,886,357,315,698,198,445,602,307,414,322,67,455,860,317,681,318,931,367,768,132,697,430,632,91,651,606,189,97,460,242,335,397,83,413,279,719,44,404,801,678,266,966,175,243,989,889,604,554,91,889,117,357,298,76,691,175,304,882,257,677,217,461,472,769,867,447,520,411,738,602,457,831,167,173,67,77,839,272,229,607,441,470,568,443,896,840,791,706,663,48,214,245,829,918,391,913,18,211,154,495,187,829,510,615,671,572,476,721,984,113,818,665,327,268,684,761,619,98,402,884,883,53,446,271,338,912,412,576,967",
    "393,923,661,274,610,211,786,4,539,960,39,66,671,60,890,310,813,310,690,722,865,394,834,320,690,798,144,674,437,582,281,960,704,658,571,22,59,323,287,875,634,879,128,431,935,531,913,416,258,570,965,36,580,956,278,176,392,544,948,940,797,946,738,856,714,450,854,370,591,388,281,637,331,869,190,202,347,728,537,494,430,143,850,164,612,450,665,752,648,641,769,506,964,551,729,164,267,391,798,61,836,75,782,892,343,43,487,76,314,28,737,196,900,384,794,928,450,333,947,923,973,938,671,583,477,639,61,231,206,733,187,759,333,593,153,217,300,896,505,605,398,770,720,50,60,308,589,815,665,19,949,364,772,247,975,792,965,39,297,767,126,5,87,973,669,602,177,66,87,83,798,684,904,780,863,387,291,298,741,146,753,686,949,271,535,0,791,830,20,696,707,593,965,160,184,742,185,631,448,31,489,497,602,669,466,677,276,558,294,991,311,642,395,662,210,841,499,428,406,852,359,908,485,731,917,767,219,874,571,156,595,134,284,897,905,787,800,592,358,405,219,410,308,135,149,389,340,143,240,377,202,31,163,123,214,732,427,777,547,271,851,269,802,358,192,8,528,270,401,791,722,277,81,371,987,667,836,385,976,883,431,972,708,51,95,516,389,998,821,238,461,967,494,743,761,335,606,98,664,639,543,561,725,751,832,720,87,104,150,52,750,609,122,474,983,482,840,908,483,632,444,895,495,637,402,329,958,917,918,710,896,443,265,912,770,381,767,250,906,854,79,64,417,6,216,662,885,144,480,497,542,222,288,246,406,855,758,835,688,81,377,951,445,782,538,260,130,751,818,753,727,971,994,533,696,852,385,600,156,224,402,941,467,508,950,660,254,929,882,275,38,259,137,832,14,651,119,715,934,881,213,939,286,398,446,294,351,383,55,637,263,331,966,860,475,581,610,668,510,336,602,49,153,196,812,914,180,817,217,672,918,76,644,26,440,533,805,358,82,472,917,641,903,522,768,914,402,843,431,85,694,996,684,915,8,991,395,220,345,293,379,375,876,389,998,803,99,515,873,317,848,358,741,391,231,324,102,398,816,955,433,776,68,750,245,557,160,296,308,755,321,656,225,914,376,430,687,576,560,15,823,846,893,269,99,355,613,149,682,175,840,479,231,64,59,764,856,773,609,181,948,895,251,33,541,409,968,391,617,990,70,188,702,20,142,348,157,500,992,473,411,294,4,611,718,791,668,315,578,950,896,635,767,232,152,780,769,170,542,351,491,628,247,220,196,688,524,365,833,244,338,370,359,651,759,696,851,337,590,944,807,39,470,791,528,948,574,327,160,628,891,669,411,839,329,885,151,6,194,286,604,150,400,44,983,860,454,207,636,551,22,754,555,525,437,886,95,202,204,73,902,439,955,285,294,372,237,77,692,798,90,229,194,513,829,367,495,424,320,175,236,231,65,712,120,710,746,185,875,953,939,7,277,693,471,47,549,524,599,351,326,939,928,852,755,488,325,244,194,647,342,226,757,672,699,229,117,702,503,599,527,974,767,73,21,696,221,616,91,243,135,644,534,572,894,758,300,336,678,912,588,486,930,131,608,708,422,585,761,393,286,694,705,56,101,834,436,681,229,181,44,141,125,424,32,656,806,305,516,765,153,194,712,146,501,337,757,671,372,136,271,116,906,250,520,75,537,711,121,207,675,309,437,943,727,300,229,320,809,172,726,186,995,32,656,327,548,586,845,972,452,300,629,356,843,335,466,934,874,191,939,715,464,200,770,953,434,808,486,642,281,975,137,581,25,455,290,10,815,207,557,213,782,445,63,607,782,642,431,281,840,3,937,529,256,54,647,627,696,630,913,205,276,359,591,743,745,878,108,95,137,321,562,492,216,911,809,94,385,594,728,660,737,412,919,372,487,191,139,918,857,357,560,353,331,70,961,903,269,488,106,307,729,639,60,49,152,571,499,636,671,383,839,371,748,431,176,996,125,271,657,66,552,250,506,810,261,327,200,263,982,257,737,267,820,264,19,903,615,855,201,534,148,310,426,628,170,5,239,645,612,788,982,133,759,799,322,720,239,80,446,354,316,196,523,503,150,44,747,824,143,782,246,492,655,93,489,648,411,149,638,56,645,943,919,631,856,623,896,7,987,684,705,701,975,867,68,586,146,770,446,880,903,283,715,519,909,50,832,519,951,409,261,337,240,704,314,323,70,517,161,792,666,236,229,644,165,605,841,498,625,345,265,457,573,444,38,907,647,595",
    "678,496,852,50,966,458,580,591,207,518,786,626,168,22,360,137,281,235,752,544,459,37,395,959,874,608,474,259,837,539,869,693,621,895,53,177,702,943,35,224,237,192,761,760,140,725,105,83,557,402,148,192,653,558,266,9,531,755,910,449,36,834,334,599,13,617,818,873,278,747,389,258,239,879,846,964,158,356,416,542,549,353,691,373,363,642,141,368,232,893,769,149,535,265,677,746,207,74,612,968,728,325,74,46,972,789,252,312,406,737,475,227,477,856,888,808,776,101,560,876,320,263,977,777,841,505,44,948,265,585,128,240,415,849,980,199,795,27,144,519,31,666,940,168,97,99,384,928,514,571,79,207,291,867,535,884,949,965,598,421,960,594,45,966,157,357,619,750,47,696,616,928,544,503,715,391,132,772,660,14,486,635,756,962,598,791,0,74,66,682,746,305,516,405,275,97,495,480,826,275,463,685,977,451,945,325,727,548,233,730,840,157,195,447,412,864,517,577,35,802,751,258,572,853,706,693,609,649,853,480,786,945,625,461,587,743,771,362,722,920,36,654,29,775,705,283,510,895,785,776,448,683,188,164,816,788,335,485,216,864,839,185,857,892,809,9,465,745,58,942,483,598,180,372,177,724,579,990,573,121,986,479,617,309,406,643,646,287,61,685,962,813,476,222,628,543,802,370,276,598,111,857,218,442,605,126,40,249,399,153,80,382,667,749,841,990,251,187,319,85,57,887,597,924,469,213,329,99,151,581,172,682,305,996,373,592,160,222,939,950,408,398,235,117,446,279,563,704,73,424,493,103,989,146,5,281,159,155,963,150,298,588,431,189,743,211,121,762,356,722,283,619,347,484,257,634,666,247,632,368,329,860,818,438,209,899,999,324,799,407,515,100,121,896,85,683,758,428,910,816,424,796,756,819,578,219,311,98,360,434,375,578,490,589,558,902,231,893,540,190,895,294,25,375,537,503,788,691,285,859,621,825,391,404,338,452,630,361,493,78,535,37,461,503,412,274,126,751,187,248,365,46,60,825,961,262,846,15,430,408,453,471,11,955,344,26,447,930,482,612,148,842,902,937,657,681,179,339,629,618,405,295,964,38,245,799,950,57,952,122,560,757,649,342,387,68,84,914,446,710,51,991,113,29,635,334,492,217,141,610,841,213,425,29,556,591,660,710,866,756,52,969,763,694,511,190,13,867,978,22,436,507,351,402,123,749,108,743,839,155,133,164,539,522,827,380,43,712,769,634,242,267,296,959,736,900,437,80,285,3,44,885,737,32,202,122,50,326,935,77,229,698,664,39,684,795,322,634,955,230,421,243,248,196,257,758,383,593,964,45,839,541,388,794,475,478,550,902,202,467,121,316,333,854,597,165,473,947,513,652,821,883,290,34,273,861,501,678,484,397,515,633,857,339,592,957,607,92,794,580,450,277,992,108,964,542,773,247,79,993,953,626,865,502,71,607,121,452,282,844,40,933,360,921,733,125,360,514,29,32,886,692,805,720,19,817,749,626,516,951,873,4,43,432,994,91,28,606,550,806,760,551,431,941,302,409,741,648,69,353,419,12,619,262,720,922,946,639,93,733,918,110,387,716,717,282,835,329,770,672,130,529,507,290,715,5,132,344,99,416,304,462,425,767,694,223,744,988,574,889,13,316,866,475,301,683,949,811,680,460,853,851,761,151,647,46,423,511,377,326,689,298,800,832,718,696,581,530,137,688,311,292,44,28,267,161,272,215,599,990,436,671,917,190,642,420,917,223,794,539,168,606,872,501,104,791,796,202,11,758,976,397,157,183,722,966,222,763,412,811,340,79,469,278,186,401,70,896,761,661,994,894,585,325,149,281,677,230,614,947,205,628,806,461,941,717,951,130,429,782,52,623,249,919,120,571,906,634,695,783,506,163,501,584,831,724,951,26,802,541,656,582,927,72,406,527,819,135,64,771,257,431,173,170,538,594,780,402,195,573,164,661,428,446,875,262,89,87,593,750,906,905,469,198,395,933,546,834,967,377,633,278,3,474,911,806,975,382,865,361,237,931,594,351,7,879,981,964,557,23,289,393,765,538,891,620,99,932,929,434,759,45,47,332,606,52,569,461,646,415,971,329,623,847,143,650,820,109,252,707,841,251,185,733,319,629,259,99,668,4,332,264,25,283,742,949,943,701,286,189,214,657,780,825,328,719,328,839,348,224,755,280,788,405,905,320,769,816,391,399,393,867,596,510,289,986,308,644,695,37,57,244,774,527,825,458,738,799,751,861",
    "873,960,457,808,563,409,895,731,631,850,204,323,21,392,890,511,67,493,244,862,176,590,481,237,361,441,291,204,223,345,973,450,794,336,362,191,604,210,147,866,610,813,219,357,230,977,838,692,70,912,499,145,349,11,668,88,685,652,6,585,523,400,381,254,397,175,700,168,193,153,266,731,492,489,928,752,488,640,997,63,540,974,745,384,813,528,140,436,904,740,288,770,212,665,258,120,443,967,652,312,755,177,622,645,384,913,532,702,508,556,269,358,738,784,640,898,719,30,290,808,446,92,976,133,235,956,138,276,800,489,438,411,582,183,103,180,399,850,881,890,491,540,486,452,714,242,29,695,118,627,988,816,565,134,512,356,217,954,25,659,339,810,384,222,391,739,347,579,641,875,896,300,449,332,373,546,400,692,691,522,750,959,139,559,471,830,74,0,802,974,933,601,728,774,807,532,5,335,476,642,373,556,958,405,332,305,433,957,305,832,598,94,626,303,81,779,778,437,793,709,756,973,486,817,400,349,621,123,85,296,213,788,811,105,506,682,857,974,998,99,636,266,828,971,605,204,563,801,824,799,912,423,447,55,188,112,51,70,32,963,386,826,907,734,274,723,951,584,359,722,777,453,218,738,428,299,489,720,22,671,976,549,661,487,652,734,218,759,315,649,282,944,344,322,769,953,227,516,560,752,904,543,479,61,153,172,180,426,338,634,460,42,151,240,228,964,209,920,8,502,326,548,957,777,375,657,794,794,751,66,44,779,545,870,177,97,458,174,434,953,93,859,269,421,308,894,536,504,62,710,269,726,536,119,406,408,644,10,240,58,731,528,433,103,344,98,459,212,35,837,751,738,820,491,766,702,292,60,641,9,123,676,617,762,878,390,670,451,859,155,654,600,321,947,688,994,38,256,106,990,983,868,794,459,287,328,449,789,743,377,957,412,152,693,494,816,987,136,486,613,903,473,43,234,615,409,483,443,336,206,405,645,25,178,126,921,162,307,318,87,508,819,626,633,446,181,709,315,581,666,747,250,137,834,239,533,940,219,948,176,263,54,349,937,991,345,724,580,563,668,982,256,282,460,701,339,567,195,382,927,449,497,125,81,129,82,947,786,657,556,148,124,157,585,278,480,716,46,537,175,475,136,247,978,261,506,777,757,750,738,264,247,373,961,34,798,294,534,567,547,817,21,523,391,672,137,811,644,532,913,23,540,741,208,715,808,688,798,980,255,115,492,434,96,583,399,814,114,74,881,818,930,7,343,953,446,925,708,97,925,126,886,785,980,908,674,735,139,268,740,30,526,402,288,579,836,118,403,597,9,131,698,193,86,479,594,442,705,844,366,699,644,336,708,693,809,822,34,910,230,182,928,636,716,219,836,76,796,62,480,930,868,993,68,380,819,693,776,212,657,96,127,865,763,631,80,445,659,112,223,947,792,515,439,70,560,766,661,896,172,835,872,935,44,942,831,570,550,906,586,952,170,842,548,255,464,665,594,467,341,878,897,451,408,423,780,755,923,380,327,968,542,8,7,307,604,347,720,105,641,853,5,377,272,926,680,176,772,862,855,708,292,939,691,841,206,545,565,764,488,887,847,947,565,226,366,934,146,673,695,84,354,751,192,309,603,316,387,160,134,307,905,820,767,465,251,464,483,755,243,686,175,31,145,764,10,687,23,594,749,680,896,792,75,762,399,715,757,223,597,570,868,921,257,103,290,245,60,843,899,325,115,168,93,338,97,970,832,407,561,933,66,776,172,675,871,478,612,699,881,877,564,241,538,243,941,81,759,628,64,381,657,994,500,679,181,100,386,616,622,971,879,650,895,827,580,975,458,64,99,780,575,702,250,579,222,387,246,211,62,638,261,799,570,336,12,792,196,29,598,820,432,223,825,152,202,996,94,871,742,39,744,947,713,869,415,751,938,421,386,268,114,284,424,846,885,26,691,479,302,876,571,888,312,329,961,25,330,261,870,222,342,290,985,746,669,432,798,307,82,234,361,407,191,551,418,812,217,894,626,57,860,338,141,395,779,867,523,864,562,14,454,267,640,561,984,285,669,105,844,133,795,415,527,489,664,331,413,979,373,690,32,330,358,886,917,608,725,417,122,337,476,737,746,895,946,732,888,390,333,839,384,182,295,278,301,205,921,231,486,196,894,250,162,442,845,417,646,853,679,961,139,210,388,895,982,848,698,142,591,965,938,222,450,668,181,411,661,321,230,187,751,144,707,191,972,112,147,478,615,128,749,530,570,518,3,605,538,28,884",
    "8,488,259,973,165,222,212,795,658,870,897,817,927,784,774,494,235,166,920,238,926,598,421,653,395,296,454,785,178,759,548,144,186,935,988,699,372,974,787,540,235,356,220,900,181,958,348,305,150,86,851,908,405,60,186,288,534,992,627,550,495,761,613,38,742,900,972,576,20,490,2,610,596,499,660,555,260,123,695,942,215,953,534,869,942,39,782,965,455,539,825,636,464,739,677,430,727,166,850,846,537,848,147,576,791,658,653,994,730,849,354,575,717,804,187,306,290,805,646,554,900,413,785,502,475,980,229,84,233,916,255,132,397,295,440,602,298,142,290,809,153,247,453,496,737,555,33,394,847,691,215,478,568,819,649,238,655,964,2,462,916,640,945,917,591,601,526,729,88,230,539,392,798,618,354,350,508,284,380,270,353,180,242,52,828,20,66,802,0,458,815,870,397,298,404,829,650,961,811,926,795,629,425,313,930,96,875,555,514,747,465,73,744,364,458,565,76,572,207,607,969,379,791,180,825,684,429,423,132,638,404,550,237,761,597,520,648,326,860,391,617,865,614,93,51,454,525,396,310,471,349,385,318,576,591,132,197,38,162,456,830,891,47,632,108,968,297,163,792,717,344,118,746,500,24,389,529,644,313,861,251,916,753,576,365,729,133,773,28,948,789,998,137,66,642,778,351,174,672,265,329,225,987,540,751,318,94,258,870,726,394,988,572,510,106,24,195,859,236,795,977,599,122,857,48,95,336,39,280,538,703,989,196,522,284,681,962,285,281,851,955,462,481,747,661,110,165,979,288,147,538,891,498,742,232,288,900,625,842,56,812,654,967,462,935,826,493,955,399,775,287,686,96,42,860,256,425,561,785,823,191,965,963,604,718,814,290,934,656,25,755,435,35,59,434,617,346,111,597,968,455,417,472,157,30,829,788,226,55,580,575,327,135,94,893,757,226,295,801,934,992,588,45,791,948,949,810,229,498,852,191,709,372,606,762,80,192,634,869,358,926,162,201,470,238,686,748,341,702,278,745,835,950,257,809,357,756,870,931,448,186,464,554,829,359,95,913,661,410,529,586,767,468,438,215,736,582,943,324,286,43,534,907,36,975,258,86,618,670,878,135,978,920,802,281,552,622,303,386,351,332,781,73,438,314,311,72,517,3,499,548,672,285,992,759,859,116,88,868,324,520,614,812,352,777,615,724,477,833,945,587,360,54,582,20,446,967,604,731,390,407,851,205,186,988,8,576,314,520,273,641,285,887,453,793,326,754,69,491,825,120,210,950,469,986,784,762,685,17,373,184,875,839,367,300,661,894,483,873,121,774,655,902,450,938,337,991,380,974,349,547,265,108,184,206,246,230,483,182,22,198,292,841,712,213,884,748,142,474,604,555,484,835,917,672,637,377,626,395,153,162,82,830,879,84,473,97,811,131,272,120,247,583,702,309,950,863,704,803,255,133,154,540,587,133,167,23,854,711,186,341,290,980,60,581,163,494,45,817,571,566,403,529,46,904,941,453,865,632,312,275,326,451,233,737,520,586,820,150,614,759,427,993,521,387,204,9,640,420,158,398,616,649,355,536,311,41,720,24,642,877,136,599,2,978,594,46,949,328,763,19,933,177,10,97,733,139,63,155,881,277,298,981,57,242,307,257,583,151,647,504,620,744,603,927,512,328,404,2,172,974,821,849,736,414,126,767,56,343,264,182,455,423,530,450,875,31,142,234,56,206,707,227,175,191,413,140,937,520,39,505,979,661,699,787,413,220,858,575,998,605,559,484,692,501,737,938,959,299,930,17,279,867,125,754,571,38,484,268,550,797,440,882,756,121,476,56,275,328,12,417,788,14,123,958,405,553,837,116,393,55,134,747,588,890,609,551,426,553,539,751,322,747,44,963,28,551,32,67,689,889,139,474,487,178,959,200,670,968,45,709,778,500,927,256,511,141,853,365,631,759,143,118,877,828,557,64,849,778,644,296,317,662,734,387,851,468,264,351,936,922,623,83,199,621,588,562,872,366,530,494,856,835,250,607,800,821,756,473,521,697,601,847,857,416,4,338,587,240,569,869,97,93,120,54,308,372,661,932,378,271,234,819,399,30,124,36,145,854,527,271,486,655,378,783,874,92,816,351,397,968,219,278,186,29,427,915,822,613,495,701,156,657,459,38,284,485,62,237,593,153,105,906,35,201,647,6,996,613,353,620,735,856,739,114,537,698,420,887,561,802,573,274,834,571,267,886,251,158,212,549,437,420,205,600,743,559,621,892,472",
    "708,452,348,66,217,559,873,633,639,876,506,498,892,791,460,786,220,574,199,666,793,622,25,41,182,783,24,203,899,95,108,74,78,994,98,726,487,650,89,461,262,780,214,742,299,190,152,396,803,371,521,207,622,390,263,419,234,187,541,879,88,37,428,98,600,639,305,551,811,288,101,198,739,278,428,237,824,34,788,565,992,625,667,184,620,998,311,379,204,508,550,905,972,832,431,474,266,430,77,286,921,563,425,560,929,856,240,658,500,620,199,175,853,873,668,826,16,389,931,596,198,271,274,752,478,212,636,161,996,101,569,197,900,23,206,38,881,425,586,122,860,665,578,950,576,434,28,57,333,541,727,465,294,842,622,669,290,494,156,334,300,292,264,157,194,93,303,763,890,690,468,588,124,67,651,216,552,625,67,969,196,317,491,410,895,696,682,974,458,0,985,781,86,383,638,451,374,60,904,401,587,877,814,928,813,15,229,932,949,914,910,362,225,837,567,145,859,343,69,849,474,503,703,763,671,492,980,399,668,410,429,874,426,826,16,471,567,390,258,663,192,902,949,281,473,492,174,993,72,899,381,908,191,558,607,110,998,191,431,843,974,701,346,995,34,764,671,197,773,801,742,782,70,649,412,704,525,286,700,477,413,741,969,876,647,417,336,152,982,574,965,710,544,662,781,93,985,213,880,207,881,427,751,141,458,993,46,420,936,382,785,392,594,129,708,766,5,309,346,293,77,115,595,65,676,175,545,501,374,152,435,473,234,912,446,399,676,705,818,429,768,503,993,977,440,685,890,521,976,961,146,668,644,796,929,109,84,694,950,15,573,161,248,606,9,13,321,207,265,449,189,13,13,280,576,847,463,561,227,109,1,620,597,369,528,326,798,385,258,182,817,980,36,305,927,588,178,184,679,685,718,665,406,260,851,684,405,366,708,874,777,218,649,231,301,815,841,483,650,959,934,9,87,365,516,391,918,487,263,296,79,450,92,409,671,730,85,176,349,948,456,470,746,926,687,781,230,202,721,144,280,548,453,452,432,213,642,164,423,540,817,445,923,137,346,779,411,753,216,497,426,681,23,380,203,293,97,311,222,706,177,526,89,68,636,604,943,991,752,701,193,191,340,208,574,886,3,424,78,867,97,566,150,66,967,711,510,796,675,165,318,685,225,874,263,744,148,986,491,678,182,250,815,72,218,90,968,337,79,44,332,749,381,570,909,273,955,626,948,589,636,165,568,121,526,67,640,871,723,183,837,162,216,432,96,286,586,808,113,88,903,393,134,795,594,208,796,708,670,221,993,867,696,96,314,904,636,179,240,144,211,21,937,563,934,402,440,185,671,831,169,73,42,71,967,563,999,200,786,931,862,51,46,330,431,276,566,145,489,164,993,608,479,801,242,404,279,750,18,341,206,710,356,605,198,705,795,648,768,872,981,568,811,796,522,234,312,219,915,371,945,687,286,531,920,916,116,532,828,769,227,554,371,576,740,660,126,926,981,63,324,409,391,204,595,960,967,635,869,802,404,899,977,453,709,966,450,378,422,165,630,332,979,505,651,520,965,572,44,121,783,241,145,8,130,659,563,883,501,504,500,996,634,791,193,363,576,564,592,890,201,703,941,604,777,452,775,894,71,981,154,256,231,24,109,878,158,915,426,462,572,919,23,534,464,91,936,828,344,175,426,998,616,931,983,561,930,896,494,188,150,180,199,904,419,756,260,330,278,917,97,489,401,214,931,698,944,516,632,729,207,475,59,703,93,381,287,92,825,401,711,355,530,976,283,355,730,994,499,686,269,320,925,286,317,213,901,506,127,332,887,51,52,186,623,203,69,585,267,316,787,152,784,891,435,766,757,62,925,833,516,525,806,860,794,925,634,411,416,123,565,737,193,839,977,234,580,490,444,362,963,909,455,797,846,570,3,838,336,458,629,877,114,746,568,577,245,74,956,579,820,212,840,341,78,527,852,790,297,869,625,742,815,96,41,388,216,786,240,444,912,681,493,262,344,783,260,736,260,570,654,706,474,50,281,384,831,361,801,260,365,601,590,133,417,47,567,123,374,612,53,590,328,844,237,287,752,958,45,569,91,910,402,54,574,241,666,547,212,504,412,836,643,287,326,814,297,148,821,401,515,886,987,394,993,470,491,736,589,662,802,639,817,676,555,320,922,348,770,9,929,77,206,153,154,420,971,688,662,643,82,539,550,458,476,728,866,198,682,367,970,478,787,195,579,525,433,347,738,530,83,904,565,847,161,147,16,294,476,531",
    "268,407,990,918,116,641,589,419,993,109,84,670,512,25,506,913,452,972,6,842,482,648,297,151,810,735,619,529,708,563,549,739,331,680,130,296,231,309,348,815,500,266,636,761,603,547,543,578,700,138,930,718,650,521,520,617,114,478,797,770,353,119,794,787,190,549,445,91,760,762,140,845,646,102,924,709,324,727,620,639,360,564,138,219,434,181,974,620,922,700,674,908,708,915,624,640,934,535,788,250,975,657,495,765,723,853,601,8,152,868,989,666,248,714,818,42,819,488,752,461,290,179,451,715,700,932,897,309,765,344,929,821,625,87,570,452,620,148,866,198,257,62,15,385,651,477,207,670,418,814,245,183,331,113,314,768,488,60,522,589,504,650,68,193,660,133,188,307,730,318,777,431,387,726,541,393,142,615,613,432,39,426,941,774,121,707,746,933,815,985,0,626,305,965,764,631,227,597,297,114,869,367,709,588,607,842,222,858,36,48,484,940,654,759,220,359,173,52,580,773,638,749,348,952,510,597,123,188,879,773,397,310,815,395,288,799,322,442,827,48,895,911,837,280,195,664,682,720,750,464,292,410,632,369,550,916,838,909,348,730,141,82,601,588,50,860,757,532,206,680,751,821,949,8,629,840,617,155,330,80,519,284,63,466,171,203,804,815,57,367,515,461,687,838,255,546,205,201,281,451,180,737,708,935,883,25,735,62,888,239,238,662,713,586,134,877,639,910,631,521,131,379,803,712,443,106,23,452,804,857,41,618,776,422,313,555,106,975,905,449,984,204,649,82,517,142,604,503,193,650,874,609,432,288,621,601,716,150,535,486,411,743,539,640,290,871,948,774,135,650,285,984,514,935,291,164,959,346,633,750,980,786,763,263,723,358,210,92,747,21,405,263,760,653,203,558,405,102,616,234,535,488,218,701,51,308,68,341,840,605,271,841,646,259,482,370,906,405,515,715,872,401,791,533,856,60,866,553,369,605,276,946,135,593,245,62,230,863,172,895,562,169,462,835,53,518,900,447,867,930,758,757,600,419,759,55,598,741,138,298,910,735,228,691,938,365,407,493,171,45,944,513,807,803,434,420,194,370,702,504,557,671,597,839,302,156,819,295,427,92,248,907,46,621,64,798,819,593,394,706,719,677,959,211,394,671,668,510,282,27,406,375,387,931,832,594,512,511,844,123,501,778,789,576,937,370,182,207,427,629,894,778,171,485,411,239,745,452,163,205,168,473,43,267,403,265,779,859,114,466,793,294,932,273,79,392,768,308,477,912,367,582,671,857,650,996,66,76,110,359,274,338,913,687,982,250,441,350,146,683,920,627,111,390,853,502,111,359,409,803,708,667,161,826,815,368,247,504,548,791,674,199,515,928,455,266,916,676,358,830,562,926,928,612,873,637,934,970,668,740,925,224,480,704,913,874,951,685,905,677,691,422,740,409,278,503,784,354,816,344,846,304,860,502,28,300,696,457,631,922,842,551,739,619,699,728,559,448,926,367,691,357,314,194,889,651,271,123,579,669,395,337,357,999,183,948,971,85,692,621,414,822,735,419,775,130,953,667,766,912,42,466,588,291,255,286,168,309,437,500,428,946,848,286,639,316,128,352,991,681,28,660,873,760,195,640,302,886,95,530,117,592,60,461,717,299,405,295,568,585,521,480,919,315,545,304,74,320,505,962,906,89,811,806,160,83,191,159,919,385,764,777,70,187,711,930,55,231,204,326,417,611,339,652,338,687,39,8,511,832,770,555,72,396,163,990,18,87,130,764,314,362,142,104,609,220,130,685,594,701,888,864,190,279,901,635,67,385,266,486,284,901,483,575,819,26,639,954,630,53,295,998,15,355,362,793,130,853,820,890,43,384,571,200,887,476,727,527,904,434,215,166,830,887,718,206,733,526,675,916,14,316,55,517,979,232,859,81,761,366,221,278,278,265,622,217,169,178,705,861,101,29,943,572,657,548,769,542,975,938,540,549,435,323,891,120,571,187,916,938,285,220,386,841,34,121,842,323,320,649,588,42,170,144,866,454,352,547,741,86,314,973,57,552,245,924,31,861,476,516,24,698,708,614,753,893,338,195,185,605,594,339,262,722,175,781,595,113,884,312,968,264,915,151,691,416,871,946,246,421,594,243,683,733,300,171,597,421,867,518,579,96,426,548,666,127,190,388,668,132,391,141,680,859,687,784,228,742,374,760,574,662,503,143,638,884,74,791,788,917,546,246,593,409,293,465,424,344,672,299,200,385,708,651,165,392,360,298,749,503",
    "177,844,365,456,528,570,171,666,433,984,563,61,908,517,905,467,676,223,186,448,762,440,433,685,423,702,846,807,708,765,226,614,572,386,926,313,663,539,107,613,581,472,560,540,714,14,956,85,571,484,284,845,337,144,417,24,333,541,308,639,521,272,415,58,941,386,253,423,408,497,557,467,602,937,117,268,954,104,432,202,387,210,121,833,762,19,65,896,329,526,352,259,136,715,847,893,870,401,637,838,1,642,133,756,87,700,624,732,82,294,622,368,33,37,932,612,795,758,34,533,56,252,785,494,89,628,574,620,140,899,928,255,73,354,687,235,702,499,471,16,976,105,517,205,839,766,577,103,617,419,574,760,696,196,89,383,429,58,986,655,415,522,806,29,899,676,150,223,292,45,173,717,703,134,911,539,221,545,191,851,471,496,745,598,62,593,305,601,870,781,626,0,144,701,517,577,190,386,432,528,664,914,369,11,544,674,34,687,757,223,574,264,537,501,208,679,618,700,259,981,480,351,904,648,15,985,530,759,231,428,341,457,10,629,222,303,186,845,191,54,822,176,613,690,966,542,221,81,436,757,313,288,413,244,522,700,420,258,856,351,925,350,532,984,451,335,551,341,683,577,412,391,498,478,432,340,115,822,962,253,930,456,213,80,232,544,455,625,549,10,485,869,697,605,444,677,153,681,359,775,715,73,685,679,887,132,331,176,229,973,894,110,897,171,801,954,116,656,834,41,411,175,972,603,346,20,839,5,695,841,186,271,248,337,946,246,323,749,874,960,483,725,258,684,298,992,279,568,366,279,810,650,939,583,186,49,696,718,305,22,489,983,135,262,97,612,65,818,672,276,213,450,93,884,777,259,485,61,134,392,381,882,886,66,144,356,334,806,560,317,355,822,133,721,71,608,221,465,857,601,646,910,627,867,646,134,151,98,352,38,359,812,235,326,122,876,117,990,788,22,308,952,599,213,509,577,829,803,663,620,987,271,579,479,465,732,51,947,839,784,181,215,779,909,58,706,908,984,665,607,976,638,948,556,165,874,873,582,182,488,629,295,415,524,441,687,512,662,53,953,965,370,171,65,875,340,298,457,339,77,287,443,632,897,408,26,533,732,175,334,814,639,458,508,748,849,100,263,93,428,762,336,184,145,596,631,480,947,450,885,30,931,668,147,229,796,393,642,486,830,316,85,57,232,831,877,381,826,249,609,874,795,558,325,553,939,426,872,294,183,121,756,832,529,85,781,882,233,538,91,229,14,711,725,760,718,21,685,818,76,531,547,491,327,628,568,600,434,504,286,96,886,39,321,811,832,453,389,95,343,452,713,434,588,352,209,571,331,987,486,357,634,852,886,663,483,20,407,462,960,69,22,443,585,900,770,607,721,219,457,692,686,149,304,70,961,23,875,577,65,926,451,782,303,290,944,171,663,247,194,66,724,385,809,508,629,55,94,72,785,286,219,475,915,512,778,174,668,774,887,852,307,172,768,3,743,341,227,454,464,15,730,503,437,835,250,306,964,382,489,866,150,834,351,613,209,303,535,465,843,210,476,109,195,185,96,859,613,600,416,395,791,287,852,108,993,329,468,61,33,908,618,524,18,961,312,463,184,14,568,340,728,753,807,739,97,72,599,53,420,990,576,918,271,507,244,836,326,105,740,437,679,510,98,252,100,708,551,881,296,936,767,548,14,193,215,78,298,522,680,621,167,480,157,389,359,240,853,445,289,430,441,284,954,474,33,839,47,822,247,99,95,928,706,276,834,207,74,923,203,292,444,777,242,105,814,974,546,745,961,694,315,353,293,150,178,76,968,921,849,23,881,922,508,302,202,979,943,586,9,51,717,241,570,66,381,9,50,403,484,317,520,474,78,700,375,295,247,701,640,274,761,120,157,162,43,329,33,424,502,838,334,507,755,143,746,893,270,817,994,75,520,596,91,40,799,893,151,399,813,164,373,640,61,500,719,224,241,868,452,140,251,607,284,7,26,458,44,136,420,827,498,619,998,975,398,828,354,624,780,852,754,730,798,804,654,40,931,538,339,395,367,421,782,439,437,278,701,838,766,48,117,844,327,809,119,899,153,365,768,849,716,375,47,525,846,997,429,376,781,513,553,65,486,363,796,404,552,379,969,435,551,750,433,947,537,584,764,344,210,853,12,29,847,722,172,449,972,500,464,95,638,671,674,584,295,519,675,123,824,180,958,334,206,533,389,988,468,355,667,679,97,724,480,898,347,877,769,200,552,492,154,422,812,395,638,207,456,409,973",
    "722,390,284,370,285,185,625,491,715,273,739,545,721,115,286,697,32,418,808,591,885,791,486,240,345,991,257,618,511,416,185,318,794,325,766,526,525,615,810,391,253,345,410,240,486,329,811,196,636,350,251,694,361,941,898,500,200,39,103,867,370,647,381,132,820,39,163,241,875,844,367,840,532,279,222,638,791,483,565,426,131,187,791,781,361,271,499,118,919,805,246,818,986,281,821,57,837,429,491,455,52,476,488,732,261,897,215,768,992,375,213,726,150,764,440,722,892,437,578,393,379,656,240,435,273,64,794,94,301,159,93,172,745,201,396,820,872,482,884,225,432,425,702,707,626,635,731,610,962,950,373,912,461,482,291,705,915,680,628,876,33,628,395,995,49,364,944,351,536,349,756,342,670,612,200,77,353,523,415,424,428,519,315,786,944,965,516,728,397,86,305,144,0,641,310,951,324,847,137,484,150,320,951,695,93,881,62,315,213,63,665,426,119,470,781,585,866,727,636,640,387,959,617,89,339,269,10,327,468,766,409,713,447,493,146,361,332,910,491,928,685,537,242,700,651,469,154,195,216,668,785,340,708,67,930,149,278,427,113,536,225,306,872,343,67,61,50,569,219,438,818,813,268,263,893,678,99,620,855,516,93,22,762,414,863,301,924,161,129,246,969,776,494,319,57,191,169,953,982,355,263,429,588,69,964,12,399,150,688,55,412,511,431,574,873,329,501,69,487,228,149,377,491,900,404,687,443,154,292,818,814,959,377,605,151,623,412,882,34,689,66,109,915,435,568,446,603,151,244,309,29,845,954,760,707,879,287,791,622,868,290,675,979,453,173,499,559,649,367,708,99,456,783,20,205,72,780,483,677,692,473,196,137,880,23,827,76,305,468,646,898,459,465,487,19,104,222,667,402,574,520,812,845,319,74,998,771,590,951,967,99,548,523,753,983,203,73,430,704,349,674,888,522,41,68,287,478,512,238,667,578,981,950,341,135,430,769,346,320,607,408,144,372,50,781,964,438,469,428,5,42,345,809,482,507,227,727,605,207,488,970,562,34,546,319,841,263,315,224,932,19,136,972,816,499,808,628,182,744,476,549,420,6,29,644,542,348,873,921,617,153,845,328,691,489,39,488,520,758,821,246,633,951,395,322,650,259,427,741,838,517,595,40,714,731,499,547,306,870,539,289,812,709,918,591,800,173,376,948,320,797,599,375,944,69,442,193,663,847,149,637,561,778,153,372,511,338,374,321,387,225,528,622,98,150,434,783,284,37,550,384,571,634,65,187,893,599,631,8,584,38,541,967,272,246,445,769,943,333,658,864,586,346,226,74,871,505,972,769,229,547,331,82,863,408,137,256,313,756,221,915,392,658,613,609,885,358,226,352,413,240,339,278,654,883,441,824,779,754,493,982,389,792,175,158,225,528,419,59,717,59,757,510,317,362,174,344,88,726,930,805,754,443,631,962,799,166,956,25,862,23,80,851,508,836,434,334,729,389,383,775,275,255,796,187,891,166,397,334,578,490,805,176,809,46,214,901,414,868,272,773,752,784,720,57,369,108,315,256,947,78,679,414,603,357,459,573,388,670,592,94,154,194,691,112,418,939,554,489,851,300,97,497,767,926,706,488,928,674,42,883,266,495,88,7,184,690,446,695,962,109,779,482,985,661,52,394,539,926,987,51,713,105,210,983,334,522,36,514,346,901,602,286,723,926,385,877,122,322,220,302,274,94,585,759,644,778,106,266,811,862,481,214,680,386,628,902,729,352,15,578,704,231,968,852,551,888,767,211,414,392,551,921,384,971,678,401,250,35,509,390,89,934,384,567,526,598,224,969,812,646,394,114,368,521,954,400,807,736,689,441,785,634,334,552,909,708,448,390,641,128,678,874,257,149,516,920,671,94,490,980,964,88,58,228,175,130,157,503,105,238,597,405,73,782,205,242,544,893,638,403,658,251,547,771,42,55,224,847,731,732,461,874,471,848,348,253,59,714,513,635,230,872,94,840,626,862,256,445,409,370,505,251,542,205,964,679,64,583,394,341,98,681,678,879,742,73,615,240,53,702,553,262,22,118,744,122,166,316,764,508,775,523,599,476,18,397,697,318,831,598,439,102,383,505,876,614,536,769,966,854,110,378,502,78,599,578,856,313,406,277,143,974,763,43,897,273,679,267,332,429,677,149,171,805,136,543,626,366,403,72,346,114,54,264,438,192,537,20,588,151,698,11,368,946,191,421,631,406,923,653,565,779,48,637,777,421,253,944,116",
    "49,619,260,723,413,33,558,146,489,167,84,837,508,576,538,889,838,517,652,822,147,620,417,260,389,787,878,869,295,849,560,663,123,247,817,674,258,992,944,678,592,339,288,852,954,885,419,939,126,953,487,909,352,899,267,68,953,464,708,840,545,442,401,737,336,350,243,605,396,760,893,335,436,620,948,894,715,21,725,142,500,110,821,911,974,401,257,40,631,459,237,820,390,278,514,234,807,72,319,214,30,742,894,905,489,808,785,957,1000,558,583,819,524,81,11,711,67,161,810,710,363,230,691,492,595,876,178,129,90,210,965,736,713,487,412,538,326,922,270,849,764,14,507,150,688,815,486,282,571,360,544,964,31,10,859,304,332,644,16,874,512,333,974,725,409,625,494,860,840,650,839,392,50,462,50,940,203,889,173,743,85,111,424,929,171,160,405,774,298,383,965,701,641,0,834,959,886,542,142,240,469,722,959,583,773,447,624,340,352,853,222,488,680,131,191,90,344,307,448,899,640,40,760,188,880,466,606,138,835,155,179,372,348,842,334,930,313,186,189,690,346,548,464,532,709,7,543,989,925,660,425,82,700,395,695,517,269,705,528,784,561,833,51,498,958,378,61,549,389,226,415,573,876,658,903,160,586,903,897,476,202,808,376,744,473,888,75,785,98,660,527,588,611,946,488,18,932,891,623,625,498,132,547,196,773,787,165,535,734,777,362,290,273,102,563,513,579,916,317,799,740,717,884,198,207,262,209,866,82,331,494,800,685,950,908,411,988,878,698,748,931,810,404,747,611,286,334,925,495,467,969,506,811,61,799,240,144,459,413,378,229,807,871,273,626,80,533,806,760,82,416,148,954,211,734,475,666,581,516,76,47,359,515,962,99,760,358,647,913,956,178,621,508,926,838,937,414,725,358,497,312,103,416,190,783,506,155,779,123,854,198,860,897,79,54,170,831,219,258,694,890,56,233,89,800,434,398,436,590,182,515,421,266,476,652,824,491,235,868,217,714,857,16,111,752,195,520,770,52,83,425,266,626,269,93,494,901,618,68,612,927,987,666,739,578,181,278,711,374,95,883,409,151,210,300,391,220,363,984,252,489,641,283,173,827,494,107,993,226,364,430,136,981,493,553,938,313,703,534,471,665,678,99,944,468,329,532,840,947,262,24,353,992,837,987,464,871,465,874,94,856,907,133,492,20,884,729,738,164,629,770,536,240,707,294,685,397,535,617,62,402,730,402,313,288,597,254,780,810,211,665,163,260,658,868,672,150,317,684,797,646,47,644,4,822,865,516,655,459,54,414,96,514,225,380,909,706,637,713,499,555,519,862,604,196,968,992,943,24,624,257,908,267,126,351,816,637,638,19,842,948,904,345,459,895,102,766,926,670,186,919,994,412,609,852,100,384,695,352,189,903,343,286,953,364,194,561,962,36,126,100,737,891,589,899,16,583,731,293,44,38,707,692,905,772,22,796,175,766,193,896,304,418,474,356,752,781,320,935,68,725,594,797,341,119,325,452,416,190,755,119,117,692,803,731,568,871,482,512,724,761,15,570,153,514,388,721,92,810,675,576,253,217,922,808,106,911,247,411,558,401,186,920,887,396,337,541,727,8,925,353,552,880,624,540,371,289,983,59,80,511,782,952,345,864,714,416,670,159,399,199,215,341,279,606,595,959,80,107,371,848,294,333,36,516,262,819,327,874,73,485,748,865,440,679,944,375,868,180,642,550,493,261,484,629,789,905,905,127,936,332,61,944,887,746,249,718,150,659,385,403,501,737,548,396,657,48,587,118,509,784,765,139,823,791,16,331,600,933,722,449,221,848,928,194,72,212,735,878,610,154,718,871,915,996,857,974,454,87,9,390,794,855,430,99,32,297,846,384,853,480,432,333,32,772,192,442,927,115,910,974,611,354,541,660,699,745,835,950,839,592,353,933,713,740,78,463,139,760,883,202,714,986,883,29,499,993,111,35,254,661,447,200,825,855,289,693,598,268,902,334,640,487,438,649,15,409,85,156,939,957,679,19,112,87,95,829,456,240,784,730,935,326,506,683,863,894,877,680,121,243,471,608,560,531,71,247,974,738,985,476,687,787,369,438,125,713,218,248,541,909,156,220,121,815,427,873,338,264,933,49,780,86,198,104,683,800,412,399,661,908,976,266,751,328,75,284,827,522,543,953,207,967,342,713,300,794,859,668,574,612,525,572,97,651,242,897,641,175,682,193,159,566,329,913,605,266,787,822,206,720,121,30,102,937,959,314,486,541,849",
    "448,477,652,118,431,389,770,905,956,528,710,327,346,414,744,413,494,407,82,228,945,963,202,169,604,460,116,230,101,512,344,82,941,744,865,299,305,825,750,88,167,742,828,239,737,23,879,384,294,500,706,240,784,572,715,502,489,976,910,264,778,814,743,374,19,197,904,399,401,53,823,860,285,913,778,235,552,261,746,197,813,517,27,509,828,808,615,974,917,696,218,750,683,804,150,427,972,836,384,982,338,187,325,397,20,513,433,239,651,292,332,733,206,800,256,841,82,731,388,885,901,650,939,237,257,270,560,725,950,976,543,192,983,189,164,174,317,87,629,886,582,449,499,516,829,193,669,228,537,457,95,85,231,682,185,731,416,55,675,823,244,530,922,737,87,218,436,945,421,211,328,317,540,671,139,899,431,279,68,145,877,239,370,25,388,184,275,807,404,638,764,517,310,834,0,255,610,623,78,329,197,729,83,868,549,921,141,431,496,976,933,700,510,358,512,413,930,750,517,371,283,276,751,492,624,786,400,829,571,387,267,836,837,742,19,697,18,765,439,728,990,269,354,375,396,376,439,544,438,342,236,462,648,32,598,911,956,521,859,488,805,473,338,401,585,600,629,227,169,321,232,743,519,849,698,627,78,512,868,886,901,457,206,692,126,773,991,231,430,368,508,804,471,335,119,794,401,38,66,853,366,421,467,613,704,135,668,39,662,815,248,541,523,541,366,272,580,123,607,839,167,835,271,185,716,114,813,8,839,14,610,765,554,102,230,44,354,304,431,731,300,794,183,802,174,76,554,223,865,283,382,827,356,750,512,155,971,258,876,391,320,120,394,73,865,472,373,559,875,37,872,16,826,193,966,577,820,513,335,500,779,507,944,869,204,84,377,782,898,75,385,650,728,658,920,287,730,24,634,958,363,275,903,55,12,459,493,678,267,613,654,21,547,97,555,373,751,513,190,887,887,590,304,325,347,627,565,549,513,940,883,722,194,706,602,258,17,977,128,70,266,53,455,532,141,419,840,813,183,658,838,882,914,109,17,108,399,880,742,796,855,552,162,122,593,650,914,833,790,271,343,356,346,269,169,820,546,250,595,731,366,797,625,576,629,877,876,699,755,665,671,915,337,747,861,284,254,129,644,972,172,280,996,109,996,831,805,747,655,197,273,376,896,737,885,410,150,973,39,491,284,741,749,536,142,677,898,43,741,690,766,960,843,273,203,809,554,277,311,472,740,130,857,213,595,91,786,651,431,228,692,711,420,269,324,275,750,279,761,633,348,282,645,966,454,670,408,608,395,24,845,28,895,171,435,10,176,317,945,2,55,810,858,349,378,629,277,431,731,32,504,366,119,814,392,397,593,771,497,511,197,544,766,758,429,295,400,101,444,848,459,680,907,254,986,113,538,891,57,325,627,343,798,16,290,108,883,55,896,830,417,9,78,152,455,432,807,555,753,793,38,678,640,813,928,981,360,43,585,182,79,387,272,246,542,245,801,887,248,471,420,486,512,883,326,362,909,27,159,922,67,302,699,903,235,75,100,778,336,525,490,224,823,787,491,370,40,683,677,607,115,595,771,172,947,230,479,981,964,592,762,763,321,997,569,527,535,274,721,979,423,249,662,885,513,5,758,86,732,447,805,941,882,71,216,500,480,270,753,676,205,104,6,444,64,322,462,608,525,308,557,682,490,485,151,152,920,364,261,433,550,161,671,466,380,767,422,361,204,270,312,15,291,333,609,957,277,912,608,175,27,694,459,727,266,800,973,591,843,74,572,737,748,603,567,312,47,640,569,660,546,128,201,355,577,287,193,880,751,26,830,815,630,345,692,944,532,539,814,999,127,355,326,979,646,650,862,8,609,863,538,970,243,927,554,812,692,843,461,715,651,531,859,773,958,346,506,82,397,941,728,118,546,975,378,647,485,916,17,145,671,534,597,2,745,631,890,340,811,95,69,508,150,791,734,261,855,802,536,820,221,627,67,727,957,230,72,582,54,121,696,779,727,535,992,933,386,865,127,682,831,833,156,655,459,5,456,562,890,564,342,511,560,174,494,485,551,806,81,177,632,809,516,231,914,236,836,79,19,680,41,437,237,188,461,485,238,861,558,844,233,845,984,504,328,34,665,349,414,744,238,496,779,198,401,825,715,247,31,4,291,306,226,146,361,860,852,613,956,856,46,421,356,465,898,637,866,704,383,821,493,400,508,123,493,125,326,939,411,397,351,704,704,723,741,431,173,209,679,8,539,859,780,459,187,398,222,332,637,500",
    "849,822,788,301,736,435,356,454,860,279,906,532,279,831,307,105,375,721,808,692,86,244,606,823,306,720,512,337,580,864,760,781,728,880,148,123,723,645,898,782,122,463,733,91,747,282,481,506,112,393,61,897,846,266,126,852,921,774,423,376,318,679,645,310,319,552,595,321,303,218,448,846,251,96,195,291,233,957,317,562,397,182,352,147,695,762,311,908,474,827,989,270,112,472,915,571,802,859,116,757,672,454,209,290,452,110,32,927,785,335,242,331,759,690,585,564,374,392,56,684,928,913,501,25,879,111,574,161,246,482,262,145,661,937,175,110,126,184,895,667,764,93,868,533,133,166,485,503,168,448,651,738,491,84,118,980,598,291,14,960,521,849,984,930,623,891,916,340,646,844,633,217,395,213,219,825,656,255,138,844,502,939,260,953,87,742,97,532,829,451,631,577,951,959,255,0,852,654,988,722,437,98,881,954,366,958,448,123,908,81,799,183,792,697,500,719,878,658,299,605,13,352,458,324,722,937,286,913,932,782,36,969,528,803,65,649,489,119,561,533,813,697,439,480,679,685,291,965,184,804,132,348,325,5,763,366,952,197,407,100,846,950,169,145,517,671,464,118,755,350,58,541,979,502,800,925,922,591,82,181,540,211,92,793,657,288,263,855,530,793,386,467,658,188,263,48,60,540,881,823,929,919,579,684,699,66,869,645,411,991,99,236,327,251,508,384,126,498,992,598,120,743,290,840,548,14,664,928,643,716,438,912,900,157,54,711,166,897,397,756,805,899,84,73,374,36,761,501,84,817,280,578,984,458,682,93,280,490,644,154,45,569,449,563,83,250,743,525,763,334,212,18,677,560,904,246,846,406,719,416,360,807,620,769,998,385,725,493,207,648,735,806,262,601,780,904,767,193,200,288,101,364,730,825,444,367,634,187,157,324,947,2,506,127,89,964,985,266,429,868,804,518,64,328,9,635,257,17,936,526,954,698,313,385,537,749,419,662,326,121,771,509,178,490,737,52,261,689,953,31,666,825,85,264,726,290,500,807,609,749,334,459,295,884,170,523,790,844,882,886,782,304,154,740,118,420,284,204,559,582,202,92,533,410,639,905,291,17,165,421,641,896,891,408,418,3,731,418,91,540,981,489,815,631,463,403,974,83,24,798,150,376,945,146,514,112,251,629,208,59,35,670,798,930,857,620,949,24,219,267,78,64,554,179,996,832,770,421,882,393,932,386,767,227,378,634,555,969,298,65,54,849,512,34,443,728,217,647,60,18,121,164,911,84,513,768,439,337,183,919,434,630,925,837,199,555,834,618,23,663,337,967,645,243,533,912,359,281,961,285,71,116,210,636,754,793,641,202,711,168,766,823,510,111,701,654,603,390,591,642,764,266,128,393,720,290,703,96,768,69,227,940,131,744,511,857,209,308,901,105,229,497,280,89,778,916,47,337,744,424,759,927,776,709,996,375,222,872,591,67,986,24,888,707,966,705,330,275,829,738,29,332,797,167,80,548,139,965,403,898,496,35,934,257,312,109,340,173,500,420,481,66,892,886,841,217,32,940,596,545,98,367,637,132,419,287,421,612,713,530,585,881,436,498,789,533,104,291,343,981,309,659,20,231,624,112,323,467,279,214,901,138,647,323,219,956,16,919,874,444,442,608,862,679,763,512,599,125,209,808,327,358,484,221,450,902,487,191,493,351,561,754,421,798,345,712,722,111,67,664,640,452,288,853,992,676,952,398,585,728,660,530,163,672,890,583,954,511,711,603,412,876,820,141,790,326,122,516,405,248,716,767,116,359,431,761,411,770,708,668,725,676,351,206,501,787,766,745,648,190,471,887,919,585,713,656,283,971,800,448,547,668,182,534,338,231,269,578,44,90,12,320,347,548,619,18,454,787,264,877,476,606,139,44,881,825,896,633,243,251,289,984,846,863,473,990,165,840,937,860,684,928,935,25,282,202,972,59,450,419,309,673,685,622,845,227,443,369,578,956,760,472,125,381,698,515,331,736,73,126,1,370,942,910,26,413,60,361,820,498,600,608,595,873,806,879,594,497,724,57,776,835,245,843,997,254,171,372,505,74,474,867,530,213,592,941,130,924,519,893,746,473,407,331,485,627,636,829,271,888,650,466,291,61,90,254,475,450,421,157,40,406,197,787,124,642,577,666,73,575,679,231,723,221,620,976,788,170,603,300,235,594,603,883,708,361,357,783,985,531,804,652,954,773,298,546,278,307,723,159,332,411,143,115,588,537,673,942,291,944",
    "381,968,918,412,366,961,535,658,812,578,691,772,119,724,855,49,622,677,587,835,376,582,223,62,727,392,131,705,504,221,491,452,555,404,418,133,824,251,106,656,347,639,576,821,923,627,197,310,88,719,650,104,14,757,730,211,641,805,208,6,504,765,904,612,607,322,850,110,664,32,82,961,415,35,493,705,616,308,507,898,395,460,374,959,873,123,487,963,881,987,897,343,939,822,881,969,437,425,18,423,437,324,785,751,867,618,146,209,802,421,934,775,597,161,672,366,318,694,468,55,604,609,905,910,181,733,89,419,88,574,587,212,754,636,246,151,262,920,649,496,712,252,785,205,850,268,986,263,719,965,32,79,220,877,19,258,355,123,719,118,72,288,640,611,831,269,586,831,362,735,276,540,935,702,949,922,326,61,744,944,609,455,34,543,996,185,495,5,650,374,227,190,324,886,610,852,0,614,537,417,146,915,864,435,269,742,489,323,986,529,498,985,216,566,743,727,416,608,289,897,889,90,858,134,190,924,824,894,528,251,637,171,154,358,348,366,432,585,691,693,540,100,79,335,660,439,755,828,485,883,647,457,992,762,334,598,565,546,988,392,245,621,157,316,296,900,915,241,46,520,215,89,525,321,503,806,906,437,800,347,751,797,44,143,302,479,43,953,819,713,267,342,892,91,757,934,393,600,716,203,793,298,416,280,196,822,513,131,846,482,435,813,212,640,85,540,625,462,278,354,707,311,201,675,137,540,121,682,107,966,163,678,902,782,874,522,54,313,119,380,517,973,547,138,130,479,829,271,686,693,944,979,75,306,728,901,826,523,988,92,660,214,902,245,192,922,665,32,799,516,519,848,32,72,583,652,410,330,571,998,795,620,340,9,54,933,178,90,117,615,491,748,947,182,790,741,751,527,817,578,491,952,38,170,102,486,76,733,570,135,261,452,734,902,987,421,453,115,84,514,738,609,443,594,922,717,427,403,170,210,815,79,35,246,917,191,136,609,845,358,386,125,60,838,593,934,5,819,542,405,940,881,111,699,921,774,432,97,992,93,841,540,277,422,228,24,667,109,426,198,218,194,349,101,200,670,911,60,652,673,761,885,831,139,125,278,231,904,699,861,917,803,717,805,558,745,957,741,730,322,318,416,340,758,726,109,876,725,455,834,664,555,227,688,920,82,194,836,730,514,915,278,972,733,347,396,356,678,319,33,227,532,955,107,295,760,73,374,461,118,220,577,908,905,105,193,678,19,415,777,81,15,584,481,933,641,497,962,497,862,784,610,41,164,234,50,737,981,391,365,788,103,88,455,136,796,784,196,753,910,1000,566,81,723,280,212,691,216,65,385,137,970,48,429,942,60,951,961,341,581,84,266,417,617,273,572,962,267,94,687,529,342,538,687,574,459,428,437,380,610,628,457,487,542,10,107,552,932,367,939,203,11,140,949,658,517,541,954,679,703,412,254,711,317,583,530,704,501,635,822,148,969,172,815,592,754,699,119,239,47,33,514,259,25,822,873,861,695,686,863,174,854,869,643,995,974,829,803,613,391,941,345,279,536,986,775,7,756,790,474,580,443,715,370,960,116,64,779,330,336,334,89,987,453,579,908,715,790,559,280,31,206,462,232,368,868,844,458,608,643,31,827,327,149,661,731,28,293,407,658,575,538,85,197,506,699,379,301,787,607,857,952,105,463,222,941,260,223,19,677,993,842,438,352,347,849,497,171,512,695,427,541,54,893,448,365,658,338,766,890,478,437,782,728,450,424,324,278,204,119,642,315,67,482,667,546,214,94,916,801,831,446,412,290,638,887,866,509,642,426,348,254,773,540,211,789,847,247,546,964,974,565,621,331,936,433,601,514,56,157,851,680,700,920,535,313,699,87,718,888,779,944,761,397,562,711,778,507,297,455,217,819,929,767,818,708,248,664,598,90,721,845,116,314,995,866,956,24,818,625,696,249,960,158,358,905,564,67,54,323,745,310,141,696,243,953,992,540,820,143,148,942,198,482,5,256,513,881,556,333,298,675,576,961,861,607,219,716,302,690,854,973,514,814,700,48,456,257,370,938,852,20,803,820,33,426,176,710,883,755,984,290,963,541,650,429,210,897,255,421,169,531,974,356,773,73,36,120,383,62,244,929,161,313,901,348,923,765,459,200,980,757,695,9,295,166,80,585,882,370,736,522,633,319,993,125,830,40,112,388,70,901,835,447,380,469,581,508,692,157,148,445,257,161,436,459,684,536,467,577,903,837,421,979,500,864,31,885,973,757",
    "720,17,689,789,235,606,784,97,165,338,848,557,263,381,148,631,135,76,370,867,730,319,189,171,668,433,299,647,562,351,433,648,335,736,993,183,215,49,996,607,834,232,928,351,794,863,514,381,141,414,211,989,996,944,585,78,381,421,46,235,946,712,468,550,576,14,814,14,843,759,106,165,753,551,584,434,578,622,929,855,421,910,630,223,253,802,991,389,575,470,991,547,2,45,541,389,435,654,676,473,995,547,185,816,368,258,190,328,32,520,957,180,945,550,865,786,614,183,136,597,741,136,101,736,598,58,114,210,112,233,331,796,70,89,581,111,219,834,697,800,807,169,245,790,801,854,158,961,941,50,83,903,518,48,701,567,304,892,188,617,247,662,451,226,588,627,389,765,133,173,50,79,518,996,793,791,901,805,329,421,986,273,466,779,263,631,480,335,961,60,597,386,847,542,623,654,614,0,450,296,852,392,13,790,893,12,236,290,447,569,755,869,996,202,385,536,309,597,173,525,653,178,292,719,110,703,981,722,482,965,113,923,269,471,431,390,932,836,668,232,119,294,647,13,985,785,299,603,531,500,840,614,8,717,110,488,143,730,446,291,172,558,601,872,794,240,594,160,179,782,329,865,336,215,59,728,847,957,847,751,341,370,755,343,557,599,798,792,661,775,847,723,684,491,274,771,179,545,307,942,95,632,91,965,604,149,86,634,252,221,491,892,343,453,143,148,175,753,72,570,988,396,557,677,700,148,426,401,85,921,176,210,162,869,741,225,386,663,331,337,99,759,603,878,885,885,321,417,116,101,768,661,843,119,669,352,748,135,837,568,40,429,527,650,212,393,159,968,917,108,986,909,520,213,615,745,890,707,672,697,918,943,332,951,808,831,946,589,768,71,153,185,251,398,992,616,484,247,947,954,906,796,796,29,243,268,983,500,50,704,36,300,234,237,716,112,477,408,481,163,400,958,447,205,382,192,86,99,256,631,721,857,491,664,594,496,208,570,539,214,959,549,646,648,979,33,112,280,273,568,481,265,452,59,891,667,389,951,889,137,130,656,66,210,178,400,94,513,670,326,898,590,791,618,70,496,562,440,39,790,457,983,226,731,697,186,345,634,264,421,217,207,257,696,38,90,897,109,396,970,771,215,929,180,620,66,258,551,771,347,941,79,752,846,9,616,567,903,371,409,85,382,701,331,504,375,359,614,891,887,115,721,279,623,874,839,628,901,817,546,718,168,925,71,208,8,284,345,709,931,627,195,188,97,823,495,143,459,249,216,5,979,942,472,493,971,157,483,30,542,140,656,26,491,443,627,840,912,652,630,674,674,724,363,415,825,953,757,444,959,691,656,881,546,860,293,236,980,923,534,649,135,770,872,775,98,819,105,432,547,600,228,479,25,41,735,724,507,860,894,611,322,929,69,426,230,411,12,739,171,989,893,900,176,975,854,972,44,602,528,396,33,152,38,580,714,243,920,345,846,819,45,325,85,368,241,705,349,682,649,560,17,466,857,861,190,44,662,148,468,225,280,426,568,813,771,34,404,469,524,408,212,451,472,131,77,262,447,905,532,475,144,821,998,658,132,961,715,966,851,678,683,116,157,459,671,479,736,636,940,176,897,727,944,570,499,686,982,165,320,957,963,266,576,382,830,927,48,244,293,269,16,796,66,340,749,485,567,648,758,170,191,232,938,323,208,631,750,295,371,190,934,786,365,882,158,525,697,760,742,199,732,255,507,585,214,894,818,46,697,212,430,334,155,742,935,154,128,685,271,797,155,364,979,641,289,879,452,647,285,161,376,273,770,677,933,789,804,226,208,749,273,393,924,218,508,891,298,181,369,714,410,243,801,105,219,534,963,110,440,787,937,219,591,869,843,777,495,961,888,438,855,917,884,138,586,334,572,579,244,695,518,478,654,171,341,605,736,479,19,962,881,460,91,20,649,695,604,166,167,368,389,868,603,110,113,927,870,794,536,314,313,676,88,407,217,292,711,326,814,915,815,244,518,815,534,627,977,687,278,37,662,151,364,699,703,622,469,465,758,813,760,588,381,7,890,781,366,567,855,809,348,966,250,764,382,96,27,503,649,632,36,793,847,669,479,614,3,431,889,678,769,899,130,828,741,189,930,111,575,925,210,987,216,751,253,234,867,439,572,351,619,101,961,828,871,27,321,607,417,658,934,981,504,394,263,554,718,896,67,722,966,315,365,565,184,291,303,750,407,551,57,826,662,669,133,402,632,300,666,49,231,14,616,439,42,507,736,771,127",
    "166,632,104,988,90,170,930,210,634,938,873,269,7,143,235,639,585,534,105,440,732,536,662,504,367,678,776,484,168,194,776,304,474,279,309,321,248,769,347,300,892,418,454,558,828,816,765,901,825,481,89,25,23,127,972,912,570,374,606,726,428,348,427,802,349,383,174,691,868,516,297,774,528,759,608,140,487,104,55,984,229,310,321,821,827,70,248,800,150,647,500,380,947,176,597,178,824,256,629,199,724,687,803,885,209,315,392,742,184,693,253,161,948,708,598,831,410,46,905,717,633,618,698,420,379,27,629,787,836,865,814,457,779,515,642,653,110,855,143,75,76,546,261,966,748,266,798,738,103,757,355,662,602,203,299,556,790,326,764,644,121,527,124,868,420,145,112,218,50,482,788,953,948,636,659,339,414,213,649,79,413,396,693,920,755,448,826,476,811,904,297,432,137,142,78,988,537,450,0,788,577,789,894,950,944,738,32,866,979,815,416,87,982,435,714,953,344,31,765,959,770,720,636,773,318,826,483,393,731,518,137,377,792,996,553,424,613,751,229,548,337,589,784,505,252,702,28,53,405,420,325,545,792,681,965,502,602,969,489,739,965,575,588,644,441,993,593,647,534,247,551,148,584,462,808,663,276,486,77,859,991,560,36,869,199,79,666,45,865,293,53,139,372,194,117,904,471,603,63,109,694,317,437,614,177,722,356,31,480,390,217,440,682,114,882,438,380,626,479,772,785,106,73,571,449,995,653,472,810,9,761,923,498,263,753,824,604,597,556,45,647,893,718,718,598,139,486,959,889,852,473,531,198,822,788,370,224,136,461,799,251,928,9,120,57,2,272,73,701,41,477,696,743,726,392,120,388,421,91,634,108,850,36,42,409,330,915,843,317,778,591,282,926,373,374,458,571,411,314,182,874,450,459,401,225,495,603,310,895,627,204,239,18,13,567,661,908,175,123,543,776,83,203,550,697,392,313,816,327,422,69,526,200,443,383,265,166,140,73,669,420,723,941,962,83,740,42,910,394,280,606,504,568,761,863,321,324,887,871,189,471,876,501,928,182,778,21,414,187,151,163,73,964,449,254,121,825,899,254,845,689,769,775,445,3,92,403,20,493,28,177,94,741,860,722,147,160,410,580,720,735,806,319,798,297,49,778,876,588,745,247,83,189,585,12,590,540,368,449,940,187,523,612,401,498,834,423,594,886,529,793,818,293,440,532,734,660,778,469,931,724,200,950,588,436,465,545,163,184,690,488,100,108,530,899,227,886,896,474,882,854,192,255,58,130,147,631,485,960,493,498,796,25,126,418,910,453,540,776,393,416,738,106,100,745,999,42,882,4,874,607,831,576,91,598,797,693,854,59,497,112,397,415,481,825,210,677,269,322,57,433,707,402,510,781,755,694,700,466,414,731,712,224,156,975,941,511,758,451,306,801,997,478,732,189,472,835,531,536,788,574,664,325,379,606,759,680,897,939,397,735,652,139,831,354,675,768,857,707,43,937,712,972,371,502,97,138,888,755,117,304,800,603,447,232,43,72,607,35,284,993,429,520,571,165,934,636,899,417,890,960,727,664,810,660,566,633,826,729,491,581,834,843,887,874,212,880,718,193,144,432,896,377,299,197,98,804,891,560,540,832,465,358,659,247,436,338,430,750,803,854,753,423,636,661,677,677,945,848,242,908,326,199,10,469,464,387,728,498,744,453,42,325,514,41,87,143,917,329,830,731,527,416,642,8,641,843,362,732,794,93,384,304,79,823,728,831,598,48,500,356,683,37,998,578,313,959,284,179,972,703,356,778,249,68,915,259,250,363,570,48,493,88,92,900,593,35,416,410,182,756,387,968,166,483,224,636,40,943,522,676,830,271,678,137,174,111,712,255,68,481,371,639,969,278,426,241,740,486,87,972,720,940,567,498,425,949,253,546,694,991,646,389,508,812,387,807,797,512,41,440,245,615,894,606,591,15,535,837,255,721,565,139,90,241,141,948,128,57,245,774,18,352,569,27,804,669,443,376,34,35,500,345,836,274,462,777,864,565,152,414,418,557,38,217,471,713,789,365,265,553,484,198,739,125,925,777,475,888,506,706,384,450,855,996,345,52,451,742,146,529,253,795,248,451,79,557,441,140,497,622,249,427,455,853,130,464,960,493,958,920,151,456,868,326,69,931,472,827,214,298,950,297,866,726,81,39,551,756,35,746,394,713,504,400,530,208,871,280,206,940,209,581,674,14,863,770,125,472,621,105,526,320,830,298,568,173,304,386,67",
    "208,471,967,80,514,962,214,591,688,996,689,374,105,143,378,959,695,959,14,485,290,912,310,950,900,824,783,289,815,91,630,701,234,931,190,573,630,381,652,996,499,656,107,85,343,542,525,980,321,903,255,822,234,541,732,539,452,385,399,376,707,756,141,240,254,904,379,849,296,870,706,813,426,17,766,340,461,743,669,946,601,969,568,381,493,675,734,952,843,141,95,621,942,431,7,290,545,486,595,714,848,589,331,887,78,294,244,844,813,49,430,385,98,763,569,209,980,395,389,961,982,970,96,7,927,854,494,753,883,149,365,336,713,356,219,603,143,125,58,692,617,188,495,250,411,749,649,443,432,113,42,751,656,245,804,45,630,936,429,236,423,426,382,20,613,256,326,519,341,811,441,285,253,963,849,39,239,250,482,399,746,95,562,983,459,31,275,642,926,401,114,528,484,240,329,722,417,296,788,0,360,55,873,613,163,82,374,55,176,315,436,579,87,745,476,924,42,359,140,4,352,891,13,406,89,721,187,863,705,122,66,470,775,94,150,919,266,159,622,102,617,131,61,936,994,376,895,868,264,641,865,815,421,80,255,743,895,207,932,46,929,47,153,226,642,483,245,813,928,506,582,957,199,66,502,859,147,98,728,968,33,920,655,300,680,853,328,940,282,346,393,72,946,326,516,75,154,178,669,376,748,283,185,460,286,524,314,181,904,546,911,595,640,853,825,209,161,277,316,89,860,826,697,566,794,554,426,830,135,628,641,345,129,418,846,99,165,646,815,884,501,436,676,13,948,48,551,99,315,90,30,839,653,420,646,19,846,61,65,165,687,922,612,836,723,828,768,686,579,824,732,98,806,801,615,241,405,761,160,925,679,25,225,184,956,372,59,158,273,732,103,754,43,79,830,90,248,527,875,849,485,152,13,730,515,516,840,561,271,430,499,771,467,818,457,439,750,104,184,291,79,442,431,421,59,669,686,224,221,795,336,634,641,698,51,769,887,946,774,880,330,646,214,128,300,207,437,870,483,260,958,519,983,656,223,221,2,539,436,370,725,767,553,373,190,214,672,504,104,119,774,941,182,305,611,280,471,305,540,755,28,701,62,212,178,637,632,214,815,221,459,604,417,676,861,706,630,133,986,563,521,855,249,380,767,271,996,654,538,379,286,62,492,825,966,583,108,813,259,860,546,753,144,227,610,574,954,650,380,306,166,141,817,416,235,98,755,724,100,897,997,531,537,529,699,948,601,604,804,242,526,666,888,472,796,969,474,76,604,672,892,349,141,748,598,728,115,136,257,517,215,354,450,542,191,517,182,209,736,683,721,199,577,160,59,164,193,276,541,21,936,695,528,897,185,441,669,915,925,256,850,721,164,633,5,770,46,880,795,888,107,616,227,451,16,448,294,247,908,413,722,52,576,354,894,776,640,201,332,552,994,850,415,750,663,871,12,170,103,658,104,420,686,825,952,772,544,406,631,221,113,700,588,106,959,213,23,827,369,19,559,310,793,64,628,77,802,236,757,217,278,603,180,277,34,281,209,950,78,498,841,294,805,298,601,564,333,692,217,827,965,741,227,502,889,147,545,994,139,37,666,170,577,89,250,29,677,209,425,545,854,394,236,544,447,270,805,901,655,292,581,73,932,112,362,25,784,647,854,716,780,8,446,75,37,865,532,818,473,279,510,422,512,198,227,485,243,668,491,996,306,331,240,672,754,973,350,271,350,335,724,323,566,200,353,657,14,444,696,434,392,294,534,135,443,64,642,384,116,153,954,808,16,817,959,645,701,657,29,623,313,576,479,273,675,424,378,365,90,120,754,80,624,47,986,509,296,330,289,754,124,77,680,321,624,697,97,72,324,318,488,388,403,222,960,895,657,247,785,970,804,219,230,714,741,9,132,427,199,205,598,861,837,560,686,562,985,899,545,954,422,838,8,297,127,463,78,740,953,282,609,643,406,703,518,931,190,818,617,637,709,148,640,613,131,470,427,330,739,143,77,209,994,20,132,480,48,865,666,120,974,275,813,294,725,57,357,727,542,248,792,922,865,256,34,948,866,913,484,367,940,916,861,656,19,893,633,199,39,888,683,350,933,925,400,201,282,731,47,588,241,909,333,807,603,124,165,802,634,6,987,559,480,444,304,119,422,216,756,380,377,155,619,305,168,79,505,913,369,159,883,620,743,261,581,155,112,496,65,221,608,1000,639,970,767,553,953,901,332,636,712,760,396,290,838,256,498,239,156,286,389,471,249,765,451,222,483,438,561,48",
    "178,639,837,886,739,72,210,875,682,663,547,38,905,498,944,939,691,726,935,352,88,419,670,750,215,238,783,555,471,375,574,798,409,757,382,301,949,678,43,520,342,991,106,74,141,764,665,350,789,234,786,584,269,288,232,640,852,29,671,32,711,299,16,121,923,173,627,588,992,816,751,416,127,547,635,213,613,938,117,20,646,484,928,649,227,497,751,939,102,985,840,574,654,717,124,756,76,682,593,51,734,990,781,312,966,563,549,912,900,188,391,960,261,361,827,71,391,244,281,786,691,904,402,802,12,781,927,609,128,330,607,772,545,641,718,8,562,271,540,794,581,702,85,529,410,837,540,814,886,836,200,232,740,77,25,716,679,251,195,876,211,923,199,949,406,638,358,570,346,370,277,552,683,75,760,323,985,364,596,506,32,178,920,91,175,489,463,373,795,587,869,664,150,469,197,437,146,852,577,360,0,661,950,862,414,559,859,161,219,401,377,569,233,315,459,834,494,137,404,365,768,154,629,496,538,207,504,494,271,67,608,26,895,539,110,738,57,41,803,652,250,717,930,214,348,338,341,238,78,217,561,116,331,589,850,9,683,941,965,527,446,800,874,140,297,287,59,969,988,506,692,199,128,859,992,212,923,618,125,754,511,577,836,300,195,169,75,108,914,821,692,479,522,502,594,252,432,13,653,510,641,530,610,87,372,597,742,373,66,911,440,383,434,147,698,799,614,899,728,444,602,98,949,650,661,830,45,441,983,324,177,325,764,106,787,431,463,390,835,478,668,895,146,51,147,872,954,492,172,179,562,304,501,845,456,643,941,53,135,423,181,974,766,110,568,332,663,487,999,385,356,887,572,332,635,874,667,988,493,406,886,346,706,44,727,477,631,453,558,643,781,991,230,657,240,732,854,343,580,595,438,636,614,63,68,963,329,45,968,92,288,487,660,413,647,699,342,228,429,214,338,149,184,399,933,403,436,377,523,542,836,604,618,387,751,86,291,140,853,979,587,783,158,160,157,613,294,752,722,46,134,862,846,686,648,942,542,574,788,880,169,89,351,306,402,217,448,497,767,255,48,741,349,839,805,34,390,862,585,481,825,703,779,19,477,507,522,470,528,663,734,562,422,151,85,479,997,193,999,226,38,106,120,420,37,508,984,877,475,977,774,196,856,916,616,897,14,952,788,360,105,803,897,463,470,240,520,592,606,282,429,663,176,202,44,286,539,394,753,178,127,356,580,547,855,593,837,921,328,769,925,323,828,691,104,336,115,247,519,148,804,946,841,864,886,920,627,699,294,768,323,194,383,498,185,575,225,545,765,705,716,516,703,790,139,309,854,577,985,309,664,747,97,338,334,871,915,869,221,536,399,416,366,23,238,665,464,182,622,769,438,412,273,881,248,130,263,559,884,319,943,604,217,412,294,798,495,428,226,872,198,144,216,713,518,130,838,4,558,4,434,775,800,533,936,750,516,630,124,742,676,404,413,978,201,227,557,209,607,699,311,548,940,377,470,240,554,255,948,350,632,139,709,339,179,268,525,395,40,311,621,911,604,565,968,489,799,347,994,230,285,382,635,881,853,348,314,748,68,531,824,835,187,159,336,37,783,761,954,967,554,601,794,815,864,84,203,467,511,770,257,87,493,306,436,455,584,965,867,993,904,630,892,642,448,892,155,705,151,360,293,808,398,558,457,655,324,583,612,453,636,348,278,617,985,811,283,994,208,71,999,278,28,301,550,937,132,860,782,328,141,616,753,693,728,467,590,111,313,113,810,574,343,753,47,178,906,450,116,123,568,668,630,485,868,813,248,568,857,416,565,783,111,615,652,167,541,907,879,247,86,554,995,859,803,664,692,897,116,405,948,254,980,684,336,557,364,713,670,670,867,595,887,402,423,921,296,731,295,784,77,907,369,845,880,837,676,961,79,416,652,606,324,413,732,992,359,522,457,979,960,790,177,348,227,308,591,875,494,335,943,488,772,804,539,505,765,671,894,372,211,19,980,273,798,534,604,5,678,916,133,791,130,493,229,948,352,838,724,77,785,253,10,571,980,823,854,853,20,663,221,932,479,192,45,817,319,350,897,780,137,53,543,269,543,702,946,584,238,131,661,313,542,541,253,166,81,586,509,515,874,116,318,710,459,136,812,324,532,753,599,14,7,630,538,292,549,389,656,164,788,249,590,214,939,754,12,776,656,909,648,866,335,697,234,577,691,328,679,469,919,162,31,997,882,341,931,981,858,880,488,24,157,173,903,175,215,492,792,216",
    "634,273,653,457,357,777,51,959,306,465,480,735,122,710,1,628,642,319,1,407,866,73,459,577,240,91,307,867,841,228,287,610,339,677,445,709,831,640,378,831,628,99,424,195,429,504,134,10,272,983,500,219,328,420,115,857,308,244,864,470,612,758,995,314,947,889,401,917,648,571,44,301,636,437,457,973,30,813,114,214,267,963,438,233,404,409,694,515,544,609,886,304,329,314,839,641,385,277,250,998,387,996,535,107,307,4,459,840,286,972,211,708,468,616,84,739,635,766,507,545,968,792,359,145,577,532,823,681,709,257,110,678,723,587,519,255,814,439,150,347,947,505,391,893,388,302,132,234,724,76,538,736,594,973,475,98,227,706,421,400,677,185,886,910,157,990,943,316,470,538,215,834,608,131,444,829,355,123,249,216,18,835,490,859,987,497,685,556,629,877,367,914,320,722,729,98,915,392,789,55,661,0,973,386,208,432,20,809,860,287,431,921,941,842,689,956,69,856,373,968,131,174,179,160,340,966,636,592,28,410,150,580,40,109,130,181,844,1,46,617,379,212,862,29,42,823,553,399,452,567,797,950,797,832,352,393,745,230,830,866,759,472,470,53,643,454,166,833,701,318,928,617,996,615,624,739,214,403,559,892,817,27,65,301,423,70,173,416,793,5,576,225,17,881,721,644,616,77,434,290,643,944,769,152,448,243,744,292,440,774,545,105,406,205,685,709,148,157,755,682,268,238,268,377,845,202,99,852,900,94,792,400,395,481,10,916,650,578,656,427,707,809,6,100,844,126,747,517,527,76,751,477,301,925,262,772,941,158,542,602,117,715,199,654,56,172,379,120,346,857,348,466,572,123,722,3,152,126,472,466,419,649,503,846,417,540,431,237,347,434,209,55,364,620,342,925,577,558,161,542,143,905,729,873,859,423,726,831,183,54,318,624,14,392,189,112,482,672,890,165,520,517,255,247,960,873,304,545,858,839,464,212,785,735,205,208,985,843,184,567,75,159,629,665,596,937,64,359,252,646,610,520,173,612,646,362,53,456,246,279,84,216,401,573,754,860,329,416,168,623,77,775,101,727,202,73,659,313,935,799,396,454,451,881,333,620,976,667,455,818,626,891,131,203,844,728,710,398,861,577,181,798,852,369,644,888,805,575,398,523,310,464,420,349,577,732,896,168,212,450,91,547,776,977,543,287,583,435,246,880,464,993,83,966,975,350,327,197,945,820,146,263,634,794,418,345,603,529,884,605,478,763,337,100,273,108,989,691,222,920,488,434,280,246,832,867,475,745,480,709,579,57,937,668,567,268,680,131,482,487,278,639,62,187,852,829,490,956,859,253,843,524,949,74,414,914,818,707,870,441,506,777,113,192,798,641,510,544,528,886,136,773,558,284,480,460,620,721,146,219,360,647,957,754,906,99,311,990,995,967,582,462,131,550,118,361,731,780,725,370,34,221,840,457,714,80,769,5,811,319,223,388,428,589,602,813,566,104,155,262,24,948,196,696,293,406,379,451,291,400,598,830,137,719,676,314,184,127,867,823,293,528,749,451,442,141,840,50,741,72,590,970,322,402,565,223,59,795,238,587,809,721,549,835,6,132,866,337,823,929,967,685,369,808,14,362,894,836,551,976,183,671,159,321,981,699,685,475,308,611,657,717,647,276,729,229,986,268,123,652,783,846,569,710,420,845,371,298,735,667,217,500,986,433,338,602,73,969,954,231,444,304,141,439,390,212,272,952,779,539,482,350,837,392,244,182,610,155,189,232,409,186,273,292,611,578,336,841,252,550,291,963,806,221,341,641,798,70,121,697,623,525,281,888,447,345,26,976,815,677,79,185,578,478,9,213,33,679,512,488,723,708,809,538,640,539,704,241,344,471,755,680,765,591,551,78,427,559,883,894,289,617,643,400,663,51,808,392,501,193,347,117,981,457,813,792,593,19,987,785,505,645,105,895,159,776,737,712,706,919,786,395,747,540,702,771,315,101,357,512,420,52,190,341,445,946,352,709,508,346,408,415,235,341,340,287,840,784,664,246,44,720,707,490,284,311,610,307,372,480,338,238,194,424,522,390,539,435,155,966,360,643,817,865,748,35,768,749,355,227,133,85,462,14,40,674,664,684,921,78,339,817,149,556,705,657,385,225,676,166,401,625,895,457,572,578,227,553,645,137,847,173,38,683,374,930,299,52,487,138,683,862,137,332,718,519,735,448,641,252,429,329,574,194,304,240,668,98,463,813,233,392,103,882,883,279,203,602,270,361",
    "557,142,656,333,806,127,932,158,567,603,769,954,454,458,554,169,220,11,695,319,181,73,242,281,561,982,225,940,151,424,447,89,81,95,750,258,315,927,825,400,922,735,160,682,953,221,514,368,838,665,858,629,994,479,996,372,752,420,233,633,85,110,878,682,590,443,552,350,104,915,500,538,170,346,436,748,679,874,217,704,163,339,15,998,637,285,756,864,154,644,142,472,542,276,197,976,444,134,522,635,209,501,336,377,561,441,275,125,295,683,210,810,708,966,753,916,911,806,713,521,410,80,730,788,718,994,314,691,699,659,80,620,381,831,596,417,290,546,403,864,631,497,133,679,695,925,806,556,929,324,527,422,222,840,586,922,728,989,930,200,578,27,612,190,750,297,107,762,304,724,288,662,231,666,78,548,58,756,296,259,925,436,707,615,144,602,977,958,425,814,709,369,951,959,83,881,864,13,894,873,950,973,0,385,396,95,847,119,938,339,315,893,101,117,107,512,914,910,597,842,787,396,164,169,712,150,373,106,636,305,94,11,478,191,694,262,6,94,578,831,942,270,678,736,849,419,387,662,582,585,371,963,144,570,147,847,614,312,567,258,856,908,706,711,124,787,163,790,166,806,454,93,300,222,557,40,341,872,879,733,939,927,233,634,851,438,212,988,406,516,205,816,743,619,613,156,330,37,506,660,212,661,970,902,87,680,690,482,955,588,918,444,863,315,925,75,157,195,978,286,186,771,650,79,324,713,857,792,348,449,814,909,905,207,287,701,275,612,21,592,718,440,771,711,953,78,883,564,840,56,764,303,418,505,59,206,867,980,591,146,50,428,406,768,363,900,481,989,491,415,21,746,5,106,876,20,479,301,626,352,439,871,235,484,243,318,496,727,346,622,586,803,95,984,286,307,414,701,691,286,425,218,383,346,852,375,589,390,239,744,336,212,368,121,103,584,819,713,653,970,177,669,910,151,681,798,279,23,480,815,483,951,948,545,490,255,862,89,920,436,570,572,623,691,50,260,504,512,205,644,155,834,415,279,908,806,377,244,935,823,709,759,128,908,844,548,313,357,70,6,965,854,562,20,949,570,793,417,569,749,905,106,50,107,841,107,764,847,458,263,479,283,100,346,754,792,650,499,300,446,318,224,771,925,171,380,71,626,41,772,670,946,917,387,247,330,781,446,828,434,866,474,72,690,8,763,917,328,422,122,773,820,704,768,546,851,631,212,130,581,637,73,73,646,106,925,926,795,958,277,977,235,146,722,282,900,442,782,713,612,603,704,595,421,978,347,316,654,381,807,262,390,436,796,691,392,961,653,73,837,416,83,289,481,191,627,487,524,522,549,749,821,118,668,227,296,494,224,701,80,70,618,633,839,952,703,30,57,322,209,791,11,130,199,150,299,505,519,492,80,466,858,719,784,639,818,258,180,733,33,119,522,626,554,5,897,688,145,941,959,443,825,658,345,287,189,93,105,718,190,498,936,951,555,406,677,483,303,328,967,680,243,179,492,775,577,673,562,594,979,816,558,167,945,297,102,463,969,163,508,132,531,349,87,83,976,410,457,441,44,633,86,19,664,211,43,284,937,883,18,549,793,608,878,114,133,931,261,386,946,465,287,375,344,61,377,234,639,30,289,179,532,411,586,808,677,147,200,445,118,897,461,629,256,533,199,411,666,843,999,999,450,635,198,554,256,660,398,884,136,264,474,106,364,276,226,793,600,885,538,783,458,195,95,937,503,872,210,944,968,810,195,378,520,820,717,258,908,59,403,41,460,6,48,894,996,631,247,66,475,882,547,861,739,30,192,501,640,462,690,605,66,476,234,362,978,845,357,113,111,860,956,395,878,77,294,338,31,735,941,621,406,710,132,892,886,747,10,882,558,757,531,252,504,189,504,163,527,396,732,143,109,674,417,641,167,952,421,336,679,7,74,597,43,713,923,596,619,846,49,675,770,568,456,465,530,757,110,753,682,441,671,289,846,862,164,922,910,719,587,965,990,714,500,758,951,933,256,28,313,334,600,765,568,710,110,667,135,541,178,798,477,346,69,529,233,543,533,494,241,111,928,328,758,670,334,584,909,459,658,503,719,438,992,819,279,915,395,956,244,305,749,305,628,495,294,347,867,891,298,433,779,130,828,907,898,534,25,577,102,812,902,126,388,845,513,243,249,611,238,410,447,706,988,250,209,158,386,640,224,755,407,294,16,199,610,371,65,933,514,161,525,427,174,966,6,791,542,739,394,496,637,690,871,910,246,398,639,453,224",
    "595,763,465,603,963,261,163,278,447,70,287,340,331,409,368,732,671,963,414,955,887,53,382,887,30,872,187,676,337,983,674,347,620,139,295,469,331,612,968,749,211,228,158,873,311,711,295,520,926,641,630,899,622,519,636,153,187,734,218,269,81,968,439,618,175,312,42,115,948,712,176,962,244,274,641,715,853,800,427,585,832,950,818,413,237,194,811,895,927,455,781,709,316,3,17,532,324,694,718,266,460,490,19,903,325,854,598,671,703,510,811,120,992,389,530,26,197,228,507,966,997,622,291,97,78,727,48,199,787,510,271,556,968,309,223,922,10,928,487,528,250,551,434,20,107,484,574,544,713,660,493,714,355,425,744,456,20,941,67,395,314,530,79,501,392,210,342,428,989,17,947,402,950,455,424,333,512,719,226,799,483,228,239,317,911,669,451,405,313,928,588,11,695,583,868,954,435,790,950,613,862,386,385,0,545,543,777,614,564,616,341,976,426,969,700,157,892,165,269,147,975,821,253,602,583,112,981,823,956,218,122,450,126,143,446,439,891,462,366,68,717,215,899,420,555,197,11,602,371,856,345,375,988,289,321,527,82,815,845,985,323,98,36,875,89,108,365,502,490,98,169,981,830,873,945,9,334,382,707,383,429,991,368,651,860,810,243,43,808,239,702,68,470,105,746,561,445,401,9,309,608,764,744,962,600,541,547,780,391,156,503,721,15,607,916,131,842,383,645,462,677,546,128,685,65,675,977,985,93,910,185,996,474,591,810,299,257,319,264,906,899,696,497,355,11,464,666,405,837,906,216,348,887,14,164,83,171,79,74,439,666,345,712,665,126,194,860,986,264,181,402,73,138,920,373,138,507,732,204,747,779,751,795,3,517,988,427,300,612,634,842,36,94,210,409,847,85,715,709,447,872,179,467,20,644,682,95,993,417,300,577,433,552,84,211,16,383,195,205,992,981,141,165,429,475,769,646,850,105,142,707,325,778,759,291,124,720,563,872,515,652,927,226,304,302,974,303,84,232,734,226,463,678,419,705,445,797,771,124,903,423,108,87,530,823,649,295,177,778,398,916,105,287,710,770,952,395,237,223,60,345,433,585,894,669,702,233,780,400,93,649,683,985,124,846,299,439,814,786,850,745,979,200,253,530,924,223,439,742,905,703,268,420,981,226,18,154,767,437,19,917,534,74,75,434,111,277,594,504,591,744,974,183,250,992,765,261,332,540,887,460,884,606,80,488,178,662,762,461,820,676,875,408,347,950,203,148,357,824,591,256,915,207,390,696,196,237,800,685,134,55,135,770,703,124,715,824,942,383,957,78,687,728,75,335,467,552,641,655,208,787,693,707,890,508,278,212,471,526,57,110,940,922,259,743,926,875,487,487,47,399,372,10,9,466,736,871,66,879,537,880,933,977,402,284,368,670,288,799,199,841,423,252,336,401,102,359,833,93,91,894,484,11,90,149,998,275,539,979,174,640,935,446,792,162,734,139,970,529,522,356,573,257,431,858,215,319,577,268,299,866,95,226,480,67,457,994,865,549,234,167,742,494,788,94,835,205,981,100,252,827,532,572,34,816,357,74,438,522,713,62,590,956,985,2,16,719,984,8,733,559,937,380,455,350,792,962,677,34,391,718,121,988,668,116,573,160,729,428,264,624,544,787,731,465,684,362,396,691,57,269,851,455,484,712,54,501,436,819,916,289,754,257,531,595,958,171,557,526,716,147,358,605,508,656,591,32,959,331,205,421,114,655,63,353,96,451,957,222,291,463,517,979,201,30,981,959,51,420,713,290,182,371,253,179,434,78,971,26,54,76,109,429,289,592,371,757,118,747,48,570,299,780,266,128,169,779,208,679,42,142,889,677,631,607,883,776,146,750,745,356,222,842,400,433,271,580,131,951,342,832,814,763,453,405,778,363,211,731,160,751,623,248,188,861,467,679,490,727,926,530,459,890,180,355,562,768,445,304,987,728,839,305,488,851,118,580,422,733,59,35,627,695,946,500,449,965,75,992,955,631,107,691,631,25,519,761,143,507,929,454,480,834,852,283,781,883,416,942,673,618,832,532,531,597,401,640,27,450,195,606,875,924,750,297,872,386,707,815,599,843,144,381,497,224,632,325,145,710,657,686,308,828,333,368,323,533,978,883,882,214,708,422,523,758,41,506,78,603,552,947,377,840,996,126,56,299,582,5,927,336,260,70,68,337,355,413,629,804,334,560,270,656,914,19,862,979,972,929,168,792,537,975,913,43,476,516,682,691,444",
    "969,708,154,737,754,532,617,218,804,140,418,158,189,623,718,275,755,27,904,808,163,50,273,713,661,56,724,333,879,388,903,821,450,758,760,762,65,933,484,815,673,384,355,597,264,465,493,98,308,354,214,11,697,851,379,545,321,297,812,996,571,570,707,473,551,163,171,605,829,820,751,518,821,686,551,540,630,641,127,466,590,9,94,873,53,707,606,667,98,372,976,287,998,984,856,230,711,488,163,891,214,591,4,624,189,726,291,38,923,132,432,6,376,134,68,229,75,680,998,386,381,722,622,965,275,984,746,186,244,262,182,50,965,567,894,128,353,610,540,947,184,379,892,559,207,683,337,86,74,516,222,836,484,357,241,843,698,828,227,370,177,543,145,825,999,841,770,531,60,86,849,705,586,255,966,79,219,884,667,739,750,788,94,308,536,466,945,332,930,813,607,544,93,773,549,366,269,893,944,163,414,208,396,545,0,437,431,269,511,253,246,956,560,974,635,221,766,816,36,952,958,987,740,583,199,121,5,126,736,792,413,671,576,596,347,235,635,957,746,735,706,158,164,760,144,649,907,138,736,313,293,478,223,675,855,295,385,521,392,764,173,94,736,959,986,645,937,73,32,274,262,897,637,751,721,943,561,420,992,663,285,47,22,890,258,885,339,556,309,412,345,343,312,801,55,67,72,996,265,271,624,237,856,822,868,861,243,838,225,776,957,187,844,572,859,773,119,625,499,959,694,951,543,732,237,907,802,456,847,870,535,600,350,78,328,359,249,350,374,933,800,90,816,516,206,847,528,180,314,737,268,664,123,586,483,476,94,50,940,593,928,193,491,889,777,461,850,320,898,89,632,563,121,386,577,957,407,626,859,157,390,387,557,785,107,869,377,205,392,806,973,99,231,467,2,769,392,813,246,363,54,923,896,889,631,860,78,693,159,873,882,847,651,853,230,302,184,842,922,831,243,310,241,655,783,210,672,707,511,537,507,144,887,392,251,586,806,886,929,207,517,708,253,563,733,518,662,385,636,503,704,494,761,835,857,875,515,235,541,905,709,286,126,115,417,920,522,243,632,387,725,174,267,102,874,363,892,438,805,285,411,153,699,334,654,732,57,846,840,336,225,250,182,53,586,216,471,995,712,763,605,859,841,101,78,697,556,134,931,914,416,420,195,29,928,530,167,693,981,929,341,301,377,445,403,415,337,71,754,715,565,547,96,156,598,594,141,678,531,253,830,311,132,446,524,952,330,796,61,723,214,41,515,495,394,33,477,589,438,849,573,837,975,24,398,221,386,202,362,752,936,679,198,366,570,800,171,316,130,358,562,154,7,50,439,359,130,486,466,650,151,800,780,978,201,40,777,456,763,160,650,598,919,724,927,525,555,453,901,108,416,486,714,811,239,606,933,741,537,774,365,635,933,930,771,626,514,871,343,823,699,686,153,942,76,533,574,636,643,969,193,46,609,937,516,249,630,52,51,261,792,877,589,841,301,712,495,769,744,459,452,919,354,674,20,339,311,799,102,299,634,875,733,527,388,669,105,467,959,173,365,436,787,189,947,962,71,520,382,467,499,248,281,804,182,557,862,376,70,422,396,403,103,243,613,835,574,274,29,342,38,996,684,529,453,820,99,39,773,251,915,459,45,943,691,126,82,238,927,30,48,665,167,192,78,587,46,407,166,720,296,624,648,512,715,312,899,740,526,41,518,364,464,552,881,668,868,403,69,839,353,911,774,478,7,380,59,561,190,292,58,857,5,608,697,930,254,181,35,876,238,753,2,512,212,803,283,503,246,119,538,136,215,951,753,967,578,224,651,362,890,591,159,293,277,273,658,376,424,424,721,484,174,764,815,754,244,847,457,463,470,552,38,975,264,15,653,346,986,205,949,104,300,183,862,561,557,665,679,329,545,567,413,170,455,788,106,762,735,451,147,273,771,192,210,901,112,384,574,415,925,372,372,200,189,647,254,407,963,504,242,146,926,759,799,28,602,170,929,364,867,12,371,580,220,338,220,231,879,755,172,742,17,913,965,230,680,343,417,24,436,206,330,654,235,525,968,556,448,825,725,476,647,253,889,926,304,542,296,25,92,989,375,23,470,178,437,536,561,609,193,502,159,216,851,692,958,195,431,399,92,341,697,313,300,214,86,195,646,824,486,107,747,770,199,16,202,564,520,912,316,108,296,509,425,690,782,542,381,489,165,8,856,855,739,208,331,405,227,905,541,540,959,45,688,556,118,349,764,161,808,417,112,68,860,543,220,147,807,479",
    "968,694,321,985,213,979,662,106,698,509,185,99,635,637,69,248,236,20,912,426,771,140,883,454,119,832,827,552,442,807,381,538,674,125,32,976,978,908,7,615,46,443,273,107,146,898,686,518,400,314,953,135,77,362,285,59,647,451,753,550,91,518,944,911,106,53,766,282,37,361,546,710,569,217,643,163,785,794,25,731,520,560,8,214,7,518,398,669,893,540,486,521,240,173,98,306,83,671,286,831,640,676,449,31,441,47,90,406,793,975,532,527,881,533,403,872,250,79,747,901,160,257,392,75,950,641,717,220,92,222,723,621,409,495,34,532,222,527,62,734,115,115,986,500,481,789,485,390,376,346,325,610,637,844,11,200,899,347,129,898,511,794,748,874,352,255,993,74,55,333,443,317,650,600,877,690,542,476,780,540,382,549,565,839,876,677,325,305,96,15,842,674,881,447,921,958,742,12,738,82,559,432,95,543,437,0,491,269,650,228,111,175,380,157,12,322,280,462,838,991,457,500,812,600,434,812,525,773,450,75,76,885,857,901,848,435,947,313,703,207,824,967,379,845,86,312,812,278,940,421,196,547,845,765,578,414,606,7,808,128,499,546,666,57,201,915,401,212,164,732,65,910,574,723,135,948,356,210,493,504,195,446,494,740,233,37,401,392,928,637,739,402,815,577,340,5,220,282,93,475,837,687,488,508,584,337,319,96,877,45,659,647,157,254,558,501,372,825,34,768,28,527,667,703,520,737,525,841,689,371,174,566,363,363,402,812,340,493,12,536,958,460,240,691,54,198,240,685,875,730,318,690,736,827,783,958,887,88,289,644,774,804,209,283,34,532,476,994,260,972,279,12,14,862,859,611,622,588,279,964,305,16,278,722,490,823,485,830,126,516,951,891,942,352,880,558,772,432,510,25,624,694,773,665,346,821,489,145,817,901,68,437,805,5,532,266,18,603,622,855,507,418,349,795,832,495,912,415,267,167,620,912,355,276,937,714,378,15,954,630,478,267,267,516,550,98,384,197,906,268,249,43,899,428,332,713,839,889,775,210,942,421,278,608,523,402,559,212,134,164,654,358,977,440,730,330,86,653,298,18,147,410,197,611,250,66,739,659,493,354,73,48,650,784,794,155,524,608,423,581,704,74,548,944,643,791,61,799,734,179,499,913,960,529,961,861,982,397,519,601,669,980,55,403,333,281,204,457,271,198,131,480,572,997,416,920,8,496,705,184,979,548,890,389,734,279,115,672,672,281,587,444,26,482,428,36,414,728,673,474,966,974,843,582,426,293,316,663,920,126,936,634,51,367,833,340,828,307,163,297,819,270,423,450,833,310,835,18,905,963,795,772,769,9,181,116,148,990,224,911,774,73,609,191,457,127,158,962,390,508,452,737,337,213,426,903,356,834,777,663,660,268,887,116,691,475,559,130,815,907,478,269,278,92,582,243,611,173,716,817,409,782,65,138,195,905,187,52,985,129,586,19,636,75,906,334,954,71,501,48,172,354,271,421,333,890,770,758,716,820,901,712,499,591,859,704,364,413,356,764,985,857,197,954,935,207,913,32,545,642,268,261,807,10,913,822,781,219,145,48,619,152,898,826,557,169,111,819,898,130,170,516,793,946,685,830,481,346,245,826,360,308,829,914,456,160,990,317,892,183,744,595,63,886,3,664,377,408,173,491,740,372,915,443,605,775,561,426,910,211,226,217,982,330,182,251,494,124,563,313,834,72,504,448,647,520,928,471,671,89,350,609,970,971,237,108,626,790,213,29,730,43,25,136,390,19,321,258,237,172,788,796,213,102,704,649,181,742,233,515,133,886,311,666,958,393,759,7,28,129,555,41,770,861,596,755,812,402,227,144,753,216,921,683,215,150,709,124,979,967,876,665,859,793,385,914,841,171,696,97,69,419,520,542,241,793,230,870,649,758,380,285,796,500,852,595,987,186,888,236,978,377,28,872,922,379,150,482,119,65,736,749,734,182,213,937,91,324,939,966,893,83,346,732,513,914,597,810,487,932,245,5,608,204,965,902,100,401,308,982,685,798,398,565,733,15,584,425,279,324,859,619,469,598,149,141,96,493,553,481,901,858,676,593,510,888,520,10,957,943,560,589,389,590,820,188,828,353,528,359,707,346,316,799,624,166,152,232,240,702,251,473,736,704,566,636,240,402,769,784,567,98,445,286,799,76,532,898,127,461,950,981,134,869,457,586,422,172,637,823,663,788,389,954,289,466,508,879,768,755,848,844,796,9,291,955,68,672,808,733",
    "384,395,722,546,427,344,588,749,943,83,62,414,672,56,96,427,832,694,820,552,574,490,412,760,906,729,310,727,650,279,747,182,217,471,915,309,568,489,720,93,837,668,835,172,938,568,731,761,825,461,471,564,134,735,278,71,721,422,765,587,818,504,721,395,509,129,664,137,15,2,452,545,557,911,164,874,908,530,589,443,609,829,526,656,478,490,925,559,92,148,666,380,48,449,256,547,292,677,571,237,339,294,567,480,37,802,717,328,517,184,284,268,90,535,940,236,177,380,944,996,85,906,783,559,983,953,843,466,41,203,845,883,564,298,398,903,619,963,816,35,95,224,384,13,276,987,951,40,732,239,560,77,66,100,6,856,214,384,52,467,486,647,999,366,23,42,679,877,924,116,643,563,16,753,47,366,159,820,713,260,760,655,245,491,58,276,727,433,875,229,222,34,62,624,141,448,489,236,32,374,859,20,847,777,431,491,0,697,160,124,526,624,607,369,103,534,916,332,50,294,895,140,285,186,375,83,646,469,170,505,444,247,587,424,773,267,457,245,653,62,828,795,918,822,585,257,920,552,735,582,84,308,910,150,237,188,714,208,989,646,193,977,902,172,428,536,55,961,597,102,137,299,151,11,399,852,582,839,393,365,789,907,184,528,166,224,940,150,574,257,495,842,896,263,216,871,909,297,660,500,290,459,344,858,643,375,139,370,953,30,924,818,4,947,286,710,271,735,44,895,534,423,372,813,831,452,569,666,148,779,477,382,968,769,256,705,454,653,393,125,612,11,770,674,85,162,829,778,810,830,397,123,478,265,143,190,414,733,950,412,4,536,795,559,150,139,40,631,856,828,623,883,650,196,994,812,859,562,200,344,411,132,793,675,916,230,724,288,560,960,788,498,995,151,981,212,532,590,615,964,667,744,480,607,15,78,367,790,767,902,263,979,917,596,651,600,269,811,520,881,49,766,353,288,783,635,548,434,277,732,418,804,813,327,462,859,961,596,145,944,161,124,237,394,423,692,564,335,56,491,510,650,440,668,117,277,889,167,113,75,505,142,784,635,594,147,461,455,13,960,717,576,966,376,970,825,637,206,730,644,83,350,47,376,71,854,206,948,564,111,361,175,955,870,564,902,725,507,60,728,596,264,128,956,485,317,959,257,801,86,36,409,430,641,812,521,780,714,502,383,191,502,657,338,931,111,208,957,587,749,397,116,693,642,85,278,480,700,666,910,511,578,791,274,568,661,572,870,209,170,265,978,27,816,265,98,386,282,420,80,222,156,14,141,779,497,229,960,184,595,759,957,954,217,450,809,419,205,805,730,550,539,241,318,292,102,169,137,471,435,814,866,740,69,540,732,470,451,43,361,638,142,910,959,744,459,693,446,985,62,477,799,589,768,342,372,64,718,477,392,414,884,186,650,98,155,654,764,258,338,654,370,11,147,750,312,333,268,170,189,840,542,866,413,722,887,406,55,840,149,58,720,150,369,237,208,445,872,81,684,692,338,428,975,430,479,770,416,992,227,414,460,670,840,49,914,963,12,745,99,444,77,858,852,762,845,807,773,248,415,529,528,340,230,760,323,165,943,874,759,503,887,815,501,719,595,553,660,531,577,383,227,723,901,710,150,332,398,62,482,358,165,348,29,73,260,414,218,338,825,691,52,835,707,272,103,173,86,751,983,248,858,76,644,252,264,255,201,488,847,38,335,437,109,138,135,104,917,982,527,386,959,221,90,121,318,446,751,42,255,945,990,179,640,360,99,851,322,371,906,681,128,971,769,14,560,823,370,61,789,208,614,839,540,182,730,61,906,221,217,681,731,582,546,398,99,643,743,707,158,604,863,783,583,248,425,832,549,24,815,107,139,699,478,377,819,759,621,44,564,597,418,794,153,104,247,363,727,438,465,970,273,174,406,599,765,616,546,245,426,24,64,826,231,998,657,666,405,365,579,27,756,649,134,397,108,185,986,234,317,576,698,497,133,555,976,886,277,776,974,966,635,422,33,810,425,453,180,5,217,260,521,758,87,98,391,53,362,4,459,956,941,826,417,269,369,957,903,218,543,620,402,263,231,730,155,569,214,694,864,39,546,521,647,864,496,988,988,727,891,138,112,570,38,231,110,221,844,640,775,460,16,610,628,998,847,89,239,354,907,654,362,951,720,860,508,861,137,737,220,324,465,587,573,380,643,422,854,226,334,916,167,728,613,959,84,721,755,427,339,549,597,580,700,636,13,256,912,365,262,608,771,132,773,224,887,389,253,246,503",
    "682,53,553,479,819,263,139,847,273,122,9,601,219,732,452,593,486,15,662,722,903,319,132,113,202,903,341,17,949,521,187,102,751,719,148,719,978,341,551,382,822,337,850,561,416,459,354,690,180,384,53,899,364,333,892,29,677,453,815,183,344,998,880,733,440,636,132,950,544,182,340,536,965,86,320,35,566,595,809,356,251,352,802,518,69,156,280,187,684,732,308,870,907,741,66,501,125,85,473,280,985,143,711,928,995,509,323,606,676,95,562,230,115,472,296,191,960,316,313,783,882,434,189,886,18,826,604,544,624,594,635,595,135,838,747,983,743,437,747,143,518,217,125,93,919,762,837,734,561,363,159,144,535,605,342,538,220,734,475,100,45,727,963,177,363,144,472,915,859,12,360,923,962,663,244,906,461,225,140,630,305,404,730,758,208,558,548,957,555,932,858,687,315,340,431,123,323,290,866,55,161,809,119,614,269,269,697,0,328,415,304,305,870,745,836,77,393,364,899,890,90,613,634,242,925,41,604,462,461,811,948,373,330,225,343,105,441,941,308,791,285,576,481,636,219,142,120,322,289,221,867,913,386,838,529,6,887,676,736,76,653,282,298,110,445,12,244,521,59,153,225,470,599,245,186,776,627,901,989,551,870,523,421,950,554,469,155,557,219,543,832,798,106,873,105,734,553,280,445,598,938,948,353,174,582,69,58,620,140,153,456,824,95,756,233,101,431,695,82,599,255,132,100,623,840,245,825,604,497,388,733,354,800,185,718,250,752,229,826,414,512,205,222,66,556,250,420,684,455,802,809,555,111,416,902,408,596,996,363,967,779,736,688,275,582,934,990,480,539,676,894,859,963,293,25,407,682,669,545,378,650,957,233,597,856,444,400,322,274,968,325,584,166,396,169,411,137,669,15,286,308,645,833,59,899,623,456,527,897,215,954,635,516,540,19,616,956,269,812,764,183,971,607,803,347,272,306,530,881,754,231,826,473,627,100,339,509,16,686,714,138,605,86,186,288,357,292,24,882,104,979,345,97,418,533,577,771,959,44,733,458,834,483,786,492,885,194,512,302,221,359,34,396,957,336,314,582,714,57,565,27,865,584,699,933,410,356,403,21,500,763,56,97,656,99,843,943,921,22,742,575,938,859,494,113,26,130,367,585,954,362,78,594,921,426,166,609,433,287,812,563,643,301,585,102,345,707,320,457,981,998,303,478,761,682,327,562,846,624,487,95,872,740,523,402,745,388,189,720,52,801,813,830,718,234,940,416,349,957,496,283,636,319,491,575,992,696,437,8,100,235,907,109,89,630,562,787,514,601,920,651,51,801,709,768,521,968,480,278,844,379,618,217,298,592,522,863,562,485,992,792,778,152,884,939,821,213,834,785,8,601,395,550,849,735,861,257,512,825,23,118,660,727,600,242,937,735,117,1000,536,801,521,22,634,873,47,708,602,910,955,190,788,364,738,261,386,834,184,89,675,617,438,10,809,684,892,688,801,572,2,105,164,553,782,72,431,841,924,668,855,588,621,610,370,55,448,558,891,138,63,352,585,530,305,368,570,883,41,7,690,555,70,171,151,510,79,889,529,945,740,206,617,121,337,365,722,271,175,312,848,82,192,559,942,979,513,478,423,712,117,272,780,250,272,968,163,950,186,39,466,988,80,37,147,662,250,4,769,799,55,312,358,599,768,815,651,127,104,760,728,286,843,374,328,981,938,244,576,267,222,265,895,318,125,872,373,534,630,531,503,723,254,953,613,376,687,697,565,49,813,146,463,52,189,912,152,288,180,785,296,232,387,789,781,766,926,128,887,487,791,272,172,713,442,572,849,800,60,51,991,271,628,222,701,234,11,348,372,644,112,267,291,742,623,606,452,50,444,191,757,376,862,457,494,748,546,22,200,638,427,943,872,125,296,355,517,950,304,274,816,335,156,577,435,96,668,302,108,879,498,365,336,39,670,736,363,108,845,643,50,718,633,179,205,989,631,20,917,237,812,133,294,220,527,84,234,812,185,226,191,968,387,115,20,933,741,808,518,226,686,969,666,967,129,732,404,499,506,210,324,280,155,332,766,184,593,610,267,934,163,357,534,928,445,324,986,639,329,480,883,320,292,891,500,174,722,42,297,348,133,365,89,229,539,903,263,315,631,992,75,80,807,784,524,177,708,176,677,426,299,732,960,815,33,540,186,859,340,834,361,719,500,773,304,955,569,680,778,837,924,436,794,875,889,413,919,849,339,771,118,816,746,139,339,54,859,557,428,877,356",
    "558,976,402,222,981,451,152,63,255,85,606,618,199,322,398,256,206,137,780,389,146,836,35,141,830,445,21,424,271,318,811,544,90,153,367,195,626,153,635,778,877,33,824,908,351,392,987,462,751,678,603,496,993,82,932,398,93,327,516,730,916,682,906,139,69,289,853,314,44,425,996,963,977,309,233,434,262,950,614,852,628,903,404,792,276,777,424,458,738,38,622,756,32,890,563,696,92,736,56,247,673,53,913,209,445,856,892,402,510,96,686,54,337,345,836,635,604,845,380,226,521,625,27,348,246,7,737,242,536,96,62,816,849,507,816,91,857,728,41,568,669,871,914,448,915,704,243,394,81,354,33,864,867,838,32,43,278,226,599,665,262,22,28,698,738,58,767,781,528,677,867,428,82,332,459,537,856,192,22,585,764,116,369,67,549,294,233,305,514,949,36,757,213,352,496,908,986,447,979,176,219,860,938,564,511,650,160,328,0,381,665,153,486,30,330,560,124,216,809,86,593,867,596,948,589,167,40,906,51,134,600,78,404,570,597,577,750,782,488,145,69,925,271,581,344,884,817,765,988,285,928,129,310,777,337,889,116,786,831,708,111,75,523,305,778,468,334,437,392,214,830,772,837,829,160,881,188,118,472,914,816,336,404,311,56,194,615,84,747,897,823,15,92,431,218,999,103,135,266,249,97,922,121,647,548,277,381,565,181,646,843,906,475,709,251,940,797,941,330,246,970,822,806,75,352,173,549,142,60,292,110,376,532,11,351,956,53,377,871,815,464,548,243,502,751,462,675,372,659,877,723,298,842,669,710,635,905,253,279,162,644,38,911,108,629,69,83,701,269,811,102,130,183,302,574,884,259,320,537,621,283,716,583,594,531,298,118,55,288,382,293,550,570,31,950,28,337,776,566,855,611,763,533,606,328,642,994,831,631,688,597,35,492,553,572,250,363,786,62,290,339,93,66,617,962,507,559,907,789,702,361,570,73,357,488,248,5,150,973,354,301,925,429,971,533,175,401,641,736,247,348,461,299,42,777,147,947,705,78,361,749,668,528,873,258,732,933,975,932,446,466,165,693,279,471,265,682,941,158,61,403,530,419,883,270,42,334,51,230,150,505,447,59,682,340,895,686,234,833,40,253,890,554,866,585,950,745,232,398,520,607,309,977,772,577,870,594,946,528,164,919,381,732,983,343,249,975,666,65,336,590,516,569,58,703,257,371,619,875,978,156,963,305,724,807,595,359,250,150,638,510,732,441,63,54,94,643,376,593,36,254,773,329,794,294,350,206,172,646,867,698,368,64,871,536,837,542,2,617,403,32,851,935,407,804,650,917,4,840,311,794,937,644,719,850,589,305,993,100,624,260,317,493,552,814,787,645,225,354,625,88,775,682,718,421,894,332,478,103,330,839,849,365,702,502,617,601,815,802,895,479,539,300,435,293,235,174,18,796,23,718,349,240,759,211,248,421,47,959,250,879,759,792,479,424,796,71,219,21,692,851,245,447,174,408,516,734,75,581,776,647,503,691,619,506,236,804,790,807,841,159,903,988,803,536,508,734,12,75,554,52,313,530,220,88,818,205,246,835,858,541,979,228,572,885,83,642,104,568,890,800,815,202,317,244,44,638,683,529,493,123,173,150,73,592,137,324,685,201,131,468,862,610,220,330,647,826,200,131,552,776,582,560,420,731,843,446,668,175,571,248,372,613,327,564,55,851,450,479,75,436,673,48,299,283,384,576,267,880,883,220,883,166,460,846,753,143,227,183,759,247,706,140,183,611,80,150,180,312,616,583,465,238,970,385,497,534,409,454,270,775,693,180,453,81,232,822,611,309,519,865,409,657,786,508,860,373,554,949,795,509,285,257,80,828,104,418,211,655,992,328,606,761,40,698,400,47,978,355,424,711,898,224,254,88,162,18,326,433,682,739,310,613,348,204,214,731,311,870,195,193,499,173,713,342,105,149,298,406,918,107,961,348,413,139,542,191,276,251,899,168,538,78,767,444,328,430,483,241,379,640,621,518,670,797,607,764,450,648,794,20,871,209,327,585,381,36,324,650,405,963,163,403,192,240,927,590,970,999,197,293,317,695,626,383,194,57,366,437,503,257,791,183,920,457,347,470,938,346,962,587,252,992,515,85,261,671,763,822,17,691,350,309,385,537,295,69,792,958,628,569,906,815,512,535,933,730,922,37,14,628,285,931,391,908,805,712,559,172,837,389,949,307,674,581,285,663,620,567,118,638,935,318,591,340,979,768,529,448,159",
    "298,778,768,881,661,602,417,667,287,655,582,473,360,616,354,150,10,457,29,252,461,328,773,72,862,113,852,79,211,697,185,857,609,411,862,828,526,435,356,406,873,842,852,721,239,282,857,476,665,319,737,23,859,101,116,614,878,941,521,856,724,297,771,599,131,395,426,872,854,252,129,450,754,268,899,328,55,744,325,78,387,851,573,998,284,885,465,953,589,300,48,969,542,368,647,299,981,27,620,572,378,162,370,470,582,965,670,830,250,513,145,823,165,78,973,419,837,649,445,169,603,858,771,159,553,391,489,256,359,776,25,396,899,57,679,699,364,340,926,912,43,173,148,935,389,850,988,635,31,51,850,17,891,175,728,603,108,192,956,656,548,408,564,573,953,531,629,290,341,847,305,441,59,306,626,783,107,944,683,563,578,486,414,527,372,991,730,832,747,914,48,223,63,853,976,81,529,569,815,315,401,287,339,616,253,228,124,415,381,0,267,529,681,825,418,357,754,208,528,678,465,272,232,203,462,43,797,320,436,575,355,414,736,249,849,669,633,858,799,876,771,203,394,630,911,376,557,253,357,733,913,381,59,371,217,565,820,565,515,526,885,383,944,214,861,99,555,466,604,677,588,891,67,380,811,190,9,179,396,670,324,319,575,37,300,498,21,688,539,119,993,448,358,495,101,530,743,775,184,903,394,732,510,11,530,352,737,404,805,153,404,845,341,357,599,922,4,289,51,734,628,38,465,613,522,825,555,499,538,512,440,378,516,143,507,947,815,522,488,949,881,985,763,297,748,581,330,471,374,493,596,851,218,649,711,180,22,754,933,518,475,694,431,848,507,213,415,5,100,48,579,23,532,273,980,200,106,234,511,4,276,889,263,748,281,192,152,203,848,791,142,848,147,411,896,86,108,309,553,916,946,663,744,574,812,670,964,71,293,247,524,553,860,453,666,197,553,658,173,666,397,452,887,777,195,82,216,44,331,746,355,388,469,104,413,834,570,805,870,769,510,924,477,989,628,459,387,36,158,574,450,900,466,264,329,96,238,330,617,371,925,204,231,581,535,952,745,260,170,113,685,744,804,154,980,389,834,958,334,965,29,118,864,687,26,398,626,939,100,109,463,542,512,576,966,248,324,381,675,732,33,99,301,636,129,13,863,969,387,499,293,638,883,360,969,760,699,117,430,937,210,542,169,654,796,505,849,474,549,610,827,377,264,586,225,337,172,261,695,989,832,143,940,117,583,151,220,445,863,222,11,815,364,242,408,849,933,55,327,399,802,121,855,65,713,633,423,259,510,289,64,431,400,96,781,866,563,75,179,936,485,453,288,641,201,680,352,20,543,418,826,80,366,303,641,889,31,803,688,176,186,565,448,566,725,833,452,979,802,244,427,447,236,943,511,826,742,132,749,48,517,105,917,29,403,900,505,222,815,593,88,405,887,903,869,89,282,719,926,515,350,442,971,455,207,629,336,101,903,912,671,829,502,739,602,486,834,981,31,111,366,657,644,216,124,410,800,219,508,677,328,363,371,903,576,206,95,73,647,335,386,225,216,19,178,594,612,179,559,803,362,183,784,903,319,841,59,87,931,990,691,697,638,973,646,773,184,772,438,316,935,403,292,785,259,204,85,137,300,410,29,735,912,637,443,850,202,410,612,371,86,691,277,832,353,899,990,669,503,6,216,693,18,77,974,94,954,991,844,675,387,966,627,654,262,5,734,420,462,823,552,75,213,369,628,100,185,237,165,680,212,62,903,71,346,101,693,512,488,385,663,392,431,846,443,954,725,574,452,915,464,232,409,216,303,534,260,317,15,5,192,425,225,509,976,545,507,887,506,681,57,14,215,848,434,419,844,993,824,918,649,940,938,165,566,223,451,632,403,888,725,812,726,706,887,897,551,485,155,910,172,704,222,667,862,104,996,59,379,609,714,703,255,565,714,492,8,581,666,696,320,163,201,513,457,644,366,874,473,236,990,899,626,811,116,196,189,633,278,761,866,706,829,525,249,324,394,435,572,197,447,542,53,848,156,260,405,340,954,812,238,478,200,55,111,582,295,219,594,156,597,952,710,19,820,473,934,751,4,852,477,170,550,214,146,298,251,837,98,524,41,916,249,572,630,34,409,497,553,153,236,790,582,278,763,372,464,596,555,505,140,896,57,112,192,233,457,738,372,513,743,178,806,799,821,697,678,250,500,692,351,767,846,378,507,502,206,219,819,290,458,444,658,25,256,235,907,545,820,617,980,9,221,674,388,699,884,987,576,802",
    "726,842,262,247,158,48,852,580,147,328,451,248,362,476,191,117,776,66,348,153,369,843,880,774,140,486,941,377,593,625,985,634,888,955,658,762,935,877,768,231,748,437,53,865,510,952,485,982,81,658,598,609,922,683,617,767,991,864,779,18,356,339,682,424,297,362,633,955,58,598,564,830,371,559,277,161,915,298,94,278,629,778,42,803,886,438,314,19,766,76,3,679,175,324,593,341,481,391,585,156,311,838,210,366,856,786,916,518,728,580,677,481,954,581,633,222,677,272,916,905,630,71,12,836,966,331,96,709,244,100,853,176,191,736,222,796,701,827,509,94,232,176,176,339,202,582,346,624,741,895,283,891,615,722,56,866,581,440,798,943,878,112,900,825,772,642,163,891,237,220,348,817,757,912,612,386,759,229,280,679,593,953,810,464,840,311,840,598,465,910,484,574,665,222,933,799,498,755,416,436,377,431,315,341,246,111,526,304,665,267,0,621,692,913,633,514,557,54,267,517,710,544,951,101,432,401,24,108,202,84,452,819,134,588,751,922,159,7,411,796,554,619,171,181,56,540,354,277,737,745,79,873,257,4,175,697,326,266,79,40,970,112,516,208,476,817,793,991,166,71,885,79,264,926,771,635,235,566,975,98,636,447,570,148,165,301,389,73,801,165,541,581,148,545,252,220,236,740,456,656,56,474,797,556,28,895,591,12,379,361,5,166,361,495,640,745,248,140,643,215,544,22,884,959,534,608,459,250,359,440,914,697,604,973,735,73,360,43,290,204,252,686,689,488,314,368,184,175,846,771,869,232,877,134,404,510,447,953,494,859,868,962,800,832,894,790,913,440,46,443,954,879,777,833,526,517,34,150,372,861,618,725,385,257,659,640,149,524,799,101,448,859,927,304,642,256,368,446,967,885,409,516,416,745,82,368,155,202,212,155,415,487,169,668,271,172,448,619,462,827,140,922,789,471,605,220,76,38,712,437,994,367,49,334,691,194,160,694,336,951,334,262,688,109,817,909,879,122,485,128,509,984,141,714,270,394,378,187,14,314,913,177,564,666,492,92,886,720,866,67,635,532,109,590,587,708,203,595,575,405,540,386,198,427,4,459,552,824,892,588,550,146,228,91,965,560,451,825,286,623,870,903,317,739,506,669,365,740,939,883,261,514,720,417,384,143,378,876,588,783,303,65,52,644,250,341,76,92,613,346,398,197,998,274,529,83,776,922,573,428,312,196,639,443,283,446,678,258,197,409,870,349,93,762,362,208,60,400,242,159,642,774,363,366,859,498,595,678,101,278,216,990,281,363,36,339,938,213,636,377,403,749,462,928,219,215,561,711,886,270,806,275,541,904,747,749,967,39,635,333,693,85,566,14,912,712,631,536,383,749,922,190,337,979,576,673,833,483,745,248,962,209,184,563,82,268,455,291,608,300,591,699,722,539,213,735,578,707,412,94,230,564,825,338,271,396,752,850,546,90,794,945,884,386,923,903,653,186,967,498,630,668,679,387,232,712,541,288,680,666,349,95,89,52,830,504,403,670,829,771,97,147,902,353,312,417,703,21,669,41,665,454,216,300,297,335,442,709,383,947,61,605,644,991,870,12,310,312,622,303,18,433,818,244,439,796,674,424,90,206,469,315,78,147,84,373,831,471,725,919,641,545,467,320,452,123,638,753,267,41,144,254,861,789,831,559,643,552,375,192,85,54,76,642,261,390,762,52,240,596,202,329,170,428,534,656,363,527,45,524,485,966,747,588,495,225,436,925,336,638,183,762,965,993,116,981,784,933,88,117,5,232,999,249,752,942,542,287,760,115,757,485,229,329,14,416,646,745,580,793,410,458,613,562,196,788,648,813,378,812,481,563,269,163,823,201,993,241,695,410,887,258,213,184,473,747,216,972,24,62,104,701,320,645,414,765,165,672,203,909,100,222,687,841,742,21,629,397,249,774,489,251,19,329,878,313,384,565,987,547,143,872,590,206,840,248,653,717,422,161,890,681,453,612,326,278,531,409,492,541,355,731,891,258,894,679,816,772,340,329,335,3,113,822,152,60,204,873,983,473,513,959,731,324,703,383,40,214,286,495,987,87,247,636,97,40,432,391,387,648,332,707,775,799,474,598,464,105,344,499,754,273,847,75,598,903,995,929,251,20,308,855,20,407,378,126,839,913,923,814,643,251,169,778,163,507,934,759,833,637,762,912,30,566,60,723,908,126,477,750,87,997,831,400,939,685,930,201,593,525,904,451,844,209,477,109,623,334,190,964",
    "899,967,257,954,123,408,326,959,195,177,450,789,368,823,385,205,648,239,175,299,23,948,962,525,681,578,864,13,522,710,302,983,437,886,824,327,662,349,298,761,556,319,4,87,4,85,475,979,165,566,729,724,713,697,739,681,235,641,843,149,520,487,864,986,569,485,258,958,147,750,342,655,499,564,580,742,556,564,265,480,695,998,935,851,635,136,678,581,680,647,77,460,391,901,835,526,499,62,971,32,316,967,913,342,476,13,20,635,932,234,17,237,185,681,271,522,793,597,222,366,860,363,528,802,612,221,674,220,440,405,997,935,776,546,116,462,733,165,938,302,58,36,274,795,390,316,370,888,143,607,464,63,701,680,649,854,182,26,628,475,244,52,279,59,686,127,564,237,530,261,239,474,586,768,551,530,326,126,140,614,852,306,334,700,188,642,157,94,73,362,940,264,426,488,700,183,985,869,87,579,569,921,893,976,956,175,624,305,153,529,621,0,449,747,788,697,323,184,343,925,539,620,353,267,137,628,925,487,627,725,414,218,303,894,332,460,319,461,212,448,674,678,50,446,479,731,548,556,572,768,136,878,15,601,105,515,544,751,196,371,434,852,650,792,764,648,31,443,242,796,634,439,202,297,539,685,851,107,766,193,198,134,950,920,321,293,84,394,197,221,513,751,942,234,459,614,64,190,450,828,683,709,219,292,920,557,597,271,330,838,432,808,131,152,647,354,297,421,657,984,169,396,355,293,827,185,900,322,779,666,496,141,778,75,887,799,781,912,173,932,355,340,792,66,793,609,24,93,626,441,301,98,282,767,768,972,909,493,192,743,405,258,203,949,420,330,204,698,367,769,351,437,624,68,709,597,274,873,778,726,944,815,138,975,350,474,158,166,345,588,497,359,42,916,350,302,870,626,914,476,739,630,278,232,676,273,193,173,372,705,489,799,911,234,411,144,892,570,177,845,115,340,219,624,543,334,721,688,580,547,516,270,422,875,71,163,974,168,622,649,55,389,1000,519,254,433,50,969,132,834,633,55,18,499,820,512,167,371,617,336,596,249,957,362,277,596,814,402,993,123,287,298,216,782,665,815,797,500,644,188,724,879,359,305,482,216,448,223,543,158,51,413,150,397,954,820,377,190,537,285,792,848,904,98,44,595,932,839,462,699,807,827,280,539,342,761,449,746,479,582,951,723,621,419,320,3,753,650,232,160,546,332,900,197,801,861,32,592,647,650,757,205,866,152,592,570,611,968,750,572,217,533,960,307,258,879,340,548,464,872,372,29,620,374,912,339,554,357,973,632,813,207,830,551,625,820,354,753,163,285,369,539,451,163,224,65,927,888,118,880,222,966,691,651,621,606,932,575,500,603,920,445,136,51,924,635,927,152,371,860,690,444,763,780,18,192,272,680,353,6,703,485,416,42,219,532,52,387,950,149,362,918,5,814,654,610,741,844,480,21,742,89,641,53,767,61,288,367,366,946,621,443,647,973,650,653,986,722,703,310,216,531,905,439,164,193,236,999,718,124,551,122,730,221,265,342,512,868,377,48,950,517,958,834,957,945,162,823,579,713,404,181,717,129,998,908,342,182,978,546,556,774,149,974,287,327,860,65,908,769,293,515,919,717,467,795,906,326,416,568,887,901,367,278,192,337,805,419,824,312,555,939,648,28,364,553,499,862,264,795,991,278,660,788,24,75,829,860,901,27,346,284,544,579,247,183,208,554,51,329,602,819,667,856,13,404,918,378,837,632,741,751,574,919,330,748,666,800,343,809,642,181,724,927,813,99,732,661,662,379,167,81,224,460,88,856,519,679,422,364,95,140,9,878,389,439,176,635,381,184,381,751,573,493,338,661,527,444,227,765,850,818,976,976,796,912,686,890,297,79,767,666,331,570,262,3,860,949,554,526,232,1000,520,801,972,722,174,386,563,484,141,435,494,970,28,407,414,679,82,105,109,760,340,585,976,86,583,405,329,185,694,352,366,271,19,887,373,382,375,783,659,566,924,178,268,615,382,845,194,254,778,42,95,238,781,950,436,935,447,681,759,378,681,168,762,600,670,696,180,562,135,904,800,581,979,97,47,492,436,267,318,431,767,500,136,962,924,663,896,501,626,880,187,446,326,525,116,894,610,907,513,901,891,119,17,410,826,134,18,569,712,389,207,759,281,890,462,129,594,670,748,217,764,346,99,935,52,149,284,146,944,536,197,232,781,178,517,685,924,261,988,875,307,730,93,326,668,102,374,810,692,517,256,733,339,845,203,807,267,185",
    "444,875,941,128,21,122,445,648,163,197,475,754,711,651,742,323,169,552,724,914,535,441,339,400,790,287,447,374,37,937,70,967,440,476,364,783,790,573,564,30,314,535,701,193,298,994,966,752,100,896,135,675,939,2,98,338,664,673,640,361,863,53,136,358,198,934,933,463,945,172,68,934,40,346,535,355,742,397,500,409,118,454,45,423,171,852,785,264,422,758,479,999,37,458,840,40,705,557,354,459,929,837,32,723,82,195,148,792,753,482,910,804,489,586,787,689,8,622,918,768,381,615,681,626,237,908,940,676,893,397,710,993,805,787,297,55,954,368,412,599,101,571,444,967,234,195,128,966,266,891,865,944,267,206,60,314,424,334,500,675,978,417,199,701,763,813,729,538,106,373,527,838,11,979,655,169,117,46,723,757,54,301,418,994,912,395,195,626,744,225,654,537,119,680,510,792,216,996,982,87,233,941,101,426,560,380,607,870,486,681,692,449,0,205,914,184,337,391,685,429,467,846,649,720,85,802,878,665,20,260,718,970,205,307,32,75,597,127,844,996,305,366,976,706,65,82,903,462,341,779,582,496,201,105,58,733,66,856,575,365,917,187,6,839,448,770,340,954,26,483,382,963,207,5,32,707,815,112,581,216,63,480,571,894,293,353,90,54,435,667,981,680,44,835,954,710,818,729,378,537,495,354,75,237,833,473,685,779,468,59,880,584,369,413,695,644,327,608,121,212,986,471,361,90,219,223,382,928,452,16,411,359,780,897,962,694,735,696,187,351,889,970,341,932,170,527,160,649,972,186,965,31,918,243,393,590,597,480,585,921,615,5,649,915,825,888,789,621,189,281,427,780,573,731,438,592,589,405,584,882,118,277,812,970,626,440,587,756,775,135,273,922,966,410,286,351,291,869,934,363,16,310,990,576,455,511,862,708,49,551,629,409,471,727,502,252,155,104,510,906,155,376,916,135,677,708,300,937,481,330,606,844,88,993,862,148,422,978,755,527,248,784,451,730,209,700,146,287,717,493,432,500,371,160,742,95,207,49,755,155,831,651,347,655,164,862,857,946,186,395,164,220,183,332,266,203,21,70,360,303,57,369,465,653,191,439,89,644,541,361,299,592,927,6,175,222,814,198,400,508,59,978,187,55,595,80,569,116,375,280,352,283,928,679,824,664,348,724,31,850,8,904,246,620,408,127,531,671,465,196,234,455,546,105,817,265,773,586,673,882,624,190,885,559,57,901,758,760,131,741,305,410,659,584,452,695,445,568,282,376,717,954,622,650,989,610,234,635,17,314,107,770,758,911,395,179,389,548,827,400,223,928,997,894,932,247,755,916,236,610,765,209,355,119,286,747,7,413,414,924,377,979,618,178,852,798,89,587,970,641,340,518,947,598,172,941,157,734,19,260,999,899,559,778,299,902,487,869,487,622,462,467,464,493,410,539,924,369,21,974,619,237,17,706,436,149,110,583,389,402,438,922,626,475,780,717,251,520,462,708,549,509,280,849,883,376,712,598,859,878,332,73,387,798,40,563,622,471,869,192,870,430,907,394,531,604,62,832,9,967,930,698,866,290,968,631,79,983,315,25,73,716,958,400,289,407,236,916,511,647,165,186,727,264,26,557,310,143,292,165,608,458,209,823,97,39,523,645,362,406,394,436,852,270,266,942,590,255,417,82,419,349,3,7,728,826,598,275,754,684,758,456,498,579,200,381,147,919,822,478,967,142,632,878,697,599,543,858,105,626,886,327,677,590,989,879,16,403,852,547,738,568,813,124,9,28,618,560,837,279,71,379,600,956,302,68,366,470,944,314,267,216,880,501,824,43,651,69,273,646,336,377,244,825,321,777,712,722,128,340,229,218,884,309,392,45,595,311,595,862,58,729,808,924,384,732,614,942,770,972,835,955,171,517,243,553,269,34,781,517,631,427,66,857,264,83,855,666,749,494,324,613,296,651,303,672,834,367,831,812,932,293,502,180,387,968,25,283,170,861,307,463,912,701,951,648,294,110,141,827,304,275,180,784,65,168,827,224,572,716,74,612,931,977,658,648,53,734,654,303,453,718,788,311,596,944,828,66,228,319,861,126,282,569,111,352,311,258,682,739,647,136,834,816,925,307,615,299,318,134,786,936,103,800,443,475,235,178,319,299,623,11,898,883,988,173,660,160,580,640,543,414,71,877,811,962,916,289,229,853,315,62,951,160,292,626,633,321,888,140,472,532,309,615,555,149,479,140,954,339,619,492,122,717,645,189,627,543,691,200",
    "420,668,118,478,946,297,917,234,166,911,717,567,736,177,801,794,636,777,245,703,362,685,54,15,155,474,417,75,229,959,765,53,994,712,235,926,843,388,212,582,713,264,166,946,242,832,706,821,602,539,990,602,786,696,581,731,927,726,515,11,77,107,648,308,601,96,225,467,50,856,628,155,500,411,365,911,919,72,591,666,685,588,80,121,672,277,28,983,137,713,881,133,267,345,598,213,961,465,289,647,981,33,811,655,111,166,556,433,225,395,917,423,526,923,776,1,252,449,509,460,2,437,115,146,105,230,979,416,562,861,73,138,715,330,538,838,543,169,17,270,582,942,718,127,653,938,938,967,799,325,364,293,478,873,833,713,890,375,787,320,828,956,587,453,776,162,977,854,495,912,508,927,842,444,357,910,339,210,331,545,934,286,189,865,729,662,447,303,364,837,759,501,470,131,358,697,566,202,435,745,315,842,117,969,974,157,369,745,30,825,913,747,205,0,862,60,571,602,313,681,645,13,422,357,842,336,644,807,985,273,663,954,56,970,320,934,290,921,214,668,88,659,748,559,657,209,505,292,731,717,659,991,644,660,14,295,118,534,528,987,846,698,611,725,848,554,935,550,170,624,185,30,674,941,183,751,906,406,861,878,914,646,760,887,265,611,843,405,499,766,724,597,253,748,290,335,24,341,620,246,495,383,893,26,929,439,524,486,366,716,895,723,162,608,605,688,525,728,228,865,540,19,377,701,77,746,343,142,931,683,876,419,77,27,81,245,873,796,609,297,766,860,257,924,137,448,495,50,262,448,94,280,556,976,526,896,937,472,945,360,450,129,584,201,775,428,241,891,560,133,549,661,781,701,104,888,813,288,701,215,191,378,756,69,570,231,123,596,61,119,906,378,527,375,159,499,925,47,367,411,438,893,158,815,496,337,916,132,661,631,491,105,338,966,576,847,642,687,434,503,115,709,653,931,537,289,238,562,603,950,545,716,222,893,214,852,945,402,659,904,187,141,261,349,555,67,594,905,268,478,864,833,123,785,142,500,986,184,868,377,987,503,476,17,737,426,775,677,146,283,990,984,183,784,534,382,857,653,378,466,567,797,219,159,92,425,67,292,80,951,638,776,114,296,184,615,607,35,975,101,677,40,161,69,212,101,86,676,704,584,137,326,740,830,208,784,293,408,86,923,938,58,403,929,900,944,419,177,101,337,867,480,877,941,375,948,708,100,368,812,730,341,958,936,158,307,19,689,960,767,317,553,923,38,684,556,33,902,682,359,596,870,906,352,595,893,511,74,664,606,810,942,792,143,217,613,493,588,893,623,141,130,25,626,494,889,645,511,929,504,939,921,876,282,700,480,394,595,913,48,476,233,623,619,10,122,458,997,429,436,905,19,997,320,395,680,214,244,124,343,364,463,729,752,631,364,47,942,869,692,466,719,6,31,750,459,174,775,834,893,880,114,4,978,646,300,129,470,285,223,476,89,553,113,517,601,769,933,201,544,61,456,889,907,714,63,526,739,244,76,246,297,828,382,190,658,164,886,544,610,365,697,598,70,542,712,672,127,262,326,102,467,227,789,499,77,504,879,267,542,965,337,901,135,68,975,44,744,70,857,774,840,882,280,862,262,342,124,53,546,919,790,593,398,598,165,459,546,162,853,872,137,103,174,637,779,27,481,719,827,668,394,188,595,617,681,568,164,887,243,577,372,977,624,587,895,396,737,91,25,510,689,939,443,818,144,661,372,474,745,291,115,30,667,66,832,677,267,636,541,422,106,152,866,544,32,452,637,233,724,283,744,457,166,345,113,243,253,256,197,103,703,172,733,607,480,995,758,262,524,372,442,395,902,421,69,289,448,456,262,789,731,547,755,573,623,898,970,855,1,276,454,786,565,264,304,172,199,733,856,795,410,936,717,410,271,704,245,627,345,256,540,753,157,461,293,424,586,266,16,628,960,952,650,859,310,131,347,841,909,316,185,993,893,15,507,129,34,980,810,527,977,140,148,886,688,981,611,431,234,203,220,716,228,497,794,264,205,490,437,135,856,270,671,839,491,155,977,408,538,264,602,330,84,542,718,275,191,993,753,55,142,133,240,720,105,437,645,834,268,697,740,109,602,240,640,981,470,807,770,215,258,439,611,610,846,400,244,618,696,533,556,944,606,767,633,263,164,142,253,836,368,491,317,530,584,703,787,628,704,298,367,211,511,684,780,972,640,978,323,292,554,604,871,933,7,354,35,489,886,746,455,204,691,242,659,963,657,153,690",
    "481,980,51,249,140,161,924,62,594,890,851,381,232,177,431,59,65,226,246,564,18,479,810,344,263,338,410,873,765,58,261,108,817,687,446,230,547,530,122,168,762,930,777,977,1,846,21,358,429,551,208,97,952,486,17,309,396,928,157,718,633,629,674,593,680,135,101,306,990,478,453,727,182,7,767,814,438,448,368,317,117,650,884,24,783,732,586,984,277,977,779,234,965,462,167,268,177,516,512,700,2,855,756,838,837,245,535,497,603,825,258,929,15,943,195,499,169,758,320,531,233,461,726,650,639,515,913,905,857,480,114,880,19,438,689,118,802,539,907,256,534,576,597,184,994,170,959,203,953,113,734,412,218,979,429,345,194,104,856,295,301,825,829,173,800,690,33,688,358,931,548,401,119,446,329,178,968,895,799,128,642,948,194,935,160,210,412,81,458,567,220,208,781,191,512,500,743,385,714,476,459,689,107,700,635,12,103,836,330,418,633,788,914,862,0,205,734,427,730,989,314,513,586,962,837,65,55,101,374,463,355,370,303,598,771,990,794,835,293,624,968,242,692,791,698,660,38,823,681,372,916,831,808,292,413,70,485,297,949,584,585,791,265,765,985,683,808,472,980,521,490,143,739,412,256,606,848,360,903,926,617,737,299,667,116,219,747,963,365,815,458,697,981,534,502,55,350,76,166,10,774,279,722,602,426,32,131,108,771,539,125,817,484,355,182,888,415,769,196,890,7,617,775,328,863,714,718,713,654,799,481,539,304,163,406,125,859,30,196,579,998,914,63,178,312,533,331,519,350,158,31,461,900,643,828,183,455,460,603,164,110,12,831,75,972,516,12,78,661,503,738,885,472,170,255,59,596,614,651,906,223,844,990,376,884,565,854,726,377,885,856,845,323,854,840,173,210,355,570,984,707,467,851,708,47,78,577,437,207,295,432,868,346,645,204,822,326,407,787,480,809,745,634,357,158,281,428,672,427,410,309,196,849,633,911,828,81,885,650,950,890,830,575,848,736,391,878,854,354,57,465,615,475,864,742,824,974,696,403,412,81,497,598,836,431,333,625,821,29,25,412,586,319,519,970,557,935,944,763,701,708,762,175,32,45,609,946,727,136,968,717,998,890,93,900,69,185,786,945,293,762,176,772,848,42,764,688,403,9,540,85,623,95,937,333,590,604,521,158,178,127,126,907,379,384,737,190,541,482,391,920,217,37,752,412,326,745,724,248,430,435,634,657,629,511,313,880,365,449,784,221,780,821,29,106,21,327,936,500,971,954,377,997,147,333,948,963,541,4,40,177,791,225,953,170,836,578,954,545,728,557,676,780,15,413,572,868,157,448,236,421,698,722,138,388,523,648,691,365,526,511,843,933,197,871,730,228,628,262,667,282,557,453,646,513,761,406,948,321,859,922,894,817,909,930,462,635,741,821,873,594,58,909,845,450,17,582,27,87,826,190,953,499,933,525,724,80,559,468,601,131,476,642,843,680,968,16,821,452,750,821,300,734,283,706,106,107,698,358,203,548,269,823,260,43,395,2,884,756,925,743,99,120,142,667,111,622,915,655,144,239,737,834,145,585,548,496,583,780,999,831,561,216,540,208,964,325,720,859,968,118,88,26,644,905,11,834,445,274,709,676,256,459,404,532,620,590,192,727,272,355,220,381,629,931,540,419,206,597,559,764,569,16,118,262,905,680,86,578,17,730,348,570,395,112,450,728,703,718,86,341,642,692,526,422,95,402,206,291,143,205,843,68,147,601,297,324,572,161,786,119,659,252,82,553,129,548,38,593,174,690,294,240,484,253,479,606,930,526,71,917,883,710,654,274,9,22,987,488,264,612,910,800,922,299,151,544,817,831,199,371,140,875,558,309,174,182,897,685,863,678,855,676,417,9,696,458,413,296,167,145,49,880,635,400,839,948,634,667,276,264,163,326,193,172,103,635,690,432,434,716,301,891,256,576,208,914,943,476,627,404,212,712,502,924,757,780,94,343,284,350,813,360,275,535,999,522,906,635,179,700,366,739,583,770,179,228,776,692,432,484,116,277,853,585,416,481,758,867,951,40,611,383,929,412,643,38,599,409,877,436,222,222,269,905,141,944,105,141,60,678,220,293,33,926,690,875,643,986,116,395,306,828,541,62,57,155,714,643,198,264,589,724,111,798,101,770,183,403,730,650,71,928,600,255,395,893,910,524,679,728,65,861,522,637,59,718,693,557,139,175,440,317,445,598,9,20,779,107,482,586,753,60,154,221,72,188,413,592,456",
    "714,367,170,511,526,685,798,275,671,562,373,793,33,963,971,600,532,687,913,48,398,100,852,845,22,133,152,358,675,686,838,983,971,240,384,220,708,125,442,732,27,119,119,32,602,627,841,712,488,545,624,251,392,578,81,755,998,910,43,303,446,562,27,969,191,393,686,686,969,590,995,393,135,81,456,7,210,81,857,779,11,818,798,387,396,106,607,349,192,984,409,169,918,919,546,538,892,833,12,987,438,650,141,594,890,646,484,124,133,545,275,729,902,1,229,998,411,765,46,969,885,953,728,55,926,100,839,47,708,342,701,613,968,982,649,229,493,890,618,942,363,503,865,74,578,801,895,67,605,258,502,437,309,972,350,704,225,606,785,130,333,477,989,337,199,152,127,545,598,506,553,513,91,948,308,615,627,124,702,285,715,954,199,448,986,841,864,779,565,145,359,679,585,90,413,719,727,536,953,924,834,956,512,157,221,322,534,77,560,357,514,697,184,60,205,0,867,369,43,763,268,850,74,202,374,837,830,796,511,866,339,735,19,739,535,843,479,385,689,613,433,810,595,719,33,7,6,413,48,563,962,333,136,216,941,149,604,416,466,415,171,925,92,802,43,384,577,738,297,197,496,248,739,905,319,504,55,829,31,101,566,531,230,257,944,267,573,767,114,115,909,420,380,408,793,155,681,434,580,177,417,721,26,365,155,883,898,733,613,11,838,205,580,828,773,797,703,970,199,821,794,480,436,423,266,934,546,582,88,566,763,654,141,224,851,386,722,276,325,735,563,437,652,681,342,291,78,786,58,527,931,31,608,843,175,458,416,390,447,896,738,339,328,101,519,4,308,439,115,996,458,126,188,368,254,610,486,397,127,460,947,505,11,607,728,506,585,420,761,658,772,251,372,736,771,516,208,179,538,26,74,753,95,880,667,998,59,694,730,530,574,295,384,274,174,866,918,351,844,634,402,879,217,982,208,903,856,681,832,317,833,685,978,658,577,34,777,144,825,144,551,162,837,376,88,631,898,926,216,738,64,714,813,669,842,48,956,583,666,834,825,831,198,276,209,395,322,210,770,152,783,116,515,467,250,853,645,810,73,627,695,889,128,62,807,310,194,597,20,964,24,940,476,824,305,726,62,328,822,380,434,644,593,588,876,451,893,110,610,352,219,842,323,812,537,321,123,708,456,656,803,906,765,633,231,490,571,155,113,92,222,942,915,779,710,777,779,716,896,384,919,508,264,799,999,430,134,393,523,539,528,405,172,133,193,753,946,267,701,19,245,417,902,896,183,334,893,921,841,548,953,234,860,706,252,132,539,397,427,576,473,562,757,204,605,709,617,227,394,715,580,970,934,288,691,984,104,654,467,352,432,266,541,156,167,243,994,457,893,473,769,561,852,420,363,194,237,698,175,89,491,477,615,19,349,964,837,821,354,718,742,274,492,846,928,153,441,602,255,90,834,751,848,649,671,928,74,233,483,807,587,474,945,569,227,710,921,977,709,778,390,953,60,719,968,905,42,323,335,871,494,478,575,720,834,304,630,251,748,623,563,679,573,451,313,430,976,214,95,643,583,167,647,693,783,794,248,210,573,132,573,311,552,819,464,985,280,965,267,947,457,380,405,181,451,867,157,377,297,256,101,742,259,723,854,365,218,484,421,983,197,789,108,138,97,828,958,917,264,540,286,460,938,392,84,393,971,310,598,544,724,334,957,170,588,21,105,906,373,283,723,459,421,251,242,363,500,754,543,786,726,40,977,597,919,984,767,981,221,920,938,70,202,92,207,773,225,514,249,573,66,486,41,846,193,397,125,551,516,206,726,537,813,633,902,834,83,191,433,765,255,329,937,574,874,482,941,725,866,139,194,421,945,250,772,798,859,462,504,316,530,591,619,52,329,989,188,948,331,249,997,794,760,8,84,35,544,66,67,78,393,753,673,183,12,188,387,210,531,58,754,84,927,293,692,460,795,769,538,340,407,937,266,833,576,76,477,309,289,274,387,387,569,82,551,793,624,352,459,130,122,893,625,677,213,321,129,620,173,605,392,396,808,702,906,945,546,88,437,794,149,374,783,993,807,956,175,313,173,722,667,236,831,576,701,607,982,611,785,734,187,111,966,618,378,491,976,437,221,534,8,898,998,520,975,111,952,67,782,585,444,306,374,399,403,822,190,972,815,601,995,926,254,376,821,946,352,745,132,422,865,481,414,752,430,511,310,86,927,775,984,580,609,825,898,900,541,187,291,543,84,979,376,640,561,716,294,938,44,444",
    "524,48,229,378,149,422,425,765,903,659,989,563,812,532,906,32,116,799,731,939,648,797,368,779,715,933,194,152,26,387,133,512,261,348,118,474,607,784,221,608,949,61,801,375,813,951,382,791,853,464,489,182,882,241,243,181,359,411,564,840,606,856,817,167,209,50,715,25,734,120,274,618,485,889,389,187,963,927,643,268,836,887,282,290,582,854,757,959,362,482,707,975,179,513,99,732,992,323,305,739,978,83,83,733,802,453,956,17,794,620,284,278,993,734,347,978,92,447,87,477,242,941,936,593,901,159,631,62,20,41,779,331,92,632,112,143,20,73,227,251,834,388,252,966,558,553,135,473,967,668,275,487,218,166,846,467,65,502,641,228,336,589,897,869,587,516,745,777,370,910,399,938,160,818,835,199,229,781,180,301,287,486,709,875,904,499,517,778,76,859,173,618,866,344,930,878,416,309,344,42,494,69,914,892,766,280,916,393,124,754,557,323,337,571,734,867,0,865,443,276,817,299,525,474,593,570,383,136,727,19,602,256,601,504,160,744,185,886,388,71,202,554,557,992,131,285,205,657,258,664,209,111,176,916,24,511,8,710,944,255,104,127,128,167,848,460,9,430,762,21,787,558,308,921,729,544,570,685,166,267,775,794,619,395,677,975,319,517,847,117,91,161,356,864,157,520,555,591,172,907,422,100,569,645,739,924,725,961,302,800,617,448,636,964,898,863,353,928,826,614,514,661,258,942,164,6,115,476,10,795,758,588,959,753,511,818,384,21,146,640,605,713,658,344,619,170,921,746,661,542,475,387,874,6,672,536,226,248,134,487,748,567,48,461,321,695,636,641,808,672,374,760,224,997,213,236,610,193,220,107,70,990,222,785,137,676,260,808,526,244,247,875,614,873,442,750,222,647,740,756,273,351,886,476,441,262,578,417,318,443,520,43,499,265,282,568,215,726,683,745,242,331,379,88,737,73,332,239,333,50,318,467,955,928,219,681,225,738,484,647,154,762,693,860,649,114,92,39,258,48,415,945,826,618,385,166,189,904,778,725,198,559,205,224,988,803,155,667,283,523,913,670,361,256,920,30,496,761,209,465,940,816,821,748,363,670,873,241,623,900,114,594,592,617,815,929,914,878,186,861,138,950,98,328,313,206,835,218,549,928,559,394,449,536,112,271,944,879,78,994,610,806,844,575,777,689,161,656,510,863,844,775,725,439,98,743,548,361,405,50,529,744,678,808,560,426,439,767,904,442,647,478,319,140,822,791,760,315,615,404,111,4,790,268,30,849,789,313,751,730,720,132,698,293,16,63,283,94,939,204,563,817,749,375,610,372,836,358,848,483,491,960,21,611,85,627,343,787,911,549,820,846,53,300,323,48,274,76,394,634,763,60,88,966,200,225,861,55,318,24,417,458,690,390,660,804,463,156,43,784,653,555,620,147,731,64,421,127,745,873,464,989,784,808,194,359,909,470,117,613,967,685,773,283,739,679,442,642,172,899,509,189,71,251,640,707,97,318,291,917,278,443,284,777,264,499,630,137,498,880,972,810,294,863,989,111,593,383,619,179,235,305,79,254,506,131,42,840,136,854,877,291,900,995,368,716,475,434,909,248,972,699,994,82,304,461,964,827,988,876,866,82,872,821,852,924,534,487,475,21,959,384,542,203,119,840,28,252,809,655,602,158,889,885,140,20,178,284,880,544,664,114,666,711,972,659,377,88,235,866,342,917,168,380,872,228,607,144,804,591,142,870,237,413,916,20,92,435,623,200,308,556,31,171,687,451,590,742,137,492,409,527,690,891,399,623,129,803,102,19,336,536,878,759,783,283,70,701,965,323,661,713,77,49,107,954,878,90,624,741,518,878,528,304,144,646,624,41,351,725,414,132,972,973,207,786,850,41,803,509,520,280,570,914,335,467,561,169,104,172,238,847,3,69,502,870,13,295,379,841,698,906,216,209,606,925,24,370,903,845,16,47,141,758,244,906,22,354,225,67,153,399,508,699,561,722,800,867,838,29,699,952,964,260,32,322,333,380,437,640,800,565,141,246,428,259,951,758,424,662,493,793,984,443,600,785,987,686,983,57,937,791,232,750,607,751,446,762,6,987,430,336,499,936,306,912,360,703,419,111,779,404,839,605,956,138,410,763,1,990,865,580,290,162,894,171,864,766,504,105,617,261,972,774,979,604,441,741,699,922,618,962,532,98,504,410,373,133,480,609,119,700,613,80,922,298,27,90,509,315,488,503,860,256,662,698,230,142,969,447",
    "633,424,865,363,85,83,323,225,873,736,311,852,149,708,415,445,110,490,902,849,854,614,395,755,792,400,575,631,313,750,993,284,676,753,813,528,714,871,349,655,253,395,238,605,577,725,665,998,977,928,393,324,176,201,190,256,120,97,251,219,867,450,901,560,371,827,984,824,258,954,288,85,207,533,796,745,496,995,721,865,655,311,540,512,303,573,783,805,51,809,462,211,806,575,244,95,101,505,195,456,851,648,346,516,74,335,862,652,237,989,279,962,218,833,85,707,535,355,62,284,352,119,841,727,480,443,248,72,459,177,360,197,607,23,238,221,816,19,163,797,720,210,246,262,120,6,717,795,39,92,643,862,942,152,607,174,567,80,35,84,856,398,358,247,670,342,922,796,722,999,319,845,599,119,520,658,832,163,751,43,958,678,818,625,70,428,577,437,572,343,52,700,727,307,750,658,608,597,31,359,137,856,910,165,816,462,332,364,216,208,54,184,391,602,427,369,865,0,759,683,519,246,845,963,392,360,750,773,194,439,579,427,587,592,1,803,966,68,369,325,231,763,969,813,552,969,490,348,983,958,740,850,148,84,642,460,643,316,723,528,966,784,664,782,129,269,264,262,634,99,322,955,389,70,428,526,420,667,531,371,320,254,283,429,873,812,507,136,89,841,339,949,326,812,453,579,1,437,857,504,57,409,602,416,465,260,448,315,694,798,961,285,594,486,943,317,868,445,645,594,824,746,95,490,75,522,380,997,895,240,959,332,359,866,483,191,751,439,554,229,954,821,202,963,390,784,672,143,769,917,284,906,331,325,244,934,942,972,733,766,124,450,19,871,735,967,133,765,402,639,476,604,335,598,512,365,581,294,206,412,827,736,26,650,142,338,552,103,123,244,420,440,262,222,609,44,285,820,858,883,962,836,565,38,718,954,614,480,746,753,471,190,505,244,474,37,53,390,408,880,945,915,859,229,137,482,689,123,632,78,415,515,128,73,929,882,738,749,407,298,693,559,232,652,759,343,94,85,15,55,922,233,408,106,598,329,702,494,795,50,917,154,544,495,840,878,55,112,114,786,225,755,618,401,691,558,729,684,838,535,769,119,840,462,935,397,162,916,443,980,415,794,866,225,76,404,162,175,475,759,376,495,33,255,72,53,480,725,238,12,569,884,112,495,820,332,473,375,162,983,152,415,870,484,486,421,323,190,376,522,375,835,741,799,313,659,349,257,734,721,120,795,639,674,676,319,816,111,350,803,527,896,508,393,88,152,304,852,744,214,750,125,617,611,207,41,505,881,123,350,139,56,50,247,421,752,199,938,121,188,803,775,31,937,512,82,417,64,195,55,387,570,967,909,17,383,279,775,12,228,613,332,170,43,736,329,430,176,431,693,341,49,5,214,681,757,717,769,263,507,949,572,721,52,479,382,262,450,52,983,267,412,283,763,59,740,555,23,207,264,950,205,693,219,355,650,937,663,897,36,964,90,829,407,383,866,411,739,427,926,42,372,944,159,875,477,995,951,234,416,705,46,746,459,18,372,254,88,306,392,394,426,744,943,549,709,830,315,98,110,705,717,684,364,512,739,971,261,755,762,968,580,529,854,673,873,228,997,451,695,1000,656,819,418,202,430,135,8,903,86,429,536,663,634,68,798,236,433,937,12,767,672,213,674,379,28,693,627,644,386,262,770,329,718,825,626,538,850,740,363,211,156,311,686,947,708,754,36,517,252,654,510,929,499,619,956,922,283,963,725,693,405,281,706,610,209,185,122,719,616,852,708,191,470,679,471,244,885,561,319,56,329,651,827,311,64,962,366,446,608,175,737,720,872,537,643,476,815,256,290,580,116,139,150,343,195,55,728,86,108,957,590,706,522,461,633,792,350,958,246,529,136,188,614,871,475,685,256,76,466,533,881,637,633,366,963,93,998,843,585,955,539,594,953,300,380,844,782,396,634,62,420,786,547,55,148,312,808,965,351,685,164,428,937,825,399,156,274,986,395,986,206,497,682,898,857,840,576,396,976,46,433,20,430,366,837,800,831,934,909,775,638,247,994,914,670,828,381,183,895,334,481,627,448,840,564,664,108,236,548,809,392,316,300,501,747,893,35,66,850,429,881,809,566,427,310,643,327,550,896,170,370,510,405,454,390,513,944,998,557,453,140,474,257,204,29,760,804,608,4,428,240,440,160,993,506,16,255,831,660,141,378,62,520,885,849,373,791,277,196,587,558,388,91,535,351,738,522,469,501,535,347,514,41,102,367,866,543,980,123",
    "897,692,45,108,254,288,297,714,660,183,731,151,232,58,345,436,549,316,406,414,501,294,654,564,177,619,265,587,895,582,275,795,25,664,456,833,685,603,99,539,468,576,284,205,738,172,533,415,133,908,311,932,426,89,405,837,443,571,732,20,312,122,892,82,41,660,257,150,776,283,689,295,650,585,637,185,613,944,660,367,266,829,871,681,486,990,988,353,249,664,886,762,417,496,936,922,298,477,335,256,732,760,158,763,740,646,14,315,407,884,40,408,368,954,103,523,452,663,603,603,518,618,476,148,740,207,939,838,290,995,149,582,937,353,501,696,13,455,434,189,527,808,728,74,852,370,502,901,888,910,251,842,269,593,392,496,432,833,467,735,267,868,528,955,731,457,628,548,813,621,719,500,566,158,527,645,912,868,672,414,684,584,299,777,632,406,35,793,207,69,580,259,636,448,517,299,289,173,765,140,404,373,597,269,36,838,50,899,809,528,267,343,685,313,730,43,443,759,0,103,322,699,699,192,907,670,733,520,891,463,479,200,111,886,438,926,147,228,488,438,639,686,40,327,795,939,613,852,212,647,859,82,890,35,968,806,165,340,911,780,125,50,870,131,435,887,224,814,301,395,790,671,459,555,846,332,873,716,10,578,214,525,421,302,520,187,360,928,703,977,665,907,636,971,8,63,462,550,512,468,332,784,577,387,235,258,462,488,532,697,654,145,117,68,365,270,902,944,643,101,477,216,49,745,443,681,410,561,803,188,851,753,578,197,500,863,304,25,887,498,831,100,536,765,649,942,181,953,471,83,579,535,783,796,440,875,983,752,253,760,369,726,898,884,49,944,972,723,802,209,671,241,478,378,323,999,489,133,852,911,588,303,178,559,19,943,668,932,674,934,353,719,450,682,844,354,256,291,570,321,112,226,337,143,242,50,798,229,879,536,882,616,173,44,811,270,625,423,854,529,289,55,471,486,54,225,664,207,516,463,671,255,371,422,620,233,862,286,15,348,763,893,145,553,300,292,270,401,965,372,866,763,985,704,978,350,139,267,373,344,968,385,78,599,63,558,660,354,946,799,147,268,782,643,357,87,681,354,655,494,128,769,192,680,968,615,712,47,51,141,164,175,281,560,427,682,336,82,45,59,894,827,456,214,979,873,815,781,854,632,169,238,58,664,42,805,776,89,733,232,280,711,664,509,273,180,280,433,313,335,328,914,274,427,508,410,693,791,15,555,58,569,968,843,868,330,162,679,611,548,170,639,596,322,643,696,75,202,281,39,198,508,688,719,368,630,786,782,558,569,923,665,941,457,159,443,86,556,270,213,518,836,718,477,258,702,677,265,121,133,833,634,811,612,760,87,413,323,660,154,791,817,86,743,904,44,37,732,730,886,797,42,790,711,732,253,818,36,780,691,759,109,336,120,722,192,287,486,391,37,458,985,551,388,27,895,808,958,913,956,898,960,774,170,626,801,894,429,744,166,619,665,900,132,626,962,995,696,231,84,463,808,154,87,600,299,481,168,943,141,953,625,52,419,744,473,403,638,209,500,768,853,701,2,712,599,290,121,566,115,881,18,901,574,522,708,277,592,22,711,724,718,603,867,164,287,221,873,613,14,443,316,539,488,590,916,810,459,599,234,793,550,778,602,388,78,881,545,541,251,749,516,33,828,27,695,821,562,361,440,463,712,811,884,291,988,749,77,756,855,101,821,568,158,831,771,948,56,903,23,796,699,259,179,149,221,679,509,884,654,147,987,95,51,825,747,884,929,887,461,42,468,780,27,427,510,117,118,183,801,37,86,543,763,711,926,709,358,523,430,10,458,474,425,907,248,864,492,784,757,517,21,144,11,171,754,340,460,763,440,575,364,796,503,201,673,81,786,354,245,410,766,213,354,58,797,235,780,754,562,15,17,763,457,346,162,548,276,365,90,55,329,528,764,327,139,441,456,238,143,396,642,9,182,84,659,360,22,664,541,458,964,418,220,149,795,875,60,984,914,513,972,692,380,52,481,279,187,711,186,51,949,164,5,978,103,950,765,82,862,863,387,152,957,78,466,743,175,641,312,512,783,581,692,636,893,794,192,12,85,882,619,800,289,348,985,758,956,538,869,359,707,898,131,784,653,181,343,456,679,599,938,664,983,776,139,983,578,871,114,318,575,686,914,787,103,995,452,574,921,191,785,454,414,319,209,53,197,402,549,282,712,390,550,33,705,470,909,310,505,986,683,161,99,70,801,60,700,76,452,918,79,175,526,622,743,768,934,869,819",
    "279,958,982,577,710,284,389,273,948,903,911,66,900,356,560,272,370,72,635,573,436,137,756,36,732,809,747,875,293,532,286,128,251,919,993,292,983,334,539,264,502,639,687,739,626,456,184,221,732,964,375,262,295,146,995,750,112,229,151,933,884,64,501,322,847,460,343,24,616,182,356,796,345,673,814,477,234,959,650,357,525,88,180,131,506,578,128,613,212,495,806,851,754,827,464,923,460,237,453,503,285,457,342,554,635,966,93,474,505,325,727,18,997,11,866,410,860,210,158,396,315,70,41,623,697,745,220,724,153,42,839,798,231,248,726,426,49,819,774,89,849,22,607,318,514,9,930,888,478,498,482,300,592,271,198,403,430,262,647,394,447,270,299,783,398,396,76,88,710,17,805,833,758,186,865,118,476,967,720,501,165,602,731,88,345,852,802,709,607,849,773,981,640,899,371,605,897,525,959,4,365,968,842,147,952,991,294,890,86,678,517,925,429,681,989,763,276,683,103,0,5,491,538,443,848,849,402,411,348,806,950,199,914,187,86,343,770,846,893,5,987,623,603,854,62,430,203,359,504,342,693,773,252,583,541,432,823,157,231,610,72,282,75,823,887,412,858,479,253,203,347,371,315,900,926,982,537,15,971,25,744,234,203,610,896,276,829,18,477,45,211,538,910,454,748,116,401,477,231,402,334,543,241,775,840,235,846,750,876,40,336,133,921,440,766,251,736,26,409,774,140,351,983,15,816,877,556,953,379,631,812,907,474,708,878,88,322,957,945,699,91,773,210,186,962,601,52,861,684,652,956,355,68,760,410,939,356,268,902,609,134,200,356,744,475,625,682,6,20,781,334,116,660,131,6,365,140,882,949,128,997,78,268,649,331,442,733,856,325,443,458,94,893,275,142,732,900,129,45,211,973,310,793,729,706,836,646,4,132,520,302,936,713,552,463,894,229,485,864,626,27,315,31,570,584,478,372,12,856,355,78,121,629,158,621,129,293,231,76,959,955,414,445,928,706,501,212,410,314,429,356,852,741,872,590,903,470,273,140,335,815,253,241,360,316,107,233,809,776,461,939,800,828,671,959,267,837,440,568,178,178,804,112,386,610,354,631,252,674,242,799,472,187,100,758,631,759,967,246,282,865,915,776,772,3,655,722,787,338,285,983,565,933,211,584,806,474,457,998,450,429,343,917,95,960,750,691,930,309,602,344,481,400,645,494,382,866,303,498,273,487,748,187,425,242,277,27,173,942,881,46,816,671,821,478,808,663,392,213,359,936,117,728,827,459,403,712,767,460,780,903,164,957,670,411,80,496,744,955,446,382,765,106,74,648,765,987,540,334,177,623,619,583,894,860,718,801,787,968,213,16,399,204,460,422,581,461,873,178,533,436,645,929,163,901,437,517,783,910,247,983,330,54,308,124,415,251,855,174,741,810,49,561,427,138,213,292,384,789,131,629,189,235,585,217,247,145,894,215,64,545,100,159,863,764,900,213,190,404,765,2,498,750,228,664,938,929,163,667,232,119,71,642,614,246,656,543,438,689,598,671,99,640,358,686,299,462,661,344,718,124,920,607,697,604,796,611,71,445,801,594,921,532,188,852,171,658,661,96,505,515,650,794,388,53,77,603,208,842,836,762,628,362,399,622,146,268,173,771,666,265,61,344,703,560,509,551,585,66,462,34,805,336,341,398,375,576,835,298,705,770,406,96,198,805,620,398,911,2,734,219,798,636,56,480,898,934,491,764,744,483,989,895,566,418,718,741,283,207,185,150,518,186,605,695,338,587,757,779,53,663,790,311,589,358,345,499,304,98,316,475,612,139,521,910,609,398,300,913,866,845,777,756,552,66,155,314,716,203,758,974,541,118,607,573,467,58,945,320,579,593,205,421,257,251,295,688,672,273,558,17,469,62,259,742,349,275,969,360,541,95,227,540,689,991,450,51,12,343,380,121,379,948,510,443,710,716,136,426,689,513,639,220,814,894,299,366,636,581,495,145,156,245,811,391,443,347,120,943,109,206,854,926,74,716,358,641,930,538,990,764,209,894,802,660,372,612,951,121,994,82,15,247,488,709,986,116,329,549,532,985,623,359,993,739,696,627,646,643,516,251,229,672,114,834,144,139,638,970,232,339,431,819,258,778,722,987,52,489,554,899,530,642,545,521,771,151,283,293,218,329,319,988,867,971,759,809,273,156,986,872,771,886,485,764,976,363,149,290,798,922,13,392,754,117,726,884,926,567,603,630,582,537,58,33,614,170,805,99,784",
    "760,223,109,932,100,866,483,324,695,747,300,537,183,615,932,209,315,102,370,193,102,108,643,162,551,694,529,691,458,433,461,880,985,163,347,546,651,853,146,286,269,403,345,714,251,241,432,26,598,81,584,770,37,616,787,299,34,482,641,653,805,48,797,386,104,414,987,875,881,663,288,455,49,576,328,508,88,27,155,349,858,868,621,134,928,481,523,64,909,423,261,326,725,526,769,123,213,474,364,478,609,944,294,528,684,884,520,699,61,5,865,368,413,223,697,667,520,868,567,994,688,139,339,12,18,535,146,743,642,676,613,165,291,775,810,4,348,564,884,668,679,802,275,174,45,183,666,695,767,136,639,116,294,538,602,927,72,71,992,743,599,707,551,189,769,647,204,812,594,124,20,559,904,400,604,881,704,161,448,767,161,446,701,369,393,359,751,756,969,474,638,480,387,640,283,13,889,653,770,352,768,131,787,975,958,457,895,90,593,465,710,539,467,645,314,268,817,519,322,5,0,897,313,104,322,65,378,243,688,959,947,365,398,783,164,617,630,229,680,232,475,778,388,546,242,606,990,168,951,473,939,611,312,42,658,586,384,998,774,89,365,108,656,596,897,348,419,402,593,864,172,838,934,386,390,723,580,596,146,81,23,357,896,576,81,458,199,773,114,242,148,641,536,951,666,268,822,364,224,814,968,370,124,151,997,133,669,379,138,723,865,92,559,589,248,109,215,142,533,779,648,224,131,581,248,440,57,666,698,288,773,660,806,912,433,679,920,465,793,689,1000,679,841,341,307,848,738,892,456,711,817,990,685,185,789,729,331,801,441,157,588,217,829,803,89,503,842,200,550,798,841,339,937,923,961,881,810,444,587,581,106,675,258,480,633,546,692,580,757,212,531,435,504,12,935,771,104,552,14,232,946,182,834,252,16,164,729,454,650,538,432,937,957,547,797,261,727,998,351,4,25,852,755,520,17,916,2,297,350,355,152,923,474,925,147,710,744,729,278,514,15,586,91,290,24,922,576,653,776,853,534,444,898,428,841,931,14,511,816,157,469,506,947,277,616,370,433,637,953,418,979,797,139,374,83,762,212,897,891,311,413,855,714,942,338,929,364,579,165,944,362,943,233,71,242,768,677,981,896,959,114,419,686,372,76,466,513,434,16,649,937,359,332,760,259,482,506,945,61,428,442,524,203,888,949,684,655,595,964,633,558,938,557,24,1,518,153,744,742,457,273,127,582,51,113,766,723,117,673,877,382,525,105,298,547,160,456,479,315,356,663,35,588,735,111,556,749,469,433,879,102,343,956,250,589,3,236,474,918,616,436,97,76,771,386,283,828,633,921,341,561,323,95,347,549,635,829,392,57,342,427,306,379,617,692,438,667,371,90,42,527,675,610,567,762,769,569,457,605,619,775,641,588,511,421,298,510,596,489,677,450,876,79,931,162,825,65,649,671,543,168,722,106,108,780,13,557,32,769,612,585,451,610,319,650,304,556,863,258,929,381,121,859,509,628,369,214,188,94,647,753,98,985,880,669,179,542,335,919,423,775,762,47,819,529,341,644,236,738,171,510,823,803,997,356,751,458,584,93,79,493,471,700,742,806,8,389,222,730,799,470,420,403,716,26,849,518,794,470,340,424,704,688,193,760,575,107,412,243,256,594,633,565,588,221,652,866,591,915,384,566,942,481,700,937,727,355,360,714,318,63,526,909,148,452,275,313,275,684,773,177,652,373,193,765,893,336,651,885,781,617,428,428,661,937,147,928,573,987,877,71,323,12,277,905,835,49,857,131,428,577,733,389,503,644,380,774,979,725,464,927,855,592,862,504,465,778,360,842,983,580,357,255,852,39,262,256,598,80,4,978,397,435,727,820,279,647,388,349,489,350,915,685,55,279,407,641,72,616,389,601,515,185,651,494,109,215,732,946,536,297,594,375,262,189,381,827,498,3,680,542,3,608,433,941,578,122,481,290,719,409,790,291,346,268,902,332,813,970,819,553,323,733,260,284,544,495,674,722,39,415,991,186,566,94,977,833,647,816,761,911,232,918,225,537,188,83,159,445,705,71,179,198,324,514,250,926,539,208,369,202,895,350,347,290,321,12,127,422,393,663,320,992,609,106,475,816,218,971,721,540,498,918,83,332,219,703,365,938,507,521,756,68,91,151,332,344,769,586,158,28,224,278,322,72,22,474,168,768,26,377,305,401,506,56,714,161,936,709,626,642,312,295,589,51,720,514,860,160,174,380,846,662,809,116,312,326,708,283,461",
    "755,374,575,170,317,289,151,532,589,944,882,192,916,888,630,371,775,762,657,871,12,551,745,224,995,670,788,446,235,342,694,443,956,903,536,877,374,195,830,42,4,58,309,179,950,270,345,613,203,527,134,835,103,67,399,864,379,349,271,790,358,768,267,513,945,88,755,858,240,956,586,197,8,975,200,112,288,949,332,337,814,673,217,480,612,628,468,291,272,653,216,770,478,413,640,5,929,581,596,848,365,564,914,671,951,335,777,294,53,251,271,907,443,220,740,57,95,49,70,520,992,64,143,156,985,747,569,434,533,68,298,737,221,785,212,256,405,275,884,232,574,585,877,528,692,341,430,525,511,196,759,534,137,108,166,234,367,985,984,486,682,285,637,441,46,564,370,410,179,729,766,785,401,301,588,422,316,28,93,353,601,338,799,559,549,908,258,973,379,503,749,351,959,40,276,352,90,178,720,891,154,174,396,821,987,500,140,613,867,272,544,620,846,13,513,850,299,246,699,491,897,0,249,817,561,545,138,248,37,681,836,496,198,333,99,652,252,282,864,505,430,266,417,576,528,304,674,44,734,507,313,146,285,900,371,605,530,303,451,609,246,847,369,966,924,137,320,121,35,390,330,354,234,879,426,849,248,354,811,54,946,138,559,315,894,759,698,267,237,151,696,928,802,278,741,65,890,764,243,563,393,74,905,509,760,556,965,950,196,354,939,727,118,616,652,840,881,380,819,110,658,996,851,229,992,755,360,946,866,140,406,392,238,510,959,484,897,358,953,892,140,715,813,699,550,103,608,846,440,686,471,846,296,720,877,468,666,204,807,160,254,62,783,148,800,256,886,152,457,382,263,541,943,935,685,31,999,54,856,228,407,89,297,158,149,143,227,419,80,933,610,259,448,322,643,73,720,793,359,238,239,871,795,932,170,882,469,208,734,682,82,593,124,421,749,337,650,214,513,116,252,275,285,983,570,954,606,997,396,959,50,394,893,950,540,403,468,75,848,531,289,546,919,405,580,677,424,560,613,10,173,113,685,675,358,192,301,712,245,444,854,778,396,306,594,866,591,851,824,844,582,550,149,898,470,470,800,141,363,636,402,291,871,814,593,599,13,992,727,437,35,59,876,766,132,925,655,192,146,621,230,218,204,769,36,255,149,26,262,319,351,690,166,569,938,297,477,294,78,791,153,985,734,200,405,575,639,563,323,624,807,897,793,894,499,639,650,729,890,926,673,720,551,128,399,867,227,548,190,796,331,767,752,738,743,723,706,43,491,312,484,207,556,184,398,869,207,873,606,756,125,182,861,637,615,465,539,509,573,96,120,580,618,937,694,238,34,424,566,473,653,363,695,371,228,688,941,129,617,981,818,505,768,791,615,856,658,989,256,518,983,783,41,244,943,940,962,798,888,269,434,52,616,26,282,461,541,295,43,466,315,234,99,858,294,951,507,207,710,248,563,700,420,128,135,284,948,140,873,212,6,912,980,52,176,296,207,392,883,91,833,377,368,420,518,884,464,806,580,504,712,808,70,826,988,791,626,319,649,530,501,148,515,367,521,445,697,480,455,751,584,422,128,51,677,380,348,931,630,873,91,277,207,25,357,879,347,149,61,420,273,22,31,960,685,13,363,255,338,780,793,114,199,309,537,203,467,437,980,9,150,850,408,613,276,945,528,342,959,75,39,274,631,559,999,959,980,537,38,949,612,597,765,307,634,779,373,286,840,692,608,866,510,368,446,894,640,575,12,894,961,2,193,554,577,731,49,205,426,396,213,262,436,717,3,692,592,63,8,529,46,530,30,94,810,857,89,791,816,825,796,104,234,697,601,742,190,858,548,639,217,250,275,305,176,650,866,990,921,506,13,423,35,123,447,721,222,704,15,650,179,450,281,395,139,508,66,845,239,734,576,145,887,41,3,665,925,531,282,558,22,659,204,540,483,352,698,212,800,987,429,147,37,927,613,944,569,589,85,778,176,485,779,780,355,218,166,651,711,14,502,72,865,835,266,153,390,567,805,286,436,415,456,953,716,826,906,463,89,430,317,455,992,97,913,187,407,819,581,560,29,859,243,459,444,924,896,747,933,578,694,129,245,15,795,701,630,860,839,620,268,691,288,345,838,936,662,701,450,402,736,840,914,369,791,121,681,163,13,956,434,868,10,413,738,899,198,229,460,353,97,241,739,736,724,481,17,956,736,1,483,741,234,468,431,289,506,515,710,484,971,369,114,274,258,590,177,784,618,518,608,614,603,379,605,914,277,983,385,196",
    "101,929,320,241,239,828,569,228,81,880,402,881,943,120,189,361,99,762,138,940,683,411,668,805,604,818,563,764,414,269,562,334,366,775,61,742,258,440,766,645,172,632,125,339,811,277,677,23,826,250,240,29,969,482,566,136,29,565,816,541,243,190,34,9,329,253,704,251,355,749,713,200,587,434,626,561,682,257,138,198,465,511,310,726,875,672,568,283,843,644,166,413,679,382,126,280,671,451,245,44,287,539,565,992,416,427,990,966,94,838,349,165,13,852,779,460,24,262,407,375,604,844,464,518,627,415,936,930,223,413,386,607,963,754,396,479,123,978,881,429,679,512,379,354,464,99,389,949,150,17,663,726,268,451,541,229,699,703,792,841,686,798,413,896,621,99,111,533,805,557,835,675,264,354,980,912,704,639,15,883,222,379,806,213,911,485,572,486,791,703,348,904,617,760,751,458,858,292,636,13,629,179,164,253,740,812,285,634,596,232,951,353,649,422,586,74,525,845,699,538,313,249,0,232,372,60,78,417,703,400,789,454,930,798,727,451,474,169,984,417,772,33,806,72,860,154,821,473,644,316,242,767,963,858,734,85,960,311,288,327,967,650,862,332,138,121,234,160,758,851,909,10,793,377,174,433,654,407,51,704,573,941,12,744,853,404,831,654,392,650,986,946,156,505,537,86,901,992,541,204,84,402,273,973,850,8,136,506,912,154,574,639,430,794,395,26,13,41,265,103,880,592,232,97,62,635,237,149,833,784,197,695,11,802,225,399,486,828,716,862,628,93,320,515,25,442,724,268,190,631,499,858,290,690,252,160,705,787,360,539,324,422,988,968,489,133,848,121,789,362,974,789,512,92,423,853,205,243,584,971,315,39,458,292,176,398,780,321,922,804,225,197,202,390,465,73,188,508,232,274,911,52,332,688,112,996,862,910,610,404,702,227,701,779,484,75,818,584,629,273,478,630,607,571,318,713,928,355,946,922,53,881,760,287,126,612,504,792,683,419,647,998,533,102,969,476,758,611,684,542,958,54,871,877,363,985,434,449,820,272,434,922,799,231,519,774,73,139,677,70,312,254,878,390,700,854,52,73,830,582,786,55,184,500,459,365,611,181,268,224,809,999,31,989,477,865,348,226,370,902,237,412,550,993,60,595,681,225,121,984,588,624,706,112,413,839,932,415,243,522,40,216,990,259,54,809,292,327,142,555,471,196,534,260,563,516,93,511,997,877,81,592,622,405,265,682,366,495,270,941,881,934,497,745,377,513,636,961,198,907,887,212,594,138,259,624,786,22,706,377,456,388,152,257,542,801,483,997,370,679,250,663,554,5,993,998,510,108,355,909,414,447,191,231,51,294,578,983,572,31,106,287,869,476,964,591,573,297,270,511,898,159,962,366,650,419,969,846,120,987,955,699,618,806,725,376,293,344,126,973,608,540,141,933,158,542,701,231,882,127,243,106,574,811,538,20,385,102,143,536,403,508,161,187,768,188,337,484,362,425,512,161,129,315,46,682,664,859,7,438,990,263,321,715,617,821,889,233,216,386,436,564,870,690,454,63,109,623,655,611,43,193,189,650,779,340,37,892,163,926,562,41,195,101,810,841,541,932,972,937,823,128,474,881,43,55,551,380,476,829,849,732,436,974,592,752,241,449,377,607,445,791,379,810,216,300,172,560,252,231,384,667,325,911,964,573,122,214,320,257,361,432,791,499,347,424,671,860,354,579,179,772,102,790,182,896,726,875,347,795,94,41,517,139,606,799,653,818,510,675,988,914,442,891,572,303,484,882,32,94,799,468,30,172,662,332,227,656,975,578,740,439,101,930,429,410,50,617,404,924,770,634,686,957,460,267,835,874,491,538,135,159,639,927,850,564,10,250,458,81,962,191,220,211,477,169,672,54,894,522,259,206,320,963,971,70,277,455,550,543,239,15,351,174,288,73,689,596,613,911,312,535,876,229,635,348,271,963,934,227,758,269,827,333,704,337,525,517,332,907,785,653,643,500,705,566,213,705,946,384,668,540,394,345,673,851,650,328,50,938,414,960,122,278,918,839,463,658,464,819,69,638,812,743,804,995,233,996,919,336,874,809,765,263,300,910,89,236,160,584,658,519,603,694,188,558,606,838,659,955,234,974,606,205,98,177,631,175,565,590,589,276,889,458,49,432,46,101,956,659,630,462,194,200,620,106,794,101,652,38,991,991,512,600,315,242,393,316,863,643,123,49,938,77,154,288,894,529,761,609,424,332,638,667,108,685,197,6,917,430",
    "37,828,879,733,471,859,529,819,54,324,442,421,257,815,160,625,423,16,795,468,878,678,113,381,640,30,913,388,405,592,430,37,830,804,685,305,903,692,352,706,641,873,643,818,275,755,493,1000,753,529,84,73,86,468,291,904,127,222,689,869,22,332,964,696,971,796,732,128,196,284,676,180,631,550,558,634,817,235,336,658,564,640,33,188,334,479,41,18,44,935,295,770,43,378,205,354,560,235,457,297,889,175,822,91,662,513,477,4,604,800,908,692,570,66,464,936,290,475,289,667,590,537,619,753,934,467,849,355,951,856,425,778,334,593,531,258,876,946,978,663,482,787,123,810,860,133,446,677,337,521,539,895,277,991,588,383,798,45,553,975,64,761,382,692,701,325,126,751,50,186,536,783,220,196,659,614,51,371,711,698,607,560,741,771,224,731,853,817,180,763,952,648,89,188,492,324,134,719,773,406,496,160,169,602,583,600,186,242,948,203,101,267,720,357,962,202,474,963,192,443,104,817,232,0,420,462,801,385,335,176,146,668,367,652,931,650,645,740,430,702,308,186,453,492,326,721,20,264,473,663,882,778,416,357,244,800,936,98,221,677,145,994,598,599,4,107,413,887,127,711,80,599,898,181,704,77,549,667,867,734,639,779,388,575,878,431,181,443,949,939,236,33,878,13,176,627,259,188,722,216,951,263,704,258,911,41,347,814,272,255,678,770,851,396,774,838,667,631,994,906,628,819,354,432,414,334,795,169,229,972,985,814,305,406,586,117,576,598,888,476,118,208,477,375,306,672,18,490,748,41,2,26,934,407,409,869,135,384,396,235,986,326,920,492,80,376,328,543,767,399,546,750,296,912,639,157,59,166,48,549,104,303,960,367,583,373,888,560,505,382,177,109,956,794,354,465,73,339,941,82,783,353,300,314,853,987,677,958,653,931,439,542,125,723,91,964,309,896,875,712,767,747,729,538,944,688,197,519,580,159,482,210,109,954,869,657,487,319,482,785,852,63,853,35,679,589,505,126,521,222,264,767,864,265,593,617,714,298,762,87,309,773,334,574,877,178,272,69,678,215,996,307,469,26,175,224,656,869,527,338,509,878,653,898,369,75,498,887,644,97,161,925,585,567,224,252,417,482,309,303,696,636,898,454,880,76,979,330,403,201,298,807,479,114,127,74,907,880,634,196,277,477,177,670,218,840,994,121,560,884,840,240,96,383,859,65,233,249,536,694,366,103,850,315,110,770,782,269,338,860,246,496,806,394,142,331,887,373,412,36,776,415,420,912,798,129,316,209,399,261,276,820,130,607,194,169,22,120,766,964,828,517,725,919,752,412,456,755,657,825,857,253,990,235,322,771,816,876,427,84,239,481,527,491,497,829,449,524,72,819,750,273,4,712,303,969,322,88,627,677,67,599,303,346,899,190,372,811,884,304,640,492,347,395,15,134,263,577,718,929,830,130,666,139,882,126,945,550,797,894,902,737,165,349,432,990,836,24,394,595,191,390,461,642,796,430,721,234,524,77,817,163,118,636,265,723,220,33,820,61,190,217,174,231,727,662,29,352,701,466,892,153,589,400,419,294,71,25,872,660,731,792,695,77,421,205,309,783,6,847,968,160,414,80,202,719,577,339,816,437,56,977,994,769,863,848,278,970,434,231,368,790,821,448,894,461,907,579,383,471,411,774,677,659,271,932,509,234,52,777,649,692,501,502,902,530,807,573,18,628,994,865,962,762,763,291,687,756,17,851,515,151,832,736,273,509,818,185,397,749,159,804,919,62,342,221,834,539,591,142,599,321,729,673,27,21,505,544,129,99,885,822,817,269,624,216,104,568,643,123,79,237,965,370,232,222,650,700,847,876,271,122,603,670,383,434,619,485,969,180,755,122,924,982,241,724,892,168,867,289,365,333,459,235,618,515,296,711,242,294,249,162,689,571,130,236,148,556,977,858,196,772,863,367,555,120,113,829,622,479,627,114,32,864,347,896,550,373,559,301,61,665,382,973,405,582,578,8,502,759,331,457,718,879,638,508,619,67,452,308,367,35,205,232,99,610,322,338,723,40,362,302,712,217,553,475,1000,450,373,372,385,460,410,14,810,881,828,555,587,249,725,205,305,759,114,44,522,989,783,322,691,720,427,93,309,43,118,145,142,180,131,413,326,879,853,559,419,35,958,259,474,889,728,997,308,439,488,93,814,189,438,585,631,605,564,659,261,53,392,984,350,86,601,489,193,198,334,455,42,888,7,359,284,832,80,404,813,52,101,627",
    "769,450,331,500,217,732,292,588,718,820,805,920,953,324,393,559,812,11,520,404,655,37,29,34,771,184,614,396,501,612,632,172,679,697,558,86,620,68,131,207,497,865,413,714,476,743,111,725,878,739,825,557,251,251,216,358,14,366,352,197,888,131,589,923,872,501,392,789,661,824,938,865,796,349,626,182,608,8,57,953,435,587,788,778,513,971,978,632,539,939,257,308,866,971,82,789,882,472,302,138,807,116,877,290,993,45,335,837,89,622,77,764,435,391,407,899,59,237,55,473,5,729,699,958,971,186,139,584,795,814,369,686,981,612,824,940,237,657,67,50,968,500,420,813,48,207,488,257,92,497,85,371,903,267,210,487,538,937,200,999,825,320,174,49,445,332,129,473,540,224,395,964,809,146,375,60,345,53,456,652,823,402,3,229,7,917,706,400,825,671,510,15,339,880,624,722,190,110,318,89,538,340,712,583,199,434,375,925,589,462,432,137,85,842,837,374,593,392,907,848,322,561,372,420,0,94,513,348,251,634,236,367,384,337,467,259,499,140,813,40,751,887,781,191,17,703,386,476,586,690,36,570,603,917,223,749,467,91,310,864,694,985,424,516,952,991,571,241,349,520,731,280,992,313,907,959,26,918,246,471,683,932,136,590,518,251,426,559,454,699,364,646,8,675,495,730,59,390,147,469,436,32,349,361,277,475,649,456,599,53,805,33,937,314,741,651,851,300,955,617,404,24,825,825,734,500,615,1,659,114,454,264,619,9,708,872,125,265,26,465,833,6,114,569,661,316,405,595,184,798,26,229,910,461,525,187,686,777,512,554,883,874,969,345,511,383,50,556,6,670,934,543,213,767,626,201,483,838,386,465,788,109,539,664,753,493,278,613,894,80,104,908,349,191,183,829,7,330,691,121,574,691,176,115,195,601,204,611,84,111,400,465,202,740,393,81,895,8,693,16,588,227,778,707,924,894,282,4,828,139,151,308,193,948,993,418,953,245,42,798,708,572,193,868,610,235,991,761,846,86,373,692,524,361,41,49,757,970,654,291,740,687,167,434,925,130,830,944,425,636,755,154,186,754,256,610,280,958,865,908,301,795,922,131,701,527,493,100,325,184,578,705,839,389,34,438,888,628,529,25,843,806,920,148,148,33,817,837,236,199,346,967,244,160,924,114,638,696,564,122,584,336,614,186,75,278,859,4,682,580,529,1000,335,125,782,225,871,610,319,359,361,976,255,715,229,529,961,11,495,409,164,830,320,781,168,635,655,781,109,550,481,351,370,971,383,579,271,278,549,930,438,477,99,314,69,376,715,295,861,665,251,811,493,532,260,63,508,585,120,982,696,653,769,274,689,165,99,289,382,846,339,787,356,931,487,453,119,936,65,79,418,362,321,660,598,773,143,665,743,979,779,2,759,852,747,602,898,451,341,868,801,610,17,108,815,640,158,478,684,604,343,541,622,754,267,338,191,835,257,445,714,527,348,214,792,141,636,768,570,472,361,652,290,483,809,308,254,235,448,623,121,655,30,842,790,690,847,118,259,452,928,166,732,555,831,518,364,18,138,317,997,616,61,456,247,125,946,406,252,415,368,328,407,289,746,294,323,713,88,734,884,186,306,204,829,643,228,181,289,125,265,755,867,233,697,381,580,489,621,987,109,581,516,406,96,149,497,314,433,776,165,986,549,291,386,676,307,533,390,337,864,175,88,354,533,865,653,155,510,398,851,337,336,911,650,420,174,715,55,452,454,128,134,520,394,86,26,698,765,585,303,687,969,786,109,759,138,97,760,43,100,80,289,997,111,179,499,229,273,841,65,549,374,567,247,859,800,390,16,707,112,908,400,698,312,170,973,592,396,879,149,929,386,114,656,771,551,957,811,744,478,404,220,669,734,391,836,107,565,995,258,427,594,818,215,903,891,361,278,469,44,307,106,249,635,583,879,883,995,824,304,908,804,440,960,674,552,345,965,859,842,777,452,798,381,999,344,503,274,278,130,448,109,514,589,904,738,387,322,761,523,810,503,943,623,836,245,908,777,210,980,528,615,284,775,837,466,722,171,925,354,834,741,193,432,100,287,265,954,564,240,331,193,667,458,592,605,232,468,837,506,420,819,602,930,754,115,295,455,825,188,477,158,808,428,591,153,117,61,377,952,190,151,886,985,815,708,207,65,482,673,830,126,521,898,227,322,223,324,835,512,997,669,865,288,19,358,763,247,305,807,64,948,640,424,174,740,666,607,502,129,617,883,682,987,755,845,532,20,883",
    "460,465,990,357,966,853,298,784,408,773,102,487,108,718,933,62,864,639,629,567,324,51,129,873,232,136,94,712,528,84,848,197,913,603,930,445,565,268,394,58,365,112,320,334,945,73,129,531,940,648,531,600,247,873,792,982,207,126,964,544,986,562,289,549,467,437,271,637,554,850,30,524,725,197,489,371,437,127,871,270,395,92,107,379,313,388,440,257,944,889,883,751,664,392,869,553,689,34,82,747,132,243,226,23,334,679,292,437,482,895,21,680,527,673,899,752,416,832,379,601,859,538,604,489,339,940,252,553,464,522,67,730,612,21,532,299,731,708,116,511,838,604,958,575,801,460,316,21,656,544,71,617,123,232,710,919,351,298,238,119,187,959,175,751,423,501,424,589,372,234,55,446,626,12,143,341,90,551,453,653,132,301,224,947,830,767,693,349,684,492,597,985,269,466,786,937,924,703,826,721,207,966,150,112,121,812,83,41,167,43,401,628,802,336,65,837,570,360,670,849,65,545,60,462,94,0,641,918,470,870,352,375,458,867,675,303,270,407,350,578,746,530,678,198,588,349,334,122,532,325,340,724,729,705,994,157,211,45,980,466,26,992,538,120,710,319,585,108,919,539,158,23,948,904,627,165,306,102,774,493,706,679,871,73,769,563,586,934,188,616,508,924,675,424,761,195,957,329,619,965,208,425,894,50,2,438,508,882,137,404,35,264,473,126,37,151,212,758,108,575,706,563,727,735,360,439,161,291,969,619,104,123,849,144,816,283,24,241,901,848,612,466,975,883,784,164,530,590,282,253,508,815,875,868,7,18,626,18,716,739,910,352,314,9,409,292,219,428,230,671,287,3,818,840,246,162,270,720,594,132,656,231,139,854,975,400,501,210,30,279,259,982,814,189,349,191,972,268,397,528,955,420,248,214,786,720,788,685,33,276,358,229,195,100,151,982,281,95,111,591,841,747,855,133,481,258,893,627,671,437,639,412,214,264,477,377,490,963,343,834,97,391,63,917,225,936,248,617,893,361,784,509,884,733,532,277,110,304,184,704,207,453,381,230,324,888,20,670,110,827,636,316,529,187,29,490,215,711,341,9,591,177,550,35,818,444,988,898,141,305,503,218,481,259,923,772,244,131,240,362,583,672,469,788,32,138,995,881,499,732,601,529,442,889,863,748,760,89,530,38,274,134,275,505,123,767,393,934,29,947,821,441,464,141,272,237,691,359,115,42,186,907,139,640,374,475,639,840,548,45,984,59,743,973,820,246,674,919,137,57,824,30,272,857,303,484,316,205,193,296,505,205,42,33,241,208,706,231,772,989,910,524,416,674,659,898,447,674,205,474,219,18,455,647,31,985,198,7,61,406,566,543,281,791,529,634,850,751,625,730,544,77,124,44,185,272,758,761,165,463,893,266,636,236,296,531,772,377,238,705,204,247,975,978,285,397,582,106,748,627,785,25,703,408,159,83,761,24,170,232,535,959,602,29,393,723,689,965,434,645,468,444,525,452,320,389,145,18,672,641,820,808,849,135,586,630,637,717,866,216,603,557,167,701,313,64,753,569,142,898,816,191,485,861,834,8,232,620,479,825,797,599,991,89,887,949,198,147,683,947,191,789,57,802,220,245,693,615,567,77,138,405,352,90,434,743,496,329,324,609,384,328,441,311,336,703,889,210,540,740,915,2,801,365,543,743,833,950,711,642,412,798,140,838,766,185,369,750,312,664,847,199,941,272,793,896,787,844,43,871,375,992,32,543,100,309,441,85,59,287,867,51,871,946,966,590,97,565,491,10,677,12,498,727,273,699,785,53,331,217,829,592,160,430,561,765,97,847,554,263,874,846,41,308,785,921,733,154,581,282,635,365,549,778,581,32,627,574,237,933,366,207,528,614,682,390,248,468,84,263,222,302,124,261,754,536,96,699,551,744,824,126,218,704,908,337,552,969,86,948,122,739,454,370,670,223,525,289,502,209,388,804,103,104,523,968,822,54,365,319,156,302,203,72,215,653,18,851,783,838,860,524,568,844,109,981,791,930,464,991,926,846,549,670,39,24,983,672,224,688,934,937,586,953,865,431,587,679,869,134,424,787,369,134,388,4,37,922,219,252,988,977,821,338,349,871,544,984,204,429,103,120,259,160,900,907,527,777,778,208,890,94,692,43,27,958,160,618,93,786,124,290,501,980,526,583,860,131,801,668,511,562,52,151,435,32,944,973,384,423,22,825,728,887,957,246,219,278,195,133,79,48,426,128,71,958,36,867,954,260",
    "252,411,887,200,557,816,569,99,135,395,764,354,750,333,371,193,837,957,133,675,242,808,382,417,766,955,80,211,377,80,334,561,478,107,793,870,614,496,547,506,168,499,278,678,447,551,89,400,763,126,411,240,492,266,673,867,241,166,229,58,405,450,842,992,374,67,513,171,267,180,357,498,472,597,876,280,354,193,386,29,760,868,663,804,649,273,685,859,93,228,402,423,404,16,16,144,94,691,460,64,114,419,230,34,74,937,896,82,989,74,203,952,727,810,443,682,829,812,16,329,493,852,359,7,672,799,912,277,817,745,475,972,994,852,354,170,60,415,210,293,807,638,787,641,557,444,804,387,446,864,875,961,906,590,597,637,184,935,346,174,339,274,146,834,253,143,187,908,622,532,101,456,801,697,38,899,859,187,20,511,484,224,629,837,704,219,609,621,429,980,123,530,10,606,400,286,824,981,483,187,504,636,373,981,5,525,646,604,40,797,24,925,878,644,55,830,383,750,733,402,378,138,78,801,513,641,0,624,250,997,676,415,49,673,123,168,613,760,17,579,965,139,670,312,777,985,380,105,898,268,695,627,520,794,724,209,523,92,111,415,140,434,990,152,167,593,990,351,790,956,650,703,712,31,528,640,424,273,946,871,22,402,611,299,452,990,168,489,156,160,970,952,83,431,578,65,137,846,121,357,419,663,277,80,209,57,927,827,665,447,152,760,341,926,460,536,712,761,873,998,757,844,232,716,744,352,60,81,944,712,91,313,44,404,585,365,51,647,363,157,295,761,690,939,959,595,8,70,4,27,147,215,879,797,210,823,746,811,182,171,572,670,940,382,297,881,56,50,478,96,559,224,99,443,193,161,406,910,631,415,520,41,459,56,855,658,130,998,373,882,52,768,292,64,15,762,130,61,767,451,256,961,472,57,853,501,659,829,215,498,109,713,705,73,489,537,824,996,523,360,422,432,734,458,149,909,677,983,380,430,858,271,122,258,907,555,606,218,300,218,267,270,917,878,834,734,550,792,125,158,253,645,76,532,210,479,206,356,954,730,583,383,951,743,803,600,517,833,475,949,4,990,718,861,922,570,347,881,178,439,878,83,518,421,143,784,184,349,149,423,957,990,971,715,994,346,314,326,31,20,420,475,767,224,113,892,286,173,134,277,413,920,197,337,268,259,102,287,815,529,658,509,513,237,169,114,677,695,331,183,991,606,226,787,876,608,901,634,598,162,744,946,220,772,367,44,253,666,761,858,915,770,665,683,240,557,727,491,875,512,147,639,228,676,294,774,454,675,595,709,722,570,714,220,206,875,409,133,468,381,132,507,406,966,38,332,233,939,601,328,947,164,240,80,20,383,161,407,407,3,70,647,194,9,278,923,965,584,715,847,948,802,850,826,452,794,88,144,91,461,808,518,291,25,170,348,476,177,516,298,622,400,530,806,917,353,183,546,325,342,644,220,753,338,474,136,798,860,509,251,931,551,178,417,811,53,701,978,289,71,695,992,476,889,445,156,255,388,578,290,497,438,175,858,343,442,700,757,782,306,681,444,710,20,14,369,466,273,335,393,722,578,88,892,608,198,756,545,149,694,911,767,951,276,387,861,780,536,83,17,893,118,280,206,726,470,96,716,203,347,676,86,785,745,231,181,540,536,762,468,784,972,232,991,501,999,857,399,757,31,132,198,90,694,322,778,992,91,547,839,10,165,956,762,808,716,878,73,333,263,9,523,870,158,738,255,101,552,39,506,186,903,779,825,537,11,333,640,651,312,297,333,459,624,293,452,321,143,21,612,609,701,639,368,998,425,400,544,962,150,486,344,546,283,922,225,811,245,692,554,190,191,666,907,538,805,939,378,26,95,670,40,771,541,542,899,502,580,594,731,190,819,997,78,189,270,879,987,92,419,825,267,577,755,669,709,319,800,52,74,794,131,326,99,157,514,969,697,55,554,356,622,717,471,37,21,882,174,603,198,890,352,598,980,462,112,300,19,915,782,809,435,260,738,653,953,496,486,692,109,950,252,87,19,372,46,878,851,38,149,959,8,438,705,286,21,604,144,175,183,666,69,913,319,587,867,373,77,200,389,415,678,175,680,331,984,828,671,516,242,199,702,625,411,188,159,794,332,972,961,274,120,697,594,910,144,260,91,323,413,858,851,811,589,653,289,794,903,80,879,16,737,553,225,644,382,860,519,515,486,226,392,430,264,660,841,156,75,581,583,424,555,679,659,128,704,156,987,186,3,607,262,487,493,718,108,588,198,126,956",
    "869,75,921,713,263,481,270,727,697,676,89,240,326,44,856,825,582,36,864,711,660,828,424,500,649,632,148,23,257,683,163,663,410,962,500,917,43,947,387,606,648,835,208,294,702,300,920,954,989,785,204,944,815,671,503,437,775,515,172,305,332,484,359,827,37,662,777,521,621,73,832,666,30,817,728,901,800,398,141,316,173,858,805,391,510,238,804,595,832,991,301,196,219,79,987,42,445,191,431,268,178,756,511,826,840,859,765,551,20,335,324,273,757,137,408,502,119,266,81,94,328,14,878,819,768,234,645,95,345,835,739,719,292,22,818,889,552,982,791,19,449,404,85,1000,857,551,822,322,469,572,490,440,278,608,920,160,445,221,391,306,710,406,876,750,248,546,285,719,797,427,776,283,414,178,315,518,369,658,253,150,656,831,65,303,437,874,649,123,423,399,188,759,327,138,829,913,894,722,393,863,494,592,106,823,126,773,469,462,906,320,108,487,665,807,101,796,136,773,520,411,243,248,417,385,348,918,624,0,731,129,493,586,38,903,682,357,654,277,575,659,932,231,267,26,877,83,241,137,305,999,156,407,316,53,116,898,166,608,664,592,900,95,875,847,174,983,833,485,308,947,500,748,625,135,498,832,115,30,143,227,856,232,648,817,849,796,229,247,52,602,750,380,677,467,650,708,731,728,306,230,863,237,984,516,820,281,480,608,407,453,165,601,295,748,479,812,825,441,874,129,356,990,21,99,956,404,648,579,352,432,722,464,66,316,504,823,2,958,794,681,586,607,325,35,116,21,336,361,539,748,893,887,688,863,690,296,453,771,433,71,855,73,86,680,12,551,283,590,678,222,492,940,476,265,496,772,530,364,505,335,347,671,624,175,773,818,434,354,284,939,629,911,313,376,824,414,702,798,801,197,315,481,786,148,842,863,917,805,859,124,335,811,436,519,801,98,108,229,695,153,842,701,492,891,788,48,846,300,607,375,25,821,757,195,229,889,710,882,866,263,570,75,783,98,640,491,598,199,585,301,152,883,521,900,721,340,989,693,962,2,640,384,349,956,913,773,937,983,979,255,700,291,831,738,869,900,424,302,379,847,317,296,729,369,854,825,242,810,615,589,349,313,63,198,509,34,310,890,344,813,19,13,27,583,921,392,218,533,606,379,28,66,510,117,227,688,579,179,700,761,105,147,668,22,162,411,613,678,954,475,878,375,886,229,413,255,85,535,430,798,421,984,65,339,620,597,695,441,672,525,785,390,919,732,557,515,554,731,545,189,696,273,694,151,440,263,208,490,609,971,204,895,699,709,56,645,813,63,844,493,965,908,642,89,651,846,241,219,59,501,428,946,384,354,623,839,485,518,160,457,861,592,11,499,907,298,24,707,699,780,888,71,166,182,706,503,690,690,770,830,214,303,970,720,506,918,832,123,791,321,645,4,19,404,841,318,351,361,707,124,528,230,319,854,442,629,766,364,573,921,696,488,38,833,436,503,504,130,730,181,379,891,995,906,728,993,398,275,12,484,934,465,683,891,737,691,526,997,720,420,417,828,292,48,218,933,126,355,748,642,322,720,316,209,701,650,887,119,730,92,577,257,682,633,309,545,273,214,75,132,57,351,829,118,837,307,998,421,417,523,772,898,494,371,768,758,739,831,367,722,852,157,14,608,964,741,576,132,959,537,21,544,388,890,877,49,375,788,406,502,251,918,231,693,248,551,398,448,948,655,153,739,930,318,507,142,18,376,233,711,51,395,2,477,8,320,312,368,938,50,12,987,438,241,308,487,11,400,583,546,53,515,810,100,167,932,873,861,338,374,506,253,179,451,356,722,181,637,110,192,696,162,104,430,111,993,265,987,119,6,192,100,706,238,214,436,879,400,754,526,104,758,24,236,117,104,119,787,525,150,676,332,885,710,613,938,574,894,326,219,722,996,426,169,889,253,931,490,746,10,857,660,630,652,842,395,445,236,364,156,536,715,846,873,24,770,797,414,595,985,786,809,262,426,861,583,97,207,915,854,992,483,990,788,285,196,3,432,194,975,501,721,341,490,607,593,395,290,978,400,328,493,860,100,400,214,679,67,429,63,817,19,31,558,712,601,347,261,83,865,213,346,827,230,686,804,252,601,297,53,727,567,570,48,868,998,300,98,41,836,340,838,853,484,81,353,854,330,784,746,382,101,931,434,963,509,589,250,548,496,397,527,834,8,711,577,519,13,767,434,112,368,960,504,605,740,834,392,495,674,124,746,239,724,524,144,138,506,402,882",
    "700,199,368,503,433,776,942,574,926,435,91,58,454,996,909,813,44,326,237,464,72,19,858,482,596,996,629,187,952,745,262,26,878,218,269,75,408,373,885,177,778,491,20,320,53,755,917,365,427,787,406,84,802,650,385,681,982,403,579,79,989,834,60,670,760,968,275,763,570,962,760,425,833,738,132,103,812,154,963,449,307,5,983,102,742,84,862,600,740,835,204,881,862,194,318,657,42,260,934,450,623,706,876,701,902,178,122,724,378,456,599,276,507,920,79,100,49,500,89,674,467,531,370,31,104,667,449,738,25,394,794,258,630,562,191,287,832,131,902,762,4,163,593,786,671,722,133,446,266,235,969,149,193,152,5,384,43,706,554,897,919,353,602,611,914,429,102,188,296,901,98,960,195,154,363,992,399,666,361,137,378,196,220,229,609,571,853,85,132,668,879,231,468,835,571,932,528,482,731,705,271,28,636,956,736,450,170,461,51,436,202,627,20,985,374,511,727,194,891,348,688,37,703,335,251,470,250,731,0,210,33,138,699,281,980,750,801,782,87,452,700,407,619,947,307,662,90,46,694,644,474,149,963,336,195,639,80,737,432,364,267,664,513,406,825,202,151,881,593,530,712,485,328,58,245,952,614,692,467,650,840,666,986,592,780,186,417,287,979,786,884,776,755,308,206,961,947,102,519,796,741,739,743,653,791,32,343,128,215,502,950,960,325,61,285,838,28,193,508,583,977,697,486,838,854,114,25,228,176,96,743,822,597,389,49,853,940,77,26,96,580,594,282,529,222,306,478,260,721,313,884,230,415,316,580,646,781,605,196,132,268,180,668,521,111,174,501,994,118,552,906,932,474,457,852,7,555,193,984,364,231,856,577,218,280,805,882,299,730,380,975,340,915,71,395,489,889,818,628,342,715,402,759,646,539,640,891,287,41,372,361,968,94,226,816,217,937,739,643,660,401,714,162,290,660,911,404,512,456,585,285,640,35,452,994,11,960,613,396,467,86,55,44,295,43,372,633,894,50,167,565,27,376,387,149,285,522,285,490,792,303,431,569,78,846,731,557,980,203,917,379,568,385,388,330,989,831,918,847,901,83,191,109,839,699,796,493,937,100,767,604,577,866,628,813,555,963,422,335,152,378,458,17,959,756,760,947,774,429,720,457,642,779,993,877,440,528,568,771,342,844,588,818,151,455,961,833,538,364,108,776,689,25,244,639,842,90,676,321,964,740,496,766,931,332,858,151,632,583,885,784,871,223,968,616,632,417,902,929,41,551,188,569,125,286,605,563,101,291,707,19,83,101,910,165,224,302,597,495,993,388,266,780,223,543,4,223,233,316,255,885,118,120,632,549,797,757,184,268,971,232,67,773,951,72,979,765,94,796,773,979,582,960,680,254,349,497,166,574,797,393,697,175,72,949,538,332,675,494,976,927,508,130,867,250,481,869,371,889,386,700,758,216,307,234,230,827,970,485,695,808,16,665,761,890,36,265,985,308,392,54,683,9,702,592,615,936,850,835,22,213,464,392,400,949,722,866,329,760,368,559,860,63,884,902,153,288,713,685,884,724,434,157,565,497,909,445,361,802,634,241,653,921,115,219,876,481,706,222,711,497,550,265,782,992,725,623,594,252,252,65,993,251,125,813,71,973,360,226,772,685,124,840,259,516,397,892,978,738,407,209,124,993,135,126,651,810,112,729,855,385,870,766,695,100,443,547,920,259,739,394,558,149,940,667,925,470,199,756,217,834,591,830,475,499,130,440,222,597,408,741,501,366,342,668,862,598,64,108,409,886,151,715,363,330,601,747,394,132,158,876,163,883,606,298,597,171,658,289,701,555,344,433,207,469,294,156,639,108,924,372,490,13,427,394,376,300,411,556,535,608,941,973,756,835,377,869,460,170,730,260,987,336,346,293,945,62,290,961,285,238,526,540,860,686,198,492,969,452,763,422,837,727,306,86,265,378,703,709,443,125,491,91,710,406,831,183,546,91,52,211,377,362,727,685,603,988,815,167,511,426,490,969,92,506,67,480,995,946,431,713,722,700,285,559,637,182,670,874,109,581,741,210,511,809,957,315,998,107,804,42,539,947,87,582,370,509,908,789,947,743,562,772,426,507,32,974,862,768,838,751,902,797,64,6,528,954,716,299,494,841,897,192,868,164,339,711,311,317,974,112,676,330,42,629,236,33,404,885,415,372,794,295,517,405,967,421,876,147,865,88,867,696,369,12,681,261,269,94,917,313,94,359,553,58,828,978,817,141,747",
    "182,145,294,588,940,586,671,839,581,637,847,859,652,809,844,83,641,603,731,42,119,424,327,831,519,141,93,582,887,654,156,196,313,140,715,430,616,907,885,501,637,314,165,638,927,858,515,585,656,483,819,570,825,348,205,764,796,795,894,289,912,318,980,756,197,482,501,165,924,152,4,572,700,778,539,140,209,254,131,458,762,613,4,351,368,972,682,678,933,688,17,810,233,246,787,121,429,362,528,161,246,17,716,804,507,466,569,569,715,952,899,108,109,590,506,21,734,465,459,802,596,489,406,836,388,947,856,828,952,416,330,10,321,862,423,479,564,271,636,588,886,171,829,299,804,645,303,753,657,251,81,838,759,171,300,857,377,484,471,333,554,772,426,506,284,79,891,79,306,747,926,615,272,681,251,625,739,953,442,527,31,57,203,161,489,156,480,296,638,410,773,428,766,155,387,782,251,965,518,122,67,410,305,218,792,75,505,811,134,575,84,725,260,273,463,866,19,439,463,806,959,681,400,176,634,870,997,129,210,0,381,797,620,202,74,840,171,743,138,427,545,660,660,148,885,966,433,768,902,975,823,801,866,305,376,866,642,973,39,603,510,748,352,691,689,90,70,240,817,442,293,526,553,451,340,729,932,723,112,311,780,871,523,924,603,847,274,883,92,932,715,909,863,642,692,276,39,42,761,180,241,793,836,628,262,390,16,89,916,173,1000,786,5,898,538,27,1,818,660,785,639,99,711,601,694,631,717,234,289,275,296,647,26,624,430,999,145,591,794,575,73,631,559,999,468,640,570,495,608,999,754,274,936,950,370,901,140,257,599,15,969,302,468,19,833,276,4,819,678,719,94,363,357,138,998,180,269,375,707,350,9,304,671,647,599,239,693,851,428,758,478,388,468,27,664,45,978,685,951,587,212,295,730,572,532,291,779,848,880,897,828,190,392,269,778,104,563,89,158,763,648,64,742,917,694,120,434,435,210,531,938,86,767,814,473,89,677,585,253,220,530,637,276,884,541,208,13,145,895,434,478,161,135,397,399,328,236,81,485,225,57,325,887,228,935,578,815,122,961,322,577,201,550,162,576,774,139,849,642,581,734,808,618,311,912,580,224,491,835,713,521,467,450,249,727,724,754,607,246,860,246,439,177,868,296,224,513,364,217,784,149,713,386,231,82,480,178,792,314,122,722,230,108,891,267,662,352,600,163,972,768,639,419,386,725,324,998,491,740,513,538,445,322,897,791,335,568,816,306,931,69,463,1000,308,916,947,930,847,216,152,405,57,594,882,645,948,484,313,449,566,394,315,319,416,760,226,402,98,300,97,629,908,211,688,137,961,289,879,427,652,432,663,466,964,191,640,569,622,821,833,2,66,446,816,905,97,825,164,679,714,814,869,225,726,783,686,271,269,181,308,511,369,581,888,94,195,320,784,94,969,370,393,621,5,144,403,510,588,708,187,629,360,304,348,147,750,153,257,292,648,531,129,433,565,842,146,970,428,110,562,417,106,369,246,439,985,234,370,292,675,813,83,545,912,737,472,963,461,121,732,226,115,665,250,780,571,840,689,230,215,47,216,994,442,396,568,559,874,222,209,635,437,317,326,959,864,471,81,870,184,742,413,579,119,518,275,273,250,933,571,955,622,801,347,54,487,506,23,598,172,83,890,404,154,986,537,182,283,968,615,776,886,479,545,357,636,840,54,827,256,955,977,146,548,439,332,717,818,303,967,499,718,433,987,438,648,736,659,324,30,962,783,459,898,501,418,36,636,303,69,877,273,734,13,612,622,184,415,71,354,20,488,253,863,807,65,244,523,220,726,287,587,889,871,552,492,614,376,950,833,475,306,970,429,650,13,16,66,204,657,962,400,62,383,237,755,481,42,583,313,177,913,975,916,790,830,854,841,223,736,933,633,140,476,10,835,368,793,309,285,121,36,114,837,520,160,834,641,743,581,575,448,912,46,173,503,672,7,901,506,338,932,395,265,499,895,868,892,57,214,824,623,740,479,706,712,653,827,503,930,13,899,272,417,202,728,963,202,616,522,462,444,396,854,850,296,802,744,495,647,543,59,901,165,669,914,571,373,104,989,202,22,846,821,454,714,524,275,682,393,230,218,743,708,238,189,176,998,461,940,126,142,899,904,547,730,849,472,479,738,247,106,204,759,714,476,203,248,928,331,434,867,622,216,212,905,765,857,918,985,929,497,622,345,610,357,602,873,74,389,708,534,7,686,739,4,615,401,593,661,717,833,601,207,399,76,874,813,573,861",
    "507,705,743,513,597,175,785,948,527,505,808,61,382,145,423,955,812,107,806,905,418,178,110,537,26,434,72,774,908,253,146,911,532,59,327,796,60,826,976,176,53,629,356,901,442,909,346,107,660,607,214,871,88,21,185,299,548,319,756,223,80,63,671,439,114,479,852,52,843,783,245,846,988,711,652,819,876,434,794,516,524,934,991,737,337,358,739,274,483,679,157,394,988,214,671,69,199,637,877,549,447,859,641,912,462,270,423,553,138,94,860,697,531,404,739,775,96,530,395,972,815,518,211,642,179,154,750,322,894,986,627,893,656,64,667,418,121,75,377,305,932,652,336,774,603,181,515,727,798,808,875,84,834,528,809,182,542,886,895,311,898,464,892,889,891,280,129,179,544,138,23,507,993,937,182,195,477,167,470,288,848,686,756,440,221,595,786,213,404,429,397,341,409,179,267,36,637,113,137,66,608,150,94,122,413,76,444,948,600,355,452,414,718,663,355,339,602,579,479,950,947,836,789,146,236,352,676,493,33,381,0,139,23,833,745,359,455,482,615,745,849,766,486,237,399,786,956,528,637,660,914,318,397,204,711,576,658,845,215,634,839,285,615,142,210,504,963,140,191,432,299,21,923,74,701,711,796,40,826,775,27,991,22,886,268,817,674,89,183,166,566,176,1,140,798,320,159,238,464,248,734,47,72,805,594,708,447,644,959,688,294,968,204,950,685,792,875,46,415,470,823,947,742,778,108,716,153,66,143,563,366,996,227,655,57,371,926,703,850,279,202,750,609,273,812,569,734,394,905,676,71,40,423,73,563,176,538,926,400,414,953,980,390,60,280,841,103,939,657,751,399,139,618,932,438,137,232,821,876,931,887,4,362,451,225,529,771,665,438,406,759,94,72,681,393,771,810,112,796,390,586,206,653,436,984,273,873,334,889,611,865,626,411,579,795,296,860,456,222,170,961,801,37,961,889,989,118,33,303,111,225,437,722,755,381,190,32,744,976,561,64,721,939,774,408,174,32,149,962,341,340,627,579,183,894,706,10,7,8,110,416,881,237,627,257,981,852,911,699,629,844,336,516,935,510,108,73,984,733,81,498,268,473,508,882,253,794,201,724,195,497,772,418,33,284,121,858,728,891,830,921,473,27,287,898,758,161,493,587,4,638,799,741,364,350,154,766,97,806,914,685,777,452,559,370,988,57,784,712,8,339,534,101,168,214,717,149,155,21,857,510,109,937,241,297,551,866,17,350,759,973,131,685,223,712,334,980,327,112,420,639,613,55,452,53,510,270,966,152,432,287,528,810,783,456,943,194,628,656,853,33,998,127,667,65,554,26,135,208,940,614,178,463,385,497,282,764,87,304,84,203,936,869,209,654,302,578,602,349,952,968,314,764,882,233,561,231,751,711,859,329,954,545,904,596,869,29,77,989,705,164,371,405,471,117,337,598,29,893,910,34,695,521,260,315,887,873,245,157,884,686,480,5,286,525,453,556,877,868,876,944,163,390,359,989,684,333,872,785,119,442,195,63,533,342,924,285,822,527,56,253,641,978,938,410,623,325,785,869,829,689,66,219,234,243,817,38,345,853,79,55,36,575,87,267,7,389,707,507,545,684,14,779,223,851,50,172,416,31,291,59,712,175,2,895,983,952,176,37,401,190,540,724,378,940,632,759,389,848,173,948,370,900,614,685,646,654,305,884,472,717,833,325,91,463,568,6,159,904,377,262,365,141,93,781,167,639,258,256,318,901,51,789,830,977,353,348,384,867,890,416,681,485,468,931,543,757,252,930,871,93,43,618,879,486,677,227,261,404,978,84,28,79,963,691,816,299,971,81,74,663,446,383,970,744,536,75,381,940,80,36,811,811,176,500,383,893,872,481,25,629,22,347,931,991,392,395,153,430,477,994,667,348,679,491,810,934,243,771,617,50,758,310,55,848,935,932,632,812,680,926,327,612,22,858,378,135,562,174,412,421,599,834,734,551,976,277,364,834,886,429,4,955,797,111,355,814,97,147,12,905,351,758,670,526,829,218,629,923,594,321,146,130,795,120,637,583,755,542,896,2,500,765,851,71,280,776,634,174,619,716,789,153,785,683,761,829,940,832,572,912,835,696,718,523,495,335,815,485,455,811,528,983,29,164,689,547,517,678,311,251,736,859,952,223,757,34,357,546,179,127,790,540,668,329,734,170,637,110,647,978,845,831,451,806,18,839,922,6,113,358,827,398,474,277,435,673,617,524,549,112,932,657,349,946,237,913,378,919,452",
    "389,227,78,833,706,168,569,234,309,729,110,174,618,441,998,814,895,428,973,689,212,638,912,226,948,292,289,730,769,855,764,121,132,109,10,204,201,270,694,421,884,246,221,430,252,418,711,853,44,22,236,137,447,180,491,248,148,307,763,576,274,245,881,203,343,285,93,556,478,449,459,78,665,814,579,318,303,672,668,274,701,121,908,996,948,310,61,570,501,345,369,571,548,256,572,851,317,43,306,898,572,706,444,353,579,378,398,820,695,797,775,94,439,747,898,530,367,447,937,314,274,635,287,499,155,400,801,977,295,685,49,417,861,276,460,909,274,995,730,889,457,639,822,400,61,920,646,249,788,928,731,570,243,39,265,985,266,761,843,261,777,574,385,903,837,592,782,827,62,897,64,463,968,261,364,514,887,53,110,479,43,511,260,665,120,134,945,788,550,874,310,457,713,372,836,969,171,923,377,470,26,580,11,450,671,885,247,373,78,414,819,218,970,954,370,735,256,427,200,199,365,496,454,668,367,375,415,586,138,797,139,0,441,962,243,585,134,265,319,745,730,239,493,933,711,963,92,854,401,982,71,711,422,42,887,885,392,570,806,293,763,573,639,874,975,16,753,971,141,149,585,291,59,514,698,666,897,806,657,309,552,94,13,547,155,614,150,378,316,818,830,991,283,810,628,884,576,849,705,417,756,721,836,149,431,795,489,348,818,598,175,280,709,353,776,491,168,602,348,783,488,635,259,872,704,260,25,976,400,75,559,855,915,731,775,201,893,385,105,143,485,699,266,716,476,738,457,700,898,171,212,551,83,65,153,86,67,244,540,457,541,103,307,864,533,425,597,411,995,615,908,236,867,176,355,877,645,564,42,436,376,671,126,532,635,590,560,663,157,749,838,219,752,985,9,567,515,260,31,308,988,410,335,933,91,204,615,372,783,180,833,458,679,35,21,222,472,916,216,979,859,32,715,122,77,781,520,188,124,485,550,85,88,441,484,901,57,417,854,359,175,958,861,341,480,965,541,207,350,559,335,142,342,753,657,906,350,881,109,167,873,308,257,152,132,967,771,613,817,959,106,170,551,10,54,582,622,455,784,264,632,3,390,680,414,473,84,390,560,227,401,29,661,540,867,188,65,391,724,303,677,913,315,273,502,99,585,748,855,10,716,63,187,448,808,123,643,505,399,104,737,326,102,134,323,84,718,815,550,549,821,676,16,263,794,599,150,504,859,465,990,867,498,109,990,736,607,617,168,409,512,999,154,965,793,792,495,181,687,637,49,174,923,998,60,221,139,226,836,964,254,766,77,182,546,323,976,405,657,127,890,711,173,159,8,741,580,676,812,856,645,856,795,377,763,241,199,209,763,572,89,15,491,773,969,691,626,468,878,698,488,322,637,497,573,932,363,497,38,796,427,858,767,88,870,668,347,417,267,840,534,972,956,985,913,813,715,69,321,817,636,246,414,160,175,865,513,114,289,762,654,409,856,671,756,955,251,457,832,123,788,36,536,479,917,841,838,21,760,952,938,302,951,756,253,537,155,121,716,22,891,535,179,636,434,155,481,93,864,274,183,724,624,588,789,879,287,231,437,214,452,611,726,452,144,996,777,87,28,558,961,765,760,605,91,655,234,438,44,620,224,478,814,112,161,720,584,805,24,187,891,285,397,94,85,616,452,145,836,911,18,935,795,882,639,507,988,40,638,58,208,176,159,636,366,275,772,716,73,903,199,986,812,289,746,955,275,272,900,883,863,116,737,530,255,334,360,267,278,886,37,855,252,890,201,63,768,827,254,939,875,819,662,139,522,11,119,15,986,593,399,978,729,297,286,616,495,175,136,188,170,121,41,692,400,697,539,599,832,688,562,491,385,601,867,993,628,889,646,202,751,28,281,93,122,747,825,319,6,299,784,684,770,776,535,498,696,37,405,828,60,936,906,807,405,415,533,5,622,279,692,884,482,779,535,369,985,512,186,667,428,877,268,657,374,56,725,76,494,470,586,968,151,819,588,859,565,627,955,769,467,349,960,316,671,845,805,955,550,626,332,976,765,803,801,372,938,7,555,253,299,103,473,375,756,132,517,651,660,225,511,418,476,496,331,675,803,940,658,773,929,239,64,790,22,61,720,999,823,730,26,919,742,504,661,265,471,807,718,655,789,299,774,691,47,167,859,459,260,519,372,411,734,367,176,501,953,740,552,751,629,380,970,311,97,749,852,541,250,135,394,173,710,900,518,937,39,167,242,760,92,865,205,325,348,369,810,542,549,255",
    "263,66,415,440,38,591,805,389,884,232,311,385,136,106,301,429,853,845,478,580,132,438,592,159,255,69,694,916,206,427,631,768,142,473,649,165,510,771,59,539,157,74,339,331,681,729,552,92,230,896,569,840,726,635,535,57,56,39,25,572,441,745,330,312,404,954,32,847,413,532,958,644,537,29,783,739,924,209,561,776,331,228,176,942,576,413,842,331,316,124,162,495,200,217,286,952,741,117,748,643,867,473,375,999,396,262,357,359,283,94,408,925,857,6,27,681,301,919,250,976,265,179,661,768,437,891,356,552,811,753,173,136,284,824,147,549,99,976,744,158,760,105,142,989,445,906,512,43,922,41,246,706,136,442,674,563,489,931,41,923,868,394,4,142,810,819,318,398,920,730,704,157,226,296,835,887,881,984,464,367,42,436,688,755,24,284,625,811,237,426,815,10,447,348,837,528,154,269,792,775,895,40,478,126,576,857,587,330,404,736,134,303,205,56,303,19,601,587,111,914,398,198,930,367,384,458,49,38,699,620,23,441,0,842,620,236,927,994,102,53,801,579,238,433,344,854,7,429,891,736,841,996,330,461,579,999,902,854,996,71,464,599,289,178,93,502,300,956,8,153,320,451,968,609,623,875,637,826,149,317,598,30,794,861,176,607,29,280,739,745,710,169,598,216,513,245,848,839,454,142,41,673,76,525,246,851,602,510,542,458,915,847,692,418,968,234,422,284,483,650,573,522,249,315,668,575,587,696,295,833,684,634,393,825,749,512,98,995,724,596,504,928,489,779,685,291,304,314,587,716,555,675,359,153,890,858,225,221,925,277,238,148,178,580,244,695,212,789,763,334,913,243,120,399,71,21,248,74,752,432,662,533,311,3,845,963,978,109,344,631,700,688,149,592,837,796,574,611,145,885,124,945,148,625,894,481,162,819,796,619,858,868,347,597,84,64,225,444,179,898,25,115,19,123,583,197,854,52,416,95,432,574,987,21,322,646,863,870,312,351,558,71,816,374,343,995,379,29,23,352,740,478,839,105,943,569,259,175,104,917,122,655,195,266,54,412,152,449,19,897,32,98,891,939,406,851,308,222,288,735,419,113,902,385,295,491,759,178,877,648,449,558,801,783,575,497,729,505,338,934,803,381,902,349,54,819,134,339,100,263,909,695,613,697,494,653,436,317,229,490,518,881,640,289,21,402,302,111,268,256,456,483,226,13,385,3,66,850,970,426,192,638,762,21,446,785,568,554,359,399,929,319,47,404,272,97,582,433,622,6,122,364,950,61,971,108,985,258,760,536,263,113,822,501,923,581,19,815,415,999,510,272,375,140,769,4,844,630,967,522,712,244,63,200,134,628,880,541,507,885,83,282,197,239,625,173,727,414,604,925,610,803,842,132,666,387,347,599,831,627,164,91,145,642,173,917,527,612,314,439,13,592,221,782,547,406,447,339,673,611,727,296,47,763,199,590,256,949,583,123,703,507,804,133,994,684,157,871,568,537,947,935,309,187,249,523,669,151,78,591,723,347,498,579,665,359,288,63,181,559,72,460,228,946,464,203,292,475,916,840,819,674,536,891,538,597,287,90,545,72,23,546,769,301,22,440,250,397,418,93,209,627,329,314,443,515,414,33,371,891,113,798,842,518,230,126,145,261,940,88,620,823,492,265,97,340,493,775,855,744,949,406,941,199,141,940,870,839,910,117,923,948,164,945,132,181,259,230,652,623,805,563,607,409,691,850,848,637,830,889,198,664,289,485,965,378,727,953,516,898,54,535,662,457,146,101,62,955,460,356,69,234,729,405,669,340,369,945,626,64,723,844,515,347,131,469,940,27,894,767,489,15,543,393,803,713,390,890,110,132,89,422,786,826,39,758,516,859,994,979,837,172,89,293,46,238,659,981,246,152,538,766,906,428,666,378,754,42,762,498,203,869,600,902,376,267,511,905,455,345,555,949,666,420,942,146,336,586,71,261,405,470,73,952,882,826,131,651,383,662,569,995,985,763,668,889,761,136,266,492,169,954,87,216,152,610,44,39,795,826,912,583,830,837,338,301,87,852,251,317,74,4,478,694,285,475,5,322,540,528,83,249,407,484,18,539,353,532,376,394,888,49,959,209,106,632,942,73,748,902,413,993,580,597,780,518,263,217,73,916,155,104,577,640,551,613,997,519,363,760,990,899,872,457,423,487,463,160,653,196,138,452,599,445,351,295,913,157,90,359,62,342,406,312,737,95,994,669,727,897,432,342,44,842,508,429,544,626,482,324",
    "88,620,23,189,330,206,232,326,887,145,977,73,870,40,201,679,725,444,166,213,783,332,511,489,430,542,813,623,239,841,153,930,96,528,618,767,826,292,898,636,618,315,812,852,213,676,85,395,999,764,638,465,566,40,191,538,453,261,18,944,727,884,882,566,101,587,652,362,971,977,371,441,77,204,133,676,184,580,651,258,796,664,411,872,329,70,23,147,10,205,905,930,18,89,994,316,608,692,623,245,78,764,603,905,107,784,266,947,964,900,858,184,530,618,174,130,872,776,282,437,326,941,521,368,784,664,761,285,662,828,780,841,131,313,165,424,309,569,700,886,194,818,256,577,141,716,786,699,942,547,598,4,903,657,347,307,281,32,591,296,699,385,918,225,467,675,493,330,823,129,624,694,784,179,187,275,728,724,571,810,300,511,461,114,355,897,461,105,761,826,395,629,493,842,742,803,358,471,996,94,539,109,191,143,596,901,424,225,570,249,588,894,307,970,598,739,504,592,886,187,783,333,798,652,337,867,673,903,281,202,833,962,842,0,582,496,953,101,350,965,573,778,43,179,634,594,142,578,640,496,837,687,945,837,6,307,544,113,784,138,145,548,588,33,854,858,42,893,47,256,417,495,282,936,172,523,40,286,165,920,896,957,208,744,443,663,759,541,175,424,399,387,650,766,235,370,406,662,576,690,860,361,406,282,880,48,747,164,982,500,154,760,85,780,216,369,631,518,861,4,512,288,752,53,447,360,629,19,341,238,105,829,666,348,630,841,909,717,815,328,406,923,732,315,580,64,144,891,195,179,854,642,213,507,151,340,675,660,36,369,713,560,40,606,825,639,441,954,569,173,173,395,868,453,805,561,399,990,640,14,296,40,337,250,860,639,768,511,779,862,52,508,908,711,377,309,828,70,76,583,190,210,339,740,329,15,329,458,385,204,788,594,476,861,683,659,671,639,784,42,253,707,109,87,527,458,215,898,74,433,169,75,497,156,461,934,297,299,554,180,531,129,823,23,426,109,242,713,995,603,456,947,308,859,678,954,657,835,949,599,825,870,586,787,619,246,915,722,848,424,47,33,219,601,56,97,227,24,861,319,851,810,624,947,502,824,858,27,411,40,872,209,519,756,204,714,439,198,477,896,364,352,896,145,106,971,61,40,223,463,45,377,883,332,424,346,573,296,92,626,143,900,339,742,617,816,999,584,547,492,992,749,213,129,199,744,852,508,544,972,645,73,335,528,834,620,439,632,213,341,941,473,972,289,827,78,698,213,862,300,836,104,566,826,512,910,302,18,948,131,398,200,511,651,750,948,60,944,392,909,794,996,16,121,577,282,300,343,137,703,449,322,687,324,596,666,930,135,372,835,302,909,886,926,771,514,814,679,150,137,275,576,203,435,18,180,200,894,138,611,45,840,853,326,212,323,368,188,619,448,677,222,714,73,370,606,365,76,178,907,486,743,250,433,254,862,517,804,280,238,592,417,895,380,234,758,543,335,398,220,414,997,463,887,880,356,628,125,72,185,720,142,130,717,761,716,8,25,396,675,526,745,184,389,659,554,700,379,433,706,292,263,274,534,659,767,717,506,848,227,292,100,138,543,826,225,348,710,993,639,833,231,734,935,698,266,851,193,575,608,612,923,88,278,663,591,255,245,295,350,72,729,483,591,11,588,913,569,86,267,589,3,874,354,16,663,100,624,354,607,290,861,796,871,736,89,403,926,24,827,479,317,406,366,844,22,116,896,90,99,811,908,806,438,142,665,944,73,383,824,987,650,538,33,599,89,506,40,849,612,70,419,316,951,328,385,555,575,813,718,474,810,561,118,341,622,70,985,182,388,713,319,402,990,307,298,412,382,974,249,670,276,404,37,337,899,539,139,427,180,956,596,302,680,874,324,546,937,537,238,164,830,758,586,78,706,750,114,341,828,880,859,98,462,916,928,112,790,173,897,171,280,440,319,454,69,17,897,118,57,726,324,231,240,295,300,677,864,315,34,102,426,168,726,242,42,566,654,278,74,704,391,613,515,516,848,955,339,275,815,946,366,687,675,837,491,336,568,819,623,852,517,644,754,793,670,121,947,684,807,364,181,719,520,130,897,578,700,388,773,498,76,726,851,847,573,4,727,446,618,999,526,459,719,913,450,570,612,541,709,773,778,628,3,693,253,812,216,135,514,292,448,664,951,4,138,628,394,808,819,822,740,49,370,160,471,129,11,616,680,686,581,847,389,757,333,737,695,134,402,228,549,990,188,744,855,669,914,438,627",
    "291,982,434,241,856,784,328,594,512,282,927,644,184,451,319,570,276,836,932,387,266,149,65,219,80,565,811,721,883,527,146,856,359,990,597,745,23,219,823,35,477,464,817,4,403,529,390,825,754,778,904,621,270,431,43,65,810,680,241,536,416,963,916,170,814,479,873,586,743,741,493,27,114,417,986,354,761,652,132,591,76,180,387,865,691,260,675,825,863,803,457,780,917,965,322,521,429,579,659,759,26,112,418,117,302,780,751,403,541,254,919,821,69,55,223,265,360,426,537,170,154,5,553,703,362,266,487,495,333,375,726,505,282,740,550,739,631,50,713,12,148,381,233,111,862,311,67,886,772,483,548,756,44,251,162,749,113,322,474,337,770,224,237,653,412,246,595,7,783,933,350,255,53,372,673,85,430,22,159,116,466,523,508,433,92,905,587,506,597,16,288,222,146,334,19,65,348,431,553,150,110,130,694,446,347,848,773,343,597,849,751,332,32,320,771,535,160,1,438,86,164,99,727,931,467,675,123,682,980,74,745,243,620,582,0,490,99,101,331,138,681,223,977,56,967,747,895,699,319,673,992,217,745,320,643,296,904,528,400,35,619,482,10,473,594,531,895,450,457,122,424,829,370,778,562,475,925,62,799,292,947,786,53,868,390,223,679,529,571,694,749,400,477,592,678,977,568,771,145,888,687,761,72,521,209,683,606,616,380,600,235,586,220,492,56,929,975,646,999,862,823,619,84,708,386,179,168,959,558,866,468,742,853,636,971,567,697,211,802,438,523,872,814,280,379,886,148,461,267,712,281,279,157,615,536,266,331,838,614,151,7,258,25,698,4,219,532,835,498,425,341,249,393,640,293,41,67,111,945,27,142,299,961,873,561,745,312,53,937,406,595,89,102,595,508,183,672,477,828,700,188,846,994,645,11,16,385,649,218,419,648,245,376,13,718,295,130,309,872,376,174,343,176,843,503,403,414,975,152,828,982,653,960,913,297,561,396,441,967,55,233,225,190,461,809,624,386,102,4,743,175,897,650,447,921,563,444,208,629,181,174,562,112,746,643,702,919,745,561,955,741,825,414,271,108,852,882,171,529,311,303,918,433,20,123,895,918,54,293,420,373,61,392,357,883,464,523,992,519,320,449,862,657,703,651,282,647,269,80,740,779,658,813,930,806,729,105,428,308,554,148,345,784,466,729,544,152,461,644,928,216,926,917,651,731,999,679,448,53,597,873,392,579,857,439,577,885,54,341,788,53,357,373,35,785,131,106,377,63,844,779,637,902,762,453,877,308,909,383,967,317,917,594,241,221,951,225,115,916,94,535,278,852,763,420,607,492,785,985,824,717,260,76,325,783,90,985,36,876,893,162,176,867,102,449,507,65,708,238,859,32,13,398,762,711,839,378,78,924,779,744,191,974,199,634,58,58,503,910,82,894,65,725,930,43,804,728,524,378,719,503,537,903,432,120,659,802,734,185,431,233,34,581,651,923,957,154,419,803,432,215,999,792,198,538,944,671,865,565,481,695,645,144,505,715,343,924,807,160,314,488,901,266,449,611,92,525,667,339,266,806,330,992,16,879,970,933,18,262,454,42,720,438,329,794,217,572,824,133,489,946,192,30,221,153,383,799,992,751,651,496,197,883,420,336,3,57,426,785,853,8,599,973,932,862,730,727,487,874,186,15,780,271,485,397,829,903,702,237,470,737,831,719,665,3,993,709,76,566,529,210,569,361,970,852,653,350,581,471,656,582,304,432,139,785,301,949,643,825,308,384,892,775,126,956,651,763,749,35,124,83,319,789,874,108,885,216,702,999,651,559,729,652,846,296,103,645,44,837,951,319,937,228,384,395,317,877,857,369,657,720,812,419,861,536,463,106,777,714,628,328,213,516,785,575,787,305,575,634,979,643,480,44,165,260,627,434,37,165,192,535,261,44,796,408,756,100,990,411,622,80,916,323,878,67,768,811,8,132,551,462,29,504,615,160,307,382,376,434,198,15,875,165,353,374,554,254,566,15,67,709,16,15,329,308,311,144,313,261,82,363,401,453,543,122,830,670,216,623,141,369,358,790,533,138,332,394,793,244,686,636,37,158,347,74,12,493,904,850,314,906,999,862,83,81,157,417,682,998,872,24,488,30,58,692,456,34,731,767,345,748,49,449,39,141,487,437,579,475,643,656,432,5,968,11,342,184,357,242,855,731,167,118,143,869,925,304,94,889,10,95,385,488,687,156,760,597,572,293,396,703,746,259,986,866,901,617,30,657,644",
    "885,33,618,10,790,533,315,396,715,353,702,570,920,445,39,439,463,578,15,684,505,13,77,211,960,883,150,784,974,573,554,379,410,515,410,839,832,935,39,968,555,540,525,493,245,840,375,543,928,574,55,171,493,813,370,441,361,175,621,478,7,714,688,848,223,312,314,730,688,678,978,482,794,189,527,400,837,314,193,789,623,782,890,422,238,861,722,627,380,81,115,499,787,124,30,276,305,729,539,946,866,580,419,644,259,766,568,715,331,523,688,782,961,118,751,541,347,323,242,631,360,217,463,185,384,717,411,815,819,991,654,397,384,314,41,701,912,938,726,701,80,459,135,875,6,440,260,33,596,575,958,557,485,404,443,863,583,559,452,429,547,292,649,486,600,943,217,597,895,419,265,514,325,522,148,704,548,856,255,678,259,22,319,752,482,787,743,682,520,471,799,303,361,930,697,649,366,390,424,919,738,181,262,439,235,435,267,105,577,669,922,460,75,934,990,843,744,803,926,343,617,652,451,650,259,303,168,357,750,840,359,585,236,496,490,0,588,141,853,283,718,955,864,663,996,32,304,459,857,235,304,416,552,18,685,840,138,448,919,482,778,976,973,674,11,755,927,328,967,97,283,70,43,851,106,637,812,980,523,66,708,425,994,544,831,961,681,473,378,563,256,584,191,92,171,148,513,379,508,689,401,785,545,885,940,262,943,107,804,725,639,643,801,130,729,6,62,982,31,518,697,249,741,968,332,516,555,674,968,671,596,985,692,797,973,57,303,996,50,864,604,866,425,386,575,22,66,84,187,346,492,39,33,784,727,130,112,55,362,855,207,319,742,182,99,384,762,632,331,682,410,532,502,213,861,519,738,92,911,62,67,798,970,38,592,776,724,478,78,302,251,192,500,663,830,580,199,654,545,169,309,750,955,168,541,790,982,678,747,239,205,696,972,337,216,127,966,318,494,429,79,308,27,280,57,189,893,628,688,889,154,482,145,630,513,605,779,13,697,69,381,200,905,83,386,383,989,689,87,881,701,631,813,210,805,775,306,123,759,900,27,442,655,466,438,111,534,582,916,508,790,452,331,654,926,981,611,9,403,997,451,171,370,889,945,745,713,439,741,700,718,643,786,179,739,369,70,87,230,628,121,93,465,469,962,880,115,153,670,999,359,211,733,447,421,347,521,866,549,53,662,506,386,846,585,779,175,21,406,934,356,981,338,303,779,104,635,891,157,987,612,989,274,965,833,507,244,512,489,341,568,762,929,904,49,548,978,14,237,831,882,190,362,703,744,630,685,866,894,234,478,65,165,584,269,444,565,124,270,327,925,124,388,105,695,30,4,536,673,685,713,742,589,69,254,628,470,538,913,16,939,38,18,789,947,982,778,225,47,462,671,748,922,572,728,426,45,17,511,615,259,615,440,528,268,129,681,947,79,651,998,441,662,687,769,495,115,378,51,576,523,679,554,552,953,931,276,176,85,475,237,558,813,187,145,868,721,686,555,767,808,4,621,577,390,240,120,337,411,507,148,256,398,817,664,855,29,110,652,395,1,155,209,439,638,584,646,768,989,534,442,277,972,797,749,432,935,99,986,683,664,354,206,935,301,1,784,89,325,647,613,119,752,736,225,205,326,207,892,20,943,737,1000,102,520,374,831,85,669,641,87,620,946,549,862,798,521,297,287,273,94,913,348,199,100,293,567,661,112,314,607,103,787,932,422,732,913,986,907,824,80,744,828,303,791,907,434,84,282,535,79,143,430,699,227,169,885,304,665,343,94,666,109,507,848,851,917,372,204,501,683,138,861,9,348,132,687,392,812,163,473,188,694,103,772,179,863,760,533,192,869,253,466,88,726,208,208,826,134,172,249,371,130,498,927,682,279,266,381,378,909,722,47,232,731,778,986,257,373,745,703,678,845,74,625,505,512,255,881,381,403,283,19,67,439,440,628,685,312,886,349,980,252,11,155,516,887,680,211,881,469,372,444,435,927,206,176,421,619,666,415,128,723,939,104,845,156,792,787,186,250,635,471,642,808,672,743,834,253,595,113,16,557,10,289,461,731,768,848,977,574,486,270,795,269,827,242,365,641,786,586,306,586,911,764,452,496,769,704,476,815,794,121,507,330,405,305,653,39,154,78,767,676,150,677,286,287,160,874,204,698,873,367,274,381,13,382,490,949,254,224,75,785,761,258,911,839,87,304,956,737,940,500,784,630,605,775,592,738,512,402,571,676,31,643,59,911,275,411,326,487,605,528,835,233,365,746,563,267,642",
    "106,317,22,210,509,541,966,467,381,322,260,154,194,450,14,716,212,744,569,704,225,665,645,732,514,925,393,787,383,818,860,87,593,108,785,513,710,158,786,200,641,832,439,743,344,799,715,949,751,188,349,559,77,807,628,803,631,420,344,594,783,688,807,911,505,983,247,447,317,842,948,751,938,280,895,162,258,887,652,36,285,163,828,116,511,622,34,218,798,262,70,216,918,57,444,55,13,755,666,252,946,456,333,524,646,460,893,317,214,390,578,257,900,725,584,586,101,38,988,501,675,150,363,6,502,631,178,987,76,153,688,918,904,156,994,780,23,863,303,326,675,421,412,890,897,383,50,916,637,558,879,785,888,864,575,355,658,305,196,460,975,295,401,49,446,368,690,400,149,741,659,393,337,643,753,749,125,916,497,634,953,268,641,146,177,800,771,857,648,567,322,186,332,313,18,489,432,932,613,266,57,844,6,891,635,947,457,441,750,633,159,319,597,290,794,479,185,966,147,770,630,252,474,645,499,270,613,654,801,171,455,134,927,953,99,588,0,652,314,888,488,69,750,445,988,902,322,126,581,453,244,236,546,645,430,549,904,574,938,866,745,156,605,444,878,246,543,117,860,654,787,337,635,156,391,918,228,307,9,85,47,101,13,848,318,602,710,510,440,282,280,317,516,370,567,519,521,664,442,784,422,712,330,685,120,415,599,338,781,153,469,332,868,133,355,301,563,805,113,265,72,99,220,560,837,516,684,5,733,189,762,171,738,856,626,900,488,940,14,731,352,554,8,453,584,168,199,429,116,759,777,79,559,902,610,747,831,344,689,325,267,299,519,907,172,532,142,720,443,575,867,331,221,827,296,96,756,261,662,797,564,772,216,668,319,343,220,738,19,841,375,881,208,557,817,394,946,128,695,433,838,393,318,916,935,33,608,25,594,680,984,133,101,260,216,385,715,863,836,303,530,748,531,408,498,327,718,89,443,971,735,277,139,458,785,193,925,155,646,798,474,805,886,731,580,607,308,517,879,545,821,884,358,149,683,363,105,709,284,667,761,375,361,327,740,115,265,857,195,12,578,461,4,870,373,192,637,35,201,157,279,224,888,469,91,476,937,81,744,519,140,682,674,46,907,995,307,296,277,617,666,815,389,240,818,275,125,271,771,145,412,562,634,688,283,925,863,87,651,939,93,51,375,589,58,257,231,403,99,422,594,500,378,391,297,956,43,971,237,55,204,573,220,223,595,677,576,27,449,937,72,680,68,955,374,636,643,462,581,727,326,495,723,225,261,295,789,841,429,301,863,889,600,233,856,187,129,547,78,49,660,233,782,388,979,315,627,912,908,527,504,314,630,233,171,577,867,580,188,870,549,253,260,132,976,404,703,141,275,812,335,647,333,329,630,206,36,188,244,704,641,277,986,300,481,192,755,423,147,373,42,583,989,787,677,907,862,967,673,504,384,549,279,294,178,16,462,208,3,362,143,966,185,992,596,685,451,393,770,3,108,683,379,593,508,803,834,590,690,548,18,902,829,288,258,487,894,251,663,950,801,751,99,323,38,473,320,63,732,272,621,88,817,906,132,288,324,239,885,767,146,56,590,89,397,790,955,94,707,810,836,657,640,2,299,376,403,298,473,418,163,600,713,319,357,805,134,730,818,327,209,720,219,343,367,870,707,903,580,585,5,289,737,197,744,768,785,596,796,511,635,651,192,720,494,126,394,150,854,550,475,462,159,547,422,509,401,457,600,792,595,152,764,864,409,170,475,39,470,217,131,114,721,906,691,667,690,227,720,900,367,594,548,936,935,713,386,12,563,184,616,884,234,543,808,814,558,655,692,407,70,580,505,314,148,190,303,902,673,511,94,907,105,513,808,389,820,249,374,182,74,622,390,791,258,340,320,527,702,410,523,465,728,99,604,161,293,356,27,430,689,88,473,664,443,451,756,300,648,468,25,87,374,932,545,883,419,761,40,841,464,314,17,760,145,998,899,597,791,719,119,463,863,480,62,888,88,386,196,756,454,318,400,456,539,910,80,83,1,661,587,453,718,821,779,483,347,134,76,303,771,114,200,513,310,185,156,319,853,975,874,999,131,142,241,356,561,822,821,46,604,956,259,456,76,244,538,65,313,504,569,192,947,856,582,581,159,704,513,848,829,167,264,695,158,596,845,880,153,875,110,200,211,753,705,962,199,49,642,621,49,631,697,528,903,736,880,425,205,370,420,981,582,973,723,613,77,412,166,171,918,160,474,446,787,977,392,644,914,181",
    "638,925,574,56,460,381,65,750,960,285,517,570,204,204,815,83,295,769,160,814,670,3,292,328,462,46,13,705,497,1,110,866,747,188,703,495,36,992,396,91,500,907,265,435,789,718,829,569,504,453,555,741,45,383,544,644,265,318,899,922,925,83,613,387,7,667,212,690,699,45,190,746,150,53,572,933,46,387,116,282,228,403,605,302,686,805,316,704,641,921,891,648,825,717,158,681,219,439,59,360,383,131,368,664,614,550,487,31,459,963,232,372,792,855,6,936,681,816,412,614,577,250,326,457,764,490,897,586,543,776,157,270,627,856,986,920,444,845,885,993,292,118,461,467,990,846,479,179,110,542,132,121,418,694,791,768,989,107,223,294,204,937,37,87,57,558,885,231,815,352,546,708,846,597,370,58,224,573,477,673,595,115,88,395,82,592,362,974,326,390,442,845,910,186,765,119,585,836,751,159,41,1,94,462,957,313,245,941,782,858,7,461,127,921,835,385,886,68,228,846,229,282,169,740,140,407,760,277,782,743,482,265,994,101,101,141,652,0,810,578,14,257,582,40,696,14,172,574,968,681,459,564,728,244,851,62,583,511,486,158,301,52,12,318,850,21,302,387,929,798,717,351,241,971,803,208,199,745,728,473,443,988,970,226,856,295,600,620,697,739,425,235,204,568,181,733,730,155,291,714,314,22,999,527,128,300,866,76,644,440,693,426,744,34,829,162,679,701,866,294,211,347,723,68,894,236,606,762,337,97,792,261,954,61,83,826,520,384,280,765,486,696,657,106,98,220,984,710,619,351,169,928,94,958,211,456,243,350,714,75,588,141,221,616,193,702,797,6,603,353,555,26,308,28,903,21,256,590,796,586,320,87,377,564,512,666,669,591,228,652,156,909,289,863,678,588,921,62,578,352,928,420,164,234,27,535,913,751,484,666,866,343,65,164,173,745,359,314,26,994,22,198,157,230,469,819,870,807,480,95,571,579,668,702,192,171,112,284,474,731,34,554,568,287,508,363,630,534,230,525,902,770,546,110,577,877,626,34,78,999,217,58,627,838,517,959,125,77,984,107,966,380,434,806,396,947,747,132,805,887,238,331,620,708,190,153,476,634,162,30,577,70,167,65,989,342,514,389,313,248,92,361,454,516,473,606,577,713,214,851,650,407,120,916,723,63,137,943,632,630,909,757,809,265,812,695,844,301,839,310,255,427,714,702,872,230,100,249,780,269,368,326,133,971,835,693,243,34,663,537,578,590,185,988,944,626,733,771,334,776,773,109,925,292,211,45,78,234,549,79,119,887,505,817,316,955,905,826,202,29,684,868,439,436,807,233,552,827,614,127,927,528,325,59,198,109,825,411,507,180,243,45,52,247,879,805,663,159,887,81,141,637,760,671,515,181,396,355,16,788,415,960,829,486,979,387,955,823,79,903,764,351,605,603,797,139,851,401,501,696,407,103,670,442,934,78,298,118,289,942,612,80,15,311,288,714,604,908,266,470,328,411,467,533,543,509,613,509,832,402,271,975,674,729,159,951,586,583,737,880,323,876,640,70,101,4,94,70,153,780,472,418,69,974,242,963,380,437,443,127,406,449,25,423,504,595,453,42,103,293,565,644,593,803,427,433,150,389,203,655,417,101,86,454,793,445,242,989,11,525,935,253,434,365,562,579,828,668,601,771,908,655,4,921,297,55,510,929,456,161,983,569,751,97,721,436,649,338,120,204,263,44,865,332,458,486,993,181,107,279,594,185,556,623,945,473,853,319,935,739,443,885,66,847,937,552,259,824,455,542,781,790,967,325,332,131,724,611,897,934,966,131,959,932,471,590,772,962,479,863,10,729,385,286,689,172,438,681,8,81,317,624,881,931,412,580,218,128,864,699,139,845,434,145,859,691,724,187,719,649,855,311,695,357,562,852,145,949,409,510,515,286,809,618,270,376,412,521,867,214,625,962,501,457,462,261,18,970,252,989,338,32,274,279,837,304,670,535,586,836,279,85,925,25,521,637,250,371,11,857,152,561,992,102,645,303,227,53,534,754,408,907,90,142,441,264,698,909,444,111,758,426,509,278,184,469,941,482,176,447,821,702,87,457,245,447,707,126,971,172,177,659,397,47,479,50,604,98,435,921,241,360,979,746,402,157,1000,767,385,294,757,275,146,474,747,931,436,562,531,429,242,25,514,204,96,983,314,687,240,459,79,64,819,825,674,844,42,547,183,553,576,500,312,47,224,495,797,72,592,228,623,991,389,840,1,491,227,454,816,823",
    "375,780,692,331,143,423,852,366,181,771,71,199,774,698,505,493,64,237,803,223,928,195,855,651,88,207,467,880,907,122,128,831,716,410,433,204,916,418,81,27,160,515,795,390,735,130,995,110,772,589,950,253,690,728,83,756,803,186,391,233,398,901,83,238,419,366,690,595,676,621,201,390,242,327,960,749,776,43,676,94,12,696,188,942,673,317,831,478,925,642,563,124,650,215,598,736,152,887,824,479,751,150,520,977,657,107,232,878,397,320,491,798,198,226,448,872,167,726,320,973,727,819,128,382,762,826,263,606,95,972,969,381,644,486,793,733,994,975,455,892,333,731,372,118,910,350,873,210,143,891,994,6,170,756,65,167,301,172,608,623,623,256,286,426,944,148,648,683,211,176,608,764,969,67,926,67,766,997,133,48,775,243,819,902,544,358,722,998,860,258,827,191,491,189,439,561,691,668,229,622,803,46,578,366,746,703,653,308,488,799,411,212,844,214,293,689,388,369,488,893,680,864,984,430,813,350,17,575,87,138,615,319,102,350,331,853,314,810,0,134,366,741,749,164,507,209,881,257,112,803,764,759,935,531,259,222,717,38,518,296,321,677,335,687,542,491,292,589,336,208,489,393,570,453,490,700,329,464,352,875,538,634,298,219,864,372,647,226,810,368,77,651,178,209,730,329,923,730,702,52,612,808,32,684,214,665,961,746,414,8,367,118,539,213,296,426,283,34,480,129,955,274,717,658,715,966,537,551,975,284,253,29,808,217,517,698,292,291,379,830,887,331,922,753,356,904,657,392,648,204,868,574,106,249,857,537,585,837,19,943,861,768,696,806,95,172,432,687,157,787,78,41,810,481,319,380,89,896,817,944,217,90,258,874,355,6,847,177,241,709,767,998,79,249,869,544,952,676,134,24,358,647,634,565,241,420,347,518,888,473,814,508,763,568,111,289,695,117,507,464,782,37,821,113,698,662,256,538,624,371,583,154,685,94,195,796,901,360,292,782,848,763,990,222,869,56,942,585,447,768,351,609,393,44,590,131,43,944,571,516,76,870,31,389,927,84,977,877,509,523,251,481,909,301,686,961,681,102,683,163,935,48,475,788,650,137,433,966,557,945,515,798,588,923,694,983,236,302,656,383,423,58,946,101,214,306,159,52,252,883,253,667,908,762,874,507,645,738,190,159,647,116,787,862,420,665,966,836,532,249,322,554,302,538,578,284,402,155,237,522,346,256,668,790,120,976,716,511,891,7,28,417,268,52,253,815,712,59,629,742,799,74,742,10,711,729,512,791,473,987,91,217,7,708,649,134,649,665,354,909,677,474,756,856,511,538,629,815,10,791,662,638,701,198,604,45,937,153,211,675,323,631,880,519,249,579,484,938,308,873,831,18,35,442,479,443,333,736,632,247,593,24,785,824,746,169,754,128,213,128,48,753,181,138,930,608,757,611,598,418,82,477,5,382,295,271,571,161,960,195,216,417,566,577,701,697,155,278,376,824,991,820,458,98,816,577,823,518,632,954,261,706,421,787,944,308,345,326,183,340,213,437,679,428,371,87,587,951,785,54,995,736,290,645,443,803,669,771,559,499,510,120,878,577,245,442,385,545,800,194,997,128,622,652,561,955,169,175,980,277,467,186,513,28,985,960,96,111,566,364,773,307,215,131,810,830,572,327,526,20,204,145,459,920,22,966,738,943,766,470,149,370,560,918,393,136,109,49,621,233,334,69,787,128,573,347,104,450,284,166,85,634,17,11,582,637,295,715,457,307,941,218,233,361,154,631,165,243,823,181,35,656,65,979,948,561,531,801,453,558,463,359,575,692,338,928,216,669,921,127,555,399,365,313,396,742,157,237,718,613,238,76,249,783,223,103,81,175,130,367,729,886,543,384,833,51,941,320,349,638,333,914,482,784,848,837,883,787,344,69,86,459,892,234,965,260,61,245,559,686,151,670,184,644,436,800,822,307,354,743,970,145,233,144,316,759,326,208,495,196,413,589,529,496,471,307,28,497,515,437,779,405,346,910,511,391,875,191,998,601,454,339,372,60,525,896,160,478,110,388,33,739,777,428,538,77,393,714,269,498,649,890,976,615,844,234,715,115,385,439,159,856,276,156,797,943,361,486,843,445,243,664,465,194,795,60,610,985,173,307,812,63,905,597,443,783,165,733,897,423,660,725,261,479,409,410,537,782,801,496,314,925,950,653,26,562,154,894,75,108,974,148,832,55,410,864,628,217,110,415,124,133,10,606,824,759,860,290,551,801",
    "661,598,953,982,434,522,726,713,236,498,656,136,47,356,893,864,892,688,867,366,923,341,71,275,753,663,993,493,824,276,941,517,268,911,832,801,465,345,217,890,745,58,646,237,331,363,600,542,613,4,540,831,709,290,915,978,936,991,412,344,65,181,444,484,792,38,439,195,236,548,219,79,238,72,564,745,94,25,791,950,213,881,545,619,370,808,444,851,354,997,905,209,235,63,142,982,230,997,763,894,998,616,930,451,444,480,483,280,677,277,61,846,219,110,203,126,886,514,808,830,661,957,895,648,316,423,884,751,741,424,150,425,1000,181,555,124,7,495,803,704,945,391,571,488,294,487,466,416,741,163,145,861,736,635,746,718,653,73,24,989,864,561,698,452,607,85,243,422,783,202,975,771,783,246,653,555,931,187,59,84,337,928,922,468,7,405,920,99,391,663,48,54,928,690,728,533,693,232,548,102,652,617,831,68,735,207,62,791,145,876,796,448,996,668,624,613,71,325,438,5,232,505,417,702,40,578,579,659,452,427,745,745,53,965,138,283,888,578,134,0,144,180,22,805,671,861,413,634,238,523,184,688,977,168,283,81,3,487,657,208,82,167,621,760,890,486,508,139,165,601,655,430,149,491,500,47,729,319,719,488,474,73,892,992,371,217,496,244,909,802,208,683,17,981,932,333,14,38,301,247,325,282,170,675,215,282,280,584,966,775,235,178,676,478,607,350,258,485,618,520,743,230,210,900,224,452,531,771,587,861,186,213,437,441,156,423,800,463,225,30,805,217,596,22,391,258,892,482,231,420,542,96,95,54,690,228,274,610,62,899,891,495,226,508,42,646,629,222,557,949,262,210,842,731,246,225,686,136,570,135,194,679,663,751,155,118,159,637,818,685,312,191,185,968,56,623,73,900,910,400,36,405,272,471,295,679,899,586,932,744,133,410,506,842,179,209,662,295,484,665,238,382,972,90,48,812,661,237,124,861,430,560,365,853,205,356,765,418,407,183,534,472,783,481,586,309,150,227,670,760,891,51,474,153,11,64,477,102,440,559,704,782,670,84,380,863,916,796,627,396,61,943,160,865,557,369,590,74,430,975,108,530,409,575,792,623,994,405,297,464,292,375,2,922,31,742,830,23,710,785,673,610,603,708,936,238,354,517,608,962,800,874,685,526,270,239,165,616,537,46,195,466,243,707,104,123,208,605,324,948,28,493,225,370,382,398,95,81,80,313,361,797,236,677,904,519,828,484,66,957,718,612,501,726,714,369,39,654,71,449,745,878,19,892,910,957,490,575,580,506,195,283,773,814,397,159,582,434,690,864,548,594,465,382,177,180,91,184,805,670,217,25,546,783,228,350,591,161,406,885,134,707,209,607,207,88,382,534,246,348,803,795,109,603,946,585,250,813,845,486,927,129,667,240,378,909,721,511,706,152,146,373,630,334,628,299,963,197,266,159,845,98,553,916,403,944,353,844,39,566,284,839,18,391,265,959,830,37,305,27,497,421,836,6,466,624,149,392,906,21,922,130,839,290,991,529,624,127,354,123,808,178,167,583,834,732,860,922,6,53,254,956,593,191,981,143,581,908,735,112,264,666,475,11,657,652,613,207,629,389,656,769,548,734,497,95,492,413,452,582,986,217,376,908,562,545,283,912,371,431,270,889,611,656,377,250,54,28,499,126,448,146,91,212,368,57,309,602,387,218,544,527,933,123,972,457,141,663,394,584,659,494,801,534,108,846,370,869,573,449,914,884,291,346,250,435,893,975,240,499,216,137,318,548,605,452,846,448,818,460,932,274,862,905,110,245,775,416,439,794,966,655,641,157,977,63,333,598,396,871,717,50,804,693,451,213,175,111,634,126,128,386,586,924,566,922,593,623,32,407,895,648,248,634,19,443,395,738,71,892,972,944,351,513,588,125,473,166,295,450,610,21,79,26,335,215,191,887,497,535,527,652,685,908,38,540,257,306,435,984,721,882,250,59,344,865,510,659,339,863,110,41,42,49,485,937,368,189,534,743,897,33,836,275,435,940,990,69,410,544,271,488,509,693,314,2,60,885,640,477,197,852,431,21,575,363,696,257,593,79,407,770,181,338,125,152,842,992,483,708,578,626,874,762,85,763,962,880,683,419,723,961,241,781,726,196,793,701,399,49,894,671,731,68,60,640,569,855,700,855,597,995,924,725,560,275,118,951,299,330,660,242,517,859,572,758,105,597,565,126,241,544,337,803,919,797,861,132,682,355,548,48,842,161,183,485,514,535,321,167",
    "375,712,154,572,493,148,788,4,39,64,487,920,642,912,233,879,770,796,207,284,217,472,431,976,597,611,102,544,384,141,660,489,932,601,943,823,853,42,511,359,798,561,172,298,672,119,237,982,176,445,823,777,514,3,726,606,187,586,770,512,885,848,24,707,954,291,786,264,301,245,335,448,365,90,490,583,268,349,830,473,877,528,552,114,366,47,967,686,298,168,767,702,865,225,135,24,851,93,777,729,419,884,898,362,131,211,357,621,252,902,695,140,623,458,481,503,29,577,609,845,471,693,200,954,786,814,954,635,844,512,833,63,124,979,929,637,333,463,400,301,605,178,771,697,96,661,517,521,914,555,452,378,197,170,666,773,144,827,235,55,869,6,37,111,250,416,137,517,27,641,181,541,994,529,809,734,604,373,169,651,169,610,98,969,396,219,36,636,617,192,895,822,685,346,990,813,540,119,337,617,250,379,942,717,706,824,828,285,69,771,554,674,305,88,968,433,202,231,639,987,475,430,772,308,751,746,965,932,700,545,849,730,801,573,681,718,488,14,366,144,0,84,863,965,871,786,401,473,56,340,416,173,70,175,524,650,509,23,680,20,883,283,549,314,30,238,357,123,9,842,584,52,562,907,681,697,836,581,568,60,849,996,100,813,980,854,233,268,235,328,125,238,867,194,599,388,177,931,18,93,796,34,522,29,93,122,503,573,90,236,674,760,475,486,965,510,313,287,535,167,640,451,529,982,491,244,143,960,438,939,919,801,317,564,895,960,940,985,124,997,349,184,565,476,274,889,572,744,911,219,681,682,307,175,247,903,875,394,63,236,273,686,678,417,911,676,869,171,368,11,447,901,843,710,640,334,441,421,846,620,626,786,388,451,910,265,562,206,169,330,781,62,798,340,235,994,140,953,881,777,75,902,80,877,51,399,648,884,28,998,970,43,10,158,350,119,835,847,499,947,589,895,43,676,582,654,124,660,585,505,708,171,909,650,139,231,832,338,835,99,864,798,143,595,893,686,930,657,562,758,964,32,605,452,951,471,725,282,725,265,666,9,545,509,826,866,631,502,359,543,63,947,426,207,140,579,869,412,253,787,588,970,391,100,315,799,409,330,719,294,741,514,97,81,353,167,594,642,436,263,845,921,406,349,775,849,584,710,22,876,583,681,952,779,526,178,856,362,425,700,576,822,219,387,576,256,326,699,56,704,725,736,490,892,617,265,777,306,770,89,950,466,578,249,876,189,396,93,98,826,98,135,82,107,441,159,563,307,456,776,71,52,272,585,444,602,233,223,122,920,373,874,886,65,327,712,255,345,640,782,241,17,152,663,957,246,112,622,530,618,317,650,65,550,929,109,276,897,932,777,770,700,118,648,102,294,812,579,148,655,708,756,177,397,973,920,13,108,61,797,865,870,178,61,671,100,755,321,72,168,568,572,236,446,528,115,383,989,582,564,671,61,721,154,505,189,357,212,334,659,131,245,201,690,36,498,724,691,917,139,734,75,829,386,667,896,198,606,330,375,306,164,857,42,2,834,998,36,466,409,210,926,903,543,653,493,245,929,992,130,737,449,776,451,251,846,209,595,356,113,887,930,686,59,243,315,570,442,680,826,959,137,405,641,427,320,634,898,73,598,726,456,959,115,250,608,490,19,928,383,595,668,137,175,198,726,159,984,779,127,852,371,186,660,376,287,774,216,873,406,352,288,938,196,692,274,850,824,264,52,130,97,852,387,256,940,544,697,465,882,438,551,457,685,491,720,279,486,80,913,730,86,602,133,688,296,87,782,792,607,633,557,909,847,384,998,354,58,485,616,464,731,68,778,808,98,649,433,401,734,945,229,673,356,631,210,743,774,870,395,58,229,690,964,920,269,437,407,392,617,651,343,779,892,758,48,610,546,461,977,773,974,586,288,176,426,270,890,403,380,350,638,317,698,49,524,689,622,522,799,462,422,80,375,532,415,883,653,957,599,593,671,534,679,101,424,580,596,78,482,363,321,156,812,228,148,207,333,200,927,363,984,503,604,22,451,406,642,250,333,600,457,372,98,884,475,925,214,382,458,784,772,926,831,422,535,85,424,569,536,33,19,590,845,77,575,882,989,893,114,885,775,940,110,722,593,269,479,634,599,247,529,352,746,486,214,198,233,433,286,924,80,659,513,872,451,367,604,730,251,954,657,618,444,646,476,525,658,428,30,12,655,938,32,466,531,309,102,376,104,348,277,632,477,854,869,653,644,776,962,853,737,124,505,63,71,395,687,637,366",
    "941,29,656,675,332,569,349,265,358,112,158,70,203,827,930,374,693,70,650,118,784,993,759,432,557,381,885,768,966,377,109,332,303,722,281,969,24,884,149,658,892,790,696,202,470,937,803,566,893,477,844,447,405,660,42,812,880,835,860,724,763,957,325,553,971,333,89,896,171,595,916,713,914,524,9,352,314,345,180,473,320,770,687,129,992,5,919,345,181,427,235,216,842,261,575,137,381,892,938,875,752,613,127,918,473,303,708,117,86,301,413,454,438,37,739,174,156,267,966,579,176,333,68,921,982,298,430,115,845,247,804,217,841,352,290,496,241,591,749,392,353,772,641,838,786,396,478,390,508,687,404,502,958,337,540,984,949,850,407,828,74,439,91,35,21,876,890,307,499,416,255,873,160,775,374,650,367,525,212,236,355,440,52,967,692,410,654,266,865,902,911,176,537,548,269,697,100,294,589,131,717,212,270,215,158,967,795,576,925,203,619,678,366,659,242,810,554,763,686,623,778,266,33,186,887,530,139,231,407,660,766,239,579,778,223,955,69,257,741,180,84,0,58,952,424,534,493,747,463,699,431,247,888,286,972,916,182,49,1000,651,401,225,606,360,289,472,365,575,366,376,423,71,777,62,738,430,553,93,387,150,201,336,85,938,125,582,561,939,885,573,325,836,857,82,231,914,546,684,11,12,479,672,243,877,100,525,794,353,151,666,799,402,328,8,931,692,289,7,309,643,112,716,934,446,741,55,360,965,211,433,89,87,434,107,960,540,915,401,664,240,482,959,196,467,180,696,702,579,52,596,284,508,176,436,770,270,537,519,237,110,675,945,724,429,12,789,312,435,923,497,94,908,355,676,379,260,246,629,214,752,688,578,315,376,455,105,699,645,390,654,463,436,659,359,472,284,848,726,705,177,416,110,102,890,629,359,629,543,852,578,204,105,333,378,525,992,159,505,980,925,245,409,454,348,877,857,157,949,295,264,909,767,386,471,953,958,7,719,902,466,126,220,498,69,941,613,471,581,576,616,897,960,925,493,648,796,44,74,822,933,834,565,65,342,615,119,460,766,552,384,617,971,49,133,477,961,615,583,221,473,69,936,272,827,973,797,614,83,637,95,622,261,328,556,210,775,253,888,723,540,221,853,333,950,705,305,994,684,815,845,371,18,622,404,461,956,355,662,538,655,109,80,641,112,974,438,812,53,262,525,635,177,2,589,359,781,543,537,70,41,352,176,347,232,942,587,120,230,256,808,929,310,705,731,820,47,257,501,594,607,694,664,559,960,152,914,717,189,293,401,447,6,613,790,271,221,461,761,785,380,306,231,777,456,887,984,902,462,986,556,975,346,875,966,490,262,438,17,470,502,659,21,141,715,457,318,326,565,651,403,735,518,31,399,466,351,881,40,416,354,147,711,550,903,788,538,403,603,106,66,575,558,421,416,262,298,488,667,244,158,152,625,194,580,595,681,259,337,255,488,987,20,822,186,329,525,474,264,49,44,370,380,275,544,499,255,577,559,621,977,788,245,278,698,867,630,81,735,127,812,183,883,304,235,241,638,252,497,810,772,467,702,26,678,173,417,724,720,658,218,315,948,924,158,772,521,914,187,942,43,907,147,319,252,48,119,170,525,101,355,39,132,694,649,779,156,233,193,568,110,822,607,115,109,998,959,429,986,378,529,193,485,93,703,13,375,591,889,485,639,265,2,10,233,97,211,89,317,634,184,771,231,376,140,911,571,770,355,553,384,623,201,87,74,218,765,674,474,300,184,663,26,764,566,526,69,115,422,766,554,5,187,15,233,344,283,546,35,244,303,379,916,465,742,244,96,539,702,109,473,442,145,266,2,930,966,947,70,164,70,56,267,3,865,48,537,505,763,112,484,623,844,13,675,89,813,998,538,394,974,355,166,394,520,316,274,831,576,167,231,684,120,247,139,545,124,781,172,333,179,193,725,639,246,818,476,256,849,740,433,190,832,721,137,200,768,18,684,934,923,258,884,817,192,64,759,398,437,924,544,975,56,550,307,526,968,159,194,646,922,343,629,469,264,219,439,784,819,803,302,258,883,318,343,277,369,246,76,943,904,38,393,800,758,826,134,405,591,585,559,687,326,953,527,377,467,249,523,821,438,582,222,503,300,292,980,458,158,338,62,757,589,401,821,1,10,113,350,196,619,550,40,663,287,371,712,644,397,366,629,730,582,131,198,304,611,950,756,884,671,43,230,341,185,558,433,597,981,449,576,857,738,194,161,878,572,264,480,504,878",
    "487,191,121,525,536,995,169,870,579,157,191,917,113,438,42,159,848,910,268,126,914,371,156,472,985,394,908,619,477,225,676,301,727,542,518,666,623,1,64,974,289,423,572,941,510,484,558,611,745,536,862,825,66,642,392,319,168,800,543,78,419,894,969,554,779,818,297,802,614,516,546,116,160,234,284,560,34,249,840,547,142,736,203,407,432,637,240,42,533,717,226,433,215,307,391,797,376,317,878,782,911,998,586,583,253,712,497,820,918,511,637,311,578,105,503,604,123,143,978,120,775,423,347,379,142,115,868,822,163,642,556,260,668,106,631,914,38,629,881,292,209,144,102,497,713,489,21,708,737,995,991,686,171,610,829,99,716,135,851,5,862,190,325,913,786,142,731,512,777,259,461,89,484,727,361,44,448,984,935,516,651,488,763,166,732,308,29,828,614,949,837,613,242,464,354,439,79,647,784,61,930,862,678,899,164,379,918,481,271,394,171,50,976,748,692,595,557,969,40,603,388,417,806,453,781,678,670,267,619,660,486,493,238,43,977,864,750,582,749,22,863,58,0,575,502,372,103,386,511,440,806,122,343,301,498,706,798,902,730,452,72,271,996,876,157,760,833,472,23,905,430,829,760,578,618,934,249,768,789,570,846,477,233,505,816,133,140,298,662,26,833,684,755,111,239,896,662,657,144,201,593,235,564,509,507,556,630,297,189,504,200,881,442,456,328,791,8,651,166,60,282,938,39,142,364,871,33,539,260,624,332,700,734,590,102,842,702,818,220,292,341,829,382,427,37,339,280,335,616,468,518,292,739,327,459,654,749,491,244,309,160,877,430,67,101,246,217,913,352,22,428,3,210,173,753,459,17,302,127,751,54,953,568,742,899,685,783,260,567,945,296,211,441,782,649,586,269,259,522,627,209,242,639,697,165,966,459,646,551,613,952,539,957,452,990,400,18,368,392,368,465,164,109,265,761,573,727,921,498,605,834,994,552,422,867,609,943,977,66,345,258,607,281,13,447,309,546,487,344,27,440,584,630,171,94,222,859,568,917,550,154,547,739,19,203,250,718,902,966,53,552,633,365,913,839,27,820,371,747,983,314,82,626,126,452,408,257,523,713,749,516,460,538,824,297,367,90,825,315,209,304,431,626,950,62,348,867,550,737,759,222,715,88,376,69,664,979,262,781,777,246,876,56,142,540,290,902,427,803,531,554,3,621,766,401,502,700,308,995,921,742,148,662,718,880,311,700,605,226,648,795,775,546,195,709,190,907,183,841,155,993,985,336,260,243,461,968,954,305,244,145,31,511,668,158,384,880,545,858,803,336,660,127,786,591,491,731,543,78,534,905,532,84,390,307,662,145,572,444,872,9,392,374,652,866,86,346,313,380,622,22,501,590,595,431,885,790,72,969,120,936,765,652,147,75,936,428,173,764,899,98,759,507,437,797,417,894,796,325,251,49,743,365,656,422,110,756,12,355,947,795,328,277,378,213,777,514,332,459,353,609,512,135,817,174,703,610,477,31,31,204,795,566,830,491,336,10,236,430,671,820,909,545,399,131,622,610,436,354,182,732,445,133,549,24,436,908,438,928,615,932,94,983,488,272,472,456,359,308,104,449,738,283,91,333,226,809,565,386,576,817,883,789,775,973,999,935,948,338,893,221,991,527,507,316,222,617,421,829,519,985,516,20,692,403,180,770,553,508,716,508,989,327,153,877,349,362,755,605,516,488,201,786,614,835,208,427,856,688,360,563,70,81,969,705,493,687,155,170,390,548,387,811,502,193,243,664,286,349,944,232,471,648,764,715,130,54,375,336,689,758,554,100,750,532,151,179,884,315,390,271,491,772,889,282,285,412,253,415,421,377,366,75,918,949,967,819,733,46,382,491,321,453,218,98,112,416,566,571,130,687,895,127,266,439,839,964,595,258,261,876,628,733,679,77,195,82,387,943,635,203,131,832,154,603,461,138,786,126,98,878,729,141,126,422,138,264,857,542,19,205,545,15,835,514,285,533,82,359,405,259,290,204,164,583,712,86,966,221,726,228,338,679,982,947,502,604,385,829,454,56,855,732,808,943,254,396,977,956,930,387,797,941,381,192,892,91,541,300,5,249,464,672,537,753,323,29,366,128,779,91,119,573,402,492,205,788,212,107,883,667,757,530,149,783,508,126,459,879,193,686,863,819,427,157,476,214,740,726,792,560,820,408,407,680,705,66,673,162,197,524,560,84,394,169,158,935,777,89,60,252,754,590,638,586,325,606,7,172,369",
    "58,650,543,801,989,386,870,415,427,333,280,712,199,785,306,429,362,357,340,721,516,12,604,972,61,875,184,755,630,911,609,291,851,256,129,255,646,331,674,842,862,855,108,215,396,974,395,466,410,686,844,961,737,958,241,308,371,152,209,639,660,561,230,457,759,158,502,257,845,963,748,400,427,262,197,510,230,826,537,763,920,920,937,390,810,715,190,398,566,110,639,276,479,487,548,114,664,251,120,835,244,580,345,336,914,753,417,556,238,269,554,384,355,536,267,257,148,765,530,448,952,209,164,818,162,181,237,915,123,277,851,838,586,512,259,84,896,602,982,306,549,210,174,127,765,836,177,916,642,139,939,616,475,684,39,640,456,890,884,63,333,654,314,918,338,489,327,778,951,668,775,154,430,180,754,674,230,745,628,722,826,999,108,428,686,135,775,971,93,281,280,690,700,532,375,480,335,13,505,936,214,29,736,420,760,845,822,636,581,630,181,446,706,559,791,719,992,813,327,854,546,576,72,492,191,198,312,26,947,148,237,933,433,179,56,663,445,40,164,805,965,952,575,0,84,980,269,762,969,736,845,667,632,740,989,569,925,674,465,772,803,861,293,169,68,407,132,958,979,819,85,609,814,673,373,445,497,66,621,180,741,150,851,650,832,179,187,810,411,936,289,858,870,466,829,266,496,328,193,633,251,284,831,183,554,44,150,66,178,441,273,120,452,879,362,82,766,882,345,135,798,747,940,186,357,739,635,328,903,277,934,92,36,43,281,517,590,576,234,496,256,587,887,800,899,549,944,913,428,856,380,720,930,819,584,917,624,46,289,220,856,70,391,454,554,172,853,916,86,491,843,422,797,188,819,772,803,131,476,167,568,887,69,651,551,696,303,664,535,811,727,375,314,446,741,245,806,391,315,801,980,254,612,678,640,346,834,810,461,713,949,183,216,180,726,500,409,732,151,835,575,287,551,461,190,412,258,344,78,56,639,108,317,607,200,57,768,133,274,24,556,166,842,895,857,892,521,596,485,476,991,799,186,606,38,780,426,557,521,354,836,444,677,988,436,150,887,884,697,21,320,30,824,5,985,199,379,29,95,480,161,505,374,670,925,957,337,409,329,307,450,951,161,572,887,712,843,1,633,979,217,980,598,333,228,248,547,14,565,865,719,391,223,34,812,203,515,619,549,369,746,476,332,759,243,73,323,642,872,182,88,986,497,726,645,574,42,591,781,902,589,619,755,1,331,223,908,671,823,635,445,44,813,799,523,395,78,943,787,119,946,347,936,29,708,870,160,37,232,718,206,710,41,913,142,408,998,844,846,355,87,309,922,899,87,460,67,13,823,119,52,909,623,68,85,583,172,565,776,497,235,776,233,748,55,63,729,563,330,556,218,515,24,793,784,416,950,833,417,710,729,893,773,151,959,600,809,737,54,965,447,91,574,370,496,152,322,328,659,362,287,488,193,274,53,904,815,672,302,369,371,707,855,892,45,850,225,685,793,926,834,681,232,335,915,423,847,145,278,989,117,835,2,81,998,889,489,82,772,39,356,530,200,606,600,134,212,894,143,582,664,404,322,141,607,418,555,944,14,358,675,28,260,638,617,323,928,171,734,3,950,827,355,356,414,584,5,370,452,902,748,571,968,297,770,475,307,635,762,989,219,262,300,827,829,237,940,950,886,671,3,77,312,599,3,312,136,583,459,947,794,762,962,159,110,165,413,356,270,527,903,919,65,705,519,171,727,363,823,811,566,831,273,926,803,845,226,957,992,495,270,638,66,659,834,95,17,691,124,248,4,555,88,115,224,556,803,865,68,585,193,651,766,380,572,824,1000,144,653,621,463,367,614,808,299,801,929,697,322,71,148,369,760,540,396,926,324,154,286,697,316,452,367,585,965,134,864,122,428,62,53,702,953,780,726,292,786,575,263,352,704,94,919,505,938,429,186,558,8,737,853,545,159,240,175,71,112,732,332,416,97,754,761,805,829,440,160,769,171,930,15,746,799,286,248,839,244,518,457,351,19,238,954,745,426,309,254,571,680,442,994,362,785,476,860,299,798,137,810,187,388,829,663,325,489,248,566,372,929,160,76,504,478,677,273,421,297,201,410,523,67,93,544,596,95,843,111,421,456,164,747,66,114,374,947,475,854,175,113,662,956,595,793,21,517,621,366,759,237,937,703,324,169,823,647,716,832,874,714,532,110,55,400,633,305,171,704,94,262,529,173,513,448,433,754,16,627,608,866,767,718,669,609,17,991,341,260,588,716,792",
    "890,287,160,546,647,382,783,442,377,211,797,603,514,49,780,730,535,685,942,741,712,636,537,812,60,119,688,299,958,804,187,340,891,423,707,978,481,944,959,33,659,127,177,422,269,572,736,716,886,14,498,978,522,280,934,163,89,346,185,170,373,253,853,860,331,201,375,955,679,362,963,624,267,604,806,511,318,408,587,840,643,65,238,288,133,730,677,134,366,99,381,330,13,953,470,279,935,236,328,392,668,111,178,486,902,830,145,365,505,375,443,694,269,936,80,314,840,741,184,746,142,385,947,846,41,804,254,791,662,341,202,925,10,264,935,143,146,544,887,68,576,907,757,233,722,951,502,194,396,897,50,507,729,758,716,777,496,432,111,39,668,241,230,875,530,878,7,743,826,162,549,649,40,760,210,768,646,680,521,771,77,192,883,747,96,149,705,605,51,473,195,966,651,709,396,679,660,985,252,994,348,42,849,555,144,86,585,219,344,911,56,479,65,657,698,33,131,552,795,62,242,528,860,326,17,588,777,877,307,885,399,711,344,634,967,996,988,696,507,671,871,424,502,84,0,514,180,485,437,259,232,106,789,503,148,245,334,288,640,298,762,874,775,900,484,530,690,780,26,411,951,676,432,406,670,653,930,314,449,922,453,877,212,182,478,537,595,240,944,910,73,336,353,514,541,113,187,463,590,30,129,227,819,695,409,794,389,137,222,897,434,442,257,160,511,455,426,768,659,615,430,539,255,262,906,278,111,229,710,309,295,769,466,946,403,366,654,1000,313,156,511,419,214,964,457,687,687,119,993,754,469,355,908,399,86,875,490,673,447,147,949,801,317,374,668,469,809,143,487,241,595,609,476,246,860,955,582,675,500,328,632,107,970,948,271,371,178,451,238,375,573,681,145,839,666,648,691,909,968,27,883,875,717,453,374,921,855,606,896,590,878,407,58,371,306,638,730,953,373,775,885,963,541,3,625,167,64,96,585,399,966,119,132,206,934,609,831,952,734,92,28,697,473,914,473,406,606,107,771,771,822,595,70,349,886,861,121,681,193,273,56,980,363,411,217,320,48,57,338,728,219,550,892,854,535,764,903,63,172,673,82,590,747,67,935,91,183,916,841,246,129,541,506,11,505,993,416,512,644,60,974,733,279,97,511,986,848,621,422,451,190,807,61,674,485,108,415,269,346,743,785,911,243,784,656,753,723,52,442,919,93,512,3,49,500,668,179,344,966,902,662,840,820,779,506,119,484,731,349,728,575,528,834,806,1,940,840,631,274,547,168,182,935,561,987,369,708,632,883,75,659,735,481,727,667,854,320,509,597,909,455,246,781,950,182,516,252,688,324,556,292,799,304,146,188,78,898,420,290,280,313,882,579,186,815,361,597,556,635,120,886,9,222,755,419,434,244,169,168,957,734,166,719,951,896,819,251,937,904,924,626,809,388,234,170,756,185,433,121,93,419,372,470,888,65,65,390,814,514,566,779,128,927,720,173,859,351,702,191,47,495,190,589,175,931,407,200,348,600,693,544,441,161,899,450,696,496,558,874,889,799,34,586,248,349,143,996,637,159,698,326,104,822,854,65,122,986,707,530,656,427,387,10,741,580,609,312,614,530,909,420,192,538,890,153,714,864,10,426,726,608,221,585,290,740,678,420,647,796,148,342,940,124,662,833,310,794,223,187,701,263,905,208,535,810,299,889,370,867,668,354,467,741,765,3,555,284,109,436,86,439,435,794,420,833,940,55,613,184,188,994,172,3,987,637,92,927,857,744,658,181,525,816,816,251,259,155,279,477,849,905,882,624,446,713,142,279,49,446,4,179,70,603,317,21,20,633,361,767,748,108,991,274,327,834,731,489,358,583,618,845,122,433,46,851,879,344,302,903,463,279,497,805,90,814,778,30,174,207,219,973,780,542,818,531,694,582,901,972,276,808,670,700,248,665,786,878,822,189,671,367,311,592,957,853,548,380,826,988,563,232,710,946,724,560,383,600,606,25,880,660,669,82,853,848,735,544,669,76,139,970,785,633,631,874,322,723,603,87,286,112,945,511,579,134,464,80,577,359,34,264,602,17,641,677,378,678,44,832,694,166,827,570,777,663,852,278,66,209,275,805,200,425,24,612,579,466,698,453,541,866,582,585,99,360,984,832,537,926,152,706,234,62,413,559,64,449,420,500,56,657,842,374,715,859,823,720,24,281,408,830,364,534,241,335,60,208,471,600,193,232,669,954,940,814,343,530,603,324,40,610,48,318,295,831,624,416,510,111,451",
    "700,301,848,827,96,119,324,696,734,631,416,907,789,268,455,78,674,665,455,663,322,621,557,483,184,517,615,299,57,568,217,474,422,259,744,347,191,318,934,910,244,717,557,883,717,548,861,348,589,709,532,88,247,73,907,774,89,276,116,540,769,119,747,981,24,658,534,378,790,742,464,247,38,490,737,589,246,176,421,228,237,265,715,611,524,347,775,80,333,12,188,765,72,414,767,628,749,294,665,124,105,589,27,159,710,776,205,692,63,417,956,357,459,981,157,34,132,240,647,928,597,375,379,206,258,149,43,984,448,116,709,923,282,706,998,542,504,190,280,405,772,254,90,464,977,115,870,106,945,146,144,356,43,382,136,614,811,19,468,594,908,646,956,795,276,752,397,148,848,377,694,345,9,958,876,145,980,509,710,824,430,639,819,303,963,389,283,204,454,492,664,542,469,7,376,685,439,785,702,376,338,823,419,197,649,312,257,142,884,376,540,731,82,209,660,7,285,969,939,430,606,304,154,721,703,349,985,83,662,966,786,963,854,594,747,32,902,14,209,861,786,534,372,980,514,0,610,322,827,802,249,60,206,680,738,418,151,469,571,11,564,219,622,604,906,34,871,935,366,831,157,325,574,181,952,951,241,675,489,901,368,947,609,54,530,596,407,376,279,481,876,562,906,355,21,92,134,303,600,759,365,734,16,672,982,133,176,954,77,392,41,921,273,819,653,764,638,814,626,308,74,723,64,414,20,674,515,481,488,825,910,763,745,535,915,398,211,725,394,21,5,125,757,970,705,815,208,904,692,48,225,208,467,247,917,422,236,655,582,646,870,606,707,381,519,216,459,695,978,710,433,479,978,454,432,394,281,600,4,386,366,590,154,195,607,779,91,993,94,957,579,989,831,36,460,615,886,676,810,90,892,342,636,697,472,866,673,194,251,739,352,949,69,187,111,573,308,138,492,559,39,866,713,637,274,377,61,754,46,558,373,981,413,655,330,523,446,825,33,345,930,664,76,916,115,863,696,262,234,948,646,549,666,376,387,287,190,970,724,627,36,72,388,604,57,19,8,401,613,317,77,917,622,42,837,632,649,466,335,112,99,282,1000,117,621,700,299,431,844,93,527,827,939,41,846,468,822,745,4,465,815,127,178,506,365,978,288,280,879,922,445,515,476,622,945,695,632,98,259,52,48,875,838,267,632,786,955,899,459,123,676,767,4,743,900,258,208,216,714,433,334,956,311,674,726,630,275,338,312,164,59,576,262,57,936,613,774,560,285,803,636,332,856,517,8,848,724,565,679,645,689,127,316,63,577,60,267,734,830,512,831,860,146,47,439,914,566,548,552,959,533,490,407,926,409,6,202,932,649,308,676,200,707,533,709,573,354,118,987,348,981,454,754,556,194,849,70,185,709,878,31,119,472,600,238,869,370,113,875,89,948,852,290,369,364,276,934,980,100,861,542,9,38,301,637,119,486,965,766,539,312,91,656,463,483,971,72,464,583,733,364,408,111,57,254,183,117,352,200,451,281,806,455,4,37,407,215,153,854,761,79,666,377,337,697,10,503,87,737,683,175,557,695,743,751,97,483,897,140,644,340,70,291,160,720,895,516,985,945,698,593,254,471,383,927,939,409,788,46,448,728,289,262,227,112,302,951,542,827,493,320,116,845,285,231,422,412,273,166,64,995,663,378,90,442,437,131,209,345,76,879,196,637,251,47,608,815,643,177,923,466,365,250,164,322,997,157,655,101,696,891,523,623,704,701,144,752,247,988,245,91,245,62,712,787,334,638,922,763,959,741,204,261,103,623,31,373,27,328,536,140,395,200,60,388,253,356,671,46,960,345,641,589,397,865,264,627,399,624,695,119,568,882,943,947,999,348,278,727,505,72,288,978,381,8,767,279,279,591,594,498,974,310,379,859,529,929,685,155,520,5,696,444,871,916,489,335,158,338,521,254,321,755,282,607,655,465,775,270,306,614,460,337,405,26,959,56,345,455,255,349,221,990,764,255,273,568,811,950,566,8,767,708,984,302,990,813,359,714,184,207,157,805,656,600,889,316,232,186,67,937,691,991,969,165,885,471,962,350,120,559,357,818,495,748,449,412,49,516,847,208,104,878,997,58,958,800,399,804,601,166,527,402,367,805,914,759,289,920,96,455,965,577,442,215,62,184,620,701,386,585,559,672,274,608,82,441,208,902,545,473,343,313,144,936,994,86,567,347,267,841,433,811,739,803,486,531,244,353,424,11,769,308,106,125,401,184,197,322,561",
    "121,735,724,99,247,123,281,720,814,751,64,475,31,600,635,782,31,519,325,19,859,985,909,217,783,327,357,833,62,952,796,355,139,140,391,578,343,903,459,526,34,71,317,444,282,895,198,873,86,858,564,952,692,337,739,496,116,314,377,884,768,113,350,452,579,546,722,152,473,726,914,627,912,261,774,967,994,472,327,294,816,68,305,876,20,36,423,230,290,12,961,944,503,715,118,752,891,116,24,628,659,348,912,276,224,174,17,9,122,659,89,338,681,912,435,728,357,123,427,21,885,820,784,469,233,251,150,713,457,24,521,439,582,235,102,448,614,200,337,6,907,744,450,22,258,992,462,379,91,695,500,715,162,367,605,738,336,978,245,718,855,423,795,526,960,127,200,22,161,606,427,892,924,57,265,993,668,165,308,745,663,600,906,223,85,340,510,563,525,174,682,221,154,543,439,291,755,299,28,895,341,553,387,11,907,812,920,120,817,557,354,548,903,505,38,6,205,490,613,203,990,674,821,20,386,334,380,241,90,433,956,92,7,142,895,304,322,172,881,413,401,493,103,269,180,610,0,515,379,601,665,136,929,878,157,401,179,409,99,85,364,369,725,214,180,115,548,846,860,614,992,621,433,966,859,255,211,61,659,952,874,381,568,640,921,96,127,599,656,280,740,925,493,198,83,139,194,719,819,323,563,852,68,579,791,714,996,485,300,25,429,573,505,506,733,416,975,886,375,292,450,439,89,883,820,455,725,53,103,527,494,613,688,670,358,754,352,995,125,506,140,617,420,586,924,629,616,770,275,276,427,144,728,228,604,206,941,593,272,412,967,228,35,573,394,731,798,339,496,997,208,951,185,870,725,830,877,998,618,252,999,9,69,680,533,417,16,192,666,679,710,954,631,143,132,611,718,110,304,412,671,753,140,673,896,777,616,729,167,875,37,428,84,620,444,990,589,271,508,990,618,570,626,278,497,190,420,114,172,642,661,351,838,575,733,200,466,985,959,149,85,842,529,622,672,461,470,58,387,222,548,893,203,919,139,684,901,930,725,398,518,699,649,401,274,185,59,329,770,33,996,556,237,557,31,277,533,585,189,670,398,101,862,815,6,931,406,35,646,621,482,891,429,882,50,102,318,985,999,460,375,151,927,889,802,555,917,18,673,366,915,94,326,690,282,717,96,871,10,21,214,868,240,808,26,279,596,95,278,125,950,255,856,797,916,249,908,697,288,131,977,90,150,566,165,584,196,22,183,867,94,205,371,327,175,706,281,680,673,965,963,184,844,194,787,112,342,191,927,365,170,621,437,924,184,512,374,62,825,949,935,965,286,596,344,404,410,270,755,722,816,476,365,435,561,707,926,295,773,428,245,91,928,452,276,758,128,420,186,196,920,853,341,310,929,495,913,412,559,221,425,455,993,841,601,705,276,690,717,718,202,725,696,277,130,485,865,385,303,141,545,520,122,633,903,842,639,796,468,430,176,963,815,102,512,738,301,400,156,291,100,48,627,673,834,817,111,661,521,615,276,982,809,363,640,1000,216,366,441,280,384,121,394,67,336,137,506,492,874,619,562,818,666,507,900,514,290,628,606,642,811,646,609,218,448,750,185,910,615,706,109,831,979,288,27,513,797,38,562,498,439,376,79,912,725,803,333,984,371,752,339,810,254,631,996,703,767,665,420,456,756,443,911,329,389,81,237,961,11,595,669,808,988,246,858,328,252,711,875,201,78,318,386,589,456,902,345,257,729,300,22,455,643,461,528,401,19,168,836,752,27,892,968,841,183,921,48,720,911,895,823,157,614,315,643,629,504,410,225,309,524,765,657,439,671,688,263,793,66,261,883,89,827,454,679,701,969,470,473,719,273,644,166,226,363,224,727,629,196,765,562,551,973,971,924,67,721,733,833,47,878,559,165,211,151,298,276,106,834,184,101,578,756,844,50,134,895,796,202,110,368,695,382,848,156,270,812,45,628,559,618,820,517,662,44,452,392,79,165,673,738,621,193,932,236,670,212,36,196,547,241,5,88,189,664,490,226,226,332,343,946,172,832,441,177,123,376,547,82,516,677,325,729,903,388,326,447,453,709,963,83,115,843,569,315,296,418,619,552,724,771,821,492,985,956,543,890,223,579,259,116,429,162,222,962,78,633,581,931,16,188,406,597,94,240,396,281,758,620,324,67,95,844,862,643,888,862,359,874,948,654,367,318,460,546,269,790,307,489,5,471,834,872,465,185,376,427,415,177,365,705,734,53,732,914,466,565,999,764,244",
    "852,765,366,741,695,876,584,732,621,496,286,901,26,613,671,852,193,584,869,32,241,962,379,171,787,416,145,76,87,355,455,541,674,355,927,644,584,757,354,122,127,408,428,173,145,771,477,542,285,624,831,530,265,244,888,945,215,40,703,84,452,105,758,597,251,915,826,551,458,704,871,368,288,944,461,103,187,59,380,131,662,363,338,654,741,183,175,680,448,481,886,68,595,206,195,426,622,882,810,776,371,866,561,907,575,619,877,915,27,391,927,844,127,471,139,614,12,781,6,318,576,310,23,283,468,938,189,469,439,789,639,435,151,238,53,20,378,521,788,545,122,376,313,935,823,442,105,361,637,940,162,983,292,19,246,993,937,843,14,475,198,380,374,127,228,18,569,233,738,573,65,38,487,486,697,287,939,408,576,998,293,836,626,648,104,143,895,801,396,993,720,81,195,989,544,965,828,603,53,868,238,399,662,602,138,278,552,322,765,253,277,556,462,292,823,413,657,348,852,359,168,44,473,264,476,122,105,137,46,768,528,854,429,578,699,459,126,574,257,634,473,747,386,762,485,322,515,0,959,833,845,136,304,805,583,59,47,854,75,693,203,693,462,858,82,509,287,765,322,390,124,977,468,180,50,741,508,621,816,240,917,703,811,956,429,933,818,288,495,572,789,312,2,418,322,887,230,544,539,436,347,888,189,517,174,691,930,840,710,713,186,194,759,707,124,1,307,259,309,150,406,730,455,499,506,793,91,598,654,20,522,163,614,117,47,649,489,127,656,355,326,227,287,396,456,403,415,430,797,924,576,183,224,208,402,900,274,938,631,939,227,942,585,780,791,941,838,227,536,569,828,548,758,413,463,333,955,156,987,186,573,448,644,604,312,470,909,393,459,11,634,998,999,191,973,862,186,301,618,622,60,485,768,979,870,761,21,352,536,374,44,780,577,480,733,912,409,171,604,42,515,180,47,256,469,636,546,358,275,214,670,504,958,488,531,275,162,869,623,665,218,563,408,322,273,582,111,707,999,898,983,571,171,911,548,455,410,192,520,556,93,719,162,592,969,744,925,604,509,266,736,750,405,160,658,63,284,548,485,546,839,55,636,138,43,431,708,657,455,535,999,520,614,15,954,826,488,711,438,89,479,932,2,154,6,690,912,989,599,602,577,33,283,605,806,276,786,390,293,374,61,51,450,668,484,609,457,336,549,816,92,639,815,80,898,366,513,448,378,192,636,280,549,638,69,724,890,892,115,498,819,861,444,980,819,817,339,351,836,659,994,609,410,369,612,763,622,813,151,633,710,294,729,896,229,21,950,328,522,758,461,609,615,590,571,479,567,929,336,206,830,644,111,603,393,151,300,624,979,752,708,392,102,572,566,888,580,894,141,578,499,80,676,846,870,257,319,222,975,523,720,274,34,770,824,3,125,22,447,212,402,330,860,671,927,164,281,795,117,159,101,256,407,608,651,324,12,26,113,557,438,806,961,805,222,346,902,993,837,61,633,756,700,780,277,812,653,21,568,462,740,983,124,333,852,869,101,706,885,509,350,827,481,681,769,317,297,609,479,250,452,57,411,394,30,269,489,184,696,789,106,444,740,943,293,657,672,193,137,130,267,579,181,591,136,606,397,419,609,843,913,871,834,119,174,894,575,980,999,188,754,210,908,978,104,826,541,714,186,685,399,165,783,375,897,95,584,921,772,503,849,83,400,900,885,67,71,309,425,67,144,768,660,841,643,141,406,90,3,873,328,854,268,409,726,225,982,14,511,253,408,138,608,378,840,832,994,827,149,67,794,820,895,620,895,387,341,249,427,342,918,306,920,750,493,15,189,427,96,354,53,890,508,359,607,887,341,341,999,850,233,419,462,887,622,967,922,52,206,622,147,619,239,583,941,522,127,700,484,18,796,452,839,747,152,227,75,222,271,114,289,656,106,311,699,580,621,218,396,736,436,386,267,506,840,501,980,26,299,734,340,168,838,695,656,678,499,359,828,601,343,575,140,868,899,836,537,15,795,978,793,326,418,432,779,920,711,572,246,188,183,67,312,354,322,324,486,877,441,611,114,947,993,860,653,684,365,190,920,277,17,225,936,739,789,725,392,710,759,960,709,516,658,113,768,110,917,268,354,171,342,48,690,906,72,303,280,773,776,293,245,332,755,662,592,11,988,537,345,264,483,448,277,215,934,866,210,504,404,792,9,588,727,237,349,350,838,963,271,207,2,32,581,431,92,514,569,232,292,122,677,887,454,180,720,750,23,185,636,825",
    "700,117,765,130,888,440,278,153,370,404,23,368,708,827,116,374,605,16,3,475,515,316,949,78,861,460,102,906,274,844,115,59,911,918,489,951,782,249,510,487,545,410,310,456,884,477,843,602,384,656,228,159,633,431,720,715,625,11,702,778,887,498,724,299,713,427,224,448,677,441,381,594,599,13,315,719,752,877,953,451,606,442,480,340,150,231,693,382,590,64,335,598,687,667,564,530,819,641,2,405,690,768,297,721,198,192,743,204,818,7,865,206,21,63,805,812,719,402,136,361,492,770,857,868,300,92,893,72,226,240,866,538,489,69,806,686,953,352,207,803,170,182,65,93,968,717,609,146,684,679,527,1,917,72,386,343,885,953,507,853,325,944,948,447,262,463,580,749,722,87,475,745,633,103,788,886,83,362,903,45,392,283,803,155,228,240,785,824,310,72,750,436,216,925,438,184,485,531,405,264,78,452,582,371,736,940,735,289,988,357,737,572,341,731,681,48,258,983,212,504,951,734,644,473,586,532,898,305,694,902,637,401,891,640,319,857,581,968,112,238,56,463,511,969,437,827,379,959,0,427,33,146,383,567,38,851,99,175,744,540,382,682,675,348,434,807,1000,257,85,313,5,863,941,381,953,840,685,130,42,66,384,568,128,971,513,191,451,562,352,251,840,498,433,473,770,482,966,212,485,627,596,902,997,624,649,402,903,6,28,862,174,544,188,498,237,305,144,456,739,66,955,456,344,816,895,14,911,17,641,746,608,982,140,926,235,400,275,580,301,682,123,156,759,47,481,755,453,496,177,985,472,93,868,352,354,308,387,2,780,223,974,154,472,808,961,683,756,311,785,217,693,615,143,850,28,944,110,473,240,667,249,823,83,436,764,452,897,770,819,567,792,264,865,847,356,811,728,70,469,534,891,132,810,911,278,504,887,65,525,653,627,500,548,884,867,61,418,598,142,517,264,44,323,477,281,336,824,531,932,929,345,249,774,313,804,655,131,199,726,798,173,522,185,270,447,967,488,52,191,904,161,75,541,207,646,104,591,615,723,695,784,612,509,398,5,55,589,499,330,658,532,239,980,633,126,449,818,643,301,664,332,577,814,447,112,543,509,955,214,949,393,588,80,223,289,558,415,813,280,695,760,213,370,953,191,447,836,783,181,474,640,142,852,382,770,408,323,983,918,981,356,938,498,89,891,299,949,811,772,861,144,384,401,117,252,206,241,956,853,661,825,515,43,705,592,335,519,651,413,90,161,979,96,937,167,710,918,410,29,754,733,477,187,256,779,422,13,674,194,383,506,82,938,21,753,168,553,755,78,816,341,123,451,618,906,941,482,188,208,721,660,375,946,976,364,493,294,703,730,145,162,807,104,466,803,712,589,97,286,728,259,349,787,62,364,261,261,47,37,87,373,925,152,746,372,525,485,679,580,502,299,976,208,930,323,264,581,986,600,838,806,611,521,57,533,737,237,228,623,611,547,517,613,310,760,487,43,55,971,796,131,288,991,359,939,321,91,832,407,499,614,635,583,751,524,119,779,119,650,570,871,666,907,713,90,37,581,222,108,39,424,888,195,714,270,881,614,481,823,877,413,868,902,411,604,237,520,282,389,53,143,570,855,150,448,900,473,435,880,416,644,643,128,307,17,452,832,3,234,268,638,364,598,718,563,698,24,368,228,887,303,156,349,507,487,523,25,120,200,152,985,791,614,202,147,18,698,55,304,427,508,968,112,243,696,444,181,386,738,330,852,170,115,126,730,434,354,859,960,808,951,206,316,188,404,155,43,117,96,508,59,431,633,320,850,6,9,426,150,631,188,403,394,389,540,903,246,258,430,786,957,368,991,933,927,539,136,171,685,405,998,42,990,743,113,252,697,723,131,455,573,566,777,932,100,694,409,210,674,818,677,997,662,540,108,914,391,825,746,531,610,611,104,683,439,410,610,640,435,561,522,757,855,497,715,4,388,249,89,720,537,156,426,298,395,45,129,409,582,297,603,922,102,540,296,570,371,484,411,27,305,315,387,301,442,179,264,908,365,699,222,239,224,651,186,368,717,535,746,38,631,139,510,162,616,181,756,841,815,504,779,300,998,123,761,586,408,859,184,376,868,86,100,251,190,338,724,30,117,562,767,53,973,502,47,450,469,878,14,713,89,629,837,30,853,955,873,693,906,405,241,361,681,124,356,583,978,22,721,816,364,742,959,185,398,742,467,914,429,712,815,430,689,623,478,844,736,13,277,92,336,404,709,623,860,549,649,731,539,81",
    "888,501,598,337,276,695,219,578,279,917,26,40,46,643,194,237,18,574,35,4,312,226,444,440,768,766,64,372,550,937,707,974,312,41,711,918,737,797,399,237,346,144,509,828,561,444,470,502,257,28,753,50,504,5,429,468,32,495,403,79,414,262,664,262,321,737,764,594,628,112,203,573,745,20,419,993,812,647,723,424,316,846,116,86,275,759,676,515,287,10,911,763,927,526,656,142,359,515,561,435,186,476,686,564,150,713,61,347,616,508,397,387,327,73,493,218,828,793,866,522,517,851,712,424,326,25,676,269,188,963,556,72,702,702,880,964,685,694,997,352,221,58,660,960,348,927,802,943,468,881,957,997,127,824,924,649,588,479,514,425,181,515,150,303,981,16,159,795,501,824,162,20,732,986,713,261,574,542,106,82,952,291,263,681,720,377,776,799,471,899,464,757,668,660,342,804,883,500,420,641,217,567,585,856,313,421,582,221,285,733,745,768,779,717,372,563,664,958,647,342,473,507,316,663,690,325,268,999,644,975,660,982,736,496,673,235,453,681,803,523,340,699,440,736,259,802,601,833,427,0,252,343,286,849,186,310,234,689,140,797,599,484,667,316,331,246,933,590,590,610,385,979,260,924,245,260,809,175,921,669,838,456,849,766,957,684,304,386,514,936,270,437,813,624,904,1,184,726,405,830,222,703,904,507,798,643,51,654,891,325,114,242,285,189,449,908,548,745,940,545,408,677,988,177,342,912,837,291,671,496,790,9,933,773,789,902,806,859,930,855,185,86,152,428,658,915,861,787,634,438,985,586,375,761,812,746,450,595,711,478,396,180,4,708,278,211,781,338,450,553,870,780,670,396,131,403,364,939,804,751,331,453,530,216,725,292,545,534,263,203,7,49,605,414,67,26,997,961,739,903,189,697,757,916,841,60,530,807,719,999,670,450,884,519,161,756,81,355,60,549,452,144,782,590,191,800,593,720,76,315,507,832,584,402,847,466,249,713,780,540,702,387,442,387,102,492,141,588,884,61,975,345,78,681,430,157,946,939,448,903,917,200,526,762,113,735,110,944,910,277,51,545,419,185,314,782,327,972,161,208,46,642,123,338,821,371,95,148,366,977,885,424,954,498,795,32,805,281,30,962,265,747,388,745,474,701,694,4,492,590,102,445,524,924,193,358,657,883,254,176,798,414,449,989,370,970,920,841,149,700,271,556,681,671,261,390,622,582,602,376,204,173,894,20,133,789,704,169,143,910,448,59,709,742,899,734,600,485,386,123,908,149,945,590,497,176,177,59,966,404,319,814,935,413,248,297,909,810,260,70,141,635,105,667,1,10,456,676,181,657,304,67,519,150,20,381,907,242,312,433,965,792,95,80,532,329,159,104,155,938,654,233,423,724,110,322,975,842,681,667,391,491,355,884,711,240,970,995,880,920,602,761,481,46,908,58,926,322,265,119,762,383,69,770,342,562,739,85,544,849,902,897,883,861,454,424,284,483,327,854,589,488,187,87,146,923,7,816,956,771,940,511,949,254,499,341,455,443,523,3,222,137,692,319,184,316,34,177,807,806,13,443,336,69,910,42,204,684,496,394,537,763,402,663,772,139,854,730,948,535,149,399,226,752,202,554,131,678,896,442,117,733,190,904,27,330,637,683,892,514,411,904,856,94,48,443,828,325,664,397,789,479,790,649,619,600,577,305,298,256,568,884,119,974,60,649,999,380,167,404,701,22,254,956,132,836,285,538,946,291,986,39,827,606,282,17,564,317,953,477,978,76,134,26,474,981,803,661,690,426,357,478,289,191,882,443,708,582,184,720,769,213,940,565,449,432,925,852,441,805,654,599,66,174,372,904,543,164,901,398,198,328,596,71,921,343,194,685,185,792,369,857,950,256,898,94,285,169,398,490,255,918,671,672,936,193,498,301,730,874,81,591,346,736,317,436,280,205,794,197,646,698,271,388,795,558,84,576,338,835,9,635,182,380,602,19,919,671,507,89,591,511,406,649,672,180,685,576,153,375,96,192,580,477,69,351,667,823,167,333,858,990,389,457,589,272,664,465,487,157,137,829,32,973,423,17,387,814,339,40,508,841,815,255,581,308,611,332,490,689,304,6,212,346,918,97,202,225,897,305,703,3,300,281,100,383,663,360,982,453,440,157,843,262,383,322,827,92,919,61,875,714,16,857,934,155,682,448,111,694,642,491,929,658,648,718,981,894,762,476,89,532,664,637,769,526,366,678,487,610,306,662,373,903,158,910,606,210,185,905",
    "814,324,880,869,684,838,120,286,218,846,751,288,854,647,90,901,604,502,23,193,731,67,113,452,820,296,361,596,963,441,514,164,502,463,761,584,37,847,98,501,116,14,232,248,715,416,255,972,129,661,327,932,386,651,512,335,828,183,387,743,301,223,287,587,972,744,302,422,755,236,698,516,858,609,250,533,46,61,977,284,641,919,385,18,937,911,319,904,650,417,289,248,114,803,661,798,891,540,272,413,453,792,572,313,103,678,747,241,458,228,481,669,155,982,304,187,963,298,107,48,424,409,395,778,208,41,426,740,267,319,616,858,304,474,970,436,104,542,644,489,216,60,500,837,393,647,749,226,800,208,384,667,382,701,763,746,382,879,812,172,908,495,488,618,788,333,481,917,354,530,844,730,796,64,303,671,619,643,282,681,525,10,13,150,55,202,448,912,349,381,292,313,785,425,236,132,647,840,325,865,561,797,371,345,293,196,84,867,928,913,79,136,582,659,916,962,209,740,859,693,939,313,242,882,36,340,695,156,474,823,914,71,841,837,992,304,244,459,764,184,416,431,806,845,232,249,665,845,33,252,0,138,796,155,588,131,724,556,793,351,295,831,210,821,27,398,215,782,96,499,784,131,721,128,410,305,566,157,750,263,30,92,40,228,618,747,188,171,5,160,861,990,311,605,402,391,966,870,575,459,949,260,564,667,513,264,700,523,252,394,495,329,976,489,990,344,445,807,483,505,521,852,818,688,404,466,698,228,17,278,701,456,631,658,575,86,921,827,643,179,867,952,565,682,344,373,910,805,813,488,618,415,742,563,785,487,48,968,345,933,232,743,798,305,928,431,74,274,336,95,574,347,32,40,463,9,858,90,946,869,635,650,888,41,863,49,332,484,243,274,948,278,26,303,365,564,27,720,593,913,176,327,152,334,875,427,905,504,418,264,133,289,596,703,30,923,296,266,198,120,591,982,412,751,369,178,803,63,608,680,510,140,427,852,607,983,741,396,699,164,384,694,915,738,648,539,484,514,735,582,633,713,549,375,330,124,935,684,295,299,583,407,500,642,575,61,152,496,639,423,687,309,317,380,872,128,148,332,498,925,324,27,258,756,716,659,275,417,805,433,608,827,955,616,569,823,690,473,615,187,74,421,963,960,551,645,282,715,372,790,601,859,783,566,100,640,936,590,492,943,676,891,842,955,713,352,358,391,617,842,171,380,137,875,608,436,62,822,106,365,14,828,855,254,335,752,535,440,435,849,547,210,216,894,17,16,19,676,723,560,835,579,927,356,577,576,544,97,850,695,153,541,834,147,577,234,685,354,24,900,414,639,166,648,400,320,883,641,279,231,286,825,670,925,921,732,947,791,454,862,854,997,184,206,763,545,678,644,714,969,625,232,46,305,861,42,112,870,344,726,781,6,62,699,977,352,593,763,350,653,350,489,222,885,951,188,266,828,37,1,518,334,825,17,712,827,599,830,87,454,676,985,814,612,953,222,903,649,221,288,187,591,446,201,506,797,3,699,122,402,599,963,180,539,59,925,197,737,345,816,241,735,966,867,709,286,959,724,619,744,968,559,92,375,953,134,364,220,751,306,998,317,242,282,648,625,323,301,84,939,471,797,346,34,447,436,95,347,191,736,950,558,193,798,733,114,11,820,598,890,301,900,700,411,312,414,448,66,180,97,696,548,493,575,509,238,307,785,692,314,621,644,136,929,844,786,240,164,387,685,439,741,884,429,541,499,417,970,357,62,412,731,820,558,793,293,339,683,653,480,11,709,596,420,508,387,149,898,420,525,433,830,724,573,34,597,955,631,338,753,798,329,348,9,742,573,674,708,361,746,468,817,120,216,327,908,256,141,110,682,149,645,660,742,547,35,603,612,66,584,413,606,268,432,474,712,647,129,262,524,32,428,68,920,589,906,806,189,620,591,82,877,437,327,62,842,789,809,402,207,399,896,875,672,512,430,307,25,778,865,483,980,943,778,363,706,854,624,466,306,367,740,936,866,833,852,434,258,68,628,373,81,157,711,512,556,354,243,618,228,542,528,210,465,194,755,64,911,983,489,214,86,395,104,229,225,231,604,773,548,230,287,42,559,638,631,556,168,729,741,720,155,802,422,550,534,535,773,370,862,920,698,239,5,840,213,506,265,975,769,327,853,317,938,703,908,558,844,432,699,852,238,988,94,656,758,239,203,187,327,317,743,837,833,558,382,630,978,895,866,306,465,913,364,461,190,413,296,639,824,350,688,425,727,826,309,474,2,167,894",
    "546,264,5,743,916,940,909,636,107,328,700,313,492,267,292,587,928,38,211,288,939,381,117,939,732,287,316,289,993,678,199,991,898,354,698,75,992,74,571,742,761,464,672,750,248,548,945,305,242,59,564,16,81,214,872,353,195,886,845,144,202,787,900,796,583,612,198,753,38,964,161,138,416,126,654,605,414,121,289,427,101,616,306,767,380,195,890,584,545,345,963,839,302,366,587,546,570,18,881,236,782,166,31,810,802,976,643,616,127,712,12,891,295,116,52,163,618,977,593,853,218,102,579,397,893,897,657,611,339,273,835,379,182,923,912,802,111,209,822,330,464,891,875,49,581,604,79,302,974,98,268,99,335,55,822,986,234,928,921,239,361,847,986,67,443,60,201,754,685,282,666,433,512,930,138,649,652,433,484,787,450,913,795,598,519,31,683,423,385,908,410,288,340,82,462,348,457,614,545,815,116,950,963,375,478,547,308,913,129,381,873,878,496,991,831,333,111,850,82,773,611,146,767,778,570,724,627,407,149,801,318,711,996,687,217,416,236,564,759,688,173,247,122,667,106,60,136,136,146,343,138,0,375,145,248,372,887,401,463,791,409,194,289,200,223,259,937,496,353,153,234,239,838,951,582,346,871,497,858,840,887,323,558,248,522,997,604,35,787,61,937,509,630,276,690,808,335,564,448,988,564,540,811,280,426,319,548,621,863,131,682,880,516,261,881,319,348,92,708,666,521,267,268,119,224,520,782,560,580,383,177,466,794,258,460,595,955,366,771,835,881,529,565,252,868,952,240,456,239,164,252,493,248,343,15,58,829,556,736,677,444,922,450,457,310,290,956,838,149,674,893,89,395,791,590,323,797,986,308,12,756,394,389,335,496,595,467,351,154,204,476,76,569,639,343,604,769,873,624,556,133,891,646,730,774,884,662,457,560,433,958,304,602,527,671,694,284,334,698,574,509,280,610,38,435,813,630,891,397,35,11,359,879,622,368,934,58,270,685,16,501,441,99,170,147,308,270,975,982,967,641,223,506,584,790,482,494,320,128,97,587,308,399,868,542,808,130,780,552,432,840,23,514,483,830,75,928,47,159,839,71,314,327,217,122,826,370,931,917,960,610,55,34,879,501,409,852,932,82,236,797,62,435,994,802,391,917,996,27,679,969,37,169,747,828,513,49,519,976,148,667,731,125,801,361,12,612,395,156,531,753,511,817,73,922,252,786,830,568,224,812,955,675,823,511,69,621,363,174,861,783,183,34,920,311,638,363,624,11,699,418,576,170,368,898,98,125,284,690,822,10,263,133,911,495,311,552,615,603,544,758,437,131,206,172,672,85,364,963,173,645,660,497,854,341,259,573,301,49,818,524,55,381,557,752,589,396,74,920,835,617,860,690,697,39,276,104,273,575,554,483,465,484,819,806,647,596,968,892,955,642,144,704,12,387,739,494,873,843,966,599,300,435,287,9,613,794,19,295,777,166,357,304,891,829,221,465,443,490,968,669,696,79,169,331,964,18,690,596,828,146,580,551,137,954,300,24,54,882,188,518,964,107,589,555,794,759,682,706,462,117,545,416,403,138,783,216,440,604,346,159,979,234,691,645,201,423,416,742,388,277,13,716,223,132,635,398,639,428,45,423,268,52,239,471,76,507,209,753,982,392,985,615,452,526,432,659,153,748,312,110,299,989,673,920,989,246,47,864,723,743,300,255,356,983,705,94,758,470,869,791,223,406,290,938,123,234,235,128,795,918,613,623,88,883,33,689,199,322,496,928,583,547,112,173,966,231,292,98,70,809,568,15,380,679,13,642,650,689,593,844,874,634,860,582,789,960,15,94,694,927,599,319,621,4,844,703,927,733,193,376,71,274,33,827,91,641,790,7,329,685,114,462,388,718,259,367,585,212,958,545,544,640,359,655,256,313,57,65,750,397,482,916,635,143,614,765,162,556,551,38,610,499,576,454,407,836,782,639,296,806,751,230,480,288,633,439,704,707,645,309,766,540,965,136,935,889,236,908,645,444,138,917,559,607,580,719,16,521,688,515,993,436,235,623,619,552,112,271,747,853,98,851,303,783,324,57,434,619,535,281,469,363,99,884,235,85,639,332,355,301,209,187,736,831,921,866,798,835,204,11,595,356,9,947,480,36,206,666,678,410,458,635,598,542,667,279,135,42,820,749,605,741,430,683,140,895,430,489,720,753,822,445,542,646,671,865,593,950,214,142,354,263,974,805,633,738,445,178,966,766,260,34,489,310,971,293,565,474,612",
    "374,10,389,702,343,338,935,400,442,238,179,235,577,626,11,499,56,975,983,265,51,510,960,483,91,571,468,59,71,129,383,851,551,283,98,702,12,408,984,576,826,73,3,343,498,839,964,354,495,456,782,213,65,644,954,765,597,536,285,453,34,191,474,114,690,694,660,177,657,301,481,424,227,270,856,652,247,3,139,124,974,649,430,873,78,149,798,844,685,840,240,596,559,212,4,192,66,821,872,120,406,352,23,876,872,617,72,732,724,146,121,542,674,216,146,949,76,432,850,899,694,743,720,240,74,795,997,588,227,560,70,461,584,176,868,516,135,625,132,127,703,809,349,503,151,58,843,280,767,732,647,869,170,922,295,302,139,619,940,699,751,719,408,650,477,763,874,693,700,835,802,928,929,650,207,793,920,292,494,876,560,161,596,252,34,163,188,447,318,191,632,413,708,700,648,325,992,8,792,421,331,797,144,988,223,845,910,386,310,59,257,15,201,644,808,136,176,148,890,252,312,285,963,416,603,729,520,316,963,866,397,422,330,945,745,552,546,728,935,977,70,888,343,632,789,206,929,304,383,286,796,375,0,558,137,689,748,837,530,13,273,810,660,773,242,72,2,438,264,31,333,960,223,985,721,458,59,376,730,198,968,861,320,281,38,448,825,872,63,317,320,598,625,218,326,491,176,356,788,17,76,999,129,895,162,149,986,337,379,569,866,64,670,75,296,235,338,400,206,177,737,832,845,453,661,578,897,47,437,299,160,932,290,197,842,243,201,121,249,483,480,383,845,361,877,835,865,561,239,309,832,581,668,326,654,180,46,440,367,92,107,492,584,399,57,683,358,623,949,805,195,902,845,106,965,567,908,869,923,655,900,82,944,292,711,429,913,887,55,485,962,976,514,541,721,748,951,568,390,872,721,230,857,57,20,149,894,806,143,867,787,771,523,625,672,235,892,210,701,570,594,656,747,151,665,697,909,479,251,848,942,151,400,914,20,871,328,426,761,717,383,177,752,325,946,135,724,678,859,579,813,458,778,782,306,488,594,19,27,699,386,544,986,745,571,83,455,774,693,791,150,778,645,768,548,528,666,810,708,907,56,441,447,58,811,479,324,635,226,828,401,691,81,976,922,679,260,135,496,746,361,274,303,322,869,579,877,446,282,862,184,962,154,997,949,57,530,328,802,131,307,760,933,689,497,301,603,897,414,171,909,601,247,237,711,269,699,239,287,838,579,896,662,221,211,718,730,2,20,366,570,1,688,290,87,669,474,811,685,850,720,968,221,887,373,688,101,743,933,944,381,310,966,129,232,138,749,209,290,385,196,439,306,138,840,745,408,375,812,131,347,501,213,655,967,99,455,31,384,149,572,4,853,697,199,296,31,155,547,447,134,995,989,674,390,134,156,49,401,88,231,864,340,519,642,651,229,48,636,199,9,663,836,490,931,80,902,290,550,82,613,527,159,226,68,643,905,171,342,264,220,660,802,110,379,99,439,788,749,998,509,376,560,652,103,61,828,955,987,443,268,916,228,991,198,787,607,949,759,406,662,195,210,458,480,188,393,888,562,863,517,210,306,811,286,188,701,371,29,867,809,509,438,144,23,911,66,124,118,436,949,923,520,845,331,177,741,882,613,609,17,602,205,779,678,655,331,620,614,112,975,900,465,775,387,484,692,373,842,811,738,418,791,819,850,512,497,240,616,673,12,8,280,608,381,595,556,666,539,502,786,857,197,372,560,777,757,471,537,637,856,150,926,512,564,313,313,318,421,342,1,224,352,966,797,109,172,955,463,281,609,561,522,627,38,930,388,489,864,905,297,952,847,739,562,344,660,120,358,567,304,875,627,825,291,500,26,366,768,522,113,24,708,871,850,209,807,555,338,55,942,873,731,48,490,165,819,288,855,355,566,785,766,127,714,734,468,527,590,523,164,916,298,71,955,57,626,895,600,738,458,97,47,204,886,470,521,720,793,6,255,583,300,558,96,43,178,645,560,545,825,468,301,400,887,18,880,611,297,240,291,759,121,861,196,673,307,779,160,466,88,964,550,341,606,700,931,925,892,637,304,478,990,491,106,319,977,469,888,205,917,303,920,888,614,161,833,854,68,608,785,888,71,550,390,988,199,867,752,184,617,140,732,841,781,383,755,852,180,93,249,340,386,63,338,463,377,389,247,335,187,637,664,758,217,460,212,575,148,487,522,663,650,279,305,334,776,464,145,767,906,714,981,581,340,475,946,339,833,625,123,951,728,245,296,766,407,251,224,576",
    "771,538,660,585,55,791,557,68,525,740,956,770,582,616,418,225,626,705,45,37,219,554,45,679,471,461,408,627,970,128,769,599,629,634,798,125,427,107,342,322,202,191,349,810,439,866,888,116,412,376,212,113,606,421,188,423,208,5,113,898,240,971,91,553,706,482,726,500,309,678,657,4,149,86,60,750,8,503,576,479,865,907,98,152,876,232,429,821,198,50,611,627,597,539,253,196,656,713,428,230,552,949,872,129,843,488,746,781,281,359,129,747,930,303,435,100,680,19,231,545,110,535,340,765,815,945,601,724,379,942,282,351,173,474,152,765,790,44,934,7,278,482,825,559,876,63,209,188,789,495,106,109,740,293,592,168,343,942,553,996,613,480,724,726,179,936,520,854,34,635,468,455,44,277,744,356,742,612,772,949,595,232,438,541,891,123,164,55,576,558,369,244,67,395,32,5,762,717,681,80,589,832,570,289,675,765,150,838,777,371,4,601,105,660,292,216,916,84,35,583,42,900,858,357,917,705,794,53,336,305,204,42,461,837,320,18,645,244,531,168,175,286,301,740,503,680,878,805,567,849,155,145,558,0,262,561,162,196,123,45,205,876,263,162,342,258,47,748,279,620,875,132,133,443,21,155,975,649,988,72,369,382,567,429,718,418,161,191,484,237,754,190,322,43,993,783,581,224,566,296,496,574,231,440,79,381,125,892,531,17,338,983,423,382,102,670,746,14,322,541,386,239,752,545,797,824,332,960,229,693,340,355,939,278,809,816,152,512,604,190,670,742,943,427,527,234,786,194,816,103,225,985,863,207,860,308,229,145,80,698,735,685,263,425,956,213,16,912,653,172,783,72,714,276,355,511,642,570,331,382,982,887,846,431,472,248,43,142,921,635,936,327,94,87,585,296,249,866,984,604,687,732,786,705,30,456,596,704,371,963,648,986,150,762,102,971,594,204,767,277,924,932,836,722,986,22,963,541,696,681,273,671,628,442,420,964,559,848,494,169,244,49,490,166,725,535,806,421,651,104,193,208,88,629,937,894,642,682,371,256,233,563,562,373,176,473,759,204,302,377,332,370,44,18,233,686,235,421,115,678,753,213,520,949,795,783,128,414,404,986,865,161,209,310,532,126,429,171,781,612,672,205,868,861,364,777,891,489,733,532,239,359,241,47,75,236,272,300,947,835,820,523,649,766,644,719,790,794,791,462,337,781,415,69,933,314,393,823,315,712,319,950,40,312,486,740,18,111,902,60,508,50,896,434,884,344,564,760,680,321,657,572,876,423,882,860,398,416,228,861,596,409,604,215,242,44,746,418,853,670,999,426,93,221,613,656,272,902,996,617,956,324,980,943,766,292,461,699,284,216,981,944,944,20,820,421,861,23,484,773,969,221,939,181,279,425,75,902,258,231,642,658,356,57,444,909,365,629,390,563,165,291,100,282,267,911,6,382,435,520,546,136,675,530,378,535,679,591,987,918,75,454,587,609,304,876,358,150,174,234,686,388,914,489,653,503,657,122,986,780,448,10,520,722,927,497,869,696,337,957,528,646,375,568,369,408,47,887,16,371,114,404,253,146,483,737,955,627,372,807,903,565,704,105,970,379,828,897,538,595,35,320,791,162,913,11,778,934,431,672,124,363,537,972,775,893,42,229,318,220,809,766,852,408,325,337,167,595,335,502,33,566,423,426,428,305,48,263,707,897,62,400,720,591,489,892,187,50,105,487,860,768,160,911,430,472,693,812,735,59,932,363,6,150,762,750,384,464,28,110,200,940,987,134,928,973,835,193,11,453,386,891,190,413,708,937,192,260,554,418,32,585,751,747,83,3,187,371,446,544,255,252,226,230,932,110,996,737,770,231,660,104,585,370,361,472,483,446,476,793,827,264,235,217,646,609,486,33,357,56,75,840,569,162,612,993,291,189,899,892,161,884,529,161,239,678,138,241,618,583,139,590,945,194,693,634,810,548,153,223,16,94,413,799,196,40,94,367,647,996,416,77,177,47,241,53,875,952,542,758,303,331,719,333,818,71,968,680,791,380,300,609,389,751,147,173,538,61,723,599,950,190,764,971,978,314,915,768,108,735,36,892,43,433,442,796,787,544,906,145,559,943,974,416,90,636,717,740,710,121,588,895,247,785,643,499,888,649,57,690,825,410,8,377,110,297,212,130,127,945,392,247,972,533,167,330,95,102,604,594,947,297,476,791,167,435,572,674,339,627,155,238,528,414,102,293,83,629,887,910,338,80,922,740,691,507,568,509,707,432",
    "683,808,481,401,767,594,541,607,533,853,955,101,701,561,360,998,433,388,108,444,103,881,412,563,868,265,862,265,655,813,488,230,994,215,263,561,905,397,902,503,478,728,767,538,673,944,883,149,988,917,826,496,733,472,619,463,638,232,349,690,984,153,235,808,373,303,805,916,647,640,927,353,231,925,19,397,630,140,292,212,926,797,522,812,692,881,374,723,252,186,152,249,849,177,219,201,171,946,72,288,160,886,887,757,73,200,340,619,301,846,698,255,191,722,880,997,350,33,425,660,439,611,514,830,536,919,63,19,18,502,609,522,548,637,276,330,758,392,332,898,114,495,121,75,443,575,133,212,969,891,810,402,836,820,751,235,994,424,12,723,231,756,609,737,391,892,280,933,561,654,793,390,76,824,88,726,29,452,204,311,664,864,491,395,887,214,816,188,591,607,550,522,930,695,598,763,334,110,965,255,850,352,147,321,855,578,237,529,337,217,175,105,58,14,413,941,24,642,968,541,658,371,734,244,223,994,724,116,195,376,711,887,579,6,643,685,430,851,259,283,524,972,498,989,148,738,157,583,38,186,588,248,137,262,0,733,827,26,758,389,468,648,45,181,702,24,274,539,578,589,792,642,883,841,920,875,73,976,973,323,751,735,32,393,512,300,736,770,788,621,363,225,794,194,431,55,989,212,880,905,729,849,273,344,643,465,394,125,951,343,892,872,849,653,447,711,555,932,646,203,528,801,941,278,401,180,866,610,301,217,903,585,140,156,569,557,271,718,103,857,436,577,5,641,886,128,615,993,830,491,759,82,439,175,549,650,568,963,132,152,756,67,778,104,939,160,964,629,306,688,118,272,920,461,573,172,395,355,147,529,944,33,402,99,466,76,470,981,798,556,948,745,79,146,88,319,846,715,264,994,685,37,594,144,852,326,656,704,356,544,390,699,325,73,999,468,412,732,487,162,864,90,38,811,87,974,273,564,404,444,245,432,974,270,362,13,599,929,691,312,892,678,237,16,340,652,347,823,659,997,684,163,803,216,848,518,309,675,169,128,634,842,954,994,820,127,131,802,535,768,990,887,430,523,623,646,106,384,633,386,16,423,428,33,943,727,433,846,45,672,240,101,171,721,60,524,582,520,159,288,592,323,670,465,405,306,183,58,368,120,472,565,475,903,126,960,761,335,219,587,708,639,459,921,776,507,569,96,470,54,837,874,824,187,601,74,915,287,999,520,285,385,420,355,111,12,770,47,617,39,259,313,682,302,820,121,502,356,560,450,445,97,626,832,89,25,286,547,286,910,418,346,506,69,184,447,32,295,8,824,703,774,990,6,211,828,242,383,569,619,418,509,288,868,997,538,449,564,269,438,55,653,702,8,334,602,108,391,835,71,301,553,751,265,331,684,295,675,957,216,12,189,324,588,560,159,484,594,801,990,828,179,641,979,97,44,40,374,998,794,274,811,442,339,77,171,782,716,157,77,61,743,398,330,178,359,133,441,463,971,89,860,124,312,365,190,255,670,431,223,481,147,836,897,272,911,856,214,512,294,264,885,594,334,726,226,933,46,417,380,412,899,471,208,25,369,345,860,157,112,617,905,525,757,217,554,41,966,465,598,286,841,545,110,228,218,222,45,906,155,211,99,168,668,638,780,938,742,939,920,289,293,118,843,900,992,856,546,962,475,65,826,953,834,492,518,802,824,632,117,107,820,368,864,730,412,483,840,720,866,728,440,39,548,829,424,111,739,21,693,13,182,72,501,304,547,129,465,6,534,221,235,24,616,624,525,828,125,85,36,699,789,18,672,263,695,435,51,866,110,246,949,443,906,143,602,987,153,701,612,813,658,754,217,752,970,874,409,570,890,644,697,841,54,904,228,440,57,312,494,671,488,839,970,7,693,289,782,964,166,359,822,159,932,474,124,387,560,213,12,778,140,996,330,921,882,352,809,659,211,888,271,431,129,918,755,980,141,446,667,211,898,925,192,388,24,464,889,406,312,700,381,251,50,689,79,446,617,81,578,460,923,994,500,645,869,243,550,559,915,541,956,341,178,321,466,267,663,126,271,804,913,719,976,514,85,931,48,168,309,385,733,280,287,517,455,388,50,617,526,879,626,463,833,605,554,852,847,651,66,116,318,363,85,490,279,946,862,399,645,615,735,284,404,800,352,826,163,752,99,678,155,653,176,766,534,322,593,301,329,262,237,828,765,66,795,531,593,717,793,361,365,365,164,246,436,101,991,266,349,492,712,907,79,301,841,151,923,913,169",
    "730,114,221,725,754,427,706,755,416,596,732,496,357,860,293,418,382,729,421,108,57,75,406,544,790,655,975,909,724,688,440,285,328,675,843,408,986,120,443,444,521,570,589,273,59,507,977,421,699,643,229,252,278,538,632,167,600,4,41,241,800,9,879,530,937,540,186,379,855,240,677,92,977,511,957,565,521,346,2,615,616,674,603,135,509,75,502,890,555,627,593,311,842,607,753,12,977,550,337,101,174,49,906,270,124,500,782,684,570,274,273,257,9,180,372,977,69,947,424,91,565,958,214,865,562,789,523,798,232,181,988,905,819,738,849,365,240,682,753,624,985,129,926,930,94,849,538,729,794,129,257,846,196,494,51,1000,642,874,899,233,136,611,353,321,668,407,562,3,764,146,214,212,146,609,347,890,981,292,567,719,872,155,432,758,213,732,788,112,132,110,916,700,149,517,911,366,598,488,502,743,9,393,847,527,295,414,188,6,889,565,697,515,733,295,70,149,511,460,806,432,586,605,85,800,749,157,209,898,639,866,576,885,999,307,296,840,549,62,222,81,650,916,706,569,245,418,401,59,851,310,131,372,689,561,733,0,821,14,913,311,651,687,503,737,156,442,799,55,513,401,204,313,395,393,332,721,117,200,578,261,259,761,418,396,691,287,240,270,818,752,443,710,90,867,112,803,630,257,918,749,33,284,416,164,443,862,994,734,139,150,971,591,457,43,560,939,880,745,191,518,576,893,143,367,298,313,484,74,844,679,42,317,494,764,967,626,916,217,109,422,816,186,541,153,579,837,282,573,132,991,204,139,996,406,231,154,125,565,674,478,552,884,737,161,376,930,581,346,193,506,142,541,726,249,483,581,28,936,48,865,552,145,331,324,102,189,218,679,239,638,942,510,635,956,529,903,278,824,632,535,502,72,157,792,560,777,675,558,185,876,475,880,676,81,920,152,943,927,264,401,508,689,769,573,749,148,781,401,859,411,53,428,667,119,753,440,534,965,653,74,620,992,796,638,51,919,614,55,819,510,900,190,696,70,603,646,961,396,763,528,980,859,937,651,474,633,595,837,446,657,887,266,172,952,320,119,276,323,408,136,31,480,621,927,901,384,6,170,293,217,979,592,161,381,542,789,610,957,144,619,260,11,276,713,390,607,995,802,490,702,371,368,87,825,90,455,523,915,414,894,939,672,49,693,223,852,735,501,85,341,554,502,872,788,640,394,18,958,190,920,964,972,313,313,474,568,630,341,581,919,243,315,914,699,298,848,355,188,986,346,409,419,358,705,97,596,440,784,833,406,623,498,432,630,296,725,570,529,645,473,805,191,132,94,818,33,673,184,218,637,751,734,813,506,361,398,314,689,160,693,196,491,751,53,969,330,497,713,542,810,136,351,185,960,940,670,956,694,204,368,261,121,9,472,450,914,151,717,338,237,837,27,269,587,443,279,521,708,770,796,328,640,361,15,625,186,180,206,480,925,190,440,871,600,268,818,44,981,134,440,449,644,329,639,919,275,790,780,146,482,155,680,699,330,153,971,895,201,389,93,838,883,277,589,39,937,161,492,669,978,261,349,387,640,988,300,596,84,649,571,452,122,904,743,218,661,903,770,431,424,247,937,123,630,944,614,553,414,425,197,57,2,252,710,457,924,321,425,621,890,402,459,142,818,350,425,212,101,497,513,653,744,211,895,401,241,658,896,715,587,34,705,8,966,413,137,293,998,61,54,756,274,759,722,178,155,268,978,340,33,828,996,830,162,504,576,386,304,693,362,650,584,821,555,519,725,279,204,938,842,806,963,645,624,275,431,711,523,217,332,913,968,992,15,328,875,377,893,322,347,122,75,337,892,28,677,35,70,308,121,925,974,39,903,321,368,495,450,294,19,842,270,264,636,79,977,125,853,823,311,786,224,932,147,409,977,600,296,216,84,458,602,408,61,582,641,512,880,993,510,424,370,486,888,163,701,605,789,89,157,971,68,60,347,447,609,713,542,240,342,642,874,676,34,535,478,807,35,108,541,611,670,988,323,53,470,997,827,70,365,82,941,736,916,464,842,443,699,955,441,782,234,404,954,890,891,308,803,89,290,205,180,665,144,816,235,631,479,708,743,201,533,649,964,721,247,419,529,231,337,374,560,389,424,802,463,643,898,375,3,462,648,424,84,589,609,188,656,754,77,625,268,779,942,19,715,381,311,80,550,407,874,980,930,795,846,821,620,18,577,29,435,912,193,780,281,257,511,256,144,903,187,909,995,217,937,577,923,407,776",
    "478,389,771,478,289,825,587,853,212,377,322,492,398,764,966,113,179,701,254,696,882,432,156,981,220,752,211,552,769,697,440,992,708,934,124,916,865,151,541,165,174,883,829,442,776,960,64,917,378,524,566,496,395,861,51,632,383,731,380,409,192,534,557,171,1000,84,589,49,973,428,207,766,151,514,308,626,66,434,842,217,807,507,64