import { DEV } from 'esm-env';
import { onMount, tick } from 'svelte';
import {
	make_trackable,
	decode_pathname,
	decode_params,
	normalize_path,
	add_data_suffix
} from '../../utils/url.js';
import {
	find_anchor,
	get_base_uri,
	get_link_info,
	get_router_options,
	is_external_url,
	scroll_state
} from './utils.js';
import {
	lock_fetch,
	unlock_fetch,
	initial_fetch,
	subsequent_fetch,
	native_fetch
} from './fetcher.js';
import { parse } from './parse.js';

import Root from '__GENERATED__/root.svelte';
import { nodes, server_loads, dictionary, matchers, hooks } from '__GENERATED__/client-manifest.js';
import { HttpError, Redirect } from '../control.js';
import { stores } from './singletons.js';
import { unwrap_promises } from '../../utils/promises.js';
import * as devalue from 'devalue';
import { INDEX_KEY, PRELOAD_PRIORITIES, SCROLL_KEY } from './constants.js';
import { validate_common_exports } from '../../utils/exports.js';

const routes = parse(nodes, server_loads, dictionary, matchers);

const default_layout_loader = nodes[0];
const default_error_loader = nodes[1];

// we import the root layout/error nodes eagerly, so that
// connectivity errors after initialisation don't nuke the app
default_layout_loader();
default_error_loader();

// We track the scroll position associated with each history entry in sessionStorage,
// rather than on history.state itself, because when navigation is driven by
// popstate it's too late to update the scroll position associated with the
// state we're navigating from

/** @typedef {{ x: number, y: number }} ScrollPosition */
/** @type {Record<number, ScrollPosition>} */
let scroll_positions = {};
try {
	scroll_positions = JSON.parse(sessionStorage[SCROLL_KEY]);
} catch {
	// do nothing
}

/** @param {number} index */
function update_scroll_positions(index) {
	scroll_positions[index] = scroll_state();
}

// TODO remove for 1.0
/** @type {Record<string, true>} */
let warned_about_attributes = {};

function check_for_removed_attributes() {
	const attrs = ['prefetch', 'noscroll', 'reload'];
	for (const attr of attrs) {
		if (document.querySelector(`[sveltekit\\:${attr}]`)) {
			if (!warned_about_attributes[attr]) {
				warned_about_attributes[attr] = true;
				console.error(
					`The sveltekit:${attr} attribute has been replaced with data-sveltekit-${
						attr === 'prefetch' ? 'preload-data' : attr
					}`
				);
			}
		}
	}
}

/**
 * @param {{
 *   target: HTMLElement;
 *   base: string;
 * }} opts
 * @returns {import('./types').Client}
 */
export function create_client({ target, base }) {
	const container = __SVELTEKIT_EMBEDDED__ ? target : document.documentElement;
	/** @type {Array<((url: URL) => boolean)>} */
	const invalidated = [];

	/** @type {{id: string, promise: Promise<import('./types').NavigationResult>} | null} */
	let load_cache = null;

	const callbacks = {
		/** @type {Array<(navigation: import('types').BeforeNavigate) => void>} */
		before_navigate: [],

		/** @type {Array<(navigation: import('types').AfterNavigate) => void>} */
		after_navigate: []
	};

	/** @type {import('./types').NavigationState} */
	let current = {
		branch: [],
		error: null,
		// @ts-ignore - we need the initial value to be null
		url: null
	};

	/** this being true means we SSR'd */
	let hydrated = false;
	let started = false;
	let autoscroll = true;
	let updating = false;
	let navigating = false;
	let hash_navigating = false;

	let force_invalidation = false;

	/** @type {import('svelte').SvelteComponent} */
	let root;

	// keeping track of the history index in order to prevent popstate navigation events if needed
	let current_history_index = history.state?.[INDEX_KEY];

	if (!current_history_index) {
		// we use Date.now() as an offset so that cross-document navigations
		// within the app don't result in data loss
		current_history_index = Date.now();

		// create initial history entry, so we can return here
		history.replaceState(
			{ ...history.state, [INDEX_KEY]: current_history_index },
			'',
			location.href
		);
	}

	// if we reload the page, or Cmd-Shift-T back to it,
	// recover scroll position
	const scroll = scroll_positions[current_history_index];
	if (scroll) {
		history.scrollRestoration = 'manual';
		scrollTo(scroll.x, scroll.y);
	}

	/** @type {import('types').Page} */
	let page;

	/** @type {{}} */
	let token;

	/** @type {Promise<void> | null} */
	let pending_invalidate;

	async function invalidate() {
		// Accept all invalidations as they come, don't swallow any while another invalidation
		// is running because subsequent invalidations may make earlier ones outdated,
		// but batch multiple synchronous invalidations.
		pending_invalidate = pending_invalidate || Promise.resolve();
		await pending_invalidate;
		pending_invalidate = null;

		const url = new URL(location.href);
		const intent = get_navigation_intent(url, true);
		// Clear preload, it might be affected by the invalidation.
		// Also solves an edge case where a preload is triggered, the navigation for it
		// was then triggered and is still running while the invalidation kicks in,
		// at which point the invalidation should take over and "win".
		load_cache = null;
		await update(intent, url, []);
	}

	/**
	 * @param {string | URL} url
	 * @param {{ noScroll?: boolean; replaceState?: boolean; keepFocus?: boolean; state?: any; invalidateAll?: boolean }} opts
	 * @param {string[]} redirect_chain
	 * @param {{}} [nav_token]
	 */
	async function goto(
		url,
		{
			noScroll = false,
			replaceState = false,
			keepFocus = false,
			state = {},
			invalidateAll = false
		},
		redirect_chain,
		nav_token
	) {
		if (typeof url === 'string') {
			url = new URL(url, get_base_uri(document));
		}

		return navigate({
			url,
			scroll: noScroll ? scroll_state() : null,
			keepfocus: keepFocus,
			redirect_chain,
			details: {
				state,
				replaceState
			},
			nav_token,
			accepted: () => {
				if (invalidateAll) {
					force_invalidation = true;
				}
			},
			blocked: () => {},
			type: 'goto'
		});
	}

	/** @param {URL} url */
	async function preload_data(url) {
		const intent = get_navigation_intent(url, false);

		if (!intent) {
			throw new Error(`Attempted to preload a URL that does not belong to this app: ${url}`);
		}

		load_cache = {
			id: intent.id,
			promise: load_route(intent).then((result) => {
				if (result.type === 'loaded' && result.state.error) {
					// Don't cache errors, because they might be transient
					load_cache = null;
				}
				return result;
			})
		};

		return load_cache.promise;
	}

	/** @param {...string} pathnames */
	async function preload_code(...pathnames) {
		const matching = routes.filter((route) => pathnames.some((pathname) => route.exec(pathname)));

		const promises = matching.map((r) => {
			return Promise.all([...r.layouts, r.leaf].map((load) => load?.[1]()));
		});

		await Promise.all(promises);
	}

	/**
	 * Returns `true` if update completes, `false` if it is aborted
	 * @param {import('./types').NavigationIntent | undefined} intent
	 * @param {URL} url
	 * @param {string[]} redirect_chain
	 * @param {{hash?: string, scroll: { x: number, y: number } | null, keepfocus: boolean, details: { replaceState: boolean, state: any } | null}} [opts]
	 * @param {{}} [nav_token] To distinguish between different navigation events and determine the latest. Needed for example for redirects to keep the original token
	 * @param {() => void} [callback]
	 */
	async function update(intent, url, redirect_chain, opts, nav_token = {}, callback) {
		token = nav_token;
		let navigation_result = intent && (await load_route(intent));

		if (!navigation_result) {
			navigation_result = await server_fallback(
				url,
				{ id: null },
				await handle_error(new Error(`Not found: ${url.pathname}`), {
					url,
					params: {},
					route: { id: null }
				}),
				404
			);
		}

		// if this is an internal navigation intent, use the normalized
		// URL for the rest of the function
		url = intent?.url || url;

		// abort if user navigated during update
		if (token !== nav_token) return false;

		if (navigation_result.type === 'redirect') {
			if (redirect_chain.length > 10 || redirect_chain.includes(url.pathname)) {
				navigation_result = await load_root_error_page({
					status: 500,
					error: await handle_error(new Error('Redirect loop'), {
						url,
						params: {},
						route: { id: null }
					}),
					url,
					route: { id: null }
				});
			} else {
				goto(
					new URL(navigation_result.location, url).href,
					{},
					[...redirect_chain, url.pathname],
					nav_token
				);
				return false;
			}
		} else if (navigation_result.props?.page?.status >= 400) {
			const updated = await stores.updated.check();
			if (updated) {
				await native_navigation(url);
			}
		}

		// reset invalidation only after a finished navigation. If there are redirects or
		// additional invalidations, they should get the same invalidation treatment
		invalidated.length = 0;
		force_invalidation = false;

		updating = true;

		if (opts && opts.details) {
			const { details } = opts;
			const change = details.replaceState ? 0 : 1;
			details.state[INDEX_KEY] = current_history_index += change;
			history[details.replaceState ? 'replaceState' : 'pushState'](details.state, '', url);
		}

		// reset preload synchronously after the history state has been set to avoid race conditions
		load_cache = null;

		if (started) {
			current = navigation_result.state;

			if (navigation_result.props.page) {
				navigation_result.props.page.url = url;
			}

			const post_update = pre_update();
			root.$set(navigation_result.props);
			post_update();
		} else {
			initialize(navigation_result);
		}

		// opts must be passed if we're navigating
		if (opts) {
			const { scroll, keepfocus } = opts;

			// reset focus first, so that manual focus management can override it
			if (!keepfocus) reset_focus();

			// need to render the DOM before we can scroll to the rendered elements
			await tick();

			if (autoscroll) {
				const deep_linked = url.hash && document.getElementById(url.hash.slice(1));
				if (scroll) {
					scrollTo(scroll.x, scroll.y);
				} else if (deep_linked) {
					// Here we use `scrollIntoView` on the element instead of `scrollTo`
					// because it natively supports the `scroll-margin` and `scroll-behavior`
					// CSS properties.
					deep_linked.scrollIntoView();
				} else {
					scrollTo(0, 0);
				}
			}
		} else {
			await tick();
		}

		autoscroll = true;

		if (navigation_result.props.page) {
			page = navigation_result.props.page;
		}

		if (callback) callback();

		updating = false;
	}

	/** @param {import('./types').NavigationFinished} result */
	function initialize(result) {
		if (DEV && document.querySelector('vite-error-overlay')) return;

		current = result.state;

		const style = document.querySelector('style[data-sveltekit]');
		if (style) style.remove();

		page = result.props.page;

		const post_update = pre_update();
		root = new Root({
			target,
			props: { ...result.props, stores },
			hydrate: true
		});
		post_update();

		/** @type {import('types').AfterNavigate} */
		const navigation = {
			from: null,
			to: add_url_properties('to', {
				params: current.params,
				route: { id: current.route?.id ?? null },
				url: new URL(location.href)
			}),
			willUnload: false,
			type: 'enter'
		};
		callbacks.after_navigate.forEach((fn) => fn(navigation));

		started = true;
	}

	/**
	 *
	 * @param {{
	 *   url: URL;
	 *   params: Record<string, string>;
	 *   branch: Array<import('./types').BranchNode | undefined>;
	 *   status: number;
	 *   error: App.Error | null;
	 *   route: import('types').CSRRoute | null;
	 *   form?: Record<string, any> | null;
	 * }} opts
	 */
	async function get_navigation_result_from_branch({
		url,
		params,
		branch,
		status,
		error,
		route,
		form
	}) {
		const filtered = /** @type {import('./types').BranchNode[] } */ (branch.filter(Boolean));

		/** @type {import('types').TrailingSlash} */
		let slash = 'never';
		for (const node of branch) {
			if (node?.slash !== undefined) slash = node.slash;
		}
		url.pathname = normalize_path(url.pathname, slash);
		url.search = url.search; // turn `/?` into `/`

		/** @type {import('./types').NavigationFinished} */
		const result = {
			type: 'loaded',
			state: {
				url,
				params,
				branch,
				error,
				route
			},
			props: {
				components: filtered.map((branch_node) => branch_node.node.component)
			}
		};

		if (form !== undefined) {
			result.props.form = form;
		}

		let data = {};
		let data_changed = !page;
		for (let i = 0; i < filtered.length; i += 1) {
			const node = filtered[i];
			data = { ...data, ...node.data };

			// Only set props if the node actually updated. This prevents needless rerenders.
			if (data_changed || !current.branch.some((previous) => previous === node)) {
				result.props[`data_${i}`] = data;
				data_changed = data_changed || Object.keys(node.data ?? {}).length > 0;
			}
		}
		if (!data_changed) {
			// If nothing was added, and the object entries are the same length, this means
			// that nothing was removed either and therefore the data is the same as the previous one.
			// This would be more readable with a separate boolean but that would cost us some bytes.
			data_changed = Object.keys(page.data).length !== Object.keys(data).length;
		}

		const page_changed =
			!current.url ||
			url.href !== current.url.href ||
			current.error !== error ||
			form !== undefined ||
			data_changed;

		if (page_changed) {
			result.props.page = {
				error,
				params,
				route,
				status,
				url: new URL(url),
				form: form ?? null,
				// The whole page store is updated, but this way the object reference stays the same
				data: data_changed ? data : page.data
			};

			// TODO remove this for 1.0
			Object.defineProperty(result.props.page, 'routeId', {
				get() {
					throw new Error('$page.routeId has been replaced by $page.route.id');
				},
				enumerable: false
			});
			/**
			 * @param {string} property
			 * @param {string} replacement
			 */
			const print_error = (property, replacement) => {
				Object.defineProperty(result.props.page, property, {
					get: () => {
						throw new Error(`$page.${property} has been replaced by $page.url.${replacement}`);
					}
				});
			};

			print_error('origin', 'origin');
			print_error('path', 'pathname');
			print_error('query', 'searchParams');
		}

		return result;
	}

	/**
	 * Call the load function of the given node, if it exists.
	 * If `server_data` is passed, this is treated as the initial run and the page endpoint is not requested.
	 *
	 * @param {{
	 *   loader: import('types').CSRPageNodeLoader;
	 * 	 parent: () => Promise<Record<string, any>>;
	 *   url: URL;
	 *   params: Record<string, string>;
	 *   route: { id: string | null };
	 * 	 server_data_node: import('./types').DataNode | null;
	 * }} options
	 * @returns {Promise<import('./types').BranchNode>}
	 */
	async function load_node({ loader, parent, url, params, route, server_data_node }) {
		/** @type {Record<string, any> | null} */
		let data = null;

		/** @type {import('types').Uses} */
		const uses = {
			dependencies: new Set(),
			params: new Set(),
			parent: false,
			route: false,
			url: false
		};

		const node = await loader();

		if (DEV) {
			validate_common_exports(node.shared);
		}

		if (node.shared?.load) {
			/** @param {string[]} deps */
			function depends(...deps) {
				for (const dep of deps) {
					const { href } = new URL(dep, url);
					uses.dependencies.add(href);
				}
			}

			/** @type {import('types').LoadEvent} */
			const load_input = {
				route: {
					get id() {
						uses.route = true;
						return route.id;
					}
				},
				params: new Proxy(params, {
					get: (target, key) => {
						uses.params.add(/** @type {string} */ (key));
						return target[/** @type {string} */ (key)];
					}
				}),
				data: server_data_node?.data ?? null,
				url: make_trackable(url, () => {
					uses.url = true;
				}),
				async fetch(resource, init) {
					/** @type {URL | string} */
					let requested;

					if (resource instanceof Request) {
						requested = resource.url;

						// we're not allowed to modify the received `Request` object, so in order
						// to fixup relative urls we create a new equivalent `init` object instead
						init = {
							// the request body must be consumed in memory until browsers
							// implement streaming request bodies and/or the body getter
							body:
								resource.method === 'GET' || resource.method === 'HEAD'
									? undefined
									: await resource.blob(),
							cache: resource.cache,
							credentials: resource.credentials,
							headers: resource.headers,
							integrity: resource.integrity,
							keepalive: resource.keepalive,
							method: resource.method,
							mode: resource.mode,
							redirect: resource.redirect,
							referrer: resource.referrer,
							referrerPolicy: resource.referrerPolicy,
							signal: resource.signal,
							...init
						};
					} else {
						requested = resource;
					}

					// we must fixup relative urls so they are resolved from the target page
					const resolved = new URL(requested, url).href;
					depends(resolved);

					// prerendered pages may be served from any origin, so `initial_fetch` urls shouldn't be resolved
					return started
						? subsequent_fetch(requested, resolved, init)
						: initial_fetch(requested, init);
				},
				setHeaders: () => {}, // noop
				depends,
				parent() {
					uses.parent = true;
					return parent();
				}
			};

			// TODO remove this for 1.0
			Object.defineProperties(load_input, {
				props: {
					get() {
						throw new Error(
							'@migration task: Replace `props` with `data` stuff https://github.com/sveltejs/kit/discussions/5774#discussioncomment-3292693'
						);
					},
					enumerable: false
				},
				session: {
					get() {
						throw new Error(
							'session is no longer available. See https://github.com/sveltejs/kit/discussions/5883'
						);
					},
					enumerable: false
				},
				stuff: {
					get() {
						throw new Error(
							'@migration task: Remove stuff https://github.com/sveltejs/kit/discussions/5774#discussioncomment-3292693'
						);
					},
					enumerable: false
				},
				routeId: {
					get() {
						throw new Error('routeId has been replaced by route.id');
					},
					enumerable: false
				}
			});

			if (DEV) {
				try {
					lock_fetch();
					data = (await node.shared.load.call(null, load_input)) ?? null;
					if (data != null && Object.getPrototypeOf(data) !== Object.prototype) {
						throw new Error(
							`a load function related to route '${route.id}' returned ${
								typeof data !== 'object'
									? `a ${typeof data}`
									: data instanceof Response
									? 'a Response object'
									: Array.isArray(data)
									? 'an array'
									: 'a non-plain object'
							}, but must return a plain object at the top level (i.e. \`return {...}\`)`
						);
					}
				} finally {
					unlock_fetch();
				}
			} else {
				data = (await node.shared.load.call(null, load_input)) ?? null;
			}
			data = data ? await unwrap_promises(data) : null;
		}

		return {
			node,
			loader,
			server: server_data_node,
			shared: node.shared?.load ? { type: 'data', data, uses } : null,
			data: data ?? server_data_node?.data ?? null,
			slash: node.shared?.trailingSlash ?? server_data_node?.slash
		};
	}

	/**
	 * @param {boolean} parent_changed
	 * @param {boolean} route_changed
	 * @param {boolean} url_changed
	 * @param {import('types').Uses | undefined} uses
	 * @param {Record<string, string>} params
	 */
	function has_changed(parent_changed, route_changed, url_changed, uses, params) {
		if (force_invalidation) return true;

		if (!uses) return false;

		if (uses.parent && parent_changed) return true;
		if (uses.route && route_changed) return true;
		if (uses.url && url_changed) return true;

		for (const param of uses.params) {
			if (params[param] !== current.params[param]) return true;
		}

		for (const href of uses.dependencies) {
			if (invalidated.some((fn) => fn(new URL(href)))) return true;
		}

		return false;
	}

	/**
	 * @param {import('types').ServerDataNode | import('types').ServerDataSkippedNode | null} node
	 * @param {import('./types').DataNode | null} [previous]
	 * @returns {import('./types').DataNode | null}
	 */
	function create_data_node(node, previous) {
		if (node?.type === 'data') {
			return {
				type: 'data',
				data: node.data,
				uses: {
					dependencies: new Set(node.uses.dependencies ?? []),
					params: new Set(node.uses.params ?? []),
					parent: !!node.uses.parent,
					route: !!node.uses.route,
					url: !!node.uses.url
				},
				slash: node.slash
			};
		} else if (node?.type === 'skip') {
			return previous ?? null;
		}
		return null;
	}

	/**
	 * @param {import('./types').NavigationIntent} intent
	 * @returns {Promise<import('./types').NavigationResult>}
	 */
	async function load_route({ id, invalidating, url, params, route }) {
		if (load_cache?.id === id) {
			return load_cache.promise;
		}

		const { errors, layouts, leaf } = route;

		const loaders = [...layouts, leaf];

		// preload modules to avoid waterfall, but handle rejections
		// so they don't get reported to Sentry et al (we don't need
		// to act on the failures at this point)
		errors.forEach((loader) => loader?.().catch(() => {}));
		loaders.forEach((loader) => loader?.[1]().catch(() => {}));

		/** @type {import('types').ServerData | null} */
		let server_data = null;

		const url_changed = current.url ? id !== current.url.pathname + current.url.search : false;
		const route_changed = current.route ? id !== current.route.id : false;

		const invalid_server_nodes = loaders.reduce((acc, loader, i) => {
			const previous = current.branch[i];

			const invalid =
				!!loader?.[0] &&
				(previous?.loader !== loader[1] ||
					has_changed(
						acc.some(Boolean),
						route_changed,
						url_changed,
						previous.server?.uses,
						params
					));

			acc.push(invalid);
			return acc;
		}, /** @type {boolean[]} */ ([]));

		if (invalid_server_nodes.some(Boolean)) {
			try {
				server_data = await load_data(url, invalid_server_nodes);
			} catch (error) {
				return load_root_error_page({
					status: 500,
					error: await handle_error(error, { url, params, route: { id: route.id } }),
					url,
					route
				});
			}

			if (server_data.type === 'redirect') {
				return server_data;
			}
		}

		const server_data_nodes = server_data?.nodes;

		let parent_changed = false;

		const branch_promises = loaders.map(async (loader, i) => {
			if (!loader) return;

			/** @type {import('./types').BranchNode | undefined} */
			const previous = current.branch[i];

			const server_data_node = server_data_nodes?.[i];

			// re-use data from previous load if it's still valid
			const valid =
				(!server_data_node || server_data_node.type === 'skip') &&
				loader[1] === previous?.loader &&
				!has_changed(parent_changed, route_changed, url_changed, previous.shared?.uses, params);
			if (valid) return previous;

			parent_changed = true;

			if (server_data_node?.type === 'error') {
				// rethrow and catch below
				throw server_data_node;
			}

			return load_node({
				loader: loader[1],
				url,
				params,
				route,
				parent: async () => {
					const data = {};
					for (let j = 0; j < i; j += 1) {
						Object.assign(data, (await branch_promises[j])?.data);
					}
					return data;
				},
				server_data_node: create_data_node(
					// server_data_node is undefined if it wasn't reloaded from the server;
					// and if current loader uses server data, we want to reuse previous data.
					server_data_node === undefined && loader[0] ? { type: 'skip' } : server_data_node ?? null,
					previous?.server
				)
			});
		});

		// if we don't do this, rejections will be unhandled
		for (const p of branch_promises) p.catch(() => {});

		/** @type {Array<import('./types').BranchNode | undefined>} */
		const branch = [];

		for (let i = 0; i < loaders.length; i += 1) {
			if (loaders[i]) {
				try {
					branch.push(await branch_promises[i]);
				} catch (err) {
					if (err instanceof Redirect) {
						return {
							type: 'redirect',
							location: err.location
						};
					}

					let status = 500;
					/** @type {App.Error} */
					let error;

					if (server_data_nodes?.includes(/** @type {import('types').ServerErrorNode} */ (err))) {
						// this is the server error rethrown above, reconstruct but don't invoke
						// the client error handler; it should've already been handled on the server
						status = /** @type {import('types').ServerErrorNode} */ (err).status ?? status;
						error = /** @type {import('types').ServerErrorNode} */ (err).error;
					} else if (err instanceof HttpError) {
						status = err.status;
						error = err.body;
					} else {
						error = await handle_error(err, { params, url, route: { id: route.id } });
					}

					const error_load = await load_nearest_error_page(i, branch, errors);
					if (error_load) {
						return await get_navigation_result_from_branch({
							url,
							params,
							branch: branch.slice(0, error_load.idx).concat(error_load.node),
							status,
							error,
							route
						});
					} else {
						// if we get here, it's because the root `load` function failed,
						// and we need to fall back to the server
						return await server_fallback(url, { id: route.id }, error, status);
					}
				}
			} else {
				// push an empty slot so we can rewind past gaps to the
				// layout that corresponds with an +error.svelte page
				branch.push(undefined);
			}
		}

		return await get_navigation_result_from_branch({
			url,
			params,
			branch,
			status: 200,
			error: null,
			route,
			// Reset `form` on navigation, but not invalidation
			form: invalidating ? undefined : null
		});
	}

	/**
	 * @param {number} i Start index to backtrack from
	 * @param {Array<import('./types').BranchNode | undefined>} branch Branch to backtrack
	 * @param {Array<import('types').CSRPageNodeLoader | undefined>} errors All error pages for this branch
	 * @returns {Promise<{idx: number; node: import('./types').BranchNode} | undefined>}
	 */
	async function load_nearest_error_page(i, branch, errors) {
		while (i--) {
			if (errors[i]) {
				let j = i;
				while (!branch[j]) j -= 1;
				try {
					return {
						idx: j + 1,
						node: {
							node: await /** @type {import('types').CSRPageNodeLoader } */ (errors[i])(),
							loader: /** @type {import('types').CSRPageNodeLoader } */ (errors[i]),
							data: {},
							server: null,
							shared: null
						}
					};
				} catch (e) {
					continue;
				}
			}
		}
	}

	/**
	 * @param {{
	 *   status: number;
	 *   error: App.Error;
	 *   url: URL;
	 *   route: { id: string | null }
	 * }} opts
	 * @returns {Promise<import('./types').NavigationFinished>}
	 */
	async function load_root_error_page({ status, error, url, route }) {
		/** @type {Record<string, string>} */
		const params = {}; // error page does not have params

		const node = await default_layout_loader();

		/** @type {import('types').ServerDataNode | null} */
		let server_data_node = null;

		if (node.server) {
			// TODO post-https://github.com/sveltejs/kit/discussions/6124 we can use
			// existing root layout data
			try {
				const server_data = await load_data(url, [true]);

				if (
					server_data.type !== 'data' ||
					(server_data.nodes[0] && server_data.nodes[0].type !== 'data')
				) {
					throw 0;
				}

				server_data_node = server_data.nodes[0] ?? null;
			} catch {
				// at this point we have no choice but to fall back to the server, if it wouldn't
				// bring us right back here, turning this into an endless loop
				if (url.origin !== location.origin || url.pathname !== location.pathname || hydrated) {
					await native_navigation(url);
				}
			}
		}

		const root_layout = await load_node({
			loader: default_layout_loader,
			url,
			params,
			route,
			parent: () => Promise.resolve({}),
			server_data_node: create_data_node(server_data_node)
		});

		/** @type {import('./types').BranchNode} */
		const root_error = {
			node: await default_error_loader(),
			loader: default_error_loader,
			shared: null,
			server: null,
			data: null
		};

		return await get_navigation_result_from_branch({
			url,
			params,
			branch: [root_layout, root_error],
			status,
			error,
			route: null
		});
	}

	/**
	 * @param {URL} url
	 * @param {boolean} invalidating
	 */
	function get_navigation_intent(url, invalidating) {
		if (is_external_url(url, base)) return;

		const path = decode_pathname(url.pathname.slice(base.length) || '/');

		for (const route of routes) {
			const params = route.exec(path);

			if (params) {
				const id = url.pathname + url.search;
				/** @type {import('./types').NavigationIntent} */
				const intent = { id, invalidating, route, params: decode_params(params), url };
				return intent;
			}
		}
	}

	/**
	 * @param {{
	 *   url: URL;
	 *   type: import('types').NavigationType;
	 *   intent?: import('./types').NavigationIntent;
	 *   delta?: number;
	 * }} opts
	 */
	function before_navigate({ url, type, intent, delta }) {
		let should_block = false;

		/** @type {import('types').Navigation} */
		const navigation = {
			from: add_url_properties('from', {
				params: current.params,
				route: { id: current.route?.id ?? null },
				url: current.url
			}),
			to: add_url_properties('to', {
				params: intent?.params ?? null,
				route: { id: intent?.route?.id ?? null },
				url
			}),
			willUnload: !intent,
			type
		};

		if (delta !== undefined) {
			navigation.delta = delta;
		}

		const cancellable = {
			...navigation,
			cancel: () => {
				should_block = true;
			}
		};

		if (!navigating) {
			// Don't run the event during redirects
			callbacks.before_navigate.forEach((fn) => fn(cancellable));
		}

		return should_block ? null : navigation;
	}

	/**
	 * @param {{
	 *   url: URL;
	 *   scroll: { x: number, y: number } | null;
	 *   keepfocus: boolean;
	 *   redirect_chain: string[];
	 *   details: {
	 *     replaceState: boolean;
	 *     state: any;
	 *   } | null;
	 *   type: import('types').NavigationType;
	 *   delta?: number;
	 *   nav_token?: {};
	 *   accepted: () => void;
	 *   blocked: () => void;
	 * }} opts
	 */
	async function navigate({
		url,
		scroll,
		keepfocus,
		redirect_chain,
		details,
		type,
		delta,
		nav_token,
		accepted,
		blocked
	}) {
		const intent = get_navigation_intent(url, false);
		const navigation = before_navigate({ url, type, delta, intent });

		if (!navigation) {
			blocked();
			return;
		}

		update_scroll_positions(current_history_index);

		accepted();

		navigating = true;

		if (started) {
			stores.navigating.set(navigation);
		}

		await update(
			intent,
			url,
			redirect_chain,
			{
				scroll,
				keepfocus,
				details
			},
			nav_token,
			() => {
				navigating = false;
				callbacks.after_navigate.forEach((fn) =>
					fn(/** @type {import('types').AfterNavigate} */ (navigation))
				);
				stores.navigating.set(null);
			}
		);
	}

	/**
	 * Does a full page reload if it wouldn't result in an endless loop in the SPA case
	 * @param {URL} url
	 * @param {{ id: string | null }} route
	 * @param {App.Error} error
	 * @param {number} status
	 * @returns {Promise<import('./types').NavigationFinished>}
	 */
	async function server_fallback(url, route, error, status) {
		if (url.origin === location.origin && url.pathname === location.pathname && !hydrated) {
			// We would reload the same page we're currently on, which isn't hydrated,
			// which means no SSR, which means we would end up in an endless loop
			return await load_root_error_page({
				status,
				error,
				url,
				route
			});
		}
		return await native_navigation(url);
	}

	/**
	 * Loads `href` the old-fashioned way, with a full page reload.
	 * Returns a `Promise` that never resolves (to prevent any
	 * subsequent work, e.g. history manipulation, from happening)
	 * @param {URL} url
	 */
	function native_navigation(url) {
		location.href = url.href;
		return new Promise(() => {});
	}

	if (import.meta.hot) {
		import.meta.hot.on('vite:beforeUpdate', () => {
			if (current.error) location.reload();
		});
	}

	function setup_preload() {
		/** @type {NodeJS.Timeout} */
		let mousemove_timeout;

		container.addEventListener('mousemove', (event) => {
			const target = /** @type {Element} */ (event.target);

			clearTimeout(mousemove_timeout);
			mousemove_timeout = setTimeout(() => {
				preload(target, 2);
			}, 20);
		});

		/** @param {Event} event */
		function tap(event) {
			preload(/** @type {Element} */ (event.composedPath()[0]), 1);
		}

		container.addEventListener('mousedown', tap);
		container.addEventListener('touchstart', tap, { passive: true });

		const observer = new IntersectionObserver(
			(entries) => {
				for (const entry of entries) {
					if (entry.isIntersecting) {
						preload_code(new URL(/** @type {HTMLAnchorElement} */ (entry.target).href).pathname);
						observer.unobserve(entry.target);
					}
				}
			},
			{ threshold: 0 }
		);

		/**
		 * @param {Element} element
		 * @param {number} priority
		 */
		function preload(element, priority) {
			const a = find_anchor(element, container);
			if (!a) return;

			const { url, external } = get_link_info(a, base);
			if (external) return;

			const options = get_router_options(a);

			if (!options.reload) {
				if (priority <= options.preload_data) {
					preload_data(/** @type {URL} */ (url));
				} else if (priority <= options.preload_code) {
					preload_code(/** @type {URL} */ (url).pathname);
				}
			}
		}

		function after_navigate() {
			observer.disconnect();

			for (const a of container.querySelectorAll('a')) {
				const { url, external } = get_link_info(a, base);
				if (external) continue;

				const options = get_router_options(a);
				if (options.reload) continue;

				if (options.preload_code === PRELOAD_PRIORITIES.viewport) {
					observer.observe(a);
				}

				if (options.preload_code === PRELOAD_PRIORITIES.eager) {
					preload_code(/** @type {URL} */ (url).pathname);
				}
			}
		}

		callbacks.after_navigate.push(after_navigate);
		after_navigate();
	}

	return {
		after_navigate: (fn) => {
			onMount(() => {
				callbacks.after_navigate.push(fn);

				return () => {
					const i = callbacks.after_navigate.indexOf(fn);
					callbacks.after_navigate.splice(i, 1);
				};
			});
		},

		before_navigate: (fn) => {
			onMount(() => {
				callbacks.before_navigate.push(fn);

				return () => {
					const i = callbacks.before_navigate.indexOf(fn);
					callbacks.before_navigate.splice(i, 1);
				};
			});
		},

		disable_scroll_handling: () => {
			if (DEV && started && !updating) {
				throw new Error('Can only disable scroll handling during navigation');
			}

			if (updating || !started) {
				autoscroll = false;
			}
		},

		goto: (href, opts = {}) => {
			// TODO remove for 1.0
			if ('keepfocus' in opts && !('keepFocus' in opts)) {
				throw new Error(
					'`keepfocus` has been renamed to `keepFocus` (note the difference in casing)'
				);
			}

			if ('noscroll' in opts && !('noScroll' in opts)) {
				throw new Error(
					'`noscroll` has been renamed to `noScroll` (note the difference in casing)'
				);
			}

			return goto(href, opts, []);
		},

		invalidate: (resource) => {
			if (resource === undefined) {
				// TODO remove for 1.0
				throw new Error(
					'`invalidate()` (with no arguments) has been replaced by `invalidateAll()`'
				);
			}

			if (typeof resource === 'function') {
				invalidated.push(resource);
			} else {
				const { href } = new URL(resource, location.href);
				invalidated.push((url) => url.href === href);
			}

			return invalidate();
		},

		invalidateAll: () => {
			force_invalidation = true;
			return invalidate();
		},

		preload_data: async (href) => {
			const url = new URL(href, get_base_uri(document));
			await preload_data(url);
		},

		preload_code,

		apply_action: async (result) => {
			if (result.type === 'error') {
				const url = new URL(location.href);

				const { branch, route } = current;
				if (!route) return;

				const error_load = await load_nearest_error_page(
					current.branch.length,
					branch,
					route.errors
				);
				if (error_load) {
					const navigation_result = await get_navigation_result_from_branch({
						url,
						params: current.params,
						branch: branch.slice(0, error_load.idx).concat(error_load.node),
						status: 500, // TODO might not be 500?
						error: result.error,
						route
					});

					current = navigation_result.state;

					const post_update = pre_update();
					root.$set(navigation_result.props);
					post_update();

					tick().then(reset_focus);
				}
			} else if (result.type === 'redirect') {
				goto(result.location, { invalidateAll: true }, []);
			} else {
				/** @type {Record<string, any>} */
				const props = {
					form: result.data,
					page: { ...page, form: result.data, status: result.status }
				};
				const post_update = pre_update();
				root.$set(props);
				post_update();

				if (result.type === 'success') {
					tick().then(reset_focus);
				}
			}
		},

		_start_router: () => {
			history.scrollRestoration = 'manual';

			// Adopted from Nuxt.js
			// Reset scrollRestoration to auto when leaving page, allowing page reload
			// and back-navigation from other pages to use the browser to restore the
			// scrolling position.
			addEventListener('beforeunload', (e) => {
				let should_block = false;

				if (!navigating) {
					// If we're navigating, beforeNavigate was already called. If we end up in here during navigation,
					// it's due to an external or full-page-reload link, for which we don't want to call the hook again.
					/** @type {import('types').BeforeNavigate} */
					const navigation = {
						from: add_url_properties('from', {
							params: current.params,
							route: { id: current.route?.id ?? null },
							url: current.url
						}),
						to: null,
						willUnload: true,
						type: 'leave',
						cancel: () => (should_block = true)
					};

					callbacks.before_navigate.forEach((fn) => fn(navigation));
				}

				if (should_block) {
					e.preventDefault();
					e.returnValue = '';
				} else {
					history.scrollRestoration = 'auto';
				}
			});

			addEventListener('visibilitychange', () => {
				if (document.visibilityState === 'hidden') {
					update_scroll_positions(current_history_index);

					try {
						sessionStorage[SCROLL_KEY] = JSON.stringify(scroll_positions);
					} catch {
						// do nothing
					}
				}
			});

			// @ts-expect-error this isn't supported everywhere yet
			if (!navigator.connection?.saveData) {
				setup_preload();
			}

			/** @param {MouseEvent} event */
			container.addEventListener('click', (event) => {
				// Adapted from https://github.com/visionmedia/page.js
				// MIT license https://github.com/visionmedia/page.js#license
				if (event.button || event.which !== 1) return;
				if (event.metaKey || event.ctrlKey || event.shiftKey || event.altKey) return;
				if (event.defaultPrevented) return;

				const a = find_anchor(/** @type {Element} */ (event.composedPath()[0]), container);
				if (!a) return;

				const { url, external, has } = get_link_info(a, base);
				const options = get_router_options(a);
				if (!url) return;

				const is_svg_a_element = a instanceof SVGAElement;

				// Ignore URL protocols that differ to the current one and are not http(s) (e.g. `mailto:`, `tel:`, `myapp:`, etc.)
				// This may be wrong when the protocol is x: and the link goes to y:.. which should be treated as an external
				// navigation, but it's not clear how to handle that case and it's not likely to come up in practice.
				// MEMO: Without this condition, firefox will open mailer twice.
				// See:
				// - https://github.com/sveltejs/kit/issues/4045
				// - https://github.com/sveltejs/kit/issues/5725
				// - https://github.com/sveltejs/kit/issues/6496
				if (
					!is_svg_a_element &&
					url.protocol !== location.protocol &&
					!(url.protocol === 'https:' || url.protocol === 'http:')
				)
					return;

				if (has.download) return;

				// Ignore the following but fire beforeNavigate
				if (external || options.reload) {
					const navigation = before_navigate({ url, type: 'link' });
					if (!navigation) {
						event.preventDefault();
					}
					navigating = true;
					return;
				}

				// Check if new url only differs by hash and use the browser default behavior in that case
				// This will ensure the `hashchange` event is fired
				// Removing the hash does a full page navigation in the browser, so make sure a hash is present
				const [nonhash, hash] = url.href.split('#');
				if (hash !== undefined && nonhash === location.href.split('#')[0]) {
					// set this flag to distinguish between navigations triggered by
					// clicking a hash link and those triggered by popstate
					// TODO why not update history here directly?
					hash_navigating = true;

					update_scroll_positions(current_history_index);

					current.url = url;
					stores.page.set({ ...page, url });
					stores.page.notify();

					return;
				}

				navigate({
					url,
					scroll: options.noscroll ? scroll_state() : null,
					keepfocus: false,
					redirect_chain: [],
					details: {
						state: {},
						replaceState: url.href === location.href
					},
					accepted: () => event.preventDefault(),
					blocked: () => event.preventDefault(),
					type: 'link'
				});
			});

			container.addEventListener('submit', (event) => {
				if (event.defaultPrevented) return;

				const form = /** @type {HTMLFormElement} */ (
					HTMLFormElement.prototype.cloneNode.call(event.target)
				);

				const submitter = /** @type {HTMLButtonElement | HTMLInputElement | null} */ (
					event.submitter
				);

				const method = submitter?.formMethod || form.method;

				if (method !== 'get') return;

				const url = new URL(
					(event.submitter?.hasAttribute('formaction') && submitter?.formAction) || form.action
				);

				if (is_external_url(url, base)) return;

				const { noscroll, reload } = get_router_options(
					/** @type {HTMLFormElement} */ (event.target)
				);
				if (reload) return;

				event.preventDefault();
				event.stopPropagation();

				// @ts-expect-error `URLSearchParams(fd)` is kosher, but typescript doesn't know that
				url.search = new URLSearchParams(new FormData(event.target)).toString();

				navigate({
					url,
					scroll: noscroll ? scroll_state() : null,
					keepfocus: false,
					redirect_chain: [],
					details: {
						state: {},
						replaceState: false
					},
					nav_token: {},
					accepted: () => {},
					blocked: () => {},
					type: 'form'
				});
			});

			addEventListener('popstate', (event) => {
				if (event.state?.[INDEX_KEY]) {
					// if a popstate-driven navigation is cancelled, we need to counteract it
					// with history.go, which means we end up back here, hence this check
					if (event.state[INDEX_KEY] === current_history_index) return;

					const delta = event.state[INDEX_KEY] - current_history_index;

					navigate({
						url: new URL(location.href),
						scroll: scroll_positions[event.state[INDEX_KEY]],
						keepfocus: false,
						redirect_chain: [],
						details: null,
						accepted: () => {
							current_history_index = event.state[INDEX_KEY];
						},
						blocked: () => {
							history.go(-delta);
						},
						type: 'popstate',
						delta
					});
				}
			});

			addEventListener('hashchange', () => {
				// if the hashchange happened as a result of clicking on a link,
				// we need to update history, otherwise we have to leave it alone
				if (hash_navigating) {
					hash_navigating = false;
					history.replaceState(
						{ ...history.state, [INDEX_KEY]: ++current_history_index },
						'',
						location.href
					);
				}
			});

			// fix link[rel=icon], because browsers will occasionally try to load relative
			// URLs after a pushState/replaceState, resulting in a 404 — see
			// https://github.com/sveltejs/kit/issues/3748#issuecomment-1125980897
			for (const link of document.querySelectorAll('link')) {
				if (link.rel === 'icon') link.href = link.href;
			}

			addEventListener('pageshow', (event) => {
				// If the user navigates to another site and then uses the back button and
				// bfcache hits, we need to set navigating to null, the site doesn't know
				// the navigation away from it was successful.
				// Info about bfcache here: https://web.dev/bfcache
				if (event.persisted) {
					stores.navigating.set(null);
				}
			});
		},

		_hydrate: async ({
			status = 200,
			error,
			node_ids,
			params,
			route,
			data: server_data_nodes,
			form
		}) => {
			hydrated = true;

			const url = new URL(location.href);

			if (!__SVELTEKIT_EMBEDDED__) {
				// See https://github.com/sveltejs/kit/pull/4935#issuecomment-1328093358 for one motivation
				// of determining the params on the client side.
				({ params = {}, route = { id: null } } = get_navigation_intent(url, false) || {});
			}

			/** @type {import('./types').NavigationFinished | undefined} */
			let result;

			try {
				const branch_promises = node_ids.map(async (n, i) => {
					const server_data_node = server_data_nodes[i];

					return load_node({
						loader: nodes[n],
						url,
						params,
						route,
						parent: async () => {
							const data = {};
							for (let j = 0; j < i; j += 1) {
								Object.assign(data, (await branch_promises[j]).data);
							}
							return data;
						},
						server_data_node: create_data_node(server_data_node)
					});
				});

				result = await get_navigation_result_from_branch({
					url,
					params,
					branch: await Promise.all(branch_promises),
					status,
					error,
					form,
					route: routes.find(({ id }) => id === route.id) ?? null
				});
			} catch (error) {
				if (error instanceof Redirect) {
					// this is a real edge case — `load` would need to return
					// a redirect but only in the browser
					await native_navigation(new URL(error.location, location.href));
					return;
				}

				result = await load_root_error_page({
					status: error instanceof HttpError ? error.status : 500,
					error: await handle_error(error, { url, params, route }),
					url,
					route
				});
			}

			initialize(result);
		}
	};
}

/**
 * @param {URL} url
 * @param {boolean[]} invalid
 * @returns {Promise<import('types').ServerData>}
 */
async function load_data(url, invalid) {
	const data_url = new URL(url);
	data_url.pathname = add_data_suffix(url.pathname);
	if (DEV && url.searchParams.has('x-sveltekit-invalidated')) {
		throw new Error('Cannot used reserved query parameter "x-sveltekit-invalidated"');
	}
	data_url.searchParams.append(
		'x-sveltekit-invalidated',
		invalid.map((x) => (x ? '1' : '')).join('_')
	);

	const res = await native_fetch(data_url.href);
	const data = await res.json();

	if (!res.ok) {
		// error message is a JSON-stringified string which devalue can't handle at the top level
		throw new Error(data);
	}

	// revive devalue-flattened data
	data.nodes?.forEach((/** @type {any} */ node) => {
		if (node?.type === 'data') {
			node.data = devalue.unflatten(node.data);
			node.uses = {
				dependencies: new Set(node.uses.dependencies ?? []),
				params: new Set(node.uses.params ?? []),
				parent: !!node.uses.parent,
				route: !!node.uses.route,
				url: !!node.uses.url
			};
		}
	});

	return data;
}

/**
 * @param {unknown} error
 * @param {import('types').NavigationEvent} event
 * @returns {import('../../../types/private.js').MaybePromise<App.Error>}
 */
function handle_error(error, event) {
	if (error instanceof HttpError) {
		return error.body;
	}
	return (
		hooks.handleError({ error, event }) ??
		/** @type {any} */ ({ message: event.route.id != null ? 'Internal Error' : 'Not Found' })
	);
}

// TODO remove for 1.0
const properties = [
	'hash',
	'href',
	'host',
	'hostname',
	'origin',
	'pathname',
	'port',
	'protocol',
	'search',
	'searchParams',
	'toString',
	'toJSON'
];

/**
 * @param {'from' | 'to'} type
 * @param {import('types').NavigationTarget} target
 */
function add_url_properties(type, target) {
	for (const prop of properties) {
		Object.defineProperty(target, prop, {
			get() {
				throw new Error(
					`The navigation shape changed - ${type}.${prop} should now be ${type}.url.${prop}`
				);
			},
			enumerable: false
		});
	}

	Object.defineProperty(target, 'routeId', {
		get() {
			throw new Error(
				`The navigation shape changed - ${type}.routeId should now be ${type}.route.id`
			);
		},
		enumerable: false
	});

	return target;
}

function pre_update() {
	if (DEV) {
		return () => {
			check_for_removed_attributes();
		};
	}

	return () => {};
}

function reset_focus() {
	const autofocus = document.querySelector('[autofocus]');
	if (autofocus) {
		// @ts-ignore
		autofocus.focus();
	} else {
		// Reset page selection and focus
		// We try to mimic browsers' behaviour as closely as possible by targeting the
		// first scrollable region, but unfortunately it's not a perfect match — e.g.
		// shift-tabbing won't immediately cycle up from the end of the page on Chromium
		// See https://html.spec.whatwg.org/multipage/interaction.html#get-the-focusable-area
		const root = document.body;
		const tabindex = root.getAttribute('tabindex');

		root.tabIndex = -1;
		root.focus({ preventScroll: true });

		setTimeout(() => {
			getSelection()?.removeAllRanges();
		});

		// restore `tabindex` as to prevent `root` from stealing input from elements
		if (tabindex !== null) {
			root.setAttribute('tabindex', tabindex);
		} else {
			root.removeAttribute('tabindex');
		}
	}
}

if (DEV) {
	// Nasty hack to silence harmless warnings the user can do nothing about
	const console_warn = console.warn;
	console.warn = function warn(...args) {
		if (
			args.length === 1 &&
			/<(Layout|Page)(_[\w$]+)?> was created (with unknown|without expected) prop '(data|form)'/.test(
				args[0]
			)
		) {
			return;
		}
		console_warn(...args);
	};
}
