//
//  MIKMIDIErrors.h
//  Danceability
//
//  Created by Andrew Madsen on 7/19/13.
//  Copyright (c) 2013 Mixed In Key. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "MIKMIDICompilerCompatibility.h"

NS_ASSUME_NONNULL_BEGIN

/**
 *  Error domain for errors generated by MIKMIDI.
 */
extern NSString * const MIKMIDIErrorDomain;

/**
 *  Error code values for NSError objects in the MIKMIDI error domain.
 */
typedef NS_ENUM(NSInteger, MIKMIDIErrorCode) {
	/**
	 *  Unknown error.
	 */
	MIKMIDIUnknownErrorCode = 1,
	/**
	 *	Invalid argument error.
	 */
	MIKMIDIInvalidArgumentError,
	/**
	 *  An error occurred because the connection to a device was lost.
	 */
	MIKMIDIDeviceConnectionLostErrorCode,
	/**
	 *  An connection error occurred because the device has no source endpoints.
	 */
	MIKMIDIDeviceHasNoSourcesErrorCode,
	/**
	 *  MIDI mapping using MIKMIDIMappingGenerator failed.
	 */
	MIKMIDIMappingFailedErrorCode,
	/**
	 *  The mapping file did not have the correct file extension (".midimap").
	 */
	MIKMIDIMappingIncorrectFileExtensionErrorCode,
	
	/**
	 *  An error ocurred while creating a new track in an MIKMIDISequence.
	 *  The error's info dictionary may contain an underlying error with 
	 *  more informationin for its NSUnderlyingErrorKey.
	 */
	MIKMIDISequenceAddTrackFailedErrorCode,
	
	/**
	 *  An error ocurred during an operation on event(s) in an MIKMIDITrack
	 *  because the event(s) could not be found in the track.
	 */
	MIKMIDITrackEventNotFoundErrorCode,
	
	/**
	 *  An error occurred selecting an instrument on MIKMIDISynthesizer,
	 *  because the synthesizer's underlying audio unit does not support
	 *  instrument selection, or MIKMIDI doesn't know how to select its
	 *  instruments.
	 */
	MIKMIDISynthesizerDoesNotSupportInstrumentSelectionError,
};

NSString *MIKMIDIDefaultLocalizedErrorDescriptionForErrorCode(MIKMIDIErrorCode code);

/**
 *  Category to ease creation of NSError instances in the MIKMIDI error domain.
 */
@interface NSError (MIKMIDI)

/**
 *  Creates an NSError instance whose domain is MIKMIDIErrorDomain. If the userInfo dictionary
 *  is nil, or does not contain a value for NSLocalizedDescriptionKey, a default description
 *  provided by MIKMIDIDefaultLocalizedErrorDescriptionForErrorCode() will be inserted.
 *
 *  @param code     An error code. Should be one of the values defined for MIKMIDIErrorCode.
 *  @param userInfo A user info dictionary. Can pass nil.
 *
 *  @return An initialized NSError instance whose domain is MIKMIDIErrorDomain.
 */
+ (instancetype)MIKMIDIErrorWithCode:(MIKMIDIErrorCode)code userInfo:(nullable NSDictionary *)userInfo;

@end

NS_ASSUME_NONNULL_END