<?php

declare(strict_types=1);

/**
 * NOTICE OF LICENSE.
 *
 * UNIT3D Community Edition is open-sourced software licensed under the GNU Affero General Public License v3.0
 * The details is bundled with this project in the file LICENSE.txt.
 *
 * @project    UNIT3D Community Edition
 *
 * @author     HDVinnie <hdinnovations@protonmail.com>
 * @license    https://www.gnu.org/licenses/agpl-3.0.en.html/ GNU Affero General Public License v3.0
 */

use App\Http\Controllers\Staff\PollController;
use App\Http\Requests\StorePoll;
use App\Http\Requests\UpdatePollRequest;
use App\Models\Bot;
use App\Models\Chatroom;
use App\Models\Group;
use App\Models\Poll;
use App\Models\User;
use Database\Seeders\UsersTableSeeder;

test('create a poll returns an ok response', function (): void {
    $this->seed(UsersTableSeeder::class);
    // Poll chat announcements needs a system user, a bot and a chatroom
    $bot = Bot::factory()->create([
        'command' => 'Systembot',
    ]);
    $chat = Chatroom::factory()->create([
        'name' => config('chat.system_chatroom'),
    ]);

    $group = Group::factory()->create([
        'is_modo' => 1,
    ]);
    $staff = User::factory()->create([
        'group_id' => $group->id,
        'active'   => 1,
    ]);

    $pollTitle = 'Test Poll Title';
    $pollOption1 = 'Option 1';
    $pollOption2 = 'Option 2';

    $response = $this->actingAs($staff)->withSession(['banned' => false])->post(route('staff.polls.store'), [
        'title'   => $pollTitle,
        'options' => [
            ['id' => 0, 'name' => $pollOption1],
            ['id' => 1, 'name' => $pollOption2],
        ],
        'multiple_choice' => 0,
    ]);

    $response->assertRedirectToRoute('staff.polls.index');
    $this->assertDatabaseHas('polls', [
        'title'      => $pollTitle,
        'user_id'    => $staff->id,
        'expires_at' => null,
    ]);
    $this->assertDatabaseHas('options', [
        'name' => $pollOption1,
    ]);
    $this->assertDatabaseHas('options', [
        'name' => $pollOption2,
    ]);
});

test('create a poll with expiration date returns an ok response', function (): void {
    $this->seed(UsersTableSeeder::class);
    // Poll chat announcements needs a system user, a bot and a chatroom
    $bot = Bot::factory()->create([
        'command' => 'Systembot',
    ]);
    $chat = Chatroom::factory()->create([
        'name' => config('chat.system_chatroom'),
    ]);

    $group = Group::factory()->create([
        'is_modo' => 1,
    ]);
    $staff = User::factory()->create([
        'group_id' => $group->id,
        'active'   => 1,
    ]);

    $pollTitle = 'Test Poll Title';
    $pollUntil = now()->addDays(1);
    $pollOption1 = 'Option 1';
    $pollOption2 = 'Option 2';

    $response = $this->actingAs($staff)->post(route('staff.polls.store'), [
        'title'      => $pollTitle,
        'expires_at' => $pollUntil,
        'options'    => [
            ['id' => 0, 'name' => $pollOption1],
            ['id' => 1, 'name' => $pollOption2],
        ],
        'multiple_choice' => 0,
    ]);

    $this->assertDatabaseHas('polls', [
        'title'      => $pollTitle,
        'user_id'    => $staff->id,
        'expires_at' => $pollUntil,
    ]);
    $this->assertDatabaseHas('options', [
        'name' => $pollOption1,
    ]);
    $this->assertDatabaseHas('options', [
        'name' => $pollOption2,
    ]);
});

test('destroy returns an ok response', function (): void {
    $this->markTestIncomplete('This test case was generated by Shift. When you are ready, remove this line and complete this test case.');

    $poll = Poll::factory()->create();
    $user = User::factory()->create();

    $response = $this->actingAs($user)->delete(route('staff.polls.destroy', [$poll]));

    $response->assertOk();
    $this->assertModelMissing($poll);

    // TODO: perform additional assertions
});

test('edit returns an ok response', function (): void {
    $this->markTestIncomplete('This test case was generated by Shift. When you are ready, remove this line and complete this test case.');

    $poll = Poll::factory()->create();
    $user = User::factory()->create();

    $response = $this->actingAs($user)->get(route('staff.polls.edit', [$poll]));

    $response->assertOk();
    $response->assertViewIs('Staff.poll.edit');
    $response->assertViewHas('poll', $poll);

    // TODO: perform additional assertions
});

test('index returns an ok response', function (): void {
    $polls = Poll::factory()->times(3)->create([
        'expires_at' => null,
    ]);
    $pollExpired = Poll::factory()->create([
        'expires_at' => now()->subDays(1),
    ]);

    $group = Group::factory()->create([
        'is_modo' => 1,
    ]);
    $staff = User::factory()->create([
        'group_id' => $group->id,
        'active'   => 1,
    ]);

    $response = $this->actingAs($staff)->get(route('staff.polls.index'));

    $response->assertOk();
    $response->assertViewIs('Staff.poll.index');
    $response->assertSee('Closed');
    $response->assertSee('Open');
});

test('show returns an ok response', function (): void {
    $this->markTestIncomplete('This test case was generated by Shift. When you are ready, remove this line and complete this test case.');

    $poll = Poll::factory()->create();
    $user = User::factory()->create();

    $response = $this->actingAs($user)->get(route('staff.polls.show', [$poll]));

    $response->assertOk();
    $response->assertViewIs('Staff.poll.show');
    $response->assertViewHas('poll', $poll);

    // TODO: perform additional assertions
});

test('store validates with a form request', function (): void {
    $this->assertActionUsesFormRequest(
        PollController::class,
        'store',
        StorePoll::class
    );
});

test('store returns an ok response', function (): void {
    $this->markTestIncomplete('This test case was generated by Shift. When you are ready, remove this line and complete this test case.');

    $user = User::factory()->create();

    $response = $this->actingAs($user)->post(route('staff.polls.store'), [
        // TODO: send request data
    ]);

    $response->assertOk();

    // TODO: perform additional assertions
});

test('update validates with a form request', function (): void {
    $this->assertActionUsesFormRequest(
        PollController::class,
        'update',
        UpdatePollRequest::class
    );
});

test('update returns an ok response', function (): void {
    $this->markTestIncomplete('This test case was generated by Shift. When you are ready, remove this line and complete this test case.');

    $poll = Poll::factory()->create();
    $user = User::factory()->create();

    $response = $this->actingAs($user)->patch(route('staff.polls.update', [$poll]), [
        // TODO: send request data
    ]);

    $response->assertOk();

    // TODO: perform additional assertions
});

// test cases...
