/* Copyright (c) 2019-2024 Griefer@Work                                       *
 *                                                                            *
 * This software is provided 'as-is', without any express or implied          *
 * warranty. In no event will the authors be held liable for any damages      *
 * arising from the use of this software.                                     *
 *                                                                            *
 * Permission is granted to anyone to use this software for any purpose,      *
 * including commercial applications, and to alter it and redistribute it     *
 * freely, subject to the following restrictions:                             *
 *                                                                            *
 * 1. The origin of this software must not be misrepresented; you must not    *
 *    claim that you wrote the original software. If you use this software    *
 *    in a product, an acknowledgement (see the following) in the product     *
 *    documentation is required:                                              *
 *    Portions Copyright (c) 2019-2024 Griefer@Work                           *
 * 2. Altered source versions must be plainly marked as such, and must not be *
 *    misrepresented as being the original software.                          *
 * 3. This notice may not be removed or altered from any source distribution. *
 */
#ifndef GUARD_LIBDISASM_ARM_DB_H
#define GUARD_LIBDISASM_ARM_DB_H 1
#define _KOS_SOURCE 1

#include "api.h"
/**/

#include <hybrid/byteorder.h>
#include <hybrid/typecore.h>

#include <kos/types.h>

#include <string.h>

DECL_BEGIN

struct arm_instruction {
	/* The `ai_repr' field is set-up as follow:
	 * >> [F_*] "mnemonic" [S_*] ["\t" [O_*]] */
	uint32_t                      ai_mask;  /* Mask of bits to test with `ai_bits' */
	uint32_t                      ai_bits;  /* Mask of mandatory 1-bits */
	COMPILER_FLEXIBLE_ARRAY(char, ai_repr); /* Opcode repr */
	/* Next instruction begins at `CEIL_ALIGN(strend(ai_repr) + 1, 4)' */
};

/* Return a pointer to the instruction that follows `self' */
#define arm_instruction_next(self) \
	((struct arm_instruction *)((uintptr_t)(strend((self)->ai_repr) + 1 + 3) & ~3))

struct arm_instruction_group {
	uint32_t aig_mask;  /* Mask of bits to test with `aig_bits' */
	uint32_t aig_bits;  /* Mask of mandatory 1-bits */
	uint16_t aig_size;  /* # of instructions in this group */
	uint16_t aig_offs;  /* Offset of group into instruction database */
};


#define U16 __UINT16_C
#define U32 __UINT32_C

/* Encode a 32-bit unsigned integer into a byte-stream */
#ifdef __INTELLISENSE__
#define P32(w) 0, 0, 0, 0
#elif __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
#define P32(w) U32(w) & 0xff, (U32(w) >> 8) & 0xff, (U32(w) >> 16) & 0xff, (U32(w) >> 24) & 0xff
#else /* __BYTE_ORDER__ == ... */
#define P32(w) (U32(w) >> 24) & 0xff, (U32(w) >> 16) & 0xff, (U32(w) >> 8) & 0xff, U32(w) & 0xff
#endif /* __BYTE_ORDER__ != ... */




/************************************************************************/
/* AUTO-GENERATED CODE                                                  */
/************************************************************************/
/* compiler: "./arm-db.dee"                                             */
/* source:   "./arm-db"                                                 */
/************************************************************************/

/* clang-format off */
/*[[[deemon import * from ".arm-db"; printInstructionDatabase();]]]*/
PRIVATE struct arm_instruction_group const t16_instruction_groups[72] = {
	{ U32(0x0000ffc0), U32(0x00004140), U16(2), U16(0x0000) },
	{ U32(0x0000f800), U32(0x0000a800), U16(1), U16(0x0028) },
	{ U32(0x0000ff00), U32(0x00004400), U16(1), U16(0x003d) },
	{ U32(0x0000ff80), U32(0x0000b000), U16(1), U16(0x004e) },
	{ U32(0x0000fe00), U32(0x00001c00), U16(2), U16(0x0062) },
	{ U32(0x0000fe00), U32(0x00001800), U16(2), U16(0x008c) },
	{ U32(0x0000f800), U32(0x00003000), U16(2), U16(0x00b6) },
	{ U32(0x0000f800), U32(0x0000a000), U16(1), U16(0x00de) },
	{ U32(0x0000ffc0), U32(0x00004000), U16(2), U16(0x00ef) },
	{ U32(0x0000ffc0), U32(0x00004100), U16(2), U16(0x0117) },
	{ U32(0x0000f800), U32(0x00001000), U16(1), U16(0x013f) },
	{ U32(0x0000f000), U32(0x0000d000), U16(2), U16(0x0154) },
	{ U32(0x0000f800), U32(0x0000e000), U16(1), U16(0x0174) },
	{ U32(0x0000ffc0), U32(0x00004380), U16(2), U16(0x0184) },
	{ U32(0x0000ff00), U32(0x0000be00), U16(1), U16(0x01ac) },
	{ U32(0x0000ff87), U32(0x00004780), U16(1), U16(0x01bc) },
	{ U32(0x0000ff87), U32(0x00004700), U16(1), U16(0x01cc) },
	{ U32(0x0000fd00), U32(0x0000b100), U16(1), U16(0x01dd) },
	{ U32(0x0000fd00), U32(0x0000b900), U16(1), U16(0x01ed) },
	{ U32(0x0000ffc0), U32(0x000042c0), U16(1), U16(0x01fe) },
	{ U32(0x0000f800), U32(0x00002800), U16(1), U16(0x020f) },
	{ U32(0x0000ffc0), U32(0x00004280), U16(1), U16(0x0220) },
	{ U32(0x0000ff00), U32(0x00004500), U16(1), U16(0x0231) },
	{ U32(0x0000ffc0), U32(0x00004040), U16(2), U16(0x0242) },
	{ U32(0x0000ff00), U32(0x0000bf00), U16(6), U16(0x026a) },
	{ U32(0x0000f800), U32(0x0000c800), U16(1), U16(0x02cb) },
	{ U32(0x0000f800), U32(0x00006800), U16(1), U16(0x02df) },
	{ U32(0x0000f800), U32(0x00004800), U16(1), U16(0x02f3) },
	{ U32(0x0000fe00), U32(0x00005800), U16(1), U16(0x0304) },
	{ U32(0x0000f800), U32(0x00009800), U16(1), U16(0x0318) },
	{ U32(0x0000fe00), U32(0x00005c00), U16(1), U16(0x032d) },
	{ U32(0x0000f800), U32(0x00007800), U16(1), U16(0x0342) },
	{ U32(0x0000fe00), U32(0x00005a00), U16(1), U16(0x0357) },
	{ U32(0x0000f800), U32(0x00008800), U16(1), U16(0x036c) },
	{ U32(0x0000fe00), U32(0x00005600), U16(1), U16(0x0381) },
	{ U32(0x0000fe00), U32(0x00005e00), U16(1), U16(0x0399) },
	{ U32(0x0000ffc0), U32(0x00004080), U16(2), U16(0x03b1) },
	{ U32(0x0000f800), U32(0x00000000), U16(3), U16(0x03d9) },
	{ U32(0x0000ffc0), U32(0x000040c0), U16(2), U16(0x0416) },
	{ U32(0x0000f800), U32(0x00000800), U16(2), U16(0x043e) },
	{ U32(0x0000ff00), U32(0x00004600), U16(1), U16(0x0468) },
	{ U32(0x0000f800), U32(0x00002000), U16(2), U16(0x0479) },
	{ U32(0x0000ffc0), U32(0x00004340), U16(2), U16(0x04a1) },
	{ U32(0x0000ffc0), U32(0x000043c0), U16(2), U16(0x04cb) },
	{ U32(0x0000ffc0), U32(0x00004300), U16(2), U16(0x04f3) },
	{ U32(0x0000fe00), U32(0x0000bc00), U16(1), U16(0x051b) },
	{ U32(0x0000fe00), U32(0x0000b400), U16(1), U16(0x052b) },
	{ U32(0x0000ffc0), U32(0x0000ba40), U16(1), U16(0x053c) },
	{ U32(0x0000ffc0), U32(0x0000ba00), U16(1), U16(0x0551) },
	{ U32(0x0000ffc0), U32(0x0000bac0), U16(1), U16(0x0562) },
	{ U32(0x0000ffc0), U32(0x000041c0), U16(2), U16(0x0577) },
	{ U32(0x0000ffc0), U32(0x00004240), U16(2), U16(0x059f) },
	{ U32(0x0000ffc0), U32(0x00004180), U16(2), U16(0x05c7) },
	{ U32(0x0000ffff), U32(0x0000b650), U16(1), U16(0x05ef) },
	{ U32(0x0000ffff), U32(0x0000b658), U16(1), U16(0x0604) },
	{ U32(0x0000f800), U32(0x0000c000), U16(1), U16(0x0619) },
	{ U32(0x0000f800), U32(0x00006000), U16(1), U16(0x062d) },
	{ U32(0x0000fe00), U32(0x00005000), U16(1), U16(0x0641) },
	{ U32(0x0000f800), U32(0x00009000), U16(1), U16(0x0655) },
	{ U32(0x0000f800), U32(0x00007000), U16(1), U16(0x066a) },
	{ U32(0x0000fe00), U32(0x00005400), U16(1), U16(0x067f) },
	{ U32(0x0000fe00), U32(0x00005200), U16(1), U16(0x0694) },
	{ U32(0x0000f800), U32(0x00008000), U16(1), U16(0x06a9) },
	{ U32(0x0000ff80), U32(0x0000b080), U16(1), U16(0x06be) },
	{ U32(0x0000fe00), U32(0x00001e00), U16(2), U16(0x06d2) },
	{ U32(0x0000fe00), U32(0x00001a00), U16(2), U16(0x06fc) },
	{ U32(0x0000f800), U32(0x00003800), U16(2), U16(0x0726) },
	{ U32(0x0000ffc0), U32(0x0000b240), U16(1), U16(0x074e) },
	{ U32(0x0000ffc0), U32(0x0000b200), U16(1), U16(0x0762) },
	{ U32(0x0000ffc0), U32(0x00004200), U16(1), U16(0x0776) },
	{ U32(0x0000ffc0), U32(0x0000b2c0), U16(1), U16(0x0787) },
	{ U32(0x0000ffc0), U32(0x0000b280), U16(1), U16(0x079b) },
}; /* t16_instruction_groups */

PRIVATE struct arm_instruction_group const t32_instruction_groups[174] = {
	{ U32(0xfbe08000), U32(0xf1400000), U16(1), U16(0x07af) },
	{ U32(0xffe08000), U32(0xeb400000), U16(1), U16(0x07c4) },
	{ U32(0xfbf08000), U32(0xf2000000), U16(2), U16(0x07d9) },
	{ U32(0xfbe08000), U32(0xf1000000), U16(2), U16(0x0803) },
	{ U32(0xffe08000), U32(0xeb000000), U16(2), U16(0x0828) },
	{ U32(0xfbf08000), U32(0xf2a00000), U16(2), U16(0x0852) },
	{ U32(0xfbe08000), U32(0xf0000000), U16(2), U16(0x087c) },
	{ U32(0xffe08000), U32(0xea000000), U16(2), U16(0x08a2) },
	{ U32(0xffe08000), U32(0xea400000), U16(7), U16(0x08c8) },
	{ U32(0xffe0f0f0), U32(0xfa40f000), U16(1), U16(0x0957) },
	{ U32(0xf800d000), U32(0xf0008000), U16(17), U16(0x096b) },
	{ U32(0xf800d000), U32(0xf0009000), U16(1), U16(0x0aa5) },
	{ U32(0xfff08020), U32(0xf3600000), U16(2), U16(0x0ab6) },
	{ U32(0xfbe08000), U32(0xf0200000), U16(1), U16(0x0adf) },
	{ U32(0xffe08000), U32(0xea200000), U16(1), U16(0x0af4) },
	{ U32(0xf800d000), U32(0xf000d000), U16(1), U16(0x0b09) },
	{ U32(0xf800d001), U32(0xf000c000), U16(1), U16(0x0b1a) },
	{ U32(0xff000010), U32(0xfe000000), U16(1), U16(0x0b2a) },
	{ U32(0xff000010), U32(0xee000000), U16(1), U16(0x0b42) },
	{ U32(0xfff0f0f0), U32(0xfab0f080), U16(1), U16(0x0b57) },
	{ U32(0xffe08000), U32(0xeba00000), U16(2), U16(0x0b68) },
	{ U32(0xfbe08000), U32(0xf1a00000), U16(2), U16(0x0b92) },
	{ U32(0xfbe08000), U32(0xf0800000), U16(2), U16(0x0bba) },
	{ U32(0xffe08000), U32(0xea800000), U16(2), U16(0x0be0) },
	{ U32(0xfe100000), U32(0xfc100000), U16(3), U16(0x0c09) },
	{ U32(0xfe100000), U32(0xec100000), U16(3), U16(0x0c4e) },
	{ U32(0xffd00000), U32(0xe8900000), U16(2), U16(0x0c90) },
	{ U32(0xffd02000), U32(0xe9100000), U16(1), U16(0x0cb6) },
	{ U32(0xff700000), U32(0xf8500000), U16(7), U16(0x0cca) },
	{ U32(0xff700000), U32(0xf8100000), U16(10), U16(0x0d67) },
	{ U32(0xff500000), U32(0xe9500000), U16(2), U16(0x0e4c) },
	{ U32(0xff700000), U32(0xe8700000), U16(1), U16(0x0e79) },
	{ U32(0xfff00f00), U32(0xe8500f00), U16(1), U16(0x0e92) },
	{ U32(0xfff00fff), U32(0xe8d00f4f), U16(1), U16(0x0eaa) },
	{ U32(0xfff000ff), U32(0xe8d0007f), U16(1), U16(0x0ec2) },
	{ U32(0xfff00fff), U32(0xe8d00f5f), U16(1), U16(0x0edb) },
	{ U32(0xff700000), U32(0xf8300000), U16(9), U16(0x0ef3) },
	{ U32(0xff700000), U32(0xf9100000), U16(10), U16(0x0fcc) },
	{ U32(0xff700000), U32(0xf9300000), U16(6), U16(0x10b4) },
	{ U32(0xffe0f0f0), U32(0xfa00f000), U16(1), U16(0x1148) },
	{ U32(0xffe0f0f0), U32(0xfa20f000), U16(1), U16(0x115c) },
	{ U32(0xff100010), U32(0xfe000010), U16(1), U16(0x1170) },
	{ U32(0xff100010), U32(0xee000010), U16(1), U16(0x1188) },
	{ U32(0xfff00000), U32(0xfc400000), U16(1), U16(0x11a1) },
	{ U32(0xfff00000), U32(0xec400000), U16(1), U16(0x11b9) },
	{ U32(0xfff000f0), U32(0xfb000000), U16(2), U16(0x11ce) },
	{ U32(0xfff000f0), U32(0xfb000010), U16(1), U16(0x11f7) },
	{ U32(0xfbf08000), U32(0xf2c00000), U16(1), U16(0x120c) },
	{ U32(0xfbf08000), U32(0xf2400000), U16(1), U16(0x1220) },
	{ U32(0xfbe08000), U32(0xf0400000), U16(2), U16(0x1234) },
	{ U32(0xff100010), U32(0xfe100010), U16(1), U16(0x125e) },
	{ U32(0xff100010), U32(0xee100010), U16(1), U16(0x1276) },
	{ U32(0xfbe08000), U32(0xf0600000), U16(2), U16(0x128f) },
	{ U32(0xffe08000), U32(0xea600000), U16(2), U16(0x12b8) },
	{ U32(0xfff08010), U32(0xeac00000), U16(1), U16(0x12e0) },
	{ U32(0xffd00000), U32(0xe8800000), U16(2), U16(0x12fc) },
	{ U32(0xfff00c00), U32(0xf8400c00), U16(3), U16(0x1325) },
	{ U32(0xfff0f0f0), U32(0xfa90f010), U16(1), U16(0x1366) },
	{ U32(0xfff0f0f0), U32(0xfa80f010), U16(1), U16(0x137b) },
	{ U32(0xfff0f0f0), U32(0xfa80f080), U16(1), U16(0x138f) },
	{ U32(0xfff0f0f0), U32(0xfaa0f010), U16(1), U16(0x13a4) },
	{ U32(0xfff0f0f0), U32(0xfa80f090), U16(1), U16(0x13b9) },
	{ U32(0xfff0f0f0), U32(0xfa80f0b0), U16(1), U16(0x13cd) },
	{ U32(0xfff0f0f0), U32(0xfae0f010), U16(1), U16(0x13e1) },
	{ U32(0xfff0f0f0), U32(0xfad0f010), U16(1), U16(0x13f6) },
	{ U32(0xfff0f0f0), U32(0xfac0f010), U16(1), U16(0x140b) },
	{ U32(0xfff0f0f0), U32(0xfa80f0a0), U16(1), U16(0x141f) },
	{ U32(0xfff0f0f0), U32(0xfa90f0a0), U16(1), U16(0x1434) },
	{ U32(0xfff0f0f0), U32(0xfa90f090), U16(1), U16(0x1448) },
	{ U32(0xfff0f0f0), U32(0xfa90f080), U16(1), U16(0x145c) },
	{ U32(0xfff0f0f0), U32(0xfa90f0b0), U16(1), U16(0x1470) },
	{ U32(0xffe0f0f0), U32(0xfa60f000), U16(1), U16(0x1484) },
	{ U32(0xffe08000), U32(0xebc00000), U16(1), U16(0x1498) },
	{ U32(0xfbe08000), U32(0xf1c00000), U16(1), U16(0x14ac) },
	{ U32(0xfff0f0f0), U32(0xfa90f000), U16(1), U16(0x14c0) },
	{ U32(0xfff0f0f0), U32(0xfa80f000), U16(1), U16(0x14d5) },
	{ U32(0xfff0f0f0), U32(0xfaa0f000), U16(1), U16(0x14e9) },
	{ U32(0xfbe08000), U32(0xf1600000), U16(1), U16(0x14fe) },
	{ U32(0xffe08000), U32(0xeb600000), U16(1), U16(0x1513) },
	{ U32(0xfff08020), U32(0xf3400000), U16(1), U16(0x1528) },
	{ U32(0xfff0f0f0), U32(0xfb90f0f0), U16(1), U16(0x1540) },
	{ U32(0xfff0f0f0), U32(0xfaa0f080), U16(1), U16(0x1555) },
	{ U32(0xfff0f0f0), U32(0xfa90f020), U16(1), U16(0x1569) },
	{ U32(0xfff0f0f0), U32(0xfa80f020), U16(1), U16(0x1581) },
	{ U32(0xfff0f0f0), U32(0xfaa0f020), U16(1), U16(0x1596) },
	{ U32(0xfff0f0f0), U32(0xfae0f020), U16(1), U16(0x15aa) },
	{ U32(0xfff0f0f0), U32(0xfad0f020), U16(1), U16(0x15be) },
	{ U32(0xfff0f0f0), U32(0xfac0f020), U16(1), U16(0x15d6) },
	{ U32(0xfff000f0), U32(0xfb100030), U16(2), U16(0x15eb) },
	{ U32(0xfff000f0), U32(0xfb100010), U16(2), U16(0x1618) },
	{ U32(0xfff000f0), U32(0xfb100020), U16(2), U16(0x1645) },
	{ U32(0xfff000f0), U32(0xfb100000), U16(2), U16(0x1672) },
	{ U32(0xfff000f0), U32(0xfb200010), U16(2), U16(0x169f) },
	{ U32(0xfff000f0), U32(0xfb200000), U16(2), U16(0x16cc) },
	{ U32(0xfff000f0), U32(0xfbc00000), U16(1), U16(0x16f5) },
	{ U32(0xfff000f0), U32(0xfbc000b0), U16(1), U16(0x170a) },
	{ U32(0xfff000f0), U32(0xfbc00090), U16(1), U16(0x1723) },
	{ U32(0xfff000f0), U32(0xfbc000a0), U16(1), U16(0x173c) },
	{ U32(0xfff000f0), U32(0xfbc00080), U16(1), U16(0x1755) },
	{ U32(0xfff000f0), U32(0xfbc000d0), U16(1), U16(0x176e) },
	{ U32(0xfff000f0), U32(0xfbc000c0), U16(1), U16(0x1787) },
	{ U32(0xfff000f0), U32(0xfb300010), U16(2), U16(0x179f) },
	{ U32(0xfff000f0), U32(0xfb300000), U16(2), U16(0x17cc) },
	{ U32(0xfff000f0), U32(0xfb400010), U16(2), U16(0x17f9) },
	{ U32(0xfff000f0), U32(0xfb400000), U16(2), U16(0x1826) },
	{ U32(0xfff000f0), U32(0xfbd000d0), U16(1), U16(0x1852) },
	{ U32(0xfff000f0), U32(0xfbd000c0), U16(1), U16(0x186b) },
	{ U32(0xfff000f0), U32(0xfb500010), U16(2), U16(0x1884) },
	{ U32(0xfff000f0), U32(0xfb500000), U16(2), U16(0x18b1) },
	{ U32(0xfff000f0), U32(0xfb600010), U16(1), U16(0x18da) },
	{ U32(0xfff000f0), U32(0xfb600000), U16(1), U16(0x18f2) },
	{ U32(0xfff000f0), U32(0xfb800000), U16(1), U16(0x1907) },
	{ U32(0xffd08020), U32(0xf3000000), U16(2), U16(0x191c) },
	{ U32(0xfff0f0f0), U32(0xfae0f000), U16(1), U16(0x1945) },
	{ U32(0xfff0f0f0), U32(0xfad0f000), U16(1), U16(0x195a) },
	{ U32(0xfff0f0f0), U32(0xfac0f000), U16(1), U16(0x196f) },
	{ U32(0xff500000), U32(0xfd000000), U16(1), U16(0x1983) },
	{ U32(0xff700000), U32(0xfc200000), U16(1), U16(0x199b) },
	{ U32(0xff500000), U32(0xfd400000), U16(1), U16(0x19b4) },
	{ U32(0xff700000), U32(0xfc600000), U16(1), U16(0x19cd) },
	{ U32(0xff500000), U32(0xed000000), U16(1), U16(0x19e5) },
	{ U32(0xff700000), U32(0xec200000), U16(1), U16(0x19fe) },
	{ U32(0xff500000), U32(0xed400000), U16(1), U16(0x1a16) },
	{ U32(0xff700000), U32(0xec600000), U16(1), U16(0x1a2e) },
	{ U32(0xffd0a000), U32(0xe9000000), U16(1), U16(0x1a47) },
	{ U32(0xfff00d00), U32(0xf8400900), U16(1), U16(0x1a5b) },
	{ U32(0xfff00000), U32(0xf8c00000), U16(1), U16(0x1a70) },
	{ U32(0xfff00fc0), U32(0xf8400000), U16(1), U16(0x1a85) },
	{ U32(0xfff00c00), U32(0xf8000c00), U16(2), U16(0x1aa2) },
	{ U32(0xfff00d00), U32(0xf8000900), U16(1), U16(0x1ad3) },
	{ U32(0xfff00000), U32(0xf8800000), U16(1), U16(0x1aeb) },
	{ U32(0xfff00fc0), U32(0xf8000000), U16(1), U16(0x1b03) },
	{ U32(0xff500000), U32(0xe9400000), U16(1), U16(0x1b1f) },
	{ U32(0xff700000), U32(0xe8600000), U16(1), U16(0x1b37) },
	{ U32(0xfff00000), U32(0xe8400000), U16(1), U16(0x1b50) },
	{ U32(0xfff00ff0), U32(0xe8c00f40), U16(1), U16(0x1b69) },
	{ U32(0xfff000f0), U32(0xe8c00070), U16(1), U16(0x1b82) },
	{ U32(0xfff00ff0), U32(0xe8c00f50), U16(1), U16(0x1b9a) },
	{ U32(0xfff00c00), U32(0xf8200c00), U16(2), U16(0x1bb3) },
	{ U32(0xfff00d00), U32(0xf8200900), U16(1), U16(0x1be4) },
	{ U32(0xfff00fc0), U32(0xf8200000), U16(1), U16(0x1bfc) },
	{ U32(0xfff00000), U32(0xf8a00000), U16(1), U16(0x1c18) },
	{ U32(0xfff0f0c0), U32(0xfa20f080), U16(2), U16(0x1c30) },
	{ U32(0xfff0f0c0), U32(0xfa40f080), U16(2), U16(0x1c5e) },
	{ U32(0xfff0f0c0), U32(0xfa00f080), U16(2), U16(0x1c87) },
	{ U32(0xfff0ffe0), U32(0xe8d0f000), U16(1), U16(0x1cb0) },
	{ U32(0xfff0f0f0), U32(0xfa90f040), U16(1), U16(0x1cc5) },
	{ U32(0xfff0f0f0), U32(0xfa80f040), U16(1), U16(0x1cda) },
	{ U32(0xfff0f0f0), U32(0xfaa0f040), U16(1), U16(0x1cee) },
	{ U32(0xfff08020), U32(0xf3c00000), U16(1), U16(0x1d03) },
	{ U32(0xfff0f0f0), U32(0xfbb0f0f0), U16(1), U16(0x1d1b) },
	{ U32(0xfff0f0f0), U32(0xfa90f060), U16(1), U16(0x1d30) },
	{ U32(0xfff0f0f0), U32(0xfa80f060), U16(1), U16(0x1d48) },
	{ U32(0xfff0f0f0), U32(0xfaa0f060), U16(1), U16(0x1d5d) },
	{ U32(0xfff0f0f0), U32(0xfae0f060), U16(1), U16(0x1d71) },
	{ U32(0xfff0f0f0), U32(0xfad0f060), U16(1), U16(0x1d85) },
	{ U32(0xfff0f0f0), U32(0xfac0f060), U16(1), U16(0x1d9d) },
	{ U32(0xfff000f0), U32(0xfbe00060), U16(1), U16(0x1db2) },
	{ U32(0xfff000f0), U32(0xfbe00000), U16(1), U16(0x1dc7) },
	{ U32(0xfff000f0), U32(0xfba00000), U16(1), U16(0x1ddc) },
	{ U32(0xfff0f0f0), U32(0xfa90f050), U16(1), U16(0x1df1) },
	{ U32(0xfff0f0f0), U32(0xfa80f050), U16(1), U16(0x1e09) },
	{ U32(0xfff0f0f0), U32(0xfaa0f050), U16(1), U16(0x1e1e) },
	{ U32(0xfff0f0f0), U32(0xfae0f050), U16(1), U16(0x1e32) },
	{ U32(0xfff0f0f0), U32(0xfad0f050), U16(1), U16(0x1e46) },
	{ U32(0xfff0f0f0), U32(0xfac0f050), U16(1), U16(0x1e5e) },
	{ U32(0xfff000f0), U32(0xfb700000), U16(2), U16(0x1e73) },
	{ U32(0xffd08020), U32(0xf3800000), U16(2), U16(0x1e9f) },
	{ U32(0xfff0f0f0), U32(0xfae0f040), U16(1), U16(0x1ec9) },
	{ U32(0xfff0f0f0), U32(0xfad0f040), U16(1), U16(0x1ede) },
	{ U32(0xfff0f0f0), U32(0xfac0f040), U16(1), U16(0x1ef3) },
	{ U32(0xfff0f0c0), U32(0xfa30f080), U16(2), U16(0x1f07) },
	{ U32(0xfff0f0c0), U32(0xfa50f080), U16(2), U16(0x1f35) },
	{ U32(0xfff0f0c0), U32(0xfa10f080), U16(2), U16(0x1f5e) },
}; /* t32_instruction_groups */

PRIVATE struct arm_instruction_group const a32_instruction_groups[267] = {
	{ U32(0x0fe00000), U32(0x02a00000), U16(1), U16(0x1f87) },
	{ U32(0x0fe00090), U32(0x00a00010), U16(1), U16(0x1f9c) },
	{ U32(0x0fe00010), U32(0x00a00000), U16(1), U16(0x1fb1) },
	{ U32(0x0fe00000), U32(0x02800000), U16(2), U16(0x1fc5) },
	{ U32(0x0fe00090), U32(0x00800010), U16(1), U16(0x1fee) },
	{ U32(0x0fe00010), U32(0x00800000), U16(1), U16(0x2003) },
	{ U32(0x0fe00000), U32(0x02400000), U16(2), U16(0x2017) },
	{ U32(0x0fe00000), U32(0x02000000), U16(1), U16(0x2040) },
	{ U32(0x0fe00090), U32(0x00000010), U16(1), U16(0x2055) },
	{ U32(0x0fe00010), U32(0x00000000), U16(1), U16(0x206a) },
	{ U32(0x0fef0070), U32(0x01a00040), U16(1), U16(0x207e) },
	{ U32(0x0fef00f0), U32(0x01a00050), U16(1), U16(0x2093) },
	{ U32(0x0e000000), U32(0x0a000000), U16(3), U16(0x20a8) },
	{ U32(0x0fe00070), U32(0x07c00010), U16(2), U16(0x20da) },
	{ U32(0x0fe00000), U32(0x03c00000), U16(1), U16(0x2103) },
	{ U32(0x0fe00090), U32(0x01c00010), U16(1), U16(0x2118) },
	{ U32(0x0fe00010), U32(0x01c00000), U16(1), U16(0x212d) },
	{ U32(0x0ff000f0), U32(0x01200070), U16(1), U16(0x2141) },
	{ U32(0x0ffffff0), U32(0x012fff30), U16(1), U16(0x2151) },
	{ U32(0x0ffffff0), U32(0x012fff10), U16(1), U16(0x2161) },
	{ U32(0x0ffffff0), U32(0x012fff20), U16(1), U16(0x2172) },
	{ U32(0xff000010), U32(0xfe000000), U16(1), U16(0x0b2a) },
	{ U32(0xff000010), U32(0xee000000), U16(1), U16(0x0b42) },
	{ U32(0x0f500000), U32(0x05500000), U16(8), U16(0x2182) },
	{ U32(0x0fff0ff0), U32(0x016f0f10), U16(1), U16(0x2214) },
	{ U32(0x0ff0f000), U32(0x03700000), U16(1), U16(0x2225) },
	{ U32(0x0ff0f090), U32(0x01700010), U16(1), U16(0x2236) },
	{ U32(0x0ff0f010), U32(0x01700000), U16(1), U16(0x224b) },
	{ U32(0x0ff0f000), U32(0x03500000), U16(1), U16(0x225f) },
	{ U32(0x0ff0f090), U32(0x01500010), U16(1), U16(0x2270) },
	{ U32(0x0ff0f010), U32(0x01500000), U16(1), U16(0x2285) },
	{ U32(0x0ffffff0), U32(0x0320f0f0), U16(1), U16(0x2299) },
	{ U32(0x0fe00000), U32(0x02200000), U16(1), U16(0x22a9) },
	{ U32(0x0fe00090), U32(0x00200010), U16(1), U16(0x22be) },
	{ U32(0x0fe00010), U32(0x00200000), U16(1), U16(0x22d3) },
	{ U32(0xfe100000), U32(0xfc100000), U16(3), U16(0x0c09) },
	{ U32(0xfe100000), U32(0xec100000), U16(3), U16(0x0c4e) },
	{ U32(0x0fd00000), U32(0x08900000), U16(2), U16(0x22e7) },
	{ U32(0x0fd00000), U32(0x08100000), U16(1), U16(0x230b) },
	{ U32(0x0fd00000), U32(0x09100000), U16(1), U16(0x231f) },
	{ U32(0x0fd00000), U32(0x09900000), U16(1), U16(0x2333) },
	{ U32(0x0f500000), U32(0x05100000), U16(3), U16(0x2347) },
	{ U32(0x0f500010), U32(0x07100000), U16(2), U16(0x2385) },
	{ U32(0x0f700010), U32(0x06100000), U16(1), U16(0x23b3) },
	{ U32(0x0f700000), U32(0x04100000), U16(2), U16(0x23cb) },
	{ U32(0x0f500010), U32(0x07500000), U16(2), U16(0x23f4) },
	{ U32(0x0f700010), U32(0x06500000), U16(2), U16(0x2424) },
	{ U32(0x0f700000), U32(0x04500000), U16(2), U16(0x2451) },
	{ U32(0x0f700000), U32(0x04700000), U16(1), U16(0x247e) },
	{ U32(0x0f700010), U32(0x06700000), U16(1), U16(0x2497) },
	{ U32(0x0f500ff0), U32(0x010000d0), U16(1), U16(0x24af) },
	{ U32(0x0f700ff0), U32(0x000000d0), U16(1), U16(0x24c7) },
	{ U32(0x0f5000f0), U32(0x014000d0), U16(2), U16(0x24e0) },
	{ U32(0x0f7000f0), U32(0x004000d0), U16(1), U16(0x250d) },
	{ U32(0x0ff00fff), U32(0x01900f9f), U16(1), U16(0x2526) },
	{ U32(0x0ff00fff), U32(0x01d00f9f), U16(1), U16(0x253b) },
	{ U32(0x0ff00fff), U32(0x01b00f9f), U16(1), U16(0x2553) },
	{ U32(0x0ff00fff), U32(0x01f00f9f), U16(1), U16(0x256c) },
	{ U32(0x0f5000f0), U32(0x015000b0), U16(2), U16(0x2584) },
	{ U32(0x0f500ff0), U32(0x011000b0), U16(1), U16(0x25b1) },
	{ U32(0x0f700ff0), U32(0x001000b0), U16(1), U16(0x25ca) },
	{ U32(0x0f7000f0), U32(0x005000b0), U16(1), U16(0x25e2) },
	{ U32(0x0f7000f0), U32(0x007000b0), U16(1), U16(0x25fa) },
	{ U32(0x0f700ff0), U32(0x003000b0), U16(1), U16(0x2613) },
	{ U32(0x0f5000f0), U32(0x015000d0), U16(2), U16(0x262c) },
	{ U32(0x0f500ff0), U32(0x011000d0), U16(1), U16(0x2659) },
	{ U32(0x0f700ff0), U32(0x001000d0), U16(1), U16(0x2671) },
	{ U32(0x0f7000f0), U32(0x005000d0), U16(1), U16(0x268a) },
	{ U32(0x0f7000f0), U32(0x007000d0), U16(1), U16(0x26a3) },
	{ U32(0x0f700ff0), U32(0x003000d0), U16(1), U16(0x26bb) },
	{ U32(0x0f5000f0), U32(0x015000f0), U16(2), U16(0x26d3) },
	{ U32(0x0f500ff0), U32(0x011000f0), U16(1), U16(0x2700) },
	{ U32(0x0f700ff0), U32(0x001000f0), U16(1), U16(0x2718) },
	{ U32(0x0f7000f0), U32(0x005000f0), U16(1), U16(0x2731) },
	{ U32(0x0f7000f0), U32(0x007000f0), U16(1), U16(0x274a) },
	{ U32(0x0f700ff0), U32(0x003000f0), U16(1), U16(0x2762) },
	{ U32(0x0f700000), U32(0x04300000), U16(1), U16(0x277a) },
	{ U32(0x0f700010), U32(0x06300000), U16(1), U16(0x2792) },
	{ U32(0x0fef0070), U32(0x01a00000), U16(2), U16(0x27ab) },
	{ U32(0x0fef00f0), U32(0x01a00010), U16(1), U16(0x27d4) },
	{ U32(0x0fef0070), U32(0x01a00020), U16(1), U16(0x27e9) },
	{ U32(0x0fef00f0), U32(0x01a00030), U16(1), U16(0x27fe) },
	{ U32(0xff100010), U32(0xfe000010), U16(1), U16(0x1170) },
	{ U32(0xff100010), U32(0xee000010), U16(1), U16(0x1188) },
	{ U32(0xfff00000), U32(0xfc400000), U16(1), U16(0x11a1) },
	{ U32(0xfff00000), U32(0xec400000), U16(1), U16(0x11b9) },
	{ U32(0x0fe000f0), U32(0x00200090), U16(1), U16(0x2813) },
	{ U32(0x0ff000f0), U32(0x00600090), U16(1), U16(0x2827) },
	{ U32(0x0ff00000), U32(0x03400000), U16(1), U16(0x283c) },
	{ U32(0x0ff00000), U32(0x03000000), U16(1), U16(0x2850) },
	{ U32(0x0fef0000), U32(0x03a00000), U16(1), U16(0x2864) },
	{ U32(0xff100010), U32(0xfe100010), U16(1), U16(0x125e) },
	{ U32(0xff100010), U32(0xee100010), U16(1), U16(0x1276) },
	{ U32(0x0fff0fff), U32(0x010f0000), U16(1), U16(0x2878) },
	{ U32(0x0fff0fff), U32(0x014f0000), U16(1), U16(0x288c) },
	{ U32(0x0ffff000), U32(0x0328f000), U16(1), U16(0x28a0) },
	{ U32(0x0ffff000), U32(0x0324f000), U16(1), U16(0x28b8) },
	{ U32(0x0ffff000), U32(0x032cf000), U16(1), U16(0x28d0) },
	{ U32(0x0ffffff0), U32(0x0128f000), U16(1), U16(0x28e9) },
	{ U32(0x0ffffff0), U32(0x0124f000), U16(1), U16(0x2901) },
	{ U32(0x0ffffff0), U32(0x012cf000), U16(1), U16(0x2919) },
	{ U32(0x0fe0f0f0), U32(0x00000090), U16(1), U16(0x2932) },
	{ U32(0x0fef0000), U32(0x03e00000), U16(1), U16(0x2947) },
	{ U32(0x0fef0090), U32(0x01e00010), U16(1), U16(0x295b) },
	{ U32(0x0fef0010), U32(0x01e00000), U16(1), U16(0x296f) },
	{ U32(0x0fffffff), U32(0x0320f000), U16(1), U16(0x2984) },
	{ U32(0x0fe00000), U32(0x03800000), U16(1), U16(0x2994) },
	{ U32(0x0fe00090), U32(0x01800010), U16(1), U16(0x29a9) },
	{ U32(0x0fe00010), U32(0x01800000), U16(1), U16(0x29be) },
	{ U32(0x0ff00030), U32(0x06800010), U16(1), U16(0x29d2) },
	{ U32(0x0f500000), U32(0x05000000), U16(2), U16(0x29ee) },
	{ U32(0x0fd00000), U32(0x09000000), U16(2), U16(0x2a1b) },
	{ U32(0x0ff00ff0), U32(0x06200f10), U16(1), U16(0x2a40) },
	{ U32(0x0ff00ff0), U32(0x06200f90), U16(1), U16(0x2a55) },
	{ U32(0x0ff00ff0), U32(0x01000050), U16(1), U16(0x2a69) },
	{ U32(0x0ff00ff0), U32(0x06200f30), U16(1), U16(0x2a7e) },
	{ U32(0x0ff00ff0), U32(0x01400050), U16(1), U16(0x2a93) },
	{ U32(0x0ff00ff0), U32(0x01600050), U16(1), U16(0x2aa7) },
	{ U32(0x0ff00ff0), U32(0x06200f50), U16(1), U16(0x2abb) },
	{ U32(0x0ff00ff0), U32(0x06200f70), U16(1), U16(0x2ad0) },
	{ U32(0x0ff00ff0), U32(0x06200ff0), U16(1), U16(0x2ae5) },
	{ U32(0x0ff00ff0), U32(0x01200050), U16(1), U16(0x2af9) },
	{ U32(0x0fff0ff0), U32(0x06ff0f30), U16(1), U16(0x2b0e) },
	{ U32(0x0fff0ff0), U32(0x06bf0fb0), U16(1), U16(0x2b22) },
	{ U32(0x0fff0ff0), U32(0x06bf0f30), U16(1), U16(0x2b37) },
	{ U32(0x0fff0ff0), U32(0x06ff0fb0), U16(1), U16(0x2b48) },
	{ U32(0x0fef0070), U32(0x01a00060), U16(2), U16(0x2b5d) },
	{ U32(0x0fef00f0), U32(0x01a00070), U16(1), U16(0x2b86) },
	{ U32(0x0fe00000), U32(0x02600000), U16(1), U16(0x2b9b) },
	{ U32(0x0fe00090), U32(0x00600010), U16(1), U16(0x2bb0) },
	{ U32(0x0fe00010), U32(0x00600000), U16(1), U16(0x2bc5) },
	{ U32(0x0fe00000), U32(0x02e00000), U16(1), U16(0x2bd9) },
	{ U32(0x0fe00090), U32(0x00e00010), U16(1), U16(0x2bee) },
	{ U32(0x0fe00010), U32(0x00e00000), U16(1), U16(0x2c03) },
	{ U32(0x0ff00ff0), U32(0x06100f10), U16(1), U16(0x2c17) },
	{ U32(0x0ff00ff0), U32(0x06100f90), U16(1), U16(0x2c2c) },
	{ U32(0x0ff00ff0), U32(0x06100f30), U16(1), U16(0x2c40) },
	{ U32(0x0fe00000), U32(0x02c00000), U16(1), U16(0x2c55) },
	{ U32(0x0fe00090), U32(0x00c00010), U16(1), U16(0x2c6a) },
	{ U32(0x0fe00010), U32(0x00c00000), U16(1), U16(0x2c7f) },
	{ U32(0x0fe00070), U32(0x07a00050), U16(1), U16(0x2c93) },
	{ U32(0x0ff00ff0), U32(0x06800fb0), U16(1), U16(0x2cab) },
	{ U32(0xffffffff), U32(0xf1010000), U16(1), U16(0x2cbf) },
	{ U32(0xffffffff), U32(0xf1010200), U16(1), U16(0x2cd4) },
	{ U32(0x0fffffff), U32(0x0320f004), U16(1), U16(0x2ce9) },
	{ U32(0x0ff00ff0), U32(0x06300f10), U16(1), U16(0x2cf9) },
	{ U32(0x0ff00ff0), U32(0x06300f90), U16(1), U16(0x2d11) },
	{ U32(0x0ff00ff0), U32(0x06300f30), U16(1), U16(0x2d26) },
	{ U32(0x0ff00ff0), U32(0x06300f50), U16(1), U16(0x2d3a) },
	{ U32(0x0ff00ff0), U32(0x06300f70), U16(1), U16(0x2d4e) },
	{ U32(0x0ff00ff0), U32(0x06300ff0), U16(1), U16(0x2d66) },
	{ U32(0x0ff000f0), U32(0x010000e0), U16(1), U16(0x2d7b) },
	{ U32(0x0ff000f0), U32(0x010000c0), U16(1), U16(0x2d93) },
	{ U32(0x0ff000f0), U32(0x010000a0), U16(1), U16(0x2dab) },
	{ U32(0x0ff000f0), U32(0x01000080), U16(1), U16(0x2dc3) },
	{ U32(0x0ff000f0), U32(0x07000030), U16(2), U16(0x2ddb) },
	{ U32(0x0ff000f0), U32(0x07000010), U16(2), U16(0x2e08) },
	{ U32(0x0ff000f0), U32(0x014000e0), U16(1), U16(0x2e31) },
	{ U32(0x0ff000f0), U32(0x014000c0), U16(1), U16(0x2e4a) },
	{ U32(0x0ff000f0), U32(0x014000a0), U16(1), U16(0x2e63) },
	{ U32(0x0ff000f0), U32(0x01400080), U16(1), U16(0x2e7c) },
	{ U32(0x0ff000f0), U32(0x07400030), U16(1), U16(0x2e95) },
	{ U32(0x0ff000f0), U32(0x07400010), U16(1), U16(0x2eae) },
	{ U32(0x0fe000f0), U32(0x00e00090), U16(1), U16(0x2ec6) },
	{ U32(0x0ff000f0), U32(0x012000c0), U16(1), U16(0x2ede) },
	{ U32(0x0ff000f0), U32(0x01200080), U16(1), U16(0x2ef6) },
	{ U32(0x0ff000f0), U32(0x07000070), U16(2), U16(0x2f0e) },
	{ U32(0x0ff000f0), U32(0x07000050), U16(2), U16(0x2f3b) },
	{ U32(0x0ff000f0), U32(0x07400070), U16(1), U16(0x2f67) },
	{ U32(0x0ff000f0), U32(0x07400050), U16(1), U16(0x2f80) },
	{ U32(0x0ff000f0), U32(0x07500030), U16(2), U16(0x2f99) },
	{ U32(0x0ff000f0), U32(0x07500010), U16(2), U16(0x2fc6) },
	{ U32(0x0ff000f0), U32(0x075000f0), U16(1), U16(0x2fef) },
	{ U32(0x0ff000f0), U32(0x075000d0), U16(1), U16(0x3007) },
	{ U32(0x0ff000f0), U32(0x016000e0), U16(1), U16(0x301c) },
	{ U32(0x0ff000f0), U32(0x016000c0), U16(1), U16(0x3031) },
	{ U32(0x0ff000f0), U32(0x016000a0), U16(1), U16(0x3046) },
	{ U32(0x0ff000f0), U32(0x01600080), U16(1), U16(0x305b) },
	{ U32(0x0fe000f0), U32(0x00c00090), U16(1), U16(0x3070) },
	{ U32(0x0ff000f0), U32(0x012000e0), U16(1), U16(0x3088) },
	{ U32(0x0ff000f0), U32(0x012000a0), U16(1), U16(0x309d) },
	{ U32(0x0ff00ff0), U32(0x06a00f30), U16(1), U16(0x30b2) },
	{ U32(0x0fe00030), U32(0x06a00010), U16(1), U16(0x30c7) },
	{ U32(0x0ff00ff0), U32(0x06100f50), U16(1), U16(0x30db) },
	{ U32(0x0ff00ff0), U32(0x06100f70), U16(1), U16(0x30f0) },
	{ U32(0x0ff00ff0), U32(0x06100ff0), U16(1), U16(0x3105) },
	{ U32(0xff500000), U32(0xfd000000), U16(1), U16(0x1983) },
	{ U32(0xff700000), U32(0xfc200000), U16(1), U16(0x199b) },
	{ U32(0xff500000), U32(0xfd400000), U16(1), U16(0x19b4) },
	{ U32(0xff700000), U32(0xfc600000), U16(1), U16(0x19cd) },
	{ U32(0xff500000), U32(0xed000000), U16(1), U16(0x19e5) },
	{ U32(0xff700000), U32(0xec200000), U16(1), U16(0x19fe) },
	{ U32(0xff500000), U32(0xed400000), U16(1), U16(0x1a16) },
	{ U32(0xff700000), U32(0xec600000), U16(1), U16(0x1a2e) },
	{ U32(0x0fd00000), U32(0x08800000), U16(1), U16(0x3119) },
	{ U32(0x0fd00000), U32(0x08000000), U16(1), U16(0x312d) },
	{ U32(0x0fd00000), U32(0x09800000), U16(1), U16(0x3141) },
	{ U32(0x0f500010), U32(0x07000000), U16(1), U16(0x3155) },
	{ U32(0x0f700010), U32(0x06000000), U16(1), U16(0x316e) },
	{ U32(0x0f700000), U32(0x04000000), U16(1), U16(0x3186) },
	{ U32(0x0f500010), U32(0x07400000), U16(1), U16(0x319b) },
	{ U32(0x0f700010), U32(0x06400000), U16(1), U16(0x31b3) },
	{ U32(0x0f500000), U32(0x05400000), U16(1), U16(0x31cc) },
	{ U32(0x0f700000), U32(0x04400000), U16(1), U16(0x31e5) },
	{ U32(0x0f700000), U32(0x04600000), U16(1), U16(0x31fd) },
	{ U32(0x0f700010), U32(0x06600000), U16(1), U16(0x3216) },
	{ U32(0x0f500ff0), U32(0x010000f0), U16(1), U16(0x322e) },
	{ U32(0x0f700ff0), U32(0x000000f0), U16(1), U16(0x3246) },
	{ U32(0x0f5000f0), U32(0x014000f0), U16(1), U16(0x325f) },
	{ U32(0x0f7000f0), U32(0x004000f0), U16(1), U16(0x3277) },
	{ U32(0x0ff00ff0), U32(0x01800f90), U16(1), U16(0x3290) },
	{ U32(0x0ff00ff0), U32(0x01c00f90), U16(1), U16(0x32a8) },
	{ U32(0x0ff00ff0), U32(0x01a00f90), U16(1), U16(0x32c1) },
	{ U32(0x0ff00ff0), U32(0x01e00f90), U16(1), U16(0x32d9) },
	{ U32(0x0f500ff0), U32(0x010000b0), U16(1), U16(0x32f2) },
	{ U32(0x0f700ff0), U32(0x000000b0), U16(1), U16(0x330b) },
	{ U32(0x0f5000f0), U32(0x014000b0), U16(1), U16(0x3323) },
	{ U32(0x0f7000f0), U32(0x004000b0), U16(1), U16(0x333c) },
	{ U32(0x0f7000f0), U32(0x006000b0), U16(1), U16(0x3354) },
	{ U32(0x0f700ff0), U32(0x002000b0), U16(1), U16(0x336d) },
	{ U32(0x0f700000), U32(0x04200000), U16(1), U16(0x3386) },
	{ U32(0x0f700010), U32(0x06200000), U16(1), U16(0x339e) },
	{ U32(0x0fe00090), U32(0x00400010), U16(1), U16(0x33b7) },
	{ U32(0x0fe00010), U32(0x00400000), U16(1), U16(0x33cc) },
	{ U32(0x0f000000), U32(0x0f000000), U16(1), U16(0x33e0) },
	{ U32(0x0ff00ff0), U32(0x01000090), U16(1), U16(0x33f0) },
	{ U32(0x0ff00ff0), U32(0x01400090), U16(1), U16(0x3404) },
	{ U32(0x0ff003f0), U32(0x06800070), U16(2), U16(0x3419) },
	{ U32(0x0ff003f0), U32(0x06a00070), U16(2), U16(0x3447) },
	{ U32(0x0ff003f0), U32(0x06b00070), U16(2), U16(0x3471) },
	{ U32(0x0ff0f000), U32(0x03300000), U16(1), U16(0x349b) },
	{ U32(0x0ff0f090), U32(0x01300010), U16(1), U16(0x34ac) },
	{ U32(0x0ff0f010), U32(0x01300000), U16(1), U16(0x34c1) },
	{ U32(0x0ff0f000), U32(0x03100000), U16(1), U16(0x34d5) },
	{ U32(0x0ff0f090), U32(0x01100010), U16(1), U16(0x34e6) },
	{ U32(0x0ff0f010), U32(0x01100000), U16(1), U16(0x34fb) },
	{ U32(0x0ff00ff0), U32(0x06500f10), U16(1), U16(0x350f) },
	{ U32(0x0ff00ff0), U32(0x06500f90), U16(1), U16(0x3524) },
	{ U32(0x0ff00ff0), U32(0x06500f30), U16(1), U16(0x3538) },
	{ U32(0x0fe00070), U32(0x07e00050), U16(1), U16(0x354d) },
	{ U32(0x0ff00ff0), U32(0x06700f10), U16(1), U16(0x3565) },
	{ U32(0x0ff00ff0), U32(0x06700f90), U16(1), U16(0x357d) },
	{ U32(0x0ff00ff0), U32(0x06700f30), U16(1), U16(0x3592) },
	{ U32(0x0ff00ff0), U32(0x06700f50), U16(1), U16(0x35a6) },
	{ U32(0x0ff00ff0), U32(0x06700f70), U16(1), U16(0x35ba) },
	{ U32(0x0ff00ff0), U32(0x06700ff0), U16(1), U16(0x35d2) },
	{ U32(0x0ff000f0), U32(0x00400090), U16(1), U16(0x35e7) },
	{ U32(0x0fe000f0), U32(0x00a00090), U16(1), U16(0x35fc) },
	{ U32(0x0fe000f0), U32(0x00800090), U16(1), U16(0x3614) },
	{ U32(0x0ff00ff0), U32(0x06600f10), U16(1), U16(0x362c) },
	{ U32(0x0ff00ff0), U32(0x06600f90), U16(1), U16(0x3644) },
	{ U32(0x0ff00ff0), U32(0x06600f30), U16(1), U16(0x3659) },
	{ U32(0x0ff00ff0), U32(0x06600f50), U16(1), U16(0x366d) },
	{ U32(0x0ff00ff0), U32(0x06600f70), U16(1), U16(0x3681) },
	{ U32(0x0ff00ff0), U32(0x06600ff0), U16(1), U16(0x3699) },
	{ U32(0x0ff000f0), U32(0x07800010), U16(2), U16(0x36ae) },
	{ U32(0x0ff00ff0), U32(0x06e00f30), U16(1), U16(0x36da) },
	{ U32(0x0fe00030), U32(0x06e00010), U16(1), U16(0x36ef) },
	{ U32(0x0ff00ff0), U32(0x06500f50), U16(1), U16(0x3704) },
	{ U32(0x0ff00ff0), U32(0x06500f70), U16(1), U16(0x3719) },
	{ U32(0x0ff00ff0), U32(0x06500ff0), U16(1), U16(0x372e) },
	{ U32(0x0ff003f0), U32(0x06c00070), U16(2), U16(0x3742) },
	{ U32(0x0ff003f0), U32(0x06e00070), U16(2), U16(0x3770) },
	{ U32(0x0ff003f0), U32(0x06f00070), U16(2), U16(0x379a) },
	{ U32(0x0fffffff), U32(0x0320f002), U16(1), U16(0x37c4) },
	{ U32(0x0fffffff), U32(0x0320f003), U16(1), U16(0x37d4) },
	{ U32(0x0fffffff), U32(0x0320f001), U16(1), U16(0x37e4) },
}; /* a32_instruction_groups */

/* Instruction matching flags */
#define F_IT0 1
#define F_IT1 2
#define ISFLAG(b) ((b) < 3)

/* Instruction suffix flags */
#define S_L 1
#define S_S 2
#define S_TODO_x 3
#define S_TODO_y 4
#define S_TODO_z 5
#define S_c 6
#define S_dW 7

/* Instruction operand encodings */
#define O_ARMEI_C_S0N12 1
#define O_ARMEI_REL_S0N12 2
#define O_ARMEI_S0N12 3
#define O_DIS10_S6N5 4
#define O_IMM00_REL_S0N8 5
#define O_IMM00_S0N7 6
#define O_IMM00_S0N8 7
#define O_IMM0_REL_X_XXXXX___ 8
#define O_IMMP1_S0N4 9
#define O_IMMP1_S16N4 10
#define O_IMM_S0N12 11
#define O_IMM_S0N24 12
#define O_IMM_S0N4 13
#define O_IMM_S0N7 14
#define O_IMM_S0N8 15
#define O_IMM_S16N4 16
#define O_IMM_S6N3 17
#define O_IMM_S6N5 18
#define O_IMM_S7N5 19
#define O_IMM_XXXXXXXXXXXX____XXXX 20
#define O_IMM_XXXX____XXXX 21
#define O_IMM_XXXX____XXXXXXXXXXXX 22
#define O_IMM_XXX____XX______ 23
#define O_IMM_X______XXXX_XXX____XXXXXXXX 24
#define O_IMM_X___________XXX____XXXXXXXX 25
#define O_MB_OPTION_S0N4 26
#define O_OPT_IA 27
#define O_OPT_ROTATION_S10N2 28
#define O_OPT_ROTATION_S4N2 29
#define O_OPT_SHIFT_ 30
#define O_OPT_SHIFT_S21N1 31
#define O_OPT_SHIFT_S4N2 32
#define O_OPT_SHIFT_S5N2 127
#define O_OPT_SHIFT_S6N1 128
#define O_PLUSMINUS_S23N1 129
#define O_PLUSMINUS_S9N1 130
#define O_RLIST_L_S0N8 131
#define O_RLIST_P_S0N13 132
#define O_RLIST_P_S0N8 133
#define O_RLIST_S0N13 134
#define O_RLIST_S0N16 135
#define O_RLIST_S0N8 136
#define O_RP1_S0N4 137
#define O_RP1_S12N4 138
#define O_R_S0N3 139
#define O_R_S0N4 140
#define O_R_S12N4 141
#define O_R_S16N4 142
#define O_R_S3N3 143
#define O_R_S3N4 144
#define O_R_S6N3 145
#define O_R_S8N3 146
#define O_R_S8N4 147
#define O_R_X____XXX 148
#define O_SIMM00_REL_S0N24 149
#define O_SIMM00_SJ2J1_REL_XXXXXXXXXX_____XXXXXXXXXX_ 150
#define O_SIMM0_H_REL_S0N25 151
#define O_SIMM0_REL_S0N11 152
#define O_SIMM0_REL_S0N8 153
#define O_SIMM0_SJ2J1_REL_XXXXXXXXXX_____XXXXXXXXXXX 154
#define O_SIMM0_SJ2J1_REL_XXXXXX_____XXXXXXXXXXX 155
#define O_THUMBEI_C_X___________XXX____XXXXXXXX 156
#define O_THUMBEI_X___________XXX____XXXXXXXX 157
#define O_TODO(...) 158

PRIVATE byte_t const arm_instruction_db[14324] = {
	/* adcs    */ P32(0x0000ffc0), P32(0x00004140), F_IT0, 'a', 'd', 'c', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* adc     */ P32(0x0000ffc0), P32(0x00004140), F_IT1, 'a', 'd', 'c', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* add     */ P32(0x0000f800), P32(0x0000a800), 'a', 'd', 'd', S_c, '\t', O_R_S8N3, 'S', 'P', O_IMM00_S0N8, 0, 0, 0, 0,
	/* add     */ P32(0x0000ff00), P32(0x00004400), 'a', 'd', 'd', S_c, '\t', O_R_X____XXX, O_R_S3N4, 0, 0,
	/* add     */ P32(0x0000ff80), P32(0x0000b000), 'a', 'd', 'd', S_c, '\t', 'S', 'P', 'S', 'P', O_IMM00_S0N7, 0, 0,
	/* adds    */ P32(0x0000fe00), P32(0x00001c00), F_IT0, 'a', 'd', 'd', 's', '\t', O_R_S0N3, O_R_S3N3, O_IMM_S6N3, 0, 0, 0, 0,
	/* add     */ P32(0x0000fe00), P32(0x00001c00), F_IT1, 'a', 'd', 'd', S_c, '\t', O_R_S0N3, O_R_S3N3, O_IMM_S6N3, 0, 0, 0, 0,
	/* adds    */ P32(0x0000fe00), P32(0x00001800), F_IT0, 'a', 'd', 'd', 's', '\t', O_R_S0N3, O_R_S3N3, O_R_S6N3, 0, 0, 0, 0,
	/* add     */ P32(0x0000fe00), P32(0x00001800), F_IT1, 'a', 'd', 'd', S_c, '\t', O_R_S0N3, O_R_S3N3, O_R_S6N3, 0, 0, 0, 0,
	/* adds    */ P32(0x0000f800), P32(0x00003000), F_IT0, 'a', 'd', 'd', 's', '\t', O_R_S8N3, O_IMM_S0N8, 0, 0, 0, 0,
	/* add     */ P32(0x0000f800), P32(0x00003000), F_IT1, 'a', 'd', 'd', S_c, '\t', O_R_S8N3, O_IMM_S0N8, 0, 0, 0, 0,
	/* adr     */ P32(0x0000f800), P32(0x0000a000), 'a', 'd', 'r', S_c, '\t', O_R_S8N3, O_IMM00_REL_S0N8, 0, 0,
	/* ands    */ P32(0x0000ffc0), P32(0x00004000), F_IT0, 'a', 'n', 'd', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* and     */ P32(0x0000ffc0), P32(0x00004000), F_IT1, 'a', 'n', 'd', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* asrs    */ P32(0x0000ffc0), P32(0x00004100), F_IT0, 'a', 's', 'r', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* asr     */ P32(0x0000ffc0), P32(0x00004100), F_IT1, 'a', 's', 'r', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* asrs    */ P32(0x0000f800), P32(0x00001000), 'a', 's', 'r', 's', '\t', O_R_S0N3, O_R_S3N3, '#', O_DIS10_S6N5, 0, 0, 0, 0,
	/* svc     */ P32(0x0000ff00), P32(0x0000df00), 's', 'v', 'c', S_c, '\t', O_IMM_S0N8, 0, 0,
	/* b       */ P32(0x0000f000), P32(0x0000d000), 'b', S_c, '\t', O_SIMM0_REL_S0N8, 0, 0, 0, 0,
	/* b       */ P32(0x0000f800), P32(0x0000e000), 'b', S_c, '\t', O_SIMM0_REL_S0N11, 0, 0, 0, 0,
	/* bics    */ P32(0x0000ffc0), P32(0x00004380), F_IT0, 'b', 'i', 'c', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* bic     */ P32(0x0000ffc0), P32(0x00004380), F_IT1, 'b', 'i', 'c', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* bkpt    */ P32(0x0000ff00), P32(0x0000be00), 'b', 'k', 'p', 't', '\t', O_IMM_S0N8, 0, 0,
	/* blx     */ P32(0x0000ff87), P32(0x00004780), 'b', 'l', 'x', S_c, '\t', O_R_S3N4, 0, 0,
	/* bx      */ P32(0x0000ff87), P32(0x00004700), 'b', 'x', S_c, '\t', O_R_S3N4, 0, 0, 0, 0,
	/* cbz     */ P32(0x0000fd00), P32(0x0000b100), 'c', 'b', 'z', '\t', O_R_S0N3, O_IMM0_REL_X_XXXXX___, 0, 0,
	/* cbnz    */ P32(0x0000fd00), P32(0x0000b900), 'c', 'b', 'n', 'z', '\t', O_R_S0N3, O_IMM0_REL_X_XXXXX___, 0, 0,
	/* cmn     */ P32(0x0000ffc0), P32(0x000042c0), 'c', 'm', 'n', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0,
	/* cmp     */ P32(0x0000f800), P32(0x00002800), 'c', 'm', 'p', S_c, '\t', O_R_S8N3, O_IMM_S0N8, 0, 0,
	/* cmp     */ P32(0x0000ffc0), P32(0x00004280), 'c', 'm', 'p', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0,
	/* cmp     */ P32(0x0000ff00), P32(0x00004500), 'c', 'm', 'p', S_c, '\t', O_R_X____XXX, O_R_S3N4, 0, 0,
	/* eors    */ P32(0x0000ffc0), P32(0x00004040), F_IT0, 'e', 'o', 'r', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* eor     */ P32(0x0000ffc0), P32(0x00004040), F_IT1, 'e', 'o', 'r', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* nop     */ P32(0x0000ffff), P32(0x0000bf00), 'n', 'o', 'p', S_c, 0, 0, 0, 0,
	/* sev     */ P32(0x0000ffff), P32(0x0000bf40), 's', 'e', 'v', S_c, 0, 0, 0, 0,
	/* wfe     */ P32(0x0000ffff), P32(0x0000bf20), 'w', 'f', 'e', S_c, 0, 0, 0, 0,
	/* wfi     */ P32(0x0000ffff), P32(0x0000bf30), 'w', 'f', 'i', S_c, 0, 0, 0, 0,
	/* yield   */ P32(0x0000ffff), P32(0x0000bf10), 'y', 'i', 'e', 'l', 'd', S_c, 0, 0,
	/* it      */ P32(0x0000ff00), P32(0x0000bf00), 'i', 't', S_TODO_x, S_TODO_y, S_TODO_z, '\t', O_TODO("firstcond"), 0, 0,
	/* ldm     */ P32(0x0000f800), P32(0x0000c800), 'l', 'd', 'm', S_c, '\t', O_R_S8N3, '!', O_RLIST_S0N8, 0, 0, 0, 0,
	/* ldr     */ P32(0x0000f800), P32(0x00006800), 'l', 'd', 'r', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_IMM_S6N5, ']', 0, 0,
	/* ldr     */ P32(0x0000f800), P32(0x00004800), 'l', 'd', 'r', S_c, '\t', O_R_S8N3, O_IMM00_REL_S0N8, 0, 0,
	/* ldr     */ P32(0x0000fe00), P32(0x00005800), 'l', 'd', 'r', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_R_S6N3, ']', 0, 0,
	/* ldr     */ P32(0x0000f800), P32(0x00009800), 'l', 'd', 'r', S_c, '\t', O_R_S8N3, '[', 'S', 'P', O_IMM_S0N8, ']', 0, 0,
	/* ldrb    */ P32(0x0000fe00), P32(0x00005c00), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_R_S6N3, ']', 0, 0,
	/* ldrb    */ P32(0x0000f800), P32(0x00007800), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_IMM_S6N5, ']', 0, 0,
	/* ldrh    */ P32(0x0000fe00), P32(0x00005a00), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_R_S6N3, ']', 0, 0,
	/* ldrh    */ P32(0x0000f800), P32(0x00008800), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_IMM_S6N5, ']', 0, 0,
	/* ldrsb   */ P32(0x0000fe00), P32(0x00005600), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_R_S6N3, ']', 0, 0, 0, 0,
	/* ldrsh   */ P32(0x0000fe00), P32(0x00005e00), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_R_S6N3, ']', 0, 0, 0, 0,
	/* lsls    */ P32(0x0000ffc0), P32(0x00004080), F_IT0, 'l', 's', 'l', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* lsl     */ P32(0x0000ffc0), P32(0x00004080), F_IT1, 'l', 's', 'l', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* movs    */ P32(0x0000ffc0), P32(0x00000000), F_IT0, 'm', 'o', 'v', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* lsls    */ P32(0x0000f800), P32(0x00000000), F_IT0, 'l', 's', 'l', 's', '\t', O_R_S0N3, O_R_S3N3, O_IMM_S6N5, 0, 0, 0, 0,
	/* lsl     */ P32(0x0000f800), P32(0x00000000), F_IT1, 'l', 's', 'l', '\t', O_R_S0N3, O_R_S3N3, O_IMM_S6N5, 0, 0, 0, 0,
	/* lsrs    */ P32(0x0000ffc0), P32(0x000040c0), F_IT0, 'l', 's', 'r', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* lsr     */ P32(0x0000ffc0), P32(0x000040c0), F_IT1, 'l', 's', 'r', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* lsrs    */ P32(0x0000f800), P32(0x00000800), F_IT0, 'l', 's', 'r', 's', '\t', O_R_S0N3, O_R_S3N3, O_IMM_S6N5, 0, 0, 0, 0,
	/* lsr     */ P32(0x0000f800), P32(0x00000800), F_IT1, 'l', 's', 'r', S_c, '\t', O_R_S0N3, O_R_S3N3, O_IMM_S6N5, 0, 0, 0, 0,
	/* mov     */ P32(0x0000ff00), P32(0x00004600), 'm', 'o', 'v', S_c, '\t', O_R_X____XXX, O_R_S3N4, 0, 0,
	/* movs    */ P32(0x0000f800), P32(0x00002000), F_IT0, 'm', 'o', 'v', 's', '\t', O_R_S8N3, O_IMM_S0N8, 0, 0, 0, 0,
	/* mov     */ P32(0x0000f800), P32(0x00002000), F_IT1, 'm', 'o', 'v', S_c, '\t', O_R_S8N3, O_IMM_S0N8, 0, 0, 0, 0,
	/* muls    */ P32(0x0000ffc0), P32(0x00004340), F_IT0, 'm', 'u', 'l', 's', '\t', O_R_S0N3, O_R_S3N3, O_R_S0N3, 0, 0, 0, 0,
	/* mul     */ P32(0x0000ffc0), P32(0x00004340), F_IT1, 'm', 'u', 'l', S_c, '\t', O_R_S0N3, O_R_S3N3, O_R_S0N3, 0, 0, 0, 0,
	/* mvns    */ P32(0x0000ffc0), P32(0x000043c0), F_IT0, 'm', 'v', 'n', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* mvn     */ P32(0x0000ffc0), P32(0x000043c0), F_IT1, 'm', 'v', 'n', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* orrs    */ P32(0x0000ffc0), P32(0x00004300), F_IT0, 'o', 'r', 'r', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* orr     */ P32(0x0000ffc0), P32(0x00004300), F_IT1, 'o', 'r', 'r', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* pop     */ P32(0x0000fe00), P32(0x0000bc00), 'p', 'o', 'p', S_c, '\t', O_RLIST_P_S0N8, 0, 0,
	/* push    */ P32(0x0000fe00), P32(0x0000b400), 'p', 'u', 's', 'h', S_c, '\t', O_RLIST_L_S0N8, 0, 0,
	/* rev16   */ P32(0x0000ffc0), P32(0x0000ba40), 'r', 'e', 'v', '1', '6', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* rev     */ P32(0x0000ffc0), P32(0x0000ba00), 'r', 'e', 'v', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0,
	/* revsh   */ P32(0x0000ffc0), P32(0x0000bac0), 'r', 'e', 'v', 's', 'h', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* rors    */ P32(0x0000ffc0), P32(0x000041c0), F_IT0, 'r', 'o', 'r', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* ror     */ P32(0x0000ffc0), P32(0x000041c0), F_IT1, 'r', 'o', 'r', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* rsbs    */ P32(0x0000ffc0), P32(0x00004240), F_IT0, 'r', 's', 'b', 's', '\t', O_R_S0N3, O_R_S3N3, '#', '0', 0, 0,
	/* rsb     */ P32(0x0000ffc0), P32(0x00004240), F_IT1, 'r', 's', 'b', S_c, '\t', O_R_S0N3, O_R_S3N3, '#', '0', 0, 0,
	/* sbcs    */ P32(0x0000ffc0), P32(0x00004180), F_IT0, 's', 'b', 'c', 's', '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* sbc     */ P32(0x0000ffc0), P32(0x00004180), F_IT1, 's', 'b', 'c', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* setend  */ P32(0x0000ffff), P32(0x0000b650), 's', 'e', 't', 'e', 'n', 'd', '\t', 'L', 'E', 0, 0, 0, 0,
	/* setend  */ P32(0x0000ffff), P32(0x0000b658), 's', 'e', 't', 'e', 'n', 'd', '\t', 'B', 'E', 0, 0, 0, 0,
	/* stm     */ P32(0x0000f800), P32(0x0000c000), 's', 't', 'm', S_c, '\t', O_R_S8N3, '!', O_RLIST_S0N8, 0, 0, 0, 0,
	/* str     */ P32(0x0000f800), P32(0x00006000), 's', 't', 'r', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_IMM_S6N5, ']', 0, 0,
	/* str     */ P32(0x0000fe00), P32(0x00005000), 's', 't', 'r', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_R_S6N3, ']', 0, 0,
	/* str     */ P32(0x0000f800), P32(0x00009000), 's', 't', 'r', S_c, '\t', O_R_S8N3, '[', 'S', 'P', O_IMM_S0N8, ']', 0, 0,
	/* strb    */ P32(0x0000f800), P32(0x00007000), 's', 't', 'r', 'b', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_IMM_S6N5, ']', 0, 0,
	/* strb    */ P32(0x0000fe00), P32(0x00005400), 's', 't', 'r', 'b', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_R_S6N3, ']', 0, 0,
	/* strh    */ P32(0x0000fe00), P32(0x00005200), 's', 't', 'r', 'h', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_R_S6N3, ']', 0, 0,
	/* strh    */ P32(0x0000f800), P32(0x00008000), 's', 't', 'r', 'h', S_c, '\t', O_R_S0N3, '[', O_R_S3N3, O_IMM_S6N5, ']', 0, 0,
	/* sub     */ P32(0x0000ff80), P32(0x0000b080), 's', 'u', 'b', S_c, '\t', 'S', 'P', 'S', 'P', O_IMM_S0N7, 0, 0,
	/* subs    */ P32(0x0000fe00), P32(0x00001e00), F_IT0, 's', 'u', 'b', 's', '\t', O_R_S0N3, O_R_S3N3, O_IMM_S6N3, 0, 0, 0, 0,
	/* sub     */ P32(0x0000fe00), P32(0x00001e00), F_IT1, 's', 'u', 'b', S_c, '\t', O_R_S0N3, O_R_S3N3, O_IMM_S6N3, 0, 0, 0, 0,
	/* subs    */ P32(0x0000fe00), P32(0x00001a00), F_IT0, 's', 'u', 'b', 's', '\t', O_R_S0N3, O_R_S3N3, O_R_S6N3, 0, 0, 0, 0,
	/* sub     */ P32(0x0000fe00), P32(0x00001a00), F_IT1, 's', 'u', 'b', S_c, '\t', O_R_S0N3, O_R_S3N3, O_R_S6N3, 0, 0, 0, 0,
	/* subs    */ P32(0x0000f800), P32(0x00003800), F_IT0, 's', 'u', 'b', 's', '\t', O_R_S8N3, O_IMM_S0N8, 0, 0, 0, 0,
	/* sub     */ P32(0x0000f800), P32(0x00003800), F_IT1, 's', 'u', 'b', S_c, '\t', O_R_S8N3, O_IMM_S0N8, 0, 0, 0, 0,
	/* sxtb    */ P32(0x0000ffc0), P32(0x0000b240), 's', 'x', 't', 'b', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* sxth    */ P32(0x0000ffc0), P32(0x0000b200), 's', 'x', 't', 'h', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* tst     */ P32(0x0000ffc0), P32(0x00004200), 't', 's', 't', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0,
	/* uxtb    */ P32(0x0000ffc0), P32(0x0000b2c0), 'u', 'x', 't', 'b', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* uxth    */ P32(0x0000ffc0), P32(0x0000b280), 'u', 'x', 't', 'h', S_c, '\t', O_R_S0N3, O_R_S3N3, 0, 0, 0, 0,
	/* adc     */ P32(0xfbe08000), P32(0xf1400000), 'a', 'd', 'c', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* adc     */ P32(0xffe08000), P32(0xeb400000), 'a', 'd', 'c', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* adr     */ P32(0xfbff8000), P32(0xf20f0000), 'a', 'd', 'r', S_c, S_dW, '\t', O_R_S8N4, '+', O_IMM_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* addw    */ P32(0xfbf08000), P32(0xf2000000), 'a', 'd', 'd', 'w', S_c, '\t', O_R_S8N4, O_R_S16N4, O_IMM_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* cmn     */ P32(0xfbf08f00), P32(0xf1100f00), 'c', 'm', 'n', S_c, '\t', O_R_S16N4, O_THUMBEI_X___________XXX____XXXXXXXX, 0, 0,
	/* add     */ P32(0xfbe08000), P32(0xf1000000), 'a', 'd', 'd', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_X___________XXX____XXXXXXXX, 0, 0,
	/* cmn     */ P32(0xfff08f00), P32(0xeb100f00), 'c', 'm', 'n', S_c, S_dW, '\t', O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0, 0, 0,
	/* add     */ P32(0xffe08000), P32(0xeb000000), 'a', 'd', 'd', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* adr     */ P32(0xfbff8000), P32(0xf2af0000), 'a', 'd', 'r', S_c, S_dW, '\t', O_R_S8N4, '-', O_IMM_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* subw    */ P32(0xfbf08000), P32(0xf2a00000), 's', 'u', 'b', 'w', S_c, '\t', O_R_S8N4, O_R_S16N4, O_IMM_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* tst     */ P32(0xfbf08f00), P32(0xf0100f00), 't', 's', 't', S_c, '\t', O_R_S16N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0,
	/* and     */ P32(0xfbe08000), P32(0xf0000000), 'a', 'n', 'd', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* tst     */ P32(0xfff08f00), P32(0xea100f00), 't', 's', 't', S_c, S_dW, 0, 0, 0, 0,
	/* and     */ P32(0xffe08000), P32(0xea000000), 'a', 'n', 'd', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* mov     */ P32(0xffeff0f0), P32(0xea4f0000), 'm', 'o', 'v', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, 0, 0, 0, 0,
	/* rrx     */ P32(0xffeff0f0), P32(0xea4f0030), 'r', 'r', 'x', S_S, S_c, '\t', O_R_S8N4, O_R_S0N4, 0, 0, 0, 0,
	/* asr     */ P32(0xffef8030), P32(0xea4f0020), 'a', 's', 'r', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, O_IMM_XXX____XX______, 0, 0,
	/* lsl     */ P32(0xffef8030), P32(0xea4f0000), 'l', 's', 'l', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, O_IMM_XXX____XX______, 0, 0,
	/* lsr     */ P32(0xffef8030), P32(0xea4f0010), 'l', 's', 'r', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, O_IMM_XXX____XX______, 0, 0,
	/* ror     */ P32(0xffef8030), P32(0xea4f0030), 'r', 'o', 'r', S_S, S_c, '\t', O_R_S8N4, O_R_S0N4, O_IMM_XXX____XX______, 0, 0, 0, 0,
	/* orr     */ P32(0xffe08000), P32(0xea400000), 'o', 'r', 'r', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* asr     */ P32(0xffe0f0f0), P32(0xfa40f000), 'a', 's', 'r', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* clrex   */ P32(0xffffffff), P32(0xf3bf8f2f), 'c', 'l', 'r', 'e', 'x', S_c, 0, 0,
	/* nop     */ P32(0xffffffff), P32(0xf3af8000), 'n', 'o', 'p', S_c, S_dW, 0, 0, 0, 0,
	/* sev     */ P32(0xffffffff), P32(0xf3af8004), 's', 'e', 'v', S_c, S_dW, 0, 0, 0, 0,
	/* wfe     */ P32(0xffffffff), P32(0xf3af8002), 'w', 'f', 'e', S_c, S_dW, 0, 0, 0, 0,
	/* wfi     */ P32(0xffffffff), P32(0xf3af8003), 'w', 'f', 'i', S_c, S_dW, 0, 0, 0, 0,
	/* yield   */ P32(0xffffffff), P32(0xf3af8001), 'y', 'i', 'e', 'l', 'd', S_c, S_dW, 0, 0,
	/* dbg     */ P32(0xfffffff0), P32(0xf3af80f0), 'd', 'b', 'g', S_c, '\t', O_IMM_S0N4, 0, 0,
	/* dmb     */ P32(0xfffffff0), P32(0xf3bf8f50), 'd', 'm', 'b', S_c, '\t', '#', O_MB_OPTION_S0N4, 0, 0,
	/* dsb     */ P32(0xfffffff0), P32(0xf3bf8f40), 'd', 's', 'b', S_c, '\t', '#', O_MB_OPTION_S0N4, 0, 0,
	/* isb     */ P32(0xfffffff0), P32(0xf3bf8f60), 'i', 's', 'b', S_c, '\t', '#', O_MB_OPTION_S0N4, 0, 0,
	/* mrs     */ P32(0xfffff0ff), P32(0xf3ef8000), 'm', 'r', 's', S_c, '\t', O_R_S8N4, 'C', 'P', 'S', 'R', 0, 0,
	/* mrs     */ P32(0xfffff0ff), P32(0xf3ff8000), 'm', 'r', 's', S_c, '\t', O_R_S8N4, 'S', 'P', 'S', 'R', 0, 0,
	/* bxj     */ P32(0xfff0ffff), P32(0xf3c08f00), 'b', 'x', 'j', S_c, '\t', O_R_S16N4, 0, 0,
	/* msr     */ P32(0xfff0ffff), P32(0xf3808800), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 'f', O_R_S16N4, 0, 0, 0, 0,
	/* msr     */ P32(0xfff0ffff), P32(0xf3808400), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 's', O_R_S16N4, 0, 0, 0, 0,
	/* msr     */ P32(0xfff0ffff), P32(0xf3808c00), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 'f', 's', O_R_S16N4, 0, 0, 0, 0,
	/* b       */ P32(0xf800d000), P32(0xf0008000), 'b', S_c, S_dW, '\t', O_SIMM0_SJ2J1_REL_XXXXXX_____XXXXXXXXXXX, 0, 0, 0, 0,
	/* b       */ P32(0xf800d000), P32(0xf0009000), 'b', S_c, S_dW, '\t', O_SIMM0_SJ2J1_REL_XXXXXXXXXX_____XXXXXXXXXXX, 0, 0, 0, 0,
	/* bfc     */ P32(0xffff8020), P32(0xf36f0000), 'b', 'f', 'c', S_c, '\t', O_R_S8N4, '#', O_TODO("lsb"), '#', O_TODO("width"), 0, 0,
	/* bfi     */ P32(0xfff08020), P32(0xf3600000), 'b', 'f', 'i', S_c, '\t', O_R_S8N4, O_R_S16N4, '#', O_TODO("lsb"), '#', O_TODO("width"), 0, 0,
	/* bic     */ P32(0xfbe08000), P32(0xf0200000), 'b', 'i', 'c', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* bic     */ P32(0xffe08000), P32(0xea200000), 'b', 'i', 'c', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* bl      */ P32(0xf800d000), P32(0xf000d000), 'b', 'l', S_c, '\t', O_SIMM0_SJ2J1_REL_XXXXXXXXXX_____XXXXXXXXXXX, 0, 0, 0, 0,
	/* blx     */ P32(0xf800d001), P32(0xf000c000), 'b', 'l', 'x', S_c, '\t', O_SIMM00_SJ2J1_REL_XXXXXXXXXX_____XXXXXXXXXX_, 0, 0,
	/* cdp2    */ P32(0xff000010), P32(0xfe000000), 'c', 'd', 'p', '2', S_c, '\t', O_TODO("coproc"), O_TODO("opc1"), O_TODO("CRd"), O_TODO("CRn"), O_TODO("CRm"), O_TODO("opc2"), 0, 0, 0, 0,
	/* cdp     */ P32(0xff000010), P32(0xee000000), 'c', 'd', 'p', S_c, '\t', O_TODO("coproc"), O_TODO("opc1"), O_TODO("CRd"), O_TODO("CRn"), O_TODO("CRm"), O_TODO("opc2"), 0, 0,
	/* clz     */ P32(0xfff0f0f0), P32(0xfab0f080), 'c', 'l', 'z', S_c, '\t', O_R_S8N4, O_R_S0N4, 0, 0,
	/* cmp     */ P32(0xfff08f00), P32(0xebb00f00), 'c', 'm', 'p', S_c, S_dW, '\t', O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0, 0, 0,
	/* sub     */ P32(0xffe08000), P32(0xeba00000), 's', 'u', 'b', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* cmp     */ P32(0xfbf08f00), P32(0xf1b00f00), 'c', 'm', 'p', S_c, S_dW, '\t', O_R_S16N4, O_THUMBEI_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* sub     */ P32(0xfbe08000), P32(0xf1a00000), 's', 'u', 'b', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_X___________XXX____XXXXXXXX, 0, 0,
	/* teq     */ P32(0xfbf08f00), P32(0xf0900f00), 't', 'e', 'q', S_c, '\t', O_R_S16N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0,
	/* eor     */ P32(0xfbe08000), P32(0xf0800000), 'e', 'o', 'r', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* teq     */ P32(0xfff08f00), P32(0xea900f00), 't', 'e', 'q', S_c, '\t', O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0, 0, 0,
	/* eor     */ P32(0xffe08000), P32(0xea800000), 'e', 'o', 'r', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* mrrc2   */ P32(0xfff00000), P32(0xfc500000), 'm', 'r', 'r', 'c', '2', S_c, '\t', O_TODO("coproc"), O_TODO("opc"), O_R_S12N4, O_R_S16N4, O_TODO("CRm"), 0, 0, 0, 0,
	/* ldc2    */ P32(0xfe1f0000), P32(0xfc1f0000), 'l', 'd', 'c', '2', S_L, S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), O_TODO("label"), 0, 0,
	/* ldc2    */ P32(0xfe100000), P32(0xfc100000), 'l', 'd', 'c', '2', S_L, S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* mrrc    */ P32(0xfff00000), P32(0xec500000), 'm', 'r', 'r', 'c', S_c, '\t', O_TODO("coproc"), O_TODO("opc"), O_R_S12N4, O_R_S16N4, O_TODO("CRm"), 0, 0,
	/* ldc     */ P32(0xfe1f0000), P32(0xec1f0000), 'l', 'd', 'c', S_L, S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), O_TODO("label"), 0, 0, 0, 0,
	/* ldc     */ P32(0xfe100000), P32(0xec100000), 'l', 'd', 'c', S_L, S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* pop     */ P32(0xffff0000), P32(0xe8bd0000), 'p', 'o', 'p', S_c, S_dW, '\t', O_RLIST_S0N16, 0, 0,
	/* ldm     */ P32(0xffd02000), P32(0xe8900000), 'l', 'd', 'm', S_c, S_dW, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_P_S0N13, 0, 0, 0, 0,
	/* ldmdb   */ P32(0xffd02000), P32(0xe9100000), 'l', 'd', 'm', 'd', 'b', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_P_S0N13, 0, 0,
	/* pop     */ P32(0xffff0fff), P32(0xf85d0b04), 'p', 'o', 'p', S_c, S_dW, '\t', '{', O_R_S12N4, '}', 0, 0, 0, 0,
	/* ldr     */ P32(0xfff00fc0), P32(0xf8500000), 'l', 'd', 'r', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0, 0, 0,
	/* ldrt    */ P32(0xfff00f00), P32(0xf8500e00), 'l', 'd', 'r', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0,
	/* ldr     */ P32(0xfff00d00), P32(0xf8500900), 'l', 'd', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S9N1, O_IMM_S0N8, 0, 0,
	/* ldr     */ P32(0xfff00c00), P32(0xf8500c00), 'l', 'd', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S9N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0, 0, 0,
	/* ldr     */ P32(0xfff00000), P32(0xf8d00000), 'l', 'd', 'r', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0,
	/* ldr     */ P32(0xff7f0000), P32(0xf85f0000), 'l', 'd', 'r', S_c, S_dW, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* pld     */ P32(0xfff0ffc0), P32(0xf810f000), 'p', 'l', 'd', S_c, '\t', '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0,
	/* pld     */ P32(0xfff0ff00), P32(0xf810fc00), 'p', 'l', 'd', S_c, '\t', '[', O_R_S16N4, '#', '-', O_IMM_S0N8, ']', 0, 0,
	/* ldrb    */ P32(0xfff00fc0), P32(0xf8100000), 'l', 'd', 'r', 'b', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0,
	/* ldrbt   */ P32(0xfff00f00), P32(0xf8100e00), 'l', 'd', 'r', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* pld     */ P32(0xfff0f000), P32(0xf890f000), 'p', 'l', 'd', S_c, '\t', '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* ldrb    */ P32(0xfff00d00), P32(0xf8100900), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S9N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* ldrb    */ P32(0xfff00c00), P32(0xf8100c00), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S9N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0, 0, 0,
	/* ldrb    */ P32(0xfff00000), P32(0xf8900000), 'l', 'd', 'r', 'b', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* pld     */ P32(0xff7ff000), P32(0xf81ff000), 'p', 'l', 'd', S_c, '\t', O_PLUSMINUS_S23N1, O_TODO("label"), 0, 0,
	/* ldrb    */ P32(0xff7f0000), P32(0xf81f0000), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* ldrd    */ P32(0xff7f0000), P32(0xe95f0000), 'l', 'd', 'r', 'd', S_c, '\t', O_R_S12N4, O_R_S8N4, O_TODO("label"), 0, 0, 0, 0,
	/* ldrd    */ P32(0xff500000), P32(0xe9500000), 'l', 'd', 'r', 'd', S_c, '\t', O_R_S12N4, O_R_S8N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* ldrd    */ P32(0xff700000), P32(0xe8700000), 'l', 'd', 'r', 'd', S_c, '\t', O_R_S12N4, O_R_S8N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* ldrex   */ P32(0xfff00f00), P32(0xe8500f00), 'l', 'd', 'r', 'e', 'x', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* ldrexb  */ P32(0xfff00fff), P32(0xe8d00f4f), 'l', 'd', 'r', 'e', 'x', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* ldrexd  */ P32(0xfff000ff), P32(0xe8d0007f), 'l', 'd', 'r', 'e', 'x', 'd', S_c, '\t', O_R_S12N4, O_R_S8N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* ldrexh  */ P32(0xfff00fff), P32(0xe8d00f5f), 'l', 'd', 'r', 'e', 'x', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* pldw    */ P32(0xfff0ffc0), P32(0xf830f000), 'p', 'l', 'd', 'w', S_c, '\t', '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0, 0, 0,
	/* pldw    */ P32(0xfff0ff00), P32(0xf830fc00), 'p', 'l', 'd', 'w', S_c, '\t', '[', O_R_S16N4, '#', '-', O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* ldrh    */ P32(0xfff00fc0), P32(0xf8300000), 'l', 'd', 'r', 'h', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0,
	/* ldrht   */ P32(0xfff00f00), P32(0xf8300e00), 'l', 'd', 'r', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* pldw    */ P32(0xfff0f000), P32(0xf8b0f000), 'p', 'l', 'd', 'w', S_c, '\t', '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0,
	/* ldrh    */ P32(0xfff00d00), P32(0xf8300900), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S9N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* ldrh    */ P32(0xfff00c00), P32(0xf8300c00), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S9N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0, 0, 0,
	/* ldrh    */ P32(0xfff00000), P32(0xf8b00000), 'l', 'd', 'r', 'h', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* ldrh    */ P32(0xff7f0000), P32(0xf83f0000), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* pli     */ P32(0xfff0ffc0), P32(0xf910f000), 'p', 'l', 'i', S_c, '\t', '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0,
	/* pli     */ P32(0xfff0ff00), P32(0xf910fc00), 'p', 'l', 'i', S_c, '\t', '[', O_R_S16N4, '#', '-', O_IMM_S0N8, ']', 0, 0,
	/* ldrsb   */ P32(0xfff00fc0), P32(0xf9100000), 'l', 'd', 'r', 's', 'b', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0,
	/* ldrsbt  */ P32(0xfff00f00), P32(0xf9100e00), 'l', 'd', 'r', 's', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* pli     */ P32(0xfff0f000), P32(0xf990f000), 'p', 'l', 'i', S_c, '\t', '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* ldrsb   */ P32(0xfff00d00), P32(0xf9100900), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S9N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* ldrsb   */ P32(0xfff00c00), P32(0xf9100c00), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S9N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* ldrsb   */ P32(0xfff00000), P32(0xf9900000), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* pli     */ P32(0xff7ff000), P32(0xf91ff000), 'p', 'l', 'i', S_c, '\t', O_PLUSMINUS_S23N1, O_TODO("label"), 0, 0,
	/* ldrsb   */ P32(0xff7f0000), P32(0xf91f0000), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* ldrsh   */ P32(0xfff00fc0), P32(0xf9300000), 'l', 'd', 'r', 's', 'h', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0,
	/* ldrsht  */ P32(0xfff00f00), P32(0xf9300e00), 'l', 'd', 'r', 's', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* ldrsh   */ P32(0xfff00d00), P32(0xf9300900), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S9N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* ldrsh   */ P32(0xfff00c00), P32(0xf9300c00), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S9N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* ldrsh   */ P32(0xfff00000), P32(0xf9b00000), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* ldrsh   */ P32(0xff7f0000), P32(0xf93f0000), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* lsl     */ P32(0xffe0f0f0), P32(0xfa00f000), 'l', 's', 'l', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* lsr     */ P32(0xffe0f0f0), P32(0xfa20f000), 'l', 's', 'r', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* mcr2    */ P32(0xff100010), P32(0xfe000010), 'm', 'c', 'r', '2', S_c, '\t', O_TODO("coproc"), O_TODO("opc1"), O_R_S12N4, O_TODO("CRn"), O_TODO("CRm"), '{', O_TODO("opc2"), '}', 0, 0,
	/* mcr     */ P32(0xff100010), P32(0xee000010), 'm', 'c', 'r', S_c, '\t', O_TODO("coproc"), O_TODO("opc1"), O_R_S12N4, O_TODO("CRn"), O_TODO("CRm"), '{', O_TODO("opc2"), '}', 0, 0, 0, 0,
	/* mcrr2   */ P32(0xfff00000), P32(0xfc400000), 'm', 'c', 'r', 'r', '2', S_c, '\t', O_TODO("coproc"), O_TODO("opc1"), O_R_S12N4, O_R_S16N4, O_TODO("CRm"), 0, 0, 0, 0,
	/* mcrr    */ P32(0xfff00000), P32(0xec400000), 'm', 'c', 'r', 'r', S_c, '\t', O_TODO("coproc"), O_TODO("opc1"), O_R_S12N4, O_R_S16N4, O_TODO("CRm"), 0, 0,
	/* mul     */ P32(0xfff0f0f0), P32(0xfb00f000), 'm', 'u', 'l', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* mla     */ P32(0xfff000f0), P32(0xfb000000), 'm', 'l', 'a', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* mls     */ P32(0xfff000f0), P32(0xfb000010), 'm', 'l', 's', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* movt    */ P32(0xfbf08000), P32(0xf2c00000), 'm', 'o', 'v', 't', S_c, '\t', O_R_S8N4, O_IMM_X______XXXX_XXX____XXXXXXXX, 0, 0, 0, 0,
	/* movw    */ P32(0xfbf08000), P32(0xf2400000), 'm', 'o', 'v', 'w', S_c, '\t', O_R_S8N4, O_IMM_X______XXXX_XXX____XXXXXXXX, 0, 0, 0, 0,
	/* mov     */ P32(0xfbef8000), P32(0xf04f0000), 'm', 'o', 'v', S_S, S_c, S_dW, '\t', O_R_S8N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* orr     */ P32(0xfbe08000), P32(0xf0400000), 'o', 'r', 'r', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* mrc2    */ P32(0xff100010), P32(0xfe100010), 'm', 'r', 'c', '2', S_c, '\t', O_TODO("coproc"), O_TODO("opc1"), O_R_S12N4, O_TODO("CRn"), O_TODO("CRm"), '{', O_TODO("opc2"), '}', 0, 0,
	/* mrc     */ P32(0xff100010), P32(0xee100010), 'm', 'r', 'c', S_c, '\t', O_TODO("coproc"), O_TODO("opc1"), O_R_S12N4, O_TODO("CRn"), O_TODO("CRm"), '{', O_TODO("opc2"), '}', 0, 0, 0, 0,
	/* mvn     */ P32(0xfbef8000), P32(0xf06f0000), 'm', 'v', 'n', S_S, S_c, '\t', O_R_S8N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* orn     */ P32(0xfbe08000), P32(0xf0600000), 'o', 'r', 'n', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_C_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* mvn     */ P32(0xffef8000), P32(0xea6f0000), 'm', 'v', 'n', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* orn     */ P32(0xffe08000), P32(0xea600000), 'o', 'r', 'n', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* pkhbt   */ P32(0xfff08010), P32(0xeac00000), 'p', 'k', 'h', 'b', 't', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_IMM_XXX____XX______, '}', 0, 0, 0, 0,
	/* push    */ P32(0xffff0000), P32(0xe8ad0000), 'p', 'u', 's', 'h', S_c, S_dW, '\t', O_RLIST_S0N16, 0, 0, 0, 0,
	/* stm     */ P32(0xffd0a000), P32(0xe8800000), 's', 't', 'm', S_c, S_dW, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N13, 0, 0, 0, 0,
	/* push    */ P32(0xffff0fff), P32(0xf84d0d04), 'p', 'u', 's', 'h', S_c, S_dW, '\t', '{', O_R_S12N4, '}', 0, 0,
	/* strt    */ P32(0xfff00f00), P32(0xf8400e00), 's', 't', 'r', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0,
	/* str     */ P32(0xfff00c00), P32(0xf8400c00), 's', 't', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S9N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0, 0, 0,
	/* qadd16  */ P32(0xfff0f0f0), P32(0xfa90f010), 'q', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* qadd8   */ P32(0xfff0f0f0), P32(0xfa80f010), 'q', 'a', 'd', 'd', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* qadd    */ P32(0xfff0f0f0), P32(0xfa80f080), 'q', 'a', 'd', 'd', S_c, '\t', O_R_S8N4, O_R_S0N4, O_R_S16N4, 0, 0, 0, 0,
	/* qasx    */ P32(0xfff0f0f0), P32(0xfaa0f010), 'q', 'a', 's', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* qdadd   */ P32(0xfff0f0f0), P32(0xfa80f090), 'q', 'd', 'a', 'd', 'd', S_c, '\t', O_R_S8N4, O_R_S0N4, O_R_S16N4, 0, 0,
	/* qdsub   */ P32(0xfff0f0f0), P32(0xfa80f0b0), 'q', 'd', 's', 'u', 'b', S_c, '\t', O_R_S8N4, O_R_S0N4, O_R_S16N4, 0, 0,
	/* qsax    */ P32(0xfff0f0f0), P32(0xfae0f010), 'q', 's', 'a', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* qsub16  */ P32(0xfff0f0f0), P32(0xfad0f010), 'q', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* qsub8   */ P32(0xfff0f0f0), P32(0xfac0f010), 'q', 's', 'u', 'b', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* qsub    */ P32(0xfff0f0f0), P32(0xfa80f0a0), 'q', 's', 'u', 'b', S_c, '\t', O_R_S8N4, O_R_S0N4, O_R_S16N4, 0, 0, 0, 0,
	/* rbit    */ P32(0xfff0f0f0), P32(0xfa90f0a0), 'r', 'b', 'i', 't', S_c, '\t', O_R_S8N4, O_R_S0N4, 0, 0, 0, 0,
	/* rev16   */ P32(0xfff0f0f0), P32(0xfa90f090), 'r', 'e', 'v', '1', '6', S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, 0, 0,
	/* rev     */ P32(0xfff0f0f0), P32(0xfa90f080), 'r', 'e', 'v', S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, 0, 0, 0, 0,
	/* revsh   */ P32(0xfff0f0f0), P32(0xfa90f0b0), 'r', 'e', 'v', 's', 'h', S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, 0, 0,
	/* ror     */ P32(0xffe0f0f0), P32(0xfa60f000), 'r', 'o', 'r', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* rsb     */ P32(0xffe08000), P32(0xebc00000), 'r', 's', 'b', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* rsb     */ P32(0xfbe08000), P32(0xf1c00000), 'r', 's', 'b', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_X___________XXX____XXXXXXXX, 0, 0,
	/* sadd16  */ P32(0xfff0f0f0), P32(0xfa90f000), 's', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* sadd8   */ P32(0xfff0f0f0), P32(0xfa80f000), 's', 'a', 'd', 'd', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* sasx    */ P32(0xfff0f0f0), P32(0xfaa0f000), 's', 'a', 's', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* sbc     */ P32(0xfbe08000), P32(0xf1600000), 's', 'b', 'c', S_S, S_c, '\t', O_R_S8N4, O_R_S16N4, O_THUMBEI_X___________XXX____XXXXXXXX, 0, 0, 0, 0,
	/* sbc     */ P32(0xffe08000), P32(0xeb600000), 's', 'b', 'c', S_S, S_c, S_dW, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S4N2, 0, 0,
	/* sbfx    */ P32(0xfff08020), P32(0xf3400000), 's', 'b', 'f', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, '#', O_TODO("lsb"), '#', O_TODO("width"), 0, 0, 0, 0,
	/* sdiv    */ P32(0xfff0f0f0), P32(0xfb90f0f0), 's', 'd', 'i', 'v', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* sel     */ P32(0xfff0f0f0), P32(0xfaa0f080), 's', 'e', 'l', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* shadd16 */ P32(0xfff0f0f0), P32(0xfa90f020), 's', 'h', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* shadd8  */ P32(0xfff0f0f0), P32(0xfa80f020), 's', 'h', 'a', 'd', 'd', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* shasx   */ P32(0xfff0f0f0), P32(0xfaa0f020), 's', 'h', 'a', 's', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* shsax   */ P32(0xfff0f0f0), P32(0xfae0f020), 's', 'h', 's', 'a', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* shsub16 */ P32(0xfff0f0f0), P32(0xfad0f020), 's', 'h', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* shsub8  */ P32(0xfff0f0f0), P32(0xfac0f020), 's', 'h', 's', 'u', 'b', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smultt  */ P32(0xfff0f0f0), P32(0xfb10f030), 's', 'm', 'u', 'l', 't', 't', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlatt  */ P32(0xfff000f0), P32(0xfb100030), 's', 'm', 'l', 'a', 't', 't', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smulbt  */ P32(0xfff0f0f0), P32(0xfb10f010), 's', 'm', 'u', 'l', 'b', 't', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlabt  */ P32(0xfff000f0), P32(0xfb100010), 's', 'm', 'l', 'a', 'b', 't', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smultb  */ P32(0xfff0f0f0), P32(0xfb10f020), 's', 'm', 'u', 'l', 't', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlatb  */ P32(0xfff000f0), P32(0xfb100020), 's', 'm', 'l', 'a', 't', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smulbb  */ P32(0xfff0f0f0), P32(0xfb10f000), 's', 'm', 'u', 'l', 'b', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlabb  */ P32(0xfff000f0), P32(0xfb100000), 's', 'm', 'l', 'a', 'b', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smuadx  */ P32(0xfff0f0f0), P32(0xfb20f010), 's', 'm', 'u', 'a', 'd', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smladx  */ P32(0xfff000f0), P32(0xfb200010), 's', 'm', 'l', 'a', 'd', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smuad   */ P32(0xfff0f0f0), P32(0xfb20f000), 's', 'm', 'u', 'a', 'd', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlad   */ P32(0xfff000f0), P32(0xfb200000), 's', 'm', 'l', 'a', 'd', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0,
	/* smlal   */ P32(0xfff000f0), P32(0xfbc00000), 's', 'm', 'l', 'a', 'l', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlaltt */ P32(0xfff000f0), P32(0xfbc000b0), 's', 'm', 'l', 'a', 'l', 't', 't', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* smlalbt */ P32(0xfff000f0), P32(0xfbc00090), 's', 'm', 'l', 'a', 'l', 'b', 't', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* smlaltb */ P32(0xfff000f0), P32(0xfbc000a0), 's', 'm', 'l', 'a', 'l', 't', 'b', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* smlalbb */ P32(0xfff000f0), P32(0xfbc00080), 's', 'm', 'l', 'a', 'l', 'b', 'b', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* smlaldx */ P32(0xfff000f0), P32(0xfbc000d0), 's', 'm', 'l', 'a', 'l', 'd', 'x', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* smlald  */ P32(0xfff000f0), P32(0xfbc000c0), 's', 'm', 'l', 'a', 'l', 'd', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* smulwt  */ P32(0xfff0f0f0), P32(0xfb30f010), 's', 'm', 'u', 'l', 'w', 't', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlawt  */ P32(0xfff000f0), P32(0xfb300010), 's', 'm', 'l', 'a', 'w', 't', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smulwb  */ P32(0xfff0f0f0), P32(0xfb30f000), 's', 'm', 'u', 'l', 'w', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlawb  */ P32(0xfff000f0), P32(0xfb300000), 's', 'm', 'l', 'a', 'w', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smusdx  */ P32(0xfff0f0f0), P32(0xfb40f010), 's', 'm', 'u', 's', 'd', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlsdt  */ P32(0xfff000f0), P32(0xfb400010), 's', 'm', 'l', 's', 'd', 't', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smusd   */ P32(0xfff0f0f0), P32(0xfb40f000), 's', 'm', 'u', 's', 'd', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlsdb  */ P32(0xfff000f0), P32(0xfb400000), 's', 'm', 'l', 's', 'd', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smlsldt */ P32(0xfff000f0), P32(0xfbd000d0), 's', 'm', 'l', 's', 'l', 'd', 't', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* smlsldb */ P32(0xfff000f0), P32(0xfbd000c0), 's', 'm', 'l', 's', 'l', 'd', 'b', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* smmulr  */ P32(0xfff0f0f0), P32(0xfb50f010), 's', 'm', 'm', 'u', 'l', 'r', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smmlar  */ P32(0xfff000f0), P32(0xfb500010), 's', 'm', 'm', 'l', 'a', 'r', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smmul   */ P32(0xfff0f0f0), P32(0xfb50f000), 's', 'm', 'm', 'u', 'l', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smmla   */ P32(0xfff000f0), P32(0xfb500000), 's', 'm', 'm', 'l', 'a', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0,
	/* smmlsr  */ P32(0xfff000f0), P32(0xfb600010), 's', 'm', 'm', 'l', 's', 'r', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* smmls   */ P32(0xfff000f0), P32(0xfb600000), 's', 'm', 'm', 'l', 's', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0,
	/* smull   */ P32(0xfff000f0), P32(0xfb800000), 's', 'm', 'u', 'l', 'l', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* ssat16  */ P32(0xfff0f0f0), P32(0xf3200000), 's', 's', 'a', 't', '1', '6', S_c, '\t', O_R_S8N4, O_IMMP1_S0N4, O_R_S16N4, 0, 0,
	/* ssat    */ P32(0xffd08020), P32(0xf3000000), 's', 's', 'a', 't', S_c, '\t', O_R_S8N4, O_IMM_XXX____XX______, O_R_S16N4, O_OPT_SHIFT_S21N1, 0, 0,
	/* ssax    */ P32(0xfff0f0f0), P32(0xfae0f000), 's', 's', 'a', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* ssub16  */ P32(0xfff0f0f0), P32(0xfad0f000), 's', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* ssub8   */ P32(0xfff0f0f0), P32(0xfac0f000), 's', 's', 'u', 'b', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* stc2    */ P32(0xff500000), P32(0xfd000000), 's', 't', 'c', '2', S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* stc2    */ P32(0xff700000), P32(0xfc200000), 's', 't', 'c', '2', S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* stc2l   */ P32(0xff500000), P32(0xfd400000), 's', 't', 'c', '2', 'l', S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* stc2l   */ P32(0xff700000), P32(0xfc600000), 's', 't', 'c', '2', 'l', S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N8, 0, 0,
	/* stc     */ P32(0xff500000), P32(0xed000000), 's', 't', 'c', S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0, 0, 0,
	/* stc     */ P32(0xff700000), P32(0xec200000), 's', 't', 'c', S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* stcl    */ P32(0xff500000), P32(0xed400000), 's', 't', 'c', 'l', S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* stcl    */ P32(0xff700000), P32(0xec600000), 's', 't', 'c', 'l', S_c, '\t', O_TODO("coproc"), O_TODO("CRd"), '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* stmdb   */ P32(0xffd0a000), P32(0xe9000000), 's', 't', 'm', 'd', 'b', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N13, 0, 0,
	/* str     */ P32(0xfff00d00), P32(0xf8400900), 's', 't', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S9N1, O_IMM_S0N8, 0, 0,
	/* str     */ P32(0xfff00000), P32(0xf8c00000), 's', 't', 'r', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0,
	/* str     */ P32(0xfff00fc0), P32(0xf8400000), 's', 't', 'r', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0, 0, 0,
	/* strbt   */ P32(0xfff00f00), P32(0xf8000e00), 's', 't', 'r', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* strb    */ P32(0xfff00c00), P32(0xf8000c00), 's', 't', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S9N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0, 0, 0,
	/* strb    */ P32(0xfff00d00), P32(0xf8000900), 's', 't', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S9N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* strb    */ P32(0xfff00000), P32(0xf8800000), 's', 't', 'r', 'b', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* strb    */ P32(0xfff00fc0), P32(0xf8000000), 's', 't', 'r', 'b', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0,
	/* strd    */ P32(0xff500000), P32(0xe9400000), 's', 't', 'r', 'd', S_c, '\t', O_R_S12N4, O_R_S8N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0,
	/* strd    */ P32(0xff700000), P32(0xe8600000), 's', 't', 'r', 'd', S_c, '\t', O_R_S12N4, O_R_S8N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* strex   */ P32(0xfff00000), P32(0xe8400000), 's', 't', 'r', 'e', 'x', S_c, '\t', O_R_S8N4, O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* strexb  */ P32(0xfff00ff0), P32(0xe8c00f40), 's', 't', 'r', 'e', 'x', 'b', S_c, '\t', O_R_S0N4, O_R_S12N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* strexd  */ P32(0xfff000f0), P32(0xe8c00070), 's', 't', 'r', 'e', 'x', 'd', S_c, '\t', O_R_S0N4, O_R_S12N4, O_R_S8N4, '[', O_R_S16N4, ']', 0, 0,
	/* strexh  */ P32(0xfff00ff0), P32(0xe8c00f50), 's', 't', 'r', 'e', 'x', 'h', S_c, '\t', O_R_S0N4, O_R_S12N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* strht   */ P32(0xfff00f00), P32(0xf8200e00), 's', 't', 'r', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N8, ']', 0, 0, 0, 0,
	/* strh    */ P32(0xfff00c00), P32(0xf8200c00), 's', 't', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S9N1, O_IMM_S0N8, ']', O_OPT_IA, 0, 0, 0, 0,
	/* strh    */ P32(0xfff00d00), P32(0xf8200900), 's', 't', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S9N1, O_IMM_S0N8, 0, 0, 0, 0,
	/* strh    */ P32(0xfff00fc0), P32(0xf8200000), 's', 't', 'r', 'h', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_TODO("imm2"), '}', ']', 0, 0,
	/* strh    */ P32(0xfff00000), P32(0xf8a00000), 's', 't', 'r', 'h', S_c, S_dW, '\t', O_R_S12N4, '[', O_R_S16N4, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* sxtb16  */ P32(0xfffff0c0), P32(0xfa2ff080), 's', 'x', 't', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* sxtab16 */ P32(0xfff0f0c0), P32(0xfa20f080), 's', 'x', 't', 'a', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0, 0, 0,
	/* sxtb    */ P32(0xfffff0c0), P32(0xfa4ff080), 's', 'x', 't', 'b', S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* sxtab   */ P32(0xfff0f0c0), P32(0xfa40f080), 's', 'x', 't', 'a', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* sxth    */ P32(0xfffff0c0), P32(0xfa0ff080), 's', 'x', 't', 'h', S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* sxtah   */ P32(0xfff0f0c0), P32(0xfa00f080), 's', 'x', 't', 'a', 'h', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* tbb     */ P32(0xfff0ffe0), P32(0xe8d0f000), 't', 'b', 'b', S_c, '\t', '[', O_R_S16N4, O_R_S0N4, ']', 0, 0, 0, 0,
	/* uadd16  */ P32(0xfff0f0f0), P32(0xfa90f040), 'u', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uadd8   */ P32(0xfff0f0f0), P32(0xfa80f040), 'u', 'a', 'd', 'd', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uasx    */ P32(0xfff0f0f0), P32(0xfaa0f040), 'u', 'a', 's', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* ubfx    */ P32(0xfff08020), P32(0xf3c00000), 'u', 'b', 'f', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, '#', O_TODO("lsb"), '#', O_TODO("width"), 0, 0, 0, 0,
	/* udiv    */ P32(0xfff0f0f0), P32(0xfbb0f0f0), 'u', 'd', 'i', 'v', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uhadd16 */ P32(0xfff0f0f0), P32(0xfa90f060), 'u', 'h', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uhadd8  */ P32(0xfff0f0f0), P32(0xfa80f060), 'u', 'h', 'a', 'd', 'd', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uhasx   */ P32(0xfff0f0f0), P32(0xfaa0f060), 'u', 'h', 'a', 's', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uhsax   */ P32(0xfff0f0f0), P32(0xfae0f060), 'u', 'h', 's', 'a', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uhsub16 */ P32(0xfff0f0f0), P32(0xfad0f060), 'u', 'h', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uhsub8  */ P32(0xfff0f0f0), P32(0xfac0f060), 'u', 'h', 's', 'u', 'b', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* umaal   */ P32(0xfff000f0), P32(0xfbe00060), 'u', 'm', 'a', 'a', 'l', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* umlal   */ P32(0xfff000f0), P32(0xfbe00000), 'u', 'm', 'l', 'a', 'l', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* umull   */ P32(0xfff000f0), P32(0xfba00000), 'u', 'm', 'u', 'l', 'l', S_c, '\t', O_R_S12N4, O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uqadd16 */ P32(0xfff0f0f0), P32(0xfa90f050), 'u', 'q', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uqadd8  */ P32(0xfff0f0f0), P32(0xfa80f050), 'u', 'q', 'a', 'd', 'd', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uqasx   */ P32(0xfff0f0f0), P32(0xfaa0f050), 'u', 'q', 'a', 's', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uqsax   */ P32(0xfff0f0f0), P32(0xfae0f050), 'u', 'q', 's', 'a', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uqsub16 */ P32(0xfff0f0f0), P32(0xfad0f050), 'u', 'q', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uqsub8  */ P32(0xfff0f0f0), P32(0xfac0f050), 'u', 'q', 's', 'u', 'b', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* usad8   */ P32(0xfff0f0f0), P32(0xfb70f000), 'u', 's', 'a', 'd', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* usada8  */ P32(0xfff000f0), P32(0xfb700000), 'u', 's', 'a', 'd', 'a', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_R_S12N4, 0, 0, 0, 0,
	/* usat16  */ P32(0xfff0f0f0), P32(0xf3a00000), 'u', 's', 'a', 't', '1', '6', S_c, '\t', O_R_S8N4, O_IMM_S0N4, O_R_S16N4, 0, 0,
	/* usat    */ P32(0xffd08020), P32(0xf3800000), 'u', 's', 'a', 't', S_c, '\t', O_R_S8N4, '#', O_TODO("DecodeImmShift(sh0,imm)"), O_R_S16N4, O_OPT_SHIFT_, 0, 0,
	/* usax    */ P32(0xfff0f0f0), P32(0xfae0f040), 'u', 's', 'a', 'x', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* usub16  */ P32(0xfff0f0f0), P32(0xfad0f040), 'u', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* usub8   */ P32(0xfff0f0f0), P32(0xfac0f040), 'u', 's', 'u', 'b', '8', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uxtb16  */ P32(0xfffff0c0), P32(0xfa3ff080), 'u', 'x', 't', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* uxtab16 */ P32(0xfff0f0c0), P32(0xfa30f080), 'u', 'x', 't', 'a', 'b', '1', '6', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0, 0, 0,
	/* uxtb    */ P32(0xfffff0c0), P32(0xfa5ff080), 'u', 'x', 't', 'b', S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* uxtab   */ P32(0xfff0f0c0), P32(0xfa50f080), 'u', 'x', 't', 'a', 'b', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* uxth    */ P32(0xfffff0c0), P32(0xfa1ff080), 'u', 'x', 't', 'h', S_c, S_dW, '\t', O_R_S8N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* uxtah   */ P32(0xfff0f0c0), P32(0xfa10f080), 'u', 'x', 't', 'a', 'h', S_c, '\t', O_R_S8N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S4N2, 0, 0,
	/* adc     */ P32(0x0fe00000), P32(0x02a00000), 'a', 'd', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_S0N12, 0, 0, 0, 0,
	/* adc     */ P32(0x0fe00090), P32(0x00a00010), 'a', 'd', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* adc     */ P32(0x0fe00010), P32(0x00a00000), 'a', 'd', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* adr     */ P32(0x0fff0000), P32(0x028f0000), 'a', 'd', 'r', S_c, '\t', O_R_S12N4, '+', O_ARMEI_REL_S0N12, 0, 0, 0, 0,
	/* add     */ P32(0x0fe00000), P32(0x02800000), 'a', 'd', 'd', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_S0N12, 0, 0, 0, 0,
	/* add     */ P32(0x0fe00090), P32(0x00800010), 'a', 'd', 'd', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* add     */ P32(0x0fe00010), P32(0x00800000), 'a', 'd', 'd', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* adr     */ P32(0x0fff0000), P32(0x024f0000), 'a', 'd', 'r', S_c, '\t', O_R_S12N4, '-', O_ARMEI_REL_S0N12, 0, 0, 0, 0,
	/* sub     */ P32(0x0fe00000), P32(0x02400000), 's', 'u', 'b', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_S0N12, 0, 0, 0, 0,
	/* and     */ P32(0x0fe00000), P32(0x02000000), 'a', 'n', 'd', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_C_S0N12, 0, 0, 0, 0,
	/* and     */ P32(0x0fe00090), P32(0x00000010), 'a', 'n', 'd', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* and     */ P32(0x0fe00010), P32(0x00000000), 'a', 'n', 'd', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* asr     */ P32(0x0fef0070), P32(0x01a00040), 'a', 's', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_IMM_S7N5, 0, 0, 0, 0,
	/* asr     */ P32(0x0fef00f0), P32(0x01a00050), 'a', 's', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* b       */ P32(0x0f000000), P32(0x0a000000), 'b', S_c, '\t', O_SIMM00_REL_S0N24, 0, 0, 0, 0,
	/* bl      */ P32(0x0f000000), P32(0x0b000000), 'b', 'l', S_c, '\t', O_SIMM00_REL_S0N24, 0, 0, 0, 0,
	/* blx     */ P32(0xfe000000), P32(0xfa000000), 'b', 'l', 'x', '\t', O_SIMM0_H_REL_S0N25, 0, 0, 0, 0,
	/* bfc     */ P32(0x0fe0007f), P32(0x07c0001f), 'b', 'f', 'c', S_c, '\t', O_R_S12N4, '#', O_TODO("lsb"), '#', O_TODO("width"), 0, 0,
	/* bfi     */ P32(0x0fe00070), P32(0x07c00010), 'b', 'f', 'i', S_c, '\t', O_R_S12N4, O_R_S0N4, '#', O_TODO("lsb"), '#', O_TODO("width"), 0, 0,
	/* bic     */ P32(0x0fe00000), P32(0x03c00000), 'b', 'i', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_C_S0N12, 0, 0, 0, 0,
	/* bic     */ P32(0x0fe00090), P32(0x01c00010), 'b', 'i', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* bic     */ P32(0x0fe00010), P32(0x01c00000), 'b', 'i', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* bkpt    */ P32(0x0ff000f0), P32(0x01200070), 'b', 'k', 'p', 't', '\t', O_IMM_XXXXXXXXXXXX____XXXX, 0, 0,
	/* blx     */ P32(0x0ffffff0), P32(0x012fff30), 'b', 'l', 'x', S_c, '\t', O_R_S0N4, 0, 0,
	/* bx      */ P32(0x0ffffff0), P32(0x012fff10), 'b', 'x', S_c, '\t', O_R_S0N4, 0, 0, 0, 0,
	/* bxj     */ P32(0x0ffffff0), P32(0x012fff20), 'b', 'x', 'j', S_c, '\t', O_R_S0N4, 0, 0,
	/* clrex   */ P32(0xffffffff), P32(0xf57ff01f), 'c', 'l', 'r', 'e', 'x', 0, 0, 0, 0,
	/* dmb     */ P32(0xfffffff0), P32(0xf57ff050), 'd', 'm', 'b', '\t', '#', O_MB_OPTION_S0N4, 0, 0,
	/* dsb     */ P32(0xfffffff0), P32(0xf57ff040), 'd', 's', 'b', '\t', '#', O_MB_OPTION_S0N4, 0, 0,
	/* isb     */ P32(0xfffffff0), P32(0xf57ff060), 'i', 's', 'b', '\t', '#', O_MB_OPTION_S0N4, 0, 0,
	/* pld     */ P32(0xff7ff000), P32(0xf55ff000), 'p', 'l', 'd', '\t', O_PLUSMINUS_S23N1, O_TODO("label"), 0, 0,
	/* pldw    */ P32(0xff70f000), P32(0xf550f000), 'p', 'l', 'd', 'w', '\t', '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N12, ']', 0, 0,
	/* ldrb    */ P32(0x0f7f0000), P32(0x055f0000), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* ldrb    */ P32(0x0f500000), P32(0x05500000), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N12, ']', O_OPT_IA, 0, 0, 0, 0,
	/* clz     */ P32(0x0fff0ff0), P32(0x016f0f10), 'c', 'l', 'z', S_c, '\t', O_R_S12N4, O_R_S0N4, 0, 0,
	/* cmn     */ P32(0x0ff0f000), P32(0x03700000), 'c', 'm', 'n', S_c, '\t', O_R_S16N4, O_ARMEI_S0N12, 0, 0,
	/* cmn     */ P32(0x0ff0f090), P32(0x01700010), 'c', 'm', 'n', S_c, '\t', O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0, 0, 0,
	/* cmn     */ P32(0x0ff0f010), P32(0x01700000), 'c', 'm', 'n', S_c, '\t', O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* cmp     */ P32(0x0ff0f000), P32(0x03500000), 'c', 'm', 'p', S_c, '\t', O_R_S16N4, O_ARMEI_S0N12, 0, 0,
	/* cmp     */ P32(0x0ff0f090), P32(0x01500010), 'c', 'm', 'p', S_c, '\t', O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0, 0, 0,
	/* cmp     */ P32(0x0ff0f010), P32(0x01500000), 'c', 'm', 'p', S_c, '\t', O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* dbg     */ P32(0x0ffffff0), P32(0x0320f0f0), 'd', 'b', 'g', S_c, '\t', O_IMM_S0N4, 0, 0,
	/* eor     */ P32(0x0fe00000), P32(0x02200000), 'e', 'o', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_C_S0N12, 0, 0, 0, 0,
	/* eor     */ P32(0x0fe00090), P32(0x00200010), 'e', 'o', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* eor     */ P32(0x0fe00010), P32(0x00200000), 'e', 'o', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* pop     */ P32(0x0fff0000), P32(0x08bd0000), 'p', 'o', 'p', S_c, '\t', O_RLIST_S0N16, 0, 0,
	/* ldm     */ P32(0x0fd00000), P32(0x08900000), 'l', 'd', 'm', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N16, 0, 0, 0, 0,
	/* ldmda   */ P32(0x0fd00000), P32(0x08100000), 'l', 'd', 'm', 'd', 'a', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N16, 0, 0,
	/* ldmdb   */ P32(0x0fd00000), P32(0x09100000), 'l', 'd', 'm', 'd', 'b', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N16, 0, 0,
	/* ldmib   */ P32(0x0fd00000), P32(0x09900000), 'l', 'd', 'm', 'i', 'b', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N16, 0, 0,
	/* pld     */ P32(0xff70f000), P32(0xf510f000), 'p', 'l', 'd', '\t', '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* ldr     */ P32(0x0f7f0000), P32(0x051f0000), 'l', 'd', 'r', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0,
	/* ldr     */ P32(0x0f500000), P32(0x05100000), 'l', 'd', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N12, ']', O_OPT_IA, 0, 0, 0, 0,
	/* pld     */ P32(0xff70f010), P32(0xf710f000), 'p', 'l', 'd', S_c, '\t', '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, ']', 0, 0,
	/* ldr     */ P32(0x0f500010), P32(0x07100000), 'l', 'd', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, ']', O_OPT_IA, 0, 0, 0, 0,
	/* ldr     */ P32(0x0f700010), P32(0x06100000), 'l', 'd', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* pop     */ P32(0x0fff0fff), P32(0x049d0004), 'p', 'o', 'p', S_c, '\t', '{', O_R_S12N4, '}', 0, 0, 0, 0,
	/* ldr     */ P32(0x0f700000), P32(0x04100000), 'l', 'd', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N12, 0, 0,
	/* pldw    */ P32(0xff70f010), P32(0xf750f000), 'p', 'l', 'd', 'w', S_c, '\t', '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, ']', 0, 0, 0, 0,
	/* ldrb    */ P32(0x0f500010), P32(0x07500000), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, ']', O_OPT_IA, 0, 0,
	/* pli     */ P32(0xff70f010), P32(0xf650f000), 'p', 'l', 'i', '\t', '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, ']', 0, 0,
	/* ldrb    */ P32(0x0f700010), P32(0x06500000), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* pli     */ P32(0xff70f000), P32(0xf450f000), 'p', 'l', 'i', '\t', '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N12, ']', 0, 0, 0, 0,
	/* ldrb    */ P32(0x0f700000), P32(0x04500000), 'l', 'd', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N12, 0, 0, 0, 0,
	/* ldrbt   */ P32(0x0f700000), P32(0x04700000), 'l', 'd', 'r', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N12, 0, 0, 0, 0,
	/* ldrbt   */ P32(0x0f700010), P32(0x06700000), 'l', 'd', 'r', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* ldrd    */ P32(0x0f500ff0), P32(0x010000d0), 'l', 'd', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, ']', O_OPT_IA, 0, 0,
	/* ldrd    */ P32(0x0f700ff0), P32(0x000000d0), 'l', 'd', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0, 0, 0,
	/* ldrd    */ P32(0x0f7f00f0), P32(0x014f00d0), 'l', 'd', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* ldrd    */ P32(0x0f5000f0), P32(0x014000d0), 'l', 'd', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, ']', O_OPT_IA, 0, 0,
	/* ldrd    */ P32(0x0f7000f0), P32(0x004000d0), 'l', 'd', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0, 0, 0,
	/* ldrex   */ P32(0x0ff00fff), P32(0x01900f9f), 'l', 'd', 'r', 'e', 'x', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', 0, 0,
	/* ldrexb  */ P32(0x0ff00fff), P32(0x01d00f9f), 'l', 'd', 'r', 'e', 'x', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* ldrexd  */ P32(0x0ff00fff), P32(0x01b00f9f), 'l', 'd', 'r', 'e', 'x', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* ldrexh  */ P32(0x0ff00fff), P32(0x01f00f9f), 'l', 'd', 'r', 'e', 'x', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* ldrh    */ P32(0x0f7f00f0), P32(0x015f00b0), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* ldrh    */ P32(0x0f5000f0), P32(0x015000b0), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, ']', O_OPT_IA, 0, 0, 0, 0,
	/* ldrh    */ P32(0x0f500ff0), P32(0x011000b0), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, ']', O_OPT_IA, 0, 0, 0, 0,
	/* ldrh    */ P32(0x0f700ff0), P32(0x001000b0), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0, 0, 0,
	/* ldrh    */ P32(0x0f7000f0), P32(0x005000b0), 'l', 'd', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0, 0, 0,
	/* ldrht   */ P32(0x0f7000f0), P32(0x007000b0), 'l', 'd', 'r', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0, 0, 0,
	/* ldrht   */ P32(0x0f700ff0), P32(0x003000b0), 'l', 'd', 'r', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0, 0, 0,
	/* ldrsb   */ P32(0x0f7f00f0), P32(0x015f00d0), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* ldrsb   */ P32(0x0f5000f0), P32(0x015000d0), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, ']', O_OPT_IA, 0, 0,
	/* ldrsb   */ P32(0x0f500ff0), P32(0x011000d0), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, ']', O_OPT_IA, 0, 0,
	/* ldrsb   */ P32(0x0f700ff0), P32(0x001000d0), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0, 0, 0,
	/* ldrsb   */ P32(0x0f7000f0), P32(0x005000d0), 'l', 'd', 'r', 's', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0, 0, 0,
	/* ldrsbt  */ P32(0x0f7000f0), P32(0x007000d0), 'l', 'd', 'r', 's', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0,
	/* ldrsbt  */ P32(0x0f700ff0), P32(0x003000d0), 'l', 'd', 'r', 's', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0,
	/* ldrsh   */ P32(0x0f7f00f0), P32(0x015f00f0), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, O_TODO("label"), 0, 0, 0, 0,
	/* ldrsh   */ P32(0x0f5000f0), P32(0x015000f0), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, ']', O_OPT_IA, 0, 0,
	/* ldrsh   */ P32(0x0f500ff0), P32(0x011000f0), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, ']', O_OPT_IA, 0, 0,
	/* ldrsh   */ P32(0x0f700ff0), P32(0x001000f0), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0, 0, 0,
	/* ldrsh   */ P32(0x0f7000f0), P32(0x005000f0), 'l', 'd', 'r', 's', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0, 0, 0,
	/* ldrsht  */ P32(0x0f7000f0), P32(0x007000f0), 'l', 'd', 'r', 's', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0,
	/* ldrsht  */ P32(0x0f700ff0), P32(0x003000f0), 'l', 'd', 'r', 's', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0,
	/* ldrt    */ P32(0x0f700000), P32(0x04300000), 'l', 'd', 'r', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N12, 0, 0, 0, 0,
	/* ldrt    */ P32(0x0f700010), P32(0x06300000), 'l', 'd', 'r', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* mov     */ P32(0x0fef0ff0), P32(0x01a00000), 'm', 'o', 'v', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, 0, 0, 0, 0,
	/* lsl     */ P32(0x0fef0070), P32(0x01a00000), 'l', 's', 'l', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_IMM_S7N5, 0, 0, 0, 0,
	/* lsl     */ P32(0x0fef00f0), P32(0x01a00010), 'l', 's', 'l', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* lsr     */ P32(0x0fef0070), P32(0x01a00020), 'l', 's', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_IMM_S7N5, 0, 0, 0, 0,
	/* lsr     */ P32(0x0fef00f0), P32(0x01a00030), 'l', 's', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* mla     */ P32(0x0fe000f0), P32(0x00200090), 'm', 'l', 'a', S_S, S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0,
	/* mls     */ P32(0x0ff000f0), P32(0x00600090), 'm', 'l', 's', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* movt    */ P32(0x0ff00000), P32(0x03400000), 'm', 'o', 'v', 't', S_c, '\t', O_R_S12N4, O_IMM_XXXX____XXXXXXXXXXXX, 0, 0, 0, 0,
	/* movw    */ P32(0x0ff00000), P32(0x03000000), 'm', 'o', 'v', 'w', S_c, '\t', O_R_S12N4, O_IMM_XXXX____XXXXXXXXXXXX, 0, 0, 0, 0,
	/* mov     */ P32(0x0fef0000), P32(0x03a00000), 'm', 'o', 'v', S_S, S_c, '\t', O_R_S12N4, O_ARMEI_C_S0N12, 0, 0, 0, 0,
	/* mrs     */ P32(0x0fff0fff), P32(0x010f0000), 'm', 'r', 's', S_c, '\t', O_R_S12N4, 'C', 'P', 'S', 'R', 0, 0,
	/* mrs     */ P32(0x0fff0fff), P32(0x014f0000), 'm', 'r', 's', S_c, '\t', O_R_S12N4, 'S', 'P', 'S', 'R', 0, 0,
	/* msr     */ P32(0x0ffff000), P32(0x0328f000), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 'f', O_ARMEI_S0N12, 0, 0, 0, 0,
	/* msr     */ P32(0x0ffff000), P32(0x0324f000), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 's', O_ARMEI_S0N12, 0, 0, 0, 0,
	/* msr     */ P32(0x0ffff000), P32(0x032cf000), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 'f', 's', O_ARMEI_S0N12, 0, 0, 0, 0,
	/* msr     */ P32(0x0ffffff0), P32(0x0128f000), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 'f', O_R_S0N4, 0, 0, 0, 0,
	/* msr     */ P32(0x0ffffff0), P32(0x0124f000), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 's', O_R_S0N4, 0, 0, 0, 0,
	/* msr     */ P32(0x0ffffff0), P32(0x012cf000), 'm', 's', 'r', S_c, '\t', 'C', 'P', 'S', 'R', '_', 'f', 's', O_R_S0N4, 0, 0, 0, 0,
	/* mul     */ P32(0x0fe0f0f0), P32(0x00000090), 'm', 'u', 'l', S_S, S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* mvn     */ P32(0x0fef0000), P32(0x03e00000), 'm', 'v', 'n', S_S, S_c, '\t', O_R_S12N4, O_ARMEI_C_S0N12, 0, 0, 0, 0,
	/* mvn     */ P32(0x0fef0090), P32(0x01e00010), 'm', 'v', 'n', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* mvn     */ P32(0x0fef0010), P32(0x01e00000), 'm', 'v', 'n', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* nop     */ P32(0x0fffffff), P32(0x0320f000), 'n', 'o', 'p', S_c, 0, 0, 0, 0,
	/* orr     */ P32(0x0fe00000), P32(0x03800000), 'o', 'r', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_C_S0N12, 0, 0, 0, 0,
	/* orr     */ P32(0x0fe00090), P32(0x01800010), 'o', 'r', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* orr     */ P32(0x0fe00010), P32(0x01800000), 'o', 'r', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* pkhbt   */ P32(0x0ff00030), P32(0x06800010), 'p', 'k', 'h', 'b', 't', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, '{', 'L', 'S', 'L', O_IMM_S7N5, '}', 0, 0, 0, 0,
	/* push    */ P32(0x0fff0fff), P32(0x052d0004), 'p', 'u', 's', 'h', S_c, '\t', '{', O_R_S12N4, '}', 0, 0, 0, 0,
	/* str     */ P32(0x0f500000), P32(0x05000000), 's', 't', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N12, ']', O_OPT_IA, 0, 0, 0, 0,
	/* push    */ P32(0x0fff0000), P32(0x092d0000), 'p', 'u', 's', 'h', S_c, '\t', O_RLIST_S0N16, 0, 0,
	/* stmdb   */ P32(0x0fd00000), P32(0x09000000), 's', 't', 'm', 'd', 'b', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N16, 0, 0,
	/* qadd16  */ P32(0x0ff00ff0), P32(0x06200f10), 'q', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* qadd8   */ P32(0x0ff00ff0), P32(0x06200f90), 'q', 'a', 'd', 'd', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* qadd    */ P32(0x0ff00ff0), P32(0x01000050), 'q', 'a', 'd', 'd', S_c, '\t', O_R_S12N4, O_R_S0N4, O_R_S16N4, 0, 0, 0, 0,
	/* qasx    */ P32(0x0ff00ff0), P32(0x06200f30), 'q', 'a', 's', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* qdadd   */ P32(0x0ff00ff0), P32(0x01400050), 'q', 'd', 'a', 'd', 'd', S_c, '\t', O_R_S12N4, O_R_S0N4, O_R_S16N4, 0, 0,
	/* qdsub   */ P32(0x0ff00ff0), P32(0x01600050), 'q', 'd', 's', 'u', 'b', S_c, '\t', O_R_S12N4, O_R_S0N4, O_R_S16N4, 0, 0,
	/* qsax    */ P32(0x0ff00ff0), P32(0x06200f50), 'q', 's', 'a', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* qsub16  */ P32(0x0ff00ff0), P32(0x06200f70), 'q', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* qsub8   */ P32(0x0ff00ff0), P32(0x06200ff0), 'q', 's', 'u', 'b', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* qsub    */ P32(0x0ff00ff0), P32(0x01200050), 'q', 's', 'u', 'b', S_c, '\t', O_R_S12N4, O_R_S0N4, O_R_S16N4, 0, 0, 0, 0,
	/* rbit    */ P32(0x0fff0ff0), P32(0x06ff0f30), 'r', 'b', 'i', 't', S_c, '\t', O_R_S12N4, O_R_S0N4, 0, 0, 0, 0,
	/* rev16   */ P32(0x0fff0ff0), P32(0x06bf0fb0), 'r', 'e', 'v', '1', '6', S_c, '\t', O_R_S12N4, O_R_S0N4, 0, 0, 0, 0,
	/* rev     */ P32(0x0fff0ff0), P32(0x06bf0f30), 'r', 'e', 'v', S_c, '\t', O_R_S12N4, O_R_S0N4, 0, 0,
	/* revsh   */ P32(0x0fff0ff0), P32(0x06ff0fb0), 'r', 'e', 'v', 's', 'h', S_c, '\t', O_R_S12N4, O_R_S0N4, 0, 0, 0, 0,
	/* rrx     */ P32(0x0fef0ff0), P32(0x01a00060), 'r', 'r', 'x', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, 0, 0, 0, 0,
	/* ror     */ P32(0x0fef0070), P32(0x01a00060), 'r', 'o', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_IMM_S7N5, 0, 0, 0, 0,
	/* ror     */ P32(0x0fef00f0), P32(0x01a00070), 'r', 'o', 'r', S_S, S_c, '\t', O_R_S12N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* rsb     */ P32(0x0fe00000), P32(0x02600000), 'r', 's', 'b', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_S0N12, 0, 0, 0, 0,
	/* rsb     */ P32(0x0fe00090), P32(0x00600010), 'r', 's', 'b', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* rsb     */ P32(0x0fe00010), P32(0x00600000), 'r', 's', 'b', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* rsc     */ P32(0x0fe00000), P32(0x02e00000), 'r', 's', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_S0N12, 0, 0, 0, 0,
	/* rsc     */ P32(0x0fe00090), P32(0x00e00010), 'r', 's', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* rsc     */ P32(0x0fe00010), P32(0x00e00000), 'r', 's', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* sadd16  */ P32(0x0ff00ff0), P32(0x06100f10), 's', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* sadd8   */ P32(0x0ff00ff0), P32(0x06100f90), 's', 'a', 'd', 'd', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* sasx    */ P32(0x0ff00ff0), P32(0x06100f30), 's', 'a', 's', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* sbc     */ P32(0x0fe00000), P32(0x02c00000), 's', 'b', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_ARMEI_S0N12, 0, 0, 0, 0,
	/* sbc     */ P32(0x0fe00090), P32(0x00c00010), 's', 'b', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* sbc     */ P32(0x0fe00010), P32(0x00c00000), 's', 'b', 'c', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* sbfx    */ P32(0x0fe00070), P32(0x07a00050), 's', 'b', 'f', 'x', S_c, '\t', O_R_S12N4, O_R_S0N4, '#', O_TODO("lsb"), '#', O_TODO("width"), 0, 0, 0, 0,
	/* sel     */ P32(0x0ff00ff0), P32(0x06800fb0), 's', 'e', 'l', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* setend  */ P32(0xffffffff), P32(0xf1010000), 's', 'e', 't', 'e', 'n', 'd', '\t', 'L', 'E', 0, 0, 0, 0,
	/* setend  */ P32(0xffffffff), P32(0xf1010200), 's', 'e', 't', 'e', 'n', 'd', '\t', 'B', 'E', 0, 0, 0, 0,
	/* sev     */ P32(0x0fffffff), P32(0x0320f004), 's', 'e', 'v', S_c, 0, 0, 0, 0,
	/* shadd16 */ P32(0x0ff00ff0), P32(0x06300f10), 's', 'h', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* shadd8  */ P32(0x0ff00ff0), P32(0x06300f90), 's', 'h', 'a', 'd', 'd', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* shasx   */ P32(0x0ff00ff0), P32(0x06300f30), 's', 'h', 'a', 's', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* shsax   */ P32(0x0ff00ff0), P32(0x06300f50), 's', 'h', 's', 'a', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* shsub16 */ P32(0x0ff00ff0), P32(0x06300f70), 's', 'h', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* shsub8  */ P32(0x0ff00ff0), P32(0x06300ff0), 's', 'h', 's', 'u', 'b', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* smlatt  */ P32(0x0ff000f0), P32(0x010000e0), 's', 'm', 'l', 'a', 't', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smlabt  */ P32(0x0ff000f0), P32(0x010000c0), 's', 'm', 'l', 'a', 'b', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smlatb  */ P32(0x0ff000f0), P32(0x010000a0), 's', 'm', 'l', 'a', 't', 'b', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smlabb  */ P32(0x0ff000f0), P32(0x01000080), 's', 'm', 'l', 'a', 'b', 'b', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smuadx  */ P32(0x0ff0f0f0), P32(0x0700f030), 's', 'm', 'u', 'a', 'd', 'x', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smladx  */ P32(0x0ff000f0), P32(0x07000030), 's', 'm', 'l', 'a', 'd', 'x', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smuad   */ P32(0x0ff0f0f0), P32(0x0700f010), 's', 'm', 'u', 'a', 'd', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smlad   */ P32(0x0ff000f0), P32(0x07000010), 's', 'm', 'l', 'a', 'd', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0,
	/* smlaltt */ P32(0x0ff000f0), P32(0x014000e0), 's', 'm', 'l', 'a', 'l', 't', 't', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smlalbt */ P32(0x0ff000f0), P32(0x014000c0), 's', 'm', 'l', 'a', 'l', 'b', 't', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smlaltb */ P32(0x0ff000f0), P32(0x014000a0), 's', 'm', 'l', 'a', 'l', 't', 'b', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smlalbb */ P32(0x0ff000f0), P32(0x01400080), 's', 'm', 'l', 'a', 'l', 'b', 'b', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smlaldx */ P32(0x0ff000f0), P32(0x07400030), 's', 'm', 'l', 'a', 'l', 'd', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smlald  */ P32(0x0ff000f0), P32(0x07400010), 's', 'm', 'l', 'a', 'l', 'd', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smlal   */ P32(0x0fe000f0), P32(0x00e00090), 's', 'm', 'l', 'a', 'l', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smlawt  */ P32(0x0ff000f0), P32(0x012000c0), 's', 'm', 'l', 'a', 'w', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smlawb  */ P32(0x0ff000f0), P32(0x01200080), 's', 'm', 'l', 'a', 'w', 'b', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smusdx  */ P32(0x0ff0f0f0), P32(0x0700f070), 's', 'm', 'u', 's', 'd', 'x', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smlsdt  */ P32(0x0ff000f0), P32(0x07000070), 's', 'm', 'l', 's', 'd', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smusd   */ P32(0x0ff0f0f0), P32(0x0700f050), 's', 'm', 'u', 's', 'd', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smlsdb  */ P32(0x0ff000f0), P32(0x07000050), 's', 'm', 'l', 's', 'd', 'b', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smlsldt */ P32(0x0ff000f0), P32(0x07400070), 's', 'm', 'l', 's', 'l', 'd', 't', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smlsldb */ P32(0x0ff000f0), P32(0x07400050), 's', 'm', 'l', 's', 'l', 'd', 'b', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smmulr  */ P32(0x0ff0f0f0), P32(0x0750f030), 's', 'm', 'm', 'u', 'l', 'r', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smmlar  */ P32(0x0ff000f0), P32(0x07500030), 's', 'm', 'm', 'l', 'a', 'r', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smmul   */ P32(0x0ff0f0f0), P32(0x0750f010), 's', 'm', 'm', 'u', 'l', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smmla   */ P32(0x0ff000f0), P32(0x07500010), 's', 'm', 'm', 'l', 'a', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0,
	/* smmlsr  */ P32(0x0ff000f0), P32(0x075000f0), 's', 'm', 'm', 'l', 's', 'r', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* smmls   */ P32(0x0ff000f0), P32(0x075000d0), 's', 'm', 'm', 'l', 's', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0,
	/* smultt  */ P32(0x0ff000f0), P32(0x016000e0), 's', 'm', 'u', 'l', 't', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smulbt  */ P32(0x0ff000f0), P32(0x016000c0), 's', 'm', 'u', 'l', 'b', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smultb  */ P32(0x0ff000f0), P32(0x016000a0), 's', 'm', 'u', 'l', 't', 'b', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smulbb  */ P32(0x0ff000f0), P32(0x01600080), 's', 'm', 'u', 'l', 'b', 'b', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smull   */ P32(0x0fe000f0), P32(0x00c00090), 's', 'm', 'u', 'l', 'l', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* smulwt  */ P32(0x0ff000f0), P32(0x012000e0), 's', 'm', 'u', 'l', 'w', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* smulwb  */ P32(0x0ff000f0), P32(0x012000a0), 's', 'm', 'u', 'l', 'w', 'b', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* ssat16  */ P32(0x0ff00ff0), P32(0x06a00f30), 's', 's', 'a', 't', '1', '6', S_c, '\t', O_R_S12N4, O_IMMP1_S16N4, O_R_S0N4, 0, 0,
	/* ssat    */ P32(0x0fe00030), P32(0x06a00010), 's', 's', 'a', 't', S_c, '\t', O_R_S12N4, O_IMM_S7N5, O_R_S0N4, O_OPT_SHIFT_S6N1, 0, 0,
	/* ssax    */ P32(0x0ff00ff0), P32(0x06100f50), 's', 's', 'a', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* ssub16  */ P32(0x0ff00ff0), P32(0x06100f70), 's', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* ssub8   */ P32(0x0ff00ff0), P32(0x06100ff0), 's', 's', 'u', 'b', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* stm     */ P32(0x0fd00000), P32(0x08800000), 's', 't', 'm', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N16, 0, 0, 0, 0,
	/* stmda   */ P32(0x0fd00000), P32(0x08000000), 's', 't', 'm', 'd', 'a', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N16, 0, 0,
	/* stmib   */ P32(0x0fd00000), P32(0x09800000), 's', 't', 'm', 'i', 'b', S_c, '\t', O_R_S16N4, O_OPT_IA, O_RLIST_S0N16, 0, 0,
	/* str     */ P32(0x0f500010), P32(0x07000000), 's', 't', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, ']', O_OPT_IA, 0, 0, 0, 0,
	/* str     */ P32(0x0f700010), P32(0x06000000), 's', 't', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* str     */ P32(0x0f700000), P32(0x04000000), 's', 't', 'r', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N12, 0, 0,
	/* strb    */ P32(0x0f500010), P32(0x07400000), 's', 't', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, ']', O_OPT_IA, 0, 0,
	/* strb    */ P32(0x0f700010), P32(0x06400000), 's', 't', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* strb    */ P32(0x0f500000), P32(0x05400000), 's', 't', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_S0N12, ']', O_OPT_IA, 0, 0, 0, 0,
	/* strb    */ P32(0x0f700000), P32(0x04400000), 's', 't', 'r', 'b', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N12, 0, 0, 0, 0,
	/* strbt   */ P32(0x0f700000), P32(0x04600000), 's', 't', 'r', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_S0N12, 0, 0, 0, 0,
	/* strbt   */ P32(0x0f700010), P32(0x06600000), 's', 't', 'r', 'b', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* strd    */ P32(0x0f500ff0), P32(0x010000f0), 's', 't', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, ']', O_OPT_IA, 0, 0,
	/* strd    */ P32(0x0f700ff0), P32(0x000000f0), 's', 't', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0, 0, 0,
	/* strd    */ P32(0x0f5000f0), P32(0x014000f0), 's', 't', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, ']', O_OPT_IA, 0, 0,
	/* strd    */ P32(0x0f7000f0), P32(0x004000f0), 's', 't', 'r', 'd', S_c, '\t', O_R_S12N4, O_RP1_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0, 0, 0,
	/* strex   */ P32(0x0ff00ff0), P32(0x01800f90), 's', 't', 'r', 'e', 'x', S_c, '\t', O_R_S12N4, O_R_S0N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* strexb  */ P32(0x0ff00ff0), P32(0x01c00f90), 's', 't', 'r', 'e', 'x', 'b', S_c, '\t', O_R_S12N4, O_R_S0N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* strexd  */ P32(0x0ff00ff0), P32(0x01a00f90), 's', 't', 'r', 'e', 'x', 'd', S_c, '\t', O_R_S12N4, O_R_S0N4, O_RP1_S0N4, '[', O_R_S16N4, ']', 0, 0,
	/* strexh  */ P32(0x0ff00ff0), P32(0x01e00f90), 's', 't', 'r', 'e', 'x', 'h', S_c, '\t', O_R_S12N4, O_R_S0N4, '[', O_R_S16N4, ']', 0, 0, 0, 0,
	/* strh    */ P32(0x0f500ff0), P32(0x010000b0), 's', 't', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_R_S0N4, ']', O_OPT_IA, 0, 0, 0, 0,
	/* strh    */ P32(0x0f700ff0), P32(0x000000b0), 's', 't', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0, 0, 0,
	/* strh    */ P32(0x0f5000f0), P32(0x014000b0), 's', 't', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, ']', O_OPT_IA, 0, 0, 0, 0,
	/* strh    */ P32(0x0f7000f0), P32(0x004000b0), 's', 't', 'r', 'h', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0, 0, 0,
	/* strht   */ P32(0x0f7000f0), P32(0x006000b0), 's', 't', 'r', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_IMM_XXXX____XXXX, 0, 0, 0, 0,
	/* strht   */ P32(0x0f700ff0), P32(0x002000b0), 's', 't', 'r', 'h', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, 0, 0, 0, 0,
	/* strt    */ P32(0x0f700000), P32(0x04200000), 's', 't', 'r', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', '{', O_PLUSMINUS_S23N1, O_IMM_S0N12, '}', 0, 0,
	/* strt    */ P32(0x0f700010), P32(0x06200000), 's', 't', 'r', 't', S_c, '\t', O_R_S12N4, '[', O_R_S16N4, ']', O_PLUSMINUS_S23N1, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* sub     */ P32(0x0fe00090), P32(0x00400010), 's', 'u', 'b', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0,
	/* sub     */ P32(0x0fe00010), P32(0x00400000), 's', 'u', 'b', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0,
	/* svc     */ P32(0x0f000000), P32(0x0f000000), 's', 'v', 'c', S_c, '\t', O_IMM_S0N24, 0, 0,
	/* swp     */ P32(0x0ff00ff0), P32(0x01000090), 's', 'w', 'p', S_c, '\t', O_R_S12N4, O_R_S0N4, '[', O_R_S16N4, ']', 0, 0,
	/* swpb    */ P32(0x0ff00ff0), P32(0x01400090), 's', 'w', 'p', 'b', S_c, '\t', O_R_S12N4, O_R_S0N4, '[', O_R_S16N4, ']', 0, 0,
	/* sxtb16  */ P32(0x0fff03f0), P32(0x068f0070), 's', 'x', 't', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0,
	/* sxtab16 */ P32(0x0ff003f0), P32(0x06800070), 's', 'x', 't', 'a', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0, 0, 0,
	/* sxtb    */ P32(0x0fff03f0), P32(0x06af0070), 's', 'x', 't', 'b', S_c, '\t', O_R_S12N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0, 0, 0,
	/* sxtab   */ P32(0x0ff003f0), P32(0x06a00070), 's', 'x', 't', 'a', 'b', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0,
	/* sxth    */ P32(0x0fff03f0), P32(0x06bf0070), 's', 'x', 't', 'h', S_c, '\t', O_R_S12N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0, 0, 0,
	/* sxtah   */ P32(0x0ff003f0), P32(0x06b00070), 's', 'x', 't', 'a', 'h', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0,
	/* teq     */ P32(0x0ff0f000), P32(0x03300000), 't', 'e', 'q', S_c, '\t', O_R_S16N4, O_ARMEI_C_S0N12, 0, 0,
	/* teq     */ P32(0x0ff0f090), P32(0x01300010), 't', 'e', 'q', S_c, '\t', O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0, 0, 0,
	/* teq     */ P32(0x0ff0f010), P32(0x01300000), 't', 'e', 'q', S_c, '\t', O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* tst     */ P32(0x0ff0f000), P32(0x03100000), 't', 's', 't', S_c, '\t', O_R_S16N4, O_ARMEI_C_S0N12, 0, 0,
	/* tst     */ P32(0x0ff0f090), P32(0x01100010), 't', 's', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_TODO("type"), O_R_S8N4, 0, 0, 0, 0,
	/* tst     */ P32(0x0ff0f010), P32(0x01100000), 't', 's', 't', S_c, '\t', O_R_S16N4, O_R_S0N4, O_OPT_SHIFT_S5N2, 0, 0, 0, 0,
	/* uadd16  */ P32(0x0ff00ff0), P32(0x06500f10), 'u', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uadd8   */ P32(0x0ff00ff0), P32(0x06500f90), 'u', 'a', 'd', 'd', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uasx    */ P32(0x0ff00ff0), P32(0x06500f30), 'u', 'a', 's', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* ubfx    */ P32(0x0fe00070), P32(0x07e00050), 'u', 'b', 'f', 'x', S_c, '\t', O_R_S12N4, O_R_S0N4, '#', O_TODO("lsb"), '#', O_TODO("width"), 0, 0, 0, 0,
	/* uhadd16 */ P32(0x0ff00ff0), P32(0x06700f10), 'u', 'h', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uhadd8  */ P32(0x0ff00ff0), P32(0x06700f90), 'u', 'h', 'a', 'd', 'd', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uhasx   */ P32(0x0ff00ff0), P32(0x06700f30), 'u', 'h', 'a', 's', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uhsax   */ P32(0x0ff00ff0), P32(0x06700f50), 'u', 'h', 's', 'a', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uhsub16 */ P32(0x0ff00ff0), P32(0x06700f70), 'u', 'h', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uhsub8  */ P32(0x0ff00ff0), P32(0x06700ff0), 'u', 'h', 's', 'u', 'b', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* umaal   */ P32(0x0ff000f0), P32(0x00400090), 'u', 'm', 'a', 'a', 'l', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* umlal   */ P32(0x0fe000f0), P32(0x00a00090), 'u', 'm', 'l', 'a', 'l', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* umull   */ P32(0x0fe000f0), P32(0x00800090), 'u', 'm', 'u', 'l', 'l', S_S, S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0, 0, 0,
	/* uqadd16 */ P32(0x0ff00ff0), P32(0x06600f10), 'u', 'q', 'a', 'd', 'd', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uqadd8  */ P32(0x0ff00ff0), P32(0x06600f90), 'u', 'q', 'a', 'd', 'd', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uqasx   */ P32(0x0ff00ff0), P32(0x06600f30), 'u', 'q', 'a', 's', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uqsax   */ P32(0x0ff00ff0), P32(0x06600f50), 'u', 'q', 's', 'a', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uqsub16 */ P32(0x0ff00ff0), P32(0x06600f70), 'u', 'q', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* uqsub8  */ P32(0x0ff00ff0), P32(0x06600ff0), 'u', 'q', 's', 'u', 'b', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* usad8   */ P32(0x0ff0f0f0), P32(0x0780f010), 'u', 's', 'a', 'd', '8', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, 0, 0,
	/* usada8  */ P32(0x0ff000f0), P32(0x07800010), 'u', 's', 'a', 'd', 'a', '8', S_c, '\t', O_R_S16N4, O_R_S0N4, O_R_S8N4, O_R_S12N4, 0, 0, 0, 0,
	/* usat16  */ P32(0x0ff00ff0), P32(0x06e00f30), 'u', 's', 'a', 't', '1', '6', S_c, '\t', O_R_S12N4, O_IMM_S16N4, O_R_S0N4, 0, 0,
	/* usat    */ P32(0x0fe00030), P32(0x06e00010), 'u', 's', 'a', 't', S_c, '\t', O_R_S12N4, '#', O_TODO("DecodeImmShift(sh0,imm)"), O_R_S0N4, O_OPT_SHIFT_, 0, 0,
	/* usax    */ P32(0x0ff00ff0), P32(0x06500f50), 'u', 's', 'a', 'x', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0, 0, 0,
	/* usub16  */ P32(0x0ff00ff0), P32(0x06500f70), 'u', 's', 'u', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* usub8   */ P32(0x0ff00ff0), P32(0x06500ff0), 'u', 's', 'u', 'b', '8', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, 0, 0,
	/* uxtb16  */ P32(0x0fff03f0), P32(0x06cf0070), 'u', 'x', 't', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0,
	/* uxtab16 */ P32(0x0ff003f0), P32(0x06c00070), 'u', 'x', 't', 'a', 'b', '1', '6', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0, 0, 0,
	/* uxtb    */ P32(0x0fff03f0), P32(0x06ef0070), 'u', 'x', 't', 'b', S_c, '\t', O_R_S12N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0, 0, 0,
	/* uxtab   */ P32(0x0ff003f0), P32(0x06e00070), 'u', 'x', 't', 'a', 'b', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0,
	/* uxth    */ P32(0x0fff03f0), P32(0x06ff0070), 'u', 'x', 't', 'h', S_c, '\t', O_R_S12N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0, 0, 0,
	/* uxtah   */ P32(0x0ff003f0), P32(0x06f00070), 'u', 'x', 't', 'a', 'h', S_c, '\t', O_R_S12N4, O_R_S16N4, O_R_S0N4, O_OPT_ROTATION_S10N2, 0, 0,
	/* wfe     */ P32(0x0fffffff), P32(0x0320f002), 'w', 'f', 'e', S_c, 0, 0, 0, 0,
	/* wfi     */ P32(0x0fffffff), P32(0x0320f003), 'w', 'f', 'i', S_c, 0, 0, 0, 0,
	/* yield   */ P32(0x0fffffff), P32(0x0320f001), 'y', 'i', 'e', 'l', 'd', S_c, 0, 0,
};
/* Database size:
 * groups: 6156 = 513 * 12
 * blob:   14324
 * total:  20480 (20.00KiB) */
/*[[[end]]]*/
/* clang-format on */

DECL_END

#endif /* !GUARD_LIBDISASM_ARM_DB_H */
