/* Copyright (c) 2019-2024 Griefer@Work                                       *
 *                                                                            *
 * This software is provided 'as-is', without any express or implied          *
 * warranty. In no event will the authors be held liable for any damages      *
 * arising from the use of this software.                                     *
 *                                                                            *
 * Permission is granted to anyone to use this software for any purpose,      *
 * including commercial applications, and to alter it and redistribute it     *
 * freely, subject to the following restrictions:                             *
 *                                                                            *
 * 1. The origin of this software must not be misrepresented; you must not    *
 *    claim that you wrote the original software. If you use this software    *
 *    in a product, an acknowledgement (see the following) in the product     *
 *    documentation is required:                                              *
 *    Portions Copyright (c) 2019-2024 Griefer@Work                           *
 * 2. Altered source versions must be plainly marked as such, and must not be *
 *    misrepresented as being the original software.                          *
 * 3. This notice may not be removed or altered from any source distribution. *
 */
#ifndef GUARD_MODSVGA_BASEFONT_C
#define GUARD_MODSVGA_BASEFONT_C 1
#define _KOS_SOURCE 1

#include <kernel/compiler.h>

#include "svga.h"

#include <stddef.h>

DECL_BEGIN


/* NOTE: ANSI tty for VGA requires colors as follows:
 *     [0x0] = { 0x00, 0x00, 0x00 }, [0x1] = { 0xaa, 0x00, 0x00 },
 *     [0x2] = { 0x00, 0xaa, 0x00 }, [0x3] = { 0xaa, 0x55, 0x00 },
 *     [0x4] = { 0x00, 0x00, 0xaa }, [0x5] = { 0xaa, 0x00, 0xaa },
 *     [0x6] = { 0x00, 0xaa, 0xaa }, [0x7] = { 0xaa, 0xaa, 0xaa },
 *     [0x8] = { 0x55, 0x55, 0x55 }, [0x9] = { 0xff, 0x55, 0x55 },
 *     [0xa] = { 0x55, 0xff, 0x55 }, [0xb] = { 0xff, 0xff, 0x55 },
 *     [0xc] = { 0x55, 0x55, 0xff }, [0xd] = { 0xff, 0x55, 0xff },
 *     [0xe] = { 0x55, 0xff, 0xff }, [0xf] = { 0xff, 0xff, 0xff },
 * All of these colors are present in CGA mode, however their indices differ.
 * As such, use the palette index registers to remap them:
 *    CGA color palette indices for ANSI color codes (for `vm_att_pal'). */

/* Default text-mode palette colors for KOS (these are set-up for ANSI TTY compatibility) */
INTERN_CONST struct vga_palcolor const basevga_defaultpal[16] = {
#define RGB(r, g, b) { r >> 2, g >> 2, b >> 2 }
	[0x0] = RGB(0x00, 0x00, 0x00),
	[0x1] = RGB(0xaa, 0x00, 0x00),
	[0x2] = RGB(0x00, 0xaa, 0x00),
	[0x3] = RGB(0xaa, 0x55, 0x00),
	[0x4] = RGB(0x00, 0x00, 0xaa),
	[0x5] = RGB(0xaa, 0x00, 0xaa),
	[0x6] = RGB(0x00, 0xaa, 0xaa),
	[0x7] = RGB(0xaa, 0xaa, 0xaa),
	[0x8] = RGB(0x55, 0x55, 0x55),
	[0x9] = RGB(0xff, 0x55, 0x55),
	[0xa] = RGB(0x55, 0xff, 0x55),
	[0xb] = RGB(0xff, 0xff, 0x55),
	[0xc] = RGB(0x55, 0x55, 0xff),
	[0xd] = RGB(0xff, 0x55, 0xff),
	[0xe] = RGB(0x55, 0xff, 0xff),
	[0xf] = RGB(0xff, 0xff, 0xff),
#undef RGB
};

/* Monochrome emulation palette. */
INTERN_CONST struct vga_palcolor const basevga_monopal[16] = {
#define RGB(r, g, b) { r >> 2, g >> 2, b >> 2 }
	/* Everything is white except for palette entry #0 */
	[0x0] = RGB(0x00, 0x00, 0x00),
	[0x1] = RGB(0xff, 0xff, 0xff),
	[0x2] = RGB(0xff, 0xff, 0xff),
	[0x3] = RGB(0xff, 0xff, 0xff),
	[0x4] = RGB(0xff, 0xff, 0xff),
	[0x5] = RGB(0xff, 0xff, 0xff),
	[0x6] = RGB(0xff, 0xff, 0xff),
	[0x7] = RGB(0xff, 0xff, 0xff),
	[0x8] = RGB(0xff, 0xff, 0xff),
	[0x9] = RGB(0xff, 0xff, 0xff),
	[0xa] = RGB(0xff, 0xff, 0xff),
	[0xb] = RGB(0xff, 0xff, 0xff),
	[0xc] = RGB(0xff, 0xff, 0xff),
	[0xd] = RGB(0xff, 0xff, 0xff),
	[0xe] = RGB(0xff, 0xff, 0xff),
	[0xf] = RGB(0xff, 0xff, 0xff),
#undef RGB
};



/* Map from `VGA_SR03_CSET_*' to the actual offset within plane #2. */
INTERN_CONST uint16_t const basevga_fontoffset[8] = {
	[VGA_SR03_CSET_0000_1FFF] = 0x0000,
	[VGA_SR03_CSET_4000_5FFF] = 0x4000,
	[VGA_SR03_CSET_8000_9FFF] = 0x8000,
	[VGA_SR03_CSET_C000_DFFF] = 0xC000,
	[VGA_SR03_CSET_2000_3FFF] = 0x2000,
	[VGA_SR03_CSET_6000_7FFF] = 0x6000,
	[VGA_SR03_CSET_A000_BFFF] = 0xA000,
	[VGA_SR03_CSET_E000_FFFF] = 0xE000,
};



#define CPVGA_ASCII_MIN 0x20 /* First ASCII-compatible ordinal */
#define CPVGA_ASCII_MAX 0x7E /* Last ASCII-compatible ordinal */

struct cpvga_encode_entry {
	char16_t cee_uni; /* Unicode ordinal */
	byte_t   cee_vga; /* VGA ordinal. */
	byte_t  _cee_pad; /* ... */
};


/* DISCLAIMER: parts of this font data have been generated from univga:
 * https://star.inp.nsk.su/~bolkhov/files/fonts/univga/ */
/*[[[deemon
// Somewhat based on cp437, but altered, including more box drawing
// stuff and less greek/math stuff. This is nothing official and this
// code page in particular cannot be found in official listings as it
// has been created for the sole purpose of existing as KOS's display
// VGA codepage
//
// There may or may not be an easteregg in here as well... ;)
local CPVGA = {
	0xFFFD, 0x263A, 0x263B, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
	0x25D8, 0x25CB, 0x25D9, 0x2642, 0x2640, 0x266A, 0x266B, 0x263C,
	0x25BA, 0x25C4, 0x2195, 0x203C, 0x00B6, 0x00A7, 0x25AC, 0x21A8,
	0x2191, 0x2193, 0x2192, 0x2190, 0x221F, 0x2194, 0x25B2, 0x25BC,
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005A, 0x005B, 0x005C, 0x005D, 0x005E, 0x005F,
	0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x007B, 0x007C, 0x007D, 0x007E, 0x2302,
	0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7,
	0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x00EC, 0x00C4, 0x00C5,
	0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9,
	0x00FF, 0x00D6, 0x00DC, 0x00A2, 0x00A3, 0x00A5, 0x20A7, 0x0192,
	0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA,
	0x00BF, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510,
	0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F,
	0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567,
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B,
	0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580,
	0x03B1, 0x00DF, 0x2596, 0x2597, 0x2598, 0x2599, 0x259A, 0x259B,
	0x259C, 0x259D, 0x259E, 0x259F, 0x221E, 0x03C6, 0x03B5, 0x2229,
	0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248,
	0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0xE888,
};

import * from deemon;
import * from ....misc.libgen.libtlft.compiler;
import uniEscape from ...libiconv.iconvdata.util;

local tlft;
with (local fp = File.open("../../../misc/fonts/tlft/uni_vga/u_vga16.bdf"))
	tlft = Dict(tlftParseBDF(fp).map);

// A couple of glyphs not ~really~ contained (the way we want them to be) in uni_vga
tlft[0x2588] = Bytes({ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }); // █
tlft[0x2584] = Bytes({ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }); // ▄
tlft[0x258C] = Bytes({ 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 }); // ▌
tlft[0x2590] = Bytes({ 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F }); // ▐
tlft[0x2580] = Bytes({ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }); // ▀
tlft[0x2596] = Bytes({ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 }); // ▖
tlft[0x2597] = Bytes({ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F }); // ▗
tlft[0x2598] = Bytes({ 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }); // ▘
tlft[0x2599] = Bytes({ 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }); // ▙
tlft[0x259A] = Bytes({ 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F }); // ▚
tlft[0x259B] = Bytes({ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 }); // ▛
tlft[0x259C] = Bytes({ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F }); // ▜
tlft[0x259D] = Bytes({ 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }); // ▝
tlft[0x259E] = Bytes({ 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 }); // ▞
tlft[0x259F] = Bytes({ 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }); // ▟
tlft[0xE888] = Bytes({ 0x10, 0x28, 0x44, 0x92, 0xAA, 0xAA, 0xA2, 0x54, 0x8A, 0xAA, 0xAA, 0x92, 0x44, 0x28, 0x10, 0x00 }); // It's a secret...

print("INTERN_CONST byte_t const basevga_defaultfont[256][32] = {");
for (local i: [:256]) {
	local ord = CPVGA[i];
	local bytes = tlft.get(ord);
	if (bytes is none)
		bytes = Bytes(16, 0);
	if (#bytes != 16)
		bytes = bytes.resized(16);
	print("\t{ ", ", ".join(for (local b: bytes) "0x" + b.tostr(16, 2, "X")), " }, "
		"/" "* U+", ord.tostr(16, 4, "X"), " ", uniEscape(ord), " *" "/");
}
print("};");
local uniToCpVGA: {int: int} = Dict();
for (local i: [:256]) {
	if (i >= CPVGA_ASCII_MIN && i <= CPVGA_ASCII_MAX)
		assert CPVGA[i] == i;
	uniToCpVGA[CPVGA[i]] = i;
}

// Unicode -> vga w/ transliteration aliases
local final UNI_EQ_PAIRS: {(int, int)...} = {

	// Space characters
	(0x00A0, 0x0020), // 𝘕𝘉𝘚𝘗 → ␠
	(0x2007, 0x0020), //   → ␠

	// Pointing-triang characters
	(0x25B2, 0x25B4), (0x25B2, 0x25B3), // ▲ → ▴
	(0x25BA, 0x25B6), (0x25BA, 0x25B8), // ► → ▶, ▸
	(0x25BC, 0x25BE),                   // ▼ → ▾
	(0x25C4, 0x25C0), (0x25BA, 0x25C2), // ◄ → ◀, ◂
	(0x25A0, 0x25FC)
};

for (local a, b: UNI_EQ_PAIRS) {
	if (a !in uniToCpVGA && b in uniToCpVGA) {
		uniToCpVGA[a] = uniToCpVGA[b];
	} else if (b !in uniToCpVGA && a in uniToCpVGA) {
		uniToCpVGA[b] = uniToCpVGA[a];
	}
}

print("PRIVATE struct cpvga_encode_entry const basevga_encode_db[] = {");
for (local uni: uniToCpVGA.keys.sorted()) {
	if (uni >= CPVGA_ASCII_MIN && uni <= CPVGA_ASCII_MAX)
		continue; // Handled explicitly
	assert uni <= 0xffff;
	print("	{ 0x", uni.tostr(16, 4, "X"), ", 0x", uniToCpVGA[uni].tostr(16, 4, "X"),
		" }, /" "* ", uniEscape(uni)),;
	print(" *" "/");
}
print("};");
]]]*/
INTERN_CONST byte_t const basevga_defaultfont[256][32] = {
	{ 0x00, 0x38, 0x7C, 0x7C, 0xC6, 0x92, 0xF2, 0xE6, 0xFE, 0xE6, 0x7C, 0x7C, 0x38, 0x00, 0x00, 0x00 }, /* U+FFFD � */
	{ 0x00, 0x00, 0x7E, 0x81, 0xA5, 0x81, 0x81, 0xA5, 0x99, 0x81, 0x81, 0x7E, 0x00, 0x00, 0x00, 0x00 }, /* U+263A ☺ */
	{ 0x00, 0x00, 0x7E, 0xFF, 0xDB, 0xFF, 0xFF, 0xDB, 0xE7, 0xFF, 0xFF, 0x7E, 0x00, 0x00, 0x00, 0x00 }, /* U+263B ☻ */
	{ 0x00, 0x00, 0x00, 0x00, 0x6C, 0xFE, 0xFE, 0xFE, 0xFE, 0x7C, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00 }, /* U+2665 ♥ */
	{ 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7C, 0xFE, 0x7C, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2666 ♦ */
	{ 0x00, 0x00, 0x00, 0x18, 0x3C, 0x3C, 0xE7, 0xE7, 0xE7, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+2663 ♣ */
	{ 0x00, 0x00, 0x00, 0x18, 0x3C, 0x7E, 0xFF, 0xFF, 0x7E, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+2660 ♠ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3C, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2022 • */
	{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xC3, 0xC3, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }, /* U+25D8 ◘ */
	{ 0x00, 0x00, 0x3C, 0x42, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x42, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+25CB ○ */
	{ 0xFF, 0xFF, 0xC3, 0xBD, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0xBD, 0xC3, 0xFF, 0xFF, 0xFF, 0xFF }, /* U+25D9 ◙ */
	{ 0x00, 0x00, 0x1E, 0x06, 0x0E, 0x1A, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x00, 0x00, 0x00 }, /* U+2642 ♂ */
	{ 0x00, 0x00, 0x3C, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x18, 0x7E, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+2640 ♀ */
	{ 0x00, 0x00, 0x30, 0x3C, 0x3E, 0x32, 0x30, 0x30, 0x30, 0x70, 0xF0, 0xE0, 0x00, 0x00, 0x00, 0x00 }, /* U+266A ♪ */
	{ 0x00, 0x00, 0x70, 0x7F, 0x6F, 0x63, 0x63, 0x63, 0x63, 0xE3, 0xE7, 0xC7, 0x06, 0x00, 0x00, 0x00 }, /* U+266B ♫ */
	{ 0x00, 0x00, 0x00, 0x10, 0x10, 0x54, 0x28, 0xC6, 0x28, 0x54, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00 }, /* U+263C ☼ */
	{ 0x00, 0x00, 0x00, 0x00, 0x80, 0xE0, 0xF8, 0xFE, 0xF8, 0xE0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+25BA ► */
	{ 0x00, 0x00, 0x00, 0x00, 0x02, 0x0E, 0x3E, 0xFE, 0x3E, 0x0E, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+25C4 ◄ */
	{ 0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2195 ↕ */
	{ 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00 }, /* U+203C ‼ */
	{ 0x00, 0x00, 0x7F, 0xDB, 0xDB, 0xDB, 0x7B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x00, 0x00, 0x00, 0x00 }, /* U+00B6 ¶ */
	{ 0x00, 0x7C, 0xC6, 0x60, 0x38, 0x6C, 0xC6, 0xC6, 0x6C, 0x38, 0x0C, 0xC6, 0x7C, 0x00, 0x00, 0x00 }, /* U+00A7 § */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+25AC ▬ */
	{ 0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x7E, 0x00, 0x00, 0x00, 0x00 }, /* U+21A8 ↨ */
	{ 0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+2191 ↑ */
	{ 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+2193 ↓ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x06, 0xFF, 0x06, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2192 → */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xFF, 0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2190 ← */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+221F ∟ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xFF, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2194 ↔ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7C, 0x7C, 0xFE, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+25B2 ▲ */
	{ 0x00, 0x00, 0xFE, 0xFE, 0x7C, 0x7C, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+25BC ▼ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+0020 ␠ */
	{ 0x00, 0x00, 0x18, 0x3C, 0x3C, 0x3C, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+0021 ! */
	{ 0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+0022 " */
	{ 0x00, 0x00, 0x00, 0x6C, 0x6C, 0xFE, 0x6C, 0x6C, 0x6C, 0xFE, 0x6C, 0x6C, 0x00, 0x00, 0x00, 0x00 }, /* U+0023 # */
	{ 0x18, 0x18, 0x7C, 0xC6, 0xC2, 0xC0, 0x7C, 0x06, 0x06, 0x86, 0xC6, 0x7C, 0x18, 0x18, 0x00, 0x00 }, /* U+0024 $ */
	{ 0x00, 0x00, 0x00, 0x00, 0xC2, 0xC6, 0x0C, 0x18, 0x30, 0x60, 0xC6, 0x86, 0x00, 0x00, 0x00, 0x00 }, /* U+0025 % */
	{ 0x00, 0x00, 0x38, 0x6C, 0x6C, 0x38, 0x76, 0xDC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+0026 & */
	{ 0x00, 0x30, 0x30, 0x30, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+0027 ' */
	{ 0x00, 0x00, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x18, 0x0C, 0x00, 0x00, 0x00, 0x00 }, /* U+0028 ( */
	{ 0x00, 0x00, 0x30, 0x18, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00 }, /* U+0029 ) */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3C, 0xFF, 0x3C, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+002A * */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7E, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+002B + */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00 }, /* U+002C , */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+002D - */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+002E . */
	{ 0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0x80, 0x00, 0x00, 0x00, 0x00 }, /* U+002F / */
	{ 0x00, 0x00, 0x38, 0x6C, 0xC6, 0xC6, 0xD6, 0xD6, 0xC6, 0xC6, 0x6C, 0x38, 0x00, 0x00, 0x00, 0x00 }, /* U+0030 0 */
	{ 0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7E, 0x00, 0x00, 0x00, 0x00 }, /* U+0031 1 */
	{ 0x00, 0x00, 0x7C, 0xC6, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xC6, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+0032 2 */
	{ 0x00, 0x00, 0x7C, 0xC6, 0x06, 0x06, 0x3C, 0x06, 0x06, 0x06, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0033 3 */
	{ 0x00, 0x00, 0x0C, 0x1C, 0x3C, 0x6C, 0xCC, 0xFE, 0x0C, 0x0C, 0x0C, 0x1E, 0x00, 0x00, 0x00, 0x00 }, /* U+0034 4 */
	{ 0x00, 0x00, 0xFE, 0xC0, 0xC0, 0xC0, 0xFC, 0x06, 0x06, 0x06, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0035 5 */
	{ 0x00, 0x00, 0x38, 0x60, 0xC0, 0xC0, 0xFC, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0036 6 */
	{ 0x00, 0x00, 0xFE, 0xC6, 0x06, 0x06, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00 }, /* U+0037 7 */
	{ 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0038 8 */
	{ 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0x7E, 0x06, 0x06, 0x06, 0x0C, 0x78, 0x00, 0x00, 0x00, 0x00 }, /* U+0039 9 */
	{ 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+003A : */
	{ 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00 }, /* U+003B ; */
	{ 0x00, 0x00, 0x00, 0x06, 0x0C, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x00, 0x00, 0x00, 0x00 }, /* U+003C < */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+003D = */
	{ 0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x0C, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00 }, /* U+003E > */
	{ 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0x0C, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+003F ? */
	{ 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xDE, 0xDE, 0xDE, 0xDC, 0xC0, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0040 @ */
	{ 0x00, 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+0041 A */
	{ 0x00, 0x00, 0xFC, 0x66, 0x66, 0x66, 0x7C, 0x66, 0x66, 0x66, 0x66, 0xFC, 0x00, 0x00, 0x00, 0x00 }, /* U+0042 B */
	{ 0x00, 0x00, 0x3C, 0x66, 0xC2, 0xC0, 0xC0, 0xC0, 0xC0, 0xC2, 0x66, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+0043 C */
	{ 0x00, 0x00, 0xF8, 0x6C, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x6C, 0xF8, 0x00, 0x00, 0x00, 0x00 }, /* U+0044 D */
	{ 0x00, 0x00, 0xFE, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x62, 0x66, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+0045 E */
	{ 0x00, 0x00, 0xFE, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x60, 0x60, 0xF0, 0x00, 0x00, 0x00, 0x00 }, /* U+0046 F */
	{ 0x00, 0x00, 0x3C, 0x66, 0xC2, 0xC0, 0xC0, 0xDE, 0xC6, 0xC6, 0x66, 0x3A, 0x00, 0x00, 0x00, 0x00 }, /* U+0047 G */
	{ 0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+0048 H */
	{ 0x00, 0x00, 0x3C, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+0049 I */
	{ 0x00, 0x00, 0x1E, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x00, 0x00, 0x00 }, /* U+004A J */
	{ 0x00, 0x00, 0xE6, 0x66, 0x66, 0x6C, 0x78, 0x78, 0x6C, 0x66, 0x66, 0xE6, 0x00, 0x00, 0x00, 0x00 }, /* U+004B K */
	{ 0x00, 0x00, 0xF0, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x62, 0x66, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+004C L */
	{ 0x00, 0x00, 0xC6, 0xEE, 0xFE, 0xFE, 0xD6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+004D M */
	{ 0x00, 0x00, 0xC6, 0xE6, 0xF6, 0xFE, 0xDE, 0xCE, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+004E N */
	{ 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+004F O */
	{ 0x00, 0x00, 0xFC, 0x66, 0x66, 0x66, 0x7C, 0x60, 0x60, 0x60, 0x60, 0xF0, 0x00, 0x00, 0x00, 0x00 }, /* U+0050 P */
	{ 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xD6, 0xDE, 0x7C, 0x0C, 0x0E, 0x00, 0x00 }, /* U+0051 Q */
	{ 0x00, 0x00, 0xFC, 0x66, 0x66, 0x66, 0x7C, 0x6C, 0x66, 0x66, 0x66, 0xE6, 0x00, 0x00, 0x00, 0x00 }, /* U+0052 R */
	{ 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0x60, 0x38, 0x0C, 0x06, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0053 S */
	{ 0x00, 0x00, 0x7E, 0x7E, 0x5A, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+0054 T */
	{ 0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0055 U */
	{ 0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x6C, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00 }, /* U+0056 V */
	{ 0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xD6, 0xD6, 0xD6, 0xFE, 0xEE, 0x6C, 0x00, 0x00, 0x00, 0x00 }, /* U+0057 W */
	{ 0x00, 0x00, 0xC6, 0xC6, 0x6C, 0x7C, 0x38, 0x38, 0x7C, 0x6C, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+0058 X */
	{ 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+0059 Y */
	{ 0x00, 0x00, 0xFE, 0xC6, 0x86, 0x0C, 0x18, 0x30, 0x60, 0xC2, 0xC6, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+005A Z */
	{ 0x00, 0x00, 0x3C, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+005B [ */
	{ 0x00, 0x00, 0x00, 0x80, 0xC0, 0xE0, 0x70, 0x38, 0x1C, 0x0E, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00 }, /* U+005C \ */
	{ 0x00, 0x00, 0x3C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+005D ] */
	{ 0x10, 0x38, 0x6C, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+005E ^ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00 }, /* U+005F _ */
	{ 0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+0060 ` */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+0061 a */
	{ 0x00, 0x00, 0xE0, 0x60, 0x60, 0x78, 0x6C, 0x66, 0x66, 0x66, 0x66, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0062 b */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC0, 0xC0, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0063 c */
	{ 0x00, 0x00, 0x1C, 0x0C, 0x0C, 0x3C, 0x6C, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+0064 d */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xFE, 0xC0, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0065 e */
	{ 0x00, 0x00, 0x38, 0x6C, 0x64, 0x60, 0xF0, 0x60, 0x60, 0x60, 0x60, 0xF0, 0x00, 0x00, 0x00, 0x00 }, /* U+0066 f */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x0C, 0xCC, 0x78, 0x00 }, /* U+0067 g */
	{ 0x00, 0x00, 0xE0, 0x60, 0x60, 0x6C, 0x76, 0x66, 0x66, 0x66, 0x66, 0xE6, 0x00, 0x00, 0x00, 0x00 }, /* U+0068 h */
	{ 0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+0069 i */
	{ 0x00, 0x00, 0x06, 0x06, 0x00, 0x0E, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3C, 0x00 }, /* U+006A j */
	{ 0x00, 0x00, 0xE0, 0x60, 0x60, 0x66, 0x6C, 0x78, 0x78, 0x6C, 0x66, 0xE6, 0x00, 0x00, 0x00, 0x00 }, /* U+006B k */
	{ 0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+006C l */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xEC, 0xFE, 0xD6, 0xD6, 0xD6, 0xD6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+006D m */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xDC, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00 }, /* U+006E n */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+006F o */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xDC, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7C, 0x60, 0x60, 0xF0, 0x00 }, /* U+0070 p */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x0C, 0x0C, 0x1E, 0x00 }, /* U+0071 q */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xDC, 0x76, 0x66, 0x60, 0x60, 0x60, 0xF0, 0x00, 0x00, 0x00, 0x00 }, /* U+0072 r */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0x60, 0x38, 0x0C, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+0073 s */
	{ 0x00, 0x00, 0x10, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x30, 0x30, 0x36, 0x1C, 0x00, 0x00, 0x00, 0x00 }, /* U+0074 t */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+0075 u */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+0076 v */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xD6, 0xD6, 0xD6, 0xFE, 0x6C, 0x00, 0x00, 0x00, 0x00 }, /* U+0077 w */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0x6C, 0x38, 0x38, 0x38, 0x6C, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+0078 x */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7E, 0x06, 0x0C, 0xF8, 0x00 }, /* U+0079 y */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xCC, 0x18, 0x30, 0x60, 0xC6, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+007A z */
	{ 0x00, 0x00, 0x0E, 0x18, 0x18, 0x18, 0x70, 0x18, 0x18, 0x18, 0x18, 0x0E, 0x00, 0x00, 0x00, 0x00 }, /* U+007B { */
	{ 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+007C | */
	{ 0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x0E, 0x18, 0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00, 0x00 }, /* U+007D } */
	{ 0x00, 0x00, 0x76, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+007E ~ */
	{ 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xC6, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2302 ⌂ */
	{ 0x00, 0x00, 0x3C, 0x66, 0xC2, 0xC0, 0xC0, 0xC0, 0xC0, 0xC2, 0x66, 0x3C, 0x18, 0x0C, 0x38, 0x00 }, /* U+00C7 Ç */
	{ 0x00, 0x00, 0xCC, 0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00FC ü */
	{ 0x00, 0x00, 0x0C, 0x18, 0x00, 0x7C, 0xC6, 0xFE, 0xC0, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00E9 é */
	{ 0x00, 0x10, 0x38, 0x6C, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00E2 â */
	{ 0x00, 0x00, 0x6C, 0x6C, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00E4 ä */
	{ 0x00, 0x00, 0x60, 0x30, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00E0 à */
	{ 0x00, 0x38, 0x6C, 0x38, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00E5 å */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC0, 0xC0, 0xC0, 0xC6, 0x7C, 0x18, 0x0C, 0x38, 0x00 }, /* U+00E7 ç */
	{ 0x00, 0x10, 0x38, 0x6C, 0x00, 0x7C, 0xC6, 0xFE, 0xC0, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00EA ê */
	{ 0x00, 0x00, 0x6C, 0x6C, 0x00, 0x7C, 0xC6, 0xFE, 0xC0, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00EB ë */
	{ 0x00, 0x00, 0x60, 0x30, 0x00, 0x7C, 0xC6, 0xFE, 0xC0, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00E8 è */
	{ 0x00, 0x00, 0x66, 0x66, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+00EF ï */
	{ 0x00, 0x10, 0x38, 0x6C, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+00EE î */
	{ 0x00, 0x00, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+00EC ì */
	{ 0x6C, 0x6C, 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+00C4 Ä */
	{ 0x38, 0x6C, 0x38, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+00C5 Å */
	{ 0x0C, 0x18, 0x00, 0xFE, 0x66, 0x62, 0x68, 0x78, 0x68, 0x62, 0x66, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+00C9 É */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xCC, 0x76, 0x36, 0x7E, 0xD8, 0xD8, 0x6E, 0x00, 0x00, 0x00, 0x00 }, /* U+00E6 æ */
	{ 0x00, 0x00, 0x3E, 0x6C, 0xCC, 0xCC, 0xFE, 0xCC, 0xCC, 0xCC, 0xCC, 0xCE, 0x00, 0x00, 0x00, 0x00 }, /* U+00C6 Æ */
	{ 0x00, 0x10, 0x38, 0x6C, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00F4 ô */
	{ 0x00, 0x00, 0x6C, 0x6C, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00F6 ö */
	{ 0x00, 0x00, 0x60, 0x30, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00F2 ò */
	{ 0x00, 0x10, 0x38, 0x6C, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00FB û */
	{ 0x00, 0x00, 0x60, 0x30, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00F9 ù */
	{ 0x00, 0x00, 0x6C, 0x6C, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7E, 0x06, 0x0C, 0xF8, 0x00 }, /* U+00FF ÿ */
	{ 0x6C, 0x6C, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00D6 Ö */
	{ 0x6C, 0x6C, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00DC Ü */
	{ 0x00, 0x18, 0x18, 0x3C, 0x66, 0x60, 0x60, 0x60, 0x66, 0x3C, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+00A2 ¢ */
	{ 0x00, 0x38, 0x6C, 0x64, 0x60, 0xF0, 0x60, 0x60, 0x60, 0x60, 0xE6, 0xFC, 0x00, 0x00, 0x00, 0x00 }, /* U+00A3 £ */
	{ 0x00, 0x00, 0x66, 0x66, 0x3C, 0x18, 0x7E, 0x18, 0x7E, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+00A5 ¥ */
	{ 0x00, 0x00, 0xFC, 0x66, 0x66, 0x7C, 0x62, 0x66, 0x6F, 0x66, 0x66, 0xF3, 0x00, 0x00, 0x00, 0x00 }, /* U+20A7 ₧ */
	{ 0x00, 0x00, 0x1C, 0x36, 0x32, 0x30, 0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0xE0, 0x00 }, /* U+0192 ƒ */
	{ 0x00, 0x00, 0x18, 0x30, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00E1 á */
	{ 0x00, 0x00, 0x0C, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00, 0x00 }, /* U+00ED í */
	{ 0x00, 0x00, 0x0C, 0x18, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00F3 ó */
	{ 0x00, 0x00, 0x18, 0x30, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+00FA ú */
	{ 0x00, 0x00, 0x76, 0xDC, 0x00, 0xDC, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00 }, /* U+00F1 ñ */
	{ 0x76, 0xDC, 0x00, 0xC6, 0xE6, 0xF6, 0xFE, 0xDE, 0xCE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+00D1 Ñ */
	{ 0x00, 0x3C, 0x6C, 0x6C, 0x3E, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00AA ª */
	{ 0x00, 0x38, 0x6C, 0x6C, 0x38, 0x00, 0x7C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00BA º */
	{ 0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60, 0xC0, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+00BF ¿ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2310 ⌐ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x06, 0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00AC ¬ */
	{ 0x00, 0xC0, 0xC0, 0xC2, 0xC6, 0xCC, 0x18, 0x30, 0x60, 0xDC, 0x86, 0x0C, 0x18, 0x3E, 0x00, 0x00 }, /* U+00BD ½ */
	{ 0x00, 0xC0, 0xC0, 0xC2, 0xC6, 0xCC, 0x18, 0x30, 0x66, 0xCE, 0x9E, 0x3E, 0x06, 0x06, 0x00, 0x00 }, /* U+00BC ¼ */
	{ 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x3C, 0x3C, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+00A1 ¡ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6C, 0xD8, 0x6C, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00AB « */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0xD8, 0x6C, 0x36, 0x6C, 0xD8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00BB » */
	{ 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44 }, /* U+2591 ░ */
	{ 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA }, /* U+2592 ▒ */
	{ 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77 }, /* U+2593 ▓ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+2502 │ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xF8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+2524 ┤ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xF8, 0x18, 0xF8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+2561 ╡ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xF6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2562 ╢ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2556 ╖ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x18, 0xF8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+2555 ╕ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xF6, 0x06, 0xF6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2563 ╣ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2551 ║ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x06, 0xF6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2557 ╗ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xF6, 0x06, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+255D ╝ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+255C ╜ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xF8, 0x18, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+255B ╛ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+2510 ┐ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2514 └ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2534 ┴ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+252C ┬ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1F, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+251C ├ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2500 ─ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xFF, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+253C ┼ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1F, 0x18, 0x1F, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+255E ╞ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+255F ╟ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+255A ╚ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3F, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2554 ╔ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xF7, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2569 ╩ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0xF7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2566 ╦ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2560 ╠ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2550 ═ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xF7, 0x00, 0xF7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+256C ╬ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xFF, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2567 ╧ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2568 ╨ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0xFF, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+2564 ╤ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2565 ╥ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2559 ╙ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1F, 0x18, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2558 ╘ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1F, 0x18, 0x1F, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+2552 ╒ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3F, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+2553 ╓ */
	{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xFF, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 }, /* U+256B ╫ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xFF, 0x18, 0xFF, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+256A ╪ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2518 ┘ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1F, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+250C ┌ */
	{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }, /* U+2588 █ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }, /* U+2584 ▄ */
	{ 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 }, /* U+258C ▌ */
	{ 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F }, /* U+2590 ▐ */
	{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2580 ▀ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00, 0x00 }, /* U+03B1 α */
	{ 0x00, 0x00, 0x3C, 0x66, 0x66, 0x66, 0x6C, 0x66, 0x66, 0x66, 0x66, 0xEC, 0x00, 0x00, 0x00, 0x00 }, /* U+00DF ß */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 }, /* U+2596 ▖ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F }, /* U+2597 ▗ */
	{ 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2598 ▘ */
	{ 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }, /* U+2599 ▙ */
	{ 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F }, /* U+259A ▚ */
	{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 }, /* U+259B ▛ */
	{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F }, /* U+259C ▜ */
	{ 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+259D ▝ */
	{ 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 }, /* U+259E ▞ */
	{ 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF }, /* U+259F ▟ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xDB, 0xDB, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+221E ∞ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0x7C, 0x10, 0x10, 0x10, 0x00 }, /* U+03C6 φ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC0, 0x78, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00, 0x00 }, /* U+03B5 ε */
	{ 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00, 0x00 }, /* U+2229 ∩ */
	{ 0x00, 0x00, 0x00, 0x00, 0xFE, 0x00, 0x00, 0xFE, 0x00, 0x00, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2261 ≡ */
	{ 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7E, 0x18, 0x18, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00B1 ± */
	{ 0x00, 0x00, 0x00, 0x00, 0xE0, 0x38, 0x0E, 0x38, 0xE0, 0x00, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2265 ≥ */
	{ 0x00, 0x00, 0x00, 0x00, 0x0E, 0x38, 0xE0, 0x38, 0x0E, 0x00, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2264 ≤ */
	{ 0x00, 0x00, 0x0E, 0x1B, 0x1B, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 }, /* U+2320 ⌠ */
	{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xD8, 0xD8, 0xD8, 0x70, 0x00, 0x00, 0x00, 0x00 }, /* U+2321 ⌡ */
	{ 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7E, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00F7 ÷ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xDC, 0x00, 0x76, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2248 ≈ */
	{ 0x00, 0x38, 0x6C, 0x6C, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00B0 ° */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3C, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+2219 ∙ */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00B7 · */
	{ 0x00, 0x00, 0x03, 0x03, 0x06, 0x06, 0x06, 0x0C, 0xCC, 0x6C, 0x38, 0x18, 0x00, 0x00, 0x00, 0x00 }, /* U+221A √ */
	{ 0x00, 0x00, 0xB0, 0xD8, 0xD8, 0xD8, 0xD8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+207F ⁿ */
	{ 0x00, 0x70, 0xD8, 0x30, 0x60, 0xC8, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, /* U+00B2 ² */
	{ 0x00, 0x00, 0x00, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0x00, 0x00, 0x00, 0x00 }, /* U+25A0 ■ */
	{ 0x10, 0x28, 0x44, 0x92, 0xAA, 0xAA, 0xA2, 0x54, 0x8A, 0xAA, 0xAA, 0x92, 0x44, 0x28, 0x10, 0x00 }, /* U+E888  */
};
PRIVATE struct cpvga_encode_entry const basevga_encode_db[] = {
	{ 0x00A0, 0x0020 }, /* 𝘕𝘉𝘚𝘗 */
	{ 0x00A1, 0x00AD }, /* ¡ */
	{ 0x00A2, 0x009B }, /* ¢ */
	{ 0x00A3, 0x009C }, /* £ */
	{ 0x00A5, 0x009D }, /* ¥ */
	{ 0x00A7, 0x0015 }, /* § */
	{ 0x00AA, 0x00A6 }, /* ª */
	{ 0x00AB, 0x00AE }, /* « */
	{ 0x00AC, 0x00AA }, /* ¬ */
	{ 0x00B0, 0x00F8 }, /* ° */
	{ 0x00B1, 0x00F1 }, /* ± */
	{ 0x00B2, 0x00FD }, /* ² */
	{ 0x00B6, 0x0014 }, /* ¶ */
	{ 0x00B7, 0x00FA }, /* · */
	{ 0x00BA, 0x00A7 }, /* º */
	{ 0x00BB, 0x00AF }, /* » */
	{ 0x00BC, 0x00AC }, /* ¼ */
	{ 0x00BD, 0x00AB }, /* ½ */
	{ 0x00BF, 0x00A8 }, /* ¿ */
	{ 0x00C4, 0x008E }, /* Ä */
	{ 0x00C5, 0x008F }, /* Å */
	{ 0x00C6, 0x0092 }, /* Æ */
	{ 0x00C7, 0x0080 }, /* Ç */
	{ 0x00C9, 0x0090 }, /* É */
	{ 0x00D1, 0x00A5 }, /* Ñ */
	{ 0x00D6, 0x0099 }, /* Ö */
	{ 0x00DC, 0x009A }, /* Ü */
	{ 0x00DF, 0x00E1 }, /* ß */
	{ 0x00E0, 0x0085 }, /* à */
	{ 0x00E1, 0x00A0 }, /* á */
	{ 0x00E2, 0x0083 }, /* â */
	{ 0x00E4, 0x0084 }, /* ä */
	{ 0x00E5, 0x0086 }, /* å */
	{ 0x00E6, 0x0091 }, /* æ */
	{ 0x00E7, 0x0087 }, /* ç */
	{ 0x00E8, 0x008A }, /* è */
	{ 0x00E9, 0x0082 }, /* é */
	{ 0x00EA, 0x0088 }, /* ê */
	{ 0x00EB, 0x0089 }, /* ë */
	{ 0x00EC, 0x008D }, /* ì */
	{ 0x00ED, 0x00A1 }, /* í */
	{ 0x00EE, 0x008C }, /* î */
	{ 0x00EF, 0x008B }, /* ï */
	{ 0x00F1, 0x00A4 }, /* ñ */
	{ 0x00F2, 0x0095 }, /* ò */
	{ 0x00F3, 0x00A2 }, /* ó */
	{ 0x00F4, 0x0093 }, /* ô */
	{ 0x00F6, 0x0094 }, /* ö */
	{ 0x00F7, 0x00F6 }, /* ÷ */
	{ 0x00F9, 0x0097 }, /* ù */
	{ 0x00FA, 0x00A3 }, /* ú */
	{ 0x00FB, 0x0096 }, /* û */
	{ 0x00FC, 0x0081 }, /* ü */
	{ 0x00FF, 0x0098 }, /* ÿ */
	{ 0x0192, 0x009F }, /* ƒ */
	{ 0x03B1, 0x00E0 }, /* α */
	{ 0x03B5, 0x00EE }, /* ε */
	{ 0x03C6, 0x00ED }, /* φ */
	{ 0x2007, 0x0020 }, /*   */
	{ 0x2022, 0x0007 }, /* • */
	{ 0x203C, 0x0013 }, /* ‼ */
	{ 0x207F, 0x00FC }, /* ⁿ */
	{ 0x20A7, 0x009E }, /* ₧ */
	{ 0x2190, 0x001B }, /* ← */
	{ 0x2191, 0x0018 }, /* ↑ */
	{ 0x2192, 0x001A }, /* → */
	{ 0x2193, 0x0019 }, /* ↓ */
	{ 0x2194, 0x001D }, /* ↔ */
	{ 0x2195, 0x0012 }, /* ↕ */
	{ 0x21A8, 0x0017 }, /* ↨ */
	{ 0x2219, 0x00F9 }, /* ∙ */
	{ 0x221A, 0x00FB }, /* √ */
	{ 0x221E, 0x00EC }, /* ∞ */
	{ 0x221F, 0x001C }, /* ∟ */
	{ 0x2229, 0x00EF }, /* ∩ */
	{ 0x2248, 0x00F7 }, /* ≈ */
	{ 0x2261, 0x00F0 }, /* ≡ */
	{ 0x2264, 0x00F3 }, /* ≤ */
	{ 0x2265, 0x00F2 }, /* ≥ */
	{ 0x2302, 0x007F }, /* ⌂ */
	{ 0x2310, 0x00A9 }, /* ⌐ */
	{ 0x2320, 0x00F4 }, /* ⌠ */
	{ 0x2321, 0x00F5 }, /* ⌡ */
	{ 0x2500, 0x00C4 }, /* ─ */
	{ 0x2502, 0x00B3 }, /* │ */
	{ 0x250C, 0x00DA }, /* ┌ */
	{ 0x2510, 0x00BF }, /* ┐ */
	{ 0x2514, 0x00C0 }, /* └ */
	{ 0x2518, 0x00D9 }, /* ┘ */
	{ 0x251C, 0x00C3 }, /* ├ */
	{ 0x2524, 0x00B4 }, /* ┤ */
	{ 0x252C, 0x00C2 }, /* ┬ */
	{ 0x2534, 0x00C1 }, /* ┴ */
	{ 0x253C, 0x00C5 }, /* ┼ */
	{ 0x2550, 0x00CD }, /* ═ */
	{ 0x2551, 0x00BA }, /* ║ */
	{ 0x2552, 0x00D5 }, /* ╒ */
	{ 0x2553, 0x00D6 }, /* ╓ */
	{ 0x2554, 0x00C9 }, /* ╔ */
	{ 0x2555, 0x00B8 }, /* ╕ */
	{ 0x2556, 0x00B7 }, /* ╖ */
	{ 0x2557, 0x00BB }, /* ╗ */
	{ 0x2558, 0x00D4 }, /* ╘ */
	{ 0x2559, 0x00D3 }, /* ╙ */
	{ 0x255A, 0x00C8 }, /* ╚ */
	{ 0x255B, 0x00BE }, /* ╛ */
	{ 0x255C, 0x00BD }, /* ╜ */
	{ 0x255D, 0x00BC }, /* ╝ */
	{ 0x255E, 0x00C6 }, /* ╞ */
	{ 0x255F, 0x00C7 }, /* ╟ */
	{ 0x2560, 0x00CC }, /* ╠ */
	{ 0x2561, 0x00B5 }, /* ╡ */
	{ 0x2562, 0x00B6 }, /* ╢ */
	{ 0x2563, 0x00B9 }, /* ╣ */
	{ 0x2564, 0x00D1 }, /* ╤ */
	{ 0x2565, 0x00D2 }, /* ╥ */
	{ 0x2566, 0x00CB }, /* ╦ */
	{ 0x2567, 0x00CF }, /* ╧ */
	{ 0x2568, 0x00D0 }, /* ╨ */
	{ 0x2569, 0x00CA }, /* ╩ */
	{ 0x256A, 0x00D8 }, /* ╪ */
	{ 0x256B, 0x00D7 }, /* ╫ */
	{ 0x256C, 0x00CE }, /* ╬ */
	{ 0x2580, 0x00DF }, /* ▀ */
	{ 0x2584, 0x00DC }, /* ▄ */
	{ 0x2588, 0x00DB }, /* █ */
	{ 0x258C, 0x00DD }, /* ▌ */
	{ 0x2590, 0x00DE }, /* ▐ */
	{ 0x2591, 0x00B0 }, /* ░ */
	{ 0x2592, 0x00B1 }, /* ▒ */
	{ 0x2593, 0x00B2 }, /* ▓ */
	{ 0x2596, 0x00E2 }, /* ▖ */
	{ 0x2597, 0x00E3 }, /* ▗ */
	{ 0x2598, 0x00E4 }, /* ▘ */
	{ 0x2599, 0x00E5 }, /* ▙ */
	{ 0x259A, 0x00E6 }, /* ▚ */
	{ 0x259B, 0x00E7 }, /* ▛ */
	{ 0x259C, 0x00E8 }, /* ▜ */
	{ 0x259D, 0x00E9 }, /* ▝ */
	{ 0x259E, 0x00EA }, /* ▞ */
	{ 0x259F, 0x00EB }, /* ▟ */
	{ 0x25A0, 0x00FE }, /* ■ */
	{ 0x25AC, 0x0016 }, /* ▬ */
	{ 0x25B2, 0x001E }, /* ▲ */
	{ 0x25B3, 0x001E }, /* △ */
	{ 0x25B4, 0x001E }, /* ▴ */
	{ 0x25B6, 0x0010 }, /* ▶ */
	{ 0x25B8, 0x0010 }, /* ▸ */
	{ 0x25BA, 0x0010 }, /* ► */
	{ 0x25BC, 0x001F }, /* ▼ */
	{ 0x25BE, 0x001F }, /* ▾ */
	{ 0x25C0, 0x0011 }, /* ◀ */
	{ 0x25C2, 0x0010 }, /* ◂ */
	{ 0x25C4, 0x0011 }, /* ◄ */
	{ 0x25CB, 0x0009 }, /* ○ */
	{ 0x25D8, 0x0008 }, /* ◘ */
	{ 0x25D9, 0x000A }, /* ◙ */
	{ 0x25FC, 0x00FE }, /* ◼ */
	{ 0x263A, 0x0001 }, /* ☺ */
	{ 0x263B, 0x0002 }, /* ☻ */
	{ 0x263C, 0x000F }, /* ☼ */
	{ 0x2640, 0x000C }, /* ♀ */
	{ 0x2642, 0x000B }, /* ♂ */
	{ 0x2660, 0x0006 }, /* ♠ */
	{ 0x2663, 0x0005 }, /* ♣ */
	{ 0x2665, 0x0003 }, /* ♥ */
	{ 0x2666, 0x0004 }, /* ♦ */
	{ 0x266A, 0x000D }, /* ♪ */
	{ 0x266B, 0x000E }, /* ♫ */
	{ 0xE888, 0x00FF }, /*  */
	{ 0xFFFD, 0x0000 }, /* � */
};
/*[[[end]]]*/


/* Return the font-map index (iow: the value that should be written
 * to plane#0) in order to represent a given UTF-32 character `ch'. */
INTERN ATTR_CONST byte_t
NOTHROW(FCALL basevga_defaultfont_encode)(char32_t ch) {
	size_t lo, hi;
	if likely(ch >= CPVGA_ASCII_MIN && ch <= CPVGA_ASCII_MAX)
		return (byte_t)ch;
	/* Binary search through `basevga_encode_db' */
	lo = 0;
	hi = lengthof(basevga_encode_db);
	while (lo < hi) {
		size_t i;
		i = (lo + hi) / 2;
		if (ch < basevga_encode_db[i].cee_uni) {
			hi = i;
		} else if (ch > basevga_encode_db[i].cee_uni) {
			lo = i + 1;
		} else {
			return basevga_encode_db[i].cee_vga;
		}
	}

	/* "full width" characters (we're VGA, so all characters are equal-sized, or "full width"):
	 * ！＂＃＄％＆＇（）＊＋，－．／０１２３４５６７８９：；＜＝＞？＠ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯ
	 * ＰＱＲＳＴＵＶＷＸＹＺ［＼］＾＿｀ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ｛｜｝～ */
	if (ch >= 0xFF01 && ch <= 0xFF5E)
		return 0x0021 + (ch - 0xFF01);

	return 0x00; /* U+FFFD: � */
}

DECL_END

#endif /* !GUARD_MODSVGA_BASEFONT_C */
