/* Copyright (c) 2019-2024 Griefer@Work                                       *
 *                                                                            *
 * This software is provided 'as-is', without any express or implied          *
 * warranty. In no event will the authors be held liable for any damages      *
 * arising from the use of this software.                                     *
 *                                                                            *
 * Permission is granted to anyone to use this software for any purpose,      *
 * including commercial applications, and to alter it and redistribute it     *
 * freely, subject to the following restrictions:                             *
 *                                                                            *
 * 1. The origin of this software must not be misrepresented; you must not    *
 *    claim that you wrote the original software. If you use this software    *
 *    in a product, an acknowledgement (see the following) in the product     *
 *    documentation is required:                                              *
 *    Portions Copyright (c) 2019-2024 Griefer@Work                           *
 * 2. Altered source versions must be plainly marked as such, and must not be *
 *    misrepresented as being the original software.                          *
 * 3. This notice may not be removed or altered from any source distribution. *
 */
#ifndef GUARD_APPS_SNAKE_MAIN_C
#define GUARD_APPS_SNAKE_MAIN_C 1
#define _KOS_SOURCE 1

/* Port of the old, incomplete snake game from KOSmk1
 * https://github.com/GrieferAtWork/KOSmk1/blob/master/userland/snake.c */

#include <hybrid/compiler.h>

#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>
#include <pthread.h>

__pragma_GCC_diagnostic_ignored(Wstrict_overflow)

DECL_BEGIN

#define IN  STDIN_FILENO
#define OUT STDOUT_FILENO

#define print(s) write(OUT, s, sizeof(s) - sizeof(char))

#define CELL_NONE   0
#define CELL_SNAKE  1
#define CELL_CHERRY 2
static char const *cell_kind[] = {
	"\033[0m",   /* reset */
	"\033[107m", /* white */
	"\033[101m", /* red */
};

static int cursor_x          = 0;
static int cursor_y          = 0;
static int screen_x          = 80;
static int screen_y          = 25;
static char const *last_cell = NULL;
static void setcursor(int x, int y) {
	if (cursor_x != x || cursor_y != y) {
		dprintf(OUT, "\033[%d;%dH",
		        y + 1, x + 1);
		cursor_x = x;
		cursor_y = y;
	}
}
static void cell_set(int x, int y, int kind) {
	char const *newcell = cell_kind[kind];
	if (newcell != last_cell) {
		last_cell = newcell;
		dprintf(OUT, "%s", newcell);
	}
	setcursor(x, y);
	dprintf(OUT, " \033[D");
}

struct cell {
	int x, y;
};
struct snake {
	size_t       cellc;
	struct cell *cellv;
	int          dir;
};
#define DIR_NONE  0
#define DIR_UP    1
#define DIR_DOWN  2
#define DIR_LEFT  3
#define DIR_RIGHT 4

static struct snake my_snake = { 0, NULL, DIR_NONE };

static void extend(size_t n) {
	struct cell *newcellv, *iter, *end;
	size_t newcellc = my_snake.cellc + n;
	newcellv = (struct cell *)realloc(my_snake.cellv,
	                                  newcellc *
	                                  sizeof(struct cell));
	my_snake.cellv  = newcellv;
	newcellv += my_snake.cellc;
	my_snake.cellc = newcellc;
	end = (iter = newcellv) + n;
	--newcellv;
	for (; iter != end; ++iter)
		*iter = *newcellv;
}

static void drawhead(void) {
	cell_set(my_snake.cellv[0].x,
	         my_snake.cellv[0].y,
	         CELL_SNAKE);
}

static void drawtail(void) {
	cell_set(my_snake.cellv[my_snake.cellc - 1].x,
	         my_snake.cellv[my_snake.cellc - 1].y,
	         CELL_NONE);
}

static void domove(void) {
	struct cell *head = my_snake.cellv;
	memmove(head + 1, head,
	        my_snake.cellc - 1,
	        sizeof(struct cell));
	switch (my_snake.dir) {

	case DIR_UP:
		--head->y;
		if (head->y < 0)
			head->y = screen_y - 1;
		break;

	case DIR_DOWN:
		++head->y;
		if ((unsigned int)head->y >= (unsigned int)screen_y)
			head->y = 0;
		break;

	case DIR_LEFT:
		--head->x;
		if (head->x < 0)
			head->x = screen_x - 1;
		break;

	case DIR_RIGHT:
		++head->x;
		if ((unsigned int)head->x >= (unsigned int)screen_x)
			head->x = 0;
		break;

	default: break;
	}
}

static void move(void) {
	if (my_snake.dir == DIR_NONE)
		return;
	drawtail();
	domove();
	drawhead();
}

static pthread_t mover_thread;

static void *mover_threadmain(void *closure) {
	struct timespec tmo = { 0, 10000000l };
	for (;;) {
		nanosleep(&tmo, NULL);
		move();
	}
	return closure;
}

static int process_input(void) {
#define handle(str)                                \
	((memcmp(iter, str, COMPILER_STRLEN(str)) == 0 \
	  ? (iter += COMPILER_STRLEN(str), 1)          \
	  : 0))
	char buf[16];
	ssize_t s;
	char *iter, *end;
	s = read(IN, buf, sizeof(buf));
	if (s <= 0)
		return 0;
	for (end = (iter = buf) + s; iter != end;) {
		if (handle("\033[D")) {
			my_snake.dir = DIR_LEFT;
		} else if (handle("\033[C")) {
			my_snake.dir = DIR_RIGHT;
		} else if (handle("\033[A")) {
			my_snake.dir = DIR_UP;
		} else if (handle("\033[B")) {
			my_snake.dir = DIR_DOWN;
		} else if (handle("k")) {
			extend(20);
		} else if (handle("\3")) {
			return 0;
		} else {
			++iter;
		}
	}
#undef handle
	return 1;
}




static struct termios oldios;
static void init(void) {
	struct termios newios;
	char answer[64];
	ssize_t rsize;
	tcgetattr(STDIN_FILENO, &oldios);
	memcpy(&newios, &oldios, sizeof(struct termios));
	newios.c_lflag &= ~(ICANON | ECHO);
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &newios);
	print("\033[25l"
	      "\033[s"
	      "\033[4096;4096H"
	      "\033[n");
	rsize = read(IN, answer, sizeof(answer) - 1);
	if (rsize >= 0) {
		answer[rsize] = '\0';
		sscanf(answer, "\033[%d;%dR", &screen_y, &screen_x);
		++screen_x;
		++screen_y;
	}
	print("\033[1;1H");
	my_snake.cellc = 1;
	my_snake.cellv = (struct cell *)calloc(my_snake.cellc,
	                                       sizeof(struct cell));
	pthread_create(&mover_thread, NULL, &mover_threadmain, NULL);
}

static void fini(void) {
	pthread_cancel(mover_thread);
	pthread_join(mover_thread, NULL);
	free(my_snake.cellv);
	print("\033[0m"
	      "\033[25h"
	      "\033[u");
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &oldios);
}

int main(int argc, char *argv[]) {
	(void)argc;
	(void)argv;
	setenv("why", "all work and no play makes jack a dull boy", 1);
	init();
	while (process_input())
		;
	fini();
	return EXIT_SUCCESS;
}

DECL_END

#endif /* !GUARD_APPS_SNAKE_MAIN_C */
