/* Copyright (c) 2019-2024 Griefer@Work                                       *
 *                                                                            *
 * This software is provided 'as-is', without any express or implied          *
 * warranty. In no event will the authors be held liable for any damages      *
 * arising from the use of this software.                                     *
 *                                                                            *
 * Permission is granted to anyone to use this software for any purpose,      *
 * including commercial applications, and to alter it and redistribute it     *
 * freely, subject to the following restrictions:                             *
 *                                                                            *
 * 1. The origin of this software must not be misrepresented; you must not    *
 *    claim that you wrote the original software. If you use this software    *
 *    in a product, an acknowledgement (see the following) in the product     *
 *    documentation is required:                                              *
 *    Portions Copyright (c) 2019-2024 Griefer@Work                           *
 * 2. Altered source versions must be plainly marked as such, and must not be *
 *    misrepresented as being the original software.                          *
 * 3. This notice may not be removed or altered from any source distribution. *
 */
#ifndef _LIBM_PZERO_H
#define _LIBM_PZERO_H 1

#include <__crt.h>

#ifndef __NO_FPU
#include <hybrid/__bit.h>
#include <hybrid/typecore.h>

#include <bits/types.h>

#include <libm/fdlibm.h>

#ifdef __CC__
__DECL_BEGIN

#ifdef __IEEE754_FLOAT_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
__LIBM_LOCAL_DECLARE_BEGIN
#ifndef __libm_onef_defined
#define __libm_onef_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, onef, __IEEE754_FLOAT_C(1.0000000000e+00)) /* 0x3F800000 */
#endif /* !__libm_onef_defined */
/* The asymptotic expansions of pzero is
 *    1 - 9/128 s^2 + 11025/98304 s^4 - ...,    where s = 1/x.
 * For x >= 2, We approximate pzero by
 *    pzero(x) = 1 + (R/S)
 * where  R = pR0 + pR1*s^2 + pR2*s^4 + ... + pR5*s^10
 *      S = 1 + pS0*s^2 + ... + pS4*s^10
 * and
 *    | pzero(x)-1-R/S | <= 2  ** ( -60.26)
 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_pR8f, 6) {
	/* for x in [inf, 8]=1/[0,0.125] */
	__IEEE754_FLOAT_C(0.0000000000e+00),  /* 0x00000000 */
	__IEEE754_FLOAT_C(-7.0312500000e-02), /* 0xbd900000 */
	__IEEE754_FLOAT_C(-8.0816707611e+00), /* 0xc1014e86 */
	__IEEE754_FLOAT_C(-2.5706311035e+02), /* 0xc3808814 */
	__IEEE754_FLOAT_C(-2.4852163086e+03), /* 0xc51b5376 */
	__IEEE754_FLOAT_C(-5.2530439453e+03)  /* 0xc5a4285a */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_pS8f, 5) {
	__IEEE754_FLOAT_C(1.1653436279e+02), /* 0x42e91198 */
	__IEEE754_FLOAT_C(3.8337448730e+03), /* 0x456f9beb */
	__IEEE754_FLOAT_C(4.0597855469e+04), /* 0x471e95db */
	__IEEE754_FLOAT_C(1.1675296875e+05), /* 0x47e4087c */
	__IEEE754_FLOAT_C(4.7627726562e+04)  /* 0x473a0bba */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_pR5f, 6) {
	/* for x in [8,4.5454]=1/[0.125,0.22001] */
	__IEEE754_FLOAT_C(-1.1412546255e-11), /* 0xad48c58a */
	__IEEE754_FLOAT_C(-7.0312492549e-02), /* 0xbd8fffff */
	__IEEE754_FLOAT_C(-4.1596107483e+00), /* 0xc0851b88 */
	__IEEE754_FLOAT_C(-6.7674766541e+01), /* 0xc287597b */
	__IEEE754_FLOAT_C(-3.3123129272e+02), /* 0xc3a59d9b */
	__IEEE754_FLOAT_C(-3.4643338013e+02)  /* 0xc3ad3779 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_pS5f, 5) {
	__IEEE754_FLOAT_C(6.0753936768e+01), /* 0x42730408 */
	__IEEE754_FLOAT_C(1.0512523193e+03), /* 0x44836813 */
	__IEEE754_FLOAT_C(5.9789707031e+03), /* 0x45bad7c4 */
	__IEEE754_FLOAT_C(9.6254453125e+03), /* 0x461665c8 */
	__IEEE754_FLOAT_C(2.4060581055e+03)  /* 0x451660ee */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_pR3f, 6) {
	/* for x in [4.547,2.8571]=1/[0.2199,0.35001] */
	__IEEE754_FLOAT_C(-2.5470459075e-09), /* 0xb12f081b */
	__IEEE754_FLOAT_C(-7.0311963558e-02), /* 0xbd8fffb8 */
	__IEEE754_FLOAT_C(-2.4090321064e+00), /* 0xc01a2d95 */
	__IEEE754_FLOAT_C(-2.1965976715e+01), /* 0xc1afba52 */
	__IEEE754_FLOAT_C(-5.8079170227e+01), /* 0xc2685112 */
	__IEEE754_FLOAT_C(-3.1447946548e+01)  /* 0xc1fb9565 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_pS3f, 5) {
	__IEEE754_FLOAT_C(3.5856033325e+01), /* 0x420f6c94 */
	__IEEE754_FLOAT_C(3.6151397705e+02), /* 0x43b4c1ca */
	__IEEE754_FLOAT_C(1.1936077881e+03), /* 0x44953373 */
	__IEEE754_FLOAT_C(1.1279968262e+03), /* 0x448cffe6 */
	__IEEE754_FLOAT_C(1.7358093262e+02)  /* 0x432d94b8 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_pR2f, 6) {
	/* for x in [2.8570,2]=1/[0.3499,0.5] */
	__IEEE754_FLOAT_C(-8.8753431271e-08), /* 0xb3be98b7 */
	__IEEE754_FLOAT_C(-7.0303097367e-02), /* 0xbd8ffb12 */
	__IEEE754_FLOAT_C(-1.4507384300e+00), /* 0xbfb9b1cc */
	__IEEE754_FLOAT_C(-7.6356959343e+00), /* 0xc0f4579f */
	__IEEE754_FLOAT_C(-1.1193166733e+01), /* 0xc1331736 */
	__IEEE754_FLOAT_C(-3.2336456776e+00)  /* 0xc04ef40d */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_pS2f, 5) {
	__IEEE754_FLOAT_C(2.2220300674e+01), /* 0x41b1c32d */
	__IEEE754_FLOAT_C(1.3620678711e+02), /* 0x430834f0 */
	__IEEE754_FLOAT_C(2.7047027588e+02), /* 0x43873c32 */
	__IEEE754_FLOAT_C(1.5387539673e+02), /* 0x4319e01a */
	__IEEE754_FLOAT_C(1.4657617569e+01)  /* 0x416a859a */
};
/* For x >= 8, the asymptotic expansions of qzero is
 *    -1/8 s + 75/1024 s^3 - ..., where s = 1/x.
 * We approximate qzero by
 *    qzero(x) = s*(-1.25 + (R/S))
 * where  R = qR0 + qR1*s^2 + qR2*s^4 + ... + qR5*s^10
 *      S = 1 + qS0*s^2 + ... + qS5*s^12
 * and
 *    | qzero(x)/s +1.25-R/S | <= 2  ** ( -61.22)
 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_qR8f, 6) {
	/* for x in [inf, 8]=1/[0,0.125] */
	__IEEE754_FLOAT_C(0.0000000000e+00), /* 0x00000000 */
	__IEEE754_FLOAT_C(7.3242187500e-02), /* 0x3d960000 */
	__IEEE754_FLOAT_C(1.1768206596e+01), /* 0x413c4a93 */
	__IEEE754_FLOAT_C(5.5767340088e+02), /* 0x440b6b19 */
	__IEEE754_FLOAT_C(8.8591972656e+03), /* 0x460a6cca */
	__IEEE754_FLOAT_C(3.7014625000e+04)  /* 0x471096a0 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_qS8f, 6) {
	__IEEE754_FLOAT_C(1.6377603149e+02), /* 0x4323c6aa */
	__IEEE754_FLOAT_C(8.0983447266e+03), /* 0x45fd12c2 */
	__IEEE754_FLOAT_C(1.4253829688e+05), /* 0x480b3293 */
	__IEEE754_FLOAT_C(8.0330925000e+05), /* 0x49441ed4 */
	__IEEE754_FLOAT_C(8.4050156250e+05), /* 0x494d3359 */
	__IEEE754_FLOAT_C(-3.4389928125e+05) /* 0xc8a7eb69 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_qR5f, 6) {
	/* for x in [8,4.5454]=1/[0.125,0.22001] */
	__IEEE754_FLOAT_C(1.8408595828e-11), /* 0x2da1ec79 */
	__IEEE754_FLOAT_C(7.3242180049e-02), /* 0x3d95ffff */
	__IEEE754_FLOAT_C(5.8356351852e+00), /* 0x40babd86 */
	__IEEE754_FLOAT_C(1.3511157227e+02), /* 0x43071c90 */
	__IEEE754_FLOAT_C(1.0272437744e+03), /* 0x448067cd */
	__IEEE754_FLOAT_C(1.9899779053e+03)  /* 0x44f8bf4b */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_qS5f, 6) {
	__IEEE754_FLOAT_C(8.2776611328e+01), /* 0x42a58da0 */
	__IEEE754_FLOAT_C(2.0778142090e+03), /* 0x4501dd07 */
	__IEEE754_FLOAT_C(1.8847289062e+04), /* 0x46933e94 */
	__IEEE754_FLOAT_C(5.6751113281e+04), /* 0x475daf1d */
	__IEEE754_FLOAT_C(3.5976753906e+04), /* 0x470c88c1 */
	__IEEE754_FLOAT_C(-5.3543427734e+03) /* 0xc5a752be */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_qR3f, 6) {
	/* for x in [4.547,2.8571]=1/[0.2199,0.35001] */
	__IEEE754_FLOAT_C(4.3774099900e-09), /* 0x3196681b */
	__IEEE754_FLOAT_C(7.3241114616e-02), /* 0x3d95ff70 */
	__IEEE754_FLOAT_C(3.3442313671e+00), /* 0x405607e3 */
	__IEEE754_FLOAT_C(4.2621845245e+01), /* 0x422a7cc5 */
	__IEEE754_FLOAT_C(1.7080809021e+02), /* 0x432acedf */
	__IEEE754_FLOAT_C(1.6673394775e+02)  /* 0x4326bbe4 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_qS3f, 6) {
	__IEEE754_FLOAT_C(4.8758872986e+01), /* 0x42430916 */
	__IEEE754_FLOAT_C(7.0968920898e+02), /* 0x44316c1c */
	__IEEE754_FLOAT_C(3.7041481934e+03), /* 0x4567825f */
	__IEEE754_FLOAT_C(6.4604252930e+03), /* 0x45c9e367 */
	__IEEE754_FLOAT_C(2.5163337402e+03), /* 0x451d4557 */
	__IEEE754_FLOAT_C(-1.4924745178e+02) /* 0xc3153f59 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_qR2f, 6) {
	/* for x in [2.8570,2]=1/[0.3499,0.5] */
	__IEEE754_FLOAT_C(1.5044444979e-07), /* 0x342189db */
	__IEEE754_FLOAT_C(7.3223426938e-02), /* 0x3d95f62a */
	__IEEE754_FLOAT_C(1.9981917143e+00), /* 0x3fffc4bf */
	__IEEE754_FLOAT_C(1.4495602608e+01), /* 0x4167edfd */
	__IEEE754_FLOAT_C(3.1666231155e+01), /* 0x41fd5471 */
	__IEEE754_FLOAT_C(1.6252708435e+01)  /* 0x4182058c */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pzero_qS2f, 6) {
	__IEEE754_FLOAT_C(3.0365585327e+01), /* 0x41f2ecb8 */
	__IEEE754_FLOAT_C(2.6934811401e+02), /* 0x4386ac8f */
	__IEEE754_FLOAT_C(8.4478375244e+02), /* 0x44533229 */
	__IEEE754_FLOAT_C(8.8293585205e+02), /* 0x445cbbe5 */
	__IEEE754_FLOAT_C(2.1266638184e+02), /* 0x4354aa98 */
	__IEEE754_FLOAT_C(-5.3109550476e+00) /* 0xc0a9f358 */
};
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(j0_y0_pzerof) __ATTR_WUNUSED __IEEE754_FLOAT_TYPE__
(__LIBCCALL __ieee754_j0_y0_pzerof)(__IEEE754_FLOAT_TYPE__ __x) {
	__IEEE754_FLOAT_TYPE__ const *__p, *__q;
	__IEEE754_FLOAT_TYPE__ __z, __r, __s;
	__int32_t __ix;
	__LIBM_GET_FLOAT_WORD(__ix, __x);
	__ix &= __INT32_C(0x7fffffff);
	if (__ix >= __INT32_C(0x41000000)) {
		/* x >= 2**16 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR8f);
		__q = __LIBM_LOCAL_VALUE(pzero_pS8f);
	} else if (__ix >= __INT32_C(0x40917458)) {
		/* x >= 4.545452117919921875 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR5f);
		__q = __LIBM_LOCAL_VALUE(pzero_pS5f);
	} else if (__ix >= __INT32_C(0x4036db68)) {
		/* x >= 2.8571414947509765625 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR3f);
		__q = __LIBM_LOCAL_VALUE(pzero_pS3f);
	} else /* if (ix >= __INT32_C(0x40000000)) */ {
		/* x >= 2 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR2f);
		__q = __LIBM_LOCAL_VALUE(pzero_pS2f);
	}
	__z = __LIBM_LOCAL_VALUE(onef) / (__x * __x);
	__r = __p[0] + __z * (__p[1] + __z * (__p[2] + __z * (__p[3] + __z * (__p[4] + __z * __p[5]))));
	__s = __LIBM_LOCAL_VALUE(onef) + __z * (__q[0] + __z * (__q[1] + __z * (__q[2] + __z * (__q[3] + __z * __q[4]))));
	return __LIBM_LOCAL_VALUE(onef) + __r / __s;
}

__LIBM_LOCAL_FUNC(j0_y0_qzerof) __ATTR_WUNUSED __IEEE754_FLOAT_TYPE__
(__LIBCCALL __ieee754_j0_y0_qzerof)(__IEEE754_FLOAT_TYPE__ __x) {
	__IEEE754_FLOAT_TYPE__ const *__p, *__q;
	__IEEE754_FLOAT_TYPE__ __s, __r, __z;
	__int32_t __ix;
	__LIBM_GET_FLOAT_WORD(__ix, __x);
	__ix &= __INT32_C(0x7fffffff);
	if (__ix >= __INT32_C(0x41000000)) {
		/* x >= 2**16 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR8f);
		__q = __LIBM_LOCAL_VALUE(pzero_qS8f);
	} else if (__ix >= __INT32_C(0x40917458)) {
		/* x >= 4.545452117919921875 */
		/* x >= 4.54541015625 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR5f);
		__q = __LIBM_LOCAL_VALUE(pzero_qS5f);
	} else if (__ix >= __INT32_C(0x4036db68)) {
		/* x >= 2.8571414947509765625 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR3f);
		__q = __LIBM_LOCAL_VALUE(pzero_qS3f);
	} else /* if (ix >= __INT32_C(0x40000000)) */
	{
		/* x >= 2 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR2f);
		__q = __LIBM_LOCAL_VALUE(pzero_qS2f);
	}
	__z = __LIBM_LOCAL_VALUE(onef) / (__x * __x);
	__r = __p[0] + __z * (__p[1] + __z * (__p[2] + __z * (__p[3] + __z * (__p[4] + __z * __p[5]))));
	__s = __LIBM_LOCAL_VALUE(onef) + __z * (__q[0] + __z * (__q[1] + __z * (__q[2] + __z * (__q[3] + __z * (__q[4] + __z * __q[5])))));
	return (__IEEE754_FLOAT_C(-0.125) + __r / __s) / __x;
}
#endif /* __IEEE754_FLOAT_TYPE__ */


#ifdef __IEEE754_DOUBLE_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunSoft, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
__LIBM_LOCAL_DECLARE_BEGIN
#ifndef __libm_one_defined
#define __libm_one_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, one, __IEEE754_DOUBLE_C(1.00000000000000000000e+00)) /* 0x3FF00000, 0x00000000 */
#endif /* !__libm_one_defined */

/* The asymptotic expansions of pzero is
 *    1 - 9/128 s^2 + 11025/98304 s^4 - ...,    where s = 1/x.
 * For x >= 2, We approximate pzero by
 *  pzero(x) = 1 + (R/S)
 * where  R = pR0 + pR1*s^2 + pR2*s^4 + ... + pR5*s^10
 *      S = 1 + pS0*s^2 + ... + pS4*s^10
 * and
 *    | pzero(x)-1-R/S | <= 2  ** ( -60.26) */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_pR8, 6) {
	/* for x in [inf, 8]=1/[0,0.125] */
	__IEEE754_DOUBLE_C(0.00000000000000000000e+00),  /* 0x00000000, 0x00000000 */
	__IEEE754_DOUBLE_C(-7.03124999999900357484e-02), /* 0xBFB1FFFF, 0xFFFFFD32 */
	__IEEE754_DOUBLE_C(-8.08167041275349795626e+00), /* 0xC02029D0, 0xB44FA779 */
	__IEEE754_DOUBLE_C(-2.57063105679704847262e+02), /* 0xC0701102, 0x7B19E863 */
	__IEEE754_DOUBLE_C(-2.48521641009428822144e+03), /* 0xC0A36A6E, 0xCD4DCAFC */
	__IEEE754_DOUBLE_C(-5.25304380490729545272e+03)  /* 0xC0B4850B, 0x36CC643D */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_pS8, 5) {
	__IEEE754_DOUBLE_C(1.16534364619668181717e+02), /* 0x405D2233, 0x07A96751 */
	__IEEE754_DOUBLE_C(3.83374475364121826715e+03), /* 0x40ADF37D, 0x50596938 */
	__IEEE754_DOUBLE_C(4.05978572648472545552e+04), /* 0x40E3D2BB, 0x6EB6B05F */
	__IEEE754_DOUBLE_C(1.16752972564375915681e+05), /* 0x40FC810F, 0x8F9FA9BD */
	__IEEE754_DOUBLE_C(4.76277284146730962675e+04)  /* 0x40E74177, 0x4F2C49DC */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_pR5, 6) {
	/* for x in [8,4.5454]=1/[0.125,0.22001] */
	__IEEE754_DOUBLE_C(-1.14125464691894502584e-11), /* 0xBDA918B1, 0x47E495CC */
	__IEEE754_DOUBLE_C(-7.03124940873599280078e-02), /* 0xBFB1FFFF, 0xE69AFBC6 */
	__IEEE754_DOUBLE_C(-4.15961064470587782438e+00), /* 0xC010A370, 0xF90C6BBF */
	__IEEE754_DOUBLE_C(-6.76747652265167261021e+01), /* 0xC050EB2F, 0x5A7D1783 */
	__IEEE754_DOUBLE_C(-3.31231299649172967747e+02), /* 0xC074B3B3, 0x6742CC63 */
	__IEEE754_DOUBLE_C(-3.46433388365604912451e+02)  /* 0xC075A6EF, 0x28A38BD7 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_pS5, 5) {
	__IEEE754_DOUBLE_C(6.07539382692300335975e+01), /* 0x404E6081, 0x0C98C5DE */
	__IEEE754_DOUBLE_C(1.05125230595704579173e+03), /* 0x40906D02, 0x5C7E2864 */
	__IEEE754_DOUBLE_C(5.97897094333855784498e+03), /* 0x40B75AF8, 0x8FBE1D60 */
	__IEEE754_DOUBLE_C(9.62544514357774460223e+03), /* 0x40C2CCB8, 0xFA76FA38 */
	__IEEE754_DOUBLE_C(2.40605815922939109441e+03)  /* 0x40A2CC1D, 0xC70BE864 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_pR3, 6) {
	/* for x in [4.547,2.8571]=1/[0.2199,0.35001] */
	__IEEE754_DOUBLE_C(-2.54704601771951915620e-09), /* 0xBE25E103, 0x6FE1AA86 */
	__IEEE754_DOUBLE_C(-7.03119616381481654654e-02), /* 0xBFB1FFF6, 0xF7C0E24B */
	__IEEE754_DOUBLE_C(-2.40903221549529611423e+00), /* 0xC00345B2, 0xAEA48074 */
	__IEEE754_DOUBLE_C(-2.19659774734883086467e+01), /* 0xC035F74A, 0x4CB94E14 */
	__IEEE754_DOUBLE_C(-5.80791704701737572236e+01), /* 0xC04D0A22, 0x420A1A45 */
	__IEEE754_DOUBLE_C(-3.14479470594888503854e+01)  /* 0xC03F72AC, 0xA892D80F */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_pS3, 5) {
	__IEEE754_DOUBLE_C(3.58560338055209726349e+01), /* 0x4041ED92, 0x84077DD3 */
	__IEEE754_DOUBLE_C(3.61513983050303863820e+02), /* 0x40769839, 0x464A7C0E */
	__IEEE754_DOUBLE_C(1.19360783792111533330e+03), /* 0x4092A66E, 0x6D1061D6 */
	__IEEE754_DOUBLE_C(1.12799679856907414432e+03), /* 0x40919FFC, 0xB8C39B7E */
	__IEEE754_DOUBLE_C(1.73580930813335754692e+02)  /* 0x4065B296, 0xFC379081 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_pR2, 6) {
	/* for x in [2.8570,2]=1/[0.3499,0.5] */
	__IEEE754_DOUBLE_C(-8.87534333032526411254e-08), /* 0xBE77D316, 0xE927026D */
	__IEEE754_DOUBLE_C(-7.03030995483624743247e-02), /* 0xBFB1FF62, 0x495E1E42 */
	__IEEE754_DOUBLE_C(-1.45073846780952986357e+00), /* 0xBFF73639, 0x8A24A843 */
	__IEEE754_DOUBLE_C(-7.63569613823527770791e+00), /* 0xC01E8AF3, 0xEDAFA7F3 */
	__IEEE754_DOUBLE_C(-1.11931668860356747786e+01), /* 0xC02662E6, 0xC5246303 */
	__IEEE754_DOUBLE_C(-3.23364579351335335033e+00)  /* 0xC009DE81, 0xAF8FE70F */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_pS2, 5) {
	__IEEE754_DOUBLE_C(2.22202997532088808441e+01), /* 0x40363865, 0x908B5959 */
	__IEEE754_DOUBLE_C(1.36206794218215208048e+02), /* 0x4061069E, 0x0EE8878F */
	__IEEE754_DOUBLE_C(2.70470278658083486789e+02), /* 0x4070E786, 0x42EA079B */
	__IEEE754_DOUBLE_C(1.53875394208320329881e+02), /* 0x40633C03, 0x3AB6FAFF */
	__IEEE754_DOUBLE_C(1.46576176948256193810e+01)  /* 0x402D50B3, 0x44391809 */
};

/* For x >= 8, the asymptotic expansions of qzero is
 *    -1/8 s + 75/1024 s^3 - ..., where s = 1/x.
 * We approximate pzero by
 *    qzero(x) = s*(-1.25 + (R/S))
 * where  R = qR0 + qR1*s^2 + qR2*s^4 + ... + qR5*s^10
 *  S = 1 + qS0*s^2 + ... + qS5*s^12
 * and
 *    | qzero(x)/s +1.25-R/S | <= 2  ** ( -61.22) */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_qR8, 6) {
	/* for x in [inf, 8]=1/[0,0.125] */
	__IEEE754_DOUBLE_C(0.00000000000000000000e+00), /* 0x00000000, 0x00000000 */
	__IEEE754_DOUBLE_C(7.32421874999935051953e-02), /* 0x3FB2BFFF, 0xFFFFFE2C */
	__IEEE754_DOUBLE_C(1.17682064682252693899e+01), /* 0x40278952, 0x5BB334D6 */
	__IEEE754_DOUBLE_C(5.57673380256401856059e+02), /* 0x40816D63, 0x15301825 */
	__IEEE754_DOUBLE_C(8.85919720756468632317e+03), /* 0x40C14D99, 0x3E18F46D */
	__IEEE754_DOUBLE_C(3.70146267776887834771e+04)  /* 0x40E212D4, 0x0E901566 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_qS8, 6) {
	__IEEE754_DOUBLE_C(1.63776026895689824414e+02), /* 0x406478D5, 0x365B39BC */
	__IEEE754_DOUBLE_C(8.09834494656449805916e+03), /* 0x40BFA258, 0x4E6B0563 */
	__IEEE754_DOUBLE_C(1.42538291419120476348e+05), /* 0x41016652, 0x54D38C3F */
	__IEEE754_DOUBLE_C(8.03309257119514397345e+05), /* 0x412883DA, 0x83A52B43 */
	__IEEE754_DOUBLE_C(8.40501579819060512818e+05), /* 0x4129A66B, 0x28DE0B3D */
	__IEEE754_DOUBLE_C(-3.43899293537866615225e+05) /* 0xC114FD6D, 0x2C9530C5 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_qR5, 6) {
	/* for x in [8,4.5454]=1/[0.125,0.22001] */
	__IEEE754_DOUBLE_C(1.84085963594515531381e-11), /* 0x3DB43D8F, 0x29CC8CD9 */
	__IEEE754_DOUBLE_C(7.32421766612684765896e-02), /* 0x3FB2BFFF, 0xD172B04C */
	__IEEE754_DOUBLE_C(5.83563508962056953777e+00), /* 0x401757B0, 0xB9953DD3 */
	__IEEE754_DOUBLE_C(1.35111577286449829671e+02), /* 0x4060E392, 0x0A8788E9 */
	__IEEE754_DOUBLE_C(1.02724376596164097464e+03), /* 0x40900CF9, 0x9DC8C481 */
	__IEEE754_DOUBLE_C(1.98997785864605384631e+03)  /* 0x409F17E9, 0x53C6E3A6 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_qS5, 6) {
	__IEEE754_DOUBLE_C(8.27766102236537761883e+01), /* 0x4054B1B3, 0xFB5E1543 */
	__IEEE754_DOUBLE_C(2.07781416421392987104e+03), /* 0x40A03BA0, 0xDA21C0CE */
	__IEEE754_DOUBLE_C(1.88472887785718085070e+04), /* 0x40D267D2, 0x7B591E6D */
	__IEEE754_DOUBLE_C(5.67511122894947329769e+04), /* 0x40EBB5E3, 0x97E02372 */
	__IEEE754_DOUBLE_C(3.59767538425114471465e+04), /* 0x40E19118, 0x1F7A54A0 */
	__IEEE754_DOUBLE_C(-5.35434275601944773371e+03) /* 0xC0B4EA57, 0xBEDBC609 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_qR3, 6) {
	/* for x in [4.547,2.8571]=1/[0.2199,0.35001] */
	__IEEE754_DOUBLE_C(4.37741014089738620906e-09), /* 0x3E32CD03, 0x6ADECB82 */
	__IEEE754_DOUBLE_C(7.32411180042911447163e-02), /* 0x3FB2BFEE, 0x0E8D0842 */
	__IEEE754_DOUBLE_C(3.34423137516170720929e+00), /* 0x400AC0FC, 0x61149CF5 */
	__IEEE754_DOUBLE_C(4.26218440745412650017e+01), /* 0x40454F98, 0x962DAEDD */
	__IEEE754_DOUBLE_C(1.70808091340565596283e+02), /* 0x406559DB, 0xE25EFD1F */
	__IEEE754_DOUBLE_C(1.66733948696651168575e+02)  /* 0x4064D77C, 0x81FA21E0 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_qS3, 6) {
	__IEEE754_DOUBLE_C(4.87588729724587182091e+01), /* 0x40486122, 0xBFE343A6 */
	__IEEE754_DOUBLE_C(7.09689221056606015736e+02), /* 0x40862D83, 0x86544EB3 */
	__IEEE754_DOUBLE_C(3.70414822620111362994e+03), /* 0x40ACF04B, 0xE44DFC63 */
	__IEEE754_DOUBLE_C(6.46042516752568917582e+03), /* 0x40B93C6C, 0xD7C76A28 */
	__IEEE754_DOUBLE_C(2.51633368920368957333e+03), /* 0x40A3A8AA, 0xD94FB1C0 */
	__IEEE754_DOUBLE_C(-1.49247451836156386662e+02) /* 0xC062A7EB, 0x201CF40F */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_qR2, 6) {
	/* for x in [2.8570,2]=1/[0.3499,0.5] */
	__IEEE754_DOUBLE_C(1.50444444886983272379e-07), /* 0x3E84313B, 0x54F76BDB */
	__IEEE754_DOUBLE_C(7.32234265963079278272e-02), /* 0x3FB2BEC5, 0x3E883E34 */
	__IEEE754_DOUBLE_C(1.99819174093815998816e+00), /* 0x3FFFF897, 0xE727779C */
	__IEEE754_DOUBLE_C(1.44956029347885735348e+01), /* 0x402CFDBF, 0xAAF96FE5 */
	__IEEE754_DOUBLE_C(3.16662317504781540833e+01), /* 0x403FAA8E, 0x29FBDC4A */
	__IEEE754_DOUBLE_C(1.62527075710929267416e+01)  /* 0x403040B1, 0x71814BB4 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pzero_qS2, 6) {
	__IEEE754_DOUBLE_C(3.03655848355219184498e+01), /* 0x403E5D96, 0xF7C07AED */
	__IEEE754_DOUBLE_C(2.69348118608049844624e+02), /* 0x4070D591, 0xE4D14B40 */
	__IEEE754_DOUBLE_C(8.44783757595320139444e+02), /* 0x408A6645, 0x22B3BF22 */
	__IEEE754_DOUBLE_C(8.82935845112488550512e+02), /* 0x408B977C, 0x9C5CC214 */
	__IEEE754_DOUBLE_C(2.12666388511798828631e+02), /* 0x406A9553, 0x0E001365 */
	__IEEE754_DOUBLE_C(-5.31095493882666946917e+00) /* 0xC0153E6A, 0xF8B32931 */
};
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(j0_y0_pzero) __ATTR_WUNUSED __IEEE754_DOUBLE_TYPE__
(__LIBCCALL __ieee754_j0_y0_pzero)(__IEEE754_DOUBLE_TYPE__ __x) {
	__IEEE754_DOUBLE_TYPE__ const *__p, *__q;
	__IEEE754_DOUBLE_TYPE__ __z, __r, __s, __z2, __z4, __r1;
	__IEEE754_DOUBLE_TYPE__ __r2, __r3, __s1, __s2, __s3;
	__int32_t __ix;
	__LIBM_GET_HIGH_WORD(__ix, __x);
	__ix &= __INT32_C(0x7fffffff);
	if (__ix >= __INT32_C(0x41b00000)) {
		/* x >= 2**28 */
		return __LIBM_LOCAL_VALUE(one);
	} else if (__ix >= __INT32_C(0x40200000)) {
		/* x >= 8 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR8);
		__q = __LIBM_LOCAL_VALUE(pzero_pS8);
	} else if (__ix >= __INT32_C(0x40122E8B)) {
		/* x >= 4.545452117919921875 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR5);
		__q = __LIBM_LOCAL_VALUE(pzero_pS5);
	} else if (__ix >= __INT32_C(0x4006DB6D)) {
		/* x >= 2.8571414947509765625 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR3);
		__q = __LIBM_LOCAL_VALUE(pzero_pS3);
	} else /* if (ix >= __INT32_C(0x40000000)) */ {
		/* x >= 2 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR2);
		__q = __LIBM_LOCAL_VALUE(pzero_pS2);
	}
	__z  = __LIBM_LOCAL_VALUE(one) / (__x * __x);
	__r1 = __p[0] + __z * __p[1];
	__z2 = __z * __z;
	__r2 = __p[2] + __z * __p[3];
	__z4 = __z2 * __z2;
	__r3 = __p[4] + __z * __p[5];
	__r  = __r1 + __z2 * __r2 + __z4 * __r3;
	__s1 = __LIBM_LOCAL_VALUE(one) + __z * __q[0];
	__s2 = __q[1] + __z * __q[2];
	__s3 = __q[3] + __z * __q[4];
	__s  = __s1 + __z2 * __s2 + __z4 * __s3;
	return __LIBM_LOCAL_VALUE(one) + __r / __s;
}

__LIBM_LOCAL_FUNC(j0_y0_qzero) __ATTR_WUNUSED __IEEE754_DOUBLE_TYPE__
(__LIBCCALL __ieee754_j0_y0_qzero)(__IEEE754_DOUBLE_TYPE__ __x) {
	__IEEE754_DOUBLE_TYPE__ const *__p, *__q;
	__IEEE754_DOUBLE_TYPE__ __s, __r, __z, __z2, __z4, __z6;
	__IEEE754_DOUBLE_TYPE__ __r1, __r2, __r3, __s1, __s2, __s3;
	__int32_t __ix;
	__LIBM_GET_HIGH_WORD(__ix, __x);
	__ix &= __INT32_C(0x7fffffff);
	if (__ix >= __INT32_C(0x41b00000)) {
		/* x >= 2**28 */
		return __IEEE754_DOUBLE_C(-0.125) / __x;
	} else if (__ix >= __INT32_C(0x40200000)) {
		/* x >= 8 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR8);
		__q = __LIBM_LOCAL_VALUE(pzero_qS8);
	} else if (__ix >= __INT32_C(0x40122E8B)) {
		/* x >= 4.545452117919921875 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR5);
		__q = __LIBM_LOCAL_VALUE(pzero_qS5);
	} else if (__ix >= __INT32_C(0x4006DB6D)) {
		/* x >= 2.8571414947509765625 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR3);
		__q = __LIBM_LOCAL_VALUE(pzero_qS3);
	} else /* if (ix >= __INT32_C(0x40000000)) */ {
		/* x >= 2 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR2);
		__q = __LIBM_LOCAL_VALUE(pzero_qS2);
	}
	__z  = __LIBM_LOCAL_VALUE(one) / (__x * __x);
	__r1 = __p[0] + __z * __p[1];
	__z2 = __z * __z;
	__r2 = __p[2] + __z * __p[3];
	__z4 = __z2 * __z2;
	__r3 = __p[4] + __z * __p[5];
	__z6 = __z4 * __z2;
	__r  = __r1 + __z2 * __r2 + __z4 * __r3;
	__s1 = __LIBM_LOCAL_VALUE(one) + __z * __q[0];
	__s2 = __q[1] + __z * __q[2];
	__s3 = __q[3] + __z * __q[4];
	__s  = __s1 + __z2 * __s2 + __z4 * __s3 + __z6 * __q[5];
	return (__IEEE754_DOUBLE_C(-0.125) + __r / __s) / __x;
}
#endif /* __IEEE754_DOUBLE_TYPE__ */


#ifdef __IEEE854_LONG_DOUBLE_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
__LIBM_LOCAL_DECLARE_BEGIN
#ifndef __libm_onel_defined
#define __libm_onel_defined
__LIBM_LOCAL_DECLARE(__IEEE854_LONG_DOUBLE_TYPE__, onel, __IEEE854_LONG_DOUBLE_C(1.0))
#endif /* !__libm_onel_defined */
/* The asymptotic expansions of pzero is
 *    1 - 9/128 s^2 + 11025/98304 s^4 - ...,    where s = 1/x.
 * For x >= 2, We approximate pzero by
 *    pzero(x) = 1 + s^2 R(s^2) / S(s^2) */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_pR8l, 7) {
	/* 8 <= x <= inf  Peak relative error 4.62 */
	__IEEE854_LONG_DOUBLE_C(-4.094398895124198016684337960227780260127E-9),
	__IEEE854_LONG_DOUBLE_C(-8.929643669432412640061946338524096893089E-7),
	__IEEE854_LONG_DOUBLE_C(-6.281267456906136703868258380673108109256E-5),
	__IEEE854_LONG_DOUBLE_C(-1.736902783620362966354814353559382399665E-3),
	__IEEE854_LONG_DOUBLE_C(-1.831506216290984960532230842266070146847E-2),
	__IEEE854_LONG_DOUBLE_C(-5.827178869301452892963280214772398135283E-2),
	__IEEE854_LONG_DOUBLE_C(-2.087563267939546435460286895807046616992E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_pS8l, 6) {
	__IEEE854_LONG_DOUBLE_C(5.823145095287749230197031108839653988393E-8),
	__IEEE854_LONG_DOUBLE_C(1.279281986035060320477759999428992730280E-5),
	__IEEE854_LONG_DOUBLE_C(9.132668954726626677174825517150228961304E-4),
	__IEEE854_LONG_DOUBLE_C(2.606019379433060585351880541545146252534E-2),
	__IEEE854_LONG_DOUBLE_C(2.956262215119520464228467583516287175244E-1),
	__IEEE854_LONG_DOUBLE_C(1.149498145388256448535563278632697465675E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_pR5l, 7) {
	/* 4.54541015625 <= x <= 8  Peak relative error 6.51E-22 */
	__IEEE854_LONG_DOUBLE_C(-2.041226787870240954326915847282179737987E-7),
	__IEEE854_LONG_DOUBLE_C(-2.255373879859413325570636768224534428156E-5),
	__IEEE854_LONG_DOUBLE_C(-7.957485746440825353553537274569102059990E-4),
	__IEEE854_LONG_DOUBLE_C(-1.093205102486816696940149222095559439425E-2),
	__IEEE854_LONG_DOUBLE_C(-5.657957849316537477657603125260701114646E-2),
	__IEEE854_LONG_DOUBLE_C(-8.641175552716402616180994954177818461588E-2),
	__IEEE854_LONG_DOUBLE_C(-1.354654710097134007437166939230619726157E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_pS5l, 6) {
	__IEEE854_LONG_DOUBLE_C(2.903078099681108697057258628212823545290E-6),
	__IEEE854_LONG_DOUBLE_C(3.253948449946735405975737677123673867321E-4),
	__IEEE854_LONG_DOUBLE_C(1.181269751723085006534147920481582279979E-2),
	__IEEE854_LONG_DOUBLE_C(1.719212057790143888884745200257619469363E-1),
	__IEEE854_LONG_DOUBLE_C(1.006306498779212467670654535430694221924E0),
	__IEEE854_LONG_DOUBLE_C(2.069568808688074324555596301126375951502E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_pR3l, 7) {
	/* 2.85711669921875 <= x <= 4.54541015625  peak relative error 5.25e-21 */
	__IEEE854_LONG_DOUBLE_C(-5.755732156848468345557663552240816066802E-6),
	__IEEE854_LONG_DOUBLE_C(-3.703675625855715998827966962258113034767E-4),
	__IEEE854_LONG_DOUBLE_C(-7.390893350679637611641350096842846433236E-3),
	__IEEE854_LONG_DOUBLE_C(-5.571922144490038765024591058478043873253E-2),
	__IEEE854_LONG_DOUBLE_C(-1.531290690378157869291151002472627396088E-1),
	__IEEE854_LONG_DOUBLE_C(-1.193350853469302941921647487062620011042E-1),
	__IEEE854_LONG_DOUBLE_C(-8.567802507331578894302991505331963782905E-3)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_pS3l, 6) {
	__IEEE854_LONG_DOUBLE_C(8.185931139070086158103309281525036712419E-5),
	__IEEE854_LONG_DOUBLE_C(5.398016943778891093520574483111255476787E-3),
	__IEEE854_LONG_DOUBLE_C(1.130589193590489566669164765853409621081E-1),
	__IEEE854_LONG_DOUBLE_C(9.358652328786413274673192987670237145071E-1),
	__IEEE854_LONG_DOUBLE_C(3.091711512598349056276917907005098085273E0),
	__IEEE854_LONG_DOUBLE_C(3.594602474737921977972586821673124231111E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_pR2l, 7) {
	/* 2 <= x <= 2.85711669921875  peak relative error 2.64e-21 */
	__IEEE854_LONG_DOUBLE_C(-1.219525235804532014243621104365384992623E-4),
	__IEEE854_LONG_DOUBLE_C(-4.838597135805578919601088680065298763049E-3),
	__IEEE854_LONG_DOUBLE_C(-5.732223181683569266223306197751407418301E-2),
	__IEEE854_LONG_DOUBLE_C(-2.472947430526425064982909699406646503758E-1),
	__IEEE854_LONG_DOUBLE_C(-3.753373645974077960207588073975976327695E-1),
	__IEEE854_LONG_DOUBLE_C(-1.556241316844728872406672349347137975495E-1),
	__IEEE854_LONG_DOUBLE_C(-5.355423239526452209595316733635519506958E-3)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_pS2l, 6) {
	__IEEE854_LONG_DOUBLE_C(1.734442793664291412489066256138894953823E-3),
	__IEEE854_LONG_DOUBLE_C(7.158111826468626405416300895617986926008E-2),
	__IEEE854_LONG_DOUBLE_C(9.153839713992138340197264669867993552641E-1),
	__IEEE854_LONG_DOUBLE_C(4.539209519433011393525841956702487797582E0),
	__IEEE854_LONG_DOUBLE_C(8.868932430625331650266067101752626253644E0),
	__IEEE854_LONG_DOUBLE_C(6.067161890196324146320763844772857713502E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
/* For x >= 8, the asymptotic expansions of qzero is
 *    -1/8 s + 75/1024 s^3 - ..., where s = 1/x.
 * We approximate qzero by
 *    qzero(x) = s*(-.125 + R(s^2) / S(s^2))
 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_qR8l, 7) {
	/* 8 <= x <= inf  peak relative error 2.23e-21 */
	__IEEE854_LONG_DOUBLE_C(3.001267180483191397885272640777189348008E-10),
	__IEEE854_LONG_DOUBLE_C(8.693186311430836495238494289942413810121E-8),
	__IEEE854_LONG_DOUBLE_C(8.496875536711266039522937037850596580686E-6),
	__IEEE854_LONG_DOUBLE_C(3.482702869915288984296602449543513958409E-4),
	__IEEE854_LONG_DOUBLE_C(6.036378380706107692863811938221290851352E-3),
	__IEEE854_LONG_DOUBLE_C(3.881970028476167836382607922840452192636E-2),
	__IEEE854_LONG_DOUBLE_C(6.132191514516237371140841765561219149638E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_qS8l, 7) {
	__IEEE854_LONG_DOUBLE_C(4.097730123753051126914971174076227600212E-9),
	__IEEE854_LONG_DOUBLE_C(1.199615869122646109596153392152131139306E-6),
	__IEEE854_LONG_DOUBLE_C(1.196337580514532207793107149088168946451E-4),
	__IEEE854_LONG_DOUBLE_C(5.099074440112045094341500497767181211104E-3),
	__IEEE854_LONG_DOUBLE_C(9.577420799632372483249761659674764460583E-2),
	__IEEE854_LONG_DOUBLE_C(7.385243015344292267061953461563695918646E-1),
	__IEEE854_LONG_DOUBLE_C(1.917266424391428937962682301561699055943E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_qR5l, 7) {
	/* 4.54541015625 <= x <= 8
	   peak relative error 1.03e-21 */
	__IEEE854_LONG_DOUBLE_C(3.406256556438974327309660241748106352137E-8),
	__IEEE854_LONG_DOUBLE_C(4.855492710552705436943630087976121021980E-6),
	__IEEE854_LONG_DOUBLE_C(2.301011739663737780613356017352912281980E-4),
	__IEEE854_LONG_DOUBLE_C(4.500470249273129953870234803596619899226E-3),
	__IEEE854_LONG_DOUBLE_C(3.651376459725695502726921248173637054828E-2),
	__IEEE854_LONG_DOUBLE_C(1.071578819056574524416060138514508609805E-1),
	__IEEE854_LONG_DOUBLE_C(7.458950172851611673015774675225656063757E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_qS5l, 7) {
	__IEEE854_LONG_DOUBLE_C(4.650675622764245276538207123618745150785E-7),
	__IEEE854_LONG_DOUBLE_C(6.773573292521412265840260065635377164455E-5),
	__IEEE854_LONG_DOUBLE_C(3.340711249876192721980146877577806687714E-3),
	__IEEE854_LONG_DOUBLE_C(7.036218046856839214741678375536970613501E-2),
	__IEEE854_LONG_DOUBLE_C(6.569599559163872573895171876511377891143E-1),
	__IEEE854_LONG_DOUBLE_C(2.557525022583599204591036677199171155186E0),
	__IEEE854_LONG_DOUBLE_C(3.457237396120935674982927714210361269133E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_qR3l, 7) {
	/* 2.85711669921875 <= x <= 4.54541015625  peak relative error 5.24e-21 */
	__IEEE854_LONG_DOUBLE_C(1.749459596550816915639829017724249805242E-6),
	__IEEE854_LONG_DOUBLE_C(1.446252487543383683621692672078376929437E-4),
	__IEEE854_LONG_DOUBLE_C(3.842084087362410664036704812125005761859E-3),
	__IEEE854_LONG_DOUBLE_C(4.066369994699462547896426554180954233581E-2),
	__IEEE854_LONG_DOUBLE_C(1.721093619117980251295234795188992722447E-1),
	__IEEE854_LONG_DOUBLE_C(2.538595333972857367655146949093055405072E-1),
	__IEEE854_LONG_DOUBLE_C(8.560591367256769038905328596020118877936E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_qS3l, 7) {
	__IEEE854_LONG_DOUBLE_C(2.388596091707517488372313710647510488042E-5),
	__IEEE854_LONG_DOUBLE_C(2.048679968058758616370095132104333998147E-3),
	__IEEE854_LONG_DOUBLE_C(5.824663198201417760864458765259945181513E-2),
	__IEEE854_LONG_DOUBLE_C(6.953906394693328750931617748038994763958E-1),
	__IEEE854_LONG_DOUBLE_C(3.638186936390881159685868764832961092476E0),
	__IEEE854_LONG_DOUBLE_C(7.900169524705757837298990558459547842607E0),
	__IEEE854_LONG_DOUBLE_C(5.992718532451026507552820701127504582907E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_qR2l, 7) {
	/* 2 <= x <= 2.85711669921875  peak relative error 1.58e-21  */
	__IEEE854_LONG_DOUBLE_C(6.306524405520048545426928892276696949540E-5),
	__IEEE854_LONG_DOUBLE_C(3.209606155709930950935893996591576624054E-3),
	__IEEE854_LONG_DOUBLE_C(5.027828775702022732912321378866797059604E-2),
	__IEEE854_LONG_DOUBLE_C(3.012705561838718956481911477587757845163E-1),
	__IEEE854_LONG_DOUBLE_C(6.960544893905752937420734884995688523815E-1),
	__IEEE854_LONG_DOUBLE_C(5.431871999743531634887107835372232030655E-1),
	__IEEE854_LONG_DOUBLE_C(9.447736151202905471899259026430157211949E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pzero_qS2l, 7) {
	__IEEE854_LONG_DOUBLE_C(8.610579901936193494609755345106129102676E-4),
	__IEEE854_LONG_DOUBLE_C(4.649054352710496997203474853066665869047E-2),
	__IEEE854_LONG_DOUBLE_C(8.104282924459837407218042945106320388339E-1),
	__IEEE854_LONG_DOUBLE_C(5.807730930825886427048038146088828206852E0),
	__IEEE854_LONG_DOUBLE_C(1.795310145936848873627710102199881642939E1),
	__IEEE854_LONG_DOUBLE_C(2.281313316875375733663657188888110605044E1),
	__IEEE854_LONG_DOUBLE_C(1.011242067883822301487154844458322200143E1)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(j0_y0_pzerol) __ATTR_WUNUSED __IEEE854_LONG_DOUBLE_TYPE__
(__LIBCCALL __ieee854_j0_y0_pzerol)(__IEEE854_LONG_DOUBLE_TYPE__ __x) {
	__IEEE854_LONG_DOUBLE_TYPE__ const *__p, *__q;
	__IEEE854_LONG_DOUBLE_TYPE__ __z, __r, __s;
	__int32_t __ix;
	__uint32_t __se, __i0, __i1;
	__LIBM_GET_LDOUBLE_WORDS(__se, __i0, __i1, __x);
	__ix = __se & 0x7fff;
	if (__ix >= 0x4002) {
		/* x >= 8 */
		__p = __LIBM_LOCAL_VALUE(pzero_pR8l);
		__q = __LIBM_LOCAL_VALUE(pzero_pS8l);
	} else {
		__i1 = (__ix << 16) | (__i0 >> 16);
		if (__i1 >= __UINT32_C(0x40019174)) {
			/* x >= 4.54541015625 */
			__p = __LIBM_LOCAL_VALUE(pzero_pR5l);
			__q = __LIBM_LOCAL_VALUE(pzero_pS5l);
		} else if (__i1 >= __UINT32_C(0x4000b6db)) {
			/* x >= 2.85711669921875 */
			__p = __LIBM_LOCAL_VALUE(pzero_pR3l);
			__q = __LIBM_LOCAL_VALUE(pzero_pS3l);
		} else /* if (ix >= 0x4000) */ {
			/* x >= 2 */
			__p = __LIBM_LOCAL_VALUE(pzero_pR2l);
			__q = __LIBM_LOCAL_VALUE(pzero_pS2l);
		}
	}
	__z = __LIBM_LOCAL_VALUE(onel) / (__x * __x);
	__r = __p[0] + __z * (__p[1] + __z * (__p[2] + __z * (__p[3] + __z * (__p[4] + __z * (__p[5] + __z * __p[6])))));
	__s = __q[0] + __z * (__q[1] + __z * (__q[2] + __z * (__q[3] + __z * (__q[4] + __z * (__q[5] + __z)))));
	return (__LIBM_LOCAL_VALUE(onel) + __z * __r / __s);
}

__LIBM_LOCAL_FUNC(j0_y0_qzerol) __ATTR_WUNUSED __IEEE854_LONG_DOUBLE_TYPE__
(__LIBCCALL __ieee854_j0_y0_qzerol)(__IEEE854_LONG_DOUBLE_TYPE__ __x) {
	__IEEE854_LONG_DOUBLE_TYPE__ const *__p, *__q;
	__IEEE854_LONG_DOUBLE_TYPE__ __s, __r, __z;
	__int32_t __ix;
	__uint32_t __se, __i0, __i1;
	__LIBM_GET_LDOUBLE_WORDS(__se, __i0, __i1, __x);
	__ix = __se & 0x7fff;
	if (__ix >= 0x4002) {
		/* x >= 8 */
		__p = __LIBM_LOCAL_VALUE(pzero_qR8l);
		__q = __LIBM_LOCAL_VALUE(pzero_qS8l);
	} else {
		__i1 = (__ix << 16) | (__i0 >> 16);
		if (__i1 >= __UINT32_C(0x40019174)) {
			/* x >= 4.54541015625 */
			__p = __LIBM_LOCAL_VALUE(pzero_qR5l);
			__q = __LIBM_LOCAL_VALUE(pzero_qS5l);
		} else if (__i1 >= __UINT32_C(0x4000b6db)) {
			/* x >= 2.85711669921875 */
			__p = __LIBM_LOCAL_VALUE(pzero_qR3l);
			__q = __LIBM_LOCAL_VALUE(pzero_qS3l);
		} else /* if (ix >= 0x4000) */ {
			/* x >= 2 */
			__p = __LIBM_LOCAL_VALUE(pzero_qR2l);
			__q = __LIBM_LOCAL_VALUE(pzero_qS2l);
		}
	}
	__z = __LIBM_LOCAL_VALUE(onel) / (__x * __x);
	__r = __p[0] + __z * (__p[1] + __z * (__p[2] + __z * (__p[3] + __z * (__p[4] + __z * (__p[5] + __z * __p[6])))));
	__s = __q[0] + __z * (__q[1] + __z * (__q[2] + __z * (__q[3] + __z * (__q[4] + __z * (__q[5] + __z * (__q[6] + __z))))));
	return (-__IEEE854_LONG_DOUBLE_C(0.125) + __z * __r / __s) / __x;
}
#endif /* __IEEE854_LONG_DOUBLE_TYPE__ */

__DECL_END
#endif /* __CC__ */
#endif /* !__NO_FPU */

#endif /* !_LIBM_PZERO_H */
