/* Copyright (c) 2019-2024 Griefer@Work                                       *
 *                                                                            *
 * This software is provided 'as-is', without any express or implied          *
 * warranty. In no event will the authors be held liable for any damages      *
 * arising from the use of this software.                                     *
 *                                                                            *
 * Permission is granted to anyone to use this software for any purpose,      *
 * including commercial applications, and to alter it and redistribute it     *
 * freely, subject to the following restrictions:                             *
 *                                                                            *
 * 1. The origin of this software must not be misrepresented; you must not    *
 *    claim that you wrote the original software. If you use this software    *
 *    in a product, an acknowledgement (see the following) in the product     *
 *    documentation is required:                                              *
 *    Portions Copyright (c) 2019-2024 Griefer@Work                           *
 * 2. Altered source versions must be plainly marked as such, and must not be *
 *    misrepresented as being the original software.                          *
 * 3. This notice may not be removed or altered from any source distribution. *
 */
#ifndef _LIBM_PONE_H
#define _LIBM_PONE_H 1

#include <__crt.h>

#ifndef __NO_FPU
#include <hybrid/__bit.h>
#include <hybrid/typecore.h>

#include <bits/types.h>

#include <libm/fdlibm.h>

#ifdef __CC__
__DECL_BEGIN

#ifdef __IEEE754_FLOAT_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
__LIBM_LOCAL_DECLARE_BEGIN
#ifndef __libm_onef_defined
#define __libm_onef_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, onef, __IEEE754_FLOAT_C(1.0000000000e+00)) /* 0x3F800000 */
#endif /* !__libm_onef_defined */

/* For x >= 8, the asymptotic expansions of pone is
 *    1 + 15/128 s^2 - 4725/2^15 s^4 - ...,    where s = 1/x.
 * We approximate pone by
 *    pone(x) = 1 + (R/S)
 * where  R = pr0 + pr1*s^2 + pr2*s^4 + ... + pr5*s^10
 *      S = 1 + ps0*s^2 + ... + ps4*s^10
 * and
 *    | pone(x)-1-R/S | <= 2  ** ( -60.06) */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_pr8f, 6) {
	/* for x in [inf, 8]=1/[0,0.125] */
	__IEEE754_FLOAT_C(0.0000000000e+00), /* 0x00000000 */
	__IEEE754_FLOAT_C(1.1718750000e-01), /* 0x3df00000 */
	__IEEE754_FLOAT_C(1.3239480972e+01), /* 0x4153d4ea */
	__IEEE754_FLOAT_C(4.1205184937e+02), /* 0x43ce06a3 */
	__IEEE754_FLOAT_C(3.8747453613e+03), /* 0x45722bed */
	__IEEE754_FLOAT_C(7.9144794922e+03)  /* 0x45f753d6 */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_ps8f, 5) {
	__IEEE754_FLOAT_C(1.1420736694e+02), /* 0x42e46a2c */
	__IEEE754_FLOAT_C(3.6509309082e+03), /* 0x45642ee5 */
	__IEEE754_FLOAT_C(3.6956207031e+04), /* 0x47105c35 */
	__IEEE754_FLOAT_C(9.7602796875e+04), /* 0x47bea166 */
	__IEEE754_FLOAT_C(3.0804271484e+04)  /* 0x46f0a88b */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_pr5f, 6) {
	/* for x in [8,4.5454]=1/[0.125,0.22001] */
	__IEEE754_FLOAT_C(1.3199052094e-11), /* 0x2d68333f */
	__IEEE754_FLOAT_C(1.1718749255e-01), /* 0x3defffff */
	__IEEE754_FLOAT_C(6.8027510643e+00), /* 0x40d9b023 */
	__IEEE754_FLOAT_C(1.0830818176e+02), /* 0x42d89dca */
	__IEEE754_FLOAT_C(5.1763616943e+02), /* 0x440168b7 */
	__IEEE754_FLOAT_C(5.2871520996e+02)  /* 0x44042dc6 */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_ps5f, 5) {
	__IEEE754_FLOAT_C(5.9280597687e+01), /* 0x426d1f55 */
	__IEEE754_FLOAT_C(9.9140142822e+02), /* 0x4477d9b1 */
	__IEEE754_FLOAT_C(5.3532670898e+03), /* 0x45a74a23 */
	__IEEE754_FLOAT_C(7.8446904297e+03), /* 0x45f52586 */
	__IEEE754_FLOAT_C(1.5040468750e+03)  /* 0x44bc0180 */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_pr3f, 6) {
	__IEEE754_FLOAT_C(3.0250391081e-09), /* 0x314fe10d */
	__IEEE754_FLOAT_C(1.1718686670e-01), /* 0x3defffab */
	__IEEE754_FLOAT_C(3.9329774380e+00), /* 0x407bb5e7 */
	__IEEE754_FLOAT_C(3.5119403839e+01), /* 0x420c7a45 */
	__IEEE754_FLOAT_C(9.1055007935e+01), /* 0x42b61c2a */
	__IEEE754_FLOAT_C(4.8559066772e+01)  /* 0x42423c7c */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_ps3f, 5) {
	__IEEE754_FLOAT_C(3.4791309357e+01), /* 0x420b2a4d */
	__IEEE754_FLOAT_C(3.3676245117e+02), /* 0x43a86198 */
	__IEEE754_FLOAT_C(1.0468714600e+03), /* 0x4482dbe3 */
	__IEEE754_FLOAT_C(8.9081134033e+02), /* 0x445eb3ed */
	__IEEE754_FLOAT_C(1.0378793335e+02)  /* 0x42cf936c */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_pr2f, 6) {
	/* for x in [2.8570,2]=1/[0.3499,0.5] */
	__IEEE754_FLOAT_C(1.0771083225e-07), /* 0x33e74ea8 */
	__IEEE754_FLOAT_C(1.1717621982e-01), /* 0x3deffa16 */
	__IEEE754_FLOAT_C(2.3685150146e+00), /* 0x401795c0 */
	__IEEE754_FLOAT_C(1.2242610931e+01), /* 0x4143e1bc */
	__IEEE754_FLOAT_C(1.7693971634e+01), /* 0x418d8d41 */
	__IEEE754_FLOAT_C(5.0735230446e+00)  /* 0x40a25a4d */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_ps2f, 5) {
	__IEEE754_FLOAT_C(2.1436485291e+01), /* 0x41ab7dec */
	__IEEE754_FLOAT_C(1.2529022980e+02), /* 0x42fa9499 */
	__IEEE754_FLOAT_C(2.3227647400e+02), /* 0x436846c7 */
	__IEEE754_FLOAT_C(1.1767937469e+02), /* 0x42eb5bd7 */
	__IEEE754_FLOAT_C(8.3646392822e+00)  /* 0x4105d590 */
};

/* For x >= 8, the asymptotic expansions of qone is
 *    3/8 s - 105/1024 s^3 - ..., where s = 1/x.
 * We approximate qone by
 *    qone(x) = s*(0.375 + (R/S))
 * where  R = qr1*s^2 + qr2*s^4 + ... + qr5*s^10
 *      S = 1 + qs1*s^2 + ... + qs6*s^12
 * and
 *    | qone(x)/s -0.375-R/S | <= 2  ** ( -61.13) */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_qr8f, 6) {
	/* for x in [inf, 8]=1/[0,0.125] */
	__IEEE754_FLOAT_C(0.0000000000e+00),  /* 0x00000000 */
	__IEEE754_FLOAT_C(-1.0253906250e-01), /* 0xbdd20000 */
	__IEEE754_FLOAT_C(-1.6271753311e+01), /* 0xc1822c8d */
	__IEEE754_FLOAT_C(-7.5960174561e+02), /* 0xc43de683 */
	__IEEE754_FLOAT_C(-1.1849806641e+04), /* 0xc639273a */
	__IEEE754_FLOAT_C(-4.8438511719e+04)  /* 0xc73d3683 */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_qs8f, 6) {
	__IEEE754_FLOAT_C(1.6139537048e+02), /* 0x43216537 */
	__IEEE754_FLOAT_C(7.8253862305e+03), /* 0x45f48b17 */
	__IEEE754_FLOAT_C(1.3387534375e+05), /* 0x4802bcd6 */
	__IEEE754_FLOAT_C(7.1965775000e+05), /* 0x492fb29c */
	__IEEE754_FLOAT_C(6.6660125000e+05), /* 0x4922be94 */
	__IEEE754_FLOAT_C(-2.9449025000e+05) /* 0xc88fcb48 */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_qr5f, 6) {
	/* for x in [8,4.5454]=1/[0.125,0.22001] */
	__IEEE754_FLOAT_C(-2.0897993405e-11), /* 0xadb7d219 */
	__IEEE754_FLOAT_C(-1.0253904760e-01), /* 0xbdd1fffe */
	__IEEE754_FLOAT_C(-8.0564479828e+00), /* 0xc100e736 */
	__IEEE754_FLOAT_C(-1.8366960144e+02), /* 0xc337ab6b */
	__IEEE754_FLOAT_C(-1.3731937256e+03), /* 0xc4aba633 */
	__IEEE754_FLOAT_C(-2.6124443359e+03)  /* 0xc523471c */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_qs5f, 6) {
	__IEEE754_FLOAT_C(8.1276550293e+01), /* 0x42a28d98 */
	__IEEE754_FLOAT_C(1.9917987061e+03), /* 0x44f8f98f */
	__IEEE754_FLOAT_C(1.7468484375e+04), /* 0x468878f8 */
	__IEEE754_FLOAT_C(4.9851425781e+04), /* 0x4742bb6d */
	__IEEE754_FLOAT_C(2.7948074219e+04), /* 0x46da5826 */
	__IEEE754_FLOAT_C(-4.7191835938e+03) /* 0xc5937978 */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_qr3f, 6) {
	__IEEE754_FLOAT_C(-5.0783124372e-09), /* 0xb1ae7d4f */
	__IEEE754_FLOAT_C(-1.0253783315e-01), /* 0xbdd1ff5b */
	__IEEE754_FLOAT_C(-4.6101160049e+00), /* 0xc0938612 */
	__IEEE754_FLOAT_C(-5.7847221375e+01), /* 0xc267638e */
	__IEEE754_FLOAT_C(-2.2824453735e+02), /* 0xc3643e9a */
	__IEEE754_FLOAT_C(-2.1921012878e+02)  /* 0xc35b35cb */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_qs3f, 6) {
	__IEEE754_FLOAT_C(4.7665153503e+01), /* 0x423ea91e */
	__IEEE754_FLOAT_C(6.7386511230e+02), /* 0x4428775e */
	__IEEE754_FLOAT_C(3.3801528320e+03), /* 0x45534272 */
	__IEEE754_FLOAT_C(5.5477290039e+03), /* 0x45ad5dd5 */
	__IEEE754_FLOAT_C(1.9031191406e+03), /* 0x44ede3d0 */
	__IEEE754_FLOAT_C(-1.3520118713e+02) /* 0xc3073381 */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_qr2f, 6) {
	/* for x in [2.8570,2]=1/[0.3499,0.5] */
	__IEEE754_FLOAT_C(-1.7838172539e-07), /* 0xb43f8932 */
	__IEEE754_FLOAT_C(-1.0251704603e-01), /* 0xbdd1f475 */
	__IEEE754_FLOAT_C(-2.7522056103e+00), /* 0xc0302423 */
	__IEEE754_FLOAT_C(-1.9663616180e+01), /* 0xc19d4f16 */
	__IEEE754_FLOAT_C(-4.2325313568e+01), /* 0xc2294d1f */
	__IEEE754_FLOAT_C(-2.1371921539e+01)  /* 0xc1aaf9b2 */
};

__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, pone_qs2f, 6) {
	__IEEE754_FLOAT_C(2.9533363342e+01), /* 0x41ec4454 */
	__IEEE754_FLOAT_C(2.5298155212e+02), /* 0x437cfb47 */
	__IEEE754_FLOAT_C(7.5750280762e+02), /* 0x443d602e */
	__IEEE754_FLOAT_C(7.3939318848e+02), /* 0x4438d92a */
	__IEEE754_FLOAT_C(1.5594900513e+02), /* 0x431bf2f2 */
	__IEEE754_FLOAT_C(-4.9594988823e+00) /* 0xc09eb437 */
};
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(j1_y1_ponef) __ATTR_WUNUSED __IEEE754_FLOAT_TYPE__
(__LIBCCALL __ieee754_j1_y1_ponef)(__IEEE754_FLOAT_TYPE__ __x) {
	__IEEE754_FLOAT_TYPE__ const *__p, *__q;
	__IEEE754_FLOAT_TYPE__ __z, __r, __s;
	__int32_t __ix;
	__LIBM_GET_FLOAT_WORD(__ix, __x);
	__ix &= __INT32_C(0x7fffffff);
	if (__ix >= __INT32_C(0x41000000)) {
		__p = __LIBM_LOCAL_VALUE(pone_pr8f);
		__q = __LIBM_LOCAL_VALUE(pone_ps8f);
	} else if (__ix >= __INT32_C(0x40f71c58)) {
		__p = __LIBM_LOCAL_VALUE(pone_pr5f);
		__q = __LIBM_LOCAL_VALUE(pone_ps5f);
	} else if (__ix >= __INT32_C(0x4036db68)) {
		__p = __LIBM_LOCAL_VALUE(pone_pr3f);
		__q = __LIBM_LOCAL_VALUE(pone_ps3f);
	} else /* if (ix >= __INT32_C(0x40000000)) */ {
		__p = __LIBM_LOCAL_VALUE(pone_pr2f);
		__q = __LIBM_LOCAL_VALUE(pone_ps2f);
	}
	__z = __LIBM_LOCAL_VALUE(onef) / (__x * __x);
	__r = __p[0] + __z * (__p[1] + __z * (__p[2] + __z * (__p[3] + __z * (__p[4] + __z * __p[5]))));
	__s = __LIBM_LOCAL_VALUE(onef) + __z * (__q[0] + __z * (__q[1] + __z * (__q[2] + __z * (__q[3] + __z * __q[4]))));
	return __LIBM_LOCAL_VALUE(onef) + __r / __s;
}

__LIBM_LOCAL_FUNC(j1_y1_qonef) __ATTR_WUNUSED __IEEE754_FLOAT_TYPE__
(__LIBCCALL __ieee754_j1_y1_qonef)(__IEEE754_FLOAT_TYPE__ __x) {
	__IEEE754_FLOAT_TYPE__ const *__p, *__q;
	__IEEE754_FLOAT_TYPE__ __s, __r, __z;
	__int32_t __ix;
	__LIBM_GET_FLOAT_WORD(__ix, __x);
	__ix &= __INT32_C(0x7fffffff);
	if (__ix >= __INT32_C(0x40200000)) {
		__p = __LIBM_LOCAL_VALUE(pone_qr8f);
		__q = __LIBM_LOCAL_VALUE(pone_qs8f);
	} else if (__ix >= __INT32_C(0x40f71c58)) {
		__p = __LIBM_LOCAL_VALUE(pone_qr5f);
		__q = __LIBM_LOCAL_VALUE(pone_qs5f);
	} else if (__ix >= __INT32_C(0x4036db68)) {
		__p = __LIBM_LOCAL_VALUE(pone_qr3f);
		__q = __LIBM_LOCAL_VALUE(pone_qs3f);
	} else /* if (ix >= __INT32_C(0x40000000)) */ {
		__p = __LIBM_LOCAL_VALUE(pone_qr2f);
		__q = __LIBM_LOCAL_VALUE(pone_qs2f);
	}
	__z = __LIBM_LOCAL_VALUE(onef) / (__x * __x);
	__r = __p[0] + __z * (__p[1] + __z * (__p[2] + __z * (__p[3] + __z * (__p[4] + __z * __p[5]))));
	__s = __LIBM_LOCAL_VALUE(onef) + __z * (__q[0] + __z * (__q[1] + __z * (__q[2] + __z * (__q[3] + __z * (__q[4] + __z * __q[5])))));
	return (.375f + __r / __s) / __x;
}
#endif /* __IEEE754_FLOAT_TYPE__ */


#ifdef __IEEE754_DOUBLE_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunSoft, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
__LIBM_LOCAL_DECLARE_BEGIN
#ifndef __libm_one_defined
#define __libm_one_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, one, __IEEE754_DOUBLE_C(1.00000000000000000000e+00)) /* 0x3FF00000, 0x00000000 */
#endif /* !__libm_one_defined */

/* For x >= 8, the asymptotic expansions of pone is
  *    1 + 15/128 s^2 - 4725/2^15 s^4 - ...,    where s = 1/x.
  * We approximate pone by
  *    pone(x) = 1 + (R/S)
  * where  R = pr0 + pr1*s^2 + pr2*s^4 + ... + pr5*s^10
  *  S = 1 + ps0*s^2 + ... + ps4*s^10
  * and
  *    | pone(x)-1-R/S | <= 2  ** ( -60.06) */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_pr8, 6) {
	/* for x in [inf, 8]=1/[0,0.125] */
	__IEEE754_DOUBLE_C(0.00000000000000000000e+00), /* 0x00000000, 0x00000000 */
	__IEEE754_DOUBLE_C(1.17187499999988647970e-01), /* 0x3FBDFFFF, 0xFFFFFCCE */
	__IEEE754_DOUBLE_C(1.32394806593073575129e+01), /* 0x402A7A9D, 0x357F7FCE */
	__IEEE754_DOUBLE_C(4.12051854307378562225e+02), /* 0x4079C0D4, 0x652EA590 */
	__IEEE754_DOUBLE_C(3.87474538913960532227e+03), /* 0x40AE457D, 0xA3A532CC */
	__IEEE754_DOUBLE_C(7.91447954031891731574e+03)  /* 0x40BEEA7A, 0xC32782DD */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_ps8, 5) {
	__IEEE754_DOUBLE_C(1.14207370375678408436e+02), /* 0x405C8D45, 0x8E656CAC */
	__IEEE754_DOUBLE_C(3.65093083420853463394e+03), /* 0x40AC85DC, 0x964D274F */
	__IEEE754_DOUBLE_C(3.69562060269033463555e+04), /* 0x40E20B86, 0x97C5BB7F */
	__IEEE754_DOUBLE_C(9.76027935934950801311e+04), /* 0x40F7D42C, 0xB28F17BB */
	__IEEE754_DOUBLE_C(3.08042720627888811578e+04)  /* 0x40DE1511, 0x697A0B2D */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_pr5, 6) {
	/* for x in [8,4.5454]=1/[0.125,0.22001] */
	__IEEE754_DOUBLE_C(1.31990519556243522749e-11), /* 0x3DAD0667, 0xDAE1CA7D */
	__IEEE754_DOUBLE_C(1.17187493190614097638e-01), /* 0x3FBDFFFF, 0xE2C10043 */
	__IEEE754_DOUBLE_C(6.80275127868432871736e+00), /* 0x401B3604, 0x6E6315E3 */
	__IEEE754_DOUBLE_C(1.08308182990189109773e+02), /* 0x405B13B9, 0x452602ED */
	__IEEE754_DOUBLE_C(5.17636139533199752805e+02), /* 0x40802D16, 0xD052D649 */
	__IEEE754_DOUBLE_C(5.28715201363337541807e+02)  /* 0x408085B8, 0xBB7E0CB7 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_ps5, 5) {
	__IEEE754_DOUBLE_C(5.92805987221131331921e+01), /* 0x404DA3EA, 0xA8AF633D */
	__IEEE754_DOUBLE_C(9.91401418733614377743e+02), /* 0x408EFB36, 0x1B066701 */
	__IEEE754_DOUBLE_C(5.35326695291487976647e+03), /* 0x40B4E944, 0x5706B6FB */
	__IEEE754_DOUBLE_C(7.84469031749551231769e+03), /* 0x40BEA4B0, 0xB8A5BB15 */
	__IEEE754_DOUBLE_C(1.50404688810361062679e+03)  /* 0x40978030, 0x036F5E51 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_pr3, 6) {
	__IEEE754_DOUBLE_C(3.02503916137373618024e-09), /* 0x3E29FC21, 0xA7AD9EDD */
	__IEEE754_DOUBLE_C(1.17186865567253592491e-01), /* 0x3FBDFFF5, 0x5B21D17B */
	__IEEE754_DOUBLE_C(3.93297750033315640650e+00), /* 0x400F76BC, 0xE85EAD8A */
	__IEEE754_DOUBLE_C(3.51194035591636932736e+01), /* 0x40418F48, 0x9DA6D129 */
	__IEEE754_DOUBLE_C(9.10550110750781271918e+01), /* 0x4056C385, 0x4D2C1837 */
	__IEEE754_DOUBLE_C(4.85590685197364919645e+01)  /* 0x4048478F, 0x8EA83EE5 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_ps3, 5) {
	__IEEE754_DOUBLE_C(3.47913095001251519989e+01), /* 0x40416549, 0xA134069C */
	__IEEE754_DOUBLE_C(3.36762458747825746741e+02), /* 0x40750C33, 0x07F1A75F */
	__IEEE754_DOUBLE_C(1.04687139975775130551e+03), /* 0x40905B7C, 0x5037D523 */
	__IEEE754_DOUBLE_C(8.90811346398256432622e+02), /* 0x408BD67D, 0xA32E31E9 */
	__IEEE754_DOUBLE_C(1.03787932439639277504e+02)  /* 0x4059F26D, 0x7C2EED53 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_pr2, 6) {
	/* for x in [2.8570,2]=1/[0.3499,0.5] */
	__IEEE754_DOUBLE_C(1.07710830106873743082e-07), /* 0x3E7CE9D4, 0xF65544F4 */
	__IEEE754_DOUBLE_C(1.17176219462683348094e-01), /* 0x3FBDFF42, 0xBE760D83 */
	__IEEE754_DOUBLE_C(2.36851496667608785174e+00), /* 0x4002F2B7, 0xF98FAEC0 */
	__IEEE754_DOUBLE_C(1.22426109148261232917e+01), /* 0x40287C37, 0x7F71A964 */
	__IEEE754_DOUBLE_C(1.76939711271687727390e+01), /* 0x4031B1A8, 0x177F8EE2 */
	__IEEE754_DOUBLE_C(5.07352312588818499250e+00)  /* 0x40144B49, 0xA574C1FE */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_ps2, 5) {
	__IEEE754_DOUBLE_C(2.14364859363821409488e+01), /* 0x40356FBD, 0x8AD5ECDC */
	__IEEE754_DOUBLE_C(1.25290227168402751090e+02), /* 0x405F5293, 0x14F92CD5 */
	__IEEE754_DOUBLE_C(2.32276469057162813669e+02), /* 0x406D08D8, 0xD5A2DBD9 */
	__IEEE754_DOUBLE_C(1.17679373287147100768e+02), /* 0x405D6B7A, 0xDA1884A9 */
	__IEEE754_DOUBLE_C(8.36463893371618283368e+00)  /* 0x4020BAB1, 0xF44E5192 */
};

/* For x >= 8, the asymptotic expansions of qone is
 *    3/8 s - 105/1024 s^3 - ..., where s = 1/x.
 * We approximate qone by
 *    qone(x) = s*(0.375 + (R/S))
 * where  R = qr1*s^2 + qr2*s^4 + ... + qr5*s^10
 *  S = 1 + qs1*s^2 + ... + qs6*s^12
 * and
 *    | qone(x)/s -0.375-R/S | <= 2  ** ( -61.13) */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_qr8, 6) {
	/* for x in [inf, 8]=1/[0,0.125] */
	__IEEE754_DOUBLE_C(0.00000000000000000000e+00),  /* 0x00000000, 0x00000000 */
	__IEEE754_DOUBLE_C(-1.02539062499992714161e-01), /* 0xBFBA3FFF, 0xFFFFFDF3 */
	__IEEE754_DOUBLE_C(-1.62717534544589987888e+01), /* 0xC0304591, 0xA26779F7 */
	__IEEE754_DOUBLE_C(-7.59601722513950107896e+02), /* 0xC087BCD0, 0x53E4B576 */
	__IEEE754_DOUBLE_C(-1.18498066702429587167e+04), /* 0xC0C724E7, 0x40F87415 */
	__IEEE754_DOUBLE_C(-4.84385124285750353010e+04)  /* 0xC0E7A6D0, 0x65D09C6A */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_qs8, 6) {
	__IEEE754_DOUBLE_C(1.61395369700722909556e+02), /* 0x40642CA6, 0xDE5BCDE5 */
	__IEEE754_DOUBLE_C(7.82538599923348465381e+03), /* 0x40BE9162, 0xD0D88419 */
	__IEEE754_DOUBLE_C(1.33875336287249578163e+05), /* 0x4100579A, 0xB0B75E98 */
	__IEEE754_DOUBLE_C(7.19657723683240939863e+05), /* 0x4125F653, 0x72869C19 */
	__IEEE754_DOUBLE_C(6.66601232617776375264e+05), /* 0x412457D2, 0x7719AD5C */
	__IEEE754_DOUBLE_C(-2.94490264303834643215e+05) /* 0xC111F969, 0x0EA5AA18 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_qr5, 6) {
	/* for x in [8,4.5454]=1/[0.125,0.22001] */
	__IEEE754_DOUBLE_C(-2.08979931141764104297e-11), /* 0xBDB6FA43, 0x1AA1A098 */
	__IEEE754_DOUBLE_C(-1.02539050241375426231e-01), /* 0xBFBA3FFF, 0xCB597FEF */
	__IEEE754_DOUBLE_C(-8.05644828123936029840e+00), /* 0xC0201CE6, 0xCA03AD4B */
	__IEEE754_DOUBLE_C(-1.83669607474888380239e+02), /* 0xC066F56D, 0x6CA7B9B0 */
	__IEEE754_DOUBLE_C(-1.37319376065508163265e+03), /* 0xC09574C6, 0x6931734F */
	__IEEE754_DOUBLE_C(-2.61244440453215656817e+03)  /* 0xC0A468E3, 0x88FDA79D */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_qs5, 6) {
	__IEEE754_DOUBLE_C(8.12765501384335777857e+01), /* 0x405451B2, 0xFF5A11B2 */
	__IEEE754_DOUBLE_C(1.99179873460485964642e+03), /* 0x409F1F31, 0xE77BF839 */
	__IEEE754_DOUBLE_C(1.74684851924908907677e+04), /* 0x40D10F1F, 0x0D64CE29 */
	__IEEE754_DOUBLE_C(4.98514270910352279316e+04), /* 0x40E8576D, 0xAABAD197 */
	__IEEE754_DOUBLE_C(2.79480751638918118260e+04), /* 0x40DB4B04, 0xCF7C364B */
	__IEEE754_DOUBLE_C(-4.71918354795128470869e+03) /* 0xC0B26F2E, 0xFCFFA004 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_qr3, 6) {
	__IEEE754_DOUBLE_C(-5.07831226461766561369e-09), /* 0xBE35CFA9, 0xD38FC84F */
	__IEEE754_DOUBLE_C(-1.02537829820837089745e-01), /* 0xBFBA3FEB, 0x51AEED54 */
	__IEEE754_DOUBLE_C(-4.61011581139473403113e+00), /* 0xC01270C2, 0x3302D9FF */
	__IEEE754_DOUBLE_C(-5.78472216562783643212e+01), /* 0xC04CEC71, 0xC25D16DA */
	__IEEE754_DOUBLE_C(-2.28244540737631695038e+02), /* 0xC06C87D3, 0x4718D55F */
	__IEEE754_DOUBLE_C(-2.19210128478909325622e+02)  /* 0xC06B66B9, 0x5F5C1BF6 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_qs3, 6) {
	__IEEE754_DOUBLE_C(4.76651550323729509273e+01), /* 0x4047D523, 0xCCD367E4 */
	__IEEE754_DOUBLE_C(6.73865112676699709482e+02), /* 0x40850EEB, 0xC031EE3E */
	__IEEE754_DOUBLE_C(3.38015286679526343505e+03), /* 0x40AA684E, 0x448E7C9A */
	__IEEE754_DOUBLE_C(5.54772909720722782367e+03), /* 0x40B5ABBA, 0xA61D54A6 */
	__IEEE754_DOUBLE_C(1.90311919338810798763e+03), /* 0x409DBC7A, 0x0DD4DF4B */
	__IEEE754_DOUBLE_C(-1.35201191444307340817e+02) /* 0xC060E670, 0x290A311F */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_qr2, 6) {
	/* for x in [2.8570,2]=1/[0.3499,0.5] */
	__IEEE754_DOUBLE_C(-1.78381727510958865572e-07), /* 0xBE87F126, 0x44C626D2 */
	__IEEE754_DOUBLE_C(-1.02517042607985553460e-01), /* 0xBFBA3E8E, 0x9148B010 */
	__IEEE754_DOUBLE_C(-2.75220568278187460720e+00), /* 0xC0060484, 0x69BB4EDA */
	__IEEE754_DOUBLE_C(-1.96636162643703720221e+01), /* 0xC033A9E2, 0xC168907F */
	__IEEE754_DOUBLE_C(-4.23253133372830490089e+01), /* 0xC04529A3, 0xDE104AAA */
	__IEEE754_DOUBLE_C(-2.13719211703704061733e+01)  /* 0xC0355F36, 0x39CF6E52 */
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, pone_qs2, 6) {
	__IEEE754_DOUBLE_C(2.95333629060523854548e+01), /* 0x403D888A, 0x78AE64FF */
	__IEEE754_DOUBLE_C(2.52981549982190529136e+02), /* 0x406F9F68, 0xDB821CBA */
	__IEEE754_DOUBLE_C(7.57502834868645436472e+02), /* 0x4087AC05, 0xCE49A0F7 */
	__IEEE754_DOUBLE_C(7.39393205320467245656e+02), /* 0x40871B25, 0x48D4C029 */
	__IEEE754_DOUBLE_C(1.55949003336666123687e+02), /* 0x40637E5E, 0x3C3ED8D4 */
	__IEEE754_DOUBLE_C(-4.95949898822628210127e+00) /* 0xC013D686, 0xE71BE86B */
};
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(j1_y1_pone) __ATTR_WUNUSED __IEEE754_DOUBLE_TYPE__
(__LIBCCALL __ieee754_j1_y1_pone)(__IEEE754_DOUBLE_TYPE__ __x) {
	__IEEE754_DOUBLE_TYPE__ const *__p, *__q;
	__IEEE754_DOUBLE_TYPE__ __z, __r, __s, __r1, __r2, __r3;
	__IEEE754_DOUBLE_TYPE__ __s1, __s2, __s3, __z2, __z4;
	__int32_t __ix;
	__LIBM_GET_HIGH_WORD(__ix, __x);
	__ix &= __INT32_C(0x7fffffff);
	if (__ix >= __INT32_C(0x41b00000)) {
		return __LIBM_LOCAL_VALUE(one);
	} else if (__ix >= __INT32_C(0x40200000)) {
		__p = __LIBM_LOCAL_VALUE(pone_pr8);
		__q = __LIBM_LOCAL_VALUE(pone_ps8);
	} else if (__ix >= __INT32_C(0x40122E8B)) {
		__p = __LIBM_LOCAL_VALUE(pone_pr5);
		__q = __LIBM_LOCAL_VALUE(pone_ps5);
	} else if (__ix >= __INT32_C(0x4006DB6D)) {
		__p = __LIBM_LOCAL_VALUE(pone_pr3);
		__q = __LIBM_LOCAL_VALUE(pone_ps3);
	} else /* if (ix >= __INT32_C(0x40000000)) */ {
		__p = __LIBM_LOCAL_VALUE(pone_pr2);
		__q = __LIBM_LOCAL_VALUE(pone_ps2);
	}
	__z  = __LIBM_LOCAL_VALUE(one) / (__x * __x);
	__r1 = __p[0] + __z * __p[1];
	__z2 = __z * __z;
	__r2 = __p[2] + __z * __p[3];
	__z4 = __z2 * __z2;
	__r3 = __p[4] + __z * __p[5];
	__r  = __r1 + __z2 * __r2 + __z4 * __r3;
	__s1 = __LIBM_LOCAL_VALUE(one) + __z * __q[0];
	__s2 = __q[1] + __z * __q[2];
	__s3 = __q[3] + __z * __q[4];
	__s  = __s1 + __z2 * __s2 + __z4 * __s3;
	return __LIBM_LOCAL_VALUE(one) + __r / __s;
}

__LIBM_LOCAL_FUNC(j1_y1_qone) __ATTR_WUNUSED __IEEE754_DOUBLE_TYPE__
(__LIBCCALL __ieee754_j1_y1_qone)(__IEEE754_DOUBLE_TYPE__ __x) {
	__IEEE754_DOUBLE_TYPE__ const *__p, *__q;
	__IEEE754_DOUBLE_TYPE__ __s, __r, __z, __r1, __r2, __r3;
	__IEEE754_DOUBLE_TYPE__ __s1, __s2, __s3, __z2, __z4, __z6;
	__int32_t __ix;
	__LIBM_GET_HIGH_WORD(__ix, __x);
	__ix &= __INT32_C(0x7fffffff);
	if (__ix >= __INT32_C(0x41b00000)) {
		return __IEEE754_DOUBLE_C(0.375) / __x;
	} else if (__ix >= __INT32_C(0x40200000)) {
		__p = __LIBM_LOCAL_VALUE(pone_qr8);
		__q = __LIBM_LOCAL_VALUE(pone_qs8);
	} else if (__ix >= __INT32_C(0x40122E8B)) {
		__p = __LIBM_LOCAL_VALUE(pone_qr5);
		__q = __LIBM_LOCAL_VALUE(pone_qs5);
	} else if (__ix >= __INT32_C(0x4006DB6D)) {
		__p = __LIBM_LOCAL_VALUE(pone_qr3);
		__q = __LIBM_LOCAL_VALUE(pone_qs3);
	} else /* if (ix >= __INT32_C(0x40000000)) */
	{
		__p = __LIBM_LOCAL_VALUE(pone_qr2);
		__q = __LIBM_LOCAL_VALUE(pone_qs2);
	}
	__z  = __LIBM_LOCAL_VALUE(one) / (__x * __x);
	__r1 = __p[0] + __z * __p[1];
	__z2 = __z * __z;
	__r2 = __p[2] + __z * __p[3];
	__z4 = __z2 * __z2;
	__r3 = __p[4] + __z * __p[5];
	__z6 = __z4 * __z2;
	__r  = __r1 + __z2 * __r2 + __z4 * __r3;
	__s1 = __LIBM_LOCAL_VALUE(one) + __z * __q[0];
	__s2 = __q[1] + __z * __q[2];
	__s3 = __q[3] + __z * __q[4];
	__s  = __s1 + __z2 * __s2 + __z4 * __s3 + __z6 * __q[5];
	return (__IEEE754_DOUBLE_C(0.375) + __r / __s) / __x;
}
#endif /* __IEEE754_DOUBLE_TYPE__ */


#ifdef __IEEE854_LONG_DOUBLE_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
__LIBM_LOCAL_DECLARE_BEGIN
#ifndef __libm_onel_defined
#define __libm_onel_defined
__LIBM_LOCAL_DECLARE(__IEEE854_LONG_DOUBLE_TYPE__, onel, __IEEE854_LONG_DOUBLE_C(1.0))
#endif /* !__libm_onel_defined */
/* For x >= 8, the asymptotic expansions of pone is
 *    1 + 15/128 s^2 - 4725/2^15 s^4 - ...,    where s = 1/x.
 * We approximate pone by
 *    pone(x) = 1 + (R/S) */
/* J1(x) cosX + Y1(x) sinX = sqrt( 2/(pi x)) P1(x)
 * P1(x) = 1 + z^2 R(z^2), z=1/x
 * 8 <= x <= inf (0 <= z <= 0.125)
 * Peak  relative  error   5.2e-22 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_pr8l, 7) {
	__IEEE854_LONG_DOUBLE_C(8.402048819032978959298664869941375143163E-9),
	__IEEE854_LONG_DOUBLE_C(1.813743245316438056192649247507255996036E-6),
	__IEEE854_LONG_DOUBLE_C(1.260704554112906152344932388588243836276E-4),
	__IEEE854_LONG_DOUBLE_C(3.439294839869103014614229832700986965110E-3),
	__IEEE854_LONG_DOUBLE_C(3.576910849712074184504430254290179501209E-2),
	__IEEE854_LONG_DOUBLE_C(1.131111483254318243139953003461511308672E-1),
	__IEEE854_LONG_DOUBLE_C(4.480715825681029711521286449131671880953E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_ps8l, 6) {
	__IEEE854_LONG_DOUBLE_C(7.169748325574809484893888315707824924354E-8),
	__IEEE854_LONG_DOUBLE_C(1.556549720596672576431813934184403614817E-5),
	__IEEE854_LONG_DOUBLE_C(1.094540125521337139209062035774174565882E-3),
	__IEEE854_LONG_DOUBLE_C(3.060978962596642798560894375281428805840E-2),
	__IEEE854_LONG_DOUBLE_C(3.374146536087205506032643098619414507024E-1),
	__IEEE854_LONG_DOUBLE_C(1.253830208588979001991901126393231302559E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
/* J1(x) cosX + Y1(x) sinX = sqrt( 2/(pi x)) P1(x)
 * P1(x) = 1 + z^2 R(z^2), z=1/x
 * 4.54541015625  <=  x  <=  8
 * Peak relative error 7.7e-22 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_pr5l, 7) {
	__IEEE854_LONG_DOUBLE_C(4.318486887948814529950980396300969247900E-7),
	__IEEE854_LONG_DOUBLE_C(4.715341880798817230333360497524173929315E-5),
	__IEEE854_LONG_DOUBLE_C(1.642719430496086618401091544113220340094E-3),
	__IEEE854_LONG_DOUBLE_C(2.228688005300803935928733750456396149104E-2),
	__IEEE854_LONG_DOUBLE_C(1.142773760804150921573259605730018327162E-1),
	__IEEE854_LONG_DOUBLE_C(1.755576530055079253910829652698703791957E-1),
	__IEEE854_LONG_DOUBLE_C(3.218803858282095929559165965353784980613E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_ps5l, 6) {
	__IEEE854_LONG_DOUBLE_C(3.685108812227721334719884358034713967557E-6),
	__IEEE854_LONG_DOUBLE_C(4.069102509511177498808856515005792027639E-4),
	__IEEE854_LONG_DOUBLE_C(1.449728676496155025507893322405597039816E-2),
	__IEEE854_LONG_DOUBLE_C(2.058869213229520086582695850441194363103E-1),
	__IEEE854_LONG_DOUBLE_C(1.164890985918737148968424972072751066553E0),
	__IEEE854_LONG_DOUBLE_C(2.274776933457009446573027260373361586841E0)
	/* __IEEE854_LONG_DOUBLE_C( 1.000000000000000000000000000000000000000E0), */
};
/* J1(x) cosX + Y1(x) sinX = sqrt( 2/(pi x)) P1(x)
 * P1(x) = 1 + z^2 R(z^2), z=1/x
 * 2.85711669921875 <= x <= 4.54541015625
 * Peak relative error 6.5e-21 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_pr3l, 7) {
	__IEEE854_LONG_DOUBLE_C(1.265251153957366716825382654273326407972E-5),
	__IEEE854_LONG_DOUBLE_C(8.031057269201324914127680782288352574567E-4),
	__IEEE854_LONG_DOUBLE_C(1.581648121115028333661412169396282881035E-2),
	__IEEE854_LONG_DOUBLE_C(1.179534658087796321928362981518645033967E-1),
	__IEEE854_LONG_DOUBLE_C(3.227936912780465219246440724502790727866E-1),
	__IEEE854_LONG_DOUBLE_C(2.559223765418386621748404398017602935764E-1),
	__IEEE854_LONG_DOUBLE_C(2.277136933287817911091370397134882441046E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_ps3l, 6) {
	__IEEE854_LONG_DOUBLE_C(1.079681071833391818661952793568345057548E-4),
	__IEEE854_LONG_DOUBLE_C(6.986017817100477138417481463810841529026E-3),
	__IEEE854_LONG_DOUBLE_C(1.429403701146942509913198539100230540503E-1),
	__IEEE854_LONG_DOUBLE_C(1.148392024337075609460312658938700765074E0),
	__IEEE854_LONG_DOUBLE_C(3.643663015091248720208251490291968840882E0),
	__IEEE854_LONG_DOUBLE_C(3.990702269032018282145100741746633960737E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
/* J1(x) cosX + Y1(x) sinX = sqrt( 2/(pi x)) P1(x)
 * P1(x) = 1 + z^2 R(z^2), z=1/x
 * 2 <= x <=  2.85711669921875
 * Peak relative error 3.5e-21 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_pr2l, 7) {
	__IEEE854_LONG_DOUBLE_C(2.795623248568412225239401141338714516445E-4),
	__IEEE854_LONG_DOUBLE_C(1.092578168441856711925254839815430061135E-2),
	__IEEE854_LONG_DOUBLE_C(1.278024620468953761154963591853679640560E-1),
	__IEEE854_LONG_DOUBLE_C(5.469680473691500673112904286228351988583E-1),
	__IEEE854_LONG_DOUBLE_C(8.313769490922351300461498619045639016059E-1),
	__IEEE854_LONG_DOUBLE_C(3.544176317308370086415403567097130611468E-1),
	__IEEE854_LONG_DOUBLE_C(1.604142674802373041247957048801599740644E-2)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_ps2l, 6) {
	__IEEE854_LONG_DOUBLE_C(2.385605161555183386205027000675875235980E-3),
	__IEEE854_LONG_DOUBLE_C(9.616778294482695283928617708206967248579E-2),
	__IEEE854_LONG_DOUBLE_C(1.195215570959693572089824415393951258510E0),
	__IEEE854_LONG_DOUBLE_C(5.718412857897054829999458736064922974662E0),
	__IEEE854_LONG_DOUBLE_C(1.065626298505499086386584642761602177568E1),
	__IEEE854_LONG_DOUBLE_C(6.809140730053382188468983548092322151791E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
/* For x >= 8, the asymptotic expansions of qone is
 *    3/8 s - 105/1024 s^3 - ..., where s = 1/x.
 * We approximate qone by
 *    qone(x) = s*(0.375 + (R/S)) */
/* Y1(x)cosX - J1(x)sinX = sqrt( 2/(pi x)) Q1(x),
 * Q1(x)  =   z(.375   +   z^2   R(z^2)),   z=1/x
 * 8 <= x <= inf
 * Peak relative error 8.3e-22 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_qr8l, 7) {
	__IEEE854_LONG_DOUBLE_C(-5.691925079044209246015366919809404457380E-10),
	__IEEE854_LONG_DOUBLE_C(-1.632587664706999307871963065396218379137E-7),
	__IEEE854_LONG_DOUBLE_C(-1.577424682764651970003637263552027114600E-5),
	__IEEE854_LONG_DOUBLE_C(-6.377627959241053914770158336842725291713E-4),
	__IEEE854_LONG_DOUBLE_C(-1.087408516779972735197277149494929568768E-2),
	__IEEE854_LONG_DOUBLE_C(-6.854943629378084419631926076882330494217E-2),
	__IEEE854_LONG_DOUBLE_C(-1.055448290469180032312893377152490183203E-1)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_qs8l, 7) {
	__IEEE854_LONG_DOUBLE_C(5.550982172325019811119223916998393907513E-9),
	__IEEE854_LONG_DOUBLE_C(1.607188366646736068460131091130644192244E-6),
	__IEEE854_LONG_DOUBLE_C(1.580792530091386496626494138334505893599E-4),
	__IEEE854_LONG_DOUBLE_C(6.617859900815747303032860443855006056595E-3),
	__IEEE854_LONG_DOUBLE_C(1.212840547336984859952597488863037659161E-1),
	__IEEE854_LONG_DOUBLE_C(9.017885953937234900458186716154005541075E-1),
	__IEEE854_LONG_DOUBLE_C(2.201114489712243262000939120146436167178E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
/* Y1(x)cosX - J1(x)sinX = sqrt( 2/(pi x)) Q1(x),
 * Q1(x)  =   z(.375   +   z^2   R(z^2)),   z=1/x
 * 4.54541015625 <= x <= 8
 * Peak relative error 4.1e-22 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_qr5l, 7) {
	__IEEE854_LONG_DOUBLE_C(-6.719134139179190546324213696633564965983E-8),
	__IEEE854_LONG_DOUBLE_C(-9.467871458774950479909851595678622044140E-6),
	__IEEE854_LONG_DOUBLE_C(-4.429341875348286176950914275723051452838E-4),
	__IEEE854_LONG_DOUBLE_C(-8.539898021757342531563866270278505014487E-3),
	__IEEE854_LONG_DOUBLE_C(-6.818691805848737010422337101409276287170E-2),
	__IEEE854_LONG_DOUBLE_C(-1.964432669771684034858848142418228214855E-1),
	__IEEE854_LONG_DOUBLE_C(-1.333896496989238600119596538299938520726E-1)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_qs5l, 7) {
	__IEEE854_LONG_DOUBLE_C(6.552755584474634766937589285426911075101E-7),
	__IEEE854_LONG_DOUBLE_C(9.410814032118155978663509073200494000589E-5),
	__IEEE854_LONG_DOUBLE_C(4.561677087286518359461609153655021253238E-3),
	__IEEE854_LONG_DOUBLE_C(9.397742096177905170800336715661091535805E-2),
	__IEEE854_LONG_DOUBLE_C(8.518538116671013902180962914473967738771E-1),
	__IEEE854_LONG_DOUBLE_C(3.177729183645800174212539541058292579009E0),
	__IEEE854_LONG_DOUBLE_C(4.006745668510308096259753538973038902990E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
/* Y1(x)cosX - J1(x)sinX = sqrt( 2/(pi x)) Q1(x),
 * Q1(x)  =   z(.375   +   z^2   R(z^2)),   z=1/x
 * 2.85711669921875   <=   x   <=   4.54541015625
 * Peak relative error 2.2e-21 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_qr3l, 7) {
	__IEEE854_LONG_DOUBLE_C(-3.618746299358445926506719188614570588404E-6),
	__IEEE854_LONG_DOUBLE_C(-2.951146018465419674063882650970344502798E-4),
	__IEEE854_LONG_DOUBLE_C(-7.728518171262562194043409753656506795258E-3),
	__IEEE854_LONG_DOUBLE_C(-8.058010968753999435006488158237984014883E-2),
	__IEEE854_LONG_DOUBLE_C(-3.356232856677966691703904770937143483472E-1),
	__IEEE854_LONG_DOUBLE_C(-4.858192581793118040782557808823460276452E-1),
	__IEEE854_LONG_DOUBLE_C(-1.592399251246473643510898335746432479373E-1)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_qs3l, 7) {
	__IEEE854_LONG_DOUBLE_C(3.529139957987837084554591421329876744262E-5),
	__IEEE854_LONG_DOUBLE_C(2.973602667215766676998703687065066180115E-3),
	__IEEE854_LONG_DOUBLE_C(8.273534546240864308494062287908662592100E-2),
	__IEEE854_LONG_DOUBLE_C(9.613359842126507198241321110649974032726E-1),
	__IEEE854_LONG_DOUBLE_C(4.853923697093974370118387947065402707519E0),
	__IEEE854_LONG_DOUBLE_C(1.002671608961669247462020977417828796933E1),
	__IEEE854_LONG_DOUBLE_C(7.028927383922483728931327850683151410267E0)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
/* Y1(x)cosX - J1(x)sinX = sqrt( 2/(pi x)) Q1(x),
 * Q1(x)  =   z(.375   +   z^2   R(z^2)),   z=1/x
 * 2 <= x <= 2.85711669921875
 * Peak relative error 6.9e-22 */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_qr2l, 7) {
	__IEEE854_LONG_DOUBLE_C(-1.372751603025230017220666013816502528318E-4),
	__IEEE854_LONG_DOUBLE_C(-6.879190253347766576229143006767218972834E-3),
	__IEEE854_LONG_DOUBLE_C(-1.061253572090925414598304855316280077828E-1),
	__IEEE854_LONG_DOUBLE_C(-6.262164224345471241219408329354943337214E-1),
	__IEEE854_LONG_DOUBLE_C(-1.423149636514768476376254324731437473915E0),
	__IEEE854_LONG_DOUBLE_C(-1.087955310491078933531734062917489870754E0),
	__IEEE854_LONG_DOUBLE_C(-1.826821119773182847861406108689273719137E-1)
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, pone_qs2l, 7) {
	__IEEE854_LONG_DOUBLE_C(1.338768933634451601814048220627185324007E-3),
	__IEEE854_LONG_DOUBLE_C(7.071099998918497559736318523932241901810E-2),
	__IEEE854_LONG_DOUBLE_C(1.200511429784048632105295629933382142221E0),
	__IEEE854_LONG_DOUBLE_C(8.327301713640367079030141077172031825276E0),
	__IEEE854_LONG_DOUBLE_C(2.468479301872299311658145549931764426840E1),
	__IEEE854_LONG_DOUBLE_C(2.961179686096262083509383820557051621644E1),
	__IEEE854_LONG_DOUBLE_C(1.201402313144305153005639494661767354977E1)
	/* __IEEE854_LONG_DOUBLE_C(1.000000000000000000000000000000000000000E0), */
};
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(j1_y1_ponel) __ATTR_WUNUSED __IEEE854_LONG_DOUBLE_TYPE__
(__LIBCCALL __ieee854_j1_y1_ponel)(__IEEE854_LONG_DOUBLE_TYPE__ __x) {
	__IEEE854_LONG_DOUBLE_TYPE__ const *__p, *__q;
	__IEEE854_LONG_DOUBLE_TYPE__ __z, __r, __s;
	__int32_t __ix;
	__uint32_t __se, __i0, __i1;
	__LIBM_GET_LDOUBLE_WORDS(__se, __i0, __i1, __x);
	__ix = __se & 0x7fff;
	if (__ix >= 0x4002) { /* x >= 8 */
		__p = __LIBM_LOCAL_VALUE(pone_pr8l);
		__q = __LIBM_LOCAL_VALUE(pone_ps8l);
	} else {
		__i1 = (__ix << 16) | (__i0 >> 16);
		if (__i1 >= __UINT32_C(0x40019174)) { /* x >= 4.54541015625 */
			__p = __LIBM_LOCAL_VALUE(pone_pr5l);
			__q = __LIBM_LOCAL_VALUE(pone_ps5l);
		} else if (__i1 >= __UINT32_C(0x4000b6db)) { /* x >= 2.85711669921875 */
			__p = __LIBM_LOCAL_VALUE(pone_pr3l);
			__q = __LIBM_LOCAL_VALUE(pone_ps3l);
		} else /* if (ix >= 0x4000) */ /* x better be >= 2 */ {
			__p = __LIBM_LOCAL_VALUE(pone_pr2l);
			__q = __LIBM_LOCAL_VALUE(pone_ps2l);
		}
	}
	__z = __LIBM_LOCAL_VALUE(onel) / (__x * __x);
	__r = __p[0] + __z * (__p[1] + __z * (__p[2] + __z * (__p[3] + __z * (__p[4] + __z * (__p[5] + __z * __p[6])))));
	__s = __q[0] + __z * (__q[1] + __z * (__q[2] + __z * (__q[3] + __z * (__q[4] + __z * (__q[5] + __z)))));
	return __LIBM_LOCAL_VALUE(onel) + __z * __r / __s;
}

__LIBM_LOCAL_FUNC(j1_y1_qonel) __ATTR_WUNUSED __IEEE854_LONG_DOUBLE_TYPE__
(__LIBCCALL __ieee854_j1_y1_qonel)(__IEEE854_LONG_DOUBLE_TYPE__ __x) {
	__IEEE854_LONG_DOUBLE_TYPE__ const *__p, *__q;
	/* XXX: These  3  were  "static"  --   I  think  that  was  a   typo...
	 * s.a. https://github.com/freemint/fdlibm/issues/5 (Reported as a bug) */
	__IEEE854_LONG_DOUBLE_TYPE__ __s, __r, __z;
	__int32_t __ix;
	__uint32_t __se, __i0, __i1;
	__LIBM_GET_LDOUBLE_WORDS(__se, __i0, __i1, __x);
	__ix = __se & 0x7fff;
	if (__ix >= 0x4002) { /* x >= 8 */
		__p = __LIBM_LOCAL_VALUE(pone_qr8l);
		__q = __LIBM_LOCAL_VALUE(pone_qs8l);
	} else {
		__i1 = (__ix << 16) | (__i0 >> 16);
		if (__i1 >= __UINT32_C(0x40019174)) { /* x >= 4.54541015625 */
			__p = __LIBM_LOCAL_VALUE(pone_qr5l);
			__q = __LIBM_LOCAL_VALUE(pone_qs5l);
		} else if (__i1 >= __UINT32_C(0x4000b6db)) { /* x >= 2.85711669921875 */
			__p = __LIBM_LOCAL_VALUE(pone_qr3l);
			__q = __LIBM_LOCAL_VALUE(pone_qs3l);
		} else /* if (ix >= 0x4000) */ /* x better be >= 2 */ {
			__p = __LIBM_LOCAL_VALUE(pone_qr2l);
			__q = __LIBM_LOCAL_VALUE(pone_qs2l);
		}
	}
	__z = __LIBM_LOCAL_VALUE(onel) / (__x * __x);
	__r = __p[0] + __z * (__p[1] + __z * (__p[2] + __z * (__p[3] + __z * (__p[4] + __z * (__p[5] + __z * __p[6])))));
	__s = __q[0] + __z * (__q[1] + __z * (__q[2] + __z * (__q[3] + __z * (__q[4] + __z * (__q[5] + __z * (__q[6] + __z))))));
	return (__IEEE854_LONG_DOUBLE_C(0.375) + __z * __r / __s) / __x;
}
#endif /* __IEEE854_LONG_DOUBLE_TYPE__ */

__DECL_END
#endif /* __CC__ */
#endif /* !__NO_FPU */

#endif /* !_LIBM_PONE_H */
