/* Copyright (c) 2019-2024 Griefer@Work                                       *
 *                                                                            *
 * This software is provided 'as-is', without any express or implied          *
 * warranty. In no event will the authors be held liable for any damages      *
 * arising from the use of this software.                                     *
 *                                                                            *
 * Permission is granted to anyone to use this software for any purpose,      *
 * including commercial applications, and to alter it and redistribute it     *
 * freely, subject to the following restrictions:                             *
 *                                                                            *
 * 1. The origin of this software must not be misrepresented; you must not    *
 *    claim that you wrote the original software. If you use this software    *
 *    in a product, an acknowledgement (see the following) in the product     *
 *    documentation is required:                                              *
 *    Portions Copyright (c) 2019-2024 Griefer@Work                           *
 * 2. Altered source versions must be plainly marked as such, and must not be *
 *    misrepresented as being the original software.                          *
 * 3. This notice may not be removed or altered from any source distribution. *
 */
#ifndef _LIBM_LOG1P_H
#define _LIBM_LOG1P_H 1

#include <__crt.h>

#ifndef __NO_FPU
#include <hybrid/__bit.h>
#include <hybrid/typecore.h>

#include <bits/types.h>

#include <libm/fdlibm.h>
#include <libm/isnan.h> /* __ieee854_isnanl */
#include <libm/log.h>   /* __ieee854_logl */
#include <libm/round.h> /* __ieee854_roundl */

#ifdef __CC__
__DECL_BEGIN

#ifdef __IEEE754_FLOAT_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
__LIBM_LOCAL_DECLARE_BEGIN
#ifndef __libm_ln2_hif_defined
#define __libm_ln2_hif_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, ln2_hif, __IEEE754_FLOAT_C(6.9313812256e-01)) /* 0x3f317180 */
#endif /* !__libm_ln2_hif_defined */
#ifndef __libm_ln2_lof_defined
#define __libm_ln2_lof_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, ln2_lof, __IEEE754_FLOAT_C(9.0580006145e-06)) /* 0x3717f7d1 */
#endif /* !__libm_ln2_lof_defined */
#ifndef __libm_two25f_defined
#define __libm_two25f_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, two25f, __IEEE754_FLOAT_C(3.355443200e+07)) /* 0x4c000000 */
#endif /* !__libm_two25f_defined */
#ifndef __libm_Lg1f_defined
#define __libm_Lg1f_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, Lg1f, __IEEE754_FLOAT_C(6.6666668653e-01))    /* 3F2AAAAB */
#endif /* !__libm_Lg1f_defined */
#ifndef __libm_Lg2f_defined
#define __libm_Lg2f_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, Lg2f, __IEEE754_FLOAT_C(4.0000000596e-01))    /* 3ECCCCCD */
#endif /* !__libm_Lg2f_defined */
#ifndef __libm_Lg3f_defined
#define __libm_Lg3f_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, Lg3f, __IEEE754_FLOAT_C(2.8571429849e-01))    /* 3E924925 */
#endif /* !__libm_Lg3f_defined */
#ifndef __libm_Lg4f_defined
#define __libm_Lg4f_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, Lg4f, __IEEE754_FLOAT_C(2.2222198546e-01))    /* 3E638E29 */
#endif /* !__libm_Lg4f_defined */
#ifndef __libm_Lg5f_defined
#define __libm_Lg5f_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, Lg5f, __IEEE754_FLOAT_C(1.8183572590e-01))    /* 3E3A3325 */
#endif /* !__libm_Lg5f_defined */
#ifndef __libm_Lg6f_defined
#define __libm_Lg6f_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, Lg6f, __IEEE754_FLOAT_C(1.5313838422e-01))    /* 3E1CD04F */
#endif /* !__libm_Lg6f_defined */
#ifndef __libm_Lg7f_defined
#define __libm_Lg7f_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, Lg7f, __IEEE754_FLOAT_C(1.4798198640e-01))    /* 3E178897 */
#endif /* !__libm_Lg7f_defined */
#ifndef __libm_zerof_defined
#define __libm_zerof_defined
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, zerof, __IEEE754_FLOAT_C(0.0000000000e+00)) /* 0x00000000 */
#endif /* !__libm_zerof_defined */
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(log1pf) __ATTR_WUNUSED __IEEE754_FLOAT_TYPE__
(__LIBCCALL __ieee754_log1pf)(__IEEE754_FLOAT_TYPE__ __x) {
	float __hfsq, __f, __c, __s, __z, __R, __u;
	__int32_t __k, __hx, __hu, __ax;
	__LIBM_GET_FLOAT_WORD(__hx, __x);
	__ax = __hx & __INT32_C(0x7fffffff);
	__k = 1;
	if (__hx < __INT32_C(0x3ed413d7)) {      /* x < 0.41422  */
		if (__ax >= __INT32_C(0x3f800000)) { /* x <= -1.0 */
			if (__x == __IEEE754_FLOAT_C(1.0)) {
				return -__LIBM_LOCAL_VALUE(two25f) / (__x - __x); /* log1p(-1)=+inf */
			} else {
				return (__x - __x) / (__x - __x); /* log1p(x<-1)=NaN */
			}
		}
		if (__ax < __INT32_C(0x31000000)) {                                                     /* |x| < 2**-29 */
			__libm_math_force_eval_r(__IEEE754_FLOAT_TYPE__, __LIBM_LOCAL_VALUE(two25f) + __x); /* raise inexact */
			if (__ax < __INT32_C(0x24800000)) {                                                 /* |x| < 2**-54 */
				return __x;
			} else {
				return __x - __x * __x * __IEEE754_FLOAT_C(0.5);
			}
		}
		if (__hx > 0 || __hx <= __INT32_C(0xbe95f61f)) {
			__k  = 0;
			__f  = __x;
			__hu = 1;
			__c  = 0;
		} /* -0.2929<x<0.41422 */
	}
	if (__hx >= __INT32_C(0x7f800000))
		return __x + __x;
	if (__k != 0) {
		if (__hx < __INT32_C(0x5a000000)) {
			__u = __IEEE754_FLOAT_C(1.0) + __x;
			__LIBM_GET_FLOAT_WORD(__hu, __u);
			__k = (__hu >> 23) - 127;
			/* correction term */
			__c = (__k > 0) ? __IEEE754_FLOAT_C(1.0) - (__u - __x)
			                : __x - (__u - __IEEE754_FLOAT_C(1.0));
			__c /= __u;
		} else {
			__u = __x;
			__LIBM_GET_FLOAT_WORD(__hu, __u);
			__k = (__hu >> 23) - 127;
			__c = 0;
		}
		__hu &= __INT32_C(0x007fffff);
		if (__hu < __INT32_C(0x3504f7)) {
			__LIBM_SET_FLOAT_WORD(__u, __hu | __INT32_C(0x3f800000)); /* normalize u */
		} else {
			__k += 1;
			__LIBM_SET_FLOAT_WORD(__u, __hu | __INT32_C(0x3f000000)); /* normalize u/2 */
			__hu = (__INT32_C(0x00800000) - __hu) >> 2;
		}
		__f = __u - __IEEE754_FLOAT_C(1.0);
	}
	__hfsq = __IEEE754_FLOAT_C(0.5) * __f * __f;

	if (__hu == 0) { /* |f| < 2**-20 */
		if (__f == __LIBM_LOCAL_VALUE(zerof)) {
			if (__k == 0) {
				return __LIBM_LOCAL_VALUE(zerof);
			} else {
				__c += __k * __LIBM_LOCAL_VALUE(ln2_lof);
				return __k * __LIBM_LOCAL_VALUE(ln2_hif) + __c;
			}
		}
		__R = __hfsq * (__IEEE754_FLOAT_C(1.0) - __IEEE754_FLOAT_C(0.66666666666666666) * __f);
		if (__k == 0) {
			return __f - __R;
		} else {
			return __k * __LIBM_LOCAL_VALUE(ln2_hif) - ((__R - (__k * __LIBM_LOCAL_VALUE(ln2_lof) + __c)) - __f);
		}
	}
	__s = __f / (__IEEE754_FLOAT_C(2.0) + __f);
	__z = __s * __s;
	__R = __z * (__LIBM_LOCAL_VALUE(Lg1f) +
	             __z * (__LIBM_LOCAL_VALUE(Lg2f) +
	                    __z * (__LIBM_LOCAL_VALUE(Lg3f) +
	                           __z * (__LIBM_LOCAL_VALUE(Lg4f) +
	                                  __z * (__LIBM_LOCAL_VALUE(Lg5f) +
	                                         __z * (__LIBM_LOCAL_VALUE(Lg6f) +
	                                                __z * __LIBM_LOCAL_VALUE(Lg7f)))))));
	if (__k == 0) {
		return __f - (__hfsq - __s * (__hfsq + __R));
	} else {
		return __k * __LIBM_LOCAL_VALUE(ln2_hif) -
		       ((__hfsq - (__s * (__hfsq + __R) +
		                   (__k * __LIBM_LOCAL_VALUE(ln2_lof) + __c))) -
		        __f);
	}
}
#endif /* __IEEE754_FLOAT_TYPE__ */


#ifdef __IEEE754_DOUBLE_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunSoft, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
__LIBM_LOCAL_DECLARE_BEGIN
#ifndef __libm_ln2_hi_defined
#define __libm_ln2_hi_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, ln2_hi, __IEEE754_DOUBLE_C(6.93147180369123816490e-01)) /* 0x3fe62e42, 0xfee00000 */
#endif /* !__libm_ln2_hi_defined */
#ifndef __libm_ln2_lo_defined
#define __libm_ln2_lo_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, ln2_lo, __IEEE754_DOUBLE_C(1.90821492927058770002e-10)) /* 0x3dea39ef, 0x35793c76 */
#endif /* !__libm_ln2_lo_defined */
#ifndef __libm_two54_defined
#define __libm_two54_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, two54, __IEEE754_DOUBLE_C(1.80143985094819840000e+16)) /* 0x43500000, 0x00000000 */
#endif /* !__libm_two54_defined */
#ifndef __libm_zero_defined
#define __libm_zero_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, zero, __IEEE754_DOUBLE_C(0.00000000000000000000e+00)) /* 0x00000000, 0x00000000 */
#endif /* !__libm_zero_defined */
#ifndef __libm_Lg1_defined
#define __libm_Lg1_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, Lg1, __IEEE754_DOUBLE_C(6.666666666666735130e-01)) /* 3FE55555 55555593 */
#endif /* !__libm_Lg1_defined */
#ifndef __libm_Lg2_defined
#define __libm_Lg2_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, Lg2, __IEEE754_DOUBLE_C(3.999999999940941908e-01)) /* 3FD99999 9997FA04 */
#endif /* !__libm_Lg2_defined */
#ifndef __libm_Lg3_defined
#define __libm_Lg3_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, Lg3, __IEEE754_DOUBLE_C(2.857142874366239149e-01)) /* 3FD24924 94229359 */
#endif /* !__libm_Lg3_defined */
#ifndef __libm_Lg4_defined
#define __libm_Lg4_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, Lg4, __IEEE754_DOUBLE_C(2.222219843214978396e-01)) /* 3FCC71C5 1D8E78AF */
#endif /* !__libm_Lg4_defined */
#ifndef __libm_Lg5_defined
#define __libm_Lg5_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, Lg5, __IEEE754_DOUBLE_C(1.818357216161805012e-01)) /* 3FC74664 96CB03DE */
#endif /* !__libm_Lg5_defined */
#ifndef __libm_Lg6_defined
#define __libm_Lg6_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, Lg6, __IEEE754_DOUBLE_C(1.531383769920937332e-01)) /* 3FC39A09 D078C69F */
#endif /* !__libm_Lg6_defined */
#ifndef __libm_Lg7_defined
#define __libm_Lg7_defined
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, Lg7, __IEEE754_DOUBLE_C(1.479819860511658591e-01)) /* 3FC2F112 DF3E5244 */
#endif /* !__libm_Lg7_defined */
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(log1p) __ATTR_WUNUSED __IEEE754_DOUBLE_TYPE__
(__LIBCCALL __ieee754_log1p)(__IEEE754_DOUBLE_TYPE__ __x) {
	__IEEE754_DOUBLE_TYPE__ __hfsq, __f, __c, __s, __z, __R;
	__IEEE754_DOUBLE_TYPE__ __u, __z2, __z4, __z6;
	__IEEE754_DOUBLE_TYPE__ __R1, __R2, __R3, __R4;
	__int32_t __k, __hx, __hu, __ax;
	__LIBM_GET_HIGH_WORD(__hx, __x);
	__ax = __hx & __INT32_C(0x7fffffff);
	__k = 1;
	if (__hx < __INT32_C(0x3FDA827A)) {      /* x < 0.41422  */
		if (__ax >= __INT32_C(0x3ff00000)) { /* x <= -1.0 */
			if (__x == -__IEEE754_DOUBLE_C(1.0)) {
				return -__LIBM_LOCAL_VALUE(two54) / __LIBM_LOCAL_VALUE(zero); /* log1p(-1)=+inf */
			} else {
				return (__x - __x) / (__x - __x); /* log1p(x<-1)=NaN */
			}
		}
		if (__ax < __INT32_C(0x3e200000)) {                                                     /* |x| < 2**-29 */
			__libm_math_force_eval_r(__IEEE754_DOUBLE_TYPE__, __LIBM_LOCAL_VALUE(two54) + __x); /* raise inexact */
			if (__ax < __INT32_C(0x3c900000)) {                                                 /* |x| < 2**-54 */
				return __x;
			} else {
				return __x - __x * __x * __IEEE754_DOUBLE_C(0.5);
			}
		}
		if (__hx > 0 || __hx <= __INT32_C(0xbfd2bec3)) {
			__k  = 0;
			__f  = __x;
			__hu = 1;
			__c  = 0;
		} /* -0.2929<x<0.41422 */
	} else if (__hx >= __INT32_C(0x7ff00000))
		return __x + __x;
	if (__k != 0) {
		if (__hx < __INT32_C(0x43400000)) {
			__u = __IEEE754_DOUBLE_C(1.0) + __x;
			__LIBM_GET_HIGH_WORD(__hu, __u);
			__k = (__hu >> 20) - 1023;
			__c = (__k > 0) ? __IEEE754_DOUBLE_C(1.0) - (__u - __x)
			                : __x - (__u - __IEEE754_DOUBLE_C(1.0)); /* correction term */
			__c /= __u;
		} else {
			__u = __x;
			__LIBM_GET_HIGH_WORD(__hu, __u);
			__k = (__hu >> 20) - 1023;
			__c = 0;
		}
		__hu &= __INT32_C(0x000fffff);
		if (__hu < __INT32_C(0x6a09e)) {
			__LIBM_SET_HIGH_WORD(__u, __hu | __INT32_C(0x3ff00000)); /* normalize u */
		} else {
			__k += 1;
			__LIBM_SET_HIGH_WORD(__u, __hu | __INT32_C(0x3fe00000)); /* normalize u/2 */
			__hu = (__INT32_C(0x00100000) - __hu) >> 2;
		}
		__f = __u - __IEEE754_DOUBLE_C(1.0);
	}
	__hfsq = __IEEE754_DOUBLE_C(0.5) * __f * __f;
	if (__hu == 0) { /* |f| < 2**-20 */
		if (__f == __LIBM_LOCAL_VALUE(zero)) {
			if (__k == 0) {
				return __LIBM_LOCAL_VALUE(zero);
			} else {
				__c += __k * __LIBM_LOCAL_VALUE(ln2_lo);
				return __k * __LIBM_LOCAL_VALUE(ln2_hi) + __c;
			}
		}
		__R = __hfsq * (__IEEE754_DOUBLE_C(1.0) - __IEEE754_DOUBLE_C(0.66666666666666666) * __f);
		if (__k == 0) {
			return __f - __R;
		} else {
			return __k * __LIBM_LOCAL_VALUE(ln2_hi) -
			       ((__R - (__k * __LIBM_LOCAL_VALUE(ln2_lo) + __c)) - __f);
		}
	}
	__s  = __f / (__IEEE754_DOUBLE_C(2.0) + __f);
	__z  = __s * __s;
	__R1 = __z * __LIBM_LOCAL_VALUE(Lg1);
	__z2 = __z * __z;
	__R2 = __LIBM_LOCAL_VALUE(Lg2) + __z * __LIBM_LOCAL_VALUE(Lg3);
	__z4 = __z2 * __z2;
	__R3 = __LIBM_LOCAL_VALUE(Lg4) + __z * __LIBM_LOCAL_VALUE(Lg5);
	__z6 = __z4 * __z2;
	__R4 = __LIBM_LOCAL_VALUE(Lg6) + __z * __LIBM_LOCAL_VALUE(Lg7);
	__R  = __R1 + __z2 * __R2 + __z4 * __R3 + __z6 * __R4;
	if (__k == 0) {
		return __f - (__hfsq - __s * (__hfsq + __R));
	} else {
		return __k * __LIBM_LOCAL_VALUE(ln2_hi) -
		       ((__hfsq - (__s * (__hfsq + __R) +
		                   (__k * __LIBM_LOCAL_VALUE(ln2_lo) + __c))) -
		        __f);
	}
}
#endif /* __IEEE754_DOUBLE_TYPE__ */


#ifdef __IEEE854_LONG_DOUBLE_TYPE__
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */

/* logl_table[i] = log((i + 128) / 256).
   Computed in GNU clisp through
   (setf (long-float-digits) 128)
   (setq a 0L0)
   (setf (long-float-digits) 256)
   (dotimes (i 385)
   (format t "        ~D,~%"
   (float (log (* (/ (+ i 128) 256) 1L0)) a)))  */
__LIBM_LOCAL_DECLARE_BEGIN
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE854_LONG_DOUBLE_TYPE__, logl_table, 385) {
	__IEEE854_LONG_DOUBLE_C(-0.693147180559945309417232121458176568075),
	__IEEE854_LONG_DOUBLE_C(-0.6853650401178903604697692213970398044),
	__IEEE854_LONG_DOUBLE_C(-0.677642994023980055266378075415729732197),
	__IEEE854_LONG_DOUBLE_C(-0.669980121278410931188432960495886651496),
	__IEEE854_LONG_DOUBLE_C(-0.662375521893191621046203913861404403985),
	__IEEE854_LONG_DOUBLE_C(-0.65482831625780871022347679633437927773),
	__IEEE854_LONG_DOUBLE_C(-0.647337644528651106250552853843513225963),
	__IEEE854_LONG_DOUBLE_C(-0.639902666041133026551361927671647791137),
	__IEEE854_LONG_DOUBLE_C(-0.632522558743510466836625989417756304788),
	__IEEE854_LONG_DOUBLE_C(-0.625196518651437560022666843685547154042),
	__IEEE854_LONG_DOUBLE_C(-0.617923759322357783718626781474514153438),
	__IEEE854_LONG_DOUBLE_C(-0.61070351134887071814907205278986876216),
	__IEEE854_LONG_DOUBLE_C(-0.60353502187025817679728065207969203929),
	__IEEE854_LONG_DOUBLE_C(-0.59641755410139419712166106497071313106),
	__IEEE854_LONG_DOUBLE_C(-0.58935038687830174459117031769420187977),
	__IEEE854_LONG_DOUBLE_C(-0.582332814219655195222425952134964639978),
	__IEEE854_LONG_DOUBLE_C(-0.575364144903561854878438011987654863008),
	__IEEE854_LONG_DOUBLE_C(-0.568443702058988073553825606077313299585),
	__IEEE854_LONG_DOUBLE_C(-0.561570822771226036828515992768693405624),
	__IEEE854_LONG_DOUBLE_C(-0.554744857700826173731906247856527380683),
	__IEEE854_LONG_DOUBLE_C(-0.547965170715447412135297057717612244552),
	__IEEE854_LONG_DOUBLE_C(-0.541231138534103334345428696561292056747),
	__IEEE854_LONG_DOUBLE_C(-0.534542150383306725323860946832334992828),
	__IEEE854_LONG_DOUBLE_C(-0.527897607664638146541620672180936254347),
	__IEEE854_LONG_DOUBLE_C(-0.52129692363328608707713317540302930314),
	__IEEE854_LONG_DOUBLE_C(-0.514739523087127012297831879947234599722),
	__IEEE854_LONG_DOUBLE_C(-0.50822484206593331675332852879892694707),
	__IEEE854_LONG_DOUBLE_C(-0.50175232756031585480793331389686769463),
	__IEEE854_LONG_DOUBLE_C(-0.495321437230025429054660050261215099),
	__IEEE854_LONG_DOUBLE_C(-0.488931639131254417913411735261937295862),
	__IEEE854_LONG_DOUBLE_C(-0.482582411452595671747679308725825054355),
	__IEEE854_LONG_DOUBLE_C(-0.476273242259330949798142595713829069596),
	__IEEE854_LONG_DOUBLE_C(-0.470003629245735553650937031148342064701),
	__IEEE854_LONG_DOUBLE_C(-0.463773079495099479425751396412036696525),
	__IEEE854_LONG_DOUBLE_C(-0.457581109247178400339643902517133157939),
	__IEEE854_LONG_DOUBLE_C(-0.451427243672800141272924605544662667972),
	__IEEE854_LONG_DOUBLE_C(-0.445311016655364052636629355711651820077),
	__IEEE854_LONG_DOUBLE_C(-0.43923197057898186527990882355156990061),
	__IEEE854_LONG_DOUBLE_C(-0.4331896561230192424451526269158655235),
	__IEEE854_LONG_DOUBLE_C(-0.427183632062807368078106194920633178807),
	__IEEE854_LONG_DOUBLE_C(-0.421213465076303550585562626925177406092),
	__IEEE854_LONG_DOUBLE_C(-0.415278729556489003230882088534775334993),
	__IEEE854_LONG_DOUBLE_C(-0.409379007429300711070330899107921801414),
	__IEEE854_LONG_DOUBLE_C(-0.403513887976902632538339065932507598071),
	__IEEE854_LONG_DOUBLE_C(-0.397682967666109433030550215403212372894),
	__IEEE854_LONG_DOUBLE_C(-0.391885849981783528404356583224421075418),
	__IEEE854_LONG_DOUBLE_C(-0.386122145265033447342107580922798666387),
	__IEEE854_LONG_DOUBLE_C(-0.380391470556048421030985561769857535915),
	__IEEE854_LONG_DOUBLE_C(-0.374693449441410693606984907867576972481),
	__IEEE854_LONG_DOUBLE_C(-0.369027711905733333326561361023189215893),
	__IEEE854_LONG_DOUBLE_C(-0.363393894187477327602809309537386757124),
	__IEEE854_LONG_DOUBLE_C(-0.357791638638807479160052541644010369001),
	__IEEE854_LONG_DOUBLE_C(-0.352220593589352099112142921677820359633),
	__IEEE854_LONG_DOUBLE_C(-0.346680413213736728498769933032403617363),
	__IEEE854_LONG_DOUBLE_C(-0.341170757402767124761784665198737642087),
	__IEEE854_LONG_DOUBLE_C(-0.33569129163814153519122263131727209364),
	__IEEE854_LONG_DOUBLE_C(-0.330241686870576856279407775480686721935),
	__IEEE854_LONG_DOUBLE_C(-0.324821619401237656369001967407777741178),
	__IEEE854_LONG_DOUBLE_C(-0.31943077076636122859621528874235306143),
	__IEEE854_LONG_DOUBLE_C(-0.314068827624975851026378775827156709194),
	__IEEE854_LONG_DOUBLE_C(-0.308735481649613269682442058976885699557),
	__IEEE854_LONG_DOUBLE_C(-0.303430429419920096046768517454655701024),
	__IEEE854_LONG_DOUBLE_C(-0.298153372319076331310838085093194799765),
	__IEEE854_LONG_DOUBLE_C(-0.292904016432932602487907019463045397996),
	__IEEE854_LONG_DOUBLE_C(-0.287682072451780927439219005993827431504),
	__IEEE854_LONG_DOUBLE_C(-0.282487255574676923482925918282353780414),
	__IEEE854_LONG_DOUBLE_C(-0.277319285416234343803903228503274262719),
	__IEEE854_LONG_DOUBLE_C(-0.272177885915815673288364959951380595626),
	__IEEE854_LONG_DOUBLE_C(-0.267062785249045246292687241862699949179),
	__IEEE854_LONG_DOUBLE_C(-0.261973715741573968558059642502581569596),
	__IEEE854_LONG_DOUBLE_C(-0.256910413785027239068190798397055267412),
	__IEEE854_LONG_DOUBLE_C(-0.251872619755070079927735679796875342712),
	__IEEE854_LONG_DOUBLE_C(-0.2468600779315257978846419408385075613265),
	__IEEE854_LONG_DOUBLE_C(-0.24187253642048672427253973837916408939),
	__IEEE854_LONG_DOUBLE_C(-0.2369097470783577150364265832942468196375),
	__IEEE854_LONG_DOUBLE_C(-0.2319714654377751430492321958603212094726),
	__IEEE854_LONG_DOUBLE_C(-0.2270574506353460848586128739534071682175),
	__IEEE854_LONG_DOUBLE_C(-0.222167465341154296870334265401817316702),
	__IEEE854_LONG_DOUBLE_C(-0.2173012756899813951520225351537951559),
	__IEEE854_LONG_DOUBLE_C(-0.212458651214193401740613666010165016867),
	__IEEE854_LONG_DOUBLE_C(-0.2076393647782445016154410442673876674964),
	__IEEE854_LONG_DOUBLE_C(-0.202843192514751471266885961812429707545),
	__IEEE854_LONG_DOUBLE_C(-0.1980699137620937948192675366153429027185),
	__IEEE854_LONG_DOUBLE_C(-0.193319311003495979595900706211132426563),
	__IEEE854_LONG_DOUBLE_C(-0.188591169807550022358923589720001638093),
	__IEEE854_LONG_DOUBLE_C(-0.183885278770137362613157202229852743197),
	__IEEE854_LONG_DOUBLE_C(-0.179201429457710992616226033183958974965),
	__IEEE854_LONG_DOUBLE_C(-0.174539416351899677264255125093377869519),
	__IEEE854_LONG_DOUBLE_C(-0.169899036795397472900424896523305726435),
	__IEEE854_LONG_DOUBLE_C(-0.165280090939102924303339903679875604517),
	__IEEE854_LONG_DOUBLE_C(-0.160682381690473465543308397998034325468),
	__IEEE854_LONG_DOUBLE_C(-0.156105714663061654850502877304344269052),
	__IEEE854_LONG_DOUBLE_C(-0.1515498981272009378406898175577424691056),
	__IEEE854_LONG_DOUBLE_C(-0.1470147429618096590348349122269674042104),
	__IEEE854_LONG_DOUBLE_C(-0.142500062607283030157283942253263107981),
	__IEEE854_LONG_DOUBLE_C(-0.1380056730194437167017517619422725179055),
	__IEEE854_LONG_DOUBLE_C(-0.1335313926245226231463436209313499745895),
	__IEEE854_LONG_DOUBLE_C(-0.129077042275142343345847831367985856258),
	__IEEE854_LONG_DOUBLE_C(-0.124642445207276597338493356591214304499),
	__IEEE854_LONG_DOUBLE_C(-0.1202274269981598003244753948319154994493),
	__IEEE854_LONG_DOUBLE_C(-0.115831815525121705099120059938680166568),
	__IEEE854_LONG_DOUBLE_C(-0.1114554409253228268966213677328042273655),
	__IEEE854_LONG_DOUBLE_C(-0.1070981355563671005131126851708522185606),
	__IEEE854_LONG_DOUBLE_C(-0.1027597339577689347753154133345778104976),
	__IEEE854_LONG_DOUBLE_C(-0.098440072813252519902888574928971234883),
	__IEEE854_LONG_DOUBLE_C(-0.094138990913861910035632096996525066015),
	__IEEE854_LONG_DOUBLE_C(-0.0898563291218610470766469347968659624282),
	__IEEE854_LONG_DOUBLE_C(-0.0855919303354035139161469686670511961825),
	__IEEE854_LONG_DOUBLE_C(-0.0813456394539524058873423550293617843895),
	__IEEE854_LONG_DOUBLE_C(-0.077117303344431289769666193261475917783),
	__IEEE854_LONG_DOUBLE_C(-0.072906770808087780565737488890929711303),
	__IEEE854_LONG_DOUBLE_C(-0.0687138925480518083746933774035034481663),
	__IEEE854_LONG_DOUBLE_C(-0.064538521137571171672923915683992928129),
	__IEEE854_LONG_DOUBLE_C(-0.0603805109889074798714456529545968095868),
	__IEEE854_LONG_DOUBLE_C(-0.0562397183228760777967376942769773768851),
	__IEEE854_LONG_DOUBLE_C(-0.0521160011390140183616307870527840213665),
	__IEEE854_LONG_DOUBLE_C(-0.0480092191863606077520036253234446621373),
	__IEEE854_LONG_DOUBLE_C(-0.0439192339348354905263921515528654458042),
	__IEEE854_LONG_DOUBLE_C(-0.0398459085471996706586162402473026835046),
	__IEEE854_LONG_DOUBLE_C(-0.0357891078515852792753420982122404025613),
	__IEEE854_LONG_DOUBLE_C(-0.0317486983145803011569962827485256299276),
	__IEEE854_LONG_DOUBLE_C(-0.0277245480148548604671395114515163869272),
	__IEEE854_LONG_DOUBLE_C(-0.0237165266173160421183468505286730579517),
	__IEEE854_LONG_DOUBLE_C(-0.0197245053477785891192717326571593033246),
	__IEEE854_LONG_DOUBLE_C(-0.015748356968139168607549511460828269521),
	__IEEE854_LONG_DOUBLE_C(-0.0117879557520422404691605618900871263399),
	__IEEE854_LONG_DOUBLE_C(-0.0078431774610258928731840424909435816546),
	__IEEE854_LONG_DOUBLE_C(-0.00391389932113632909231778364357266484272),
	__IEEE854_LONG_DOUBLE_C(0.0),
	__IEEE854_LONG_DOUBLE_C(0.00389864041565732301393734309584290701073),
	__IEEE854_LONG_DOUBLE_C(0.00778214044205494894746290006113676367813),
	__IEEE854_LONG_DOUBLE_C(0.01165061721997527413559144280921434893315),
	__IEEE854_LONG_DOUBLE_C(0.0155041865359652541508540460424468358779),
	__IEEE854_LONG_DOUBLE_C(0.01934296284313093463590553454155047018545),
	__IEEE854_LONG_DOUBLE_C(0.0231670592815343782287991609622899165794),
	__IEEE854_LONG_DOUBLE_C(0.0269765876982020757480692925396595457815),
	__IEEE854_LONG_DOUBLE_C(0.0307716586667536883710282075967721640917),
	__IEEE854_LONG_DOUBLE_C(0.0345523815066597334073715005898328652816),
	__IEEE854_LONG_DOUBLE_C(0.038318864302136599193755325123797290346),
	__IEEE854_LONG_DOUBLE_C(0.042071213920687054375203805926962379448),
	__IEEE854_LONG_DOUBLE_C(0.045809536031294203166679267614663342114),
	__IEEE854_LONG_DOUBLE_C(0.049533935122276630882096208829824573267),
	__IEEE854_LONG_DOUBLE_C(0.0532445145188122828658701937865287769396),
	__IEEE854_LONG_DOUBLE_C(0.0569413764001384247590131015404494943015),
	__IEEE854_LONG_DOUBLE_C(0.0606246218164348425806061320404202632862),
	__IEEE854_LONG_DOUBLE_C(0.0642943507053972572162284502656114944857),
	__IEEE854_LONG_DOUBLE_C(0.0679506619085077493945652777726294140346),
	__IEEE854_LONG_DOUBLE_C(0.071593653187008817925605272752092034269),
	__IEEE854_LONG_DOUBLE_C(0.075223421237587525698605339983662414637),
	__IEEE854_LONG_DOUBLE_C(0.078840061707776024531540577859198294559),
	__IEEE854_LONG_DOUBLE_C(0.082443669211074591268160068668307805914),
	__IEEE854_LONG_DOUBLE_C(0.086034337341803153381797826721996075141),
	__IEEE854_LONG_DOUBLE_C(0.0896121586896871326199514693784845287854),
	__IEEE854_LONG_DOUBLE_C(0.093177224854183289768781353027759396216),
	__IEEE854_LONG_DOUBLE_C(0.096729626458551112295571056487463437015),
	__IEEE854_LONG_DOUBLE_C(0.1002694531636751493081301751297276601964),
	__IEEE854_LONG_DOUBLE_C(0.1037967936816435648260618037639746883066),
	__IEEE854_LONG_DOUBLE_C(0.1073117357890880506671750303711543368066),
	__IEEE854_LONG_DOUBLE_C(0.1108143663402901141948061693232119280986),
	__IEEE854_LONG_DOUBLE_C(0.1143047712800586336342591448151747734094),
	__IEEE854_LONG_DOUBLE_C(0.1177830356563834545387941094705217050686),
	__IEEE854_LONG_DOUBLE_C(0.1212492436328696851612122640808405265723),
	__IEEE854_LONG_DOUBLE_C(0.1247034785009572358634065153808632684918),
	__IEEE854_LONG_DOUBLE_C(0.128145822691930038174109886961074873852),
	__IEEE854_LONG_DOUBLE_C(0.1315763577887192725887161286894831624516),
	__IEEE854_LONG_DOUBLE_C(0.134995164537504830601983291147085645626),
	__IEEE854_LONG_DOUBLE_C(0.138402322859119135685325873601649187393),
	__IEEE854_LONG_DOUBLE_C(0.1417979118602573498789527352804727189846),
	__IEEE854_LONG_DOUBLE_C(0.1451820098444978972819350637405643235226),
	__IEEE854_LONG_DOUBLE_C(0.1485546943231371429098223170672938691604),
	__IEEE854_LONG_DOUBLE_C(0.151916042025841975071803424896884511328),
	__IEEE854_LONG_DOUBLE_C(0.1552661289111239515223833017101021786436),
	__IEEE854_LONG_DOUBLE_C(0.1586050301766385840933711746258415752456),
	__IEEE854_LONG_DOUBLE_C(0.161932820269313253240338285123614220592),
	__IEEE854_LONG_DOUBLE_C(0.165249572895307162875611449277240313729),
	__IEEE854_LONG_DOUBLE_C(0.1685553610298066669415865321701023169345),
	__IEEE854_LONG_DOUBLE_C(0.171850256926659222340098946055147264935),
	__IEEE854_LONG_DOUBLE_C(0.1751343321278491480142914649863898412374),
	__IEEE854_LONG_DOUBLE_C(0.1784076574728182971194002415109419683545),
	__IEEE854_LONG_DOUBLE_C(0.181670303107634678260605595617079739242),
	__IEEE854_LONG_DOUBLE_C(0.184922338494011992663903592659249621006),
	__IEEE854_LONG_DOUBLE_C(0.1881638324181829868259905803105539806714),
	__IEEE854_LONG_DOUBLE_C(0.191394852999629454609298807561308873447),
	__IEEE854_LONG_DOUBLE_C(0.194615467699671658858138593767269731516),
	__IEEE854_LONG_DOUBLE_C(0.1978257433299198803625720711969614690756),
	__IEEE854_LONG_DOUBLE_C(0.201025746060590741340908337591797808969),
	__IEEE854_LONG_DOUBLE_C(0.204215541428690891503820386196239272214),
	__IEEE854_LONG_DOUBLE_C(0.2073951943460705871587455788490062338536),
	__IEEE854_LONG_DOUBLE_C(0.210564769107349637669552812732351513721),
	__IEEE854_LONG_DOUBLE_C(0.2137243293977181388619051976331987647734),
	__IEEE854_LONG_DOUBLE_C(0.216873938300614359619089525744347498479),
	__IEEE854_LONG_DOUBLE_C(0.220013658305282095907358638661628360712),
	__IEEE854_LONG_DOUBLE_C(0.2231435513142097557662950903098345033745),
	__IEEE854_LONG_DOUBLE_C(0.226263678650453389361787082280390161607),
	__IEEE854_LONG_DOUBLE_C(0.229374101064845829991480725046139871551),
	__IEEE854_LONG_DOUBLE_C(0.232474878743094064920705078095567528222),
	__IEEE854_LONG_DOUBLE_C(0.235566071312766909077588218941043410137),
	__IEEE854_LONG_DOUBLE_C(0.2386477378501750099171491363522813392526),
	__IEEE854_LONG_DOUBLE_C(0.241719936887145168144307515913513900104),
	__IEEE854_LONG_DOUBLE_C(0.244782726417690916434704717466314811104),
	__IEEE854_LONG_DOUBLE_C(0.247836163904581256780602765746524747999),
	__IEEE854_LONG_DOUBLE_C(0.25088030628580941658844644154994089393),
	__IEEE854_LONG_DOUBLE_C(0.253915209980963444137323297906606667466),
	__IEEE854_LONG_DOUBLE_C(0.256940930897500425446759867911224262093),
	__IEEE854_LONG_DOUBLE_C(0.259957524436926066972079494542311044577),
	__IEEE854_LONG_DOUBLE_C(0.26296504550088135182072917321108602859),
	__IEEE854_LONG_DOUBLE_C(0.265963548497137941339125926537543389269),
	__IEEE854_LONG_DOUBLE_C(0.268953087345503958932974357924497845489),
	__IEEE854_LONG_DOUBLE_C(0.271933715483641758831669494532999161983),
	__IEEE854_LONG_DOUBLE_C(0.274905485872799249167009582983018668293),
	__IEEE854_LONG_DOUBLE_C(0.277868451003456306186350032923401233082),
	__IEEE854_LONG_DOUBLE_C(0.280822662900887784639519758873134832073),
	__IEEE854_LONG_DOUBLE_C(0.28376817313064459834690122235025476666),
	__IEEE854_LONG_DOUBLE_C(0.286705032803954314653250930842073965668),
	__IEEE854_LONG_DOUBLE_C(0.289633292583042676878893055525668970004),
	__IEEE854_LONG_DOUBLE_C(0.292553002686377439978201258664126644308),
	__IEEE854_LONG_DOUBLE_C(0.295464212893835876386681906054964195182),
	__IEEE854_LONG_DOUBLE_C(0.298366972551797281464900430293496918012),
	__IEEE854_LONG_DOUBLE_C(0.301261330578161781012875538233755492657),
	__IEEE854_LONG_DOUBLE_C(0.304147335467296717015819874720446989991),
	__IEEE854_LONG_DOUBLE_C(0.30702503529491186207512454053537790169),
	__IEEE854_LONG_DOUBLE_C(0.309894477722864687861624550833227164546),
	__IEEE854_LONG_DOUBLE_C(0.31275571000389688838624655968831903216),
	__IEEE854_LONG_DOUBLE_C(0.315608778986303334901366180667483174144),
	__IEEE854_LONG_DOUBLE_C(0.318453731118534615810247213590599595595),
	__IEEE854_LONG_DOUBLE_C(0.321290612453734292057863145522557457887),
	__IEEE854_LONG_DOUBLE_C(0.324119468654211976090670760434987352183),
	__IEEE854_LONG_DOUBLE_C(0.326940344995853320592356894073809191681),
	__IEEE854_LONG_DOUBLE_C(0.329753286372467981814422811920789810952),
	__IEEE854_LONG_DOUBLE_C(0.332558337300076601412275626573419425269),
	__IEEE854_LONG_DOUBLE_C(0.335355541921137830257179579814166199074),
	__IEEE854_LONG_DOUBLE_C(0.338144944008716397710235913939267433111),
	__IEEE854_LONG_DOUBLE_C(0.340926586970593210305089199780356208443),
	__IEEE854_LONG_DOUBLE_C(0.34370051385331844468019789211029452987),
	__IEEE854_LONG_DOUBLE_C(0.346466767346208580918462188425772950712),
	__IEEE854_LONG_DOUBLE_C(0.349225389785288304181275421187371759687),
	__IEEE854_LONG_DOUBLE_C(0.35197642315717818465544745625943892599),
	__IEEE854_LONG_DOUBLE_C(0.354719909102929028355011218999317665826),
	__IEEE854_LONG_DOUBLE_C(0.357455888921803774226009490140904474434),
	__IEEE854_LONG_DOUBLE_C(0.360184403575007796281574967493016620926),
	__IEEE854_LONG_DOUBLE_C(0.362905493689368453137824345977489846141),
	__IEEE854_LONG_DOUBLE_C(0.365619199560964711319396875217046453067),
	__IEEE854_LONG_DOUBLE_C(0.368325561158707653048230154050398826898),
	__IEEE854_LONG_DOUBLE_C(0.371024618127872663911964910806824955394),
	__IEEE854_LONG_DOUBLE_C(0.373716409793584080821016832715823506644),
	__IEEE854_LONG_DOUBLE_C(0.376400975164253065997877633436251593315),
	__IEEE854_LONG_DOUBLE_C(0.379078352934969458390853345631019858882),
	__IEEE854_LONG_DOUBLE_C(0.38174858149084833985966626493567607862),
	__IEEE854_LONG_DOUBLE_C(0.384411698910332039734790062481290868519),
	__IEEE854_LONG_DOUBLE_C(0.387067742968448287898902502261817665695),
	__IEEE854_LONG_DOUBLE_C(0.38971675114002521337046360400352086705),
	__IEEE854_LONG_DOUBLE_C(0.392358760602863872479379611988215363485),
	__IEEE854_LONG_DOUBLE_C(0.39499380824086897810639403636498176831),
	__IEEE854_LONG_DOUBLE_C(0.397621930647138489104829072973405554918),
	__IEEE854_LONG_DOUBLE_C(0.40024316412701270692932510199513117008),
	__IEEE854_LONG_DOUBLE_C(0.402857544701083514655197565487057707577),
	__IEEE854_LONG_DOUBLE_C(0.405465108108164381978013115464349136572),
	__IEEE854_LONG_DOUBLE_C(0.408065889808221748430198682969084124381),
	__IEEE854_LONG_DOUBLE_C(0.410659924985268385934306203175822787661),
	__IEEE854_LONG_DOUBLE_C(0.41324724855021933092547601552548590025),
	__IEEE854_LONG_DOUBLE_C(0.415827895143710965613328892954902305356),
	__IEEE854_LONG_DOUBLE_C(0.418401899138883817510763261966760106515),
	__IEEE854_LONG_DOUBLE_C(0.42096929464412963612886716150679597245),
	__IEEE854_LONG_DOUBLE_C(0.423530115505803295718430478017910109426),
	__IEEE854_LONG_DOUBLE_C(0.426084395310900063124544879595476618897),
	__IEEE854_LONG_DOUBLE_C(0.428632167389698760206812276426639053152),
	__IEEE854_LONG_DOUBLE_C(0.43117346481837134085917247895559499848),
	__IEEE854_LONG_DOUBLE_C(0.433708320421559393435847903042186017095),
	__IEEE854_LONG_DOUBLE_C(0.436236766774918070349041323061121300663),
	__IEEE854_LONG_DOUBLE_C(0.438758836207627937745575058511446738878),
	__IEEE854_LONG_DOUBLE_C(0.441274560804875229489496441661301225362),
	__IEEE854_LONG_DOUBLE_C(0.443783972410300981171768440588146426918),
	__IEEE854_LONG_DOUBLE_C(0.446287102628419511532590180619669006749),
	__IEEE854_LONG_DOUBLE_C(0.448783982827006710512822115683937186274),
	__IEEE854_LONG_DOUBLE_C(0.451274644139458585144692383079012478686),
	__IEEE854_LONG_DOUBLE_C(0.453759117467120506644794794442263270651),
	__IEEE854_LONG_DOUBLE_C(0.456237433481587594380805538163929748437),
	__IEEE854_LONG_DOUBLE_C(0.458709622626976664843883309250877913511),
	__IEEE854_LONG_DOUBLE_C(0.461175715122170166367999925597855358603),
	__IEEE854_LONG_DOUBLE_C(0.463635740963032513092182277331163919118),
	__IEEE854_LONG_DOUBLE_C(0.466089729924599224558619247504769399859),
	__IEEE854_LONG_DOUBLE_C(0.468537711563239270375665237462973542708),
	__IEEE854_LONG_DOUBLE_C(0.470979715218791012546897856056359251373),
	__IEEE854_LONG_DOUBLE_C(0.473415770016672131372578393236978550606),
	__IEEE854_LONG_DOUBLE_C(0.475845904869963914265209586304381412175),
	__IEEE854_LONG_DOUBLE_C(0.478270148481470280383546145497464809096),
	__IEEE854_LONG_DOUBLE_C(0.480688529345751907676618455448011551209),
	__IEEE854_LONG_DOUBLE_C(0.48310107575113582273837458485214554795),
	__IEEE854_LONG_DOUBLE_C(0.485507815781700807801791077190788900579),
	__IEEE854_LONG_DOUBLE_C(0.487908777319238973246173184132656942487),
	__IEEE854_LONG_DOUBLE_C(0.490303988045193838150346159645746860531),
	__IEEE854_LONG_DOUBLE_C(0.492693475442575255695076950020077845328),
	__IEEE854_LONG_DOUBLE_C(0.495077266797851514597964584842833665358),
	__IEEE854_LONG_DOUBLE_C(0.497455389202818942250859256731684928918),
	__IEEE854_LONG_DOUBLE_C(0.499827869556449329821331415247044141512),
	__IEEE854_LONG_DOUBLE_C(0.502194734566715494273584171951812573586),
	__IEEE854_LONG_DOUBLE_C(0.504556010752395287058308531738174929982),
	__IEEE854_LONG_DOUBLE_C(0.506911724444854354113196312660089270034),
	__IEEE854_LONG_DOUBLE_C(0.509261901789807946804074919228323824878),
	__IEEE854_LONG_DOUBLE_C(0.51160656874906207851888487520338193135),
	__IEEE854_LONG_DOUBLE_C(0.51394575110223431680100608827421759311),
	__IEEE854_LONG_DOUBLE_C(0.51627947444845449617281928478756106467),
	__IEEE854_LONG_DOUBLE_C(0.518607764208045632152976996364798698556),
	__IEEE854_LONG_DOUBLE_C(0.520930645624185312409809834659637709188),
	__IEEE854_LONG_DOUBLE_C(0.52324814376454783651680722493487084164),
	__IEEE854_LONG_DOUBLE_C(0.525560283522927371382427602307131424923),
	__IEEE854_LONG_DOUBLE_C(0.527867089620842385113892217778300963557),
	__IEEE854_LONG_DOUBLE_C(0.530168586609121617841419630845212405063),
	__IEEE854_LONG_DOUBLE_C(0.532464798869471843873923723460142242606),
	__IEEE854_LONG_DOUBLE_C(0.534755750616027675477923292032637111077),
	__IEEE854_LONG_DOUBLE_C(0.537041465896883654566729244153832299024),
	__IEEE854_LONG_DOUBLE_C(0.539321968595608874655355158077341155752),
	__IEEE854_LONG_DOUBLE_C(0.54159728243274437157654230390043409897),
	__IEEE854_LONG_DOUBLE_C(0.543867430967283517663338989065998323965),
	__IEEE854_LONG_DOUBLE_C(0.546132437598135650382397209231209163864),
	__IEEE854_LONG_DOUBLE_C(0.548392325565573162748150286179863158565),
	__IEEE854_LONG_DOUBLE_C(0.550647117952662279259948179204913460093),
	__IEEE854_LONG_DOUBLE_C(0.552896837686677737580717902230624314327),
	__IEEE854_LONG_DOUBLE_C(0.55514150754050159271548035951590405017),
	__IEEE854_LONG_DOUBLE_C(0.557381150134006357049816540361233647898),
	__IEEE854_LONG_DOUBLE_C(0.559615787935422686270888500526826593487),
	__IEEE854_LONG_DOUBLE_C(0.561845443262691817915664819160697456814),
	__IEEE854_LONG_DOUBLE_C(0.564070138284802966071384290090190711817),
	__IEEE854_LONG_DOUBLE_C(0.566289895023115872590849979337124343595),
	__IEEE854_LONG_DOUBLE_C(0.568504735352668712078738764866962263577),
	__IEEE854_LONG_DOUBLE_C(0.5707146810034715448536245647415894503),
	__IEEE854_LONG_DOUBLE_C(0.572919753561785509092756726626261068625),
	__IEEE854_LONG_DOUBLE_C(0.575119974471387940421742546569273429365),
	__IEEE854_LONG_DOUBLE_C(0.577315365034823604318112061519496401506),
	__IEEE854_LONG_DOUBLE_C(0.579505946414642223855274409488070989814),
	__IEEE854_LONG_DOUBLE_C(0.58169173963462248252061075372537234071),
	__IEEE854_LONG_DOUBLE_C(0.583872765580982679097413356975291104927),
	__IEEE854_LONG_DOUBLE_C(0.586049045003578208904119436287324349516),
	__IEEE854_LONG_DOUBLE_C(0.588220598517086043034868221609113995052),
	__IEEE854_LONG_DOUBLE_C(0.590387446602176374641916708123598757576),
	__IEEE854_LONG_DOUBLE_C(0.59254960960667159874199020959329739696),
	__IEEE854_LONG_DOUBLE_C(0.594707107746692789514343546529205333192),
	__IEEE854_LONG_DOUBLE_C(0.59685996110779383658731192302565801002),
	__IEEE854_LONG_DOUBLE_C(0.59900818964608339938160002446165150206),
	__IEEE854_LONG_DOUBLE_C(0.601151813189334836191674317068856441547),
	__IEEE854_LONG_DOUBLE_C(0.603290851438084262340585186661310605647),
	__IEEE854_LONG_DOUBLE_C(0.6054253239667168894375677681414899356),
	__IEEE854_LONG_DOUBLE_C(0.607555250224541795501085152791125371894),
	__IEEE854_LONG_DOUBLE_C(0.609680649536855273481833501660588408785),
	__IEEE854_LONG_DOUBLE_C(0.611801541105992903529889766428814783686),
	__IEEE854_LONG_DOUBLE_C(0.613917944012370492196929119645563790777),
	__IEEE854_LONG_DOUBLE_C(0.616029877215514019647565928196700650293),
	__IEEE854_LONG_DOUBLE_C(0.618137359555078733872689126674816271683),
	__IEEE854_LONG_DOUBLE_C(0.620240409751857528851494632567246856773),
	__IEEE854_LONG_DOUBLE_C(0.62233904640877874159710264120869663505),
	__IEEE854_LONG_DOUBLE_C(0.62443328801189350104253874405467311991),
	__IEEE854_LONG_DOUBLE_C(0.626523152931352759778820859734204069282),
	__IEEE854_LONG_DOUBLE_C(0.628608659422374137744308205774183639946),
	__IEEE854_LONG_DOUBLE_C(0.6306898256261987050837261409313532241),
	__IEEE854_LONG_DOUBLE_C(0.63276666957103782954578646850357975849),
	__IEEE854_LONG_DOUBLE_C(0.634839209173010211969493840510489008123),
	__IEEE854_LONG_DOUBLE_C(0.63690746223706923162049442718119919119),
	__IEEE854_LONG_DOUBLE_C(0.63897144645792072137962398326473680873),
	__IEEE854_LONG_DOUBLE_C(0.64103117942093129105560133440539254671),
	__IEEE854_LONG_DOUBLE_C(0.643086678603027315392053859585132960477),
	__IEEE854_LONG_DOUBLE_C(0.645137961373584701665228496134731905937),
	__IEEE854_LONG_DOUBLE_C(0.647185044995309550122320631377863036675),
	__IEEE854_LONG_DOUBLE_C(0.64922794662510981889083996990531112227),
	__IEEE854_LONG_DOUBLE_C(0.651266683314958103396333353349672108398),
	__IEEE854_LONG_DOUBLE_C(0.653301272012745638758615881210873884572),
	__IEEE854_LONG_DOUBLE_C(0.65533172956312763209494967856962559648),
	__IEEE854_LONG_DOUBLE_C(0.657358072708360030141890023245936165513),
	__IEEE854_LONG_DOUBLE_C(0.659380318089127826115336413370955804038),
	__IEEE854_LONG_DOUBLE_C(0.661398482245365008260235838709650938148),
	__IEEE854_LONG_DOUBLE_C(0.66341258161706625109695030429080128179),
	__IEEE854_LONG_DOUBLE_C(0.665422632545090448950092610006660181147),
	__IEEE854_LONG_DOUBLE_C(0.667428651271956189947234166318980478403),
	__IEEE854_LONG_DOUBLE_C(0.669430653942629267298885270929503510123),
	__IEEE854_LONG_DOUBLE_C(0.67142865660530232331713904200189252584),
	__IEEE854_LONG_DOUBLE_C(0.67342267521216672029796038880101726475),
	__IEEE854_LONG_DOUBLE_C(0.67541272562017673108090414397019748722),
	__IEEE854_LONG_DOUBLE_C(0.677398823591806140809682609997348298556),
	__IEEE854_LONG_DOUBLE_C(0.67938098479579735014710062847376425181),
	__IEEE854_LONG_DOUBLE_C(0.681359224807903068948071559568089441735),
	__IEEE854_LONG_DOUBLE_C(0.683333559111620688164363148387750369654),
	__IEEE854_LONG_DOUBLE_C(0.68530400309891941654404807896723298642),
	__IEEE854_LONG_DOUBLE_C(0.687270572070960267497006884394346103924),
	__IEEE854_LONG_DOUBLE_C(0.689233281238808980324914337814603903233),
	__IEEE854_LONG_DOUBLE_C(0.691192145724141958859604629216309755938),
	__IEEE854_LONG_DOUBLE_C(0.693147180559945309417232121458176568075)
};
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(log1pl) __ATTR_WUNUSED __IEEE854_LONG_DOUBLE_TYPE__
(__LIBCCALL __ieee854_log1pl)(__IEEE854_LONG_DOUBLE_TYPE__ __x) {
	if (__ieee854_isnanl(__x))
		return __x;
	if (__x < -__IEEE854_LONG_DOUBLE_C(0.5) || __x > __IEEE854_LONG_DOUBLE_C(1.0))
		return __ieee854_logl(__IEEE854_LONG_DOUBLE_C(1.0) + __x);
	/* Here -0.5 <= x <= 1.0.  */
	if (__x == __IEEE854_LONG_DOUBLE_C(0.0))
		return __x; /* Return a zero with the same sign as x.  */
	{
		__IEEE854_LONG_DOUBLE_TYPE__ __m = __ieee854_roundl(__x * __IEEE854_LONG_DOUBLE_C(256.0));
		__IEEE854_LONG_DOUBLE_TYPE__ __y = ((__x * __IEEE854_LONG_DOUBLE_C(256.0)) - __m) / (__m + __IEEE854_LONG_DOUBLE_C(256.0));
		__IEEE854_LONG_DOUBLE_TYPE__ __z = __y / (__IEEE854_LONG_DOUBLE_C(2.0) + __y);

/* Coefficients of the power series for atanh(z).  */
#define __LIBM_ATANH_COEFF_1  __IEEE854_LONG_DOUBLE_C(1.0)
#define __LIBM_ATANH_COEFF_3  __IEEE854_LONG_DOUBLE_C(0.333333333333333333333333333333333333334)
#define __LIBM_ATANH_COEFF_5  __IEEE854_LONG_DOUBLE_C(0.2)
#define __LIBM_ATANH_COEFF_7  __IEEE854_LONG_DOUBLE_C(0.142857142857142857142857142857142857143)
#define __LIBM_ATANH_COEFF_9  __IEEE854_LONG_DOUBLE_C(0.1111111111111111111111111111111111111113)
#define __LIBM_ATANH_COEFF_11 __IEEE854_LONG_DOUBLE_C(0.090909090909090909090909090909090909091)
#define __LIBM_ATANH_COEFF_13 __IEEE854_LONG_DOUBLE_C(0.076923076923076923076923076923076923077)
#define __LIBM_ATANH_COEFF_15 __IEEE854_LONG_DOUBLE_C(0.066666666666666666666666666666666666667)
		__IEEE854_LONG_DOUBLE_TYPE__ z2      = __z * __z;
		__IEEE854_LONG_DOUBLE_TYPE__ atanh_z = ((((((__LIBM_ATANH_COEFF_13 * z2 +
		                                             __LIBM_ATANH_COEFF_11) *
		                                            z2 +
		                                            __LIBM_ATANH_COEFF_9) *
		                                           z2 +
		                                           __LIBM_ATANH_COEFF_7) *
		                                          z2 +
		                                          __LIBM_ATANH_COEFF_5) *
		                                         z2 +
		                                         __LIBM_ATANH_COEFF_3) *
		                                        z2 +
		                                        __LIBM_ATANH_COEFF_1) *
		                                       __z;
#undef __LIBM_ATANH_COEFF_1
#undef __LIBM_ATANH_COEFF_3
#undef __LIBM_ATANH_COEFF_5
#undef __LIBM_ATANH_COEFF_7
#undef __LIBM_ATANH_COEFF_9
#undef __LIBM_ATANH_COEFF_11
#undef __LIBM_ATANH_COEFF_13
#undef __LIBM_ATANH_COEFF_15
		int mi = (int)__m;
		return __LIBM_LOCAL_VALUE(logl_table)[128 + mi] + 2.0L * atanh_z;
	}
}
#endif /* __IEEE854_LONG_DOUBLE_TYPE__ */

__DECL_END
#endif /* __CC__ */
#endif /* !__NO_FPU */

#endif /* !_LIBM_LOG1P_H */
