/* Copyright (c) 2019-2024 Griefer@Work                                       *
 *                                                                            *
 * This software is provided 'as-is', without any express or implied          *
 * warranty. In no event will the authors be held liable for any damages      *
 * arising from the use of this software.                                     *
 *                                                                            *
 * Permission is granted to anyone to use this software for any purpose,      *
 * including commercial applications, and to alter it and redistribute it     *
 * freely, subject to the following restrictions:                             *
 *                                                                            *
 * 1. The origin of this software must not be misrepresented; you must not    *
 *    claim that you wrote the original software. If you use this software    *
 *    in a product, an acknowledgement (see the following) in the product     *
 *    documentation is required:                                              *
 *    Portions Copyright (c) 2019-2024 Griefer@Work                           *
 * 2. Altered source versions must be plainly marked as such, and must not be *
 *    misrepresented as being the original software.                          *
 * 3. This notice may not be removed or altered from any source distribution. *
 */
#ifndef _LIBM_EXP2_H
#define _LIBM_EXP2_H 1

#include <__crt.h>

#ifndef __NO_FPU
#include <hybrid/typecore.h>

#include <bits/crt/fenv.h>
#include <bits/math-constants.h>
#include <bits/types.h>

#include <libc/fenv.h>
#include <libm/expm1.h>
#include <libm/fabs.h>
#include <libm/fcomp.h>
#include <libm/fdlibm.h>
#include <libm/isinf.h>
#include <libm/isnan.h>
#include <libm/scalbn.h>

#ifdef __CC__
__DECL_BEGIN

#ifdef __IEEE754_FLOAT_TYPE__
/* Single-precision floating point 2^x.
   Copyright (C) 1997-2013 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Geoffrey Keating <geoffk@ozemail.com.au>

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

__LIBM_LOCAL_DECLARE_BEGIN
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, exp2_himarkf, __IEEE754_FLOAT_C(128.0)) /* FLT_MAX_EXP */
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, exp2_lomarkf, __IEEE754_FLOAT_C(-150.0)) /* FLT_MIN_EXP - FLT_MANT_DIG - 1 */
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, TWOM100f, __IEEE754_FLOAT_C(7.88860905e-31))
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, TWO127f,  __IEEE754_FLOAT_C(1.7014118346e+38))
__LIBM_LOCAL_DECLARE(__IEEE754_FLOAT_TYPE__, THREEp14f, __IEEE754_FLOAT_C(49152.0));
/* Accurate tables for exp2f().
   Copyright (C) 1998-2015 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Geoffrey Keating <geoffk@ozemail.com.au>

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */
#define __LIBM_W30f (9.31322575e-10)
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, exp2f_deltatable, 256) {
	__INT32_C(-810) * __LIBM_W30f, __INT32_C(283) * __LIBM_W30f, __INT32_C(-1514) * __LIBM_W30f, __INT32_C(1304) * __LIBM_W30f,
	__INT32_C(-1148) * __LIBM_W30f, __INT32_C(-98) * __LIBM_W30f, __INT32_C(-744) * __LIBM_W30f, __INT32_C(-156) * __LIBM_W30f,
	__INT32_C(-419) * __LIBM_W30f, __INT32_C(-155) * __LIBM_W30f, __INT32_C(474) * __LIBM_W30f, __INT32_C(167) * __LIBM_W30f,
	__INT32_C(-1984) * __LIBM_W30f, __INT32_C(-826) * __LIBM_W30f, __INT32_C(692) * __LIBM_W30f, __INT32_C(781) * __LIBM_W30f,
	__INT32_C(-578) * __LIBM_W30f, __INT32_C(-411) * __LIBM_W30f, __INT32_C(-129) * __LIBM_W30f, __INT32_C(-1500) * __LIBM_W30f,
	__INT32_C(654) * __LIBM_W30f, __INT32_C(-141) * __LIBM_W30f, __INT32_C(-816) * __LIBM_W30f, __INT32_C(-53) * __LIBM_W30f,
	__INT32_C(148) * __LIBM_W30f, __INT32_C(493) * __LIBM_W30f, __INT32_C(-2214) * __LIBM_W30f, __INT32_C(760) * __LIBM_W30f,
	__INT32_C(260) * __LIBM_W30f, __INT32_C(750) * __LIBM_W30f, __INT32_C(-1300) * __LIBM_W30f, __INT32_C(1424) * __LIBM_W30f,
	__INT32_C(-1445) * __LIBM_W30f, __INT32_C(-339) * __LIBM_W30f, __INT32_C(-680) * __LIBM_W30f, __INT32_C(-349) * __LIBM_W30f,
	__INT32_C(-922) * __LIBM_W30f, __INT32_C(531) * __LIBM_W30f, __INT32_C(193) * __LIBM_W30f, __INT32_C(-2892) * __LIBM_W30f,
	__INT32_C(290) * __LIBM_W30f, __INT32_C(-2145) * __LIBM_W30f, __INT32_C(-276) * __LIBM_W30f, __INT32_C(485) * __LIBM_W30f,
	__INT32_C(-695) * __LIBM_W30f, __INT32_C(215) * __LIBM_W30f, __INT32_C(-7093) * __LIBM_W30f, __INT32_C(412) * __LIBM_W30f,
	__INT32_C(-4596) * __LIBM_W30f, __INT32_C(367) * __LIBM_W30f, __INT32_C(592) * __LIBM_W30f, __INT32_C(-615) * __LIBM_W30f,
	__INT32_C(-97) * __LIBM_W30f, __INT32_C(-1066) * __LIBM_W30f, __INT32_C(972) * __LIBM_W30f, __INT32_C(-226) * __LIBM_W30f,
	__INT32_C(-625) * __LIBM_W30f, __INT32_C(-374) * __LIBM_W30f, __INT32_C(-5647) * __LIBM_W30f, __INT32_C(-180) * __LIBM_W30f,
	__INT32_C(20349) * __LIBM_W30f, __INT32_C(-447) * __LIBM_W30f, __INT32_C(111) * __LIBM_W30f, __INT32_C(-4164) * __LIBM_W30f,
	__INT32_C(-87) * __LIBM_W30f, __INT32_C(-21) * __LIBM_W30f, __INT32_C(-251) * __LIBM_W30f, __INT32_C(66) * __LIBM_W30f,
	__INT32_C(-517) * __LIBM_W30f, __INT32_C(2093) * __LIBM_W30f, __INT32_C(-263) * __LIBM_W30f, __INT32_C(182) * __LIBM_W30f,
	__INT32_C(-601) * __LIBM_W30f, __INT32_C(475) * __LIBM_W30f, __INT32_C(-483) * __LIBM_W30f, __INT32_C(-1251) * __LIBM_W30f,
	__INT32_C(-373) * __LIBM_W30f, __INT32_C(1471) * __LIBM_W30f, __INT32_C(-92) * __LIBM_W30f, __INT32_C(-215) * __LIBM_W30f,
	__INT32_C(-97) * __LIBM_W30f, __INT32_C(-190) * __LIBM_W30f, __INT32_C(0) * __LIBM_W30f, __INT32_C(-290) * __LIBM_W30f,
	__INT32_C(-2647) * __LIBM_W30f, __INT32_C(1940) * __LIBM_W30f, __INT32_C(-582) * __LIBM_W30f, __INT32_C(28) * __LIBM_W30f,
	__INT32_C(833) * __LIBM_W30f, __INT32_C(1493) * __LIBM_W30f, __INT32_C(34) * __LIBM_W30f, __INT32_C(321) * __LIBM_W30f,
	__INT32_C(3327) * __LIBM_W30f, __INT32_C(-35) * __LIBM_W30f, __INT32_C(177) * __LIBM_W30f, __INT32_C(-135) * __LIBM_W30f,
	__INT32_C(-796) * __LIBM_W30f, __INT32_C(-428) * __LIBM_W30f, __INT32_C(129) * __LIBM_W30f, __INT32_C(9332) * __LIBM_W30f,
	__INT32_C(-12) * __LIBM_W30f, __INT32_C(-69) * __LIBM_W30f, __INT32_C(-1743) * __LIBM_W30f, __INT32_C(6508) * __LIBM_W30f,
	__INT32_C(-60) * __LIBM_W30f, __INT32_C(359) * __LIBM_W30f, __INT32_C(43447) * __LIBM_W30f, __INT32_C(15) * __LIBM_W30f,
	__INT32_C(-23) * __LIBM_W30f, __INT32_C(-305) * __LIBM_W30f, __INT32_C(-375) * __LIBM_W30f, __INT32_C(-652) * __LIBM_W30f,
	__INT32_C(667) * __LIBM_W30f, __INT32_C(269) * __LIBM_W30f, __INT32_C(-1575) * __LIBM_W30f, __INT32_C(185) * __LIBM_W30f,
	__INT32_C(-329) * __LIBM_W30f, __INT32_C(200) * __LIBM_W30f, __INT32_C(6002) * __LIBM_W30f, __INT32_C(163) * __LIBM_W30f,
	__INT32_C(-647) * __LIBM_W30f, __INT32_C(19) * __LIBM_W30f, __INT32_C(-603) * __LIBM_W30f, __INT32_C(-755) * __LIBM_W30f,
	__INT32_C(742) * __LIBM_W30f, __INT32_C(-438) * __LIBM_W30f, __INT32_C(3587) * __LIBM_W30f, __INT32_C(2560) * __LIBM_W30f,
	__INT32_C(0) * __LIBM_W30f, __INT32_C(-520) * __LIBM_W30f, __INT32_C(-241) * __LIBM_W30f, __INT32_C(-299) * __LIBM_W30f,
	__INT32_C(-1270) * __LIBM_W30f, __INT32_C(-991) * __LIBM_W30f, __INT32_C(-1138) * __LIBM_W30f, __INT32_C(255) * __LIBM_W30f,
	__INT32_C(-1192) * __LIBM_W30f, __INT32_C(1722) * __LIBM_W30f, __INT32_C(1023) * __LIBM_W30f, __INT32_C(3700) * __LIBM_W30f,
	__INT32_C(-1388) * __LIBM_W30f, __INT32_C(-1551) * __LIBM_W30f, __INT32_C(-2549) * __LIBM_W30f, __INT32_C(27) * __LIBM_W30f,
	__INT32_C(282) * __LIBM_W30f, __INT32_C(673) * __LIBM_W30f, __INT32_C(113) * __LIBM_W30f, __INT32_C(1561) * __LIBM_W30f,
	__INT32_C(72) * __LIBM_W30f, __INT32_C(873) * __LIBM_W30f, __INT32_C(87) * __LIBM_W30f, __INT32_C(-395) * __LIBM_W30f,
	__INT32_C(-433) * __LIBM_W30f, __INT32_C(629) * __LIBM_W30f, __INT32_C(3440) * __LIBM_W30f, __INT32_C(-284) * __LIBM_W30f,
	__INT32_C(-592) * __LIBM_W30f, __INT32_C(-103) * __LIBM_W30f, __INT32_C(-46) * __LIBM_W30f, __INT32_C(-3844) * __LIBM_W30f,
	__INT32_C(1712) * __LIBM_W30f, __INT32_C(303) * __LIBM_W30f, __INT32_C(1555) * __LIBM_W30f, __INT32_C(-631) * __LIBM_W30f,
	__INT32_C(-1400) * __LIBM_W30f, __INT32_C(-961) * __LIBM_W30f, __INT32_C(-854) * __LIBM_W30f, __INT32_C(-276) * __LIBM_W30f,
	__INT32_C(407) * __LIBM_W30f, __INT32_C(833) * __LIBM_W30f, __INT32_C(-345) * __LIBM_W30f, __INT32_C(-1501) * __LIBM_W30f,
	__INT32_C(121) * __LIBM_W30f, __INT32_C(-1581) * __LIBM_W30f, __INT32_C(400) * __LIBM_W30f, __INT32_C(150) * __LIBM_W30f,
	__INT32_C(1224) * __LIBM_W30f, __INT32_C(-139) * __LIBM_W30f, __INT32_C(-563) * __LIBM_W30f, __INT32_C(879) * __LIBM_W30f,
	__INT32_C(933) * __LIBM_W30f, __INT32_C(2939) * __LIBM_W30f, __INT32_C(788) * __LIBM_W30f, __INT32_C(211) * __LIBM_W30f,
	__INT32_C(530) * __LIBM_W30f, __INT32_C(-192) * __LIBM_W30f, __INT32_C(706) * __LIBM_W30f, __INT32_C(-13347) * __LIBM_W30f,
	__INT32_C(1065) * __LIBM_W30f, __INT32_C(3) * __LIBM_W30f, __INT32_C(111) * __LIBM_W30f, __INT32_C(-208) * __LIBM_W30f,
	__INT32_C(-360) * __LIBM_W30f, __INT32_C(-532) * __LIBM_W30f, __INT32_C(-291) * __LIBM_W30f, __INT32_C(483) * __LIBM_W30f,
	__INT32_C(987) * __LIBM_W30f, __INT32_C(-33) * __LIBM_W30f, __INT32_C(-1373) * __LIBM_W30f, __INT32_C(-166) * __LIBM_W30f,
	__INT32_C(-1174) * __LIBM_W30f, __INT32_C(-3955) * __LIBM_W30f, __INT32_C(1601) * __LIBM_W30f, __INT32_C(-280) * __LIBM_W30f,
	__INT32_C(1405) * __LIBM_W30f, __INT32_C(600) * __LIBM_W30f, __INT32_C(-1659) * __LIBM_W30f, __INT32_C(-23) * __LIBM_W30f,
	__INT32_C(390) * __LIBM_W30f, __INT32_C(449) * __LIBM_W30f, __INT32_C(570) * __LIBM_W30f, __INT32_C(-13143) * __LIBM_W30f,
	__INT32_C(-9) * __LIBM_W30f, __INT32_C(-1646) * __LIBM_W30f, __INT32_C(1201) * __LIBM_W30f, __INT32_C(294) * __LIBM_W30f,
	__INT32_C(2181) * __LIBM_W30f, __INT32_C(-1173) * __LIBM_W30f, __INT32_C(1388) * __LIBM_W30f, __INT32_C(-4504) * __LIBM_W30f,
	__INT32_C(190) * __LIBM_W30f, __INT32_C(-2304) * __LIBM_W30f, __INT32_C(211) * __LIBM_W30f, __INT32_C(239) * __LIBM_W30f,
	__INT32_C(48) * __LIBM_W30f, __INT32_C(-817) * __LIBM_W30f, __INT32_C(1018) * __LIBM_W30f, __INT32_C(1828) * __LIBM_W30f,
	__INT32_C(-663) * __LIBM_W30f, __INT32_C(1408) * __LIBM_W30f, __INT32_C(408) * __LIBM_W30f, __INT32_C(-36) * __LIBM_W30f,
	__INT32_C(1295) * __LIBM_W30f, __INT32_C(-230) * __LIBM_W30f, __INT32_C(1341) * __LIBM_W30f, __INT32_C(9) * __LIBM_W30f,
	__INT32_C(40) * __LIBM_W30f, __INT32_C(705) * __LIBM_W30f, __INT32_C(186) * __LIBM_W30f, __INT32_C(376) * __LIBM_W30f,
	__INT32_C(557) * __LIBM_W30f, __INT32_C(5866) * __LIBM_W30f, __INT32_C(363) * __LIBM_W30f, __INT32_C(-1558) * __LIBM_W30f,
	__INT32_C(718) * __LIBM_W30f, __INT32_C(669) * __LIBM_W30f, __INT32_C(1369) * __LIBM_W30f, __INT32_C(-2972) * __LIBM_W30f,
	__INT32_C(-468) * __LIBM_W30f, __INT32_C(-121) * __LIBM_W30f, __INT32_C(-219) * __LIBM_W30f, __INT32_C(667) * __LIBM_W30f,
	__INT32_C(29954) * __LIBM_W30f, __INT32_C(366) * __LIBM_W30f, __INT32_C(48) * __LIBM_W30f, __INT32_C(-203) * __LIBM_W30f
};
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, exp2f_atable, 256) {
	__IEEE754_FLOAT_C(0.707106411447), /* 0x0.b504ecfff */
	__IEEE754_FLOAT_C(0.709024071690), /* 0x0.b58299fff */
	__IEEE754_FLOAT_C(0.710945606239), /* 0x0.b60088000 */
	__IEEE754_FLOAT_C(0.712874472142), /* 0x0.b67ef1000 */
	__IEEE754_FLOAT_C(0.714806139464), /* 0x0.b6fd88fff */
	__IEEE754_FLOAT_C(0.716744661340), /* 0x0.b77c94000 */
	__IEEE754_FLOAT_C(0.718687653549), /* 0x0.b7fbea000 */
	__IEEE754_FLOAT_C(0.720636486992), /* 0x0.b87ba1fff */
	__IEEE754_FLOAT_C(0.722590208040), /* 0x0.b8fbabfff */
	__IEEE754_FLOAT_C(0.724549472323), /* 0x0.b97c12fff */
	__IEEE754_FLOAT_C(0.726514220228), /* 0x0.b9fcd5fff */
	__IEEE754_FLOAT_C(0.728483855735), /* 0x0.ba7deb000 */
	__IEEE754_FLOAT_C(0.730457961549), /* 0x0.baff4afff */
	__IEEE754_FLOAT_C(0.732438981522), /* 0x0.bb811efff */
	__IEEE754_FLOAT_C(0.734425544748), /* 0x0.bc0350000 */
	__IEEE754_FLOAT_C(0.736416816713), /* 0x0.bc85d0000 */
	__IEEE754_FLOAT_C(0.738412797450), /* 0x0.bd089efff */
	__IEEE754_FLOAT_C(0.740414917465), /* 0x0.bd8bd4fff */
	__IEEE754_FLOAT_C(0.742422521111), /* 0x0.be0f66fff */
	__IEEE754_FLOAT_C(0.744434773914), /* 0x0.be9346fff */
	__IEEE754_FLOAT_C(0.746454179287), /* 0x0.bf179f000 */
	__IEEE754_FLOAT_C(0.748477637755), /* 0x0.bf9c3afff */
	__IEEE754_FLOAT_C(0.750506639473), /* 0x0.c02133fff */
	__IEEE754_FLOAT_C(0.752541840064), /* 0x0.c0a694fff */
	__IEEE754_FLOAT_C(0.754582285889), /* 0x0.c12c4e000 */
	__IEEE754_FLOAT_C(0.756628334525), /* 0x0.c1b265000 */
	__IEEE754_FLOAT_C(0.758678436269), /* 0x0.c238bffff */
	__IEEE754_FLOAT_C(0.760736882681), /* 0x0.c2bfa6fff */
	__IEEE754_FLOAT_C(0.762799203401), /* 0x0.c346cf000 */
	__IEEE754_FLOAT_C(0.764867603790), /* 0x0.c3ce5d000 */
	__IEEE754_FLOAT_C(0.766940355298), /* 0x0.c45633fff */
	__IEEE754_FLOAT_C(0.769021093841), /* 0x0.c4de90fff */
	__IEEE754_FLOAT_C(0.771104693409), /* 0x0.c5671dfff */
	__IEEE754_FLOAT_C(0.773195922364), /* 0x0.c5f02afff */
	__IEEE754_FLOAT_C(0.775292098512), /* 0x0.c6798afff */
	__IEEE754_FLOAT_C(0.777394294745), /* 0x0.c70350000 */
	__IEEE754_FLOAT_C(0.779501736166), /* 0x0.c78d6d000 */
	__IEEE754_FLOAT_C(0.781615912910), /* 0x0.c817fafff */
	__IEEE754_FLOAT_C(0.783734917628), /* 0x0.c8a2d9fff */
	__IEEE754_FLOAT_C(0.785858273516), /* 0x0.c92e02000 */
	__IEEE754_FLOAT_C(0.787990570071), /* 0x0.c9b9c0000 */
	__IEEE754_FLOAT_C(0.790125787245), /* 0x0.ca45aefff */
	__IEEE754_FLOAT_C(0.792268991467), /* 0x0.cad223fff */
	__IEEE754_FLOAT_C(0.794417440881), /* 0x0.cb5ef0fff */
	__IEEE754_FLOAT_C(0.796570718287), /* 0x0.cbec0efff */
	__IEEE754_FLOAT_C(0.798730909811), /* 0x0.cc79a0fff */
	__IEEE754_FLOAT_C(0.800892710672), /* 0x0.cd074dfff */
	__IEEE754_FLOAT_C(0.803068041795), /* 0x0.cd95ddfff */
	__IEEE754_FLOAT_C(0.805242776881), /* 0x0.ce2464000 */
	__IEEE754_FLOAT_C(0.807428598393), /* 0x0.ceb3a3fff */
	__IEEE754_FLOAT_C(0.809617877002), /* 0x0.cf431dfff */
	__IEEE754_FLOAT_C(0.811812341211), /* 0x0.cfd2eefff */
	__IEEE754_FLOAT_C(0.814013659956), /* 0x0.d06333000 */
	__IEEE754_FLOAT_C(0.816220164311), /* 0x0.d0f3ce000 */
	__IEEE754_FLOAT_C(0.818434238424), /* 0x0.d184e7fff */
	__IEEE754_FLOAT_C(0.820652604094), /* 0x0.d21649fff */
	__IEEE754_FLOAT_C(0.822877407074), /* 0x0.d2a818000 */
	__IEEE754_FLOAT_C(0.825108587751), /* 0x0.d33a51000 */
	__IEEE754_FLOAT_C(0.827342867839), /* 0x0.d3ccbdfff */
	__IEEE754_FLOAT_C(0.829588949684), /* 0x0.d45ff1000 */
	__IEEE754_FLOAT_C(0.831849217401), /* 0x0.d4f411fff */
	__IEEE754_FLOAT_C(0.834093391880), /* 0x0.d58724fff */
	__IEEE754_FLOAT_C(0.836355149750), /* 0x0.d61b5f000 */
	__IEEE754_FLOAT_C(0.838620424257), /* 0x0.d6afd3fff */
	__IEEE754_FLOAT_C(0.840896368027), /* 0x0.d744fc000 */
	__IEEE754_FLOAT_C(0.843176305293), /* 0x0.d7da66fff */
	__IEEE754_FLOAT_C(0.845462262643), /* 0x0.d87037000 */
	__IEEE754_FLOAT_C(0.847754716864), /* 0x0.d90673fff */
	__IEEE754_FLOAT_C(0.850052893157), /* 0x0.d99d10fff */
	__IEEE754_FLOAT_C(0.852359056469), /* 0x0.da3433fff */
	__IEEE754_FLOAT_C(0.854668736446), /* 0x0.dacb91fff */
	__IEEE754_FLOAT_C(0.856986224651), /* 0x0.db6373000 */
	__IEEE754_FLOAT_C(0.859309315673), /* 0x0.dbfbb1fff */
	__IEEE754_FLOAT_C(0.861639738080), /* 0x0.dc946bfff */
	__IEEE754_FLOAT_C(0.863975346095), /* 0x0.dd2d7d000 */
	__IEEE754_FLOAT_C(0.866317391394), /* 0x0.ddc6f9fff */
	__IEEE754_FLOAT_C(0.868666708472), /* 0x0.de60f1000 */
	__IEEE754_FLOAT_C(0.871022939695), /* 0x0.defb5c000 */
	__IEEE754_FLOAT_C(0.873383641229), /* 0x0.df9611fff */
	__IEEE754_FLOAT_C(0.875751554968), /* 0x0.e03141000 */
	__IEEE754_FLOAT_C(0.878126025200), /* 0x0.e0ccde000 */
	__IEEE754_FLOAT_C(0.880506813521), /* 0x0.e168e4fff */
	__IEEE754_FLOAT_C(0.882894217966), /* 0x0.e2055afff */
	__IEEE754_FLOAT_C(0.885287821299), /* 0x0.e2a239000 */
	__IEEE754_FLOAT_C(0.887686729423), /* 0x0.e33f6ffff */
	__IEEE754_FLOAT_C(0.890096127973), /* 0x0.e3dd56fff */
	__IEEE754_FLOAT_C(0.892507970338), /* 0x0.e47b67000 */
	__IEEE754_FLOAT_C(0.894928157336), /* 0x0.e51a03000 */
	__IEEE754_FLOAT_C(0.897355020043), /* 0x0.e5b90efff */
	__IEEE754_FLOAT_C(0.899788379682), /* 0x0.e65888000 */
	__IEEE754_FLOAT_C(0.902227103705), /* 0x0.e6f85afff */
	__IEEE754_FLOAT_C(0.904673457151), /* 0x0.e798ae000 */
	__IEEE754_FLOAT_C(0.907128036008), /* 0x0.e8398afff */
	__IEEE754_FLOAT_C(0.909585535528), /* 0x0.e8da99000 */
	__IEEE754_FLOAT_C(0.912051796915), /* 0x0.e97c3a000 */
	__IEEE754_FLOAT_C(0.914524436003), /* 0x0.ea1e46000 */
	__IEEE754_FLOAT_C(0.917003571999), /* 0x0.eac0bf000 */
	__IEEE754_FLOAT_C(0.919490039339), /* 0x0.eb63b2fff */
	__IEEE754_FLOAT_C(0.921983361257), /* 0x0.ec071a000 */
	__IEEE754_FLOAT_C(0.924488604054), /* 0x0.ecab48fff */
	__IEEE754_FLOAT_C(0.926989555360), /* 0x0.ed4f30000 */
	__IEEE754_FLOAT_C(0.929502844812), /* 0x0.edf3e6000 */
	__IEEE754_FLOAT_C(0.932021975503), /* 0x0.ee98fdfff */
	__IEEE754_FLOAT_C(0.934553921208), /* 0x0.ef3eecfff */
	__IEEE754_FLOAT_C(0.937083780759), /* 0x0.efe4b8fff */
	__IEEE754_FLOAT_C(0.939624726786), /* 0x0.f08b3f000 */
	__IEEE754_FLOAT_C(0.942198514924), /* 0x0.f133ebfff */
	__IEEE754_FLOAT_C(0.944726586343), /* 0x0.f1d99a000 */
	__IEEE754_FLOAT_C(0.947287976728), /* 0x0.f28176fff */
	__IEEE754_FLOAT_C(0.949856162070), /* 0x0.f329c5fff */
	__IEEE754_FLOAT_C(0.952431440345), /* 0x0.f3d28bfff */
	__IEEE754_FLOAT_C(0.955013573175), /* 0x0.f47bc5000 */
	__IEEE754_FLOAT_C(0.957603693021), /* 0x0.f52584000 */
	__IEEE754_FLOAT_C(0.960199773321), /* 0x0.f5cfa7000 */
	__IEEE754_FLOAT_C(0.962801992906), /* 0x0.f67a31000 */
	__IEEE754_FLOAT_C(0.965413510788), /* 0x0.f72556fff */
	__IEEE754_FLOAT_C(0.968030691152), /* 0x0.f7d0dc000 */
	__IEEE754_FLOAT_C(0.970655620084), /* 0x0.f87ce2fff */
	__IEEE754_FLOAT_C(0.973290979849), /* 0x0.f92998fff */
	__IEEE754_FLOAT_C(0.975926160805), /* 0x0.f9d64bfff */
	__IEEE754_FLOAT_C(0.978571653370), /* 0x0.fa83ac000 */
	__IEEE754_FLOAT_C(0.981225252139), /* 0x0.fb3193fff */
	__IEEE754_FLOAT_C(0.983885228626), /* 0x0.fbdfe6fff */
	__IEEE754_FLOAT_C(0.986552715296), /* 0x0.fc8eb7fff */
	__IEEE754_FLOAT_C(0.989228487027), /* 0x0.fd3e14000 */
	__IEEE754_FLOAT_C(0.991909801964), /* 0x0.fdedcd000 */
	__IEEE754_FLOAT_C(0.994601726545), /* 0x0.fe9e38000 */
	__IEEE754_FLOAT_C(0.997297704209), /* 0x0.ff4ee6fff */
	__IEEE754_FLOAT_C(1.000000000000), /* 0x1.000000000 */
	__IEEE754_FLOAT_C(1.002710938457), /* 0x1.00b1aa000 */
	__IEEE754_FLOAT_C(1.005429744692), /* 0x1.0163d7ffe */
	__IEEE754_FLOAT_C(1.008155703526), /* 0x1.02167dffe */
	__IEEE754_FLOAT_C(1.010888457284), /* 0x1.02c995fff */
	__IEEE754_FLOAT_C(1.013629436498), /* 0x1.037d38000 */
	__IEEE754_FLOAT_C(1.016377568250), /* 0x1.043152000 */
	__IEEE754_FLOAT_C(1.019134163841), /* 0x1.04e5f9ffe */
	__IEEE754_FLOAT_C(1.021896362316), /* 0x1.059b00000 */
	__IEEE754_FLOAT_C(1.024668931945), /* 0x1.0650b3ffe */
	__IEEE754_FLOAT_C(1.027446627635), /* 0x1.0706be001 */
	__IEEE754_FLOAT_C(1.030234098408), /* 0x1.07bd6bffe */
	__IEEE754_FLOAT_C(1.033023953416), /* 0x1.087441ffe */
	__IEEE754_FLOAT_C(1.035824656494), /* 0x1.092bce000 */
	__IEEE754_FLOAT_C(1.038632392900), /* 0x1.09e3d0001 */
	__IEEE754_FLOAT_C(1.041450142840), /* 0x1.0a9c79ffe */
	__IEEE754_FLOAT_C(1.044273972530), /* 0x1.0b558a001 */
	__IEEE754_FLOAT_C(1.047105550795), /* 0x1.0c0f1c001 */
	__IEEE754_FLOAT_C(1.049944162390), /* 0x1.0cc924001 */
	__IEEE754_FLOAT_C(1.052791833895), /* 0x1.0d83c4001 */
	__IEEE754_FLOAT_C(1.055645227426), /* 0x1.0e3ec3fff */
	__IEEE754_FLOAT_C(1.058507919326), /* 0x1.0efa60001 */
	__IEEE754_FLOAT_C(1.061377286898), /* 0x1.0fb66bfff */
	__IEEE754_FLOAT_C(1.064254641510), /* 0x1.1072fdffe */
	__IEEE754_FLOAT_C(1.067140102389), /* 0x1.113018000 */
	__IEEE754_FLOAT_C(1.070034146304), /* 0x1.11edc1fff */
	__IEEE754_FLOAT_C(1.072937250162), /* 0x1.12ac04001 */
	__IEEE754_FLOAT_C(1.075843691823), /* 0x1.136a7dfff */
	__IEEE754_FLOAT_C(1.078760385496), /* 0x1.1429a3ffe */
	__IEEE754_FLOAT_C(1.081685543070), /* 0x1.14e958000 */
	__IEEE754_FLOAT_C(1.084618330005), /* 0x1.15a98c000 */
	__IEEE754_FLOAT_C(1.087556362176), /* 0x1.166a18001 */
	__IEEE754_FLOAT_C(1.090508937863), /* 0x1.172b98001 */
	__IEEE754_FLOAT_C(1.093464612954), /* 0x1.17ed4bfff */
	__IEEE754_FLOAT_C(1.096430182434), /* 0x1.18afa5ffe */
	__IEEE754_FLOAT_C(1.099401354802), /* 0x1.19725e000 */
	__IEEE754_FLOAT_C(1.102381587017), /* 0x1.1a35adfff */
	__IEEE754_FLOAT_C(1.105370759965), /* 0x1.1af994000 */
	__IEEE754_FLOAT_C(1.108367800686), /* 0x1.1bbdfdffe */
	__IEEE754_FLOAT_C(1.111373305331), /* 0x1.1c82f6000 */
	__IEEE754_FLOAT_C(1.114387035385), /* 0x1.1d4878001 */
	__IEEE754_FLOAT_C(1.117408752440), /* 0x1.1e0e7ffff */
	__IEEE754_FLOAT_C(1.120437502874), /* 0x1.1ed4fe000 */
	__IEEE754_FLOAT_C(1.123474478729), /* 0x1.1f9c06000 */
	__IEEE754_FLOAT_C(1.126521706601), /* 0x1.2063ba001 */
	__IEEE754_FLOAT_C(1.129574775716), /* 0x1.212bd0001 */
	__IEEE754_FLOAT_C(1.132638812065), /* 0x1.21f49e000 */
	__IEEE754_FLOAT_C(1.135709524130), /* 0x1.22bddbffe */
	__IEEE754_FLOAT_C(1.138789534565), /* 0x1.2387b5fff */
	__IEEE754_FLOAT_C(1.141876101508), /* 0x1.2451fe000 */
	__IEEE754_FLOAT_C(1.144971728301), /* 0x1.251cddffe */
	__IEEE754_FLOAT_C(1.148077130296), /* 0x1.25e861ffe */
	__IEEE754_FLOAT_C(1.151189923305), /* 0x1.26b462001 */
	__IEEE754_FLOAT_C(1.154312610610), /* 0x1.278107ffe */
	__IEEE754_FLOAT_C(1.157440662410), /* 0x1.284e08001 */
	__IEEE754_FLOAT_C(1.160578370109), /* 0x1.291baa001 */
	__IEEE754_FLOAT_C(1.163725256932), /* 0x1.29e9e6000 */
	__IEEE754_FLOAT_C(1.166879892324), /* 0x1.2ab8a3ffe */
	__IEEE754_FLOAT_C(1.170044302935), /* 0x1.2b8805fff */
	__IEEE754_FLOAT_C(1.173205971694), /* 0x1.2c5739ffe */
	__IEEE754_FLOAT_C(1.176397800428), /* 0x1.2d2867ffe */
	__IEEE754_FLOAT_C(1.179586529747), /* 0x1.2df962001 */
	__IEEE754_FLOAT_C(1.182784795737), /* 0x1.2ecafbffe */
	__IEEE754_FLOAT_C(1.185991406414), /* 0x1.2f9d21ffe */
	__IEEE754_FLOAT_C(1.189206838636), /* 0x1.306fdc001 */
	__IEEE754_FLOAT_C(1.192430973067), /* 0x1.314328000 */
	__IEEE754_FLOAT_C(1.195664167430), /* 0x1.32170c001 */
	__IEEE754_FLOAT_C(1.198906540890), /* 0x1.32eb8a001 */
	__IEEE754_FLOAT_C(1.202157497408), /* 0x1.33c098000 */
	__IEEE754_FLOAT_C(1.205416083326), /* 0x1.349625fff */
	__IEEE754_FLOAT_C(1.208683252332), /* 0x1.356c43fff */
	__IEEE754_FLOAT_C(1.211961269402), /* 0x1.364318001 */
	__IEEE754_FLOAT_C(1.215246438983), /* 0x1.371a64000 */
	__IEEE754_FLOAT_C(1.218539118740), /* 0x1.37f22dffe */
	__IEEE754_FLOAT_C(1.221847295770), /* 0x1.38cafc000 */
	__IEEE754_FLOAT_C(1.225158572187), /* 0x1.39a3fdfff */
	__IEEE754_FLOAT_C(1.228481650325), /* 0x1.3a7dc5ffe */
	__IEEE754_FLOAT_C(1.231811761846), /* 0x1.3b5803fff */
	__IEEE754_FLOAT_C(1.235149741144), /* 0x1.3c32c5ffe */
	__IEEE754_FLOAT_C(1.238499879811), /* 0x1.3d0e53ffe */
	__IEEE754_FLOAT_C(1.241858124726), /* 0x1.3dea69fff */
	__IEEE754_FLOAT_C(1.245225191102), /* 0x1.3ec713fff */
	__IEEE754_FLOAT_C(1.248601436624), /* 0x1.3fa458000 */
	__IEEE754_FLOAT_C(1.251975655584), /* 0x1.40817a001 */
	__IEEE754_FLOAT_C(1.255380749731), /* 0x1.4160a2001 */
	__IEEE754_FLOAT_C(1.258783102010), /* 0x1.423f9bffe */
	__IEEE754_FLOAT_C(1.262198328973), /* 0x1.431f6e000 */
	__IEEE754_FLOAT_C(1.265619754780), /* 0x1.43ffa7fff */
	__IEEE754_FLOAT_C(1.269052743928), /* 0x1.44e0a4001 */
	__IEEE754_FLOAT_C(1.272490739830), /* 0x1.45c1f4000 */
	__IEEE754_FLOAT_C(1.275942921659), /* 0x1.46a432001 */
	__IEEE754_FLOAT_C(1.279397487615), /* 0x1.478697ffe */
	__IEEE754_FLOAT_C(1.282870173427), /* 0x1.486a2dffe */
	__IEEE754_FLOAT_C(1.286346316319), /* 0x1.494dfdffe */
	__IEEE754_FLOAT_C(1.289836049094), /* 0x1.4a32b2001 */
	__IEEE754_FLOAT_C(1.293333172770), /* 0x1.4b17e1ffe */
	__IEEE754_FLOAT_C(1.296839594835), /* 0x1.4bfdadfff */
	__IEEE754_FLOAT_C(1.300354957560), /* 0x1.4ce40fffe */
	__IEEE754_FLOAT_C(1.303882122055), /* 0x1.4dcb38001 */
	__IEEE754_FLOAT_C(1.307417988757), /* 0x1.4eb2f1ffe */
	__IEEE754_FLOAT_C(1.310960650439), /* 0x1.4f9b1dfff */
	__IEEE754_FLOAT_C(1.314516782746), /* 0x1.50842bfff */
	__IEEE754_FLOAT_C(1.318079948424), /* 0x1.516daffff */
	__IEEE754_FLOAT_C(1.321653246888), /* 0x1.5257de000 */
	__IEEE754_FLOAT_C(1.325237751030), /* 0x1.5342c8001 */
	__IEEE754_FLOAT_C(1.328829526907), /* 0x1.542e2c000 */
	__IEEE754_FLOAT_C(1.332433700535), /* 0x1.551a5fffe */
	__IEEE754_FLOAT_C(1.336045145966), /* 0x1.56070dffe */
	__IEEE754_FLOAT_C(1.339667558645), /* 0x1.56f473ffe */
	__IEEE754_FLOAT_C(1.343300342533), /* 0x1.57e287ffe */
	__IEEE754_FLOAT_C(1.346941947961), /* 0x1.58d130001 */
	__IEEE754_FLOAT_C(1.350594043714), /* 0x1.59c087ffe */
	__IEEE754_FLOAT_C(1.354256033883), /* 0x1.5ab085fff */
	__IEEE754_FLOAT_C(1.357932448365), /* 0x1.5ba175ffe */
	__IEEE754_FLOAT_C(1.361609339707), /* 0x1.5c926dfff */
	__IEEE754_FLOAT_C(1.365299344044), /* 0x1.5d8441ffe */
	__IEEE754_FLOAT_C(1.369003057507), /* 0x1.5e76fc001 */
	__IEEE754_FLOAT_C(1.372714757920), /* 0x1.5f6a3c000 */
	__IEEE754_FLOAT_C(1.376437187179), /* 0x1.605e2fffe */
	__IEEE754_FLOAT_C(1.380165219333), /* 0x1.615282001 */
	__IEEE754_FLOAT_C(1.383909463864), /* 0x1.6247e3ffe */
	__IEEE754_FLOAT_C(1.387661933907), /* 0x1.633dd0000 */
	__IEEE754_FLOAT_C(1.391424179060), /* 0x1.64345fffe */
	__IEEE754_FLOAT_C(1.395197510706), /* 0x1.652ba9fff */
	__IEEE754_FLOAT_C(1.399006724329), /* 0x1.66254dffe */
	__IEEE754_FLOAT_C(1.402773022651), /* 0x1.671c22000 */
	__IEEE754_FLOAT_C(1.406576037403), /* 0x1.68155dfff */
	__IEEE754_FLOAT_C(1.410389423392)  /* 0x1.690f48001 */
};
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(exp2f) __ATTR_WUNUSED __ATTR_CONST __IEEE754_FLOAT_TYPE__
(__LIBCCALL __ieee754_exp2f)(__IEEE754_FLOAT_TYPE__ __x) {
	/* Check for usual case.  */
	if (__ieee754_islessf(__x, __LIBM_LOCAL_VALUE(exp2_himarkf)) &&
	    __ieee754_isgreaterequalf(__x, __LIBM_LOCAL_VALUE(exp2_lomarkf))) {
		__int32_t __tval, __unsafe;
		__IEEE754_FLOAT_TYPE__ __rx, __x22, __result;
		__IEEE754_FLOAT_TYPE__ __ex2_u, __scale_u;
		__uint32_t __exp;
		int __oldround;
		if (__ieee754_fabsf(__x) < (__IEEE754_FLOAT_C(1.19209289550781250000e-7) /* FLT_EPSILON */ / __IEEE754_FLOAT_C(4.0)))
			return __IEEE754_FLOAT_C(1.0) + __x;
		__oldround = __libc_fegetround();
		if (__oldround != FE_TONEAREST)
			__libc_fesetround(FE_TONEAREST);
		__rx = __x + __LIBM_LOCAL_VALUE(THREEp14f);
		__rx -= __LIBM_LOCAL_VALUE(THREEp14f);
		__x -= __rx; /* Compute x=x1. */
		__tval = (__int32_t)(__rx * __IEEE754_FLOAT_C(256.0) + __IEEE754_FLOAT_C(128.0));
		__x -= __LIBM_LOCAL_VALUE(exp2f_deltatable)[__tval & 255];
		__ex2_u = __LIBM_LOCAL_VALUE(exp2f_atable)[__tval & 255];
		__tval >>= 8;
		__unsafe = (__tval < 0 ? -__tval : __tval) >= -(-125 /*FLT_MIN_EXP*/) - 32;
		__LIBM_GET_FLOAT_WORD(__exp, __ex2_u);
		__exp = ((__exp & __UINT32_C(0x7F800000)) + ((__tval >> __unsafe) << IEEE754_FLOAT_SHIFT)) |
		        (__exp & ~__UINT32_C(0x7F800000));
		__LIBM_SET_FLOAT_WORD(__ex2_u, __exp);
		__exp = __UINT32_C(0x3f800000) + ((__tval - (__tval >> __unsafe)) << IEEE754_FLOAT_SHIFT);
		__LIBM_SET_FLOAT_WORD(__scale_u, __exp);
		__x22 = (__IEEE754_FLOAT_C(0.24022656679) * __x +
		         __IEEE754_FLOAT_C(0.69314736128)) *
		        __ex2_u;
		if (__oldround != FE_TONEAREST)
			__libc_fesetround(__oldround);
		__result = __x22 * __x + __ex2_u;
		if (!__unsafe)
			return __result;
		return __result * __scale_u;
	}
	/* Exceptional cases:  */
	if (__ieee754_islessf(__x, __LIBM_LOCAL_VALUE(exp2_himarkf))) {
		if (__ieee754_isinf(__x)) /* e^-inf == 0, with no error.  */
			return 0;
		/* Underflow */
		return __LIBM_LOCAL_VALUE(TWOM100f) * __LIBM_LOCAL_VALUE(TWOM100f);
	}
	/* Return x, if x is a NaN or Inf; or overflow, otherwise.  */
	return __LIBM_LOCAL_VALUE(TWO127f) * __x;
}
#endif /* __IEEE754_FLOAT_TYPE__ */


#ifdef __IEEE754_DOUBLE_TYPE__
/* Double-precision floating point 2^x.
   Copyright (C) 1997-2013 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Geoffrey Keating <geoffk@ozemail.com.au>

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

__LIBM_LOCAL_DECLARE_BEGIN
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, exp2_himark, __IEEE754_DOUBLE_C(1024.0))  /* DBL_MAX_EXP */
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, exp2_lomark, __IEEE754_DOUBLE_C(-1075.0)) /* DBL_MIN_EXP - DBL_MANT_DIG - 1 */
__LIBM_LOCAL_DECLARE(__IEEE754_DOUBLE_TYPE__, THREEp42, __IEEE754_DOUBLE_C(13194139533312.0))
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, exp2_accuratetable, 512) {
	__IEEE754_DOUBLE_C(0.707106781187802013759), /* 0x0.b504f333fb3f80007 */
	__IEEE754_DOUBLE_C(0.708064712808760599040), /* 0x0.b543baa0f71b38000 */
	__IEEE754_DOUBLE_C(0.709023942160304065938), /* 0x0.b58297d3a8d518002 */
	__IEEE754_DOUBLE_C(0.709984470998547667624), /* 0x0.b5c18ad39b4ba0001 */
	__IEEE754_DOUBLE_C(0.710946301084324217006), /* 0x0.b60093a85e8d30001 */
	__IEEE754_DOUBLE_C(0.711909434180505784637), /* 0x0.b63fb25984e628005 */
	__IEEE754_DOUBLE_C(0.712873872052760648733), /* 0x0.b67ee6eea3b5f8003 */
	__IEEE754_DOUBLE_C(0.713839616467838999908), /* 0x0.b6be316f518c98001 */
	__IEEE754_DOUBLE_C(0.714806669195984345523), /* 0x0.b6fd91e328d148007 */
	__IEEE754_DOUBLE_C(0.715775032009894562898), /* 0x0.b73d0851c69e20002 */
	__IEEE754_DOUBLE_C(0.716744706683768884058), /* 0x0.b77c94c2c9b3d0003 */
	__IEEE754_DOUBLE_C(0.717715694995770148178), /* 0x0.b7bc373dd52eb0003 */
	__IEEE754_DOUBLE_C(0.718687998724665488852), /* 0x0.b7fbefca8cd530004 */
	__IEEE754_DOUBLE_C(0.719661619652575468291), /* 0x0.b83bbe70981da8001 */
	__IEEE754_DOUBLE_C(0.720636559564428180758), /* 0x0.b87ba337a194b0006 */
	__IEEE754_DOUBLE_C(0.721612820246623098989), /* 0x0.b8bb9e27556508004 */
	__IEEE754_DOUBLE_C(0.722590403488338473025), /* 0x0.b8fbaf4762c798006 */
	__IEEE754_DOUBLE_C(0.723569311081411870036), /* 0x0.b93bd69f7be1d0000 */
	__IEEE754_DOUBLE_C(0.724549544820974333906), /* 0x0.b97c1437567828007 */
	__IEEE754_DOUBLE_C(0.725531106502312561633), /* 0x0.b9bc6816a87ae8002 */
	__IEEE754_DOUBLE_C(0.726513997924421062181), /* 0x0.b9fcd2452bee00000 */
	__IEEE754_DOUBLE_C(0.727498220889519875430), /* 0x0.ba3d52ca9e6148002 */
	__IEEE754_DOUBLE_C(0.728483777200401694265), /* 0x0.ba7de9aebe05c8003 */
	__IEEE754_DOUBLE_C(0.729470668664712662563), /* 0x0.babe96f94e62a8002 */
	__IEEE754_DOUBLE_C(0.730458897090379144517), /* 0x0.baff5ab2134df0004 */
	__IEEE754_DOUBLE_C(0.731448464287988597833), /* 0x0.bb4034e0d38ab0000 */
	__IEEE754_DOUBLE_C(0.732439372072965166897), /* 0x0.bb81258d5b2d60001 */
	__IEEE754_DOUBLE_C(0.733431622260458326859), /* 0x0.bbc22cbf75fd28001 */
	__IEEE754_DOUBLE_C(0.734425216668725511232), /* 0x0.bc034a7ef32c00001 */
	__IEEE754_DOUBLE_C(0.735420157118880535324), /* 0x0.bc447ed3a50fe0005 */
	__IEEE754_DOUBLE_C(0.736416445434497690674), /* 0x0.bc85c9c560b350001 */
	__IEEE754_DOUBLE_C(0.737414083433310718618), /* 0x0.bcc72b5bf4b4e0000 */
	__IEEE754_DOUBLE_C(0.738413072966152328496), /* 0x0.bd08a39f5417a8007 */
	__IEEE754_DOUBLE_C(0.739413415848264365956), /* 0x0.bd4a32974abcd0002 */
	__IEEE754_DOUBLE_C(0.740415113911250699637), /* 0x0.bd8bd84bb68300002 */
	__IEEE754_DOUBLE_C(0.741418168994518067562), /* 0x0.bdcd94c47ddd30003 */
	__IEEE754_DOUBLE_C(0.742422582936659858376), /* 0x0.be0f6809865968006 */
	__IEEE754_DOUBLE_C(0.743428357577745613238), /* 0x0.be515222b72530003 */
	__IEEE754_DOUBLE_C(0.744435494762383687126), /* 0x0.be935317fc6ba0002 */
	__IEEE754_DOUBLE_C(0.745443996335090397492), /* 0x0.bed56af1423de8001 */
	__IEEE754_DOUBLE_C(0.746453864145572798553), /* 0x0.bf1799b67a6248007 */
	__IEEE754_DOUBLE_C(0.747465100043933849969), /* 0x0.bf59df6f970e70002 */
	__IEEE754_DOUBLE_C(0.748477705883256683178), /* 0x0.bf9c3c248dbee8001 */
	__IEEE754_DOUBLE_C(0.749491683518965001732), /* 0x0.bfdeafdd568308000 */
	__IEEE754_DOUBLE_C(0.750507034813367890373), /* 0x0.c0213aa1f0fc38004 */
	__IEEE754_DOUBLE_C(0.751523761622240105153), /* 0x0.c063dc7a559ca0003 */
	__IEEE754_DOUBLE_C(0.752541865811731880422), /* 0x0.c0a6956e883ed8000 */
	__IEEE754_DOUBLE_C(0.753561349247157341600), /* 0x0.c0e965868bd220006 */
	__IEEE754_DOUBLE_C(0.754582213796583967110), /* 0x0.c12c4cca664cb8002 */
	__IEEE754_DOUBLE_C(0.755604461332336940791), /* 0x0.c16f4b42225350006 */
	__IEEE754_DOUBLE_C(0.756628093726406381068), /* 0x0.c1b260f5ca2c48002 */
	__IEEE754_DOUBLE_C(0.757653112855631305506), /* 0x0.c1f58ded6d72d8001 */
	__IEEE754_DOUBLE_C(0.758679520599333412360), /* 0x0.c238d2311e7d08001 */
	__IEEE754_DOUBLE_C(0.759707318837184453227), /* 0x0.c27c2dc8f00368005 */
	__IEEE754_DOUBLE_C(0.760736509456435783249), /* 0x0.c2bfa0bcfd1400000 */
	__IEEE754_DOUBLE_C(0.761767094336480043995), /* 0x0.c3032b155818d0000 */
	__IEEE754_DOUBLE_C(0.762799075372231349951), /* 0x0.c346ccda248cc0001 */
	__IEEE754_DOUBLE_C(0.763832454453522768941), /* 0x0.c38a8613805488005 */
	__IEEE754_DOUBLE_C(0.764867233473625618441), /* 0x0.c3ce56c98d1ca8005 */
	__IEEE754_DOUBLE_C(0.765903414329434539816), /* 0x0.c4123f04708d80002 */
	__IEEE754_DOUBLE_C(0.766940998920452976510), /* 0x0.c4563ecc532dc0001 */
	__IEEE754_DOUBLE_C(0.767979989148100838946), /* 0x0.c49a56295f9f88006 */
	__IEEE754_DOUBLE_C(0.769020386915772125040), /* 0x0.c4de8523c2b0a0001 */
	__IEEE754_DOUBLE_C(0.770062194131770905170), /* 0x0.c522cbc3ae94e0003 */
	__IEEE754_DOUBLE_C(0.771105412703856241146), /* 0x0.c5672a1154e6b8004 */
	__IEEE754_DOUBLE_C(0.772150044545352520777), /* 0x0.c5aba014ed5f18003 */
	__IEEE754_DOUBLE_C(0.773196091570364285606), /* 0x0.c5f02dd6b09288003 */
	__IEEE754_DOUBLE_C(0.774243555696622731700), /* 0x0.c634d35edb1260003 */
	__IEEE754_DOUBLE_C(0.775292438842697939641), /* 0x0.c67990b5aa5c18004 */
	__IEEE754_DOUBLE_C(0.776342742931542928455), /* 0x0.c6be65e360bed8000 */
	__IEEE754_DOUBLE_C(0.777394469888802008854), /* 0x0.c70352f0437f50004 */
	__IEEE754_DOUBLE_C(0.778447621641124243320), /* 0x0.c74857e498fd00006 */
	__IEEE754_DOUBLE_C(0.779502200118583399303), /* 0x0.c78d74c8ab5b60000 */
	__IEEE754_DOUBLE_C(0.780558207255445668515), /* 0x0.c7d2a9a4c959f8000 */
	__IEEE754_DOUBLE_C(0.781615644985491186966), /* 0x0.c817f681412f80002 */
	__IEEE754_DOUBLE_C(0.782674515247667956808), /* 0x0.c85d5b6666c150006 */
	__IEEE754_DOUBLE_C(0.783734819983036512536), /* 0x0.c8a2d85c904760003 */
	__IEEE754_DOUBLE_C(0.784796561133562109454), /* 0x0.c8e86d6c14f850002 */
	__IEEE754_DOUBLE_C(0.785859740645942328471), /* 0x0.c92e1a9d513ec8002 */
	__IEEE754_DOUBLE_C(0.786924360469767103536), /* 0x0.c973dff8a4b390007 */
	__IEEE754_DOUBLE_C(0.787990422552312885808), /* 0x0.c9b9bd866c6440007 */
	__IEEE754_DOUBLE_C(0.789057928854407064640), /* 0x0.c9ffb34f1444b0001 */
	__IEEE754_DOUBLE_C(0.790126881326406182996), /* 0x0.ca45c15afcc570001 */
	__IEEE754_DOUBLE_C(0.791197281930050233534), /* 0x0.ca8be7b292db38000 */
	__IEEE754_DOUBLE_C(0.792269132620954885659), /* 0x0.cad2265e3cbee8000 */
	__IEEE754_DOUBLE_C(0.793342435380726906957), /* 0x0.cb187d667d3d38006 */
	__IEEE754_DOUBLE_C(0.794417192158282659010), /* 0x0.cb5eecd3b33158006 */
	__IEEE754_DOUBLE_C(0.795493404931386649540), /* 0x0.cba574ae5d2e80001 */
	__IEEE754_DOUBLE_C(0.796571075671306805268), /* 0x0.cbec14fef2a348004 */
	__IEEE754_DOUBLE_C(0.797650206352955137846), /* 0x0.cc32cdcdef0000000 */
	__IEEE754_DOUBLE_C(0.798730798954342069432), /* 0x0.cc799f23d11d18000 */
	__IEEE754_DOUBLE_C(0.799812855456121796232), /* 0x0.ccc089091abb28004 */
	__IEEE754_DOUBLE_C(0.800896377841454287795), /* 0x0.cd078b86505c18003 */
	__IEEE754_DOUBLE_C(0.801981368096190028208), /* 0x0.cd4ea6a3f97720007 */
	__IEEE754_DOUBLE_C(0.803067828208752554378), /* 0x0.cd95da6aa057b8007 */
	__IEEE754_DOUBLE_C(0.804155760170129796375), /* 0x0.cddd26e2d21b28001 */
	__IEEE754_DOUBLE_C(0.805245165974338261710), /* 0x0.ce248c151f3330001 */
	__IEEE754_DOUBLE_C(0.806336047619038653883), /* 0x0.ce6c0a0a1c1350001 */
	__IEEE754_DOUBLE_C(0.807428407102107836855), /* 0x0.ceb3a0ca5d6be0006 */
	__IEEE754_DOUBLE_C(0.808522246427078927792), /* 0x0.cefb505e7e2550007 */
	__IEEE754_DOUBLE_C(0.809617567597010201484), /* 0x0.cf4318cf18a268002 */
	__IEEE754_DOUBLE_C(0.810714372621179513182), /* 0x0.cf8afa24ce1c98004 */
	__IEEE754_DOUBLE_C(0.811812663508675536069), /* 0x0.cfd2f4683f9810005 */
	__IEEE754_DOUBLE_C(0.812912442272482604912), /* 0x0.d01b07a2126188003 */
	__IEEE754_DOUBLE_C(0.814013710929394895825), /* 0x0.d06333daeff618001 */
	__IEEE754_DOUBLE_C(0.815116471495287542325), /* 0x0.d0ab791b80d028006 */
	__IEEE754_DOUBLE_C(0.816220725993571205593), /* 0x0.d0f3d76c75b330000 */
	__IEEE754_DOUBLE_C(0.817326476447408967199), /* 0x0.d13c4ed67f1cf8000 */
	__IEEE754_DOUBLE_C(0.818433724883006474832), /* 0x0.d184df6250e3b0001 */
	__IEEE754_DOUBLE_C(0.819542473330909460055), /* 0x0.d1cd8918a3a328004 */
	__IEEE754_DOUBLE_C(0.820652723822034690935), /* 0x0.d2164c02305fa0002 */
	__IEEE754_DOUBLE_C(0.821764478391968422618), /* 0x0.d25f2827b53fb0005 */
	__IEEE754_DOUBLE_C(0.822877739077315761840), /* 0x0.d2a81d91f188b8000 */
	__IEEE754_DOUBLE_C(0.823992507918612782109), /* 0x0.d2f12c49a8d290005 */
	__IEEE754_DOUBLE_C(0.825108786960634610365), /* 0x0.d33a5457a35e40003 */
	__IEEE754_DOUBLE_C(0.826226578247117093869), /* 0x0.d38395c4a84848007 */
	__IEEE754_DOUBLE_C(0.827345883828319528258), /* 0x0.d3ccf09985d958004 */
	__IEEE754_DOUBLE_C(0.828466705754248966560), /* 0x0.d41664df0a1320005 */
	__IEEE754_DOUBLE_C(0.829589046080638992111), /* 0x0.d45ff29e094330000 */
	__IEEE754_DOUBLE_C(0.830712906863802391671), /* 0x0.d4a999df585a20005 */
	__IEEE754_DOUBLE_C(0.831838290163696481037), /* 0x0.d4f35aabd04a60006 */
	__IEEE754_DOUBLE_C(0.832965198041969556729), /* 0x0.d53d350c4be258002 */
	__IEEE754_DOUBLE_C(0.834093632565442222342), /* 0x0.d5872909aba050007 */
	__IEEE754_DOUBLE_C(0.835223595802037643865), /* 0x0.d5d136acd138e8006 */
	__IEEE754_DOUBLE_C(0.836355089820669306292), /* 0x0.d61b5dfe9f7780004 */
	__IEEE754_DOUBLE_C(0.837488116698010487424), /* 0x0.d6659f0801afa8005 */
	__IEEE754_DOUBLE_C(0.838622678508982644113), /* 0x0.d6aff9d1e147d8004 */
	__IEEE754_DOUBLE_C(0.839758777333464490056), /* 0x0.d6fa6e652d19e0000 */
	__IEEE754_DOUBLE_C(0.840896415254110962690), /* 0x0.d744fccad70d00003 */
	__IEEE754_DOUBLE_C(0.842035594355151628676), /* 0x0.d78fa50bd2c3b0000 */
	__IEEE754_DOUBLE_C(0.843176316724478125433), /* 0x0.d7da673117e730007 */
	__IEEE754_DOUBLE_C(0.844318584453106590905), /* 0x0.d8254343a19038003 */
	__IEEE754_DOUBLE_C(0.845462399634695271912), /* 0x0.d870394c6dbf30003 */
	__IEEE754_DOUBLE_C(0.846607764365415071965), /* 0x0.d8bb49547d37c0004 */
	__IEEE754_DOUBLE_C(0.847754680744707056494), /* 0x0.d9067364d45608003 */
	__IEEE754_DOUBLE_C(0.848903150873708822763), /* 0x0.d951b7867953b0006 */
	__IEEE754_DOUBLE_C(0.850053176859071113491), /* 0x0.d99d15c2787a30006 */
	__IEEE754_DOUBLE_C(0.851204760807439786431), /* 0x0.d9e88e21de11a0003 */
	__IEEE754_DOUBLE_C(0.852357904828824897169), /* 0x0.da3420adba1508003 */
	__IEEE754_DOUBLE_C(0.853512611037803181642), /* 0x0.da7fcd6f2184d8005 */
	__IEEE754_DOUBLE_C(0.854668881550406100980), /* 0x0.dacb946f2afaf8000 */
	__IEEE754_DOUBLE_C(0.855826718478671755185), /* 0x0.db1775b6e8ad48000 */
	__IEEE754_DOUBLE_C(0.856986123964844970247), /* 0x0.db63714f8e0818006 */
	__IEEE754_DOUBLE_C(0.858147100114499461478), /* 0x0.dbaf87422625b8000 */
	__IEEE754_DOUBLE_C(0.859309649060962410524), /* 0x0.dbfbb797daa460002 */
	__IEEE754_DOUBLE_C(0.860473772936213743282), /* 0x0.dc480259d3a710001 */
	__IEEE754_DOUBLE_C(0.861639473872910177676), /* 0x0.dc9467913a0f48006 */
	__IEEE754_DOUBLE_C(0.862806754008130227807), /* 0x0.dce0e7473b9b28003 */
	__IEEE754_DOUBLE_C(0.863975615481124226159), /* 0x0.dd2d8185086c20006 */
	__IEEE754_DOUBLE_C(0.865146060433749419813), /* 0x0.dd7a3653d38168005 */
	__IEEE754_DOUBLE_C(0.866318091005120138881), /* 0x0.ddc705bcccd628000 */
	__IEEE754_DOUBLE_C(0.867491709362415264210), /* 0x0.de13efc9434100004 */
	__IEEE754_DOUBLE_C(0.868666917636779056818), /* 0x0.de60f4825df9b8005 */
	__IEEE754_DOUBLE_C(0.869843717989716047624), /* 0x0.deae13f16599c0003 */
	__IEEE754_DOUBLE_C(0.871022112578215268471), /* 0x0.defb4e1f9dc388002 */
	__IEEE754_DOUBLE_C(0.872202103559697183859), /* 0x0.df48a3164a92f0001 */
	__IEEE754_DOUBLE_C(0.873383693097737778847), /* 0x0.df9612deb6e878007 */
	__IEEE754_DOUBLE_C(0.874566883362160263365), /* 0x0.dfe39d82348310001 */
	__IEEE754_DOUBLE_C(0.875751676517234511901), /* 0x0.e031430a0f0688000 */
	__IEEE754_DOUBLE_C(0.876938074732511840819), /* 0x0.e07f037f97e548001 */
	__IEEE754_DOUBLE_C(0.878126080186539592654), /* 0x0.e0ccdeec2a75e0006 */
	__IEEE754_DOUBLE_C(0.879315695055312818168), /* 0x0.e11ad5591f4078001 */
	__IEEE754_DOUBLE_C(0.880506921518618312932), /* 0x0.e168e6cfd2f880004 */
	__IEEE754_DOUBLE_C(0.881699761760385225541), /* 0x0.e1b71359a6df60003 */
	__IEEE754_DOUBLE_C(0.882894217964411143207), /* 0x0.e2055afffc1178000 */
	__IEEE754_DOUBLE_C(0.884090292325693805080), /* 0x0.e253bdcc3ffbb8001 */
	__IEEE754_DOUBLE_C(0.885287987031581180559), /* 0x0.e2a23bc7d7a1d8002 */
	__IEEE754_DOUBLE_C(0.886487304278189114386), /* 0x0.e2f0d4fc31ab80004 */
	__IEEE754_DOUBLE_C(0.887688246263368285778), /* 0x0.e33f8972bea8a8005 */
	__IEEE754_DOUBLE_C(0.888890815189881999840), /* 0x0.e38e5934f49010007 */
	__IEEE754_DOUBLE_C(0.890095013257492739835), /* 0x0.e3dd444c460bd0007 */
	__IEEE754_DOUBLE_C(0.891300842677948068626), /* 0x0.e42c4ac232f380000 */
	__IEEE754_DOUBLE_C(0.892508305659222567226), /* 0x0.e47b6ca036f8b8005 */
	__IEEE754_DOUBLE_C(0.893717404414979710310), /* 0x0.e4caa9efd40e58002 */
	__IEEE754_DOUBLE_C(0.894928141160697743242), /* 0x0.e51a02ba8e2610007 */
	__IEEE754_DOUBLE_C(0.896140518115016826430), /* 0x0.e5697709ecab90000 */
	__IEEE754_DOUBLE_C(0.897354537501434679237), /* 0x0.e5b906e77c61d0006 */
	__IEEE754_DOUBLE_C(0.898570201543732793877), /* 0x0.e608b25cca5ba8005 */
	__IEEE754_DOUBLE_C(0.899787512470129891014), /* 0x0.e6587973688ce8002 */
	__IEEE754_DOUBLE_C(0.901006472512270728537), /* 0x0.e6a85c34ecadb8000 */
	__IEEE754_DOUBLE_C(0.902227083902570559127), /* 0x0.e6f85aaaed4f20006 */
	__IEEE754_DOUBLE_C(0.903449348881299796343), /* 0x0.e74874df09a530003 */
	__IEEE754_DOUBLE_C(0.904673269686823378091), /* 0x0.e798aadadecba0007 */
	__IEEE754_DOUBLE_C(0.905898848559668845585), /* 0x0.e7e8fca80c3ee0001 */
	__IEEE754_DOUBLE_C(0.907126087750156795426), /* 0x0.e8396a503c3fe0005 */
	__IEEE754_DOUBLE_C(0.908354989505901100354), /* 0x0.e889f3dd1615b0002 */
	__IEEE754_DOUBLE_C(0.909585556079328783087), /* 0x0.e8da9958465228007 */
	__IEEE754_DOUBLE_C(0.910817789726044213523), /* 0x0.e92b5acb7d0578001 */
	__IEEE754_DOUBLE_C(0.912051692703457872481), /* 0x0.e97c38406c3c30003 */
	__IEEE754_DOUBLE_C(0.913287267274154990210), /* 0x0.e9cd31c0cbb370001 */
	__IEEE754_DOUBLE_C(0.914524515702244578108), /* 0x0.ea1e475654d540000 */
	__IEEE754_DOUBLE_C(0.915763440256158633982), /* 0x0.ea6f790ac5cc78001 */
	__IEEE754_DOUBLE_C(0.917004043205012497909), /* 0x0.eac0c6e7dd8448007 */
	__IEEE754_DOUBLE_C(0.918246326823137892807), /* 0x0.eb1230f760a428007 */
	__IEEE754_DOUBLE_C(0.919490293387826285200), /* 0x0.eb63b7431714a8007 */
	__IEEE754_DOUBLE_C(0.920735945178816406225), /* 0x0.ebb559d4cb6f30007 */
	__IEEE754_DOUBLE_C(0.921983284479243714322), /* 0x0.ec0718b64c0940002 */
	__IEEE754_DOUBLE_C(0.923232313574974705626), /* 0x0.ec58f3f16a3910002 */
	__IEEE754_DOUBLE_C(0.924483034755387955725), /* 0x0.ecaaeb8ffb3168005 */
	__IEEE754_DOUBLE_C(0.925735450311948926408), /* 0x0.ecfcff9bd67078000 */
	__IEEE754_DOUBLE_C(0.926989562542820610982), /* 0x0.ed4f301edad1a0007 */
	__IEEE754_DOUBLE_C(0.928245373740515189457), /* 0x0.eda17d22e0f9b0001 */
	__IEEE754_DOUBLE_C(0.929502886213858126045), /* 0x0.edf3e6b1d37d40001 */
	__IEEE754_DOUBLE_C(0.930762102264245716494), /* 0x0.ee466cd594c5c8005 */
	__IEEE754_DOUBLE_C(0.932023024199046146183), /* 0x0.ee990f980dcdb0005 */
	__IEEE754_DOUBLE_C(0.933285654329454095216), /* 0x0.eeebcf032bc470007 */
	__IEEE754_DOUBLE_C(0.934549994971191289044), /* 0x0.ef3eab20e0d3c0001 */
	__IEEE754_DOUBLE_C(0.935816048439005676599), /* 0x0.ef91a3fb1e1340004 */
	__IEEE754_DOUBLE_C(0.937083817055075818404), /* 0x0.efe4b99bdcc618006 */
	__IEEE754_DOUBLE_C(0.938353303143720007819), /* 0x0.f037ec0d1889b8000 */
	__IEEE754_DOUBLE_C(0.939624509028518128972), /* 0x0.f08b3b58cc2bb8006 */
	__IEEE754_DOUBLE_C(0.940897437041863904384), /* 0x0.f0dea788fc2a90000 */
	__IEEE754_DOUBLE_C(0.942172089516254085427), /* 0x0.f13230a7ad21b8003 */
	__IEEE754_DOUBLE_C(0.943448468787511540534), /* 0x0.f185d6bee754e0006 */
	__IEEE754_DOUBLE_C(0.944726577195256100890), /* 0x0.f1d999d8b73478005 */
	__IEEE754_DOUBLE_C(0.946006417082291717338), /* 0x0.f22d79ff2cb130000 */
	__IEEE754_DOUBLE_C(0.947287990793413858827), /* 0x0.f281773c59ec48007 */
	__IEEE754_DOUBLE_C(0.948571300678290207925), /* 0x0.f2d5919a566268001 */
	__IEEE754_DOUBLE_C(0.949856349088629370320), /* 0x0.f329c9233bceb0001 */
	__IEEE754_DOUBLE_C(0.951143138379053731954), /* 0x0.f37e1de1272068002 */
	__IEEE754_DOUBLE_C(0.952431670908847949364), /* 0x0.f3d28fde3a6728006 */
	__IEEE754_DOUBLE_C(0.953721949039916472305), /* 0x0.f4271f249a93f0001 */
	__IEEE754_DOUBLE_C(0.955013975135367898520), /* 0x0.f47bcbbe6deab0001 */
	__IEEE754_DOUBLE_C(0.956307751564417496418), /* 0x0.f4d095b5e16638004 */
	__IEEE754_DOUBLE_C(0.957603280698967163097), /* 0x0.f5257d1524f590006 */
	__IEEE754_DOUBLE_C(0.958900564911197350604), /* 0x0.f57a81e668d628000 */
	__IEEE754_DOUBLE_C(0.960199606581278120057), /* 0x0.f5cfa433e60e50007 */
	__IEEE754_DOUBLE_C(0.961500408088936442422), /* 0x0.f624e407d527a0007 */
	__IEEE754_DOUBLE_C(0.962802971817578789903), /* 0x0.f67a416c72b760006 */
	__IEEE754_DOUBLE_C(0.964107300155846558292), /* 0x0.f6cfbc6c011458004 */
	__IEEE754_DOUBLE_C(0.965413395493874504368), /* 0x0.f7255510c439a8002 */
	__IEEE754_DOUBLE_C(0.966721260225105960572), /* 0x0.f77b0b6503c5b8006 */
	__IEEE754_DOUBLE_C(0.968030896745834645873), /* 0x0.f7d0df730a7940005 */
	__IEEE754_DOUBLE_C(0.969342307458006424716), /* 0x0.f826d145294be8003 */
	__IEEE754_DOUBLE_C(0.970655494764855020231), /* 0x0.f87ce0e5b29fd8000 */
	__IEEE754_DOUBLE_C(0.971970461071268720958), /* 0x0.f8d30e5efaa8f0004 */
	__IEEE754_DOUBLE_C(0.973287208789983648852), /* 0x0.f92959bb5e3c08001 */
	__IEEE754_DOUBLE_C(0.974605740331924708124), /* 0x0.f97fc305383028004 */
	__IEEE754_DOUBLE_C(0.975926058115625383329), /* 0x0.f9d64a46ebb9f8004 */
	__IEEE754_DOUBLE_C(0.977248164559556209435), /* 0x0.fa2cef8adbfc68004 */
	__IEEE754_DOUBLE_C(0.978572062087848637573), /* 0x0.fa83b2db7253d0007 */
	__IEEE754_DOUBLE_C(0.979897753126343307191), /* 0x0.fada944319fda0005 */
	__IEEE754_DOUBLE_C(0.981225240104636631254), /* 0x0.fb3193cc425870002 */
	__IEEE754_DOUBLE_C(0.982554525455618277276), /* 0x0.fb88b1815e61d0003 */
	__IEEE754_DOUBLE_C(0.983885611617111077747), /* 0x0.fbdfed6ce683e0007 */
	__IEEE754_DOUBLE_C(0.985218501026348891812), /* 0x0.fc3747995282f8006 */
	__IEEE754_DOUBLE_C(0.986553196127724962867), /* 0x0.fc8ec0112202a0005 */
	__IEEE754_DOUBLE_C(0.987889699367056062238), /* 0x0.fce656ded63710002 */
	__IEEE754_DOUBLE_C(0.989228013193998778636), /* 0x0.fd3e0c0cf48d50005 */
	__IEEE754_DOUBLE_C(0.990568140061241164686), /* 0x0.fd95dfa605c7b0003 */
	__IEEE754_DOUBLE_C(0.991910082424819927754), /* 0x0.fdedd1b4965710004 */
	__IEEE754_DOUBLE_C(0.993253842749249660216), /* 0x0.fe45e2433bfea0000 */
	__IEEE754_DOUBLE_C(0.994599423484053835071), /* 0x0.fe9e115c7c05f0005 */
	__IEEE754_DOUBLE_C(0.995946827107488830167), /* 0x0.fef65f0afb4c28006 */
	__IEEE754_DOUBLE_C(0.997296056085008264529), /* 0x0.ff4ecb59509cc8001 */
	__IEEE754_DOUBLE_C(0.998647112892057764479), /* 0x0.ffa756521dbfd0007 */
	__IEEE754_DOUBLE_C(1.000000000000000000000), /* 0x1.00000000000000000 */
	__IEEE754_DOUBLE_C(1.001354719891689004659), /* 0x1.0058c86da14aa0005 */
	__IEEE754_DOUBLE_C(1.002711275050312211844), /* 0x1.00b1afa5abead0003 */
	__IEEE754_DOUBLE_C(1.004069667960743483835), /* 0x1.010ab5b2cc0660009 */
	__IEEE754_DOUBLE_C(1.005429901112333324093), /* 0x1.0163da9fb2af30008 */
	__IEEE754_DOUBLE_C(1.006791976999887428009), /* 0x1.01bd1e7716f6a0008 */
	__IEEE754_DOUBLE_C(1.008155898118476168101), /* 0x1.02168143b03890006 */
	__IEEE754_DOUBLE_C(1.009521666967782227439), /* 0x1.027003103ae320002 */
	__IEEE754_DOUBLE_C(1.010889286051850133326), /* 0x1.02c9a3e7783030002 */
	__IEEE754_DOUBLE_C(1.012258757875921233497), /* 0x1.032363d42aaa8000e */
	__IEEE754_DOUBLE_C(1.013630084952214405194), /* 0x1.037d42e11c88d0000 */
	__IEEE754_DOUBLE_C(1.015003269791313389451), /* 0x1.03d741191635a0001 */
	__IEEE754_DOUBLE_C(1.016378314911229763267), /* 0x1.04315e86e84630008 */
	__IEEE754_DOUBLE_C(1.017755222831652872635), /* 0x1.048b9b35652800002 */
	__IEEE754_DOUBLE_C(1.019133996077934645224), /* 0x1.04e5f72f65827000b */
	__IEEE754_DOUBLE_C(1.020514637175266248212), /* 0x1.0540727fc1cfa0006 */
	__IEEE754_DOUBLE_C(1.021897148653734488385), /* 0x1.059b0d3157ebb0002 */
	__IEEE754_DOUBLE_C(1.023281533050062419584), /* 0x1.05f5c74f0cfeb0002 */
	__IEEE754_DOUBLE_C(1.024667792897328677539), /* 0x1.0650a0e3c22ee0003 */
	__IEEE754_DOUBLE_C(1.026055930738840826806), /* 0x1.06ab99fa63e1b0008 */
	__IEEE754_DOUBLE_C(1.027445949118511947550), /* 0x1.0706b29ddf2700009 */
	__IEEE754_DOUBLE_C(1.028837850584049418178), /* 0x1.0761ead9253ab0009 */
	__IEEE754_DOUBLE_C(1.030231637685799839262), /* 0x1.07bd42b72a3f80008 */
	__IEEE754_DOUBLE_C(1.031627312979383592802), /* 0x1.0818ba42e824a000c */
	__IEEE754_DOUBLE_C(1.033024879021186448496), /* 0x1.0874518759b0b0008 */
	__IEEE754_DOUBLE_C(1.034424338374263729911), /* 0x1.08d0088f80ffa0006 */
	__IEEE754_DOUBLE_C(1.035825693601787333992), /* 0x1.092bdf66604e30005 */
	__IEEE754_DOUBLE_C(1.037228947273990842283), /* 0x1.0987d617019cd000a */
	__IEEE754_DOUBLE_C(1.038634101961269928846), /* 0x1.09e3ecac6f199000f */
	__IEEE754_DOUBLE_C(1.040041160239590700707), /* 0x1.0a402331b91270002 */
	__IEEE754_DOUBLE_C(1.041450124688240164200), /* 0x1.0a9c79b1f37c3000b */
	__IEEE754_DOUBLE_C(1.042860997889083929381), /* 0x1.0af8f038352160000 */
	__IEEE754_DOUBLE_C(1.044273782427270314011), /* 0x1.0b5586cf986890006 */
	__IEEE754_DOUBLE_C(1.045688480893644856116), /* 0x1.0bb23d833dfbf0006 */
	__IEEE754_DOUBLE_C(1.047105095879385272564), /* 0x1.0c0f145e46e330007 */
	__IEEE754_DOUBLE_C(1.048523629981608529302), /* 0x1.0c6c0b6bdaadc000f */
	__IEEE754_DOUBLE_C(1.049944085800634585634), /* 0x1.0cc922b72470a000f */
	__IEEE754_DOUBLE_C(1.051366465939483019223), /* 0x1.0d265a4b5238b0007 */
	__IEEE754_DOUBLE_C(1.052790773004648849929), /* 0x1.0d83b23395e510002 */
	__IEEE754_DOUBLE_C(1.054217009607077093512), /* 0x1.0de12a7b263970006 */
	__IEEE754_DOUBLE_C(1.055645178360430591625), /* 0x1.0e3ec32d3cf680000 */
	__IEEE754_DOUBLE_C(1.057075281882416506511), /* 0x1.0e9c7c55184f5000e */
	__IEEE754_DOUBLE_C(1.058507322794714378170), /* 0x1.0efa55fdfad51000a */
	__IEEE754_DOUBLE_C(1.059941303721639416236), /* 0x1.0f58503329fed0003 */
	__IEEE754_DOUBLE_C(1.061377227289284297385), /* 0x1.0fb66affed37f0000 */
	__IEEE754_DOUBLE_C(1.062815096132297298980), /* 0x1.1014a66f95540000c */
	__IEEE754_DOUBLE_C(1.064254912884593951029), /* 0x1.1073028d725850007 */
	__IEEE754_DOUBLE_C(1.065696680185205469411), /* 0x1.10d17f64d9ea2000b */
	__IEEE754_DOUBLE_C(1.067140400676658718053), /* 0x1.11301d012586a0007 */
	__IEEE754_DOUBLE_C(1.068586077004890055886), /* 0x1.118edb6db26ab0003 */
	__IEEE754_DOUBLE_C(1.070033711820396415998), /* 0x1.11edbab5e2d6e000b */
	__IEEE754_DOUBLE_C(1.071483307775789262099), /* 0x1.124cbae51b5ef0001 */
	__IEEE754_DOUBLE_C(1.072934867526001312439), /* 0x1.12abdc06c3240000c */
	__IEEE754_DOUBLE_C(1.074388393734249103080), /* 0x1.130b1e264a62e0005 */
	__IEEE754_DOUBLE_C(1.075843889063253344684), /* 0x1.136a814f20ccd0003 */
	__IEEE754_DOUBLE_C(1.077301356179926061823), /* 0x1.13ca058cbaaed000b */
	__IEEE754_DOUBLE_C(1.078760797756675327056), /* 0x1.1429aaea9260e000e */
	__IEEE754_DOUBLE_C(1.080222216468626150775), /* 0x1.148971742537c0009 */
	__IEEE754_DOUBLE_C(1.081685614993597610617), /* 0x1.14e95934f37e8000b */
	__IEEE754_DOUBLE_C(1.083150996013011013776), /* 0x1.1549623881762000d */
	__IEEE754_DOUBLE_C(1.084618362213087383633), /* 0x1.15a98c8a58a6a000b */
	__IEEE754_DOUBLE_C(1.086087716284427351384), /* 0x1.1609d8360768c0008 */
	__IEEE754_DOUBLE_C(1.087559060917626885283), /* 0x1.166a45471c13f0008 */
	__IEEE754_DOUBLE_C(1.089032398810997337465), /* 0x1.16cad3c92d7b50009 */
	__IEEE754_DOUBLE_C(1.090507732647478578212), /* 0x1.172b83c7c18b5000f */
	__IEEE754_DOUBLE_C(1.091985065182095926460), /* 0x1.178c554ead72a000c */
	__IEEE754_DOUBLE_C(1.093464399073070136880), /* 0x1.17ed48695befe000c */
	__IEEE754_DOUBLE_C(1.094945737045367906172), /* 0x1.184e5d23812500007 */
	__IEEE754_DOUBLE_C(1.096429081816546080591), /* 0x1.18af9388c90e40005 */
	__IEEE754_DOUBLE_C(1.097914436104650892651), /* 0x1.1910eba4e031a0001 */
	__IEEE754_DOUBLE_C(1.099401802629782043408), /* 0x1.19726583755720003 */
	__IEEE754_DOUBLE_C(1.100891184121537858001), /* 0x1.19d4013041b860007 */
	__IEEE754_DOUBLE_C(1.102382583308144647940), /* 0x1.1a35beb6fd0cd0007 */
	__IEEE754_DOUBLE_C(1.103876002922312915544), /* 0x1.1a979e2363fa10000 */
	__IEEE754_DOUBLE_C(1.105371445702084232160), /* 0x1.1af99f8139025000e */
	__IEEE754_DOUBLE_C(1.106868914387219016199), /* 0x1.1b5bc2dc408b9000e */
	__IEEE754_DOUBLE_C(1.108368411723785085252), /* 0x1.1bbe084045eb30002 */
	__IEEE754_DOUBLE_C(1.109869940458469095340), /* 0x1.1c206fb91524c000e */
	__IEEE754_DOUBLE_C(1.111373503344554869449), /* 0x1.1c82f952817cc0001 */
	__IEEE754_DOUBLE_C(1.112879103137133007859), /* 0x1.1ce5a51860344000f */
	__IEEE754_DOUBLE_C(1.114386742595953938610), /* 0x1.1d4873168babf000e */
	__IEEE754_DOUBLE_C(1.115896424484008608911), /* 0x1.1dab6358e1d4a000f */
	__IEEE754_DOUBLE_C(1.117408151567338414664), /* 0x1.1e0e75eb43f9c000c */
	__IEEE754_DOUBLE_C(1.118921926613465345265), /* 0x1.1e71aad995078000f */
	__IEEE754_DOUBLE_C(1.120437752409564780022), /* 0x1.1ed5022fcd8600003 */
	__IEEE754_DOUBLE_C(1.121955631720569668277), /* 0x1.1f387bf9cd88b0000 */
	__IEEE754_DOUBLE_C(1.123475567332998359439), /* 0x1.1f9c18438cdec000a */
	__IEEE754_DOUBLE_C(1.124997562033035469759), /* 0x1.1fffd71902f970002 */
	__IEEE754_DOUBLE_C(1.126521618608448571713), /* 0x1.2063b88629079000e */
	__IEEE754_DOUBLE_C(1.128047739853580200284), /* 0x1.20c7bc96ff72a0002 */
	__IEEE754_DOUBLE_C(1.129575928566289189112), /* 0x1.212be3578a81e0006 */
	__IEEE754_DOUBLE_C(1.131106187546149888259), /* 0x1.21902cd3d05f70007 */
	__IEEE754_DOUBLE_C(1.132638519598779369743), /* 0x1.21f49917ddda5000c */
	__IEEE754_DOUBLE_C(1.134172927531616359481), /* 0x1.2259282fc1c24000e */
	__IEEE754_DOUBLE_C(1.135709414157753949251), /* 0x1.22bdda27911e90007 */
	__IEEE754_DOUBLE_C(1.137247982292643566662), /* 0x1.2322af0b638e60007 */
	__IEEE754_DOUBLE_C(1.138788634756517259562), /* 0x1.2387a6e755f270000 */
	__IEEE754_DOUBLE_C(1.140331374372893558110), /* 0x1.23ecc1c788c890006 */
	__IEEE754_DOUBLE_C(1.141876203969685699176), /* 0x1.2451ffb821639000c */
	__IEEE754_DOUBLE_C(1.143423126377846266197), /* 0x1.24b760c5486dc0009 */
	__IEEE754_DOUBLE_C(1.144972144431494420774), /* 0x1.251ce4fb2a0cc0005 */
	__IEEE754_DOUBLE_C(1.146523260971646252006), /* 0x1.25828c65f9fb8000d */
	__IEEE754_DOUBLE_C(1.148076478839068270690), /* 0x1.25e85711ebaeb0000 */
	__IEEE754_DOUBLE_C(1.149631800883562204903), /* 0x1.264e450b3c8a30008 */
	__IEEE754_DOUBLE_C(1.151189229953253789786), /* 0x1.26b4565e281a20003 */
	__IEEE754_DOUBLE_C(1.152748768902654319399), /* 0x1.271a8b16f0f000002 */
	__IEEE754_DOUBLE_C(1.154310420590433317050), /* 0x1.2780e341de2fc0001 */
	__IEEE754_DOUBLE_C(1.155874187878668246681), /* 0x1.27e75eeb3abc90007 */
	__IEEE754_DOUBLE_C(1.157440073633736243899), /* 0x1.284dfe1f5633e000a */
	__IEEE754_DOUBLE_C(1.159008080725518974322), /* 0x1.28b4c0ea840d90001 */
	__IEEE754_DOUBLE_C(1.160578212048386514965), /* 0x1.291ba75932ae60000 */
	__IEEE754_DOUBLE_C(1.162150470417516290340), /* 0x1.2982b177796850008 */
	__IEEE754_DOUBLE_C(1.163724858777502646494), /* 0x1.29e9df51fdd900001 */
	__IEEE754_DOUBLE_C(1.165301379991388053320), /* 0x1.2a5130f50bf34000e */
	__IEEE754_DOUBLE_C(1.166880036952526289469), /* 0x1.2ab8a66d10fdc0008 */
	__IEEE754_DOUBLE_C(1.168460832550151540268), /* 0x1.2b203fc675b7a000a */
	__IEEE754_DOUBLE_C(1.170043769683112966389), /* 0x1.2b87fd0dad7260008 */
	__IEEE754_DOUBLE_C(1.171628851252754177681), /* 0x1.2befde4f2e3da000d */
	__IEEE754_DOUBLE_C(1.173216080163546060084), /* 0x1.2c57e397719940002 */
	__IEEE754_DOUBLE_C(1.174805459325657830448), /* 0x1.2cc00cf2f7491000c */
	__IEEE754_DOUBLE_C(1.176396991650083379037), /* 0x1.2d285a6e3ff90000b */
	__IEEE754_DOUBLE_C(1.177990680055698513602), /* 0x1.2d90cc15d4ff90005 */
	__IEEE754_DOUBLE_C(1.179586527463262646306), /* 0x1.2df961f641c57000c */
	__IEEE754_DOUBLE_C(1.181184536796979545103), /* 0x1.2e621c1c157cd000d */
	__IEEE754_DOUBLE_C(1.182784710984701836994), /* 0x1.2ecafa93e35af0004 */
	__IEEE754_DOUBLE_C(1.184387052960675701386), /* 0x1.2f33fd6a459cb0000 */
	__IEEE754_DOUBLE_C(1.185991565661414393112), /* 0x1.2f9d24abd8fd1000e */
	__IEEE754_DOUBLE_C(1.187598252026902612178), /* 0x1.300670653e083000a */
	__IEEE754_DOUBLE_C(1.189207115003001469262), /* 0x1.306fe0a31bc040008 */
	__IEEE754_DOUBLE_C(1.190818157535919796833), /* 0x1.30d9757219895000e */
	__IEEE754_DOUBLE_C(1.192431382587621380206), /* 0x1.31432edef01a1000f */
	__IEEE754_DOUBLE_C(1.194046793097208292195), /* 0x1.31ad0cf63f0630008 */
	__IEEE754_DOUBLE_C(1.195664392040319823392), /* 0x1.32170fc4ce0db000c */
	__IEEE754_DOUBLE_C(1.197284182375793593084), /* 0x1.32813757527750005 */
	__IEEE754_DOUBLE_C(1.198906167074650808198), /* 0x1.32eb83ba8eef3000f */
	__IEEE754_DOUBLE_C(1.200530349107333139048), /* 0x1.3355f4fb457e5000d */
	__IEEE754_DOUBLE_C(1.202156731453099647353), /* 0x1.33c08b2641df9000c */
	__IEEE754_DOUBLE_C(1.203785317090505513368), /* 0x1.342b46484f07b0005 */
	__IEEE754_DOUBLE_C(1.205416109005122526928), /* 0x1.3496266e3fa270005 */
	__IEEE754_DOUBLE_C(1.207049110184904572310), /* 0x1.35012ba4e8fa10000 */
	__IEEE754_DOUBLE_C(1.208684323627194912036), /* 0x1.356c55f92aabb0004 */
	__IEEE754_DOUBLE_C(1.210321752322854882437), /* 0x1.35d7a577dd33f0004 */
	__IEEE754_DOUBLE_C(1.211961399276747286580), /* 0x1.36431a2de8748000d */
	__IEEE754_DOUBLE_C(1.213603267492579629347), /* 0x1.36aeb4283309e000c */
	__IEEE754_DOUBLE_C(1.215247359985374142610), /* 0x1.371a7373b00160000 */
	__IEEE754_DOUBLE_C(1.216893679753690671322), /* 0x1.3786581d404e90000 */
	__IEEE754_DOUBLE_C(1.218542229828181611183), /* 0x1.37f26231e82e4000c */
	__IEEE754_DOUBLE_C(1.220193013225231215567), /* 0x1.385e91be9c2d20002 */
	__IEEE754_DOUBLE_C(1.221846032973555429280), /* 0x1.38cae6d05e66f0000 */
	__IEEE754_DOUBLE_C(1.223501292099485437962), /* 0x1.393761742e5830001 */
	__IEEE754_DOUBLE_C(1.225158793636904830441), /* 0x1.39a401b713cb3000e */
	__IEEE754_DOUBLE_C(1.226818540625497444577), /* 0x1.3a10c7a61ceae0007 */
	__IEEE754_DOUBLE_C(1.228480536107136034131), /* 0x1.3a7db34e5a4a50003 */
	__IEEE754_DOUBLE_C(1.230144783126481566885), /* 0x1.3aeac4bcdf8d60001 */
	__IEEE754_DOUBLE_C(1.231811284734168454619), /* 0x1.3b57fbfec6e950008 */
	__IEEE754_DOUBLE_C(1.233480043984379381835), /* 0x1.3bc559212e7a2000f */
	__IEEE754_DOUBLE_C(1.235151063936380300149), /* 0x1.3c32dc3139f2a0004 */
	__IEEE754_DOUBLE_C(1.236824347652524913647), /* 0x1.3ca0853c106ac000e */
	__IEEE754_DOUBLE_C(1.238499898199571624970), /* 0x1.3d0e544eddd240003 */
	__IEEE754_DOUBLE_C(1.240177718649636107175), /* 0x1.3d7c4976d3fcd0000 */
	__IEEE754_DOUBLE_C(1.241857812073360767273), /* 0x1.3dea64c1231f70004 */
	__IEEE754_DOUBLE_C(1.243540181554270152039), /* 0x1.3e58a63b099920005 */
	__IEEE754_DOUBLE_C(1.245224830175077013244), /* 0x1.3ec70df1c4e46000e */
	__IEEE754_DOUBLE_C(1.246911761022835740725), /* 0x1.3f359bf29741c000e */
	__IEEE754_DOUBLE_C(1.248600977188942806639), /* 0x1.3fa4504ac7b800009 */
	__IEEE754_DOUBLE_C(1.250292481770148400634), /* 0x1.40132b07a330d000a */
	__IEEE754_DOUBLE_C(1.251986277866492969263), /* 0x1.40822c367a340000b */
	__IEEE754_DOUBLE_C(1.253682368581898742876), /* 0x1.40f153e4a18e0000d */
	__IEEE754_DOUBLE_C(1.255380757024939564249), /* 0x1.4160a21f73289000d */
	__IEEE754_DOUBLE_C(1.257081446308726757662), /* 0x1.41d016f44deaa000c */
	__IEEE754_DOUBLE_C(1.258784439550028944083), /* 0x1.423fb27094c090008 */
	__IEEE754_DOUBLE_C(1.260489739869405489991), /* 0x1.42af74a1aec1c0006 */
	__IEEE754_DOUBLE_C(1.262197350394008266193), /* 0x1.431f5d950a453000c */
	__IEEE754_DOUBLE_C(1.263907274252603851764), /* 0x1.438f6d58176860004 */
	__IEEE754_DOUBLE_C(1.265619514578811388761), /* 0x1.43ffa3f84b9eb000d */
	__IEEE754_DOUBLE_C(1.267334074511444086425), /* 0x1.44700183221180008 */
	__IEEE754_DOUBLE_C(1.269050957191869555296), /* 0x1.44e0860618b930006 */
	__IEEE754_DOUBLE_C(1.270770165768063009230), /* 0x1.4551318eb4d20000e */
	__IEEE754_DOUBLE_C(1.272491703389059036805), /* 0x1.45c2042a7cc26000b */
	__IEEE754_DOUBLE_C(1.274215573211836316547), /* 0x1.4632fde6ffacd000d */
	__IEEE754_DOUBLE_C(1.275941778396075143580), /* 0x1.46a41ed1cfac40001 */
	__IEEE754_DOUBLE_C(1.277670322103555911043), /* 0x1.471566f8812ac0000 */
	__IEEE754_DOUBLE_C(1.279401207505722393185), /* 0x1.4786d668b33260005 */
	__IEEE754_DOUBLE_C(1.281134437771823675369), /* 0x1.47f86d3002637000a */
	__IEEE754_DOUBLE_C(1.282870016078732078362), /* 0x1.486a2b5c13c00000e */
	__IEEE754_DOUBLE_C(1.284607945607987078432), /* 0x1.48dc10fa916bd0004 */
	__IEEE754_DOUBLE_C(1.286348229545787758022), /* 0x1.494e1e192aaa30007 */
	__IEEE754_DOUBLE_C(1.288090871080605159846), /* 0x1.49c052c5913df000c */
	__IEEE754_DOUBLE_C(1.289835873406902644341), /* 0x1.4a32af0d7d8090002 */
	__IEEE754_DOUBLE_C(1.291583239722392528754), /* 0x1.4aa532feab5e10002 */
	__IEEE754_DOUBLE_C(1.293332973229098792374), /* 0x1.4b17dea6db8010008 */
	__IEEE754_DOUBLE_C(1.295085077135345708087), /* 0x1.4b8ab213d57d9000d */
	__IEEE754_DOUBLE_C(1.296839554650994097442), /* 0x1.4bfdad53629e10003 */
	__IEEE754_DOUBLE_C(1.298596408992440220988), /* 0x1.4c70d0735358a000d */
	__IEEE754_DOUBLE_C(1.300355643380135983739), /* 0x1.4ce41b817c99e0001 */
	__IEEE754_DOUBLE_C(1.302117261036232376282), /* 0x1.4d578e8bb52cb0003 */
	__IEEE754_DOUBLE_C(1.303881265192249561154), /* 0x1.4dcb299fde2920008 */
	__IEEE754_DOUBLE_C(1.305647659079073541490), /* 0x1.4e3eeccbd7f4c0003 */
	__IEEE754_DOUBLE_C(1.307416445934474813521), /* 0x1.4eb2d81d8a86f000b */
	__IEEE754_DOUBLE_C(1.309187629001237640529), /* 0x1.4f26eba2e35a5000e */
	__IEEE754_DOUBLE_C(1.310961211525240921493), /* 0x1.4f9b2769d35090009 */
	__IEEE754_DOUBLE_C(1.312737196755087820678), /* 0x1.500f8b804e4a30000 */
	__IEEE754_DOUBLE_C(1.314515587949291131086), /* 0x1.508417f4530d00009 */
	__IEEE754_DOUBLE_C(1.316296388365203462468), /* 0x1.50f8ccd3df1840003 */
	__IEEE754_DOUBLE_C(1.318079601265708777911), /* 0x1.516daa2cf60020002 */
	__IEEE754_DOUBLE_C(1.319865229921343141607), /* 0x1.51e2b00da3c2b0007 */
	__IEEE754_DOUBLE_C(1.321653277603506371251), /* 0x1.5257de83f5512000d */
	__IEEE754_DOUBLE_C(1.323443747588034513690), /* 0x1.52cd359dfc7d5000e */
	__IEEE754_DOUBLE_C(1.325236643161341820781), /* 0x1.5342b569d6baa000f */
	__IEEE754_DOUBLE_C(1.327031967602244177939), /* 0x1.53b85df59921b0000 */
	__IEEE754_DOUBLE_C(1.328829724206201046165), /* 0x1.542e2f4f6b17e0006 */
	__IEEE754_DOUBLE_C(1.330629916266568235675), /* 0x1.54a4298571b27000e */
	__IEEE754_DOUBLE_C(1.332432547083447937938), /* 0x1.551a4ca5d97190009 */
	__IEEE754_DOUBLE_C(1.334237619959296017340), /* 0x1.559098bed16bf0008 */
	__IEEE754_DOUBLE_C(1.336045138203900251029), /* 0x1.56070dde90c800000 */
	__IEEE754_DOUBLE_C(1.337855105129210686631), /* 0x1.567dac13510cd0009 */
	__IEEE754_DOUBLE_C(1.339667524053662184301), /* 0x1.56f4736b52e2c000c */
	__IEEE754_DOUBLE_C(1.341482398296830025383), /* 0x1.576b63f4d8333000f */
	__IEEE754_DOUBLE_C(1.343299731186792467254), /* 0x1.57e27dbe2c40e0003 */
	__IEEE754_DOUBLE_C(1.345119526053918823702), /* 0x1.5859c0d59cd37000f */
	__IEEE754_DOUBLE_C(1.346941786233264881662), /* 0x1.58d12d497cd9a0005 */
	__IEEE754_DOUBLE_C(1.348766515064854010261), /* 0x1.5948c32824b87000c */
	__IEEE754_DOUBLE_C(1.350593715891792223641), /* 0x1.59c0827ff03890007 */
	__IEEE754_DOUBLE_C(1.352423392064920459908), /* 0x1.5a386b5f43a3e0006 */
	__IEEE754_DOUBLE_C(1.354255546937278120764), /* 0x1.5ab07dd485af1000c */
	__IEEE754_DOUBLE_C(1.356090183865519494030), /* 0x1.5b28b9ee21085000f */
	__IEEE754_DOUBLE_C(1.357927306213322804534), /* 0x1.5ba11fba8816e000b */
	__IEEE754_DOUBLE_C(1.359766917346459269620), /* 0x1.5c19af482f8f2000f */
	__IEEE754_DOUBLE_C(1.361609020638567812980), /* 0x1.5c9268a594cc00004 */
	__IEEE754_DOUBLE_C(1.363453619463660171403), /* 0x1.5d0b4be135916000c */
	__IEEE754_DOUBLE_C(1.365300717204201985683), /* 0x1.5d84590998eeb0005 */
	__IEEE754_DOUBLE_C(1.367150317245710233754), /* 0x1.5dfd902d494e40001 */
	__IEEE754_DOUBLE_C(1.369002422974674892971), /* 0x1.5e76f15ad22c40008 */
	__IEEE754_DOUBLE_C(1.370857037789471544224), /* 0x1.5ef07ca0cc166000b */
	__IEEE754_DOUBLE_C(1.372714165088220639199), /* 0x1.5f6a320dcf5280006 */
	__IEEE754_DOUBLE_C(1.374573808273481745378), /* 0x1.5fe411b0790800009 */
	__IEEE754_DOUBLE_C(1.376435970755022220096), /* 0x1.605e1b976e4b1000e */
	__IEEE754_DOUBLE_C(1.378300655944092456600), /* 0x1.60d84fd155d15000e */
	__IEEE754_DOUBLE_C(1.380167867259843417228), /* 0x1.6152ae6cdf0030003 */
	__IEEE754_DOUBLE_C(1.382037608124419003675), /* 0x1.61cd3778bc879000d */
	__IEEE754_DOUBLE_C(1.383909881963391264069), /* 0x1.6247eb03a4dc40009 */
	__IEEE754_DOUBLE_C(1.385784692209972801544), /* 0x1.62c2c91c56d9b0002 */
	__IEEE754_DOUBLE_C(1.387662042298923203992), /* 0x1.633dd1d1930ec0001 */
	__IEEE754_DOUBLE_C(1.389541935670444372533), /* 0x1.63b90532200630004 */
	__IEEE754_DOUBLE_C(1.391424375772021271329), /* 0x1.6434634ccc4cc0007 */
	__IEEE754_DOUBLE_C(1.393309366052102982208), /* 0x1.64afec30677e90008 */
	__IEEE754_DOUBLE_C(1.395196909966106124701), /* 0x1.652b9febc8e0f000d */
	__IEEE754_DOUBLE_C(1.397087010973788290271), /* 0x1.65a77e8dcc7f10004 */
	__IEEE754_DOUBLE_C(1.398979672539331309267), /* 0x1.66238825534170000 */
	__IEEE754_DOUBLE_C(1.400874898129892187656), /* 0x1.669fbcc1415600008 */
	__IEEE754_DOUBLE_C(1.402772691220124823310), /* 0x1.671c1c708328e000a */
	__IEEE754_DOUBLE_C(1.404673055288671035301), /* 0x1.6798a7420988b000d */
	__IEEE754_DOUBLE_C(1.406575993818903302975), /* 0x1.68155d44ca77a000f */
	__IEEE754_DOUBLE_C(1.408481510297352468121), /* 0x1.68923e87bf70e000a */
	__IEEE754_DOUBLE_C(1.410389608216942924956), /* 0x1.690f4b19e8f74000c */
	__IEEE754_DOUBLE_C(1.412300291075172076232), /* 0x1.698c830a4c94c0008 */
};
#define __LIBM_EXP2S (__IEEE754_DOUBLE_C(1.0) / __IEEE754_DOUBLE_C(4503599627370496.0)) /* 2^-52 */
#ifdef __IEEE754_FLOAT_TYPE__
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_FLOAT_TYPE__, exp2_deltatable, 512)
#else /* __IEEE754_FLOAT_TYPE__ */
__LIBM_LOCAL_DECLARE_ARRAY(__IEEE754_DOUBLE_TYPE__, exp2_deltatable, 512)
#endif /* !__IEEE754_FLOAT_TYPE__ */
{
	__INT32_C(11527) * __LIBM_EXP2S, __INT32_C(-963) * __LIBM_EXP2S, __INT32_C(884) * __LIBM_EXP2S, __INT32_C(-781) * __LIBM_EXP2S, __INT32_C(-2363) * __LIBM_EXP2S, __INT32_C(-3441) * __LIBM_EXP2S, __INT32_C(123) * __LIBM_EXP2S, __INT32_C(526) * __LIBM_EXP2S,
	__INT32_C(-6) * __LIBM_EXP2S, __INT32_C(1254) * __LIBM_EXP2S, __INT32_C(-1138) * __LIBM_EXP2S, __INT32_C(1519) * __LIBM_EXP2S, __INT32_C(1576) * __LIBM_EXP2S, __INT32_C(-65) * __LIBM_EXP2S, __INT32_C(1040) * __LIBM_EXP2S, __INT32_C(793) * __LIBM_EXP2S,
	__INT32_C(-1662) * __LIBM_EXP2S, __INT32_C(-5063) * __LIBM_EXP2S, __INT32_C(-387) * __LIBM_EXP2S, __INT32_C(968) * __LIBM_EXP2S, __INT32_C(-941) * __LIBM_EXP2S, __INT32_C(984) * __LIBM_EXP2S, __INT32_C(-2856) * __LIBM_EXP2S, __INT32_C(-545) * __LIBM_EXP2S,
	__INT32_C(495) * __LIBM_EXP2S, __INT32_C(-5246) * __LIBM_EXP2S, __INT32_C(-2109) * __LIBM_EXP2S, __INT32_C(1281) * __LIBM_EXP2S, __INT32_C(2075) * __LIBM_EXP2S, __INT32_C(909) * __LIBM_EXP2S, __INT32_C(-1642) * __LIBM_EXP2S, __INT32_C(-78233) * __LIBM_EXP2S,
	__INT32_C(-31653) * __LIBM_EXP2S, __INT32_C(-265) * __LIBM_EXP2S, __INT32_C(130) * __LIBM_EXP2S, __INT32_C(430) * __LIBM_EXP2S, __INT32_C(2482) * __LIBM_EXP2S, __INT32_C(-742) * __LIBM_EXP2S, __INT32_C(1616) * __LIBM_EXP2S, __INT32_C(-2213) * __LIBM_EXP2S,
	__INT32_C(-519) * __LIBM_EXP2S, __INT32_C(20) * __LIBM_EXP2S, __INT32_C(-3134) * __LIBM_EXP2S, __INT32_C(-13981) * __LIBM_EXP2S, __INT32_C(1343) * __LIBM_EXP2S, __INT32_C(-1740) * __LIBM_EXP2S, __INT32_C(247) * __LIBM_EXP2S, __INT32_C(1679) * __LIBM_EXP2S,
	__INT32_C(-1097) * __LIBM_EXP2S, __INT32_C(3131) * __LIBM_EXP2S, __INT32_C(871) * __LIBM_EXP2S, __INT32_C(-1480) * __LIBM_EXP2S, __INT32_C(1936) * __LIBM_EXP2S, __INT32_C(-1827) * __LIBM_EXP2S, __INT32_C(17325) * __LIBM_EXP2S, __INT32_C(528) * __LIBM_EXP2S,
	__INT32_C(-322) * __LIBM_EXP2S, __INT32_C(1404) * __LIBM_EXP2S, __INT32_C(-152) * __LIBM_EXP2S, __INT32_C(-1845) * __LIBM_EXP2S, __INT32_C(-212) * __LIBM_EXP2S, __INT32_C(2639) * __LIBM_EXP2S, __INT32_C(-476) * __LIBM_EXP2S, __INT32_C(2960) * __LIBM_EXP2S,
	__INT32_C(-962) * __LIBM_EXP2S, __INT32_C(-1012) * __LIBM_EXP2S, __INT32_C(-1231) * __LIBM_EXP2S, __INT32_C(3030) * __LIBM_EXP2S, __INT32_C(1659) * __LIBM_EXP2S, __INT32_C(-486) * __LIBM_EXP2S, __INT32_C(2154) * __LIBM_EXP2S, __INT32_C(1728) * __LIBM_EXP2S,
	__INT32_C(-2793) * __LIBM_EXP2S, __INT32_C(699) * __LIBM_EXP2S, __INT32_C(-1560) * __LIBM_EXP2S, __INT32_C(-2125) * __LIBM_EXP2S, __INT32_C(2156) * __LIBM_EXP2S, __INT32_C(142) * __LIBM_EXP2S, __INT32_C(-1888) * __LIBM_EXP2S, __INT32_C(4426) * __LIBM_EXP2S,
	__INT32_C(-13443) * __LIBM_EXP2S, __INT32_C(1970) * __LIBM_EXP2S, __INT32_C(-50) * __LIBM_EXP2S, __INT32_C(1771) * __LIBM_EXP2S, __INT32_C(-43399) * __LIBM_EXP2S, __INT32_C(4979) * __LIBM_EXP2S, __INT32_C(-2448) * __LIBM_EXP2S, __INT32_C(-370) * __LIBM_EXP2S,
	__INT32_C(1414) * __LIBM_EXP2S, __INT32_C(1075) * __LIBM_EXP2S, __INT32_C(232) * __LIBM_EXP2S, __INT32_C(206) * __LIBM_EXP2S, __INT32_C(873) * __LIBM_EXP2S, __INT32_C(2141) * __LIBM_EXP2S, __INT32_C(2970) * __LIBM_EXP2S, __INT32_C(1279) * __LIBM_EXP2S,
	__INT32_C(-2331) * __LIBM_EXP2S, __INT32_C(336) * __LIBM_EXP2S, __INT32_C(-2595) * __LIBM_EXP2S, __INT32_C(753) * __LIBM_EXP2S, __INT32_C(-3384) * __LIBM_EXP2S, __INT32_C(-616) * __LIBM_EXP2S, __INT32_C(89) * __LIBM_EXP2S, __INT32_C(-818) * __LIBM_EXP2S,
	__INT32_C(5755) * __LIBM_EXP2S, __INT32_C(-241) * __LIBM_EXP2S, __INT32_C(-528) * __LIBM_EXP2S, __INT32_C(-661) * __LIBM_EXP2S, __INT32_C(-3777) * __LIBM_EXP2S, __INT32_C(-354) * __LIBM_EXP2S, __INT32_C(250) * __LIBM_EXP2S, __INT32_C(3881) * __LIBM_EXP2S,
	__INT32_C(2632) * __LIBM_EXP2S, __INT32_C(-2131) * __LIBM_EXP2S, __INT32_C(2565) * __LIBM_EXP2S, __INT32_C(-316) * __LIBM_EXP2S, __INT32_C(1746) * __LIBM_EXP2S, __INT32_C(-2541) * __LIBM_EXP2S, __INT32_C(-1324) * __LIBM_EXP2S, __INT32_C(-50) * __LIBM_EXP2S,
	__INT32_C(2564) * __LIBM_EXP2S, __INT32_C(-782) * __LIBM_EXP2S, __INT32_C(1176) * __LIBM_EXP2S, __INT32_C(6452) * __LIBM_EXP2S, __INT32_C(-1002) * __LIBM_EXP2S, __INT32_C(1288) * __LIBM_EXP2S, __INT32_C(336) * __LIBM_EXP2S, __INT32_C(-185) * __LIBM_EXP2S,
	__INT32_C(3063) * __LIBM_EXP2S, __INT32_C(3784) * __LIBM_EXP2S, __INT32_C(2169) * __LIBM_EXP2S, __INT32_C(686) * __LIBM_EXP2S, __INT32_C(328) * __LIBM_EXP2S, __INT32_C(-400) * __LIBM_EXP2S, __INT32_C(312) * __LIBM_EXP2S, __INT32_C(-4517) * __LIBM_EXP2S,
	__INT32_C(-1457) * __LIBM_EXP2S, __INT32_C(1046) * __LIBM_EXP2S, __INT32_C(-1530) * __LIBM_EXP2S, __INT32_C(-685) * __LIBM_EXP2S, __INT32_C(1328) * __LIBM_EXP2S, __INT32_C(-49815) * __LIBM_EXP2S, __INT32_C(-895) * __LIBM_EXP2S, __INT32_C(1063) * __LIBM_EXP2S,
	__INT32_C(-2091) * __LIBM_EXP2S, __INT32_C(-672) * __LIBM_EXP2S, __INT32_C(-1710) * __LIBM_EXP2S, __INT32_C(-665) * __LIBM_EXP2S, __INT32_C(1545) * __LIBM_EXP2S, __INT32_C(1819) * __LIBM_EXP2S, __INT32_C(-45265) * __LIBM_EXP2S, __INT32_C(3548) * __LIBM_EXP2S,
	__INT32_C(-554) * __LIBM_EXP2S, __INT32_C(-568) * __LIBM_EXP2S, __INT32_C(4752) * __LIBM_EXP2S, __INT32_C(-1907) * __LIBM_EXP2S, __INT32_C(-13738) * __LIBM_EXP2S, __INT32_C(675) * __LIBM_EXP2S, __INT32_C(9611) * __LIBM_EXP2S, __INT32_C(-1115) * __LIBM_EXP2S,
	__INT32_C(-815) * __LIBM_EXP2S, __INT32_C(408) * __LIBM_EXP2S, __INT32_C(-1281) * __LIBM_EXP2S, __INT32_C(-937) * __LIBM_EXP2S, __INT32_C(-16376) * __LIBM_EXP2S, __INT32_C(-4772) * __LIBM_EXP2S, __INT32_C(-1440) * __LIBM_EXP2S, __INT32_C(992) * __LIBM_EXP2S,
	__INT32_C(788) * __LIBM_EXP2S, __INT32_C(10364) * __LIBM_EXP2S, __INT32_C(-1602) * __LIBM_EXP2S, __INT32_C(-661) * __LIBM_EXP2S, __INT32_C(-1783) * __LIBM_EXP2S, __INT32_C(-265) * __LIBM_EXP2S, __INT32_C(-20) * __LIBM_EXP2S, __INT32_C(-3781) * __LIBM_EXP2S,
	__INT32_C(-861) * __LIBM_EXP2S, __INT32_C(-345) * __LIBM_EXP2S, __INT32_C(-994) * __LIBM_EXP2S, __INT32_C(1364) * __LIBM_EXP2S, __INT32_C(-5339) * __LIBM_EXP2S, __INT32_C(1620) * __LIBM_EXP2S, __INT32_C(9390) * __LIBM_EXP2S, __INT32_C(-1066) * __LIBM_EXP2S,
	__INT32_C(-305) * __LIBM_EXP2S, __INT32_C(-170) * __LIBM_EXP2S, __INT32_C(175) * __LIBM_EXP2S, __INT32_C(2461) * __LIBM_EXP2S, __INT32_C(-490) * __LIBM_EXP2S, __INT32_C(-769) * __LIBM_EXP2S, __INT32_C(-1450) * __LIBM_EXP2S, __INT32_C(3315) * __LIBM_EXP2S,
	__INT32_C(2418) * __LIBM_EXP2S, __INT32_C(-45) * __LIBM_EXP2S, __INT32_C(-852) * __LIBM_EXP2S, __INT32_C(-1295) * __LIBM_EXP2S, __INT32_C(-488) * __LIBM_EXP2S, __INT32_C(-96) * __LIBM_EXP2S, __INT32_C(1142) * __LIBM_EXP2S, __INT32_C(-2639) * __LIBM_EXP2S,
	__INT32_C(7905) * __LIBM_EXP2S, __INT32_C(-9306) * __LIBM_EXP2S, __INT32_C(-3859) * __LIBM_EXP2S, __INT32_C(760) * __LIBM_EXP2S, __INT32_C(1057) * __LIBM_EXP2S, __INT32_C(-1570) * __LIBM_EXP2S, __INT32_C(3977) * __LIBM_EXP2S, __INT32_C(209) * __LIBM_EXP2S,
	__INT32_C(-514) * __LIBM_EXP2S, __INT32_C(7151) * __LIBM_EXP2S, __INT32_C(1646) * __LIBM_EXP2S, __INT32_C(627) * __LIBM_EXP2S, __INT32_C(599) * __LIBM_EXP2S, __INT32_C(-774) * __LIBM_EXP2S, __INT32_C(-1468) * __LIBM_EXP2S, __INT32_C(633) * __LIBM_EXP2S,
	__INT32_C(-473) * __LIBM_EXP2S, __INT32_C(851) * __LIBM_EXP2S, __INT32_C(2406) * __LIBM_EXP2S, __INT32_C(143) * __LIBM_EXP2S, __INT32_C(74) * __LIBM_EXP2S, __INT32_C(4260) * __LIBM_EXP2S, __INT32_C(1177) * __LIBM_EXP2S, __INT32_C(-913) * __LIBM_EXP2S,
	__INT32_C(2670) * __LIBM_EXP2S, __INT32_C(-3298) * __LIBM_EXP2S, __INT32_C(-1662) * __LIBM_EXP2S, __INT32_C(-120) * __LIBM_EXP2S, __INT32_C(-3264) * __LIBM_EXP2S, __INT32_C(-2148) * __LIBM_EXP2S, __INT32_C(410) * __LIBM_EXP2S, __INT32_C(2078) * __LIBM_EXP2S,
	__INT32_C(-2098) * __LIBM_EXP2S, __INT32_C(-926) * __LIBM_EXP2S, __INT32_C(3580) * __LIBM_EXP2S, __INT32_C(-1289) * __LIBM_EXP2S, __INT32_C(2450) * __LIBM_EXP2S, __INT32_C(-1158) * __LIBM_EXP2S, __INT32_C(907) * __LIBM_EXP2S, __INT32_C(-590) * __LIBM_EXP2S,
	__INT32_C(986) * __LIBM_EXP2S, __INT32_C(1801) * __LIBM_EXP2S, __INT32_C(1145) * __LIBM_EXP2S, __INT32_C(-1677) * __LIBM_EXP2S, __INT32_C(3455) * __LIBM_EXP2S, __INT32_C(956) * __LIBM_EXP2S, __INT32_C(710) * __LIBM_EXP2S, __INT32_C(144) * __LIBM_EXP2S,
	__INT32_C(153) * __LIBM_EXP2S, __INT32_C(-255) * __LIBM_EXP2S, __INT32_C(-1898) * __LIBM_EXP2S, __INT32_C(28102) * __LIBM_EXP2S, __INT32_C(2748) * __LIBM_EXP2S, __INT32_C(1194) * __LIBM_EXP2S, __INT32_C(-3009) * __LIBM_EXP2S, __INT32_C(7076) * __LIBM_EXP2S,
	__INT32_C(0) * __LIBM_EXP2S, __INT32_C(-2720) * __LIBM_EXP2S, __INT32_C(711) * __LIBM_EXP2S, __INT32_C(1225) * __LIBM_EXP2S, __INT32_C(-3034) * __LIBM_EXP2S, __INT32_C(-473) * __LIBM_EXP2S, __INT32_C(378) * __LIBM_EXP2S, __INT32_C(-1046) * __LIBM_EXP2S,
	__INT32_C(962) * __LIBM_EXP2S, __INT32_C(-2006) * __LIBM_EXP2S, __INT32_C(4647) * __LIBM_EXP2S, __INT32_C(3206) * __LIBM_EXP2S, __INT32_C(1769) * __LIBM_EXP2S, __INT32_C(-2665) * __LIBM_EXP2S, __INT32_C(1254) * __LIBM_EXP2S, __INT32_C(2025) * __LIBM_EXP2S,
	__INT32_C(-2430) * __LIBM_EXP2S, __INT32_C(6193) * __LIBM_EXP2S, __INT32_C(1224) * __LIBM_EXP2S, __INT32_C(-856) * __LIBM_EXP2S, __INT32_C(-1592) * __LIBM_EXP2S, __INT32_C(-325) * __LIBM_EXP2S, __INT32_C(-1521) * __LIBM_EXP2S, __INT32_C(1827) * __LIBM_EXP2S,
	__INT32_C(-264) * __LIBM_EXP2S, __INT32_C(2403) * __LIBM_EXP2S, __INT32_C(-1065) * __LIBM_EXP2S, __INT32_C(967) * __LIBM_EXP2S, __INT32_C(-681) * __LIBM_EXP2S, __INT32_C(-2106) * __LIBM_EXP2S, __INT32_C(-474) * __LIBM_EXP2S, __INT32_C(1333) * __LIBM_EXP2S,
	__INT32_C(-893) * __LIBM_EXP2S, __INT32_C(2296) * __LIBM_EXP2S, __INT32_C(592) * __LIBM_EXP2S, __INT32_C(-1220) * __LIBM_EXP2S, __INT32_C(-326) * __LIBM_EXP2S, __INT32_C(990) * __LIBM_EXP2S, __INT32_C(139) * __LIBM_EXP2S, __INT32_C(206) * __LIBM_EXP2S,
	__INT32_C(-779) * __LIBM_EXP2S, __INT32_C(-1683) * __LIBM_EXP2S, __INT32_C(1238) * __LIBM_EXP2S, __INT32_C(6098) * __LIBM_EXP2S, __INT32_C(136) * __LIBM_EXP2S, __INT32_C(1197) * __LIBM_EXP2S, __INT32_C(790) * __LIBM_EXP2S, __INT32_C(-107) * __LIBM_EXP2S,
	__INT32_C(-1004) * __LIBM_EXP2S, __INT32_C(-2449) * __LIBM_EXP2S, __INT32_C(939) * __LIBM_EXP2S, __INT32_C(5568) * __LIBM_EXP2S, __INT32_C(156) * __LIBM_EXP2S, __INT32_C(1812) * __LIBM_EXP2S, __INT32_C(2792) * __LIBM_EXP2S, __INT32_C(-1094) * __LIBM_EXP2S,
	__INT32_C(-2677) * __LIBM_EXP2S, __INT32_C(-251) * __LIBM_EXP2S, __INT32_C(2297) * __LIBM_EXP2S, __INT32_C(943) * __LIBM_EXP2S, __INT32_C(-1329) * __LIBM_EXP2S, __INT32_C(2883) * __LIBM_EXP2S, __INT32_C(-853) * __LIBM_EXP2S, __INT32_C(-2626) * __LIBM_EXP2S,
	__INT32_C(-105929) * __LIBM_EXP2S, __INT32_C(-6552) * __LIBM_EXP2S, __INT32_C(1095) * __LIBM_EXP2S, __INT32_C(-1508) * __LIBM_EXP2S, __INT32_C(1003) * __LIBM_EXP2S, __INT32_C(5039) * __LIBM_EXP2S, __INT32_C(-2600) * __LIBM_EXP2S, __INT32_C(-749) * __LIBM_EXP2S,
	__INT32_C(1790) * __LIBM_EXP2S, __INT32_C(890) * __LIBM_EXP2S, __INT32_C(2016) * __LIBM_EXP2S, __INT32_C(-1073) * __LIBM_EXP2S, __INT32_C(624) * __LIBM_EXP2S, __INT32_C(-2084) * __LIBM_EXP2S, __INT32_C(-1536) * __LIBM_EXP2S, __INT32_C(-1330) * __LIBM_EXP2S,
	__INT32_C(358) * __LIBM_EXP2S, __INT32_C(2444) * __LIBM_EXP2S, __INT32_C(-179) * __LIBM_EXP2S, __INT32_C(-25759) * __LIBM_EXP2S, __INT32_C(-243) * __LIBM_EXP2S, __INT32_C(-552) * __LIBM_EXP2S, __INT32_C(-124) * __LIBM_EXP2S, __INT32_C(3766) * __LIBM_EXP2S,
	__INT32_C(1192) * __LIBM_EXP2S, __INT32_C(-1614) * __LIBM_EXP2S, __INT32_C(6) * __LIBM_EXP2S, __INT32_C(-1227) * __LIBM_EXP2S, __INT32_C(345) * __LIBM_EXP2S, __INT32_C(-981) * __LIBM_EXP2S, __INT32_C(-295) * __LIBM_EXP2S, __INT32_C(-1006) * __LIBM_EXP2S,
	__INT32_C(-995) * __LIBM_EXP2S, __INT32_C(-1195) * __LIBM_EXP2S, __INT32_C(706) * __LIBM_EXP2S, __INT32_C(2512) * __LIBM_EXP2S, __INT32_C(-1758) * __LIBM_EXP2S, __INT32_C(-734) * __LIBM_EXP2S, __INT32_C(-6286) * __LIBM_EXP2S, __INT32_C(-922) * __LIBM_EXP2S,
	__INT32_C(1530) * __LIBM_EXP2S, __INT32_C(1542) * __LIBM_EXP2S, __INT32_C(1223) * __LIBM_EXP2S, __INT32_C(61) * __LIBM_EXP2S, __INT32_C(-83) * __LIBM_EXP2S, __INT32_C(522) * __LIBM_EXP2S, __INT32_C(116937) * __LIBM_EXP2S, __INT32_C(-914) * __LIBM_EXP2S,
	__INT32_C(-418) * __LIBM_EXP2S, __INT32_C(-7339) * __LIBM_EXP2S, __INT32_C(249) * __LIBM_EXP2S, __INT32_C(-520) * __LIBM_EXP2S, __INT32_C(-762) * __LIBM_EXP2S, __INT32_C(426) * __LIBM_EXP2S, __INT32_C(-505) * __LIBM_EXP2S, __INT32_C(2664) * __LIBM_EXP2S,
	__INT32_C(-1093) * __LIBM_EXP2S, __INT32_C(-1035) * __LIBM_EXP2S, __INT32_C(2130) * __LIBM_EXP2S, __INT32_C(4878) * __LIBM_EXP2S, __INT32_C(1982) * __LIBM_EXP2S, __INT32_C(1551) * __LIBM_EXP2S, __INT32_C(2304) * __LIBM_EXP2S, __INT32_C(193) * __LIBM_EXP2S,
	__INT32_C(1532) * __LIBM_EXP2S, __INT32_C(-7268) * __LIBM_EXP2S, __INT32_C(24357) * __LIBM_EXP2S, __INT32_C(531) * __LIBM_EXP2S, __INT32_C(2676) * __LIBM_EXP2S, __INT32_C(-1170) * __LIBM_EXP2S, __INT32_C(1465) * __LIBM_EXP2S, __INT32_C(-1917) * __LIBM_EXP2S,
	__INT32_C(2143) * __LIBM_EXP2S, __INT32_C(1466) * __LIBM_EXP2S, __INT32_C(-7) * __LIBM_EXP2S, __INT32_C(-7300) * __LIBM_EXP2S, __INT32_C(3297) * __LIBM_EXP2S, __INT32_C(-1197) * __LIBM_EXP2S, __INT32_C(-289) * __LIBM_EXP2S, __INT32_C(-1548) * __LIBM_EXP2S,
	__INT32_C(26226) * __LIBM_EXP2S, __INT32_C(4401) * __LIBM_EXP2S, __INT32_C(4123) * __LIBM_EXP2S, __INT32_C(-1588) * __LIBM_EXP2S, __INT32_C(4243) * __LIBM_EXP2S, __INT32_C(4069) * __LIBM_EXP2S, __INT32_C(-1276) * __LIBM_EXP2S, __INT32_C(-2010) * __LIBM_EXP2S,
	__INT32_C(1407) * __LIBM_EXP2S, __INT32_C(1478) * __LIBM_EXP2S, __INT32_C(488) * __LIBM_EXP2S, __INT32_C(-2366) * __LIBM_EXP2S, __INT32_C(-2909) * __LIBM_EXP2S, __INT32_C(-2534) * __LIBM_EXP2S, __INT32_C(-1285) * __LIBM_EXP2S, __INT32_C(7095) * __LIBM_EXP2S,
	__INT32_C(-645) * __LIBM_EXP2S, __INT32_C(-2089) * __LIBM_EXP2S, __INT32_C(-944) * __LIBM_EXP2S, __INT32_C(-40) * __LIBM_EXP2S, __INT32_C(-1363) * __LIBM_EXP2S, __INT32_C(-833) * __LIBM_EXP2S, __INT32_C(917) * __LIBM_EXP2S, __INT32_C(1609) * __LIBM_EXP2S,
	__INT32_C(1286) * __LIBM_EXP2S, __INT32_C(1677) * __LIBM_EXP2S, __INT32_C(1613) * __LIBM_EXP2S, __INT32_C(-2295) * __LIBM_EXP2S, __INT32_C(-1248) * __LIBM_EXP2S, __INT32_C(40) * __LIBM_EXP2S, __INT32_C(26) * __LIBM_EXP2S, __INT32_C(2038) * __LIBM_EXP2S,
	__INT32_C(698) * __LIBM_EXP2S, __INT32_C(2675) * __LIBM_EXP2S, __INT32_C(-1755) * __LIBM_EXP2S, __INT32_C(-3522) * __LIBM_EXP2S, __INT32_C(-1614) * __LIBM_EXP2S, __INT32_C(-6111) * __LIBM_EXP2S, __INT32_C(270) * __LIBM_EXP2S, __INT32_C(1822) * __LIBM_EXP2S,
	__INT32_C(-234) * __LIBM_EXP2S, __INT32_C(-2844) * __LIBM_EXP2S, __INT32_C(-1201) * __LIBM_EXP2S, __INT32_C(-830) * __LIBM_EXP2S, __INT32_C(1193) * __LIBM_EXP2S, __INT32_C(2354) * __LIBM_EXP2S, __INT32_C(47) * __LIBM_EXP2S, __INT32_C(1522) * __LIBM_EXP2S,
	__INT32_C(-78) * __LIBM_EXP2S, __INT32_C(-640) * __LIBM_EXP2S, __INT32_C(2425) * __LIBM_EXP2S, __INT32_C(-1596) * __LIBM_EXP2S, __INT32_C(1563) * __LIBM_EXP2S, __INT32_C(1169) * __LIBM_EXP2S, __INT32_C(-1006) * __LIBM_EXP2S, __INT32_C(-83) * __LIBM_EXP2S,
	__INT32_C(2362) * __LIBM_EXP2S, __INT32_C(-3521) * __LIBM_EXP2S, __INT32_C(-314) * __LIBM_EXP2S, __INT32_C(1814) * __LIBM_EXP2S, __INT32_C(-1751) * __LIBM_EXP2S, __INT32_C(305) * __LIBM_EXP2S, __INT32_C(1715) * __LIBM_EXP2S, __INT32_C(-3741) * __LIBM_EXP2S,
	__INT32_C(7847) * __LIBM_EXP2S, __INT32_C(1291) * __LIBM_EXP2S, __INT32_C(1206) * __LIBM_EXP2S, __INT32_C(36) * __LIBM_EXP2S, __INT32_C(1397) * __LIBM_EXP2S, __INT32_C(-1419) * __LIBM_EXP2S, __INT32_C(-1194) * __LIBM_EXP2S, __INT32_C(-2014) * __LIBM_EXP2S,
	__INT32_C(1742) * __LIBM_EXP2S, __INT32_C(-578) * __LIBM_EXP2S, __INT32_C(-207) * __LIBM_EXP2S, __INT32_C(875) * __LIBM_EXP2S, __INT32_C(1539) * __LIBM_EXP2S, __INT32_C(2826) * __LIBM_EXP2S, __INT32_C(-1165) * __LIBM_EXP2S, __INT32_C(-909) * __LIBM_EXP2S,
	__INT32_C(1849) * __LIBM_EXP2S, __INT32_C(927) * __LIBM_EXP2S, __INT32_C(2018) * __LIBM_EXP2S, __INT32_C(-981) * __LIBM_EXP2S, __INT32_C(1637) * __LIBM_EXP2S, __INT32_C(-463) * __LIBM_EXP2S, __INT32_C(905) * __LIBM_EXP2S, __INT32_C(6618) * __LIBM_EXP2S,
	__INT32_C(400) * __LIBM_EXP2S, __INT32_C(630) * __LIBM_EXP2S, __INT32_C(2614) * __LIBM_EXP2S, __INT32_C(900) * __LIBM_EXP2S, __INT32_C(2323) * __LIBM_EXP2S, __INT32_C(-1094) * __LIBM_EXP2S, __INT32_C(-1858) * __LIBM_EXP2S, __INT32_C(-212) * __LIBM_EXP2S,
	__INT32_C(-2069) * __LIBM_EXP2S, __INT32_C(747) * __LIBM_EXP2S, __INT32_C(1845) * __LIBM_EXP2S, __INT32_C(-1450) * __LIBM_EXP2S, __INT32_C(444) * __LIBM_EXP2S, __INT32_C(-213) * __LIBM_EXP2S, __INT32_C(-438) * __LIBM_EXP2S, __INT32_C(1158) * __LIBM_EXP2S,
	__INT32_C(4738) * __LIBM_EXP2S, __INT32_C(2497) * __LIBM_EXP2S, __INT32_C(-370) * __LIBM_EXP2S, __INT32_C(-2016) * __LIBM_EXP2S, __INT32_C(-518) * __LIBM_EXP2S, __INT32_C(-1160) * __LIBM_EXP2S, __INT32_C(-1510) * __LIBM_EXP2S, __INT32_C(123) * __LIBM_EXP2S
};
#undef __LIBM_EXP2S
__LIBM_LOCAL_DECLARE_END

__LIBM_LOCAL_FUNC(exp2) __ATTR_WUNUSED __ATTR_CONST __IEEE754_DOUBLE_TYPE__
(__LIBCCALL __ieee754_exp2)(__IEEE754_DOUBLE_TYPE__ __x) {
	/* Check for usual case.  */
	if (__ieee754_isless(__x, __LIBM_LOCAL_VALUE(exp2_himark))) {
		__int32_t __tval, __unsafe;
		__IEEE754_DOUBLE_TYPE__ __rx, __x22, __result;
		__IEEE754_DOUBLE_TYPE__ __ex2_u, __scale_u, __temp_x22;
		__uint32_t __exp;
		int __oldround;
		if (!__ieee754_isgreaterequal(__x, __LIBM_LOCAL_VALUE(exp2_lomark))) {
			if (__ieee754_isinf(__x)) /* e^-inf == 0, with no error.  */
				return 0;
			__libc_feraiseexcept(FE_UNDERFLOW);
			return 0;
		}
		if (__ieee754_fabs(__x) < (__CCAST(__IEEE754_DOUBLE_TYPE__)2.22044604925031308085e-16L /*DBL_EPSILON*/) / __IEEE754_DOUBLE_C(4.0))
			return __IEEE754_DOUBLE_C(1.0) + __x;
		__oldround = __libc_fegetround();
		if (__oldround != FE_TONEAREST)
			__libc_fesetround(FE_TONEAREST);
		__rx = __x + __LIBM_LOCAL_VALUE(THREEp42);
		__rx -= __LIBM_LOCAL_VALUE(THREEp42);
		__x -= __rx; /* Compute x=x1. */
		__tval = (__int32_t)(__rx * 512.0 + 256.0);
		__x -= __LIBM_LOCAL_VALUE(exp2_deltatable)[__tval & 511];
		__ex2_u = __LIBM_LOCAL_VALUE(exp2_accuratetable)[__tval & 511];
		__tval >>= 9;
		__unsafe = (__tval < 0 ? -__tval : __tval) >= -(-1021 /*DBL_MIN_EXP*/) - 56;
		__LIBM_GET_HIGH_WORD(__exp, __ex2_u);
		__exp = ((__exp & __UINT32_C(0x7ff00000)) +
		         ((__tval >> __unsafe) << IEEE754_DOUBLE_SHIFT)) |
		        (__exp & ~__UINT32_C(0x7ff00000));
		__LIBM_SET_HIGH_WORD(__ex2_u, __exp);
		__exp = __UINT32_C(0x3ff00000) + ((__tval - (__tval >> __unsafe)) << IEEE754_DOUBLE_SHIFT);
		__LIBM_INSERT_WORDS(__scale_u, __exp, 0);
		__x22 = (((__IEEE754_DOUBLE_C(0.0096181293647031180) * __x +
		           __IEEE754_DOUBLE_C(0.055504110254308625)) *
		          __x +
		          __IEEE754_DOUBLE_C(0.240226506959100583)) *
		         __x +
		         __IEEE754_DOUBLE_C(0.69314718055994495)) *
		        __ex2_u;
		__libm_math_opt_barrier(__x22, __temp_x22);
		__temp_x22 = __x22;
		if (__oldround != FE_TONEAREST)
			__libc_fesetround(__oldround);
		__result = __x22 * __x + __ex2_u;
		if (!__unsafe)
			return __result;
		return __result * __scale_u;
	}
	/* Return x, if x is a NaN or Inf; or overflow, otherwise.  */
	if (__ieee754_isinf(__x))
		return __HUGE_VAL;
	if (__ieee754_isnan(__x))
		return __x;
	__libc_feraiseexcept(FE_OVERFLOW);
	return __HUGE_VAL;
}
#endif /* __IEEE754_DOUBLE_TYPE__ */


#ifdef __IEEE854_LONG_DOUBLE_TYPE__
/* Compute 2^x.
   Copyright (C) 2012-2013 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

__LIBM_LOCAL_FUNC(exp2l) __ATTR_WUNUSED __ATTR_CONST __IEEE854_LONG_DOUBLE_TYPE__
(__LIBCCALL __ieee854_exp2l)(__IEEE854_LONG_DOUBLE_TYPE__ __x) {
	if (__ieee854_islessl(__x, __IEEE854_LONG_DOUBLE_C(16384.0))) {              /* LDBL_MAX_EXP */
		if (__ieee854_isgreaterequall(__x, __IEEE854_LONG_DOUBLE_C(-16446.0))) { /* LDBL_MIN_EXP - LDBL_MANT_DIG - 1 */
			int intx = (int)__x;
			__IEEE854_LONG_DOUBLE_TYPE__ __fractx = __x - intx;
			return __ieee854_scalbnl(__ieee854_expm1l(__IEEE854_LONG_DOUBLE_C(0.693147180559945309417232121458176568) /*M_LN2l*/ *
			                                          __fractx) +
			                         __IEEE854_LONG_DOUBLE_C(1.0),
			                         intx);
		}
		/* Underflow or exact zero.  */
		if (__ieee854_isinfl(__x))
			return __IEEE854_LONG_DOUBLE_C(0.0);
		__libc_feraiseexcept(FE_UNDERFLOW);
		return __IEEE854_LONG_DOUBLE_C(0.0);
	}
	/* Infinity, NaN or overflow.  */
	return __IEEE854_LONG_DOUBLE_C(1.18973149535723176502e+4932) /*LDBL_MAX*/ * __x;
}
#endif /* __IEEE854_LONG_DOUBLE_TYPE__ */

__DECL_END
#endif /* __CC__ */
#endif /* !__NO_FPU */

#endif /* !_LIBM_EXP2_H */
