/* Copyright (c) 2019-2024 Griefer@Work                                       *
 *                                                                            *
 * This software is provided 'as-is', without any express or implied          *
 * warranty. In no event will the authors be held liable for any damages      *
 * arising from the use of this software.                                     *
 *                                                                            *
 * Permission is granted to anyone to use this software for any purpose,      *
 * including commercial applications, and to alter it and redistribute it     *
 * freely, subject to the following restrictions:                             *
 *                                                                            *
 * 1. The origin of this software must not be misrepresented; you must not    *
 *    claim that you wrote the original software. If you use this software    *
 *    in a product, an acknowledgement (see the following) in the product     *
 *    documentation is required:                                              *
 *    Portions Copyright (c) 2019-2024 Griefer@Work                           *
 * 2. Altered source versions must be plainly marked as such, and must not be *
 *    misrepresented as being the original software.                          *
 * 3. This notice may not be removed or altered from any source distribution. *
 */
#ifdef __INTELLISENSE__
#ifndef __LIBEMU86_STATIC
#define LIBEMU86_WANT_PROTOTYPES
#define __LIBEMU86_STATIC 1
#define EMU86_EMULATE_CONFIG_CHECKERROR 1
#define LIBEMU86_CONFIG_WANT_16BIT 1
#define LIBEMU86_CONFIG_WANT_32BIT 1
#ifdef __x86_64__
#define LIBEMU86_CONFIG_WANT_64BIT 1
#endif /* __x86_64__ */
#endif /* !__LIBEMU86_STATIC */
#endif /* __INTELLISENSE__ */

#include <hybrid/compiler.h>

#include <hybrid/__atomic.h>
#include <hybrid/__rotate.h>
#include <hybrid/align.h>
#include <hybrid/host.h>
#include <hybrid/overflow.h>
#include <hybrid/typecore.h>
#include <hybrid/unaligned.h>
#include <hybrid/wordbits.h>

#include <i386-kos/asm/cpu-cpuid.h>
#include <i386-kos/asm/cpu-flags.h>
#include <i386-kos/asm/cpu-msr.h>
#include <i386-kos/asm/registers.h>
#include <i386-kos/asm/rtm.h>
#include <kos/except.h>
#include <kos/except/reason/inval.h>
#include <kos/kernel/types.h>
#include <kos/types.h>

#include <stdbool.h>
#include <stdint.h>

#include "eflags.h"
#include "emu86.h"
#include "helpers.h"

/* When `EMU86_EMULATE_CONFIG_DONT_USE_HYBRID_BIT' is defined, don't use the bit
 * functions from `<hybrid/__bit.h>'. */
#ifndef EMU86_EMULATE_CONFIG_DONT_USE_HYBRID_BIT
#include <hybrid/__bit.h>
#endif /* !EMU86_EMULATE_CONFIG_DONT_USE_HYBRID_BIT */

/* When `EMU86_EMULATE_CONFIG_DONT_USE_HYBRID_BYTESWAP' is defined, don't
 * use the BSWAP() functions  from `<hybrid/byteswap.h>'. This should  be
 * configured  when libemu86  is being  used to  emulate instructions not
 * supported by the  host, in which  case it couldn't  very well use  the
 * instructions it's trying to emulate. */
#if !defined(EMU86_EMULATE_CONFIG_DONT_USE_HYBRID_BYTESWAP) || 1
#include <hybrid/byteswap.h>
#endif /* !EMU86_EMULATE_CONFIG_DONT_USE_HYBRID_BYTESWAP || 1 */

#if defined(__x86_64__) || defined(__i386__)
#if !defined(EMU86_EMULATE_CONFIG_NO_INTRIN) || !(EMU86_EMULATE_CONFIG_NO_INTRIN + 0)
#include <asm/intrin.h>
#include <asm/intrin-fpu.h>
#endif /* !EMU86_EMULATE_CONFIG_NO_INTRIN */
#include <kos/kernel/cpu-state-helpers.h>
#include <kos/kernel/cpu-state.h>
#ifdef __KERNEL__
#include <sched/cred.h>
#endif /* __KERNEL__ */
#endif /* __x86_64__ || __i386__ */

#if LIBEMU86_CONFIG_WANT_64BIT
#include <int128.h>
#endif /* LIBEMU86_CONFIG_WANT_64BIT */

#if LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_UREG_TYPE u64
#define EMU86_SREG_TYPE s64
#else /* LIBEMU86_CONFIG_WANT_64BIT */
#define EMU86_UREG_TYPE u32
#define EMU86_SREG_TYPE s32
#endif /* !LIBEMU86_CONFIG_WANT_64BIT */


__DECL_BEGIN

/* Define to 1 to only emulate instructions that access memory. (used to implement VIO) */
#ifndef EMU86_EMULATE_CONFIG_ONLY_MEMORY
#define EMU86_EMULATE_CONFIG_ONLY_MEMORY 0
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */

/* Define to non-zero if user access checks should be performed. */
#ifndef EMU86_EMULATE_CONFIG_CHECKUSER
#if defined(__KERNEL__) || defined(__INTELLISENSE__)
#define EMU86_EMULATE_CONFIG_CHECKUSER 1
#else /* __KERNEL__ */
#define EMU86_EMULATE_CONFIG_CHECKUSER 0
#endif /* !__KERNEL__ */
#endif /* !EMU86_EMULATE_CONFIG_CHECKUSER */

/* Enable permission/usage/register checks  for instructions that  could
 * only ever result in  `EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION()'
 * Without this, such instructions may instead be handled as though they
 * were unknown.
 * WARNING: Some instructions may still not be checked unless the option
 *          `EMU86_EMULATE_CONFIG_ONLY_MEMORY'   is   defined   to   `0' */
#ifndef EMU86_EMULATE_CONFIG_CHECKERROR
#define EMU86_EMULATE_CONFIG_CHECKERROR 0
#endif /* !EMU86_EMULATE_CONFIG_CHECKERROR */

/* _only_ check for errors, and handle every instruction as being unsupported.
 * As an exception to this, support for certain instructions can be re-enabled
 * explicitly using other config options.
 * This is mainly intended to implement the #UD and #GP exception handlers in
 * order to */
#ifndef EMU86_EMULATE_CONFIG_ONLY_CHECKERROR
#define EMU86_EMULATE_CONFIG_ONLY_CHECKERROR 0
#endif /* !EMU86_EMULATE_CONFIG_ONLY_CHECKERROR */

/* Don't  necessarily include  basic instructions  that _only_  ever result in
 * an  unsupported  instruction  case in  the  only-check-error configuration.
 * This means that no additional code is generated for (e.g.) `add $imm8, %al'
 * when configured for `EMU86_EMULATE_CONFIG_ONLY_CHECKERROR'
 *
 * This option  is  ignored  when  `EMU86_EMULATE_CONFIG_ONLY_CHECKERROR == 0' */
#ifndef EMU86_EMULATE_CONFIG_ONLY_CHECKERROR_NO_BASIC
#define EMU86_EMULATE_CONFIG_ONLY_CHECKERROR_NO_BASIC 0
#endif /* !EMU86_EMULATE_CONFIG_ONLY_CHECKERROR_NO_BASIC */

/* Check that the `lock' prefix is only used by one of the white-listed lock-able instructions.
 * Enabling this option matches hardware behavior, but adds a small amount of overhead to every
 * instruction  that gets emulated. Disabling this option causes a lock prefix to be ignored by
 * instructions that don't actually support it */
#ifndef EMU86_EMULATE_CONFIG_CHECKLOCK
#define EMU86_EMULATE_CONFIG_CHECKLOCK 1
#endif /* !EMU86_EMULATE_CONFIG_CHECKLOCK */


/* Ignore the `lock' prefix in individual instructions, which will instead always operate
 * identical  to how they would when no `lock'-prefix would have been given. Note however
 * that this option does not  affect `EMU86_EMULATE_CONFIG_CHECKLOCK', meaning that  when
 * that  option is enabled, the emulator will  still ensure that only certain instruction
 * are allowed to make use of `lock' prefixes.
 *
 * Note  however that  this does _NOT_  affect instructions that  use the `lock'
 * prefix to alter their actual behavior in some way other than becoming atomic. */
#ifndef EMU86_EMULATE_CONFIG_IGNORE_LOCK
#define EMU86_EMULATE_CONFIG_IGNORE_LOCK 0
#endif /* !EMU86_EMULATE_CONFIG_IGNORE_LOCK */


/* Accept the  `lock' prefix  for various  instructions where  hardware  doesn't
 * normally support  this. However  support can  be augmented  by enabling  this
 * option as part of the #UD handler, alongside enabling emulation of individual
 * instruction  affected by this  (NOTE: By default, KOS  is _NOT_ configured to
 * do this) */

/* Accept the `lock' prefix for the rol/ror/rcl/rcr/shl/shr/sal/sar instructions. */
#ifndef EMU86_EMULATE_CONFIG_LOCK_SHIFT
#define EMU86_EMULATE_CONFIG_LOCK_SHIFT 0
#endif /* !EMU86_EMULATE_CONFIG_LOCK_SHIFT */

/* Accept the `lock' prefix for the shld/shrd instructions. */
#ifndef EMU86_EMULATE_CONFIG_LOCK_SHIFT2
#define EMU86_EMULATE_CONFIG_LOCK_SHIFT2 0
#endif /* !EMU86_EMULATE_CONFIG_LOCK_SHIFT2 */

/* Accept the `lock' prefix for the arpl instructions. */
#ifndef EMU86_EMULATE_CONFIG_LOCK_ARPL
#define EMU86_EMULATE_CONFIG_LOCK_ARPL 0
#endif /* !EMU86_EMULATE_CONFIG_LOCK_ARPL */


/* Allow use of `stac' and `clac' from user-space (by having the instruction
 * enable/disable the EFLAGS.AC bit as normal). Technically, this isn't done
 * on  real hardware, since  the instruction pair was  introduced to allow a
 * kernel to  temporarily enable/disable  the effects  of `CR4_SMAP'  (which
 * causes any kernel-access to pages marked with the U-bit to generate a #PF
 * so-long as `CR4.SMAP == true && EFLAGS.AC == true')
 * However, EFLAGS.AC also has meaning in user-space, where it can be used
 * to enable/disable AlignmentChecking  (hence the name  AC), causing  any
 * unaligned memory access to trigger a #AC fault.
 * Furthermore, regardless  of  this  these instructions  being  allowed  to  be
 * executed outside of  ring #0, ring  #3 already has  the ability to  set/clear
 * the  EFLAGS.AC  bit  freely,  as  that   bit  can  be  modified  by   `popf'.
 * s.a. the implementation of `__stac()' / `__clac()' in <i386-kos/asm/intrin.h> */
#ifndef EMU86_EMULATE_CONFIG_ALLOW_USER_STAC_CLAC
#define EMU86_EMULATE_CONFIG_ALLOW_USER_STAC_CLAC 1
#endif /* !EMU86_EMULATE_CONFIG_ALLOW_USER_STAC_CLAC */


/* Explicit feature-support configuration for individual instructions */
#ifndef EMU86_EMULATE_CONFIG_WANT_ADCX
#define EMU86_EMULATE_CONFIG_WANT_ADCX (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ADCX */
#ifndef EMU86_EMULATE_CONFIG_WANT_ADOX
#define EMU86_EMULATE_CONFIG_WANT_ADOX (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ADOX */
#ifndef EMU86_EMULATE_CONFIG_WANT_MULX
#define EMU86_EMULATE_CONFIG_WANT_MULX (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MULX */
#ifndef EMU86_EMULATE_CONFIG_WANT_ANDN
#define EMU86_EMULATE_CONFIG_WANT_ANDN (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ANDN */
#ifndef EMU86_EMULATE_CONFIG_WANT_ARITH
#define EMU86_EMULATE_CONFIG_WANT_ARITH (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ARITH */
#ifndef EMU86_EMULATE_CONFIG_WANT_ARITH2
#define EMU86_EMULATE_CONFIG_WANT_ARITH2 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ARITH2 */
#ifndef EMU86_EMULATE_CONFIG_WANT_DAA
#define EMU86_EMULATE_CONFIG_WANT_DAA (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_DAA */
#ifndef EMU86_EMULATE_CONFIG_WANT_DAS
#define EMU86_EMULATE_CONFIG_WANT_DAS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_DAS */
#ifndef EMU86_EMULATE_CONFIG_WANT_AAA
#define EMU86_EMULATE_CONFIG_WANT_AAA (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_AAA */
#ifndef EMU86_EMULATE_CONFIG_WANT_AAM
#define EMU86_EMULATE_CONFIG_WANT_AAM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_AAM */
#ifndef EMU86_EMULATE_CONFIG_WANT_AAD
#define EMU86_EMULATE_CONFIG_WANT_AAD (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_AAD */
#ifndef EMU86_EMULATE_CONFIG_WANT_TZCNT
#define EMU86_EMULATE_CONFIG_WANT_TZCNT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_TZCNT */
#ifndef EMU86_EMULATE_CONFIG_WANT_BSF
#define EMU86_EMULATE_CONFIG_WANT_BSF (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BSF */
#ifndef EMU86_EMULATE_CONFIG_WANT_LZCNT
#define EMU86_EMULATE_CONFIG_WANT_LZCNT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LZCNT */
#ifndef EMU86_EMULATE_CONFIG_WANT_BSR
#define EMU86_EMULATE_CONFIG_WANT_BSR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BSR */
#ifndef EMU86_EMULATE_CONFIG_WANT_BITTEST
#define EMU86_EMULATE_CONFIG_WANT_BITTEST (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BITTEST */
#ifndef EMU86_EMULATE_CONFIG_WANT_BLSR
#define EMU86_EMULATE_CONFIG_WANT_BLSR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BLSR */
#ifndef EMU86_EMULATE_CONFIG_WANT_BLSMSK
#define EMU86_EMULATE_CONFIG_WANT_BLSMSK (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BLSMSK */
#ifndef EMU86_EMULATE_CONFIG_WANT_BLSI
#define EMU86_EMULATE_CONFIG_WANT_BLSI (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BLSI */
#ifndef EMU86_EMULATE_CONFIG_WANT_BOUND
#define EMU86_EMULATE_CONFIG_WANT_BOUND (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BOUND */
#ifndef EMU86_EMULATE_CONFIG_WANT_BSWAP
#define EMU86_EMULATE_CONFIG_WANT_BSWAP (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BSWAP */
#ifndef EMU86_EMULATE_CONFIG_WANT_CALL
#define EMU86_EMULATE_CONFIG_WANT_CALL (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CALL */
#ifndef EMU86_EMULATE_CONFIG_WANT_CBW /* CBW, CWDE, CDQE */
#define EMU86_EMULATE_CONFIG_WANT_CBW (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CBW */
#ifndef EMU86_EMULATE_CONFIG_WANT_CWD /* CWD, CDQ, CQO */
#define EMU86_EMULATE_CONFIG_WANT_CWD (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CWD */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLTS
#define EMU86_EMULATE_CONFIG_WANT_CLTS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLTS */
#ifndef EMU86_EMULATE_CONFIG_WANT_SETCC
#define EMU86_EMULATE_CONFIG_WANT_SETCC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SETCC */
#ifndef EMU86_EMULATE_CONFIG_WANT_CMOVCC
#define EMU86_EMULATE_CONFIG_WANT_CMOVCC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CMOVCC */
#ifndef EMU86_EMULATE_CONFIG_WANT_CMPS
#define EMU86_EMULATE_CONFIG_WANT_CMPS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CMPS */
#ifndef EMU86_EMULATE_CONFIG_WANT_CMPXCHG
#define EMU86_EMULATE_CONFIG_WANT_CMPXCHG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CMPXCHG */
#ifndef EMU86_EMULATE_CONFIG_WANT_CMPXCHG8B
#define EMU86_EMULATE_CONFIG_WANT_CMPXCHG8B (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CMPXCHG8B */
#ifndef EMU86_EMULATE_CONFIG_WANT_CMPXCHG16B
#define EMU86_EMULATE_CONFIG_WANT_CMPXCHG16B (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CMPXCHG16B */
#ifndef EMU86_EMULATE_CONFIG_WANT_CPUID
#define EMU86_EMULATE_CONFIG_WANT_CPUID (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CPUID */
#ifndef EMU86_EMULATE_CONFIG_WANT_ENTER
#define EMU86_EMULATE_CONFIG_WANT_ENTER (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ENTER */
#ifndef EMU86_EMULATE_CONFIG_WANT_INVD
#define EMU86_EMULATE_CONFIG_WANT_INVD (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_INVD */
#ifndef EMU86_EMULATE_CONFIG_WANT_WBINVD
#define EMU86_EMULATE_CONFIG_WANT_WBINVD (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_WBINVD */
#ifndef EMU86_EMULATE_CONFIG_WANT_PREFETCHW
#define EMU86_EMULATE_CONFIG_WANT_PREFETCHW (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PREFETCHW */
#ifndef EMU86_EMULATE_CONFIG_WANT_PREFETCHNTA
#define EMU86_EMULATE_CONFIG_WANT_PREFETCHNTA (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PREFETCHNTA */
#ifndef EMU86_EMULATE_CONFIG_WANT_PREFETCH0
#define EMU86_EMULATE_CONFIG_WANT_PREFETCH0 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PREFETCH0 */
#ifndef EMU86_EMULATE_CONFIG_WANT_PREFETCH1
#define EMU86_EMULATE_CONFIG_WANT_PREFETCH1 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PREFETCH1 */
#ifndef EMU86_EMULATE_CONFIG_WANT_PREFETCH2
#define EMU86_EMULATE_CONFIG_WANT_PREFETCH2 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PREFETCH2 */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLDEMOTE
#define EMU86_EMULATE_CONFIG_WANT_CLDEMOTE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLDEMOTE */
#ifndef EMU86_EMULATE_CONFIG_WANT_HLT
#define EMU86_EMULATE_CONFIG_WANT_HLT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_HLT */
#ifndef EMU86_EMULATE_CONFIG_WANT_INC_REG
#define EMU86_EMULATE_CONFIG_WANT_INC_REG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_INC_REG */
#ifndef EMU86_EMULATE_CONFIG_WANT_DEC_REG
#define EMU86_EMULATE_CONFIG_WANT_DEC_REG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_DEC_REG */
#ifndef EMU86_EMULATE_CONFIG_WANT_INT1
#define EMU86_EMULATE_CONFIG_WANT_INT1 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_INT1 */
#ifndef EMU86_EMULATE_CONFIG_WANT_INT3
#define EMU86_EMULATE_CONFIG_WANT_INT3 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_INT3 */
#ifndef EMU86_EMULATE_CONFIG_WANT_INT
#define EMU86_EMULATE_CONFIG_WANT_INT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_INT */
#ifndef EMU86_EMULATE_CONFIG_WANT_INTO
#define EMU86_EMULATE_CONFIG_WANT_INTO (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_INTO */
#ifndef EMU86_EMULATE_CONFIG_WANT_IO
#define EMU86_EMULATE_CONFIG_WANT_IO (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_IO */
#ifndef EMU86_EMULATE_CONFIG_WANT_IRET
#define EMU86_EMULATE_CONFIG_WANT_IRET (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_IRET */
#ifndef EMU86_EMULATE_CONFIG_WANT_JCC_DISP8
#define EMU86_EMULATE_CONFIG_WANT_JCC_DISP8 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_JCC_DISP8 */
#ifndef EMU86_EMULATE_CONFIG_WANT_JCC_DISP32
#define EMU86_EMULATE_CONFIG_WANT_JCC_DISP32 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_JCC_DISP32 */
#ifndef EMU86_EMULATE_CONFIG_WANT_JCXZ
#define EMU86_EMULATE_CONFIG_WANT_JCXZ (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_JCXZ */
#ifndef EMU86_EMULATE_CONFIG_WANT_JMP_DISP8
#define EMU86_EMULATE_CONFIG_WANT_JMP_DISP8 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_JMP_DISP8 */
#ifndef EMU86_EMULATE_CONFIG_WANT_JMP_DISP32
#define EMU86_EMULATE_CONFIG_WANT_JMP_DISP32 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_JMP_DISP32 */
#ifndef EMU86_EMULATE_CONFIG_WANT_LCALL
#define EMU86_EMULATE_CONFIG_WANT_LCALL (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LCALL */
#ifndef EMU86_EMULATE_CONFIG_WANT_LEA
#define EMU86_EMULATE_CONFIG_WANT_LEA (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LEA */
#ifndef EMU86_EMULATE_CONFIG_WANT_LEAVE
#define EMU86_EMULATE_CONFIG_WANT_LEAVE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LEAVE */
#ifndef EMU86_EMULATE_CONFIG_WANT_LJMP
#define EMU86_EMULATE_CONFIG_WANT_LJMP (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LJMP */
#ifndef EMU86_EMULATE_CONFIG_WANT_LODS
#define EMU86_EMULATE_CONFIG_WANT_LODS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LODS */
#ifndef EMU86_EMULATE_CONFIG_WANT_LOOPNE
#define EMU86_EMULATE_CONFIG_WANT_LOOPNE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LOOPNE */
#ifndef EMU86_EMULATE_CONFIG_WANT_LOOPE
#define EMU86_EMULATE_CONFIG_WANT_LOOPE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LOOPE */
#ifndef EMU86_EMULATE_CONFIG_WANT_LOOP
#define EMU86_EMULATE_CONFIG_WANT_LOOP (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LOOP */
#ifndef EMU86_EMULATE_CONFIG_WANT_LRET
#define EMU86_EMULATE_CONFIG_WANT_LRET (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LRET */
#ifndef EMU86_EMULATE_CONFIG_WANT_LXS
#define EMU86_EMULATE_CONFIG_WANT_LXS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LXS */
#ifndef EMU86_EMULATE_CONFIG_WANT_INC_RM
#define EMU86_EMULATE_CONFIG_WANT_INC_RM EMU86_EMULATE_CONFIG_WANT_INC_REG
#endif /* !EMU86_EMULATE_CONFIG_WANT_INC_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_DEC_RM
#define EMU86_EMULATE_CONFIG_WANT_DEC_RM EMU86_EMULATE_CONFIG_WANT_DEC_REG
#endif /* !EMU86_EMULATE_CONFIG_WANT_DEC_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_CALL_RM
#define EMU86_EMULATE_CONFIG_WANT_CALL_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CALL_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_LCALL_RM
#define EMU86_EMULATE_CONFIG_WANT_LCALL_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LCALL_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_JMP_RM
#define EMU86_EMULATE_CONFIG_WANT_JMP_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_JMP_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_LJMP_RM
#define EMU86_EMULATE_CONFIG_WANT_LJMP_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LJMP_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_RM
#define EMU86_EMULATE_CONFIG_WANT_PUSH_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDFSBASE
#define EMU86_EMULATE_CONFIG_WANT_RDFSBASE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDFSBASE */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDGSBASE
#define EMU86_EMULATE_CONFIG_WANT_RDGSBASE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDGSBASE */
#ifndef EMU86_EMULATE_CONFIG_WANT_WRFSBASE
#define EMU86_EMULATE_CONFIG_WANT_WRFSBASE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_WRFSBASE */
#ifndef EMU86_EMULATE_CONFIG_WANT_WRGSBASE
#define EMU86_EMULATE_CONFIG_WANT_WRGSBASE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_WRGSBASE */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLWB
#define EMU86_EMULATE_CONFIG_WANT_CLWB (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLWB */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLFLUSH
#define EMU86_EMULATE_CONFIG_WANT_CLFLUSH (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLFLUSH */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLFLUSHOPT
#define EMU86_EMULATE_CONFIG_WANT_CLFLUSHOPT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLFLUSHOPT */
#ifndef EMU86_EMULATE_CONFIG_WANT_LFENCE
#define EMU86_EMULATE_CONFIG_WANT_LFENCE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LFENCE */
#ifndef EMU86_EMULATE_CONFIG_WANT_SFENCE
#define EMU86_EMULATE_CONFIG_WANT_SFENCE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SFENCE */
#ifndef EMU86_EMULATE_CONFIG_WANT_MFENCE
#define EMU86_EMULATE_CONFIG_WANT_MFENCE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MFENCE */
#ifndef EMU86_EMULATE_CONFIG_WANT_TPAUSE
#define EMU86_EMULATE_CONFIG_WANT_TPAUSE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_TPAUSE */
#ifndef EMU86_EMULATE_CONFIG_WANT_STMXCSR
#define EMU86_EMULATE_CONFIG_WANT_STMXCSR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_STMXCSR */
#ifndef EMU86_EMULATE_CONFIG_WANT_LDMXCSR
#define EMU86_EMULATE_CONFIG_WANT_LDMXCSR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LDMXCSR */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOV_RM
#define EMU86_EMULATE_CONFIG_WANT_MOV_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOV_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOVBE
#define EMU86_EMULATE_CONFIG_WANT_MOVBE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOVBE */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOV_CREG
#define EMU86_EMULATE_CONFIG_WANT_MOV_CREG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOVDIR64B
#define EMU86_EMULATE_CONFIG_WANT_MOVDIR64B (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOVDIR64B */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOV_DREG
#define EMU86_EMULATE_CONFIG_WANT_MOV_DREG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOV_IMM
#define EMU86_EMULATE_CONFIG_WANT_MOV_IMM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOV_IMM */
#ifndef EMU86_EMULATE_CONFIG_WANT_XABORT
#define EMU86_EMULATE_CONFIG_WANT_XABORT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XABORT */
#ifndef EMU86_EMULATE_CONFIG_WANT_XBEGIN
#define EMU86_EMULATE_CONFIG_WANT_XBEGIN (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XBEGIN */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOV_MOFFS
#define EMU86_EMULATE_CONFIG_WANT_MOV_MOFFS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOV_MOFFS */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOVNTI
#define EMU86_EMULATE_CONFIG_WANT_MOVNTI (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOVNTI */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOVS
#define EMU86_EMULATE_CONFIG_WANT_MOVS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOVS */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOV_SREG
#define EMU86_EMULATE_CONFIG_WANT_MOV_SREG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOV_SREG */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOVSX
#define EMU86_EMULATE_CONFIG_WANT_MOVSX (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOVSX */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOVSXD
#define EMU86_EMULATE_CONFIG_WANT_MOVSXD (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOVSXD */
#ifndef EMU86_EMULATE_CONFIG_WANT_MOVZX
#define EMU86_EMULATE_CONFIG_WANT_MOVZX (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MOVZX */
#ifndef EMU86_EMULATE_CONFIG_WANT_ARPL
#define EMU86_EMULATE_CONFIG_WANT_ARPL (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ARPL */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDMSR
#define EMU86_EMULATE_CONFIG_WANT_RDMSR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDMSR */
#ifndef EMU86_EMULATE_CONFIG_WANT_WRMSR
#define EMU86_EMULATE_CONFIG_WANT_WRMSR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_WRMSR */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDTSC
#define EMU86_EMULATE_CONFIG_WANT_RDTSC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDTSC */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDPMC
#define EMU86_EMULATE_CONFIG_WANT_RDPMC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDPMC */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDPID
#define EMU86_EMULATE_CONFIG_WANT_RDPID (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDPID */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDTSCP
#define EMU86_EMULATE_CONFIG_WANT_RDTSCP (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDTSCP */
#ifndef EMU86_EMULATE_CONFIG_WANT_SWAPGS
#define EMU86_EMULATE_CONFIG_WANT_SWAPGS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SWAPGS */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDRAND
#define EMU86_EMULATE_CONFIG_WANT_RDRAND (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDRAND */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDSEED
#define EMU86_EMULATE_CONFIG_WANT_RDSEED (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDSEED */
#ifndef EMU86_EMULATE_CONFIG_WANT_NOP_RM
#define EMU86_EMULATE_CONFIG_WANT_NOP_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_NOP_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_PEXT
#define EMU86_EMULATE_CONFIG_WANT_PEXT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PEXT */
#ifndef EMU86_EMULATE_CONFIG_WANT_PDEP
#define EMU86_EMULATE_CONFIG_WANT_PDEP (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PDEP */
#ifndef EMU86_EMULATE_CONFIG_WANT_BZHI
#define EMU86_EMULATE_CONFIG_WANT_BZHI (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BZHI */
#ifndef EMU86_EMULATE_CONFIG_WANT_SLDT
#define EMU86_EMULATE_CONFIG_WANT_SLDT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SLDT */
#ifndef EMU86_EMULATE_CONFIG_WANT_LLDT
#define EMU86_EMULATE_CONFIG_WANT_LLDT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LLDT */
#ifndef EMU86_EMULATE_CONFIG_WANT_STR
#define EMU86_EMULATE_CONFIG_WANT_STR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_STR */
#ifndef EMU86_EMULATE_CONFIG_WANT_LTR
#define EMU86_EMULATE_CONFIG_WANT_LTR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LTR */
#ifndef EMU86_EMULATE_CONFIG_WANT_SGDT
#define EMU86_EMULATE_CONFIG_WANT_SGDT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SGDT */
#ifndef EMU86_EMULATE_CONFIG_WANT_LGDT
#define EMU86_EMULATE_CONFIG_WANT_LGDT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LGDT */
#ifndef EMU86_EMULATE_CONFIG_WANT_XEND
#define EMU86_EMULATE_CONFIG_WANT_XEND (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XEND */
#ifndef EMU86_EMULATE_CONFIG_WANT_XTEST
#define EMU86_EMULATE_CONFIG_WANT_XTEST (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XTEST */
#ifndef EMU86_EMULATE_CONFIG_WANT_XGETBV
#define EMU86_EMULATE_CONFIG_WANT_XGETBV (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XGETBV */
#ifndef EMU86_EMULATE_CONFIG_WANT_XSETBV
#define EMU86_EMULATE_CONFIG_WANT_XSETBV (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XSETBV */
#ifndef EMU86_EMULATE_CONFIG_WANT_MCOMMIT
#define EMU86_EMULATE_CONFIG_WANT_MCOMMIT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MCOMMIT */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLZERO
#define EMU86_EMULATE_CONFIG_WANT_CLZERO (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLZERO */
#ifndef EMU86_EMULATE_CONFIG_WANT_SIDT
#define EMU86_EMULATE_CONFIG_WANT_SIDT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SIDT */
#ifndef EMU86_EMULATE_CONFIG_WANT_LIDT
#define EMU86_EMULATE_CONFIG_WANT_LIDT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LIDT */
#ifndef EMU86_EMULATE_CONFIG_WANT_VERR
#define EMU86_EMULATE_CONFIG_WANT_VERR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_VERR */
#ifndef EMU86_EMULATE_CONFIG_WANT_VERW
#define EMU86_EMULATE_CONFIG_WANT_VERW (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_VERW */
#ifndef EMU86_EMULATE_CONFIG_WANT_SMSW
#define EMU86_EMULATE_CONFIG_WANT_SMSW (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SMSW */
#ifndef EMU86_EMULATE_CONFIG_WANT_LMSW
#define EMU86_EMULATE_CONFIG_WANT_LMSW (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LMSW */
#ifndef EMU86_EMULATE_CONFIG_WANT_INVLPG
#define EMU86_EMULATE_CONFIG_WANT_INVLPG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_INVLPG */
#ifndef EMU86_EMULATE_CONFIG_WANT_LAR
#define EMU86_EMULATE_CONFIG_WANT_LAR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LAR */
#ifndef EMU86_EMULATE_CONFIG_WANT_LSL
#define EMU86_EMULATE_CONFIG_WANT_LSL (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LSL */
#ifndef EMU86_EMULATE_CONFIG_WANT_POPCNT
#define EMU86_EMULATE_CONFIG_WANT_POPCNT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POPCNT */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSHA
#define EMU86_EMULATE_CONFIG_WANT_PUSHA (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSHA */
#ifndef EMU86_EMULATE_CONFIG_WANT_POPA
#define EMU86_EMULATE_CONFIG_WANT_POPA (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POPA */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSHF
#define EMU86_EMULATE_CONFIG_WANT_PUSHF (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSHF */
#ifndef EMU86_EMULATE_CONFIG_WANT_POPF
#define EMU86_EMULATE_CONFIG_WANT_POPF (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POPF */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_IMM
#define EMU86_EMULATE_CONFIG_WANT_PUSH_IMM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_IMM */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_REG
#define EMU86_EMULATE_CONFIG_WANT_PUSH_REG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_REG */
#ifndef EMU86_EMULATE_CONFIG_WANT_POP_REG
#define EMU86_EMULATE_CONFIG_WANT_POP_REG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POP_REG */
#ifndef EMU86_EMULATE_CONFIG_WANT_POP_RM
#define EMU86_EMULATE_CONFIG_WANT_POP_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POP_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_ES
#define EMU86_EMULATE_CONFIG_WANT_PUSH_ES (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_ES */
#ifndef EMU86_EMULATE_CONFIG_WANT_POP_ES
#define EMU86_EMULATE_CONFIG_WANT_POP_ES (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POP_ES */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_CS
#define EMU86_EMULATE_CONFIG_WANT_PUSH_CS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_CS */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_SS
#define EMU86_EMULATE_CONFIG_WANT_PUSH_SS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_SS */
#ifndef EMU86_EMULATE_CONFIG_WANT_POP_SS
#define EMU86_EMULATE_CONFIG_WANT_POP_SS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POP_SS */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_DS
#define EMU86_EMULATE_CONFIG_WANT_PUSH_DS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_DS */
#ifndef EMU86_EMULATE_CONFIG_WANT_POP_DS
#define EMU86_EMULATE_CONFIG_WANT_POP_DS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POP_DS */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_FS
#define EMU86_EMULATE_CONFIG_WANT_PUSH_FS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_FS */
#ifndef EMU86_EMULATE_CONFIG_WANT_POP_FS
#define EMU86_EMULATE_CONFIG_WANT_POP_FS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POP_FS */
#ifndef EMU86_EMULATE_CONFIG_WANT_PUSH_GS
#define EMU86_EMULATE_CONFIG_WANT_PUSH_GS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PUSH_GS */
#ifndef EMU86_EMULATE_CONFIG_WANT_POP_GS
#define EMU86_EMULATE_CONFIG_WANT_POP_GS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_POP_GS */
#ifndef EMU86_EMULATE_CONFIG_WANT_RET
#define EMU86_EMULATE_CONFIG_WANT_RET (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RET */
#ifndef EMU86_EMULATE_CONFIG_WANT_RORX
#define EMU86_EMULATE_CONFIG_WANT_RORX (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RORX */
#ifndef EMU86_EMULATE_CONFIG_WANT_SAHF
#define EMU86_EMULATE_CONFIG_WANT_SAHF (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SAHF */
#ifndef EMU86_EMULATE_CONFIG_WANT_LAHF
#define EMU86_EMULATE_CONFIG_WANT_LAHF (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_LAHF */
#ifndef EMU86_EMULATE_CONFIG_WANT_SALC
#define EMU86_EMULATE_CONFIG_WANT_SALC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SALC */
#ifndef EMU86_EMULATE_CONFIG_WANT_SCAS
#define EMU86_EMULATE_CONFIG_WANT_SCAS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SCAS */
#ifndef EMU86_EMULATE_CONFIG_WANT_SHIFT
#define EMU86_EMULATE_CONFIG_WANT_SHIFT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SHIFT */
#ifndef EMU86_EMULATE_CONFIG_WANT_SHIFT2
#define EMU86_EMULATE_CONFIG_WANT_SHIFT2 (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SHIFT2 */
#ifndef EMU86_EMULATE_CONFIG_WANT_BEXTR
#define EMU86_EMULATE_CONFIG_WANT_BEXTR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_BEXTR */
#ifndef EMU86_EMULATE_CONFIG_WANT_SHIFTX
#define EMU86_EMULATE_CONFIG_WANT_SHIFTX (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SHIFTX */
#ifndef EMU86_EMULATE_CONFIG_WANT_STAC
#define EMU86_EMULATE_CONFIG_WANT_STAC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_STAC */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLAC
#define EMU86_EMULATE_CONFIG_WANT_CLAC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLAC */
#ifndef EMU86_EMULATE_CONFIG_WANT_ENCLS
#define EMU86_EMULATE_CONFIG_WANT_ENCLS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ENCLS */
#ifndef EMU86_EMULATE_CONFIG_WANT_ENCLU
#define EMU86_EMULATE_CONFIG_WANT_ENCLU (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_ENCLU */
#ifndef EMU86_EMULATE_CONFIG_WANT_MONITOR
#define EMU86_EMULATE_CONFIG_WANT_MONITOR (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MONITOR */
#ifndef EMU86_EMULATE_CONFIG_WANT_MWAIT
#define EMU86_EMULATE_CONFIG_WANT_MWAIT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_MWAIT */
#ifndef EMU86_EMULATE_CONFIG_WANT_STGI
#define EMU86_EMULATE_CONFIG_WANT_STGI (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_STGI */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLGI
#define EMU86_EMULATE_CONFIG_WANT_CLGI (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLGI */
#ifndef EMU86_EMULATE_CONFIG_WANT_CMC
#define EMU86_EMULATE_CONFIG_WANT_CMC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CMC */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLC
#define EMU86_EMULATE_CONFIG_WANT_CLC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLC */
#ifndef EMU86_EMULATE_CONFIG_WANT_STC
#define EMU86_EMULATE_CONFIG_WANT_STC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_STC */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLD
#define EMU86_EMULATE_CONFIG_WANT_CLD (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLD */
#ifndef EMU86_EMULATE_CONFIG_WANT_STD
#define EMU86_EMULATE_CONFIG_WANT_STD (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_STD */
#ifndef EMU86_EMULATE_CONFIG_WANT_CLI
#define EMU86_EMULATE_CONFIG_WANT_CLI (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_CLI */
#ifndef EMU86_EMULATE_CONFIG_WANT_STI
#define EMU86_EMULATE_CONFIG_WANT_STI (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_STI */
#ifndef EMU86_EMULATE_CONFIG_WANT_STOS
#define EMU86_EMULATE_CONFIG_WANT_STOS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_STOS */
#ifndef EMU86_EMULATE_CONFIG_WANT_SYSCALL
#define EMU86_EMULATE_CONFIG_WANT_SYSCALL (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SYSCALL */
#ifndef EMU86_EMULATE_CONFIG_WANT_SYSRET
#define EMU86_EMULATE_CONFIG_WANT_SYSRET (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SYSRET */
#ifndef EMU86_EMULATE_CONFIG_WANT_SYSENTER
#define EMU86_EMULATE_CONFIG_WANT_SYSENTER (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SYSENTER */
#ifndef EMU86_EMULATE_CONFIG_WANT_SYSEXIT
#define EMU86_EMULATE_CONFIG_WANT_SYSEXIT (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_SYSEXIT */
#ifndef EMU86_EMULATE_CONFIG_WANT_XADD
#define EMU86_EMULATE_CONFIG_WANT_XADD (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XADD */
#ifndef EMU86_EMULATE_CONFIG_WANT_XCHG_RM
#define EMU86_EMULATE_CONFIG_WANT_XCHG_RM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XCHG_RM */
#ifndef EMU86_EMULATE_CONFIG_WANT_NOP
#define EMU86_EMULATE_CONFIG_WANT_NOP (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_NOP */
#ifndef EMU86_EMULATE_CONFIG_WANT_PAUSE
#define EMU86_EMULATE_CONFIG_WANT_PAUSE (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_PAUSE */
#ifndef EMU86_EMULATE_CONFIG_WANT_XCHG_REG
#define EMU86_EMULATE_CONFIG_WANT_XCHG_REG (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XCHG_REG */
#ifndef EMU86_EMULATE_CONFIG_WANT_XLATB
#define EMU86_EMULATE_CONFIG_WANT_XLATB (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XLATB */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_BLCFILL
#define EMU86_EMULATE_CONFIG_WANT_XOP_BLCFILL (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_BLCFILL */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_BLSFILL
#define EMU86_EMULATE_CONFIG_WANT_XOP_BLSFILL (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_BLSFILL */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_BLCS
#define EMU86_EMULATE_CONFIG_WANT_XOP_BLCS (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_BLCS */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_TZMSK
#define EMU86_EMULATE_CONFIG_WANT_XOP_TZMSK (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_TZMSK */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_BLCIC
#define EMU86_EMULATE_CONFIG_WANT_XOP_BLCIC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_BLCIC */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_BLSIC
#define EMU86_EMULATE_CONFIG_WANT_XOP_BLSIC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_BLSIC */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_T1MSKC
#define EMU86_EMULATE_CONFIG_WANT_XOP_T1MSKC (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_T1MSKC */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_BLCMSK
#define EMU86_EMULATE_CONFIG_WANT_XOP_BLCMSK (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_BLCMSK */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_BLCI
#define EMU86_EMULATE_CONFIG_WANT_XOP_BLCI (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_BLCI */
#ifndef EMU86_EMULATE_CONFIG_WANT_XOP_BEXTR_IMM
#define EMU86_EMULATE_CONFIG_WANT_XOP_BEXTR_IMM (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_BEXTR_IMM */

/* Emulate `rdmsr' for the following MSRs (if possible):
 *  - IA32_FS_BASE            (#if EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED_FSGSBASE; uses `EMU86_GETFSBASE()')
 *  - IA32_GS_BASE            (#if EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED_FSGSBASE; uses `EMU86_GETGSBASE()')
 *  - IA32_TIME_STAMP_COUNTER (#ifdef EMU86_EMULATE_RDTSC_INDIRECT)
 *  - IA32_TSC_AUX            (#ifdef EMU86_EMULATE_RDPID) */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED
#define EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED */

/* When `EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED'
 * is also enabled, emulate FSGSBASE MSR accesses. */
#ifndef EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED_FSGSBASE
#define EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED_FSGSBASE 1
#endif /* !EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED_FSGSBASE */

/* Emulate `wrmsr' for the following MSRs (if possible):
 *  - IA32_FS_BASE            (#if EMU86_SETFSBASE; uses `EMU86_SETFSBASE()')
 *  - IA32_GS_BASE            (#if EMU86_SETGSBASE; uses `EMU86_SETGSBASE()') */
#ifndef EMU86_EMULATE_CONFIG_WANT_WRMSR_EMULATED /* emulate specific MSR accesses for user-space */
#define EMU86_EMULATE_CONFIG_WANT_WRMSR_EMULATED (!EMU86_EMULATE_CONFIG_ONLY_CHECKERROR)
#endif /* !EMU86_EMULATE_CONFIG_WANT_WRMSR_EMULATED */

#ifndef EMU86_EMULATE_CONFIG_WANT_XOP
#if (EMU86_EMULATE_CONFIG_WANT_XOP_BLCFILL || EMU86_EMULATE_CONFIG_WANT_XOP_BLSFILL || \
     EMU86_EMULATE_CONFIG_WANT_XOP_BLCS || EMU86_EMULATE_CONFIG_WANT_XOP_TZMSK ||      \
     EMU86_EMULATE_CONFIG_WANT_XOP_BLCIC || EMU86_EMULATE_CONFIG_WANT_XOP_BLSIC ||     \
     EMU86_EMULATE_CONFIG_WANT_XOP_T1MSKC || EMU86_EMULATE_CONFIG_WANT_XOP_BLCMSK ||   \
     EMU86_EMULATE_CONFIG_WANT_XOP_BLCI || EMU86_EMULATE_CONFIG_WANT_XOP_BEXTR_IMM)
#define EMU86_EMULATE_CONFIG_WANT_XOP 1
#else /* EMU86_EMULATE_CONFIG_WANT_XOP_* */
#define EMU86_EMULATE_CONFIG_WANT_XOP 0
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP_* */
#endif /* !EMU86_EMULATE_CONFIG_WANT_XOP */


/* Enable support for  rdfsbase/rdgsbase/wrfsbase/wrgsbase
 * in 32-bit and 16-bit modes (with the instructions using
 * the  32-bit   registers   in   both   execution   modi)
 * s.a.   `EMU86_SETFSBASE()'   and    `EMU86_SETGSBASE()' */
#ifndef EMU86_EMULATE_CONFIG_FSGSBASE_32BIT
#ifdef __KOS__
#define EMU86_EMULATE_CONFIG_FSGSBASE_32BIT 1
#else /* __KOS__ */
#define EMU86_EMULATE_CONFIG_FSGSBASE_32BIT 0
#endif /* !__KOS__ */
#endif /* !EMU86_EMULATE_CONFIG_FSGSBASE_32BIT */


/* Define to `1' to have `xchg' require a lock prefix in order to be considered atomic.
 * Real hardware always has  xchg behave as  atomic, regardless of  the presence of  an
 * optional  lock-prefix, however this also makes it impossible to implement non-atomic
 * xchg in VIO. (though VIO is default configured to still match the hardware standard,
 * meaning that this is only a convenience option that isn't actually ever altered) */
#ifndef EMU86_EMULATE_CONFIG_ATOMIC_XCHG_REQUIRES_LOCK
#define EMU86_EMULATE_CONFIG_ATOMIC_XCHG_REQUIRES_LOCK 0
#endif /* !EMU86_EMULATE_CONFIG_ATOMIC_XCHG_REQUIRES_LOCK */


/* Declaration visibility of `emu86_emulate()' */
#ifndef EMU86_EMULATE_DECL
#define EMU86_EMULATE_DECL __PRIVATE
#endif /* !EMU86_EMULATE_DECL */

/* Attributes applied to `emu86_emulate()' */
#ifndef EMU86_EMULATE_ATTR
#define EMU86_EMULATE_ATTR __ATTR_NONNULL((1))
#endif /* !EMU86_EMULATE_ATTR */

/* How and what `emu86_emulate()' should return */
#ifndef EMU86_EMULATE_RETURN
#define EMU86_EMULATE_RETURN()    return _state
#define EMU86_EMULATE_RETURN_TYPE struct icpustate *
#endif /* !EMU86_EMULATE_RETURN */

/* The NOTHROW annotation of `emu86_emulate()' */
#ifndef EMU86_EMULATE_NOTHROW
#define EMU86_EMULATE_NOTHROW /* nothing */
#endif /* !EMU86_EMULATE_NOTHROW */

/* The calling convention of `emu86_emulate()' */
#ifndef EMU86_EMULATE_CC
#define EMU86_EMULATE_CC LIBEMU86_CC
#endif /* !EMU86_EMULATE_CC */

/* An override for the name of `emu86_emulate()' */
#ifndef EMU86_EMULATE_NAME
#define EMU86_EMULATE_NAME emu86_emulate
#endif /* !EMU86_EMULATE_NAME */

/* Declaration visibility of helper functions */
#ifndef EMU86_EMULATE_HELPER_DECL
#define EMU86_EMULATE_HELPER_DECL __PRIVATE
#endif /* !EMU86_EMULATE_HELPER_DECL */

/* Transform the name of helper functions */
#ifndef EMU86_EMULATE_HELPER_NAME
#define EMU86_EMULATE_HELPER_NAME(x) x
#endif /* !EMU86_EMULATE_HELPER_NAME */

/* A list of arguments taken by `emu86_emulate()' */
#ifndef EMU86_EMULATE_ARGS
#define EMU86_EMULATE_ARGS struct icpustate *__restrict _state
#endif /* !EMU86_EMULATE_ARGS */

/* Hint at a spin-loop that may not necessarily break on its  own.
 * This is  placed  in  `do { read(); } while (!atomic_cmpxch());'
 * loops after the  every time that  the `atomic_cmpxch()'  fails.
 * For kernel-space, this should contain a call to  `task_serve()'
 * in  order to allow  the calling thread to  be terminated in the
 * event that the backing memory is set-up to return random values
 * for every invocation. */
#ifndef EMU86_EMULATE_LOOPHINT
#ifdef __KERNEL__
#include <sched/rpc.h>
#include <sched/task.h>
#define EMU86_EMULATE_LOOPHINT() (task_serve(), task_yield())
#else /* __KERNEL__ */
#include <hybrid/sched/__yield.h>
#define EMU86_EMULATE_LOOPHINT() __hybrid_yield()
#endif /* !__KERNEL__ */
#endif /* !EMU86_EMULATE_LOOPHINT */

/* Arguments taken by helper functions */
#ifndef EMU86_EMULATE_HELPER_ARGS
#define EMU86_EMULATE_HELPER_ATTR   __ATTR_NONNULL((1))
#define EMU86_EMULATE_HELPER_ARGS   struct icpustate *__restrict _state /* Arguments for helper functions */
#define EMU86_EMULATE_HELPER_PARAM  _state                              /* Parameters passed to helper functions */
#endif /* !EMU86_EMULATE_HELPER_ARGS */
#ifndef EMU86_EMULATE_HELPER_ARGS_
#define EMU86_EMULATE_HELPER_ARGS_  EMU86_EMULATE_HELPER_ARGS,
#define EMU86_EMULATE_HELPER_PARAM_ EMU86_EMULATE_HELPER_PARAM,
#endif /* !EMU86_EMULATE_HELPER_ARGS_ */

/* Additional attributes for annotating helper functions */
#ifndef EMU86_EMULATE_HELPER_ATTR
#define EMU86_EMULATE_HELPER_ATTR /* nothing */
#endif /* !EMU86_EMULATE_HELPER_ATTR */

/* Declare additional variables used during emulation */
#ifndef EMU86_EMULATE_VARIABLES
#define EMU86_EMULATE_VARIABLES /* nothing */
#endif /* !EMU86_EMULATE_VARIABLES */

/* Additional code to insert at the start of emu86_emulate() */
/* #define EMU86_EMULATE_SETUP() ... */

/* Additional code to insert at the end of emu86_emulate() */
/* #define EMU86_EMULATE_FOOTER() ... */

/* May be used to add a TRY ... EXCEPT block around emulation */
/* #define EMU86_EMULATE_TRY           TRY */
/* #define EMU86_EMULATE_EXCEPT        EXCEPT { ... } */
/* #define EMU86_EMULATE_TRY_DECODE    TRY */
/* #define EMU86_EMULATE_TRY_SWITCH    EXCEPT { ... } */
/* #define EMU86_EMULATE_EXCEPT_DECODE TRY */
/* #define EMU86_EMULATE_EXCEPT_SWITCH EXCEPT { ... } */

/* Return handlers for specific instructions */
#ifdef EMU86_EMULATE_RETURN_AFTER_INT
#ifndef EMU86_EMULATE_NO_IMPLICIT_RETURN_AFTER_INT
#ifndef EMU86_EMULATE_RETURN_AFTER_INT1
#define EMU86_EMULATE_RETURN_AFTER_INT1() \
	EMU86_EMULATE_RETURN_AFTER_INT(0x01) /* #DB */
#endif /* !EMU86_EMULATE_RETURN_AFTER_INT1 */
#ifndef EMU86_EMULATE_RETURN_AFTER_INT3
#define EMU86_EMULATE_RETURN_AFTER_INT3() \
	EMU86_EMULATE_RETURN_AFTER_INT(0x03) /* #BP */
#endif /* !EMU86_EMULATE_RETURN_AFTER_INT3 */
#ifndef EMU86_EMULATE_RETURN_AFTER_INTO
#define EMU86_EMULATE_RETURN_AFTER_INTO() \
	EMU86_EMULATE_RETURN_AFTER_INT(0x04) /* #OF */
#endif /* !EMU86_EMULATE_RETURN_AFTER_INTO */
#ifndef EMU86_EMULATE_THROW_BOUNDERR
#define EMU86_EMULATE_THROW_BOUNDERR(bound_idx, bound_min, bound_max) \
	EMU86_EMULATE_RETURN_AFTER_INT(0x05) /* #BR */
#endif /* !EMU86_EMULATE_THROW_BOUNDERR */
#ifndef EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION
#define EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION() \
	EMU86_EMULATE_RETURN_AFTER_INT(0x06) /* #UD */
#endif /* !EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION */
#ifndef EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER
#define EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER(how, regno, offset, regval, regval2) \
	EMU86_EMULATE_RETURN_AFTER_INT((how) == E_ILLEGAL_INSTRUCTION_REGISTER_RDINV ||           \
	                               (how) == E_ILLEGAL_INSTRUCTION_REGISTER_WRINV              \
	                               ? 0x06 /* #UD */                                           \
	                               : 0x0d /* #GP */)
#endif /* !EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER */
#ifndef EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER
#endif /* !EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER */
#ifndef EMU86_EMULATE_THROW_SEGFAULT_UNALIGNED
#define EMU86_EMULATE_THROW_SEGFAULT_UNALIGNED(addr, context, req_alignment) \
	EMU86_EMULATE_RETURN_AFTER_INT(0x0d) /* #GP */
#endif /* !EMU86_EMULATE_THROW_SEGFAULT_UNALIGNED */
#endif /* !EMU86_EMULATE_NO_IMPLICIT_RETURN_AFTER_INT */
#endif /* EMU86_EMULATE_RETURN_AFTER_INT */



/* Helper macros for defining error handling macros */
#if LIBEMU86_CONFIG_WANT_64BIT && LIBEMU86_CONFIG_WANT_32BIT && LIBEMU86_CONFIG_WANT_16BIT
#define _EMU86_GETMODRM_REGNOSIZE()   (EMU86_F_IS64(op_flags) ? X86_REGISTER_SIZEMASK_8BYTE : EMU86_F_IS32(op_flags) ? X86_REGISTER_SIZEMASK_4BYTE : X86_REGISTER_SIZEMASK_2BYTE)
#define _EMU86_GETMODRM_RM_GPREGVAL() (EMU86_F_IS64(op_flags) ? MODRM_GETRMREGQ() : EMU86_F_IS32(op_flags) ? (u64)MODRM_GETRMREGL() : (u64)MODRM_GETRMREGW())
#elif LIBEMU86_CONFIG_WANT_64BIT && LIBEMU86_CONFIG_WANT_32BIT
#define _EMU86_GETMODRM_REGNOSIZE()   (EMU86_F_IS64(op_flags) ? X86_REGISTER_SIZEMASK_8BYTE : X86_REGISTER_SIZEMASK_4BYTE)
#define _EMU86_GETMODRM_RM_GPREGVAL() (EMU86_F_IS64(op_flags) ? MODRM_GETRMREGQ() : (u64)MODRM_GETRMREGL())
#elif LIBEMU86_CONFIG_WANT_64BIT && LIBEMU86_CONFIG_WANT_16BIT
#define _EMU86_GETMODRM_REGNOSIZE()   (EMU86_F_IS64(op_flags) ? X86_REGISTER_SIZEMASK_8BYTE : X86_REGISTER_SIZEMASK_2BYTE)
#define _EMU86_GETMODRM_RM_GPREGVAL() (EMU86_F_IS64(op_flags) ? MODRM_GETRMREGQ() : (u64)MODRM_GETRMREGW())
#elif LIBEMU86_CONFIG_WANT_32BIT && LIBEMU86_CONFIG_WANT_16BIT
#define _EMU86_GETMODRM_REGNOSIZE()   (EMU86_F_IS32(op_flags) ? X86_REGISTER_SIZEMASK_4BYTE : X86_REGISTER_SIZEMASK_2BYTE)
#define _EMU86_GETMODRM_RM_GPREGVAL() (EMU86_F_IS32(op_flags) ? MODRM_GETRMREGL() : (u32)MODRM_GETRMREGW())
#elif LIBEMU86_CONFIG_WANT_64BIT
#define _EMU86_GETMODRM_REGNOSIZE()   X86_REGISTER_SIZEMASK_8BYTE
#define _EMU86_GETMODRM_RM_GPREGVAL() MODRM_GETRMREGQ()
#elif LIBEMU86_CONFIG_WANT_32BIT
#define _EMU86_GETMODRM_REGNOSIZE()   X86_REGISTER_SIZEMASK_4BYTE
#define _EMU86_GETMODRM_RM_GPREGVAL() MODRM_GETRMREGL()
#elif LIBEMU86_CONFIG_WANT_16BIT
#define _EMU86_GETMODRM_REGNOSIZE()   X86_REGISTER_SIZEMASK_2BYTE
#define _EMU86_GETMODRM_RM_GPREGVAL() MODRM_GETRMREGW()
#else /* ... */
#error "Invalid configuration"
#endif /* !... */

/* Returns the name of a general-purpose register described by `MODRM.RM' */
#define _EMU86_GETMODRM_RM_GPREGNO() (_EMU86_GETMODRM_REGNOSIZE() | X86_REGISTER_GENERAL_PURPOSE | modrm.mi_rm)

/* Returns the memory address pointed to by a general-purpose register described by `MODRM.RM' */
#define _EMU86_GETMODRM_RM_MEMADDR() ((uintptr_t)MODRM_MEMADDR())

/* Returns the opcode for the current instruction */
#define _EMU86_GETOPCODE() EMU86_OPCODE()

/* Returns the opcode for the current instruction, while also taking `MODRM.REG' into account */
#ifdef E_ILLEGAL_INSTRUCTION_X86_OPCODE
#define _EMU86_GETOPCODE_RMREG()  E_ILLEGAL_INSTRUCTION_X86_OPCODE(EMU86_OPCODE(), modrm.mi_reg)
#else /* E_ILLEGAL_INSTRUCTION_X86_OPCODE */
#define _EMU86_GETOPCODE_RMREG()  EMU86_OPCODE()
#endif /* !E_ILLEGAL_INSTRUCTION_X86_OPCODE */

/* Return  in  the  event  of  an  unrecognized  instruction.
 * HINT: Additionally, you may use the following expressions:
 *     - REAL_START_IP()               Evaluates to the starting IP of the faulting
 *                                     instruction   (i.e.   the   fault   address)
 *     - op_flags                      Set of `EMU86_F_*'
 *     - _EMU86_GETOPCODE()            The absolute opcode of the faulting instruction
 *     - _EMU86_GETOPCODE_RMREG()      The absolute opcode of the faulting instruction (with accounting for `MODRM.REG')
 *     - modrm                         The decoded modr/m suffix (only for *_RMREG handlers)
 *     - _EMU86_GETMODRM_REGNOSIZE()   The size addend for `_EMU86_GETMODRM_RM_GPREGNO()' (one of `X86_REGISTER_SIZEMASK_(2|4|8)BYTE')
 *     - _EMU86_GETMODRM_RM_GPREGNO()  The name of the GP register pointed to by MODRM.RM (one of `X86_REGISTER_*')
 *     - _EMU86_GETMODRM_RM_GPREGVAL() The value of the exec-mode-sized GP register pointed to by MODRM.RM
 */
#ifndef EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION
#define EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION()           THROW(E_ILLEGAL_INSTRUCTION_BAD_OPCODE, _EMU86_GETOPCODE(), op_flags)
#define EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION_RMREG()     THROW(E_ILLEGAL_INSTRUCTION_BAD_OPCODE, _EMU86_GETOPCODE_RMREG(), op_flags)
#define EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION()        THROW(E_ILLEGAL_INSTRUCTION_PRIVILEGED_OPCODE, _EMU86_GETOPCODE(), op_flags)
#define EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION_RMREG()  THROW(E_ILLEGAL_INSTRUCTION_PRIVILEGED_OPCODE, _EMU86_GETOPCODE_RMREG(), op_flags)
#define EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM()         THROW(E_ILLEGAL_INSTRUCTION_BAD_OPERAND, _EMU86_GETOPCODE(), op_flags, E_ILLEGAL_INSTRUCTION_BAD_OPERAND_UNEXPECTED_REGISTER, _EMU86_GETMODRM_RM_GPREGNO(), 0, _EMU86_GETMODRM_RM_GPREGVAL())
#define EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG()   THROW(E_ILLEGAL_INSTRUCTION_BAD_OPERAND, _EMU86_GETOPCODE_RMREG(), op_flags, E_ILLEGAL_INSTRUCTION_BAD_OPERAND_UNEXPECTED_REGISTER, _EMU86_GETMODRM_RM_GPREGNO(), 0, _EMU86_GETMODRM_RM_GPREGVAL())
#define EMU86_EMULATE_RETURN_EXPECTED_REGISTER_MODRM()       THROW(E_ILLEGAL_INSTRUCTION_BAD_OPERAND, _EMU86_GETOPCODE(), op_flags, E_ILLEGAL_INSTRUCTION_BAD_OPERAND_UNEXPECTED_MEMORY, 0, _EMU86_GETMODRM_RM_MEMADDR())
#define EMU86_EMULATE_RETURN_EXPECTED_REGISTER_MODRM_RMREG() THROW(E_ILLEGAL_INSTRUCTION_BAD_OPERAND, _EMU86_GETOPCODE_RMREG(), op_flags, E_ILLEGAL_INSTRUCTION_BAD_OPERAND_UNEXPECTED_MEMORY, 0, _EMU86_GETMODRM_RM_MEMADDR())
#ifdef E_ILLEGAL_INSTRUCTION_X86_BAD_PREFIX
#define EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX()       THROW(E_ILLEGAL_INSTRUCTION_X86_BAD_PREFIX, _EMU86_GETOPCODE(), op_flags)
#define EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX_RMREG() THROW(E_ILLEGAL_INSTRUCTION_X86_BAD_PREFIX, _EMU86_GETOPCODE_RMREG(), op_flags)
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_LOCK()         THROW(E_ILLEGAL_INSTRUCTION_X86_BAD_PREFIX, _EMU86_GETOPCODE(), op_flags) */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_LOCK_RMREG()   THROW(E_ILLEGAL_INSTRUCTION_X86_BAD_PREFIX, _EMU86_GETOPCODE_RMREG(), op_flags) */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL()       THROW(E_ILLEGAL_INSTRUCTION_X86_BAD_PREFIX, _EMU86_GETOPCODE(), op_flags) */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL_RMREG() THROW(E_ILLEGAL_INSTRUCTION_X86_BAD_PREFIX, _EMU86_GETOPCODE_RMREG(), op_flags) */
#endif /* E_ILLEGAL_INSTRUCTION_X86_BAD_PREFIX */
#define EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION()       THROW(E_ILLEGAL_INSTRUCTION_UNSUPPORTED_OPCODE, _EMU86_GETOPCODE(), op_flags)
#define EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG() THROW(E_ILLEGAL_INSTRUCTION_UNSUPPORTED_OPCODE, _EMU86_GETOPCODE_RMREG(), op_flags)
#define EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER(how, regno, offset, regval, regval2) \
	THROW(E_ILLEGAL_INSTRUCTION_REGISTER, _EMU86_GETOPCODE(), op_flags, how, regno, offset, regval, regval2)
#define EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER_RMREG(how, regno, offset, regval, regval2) \
	THROW(E_ILLEGAL_INSTRUCTION_REGISTER, _EMU86_GETOPCODE_RMREG(), op_flags, how, regno, offset, regval, regval2)
#if LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_EMULATE_THROW_SEGFAULT_UNMAPPED_NONCANON(addr) \
	THROW(E_SEGFAULT_UNMAPPED, addr, E_SEGFAULT_CONTEXT_NONCANON)
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#endif /* !EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION */

/* #define EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION()       ... */
/* #define EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION_RMREG() ... */

/* Throw  an exception indicative  of a privileged instruction
 * Only used when `EMU86_EMULATE_CONFIG_CHECKUSER' is enabled. */
/* #define EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION()       ... */
/* #define EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION_RMREG() ... */

/* Expected a memory operand modr/m */
/* #define EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM()       ... */
/* #define EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG() ... */

/* Expected a register operand for modr/m */
/* #define EMU86_EMULATE_RETURN_EXPECTED_REGISTER_MODRM()       ... */
/* #define EMU86_EMULATE_RETURN_EXPECTED_REGISTER_MODRM_RMREG() ... */

/* Unexpected instruction prefix */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX()       ... */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX_RMREG() ... */

/* Unexpected LOCK prefix
 * defaults to `EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX()' */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_LOCK()       ... */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_LOCK_RMREG() ... */

/* Unexpected  value  for   `op_flags & EMU86_F_VEX_LL_M'
 * defaults to `EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX()' */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL()       ... */
/* #define EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL_RMREG() ... */

/* Instruction isn't supported due to some missing hardware feature.
 * Used by the implementations of `cmpxchg8b' and `cmpxchg16b'  when
 * no way of performing the operation was configured. */
/* #define EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION()       ... */
/* #define EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG() ... */





/* An optional, special return expression to be evaluated following
 * an  `sti' instruction that  turned on EFLAGS.IF  (may be used to
 * implement special handling  in order to  delay interrupt  checks
 * until after the next instruction)
 * NOTE: This expression mustn't return normally! (but should
 *       normally  contain a `THROW()' or `return' statement) */
/* #define EMU86_EMULATE_RETURN_AFTER_STI() ... */

/* Same as `EMU86_EMULATE_RETURN_AFTER_STI()', but used for vm86 instead. */
/* #define EMU86_EMULATE_RETURN_AFTER_STI_VM86() ... */



/* An optional, special return expression to be evaluated following
 * an   `hlt'  instruction  (with   #IF=0  or  #IF=1  respectively)
 * When not defined, `hlt' will simply return normally.
 * See also: `EMU86_EMULATE_RETURN_AFTER_HLT_VM86()' when vm86 is supported. */
/* #define EMU86_EMULATE_RETURN_AFTER_HLT_IF0() ... */
/* #define EMU86_EMULATE_RETURN_AFTER_HLT_IF1() ... */



/* Define this (and don't enable `EMU86_EMULATE_CONFIG_ONLY_MEMORY')
 * to  enable  emulation  of   `int',  `int3',  `into'  and   `int1' */
/* #define EMU86_EMULATE_RETURN_AFTER_INT(intno) ... */



/* Return handlers for `syscall' and `sysenter'
 * These hooks should be defined within the #UD handler in
 * kernel-space to emulate  the expected  behavior is  the
 * instruction had actually been supported. */
/* #define EMU86_EMULATE_RETURN_AFTER_SYSCALL() ... */
/* #define EMU86_EMULATE_RETURN_AFTER_SYSENTER() ... */



/* Define to implement custom handling of RTM instructions. */
/* #define EMU86_EMULATE_RETURN_AFTER_XBEGIN(fallback_ip) ... */
/* #define EMU86_EMULATE_RETURN_AFTER_XABORT(code)        ... // Only called when `EMU86_EMULATE_XTEST() != 0' */
/* #define EMU86_EMULATE_RETURN_AFTER_XEND()              ... // Only called when `EMU86_EMULATE_XTEST() != 0' */
/* #define EMU86_EMULATE_XTEST()                          0/1 */
/* #define EMU86_EMULATE_XTEST_IS_ONE                     1 // EMU86_EMULATE_XTEST() is always 1 */
/* #define EMU86_EMULATE_XTEST_IS_ZERO                    1 // EMU86_EMULATE_XTEST() is always 0 (default) */

/* Default-configure `EMU86_EMULATE_XTEST()' */
#ifdef EMU86_EMULATE_XTEST_IS_ONE
#undef EMU86_EMULATE_XTEST
#define EMU86_EMULATE_XTEST() 1
#elif defined(EMU86_EMULATE_XTEST_IS_ZERO)
#undef EMU86_EMULATE_XTEST
#define EMU86_EMULATE_XTEST() 0
#elif !defined(EMU86_EMULATE_XTEST)
#define EMU86_EMULATE_XTEST() 0
#define EMU86_EMULATE_XTEST_IS_ZERO 1
#endif /* ... */



/* Do what  should be  done in  order to  handle
 * the `bound' instruction failing on 16-/32-bit */
#ifndef EMU86_EMULATE_THROW_BOUNDERR
#define EMU86_EMULATE_THROW_BOUNDERR(bound_idx, bound_min, bound_max) \
	THROW(E_INDEX_ERROR_OUT_OF_BOUNDS, bound_idx, bound_min, bound_max)
#endif /* !EMU86_EMULATE_THROW_BOUNDERR */

/* Handle a misaligned memory operand. */
#ifndef EMU86_EMULATE_THROW_SEGFAULT_UNALIGNED
#define EMU86_EMULATE_THROW_SEGFAULT_UNALIGNED(addr, context, req_alignment) \
	THROW(E_SEGFAULT_UNALIGNED, addr, context, req_alignment)
#endif /* !EMU86_EMULATE_THROW_SEGFAULT_UNALIGNED */

/* Throw an exception related to attempting to access an illegal/protected register.
 * NOTE: When not defined, such cases are handled as follows:
 *   how == E_ILLEGAL_INSTRUCTION_REGISTER_RDINV:   return_unsupported_instruction
 *   how == E_ILLEGAL_INSTRUCTION_REGISTER_WRINV:   return_unsupported_instruction
 *   how == E_ILLEGAL_INSTRUCTION_REGISTER_WRBAD:   return_unsupported_instruction
 *   how == E_ILLEGAL_INSTRUCTION_REGISTER_WRNPSEG: return_unsupported_instruction
 *   how == E_ILLEGAL_INSTRUCTION_REGISTER_RDPRV:   return_privileged_instruction  (only with `EMU86_EMULATE_CONFIG_CHECKUSER')
 *   how == E_ILLEGAL_INSTRUCTION_REGISTER_WRPRV:   return_privileged_instruction  (only with `EMU86_EMULATE_CONFIG_CHECKUSER')
 * @param: how:     How was the register accessed (One of `E_ILLEGAL_INSTRUCTION_REGISTER_*')
 * @param: regno:   The accessed register index (one of `X86_REGISTER_*')
 * @param: offset:  An offset applied to the register (in case of an illegal lcall/ljmp; 0 otherwise)
 * @param: regval:  The value that was attempted to be assigned (only for write operations)
 * @param: regval2: High 32-bit of the written value in case of an MSR register write. */
/* #define EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER(how, regno, offset, regval, regval2)       ... */
/* #define EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER_RMREG(how, regno, offset, regval, regval2) ... */


/* Throw an  exception as  the result  of `addr'  being non-canonical  in 64-bit  mode.
 * Note that for normal memory accesses, an address is not checked for being canonical.
 * However, certain instructions (such as wr(fs|gs)base) have an explicit except-branch
 * that gets called whenever a non-canonical address is passed. This exception  handler
 * will then be used these (few) cases.
 * s.a. `EMU86_VALIDATE_CANONICAL()' */
//#if LIBEMU86_CONFIG_WANT_64BIT
/* #define EMU86_EMULATE_THROW_SEGFAULT_UNMAPPED_NONCANON(addr) \
	THROW(E_SEGFAULT_UNMAPPED, addr, E_SEGFAULT_CONTEXT_NONCANON) */
//#endif /* LIBEMU86_CONFIG_WANT_64BIT */



/* Allow E_DIVIDE_BY_ZERO to  be rethrown natively,  rather than  having
 * to be handled by use of `EMU86_EMULATE_THROW_DIVIDE_ERROR[B|W|L|Q]()'
 * In case the aforementioned macros only ever throw the same exception,
 * this can get rid of unnecessary bloat. */
#ifndef EMU86_EMULATE_THROW_DIVIDE_BY_ZERO_ALLOW_RETHROW
#define EMU86_EMULATE_THROW_DIVIDE_BY_ZERO_ALLOW_RETHROW 0
#endif /* !EMU86_EMULATE_THROW_DIVIDE_BY_ZERO_ALLOW_RETHROW */

#ifndef EMU86_EMULATE_THROW_DIVIDE_BY_ZEROB
#ifndef EMU86_EMULATE_THROW_DIVIDE_ERROR
#define EMU86_EMULATE_THROW_DIVIDE_ERROR() THROW(E_DIVIDE_BY_ZERO)
#endif /* !EMU86_EMULATE_THROW_DIVIDE_ERROR */
#ifndef EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZERO
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZERO() EMU86_EMULATE_THROW_DIVIDE_ERROR()
#endif /* !EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZERO */
#define EMU86_EMULATE_THROW_DIVIDE_BY_ZEROB(/*u16*/ lhs) EMU86_EMULATE_THROW_DIVIDE_ERROR()
#define EMU86_EMULATE_THROW_DIVIDE_BY_ZEROW(/*u32*/ lhs) EMU86_EMULATE_THROW_DIVIDE_ERROR()
#define EMU86_EMULATE_THROW_DIVIDE_BY_ZEROL(/*u64*/ lhs) EMU86_EMULATE_THROW_DIVIDE_ERROR()
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZEROB(/*s16*/ lhs) EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZERO()
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZEROW(/*s32*/ lhs) EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZERO()
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZEROL(/*s64*/ lhs) EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZERO()
#if LIBEMU86_CONFIG_WANT_64BIT && defined(__UINT128_TYPE__)
#define EMU86_EMULATE_THROW_DIVIDE_BY_ZEROQ(/*uint128_t*/ lhs) EMU86_EMULATE_THROW_DIVIDE_ERROR()
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZEROQ(/*int128_t*/ lhs) EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZERO()
#endif /* LIBEMU86_CONFIG_WANT_64BIT && __UINT128_TYPE__ */
#endif /* !EMU86_EMULATE_THROW_DIVIDE_ERROR */
#ifndef EMU86_EMULATE_THROW_DIVIDE_OVERFLOWB
#define EMU86_EMULATE_THROW_DIVIDE_OVERFLOWB(/*u16*/ lhs, /*u8*/ rhs, /*u16*/ exact_result) EMU86_EMULATE_THROW_DIVIDE_BY_ZEROB(lhs)
#define EMU86_EMULATE_THROW_DIVIDE_OVERFLOWW(/*u32*/ lhs, /*u16*/ rhs, /*u32*/ exact_result) EMU86_EMULATE_THROW_DIVIDE_BY_ZEROW(lhs)
#define EMU86_EMULATE_THROW_DIVIDE_OVERFLOWL(/*u64*/ lhs, /*u32*/ rhs, /*u64*/ exact_result) EMU86_EMULATE_THROW_DIVIDE_BY_ZEROL(lhs)
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_OVERFLOWB(/*s16*/ lhs, /*s8*/ rhs, /*s16*/ exact_result) EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZEROB(lhs)
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_OVERFLOWW(/*s32*/ lhs, /*s16*/ rhs, /*s32*/ exact_result) EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZEROW(lhs)
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_OVERFLOWL(/*s64*/ lhs, /*s32*/ rhs, /*s64*/ exact_result) EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZEROL(lhs)
#if LIBEMU86_CONFIG_WANT_64BIT && defined(__UINT128_TYPE__)
#define EMU86_EMULATE_THROW_DIVIDE_OVERFLOWQ(/*uint128_t*/ lhs, /*u64*/ rhs, /*uint128_t*/ exact_result) EMU86_EMULATE_THROW_DIVIDE_BY_ZEROQ(lhs)
#define EMU86_EMULATE_THROW_SIGNED_DIVIDE_OVERFLOWQ(/*int128_t*/ lhs, /*s64*/ rhs, /*int128_t*/ exact_result) EMU86_EMULATE_THROW_SIGNED_DIVIDE_BY_ZEROQ(lhs)
#endif /* LIBEMU86_CONFIG_WANT_64BIT && __UINT128_TYPE__ */
#endif /* !EMU86_EMULATE_THROW_DIVIDE_OVERFLOWB */

/* Return the initial set of opcode flags. */
#ifndef EMU86_EMULATE_GETOPFLAGS
#ifdef LIBEMU86_CONFIG_NEED_ARCHMODE
#define EMU86_EMULATE_GETOPFLAGS() emu86_opflags_from_icpustate(_state)
#else /* LIBEMU86_CONFIG_NEED_ARCHMODE */
#define EMU86_EMULATE_GETOPFLAGS() EMU86_F_NORMAL
#endif /* !LIBEMU86_CONFIG_NEED_ARCHMODE */
#endif /* !EMU86_EMULATE_GETOPFLAGS */

/* Translate  a  memory   address  to   its  real   counterpart.
 * This macro is invoked prior to any kind of memory access made
 * by hosted code, and can be used to implement software paging,
 * or  some kind of  addend added to any  kind of memory access. */
#ifndef EMU86_EMULATE_TRANSLATEADDR
#define EMU86_EMULATE_TRANSLATEADDR_IS_NOOP 1
#ifdef __INTELLISENSE__
#define EMU86_EMULATE_TRANSLATEADDR(addr) (void *)(addr)
#else /* __INTELLISENSE__ */
#define EMU86_EMULATE_TRANSLATEADDR(addr) (addr)
#endif /* !__INTELLISENSE__ */
#endif /* !EMU86_EMULATE_TRANSLATEADDR */


/* Enable emulation of VM86 helpers */
#ifndef EMU86_EMULATE_CONFIG_VM86
#define EMU86_EMULATE_CONFIG_VM86 (!LIBEMU86_CONFIG_WANT_64BIT && LIBEMU86_CONFIG_WANT_16BIT)
#endif /* !EMU86_EMULATE_CONFIG_VM86 */

#if EMU86_EMULATE_CONFIG_VM86 && LIBEMU86_CONFIG_WANT_64BIT
#error "Cannot emulate vm86 together with 64-bit support!"
#endif /* EMU86_EMULATE_CONFIG_VM86 && LIBEMU86_CONFIG_WANT_64BIT */

/* These functions are used only  when emulating an instruction for  vm86
 * They are _not_ used by libvm86 (which is a full realmode emulator that
 * doesn't actually make  use of  vm86, so-as  to also  work on  x86_64!)
 * These functions are used by */
#if EMU86_EMULATE_CONFIG_VM86
#ifndef EMU86_EMULATE_VM86_GETIF
#define EMU86_EMULATE_VM86_GETIF() 0
#endif /* !EMU86_EMULATE_VM86_GETIF */
#ifndef EMU86_EMULATE_VM86_SETIF
#define EMU86_EMULATE_VM86_SETIF(v) (void)0
#endif /* !EMU86_EMULATE_VM86_SETIF */
#ifndef EMU86_EMULATE_RETURN_AFTER_HLT_VM86
#define EMU86_EMULATE_RETURN_AFTER_HLT_VM86() goto done
#endif /* !EMU86_EMULATE_RETURN_AFTER_HLT_VM86 */
#endif /* EMU86_EMULATE_CONFIG_VM86 */



/* Emulation instruction hooks.
 * Each of these must be defined as a macro-function that behaves the same as a
 * function with a prototype identical to those found in the following disabled
 * pre-processor block.
 * Individual macros are only ever used any of the listed config options are in use
 * (i.e. only when specific instructions are being emulated), and when not defined,
 * those instructions will instead `goto return_unsupported_instruction[_rmreg]' */
#if 0
u16 EMU86_EMULATE_SLDT(void);                               /* EMU86_EMULATE_CONFIG_WANT_SLDT */
void EMU86_EMULATE_LLDT(u16 segment_index);                 /* EMU86_EMULATE_CONFIG_WANT_LLDT */
u16 EMU86_EMULATE_STR(void);                                /* EMU86_EMULATE_CONFIG_WANT_STR */
void EMU86_EMULATE_LTR(u16 segment_index);                  /* EMU86_EMULATE_CONFIG_WANT_LTR */
bool EMU86_EMULATE_VERR(u16 segment_index);                 /* EMU86_EMULATE_CONFIG_WANT_VERR */
bool EMU86_EMULATE_VERW(u16 segment_index);                 /* EMU86_EMULATE_CONFIG_WANT_VERW */
void EMU86_EMULATE_SGDT(u16 &limit, EMU86_UREG_TYPE &base); /* EMU86_EMULATE_CONFIG_WANT_SGDT */
void EMU86_EMULATE_LGDT(u16 limit, EMU86_UREG_TYPE base);   /* EMU86_EMULATE_CONFIG_WANT_LGDT */
void EMU86_EMULATE_SIDT(u16 &limit, EMU86_UREG_TYPE &base); /* EMU86_EMULATE_CONFIG_WANT_SIDT */
void EMU86_EMULATE_LIDT(u16 limit, EMU86_UREG_TYPE base);   /* EMU86_EMULATE_CONFIG_WANT_LIDT */
u16 EMU86_EMULATE_SMSW(void);                               /* EMU86_EMULATE_CONFIG_WANT_SMSW */
void EMU86_EMULATE_LMSW(u16 value);                         /* EMU86_EMULATE_CONFIG_WANT_LMSW */
void EMU86_EMULATE_INVLPG(void *addr);                      /* EMU86_EMULATE_CONFIG_WANT_INVLPG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR0(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR1(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR2(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR3(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR4(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR5(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR6(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR7(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR8(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR9(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR10(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR11(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR12(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR13(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR14(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDCR15(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR0(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR1(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR2(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR3(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR4(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR5(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR6(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR7(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR8(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR9(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR10(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR11(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR12(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR13(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR14(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
void EMU86_EMULATE_WRCR15(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_CREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR0(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR1(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR2(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR3(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR4(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR5(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR6(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR7(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR8(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR9(void);                  /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR10(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR11(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR12(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR13(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR14(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
EMU86_UREG_TYPE EMU86_EMULATE_RDDR15(void);                 /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR0(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR1(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR2(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR3(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR4(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR5(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR6(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR7(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR8(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR9(EMU86_UREG_TYPE value);            /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR10(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR11(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR12(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR13(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR14(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
void EMU86_EMULATE_WRDR15(EMU86_UREG_TYPE value);           /* EMU86_EMULATE_CONFIG_WANT_MOV_DREG */
bool EMU86_EMULATE_LAR(u16 value, u16 &result);             /* EMU86_EMULATE_CONFIG_WANT_LAR */
bool EMU86_EMULATE_LSL(u16 value, EMU86_UREG_TYPE &result); /* EMU86_EMULATE_CONFIG_WANT_LSL */
void EMU86_EMULATE_PATCH_RDFSBASE(void *pc);                /* EMU86_EMULATE_CONFIG_WANT_RDFSBASE */
void EMU86_EMULATE_PATCH_RDGSBASE(void *pc);                /* EMU86_EMULATE_CONFIG_WANT_RDGSBASE */
void EMU86_EMULATE_PATCH_WRFSBASE(void *pc);                /* EMU86_EMULATE_CONFIG_WANT_WRFSBASE */
void EMU86_EMULATE_PATCH_WRGSBASE(void *pc);                /* EMU86_EMULATE_CONFIG_WANT_WRGSBASE */
bool EMU86_SETFSBASE(EMU86_UREG_TYPE value);                /* EMU86_EMULATE_CONFIG_WANT_WRFSBASE */
bool EMU86_SETGSBASE(EMU86_UREG_TYPE value);                /* EMU86_EMULATE_CONFIG_WANT_WRGSBASE */
u64 EMU86_EMULATE_RDMSR(u32 index);                         /* EMU86_EMULATE_CONFIG_WANT_RDMSR */
void EMU86_EMULATE_WRMSR(u32 index, u64 value);             /* EMU86_EMULATE_CONFIG_WANT_WRMSR */
u64 EMU86_EMULATE_RDTSC(void);                              /* EMU86_EMULATE_CONFIG_WANT_RDTSC */
u64 EMU86_EMULATE_RDTSC_INDIRECT(void);                     /* EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED */
u32 EMU86_EMULATE_RDPID(void);                              /* EMU86_EMULATE_CONFIG_WANT_RDPID || EMU86_EMULATE_CONFIG_WANT_RDMSR_EMULATED */
u64 EMU86_EMULATE_RDTSCP(u32 &tsc_aux);                     /* EMU86_EMULATE_CONFIG_WANT_RDTSCP */
u64 EMU86_EMULATE_RDPMC(u32 index);                         /* EMU86_EMULATE_CONFIG_WANT_RDPMC */
bool EMU86_EMULATE_RDRAND16(u16 &result);                   /* EMU86_EMULATE_CONFIG_WANT_RDRAND */
bool EMU86_EMULATE_RDRAND32(u32 &result);                   /* EMU86_EMULATE_CONFIG_WANT_RDRAND */
bool EMU86_EMULATE_RDRAND64(u64 &result);                   /* EMU86_EMULATE_CONFIG_WANT_RDRAND */
bool EMU86_EMULATE_RDSEED16(u16 &result);                   /* EMU86_EMULATE_CONFIG_WANT_RDSEED */
bool EMU86_EMULATE_RDSEED32(u32 &result);                   /* EMU86_EMULATE_CONFIG_WANT_RDSEED */
bool EMU86_EMULATE_RDSEED64(u64 &result);                   /* EMU86_EMULATE_CONFIG_WANT_RDSEED */
void EMU86_EMULATE_CLTS(void);                              /* EMU86_EMULATE_CONFIG_WANT_CLTS */
void EMU86_EMULATE_MCOMMIT(void);                           /* EMU86_EMULATE_CONFIG_WANT_SWAPGS */
void EMU86_EMULATE_SWAPGS(void);                            /* EMU86_EMULATE_CONFIG_WANT_SWAPGS */
void EMU86_EMULATE_LFENCE(void);                            /* EMU86_EMULATE_CONFIG_WANT_LFENCE */
void EMU86_EMULATE_SFENCE(void);                            /* EMU86_EMULATE_CONFIG_WANT_SFENCE */
void EMU86_EMULATE_MFENCE(void);                            /* EMU86_EMULATE_CONFIG_WANT_MFENCE */
void EMU86_EMULATE_FENCELOCK(void);                         /* EMU86_EMULATE_CONFIG_WANT_LFENCE || EMU86_EMULATE_CONFIG_WANT_SFENCE || EMU86_EMULATE_CONFIG_WANT_MFENCE */
bool EMU86_EMULATE_TPAUSE(bool want_c01, u64 tsc_deadline); /* EMU86_EMULATE_CONFIG_WANT_TPAUSE */
void EMU86_EMULATE_CLZERO(void *addr);                      /* EMU86_EMULATE_CONFIG_WANT_CLZERO */
u64 EMU86_EMULATE_XGETBV(u32 index);                        /* EMU86_EMULATE_CONFIG_WANT_XGETBV */
void EMU86_EMULATE_XSETBV(u32 index, u64 value);            /* EMU86_EMULATE_CONFIG_WANT_XSETBV */
void EMU86_EMULATE_MONITOR(void *addr, u32 extensions, u32 hints); /* EMU86_EMULATE_CONFIG_WANT_MONITOR */
void EMU86_EMULATE_MWAIT(u32 ecx, u32 edx);                        /* EMU86_EMULATE_CONFIG_WANT_MWAIT */
void EMU86_EMULATE_ENCLS(u32 leaf);                         /* EMU86_EMULATE_CONFIG_WANT_ENCLS */
void EMU86_EMULATE_ENCLU(u32 leaf);                         /* EMU86_EMULATE_CONFIG_WANT_ENCLU */
void EMU86_EMULATE_LDMXCSR(u32 mxcsr);                      /* EMU86_EMULATE_CONFIG_WANT_LDMXCSR */
#endif



/* Emulate certain (harmless) instructions via native intrinsics
 * NOTE: All of these instructions are emulated within the KOS
 *       kernel when they're not supported by the hosting CPU. */
#if defined(__x86_64__) || defined(__i386__)
#if !defined(EMU86_EMULATE_CONFIG_NO_INTRIN) || !(EMU86_EMULATE_CONFIG_NO_INTRIN + 0)
#if defined(__KOS__) || defined(__x86_64__)
#ifndef EMU86_EMULATE_RDTSC
#define EMU86_EMULATE_RDTSC() __rdtsc()
#endif /* !EMU86_EMULATE_RDTSC */
#endif /* __KOS__ || __x86_64__ */
#ifdef __KOS__
#ifndef EMU86_EMULATE_RDPID
#ifdef __KERNEL__ /* Don't rely on the existence of `__rdpid()' when running from kernel-space! */
#define EMU86_EMULATE_RDPID() __rdmsr(IA32_TSC_AUX)
#else /* __KERNEL__ */
#define EMU86_EMULATE_RDPID() __rdpid()
#endif /* !__KERNEL__ */
#endif /* !EMU86_EMULATE_RDPID */
#ifndef EMU86_EMULATE_RDTSCP
#define EMU86_EMULATE_RDTSCP(tsc_aux) __rdtscp(&(tsc_aux))
#endif /* !EMU86_EMULATE_RDTSCP */
#ifndef EMU86_EMULATE_RDPMC
#define EMU86_EMULATE_RDPMC(index) __rdpmc(index)
#endif /* !EMU86_EMULATE_RDPMC */
#ifndef EMU86_EMULATE_RDRAND16
#define EMU86_EMULATE_RDRAND16(result) __rdrandw(&(result))
#endif /* !EMU86_EMULATE_RDRAND16 */
#ifndef EMU86_EMULATE_RDSEED16
#define EMU86_EMULATE_RDSEED16(result) __rdseedw(&(result))
#endif /* !EMU86_EMULATE_RDSEED16 */
#ifndef EMU86_EMULATE_RDRAND32
#define EMU86_EMULATE_RDRAND32(result) __rdrandl(&(result))
#endif /* !EMU86_EMULATE_RDRAND32 */
#ifndef EMU86_EMULATE_RDSEED32
#define EMU86_EMULATE_RDSEED32(result) __rdseedl(&(result))
#endif /* !EMU86_EMULATE_RDSEED32 */
#ifdef __x86_64__
#ifndef EMU86_EMULATE_RDRAND64
#define EMU86_EMULATE_RDRAND64(result) __rdrandq(&(result))
#endif /* !EMU86_EMULATE_RDRAND64 */
#ifndef EMU86_EMULATE_RDSEED64
#define EMU86_EMULATE_RDSEED64(result) __rdseedq(&(result))
#endif /* !EMU86_EMULATE_RDSEED64 */
#endif /* __x86_64__ */
#endif /* __KOS__ */
#ifndef EMU86_EMULATE_FENCELOCK
#define EMU86_EMULATE_FENCELOCK() __fencelock()
#endif /* !EMU86_EMULATE_FENCELOCK */
#ifndef EMU86_EMULATE_STMXCSR
#define EMU86_EMULATE_STMXCSR() __stmxcsr()
#endif /* !EMU86_EMULATE_STMXCSR */
#ifndef EMU86_EMULATE_LDMXCSR
#define EMU86_EMULATE_LDMXCSR(mxcsr) __ldmxcsr(mxcsr)
#endif /* !EMU86_EMULATE_LDMXCSR */
#endif /* !EMU86_EMULATE_CONFIG_NO_INTRIN */
#endif /* __x86_64__ || __i386__ */

#if !defined(EMU86_EMULATE_CONFIG_NO_INTRIN) || !(EMU86_EMULATE_CONFIG_NO_INTRIN + 0)
#ifndef EMU86_EMULATE_FENCELOCK
#define EMU86_EMULATE_FENCELOCK()                              \
	do {                                                       \
		static int volatile fence_word = 0;                    \
		__hybrid_atomic_xch(&fence_word, 1, __ATOMIC_SEQ_CST); \
	}	__WHILE0
#endif /* !EMU86_EMULATE_FENCELOCK */
#endif /* !EMU86_EMULATE_CONFIG_NO_INTRIN */


/* Substitute smsw/lmsw with read/write to/from %cr0 */
#if !defined(EMU86_EMULATE_SMSW) && defined(EMU86_EMULATE_RDCR0)
#define EMU86_EMULATE_SMSW() ((u16)EMU86_EMULATE_RDCR0())
#endif /* !EMU86_EMULATE_SMSW && EMU86_EMULATE_RDCR0 */
#if !defined(EMU86_EMULATE_LMSW) && defined(EMU86_EMULATE_WRCR0)
#define EMU86_EMULATE_LMSW(value)                        \
	EMU86_EMULATE_WRCR0((EMU86_EMULATE_RDCR0() & ~0xf) | \
	                    ((value) & 0xf))
#endif /* !EMU86_EMULATE_LMSW && EMU86_EMULATE_WRCR0 */

#if !defined(EMU86_EMULATE_RDTSC) && defined(EMU86_EMULATE_RDMSR)
#define EMU86_EMULATE_RDTSC() EMU86_EMULATE_RDMSR(IA32_TIME_STAMP_COUNTER)
#endif /* !EMU86_EMULATE_RDTSC && EMU86_EMULATE_RDMSR */
#if !defined(EMU86_EMULATE_RDPID) && defined(EMU86_EMULATE_RDMSR)
#define EMU86_EMULATE_RDPID() EMU86_EMULATE_RDMSR(IA32_TSC_AUX)
#endif /* !EMU86_EMULATE_RDPID && EMU86_EMULATE_RDMSR */


/* An optional override for the method used to read the TSC when read
 * from user-space via  `__rdmsr(IA32_TIME_STAMP_COUNTER)' (which  is
 * emulated as one of the white-listed MSRs that can actually be read
 * from user-space under KOS), or `__rdtscp()'
 * A  different  method is  used  for this  in  order to  allow instruction
 * emulation  to use the  native `__rdtsc()' for this  method, such that in
 * its absence the instruction emulator will fault into itself recursively,
 * at  which  point `EMU86_EMULATE_RDTSC()'  will  be used  to  emulate the
 * recursive  use-case,  such  that  `EMU86_EMULATE_RDTSC()'  may  use some
 * different  method  to  implement some  kind  of cycle  counter  (such as
 * APIC counters, or something similar). */
#if !defined(EMU86_EMULATE_RDTSC_INDIRECT) && defined(EMU86_EMULATE_RDTSC)
#define EMU86_EMULATE_RDTSC_INDIRECT() EMU86_EMULATE_RDTSC()
#endif /* !EMU86_EMULATE_RDTSC_INDIRECT && EMU86_EMULATE_RDTSC */
#if !defined(EMU86_EMULATE_RDTSC) && defined(EMU86_EMULATE_RDTSC_INDIRECT)
#define EMU86_EMULATE_RDTSC() EMU86_EMULATE_RDTSC_INDIRECT()
#endif /* !EMU86_EMULATE_RDTSC && EMU86_EMULATE_RDTSC_INDIRECT */

#if (!defined(EMU86_EMULATE_RDTSCP) && \
     (defined(EMU86_EMULATE_RDTSC_INDIRECT) && defined(EMU86_EMULATE_RDPID)))
#define EMU86_EMULATE_RDTSCP(tsc, tsc_aux)       \
	((tsc)     = EMU86_EMULATE_RDTSC_INDIRECT(), \
	 (tsc_aux) = EMU86_EMULATE_RDPID())
#endif /* !EMU86_EMULATE_RDTSCP && (EMU86_EMULATE_RDTSC_INDIRECT && EMU86_EMULATE_RDPID) */

/* Substitute clts with read/write to/from %cr0 */
#if !defined(EMU86_EMULATE_CLTS) && defined(EMU86_EMULATE_RDCR0) && defined(EMU86_EMULATE_WRCR0)
#define EMU86_EMULATE_CLTS() EMU86_EMULATE_WRCR0(EMU86_EMULATE_RDCR0() & ~(CR0_TS))
#endif /* !EMU86_EMULATE_CLTS && EMU86_EMULATE_RDCR0 && EMU86_EMULATE_WRCR0 */



/* Get the value of `CR4.UMIP', which when enabled causes the following
 * instructions  to  trigger  a  #GP  when  executed  from  user-space:
 *   - sldt
 *   - str
 *   - sgdt
 *   - sidt
 *   - smsw
 */
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_GETCR4_UMIP)
#ifdef EMU86_GETCR4_UMIP_IS_ONE
#define EMU86_GETCR4_UMIP() 1
#elif defined(EMU86_GETCR4_UMIP_IS_ZERO)
#define EMU86_GETCR4_UMIP() 0
#elif ((!defined(EMU86_EMULATE_CONFIG_NO_INTRIN) || !(EMU86_EMULATE_CONFIG_NO_INTRIN + 0)) && \
       (defined(__KERNEL__) && (defined(__x86_64__) || defined(__i386__))))
#define EMU86_GETCR4_UMIP() (__rdcr4() & CR4_UMIP)
#endif /* !EMU86_EMULATE_CONFIG_NO_INTRIN && __KERNEL__ && (__x86_64__ || __i386__) */
#ifndef EMU86_GETCR4_UMIP
#define EMU86_GETCR4_UMIP_IS_ONE 1
#define EMU86_GETCR4_UMIP() 1
#endif /* !EMU86_GETCR4_UMIP */
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_GETCR4_UMIP */

/* CR4.DE -- When disabled, read/write to/from %dr4 and %dr5 actually use %dr6 and  %dr7
 *           When enabled, such reads/writes actually affect the proper registers (which
 *           can be provided  with `EMU86_EMULATE_(RD|WR)DR(4|5)'), or  cause a call  to
 *           `EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER()' with the actually used
 *           register ID.
 * NOTE: To prevent leaking the state of CR4.DE to user-space via exceptions (attempting
 *       to access %dr4 would otherwise (correctly)  cause an exception with %dr6),  the
 *       state of CR4.DE is always handled as though it was 1 (i.e. don't re-route debug
 *       register indices) when `EMU86_ISUSER()' evaluates to true. */
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_GETCR4_DE)
#ifdef EMU86_GETCR4_DE_IS_ONE
#define EMU86_GETCR4_DE() 1
#elif defined(EMU86_GETCR4_DE_IS_ZERO)
#define EMU86_GETCR4_DE() 0
#elif ((!defined(EMU86_EMULATE_CONFIG_NO_INTRIN) || !(EMU86_EMULATE_CONFIG_NO_INTRIN + 0)) && \
       (defined(__KERNEL__) && (defined(__x86_64__) || defined(__i386__))))
#define EMU86_GETCR4_DE() (__rdcr4() & CR4_DE)
#endif /* !EMU86_EMULATE_CONFIG_NO_INTRIN && __KERNEL__ && (__x86_64__ || __i386__) */
#ifndef EMU86_GETCR4_DE
#define EMU86_GETCR4_DE_IS_ONE 1
#define EMU86_GETCR4_DE() 1
#endif /* !EMU86_GETCR4_DE */
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_GETCR4_DE */


/* CR4.TSD -- When enabled, user-space isn't allowed to access to `IA32_TIME_STAMP_COUNTER',
 *            either via  rdmsr,  rdtsc  or rdtscp,  all  of  which will  instead  call  to:
 *            >> EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER(E_ILLEGAL_INSTRUCTION_REGISTER_RDPRV,
 *            >>                                                  X86_REGISTER_MSR, IA32_TIME_STAMP_COUNTER, 0, 0) */
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_GETCR4_TSD)
#ifdef EMU86_GETCR4_TSD_IS_ONE
#define EMU86_GETCR4_TSD() 1
#elif defined(EMU86_GETCR4_TSD_IS_ZERO)
#define EMU86_GETCR4_TSD() 0
#elif ((!defined(EMU86_EMULATE_CONFIG_NO_INTRIN) || !(EMU86_EMULATE_CONFIG_NO_INTRIN + 0)) && \
       (defined(__KERNEL__) && (defined(__x86_64__) || defined(__i386__))))
#define EMU86_GETCR4_TSD() (__rdcr4() & CR4_TSD)
#endif /* !EMU86_EMULATE_CONFIG_NO_INTRIN && __KERNEL__ && (__x86_64__ || __i386__) */
#ifndef EMU86_GETCR4_TSD
#define EMU86_GETCR4_TSD_IS_ZERO 1
#define EMU86_GETCR4_TSD() 0
#endif /* !EMU86_GETCR4_TSD */
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_GETCR4_TSD */


/* CR4.PCE -- When enabled, user-space isn't allowed to access to `rdpmc', which will instead call to:
 *            >> EMU86_EMULATE_THROW_ILLEGAL_INSTRUCTION_REGISTER(E_ILLEGAL_INSTRUCTION_REGISTER_RDPRV,
 *            >>                                                  X86_REGISTER_PCR, %ecx, 0, 0) */
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_GETCR4_PCE)
#ifdef EMU86_GETCR4_PCE_IS_ONE
#define EMU86_GETCR4_PCE() 1
#elif defined(EMU86_GETCR4_PCE_IS_ZERO)
#define EMU86_GETCR4_PCE() 0
#elif ((!defined(EMU86_EMULATE_CONFIG_NO_INTRIN) || !(EMU86_EMULATE_CONFIG_NO_INTRIN + 0)) && \
       (defined(__KERNEL__) && (defined(__x86_64__) || defined(__i386__))))
#define EMU86_GETCR4_PCE() (__rdcr4() & CR4_PCE)
#endif /* !EMU86_EMULATE_CONFIG_NO_INTRIN && __KERNEL__ && (__x86_64__ || __i386__) */
#ifndef EMU86_GETCR4_PCE
#define EMU86_GETCR4_PCE_IS_ONE 1
#define EMU86_GETCR4_PCE() 1
#endif /* !EMU86_GETCR4_PCE */
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_GETCR4_PCE */




/* Read/write memory
 * NOTE: These functions aren't used when it comes to reading from PC-relative memory. */
#ifndef EMU86_MEMREADB
#define EMU86_MEMREADB(addr) \
	UNALIGNED_GETLE8(EMU86_EMULATE_TRANSLATEADDR(addr))
#endif /* !EMU86_MEMREADB */
#ifndef EMU86_MEMREADW
#define EMU86_MEMREADW(addr) \
	UNALIGNED_GETLE16(EMU86_EMULATE_TRANSLATEADDR(addr))
#endif /* !EMU86_MEMREADW */
#ifndef EMU86_MEMREADL
#define EMU86_MEMREADL(addr) \
	UNALIGNED_GETLE32(EMU86_EMULATE_TRANSLATEADDR(addr))
#endif /* !EMU86_MEMREADL */
#ifndef EMU86_MEMWRITEB
#define EMU86_MEMWRITEB(addr, v) \
	UNALIGNED_SETLE8(EMU86_EMULATE_TRANSLATEADDR(addr), v)
#endif /* !EMU86_MEMWRITEB */
#ifndef EMU86_MEMWRITEW
#define EMU86_MEMWRITEW(addr, v) \
	UNALIGNED_SETLE16(EMU86_EMULATE_TRANSLATEADDR(addr), v)
#endif /* !EMU86_MEMWRITEW */
#ifndef EMU86_MEMWRITEL
#define EMU86_MEMWRITEL(addr, v) \
	UNALIGNED_SETLE32(EMU86_EMULATE_TRANSLATEADDR(addr), v)
#endif /* !EMU86_MEMWRITEL */
#ifndef EMU86_MEM_ATOMIC_XCHB
#define EMU86_MEM_ATOMIC_XCHB(addr, addend, force_atomic) \
	__hybrid_atomic_xch8(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_XCHB */
#ifndef EMU86_MEM_ATOMIC_XCHW
#define EMU86_MEM_ATOMIC_XCHW(addr, addend, force_atomic) \
	__hybrid_atomic_xch16(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_XCHW */
#ifndef EMU86_MEM_ATOMIC_XCHL
#define EMU86_MEM_ATOMIC_XCHL(addr, addend, force_atomic) \
	__hybrid_atomic_xch32(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_XCHL */
#ifndef EMU86_MEM_ATOMIC_FETCHADDB
#define EMU86_MEM_ATOMIC_FETCHADDB(addr, addend, force_atomic) \
	__hybrid_atomic_fetchadd8(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHADDB */
#ifndef EMU86_MEM_ATOMIC_FETCHADDW
#define EMU86_MEM_ATOMIC_FETCHADDW(addr, addend, force_atomic) \
	__hybrid_atomic_fetchadd16(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHADDW */
#ifndef EMU86_MEM_ATOMIC_FETCHADDL
#define EMU86_MEM_ATOMIC_FETCHADDL(addr, addend, force_atomic) \
	__hybrid_atomic_fetchadd32(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHADDL */
#ifndef EMU86_MEM_ATOMIC_FETCHSUBB
#define EMU86_MEM_ATOMIC_FETCHSUBB(addr, addend, force_atomic) \
	__hybrid_atomic_fetchsub8(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHSUBB */
#ifndef EMU86_MEM_ATOMIC_FETCHSUBW
#define EMU86_MEM_ATOMIC_FETCHSUBW(addr, addend, force_atomic) \
	__hybrid_atomic_fetchsub16(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHSUBW */
#ifndef EMU86_MEM_ATOMIC_FETCHSUBL
#define EMU86_MEM_ATOMIC_FETCHSUBL(addr, addend, force_atomic) \
	__hybrid_atomic_fetchsub32(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHSUBL */
#ifndef EMU86_MEM_ATOMIC_FETCHANDB
#define EMU86_MEM_ATOMIC_FETCHANDB(addr, addend, force_atomic) \
	__hybrid_atomic_fetchand8(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHANDB */
#ifndef EMU86_MEM_ATOMIC_FETCHANDW
#define EMU86_MEM_ATOMIC_FETCHANDW(addr, addend, force_atomic) \
	__hybrid_atomic_fetchand16(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHANDW */
#ifndef EMU86_MEM_ATOMIC_FETCHANDL
#define EMU86_MEM_ATOMIC_FETCHANDL(addr, addend, force_atomic) \
	__hybrid_atomic_fetchand32(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHANDL */
#ifndef EMU86_MEM_ATOMIC_FETCHORB
#define EMU86_MEM_ATOMIC_FETCHORB(addr, addend, force_atomic) \
	__hybrid_atomic_fetchor8(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHORB */
#ifndef EMU86_MEM_ATOMIC_FETCHORW
#define EMU86_MEM_ATOMIC_FETCHORW(addr, addend, force_atomic) \
	__hybrid_atomic_fetchor16(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHORW */
#ifndef EMU86_MEM_ATOMIC_FETCHORL
#define EMU86_MEM_ATOMIC_FETCHORL(addr, addend, force_atomic) \
	__hybrid_atomic_fetchor32(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHORL */
#ifndef EMU86_MEM_ATOMIC_FETCHXORB
#define EMU86_MEM_ATOMIC_FETCHXORB(addr, addend, force_atomic) \
	__hybrid_atomic_fetchxor8(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHXORB */
#ifndef EMU86_MEM_ATOMIC_FETCHXORW
#define EMU86_MEM_ATOMIC_FETCHXORW(addr, addend, force_atomic) \
	__hybrid_atomic_fetchxor16(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHXORW */
#ifndef EMU86_MEM_ATOMIC_FETCHXORL
#define EMU86_MEM_ATOMIC_FETCHXORL(addr, addend, force_atomic) \
	__hybrid_atomic_fetchxor32(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHXORL */
#ifndef EMU86_MEM_ATOMIC_CMPXCHB
#define EMU86_MEM_ATOMIC_CMPXCHB(addr, oldval, newval, force_atomic) \
	__hybrid_atomic_cmpxch_val8(EMU86_EMULATE_TRANSLATEADDR(addr),   \
	                            oldval, newval, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_CMPXCHB */
#ifndef EMU86_MEM_ATOMIC_CMPXCHW
#define EMU86_MEM_ATOMIC_CMPXCHW(addr, oldval, newval, force_atomic) \
	__hybrid_atomic_cmpxch_val16(EMU86_EMULATE_TRANSLATEADDR(addr),  \
	                             oldval, newval, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_CMPXCHW */
#ifndef EMU86_MEM_ATOMIC_CMPXCHL
#define EMU86_MEM_ATOMIC_CMPXCHL(addr, oldval, newval, force_atomic) \
	__hybrid_atomic_cmpxch_val32(EMU86_EMULATE_TRANSLATEADDR(addr),  \
	                             oldval, newval, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_CMPXCHL */
#ifndef EMU86_MEM_ATOMIC_CMPXCHQ
#define EMU86_MEM_ATOMIC_CMPXCHQ(addr, oldval, newval, force_atomic) \
	__hybrid_atomic_cmpxch_val64(EMU86_EMULATE_TRANSLATEADDR(addr),  \
	                             oldval, newval, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_CMPXCHQ */
#ifndef EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEB
#define EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEB(addr, oldval, newval, force_atomic)       \
	(__hybrid_atomic_cmpxch8(EMU86_EMULATE_TRANSLATEADDR(addr),                     \
	                         oldval, newval, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST) || \
	 !(force_atomic))
#endif /* !EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEB */
#ifndef EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEW
#define EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEW(addr, oldval, newval, force_atomic)        \
	(__hybrid_atomic_cmpxch16(EMU86_EMULATE_TRANSLATEADDR(addr),                     \
	                          oldval, newval, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST) || \
	 !(force_atomic))
#endif /* !EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEW */
#ifndef EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEL
#define EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEL(addr, oldval, newval, force_atomic)      \
	(__hybrid_atomic_cmpxch((u32 *)EMU86_EMULATE_TRANSLATEADDR(addr),              \
	                        oldval, newval, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST) || \
	 !(force_atomic))
#endif /* !EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEL */
#if LIBEMU86_CONFIG_WANT_64BIT
#ifndef EMU86_MEMREADQ
#define EMU86_MEMREADQ(addr) \
	UNALIGNED_GET64(EMU86_EMULATE_TRANSLATEADDR(addr))
#endif /* !EMU86_MEMREADQ */
#ifndef EMU86_MEMWRITEQ
#define EMU86_MEMWRITEQ(addr, v) \
	UNALIGNED_SET64(EMU86_EMULATE_TRANSLATEADDR(addr), v)
#endif /* !EMU86_MEMWRITEQ */
#ifndef EMU86_MEM_ATOMIC_XCHQ
#define EMU86_MEM_ATOMIC_XCHQ(addr, addend, force_atomic) \
	__hybrid_atomic_xch64(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_XCHQ */
#ifndef EMU86_MEM_ATOMIC_FETCHADDQ
#define EMU86_MEM_ATOMIC_FETCHADDQ(addr, addend, force_atomic) \
	__hybrid_atomic_fetchadd64(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHADDQ */
#ifndef EMU86_MEM_ATOMIC_FETCHSUBQ
#define EMU86_MEM_ATOMIC_FETCHSUBQ(addr, addend, force_atomic) \
	__hybrid_atomic_fetchsub64(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHSUBQ */
#ifndef EMU86_MEM_ATOMIC_FETCHANDQ
#define EMU86_MEM_ATOMIC_FETCHANDQ(addr, addend, force_atomic) \
	__hybrid_atomic_fetchand64(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHANDQ */
#ifndef EMU86_MEM_ATOMIC_FETCHORQ
#define EMU86_MEM_ATOMIC_FETCHORQ(addr, addend, force_atomic) \
	__hybrid_atomic_fetchor64(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHORQ */
#ifndef EMU86_MEM_ATOMIC_FETCHXORQ
#define EMU86_MEM_ATOMIC_FETCHXORQ(addr, addend, force_atomic) \
	__hybrid_atomic_fetchxor64(EMU86_EMULATE_TRANSLATEADDR(addr), addend, __ATOMIC_SEQ_CST)
#endif /* !EMU86_MEM_ATOMIC_FETCHXORQ */
#ifndef EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEQ
#define EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEQ(addr, oldval, newval, force_atomic)        \
	(__hybrid_atomic_cmpxch64(EMU86_EMULATE_TRANSLATEADDR(addr),                     \
	                          oldval, newval, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST) || \
	 (force_atomic))
#endif /* !EMU86_MEM_ATOMIC_CMPXCH_OR_WRITEQ */
#ifndef EMU86_MEM_ATOMIC_CMPXCHX
#ifdef __x86_64__
#define EMU86_MEM_ATOMIC_CMPXCHX(addr, oldval, newval, force_atomic) \
	__lock_cmpxchg16b((uint128_t *)(addr), oldval, newval)
#endif /* __x86_64__ */
#endif /* !EMU86_MEM_ATOMIC_CMPXCHX */
#endif /* LIBEMU86_CONFIG_WANT_64BIT */

#ifndef EMU86_MEMREADLASW
#if 1
#define EMU86_MEMREADLASW(addr) EMU86_MEMREADW(addr)
#else
#define EMU86_MEMREADLASW(addr) (u16)EMU86_MEMREADL(addr)
#endif
#endif /* !EMU86_MEMREADLASW */

#ifndef EMU86_MEMWRITELASW
#if 1
#define EMU86_MEMWRITELASW(addr, value) EMU86_MEMWRITEW(addr, value)
#else
#define EMU86_MEMWRITELASW(addr, value) EMU86_MEMWRITEL(addr, value)
#endif
#endif /* !EMU86_MEMWRITELASW */

#if LIBEMU86_CONFIG_WANT_64BIT
#ifndef EMU86_MEMREADQASW
#if 1
#define EMU86_MEMREADQASW(addr) EMU86_MEMREADW(addr)
#else
#define EMU86_MEMREADQASW(addr) (u16)EMU86_MEMREADQ(addr)
#endif
#endif /* !EMU86_MEMREADQASW */
#ifndef EMU86_MEMREADQASL
#if 1
#define EMU86_MEMREADQASL(addr) EMU86_MEMREADL(addr)
#else
#define EMU86_MEMREADQASL(addr) (u32)EMU86_MEMREADQ(addr)
#endif
#endif /* !EMU86_MEMREADQASL */
#ifndef EMU86_MEMWRITEQASW
#if 1
#define EMU86_MEMWRITEQASW(addr, value) EMU86_MEMWRITEW(addr, value)
#else
#define EMU86_MEMWRITEQASW(addr, value) EMU86_MEMWRITEQ(addr, value)
#endif
#endif /* !EMU86_MEMWRITEQASW */
#ifndef EMU86_MEMWRITEQASL
#if 1
#define EMU86_MEMWRITEQASL(addr, value) EMU86_MEMWRITEL(addr, value)
#else
#define EMU86_MEMWRITEQASL(addr, value) EMU86_MEMWRITEQ(addr, value)
#endif
#endif /* !EMU86_MEMWRITEQASL */
#endif /* LIBEMU86_CONFIG_WANT_64BIT */

/* Return non-zero if user access checks should be performed for the caller. */
#ifndef EMU86_ISUSER
#define EMU86_ISUSER() icpustate_isuser(_state)
#endif /* !EMU86_ISUSER */
#ifndef EMU86_ISUSER_NOVM86
#if EMU86_EMULATE_CONFIG_VM86
#define EMU86_ISUSER_NOVM86() icpustate_isuser_novm86(_state)
#else /* EMU86_EMULATE_CONFIG_VM86 */
#define EMU86_ISUSER_NOVM86() EMU86_ISUSER()
#endif /* !EMU86_EMULATE_CONFIG_VM86 */
#endif /* !EMU86_ISUSER_NOVM86 */
#ifndef EMU86_ISVM86
#if EMU86_EMULATE_CONFIG_VM86
#define EMU86_ISVM86() (EMU86_GETFLAGS() & EFLAGS_VM)
#else /* EMU86_EMULATE_CONFIG_VM86 */
#define EMU86_ISVM86() 0
#endif /* !EMU86_EMULATE_CONFIG_VM86 */
#endif /* !EMU86_ISVM86 */

/* Validate that `addr ... += num_bytes' is a user-space address range. */
#ifndef EMU86_VALIDATE_READABLE
#ifdef __KERNEL__
#include <kernel/user.h>
#define EMU86_VALIDATE_READABLE(addr, num_bytes)  validate_readable((void const *)(uintptr_t)(addr), num_bytes)
#define EMU86_VALIDATE_WRITABLE(addr, num_bytes)  validate_writable((void *)(uintptr_t)(addr), num_bytes)
#define EMU86_VALIDATE_READWRITE(addr, num_bytes) validate_readwrite((void *)(uintptr_t)(addr), num_bytes)
#else /* __KERNEL__ */
#define EMU86_VALIDATE_READABLE(addr, num_bytes)  (void)0
#define EMU86_VALIDATE_WRITABLE(addr, num_bytes)  (void)0
#define EMU86_VALIDATE_READWRITE(addr, num_bytes) (void)0
#define EMU86_VALIDATE_READABLE_IS_NOOP  1
#define EMU86_VALIDATE_WRITABLE_IS_NOOP  1
#define EMU86_VALIDATE_READWRITE_IS_NOOP 1
#endif /* !__KERNEL__ */
#endif /* !EMU86_VALIDATE_READABLE */


/* Validate that a given `addr' is canonical.
 * NOTE: It is assumed that any 32-bit address is always canonical! */
#if LIBEMU86_CONFIG_WANT_64BIT
#ifndef EMU86_VALIDATE_CANONICAL
#if __SIZEOF_POINTER__ < 8
#define EMU86_VALIDATE_CANONICAL_IS_NOOP 1
#define EMU86_VALIDATE_CANONICAL(addr) (void)0
#elif defined(EMU86_EMULATE_THROW_SEGFAULT_UNMAPPED_NONCANON)
#ifndef EMU86_VALIDATE_IS_CANONICAL
#ifdef __x86_64__
#include <kos/kernel/paging.h>
#define EMU86_VALIDATE_IS_CANONICAL(addr) ADDR_IS_CANON(addr)
#else /* __x86_64__ */
#define EMU86_VALIDATE_IS_CANONICAL(addr)                     \
	((u64)(uintptr_t)(addr) < UINT64_C(0x0000800000000000) || \
	 (u64)(uintptr_t)(addr) > UINT64_C(0xffff7fffffffffff))
#endif /* !__x86_64__ */
#endif /* !EMU86_VALIDATE_IS_CANONICAL */
#define EMU86_VALIDATE_CANONICAL(addr)                            \
	do {                                                          \
		if unlikely(!EMU86_VALIDATE_IS_CANONICAL(addr))           \
			EMU86_EMULATE_THROW_SEGFAULT_UNMAPPED_NONCANON(addr); \
	}	__WHILE0
#else /* EMU86_EMULATE_THROW_SEGFAULT_UNMAPPED_NONCANON */
#define EMU86_VALIDATE_CANONICAL_IS_NOOP 1
#define EMU86_VALIDATE_CANONICAL(addr) (void)0
#endif /* !EMU86_EMULATE_THROW_SEGFAULT_UNMAPPED_NONCANON */
#endif /* !EMU86_VALIDATE_CANONICAL */
#endif /* LIBEMU86_CONFIG_WANT_64BIT */


/* Handle the case where an unsupported instruction performs a memory access.
 * @param: addr:      [type(void *)] Base address (not translated).
 * @param: num_bytes: [type(size_t)] The number of bytes being accessed.
 * @param: reading:   [type(bool)]   A read is being performed.
 * @param: writing:   [type(bool)]   A write is being performed.
 * NOTE: Both `reading' and `writing' may be `true', in which case
 *       the  instruction would have performed a read+modify+write
 *       or read+write operation.
 * WARNING: The given address range will not have been user-verified
 *          when this macro is called! */
#ifndef EMU86_UNSUPPORTED_MEMACCESS
#define EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP 1
#define EMU86_UNSUPPORTED_MEMACCESS(addr, num_bytes, reading, writing) (void)0
#endif /* !EMU86_UNSUPPORTED_MEMREAD */

/* Get/Set the value of the EFLAGS register. */
#ifndef EMU86_GETFLAGS
#define EMU86_GETFLAGS()            icpustate_getpflags(_state)
#define EMU86_SETFLAGS(v)           icpustate_setpflags(_state, v)
#define EMU86_MSKFLAGS(mask, value) icpustate_mskpflags(_state, mask, value)
#endif /* !EMU86_GETFLAGS */
#ifndef EMU86_MSKFLAGS
#define EMU86_MSKFLAGS(mask, value) \
	EMU86_SETFLAGS((EMU86_GETFLAGS() & (mask)) | (value))
#endif /* !EMU86_MSKFLAGS */

/* Get/set   the   program   counter   (IP/EIP/RIP)   register   (including  CS.BASE)
 * Note that `EMU86_GETPCPTR() == EMU86_SEGADDR(EMU86_GETCSBASE(), EMU86_GETIPREG())' */
#ifndef EMU86_GETPCPTR
#define EMU86_GETPCPTR()  icpustate_getpc(_state)
#define EMU86_SETPCPTR(v) icpustate_setpc(_state, v)
#endif /* !EMU86_GETPCPTR */

/* Same  as   `EMU86_SETPCPTR()',  but   don't  account   for
 * segment offsets, but simply set the raw %(e|r)ip register. */
#ifndef EMU86_GETIPREG
#define EMU86_GETIPREG()  (EMU86_UREG_TYPE)(uintptr_t)EMU86_GETPCPTR()
#define EMU86_SETIPREG(v) EMU86_SETPCPTR((void *)(uintptr_t)(EMU86_UREG_TYPE)(v))
#endif /* !EMU86_GETIPREG */

/* Get/Set the SP/ESP/RSP register (including SS.BASE)
 * Note that `EMU86_GETSTACKPTR() == EMU86_SEGADDR(EMU86_GETSSBASE(), EMU86_GETSPREG())' */
#ifndef EMU86_GETSTACKPTR
#define EMU86_GETSTACKPTR() icpustate_getsp(_state)
#ifdef __x86_64__
#define EMU86_SETSTACKPTR(v) icpustate_setsp(_state, v)
#else /* __x86_64__ */
#define EMU86_SETSTACKPTR(v) (_state = icpustate_setsp_p(_state, v))
#endif /* !__x86_64__ */
#endif /* !EMU86_GETSTACKPTR */

/* Same as `EMU86_GETIPREG()' is for `EMU86_GETPCPTR()', but for `EMU86_GETSTACKPTR()' */
#ifndef EMU86_GETSPREG
#define EMU86_GETSPREG()  (EMU86_UREG_TYPE)(uintptr_t)EMU86_GETSTACKPTR()
#define EMU86_SETSPREG(v) EMU86_SETSTACKPTR((void *)(uintptr_t)(EMU86_UREG_TYPE)(v))
#endif /* !EMU86_GETSPREG */

/* Return the base address address for a given segment */
#ifndef EMU86_GETSEGBASE
#define EMU86_GETSEGBASE_IS_NOOP_ES 1
#define EMU86_GETSEGBASE_IS_NOOP_CS 1
#define EMU86_GETSEGBASE_IS_NOOP_SS 1
#define EMU86_GETSEGBASE_IS_NOOP_DS 1
#define EMU86_GETSEGBASE_IS_NOOP_FS 1
#define EMU86_GETSEGBASE_IS_NOOP_GS 1
#define EMU86_GETSEGBASE(segment_regno) 0
#endif /* !EMU86_GETSEGBASE */

#if (defined(EMU86_GETSEGBASE_IS_NOOP_ES) || \
     defined(EMU86_GETSEGBASE_IS_NOOP_CS) || \
     defined(EMU86_GETSEGBASE_IS_NOOP_SS) || \
     defined(EMU86_GETSEGBASE_IS_NOOP_DS) || \
     defined(EMU86_GETSEGBASE_IS_NOOP_FS) || \
     defined(EMU86_GETSEGBASE_IS_NOOP_GS))
#if (defined(EMU86_GETSEGBASE_IS_NOOP_ES) && \
     defined(EMU86_GETSEGBASE_IS_NOOP_CS) && \
     defined(EMU86_GETSEGBASE_IS_NOOP_SS) && \
     defined(EMU86_GETSEGBASE_IS_NOOP_DS) && \
     defined(EMU86_GETSEGBASE_IS_NOOP_FS) && \
     defined(EMU86_GETSEGBASE_IS_NOOP_GS))
#define EMU86_GETSEGBASE_IS_NOOP_ALL 1
#define EMU86_GETSEGBASE_IS_NOOP(regno) 1
#else /* All-noop */
#ifdef EMU86_GETSEGBASE_IS_NOOP_ES
#define __EMU86_GETSEGBASE_IS_NOOP_ES 1
#else /* EMU86_GETSEGBASE_IS_NOOP_ES */
#define __EMU86_GETSEGBASE_IS_NOOP_ES 0
#endif /* !EMU86_GETSEGBASE_IS_NOOP_ES */
#ifdef EMU86_GETSEGBASE_IS_NOOP_CS
#define __EMU86_GETSEGBASE_IS_NOOP_CS 2
#else /* EMU86_GETSEGBASE_IS_NOOP_CS */
#define __EMU86_GETSEGBASE_IS_NOOP_CS 0
#endif /* !EMU86_GETSEGBASE_IS_NOOP_CS */
#ifdef EMU86_GETSEGBASE_IS_NOOP_SS
#define __EMU86_GETSEGBASE_IS_NOOP_SS 4
#else /* EMU86_GETSEGBASE_IS_NOOP_SS */
#define __EMU86_GETSEGBASE_IS_NOOP_SS 0
#endif /* !EMU86_GETSEGBASE_IS_NOOP_SS */
#ifdef EMU86_GETSEGBASE_IS_NOOP_DS
#define __EMU86_GETSEGBASE_IS_NOOP_DS 8
#else /* EMU86_GETSEGBASE_IS_NOOP_DS */
#define __EMU86_GETSEGBASE_IS_NOOP_DS 0
#endif /* !EMU86_GETSEGBASE_IS_NOOP_DS */
#ifdef EMU86_GETSEGBASE_IS_NOOP_FS
#define __EMU86_GETSEGBASE_IS_NOOP_FS 16
#else /* EMU86_GETSEGBASE_IS_NOOP_FS */
#define __EMU86_GETSEGBASE_IS_NOOP_FS 0
#endif /* !EMU86_GETSEGBASE_IS_NOOP_FS */
#ifdef EMU86_GETSEGBASE_IS_NOOP_GS
#define __EMU86_GETSEGBASE_IS_NOOP_GS 32
#else /* EMU86_GETSEGBASE_IS_NOOP_GS */
#define __EMU86_GETSEGBASE_IS_NOOP_GS 0
#endif /* !EMU86_GETSEGBASE_IS_NOOP_GS */
#define __EMU86_GETSEGBASE_IS_NOOP                                   \
	(__EMU86_GETSEGBASE_IS_NOOP_ES | __EMU86_GETSEGBASE_IS_NOOP_CS | \
	 __EMU86_GETSEGBASE_IS_NOOP_SS | __EMU86_GETSEGBASE_IS_NOOP_DS | \
	 __EMU86_GETSEGBASE_IS_NOOP_FS | __EMU86_GETSEGBASE_IS_NOOP_GS)
#define EMU86_GETSEGBASE_IS_NOOP_ANY 1
#define EMU86_GETSEGBASE_IS_NOOP(regno) \
	((__EMU86_GETSEGBASE_IS_NOOP >> (regno)) & 1)
#endif /* !All-noop */
#else /* Any-noop */
#define EMU86_GETSEGBASE_IS_NOOP(regno) 0
#endif /* !Any-noop */


#ifndef EMU86_GETDSBASE
#define EMU86_GETDSBASE() EMU86_GETSEGBASE(EMU86_R_DS)
#endif /* !EMU86_GETDSBASE */
#ifndef EMU86_GETESBASE
#define EMU86_GETESBASE() EMU86_GETSEGBASE(EMU86_R_ES)
#endif /* !EMU86_GETESBASE */
#ifndef EMU86_GETCSBASE
#define EMU86_GETCSBASE() EMU86_GETSEGBASE(EMU86_R_CS)
#endif /* !EMU86_GETCSBASE */
#ifndef EMU86_GETSSBASE
#define EMU86_GETSSBASE() EMU86_GETSEGBASE(EMU86_R_SS)
#endif /* !EMU86_GETSSBASE */
#ifndef EMU86_GETFSBASE
#define EMU86_GETFSBASE() EMU86_GETSEGBASE(EMU86_R_FS)
#endif /* !EMU86_GETFSBASE */
#ifndef EMU86_GETGSBASE
#define EMU86_GETGSBASE() EMU86_GETSEGBASE(EMU86_R_GS)
#endif /* !EMU86_GETGSBASE */


/* Substitute swapgs with MSR access */
#if (!defined(EMU86_EMULATE_SWAPGS) &&                               \
     defined(EMU86_EMULATE_RDMSR) && defined(EMU86_EMULATE_WRMSR) && \
     defined(EMU86_GETGSBASE) && defined(EMU86_SETGSBASE))
#define EMU86_EMULATE_SWAPGS()                                          \
	do {                                                                \
		EMU86_UREG_TYPE _gsbase, _kernel_gsbase;                        \
		_gsbase        = (EMU86_UREG_TYPE)(uintptr_t)EMU86_GETGSBASE(); \
		_kernel_gsbase = EMU86_EMULATE_RDMSR(IA32_KERNEL_GS_BASE);      \
		EMU86_SETGSBASE(_kernel_gsbase);                                \
		EMU86_EMULATE_WRMSR(IA32_KERNEL_GS_BASE, _gsbase);              \
	}	__WHILE0
#endif /* !EMU86_EMULATE_SWAPGS && ... */


/* Form a memory address from a segment, and an offset to that segment */
#ifndef EMU86_SEGADDR
#define EMU86_SEGADDR(segbase, segoffset) (byte_t *)(uintptr_t)((segbase) + (segoffset))
#endif /* !EMU86_SEGADDR */


/* Get/Set the %es register */
#ifndef EMU86_GETES
#define EMU86_GETES()  icpustate_getes(_state)
#define EMU86_SETES(v) icpustate_setes(_state, v)
#endif /* !EMU86_GETES */

/* Get/Set the %cs register */
#ifndef EMU86_GETCS
#define EMU86_GETCS()  icpustate_getcs(_state)
#define EMU86_SETCS(v) icpustate_setcs(_state, v)
#endif /* !EMU86_GETCS */

/* Get/Set the %ss register */
#ifndef EMU86_GETSS
#define EMU86_GETSS()  icpustate_getss(_state)
#ifdef icpustate_setss
#define EMU86_SETSS(v) icpustate_setss(_state, v)
#else /* icpustate_setss */
#define EMU86_SETSS(v) icpustate_trysetss(_state, v) /* Override me to handle failure! */
#endif /* !icpustate_setss */
#endif /* !EMU86_GETSTACKPTR */

/* Get/Set the %ds register */
#ifndef EMU86_GETDS
#define EMU86_GETDS()  icpustate_getds(_state)
#define EMU86_SETDS(v) icpustate_setds(_state, v)
#endif /* !EMU86_GETDS */

/* Get/Set the %fs register */
#ifndef EMU86_GETFS
#define EMU86_GETFS()  icpustate_getfs(_state)
#define EMU86_SETFS(v) icpustate_setfs(_state, v)
#endif /* !EMU86_GETFS */

/* Get/Set the %gs register */
#ifndef EMU86_GETGS
#define EMU86_GETGS()  icpustate_getgs(_state)
#define EMU86_SETGS(v) icpustate_setgs(_state, v)
#endif /* !EMU86_GETGS */

/* Notify that the instruction is about to push/pop the indicated number of bytes.
 * @param: new_sp: The new SP value (offset from `EMU86_GETSTACKPTR()')
 * @param: old_sp: The old SP value (as returned by `EMU86_GETSTACKPTR()') */
#ifndef EMU86_EMULATE_PUSH
#define EMU86_EMULATE_PUSH(new_sp, num_bytes) (void)0
#endif /* !EMU86_EMULATE_PUSH */
#ifndef EMU86_EMULATE_POP
#define EMU86_EMULATE_POP(old_sp, num_bytes) (void)0
#endif /* !EMU86_EMULATE_POP */

/* Verify user access to a specified I/O port range. (requires `EMU86_EMULATE_CONFIG_CHECKUSER') */
#ifndef EMU86_VALIDATE_IO
#define EMU86_VALIDATE_IO(portno, num_ports) (void)0
#define EMU86_VALIDATE_IO_IS_NOOP 1
#endif /* !EMU86_VALIDATE_IO */

/* Access I/O ports. */
#ifndef EMU86_EMULATE_INB
#include <sys/io.h>
#define EMU86_EMULATE_INB(portno, result) do { (result) = inb((__port_t)(portno)); } __WHILE0
#define EMU86_EMULATE_INW(portno, result) do { (result) = inw((__port_t)(portno)); } __WHILE0
#define EMU86_EMULATE_INL(portno, result) do { (result) = inl((__port_t)(portno)); } __WHILE0
#define EMU86_EMULATE_OUTB(portno, value) do { outb((__port_t)(portno), value); } __WHILE0
#define EMU86_EMULATE_OUTW(portno, value) do { outw((__port_t)(portno), value); } __WHILE0
#define EMU86_EMULATE_OUTL(portno, value) do { outl((__port_t)(portno), value); } __WHILE0
#ifdef __KERNEL__
/* Define these to allow for some minor optimizations, when it is known
 * that in/out emulation can never  throw an exception (Note that  this
 * isn't the case for user-space,  where use of these instructions  can
 * result in an `E_ILLEGAL_INSTRUCTION_PRIVILEGED_OPCODE' exception) */
#define EMU86_EMULATE_IN_IS_NOEXCEPT 1
#define EMU86_EMULATE_OUT_IS_NOEXCEPT 1
#endif /* __KERNEL__ */
#endif /* !EMU86_EMULATE_INB */







/* Get a segment register by index
 * @param: regno: One of `EMU86_R_ES', `EMU86_R_CS', ... */
#ifndef EMU86_GETSEG
#define EMU86_GETSEG(regno) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getseg)(EMU86_EMULATE_HELPER_PARAM_ regno))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u16
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getseg))(EMU86_EMULATE_HELPER_ARGS_
                                                                                u8 regno) {
	u16 result;
	switch (regno) {
	case EMU86_R_ES: result = EMU86_GETES(); break;
	case EMU86_R_CS: result = EMU86_GETCS(); break;
	case EMU86_R_SS: result = EMU86_GETSS(); break;
	case EMU86_R_DS: result = EMU86_GETDS(); break;
	case EMU86_R_FS: result = EMU86_GETFS(); break;
	case EMU86_R_GS: result = EMU86_GETGS(); break;
	default: __builtin_unreachable();
	}
	return result;
}
#endif /* !EMU86_GETSEG */

/* Set a segment register by index
 * @param: regno: One of `EMU86_R_ES', `EMU86_R_CS', ... */
#ifndef EMU86_SETSEG
#define EMU86_SETSEG(regno, value) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setseg)(EMU86_EMULATE_HELPER_PARAM_ regno, value))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setseg))(EMU86_EMULATE_HELPER_ARGS_
                                                                                u8 regno, u16 value) {
	switch (regno) {
	case EMU86_R_ES: EMU86_SETES(value); break;
	case EMU86_R_CS: EMU86_SETCS(value); break;
	case EMU86_R_SS: EMU86_SETSS(value); break;
	case EMU86_R_DS: EMU86_SETDS(value); break;
	case EMU86_R_FS: EMU86_SETFS(value); break;
	case EMU86_R_GS: EMU86_SETGS(value); break;
	default: __builtin_unreachable();
	}
}
#endif /* !EMU86_SETSEG */



/* Get/set a general purpose register by index
 * @param: regno:    One of `EMU86_R_AL', `EMU86_R_CL', ...
 * @param: op_flags: Set of `0 | EMU86_F_HASREX' (only for 8-bit registers when 64-bit support is enabled) */
#if !defined(EMU86_GETREGB) || !defined(EMU86_SETREGB)
#ifdef __x86_64__
PRIVATE u8 const EMU86_EMULATE_HELPER_NAME(emu86_regb_offsets)[16] = {
	/* [EMU86_R_AL]   = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RAX,     /* %al */
	/* [EMU86_R_CL]   = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RCX,     /* %cl */
	/* [EMU86_R_DL]   = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RDX,     /* %dl */
	/* [EMU86_R_BL]   = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RBX,     /* %bl */
	/* [EMU86_R_AH]   = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RAX + 1, /* %ah */
	/* [EMU86_R_CH]   = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RCX + 1, /* %ch */
	/* [EMU86_R_DH]   = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RDX + 1, /* %dh */
	/* [EMU86_R_BH]   = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RBX + 1, /* %bh */
	/* [EMU86_R_R8L]  = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R8,      /* %r8l */
	/* [EMU86_R_R9L]  = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R9,      /* %r9l */
	/* [EMU86_R_R10L] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R10,     /* %r10l */
	/* [EMU86_R_R11L] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R11,     /* %r11l */
	/* [EMU86_R_R12L] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R12,     /* %r12l */
	/* [EMU86_R_R13L] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R13,     /* %r13l */
	/* [EMU86_R_R14L] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R14,     /* %r14l */
	/* [EMU86_R_R15L] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R15,     /* %r15l */
};
#else /* __x86_64__ */
PRIVATE u8 const EMU86_EMULATE_HELPER_NAME(emu86_regb_offsets)[8] = {
	/* [EMU86_R_AL] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EAX,     /* %al */
	/* [EMU86_R_CL] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_ECX,     /* %cl */
	/* [EMU86_R_DL] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EDX,     /* %dl */
	/* [EMU86_R_BL] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EBX,     /* %bl */
	/* [EMU86_R_AH] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EAX + 1, /* %ah */
	/* [EMU86_R_CH] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_ECX + 1, /* %ch */
	/* [EMU86_R_DH] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EDX + 1, /* %dh */
	/* [EMU86_R_BH] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EBX + 1, /* %bh */
};
#endif /* !__x86_64__ */
#endif /* !EMU86_GETREGB || !EMU86_SETREGB */

#if (!defined(EMU86_GETREGW) || !defined(EMU86_SETREGW) ||   \
     !defined(EMU86_GETREGL) || !defined(EMU86_SETREGL) ||   \
     (LIBEMU86_CONFIG_WANT_64BIT &&                          \
      (!defined(EMU86_GETREGB) || !defined(EMU86_SETREGB) || \
       !defined(EMU86_GETREGQ) || !defined(EMU86_SETREGQ))))
#ifdef __x86_64__
PRIVATE u8 const EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[16] = {
	/* [EMU86_R_RAX] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RAX, /* %rax */
	/* [EMU86_R_RCX] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RCX, /* %rcx */
	/* [EMU86_R_RDX] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RDX, /* %rdx */
	/* [EMU86_R_RBX] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RBX, /* %rbx */
	/* [EMU86_R_RSP] = */ 0,                                                 /* %rsp */
	/* [EMU86_R_RBP] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RBP, /* %rbp */
	/* [EMU86_R_RSI] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RSI, /* %rsi */
	/* [EMU86_R_RDI] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_RDI, /* %rdi */
	/* [EMU86_R_R8]  = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R8,  /* %r8 */
	/* [EMU86_R_R9]  = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R9,  /* %r9 */
	/* [EMU86_R_R10] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R10, /* %r10 */
	/* [EMU86_R_R11] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R11, /* %r11 */
	/* [EMU86_R_R12] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R12, /* %r12 */
	/* [EMU86_R_R13] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R13, /* %r13 */
	/* [EMU86_R_R14] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R14, /* %r14 */
	/* [EMU86_R_R15] = */ OFFSET_ICPUSTATE_GPREGSNSP + OFFSET_GPREGSNSP_R15, /* %r15 */
};
#else /* __x86_64__ */
PRIVATE u8 const EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[8] = {
	/* [EMU86_R_EAX] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EAX, /* %eax */
	/* [EMU86_R_ECX] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_ECX, /* %ecx */
	/* [EMU86_R_EDX] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EDX, /* %edx */
	/* [EMU86_R_EBX] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EBX, /* %ebx */
	/* [EMU86_R_ESP] = */ 0,                                           /* %esp */
	/* [EMU86_R_EBP] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EBP, /* %ebp */
	/* [EMU86_R_ESI] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_ESI, /* %esi */
	/* [EMU86_R_EDI] = */ OFFSET_ICPUSTATE_GPREGS + OFFSET_GPREGS_EDI, /* %edi */
};
#endif /* !__x86_64__ */
#endif /* Need `emu86_reg_offsets' */

#ifdef __x86_64__
#define __EMU86_GPREG_MASK 0xf
#else /* __x86_64__ */
#define __EMU86_GPREG_MASK 0x7
#endif /* !__x86_64__ */


#ifndef EMU86_GETREGB
#if LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_GETREGB(regno, op_flags) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getregb)(EMU86_EMULATE_HELPER_PARAM_ regno, op_flags))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u8
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getregb))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno,
                                                                                 emu86_opflags_t op_flags)
#else /* LIBEMU86_CONFIG_WANT_64BIT */
#define EMU86_GETREGB(regno, ...) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getregb)(EMU86_EMULATE_HELPER_PARAM_ regno))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u8
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getregb))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno)
#endif /* !LIBEMU86_CONFIG_WANT_64BIT */
{
	regno &= __EMU86_GPREG_MASK;
#if LIBEMU86_CONFIG_WANT_64BIT
	if (op_flags & EMU86_F_HASREX) {
		if (regno == EMU86_R_SPL)
			return (u8)EMU86_GETSPREG();
		return *(u8 const *)((byte_t const *)_state + EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[regno]);
	}
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
	return *(u8 const *)((byte_t const *)_state + EMU86_EMULATE_HELPER_NAME(emu86_regb_offsets)[regno]);
}
#endif /* !EMU86_GETREGB */

#ifndef EMU86_SETREGB
#if LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_SETREGB(regno, value, op_flags) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setregb)(EMU86_EMULATE_HELPER_PARAM_ regno, value, op_flags))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setregb))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno, u8 value,
                                                                                 emu86_opflags_t op_flags)
#else /* LIBEMU86_CONFIG_WANT_64BIT */
#define EMU86_SETREGB(regno, value, ...) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setregb)(EMU86_EMULATE_HELPER_PARAM_ regno, value))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setregb))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno, u8 value)
#endif /* !LIBEMU86_CONFIG_WANT_64BIT */
{
	regno &= __EMU86_GPREG_MASK;
#if LIBEMU86_CONFIG_WANT_64BIT
	if (op_flags & EMU86_F_HASREX) {
		if (regno == EMU86_R_SPL) {
			EMU86_SETSPREG((EMU86_GETSPREG() & ~0xff) | value);
		} else {
			*(u8 *)((byte_t *)_state + EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[regno]) = value;
		}
	} else
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
	{
		*(u8 *)((byte_t *)_state + EMU86_EMULATE_HELPER_NAME(emu86_regb_offsets)[regno]) = value;
	}
}
#endif /* !EMU86_SETREGB */

#ifndef EMU86_GETREGW
#define EMU86_GETREGW(regno) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getregw)(EMU86_EMULATE_HELPER_PARAM_ regno))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u16
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getregw))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno) {
	regno &= __EMU86_GPREG_MASK;
	if (regno == EMU86_R_SP)
		return EMU86_GETSPREG();
	return (u16)(*(uintptr_t const *)((byte_t *)_state + EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[regno]));
}
#endif /* !EMU86_GETREGW */

#ifndef EMU86_SETREGW
#define EMU86_SETREGW(regno, value) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setregw)(EMU86_EMULATE_HELPER_PARAM_ regno, value))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setregw))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno, u16 value) {
	regno &= __EMU86_GPREG_MASK;
	if (regno == EMU86_R_SP) {
		EMU86_SETSPREG(value);
	} else {
		*(uintptr_t *)((byte_t *)_state + EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[regno]) = (uintptr_t)value;
	}
}
#endif /* !EMU86_SETREGW */

#ifndef EMU86_GETREGL
#define EMU86_GETREGL(regno) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getregl)(EMU86_EMULATE_HELPER_PARAM_ regno))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u32
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getregl))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno) {
	regno &= __EMU86_GPREG_MASK;
	if (regno == EMU86_R_SP)
		return (u32)EMU86_GETSPREG();
	return (u32)(*(uintptr_t const *)((byte_t const *)_state + EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[regno]));
}
#endif /* !EMU86_GETREGL */

#ifndef EMU86_SETREGL
#define EMU86_SETREGL(regno, value) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setregl)(EMU86_EMULATE_HELPER_PARAM_ regno, value))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setregl))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno, u32 value) {
	regno &= __EMU86_GPREG_MASK;
	if (regno == EMU86_R_SP) {
		EMU86_SETSPREG(value);
	} else {
		*(uintptr_t *)((byte_t *)_state + EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[regno]) = (uintptr_t)value;
	}
}
#endif /* !EMU86_SETREGL */

#if LIBEMU86_CONFIG_WANT_64BIT
#ifndef EMU86_GETREGQ
#define EMU86_GETREGQ(regno) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getregq)(EMU86_EMULATE_HELPER_PARAM_ regno))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u64
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getregq))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno) {
	regno &= __EMU86_GPREG_MASK;
	if (regno == EMU86_R_SP)
		return (u64)EMU86_GETSPREG();
	return (u64)(*(uintptr_t const *)((byte_t const *)_state + EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[regno]));
}
#endif /* !EMU86_GETREGQ */

#ifndef EMU86_SETREGQ
#define EMU86_SETREGQ(regno, value) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setregq)(EMU86_EMULATE_HELPER_PARAM_ regno, value))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setregq))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno, u64 value) {
	regno &= __EMU86_GPREG_MASK;
	if (regno == EMU86_R_SP) {
		EMU86_SETSPREG(value);
	} else {
		*(uintptr_t *)((byte_t *)_state + EMU86_EMULATE_HELPER_NAME(emu86_reg_offsets)[regno]) = (uintptr_t)value;
	}
}
#endif /* !EMU86_SETREGQ */
#endif /* LIBEMU86_CONFIG_WANT_64BIT */


/* Get/set a pointer-sized general purpose register by index
 * @param: regno: One of `EMU86_R_AX', `EMU86_R_EAX', `EMU86_R_RAX', `EMU86_R_CX', ... */
#ifndef EMU86_GETREGP
#ifndef LIBEMU86_CONFIG_NEED_ARCHMODE
#if LIBEMU86_CONFIG_WANT_16BIT
#define EMU86_GETREGP(regno, ...) EMU86_GETREGW(regno)
#elif LIBEMU86_CONFIG_WANT_32BIT
#define EMU86_GETREGP(regno, ...) EMU86_GETREGL(regno)
#elif LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_GETREGP(regno, ...) EMU86_GETREGQ(regno)
#else /* ... */
#error "Invalid configuration"
#endif /* !... */
#else /* !LIBEMU86_CONFIG_NEED_ARCHMODE */
#define EMU86_GETREGP(regno, op_flags) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getregp)(EMU86_EMULATE_HELPER_PARAM_ regno, op_flags))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR uintptr_t
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getregp))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno,
                                                                                 emu86_opflags_t op_flags) {
#if LIBEMU86_CONFIG_WANT_64BIT
	if (EMU86_F_IS64(op_flags))
		return (uintptr_t)EMU86_GETREGQ(regno);
#if LIBEMU86_CONFIG_WANT_16BIT && LIBEMU86_CONFIG_WANT_32BIT
	if (EMU86_F_IS32(op_flags))
		return (uintptr_t)EMU86_GETREGL(regno);
	return (uintptr_t)EMU86_GETREGW(regno);
#elif LIBEMU86_CONFIG_WANT_16BIT
	return (uintptr_t)EMU86_GETREGW(regno);
#elif LIBEMU86_CONFIG_WANT_32BIT
	return (uintptr_t)EMU86_GETREGL(regno);
#else
#error "Invalid configuration"
#endif
#else /* LIBEMU86_CONFIG_WANT_64BIT */
	if (EMU86_F_IS32(op_flags))
		return (uintptr_t)EMU86_GETREGL(regno);
	return (uintptr_t)EMU86_GETREGW(regno);
#endif /* !LIBEMU86_CONFIG_WANT_64BIT */
}
#endif /* LIBEMU86_CONFIG_NEED_ARCHMODE */
#endif /* !EMU86_GETREGP */

#ifndef EMU86_SETREGP
#ifndef LIBEMU86_CONFIG_NEED_ARCHMODE
#if LIBEMU86_CONFIG_WANT_16BIT
#define EMU86_SETREGP(regno, value, ...) EMU86_SETREGW(regno, value)
#elif LIBEMU86_CONFIG_WANT_32BIT
#define EMU86_SETREGP(regno, value, ...) EMU86_SETREGL(regno, value)
#elif LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_SETREGP(regno, value, ...) EMU86_SETREGQ(regno, value)
#else /* ... */
#error "Invalid configuration"
#endif /* !... */
#else /* !LIBEMU86_CONFIG_NEED_ARCHMODE */
#define EMU86_SETREGP(regno, value, op_flags) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setregp)(EMU86_EMULATE_HELPER_PARAM_ regno, value, op_flags))
EMU86_EMULATE_HELPER_DECL EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setregp))(EMU86_EMULATE_HELPER_ARGS_
                                                                                 u8 regno, uintptr_t value,
                                                                                 emu86_opflags_t op_flags) {
#if LIBEMU86_CONFIG_WANT_64BIT
	if (EMU86_F_IS64(op_flags)) {
		EMU86_SETREGQ(regno, (u64)value);
		return;
	}
#if LIBEMU86_CONFIG_WANT_16BIT && LIBEMU86_CONFIG_WANT_32BIT
	if (EMU86_F_IS32(op_flags)) {
		EMU86_SETREGL(regno, (u32)value);
		return;
	}
	EMU86_SETREGW(regno, (u16)value);
#elif LIBEMU86_CONFIG_WANT_16BIT
	EMU86_SETREGW(regno, (u16)value);
#elif LIBEMU86_CONFIG_WANT_32BIT
	EMU86_SETREGL(regno, (u32)value);
#else
#error "Invalid configuration"
#endif
#else /* LIBEMU86_CONFIG_WANT_64BIT */
	if (EMU86_F_IS32(op_flags)) {
		EMU86_SETREGL(regno, (u32)value);
	} else {
		EMU86_SETREGW(regno, (u16)value);
	}
#endif /* !LIBEMU86_CONFIG_WANT_64BIT */
}
#endif /* LIBEMU86_CONFIG_NEED_ARCHMODE */
#endif /* !EMU86_SETREGP */


/* Get/set the %(|e|r)ip register */
#ifndef EMU86_GETIP
#define EMU86_GETIP()  ((u16)EMU86_GETIPREG())
#define EMU86_SETIP(v) EMU86_SETIPREG((u16)(v))
#endif /* !EMU86_GETIP */
#ifndef EMU86_GETEIP
#define EMU86_GETEIP()  ((u32)EMU86_GETIPREG())
#define EMU86_SETEIP(v) EMU86_SETIPREG((u32)(v))
#endif /* !EMU86_GETEIP */
#if LIBEMU86_CONFIG_WANT_64BIT
#ifndef EMU86_GETRIP
#define EMU86_GETRIP()  ((u64)EMU86_GETIPREG())
#define EMU86_SETRIP(v) EMU86_SETIPREG((u64)(v))
#endif /* !EMU86_GETRIP */
#endif /* LIBEMU86_CONFIG_WANT_64BIT */

/* Get/set specific registers */
#ifndef EMU86_GETAL
#define EMU86_GETAL() EMU86_GETREGB(EMU86_R_AL, 0)
#define EMU86_GETCL() EMU86_GETREGB(EMU86_R_CL, 0)
#define EMU86_GETDL() EMU86_GETREGB(EMU86_R_DL, 0)
#define EMU86_GETBL() EMU86_GETREGB(EMU86_R_BL, 0)
#endif /* !EMU86_GETAL */
#ifndef EMU86_SETAL
#define EMU86_SETAL(value) EMU86_SETREGB(EMU86_R_AL, value, 0)
#define EMU86_SETCL(value) EMU86_SETREGB(EMU86_R_CL, value, 0)
#define EMU86_SETDL(value) EMU86_SETREGB(EMU86_R_DL, value, 0)
#define EMU86_SETBL(value) EMU86_SETREGB(EMU86_R_BL, value, 0)
#endif /* !EMU86_SETAL */
#ifndef EMU86_GETAH
#define EMU86_GETAH() EMU86_GETREGB(EMU86_R_AH, 0)
#define EMU86_GETCH() EMU86_GETREGB(EMU86_R_CH, 0)
#define EMU86_GETDH() EMU86_GETREGB(EMU86_R_DH, 0)
#define EMU86_GETBH() EMU86_GETREGB(EMU86_R_BH, 0)
#endif /* !EMU86_GETAH */
#ifndef EMU86_SETAH
#define EMU86_SETAH(value) EMU86_SETREGB(EMU86_R_AH, value, 0)
#define EMU86_SETCH(value) EMU86_SETREGB(EMU86_R_CH, value, 0)
#define EMU86_SETDH(value) EMU86_SETREGB(EMU86_R_DH, value, 0)
#define EMU86_SETBH(value) EMU86_SETREGB(EMU86_R_BH, value, 0)
#endif /* !EMU86_SETAH */
#ifndef EMU86_GETAX
#define EMU86_GETAX() EMU86_GETREGW(EMU86_R_AX)
#define EMU86_GETCX() EMU86_GETREGW(EMU86_R_CX)
#define EMU86_GETDX() EMU86_GETREGW(EMU86_R_DX)
#define EMU86_GETBX() EMU86_GETREGW(EMU86_R_BX)
#define EMU86_GETSP() EMU86_GETREGW(EMU86_R_SP)
#define EMU86_GETBP() EMU86_GETREGW(EMU86_R_BP)
#define EMU86_GETSI() EMU86_GETREGW(EMU86_R_SI)
#define EMU86_GETDI() EMU86_GETREGW(EMU86_R_DI)
#endif /* !EMU86_GETAX */
#ifndef EMU86_SETAX
#define EMU86_SETAX(value) EMU86_SETREGW(EMU86_R_AX, value)
#define EMU86_SETCX(value) EMU86_SETREGW(EMU86_R_CX, value)
#define EMU86_SETDX(value) EMU86_SETREGW(EMU86_R_DX, value)
#define EMU86_SETBX(value) EMU86_SETREGW(EMU86_R_BX, value)
#define EMU86_SETSP(value) EMU86_SETREGW(EMU86_R_SP, value)
#define EMU86_SETBP(value) EMU86_SETREGW(EMU86_R_BP, value)
#define EMU86_SETSI(value) EMU86_SETREGW(EMU86_R_SI, value)
#define EMU86_SETDI(value) EMU86_SETREGW(EMU86_R_DI, value)
#endif /* !EMU86_SETAX */
#ifndef EMU86_GETEAX
#define EMU86_GETEAX() EMU86_GETREGL(EMU86_R_EAX)
#define EMU86_GETECX() EMU86_GETREGL(EMU86_R_ECX)
#define EMU86_GETEDX() EMU86_GETREGL(EMU86_R_EDX)
#define EMU86_GETEBX() EMU86_GETREGL(EMU86_R_EBX)
#define EMU86_GETESP() EMU86_GETREGL(EMU86_R_ESP)
#define EMU86_GETEBP() EMU86_GETREGL(EMU86_R_EBP)
#define EMU86_GETESI() EMU86_GETREGL(EMU86_R_ESI)
#define EMU86_GETEDI() EMU86_GETREGL(EMU86_R_EDI)
#endif /* !EMU86_GETEAX */
#ifndef EMU86_SETEAX
#define EMU86_SETEAX(value) EMU86_SETREGL(EMU86_R_EAX, value)
#define EMU86_SETECX(value) EMU86_SETREGL(EMU86_R_ECX, value)
#define EMU86_SETEDX(value) EMU86_SETREGL(EMU86_R_EDX, value)
#define EMU86_SETEBX(value) EMU86_SETREGL(EMU86_R_EBX, value)
#define EMU86_SETESP(value) EMU86_SETREGL(EMU86_R_ESP, value)
#define EMU86_SETEBP(value) EMU86_SETREGL(EMU86_R_EBP, value)
#define EMU86_SETESI(value) EMU86_SETREGL(EMU86_R_ESI, value)
#define EMU86_SETEDI(value) EMU86_SETREGL(EMU86_R_EDI, value)
#endif /* !EMU86_SETEAX */
#if LIBEMU86_CONFIG_WANT_64BIT
#ifndef EMU86_GETSPL
#define EMU86_GETSPL() EMU86_GETREGB(EMU86_R_SP, EMU86_F_HASREX)
#define EMU86_GETBPL() EMU86_GETREGB(EMU86_R_BP, EMU86_F_HASREX)
#define EMU86_GETSIL() EMU86_GETREGB(EMU86_R_SI, EMU86_F_HASREX)
#define EMU86_GETDIL() EMU86_GETREGB(EMU86_R_DI, EMU86_F_HASREX)
#endif /* !EMU86_GETSPL */
#ifndef EMU86_SETSPL
#define EMU86_SETSPL(value) EMU86_SETREGB(EMU86_R_SP, value, EMU86_F_HASREX)
#define EMU86_SETBPL(value) EMU86_SETREGB(EMU86_R_BP, value, EMU86_F_HASREX)
#define EMU86_SETSIL(value) EMU86_SETREGB(EMU86_R_SI, value, EMU86_F_HASREX)
#define EMU86_SETDIL(value) EMU86_SETREGB(EMU86_R_DI, value, EMU86_F_HASREX)
#endif /* !EMU86_SETSPL */
#ifndef EMU86_GETRAX
#define EMU86_GETRAX() EMU86_GETREGQ(EMU86_R_RAX)
#define EMU86_GETRCX() EMU86_GETREGQ(EMU86_R_RCX)
#define EMU86_GETRDX() EMU86_GETREGQ(EMU86_R_RDX)
#define EMU86_GETRBX() EMU86_GETREGQ(EMU86_R_RBX)
#define EMU86_GETRSP() EMU86_GETREGQ(EMU86_R_RSP)
#define EMU86_GETRBP() EMU86_GETREGQ(EMU86_R_RBP)
#define EMU86_GETRSI() EMU86_GETREGQ(EMU86_R_RSI)
#define EMU86_GETRDI() EMU86_GETREGQ(EMU86_R_RDI)
#endif /* !EMU86_GETRAX */
#ifndef EMU86_SETRAX
#define EMU86_SETRAX(value) EMU86_SETREGQ(EMU86_R_RAX, value)
#define EMU86_SETRCX(value) EMU86_SETREGQ(EMU86_R_RCX, value)
#define EMU86_SETRDX(value) EMU86_SETREGQ(EMU86_R_RDX, value)
#define EMU86_SETRBX(value) EMU86_SETREGQ(EMU86_R_RBX, value)
#define EMU86_SETRSP(value) EMU86_SETREGQ(EMU86_R_RSP, value)
#define EMU86_SETRBP(value) EMU86_SETREGQ(EMU86_R_RBP, value)
#define EMU86_SETRSI(value) EMU86_SETREGQ(EMU86_R_RSI, value)
#define EMU86_SETRDI(value) EMU86_SETREGQ(EMU86_R_RDI, value)
#endif /* !EMU86_SETRAX */
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#ifndef EMU86_GETPAX
#define EMU86_GETPAX(op_flags) EMU86_GETREGP(EMU86_R_AX, op_flags)
#define EMU86_GETPCX(op_flags) EMU86_GETREGP(EMU86_R_CX, op_flags)
#define EMU86_GETPDX(op_flags) EMU86_GETREGP(EMU86_R_DX, op_flags)
#define EMU86_GETPBX(op_flags) EMU86_GETREGP(EMU86_R_BX, op_flags)
#define EMU86_GETPSP(op_flags) EMU86_GETREGP(EMU86_R_SP, op_flags)
#define EMU86_GETPBP(op_flags) EMU86_GETREGP(EMU86_R_BP, op_flags)
#define EMU86_GETPSI(op_flags) EMU86_GETREGP(EMU86_R_SI, op_flags)
#define EMU86_GETPDI(op_flags) EMU86_GETREGP(EMU86_R_DI, op_flags)
#endif /* !EMU86_GETPAX */
#ifndef EMU86_SETPAX
#define EMU86_SETPAX(value, op_flags) EMU86_SETREGP(EMU86_R_AX, value, op_flags)
#define EMU86_SETPCX(value, op_flags) EMU86_SETREGP(EMU86_R_CX, value, op_flags)
#define EMU86_SETPDX(value, op_flags) EMU86_SETREGP(EMU86_R_DX, value, op_flags)
#define EMU86_SETPBX(value, op_flags) EMU86_SETREGP(EMU86_R_BX, value, op_flags)
#define EMU86_SETPSP(value, op_flags) EMU86_SETREGP(EMU86_R_SP, value, op_flags)
#define EMU86_SETPBP(value, op_flags) EMU86_SETREGP(EMU86_R_BP, value, op_flags)
#define EMU86_SETPSI(value, op_flags) EMU86_SETREGP(EMU86_R_SI, value, op_flags)
#define EMU86_SETPDI(value, op_flags) EMU86_SETREGP(EMU86_R_DI, value, op_flags)
#endif /* !EMU86_SETPAX */



/* Calculate the memory address of a given modrm */
#ifdef EMU86_MODRM_MEMADDR
#ifndef EMU86_MODRM_MEMADDR_NEED_REAL_IP
#define EMU86_MODRM_MEMADDR_NEED_REAL_IP LIBEMU86_CONFIG_WANT_64BIT
#endif /* !EMU86_MODRM_MEMADDR_NEED_REAL_IP */
#else /* EMU86_MODRM_MEMADDR */
#undef EMU86_MODRM_MEMADDR_NEED_REAL_IP
#define EMU86_MODRM_MEMADDR_NEED_REAL_IP LIBEMU86_CONFIG_WANT_64BIT
#if LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_modrm_memaddr)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags, real_ip))
#else /* LIBEMU86_CONFIG_WANT_64BIT */
#define EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_modrm_memaddr)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags))
#endif /* !LIBEMU86_CONFIG_WANT_64BIT */
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR byte_t *
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_modrm_memaddr))(EMU86_EMULATE_HELPER_ARGS_
                                                                                       struct emu86_modrm const *__restrict modrm,
                                                                                       emu86_opflags_t op_flags
#if LIBEMU86_CONFIG_WANT_64BIT
                                                                                       ,
                                                                                       uintptr_t real_ip
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
                                                                                       ) {
	uintptr_t result;
	(void)op_flags;
	result = modrm->mi_offset;
	if (modrm->mi_rm != 0xff) {
#if LIBEMU86_CONFIG_WANT_64BIT
		if (modrm->mi_rm == EMU86_R_RIP) {
#if __SIZEOF_POINTER__ > 4
			if unlikely(op_flags & EMU86_F_67) {
				result += real_ip & UINT32_C(0xffffffff);
			} else
#endif /* __SIZEOF_POINTER__ > 4 */
			{
				result += real_ip;
			}
		} else
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
		{
			result += EMU86_GETREGP(modrm->mi_rm, op_flags);
		}
	}
	if (modrm->mi_index != 0xff)
		result += EMU86_GETREGP(modrm->mi_index, op_flags) << modrm->mi_shift;

	/* Mask to 16 bits _BEFORE_ segmentation.
	 * -> Realmode is limited to 1MiB with segmentation and 64KiB without */
#if __SIZEOF_POINTER__ > 2 && LIBEMU86_CONFIG_WANT_16BIT
	if (EMU86_F_IS16(op_flags))
		result &= UINT16_C(0xffff);
#endif /* __SIZEOF_POINTER__ > 2 && LIBEMU86_CONFIG_WANT_16BIT */

	/* Form full (segmented) memory address */
#ifndef EMU86_GETSEGBASE_IS_NOOP_ALL
	{
		u8 reg = EMU86_F_SEGREG(op_flags);
		if (!EMU86_F_HASSEG(op_flags)) {
			reg = EMU86_R_DS;
#if LIBEMU86_CONFIG_WANT_16BIT || LIBEMU86_CONFIG_WANT_32BIT
			/* When  no  segment  override  is  given,  and  `modrm->mi_rm'  is
			 * either ESP or EBP, then %ss must be used; else %ds must be used. */
			if (modrm->mi_rm == EMU86_R_BP || modrm->mi_rm == EMU86_R_SP)
				reg = EMU86_R_SS;
#endif /* LIBEMU86_CONFIG_WANT_16BIT || LIBEMU86_CONFIG_WANT_32BIT */
		}
		if (!EMU86_GETSEGBASE_IS_NOOP(reg)) {
			uintptr_t segment_base;
			segment_base = (uintptr_t)EMU86_GETSEGBASE(reg);
			result       = (uintptr_t)EMU86_SEGADDR(segment_base, result);
		}
	}
#endif /* !EMU86_GETSEGBASE_IS_NOOP_ALL */

	/* Mask to 32 bits _AFTER_ segmentation.
	 * -> 32-bit protected mode can never access more than 4GiB, even when using segmentation! */
#if __SIZEOF_POINTER__ > 4 && LIBEMU86_CONFIG_WANT_32BIT
	if (EMU86_F_IS32(op_flags))
		result &= UINT32_C(0xffffffff);
#endif /* __SIZEOF_POINTER__ > 4 && LIBEMU86_CONFIG_WANT_32BIT */

	return (byte_t *)result;
}
#endif /* !EMU86_MODRM_MEMADDR */

#undef EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP
#if EMU86_MODRM_MEMADDR_NEED_REAL_IP
#define EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(...) __VA_ARGS__
#else /* EMU86_MODRM_MEMADDR_NEED_REAL_IP */
#define EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(...) /* nothing */
#endif /* !EMU86_MODRM_MEMADDR_NEED_REAL_IP */

#ifndef EMU86_MODRM_MEMADDR_NOSEGBASE
#ifdef EMU86_GETSEGBASE_IS_NOOP_ALL
#define EMU86_MODRM_MEMADDR_NOSEGBASE(modrm, op_flags, real_ip) \
	EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip)
#else /* EMU86_GETSEGBASE_IS_NOOP_ALL */
#define EMU86_MODRM_MEMADDR_NOSEGBASE(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_modrm_memaddr_nosegbase)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR byte_t *
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_modrm_memaddr_nosegbase))(EMU86_EMULATE_HELPER_ARGS_
                                                                                                 struct emu86_modrm const *__restrict modrm,
                                                                                                 emu86_opflags_t op_flags
                                                                                                 EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	uintptr_t result;
	(void)op_flags;
	result = modrm->mi_offset;
	if (modrm->mi_rm != 0xff) {
#if EMU86_MODRM_MEMADDR_NEED_REAL_IP
		if (modrm->mi_rm == EMU86_R_RIP) {
#if __SIZEOF_POINTER__ > 4
			if unlikely(op_flags & EMU86_F_67) {
				result += real_ip & UINT32_C(0xffffffff);
			} else
#endif /* __SIZEOF_POINTER__ > 4 */
			{
				result += real_ip;
			}
		} else
#endif /* EMU86_MODRM_MEMADDR_NEED_REAL_IP */
		{
			result += EMU86_GETREGP(modrm->mi_rm, op_flags);
		}
	}
	if (modrm->mi_index != 0xff)
		result += EMU86_GETREGP(modrm->mi_index, op_flags) << modrm->mi_shift;
#if __SIZEOF_POINTER__ > 4 && LIBEMU86_CONFIG_WANT_32BIT
	if (EMU86_F_IS32(op_flags))
		result &= UINT32_C(0xffffffff);
#endif /* __SIZEOF_POINTER__ > 4 && LIBEMU86_CONFIG_WANT_32BIT */
#if __SIZEOF_POINTER__ > 2 && LIBEMU86_CONFIG_WANT_16BIT
	if (EMU86_F_IS16(op_flags))
		result &= UINT16_C(0xffff);
#endif /* __SIZEOF_POINTER__ > 2 && LIBEMU86_CONFIG_WANT_16BIT */
	return (byte_t *)result;
}
#endif /* !EMU86_GETSEGBASE_IS_NOOP_ALL */
#endif /* !EMU86_MODRM_MEMADDR_NOSEGBASE */





/* Get/Set the R/M operand from a given modrm */
#ifndef EMU86_GETMODRM_RMB
#define EMU86_GETMODRM_RMB(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmb)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u8
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmb))(EMU86_EMULATE_HELPER_ARGS_
                                                                                      struct emu86_modrm const *__restrict modrm,
                                                                                      emu86_opflags_t op_flags
                                                                                      EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
	if (EMU86_MODRM_ISREG(modrm->mi_type))
		return EMU86_GETREGB(modrm->mi_rm, op_flags);
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_READABLE(addr, 1);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
	return EMU86_MEMREADB(addr);
}
#endif /* !EMU86_GETMODRM_RMB */

#ifndef EMU86_SETMODRM_RMB
#define EMU86_SETMODRM_RMB(modrm, value, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmb)(EMU86_EMULATE_HELPER_PARAM_ modrm, value, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmb))(EMU86_EMULATE_HELPER_ARGS_
                                                                                      struct emu86_modrm const *__restrict modrm,
                                                                                      u8 value, emu86_opflags_t op_flags
                                                                                      EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
	if (EMU86_MODRM_ISREG(modrm->mi_type)) {
		EMU86_SETREGB(modrm->mi_rm, value, op_flags);
	} else
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
	{
		byte_t *addr;
		addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
		if (EMU86_ISUSER())
			EMU86_VALIDATE_WRITABLE(addr, 1);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
		EMU86_MEMWRITEB(addr, value);
	}
}
#endif /* !EMU86_SETMODRM_RMB */

#ifndef EMU86_GETMODRM_RMW
#define EMU86_GETMODRM_RMW(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmw)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u16
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmw))(EMU86_EMULATE_HELPER_ARGS_
                                                                                      struct emu86_modrm const *__restrict modrm,
                                                                                      emu86_opflags_t op_flags
                                                                                      EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
	if (EMU86_MODRM_ISREG(modrm->mi_type))
		return EMU86_GETREGW(modrm->mi_rm);
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_READABLE(addr, 2);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
	return EMU86_MEMREADW(addr);
}
#endif /* !EMU86_GETMODRM_RMW */

#ifndef EMU86_SETMODRM_RMW
#define EMU86_SETMODRM_RMW(modrm, value, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmw)(EMU86_EMULATE_HELPER_PARAM_ modrm, value, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmw))(EMU86_EMULATE_HELPER_ARGS_
                                                                                      struct emu86_modrm const *__restrict modrm,
                                                                                      u16 value, emu86_opflags_t op_flags
                                                                                      EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
	if (EMU86_MODRM_ISREG(modrm->mi_type)) {
		EMU86_SETREGW(modrm->mi_rm, value);
	} else
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
	{
		byte_t *addr;
		addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
		if (EMU86_ISUSER())
			EMU86_VALIDATE_WRITABLE(addr, 2);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
		EMU86_MEMWRITEW(addr, value);
	}
}
#endif /* !EMU86_SETMODRM_RMW */

#ifndef EMU86_GETMODRM_RML
#define EMU86_GETMODRM_RML(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rml)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u32
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rml))(EMU86_EMULATE_HELPER_ARGS_
                                                                                      struct emu86_modrm const *__restrict modrm,
                                                                                      emu86_opflags_t op_flags
                                                                                      EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
	if (EMU86_MODRM_ISREG(modrm->mi_type))
		return EMU86_GETREGL(modrm->mi_rm);
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_READABLE(addr, 4);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
	return EMU86_MEMREADL(addr);
}
#endif /* !EMU86_GETMODRM_RML */

#ifndef EMU86_SETMODRM_RML
#define EMU86_SETMODRM_RML(modrm, value, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rml)(EMU86_EMULATE_HELPER_PARAM_ modrm, value, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rml))(EMU86_EMULATE_HELPER_ARGS_
                                                                                      struct emu86_modrm const *__restrict modrm,
                                                                                      u32 value, emu86_opflags_t op_flags
                                                                                      EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
	if (EMU86_MODRM_ISREG(modrm->mi_type)) {
		EMU86_SETREGL(modrm->mi_rm, value);
	} else
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
	{
		byte_t *addr;
		addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
		if (EMU86_ISUSER())
			EMU86_VALIDATE_WRITABLE(addr, 4);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
		EMU86_MEMWRITEL(addr, value);
	}
}
#endif /* !EMU86_SETMODRM_RML */

#if LIBEMU86_CONFIG_WANT_64BIT
#ifndef EMU86_GETMODRM_RMQ
#define EMU86_GETMODRM_RMQ(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmq)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u64
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmq))(EMU86_EMULATE_HELPER_ARGS_
                                                                                      struct emu86_modrm const *__restrict modrm,
                                                                                      emu86_opflags_t op_flags
                                                                                      EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
	if (EMU86_MODRM_ISREG(modrm->mi_type))
		return EMU86_GETREGQ(modrm->mi_rm);
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_READABLE(addr, 8);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
	return EMU86_MEMREADQ(addr);
}
#endif /* !EMU86_GETMODRM_RMQ */

#ifndef EMU86_SETMODRM_RMQ
#define EMU86_SETMODRM_RMQ(modrm, value, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmq)(EMU86_EMULATE_HELPER_PARAM_ modrm, value, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmq))(EMU86_EMULATE_HELPER_ARGS_
                                                                                      struct emu86_modrm const *__restrict modrm,
                                                                                      u64 value, emu86_opflags_t op_flags
                                                                                      EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
	if (EMU86_MODRM_ISREG(modrm->mi_type)) {
		EMU86_SETREGQ(modrm->mi_rm, value);
	} else
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
	{
		byte_t *addr;
		addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
		if (EMU86_ISUSER())
			EMU86_VALIDATE_WRITABLE(addr, 8);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
		EMU86_MEMWRITEQ(addr, value);
	}
}
#endif /* !EMU86_SETMODRM_RMQ */
#endif /* LIBEMU86_CONFIG_WANT_64BIT */



/* Get/Set the R/M operand from a given modrm (when the modrm is known to be memory) */
#if !EMU86_EMULATE_CONFIG_ONLY_MEMORY
#ifndef EMU86_GETMODRM_RMMEMB
#define EMU86_GETMODRM_RMMEMB(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmmemb)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u8
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmmemb))(EMU86_EMULATE_HELPER_ARGS_
                                                                                         struct emu86_modrm const *__restrict modrm,
                                                                                         emu86_opflags_t op_flags
                                                                                         EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_READABLE(addr, 1);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
	return EMU86_MEMREADB(addr);
}
#endif /* !EMU86_GETMODRM_RMMEMB */

#ifndef EMU86_SETMODRM_RMMEMB
#define EMU86_SETMODRM_RMMEMB(modrm, value, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmmemb)(EMU86_EMULATE_HELPER_PARAM_ modrm, value, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmmemb))(EMU86_EMULATE_HELPER_ARGS_
                                                                                         struct emu86_modrm const *__restrict modrm,
                                                                                         u8 value, emu86_opflags_t op_flags
                                                                                         EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_WRITABLE(addr, 1);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
	EMU86_MEMWRITEB(addr, value);
}
#endif /* !EMU86_SETMODRM_RMMEMB */

#ifndef EMU86_GETMODRM_RMMEMW
#define EMU86_GETMODRM_RMMEMW(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmmemw)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u16
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmmemw))(EMU86_EMULATE_HELPER_ARGS_
                                                                                         struct emu86_modrm const *__restrict modrm,
                                                                                         emu86_opflags_t op_flags
                                                                                         EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_READABLE(addr, 2);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
	return EMU86_MEMREADW(addr);
}
#endif /* !EMU86_GETMODRM_RMMEMW */

#ifndef EMU86_SETMODRM_RMMEMW
#define EMU86_SETMODRM_RMMEMW(modrm, value, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmmemw)(EMU86_EMULATE_HELPER_PARAM_ modrm, value, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmmemw))(EMU86_EMULATE_HELPER_ARGS_
                                                                                         struct emu86_modrm const *__restrict modrm,
                                                                                         u16 value, emu86_opflags_t op_flags
                                                                                         EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_WRITABLE(addr, 2);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
	EMU86_MEMWRITEW(addr, value);
}
#endif /* !EMU86_SETMODRM_RMMEMW */

#ifndef EMU86_GETMODRM_RMMEML
#define EMU86_GETMODRM_RMMEML(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rml)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u32
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmmeml))(EMU86_EMULATE_HELPER_ARGS_
                                                                                         struct emu86_modrm const *__restrict modrm,
                                                                                         emu86_opflags_t op_flags
                                                                                         EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_READABLE(addr, 4);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
	return EMU86_MEMREADL(addr);
}
#endif /* !EMU86_GETMODRM_RMMEML */

#ifndef EMU86_SETMODRM_RMMEML
#define EMU86_SETMODRM_RMMEML(modrm, value, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmmeml)(EMU86_EMULATE_HELPER_PARAM_ modrm, value, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmmeml))(EMU86_EMULATE_HELPER_ARGS_
                                                                                         struct emu86_modrm const *__restrict modrm,
                                                                                         u32 value, emu86_opflags_t op_flags
                                                                                         EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_WRITABLE(addr, 4);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
	EMU86_MEMWRITEL(addr, value);
}
#endif /* !EMU86_SETMODRM_RMMEML */

#if LIBEMU86_CONFIG_WANT_64BIT
#ifndef EMU86_GETMODRM_RMMEMQ
#define EMU86_GETMODRM_RMMEMQ(modrm, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmmemq)(EMU86_EMULATE_HELPER_PARAM_ modrm, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED ATTR_PURE EMU86_EMULATE_HELPER_ATTR u64
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_getmodrm_rmmemq))(EMU86_EMULATE_HELPER_ARGS_
                                                                                         struct emu86_modrm const *__restrict modrm,
                                                                                         emu86_opflags_t op_flags
                                                                                         EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_READABLE(addr, 8);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
	return EMU86_MEMREADQ(addr);
}
#endif /* !EMU86_GETMODRM_RMMEMQ */

#ifndef EMU86_SETMODRM_RMMEMQ
#define EMU86_SETMODRM_RMMEMQ(modrm, value, op_flags, real_ip) \
	(EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmmemq)(EMU86_EMULATE_HELPER_PARAM_ modrm, value, op_flags EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, real_ip)))
EMU86_EMULATE_HELPER_DECL ATTR_UNUSED EMU86_EMULATE_HELPER_ATTR void
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(emu86_setmodrm_rmmemq))(EMU86_EMULATE_HELPER_ARGS_
                                                                                         struct emu86_modrm const *__restrict modrm,
                                                                                         u64 value, emu86_opflags_t op_flags
                                                                                         EMU86_IF_MODRM_MEMADDR_NEED_REAL_IP(, uintptr_t real_ip)) {
	byte_t *addr;
	addr = EMU86_MODRM_MEMADDR(modrm, op_flags, real_ip);
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
	if (EMU86_ISUSER())
		EMU86_VALIDATE_WRITABLE(addr, 8);
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
	EMU86_MEMWRITEQ(addr, value);
}
#endif /* !EMU86_SETMODRM_RMMEMQ */
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */



#ifdef __INTELLISENSE__
#ifndef EMU86_EMULATE_TRANSLATEADDR_IS_NOOP
#define _EMU86_INTELLISENSE_BEGIN_REAL_START_PC , *real_start_pc
#else /* !EMU86_EMULATE_TRANSLATEADDR_IS_NOOP */
#define _EMU86_INTELLISENSE_BEGIN_REAL_START_PC /* nothing */
#endif /* EMU86_EMULATE_TRANSLATEADDR_IS_NOOP */


#if 0
#define _EMU86_INTELLISENSE_DEFINE_LABEL(name) name:
#else
#define _EMU86_INTELLISENSE_ARG_PLACEHOLDER_ ,
#define _EMU86_INTELLISENSE_TAKE_SECOND_ARG_IMPL(x, val, ...) val
#define _EMU86_INTELLISENSE_TAKE_SECOND_ARG(x) _EMU86_INTELLISENSE_TAKE_SECOND_ARG_IMPL x
#define _EMU86_INTELLISENSE_IS_DEFINED3(...) _EMU86_INTELLISENSE_TAKE_SECOND_ARG((__VA_ARGS__ 1,0))
#define _EMU86_INTELLISENSE_IS_DEFINED2(x) _EMU86_INTELLISENSE_IS_DEFINED3(_EMU86_INTELLISENSE_ARG_PLACEHOLDER_##x)
#define _EMU86_INTELLISENSE_IS_DEFINED(x) _EMU86_INTELLISENSE_IS_DEFINED2(x)
#define _EMU86_INTELLISENSE_DEFINE_LABEL3_0(name) /* nothing */
#define _EMU86_INTELLISENSE_DEFINE_LABEL3_1(name) name:
#define _EMU86_INTELLISENSE_DEFINE_LABEL3(name, x) _EMU86_INTELLISENSE_DEFINE_LABEL3_##x(name)
#define _EMU86_INTELLISENSE_DEFINE_LABEL2(name, x) _EMU86_INTELLISENSE_DEFINE_LABEL3(name, x)
#define _EMU86_INTELLISENSE_DEFINE_LABEL(name) \
	_EMU86_INTELLISENSE_DEFINE_LABEL2(name, _EMU86_INTELLISENSE_IS_DEFINED(NEED_##name))
#endif

#define EMU86_INTELLISENSE_BEGIN(name)                                                                \
	__DECL_BEGIN                                                                                      \
	EMU86_EMULATE_DECL EMU86_EMULATE_RETURN_TYPE                                                      \
	EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_HELPER_NAME(_i##name))(EMU86_EMULATE_ARGS) { \
		byte_t const *start_pc, *pc _EMU86_INTELLISENSE_BEGIN_REAL_START_PC;                          \
		emu86_opcode_t tiny_opcode;                                                                   \
		emu86_opflags_t op_flags;                                                                     \
		struct emu86_modrm modrm;                                                                     \
		switch (tiny_opcode)
#define EMU86_INTELLISENSE_END                                         \
done:                                                                  \
return_unknown_instruction:                                            \
_EMU86_INTELLISENSE_DEFINE_LABEL(done_dont_set_pc)                     \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_privileged_instruction_rmreg)  \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_privileged_instruction)        \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_expected_memory_modrm_rmreg)   \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_expected_memory_modrm)         \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_expected_register_modrm_rmreg) \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_expected_register_modrm)       \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unexpected_vex_ll_rmreg)       \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unexpected_vex_ll)             \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unexpected_prefix_rmreg)       \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unexpected_prefix)             \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unsupported_instruction_rmreg) \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unsupported_instruction)       \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unavailable_instruction_rmreg) \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unsupported_instruction)       \
_EMU86_INTELLISENSE_DEFINE_LABEL(return_unknown_instruction_rmreg)     \
		(void)0;                                                       \
	}                                                                  \
	DECL_END
#else /* __INTELLISENSE__ */
#define EMU86_INTELLISENSE_BEGIN(name) /* nothing */
#define EMU86_INTELLISENSE_END         /* nothing */
#endif /* !__INTELLISENSE__ */

EMU86_EMULATE_DECL EMU86_EMULATE_ATTR EMU86_EMULATE_RETURN_TYPE
EMU86_EMULATE_NOTHROW(EMU86_EMULATE_CC EMU86_EMULATE_NAME)(EMU86_EMULATE_ARGS) {
	EMU86_EMULATE_VARIABLES
#ifndef EMU86_EMULATE_TRANSLATEADDR_IS_NOOP
	byte_t const *real_start_pc;
#endif /* !EMU86_EMULATE_TRANSLATEADDR_IS_NOOP */
	byte_t const *start_pc, *pc;
	struct emu86_modrm modrm;
	emu86_opflags_t op_flags;
	emu86_opcode_t tiny_opcode;
#ifdef EMU86_EMULATE_SETUP
	EMU86_EMULATE_SETUP();
#endif /* EMU86_EMULATE_SETUP */
	op_flags = EMU86_EMULATE_GETOPFLAGS();
#ifdef EMU86_EMULATE_TRANSLATEADDR_IS_NOOP
	start_pc = (byte_t const *)EMU86_GETPCPTR();
#else /* EMU86_EMULATE_TRANSLATEADDR_IS_NOOP */
	real_start_pc = (byte_t const *)EMU86_GETPCPTR();
	start_pc      = (byte_t const *)EMU86_EMULATE_TRANSLATEADDR(real_start_pc);
#endif /* !EMU86_EMULATE_TRANSLATEADDR_IS_NOOP */
#ifdef EMU86_EMULATE_TRANSLATEADDR_IS_NOOP
#define REAL_START_IP() (uintptr_t)start_pc
#define REAL_IP()       (uintptr_t)pc
#else /* EMU86_EMULATE_TRANSLATEADDR_IS_NOOP */
#define REAL_START_IP() (uintptr_t)real_start_pc
#define REAL_IP()       (uintptr_t)(real_start_pc + (size_t)(pc - start_pc))
#endif /* !EMU86_EMULATE_TRANSLATEADDR_IS_NOOP */

#ifdef EMU86_EMULATE_TRY
	EMU86_EMULATE_TRY
#endif /* EMU86_EMULATE_TRY */
	{
		/* Decode the instruction */
#ifdef EMU86_EMULATE_TRY_DECODE
		EMU86_EMULATE_TRY_DECODE
#endif /* EMU86_EMULATE_TRY_DECODE */
		{
#ifdef EMU86_EMULATE_VALIDATE_BEFORE_OPCODE_DECODE
			EMU86_EMULATE_VALIDATE_BEFORE_OPCODE_DECODE(REAL_START_IP());
#endif /* EMU86_EMULATE_VALIDATE_BEFORE_OPCODE_DECODE */
			pc = emu86_opcode_decode(start_pc, &tiny_opcode, &op_flags);
		}
#ifdef EMU86_EMULATE_EXCEPT_DECODE
		EMU86_EMULATE_EXCEPT_DECODE;
#endif /* EMU86_EMULATE_EXCEPT_DECODE */

		/* Execute the instruction */
#ifdef EMU86_EMULATE_TRY_SWITCH
		EMU86_EMULATE_TRY_SWITCH
#endif /* EMU86_EMULATE_TRY_SWITCH */
		{
#if LIBEMU86_CONFIG_WANT_32BIT || LIBEMU86_CONFIG_WANT_16BIT
#define IF_16BIT_OR_32BIT(...) __VA_ARGS__
#else /* LIBEMU86_CONFIG_WANT_32BIT || LIBEMU86_CONFIG_WANT_16BIT */
#define IF_16BIT_OR_32BIT(...) /* nothing */
#endif /* !LIBEMU86_CONFIG_WANT_32BIT && !LIBEMU86_CONFIG_WANT_16BIT */


#if LIBEMU86_CONFIG_WANT_64BIT
#define IS_64BIT()           (op_flags & EMU86_F_REX_W)
#define IF_64BIT(...)        __VA_ARGS__
#define IFELSE_64BIT(tt, ff) tt
#else /* LIBEMU86_CONFIG_WANT_64BIT */
#define IS_64BIT()           0
#define IF_64BIT(...)        /* nothing */
#define IFELSE_64BIT(tt, ff) ff
#endif /* !LIBEMU86_CONFIG_WANT_64BIT */

#if LIBEMU86_CONFIG_WANT_16BIT
#define IS_16BIT() (EMU86_F_IS16(op_flags) ? (op_flags & EMU86_F_OP16) == 0 : (op_flags & EMU86_F_OP16) != 0)
#else /* LIBEMU86_CONFIG_WANT_16BIT */
#define IS_16BIT() (op_flags & EMU86_F_OP16)
#endif /* !LIBEMU86_CONFIG_WANT_16BIT */

#define MODRM_DECODE_MEMONLY()                         \
	do {                                               \
		pc = emu86_modrm_decode(pc, &modrm, op_flags); \
		if (!EMU86_MODRM_ISMEM(modrm.mi_type))         \
			goto return_expected_memory_modrm;         \
	}	__WHILE0
#if EMU86_EMULATE_CONFIG_ONLY_MEMORY
#define NEED_return_expected_memory_modrm
#define MODRM_DECODE() MODRM_DECODE_MEMONLY()
#else /* EMU86_EMULATE_CONFIG_ONLY_MEMORY */
#define MODRM_DECODE() \
	(pc = emu86_modrm_decode(pc, &modrm, op_flags))
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */

	/* Invoke all necessary callbacks for read/write access to memory
	 * who's address has been derived */
#ifndef EMU86_READ_USER_MEMORY
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READABLE_IS_NOOP)
#define EMU86_READ_USER_MEMORY(addr, num_bytes)       \
	do {                                              \
		if (EMU86_ISUSER())                           \
			EMU86_VALIDATE_READABLE(addr, num_bytes); \
	}	__WHILE0
#else /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READABLE_IS_NOOP */
#define EMU86_READ_USER_MEMORY(addr, num_bytes)  (void)0
#endif /* !EMU86_EMULATE_CONFIG_CHECKUSER || EMU86_VALIDATE_READABLE_IS_NOOP */
#endif /* !EMU86_READ_USER_MEMORY */

#ifndef EMU86_WRITE_USER_MEMORY
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_WRITABLE_IS_NOOP)
#define EMU86_WRITE_USER_MEMORY(addr, num_bytes)      \
	do {                                              \
		if (EMU86_ISUSER())                           \
			EMU86_VALIDATE_WRITABLE(addr, num_bytes); \
	}	__WHILE0
#else /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_WRITABLE_IS_NOOP */
#define EMU86_WRITE_USER_MEMORY(addr, num_bytes) (void)0
#endif /* !EMU86_EMULATE_CONFIG_CHECKUSER || EMU86_VALIDATE_WRITABLE_IS_NOOP */
#endif /* !EMU86_WRITE_USER_MEMORY */

#ifndef EMU86_READWRITE_USER_MEMORY
#if EMU86_EMULATE_CONFIG_CHECKUSER && !defined(EMU86_VALIDATE_READWRITE_IS_NOOP)
#define EMU86_READWRITE_USER_MEMORY(addr, num_bytes)   \
	do {                                               \
		if (EMU86_ISUSER())                            \
			EMU86_VALIDATE_READWRITE(addr, num_bytes); \
	}	__WHILE0
#else /* EMU86_EMULATE_CONFIG_CHECKUSER && !EMU86_VALIDATE_READWRITE_IS_NOOP */
#define EMU86_READWRITE_USER_MEMORY(addr, num_bytes) (void)0
#endif /* !EMU86_EMULATE_CONFIG_CHECKUSER || EMU86_VALIDATE_READWRITE_IS_NOOP */
#endif /* !EMU86_READWRITE_USER_MEMORY */

#if EMU86_EMULATE_CONFIG_ONLY_MEMORY
#define IF_ONLY_MEMORY(...)  __VA_ARGS__
#define NIF_ONLY_MEMORY(...) /* nothing */
#else /* EMU86_EMULATE_CONFIG_ONLY_MEMORY */
#define IF_ONLY_MEMORY(...)  /* nothing */
#define NIF_ONLY_MEMORY(...) __VA_ARGS__
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */

#define EMU86_OPCODE() EMU86_OPCODE_DECODE(tiny_opcode)

#define MODRM_MEMADDR()           EMU86_MODRM_MEMADDR(&modrm, op_flags, REAL_IP())
#define MODRM_MEMADDR_NOSEGBASE() EMU86_MODRM_MEMADDR_NOSEGBASE(&modrm, op_flags, REAL_IP())
#define MODRM_GETRMB()   EMU86_GETMODRM_RMB(&modrm, op_flags, REAL_IP())
#define MODRM_GETRMW()   EMU86_GETMODRM_RMW(&modrm, op_flags, REAL_IP())
#define MODRM_GETRML()   EMU86_GETMODRM_RML(&modrm, op_flags, REAL_IP())
#define MODRM_SETRMB(v)  EMU86_SETMODRM_RMB(&modrm, v, op_flags, REAL_IP())
#define MODRM_SETRMW(v)  EMU86_SETMODRM_RMW(&modrm, v, op_flags, REAL_IP())
#define MODRM_SETRML(v)  EMU86_SETMODRM_RML(&modrm, v, op_flags, REAL_IP())
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_GETRMQ()   EMU86_GETMODRM_RMQ(&modrm, op_flags, REAL_IP())
#define MODRM_SETRMQ(v)  EMU86_SETMODRM_RMQ(&modrm, v, op_flags, REAL_IP())
#endif /* LIBEMU86_CONFIG_WANT_64BIT */


/* Indicate that an unsupported instruction would have accessed modrm */
#ifdef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
#define MODRM_NOSUP_GETRMB()           (void)0
#define MODRM_NOSUP_GETRMW()           (void)0
#define MODRM_NOSUP_GETRML()           (void)0
#define MODRM_NOSUP_SETRMB()           (void)0
#define MODRM_NOSUP_SETRMW()           (void)0
#define MODRM_NOSUP_SETRML()           (void)0
#define MODRM_NOSUP_GETRMMEMB()        (void)0
#define MODRM_NOSUP_GETRMMEMW()        (void)0
#define MODRM_NOSUP_GETRMMEML()        (void)0
#define MODRM_NOSUP_SETRMMEMB()        (void)0
#define MODRM_NOSUP_SETRMMEMW()        (void)0
#define MODRM_NOSUP_SETRMMEML()        (void)0
#define MODRM_NOSUP_GETSETRMB()        (void)0
#define MODRM_NOSUP_GETSETRMW()        (void)0
#define MODRM_NOSUP_GETSETRML()        (void)0
#define MODRM_NOSUP_GETSETRMMEMB()     (void)0
#define MODRM_NOSUP_GETSETRMMEMW()     (void)0
#define MODRM_NOSUP_GETSETRMMEML()     (void)0
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_NOSUP_GETRMQ()           (void)0
#define MODRM_NOSUP_SETRMQ()           (void)0
#define MODRM_NOSUP_GETRMMEMQ()        (void)0
#define MODRM_NOSUP_SETRMMEMQ()        (void)0
#define MODRM_NOSUP_GETSETRMQ()        (void)0
#define MODRM_NOSUP_GETSETRMMEMQ()     (void)0
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#define MODRM_NOSUP_GETRMZ(num_bytes)       (void)0
#define MODRM_NOSUP_SETRMZ(num_bytes)       (void)0
#define MODRM_NOSUP_GETSETRMZ(num_bytes)    (void)0
#define MODRM_NOSUP_GETRMMEMZ(num_bytes)    (void)0
#define MODRM_NOSUP_SETRMMEMZ(num_bytes)    (void)0
#define MODRM_NOSUP_GETSETRMMEMZ(num_bytes) (void)0
#else /* EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
#if EMU86_EMULATE_CONFIG_ONLY_MEMORY
#define MODRM_NOSUP_GETRMB()    MODRM_NOSUP_GETRMMEMB()
#define MODRM_NOSUP_GETRMW()    MODRM_NOSUP_GETRMMEMW()
#define MODRM_NOSUP_GETRML()    MODRM_NOSUP_GETRMMEML()
#define MODRM_NOSUP_SETRMB()    MODRM_NOSUP_SETRMMEMB()
#define MODRM_NOSUP_SETRMW()    MODRM_NOSUP_SETRMMEMW()
#define MODRM_NOSUP_SETRML()    MODRM_NOSUP_SETRMMEML()
#define MODRM_NOSUP_GETSETRMB() MODRM_NOSUP_GETSETRMMEMB()
#define MODRM_NOSUP_GETSETRMW() MODRM_NOSUP_GETSETRMMEMW()
#define MODRM_NOSUP_GETSETRML() MODRM_NOSUP_GETSETRMMEML()
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_NOSUP_GETRMQ()    MODRM_NOSUP_GETRMMEMQ()
#define MODRM_NOSUP_SETRMQ()    MODRM_NOSUP_SETRMMEMQ()
#define MODRM_NOSUP_GETSETRMQ() MODRM_NOSUP_GETSETRMMEMQ()
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#define MODRM_NOSUP_GETRMZ(num_bytes)    MODRM_NOSUP_GETRMMEMZ(num_bytes)
#define MODRM_NOSUP_SETRMZ(num_bytes)    MODRM_NOSUP_SETRMMEMZ(num_bytes)
#define MODRM_NOSUP_GETSETRMZ(num_bytes) MODRM_NOSUP_GETSETRMMEMZ(num_bytes)
#else /* EMU86_EMULATE_CONFIG_ONLY_MEMORY */
#define MODRM_NOSUP_GETRMB()    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETRMMEMB(); } __WHILE0
#define MODRM_NOSUP_GETRMW()    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETRMMEMW(); } __WHILE0
#define MODRM_NOSUP_GETRML()    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETRMMEML(); } __WHILE0
#define MODRM_NOSUP_SETRMB()    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_SETRMMEMB(); } __WHILE0
#define MODRM_NOSUP_SETRMW()    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_SETRMMEMW(); } __WHILE0
#define MODRM_NOSUP_SETRML()    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_SETRMMEML(); } __WHILE0
#define MODRM_NOSUP_GETSETRMB() do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETSETRMMEMB(); } __WHILE0
#define MODRM_NOSUP_GETSETRMW() do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETSETRMMEMW(); } __WHILE0
#define MODRM_NOSUP_GETSETRML() do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETSETRMMEML(); } __WHILE0
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_NOSUP_GETRMQ()    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETRMMEMQ(); } __WHILE0
#define MODRM_NOSUP_SETRMQ()    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_SETRMMEMQ(); } __WHILE0
#define MODRM_NOSUP_GETSETRMQ() do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETSETRMMEMQ(); } __WHILE0
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#define MODRM_NOSUP_GETRMZ(num_bytes)    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETRMMEMZ(num_bytes); } __WHILE0
#define MODRM_NOSUP_SETRMZ(num_bytes)    do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_SETRMMEMZ(num_bytes); } __WHILE0
#define MODRM_NOSUP_GETSETRMZ(num_bytes) do { if (EMU86_MODRM_ISMEM(modrm.mi_type)) MODRM_NOSUP_GETSETRMMEMZ(num_bytes); } __WHILE0
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */
#define MODRM_NOSUP_GETRMMEMB()    EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 1, true, false)
#define MODRM_NOSUP_GETRMMEMW()    EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 2, true, false)
#define MODRM_NOSUP_GETRMMEML()    EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 4, true, false)
#define MODRM_NOSUP_SETRMMEMB()    EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 1, false, true)
#define MODRM_NOSUP_SETRMMEMW()    EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 2, false, true)
#define MODRM_NOSUP_SETRMMEML()    EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 4, false, true)
#define MODRM_NOSUP_GETSETRMMEMB() EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 1, true, true)
#define MODRM_NOSUP_GETSETRMMEMW() EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 2, true, true)
#define MODRM_NOSUP_GETSETRMMEML() EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 4, true, true)
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_NOSUP_GETRMMEMQ()        EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 8, true, false)
#define MODRM_NOSUP_SETRMMEMQ()        EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 8, false, true)
#define MODRM_NOSUP_GETSETRMMEMQ()     EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), 8, true, true)
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#define MODRM_NOSUP_GETRMMEMZ(num_bytes)    EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), num_bytes, true, false)
#define MODRM_NOSUP_SETRMMEMZ(num_bytes)    EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), num_bytes, false, true)
#define MODRM_NOSUP_GETSETRMMEMZ(num_bytes) EMU86_UNSUPPORTED_MEMACCESS(MODRM_MEMADDR(), num_bytes, true, true)
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */

#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_NOSUP_GETRMZ_VEX_W()       MODRM_NOSUP_GETRMZ((op_flags & EMU86_F_VEX_W) ? 8 : 4)
#define MODRM_NOSUP_SETRMZ_VEX_W()       MODRM_NOSUP_SETRMZ((op_flags & EMU86_F_VEX_W) ? 8 : 4)
#define MODRM_NOSUP_GETSETRMZ_VEX_W()    MODRM_NOSUP_GETSETRMZ((op_flags & EMU86_F_VEX_W) ? 8 : 4)
#define MODRM_NOSUP_GETRMMEMZ_VEX_W()    MODRM_NOSUP_GETRMMEMZ((op_flags & EMU86_F_VEX_W) ? 8 : 4)
#define MODRM_NOSUP_SETRMMEMZ_VEX_W()    MODRM_NOSUP_SETRMMEMZ((op_flags & EMU86_F_VEX_W) ? 8 : 4)
#define MODRM_NOSUP_GETSETRMMEMZ_VEX_W() MODRM_NOSUP_GETSETRMMEMZ((op_flags & EMU86_F_VEX_W) ? 8 : 4)
#define MODRM_NOSUP_GETRMWLQ()           MODRM_NOSUP_GETRMZ(IS_64BIT() ? 8 : !IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_SETRMWLQ()           MODRM_NOSUP_SETRMZ(IS_64BIT() ? 8 : !IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_GETSETRMWLQ()        MODRM_NOSUP_GETSETRMZ(IS_64BIT() ? 8 : !IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_GETRMMEMWLQ()        MODRM_NOSUP_GETRMMEMZ(IS_64BIT() ? 8 : !IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_SETRMMEMWLQ()        MODRM_NOSUP_SETRMMEMZ(IS_64BIT() ? 8 : !IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_GETSETRMMEMWLQ()     MODRM_NOSUP_GETSETRMMEMZ(IS_64BIT() ? 8 : !IS_16BIT() ? 4 : 2)
#else /* LIBEMU86_CONFIG_WANT_64BIT */
#define MODRM_NOSUP_GETRMZ_VEX_W()       MODRM_NOSUP_GETRML())
#define MODRM_NOSUP_SETRMZ_VEX_W()       MODRM_NOSUP_SETRML())
#define MODRM_NOSUP_GETSETRMZ_VEX_W()    MODRM_NOSUP_GETSETRML())
#define MODRM_NOSUP_GETRMMEMZ_VEX_W()    MODRM_NOSUP_GETRMMEML())
#define MODRM_NOSUP_SETRMMEMZ_VEX_W()    MODRM_NOSUP_SETRMMEML())
#define MODRM_NOSUP_GETSETRMMEMZ_VEX_W() MODRM_NOSUP_GETSETRMMEML())
#define MODRM_NOSUP_GETRMWLQ()           MODRM_NOSUP_GETRMZ(!IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_SETRMWLQ()           MODRM_NOSUP_SETRMZ(!IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_GETSETRMWLQ()        MODRM_NOSUP_GETSETRMZ(!IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_GETRMMEMWLQ()        MODRM_NOSUP_GETRMMEMZ(!IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_SETRMMEMWLQ()        MODRM_NOSUP_SETRMMEMZ(!IS_16BIT() ? 4 : 2)
#define MODRM_NOSUP_GETSETRMMEMWLQ()     MODRM_NOSUP_GETSETRMMEMZ(!IS_16BIT() ? 4 : 2)
#endif /* !LIBEMU86_CONFIG_WANT_64BIT */


#if EMU86_EMULATE_CONFIG_ONLY_MEMORY
#define MODRM_GETRMMEMB()   EMU86_GETMODRM_RMB(&modrm, op_flags, REAL_IP())
#define MODRM_GETRMMEMW()   EMU86_GETMODRM_RMW(&modrm, op_flags, REAL_IP())
#define MODRM_GETRMMEML()   EMU86_GETMODRM_RML(&modrm, op_flags, REAL_IP())
#define MODRM_SETRMMEMB(v)  EMU86_SETMODRM_RMB(&modrm, v, op_flags, REAL_IP())
#define MODRM_SETRMMEMW(v)  EMU86_SETMODRM_RMW(&modrm, v, op_flags, REAL_IP())
#define MODRM_SETRMMEML(v)  EMU86_SETMODRM_RML(&modrm, v, op_flags, REAL_IP())
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_GETRMMEMQ()   EMU86_GETMODRM_RMQ(&modrm, op_flags, REAL_IP())
#define MODRM_SETRMMEMQ(v)  EMU86_SETMODRM_RMQ(&modrm, v, op_flags, REAL_IP())
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#else /* EMU86_EMULATE_CONFIG_ONLY_MEMORY */
#define MODRM_GETRMMEMB()   EMU86_GETMODRM_RMMEMB(&modrm, op_flags, REAL_IP())
#define MODRM_GETRMMEMW()   EMU86_GETMODRM_RMMEMW(&modrm, op_flags, REAL_IP())
#define MODRM_GETRMMEML()   EMU86_GETMODRM_RMMEML(&modrm, op_flags, REAL_IP())
#define MODRM_SETRMMEMB(v)  EMU86_SETMODRM_RMMEMB(&modrm, v, op_flags, REAL_IP())
#define MODRM_SETRMMEMW(v)  EMU86_SETMODRM_RMMEMW(&modrm, v, op_flags, REAL_IP())
#define MODRM_SETRMMEML(v)  EMU86_SETMODRM_RMMEML(&modrm, v, op_flags, REAL_IP())
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_GETRMMEMQ()   EMU86_GETMODRM_RMMEMQ(&modrm, op_flags, REAL_IP())
#define MODRM_SETRMMEMQ(v)  EMU86_SETMODRM_RMMEMQ(&modrm, v, op_flags, REAL_IP())
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#endif /* !EMU86_EMULATE_CONFIG_ONLY_MEMORY */

#define MODRM_GETRMREGB()   EMU86_GETREGB(modrm.mi_rm, op_flags)
#define MODRM_GETRMREGW()   EMU86_GETREGW(modrm.mi_rm)
#define MODRM_GETRMREGL()   EMU86_GETREGL(modrm.mi_rm)
#define MODRM_SETRMREGB(v)  EMU86_SETREGB(modrm.mi_rm, v, op_flags)
#define MODRM_SETRMREGW(v)  EMU86_SETREGW(modrm.mi_rm, v)
#define MODRM_SETRMREGL(v)  EMU86_SETREGL(modrm.mi_rm, v)
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_GETRMREGQ()   EMU86_GETREGQ(modrm.mi_rm)
#define MODRM_SETRMREGQ(v)  EMU86_SETREGQ(modrm.mi_rm, v)
#endif /* LIBEMU86_CONFIG_WANT_64BIT */

#define MODRM_GETREGB()  EMU86_GETREGB(modrm.mi_reg, op_flags)
#define MODRM_GETREGW()  EMU86_GETREGW(modrm.mi_reg)
#define MODRM_GETREGL()  EMU86_GETREGL(modrm.mi_reg)
#define MODRM_SETREGB(v) EMU86_SETREGB(modrm.mi_reg, v, op_flags)
#define MODRM_SETREGW(v) EMU86_SETREGW(modrm.mi_reg, v)
#define MODRM_SETREGL(v) EMU86_SETREGL(modrm.mi_reg, v)
#if LIBEMU86_CONFIG_WANT_64BIT
#define MODRM_GETREGQ()  EMU86_GETREGQ(modrm.mi_reg)
#define MODRM_SETREGQ(v) EMU86_SETREGQ(modrm.mi_reg, v)
#endif /* LIBEMU86_CONFIG_WANT_64BIT */

#define VEX_GETREGB()  EMU86_GETREGB(EMU86_F_VEX_VVVVV(op_flags), op_flags)
#define VEX_GETREGW()  EMU86_GETREGW(EMU86_F_VEX_VVVVV(op_flags))
#define VEX_GETREGL()  EMU86_GETREGL(EMU86_F_VEX_VVVVV(op_flags))
#define VEX_SETREGB(v) EMU86_SETREGB(EMU86_F_VEX_VVVVV(op_flags), v, op_flags)
#define VEX_SETREGW(v) EMU86_SETREGW(EMU86_F_VEX_VVVVV(op_flags), v)
#define VEX_SETREGL(v) EMU86_SETREGL(EMU86_F_VEX_VVVVV(op_flags), v)
#if LIBEMU86_CONFIG_WANT_64BIT
#define VEX_GETREGQ()  EMU86_GETREGQ(EMU86_F_VEX_VVVVV(op_flags))
#define VEX_SETREGQ(v) EMU86_SETREGQ(EMU86_F_VEX_VVVVV(op_flags), v)
#endif /* LIBEMU86_CONFIG_WANT_64BIT */



/* EMU86_ADDRSIZE_SWITCH(do64, do32, do16)
 * >> if (<running-as-64-bit-code>) {
 * >>     if (<67-prefix-byte-given>) {
 * >>         do32
 * >>     } else {
 * >>         do64
 * >>     }
 * >> } else if (<running-as-32-bit-code>) {
 * >>     if (<67-prefix-byte-given>) {
 * >>         do16
 * >>     } else {
 * >>         do32
 * >>     }
 * >> } else {
 * >>     if (<67-prefix-byte-given>) {
 * >>         do32
 * >>     } else {
 * >>         do16
 * >>     }
 * >> }
 */
#if LIBEMU86_CONFIG_WANT_64BIT && (LIBEMU86_CONFIG_WANT_16BIT || LIBEMU86_CONFIG_WANT_32BIT)
#define EMU86_ADDRSIZE_SWITCH(do64, do32, do16)                                                  \
	if (EMU86_F_IS64(op_flags) && !(op_flags & EMU86_F_67)) {                                    \
		/* 64-bit address size */                                                                \
		do64;                                                                                    \
	} else if ((EMU86_F_IS64(op_flags) || EMU86_F_IS16(op_flags)) ^ !!(op_flags & EMU86_F_67)) { \
		/* 16-bit address size */                                                                \
		do16;                                                                                    \
	} else {                                                                                     \
		/* 32-bit address size */                                                                \
		do32;                                                                                    \
	}
#elif !LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_ADDRSIZE_SWITCH(do64, do32, do16)                 \
	if (!!EMU86_F_IS16(op_flags) ^ !!(op_flags & EMU86_F_67)) { \
		/* 16-bit address size */                               \
		do16;                                                   \
	} else {                                                    \
		/* 32-bit address size */                               \
		do32;                                                   \
	}
#elif LIBEMU86_CONFIG_WANT_64BIT && !(LIBEMU86_CONFIG_WANT_16BIT || LIBEMU86_CONFIG_WANT_32BIT)
#define EMU86_ADDRSIZE_SWITCH(do64, do32, do16)               \
	if (EMU86_F_IS64(op_flags) && !(op_flags & EMU86_F_67)) { \
		/* 64-bit address size */                             \
		do64;                                                 \
	} else {                                                  \
		/* 32-bit address size */                             \
		do32;                                                 \
	}
#else
#error "Invalid configuration"
#endif


/* EMU86_ADDRSIZE_SWITCH64(do64, do32, do16)
 * >> if (<running-as-64-bit-code>) {
 * >>     do64
 * >> } else if (<running-as-32-bit-code>) {
 * >>     if (<67-prefix-byte-given>) {
 * >>         do16
 * >>     } else {
 * >>         do32
 * >>     }
 * >> } else {
 * >>     if (<67-prefix-byte-given>) {
 * >>         do32
 * >>     } else {
 * >>         do16
 * >>     }
 * >> }
 */
#if LIBEMU86_CONFIG_WANT_64BIT && (LIBEMU86_CONFIG_WANT_16BIT || LIBEMU86_CONFIG_WANT_32BIT)
#define EMU86_ADDRSIZE_SWITCH64(do64, do32, do16)                        \
	if (EMU86_F_IS64(op_flags)) {                                      \
		/* 64-bit address size */                                      \
		do64;                                                          \
	} else if (!!EMU86_F_IS16(op_flags) ^ !!(op_flags & EMU86_F_67)) { \
		/* 16-bit address size */                                      \
		do16;                                                          \
	} else {                                                           \
		/* 32-bit address size */                                      \
		do32;                                                          \
	}
#elif !LIBEMU86_CONFIG_WANT_64BIT
#define EMU86_ADDRSIZE_SWITCH64(do64, do32, do16)               \
	if (!!EMU86_F_IS16(op_flags) ^ !!(op_flags & EMU86_F_67)) { \
		/* 16-bit address size */                               \
		do16;                                                   \
	} else {                                                    \
		/* 32-bit address size */                               \
		do32;                                                   \
	}
#elif LIBEMU86_CONFIG_WANT_64BIT && !(LIBEMU86_CONFIG_WANT_16BIT || LIBEMU86_CONFIG_WANT_32BIT)
#define EMU86_ADDRSIZE_SWITCH64(do64, do32, do16) \
	{                                             \
		/* 64-bit address size */                 \
		do64;                                     \
	}
#else
#error "Invalid configuration"
#endif

/* Check if a lock-prefix has been given. */
#ifndef EMU86_HASLOCK
#if EMU86_EMULATE_CONFIG_IGNORE_LOCK
#define EMU86_HASLOCK() 0
#else /* EMU86_EMULATE_CONFIG_IGNORE_LOCK */
#define EMU86_HASLOCK() ((op_flags & EMU86_F_LOCK) != 0)
#endif /* !EMU86_EMULATE_CONFIG_IGNORE_LOCK */
#endif /* !EMU86_HASLOCK */

#if EMU86_EMULATE_CONFIG_CHECKLOCK
			if unlikely(op_flags & EMU86_F_LOCK) {
				switch (tiny_opcode) {
#define HAVEOP(config) config || EMU86_EMULATE_CONFIG_CHECKERROR


#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_BITTEST)
	/*         0F BB /r     BTC r/m16, r16      Store selected bit in CF flag and complement.
	 *         0F BB /r     BTC r/m32, r32      Store selected bit in CF flag and complement.
	 * REX.W + 0F BB /r     BTC r/m64, r64      Store selected bit in CF flag and complement. */
case EMU86_OPCODE_ENCODE(0x0fbb):

	/*         0F B3 /r     BTR r/m16, r16      Store selected bit in CF flag and clear.
	 *         0F B3 /r     BTR r/m32, r32      Store selected bit in CF flag and clear.
	 * REX.W + 0F B3 /r     BTR r/m64, r64      Store selected bit in CF flag and clear. */
case EMU86_OPCODE_ENCODE(0x0fb3):

	/*         0F AB /r     BTS r/m16, r16      Store selected bit in CF flag and set.
	 *         0F AB /r     BTS r/m32, r32      Store selected bit in CF flag and set.
	 * REX.W + 0F AB /r     BTS r/m64, r64      Store selected bit in CF flag and set. */
case EMU86_OPCODE_ENCODE(0x0fab):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory

case EMU86_OPCODE_ENCODE(0x0fba):
	emu86_modrm_decode(pc, &modrm, op_flags);
	if unlikely(modrm.mi_reg != 5 && /* bts */
	            modrm.mi_reg != 6 && /* btr */
	            modrm.mi_reg != 7)   /* btc */
		goto return_unexpected_lock_rmreg;
#define NEED_return_unexpected_lock_rmreg
	goto checklock_modrm_memory_parsed;
#define NEED_checklock_modrm_memory_parsed
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_BITTEST) */


#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_INC_RM || EMU86_EMULATE_CONFIG_WANT_DEC_RM)
	/* FE /0      INC r/m8      Increment r/m byte by 1 */
	/* FE /1      DEC r/m8      Decrement r/m8 by 1 */
case EMU86_OPCODE_ENCODE(0xfe):
	/* FF /0      INC r/m16      Increment r/m byte by 1
	 * FF /0      INC r/m32      Increment r/m byte by 1
	 * FF /0      INC r/m64      Increment r/m byte by 1 */
	/* FF /1      DEC r/m16      Decrement r/m byte by 1
	 * FF /1      DEC r/m32      Decrement r/m byte by 1
	 * FF /1      DEC r/m64      Decrement r/m byte by 1 */
case EMU86_OPCODE_ENCODE(0xff):
	emu86_modrm_decode(pc, &modrm, op_flags);
	if unlikely(modrm.mi_reg != 0 && /* inc */
	            modrm.mi_reg != 1)   /* dec */
		goto return_unexpected_lock_rmreg;
#define NEED_return_unexpected_lock_rmreg
	goto checklock_modrm_memory_parsed;
#define NEED_checklock_modrm_memory_parsed
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_INC_RM || EMU86_EMULATE_CONFIG_WANT_DEC_RM) */


#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_CMPXCHG)
	/* 0F B0 /r      CMPXCHG r/m8,r8      Compare  AL with r/m8.  If equal, ZF is  set and r8 is
	 *                                    loaded into r/m8. Else, clear ZF and load r/m8 into AL */
case EMU86_OPCODE_ENCODE(0x0fb0):
	/* 0F B1 /r      CMPXCHG r/m16,r16    Compare  AX with  r/m16. If equal,  ZF is set  and r16 is
	 *                                    loaded into r/m16. Else, clear ZF and load r/m16 into AX.
	 * 0F B1 /r      CMPXCHG r/m32,r32    Compare  EAX with  r/m32. If equal,  ZF is set  and r32 is
	 *                                    loaded into r/m32. Else, clear ZF and load r/m32 into EAX.
	 * 0F B1 /r      CMPXCHG r/m64,r64    Compare  RAX with  r/m64. If equal,  ZF is set  and r64 is
	 *                                    loaded into r/m64. Else, clear ZF and load r/m64 into RAX. */
case EMU86_OPCODE_ENCODE(0x0fb1):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_CMPXCHG) */


#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_CMPXCHG8B || EMU86_EMULATE_CONFIG_WANT_CMPXCHG16B)
	/*         0F C7 /1 CMPXCHG8B m64      Compare EDX:EAX with m64. If equal, set ZF and load ECX:EBX
	 *                                     into  m64.  Else,  clear  ZF  and  load  m64  into EDX:EAX.
	 * REX.W + 0F C7 /1 CMPXCHG16B m128    Compare RDX:RAX with m128. If equal, set ZF and load RCX:RBX
	 *                                     into m128.  Else,  clear  ZF and  load  m128  into  RDX:RAX. */
case EMU86_OPCODE_ENCODE(0x0fc7):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_CMPXCHG8B || EMU86_EMULATE_CONFIG_WANT_CMPXCHG16B) */


#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_ARITH2)
	/* F6 /2      NOT r/m8      Reverse each bit of r/m8 */
	/* F6 /3      NEG r/m8      Two's complement negate r/m8 */
case EMU86_OPCODE_ENCODE(0xf6):
	/* F7 /2      NOT r/m16      Reverse each bit of r/m16
	 * F7 /2      NOT r/m32      Reverse each bit of r/m32
	 * F7 /2      NOT r/m64      Reverse each bit of r/m64 */
	/* F7 /3      NEG r/m16      Two's complement negate r/m16
	 * F7 /3      NEG r/m32      Two's complement negate r/m32
	 * F7 /3      NEG r/m64      Two's complement negate r/m64 */
case EMU86_OPCODE_ENCODE(0xf7):
	emu86_modrm_decode(pc, &modrm, op_flags);
	if unlikely(modrm.mi_reg != 2 && /* not */
	            modrm.mi_reg != 3)   /* neg */
		goto return_unexpected_lock_rmreg;
#define NEED_return_unexpected_lock_rmreg
	goto checklock_modrm_memory_parsed;
#define NEED_checklock_modrm_memory_parsed
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_ARITH2) */


#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_XADD)
	/* 0F C0 /r      XADD r/m8, r8      Exchange r8 and r/m8; load sum into r/m8 */
case EMU86_OPCODE_ENCODE(0x0fc0):
	/* 0F C1 /r      XADD r/m16, r16      Exchange r16 and r/m16; load sum into r/m16
	 * 0F C1 /r      XADD r/m32, r32      Exchange r32 and r/m32; load sum into r/m32
	 * 0F C1 /r      XADD r/m64, r64      Exchange r64 and r/m64; load sum into r/m64 */
case EMU86_OPCODE_ENCODE(0x0fc1):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_XADD) */


#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_XCHG_RM)
	/* 86 /r      XCHG r/m8, r8      Exchange r8 (byte register) with byte from r/m8 */
	/* 86 /r      XCHG r8, r/m8      Exchange byte from r/m8 with r8 (byte register) */
case EMU86_OPCODE_ENCODE(0x86):
	/* 87 /r      XCHG r/m16, r16      Exchange r16 with word from r/m16
	 * 87 /r      XCHG r16, r/m16      Exchange word from r/m16 with r16
	 * 87 /r      XCHG r/m32, r32      Exchange r32 with doubleword from r/m32
	 * 87 /r      XCHG r32, r/m32      Exchange doubleword from r/m32 with r32
	 * 87 /r      XCHG r/m64, r64      Exchange r64 with quadword from r/m64
	 * 87 /r      XCHG r64, r/m64      Exchange quadword from r/m64 with r64 */
case EMU86_OPCODE_ENCODE(0x87):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_XCHG_RM) */


#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_ARITH)
	/* 80 /0-6 ib      FOO r/m8,imm8 */
case EMU86_OPCODE_ENCODE(0x80):
	/* 81 /0-6 iq      FOO r/m16,imm16
	 * 81 /0-6 iq      FOO r/m32,imm32
	 * 81 /0-6 iq      FOO r/m64,Simm32 */
case EMU86_OPCODE_ENCODE(0x81):
	/* 81 /0-6 iq      FOO r/m16,Simm8
	 * 81 /0-6 iq      FOO r/m32,Simm8
	 * 81 /0-6 iq      FOO r/m64,Simm8 */
case EMU86_OPCODE_ENCODE(0x83):
	emu86_modrm_decode(pc, &modrm, op_flags);
	if unlikely(modrm.mi_reg > 6) /* cmp, ... */
		goto return_unexpected_lock_rmreg;
#define NEED_return_unexpected_lock_rmreg
	goto checklock_modrm_memory_parsed;
#define NEED_checklock_modrm_memory_parsed

	/* 00 /r     ADD r/m8, r8       Add r8 to r/m8.
	 * 01 /r     ADD r/m16, r16     Add r16 to r/m16.
	 * 01 /r     ADD r/m32, r32     Add r32 to r/m32.
	 * 01 /r     ADD r/m64, r64     Add r64 to r/m64. */
case EMU86_OPCODE_ENCODE(0x00):
case EMU86_OPCODE_ENCODE(0x01):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory

	/* 08 /r     OR r/m8, r8        r/m8 OR r8.
	 * 09 /r     OR r/m16, r16      r/m16 OR r16.
	 * 09 /r     OR r/m32, r32      r/m32 OR r32.
	 * 09 /r     OR r/m64, r64      r/m64 OR r64. */
case EMU86_OPCODE_ENCODE(0x08):
case EMU86_OPCODE_ENCODE(0x09):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory

	/* 10 /r     ADC r/m8, r8       Add with carry r8 to r/m8.
	 * 11 /r     ADC r/m16, r16     Add with carry r16 to r/m16.
	 * 11 /r     ADC r/m32, r32     Add with carry r32 to r/m32.
	 * 11 /r     ADC r/m64, r64     Add with carry r64 to r/m64. */
case EMU86_OPCODE_ENCODE(0x10):
case EMU86_OPCODE_ENCODE(0x11):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory

	/* 18 /r     SBB r/m8, r8       Subtract with borrow r8 from r/m8
	 * 19 /r     SBB r/m16, r16     Subtract with borrow r16 from r/m16
	 * 19 /r     SBB r/m32, r32     Subtract with borrow r32 from r/m32
	 * 19 /r     SBB r/m64, r64     Subtract with borrow r64 from r/m64 */
case EMU86_OPCODE_ENCODE(0x18):
case EMU86_OPCODE_ENCODE(0x19):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory

	/* 20 /r     AND r/m8, r8       r/m8 AND r8.
	 * 21 /r     AND r/m16, r16     r/m16 AND r16.
	 * 21 /r     AND r/m32, r32     r/m32 AND r32.
	 * 21 /r     AND r/m64, r64     r/m64 AND r64. */
case EMU86_OPCODE_ENCODE(0x20):
case EMU86_OPCODE_ENCODE(0x21):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory

	/* 28 /r     SUB r/m8, r8       Subtract r8 from r/m8.
	 * 29 /r     SUB r/m16, r16     Subtract r16 from r/m16.
	 * 29 /r     SUB r/m32, r32     Subtract r32 from r/m32.
	 * 29 /r     SUB r/m64, r64     Subtract r64 from r/m64. */
case EMU86_OPCODE_ENCODE(0x28):
case EMU86_OPCODE_ENCODE(0x29):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory

	/* 30 /r     XOR r/m8, r8       r/m8 XOR r8.
	 * 31 /r     XOR r/m16, r16     r/m16 XOR r16.
	 * 31 /r     XOR r/m32, r32     r/m32 XOR r32.
	 * 31 /r     XOR r/m64, r64     r/m64 XOR r64. */
case EMU86_OPCODE_ENCODE(0x30):
case EMU86_OPCODE_ENCODE(0x31):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory

#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_ARITH) */


#if EMU86_EMULATE_CONFIG_LOCK_SHIFT /* Extension!!! */
#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_SHIFT)
	/* D2 /X      FOO r/m8,CL */
case EMU86_OPCODE_ENCODE(0xd2):
	/* D3 /X      FOO r/m16,CL */
	/* D3 /X      FOO r/m32,CL */
	/* D3 /X      FOO r/m64,CL */
case EMU86_OPCODE_ENCODE(0xd3):
	/* C0 /X ib   FOO r/m8,imm8 */
case EMU86_OPCODE_ENCODE(0xc0):
	/* D1 /X ib   FOO r/m16,imm8 */
	/* D1 /X ib   FOO r/m32,imm8 */
	/* D1 /X ib   FOO r/m64,imm8 */
case EMU86_OPCODE_ENCODE(0xc1):
	/* D0 /X      FOO r/m8,1 */
case EMU86_OPCODE_ENCODE(0xd0):
	/* D1 /X      FOO r/m16,1 */
	/* D1 /X      FOO r/m32,1 */
	/* D1 /X      FOO r/m64,1 */
case EMU86_OPCODE_ENCODE(0xd1):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_SHIFT) */
#endif /* EMU86_EMULATE_CONFIG_LOCK_SHIFT */


#if EMU86_EMULATE_CONFIG_LOCK_SHIFT2 /* Extension!!! */
#if HAVEOP(EMU86_EMULATE_CONFIG_WANT_SHIFT2)
	/* 0F A5      SHLD r/m16, r16, CL      Shift r/m16 to left CL places while shifting bits from r16 in from the right
	 * 0F A5      SHLD r/m32, r32, CL      Shift r/m32 to left CL places while shifting bits from r32 in from the right
	 * 0F A5      SHLD r/m64, r64, CL      Shift r/m64 to left CL places while shifting bits from r64 in from the right */
case EMU86_OPCODE_ENCODE(0x0fa5):
	/* 0F A4      SHLD r/m16, r16, imm8      Shift r/m16 to left imm8 places while shifting bits from r16 in from the right
	 * 0F A4      SHLD r/m32, r32, imm8      Shift r/m32 to left imm8 places while shifting bits from r32 in from the right
	 * 0F A4      SHLD r/m64, r64, imm8      Shift r/m64 to left imm8 places while shifting bits from r64 in from the right */
case EMU86_OPCODE_ENCODE(0x0fa4):
	/* 0F AD      SHRD r/m16, r16, CL      Shift r/m16 to right CL places while shifting bits from r16 in from the left
	 * 0F AD      SHRD r/m32, r32, CL      Shift r/m32 to right CL places while shifting bits from r32 in from the left
	 * 0F AD      SHRD r/m64, r64, CL      Shift r/m64 to right CL places while shifting bits from r64 in from the left */
case EMU86_OPCODE_ENCODE(0x0fad):
	/* 0F AC      SHRD r/m16, r16, imm8      Shift r/m16 to right imm8 places while shifting bits from r16 in from the left
	 * 0F AC      SHRD r/m32, r32, imm8      Shift r/m32 to right imm8 places while shifting bits from r32 in from the left
	 * 0F AC      SHRD r/m64, r64, imm8      Shift r/m64 to right imm8 places while shifting bits from r64 in from the left */
case EMU86_OPCODE_ENCODE(0x0fac):
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_SHIFT2) */
#endif /* EMU86_EMULATE_CONFIG_LOCK_SHIFT2 */


#if EMU86_EMULATE_CONFIG_LOCK_ARPL /* Extension!!! */
#if (HAVEOP(EMU86_EMULATE_CONFIG_WANT_ARPL)) && (LIBEMU86_CONFIG_WANT_32BIT || LIBEMU86_CONFIG_WANT_16BIT)
	/* 63 /r     ARPL r/m16, r16     Adjust RPL of r/m16 to not less than RPL of r16. */
case EMU86_OPCODE_ENCODE(0x63):
#if EMU86_EMULATE_CONFIG_WANT_MOVSXD && LIBEMU86_CONFIG_WANT_64BIT
	if (EMU86_F_IS64(op_flags)) {
		goto return_unexpected_lock;
#define NEED_return_unexpected_lock
	}
#endif /* EMU86_EMULATE_CONFIG_WANT_MOVSXD && LIBEMU86_CONFIG_WANT_64BIT */
	goto checklock_modrm_memory;
#define NEED_checklock_modrm_memory
#endif /* HAVEOP(EMU86_EMULATE_CONFIG_WANT_ARPL) */
#endif /* EMU86_EMULATE_CONFIG_LOCK_ARPL */


#undef HAVEOP
#ifdef NEED_checklock_modrm_memory
#undef NEED_checklock_modrm_memory
checklock_modrm_memory:
	emu86_modrm_decode(pc, &modrm, op_flags);
	goto checklock_modrm_memory_parsed;
#define NEED_checklock_modrm_memory_parsed
#endif /* NEED_checklock_modrm_memory */

#ifdef NEED_checklock_modrm_memory_parsed
#undef NEED_checklock_modrm_memory_parsed
checklock_modrm_memory_parsed:
	if unlikely(!EMU86_MODRM_ISMEM(modrm.mi_type))
		goto return_unexpected_lock;
#define NEED_return_unexpected_lock
	break;
#endif /* NEED_checklock_modrm_memory_parsed */

				default:
					goto return_unexpected_lock;
#define NEED_return_unexpected_lock
				}
			}
#endif /* EMU86_EMULATE_CONFIG_CHECKLOCK */
			switch (tiny_opcode) {

				/* Pull in emulated instructions. */
#ifndef __INTELLISENSE__
#ifdef EMU86_EMULATE_IMPL_HEADER
/* Allow the user to specify a custom header that gets included for
 * the  purpose  of defining  actually emulated  instructions, thus
 * allowing the  user to  restrict the  set of  instructions  being
 * emulated. */
#include EMU86_EMULATE_IMPL_HEADER
#else /* EMU86_EMULATE_IMPL_HEADER */
#include "emu/_notsup.c.inl"
#include "emu/adcx.c.inl"
#include "emu/andn.c.inl"
#include "emu/arith.c.inl"
#include "emu/arith2.c.inl"
#include "emu/bcd.c.inl"
#include "emu/bitscan.c.inl"
#include "emu/bittest.c.inl"
#include "emu/blsx.c.inl"
#include "emu/bound.c.inl"
#include "emu/bswap.c.inl"
#include "emu/call.c.inl"
#include "emu/cbw.c.inl"
#include "emu/clts.c.inl"
#include "emu/cmovcc.c.inl"
#include "emu/cmps.c.inl"
#include "emu/cmpxchg.c.inl"
#include "emu/cmpxchgb.c.inl"
#include "emu/cpuid.c.inl"
#include "emu/enter.c.inl"
#include "emu/flush.c.inl"
#include "emu/hlt.c.inl"
#include "emu/incdec.c.inl"
#include "emu/int.c.inl"
#include "emu/io.c.inl"
#include "emu/iret.c.inl"
#include "emu/jcc.c.inl"
#include "emu/jmp.c.inl"
#include "emu/lcall.c.inl"
#include "emu/lea.c.inl"
#include "emu/leave.c.inl"
#include "emu/ljmp.c.inl"
#include "emu/lods.c.inl"
#include "emu/loop.c.inl"
#include "emu/lret.c.inl"
#include "emu/lxs.c.inl"
#include "emu/misc.c.inl"
#include "emu/misc2.c.inl"
#include "emu/mov-creg.c.inl"
#include "emu/mov-dreg.c.inl"
#include "emu/mov-imm.c.inl"
#include "emu/mov-moffs.c.inl"
#include "emu/mov-sreg.c.inl"
#include "emu/mov.c.inl"
#include "emu/movbe.c.inl"
#include "emu/movdir64b.c.inl"
#include "emu/movnti.c.inl"
#include "emu/movs.c.inl"
#include "emu/movsx.c.inl"
#include "emu/movsxd.c.inl"
#include "emu/movzx.c.inl"
#include "emu/msr.c.inl"
#include "emu/nop.c.inl"
#include "emu/pext.c.inl"
#include "emu/pmode.c.inl"
#include "emu/popcnt.c.inl"
#include "emu/push-imm.c.inl"
#include "emu/push-pop-sreg.c.inl"
#include "emu/push-pop.c.inl"
#include "emu/pusha-popa.c.inl"
#include "emu/pushf-popf.c.inl"
#include "emu/ret.c.inl"
#include "emu/rorx.c.inl"
#include "emu/sahf-lahf.c.inl"
#include "emu/salc.c.inl"
#include "emu/scas.c.inl"
#include "emu/shift.c.inl"
#include "emu/shift2.c.inl"
#include "emu/shiftx.c.inl"
#include "emu/stcf.c.inl"
#include "emu/stdf.c.inl"
#include "emu/stif.c.inl"
#include "emu/stos.c.inl"
#include "emu/syscall.c.inl"
#include "emu/sysenter.c.inl"
#include "emu/sysexit.c.inl"
#include "emu/sysret.c.inl"
#include "emu/xadd.c.inl"
#include "emu/xchg.c.inl"
#include "emu/xlatb.c.inl"
#endif /* !EMU86_EMULATE_IMPL_HEADER */
#endif /* !__INTELLISENSE__ */

			/* XXX: xrstor     (if only for verbose exception messages?) */
			/* XXX: xrstor64   (if only for verbose exception messages?) */
			/* XXX: xsave      (if only for verbose exception messages?) */
			/* XXX: xsave64    (if only for verbose exception messages?) */
			/* XXX: xsaveopt   (if only for verbose exception messages?) */
			/* XXX: xsaveopt64 (if only for verbose exception messages?) */
			/* XXX: xrstors    (if only for verbose exception messages?) */
			/* XXX: xrstors64  (if only for verbose exception messages?) */
			/* XXX: xsaves     (if only for verbose exception messages?) */
			/* XXX: xsaves64   (if only for verbose exception messages?) */
			/* XXX: xsavec     (if only for verbose exception messages?) */
			/* XXX: xsavec64   (if only for verbose exception messages?) */
			/* XXX: fxrstor    (if only for verbose exception messages?) */
			/* XXX: fxrstor64  (if only for verbose exception messages?) */
			/* XXX: fxsave     (if only for verbose exception messages?) */
			/* XXX: fxsave64   (if only for verbose exception messages?) */
			/* XXX: invpcid    (if only for verbose exception messages?) */
			/* XXX: vmptrst    (if only for verbose exception messages?) */
			/* XXX: vmxon      (if only for verbose exception messages?) */
			/* XXX: vmclear    (if only for verbose exception messages?) */
			/* XXX: vmptrld    (if only for verbose exception messages?) */
			/* XXX: vmcall     (if only for verbose exception messages?) */
			/* XXX: vmlaunch   (if only for verbose exception messages?) */
			/* XXX: vmresume   (if only for verbose exception messages?) */
			/* XXX: vmxoff     (if only for verbose exception messages?) */
			/* XXX: vmfunc     (if only for verbose exception messages?) */
			/* XXX: vmrun      (if only for verbose exception messages?) */
			/* XXX: vmmcall    (if only for verbose exception messages?) */
			/* XXX: vmload     (if only for verbose exception messages?) */
			/* XXX: vmsave     (if only for verbose exception messages?) */
			/* XXX: vmreadl    (if only for verbose exception messages?) */
			/* XXX: vmreadq    (if only for verbose exception messages?) */
			/* XXX: vmwritel   (if only for verbose exception messages?) */
			/* XXX: vmwriteq   (if only for verbose exception messages?) */
			/* XXX: montmul    (if only for verbose exception messages?) */
			/* XXX: xsha1      (if only for verbose exception messages?) */
			/* XXX: xsha256    (if only for verbose exception messages?) */
			/* XXX: xstore     (if only for verbose exception messages?) */
			/* XXX: xcrypt-ecb (if only for verbose exception messages?) */
			/* XXX: xcrypt-cbc (if only for verbose exception messages?) */
			/* XXX: xcrypt-ctr (if only for verbose exception messages?) */
			/* XXX: xcrypt-cfb (if only for verbose exception messages?) */
			/* XXX: xcrypt-ofb (if only for verbose exception messages?) */
			/* XXX: umonitorl  (if only for verbose exception messages?) */
			/* XXX: umonitorq  (if only for verbose exception messages?) */
			/* XXX: umwait     (if only for verbose exception messages?) */
			/* XXX: skinit     (if only for verbose exception messages?) */
			/* XXX: invlpga    (if only for verbose exception messages?) */
			/* XXX: monitorx   (if only for verbose exception messages?) */
			/* XXX: mwaitx     (if only for verbose exception messages?) */
			/* XXX: rdpru      (if only for verbose exception messages?) */
			/* XXX: rdsspd     (if only for verbose exception messages?) */
			/* XXX: rdsspq     (if only for verbose exception messages?) */
			/* XXX: endbr64    (if only for verbose exception messages?) */
			/* XXX: endbr32    (if only for verbose exception messages?) */
			/* XXX: getsec     (if only for verbose exception messages?) */
			/* XXX: 0x0f3f:jmp (if only for verbose exception messages?) */

			/* TODO: Emulate crc32 */

// XXX: I(0x80, IF_X32|IF_66|IF_MODRM|IF_RMM,"inveptl\t" OP_MEM OP_R32),  /* 66 0F 38 80 /r     INVEPT r64, m128     Invalidates EPT-derived entries in the TLBs and paging-structure caches (in 64-bit mode). */
// XXX: I(0x80, IF_X64|IF_66|IF_MODRM|IF_RMM,"inveptq\t" OP_MEM OP_R64),  /* 66 0F 38 80 /r     INVEPT r32, m128     Invalidates EPT-derived entries in the TLBs and paging-structure caches (outside 64-bit mode). */
// XXX: I(0x81, IF_X32|IF_66|IF_MODRM|IF_RMM,"invvpidl\t" OP_MEM OP_R32), /* 66 0F 38 81 /r     INVVPID r64, m128     Invalidates entries in the TLBs and paging-structure caches based on VPID (in 64-bit mode). */
// XXX: I(0x81, IF_X64|IF_66|IF_MODRM|IF_RMM,"invvpidq\t" OP_MEM OP_R64), /* 66 0F 38 81 /r     INVVPID r32, m128     Invalidates entries in the TLBs and paging-structure caches based on VPID (outside 64-bit mode). */
// XXX: I(0x82, IF_X32|IF_66|IF_MODRM|IF_RMM,"invpcidl\t" OP_MEM OP_R32), /* 66 0F 38 82 /r     INVPCID r32, m128     Invalidates entries in the TLBs and paging-structure caches based on invalidation type in r32 and descriptor in m128. */
// XXX: I(0x82, IF_X64|IF_66|IF_MODRM|IF_RMM,"invpcidq\t" OP_MEM OP_R64), /* 66 0F 38 82 /r     INVPCID r64, m128     Invalidates entries in the TLBs and paging-structure caches based on invalidation type in r64 and descriptor in m128. */

// ???:	I(0x1a, IF_X32|IF_F2|IF_MODRM, "bndcu\t" OP_RM32 OP_RBND),
// ???:	I(0x1a, IF_X64|IF_F2|IF_MODRM, "bndcu\t" OP_RM64 OP_RBND),
// ???:	I(0x1a, IF_X32|IF_F3|IF_MODRM, "bndcl\t" OP_RM32 OP_RBND),
// ???:	I(0x1a, IF_X64|IF_F3|IF_MODRM, "bndcl\t" OP_RM64 OP_RBND),
// ???:	I(0x1a, IF_MODRM,              "bndldx\t" OP_RMBND_RANGE OP_RBND),
// ???:	I(0x1a, IF_66|IF_MODRM,        "bndmov\t" OP_RMBND OP_RBND),
// ???:	I(0x1b, IF_X32|IF_F2|IF_MODRM, "bndcn\t" OP_RM32 OP_RBND),
// ???:	I(0x1b, IF_X64|IF_F2|IF_MODRM, "bndcn\t" OP_RM64 OP_RBND),
// ???:	I(0x1b, IF_66|IF_MODRM,        "bndmov\t" OP_RBND OP_RMBND),
// ???:	I(0x1b, IF_X32|IF_F3|IF_MODRM, "bndmk\t" OP_RM32 OP_RBND),
// ???:	I(0x1b, IF_X64|IF_F3|IF_MODRM, "bndmk\t" OP_RM64 OP_RBND),

// ???:	I(0x24, IF_MODRM|IF_RMR, "movl\t" OP_RTR OP_RM32),
// ???:	I(0x26, IF_MODRM|IF_RMR, "movl\t" OP_RM32 OP_RTR),



			/* TODO: XOP instructions (from AMD):
			 *
			 * LLWPCB reg32                         8F RXB.09 0.1111.0.00 12 /0
			 * LLWPCB reg64                         8F RXB.09 1.1111.0.00 12 /0
			 * SLWPCB reg32                         8F RXB.09 0.1111.0.00 12 /1
			 * SLWPCB reg64                         8F RXB.09 1.1111.0.00 12 /1
			 *
			 * LWPINS reg32.vvvv, reg/mem32, imm32 8F RXB.0A 0.src1.0.00 12 /0 /imm32
			 * LWPINS reg64.vvvv, reg/mem32, imm32 8F RXB.0A 1.src1.0.00 12 /0 /imm32
			 * LWPVAL reg32.vvvv, reg/mem32, imm32 8F RXB.0A 0.src1.0.00 12 /1 /imm32
			 * LWPVAL reg64.vvvv, reg/mem32, imm32 8F RXB.0A 1.src1.0.00 12 /1 /imm32
			 */

			default:
				goto return_unknown_instruction;


/* Helpers for handling unsupported instructions. */
#ifdef NEED_notsup_stosb_userprivileged
#undef NEED_notsup_stosb_userprivileged
notsup_stosb_userprivileged:
#ifndef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
		{
			void *baseaddr;
			EMU86_ADDRSIZE_SWITCH({
				baseaddr = (void *)(uintptr_t)EMU86_GETRDI();
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETESBASE(), EMU86_GETEDI());
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETDSBASE(), EMU86_GETDI());
			});
			EMU86_UNSUPPORTED_MEMACCESS(baseaddr, 1, false, true);
		}
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
		goto notsup_userprivileged;
#define NEED_notsup_userprivileged
#endif /* NEED_notsup_stosb_userprivileged */


#ifdef NEED_notsup_stoswlq_userprivileged
#undef NEED_notsup_stoswlq_userprivileged
notsup_stoswlq_userprivileged:
#ifndef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
		{
			void *baseaddr;
			size_t num_bytes = IF_64BIT(IS_64BIT() ? 8 :) !IS_16BIT() ? 4 : 2;
			EMU86_ADDRSIZE_SWITCH({
				baseaddr = (void *)(uintptr_t)EMU86_GETRDI();
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETESBASE(), EMU86_GETEDI());
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETDSBASE(), EMU86_GETDI());
			});
			EMU86_UNSUPPORTED_MEMACCESS(baseaddr, num_bytes, false, true);
		}
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
		goto notsup_userprivileged;
#define NEED_notsup_userprivileged
#endif /* NEED_notsup_stoswlq_userprivileged */


#ifdef NEED_notsup_lodsb_userprivileged
#undef NEED_notsup_lodsb_userprivileged
notsup_lodsb_userprivileged:
#ifndef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
		{
			void *baseaddr;
			EMU86_ADDRSIZE_SWITCH({
				baseaddr = (void *)(uintptr_t)EMU86_GETRDI();
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETESBASE(), EMU86_GETEDI());
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETDSBASE(), EMU86_GETDI());
			});
			EMU86_UNSUPPORTED_MEMACCESS(baseaddr, 1, true, false);
		}
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
		goto notsup_userprivileged;
#define NEED_notsup_userprivileged
#endif /* NEED_notsup_lodsb_userprivileged */


#ifdef NEED_notsup_lodswlq_userprivileged
#undef NEED_notsup_lodswlq_userprivileged
notsup_lodswlq_userprivileged:
#ifndef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
		{
			void *baseaddr;
			size_t num_bytes = IF_64BIT(IS_64BIT() ? 8 :) !IS_16BIT() ? 4 : 2;
			EMU86_ADDRSIZE_SWITCH({
				baseaddr = (void *)(uintptr_t)EMU86_GETRDI();
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETESBASE(), EMU86_GETEDI());
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETDSBASE(), EMU86_GETDI());
			});
			EMU86_UNSUPPORTED_MEMACCESS(baseaddr, num_bytes, true, false);
		}
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
		goto notsup_userprivileged;
#define NEED_notsup_userprivileged
#endif /* NEED_notsup_lodswlq_userprivileged */


#ifdef NEED_notsup_userprivileged
#undef NEED_notsup_userprivileged
notsup_userprivileged:
		if (EMU86_ISUSER())
			goto return_privileged_instruction;
#define NEED_return_privileged_instruction
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_userprivileged */


#ifdef NEED_notsup_stosb
#undef NEED_notsup_stosb
notsup_stosb:
#ifndef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
		{
			void *baseaddr;
			EMU86_ADDRSIZE_SWITCH({
				baseaddr = (void *)(uintptr_t)EMU86_GETRDI();
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETESBASE(), EMU86_GETEDI());
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETDSBASE(), EMU86_GETDI());
			});
			EMU86_UNSUPPORTED_MEMACCESS(baseaddr, 1, false, true);
		}
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_stosb */


#ifdef NEED_notsup_stoswlq
#undef NEED_notsup_stoswlq
notsup_stoswlq:
#ifndef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
		{
			void *baseaddr;
			size_t num_bytes = IF_64BIT(IS_64BIT() ? 8 :) !IS_16BIT() ? 4 : 2;
			EMU86_ADDRSIZE_SWITCH({
				baseaddr = (void *)(uintptr_t)EMU86_GETRDI();
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETESBASE(), EMU86_GETEDI());
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETDSBASE(), EMU86_GETDI());
			});
			EMU86_UNSUPPORTED_MEMACCESS(baseaddr, num_bytes, false, true);
		}
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_stoswlq */


#ifdef NEED_notsup_lodsb
#undef NEED_notsup_lodsb
notsup_lodsb:
#ifndef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
		{
			void *baseaddr;
			EMU86_ADDRSIZE_SWITCH({
				baseaddr = (void *)(uintptr_t)EMU86_GETRDI();
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETESBASE(), EMU86_GETEDI());
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETDSBASE(), EMU86_GETDI());
			});
			EMU86_UNSUPPORTED_MEMACCESS(baseaddr, 1, true, false);
		}
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_lodsb */


#ifdef NEED_notsup_lodswlq
#undef NEED_notsup_lodswlq
notsup_lodswlq:
#ifndef EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP
		{
			void *baseaddr;
			size_t num_bytes = IF_64BIT(IS_64BIT() ? 8 :) !IS_16BIT() ? 4 : 2;
			EMU86_ADDRSIZE_SWITCH({
				baseaddr = (void *)(uintptr_t)EMU86_GETRDI();
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETESBASE(), EMU86_GETEDI());
			}, {
				baseaddr = EMU86_SEGADDR(EMU86_GETDSBASE(), EMU86_GETDI());
			});
			EMU86_UNSUPPORTED_MEMACCESS(baseaddr, num_bytes, true, false);
		}
#endif /* !EMU86_UNSUPPORTED_MEMACCESS_IS_NOOP */
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_lodswlq */


#ifdef NEED_notsup_modrm_getlq_vex_w_rmreg_modrm_parsed
#undef NEED_notsup_modrm_getlq_vex_w_rmreg_modrm_parsed
notsup_modrm_getlq_vex_w_rmreg_modrm_parsed:
#if LIBEMU86_CONFIG_WANT_64BIT
		if (op_flags & EMU86_F_VEX_W) {
			goto notsup_modrm_getq_rmreg_modrm_parsed;
#define NEED_notsup_modrm_getq_rmreg_modrm_parsed
		}
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
		goto notsup_modrm_getl_rmreg_modrm_parsed;
#define NEED_notsup_modrm_getl_rmreg_modrm_parsed
#endif /* NEED_notsup_modrm_getlq_vex_w_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_getq_rmreg_modrm_parsed
#undef NEED_notsup_modrm_getq_rmreg_modrm_parsed
notsup_modrm_getq_rmreg_modrm_parsed:
		MODRM_NOSUP_GETRMQ();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_getq_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_getl_rmreg_modrm_parsed
#undef NEED_notsup_modrm_getl_rmreg_modrm_parsed
notsup_modrm_getl_rmreg_modrm_parsed:
		MODRM_NOSUP_GETRML();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_getl_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_setl_rmreg_modrm_parsed
#undef NEED_notsup_modrm_setl_rmreg_modrm_parsed
notsup_modrm_setl_rmreg_modrm_parsed:
		MODRM_NOSUP_SETRML();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_setl_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_setwlq_rmreg
#undef NEED_notsup_modrm_setwlq_rmreg
notsup_modrm_setwlq_rmreg:
#if LIBEMU86_CONFIG_WANT_64BIT || defined(EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG)
		MODRM_DECODE();
#if LIBEMU86_CONFIG_WANT_64BIT
		if (modrm.mi_reg >= 8) {
#define NEED_return_unknown_instruction_rmreg
			goto return_unknown_instruction_rmreg;
		}
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
#define NEED_notsup_modrm_setwlq_rmreg_modrm_parsed
		goto notsup_modrm_setwlq_rmreg_modrm_parsed;
#else /* LIBEMU86_CONFIG_WANT_64BIT || EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
		goto notsup_modrm_setwlq;
#define NEED_notsup_modrm_setwlq
#endif /* !LIBEMU86_CONFIG_WANT_64BIT && !EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
#endif /* NEED_notsup_modrm_setwlq_rmreg */


#ifdef NEED_notsup_modrm_setwlq_rmreg_modrm_parsed_popwlq
#undef NEED_notsup_modrm_setwlq_rmreg_modrm_parsed_popwlq
notsup_modrm_setwlq_rmreg_modrm_parsed_popwlq:
		EMU86_POP163264_NOSUP();
		goto notsup_modrm_setwlq_rmreg_modrm_parsed;
#define NEED_notsup_modrm_setwlq_rmreg_modrm_parsed
#endif /* NEED_notsup_modrm_setwlq_rmreg_modrm_parsed_popwlq */


#ifdef NEED_notsup_modrm_setwlq_rmreg_modrm_parsed
#undef NEED_notsup_modrm_setwlq_rmreg_modrm_parsed
notsup_modrm_setwlq_rmreg_modrm_parsed:
		MODRM_NOSUP_SETRMWLQ();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_setwlq_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_setlq_memonly
#undef NEED_notsup_modrm_setlq_memonly
notsup_modrm_setlq_memonly:
		MODRM_DECODE_MEMONLY();
#define NEED_return_expected_memory_modrm
		MODRM_NOSUP_SETRMZ(IF_64BIT(IS_64BIT() ? 8 :) 4);
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_setlq_memonly */


#ifdef NEED_notsup_modrm_setwlq
#undef NEED_notsup_modrm_setwlq
notsup_modrm_setwlq:
		MODRM_DECODE();
		MODRM_NOSUP_SETRMWLQ();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_setwlq */


#ifdef NEED_notsup_modrm_setb_rmreg
#undef NEED_notsup_modrm_setb_rmreg
notsup_modrm_setb_rmreg:
#if LIBEMU86_CONFIG_WANT_64BIT || defined(EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG)
		MODRM_DECODE();
		goto notsup_modrm_setb_rmreg_modrm_parsed;
#define NEED_notsup_modrm_setb_rmreg_modrm_parsed
#else /* LIBEMU86_CONFIG_WANT_64BIT || EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
		goto notsup_modrm_setb;
#define NEED_notsup_modrm_setb
#endif /* !LIBEMU86_CONFIG_WANT_64BIT && !EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
#endif /* NEED_notsup_modrm_setb_rmreg */


#ifdef NEED_notsup_modrm_setb_rmreg_modrm_parsed
#undef NEED_notsup_modrm_setb_rmreg_modrm_parsed
notsup_modrm_setb_rmreg_modrm_parsed:
#if LIBEMU86_CONFIG_WANT_64BIT
		if (modrm.mi_reg >= 8) {
#define NEED_return_unknown_instruction_rmreg
			goto return_unknown_instruction_rmreg;
		}
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
		MODRM_NOSUP_SETRMB();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_setb_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_setb
#undef NEED_notsup_modrm_setb
notsup_modrm_setb:
		MODRM_DECODE();
		MODRM_NOSUP_SETRMB();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_setb */


#ifdef NEED_notsup_modrm_getz_rex_w
#undef NEED_notsup_modrm_getz_rex_w
notsup_modrm_getz_rex_w:
		MODRM_DECODE();
		goto notsup_modrm_getz_rex_w_modrm_parsed;
#define NEED_notsup_modrm_getz_rex_w_modrm_parsed
#endif /* NEED_notsup_modrm_getz_rex_w */


#ifdef NEED_notsup_modrm_getz_rex_w_modrm_parsed_rmreg
#undef NEED_notsup_modrm_getz_rex_w_modrm_parsed_rmreg
notsup_modrm_getz_rex_w_modrm_parsed_rmreg:
		MODRM_NOSUP_GETRMZ_VEX_W();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_getz_rex_w_modrm_parsed_rmreg */


#ifdef NEED_notsup_modrm_getz_rex_w_modrm_parsed
#undef NEED_notsup_modrm_getz_rex_w_modrm_parsed
notsup_modrm_getz_rex_w_modrm_parsed:
		MODRM_NOSUP_GETRMZ_VEX_W();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_getz_rex_w_modrm_parsed */


#ifdef NEED_notsup_modrm_getwlq_rmreg
#undef NEED_notsup_modrm_getwlq_rmreg
notsup_modrm_getwlq_rmreg:
#if LIBEMU86_CONFIG_WANT_64BIT || defined(EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG)
		MODRM_DECODE();
#if LIBEMU86_CONFIG_WANT_64BIT
		if (modrm.mi_reg >= 8) {
#define NEED_return_unknown_instruction_rmreg
			goto return_unknown_instruction_rmreg;
		}
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
		MODRM_NOSUP_GETRMWLQ();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#else /* LIBEMU86_CONFIG_WANT_64BIT || EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
		goto notsup_modrm_getwlq;
#define NEED_notsup_modrm_getwlq
#endif /* !LIBEMU86_CONFIG_WANT_64BIT && !EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
#endif /* NEED_notsup_modrm_getwlq_rmreg */


#ifdef NEED_notsup_modrm_getwlq_rmreg_modrm_parsed_pushwlq
#undef NEED_notsup_modrm_getwlq_rmreg_modrm_parsed_pushwlq
#include "emu/push-pop-util.h"
notsup_modrm_getwlq_rmreg_modrm_parsed_pushwlq:
		EMU86_PUSH163264_NOSUP();
		goto notsup_modrm_getwlq_rmreg_modrm_parsed;
#define NEED_notsup_modrm_getwlq_rmreg_modrm_parsed
#endif /* NEED_notsup_modrm_getwlq_rmreg_modrm_parsed_pushwlq */


#ifdef NEED_notsup_modrm_getwlq64_rmreg_modrm_parsed_pushwlq
#undef NEED_notsup_modrm_getwlq64_rmreg_modrm_parsed_pushwlq
#include "emu/push-pop-util.h"
notsup_modrm_getwlq64_rmreg_modrm_parsed_pushwlq:
		EMU86_PUSH163264_FORCE64_NOSUP();
		goto notsup_modrm_getwlq64_rmreg_modrm_parsed;
#define NEED_notsup_modrm_getwlq64_rmreg_modrm_parsed
#endif /* NEED_notsup_modrm_getwlq64_rmreg_modrm_parsed_pushwlq */


#ifdef NEED_notsup_modrm_getwlq_rmreg_modrm_parsed
#undef NEED_notsup_modrm_getwlq_rmreg_modrm_parsed
notsup_modrm_getwlq_rmreg_modrm_parsed:
		MODRM_NOSUP_GETRMWLQ();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_getwlq_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_getwlq64_rmreg_modrm_parsed
#undef NEED_notsup_modrm_getwlq64_rmreg_modrm_parsed
notsup_modrm_getwlq64_rmreg_modrm_parsed:
		/* IN_64_BIT_MODE ? 8 : PREFIX_66h_GIVEN ? 2 : 4 */
		MODRM_NOSUP_GETRMZ(IF_64BIT(IF_16BIT_OR_32BIT(EMU86_F_IS64(op_flags) ? ) 8
		                            IF_16BIT_OR_32BIT(:))
		                   IF_16BIT_OR_32BIT(IS_16BIT() ? 2 : 4));
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_getwlq64_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_getwlq
#undef NEED_notsup_modrm_getwlq
notsup_modrm_getwlq:
		MODRM_DECODE();
		MODRM_NOSUP_GETRMWLQ();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_getwlq */


#ifdef NEED_notsup_modrm_getb_rmreg
#undef NEED_notsup_modrm_getb_rmreg
notsup_modrm_getb_rmreg:
#if LIBEMU86_CONFIG_WANT_64BIT || defined(EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG)
		MODRM_DECODE();
#if LIBEMU86_CONFIG_WANT_64BIT
		if (modrm.mi_reg >= 8) {
#define NEED_return_unknown_instruction_rmreg
			goto return_unknown_instruction_rmreg;
		}
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
		MODRM_NOSUP_GETRMB();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#else /* LIBEMU86_CONFIG_WANT_64BIT || EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
		goto notsup_modrm_getb;
#define NEED_notsup_modrm_getb
#endif /* !LIBEMU86_CONFIG_WANT_64BIT && !EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
#endif /* NEED_notsup_modrm_getb_rmreg */


#ifdef NEED_notsup_modrm_getb
#undef NEED_notsup_modrm_getb
notsup_modrm_getb:
		MODRM_DECODE();
		MODRM_NOSUP_GETRMB();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_getb */


#ifdef NEED_notsup_modrm_getw
#undef NEED_notsup_modrm_getw
notsup_modrm_getw:
		MODRM_DECODE();
		MODRM_NOSUP_GETRMW();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_getw */


#ifdef NEED_notsup_modrm_getsetb_rmreg
#undef NEED_notsup_modrm_getsetb_rmreg
notsup_modrm_getsetb_rmreg:
#if (LIBEMU86_CONFIG_WANT_64BIT || defined(EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG))
		MODRM_DECODE();
#if LIBEMU86_CONFIG_WANT_64BIT
		if (modrm.mi_reg >= 8) {
#define NEED_return_unknown_instruction_rmreg
			goto return_unknown_instruction_rmreg;
		}
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
		goto notsup_modrm_getsetb_rmreg_modrm_parsed;
#define NEED_notsup_modrm_getsetb_rmreg_modrm_parsed
#else /* LIBEMU86_CONFIG_WANT_64BIT || EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
		goto notsup_modrm_getsetb;
#define NEED_notsup_modrm_getsetb
#endif /* !LIBEMU86_CONFIG_WANT_64BIT && !EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
#endif /* NEED_notsup_modrm_getsetb_rmreg */


#ifdef NEED_notsup_modrm_getsetb_rmreg_modrm_parsed
#undef NEED_notsup_modrm_getsetb_rmreg_modrm_parsed
notsup_modrm_getsetb_rmreg_modrm_parsed:
		MODRM_NOSUP_GETSETRMB();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_getsetb_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_getsetb
#undef NEED_notsup_modrm_getsetb
notsup_modrm_getsetb:
		MODRM_DECODE();
		MODRM_NOSUP_GETSETRMB();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_getsetb */


#ifdef NEED_notsup_modrm_getsetwlq_rmreg
#undef NEED_notsup_modrm_getsetwlq_rmreg
notsup_modrm_getsetwlq_rmreg:
#if LIBEMU86_CONFIG_WANT_64BIT || defined(EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG)
		MODRM_DECODE();
#if LIBEMU86_CONFIG_WANT_64BIT
		if (modrm.mi_reg >= 8) {
#define NEED_return_unknown_instruction_rmreg
			goto return_unknown_instruction_rmreg;
		}
#endif /* LIBEMU86_CONFIG_WANT_64BIT */
		goto notsup_modrm_getsetwlq_rmreg_modrm_parsed;
#define NEED_notsup_modrm_getsetwlq_rmreg_modrm_parsed
#else /* LIBEMU86_CONFIG_WANT_64BIT || EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
		goto notsup_modrm_getsetwlq;
#define NEED_notsup_modrm_getsetwlq
#endif /* !LIBEMU86_CONFIG_WANT_64BIT && !EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
#endif /* NEED_notsup_modrm_getsetwlq_rmreg */


#ifdef NEED_notsup_modrm_getsetwlq_rmreg_modrm_parsed
#undef NEED_notsup_modrm_getsetwlq_rmreg_modrm_parsed
notsup_modrm_getsetwlq_rmreg_modrm_parsed:
		MODRM_NOSUP_GETSETRMWLQ();
		goto return_unsupported_instruction_rmreg;
#define NEED_return_unsupported_instruction_rmreg
#endif /* NEED_notsup_modrm_getsetwlq_rmreg_modrm_parsed */


#ifdef NEED_notsup_modrm_getsetw
#undef NEED_notsup_modrm_getsetw
notsup_modrm_getsetw:
		MODRM_DECODE();
		MODRM_NOSUP_GETSETRMW();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_getsetw */

#ifdef NEED_notsup_modrm_getsetwlq
#undef NEED_notsup_modrm_getsetwlq
notsup_modrm_getsetwlq:
		MODRM_DECODE();
		MODRM_NOSUP_GETSETRMWLQ();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_modrm_getsetwlq */


#ifdef NEED_notsup_pushwlq
#undef NEED_notsup_pushwlq
#include "emu/push-pop-util.h"
notsup_pushwlq:
		EMU86_PUSH163264_NOSUP();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_pushwlq */


#ifdef NEED_notsup_popwlq
#undef NEED_notsup_popwlq
#include "emu/push-pop-util.h"
notsup_popwlq:
		EMU86_POP163264_NOSUP();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_popwlq */


#ifdef NEED_notsup_pushwl
#undef NEED_notsup_pushwl
#include "emu/push-pop-util.h"
#ifndef EMU86_PUSH1632_NOSUP
#error "Unsupported: notsup_pushwl"
#endif /* EMU86_PUSH1632_NOSUP */
notsup_pushwl:
		EMU86_PUSH1632_NOSUP();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_pushwl */


#ifdef NEED_notsup_popwl
#undef NEED_notsup_popwl
#include "emu/push-pop-util.h"
#ifndef EMU86_POP1632_NOSUP
#error "Unsupported: notsup_popwl"
#endif /* EMU86_POP1632_NOSUP */
notsup_popwl:
		EMU86_POP1632_NOSUP();
		goto return_unsupported_instruction;
#define NEED_return_unsupported_instruction
#endif /* NEED_notsup_popwl */


			}
		}
#ifdef EMU86_EMULATE_EXCEPT_SWITCH
		EMU86_EMULATE_EXCEPT_SWITCH;
#endif /* EMU86_EMULATE_EXCEPT_SWITCH */
	}
#ifdef EMU86_EMULATE_EXCEPT
	EMU86_EMULATE_EXCEPT;
#endif /* EMU86_EMULATE_EXCEPT */
done:
	/* Set the new instruction pointer. */
	EMU86_SETPCPTR((void *)REAL_IP());
#ifdef NEED_done_dont_set_pc
#undef NEED_done_dont_set_pc
done_dont_set_pc:
#endif /* NEED_done_dont_set_pc */
	EMU86_EMULATE_RETURN();


#if EMU86_EMULATE_CONFIG_CHECKUSER
	/* Privileged instruction
	 * Only used when `EMU86_EMULATE_CONFIG_CHECKUSER' is enabled. */
	__IF0 {
#ifdef NEED_return_privileged_instruction_rmreg
#undef NEED_return_privileged_instruction_rmreg
return_privileged_instruction_rmreg:;
#ifdef EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION_RMREG
		EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION_RMREG();
#else /* EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION_RMREG */
#define NEED_return_unknown_instruction_rmreg
		goto return_unknown_instruction_rmreg;
#endif /* !EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION_RMREG */
#endif /* NEED_return_privileged_instruction_rmreg */
#ifdef NEED_return_privileged_instruction
#undef NEED_return_privileged_instruction
return_privileged_instruction:;
#ifdef EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION
		EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION();
#endif /* EMU86_EMULATE_RETURN_PRIVILEGED_INSTRUCTION */
#endif /* NEED_return_privileged_instruction */
	}
#endif /* EMU86_EMULATE_CONFIG_CHECKUSER */


	/* Instruction isn't defined for a non-memory modr/m operand */
	__IF0 {
#ifdef NEED_return_expected_memory_modrm_rmreg
#undef NEED_return_expected_memory_modrm_rmreg
return_expected_memory_modrm_rmreg:;
#ifdef EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG
		EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG();
#else /* EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG */
#define NEED_return_unknown_instruction_rmreg
		goto return_unknown_instruction_rmreg;
#endif /* !EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG */
#endif /* NEED_return_expected_memory_modrm_rmreg */
#ifdef NEED_return_expected_memory_modrm
#undef NEED_return_expected_memory_modrm
return_expected_memory_modrm:;
#ifdef EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM
		EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM();
#endif /* EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM */
#endif /* NEED_return_expected_memory_modrm */
	}


	/* Instruction isn't defined for a non-register modr/m operand */
	__IF0 {
#ifdef NEED_return_expected_register_modrm_rmreg
#undef NEED_return_expected_register_modrm_rmreg
return_expected_register_modrm_rmreg:;
#ifdef EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG
		EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG();
#else /* EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG */
#define NEED_return_unknown_instruction_rmreg
		goto return_unknown_instruction_rmreg;
#endif /* !EMU86_EMULATE_RETURN_EXPECTED_MEMORY_MODRM_RMREG */
#endif /* NEED_return_expected_register_modrm_rmreg */
#ifdef NEED_return_expected_register_modrm
#undef NEED_return_expected_register_modrm
return_expected_register_modrm:;
#ifdef EMU86_EMULATE_RETURN_EXPECTED_REGISTER_MODRM
		EMU86_EMULATE_RETURN_EXPECTED_REGISTER_MODRM();
#endif /* EMU86_EMULATE_RETURN_EXPECTED_REGISTER_MODRM */
#endif /* NEED_return_expected_register_modrm */
	}


	/* Unexpected VEX.LL prefix */
	__IF0 {
#ifdef NEED_return_unexpected_vex_ll_rmreg
#undef NEED_return_unexpected_vex_ll_rmreg
return_unexpected_vex_ll_rmreg:;
#ifdef EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL_RMREG
		EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL_RMREG();
#else /* EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL_RMREG */
#define NEED_return_unexpected_prefix_rmreg
		goto return_unexpected_prefix_rmreg;
#endif /* !EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL_RMREG */
#endif /* NEED_return_unexpected_vex_ll_rmreg */
#ifdef NEED_return_unexpected_vex_ll
#undef NEED_return_unexpected_vex_ll
return_unexpected_vex_ll:;
#ifdef EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL
		EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL();
#else /* EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL */
		goto return_unexpected_prefix;
#define NEED_return_unexpected_prefix
#endif /* !EMU86_EMULATE_RETURN_UNEXPECTED_VEX_LL */
#endif /* NEED_return_unexpected_vex_ll */
	}


	/* Unexpected LOCK prefix */
	__IF0 {
#ifdef NEED_return_unexpected_lock_rmreg
#undef NEED_return_unexpected_lock_rmreg
return_unexpected_lock_rmreg:;
#ifdef EMU86_EMULATE_RETURN_UNEXPECTED_LOCK_RMREG
		EMU86_EMULATE_RETURN_UNEXPECTED_LOCK_RMREG();
#else /* EMU86_EMULATE_RETURN_UNEXPECTED_LOCK_RMREG */
#define NEED_return_unexpected_prefix_rmreg
		goto return_unexpected_prefix_rmreg;
#endif /* !EMU86_EMULATE_RETURN_UNEXPECTED_LOCK_RMREG */
#endif /* NEED_return_unexpected_lock_rmreg */
#ifdef NEED_return_unexpected_lock
#undef NEED_return_unexpected_lock
return_unexpected_lock:;
#ifdef EMU86_EMULATE_RETURN_UNEXPECTED_LOCK
		EMU86_EMULATE_RETURN_UNEXPECTED_LOCK();
#else /* EMU86_EMULATE_RETURN_UNEXPECTED_LOCK */
		goto return_unexpected_prefix;
#define NEED_return_unexpected_prefix
#endif /* !EMU86_EMULATE_RETURN_UNEXPECTED_LOCK */
#endif /* NEED_return_unexpected_lock */
	}


	/* Unexpected prefix */
	__IF0 {
#ifdef NEED_return_unexpected_prefix_rmreg
#undef NEED_return_unexpected_prefix_rmreg
return_unexpected_prefix_rmreg:;
#ifdef EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX
		EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX();
#else /* EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX */
#define NEED_return_unknown_instruction_rmreg
		goto return_unknown_instruction_rmreg;
#endif /* !EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX */
#endif /* NEED_return_unexpected_prefix_rmreg */
#ifdef NEED_return_unexpected_prefix
#undef NEED_return_unexpected_prefix
return_unexpected_prefix:;
#ifdef EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX
		EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX();
#endif /* EMU86_EMULATE_RETURN_UNEXPECTED_PREFIX */
#endif /* NEED_return_unexpected_prefix */
	}


	/* Instruction isn't supported due to some missing hardware feature,
	 * due  to the  current execution mode,  or due to  how libemu86 was
	 * configured. */
	__IF0 {
#ifdef NEED_return_unsupported_instruction_rmreg
#undef NEED_return_unsupported_instruction_rmreg
return_unsupported_instruction_rmreg:;
#ifdef EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG
		EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG();
#else /* EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
#define NEED_return_unknown_instruction_rmreg
		goto return_unknown_instruction_rmreg;
#endif /* !EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION_RMREG */
#endif /* NEED_return_unsupported_instruction_rmreg */
#ifdef NEED_return_unsupported_instruction
#undef NEED_return_unsupported_instruction
return_unsupported_instruction:;
#ifdef EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION
		EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION();
#endif /* EMU86_EMULATE_RETURN_UNSUPPORTED_INSTRUCTION */
#endif /* NEED_return_unsupported_instruction */
	}


	/* Unknown instruction */
	__IF0 {
#ifdef NEED_return_unknown_instruction_rmreg
#undef NEED_return_unknown_instruction_rmreg
return_unknown_instruction_rmreg:;
#ifdef EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION_RMREG
		EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION_RMREG();
#endif /* EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION_RMREG */
#endif /* NEED_return_unknown_instruction_rmreg */
	}
return_unknown_instruction:
	/* Default case: Handle as an unknown instruction */
	EMU86_EMULATE_RETURN_UNKNOWN_INSTRUCTION();


#ifdef EMU86_EMULATE_FOOTER
	EMU86_EMULATE_FOOTER();
#endif /* EMU86_EMULATE_FOOTER */

#ifndef __INTELLISENSE__
#undef REAL_START_IP
#undef REAL_IP
#endif /* !__INTELLISENSE__ */
}

DECL_END
