/*
 * Copyright 2020 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.cloud.sql.core;

import static io.r2dbc.spi.ConnectionFactoryOptions.Builder;
import static io.r2dbc.spi.ConnectionFactoryOptions.DRIVER;

import io.netty.handler.ssl.SslContextBuilder;
import io.r2dbc.mssql.MssqlConnectionFactoryProvider;
import io.r2dbc.spi.ConnectionFactory;
import io.r2dbc.spi.ConnectionFactoryOptions;
import io.r2dbc.spi.ConnectionFactoryProvider;
import java.util.function.Function;

/** {@link ConnectionFactoryProvider} for proxied access to GCP MsSQL instances. */
public class GcpConnectionFactoryProviderMssql extends GcpConnectionFactoryProvider {

  static {
    InternalConnectorRegistry.addArtifactId("cloud-sql-connector-r2dbc-mssql");
  }

  /** MsSQL driver option value. */
  private static final String MSSQL_DRIVER = "mssql";

  @Override
  boolean supportedProtocol(String protocol) {
    return protocol.equals(MSSQL_DRIVER);
  }

  @Override
  ConnectionFactory tcpSocketConnectionFactory(
      ConnectionConfig config,
      Builder builder,
      Function<SslContextBuilder, SslContextBuilder> customizer) {
    builder
        .option(MssqlConnectionFactoryProvider.SSL_TUNNEL, customizer)
        .option(MssqlConnectionFactoryProvider.TCP_NODELAY, true)
        .option(MssqlConnectionFactoryProvider.TCP_KEEPALIVE, true);

    return new CloudSqlConnectionFactory(config, MssqlConnectionFactoryProvider::new, builder);
  }

  @Override
  ConnectionFactory unixSocketConnectionFactory(Builder optionBuilder, String socket) {
    throw new RuntimeException("UNIX socket connections are not supported");
  }

  @Override
  Builder createBuilder(ConnectionFactoryOptions connectionFactoryOptions) {
    return connectionFactoryOptions.mutate().option(DRIVER, MSSQL_DRIVER);
  }
}
