//
// Copyright © 2025 Stream.io Inc. All rights reserved.
//

@testable import StreamChat
import XCTest

public let channelKey = ChannelCodingKeys.self
public let channelPayloadKey = ChannelPayload.CodingKeys.self
var autogeneratedMessagesCounter = 0

public extension StreamMockServer {
    
    private enum ChannelRequestType {
        case addMembers([String])
        case removeMembers([String])

        static func type(from body: [UInt8]) -> ChannelRequestType? {
            let json = TestData.toJson(body)

            // Add members
            if let ids = json[JSONKey.Channel.addMembers] as? [String] {
                return .addMembers(ids)
            }

            // Remove members
            if let ids = json[JSONKey.Channel.removeMembers] as? [String] {
                return .removeMembers(ids)
            }

            return nil
        }

        var eventType: EventType {
            switch self {
            case .addMembers:
                return .memberAdded
            case .removeMembers:
                return .memberRemoved
            }
        }

        var ids: [String] {
            switch self {
            case .addMembers(let ids),
                 .removeMembers(let ids):
                return ids
            }
        }
    }

    func configureChannelEndpoints() {
        server.register(MockEndpoint.query) { [weak self] request in
            let channelId = try XCTUnwrap(request.params[EndpointQuery.channelId])
            self?.channelQueryEndpointWasCalled = true
            self?.updateChannel(withId: channelId)
            return self?.mockChannelQuery(request)
        }
        server.register(MockEndpoint.channels) { [weak self] request in
            self?.channelsEndpointWasCalled = true
            self?.updateChannels()
            return self?.limitChannels(request)
        }
        server.register(MockEndpoint.channel) { [weak self] request in
            self?.handleChannelRequest(request)
        }
        server.register(MockEndpoint.truncate) { [weak self] request in
            self?.channelTruncation(request)
        }
        server.register(MockEndpoint.sync) { [weak self] _ in
            self?.handleSyncRequest()
        }
    }

    func channelIndex(withId id: String) -> Int? {
        guard
            let channels = channelList[JSONKey.channels] as? [[String: Any]],
            let index = channels.firstIndex(
                where: {
                    let channel = $0[channelPayloadKey.channel.rawValue] as? [String: Any]
                    return (channel?[channelKey.id.rawValue] as? String) == id
                })
        else {
            return nil
        }

        return index
    }

    func channel(withId id: String) -> [String: Any]? {
        guard
            let channels = channelList[JSONKey.channels] as? [[String: Any]],
            let index = channelIndex(withId: id)
        else {
            return nil
        }
        return channels[index]
    }

    func waitForChannelQueryUpdate(timeout: Double = StreamMockServer.waitTimeout) {
        let endTime = Date().timeIntervalSince1970 * 1000 + timeout * 1000
        while !channelQueryEndpointWasCalled
                && endTime > Date().timeIntervalSince1970 * 1000 {}
    }

    func waitForChannelsUpdate(timeout: Double = StreamMockServer.waitTimeout) {
        let endTime = Date().timeIntervalSince1970 * 1000 + timeout * 1000
        while !channelsEndpointWasCalled
                && endTime > Date().timeIntervalSince1970 * 1000 {}
    }

    private func updateChannel(withId id: String) {
        var json = channelList
        var channels = json[JSONKey.channels] as? [[String: Any]]
        if let index = channels?.firstIndex(where: {
            let channel = $0[channelPayloadKey.channel.rawValue] as? [String: Any]
            return (channel?[channelKey.id.rawValue] as? String) == id
        }) {
            let messageList = findMessagesByChannelId(id)
            if
                var channel = channels?[index],
                var innerChannel = channel[JSONKey.channel] as? [String: Any] {
                setCooldown(in: &innerChannel)
                channel[JSONKey.channel] = innerChannel

                channel[channelPayloadKey.messages.rawValue] = messageList

                channels?[index] = channel
                json[JSONKey.channels] = channels
            }
            currentChannelId = id
            channelList = json
        }
    }

    private func updateChannels() {
        var json = channelList
        guard var channels = json[JSONKey.channels] as? [[String: Any]] else { return }

        for (index, channel) in channels.enumerated() {
            let channelDetails = channel[channelPayloadKey.channel.rawValue] as? [String: Any]
            if let channelId = channelDetails?[channelKey.id.rawValue] as? String {
                let messageList = findMessagesByChannelId(channelId)
                var mockedChannel = channel
                mockedChannel[channelPayloadKey.messages.rawValue] = messageList
                channels[index] = mockedChannel
            }
        }
        json[JSONKey.channels] = channels
        channelList = json
    }

    private func limitChannels(_ request: HttpRequest) -> HttpResponse {
        guard
            let payloadQuery = request.queryParams.first(where: { $0.0 == JSONKey.payload }),
            let payload = payloadQuery.1.removingPercentEncoding?.json,
            let limit = payload[Pagination.CodingKeys.pageSize.rawValue] as? Int
        else {
            return .ok(.json(channelList))
        }

        let offset = payload[Pagination.CodingKeys.offset.rawValue] as? Int ?? 0

        var limitedChannelList = channelList
        let channels = limitedChannelList[JSONKey.channels] as? [[String: Any]] ?? []
        let channelCount = channels.count - 1

        if !allChannelsWereLoaded && channelCount > limit {
            allChannelsWereLoaded = (channelCount - limit - offset < 0)
            let startWith = offset > channelCount ? channelCount : offset
            let endWith = offset + limit < channelCount ? offset + limit - 1 : channelCount
            limitedChannelList[JSONKey.channels] = Array(channels[startWith...endWith])
        }

        return .ok(.json(limitedChannelList))
    }

    private func mockChannelQuery(_ request: HttpRequest) -> HttpResponse {
        let json = TestData.toJson(request.body)
        let messages = json[JSONKey.messages] as? [String: Any]

        guard let id = request.params[EndpointQuery.channelId] else { return .badRequest(nil) }
        guard var channel = findChannelById(id) else { return .badRequest(nil) }
        guard let limit = messages?[MessagesPagination.CodingKeys.pageSize.rawValue] as? Int else {
            return .ok(.json(channel))
        }

        channel[channelPayloadKey.messages.rawValue] = mockMessagePagination(
            messageList: findMessagesByChannelId(id),
            limit: limit,
            idLt: messages?[paginationKey.lessThan.rawValue] as? String,
            idGt: messages?[paginationKey.greaterThan.rawValue] as? String,
            idLte: messages?[paginationKey.lessThanOrEqual.rawValue] as? String,
            idGte: messages?[paginationKey.greaterThanOrEqual.rawValue] as? String,
            around: messages?[paginationKey.around.rawValue] as? String
        )
        return .ok(.json(channel))
    }

    // MARK: Channel Members
    private func handleChannelRequest(_ request: HttpRequest) -> HttpResponse? {
        guard let type = ChannelRequestType.type(from: request.body) else {
            print("Unhandled request: \(request)")
            return .badRequest(nil)
        }

        return updateChannelMembers(request, ids: type.ids, eventType: type.eventType)
    }

    // TODO: CIS-2230
    private func handleSyncRequest() -> HttpResponse? {
        .ok(.json([JSONKey.events: []]))
    }

    private func updateChannelMembers(_ request: HttpRequest,
                                      ids: [String],
                                      eventType: EventType) -> HttpResponse {
        guard
            let id = request.params[EndpointQuery.channelId]
        else {
            return .ok(.json(channelList))
        }

        var json = channelList
        guard
            var channels = json[JSONKey.channels] as? [[String: Any]],
            let channelIndex = channelIndex(withId: id),
            var channel = channel(withId: id),
            var innerChannel = channel[JSONKey.channel] as? [String: Any],
            var members = channel[channelKey.members.rawValue] as? [[String: Any]]
        else {
            return .badRequest(nil)
        }

        let membersWithIds = memberJSONs(for: ids)
        switch eventType {
        case .memberAdded:
            members.append(contentsOf: membersWithIds)
        case .memberRemoved:
            members.removeAll(where: {
                let memberId = $0[JSONKey.userId] as? String
                return ids.contains(memberId ?? "")
            })
        default:
            return .badRequest(nil)
        }
        innerChannel[channelKey.members.rawValue] = members
        innerChannel[channelKey.memberCount.rawValue] = members.count
        setCooldown(in: &innerChannel)
        channel[JSONKey.channel] = innerChannel

        channels[channelIndex] = channel
        json[JSONKey.channels] = channels

        if let channelId = (channel[JSONKey.channel] as? [String: Any])?[JSONKey.id] as? String {
            // Send web socket event with given event type
            membersWithIds.forEach {
                websocketMember(with: $0, channelId: channelId, eventType: eventType)
            }

            // Send channel update web socket event
            websocketChannelUpdated(with: members, channelId: channelId)
        }

        channelList = json

        return .ok(.json(json))
    }

    func mockMembers(
        userSources: [String: Any]?,
        sampleChannel: [String: Any],
        memberDetails: [[String: String]]
    ) -> [[String: Any]] {
        var members: [[String: Any]] = []
        let channelMembers = sampleChannel[channelPayloadKey.members.rawValue] as? [[String: Any]]
        guard var sampleMember = channelMembers?.first else { return members }

        for member in memberDetails {
            sampleMember[JSONKey.user] = setUpUser(source: userSources, details: member)
            sampleMember[JSONKey.userId] = member[userKey.id.rawValue]
            members.append(sampleMember)
        }
        return members
    }

    func mockChannels(
        count: Int,
        messageText: String? = nil,
        messagesCount: Int = 0,
        replyCount: Int = 0,
        author: [String: Any]?,
        members: [[String: Any]],
        sampleChannel: [String: Any],
        withAttachments: Bool = false
    ) -> [[String: Any]] {
        var channels: [[String: Any]] = []
        guard count > 0 else { return channels }

        var membership = sampleChannel[channelPayloadKey.membership.rawValue] as? [String: Any]
        membership?[JSONKey.user] = author

        for channelIndex in 1...count {
            autogeneratedMessagesCounter = 0
            var newChannel = sampleChannel
            var messages: [[String: Any]?] = []
            newChannel[channelPayloadKey.members.rawValue] = members
            newChannel[channelPayloadKey.membership.rawValue] = membership
            let channelDetails = mockChannelDetails(
                channel: newChannel,
                author: author,
                memberCount: members.count,
                channelIndex: channelIndex
            )

            if messagesCount > 0 {
                for messageIndex in 1...messagesCount {
                    let channelId = channelDetails?[channelKey.id.rawValue] as? String
                    let messageId = TestData.uniqueId
                    let newMessage = generateMessage(
                        withText: messageText,
                        withIndex: messageIndex,
                        withChannelIndex: channelIndex,
                        withId: messageId,
                        channelId: channelId,
                        author: author,
                        replyCount: replyCount,
                        withAttachments: withAttachments,
                        overallMessagesCount: messagesCount
                    )
                    messages.append(newMessage)

                    if replyCount > 0 {
                        for replyIndex in 1...replyCount {
                            generateMessage(
                                withIndex: replyIndex,
                                withChannelIndex: channelIndex,
                                withId: TestData.uniqueId,
                                parentId: messageId,
                                channelId: channelId,
                                author: author
                            )
                        }
                    }
                }
            }

            newChannel[channelPayloadKey.messages.rawValue] = messages
            newChannel[channelPayloadKey.channel.rawValue] = channelDetails
            channels.append(newChannel)
        }

        return channels
    }

    @discardableResult
    private func generateMessage(
        withText text: String? = nil,
        withIndex index: Int,
        withChannelIndex channelIndex: Int,
        withId id: String?,
        parentId: String? = nil,
        channelId: String?,
        author: [String: Any]?,
        replyCount: Int? = 0,
        withAttachments: Bool = false,
        overallMessagesCount: Int = 1
    ) -> [String : Any]? {
        let timeInterval = TimeInterval(index + channelIndex * 1000 - 123_456_789)
        let timestamp = TestData.stringTimestamp(Date(timeIntervalSinceNow: timeInterval))
        let messageText = text == nil ? String(index) : text
        var message = mockMessage(
            TestData.toJson(.message)[JSONKey.message] as? [String : Any],
            channelId: channelId,
            messageId: id,
            text: messageText,
            user: author,
            createdAt: timestamp,
            updatedAt: timestamp,
            parentId: parentId,
            replyCount: replyCount
        )
        
        if withAttachments {
            var attachments: [[String: Any]] = []
            var file: [String: Any] = [:]
            var type: AttachmentType?
            
            switch autogeneratedMessagesCounter {
            case overallMessagesCount - 1, overallMessagesCount - 9:
                type = .image
                file[AttachmentCodingKeys.imageURL.rawValue] = Attachments.image
            case overallMessagesCount - 3, overallMessagesCount - 11:
                type = .giphy
                let json = TestData.getMockResponse(fromFile: MockFile.ephemeralMessage).json
                let messageObj = json["message"] as? [String: Any]
                attachments = messageObj?[messageKey.attachments.rawValue] as? [[String: Any]] ?? []
                attachments[0][GiphyAttachmentSpecificCodingKeys.actions.rawValue] = nil
            case overallMessagesCount - 5, overallMessagesCount - 13:
                type = .file
                file[AttachmentCodingKeys.assetURL.rawValue] = Attachments.file
                file[AttachmentFile.CodingKeys.mimeType.rawValue] = "application/pdf"
                file[AttachmentFile.CodingKeys.size.rawValue] = 123456
            case overallMessagesCount - 7, overallMessagesCount - 15:
                type = .video
                file[AttachmentCodingKeys.assetURL.rawValue] = Attachments.video
                file[AttachmentFile.CodingKeys.mimeType.rawValue] = "video/mp4"
                file[AttachmentFile.CodingKeys.size.rawValue] = 123456
            default:
                break
            }
            if type != nil && type != .giphy {
                for _ in 0...2 {
                    file[AttachmentCodingKeys.type.rawValue] = type?.rawValue
                    file[AttachmentCodingKeys.title.rawValue] = UUID().uuidString
                    attachments.append(file)
                }
            }
            message?[messageKey.attachments.rawValue] = attachments
            autogeneratedMessagesCounter += 1
        }
        
        parentId == nil ? saveMessage(message) : saveReply(message)
        return message
    }

    private func mockChannelDetails(
        channel: [String: Any],
        author: [String: Any]?,
        memberCount: Int,
        channelIndex: Int
    ) -> [String: Any]? {
        var channelDetails = channel[channelPayloadKey.channel.rawValue] as? [String: Any]
        let uniqueId = TestData.uniqueId
        let timeInterval = TimeInterval(123_456_789 - channelIndex * 1000)
        let timestamp = TestData.stringTimestamp(Date(timeIntervalSinceNow: timeInterval))
        channelDetails?[channelKey.name.rawValue] = "\(channelIndex)"
        channelDetails?[channelKey.id.rawValue] = uniqueId
        channelDetails?[channelKey.cid.rawValue] = "\(ChannelType.messaging.rawValue):\(uniqueId)"
        channelDetails?[channelKey.createdBy.rawValue] = author
        channelDetails?[channelKey.memberCount.rawValue] = memberCount
        channelDetails?[channelKey.createdAt.rawValue] = timestamp
        channelDetails?[channelKey.updatedAt.rawValue] = timestamp
        return channelDetails
    }

    func getFirstChannelId() -> String {
        let endTime = TestData.waitingEndTime
        while channelList.isEmpty && endTime > TestData.currentTimeInterval {}
        guard
            let channels = channelList[JSONKey.channels] as? [[String: Any]],
            let firstChannel = channels.first?[JSONKey.channel] as? [String: Any],
            let id = firstChannel[channelKey.id.rawValue] as? String
        else {
            return ""
        }
        return id
    }

    func removeChannel(_ id: String) {
        let deletedChannel = try? XCTUnwrap(findChannelById(id))
        guard var channels = channelList[JSONKey.channels] as? [[String: Any]] else { return }

        if let deletedIndex = channels.firstIndex(where: { (channel) -> Bool in
            (channel[channelKey.id.rawValue] as? String) == (deletedChannel?[channelKey.id.rawValue] as? String)
        }) {
            channels.remove(at: deletedIndex)
        }

        channelList[JSONKey.channels] = channels
    }

    private func truncateChannel(_ id: String, truncatedAt: String, truncatedBy: [String: Any]?) {
        let channelMessages = findMessagesByChannelId(id)
        for message in channelMessages {
            removeMessage(message)
        }

        var channel = findChannelById(id)
        var channelDetails = channel?[JSONKey.channel] as? [String: Any]
        channelDetails?[JSONKey.Channel.truncatedBy] = truncatedBy
        channelDetails?[channelKey.truncatedAt.rawValue] = truncatedAt

        channel?[JSONKey.channel] = channelDetails
        channel?[JSONKey.messages] = []

        if var channels = channelList[JSONKey.channels] as? [[String: Any]?] {
            removeChannel(id)
            channels.append(channel)
            channelList[JSONKey.channels] = channels
        }
    }

    private func channelTruncation(_ request: HttpRequest) -> HttpResponse? {
        waitForChannelQueryUpdate()
        guard let channelId = request.params[EndpointQuery.channelId] else { return .badRequest(nil) }
        var json = TestData.toJson(.httpTruncate)
        var truncatedMessage: [String: Any]?
        var channel = json[JSONKey.channel] as? [String: Any]
        let channelDetails = findChannelById(channelId)?[JSONKey.channel] as? [String: Any]
        let truncatedby = channelDetails?[channelKey.createdBy.rawValue] as? [String: Any]
        let truncatedAt = TestData.currentDate

        truncateChannel(
            channelId,
            truncatedAt: truncatedAt,
            truncatedBy: truncatedby
        )

        channel?[channelKey.id.rawValue] = channelId
        channel?[channelKey.cid.rawValue] = "\(ChannelType.messaging.rawValue):\(channelId)"
        channel?[JSONKey.Channel.truncatedBy] = truncatedby
        channel?[channelKey.truncatedAt.rawValue] = truncatedAt
        channel?[channelKey.name.rawValue] = channelDetails?[channelKey.name.rawValue]

        websocketEvent(
            .channelTruncated,
            user: truncatedby,
            channelId: channelId,
            channel: channel
        )

        if let message = TestData.toJson(request.body)[JSONKey.message] as? [String: Any] {
            truncatedMessage = json[JSONKey.message] as? [String: Any]
            truncatedMessage?[messageKey.id.rawValue] = message[messageKey.id.rawValue]
            if let text = message[messageKey.text.rawValue] as? String {
                truncatedMessage?[messageKey.text.rawValue] = text
                truncatedMessage?[messageKey.html.rawValue] = text.html
            }
            websocketMessage(
                truncatedMessage?[messageKey.text.rawValue] as? String,
                channelId: channelId,
                messageId: truncatedMessage?[messageKey.id.rawValue] as? String,
                messageType: .system,
                eventType: .messageNew,
                user: truncatedby,
                channel: channel
            )
        } else {
            truncatedMessage = nil
        }

        json[JSONKey.message] = truncatedMessage
        json[JSONKey.channel] = channel
        return .ok(.json(json))
    }
}
