//
//  SwiftyTokeniser.swift
//  SwiftyMarkdown
//
//  Created by Simon Fairbairn on 16/12/2019.
//  Copyright © 2019 Voyage Travel Apps. All rights reserved.
//
import Foundation
import os.log

extension OSLog {
	private static var subsystem = "SwiftyTokeniser"
	static let tokenising = OSLog(subsystem: subsystem, category: "Tokenising")
	static let styling = OSLog(subsystem: subsystem, category: "Styling")
	static let performance = OSLog(subsystem: subsystem, category: "Peformance")
}

class SwiftyTokeniser {
	let rules : [CharacterRule]
	var replacements : [String : [Token]] = [:]
	
	var enableLog = (ProcessInfo.processInfo.environment["SwiftyTokeniserLogging"] != nil)
	let totalPerfomanceLog = PerformanceLog(with: "SwiftyTokeniserPerformanceLogging", identifier: "Tokeniser Total Run Time", log: OSLog.performance)
	let currentPerfomanceLog = PerformanceLog(with: "SwiftyTokeniserPerformanceLogging", identifier: "Tokeniser Current", log: OSLog.performance)
		
	var metadataLookup : [String : String] = [:]
	
	let newlines = CharacterSet.newlines
	let spaces = CharacterSet.whitespaces

	
	init( with rules : [CharacterRule] ) {
		self.rules = rules
		
		self.totalPerfomanceLog.start()
	}
	
	deinit {
		self.totalPerfomanceLog.end()
	}
	
	
	/// This goes through every CharacterRule in order and applies it to the input string, tokenising the string
	/// if there are any matches.
	///
	/// The for loop in the while loop (yeah, I know) is there to separate strings from within tags to
	/// those outside them.
	///
	/// e.g. "A string with a \[link\]\(url\) tag" would have the "link" text tokenised separately.
	///
	/// This is to prevent situations like **\[link**\](url) from returing a bold string.
	///
	/// - Parameter inputString: A string to have the CharacterRules in `self.rules` applied to
	func process( _ inputString : String ) -> [Token] {
		let currentTokens = [Token(type: .string, inputString: inputString)]
		guard rules.count > 0 else {
			return currentTokens
		}
		var mutableRules = self.rules
		
		if inputString.isEmpty {
			return [Token(type: .string, inputString: "", characterStyles: [])]
		}
		
		self.currentPerfomanceLog.start()
	
		var elementArray : [Element] = []
		for char in inputString {
			if newlines.containsUnicodeScalars(of: char) {
				let element = Element(character: char, type: .newline)
				elementArray.append(element)
				continue
			}
			if spaces.containsUnicodeScalars(of: char) {
				let element = Element(character: char, type: .space)
				elementArray.append(element)
				continue
			}
			let element = Element(character: char, type: .string)
			elementArray.append(element)
		}
		
		while !mutableRules.isEmpty {
			let nextRule = mutableRules.removeFirst()
			if enableLog {
				os_log("------------------------------", log: .tokenising, type: .info)
				os_log("RULE: %@", log: OSLog.tokenising, type:.info , nextRule.description)
			}
			self.currentPerfomanceLog.tag(with: "(start rule %@)")
			
			let scanner = SwiftyScanner(withElements: elementArray, rule: nextRule, metadata: self.metadataLookup)
			elementArray = scanner.scan()
		}
		
		var output : [Token] = []
		var lastElement = elementArray.first!
		
		func empty( _ string : inout String, into tokens : inout [Token] )  {
			guard !string.isEmpty else {
				return
			}
			var token = Token(type: .string, inputString: string)
			token.metadataStrings.append(contentsOf: lastElement.metadata) 
			token.characterStyles = lastElement.styles
			string.removeAll()
			tokens.append(token)
		}

		var lastTag: Element? = nil
		var accumulatedString = ""
		for element in elementArray {
			var currentElement = element

			guard currentElement.type != .escape else {
				continue
			}

			let lastTagStyles = lastTag?.styles as? [CharacterStyle] ?? []
			let lastElementStyles = lastElement.styles as? [CharacterStyle] ?? []
			let currentElementStyles = currentElement.styles as? [CharacterStyle] ?? []

			if  !lastTagStyles.contains(.italic) &&
                !lastTagStyles.contains(.bold) &&
				(
                    (currentElement.character == "_" && lastElement.type == .string) ||
                    (currentElement.character != "_" && currentElement.character != "*" && currentElementStyles.contains { $0 == .italic || $0 == .bold })
                ) {
                currentElement.styles.removeAll(where: { [.italic, .bold].contains($0 as? CharacterStyle) })
				currentElement.type = .string
			}

			if currentElement.type == .tag {
				lastTag = currentElement
			}

			guard currentElement.type == .string || currentElement.type == .space || currentElement.type == .newline else {
				empty(&accumulatedString, into: &output)
				continue
			}
			if lastElementStyles != currentElementStyles {
				empty(&accumulatedString, into: &output)
			}
			accumulatedString.append(currentElement.character)
			lastElement = currentElement
		}
		empty(&accumulatedString, into: &output)
		
		self.currentPerfomanceLog.tag(with: "(finished all rules)")
		
		if enableLog {
			os_log("=====RULE PROCESSING COMPLETE=====", log: .tokenising, type: .info)
			os_log("==================================", log: .tokenising, type: .info)
		}
		return output
	}
}


extension String {
	func repeating( _ max : Int ) -> String {
		var output = self
		for _ in 1..<max {
			output += self
		}
		return output
	}
}
