//
// Copyright © 2025 Stream.io Inc. All rights reserved.
//

import Foundation

/// A sorting key protocol.
public protocol SortingKey: Encodable {}

/// Sorting options.
///
/// For example:
/// ```
/// // Sort channels by the last message date:
/// let sorting = Sorting("lastMessageDate")
/// ```
public struct Sorting<Key: SortingKey>: Encodable, CustomStringConvertible {
    /// A sorting field name.
    public let key: Key
    /// A sorting direction.
    public let direction: Int

    private enum CodingKeys: String, CodingKey {
        case key = "field"
        case direction
    }

    /// True if the sorting in ascending order, otherwise false.
    public var isAscending: Bool { direction == 1 }

    public var description: String { "\(key):\(direction)" }

    /// Init sorting options.
    ///
    /// - Parameters:
    ///     - key: a sorting key.
    ///     - isAscending: a direction of the sorting.
    public init(key: Key, isAscending: Bool = false) {
        self.key = key
        direction = isAscending ? 1 : -1
    }
}

extension Sorting: Equatable where Key: Equatable {}
extension Sorting: Hashable where Key: Hashable {}
