/*
 *  Derived from linux/kernel/compat.c. See doc/legal.rst for full GPL text.
 *
 *  Copyright (C) 2002-2003 Stephen Rothwell, IBM Corporation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

/* Test program for uninitialized stack memory usage POI */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/timex.h>
#include <unistd.h>

/*
 * CVE-2018-11508 example
 *
 * First we stub out relevant kernel functionality for the example
 */

#define EFAULT 14 /* Bad address */

int fake_copy_from_user(void *to, unsigned long n) {
  unsigned long c;
  // symbolic data for Manticore
  read(0, &c, sizeof(c));
  if (c <= n) {
    /* fake, loads data from the user process */
    memset(to, 0, n);
    /* in the real kernel, catch and return -EFAULT on page fault, otherwise 0
     */
    return 0;
  }
  return -EFAULT;
}

int fake_copy_to_user(void *from, unsigned long n) {
  /* fake, leaks data to the user process */
  printf("%.*s", n, from);
  return 0;
}

int do_adjtimex(struct timex *txc) {
  /* fake, do nothing */
  return 0;
}

/* adapted from kernel/compat.c */
/* BUG: does not initialize txc->tai */
int compat_get_timex(struct timex *txc) {
  struct timex tx32;

  /* CVE-2018-11508 bugfix
   * fix 4-byte infoleak via uninitialized struct field
   * https://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git/commit/?id=0a0b98734479aa5b3c671d5190e86273372cab95
   *
   * Uncomment the following line to apply the fix that removes the bug.
   * When this fix has been applied, our static POI analysis no longer reports
   * use of uninitialized data in these functions.
   */
  /* memset(txc, 0, sizeof(struct timex)); */

  /* initialize (non-tai fields) from userspace */
  if (fake_copy_from_user(&tx32, sizeof(struct timex))) {
    /* this causes lots of false positives! returning here fails to initialize
     * *any* of the txc fields. Though returning here prevents them from being
     * read later, but our paths aren't smart enough to figure this out. */
    return -EFAULT;
  }

  txc->modes = tx32.modes;
  txc->offset = tx32.offset;
  txc->freq = tx32.freq;
  txc->maxerror = tx32.maxerror;
  txc->esterror = tx32.esterror;
  txc->status = tx32.status;
  txc->constant = tx32.constant;
  txc->precision = tx32.precision;
  txc->tolerance = tx32.tolerance;
  txc->time.tv_sec = tx32.time.tv_sec;
  txc->time.tv_usec = tx32.time.tv_usec;
  txc->tick = tx32.tick;
  txc->ppsfreq = tx32.ppsfreq;
  txc->jitter = tx32.jitter;
  txc->shift = tx32.shift;
  txc->stabil = tx32.stabil;
  txc->jitcnt = tx32.jitcnt;
  txc->calcnt = tx32.calcnt;
  txc->errcnt = tx32.errcnt;
  txc->stbcnt = tx32.stbcnt;

  return 0;
}

/* adapted from kernel/compat.c */
int compat_put_timex(const struct timex *txc) {
  struct timex tx32;

  memset(&tx32, 0, sizeof(struct timex));
  tx32.modes = txc->modes;
  tx32.offset = txc->offset;
  tx32.freq = txc->freq;
  tx32.maxerror = txc->maxerror;
  tx32.esterror = txc->esterror;
  tx32.status = txc->status;
  tx32.constant = txc->constant;
  tx32.precision = txc->precision;
  tx32.tolerance = txc->tolerance;
  tx32.time.tv_sec = txc->time.tv_sec;
  tx32.time.tv_usec = txc->time.tv_usec;
  tx32.tick = txc->tick;
  tx32.ppsfreq = txc->ppsfreq;
  tx32.jitter = txc->jitter;
  tx32.shift = txc->shift;
  tx32.stabil = txc->stabil;
  tx32.jitcnt = txc->jitcnt;
  tx32.calcnt = txc->calcnt;
  tx32.errcnt = txc->errcnt;
  tx32.stbcnt = txc->stbcnt;
  tx32.tai = txc->tai; /* BUG: copies uninitialized data 'tai' */

  if (fake_copy_to_user(&tx32, sizeof(struct timex)))
    return -EFAULT;

  return 0;
}

/*
 * Example based on CVE-2018-11508
 *
 * Summary:
 *   - stack variable 'struct timex txc' gets initialized in syscall_adjtimex
 *   - compat_get_timex initializes some fields of txc, but not tai
 *   - compat_put_timex copies the uninitialized tai field into
 *     the struct that gets used (sent back to "userspace")
 *
 * The True Positive Bug:
 *   - tai used without initialization (in compate_put_timex)
 *
 * False Positives:
 *   - if fake_copy_from_user fails in compate_get_timex, then
 *     none of the other fields in txc get initialized either
 *   - but if that happens, syscall_adjtimex returns before accessing them
 *   - our pathfinding doesn't do that kind of reasoning, so reports
 *     paths where fake_copy_from_user, no fields are initialized,
 *     and then they all get accessed later in compate_put_timex
 *     even though that is not feasible
 */

/* adapted from kernel/time/time.c */
int syscall_adjtimex() {
  struct timex txc; /* stack object */
  int err, ret;

  /* initializes some fields of txc, but not tai */
  err = compat_get_timex(&txc);
  if (err)
    return err; /* no fields of txc were initialized, return */

  ret = do_adjtimex(&txc);
  /* txc.tai has not been initialized by this point */

  /* next: access and leak the (uninitialized) 'tai' field */
  err = compat_put_timex(&txc);
  if (err)
    return err;

  return ret;
}

int main(int argc, char **argv) { syscall_adjtimex(); }
