
import React from 'react';
import {
  localized,
  AccountStore,
  SignatureStore,
  Actions,
  FocusedPerspectiveStore,
  Utils,
  ISignatureSet,
  ISignature,
  IDefaultSignatures,
  IAliasSet,
  InlineStyleTransformer,
} from 'mailspring-exports';
import { Flexbox, EditableList } from 'mailspring-component-kit';

import { ResolveSignatureData, RenderSignatureData, DataShape } from './constants';
import SignatureAccountDefaultPicker from './signature-account-default-picker';
import SignatureTemplatePicker from './signature-template-picker';
import SignaturePhotoPicker from './signature-photo-picker';
import Templates from './templates';

interface SignatureEditorProps {
  signature: ISignature;
  defaults: IDefaultSignatures;
  accountsAndAliases: IAliasSet;
}

interface SignatureEditorState { }

class SignatureEditor extends React.Component<SignatureEditorProps, SignatureEditorState> {
  _onTitleChange = event => {
    const sig = this.props.signature;
    Actions.upsertSignature({ ...sig, title: event.target.value }, sig.id);
  };

  _onRawBodyChange = async event => {
    const sig = this.props.signature;
    let body = event.target.value;
    try {
      body = await InlineStyleTransformer.run(body);
    } catch (err) {
      //no-op
    }
    Actions.upsertSignature({ ...sig, body }, sig.id);
  };

  _onDataFieldChange = event => {
    const { id, value } = event.target;
    const sig = this.props.signature;

    // If you have raw selected and are switching back to a template,
    // display a warning UNLESS the html is an unmodified template HTML
    if (id === 'templateName' && !sig.data.templateName && value) {
      const htmlMatchesATemplate = Templates.find(
        t => sig.body === RenderSignatureData({ ...sig.data, templateName: t.name })
      );
      if (!htmlMatchesATemplate) {
        const idx = require('@electron/remote').dialog.showMessageBoxSync({
          type: 'warning',
          buttons: [localized('Cancel'), localized('Continue')],
          message: localized('Revert custom HTML?'),
          detail: localized(
            "Switching back to a signature template will overwrite the custom HTML you've entered."
          ),
        });
        if (idx === 0) {
          return;
        }
      }
    }

    // apply change
    sig.data = { ...sig.data, [id]: value };

    // re-render
    if (sig.data.templateName) {
      const template = Templates.find(t => t.name === sig.data.templateName);
      if (template) {
        sig.body = RenderSignatureData(sig.data);
      }
    }

    Actions.upsertSignature(sig, sig.id);
  };

  render() {
    const { accountsAndAliases, defaults } = this.props;

    let signature = this.props.signature;
    let empty = false;
    if (!signature) {
      signature = {
        id: '',
        body: '',
        title: '',
        data: { title: '', templateName: Templates[0].name },
      };
      empty = true;
    }
    const data = signature.data || {};
    const resolvedData = ResolveSignatureData(data);

    return (
      <div className={`signature-wrap ${empty && 'empty'}`}>
        <div className="section basic-info">
          <input
            type="text"
            id="title"
            placeholder={localized('Name')}
            value={signature.title || ''}
            onChange={this._onTitleChange}
          />
          <div style={{ flex: 1 }} />
          <SignatureAccountDefaultPicker
            signature={signature}
            accountsAndAliases={accountsAndAliases}
            defaults={defaults}
          />
        </div>

        <div className="section preview">
          <div className="label">{localized('Preview')}</div>
          <div className="preview" dangerouslySetInnerHTML={{ __html: signature.body }} />
        </div>

        <div className="section">
          <SignatureTemplatePicker resolvedData={resolvedData} onChange={this._onDataFieldChange} />
        </div>

        {!resolvedData.templateName
          ? [
            <div key="header" className="section-header">
              {localized('Raw Source')}
            </div>,
            <textarea
              id="body"
              key={`textarea ${signature.id}`}
              className="section raw-html"
              spellCheck={false}
              onChange={this._onRawBodyChange}
              defaultValue={signature.body || ''}
            />,
          ]
          : [
            <div key="header" className="section-header">
              {localized('Information')}
            </div>,
            <div key="section" className="section information">
              {DataShape.map(item => (
                <div className="field" key={item.key}>
                  <label>{item.label}</label>
                  <input
                    type="text"
                    onChange={this._onDataFieldChange}
                    placeholder={item.placeholder}
                    id={item.key}
                    value={data[item.key] || ''}
                  />
                </div>
              ))}
              <SignaturePhotoPicker
                id={signature.id}
                data={data}
                resolvedURL={resolvedData.photoURL}
                onChange={this._onDataFieldChange}
              />
            </div>,
          ]}
      </div>
    );
  }
}

interface PreferencesSignaturesState {
  signatures: ISignatureSet;
  selectedSignature: ISignature;
  defaults: IDefaultSignatures;
  accountsAndAliases: IAliasSet;
}

export default class PreferencesSignatures extends React.Component<
  Record<string, unknown>,
  PreferencesSignaturesState
> {
  static displayName = 'PreferencesSignatures';

  unsubscribers = [];

  constructor(props) {
    super(props);
    this.state = this._getStateFromStores();
  }

  componentDidMount() {
    this.unsubscribers = [SignatureStore.listen(this._onChange)];
  }

  componentWillUnmount() {
    this.unsubscribers.forEach(unsubscribe => unsubscribe());
  }

  _onChange = () => {
    this.setState(this._getStateFromStores());
  };

  _getStateFromStores() {
    return {
      signatures: SignatureStore.getSignatures(),
      selectedSignature: SignatureStore.selectedSignature(),
      defaults: SignatureStore.getDefaults(),
      accountsAndAliases: AccountStore.aliases(),
    };
  }

  _onAddSignature = () => {
    const activeIds = FocusedPerspectiveStore.current().accountIds || AccountStore.accountIds();
    const activeAccount = AccountStore.accountForId(activeIds[0]);
    const id = Utils.generateTempId();

    let data = {};
    let body = null;
    if (this.state.selectedSignature) {
      data = { ...this.state.selectedSignature.data };
      body = this.state.selectedSignature.body;
    } else {
      data = {
        templateName: Templates[0].name,
        name: activeAccount.name,
        email: activeAccount.emailAddress,
      };
      body = RenderSignatureData(data);
    }

    Actions.upsertSignature({ id, title: localized('Untitled'), body, data }, id);
    Actions.selectSignature(id);
  };

  _onDeleteSignature = signature => {
    Actions.removeSignature(signature);
  };

  _onEditSignatureTitle = nextTitle => {
    const { title, ...rest } = this.state.selectedSignature;
    Actions.upsertSignature({ title: nextTitle, ...rest }, rest.id);
  };

  _onSelectSignature = sig => {
    Actions.selectSignature(sig.id);
  };

  _renderSignatures() {
    const sigArr = Object.values(this.state.signatures);

    return (
      <Flexbox>
        <EditableList
          showEditIcon
          className="signature-list"
          items={sigArr}
          itemContent={sig => sig.title}
          onCreateItem={this._onAddSignature}
          onDeleteItem={this._onDeleteSignature}
          onItemEdited={this._onEditSignatureTitle}
          onSelectItem={this._onSelectSignature}
          selected={this.state.selectedSignature}
        />
        <SignatureEditor
          signature={this.state.selectedSignature}
          defaults={this.state.defaults}
          accountsAndAliases={this.state.accountsAndAliases}
        />
      </Flexbox>
    );
  }

  render() {
    return (
      <div className="preferences-signatures-container">
        <section>{this._renderSignatures()}</section>
      </div>
    );
  }
}
