# 1. Install/register GitLab Runner
GitLab Runner is the open source project that is used to run your jobs and
send the results back to GitLab. You have to
[install](https://docs.gitlab.com/runner/install/) it on a machine where
your analysis will be performed.

To register the runner with GitLab first you need to get a registration token
from the GitLab web interface. It is generated by GitLab and it can be found in
your projects settings under
`settings/ci_cd/Set up a specific Runner manually`.

Once GitLab Runner is installed and you get the token, you need to register the
runner with GitLab. You can use the following command:
```sh
sudo gitlab-runner register \
    --non-interactive \
    --url "https://mycompany.gitlab.com" \
    --registration-token "<REGISTRATION_TOKEN_FROM_GITLAB>" \
    --description "codechecker" \
    --executor "shell"
```

You can find more information how to
[register a GitLab Runner](https://docs.gitlab.com/ee/ci/runners/).

# 2. Define a job in your .gitlab-ci.yml
GitLab CI/CD pipelines are configured using a YAML file called `.gitlab-ci.yml`
within each project. Create this file in your project root directory:

```yml
code_quality:
  allow_failure: false
  script:
    - bash .gitlab/ci/run_codechecker.sh
  after_script:
    - cat gl-code-quality-report.json
  artifacts:
    reports:
      codequality: gl-code-quality-report.json
    paths: [gl-code-quality-report.json]
    expire_in: '2 mos'
  stage: test
```

`expire_in` allows you to specify how long artifacts should live before they
expire and are therefore deleted. You can set it to a lower or a higher value.
For more information
[see](https://docs.gitlab.com/ee/ci/yaml/README.html#artifactsexpire_in).

Script is a shell script which will be executed by the Runner.

# 3. Create a script to run CodeChecker analysis
Create a shell script in your project under `.gitlab/ci/run_codechecker.sh`.
This script will run the analysis and it has to create a json file called
`gl-code-quality-report.json` which will contain reports in Code Climate
format. CodeChecker from version `6.12.0` is able to generate Code Climate
output by using the *CodeChecker parse* or *CodeChecker cmd diff* commands:

## Using CodeChecker parse
```sh
#!/usr/bin/env bash

# Log your project.
CodeChecker log -b "make" -o ./compilation_database.json

# Analyze your project.
CodeChecker analyze --ctu \
  -o ./reports \
  ./compilation_database.json

# Create the report file by using the CodeChecker parse command.
CodeChecker parse \
  --trim-path-prefix $(pwd) \
  -e codeclimate \
  ./reports > gl-code-quality-report.json

# Exit with status code 1 if there is any report in the output file.
status=$(cat gl-code-quality-report.json)
if [[ -n "$status" && "$status" != "[]" ]]; then
  exit 1
fi
```

## Using CodeChecker cmd diff
```sh
#!/usr/bin/env bash

# Log your project.
CodeChecker log -b "make" -o ./compilation_database.json

# Analyze your project.
CodeChecker analyze --ctu \
  -o ./reports \
  ./compilation_database.json

# Or you can create a report by using the CodeChecker diff command.
CC_REPO_DIR="$(pwd)" \
CodeChecker cmd diff \
  -b my_remote_run \
  --url your_baseline_url \
  -n ./reports \
  --new \
  -o codeclimate \
  -e ./codeclimate

out_file="codeclimate/codeclimate_issues.json"

if [ ! -f "$out_file" ]; then
  echo "${out_file} does not exists."
  exit 1
fi

cp $out_file gl-code-quality-report.json

# Exit with status code 1 if there is any report in the output file.
status=$(cat gl-code-quality-report.json)
if [[ -n "$status" && "$status" != "[]" ]]; then
  exit 1
fi
```

# 4. Create a merge request
Create a merge request which will contain the `.gitlab-ci.yml` and the
`.gitlab/ci/run_codechecker.sh` files.

Once the Code Quality job has completed, potential changes to code quality are
shown directly in the merge request. If the Code Quality report doesn’t have
anything to compare to, no information will be displayed in the merge request
area. You may see the following message: *Failed to load codeclimate report*.
That is the case when you add the Code Quality job in your `.gitlab-ci.yml`
for the very first time.

For more information
[see](https://docs.gitlab.com/ee/user/project/merge_requests/code_quality.html).

# Possible errors
## Runner is outdated
The GitLab Runner version should be in sync with the GitLab version. You will
see the following message is your runner is outdated:
```sh
Error: `Your runner is outdated, please upgrade your runner`
```

You can upgrade it by using the following commands:
```sh
curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.deb.sh | sudo bash
apt-cache madison gitlab-runner
sudo apt-get install gitlab-runner=12.9.0
```

For more information
[see](https://docs.gitlab.com/runner/#compatibility-with-gitlab-versions).

## Service run failed
If something is not working properly with the runners you can see the logs by
using the following command: `sudo journalctl -u gitlab-runner`. If you see the
following error message in the logs:
```sh
FATAL: Service run failed  error=chdir /var/lib/gitlab-runner: no such file or directory
```

you have to do the following steps:
```sh
sudo mkdir /var/lib/gitlab-runner/
sudo chown -R gitlab-runner:gitlab-runner /var/lib/gitlab-runner/
```

For more information
[see](https://gitlab.com/gitlab-org/gitlab-runner/issues/3000).
