public static class ReleaseNotes
{
    public static void Create(ICakeContext context)
    {
        if (context == null)
        {
            throw new ArgumentNullException(nameof(context));
        }

        context.Information("Transforming release notes");

        var notes = context.FileReadText(".\\ReleaseNotes.md");
        var markdown = new MarkdownSharp.Markdown();
        var html = markdown.Transform(notes);

        var outPut = @"<!DOCTYPE HTML>
<html><head><meta http-equiv=""content-Type"" content=""text/html; charset=UTF-8"">
<style>.subHeader { color: #A6A6A6; font-style: italic; font-family: Roboto, 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; font-size: 1em; }
h1 { color: #0070C0; font-style: normal; font-weight: bold; font-family: Roboto, 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; font-size: 2em; }
body { font-style: normal; font-family: Roboto, 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; font-size: 1em; }
li { margin-bottom: 5px; } </style></head><body><div><p><img width=""250"" height=""65"" 
src=""data:image/png;base64,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"" alt=""""></p>
<p class=""subHeader"">The Simple Desktop Email Receiver.</p></div>" + html + "</body></html>";

        context.Information("Write readme mail");
        var body = new MimeKit.BodyBuilder { HtmlBody = outPut };
        var message = new MimeKit.MimeMessage
        {
            Subject = "Welcome to Papercut SMTP!",
            Body = body.ToMessageBody(),
        };
        message.From.Add(new MimeKit.MailboxAddress("Papercut SMTP", "PapercutSMTP@papercut-smtp.com"));
        message.To.Add(new MimeKit.MailboxAddress("Papercut User", "PapercutUser@papercut-smtp.com"));
        message.WriteTo(System.IO.Path.GetFullPath(".\\src\\Papercut.UI\\Readme.eml"));
    }
}