"use client";

import { useChat } from "ai/react";
import { useState } from "react";

export default function Home() {
  const [threadId] = useState(crypto.randomUUID());
  const [runId, setRunId] = useState(crypto.randomUUID());

  // Mapping between the message ID (generated by Vercel AI SDK) and the run ID (managed by us)
  const [messageIdMapping, setMessageIdMapping] = useState<{
    [messageId: string]: string;
  }>({});
  // IDs of the messages that the user has upvoted
  const [upvotedMessages, setUpvotedMessages] = useState<string[]>([]);

  // useChat is provided by Vercel AI SDK. It handles the chat logic and the streaming
  // of messages between the frontend & the backend.
  const { messages, input, handleInputChange, handleSubmit } = useChat({
    api: "api/chat",
    // Here we attach the thread & run IDs so that they are accessible from our API route
    body: { threadId, runId },
    onFinish: (message) => {
      // When a message is received, we will store the mapping between
      // the message ID and the run ID sent to Literal AI.
      setMessageIdMapping((prev) => ({
        ...prev,
        [message.id]: runId,
      }));

      // Then we generate a new Step ID for the next message
      setRunId(crypto.randomUUID());
    },
  });

  const handleScore = ({
    messageId,
    upvotedRunId,
    upvote,
  }: {
    messageId: string;
    upvotedRunId: string;
    upvote: boolean;
  }) => {
    fetch("api/score", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ runId: upvotedRunId, score: upvote ? 1 : 0 }),
    }).then(() => {
      if (upvote) {
        setUpvotedMessages([...upvotedMessages, messageId]);
      }
    });
  };

  return (
    <main className="min-h-screen">
      <section className="mx-auto w-2/3 mt-10 mb-20">
        {messages.map((message) => {
          const mappedRunId = messageIdMapping[message.id];
          const isAlreadyUpvoted = upvotedMessages.includes(message.id);

          return (
            <article key={message.id} className="flex gap-3 my-3">
              <div className="border border-gray-500 p-1">
                {message.role === "user" ? "🧑‍💻" : "🤖"}
              </div>
              {message.role === "user" && (
                <div className="flex-1">{message.content}</div>
              )}
              {message.role !== "user" && (
                <pre className="flex-1 text-pretty">{message.content}</pre>
              )}
              {!!mappedRunId && (
                <button
                  onClick={() =>
                    handleScore({
                      messageId: message.id,
                      upvotedRunId: mappedRunId,
                      upvote: true,
                    })
                  }
                  disabled={isAlreadyUpvoted}
                  className={isAlreadyUpvoted ? "" : "grayscale"}
                >
                  👍
                </button>
              )}
            </article>
          );
        })}
      </section>

      <form
        onSubmit={handleSubmit}
        className="fixed inset-x-0 bottom-0 mx-auto"
      >
        <div className="px-4 py-2">
          <div className="flex gap-2 w-2/3 mx-auto border border-gray-600 rounded-md">
            <input
              placeholder="Send a message."
              className="flex-1 p-2 rounded-md"
              autoComplete="off"
              value={input}
              onChange={handleInputChange}
            />
            <button
              className="bg-gray-300 rounded-md"
              type="submit"
              disabled={!input}
            >
              <span className="p-2">Send message</span>
            </button>
          </div>
        </div>
      </form>
    </main>
  );
}
