using System.Threading;
using System.Threading.Tasks;
using MessagePack;

namespace System.Net.Http.MessagePack
{
    /// <summary>
    ///     Contains the extensions methods for using MessagePack as the content-type in HttpClient.
    /// </summary>
    public static partial class HttpClientMessagePackExtensions
    {
        /// <summary>
        ///     Send a PUT request to the specified Uri containing the <paramref name="value" /> serialized as MessagePack in the
        ///     request body.
        /// </summary>
        /// <typeparam name="TValue">The type of the value to serialize.</typeparam>
        /// <param name="client">The client used to make the request.</param>
        /// <param name="requestUri">The Uri the request is sent to.</param>
        /// <param name="value">The value to serialize.</param>
        /// <param name="options">Options for running the serialization.</param>
        /// <param name="cancellationToken">
        ///     A cancellation token that can be used by other objects or threads to receive notice of
        ///     cancellation.
        /// </param>
        /// <returns>A task object representing the asynchronous operation.</returns>
        public static Task<HttpResponseMessage> PutAsMessagePackAsync<TValue>(this HttpClient client, string requestUri,
            TValue value, MessagePackSerializerOptions? options, CancellationToken cancellationToken = default)
        {
            Guard.NotNull(client, nameof(client));

            var content = MessagePackContent.Create(value, options);
            return client.PutAsync(requestUri, content, cancellationToken);
        }

        /// <summary>
        ///     Send a PUT request to the specified Uri containing the <paramref name="value" /> serialized as MessagePack in the
        ///     request body.
        /// </summary>
        /// <typeparam name="TValue">The type of the value to serialize.</typeparam>
        /// <param name="client">The client used to make the request.</param>
        /// <param name="requestUri">The Uri the request is sent to.</param>
        /// <param name="value">The value to serialize.</param>
        /// <param name="cancellationToken">
        ///     A cancellation token that can be used by other objects or threads to receive notice of
        ///     cancellation.
        /// </param>
        /// <returns>A task object representing the asynchronous operation.</returns>
        public static Task<HttpResponseMessage> PutAsMessagePackAsync<TValue>(this HttpClient client, string requestUri,
            TValue value, CancellationToken cancellationToken = default)
        {
            Guard.NotNull(client, nameof(client));

            return client.PutAsMessagePackAsync(requestUri, value, null, cancellationToken);
        }

        /// <summary>
        ///     Send a PUT request to the specified Uri containing the <paramref name="value" /> serialized as MessagePack in the
        ///     request body.
        /// </summary>
        /// <typeparam name="TValue">The type of the value to serialize.</typeparam>
        /// <param name="client">The client used to make the request.</param>
        /// <param name="requestUri">The Uri the request is sent to.</param>
        /// <param name="value">The value to serialize.</param>
        /// <param name="options">Options for running the serialization.</param>
        /// <param name="cancellationToken">
        ///     A cancellation token that can be used by other objects or threads to receive notice of
        ///     cancellation.
        /// </param>
        /// <returns>A task object representing the asynchronous operation.</returns>
        public static Task<HttpResponseMessage> PutAsMessagePackAsync<TValue>(this HttpClient client, Uri requestUri,
            TValue value, MessagePackSerializerOptions? options, CancellationToken cancellationToken = default)
        {
            Guard.NotNull(client, nameof(client));

            var content = MessagePackContent.Create(value, options);
            return client.PutAsync(requestUri, content, cancellationToken);
        }

        /// <summary>
        ///     Send a PUT request to the specified Uri containing the <paramref name="value" /> serialized as MessagePack in the
        ///     request body.
        /// </summary>
        /// <typeparam name="TValue">The type of the value to serialize.</typeparam>
        /// <param name="client">The client used to make the request.</param>
        /// <param name="requestUri">The Uri the request is sent to.</param>
        /// <param name="value">The value to serialize.</param>
        /// <param name="cancellationToken">
        ///     A cancellation token that can be used by other objects or threads to receive notice of
        ///     cancellation.
        /// </param>
        /// <returns>A task object representing the asynchronous operation.</returns>
        public static Task<HttpResponseMessage> PutAsMessagePackAsync<TValue>(this HttpClient client, Uri requestUri,
            TValue value, CancellationToken cancellationToken = default)
        {
            Guard.NotNull(client, nameof(client));

            return client.PutAsMessagePackAsync(requestUri, value, null, cancellationToken);
        }
    }
}