# Handling Dependabot PRs
We use [Dependabot](https://github.com/Automattic/woocommerce-payments/security/dependabot) to monitor our PHP and JavaScript package dependencies.
Dependabot will open a PR when an update is available for a package. We can then review the PR, test the change, and
merge to apply the update. The testing and review needed will depend on the package being updated. This document aims
to catalog our packages and provide guidance to a developer who wants to test and merge Dependabot PRs.

* [Open issues from Dependabot](https://github.com/Automattic/woocommerce-payments/security/dependabot).
* [Open PRs from Dependabot](https://github.com/Automattic/woocommerce-payments/pulls/app%2Fdependabot).
  * Currently open PRs are limited to 5 for each `package-ecosystem`.
* [Dependabot config](https://github.com/Automattic/woocommerce-payments/blob/develop/.github/dependabot.yml).

# Some Notes on Packages
* A lot of our JavaScript dev dependencies are provided by WordPress or WooCommerce globally at runtime, and this is why we include them
  only as dev dependencies. We aren't duplicating them in our build bundle but our unit tests can still pass by
  having them available. 
* We use the `@woocommerce/dependency-extraction-webpack-plugin` to make WebPack aware of what
  can be found globally at runtime. The configuration for this can be found in
  [`webpack.config.js`](https://github.com/Automattic/woocommerce-payments/blob/develop/webpack.config.js). Any `wordpress/*`, `woocommerce/*`
  and [some other packages](https://www.npmjs.com/package/@woocommerce/dependency-extraction-webpack-plugin) are removed from
  the built bundle by default, in addition to any packages listed in the configuration file. Returning `null` in the configuration
  indicates that we want to bundle the package rather than using the globally available one.
* We should keep the versions of these packages aligned on the versions used in our
  minimum supported versions of [WordPress](https://github.com/WordPress/wordpress-develop/blob/x.y.z/package.json) and [WooCommerce](https://github.com/woocommerce/woocommerce/blob/x.y.z/plugins/woocommerce/package.json) (replace `x.y.z`  by the version in the links).
  * The minimum supported version of WordPress can be found [here](https://github.com/Automattic/woocommerce-payments/blob/develop/woocommerce-payments.php#L13)
  * The minimum supported version of WooCommerce can be found [here](https://github.com/Automattic/woocommerce-payments/blob/develop/woocommerce-payments.php#L11)

# Review Process
1. Check the tables below for the package you’re reviewing.
2. Add the package to the table if isn’t already present.
3. Perform any suggested testing based on the size of the update (major / minor / patch).
4. Check that our CI tests have all passed, realistically these should find 90% of problems for us.
5. Merge the PR. If there are any lingering concerns, consider merging immediately after a release code freeze so that we benefit from several testing cycles.

## PHP Dev Dependencies
| Package Name                                  | Usage Summary                        | Testing                                                   | Notes                     |
|-----------------------------------------------|--------------------------------------|-----------------------------------------------------------|---------------------------|
| [vimeo/psalm](https://github.com/vimeo/psalm) | Used for type checking our PHP code. | Run the tool, make sure it completes and produces output. | Run with `npm run psalm`. |

## JavaScript Dev Dependencies
| Package Name                                                                                     | Usage Summary                                                                                                                                                           | Testing                                                                                                                                                                                                                                                                                    | Notes                                                                                                                                                                                                                                                                                                                                  |
|--------------------------------------------------------------------------------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| [husky](https://www.npmjs.com/package/husky)                                                     | Used to run hooks pre/post commit, like automatically running PHPCS.                                                                                                    | Ensure `composer install` runs automatically when checking out another branch.                                                                                                                                                                                                             |                                                                                                                                                                                                                                                                                                                                        |
| [lodash](https://www.npmjs.com/package/lodash)                                                   | Lodash makes JavaScript easier by taking the hassle out of working with arrays, numbers, objects, strings, etc.                                                         | JS tests should pass.                                                                                                                                                                                                                                                                      |                                                                                                                                                                                                                                                                                                                                        |
| [node](https://www.npmjs.com/package/node)                                                       | Not a package, but we declare the supported version of node in our `.nvmrc` file. We use node to build the JavaScript for the plugin and run the JavaScript unit tests. | Ensure you're running the new version of node by running the `nvm use` command or manually setting up the correct version. <br> - For minor and patch upgrades, it is sufficient to test that the build runs.<br> - For major versions, smoke testing the running plugin would be advised. |                                                                                                                                                                                                                                                                                                                                        |
| [@woocommerce/currency](https://www.npmjs.com/package/@woocommerce/currency)                     | A collection of utilities to display and work with currency values.                                                                                                     | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/url](https://www.npmjs.com/package/@wordpress/url)                                   | A collection of utilities to manipulate URLs.                                                                                                                           | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/data](https://www.npmjs.com/package/@wordpress/data)                                 | It serves as a hub to manage application state for both plugins and WordPress itself, providing tools to manage data within and between distinct modules.               | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/i18n](https://www.npmjs.com/package/@wordpress/i18n)                                 | Internationalization utilities for client-side localization.                                                                                                            | JS unit tests are passing.                                                                                                                                                                                                                                                                 | The `wpi18n` used in `postbuild:client` script comes from `node-wp-i18n` and is thus not related to this package.                                                                                                                                                                                                                      |
| [@wordpress/date](https://www.npmjs.com/package/@wordpress/date)                                 | Date module for WordPress.                                                                                                                                              | JS unit tests are passing.                                                                                                                                                                                                                                                                 | From v4.6.0, the `TZ` env var from the Jest global setup is not taken into account anymore, hence unit tests fail.                                                                                                                                                                                                                     |
| [@wordpress/api-fetch](https://www.npmjs.com/package/@wordpress/api-fetch)                       | Utility to make WordPress REST API requests.                                                                                                                            | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@woocommerce/date](https://www.npmjs.com/package/@woocommerce/date)                             | A collection of utilities to display and work with date values.                                                                                                         | JS unit tests are passing.                                                                                                                                                                                                                                                                 | Though there is no direct use of this package, it is used by [jest-test-file-setup.js](https://github.com/Automattic/woocommerce-payments/blob/b64178138d44d3bd3aa2a692d1c84e4d91e521b9/tests/js/jest-test-file-setup.js#L25)                                                                                                          |
| [@wordpress/hooks](https://www.npmjs.com/package/@wordpress/hooks)                               | A lightweight & efficient EventManager for JavaScript.                                                                                                                  | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/plugins](https://www.npmjs.com/package/@wordpress/plugins)                           | Plugins module for WordPress.                                                                                                                                           | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/icons](https://www.npmjs.com/package/@wordpress/icons)                               | WordPress Icons Library.                                                                                                                                                | JS unit tests are passing and UI isn't affected at places of usage.                                                                                                                                                                                                                        | This package is one of the few `@wordpress` packages that doesn't come with WordPress automatically (default behavior inherited from the [WP dependency extractor webpack config](https://github.com/WordPress/gutenberg/blob/a42fd75977a6469050564b8fc821e8018bd566e0/packages/dependency-extraction-webpack-plugin/lib/util.js#L2)). |
| [@wordpress/element](https://www.npmjs.com/package/@wordpress/element)                           | An abstraction layer atop React.                                                                                                                                        | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/scripts](https://www.npmjs.com/package/@wordpress/scripts)                           | Collection of reusable scripts tailored for WordPress development.                                                                                                      | JS unit tests and E2E pipeline works.                                                                                                                                                                                                                                                      | Places of usages can be found by searching for `wp-scripts`.<br/> Version `>=20` requires `jest >=27`, and we're currently using `v26.x`                                                                                                                                                                                               |
| [@woocommerce/explat](https://www.npmjs.com/package/@woocommerce/explat/)                        | Component and utility functions that can be used to run A/B Tests in WooCommerce dashboard and reports pages.                                                           | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@woocommerce/experimental](https://www.npmjs.com/package/@wordpress/plugins)                    | A collection of component imports and exports that are aliases for components transitioning from experimental to non-experimental.                                      | JS unit tests are passing.                                                                                                                                                                                                                                                                 | Needed to install `@types/react-transition-group` as JS linting could not find a declaration file for module `react-transition-group/CSSTransition`.                                                                                                                                                                                   |
| [@wordpress/dom-ready](https://www.npmjs.com/package/@wordpress/dom-ready)                       | Execute callback after the DOM is loaded.                                                                                                                               | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/html-entities](https://www.npmjs.com/package/@wordpress/html-entities)               | HTML entity utilities for WordPress.                                                                                                                                    | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/blocks](https://www.npmjs.com/package/@wordpress/blocks)                             | Blocks utilities for WordPress.                                                                                                                                         | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/block-editor](https://www.npmjs.com/package/@wordpress/block-editor)                 | Allows you to create and use standalone block editors.                                                                                                                  | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/jest-preset-default](https://www.npmjs.com/package/@wordpress/jest-preset-default)   | Default Jest preset for WordPress development.                                                                                                                          | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/babel-plugin-makepot](https://www.npmjs.com/package/@wordpress/babel-plugin-makepot) | Babel plugin used to scan JavaScript files for use of localization functions.                                                                                           | Ensure `languages/woocommerce-payments.pot` is created correctly when building a release.                                                                                                                                                                                                  | The script command for building a release is: `npm run build:release`                                                                                                                                                                                                                                                                  |
| [@wordpress/base-styles](https://www.npmjs.com/package/@wordpress/base-styles)                   | Base SCSS utilities and variables for WordPress.                                                                                                                        | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/data-controls](https://www.npmjs.com/package/@wordpress/data-controls)               | The data controls module is a module intended to simplify implementation of common controls used with the `@wordpress/data` package.                                    | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [react](https://www.npmjs.com/package/react)                                                     | JavaScript library for creating user interfaces.                                                                                                                        | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [react-dom](https://www.npmjs.com/package/react-dom)                                             | Serves as the entry point to the DOM and server renderers for React.                                                                                                    | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@types/react](https://www.npmjs.com/package/@types/react)                                       | Contains type definitions for React.                                                                                                                                    | JS unit tests are passing.                                                                                                                                                                                                                                                                 | You should pick a version `x.y.z` with `x.y` similar to the version defined for React.                                                                                                                                                                                                                                                 |
| [@woocommerce/components](https://www.npmjs.com/package/@woocommerce/components)                 | A library of components that can be used to create pages in the WooCommerce dashboard and reports pages.                                                                | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |
| [@wordpress/components](https://www.npmjs.com/package/@wordpress/components)                     | A library of generic WordPress components to be used for creating common UI elements.                                                                                   | JS unit tests are passing and UI isn't affected at places of usage.                                                                                                                                                                                                                        | This package is one of the few `@wordpress/x` packages that doesn't come from WordPress directly, because we decided to bundle it ourselves (see our [dependency extractor webpack config](https://github.com/Automattic/woocommerce-payments/blob/b7634468560d905a479d50066233f807da62413f/webpack/shared.js#L132-L150)).             |
| [@woocommerce/csv-export](https://www.npmjs.com/package/@woocommerce/csv-export)                 | A set of functions to convert data into CSV values, and enable a browser download of the CSV data. We use it to export reports for Transactions, Deposits, and Disputes.                                                                | JS unit tests are passing.                                                                                                                                                                                                                                                                 |                                                                                                                                                                                                                                                                                                                                        |

### PHP Runtime Dependencies
| Package Name                                                                      | Usage Summary                                                              | Testing                                                                                                                                                                                                                                       | Notes                                                                                                    |
|-----------------------------------------------------------------------------------|----------------------------------------------------------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------------------------------------------------------------------------------------------------|
| [automattic/jetpack-autoloader](https://github.com/Automattic/jetpack-autoloader) | Loads other Automattic Composer packages. For example, jetpack-connection. | Testing that packages load correctly should be sufficient.<br> For major version upgrades, testing this package in a number of scenarios is advised, to rule out bugs in configurations we might not usually run in development environments. | As the number of packages we’re loading with jetpack-autoloader increases, so will the required testing. |

### JavaScript Runtime Dependencies
| Package Name                                                         | Usage Summary                                                         | Testing                                                                                                                                                                                         | Notes                                                                                                                                                                         |
|----------------------------------------------------------------------|-----------------------------------------------------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| [@stripe/stripe-js](https://www.npmjs.com/package/@stripe/stripe-js) | Powers all the direct communication with Stripe in the user's browser | Reviewing Stripe's changelog is a good idea, then tailoring your testing to that. Testing payments, saving payment information, different payment methods, UPE on and off are all good options. | We configure which version of the Stripe API we want to use, although we aim to stay up to date there it's also worth ensuring the new version of this package is compatible. |
