//
//  ExtensionDoc_fromAE.swift
//  Aurora Editor
//
//  Created by Wesley de Groot on 09/07/2024.
//  Copyright © 2024 Aurora Company. All rights reserved.
//

import Foundation
import AEExtensionKit

/// This class does not execute any code, this is made to generate documentation for the extension system
/// This are items which an extension can expect from AuroraEditor.
/// To see what you can send to Aurora Editor see ``ExtensionDocumentationToAuroraEditor``.
///
/// To send a command from Aurora Editor to the extensions use:
///
/// ```swift
/// ExtensionsManager.shared.sendEvent(
///     event: "auroraeditorDidChangeFontNameTo",
///     parameters: [
///         "name": "Aurora Editor Font"
///     ]
/// )
/// ```
public protocol ExtensionDocumentationFromAuroraEditor {
    /// Build the editor view for a specific file.
    ///
    /// Please note the view will be rendered using the first "view" key provided in the return value
    ///
    /// - Parameter file: The file url (relative path)
    /// - Returns: An array containing the necessary elements to build an editor.
    func buildEditor(file: String) -> [String: Any]

    /// Can open a file
    ///
    /// - Parameter file: The file url (relative path)
    /// - Returns: Whether the file can be opened or not using a custom editor
    func canOpen(file: String) -> Bool

    /// Did open file
    ///
    /// - Parameter file: file name
    /// - Parameter contents: file contents
    func didOpen(file: String, contents: String)

    /// Did open file (in workspace)
    ///
    /// - Parameter workspace: workspace name
    /// - Parameter file: file name
    /// - Parameter contents: file contents
    func didOpen(workspace: String, file: String, contents: String)

    /// Did edit file (in workspace)
    ///
    /// - Parameter workspace: workspace name
    /// - Parameter file: file name
    /// - Parameter contents: file contents
    /// - Parameter isEdited: file is edited?
    func didBeginEditing(workspace: String, file: String, contents: String, isEdited: Bool)

    /// Did save file
    ///
    /// - Parameter file: filename
    func didSave(file: String)

    /// Did move caret to position
    ///
    /// - Parameters row: row
    /// - Parameter column: column
    func didMoveCaret(row: Int, column: Int)

    /// Did activate tab
    ///
    /// - Parameter file: filename of the tab
    func didActivateTab(file: String)

    /// Did deactivate tab
    ///
    /// - Parameter file: filename of the tab
    func didDeactivateTab(file: String)

    /// Did toggle navigator pane
    ///
    /// - Parameter visible: is the pane visible?
    func didToggleNavigatorPane(visible: Bool)

    /// Did toggle inspector pane
    ///
    /// - Parameter visible: is the pane visible?
    func didToggleInspectorPane(visible: Bool)

    /// Register callback
    ///
    /// - Parameter callback: Callback handler ``AuroraAPI``
    /// - note: Only required for Swift Extensions
    func registerCallback(callback: (AuroraAPI) -> Bool)

    /// Did open a view generated by an extension
    ///
    /// - Parameter type: View type
    /// - Parameter extension: Extension name
    /// - Parameter view: View
    func didOpenExtensionView(type: String, extension: String, view: Any)

    /// Did close a view generated by an extension
    ///
    /// - Parameter type: View type
    /// - Parameter extension: Extension name
    /// - Parameter view: View
    func didCloseExtensionView(type: String, extension: String, view: Any)

    /// Did fail to generate a view from an extension
    /// - Parameter type: View type
    /// - Parameter extension: Extension name
    /// - Parameter view: View
    func didFailToOpenExtensionView(type: String, extension: String, view: Any)

    /// Did click on notification
    ///
    /// - Parameter identifier: Notification identifier
    /// - Parameter extension: Extension name
    /// - Parameter title: Notification title
    /// - Parameter message: Notification message
    func didClickOnNotification(
        identifier: String,
        extension: String,
        title: String,
        message: String
    )

    /// Did dismiss an notification
    ///
    /// - Parameter identifier: Notification identifier
    /// - Parameter extension: Extension name
    func didDismissNotification(identifier: String, extension: String)

    /// Command pallette did appear
    func commandPalletteDidAppear()

    /// Command pallette did disappear
    func commandPalletteDidDisappear()

    /// Command palette did select (run)
    /// 
    /// - Parameter name: Command name
    func commandPalletteDidSelect(name: String)

    /// Welcome window did appear
    func welcomeDidAppear()

    /// Welcome window did disappear
    func welcomeDidDisappear()

    /// Terminal did appear
    ///
    /// - Parameter path: Current Terminal Path
    func terminalDidAppear(path: String)

    /// Terminal did appear
    ///
    /// - Parameter path: Current Terminal Path
    func terminalDidDisappear(path: String)

    /// Preferences did appear
    func preferencesDidAppear()

    /// Preferences did disappear
    func preferencesDidDisppear()

    /// Project navigator did appear
    func projectNavigatorDidAppear()

    /// Project navigator did disappear
    func projectNavigatorDidDisappear()

    /// Source control navigator did appear
    func sourceControlNavigatorDidAppear()

    /// Source control navigator did disappear
    func sourceControlNavigatorDidDisappear()

    /// Find navigator did appear
    func findNavigatorDidAppear()

    /// Find navigator did disappear
    func findNavigatorDidDisappear()

    /// Notifications navigator did appear
    func notificationsNavigatorDidAppear()

    /// Notifications navigator did disappear
    func notificationsNavigatorDidDisappear()

    /// Hierarchy navigator did appear
    func hierarchyNavigatorDidAppear()

    /// Hierarchy navigator did disappear
    func hierarchyNavigatorDidDisappear()

    /// Extension navigator did appear
    func extensionNavigatorDidAppear()

    /// Extension navigator did disappear
    func extensionNavigatorDidDisappear()
}
