package io.github.aquerr.eaglefactions.api.storage;

import io.github.aquerr.eaglefactions.api.entities.Faction;
import io.github.aquerr.eaglefactions.api.entities.FactionPlayer;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.nio.file.Path;
import java.util.List;
import java.util.Set;
import java.util.UUID;

public interface StorageManager
{
    /**
     * Adds a new faction to the storage and cache. Or, updates an existing faction in the storage and in the cache.
     * @param faction the faction that should be added or updated.
     */
    void saveFaction(final Faction faction);

    /**
     * Deletes faction from the storage and cache.
     * @param factionName name of the faction that should be deleted.
     * @return true if operation succeed or false if not
     */
    boolean deleteFaction(final String factionName);

    /**
     * Gets a faction from the storage.
     * If there is no faction with given faction name then null is returned.
     * @param factionName the name of the faction to get
     * @return faction or null if faction does not exist
     */
    @Nullable
    Faction getFaction(final String factionName);

    /**
     * Reloads storage. If any changes have been made in the files or database manually then this method loads all data into the storage cache.
     */
    void reloadStorage();

    /**
     * Gets all unique players who were registered/added to the Eagle Factions storage.
     * @return a Set of {@link FactionPlayer} that were added to the Eagle Factions storage.
     */
    Set<FactionPlayer> getServerPlayers();

    /**
     * Adds a player to the storage.
     * @param factionPlayer the faction player.
     * @return true if operation succeed or false if not.
     */
    boolean savePlayer(FactionPlayer factionPlayer);

    /**
     * Gets the faction player for the given unique id.
     *
     * @param playerUUID the UUID of the player.
     * @return faction player or null if player does not exists.
     */
    @Nullable FactionPlayer getPlayer(UUID playerUUID);

    /**
     * Creates a backup of Eagle Factions data.
     * By default, backups are placed in /config/eaglefactions/backups.
     *
     * @return backup path if operation succeed or null if not
     */
    Path createBackup();

    /**
     * Restores Eagle Factions data from the given backup.
     * @param backupName the name of the backup file.
     * @return true if operation succeed or false if not
     */
    boolean restoreBackup(String backupName);

    /**
     * Gets a list of backup files' names.
     * @return the list of backup files.
     */
    List<String> listBackups();
}
