package io.github.aquerr.eaglefactions.api.logic;

import io.github.aquerr.eaglefactions.api.entities.Claim;
import io.github.aquerr.eaglefactions.api.entities.Faction;
import io.github.aquerr.eaglefactions.api.entities.FactionChest;
import io.github.aquerr.eaglefactions.api.entities.FactionHome;
import io.github.aquerr.eaglefactions.api.entities.ProtectionFlagType;
import io.github.aquerr.eaglefactions.api.managers.claim.provider.FactionMaxClaimCountProvider;
import net.kyori.adventure.text.format.NamedTextColor;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.spongepowered.api.entity.living.player.server.ServerPlayer;
import org.spongepowered.api.world.server.ServerLocation;
import org.spongepowered.math.vector.Vector3i;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

/**
 * Interface for all faction related actions.
 *
 * Other plugins should always use this class for interacting with factions.
 */
public interface FactionLogic
{
    /**
     * Add {@link FactionMaxClaimCountProvider} to providers used in calculating maximal claim count for faction in {@link FactionLogic#getFactionMaxClaims(Faction)}.
     * @param provider new provider
     */
    void addFactionMaxClaimCountProvider(FactionMaxClaimCountProvider provider);

    /**
     * Overrides providers used in calculating maximal claim count for faction.
     * @param providers
     */
    void setFactionMaxClaimCountProviders(Set<FactionMaxClaimCountProvider> providers);

    /**
     * Gets providers used in calculating maximal claim count for faction.
     * @return an unmodifiable set of {@link FactionMaxClaimCountProvider}
     */
    Set<FactionMaxClaimCountProvider> getFactionMaxClaimCountProviders();

    /**
     * Gets {@link Faction} for the given player {@link UUID}.
     * @param playerUUID the UUID of the player whose faction should be looked for.
     * @return {@link Optional} if the given player is in a faction or
     * {@link Optional#empty()} if the given player is not in a faction.
     */
    Optional<Faction> getFactionByPlayerUUID(UUID playerUUID);

    /**
     * Gets {@link Faction} for the given chunk location {@link Vector3i} in the given world UUID {@link UUID}.
     * @param worldUUID the UUID of the world
     * @param chunk the location of the chunk
     * @return {@link Optional} if the given chunk claimed by a faction or
     * {@link Optional#empty()} if the given chunk is not claimed by any faction.
     */
    Optional<Faction> getFactionByChunk(UUID worldUUID, Vector3i chunk);

    /**
     * Gets {@link Faction} by the given faction name.
     * @param factionName the name of the faction that should be looked for.
     * @return {@link Faction} or null if factions could not be found.
     */
    @Nullable
    Faction getFactionByName(String factionName);

    /**
     * Gets a {@link List} that contains all online players in a given {@link Faction}
     * @param faction the faction that should be used to get online players from.
     * @return {@link List} list with online players in the given faction.
     */
    List<ServerPlayer> getOnlinePlayers(Faction faction);

    /**
     * Gets all faction names that exists on the server.
     * @return {@link Set} that contains all faction names on the server.
     */
    Set<String> getFactionsNames();

    /**
     * Gets a {@link Map} that contains all factions that exists on the server from the plugin cache.
     * @return {@link Map} instance where keys are factions names and values are factions objects.
     */
    Map<String, Faction> getFactions();

    /**
     * Disbands/Deletes a faction.
     * @param factionName name of the faction that should be disbanded/deleted.
     * @return true if operation succeeded or false if it did not.
     */
    boolean disbandFaction(String factionName);

    /**
     * Joins a player to the given faction.
     * This method first tries to get the faction object and then removes player from it.
     * @param playerUUID the UUID of the player that should be removed from the faction.
     * @param factionName the name of the faction.
     */
    void leaveFaction(UUID playerUUID, String factionName);

    /**
     * Sets player's faction
     * @param playerUUID the UUID of the player.
     * @param factionName the name of the faction.
     * @param rankName the rank that will be given to the player.
     */
    void setFaction(UUID playerUUID, String factionName, @Nullable String rankName);

    /**
     * Creates a truce between given factions.
     * @param playerFactionName  the name of the first faction.
     * @param invitedFactionName the name of the second faction.
     */
    void addTruce(String playerFactionName, String invitedFactionName);

    /**
     * Disbands a truce between given factions.
     * @param playerFactionName  the name of the first faction.
     * @param removedFactionName the name of the second faction.
     */
    void removeTruce(String playerFactionName, String removedFactionName);

    /**
     * Creates an alliance between given factions.
     * @param playerFactionName  the name of the first faction.
     * @param invitedFactionName the name of the second faction.
     */
    void addAlly(String playerFactionName, String invitedFactionName);

    /**
     * Disbands an alliance between given factions.
     * @param playerFactionName  the name of the first faction.
     * @param removedFactionName the name of the second faction.
     */
    void removeAlly(String playerFactionName, String removedFactionName);

    /**
     * Sets enemy state between given factions.
     * @param playerFactionName  the name of the first faction.
     * @param enemyFactionName the name of the second faction.
     */
    void addEnemy(String playerFactionName, String enemyFactionName);

    /**
     * Removes enemy state between given factions.
     * @param playerFactionName  the name of the first faction.
     * @param enemyFactionName the name of the second faction.
     */
    void removeEnemy(String playerFactionName, String enemyFactionName);

    /**
     * Gets all claimed chunks on the server with corresponding factions.
     * @return {@link Map} that contains all claimed chunks.
     */
    Map<Claim, Faction> getAllClaims();

    /**
     * Adds claims to the given faction.
     * @param faction the faction that should acquire claims.
     * @param claims list of claims that should be added to the faction.
     */
    void addClaims(Faction faction, List<Claim> claims);

    /**
     * Directly adds a claim to the given faction without performing any extra logic.
     *
     * Consider using {@link io.github.aquerr.eaglefactions.api.managers.claim.ClaimManager#claim(ServerPlayer, Faction, ServerLocation)}
     * if you want to perform the full claiming mechanism (claiming with delay, claiming by using items).
     *
     * @see io.github.aquerr.eaglefactions.api.managers.claim.ClaimManager#claim(ServerPlayer, Faction, ServerLocation)
     *
     * @param faction the faction that should acquire claim.
     * @param claim the claim that should be added to the faction.
     */
    void addClaim(Faction faction, Claim claim);

    /**
     * Directly adds a claim from the given faction.
     * @param faction the faction that should lose a claim.
     * @param claim the claim that should be removed from the faction.
     */
    void removeClaim(Faction faction, Claim claim);

    /**
     * Works the same as {@link FactionLogic#removeClaim(Faction, Claim)} but it is used after successful attack to spawn different particles.
     * @param faction the faction that should lose a claim.
     * @param claim the claim that should be removed from the faction.
     */
	void destroyClaim(Faction faction, Claim claim);

    /**
     * Checks if the chunk {@link Vector3i} in the given world {@link UUID} is claimed.
     * @param worldUUID the UUID of the world
     * @param chunk the position of the chunk
     * @return true if chunk is claimed or false if it is not.
     */
	boolean isClaimed(UUID worldUUID, Vector3i chunk);

    /**
     * Checks if a {@link Claim} is connected to other claims in the given {@link Faction}
     * @param faction the faction object to perform check against
     * @param claimToCheck the claim that should be checked
     * @return true if chunk is connected to other claims or false if it is not.
     */
    boolean isClaimConnected(Faction faction, Claim claimToCheck);

    /**
     * Adds new owner of the given claim.
     * @param faction the faction that owns the claim.
     * @param claim the claim
     * @param owner the owner that should be added as owner of the given claim.
     * @return true if operation succeeded, false if not.
     */
    boolean addClaimOwner(final Faction faction, final Claim claim, final UUID owner);

    /**
     * Removes owner of the given claim.
     * @param faction the faction that owns the claim.
     * @param claim the claim
     * @param owner the owner that should be removed from the given claim.
     * @return true if operation succeeded, false if not.
     */
    boolean removeClaimOwner(final Faction faction, final Claim claim, final UUID owner);

    /**
     * Sets claims accessibility by faction.
     * @param faction the faction that own the claim.
     * @param claim the claim.
     * @param isAccessibleByFaction the
     */
    void setClaimAccessibleByFaction(final Faction faction, final Claim claim, final boolean isAccessibleByFaction);

    /**
     * Sets home for the given {@link Faction}
     * @param faction the faction for which home should be set
     * @param home new faction's home.
     */
    void setHome(Faction faction, @Nullable FactionHome home);

    /**
     * Gets the list of all factions tags used on the server.
     * @return {@link List} that contains all factions tags.
     */
    List<String> getFactionsTags();

    /**
     * Checks if the given faction has online players on the server.
     * @param faction the faction that check should be run against.
     * @return true if there are player on the server that are in that faction or false if this faction has no players who are currently online.
     */
    boolean hasOnlinePlayers(Faction faction);

    /**
     * Removes all claims of the given faction.
     * @param faction the faction that claims should be removed.
     */
    void removeAllClaims(Faction faction);

    /**
     * Kicks player from the given faction.
     * @param playerUUID the UUId of the player that should be kicked.
     * @param factionName the name of the faction that the player should be kicked from.
     */
    void kickPlayer(UUID playerUUID, String factionName);

    /**
     * Changes color of the faction tag.
     * @param faction the faction that should be affected.
     * @param textColor new {@link NamedTextColor} that should be used for the faction tag.
     */
    void changeTagColor(Faction faction, NamedTextColor textColor);

    /**
     * Sets faction last online time.
     * @param faction that should be affected by this change.
     * @param instantTime new time as {@link Instant}
     */
    void setLastOnline(Faction faction, Instant instantTime);

    /**
     * Changes the name of the faction.
     * @param faction that should be affected by this change.
     * @param newFactionName new faction name.
     */
    void renameFaction(Faction faction, String newFactionName);

    /**
     * Changes the tag of the faction.
     * @param faction that should be affected by this change.
     * @param newTag new faction tag.
     */
    void changeTag(Faction faction, String newTag);

    /**
     * Sets faction chest.
     * @param faction that should be affected by this change.s
     * @param inventory new {@link FactionChest}
     */
    void setChest(Faction faction, FactionChest inventory);

    /**
     * Sets faction description.
     * @param faction that should be affected by this change.
     * @param description new description.
     */
    void setDescription(Faction faction, String description);

    /**
     * Sets faction message of the day.
     * @param faction that should be affected by this change.
     * @param motd new message of the day.
     */
    void setMessageOfTheDay(Faction faction, String motd);

    /**
     * Sets if faction is public (people can join it without invitation) or private (normal faction).
     * @param faction that should be affected by this change.
     * @param isPublic boolean value, true if faction should be public, false if not.
     */
	void setIsPublic(Faction faction, boolean isPublic);

    /**
     * Gets maximal amount of claims a faction can have.
     * @param faction the faction that the maximal amount of claims should be got from.
     * @return maximal amount of claims for given faction.
     */
    int getFactionMaxClaims(final Faction faction);

    /**
     * Updates faction's protection flag value
     * @param faction the faction
     * @param flagType flag type
     * @param value new value
     */
    void setFactionProtectionFlag(Faction faction, ProtectionFlagType flagType, boolean value);
}
