{*******************************************************}
{          Linkbar - Windows desktop toolbar            }
{            Copyright (c) 2010-2021 Asaq               }
{*******************************************************}

unit NewSpin;

interface

uses
  Windows, SysUtils, Classes, Controls, StdCtrls, ExtCtrls, Messages,
  Buttons, ComCtrls;

type

  TTimerSpeedButton = class;

{ TSpinButton }

  TSpinButton = class (TWinControl)
  private
    FUpDownButton: TTimerSpeedButton;
    FFocusControl: TWinControl;
    FOnUpClick: TNotifyEvent;
    FOnDownClick: TNotifyEvent;
    function CreateButton: TTimerSpeedButton;
    procedure BtnClick(Sender: TObject; Button: TUDBtnType);
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure AdjustSize (var W, H: Integer); reintroduce;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
  protected
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
  published
    property Align;
    property Anchors;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FocusControl: TWinControl read FFocusControl write FFocusControl;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnDownClick: TNotifyEvent read FOnDownClick write FOnDownClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnStartDock;
    property OnStartDrag;
    property OnUpClick: TNotifyEvent read FOnUpClick write FOnUpClick;
  end;

{ TnSpinEdit }

  TnSpinEdit = class(TCustomEdit)
  private
    FMinValue: LongInt;
    FMaxValue: LongInt;
    FButton: TSpinButton;
    FEditorEnabled: Boolean;
    function GetValue: LongInt;
    function CheckValue (NewValue: LongInt): LongInt;
    procedure SetValue (NewValue: LongInt);
    procedure SetMinValue (NewMinValue: LongInt);
    procedure SetMaxValue (NewMaxValue: LongInt);
    procedure SetEditRect;
    function GetIncrement: LongInt;
    procedure SetIncrement(NewValue: LongInt);
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure CMEnter(var Message: TCMGotFocus); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure WMPaste(var Message: TWMPaste); message WM_PASTE;
    procedure WMCut(var Message: TWMCut); message WM_CUT;
  protected
    function IsValidChar(Key: Char): Boolean; virtual;
    procedure UpClick (Sender: TObject); virtual;
    procedure DownClick (Sender: TObject); virtual;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Button: TSpinButton read FButton;
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
  published
    property Align;
    property Anchors;
    property AutoSelect;
    property AutoSize;
    property Color;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property EditorEnabled: Boolean read FEditorEnabled write FEditorEnabled default True;
    property Enabled;
    property Font;
    property Increment: LongInt read GetIncrement write SetIncrement default 1;
    property MaxLength;
    property MaxValue: LongInt read FMaxValue write SetMaxValue;
    property MinValue: LongInt read FMinValue write SetMinValue;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Value: LongInt read GetValue write SetValue;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

{ TTimerSpeedButton }

  TTimeBtnState = set of (tbFocusRect, tbAllowTimer);

  TTimerSpeedButton = class(TUpDown)
  private
    FRepeatTimer: TTimer;
    FTimeBtnState: TTimeBtnState;
    procedure TimerExpired(Sender: TObject);
  protected
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
  public
    destructor Destroy; override;
    property TimeBtnState: TTimeBtnState read FTimeBtnState write FTimeBtnState;
  end;

  procedure Register;

implementation

uses Themes;

const
  InitRepeatPause = 400;  { pause before repeat timer (ms) }
  RepeatPause     = 100;  { pause before hint window displays (ms)}

procedure Register;
begin
  RegisterComponents('Standard', [TnSpinEdit]);
end;

{ TSpinButton }

constructor TSpinButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csFramed, csOpaque];
  { Frames don't look good around the buttons when themes are on }
  if StyleServices.Enabled
  then ControlStyle := ControlStyle - [csFramed];
  FUpDownButton := CreateButton;
  Width := 15;
  Height := 25;
end;

function TSpinButton.CreateButton: TTimerSpeedButton;
begin
  Result := TTimerSpeedButton.Create(Self);
  Result.OnClick := BtnClick;
  Result.OnMouseDown := BtnMouseDown;
  Result.Visible := True;
  Result.Enabled := True;
  Result.TimeBtnState := [tbAllowTimer];
  Result.Parent := Self;
end;

procedure TSpinButton.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove)
     and (AComponent = FFocusControl)
  then FFocusControl := nil;
end;

procedure TSpinButton.AdjustSize(var W, H: Integer);
begin
  if (FUpDownButton = nil)
     or (csLoading in ComponentState)
  then Exit;
  if (W < 15)
  then W := 15;
  FUpDownButton.SetBounds(0, 0, W, H);
end;

procedure TSpinButton.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize(W, H);
  inherited SetBounds(ALeft, ATop, W, H);
end;

procedure TSpinButton.WMSize(var Message: TWMSize);
var W, H: Integer;
begin
  inherited;
  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize(W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TSpinButton.KeyDown(var Key: Word; Shift: TShiftState);
begin
  case Key of
    VK_UP: FUpDownButton.Click(btNext);
    VK_DOWN: FUpDownButton.Click(btPrev);
    VK_SPACE: exit;
  end;
end;

procedure TSpinButton.BtnMouseDown (Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft)
  then begin
    if (FFocusControl <> nil)
       and FFocusControl.TabStop
       and FFocusControl.CanFocus
       and (GetFocus <> FFocusControl.Handle)
    then FFocusControl.SetFocus
    else if TabStop
            and (GetFocus <> Handle)
            and CanFocus
         then SetFocus;
  end;
end;

procedure TSpinButton.BtnClick(Sender: TObject; Button: TUDBtnType);
begin
  if (Button = btNext)
  then begin
    if Assigned(FOnUpClick)
    then FOnUpClick(Self);
  end
  else if Assigned(FOnDownClick)
       then FOnDownClick(Self);
end;

procedure TSpinButton.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TSpinButton.Loaded;
var W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width)
     or (H <> Height)
  then inherited SetBounds (Left, Top, W, H);
end;

{ TnSpinEdit }

constructor TnSpinEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FButton := TSpinButton.Create(Self);
  FButton.Width := 17;
  FButton.Height := 22;
  FButton.Visible := True;
  FButton.Parent := Self;
  FButton.FocusControl := Self;
  FButton.OnUpClick := UpClick;
  FButton.OnDownClick := DownClick;
  Text := '0';
  ControlStyle := ControlStyle - [csSetCaption];
  FEditorEnabled := True;
  ParentBackground := False;
end;

destructor TnSpinEdit.Destroy;
begin
  FButton := nil;
  inherited Destroy;
end;

procedure TnSpinEdit.GetChildren(Proc: TGetChildProc; Root: TComponent);
begin
end;

procedure TnSpinEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_UP)
  then UpClick(Self)
  else if (Key = VK_DOWN)
       then DownClick (Self);
  inherited KeyDown(Key, Shift);
end;

procedure TnSpinEdit.KeyPress(var Key: Char);
begin
  if (Key = Chr(VK_RETURN))
  then begin
    if Assigned(OnKeyPress)
    then OnKeyPress(Self, Key);
    Key := #0;
    Exit;
  end;
  if not IsValidChar(Key)
  then begin
    Key := #0;
    MessageBeep(0)
  end;
  if (Key <> #0)
  then inherited KeyPress(Key);
end;

function TnSpinEdit.IsValidChar(Key: Char): Boolean;
begin
  Result := CharInSet(Key, [FormatSettings.DecimalSeparator, '+', '-', '0'..'9'])
    or ((Key < #32) and (Key <> Chr(VK_RETURN)));

  if (not FEditorEnabled)
     and Result
     and ((Key >= #32) or (Key = Chr(VK_BACK)) or (Key = Chr(VK_DELETE)))
  then Result := False;
end;

procedure TnSpinEdit.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.Style := Params.Style or WS_CLIPCHILDREN;
end;

procedure TnSpinEdit.CreateWnd;
begin
  inherited CreateWnd;
  SetEditRect;
end;

procedure TnSpinEdit.SetEditRect;
var
  Loc: TRect;
begin
  SendMessage(Handle, EM_GETRECT, 0, LongInt(@Loc));
  Loc.Bottom := ClientHeight + 1;  {+1 is workaround for windows paint bug}
  Loc.Right := ClientWidth - FButton.Width - 2;
  Loc.Top := 0;
  Loc.Left := 0;
  SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@Loc));
end;

procedure TnSpinEdit.WMSize(var Message: TWMSize);
begin
  inherited;
  if (FButton <> nil)
  then begin
    if NewStyleControls
       and Ctl3D
    then FButton.SetBounds(Width - FButton.Width - 3, -1, FButton.Width, Height - 2)
    else FButton.SetBounds (Width - FButton.Width, 1, FButton.Width, Height - 3);
    SetEditRect;
  end;
end;

procedure TnSpinEdit.UpClick (Sender: TObject);
begin
  if ReadOnly
  then MessageBeep(0)
  else Value := Value + Increment;
end;

procedure TnSpinEdit.DownClick (Sender: TObject);
begin
  if ReadOnly
  then MessageBeep(0)
  else Value := Value - Increment;
end;

procedure TnSpinEdit.WMPaste(var Message: TWMPaste);
begin
  if (not FEditorEnabled)
     or ReadOnly
  then Exit;
  inherited;
end;

procedure TnSpinEdit.WMCut(var Message: TWMPaste);
begin
  if (not FEditorEnabled)
     or ReadOnly
  then Exit;
  inherited;
end;

procedure TnSpinEdit.CMExit(var Message: TCMExit);
begin
  inherited;
  if (CheckValue(Value) <> Value)
  then SetValue(Value);
end;

function TnSpinEdit.GetValue: LongInt;
begin
  try
    Result := StrToInt(Text);
  except
    Result := FMinValue;
  end;
end;

procedure TnSpinEdit.SetValue (NewValue: LongInt);
begin
  Text := IntToStr(CheckValue (NewValue));
end;

function TnSpinEdit.CheckValue (NewValue: LongInt): LongInt;
begin
  Result := NewValue;
  if (FMaxValue <> FMinValue)
  then begin
    if (NewValue < FMinValue)
    then Result := FMinValue
    else if (NewValue > FMaxValue)
         then Result := FMaxValue;
  end;
  FButton.FUpDownButton.Position := Result;
end;

procedure TnSpinEdit.SetMinValue (NewMinValue: LongInt);
begin
  FButton.FUpDownButton.Min := NewMinValue;
  FMinValue := NewMinValue;
end;

procedure TnSpinEdit.SetMaxValue (NewMaxValue: LongInt);
begin
  FButton.FUpDownButton.Max := NewMaxValue;
  FMaxValue := NewMaxValue;
end;

procedure TnSpinEdit.CMEnter(var Message: TCMGotFocus);
begin
  if AutoSelect
     and (not (csLButtonDown in ControlState))
  then SelectAll;
  inherited;
end;

function TnSpinEdit.GetIncrement: LongInt;
begin
  Result := FButton.FUpDownButton.Increment;
end;

procedure TnSpinEdit.SetIncrement(NewValue: LongInt);
begin
  FButton.FUpDownButton.Increment := NewValue;
end;

{TTimerSpeedButton}

destructor TTimerSpeedButton.Destroy;
begin
  if (FRepeatTimer <> nil)
  then FRepeatTimer.Free;
  inherited Destroy;
end;

procedure TTimerSpeedButton.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  if (tbAllowTimer in FTimeBtnState)
  then begin
    if (FRepeatTimer = nil)
    then FRepeatTimer := TTimer.Create(Self);
    FRepeatTimer.OnTimer := TimerExpired;
    FRepeatTimer.Interval := InitRepeatPause;
    FRepeatTimer.Enabled  := True;
  end;
end;

procedure TTimerSpeedButton.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  if (FRepeatTimer <> nil)
  then FRepeatTimer.Enabled  := False;
end;

procedure TTimerSpeedButton.TimerExpired(Sender: TObject);
begin
end;

end.
