/*
 * Copyright (c) 2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include <cassert>
#include <cstdint>

namespace arm_gemm {

void sve_hybrid_s8qs_dot_6x4VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int col_base
)
{
    struct KernelArgs {
        const int32_t *multiplier_ptr = {};
        const int32_t *shift_ptr = {};
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->per_channel_requant) {
        flags |= 0x10;
        ka.multiplier_ptr=qp->per_channel_muls + col_base;
        ka.shift_ptr=qp->per_channel_right_shifts + col_base;
    }
    __asm__ __volatile__(
      "ptrue p2.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 61f\n"
      "cmp %x[M], #0x4\n"
      "bgt 49f\n"
      "beq 37f\n"
      "cmp %x[M], #0x2\n"
      "bgt 25f\n"
      "beq 13f\n"
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov x28, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "ble 8f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "bgt 7b\n"
      "8:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "ble 9f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "ble 9f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "ble 9f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "9:"  // Height 1: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 4b\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "add z8.s, z8.s, z0.s\n"
      "add z9.s, z9.s, z1.s\n"
      "add z10.s, z10.s, z2.s\n"
      "add z11.s, z11.s, z3.s\n"
      "tbz %x[flags], #4, 10f\n"
      "ld1w { z0.s }, p2/Z, [x13]\n"
      "ld1w { z4.s }, p2/Z, [x14]\n"
      "ld1w { z1.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x13, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "addvl x14, x14, #4\n"
      "b 11f\n"
      "10:"  // Height 1: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "11:"  // Height 1: parameters loaded
      ".inst 0x04a47108  // sqdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57129  // sqdmulh z9.s, z9.s, z5.s\n"
      "add x21, %x[qp], %[c_offset]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0x04a6714a  // sqdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7716b  // sqdmulh z11.s, z11.s, z7.s\n"
      "ld1rw { z4.s }, p2/Z, [x21]\n"
      "ld1rw { z6.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[minval]\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      "add z8.s, z8.s, z4.s\n"
      "add z9.s, z9.s, z4.s\n"
      "add z10.s, z10.s, z4.s\n"
      "add z11.s, z11.s, z4.s\n"
      "smin z8.s, p2/M, z8.s, z6.s\n"
      "smin z9.s, p2/M, z9.s, z6.s\n"
      "smin z10.s, p2/M, z10.s, z6.s\n"
      "smin z11.s, p2/M, z11.s, z6.s\n"
      "smax z8.s, p2/M, z8.s, z5.s\n"
      "smax z9.s, p2/M, z9.s, z5.s\n"
      "smax z10.s, p2/M, z10.s, z5.s\n"
      "smax z11.s, p2/M, z11.s, z5.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "uzp1 z9.h, z10.h, z11.h\n"
      "uzp1 z8.b, z8.b, z9.b\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 2b\n"
      "b 74f\n"
      "13:"  // Height 2
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "14:"  // Height 2: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov x28, #0x0\n"
      "16:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 17f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 18f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 18f\n"
      "17:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "18:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "ble 20f\n"
      "19:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "bgt 19b\n"
      "20:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "ble 21f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "ble 21f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "ble 21f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "21:"  // Height 2: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 16b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "add x27, x9, x20\n"
      "add z8.s, z8.s, z0.s\n"
      "add z12.s, z12.s, z0.s\n"
      "add z9.s, z9.s, z1.s\n"
      "add z10.s, z10.s, z2.s\n"
      "add z13.s, z13.s, z1.s\n"
      "add z11.s, z11.s, z3.s\n"
      "add z14.s, z14.s, z2.s\n"
      "add z15.s, z15.s, z3.s\n"
      "tbz %x[flags], #4, 22f\n"
      "ld1w { z0.s }, p2/Z, [x13]\n"
      "ld1w { z4.s }, p2/Z, [x14]\n"
      "ld1w { z1.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x13, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "addvl x14, x14, #4\n"
      "b 23f\n"
      "22:"  // Height 2: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "23:"  // Height 2: parameters loaded
      ".inst 0x04a47108  // sqdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57129  // sqdmulh z9.s, z9.s, z5.s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      ".inst 0x04a6714a  // sqdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7716b  // sqdmulh z11.s, z11.s, z7.s\n"
      "add x20, %x[qp], %[minval]\n"
      ".inst 0x04a4718c  // sqdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a571ad  // sqdmulh z13.s, z13.s, z5.s\n"
      "ld1rw { z4.s }, p2/Z, [x22]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      ".inst 0x04a671ce  // sqdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a771ef  // sqdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      "add z8.s, z8.s, z4.s\n"
      "add z9.s, z9.s, z4.s\n"
      "add z10.s, z10.s, z4.s\n"
      "add z11.s, z11.s, z4.s\n"
      "add z12.s, z12.s, z4.s\n"
      "add z13.s, z13.s, z4.s\n"
      "add z14.s, z14.s, z4.s\n"
      "add z15.s, z15.s, z4.s\n"
      "smin z8.s, p2/M, z8.s, z6.s\n"
      "smin z9.s, p2/M, z9.s, z6.s\n"
      "smin z10.s, p2/M, z10.s, z6.s\n"
      "smin z11.s, p2/M, z11.s, z6.s\n"
      "smin z12.s, p2/M, z12.s, z6.s\n"
      "smin z13.s, p2/M, z13.s, z6.s\n"
      "smin z14.s, p2/M, z14.s, z6.s\n"
      "smin z15.s, p2/M, z15.s, z6.s\n"
      "smax z8.s, p2/M, z8.s, z5.s\n"
      "smax z9.s, p2/M, z9.s, z5.s\n"
      "smax z10.s, p2/M, z10.s, z5.s\n"
      "smax z11.s, p2/M, z11.s, z5.s\n"
      "smax z12.s, p2/M, z12.s, z5.s\n"
      "smax z13.s, p2/M, z13.s, z5.s\n"
      "smax z14.s, p2/M, z14.s, z5.s\n"
      "smax z15.s, p2/M, z15.s, z5.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "uzp1 z9.h, z10.h, z11.h\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z13.h, z14.h, z15.h\n"
      "uzp1 z8.b, z8.b, z9.b\n"
      "uzp1 z12.b, z12.b, z13.b\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "st1b { z12.b }, p1, [x27]\n"
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 14b\n"
      "b 74f\n"
      "25:"  // Height 3
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "26:"  // Height 3: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov x28, #0x0\n"
      "28:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 29f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 30f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 30f\n"
      "29:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "30:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "ble 32f\n"
      "31:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "sdot z16.s, z6.b, z2.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z7.b, z2.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z2.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z2.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "sdot z16.s, z6.b, z2.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "sdot z17.s, z7.b, z2.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z18.s, z6.b, z2.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "sdot z19.s, z7.b, z2.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "sdot z16.s, z6.b, z2.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "sdot z17.s, z7.b, z2.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z6.b, z2.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z7.b, z2.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "sdot z16.s, z6.b, z2.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "sdot z17.s, z7.b, z2.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z18.s, z6.b, z2.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "sdot z19.s, z7.b, z2.b[3]\n"
      "bgt 31b\n"
      "32:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "sdot z16.s, z6.b, z2.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z7.b, z2.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z2.b[0]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z2.b[0]\n"
      "ble 33f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "sdot z16.s, z6.b, z2.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "sdot z17.s, z7.b, z2.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z18.s, z6.b, z2.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "sdot z19.s, z7.b, z2.b[1]\n"
      "ble 33f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "sdot z16.s, z6.b, z2.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "sdot z17.s, z7.b, z2.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z6.b, z2.b[2]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z7.b, z2.b[2]\n"
      "ble 33f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "sdot z16.s, z6.b, z2.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "sdot z17.s, z7.b, z2.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z18.s, z6.b, z2.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "sdot z19.s, z7.b, z2.b[3]\n"
      "33:"  // Height 3: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 28b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "add z8.s, z8.s, z0.s\n"
      "add z12.s, z12.s, z0.s\n"
      "add z9.s, z9.s, z1.s\n"
      "add z10.s, z10.s, z2.s\n"
      "add z11.s, z11.s, z3.s\n"
      "add z13.s, z13.s, z1.s\n"
      "add z14.s, z14.s, z2.s\n"
      "add z15.s, z15.s, z3.s\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z1.s\n"
      "add z18.s, z18.s, z2.s\n"
      "add z19.s, z19.s, z3.s\n"
      "tbz %x[flags], #4, 34f\n"
      "ld1w { z0.s }, p2/Z, [x13]\n"
      "ld1w { z4.s }, p2/Z, [x14]\n"
      "ld1w { z1.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x13, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "addvl x14, x14, #4\n"
      "b 35f\n"
      "34:"  // Height 3: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "35:"  // Height 3: parameters loaded
      ".inst 0x04a47108  // sqdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57129  // sqdmulh z9.s, z9.s, z5.s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      ".inst 0x04a6714a  // sqdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7716b  // sqdmulh z11.s, z11.s, z7.s\n"
      "add x20, %x[qp], %[minval]\n"
      ".inst 0x04a4718c  // sqdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a571ad  // sqdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a671ce  // sqdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a771ef  // sqdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x04a47210  // sqdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a57231  // sqdmulh z17.s, z17.s, z5.s\n"
      "ld1rw { z4.s }, p2/Z, [x22]\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x04a67252  // sqdmulh z18.s, z18.s, z6.s\n"
      ".inst 0x04a77273  // sqdmulh z19.s, z19.s, z7.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z8.s, z8.s, z4.s\n"
      "add z9.s, z9.s, z4.s\n"
      ".inst 0x44828831  // srshl z17.s, p2/M, z17.s, z1.s\n"
      ".inst 0x44828852  // srshl z18.s, p2/M, z18.s, z2.s\n"
      "add z10.s, z10.s, z4.s\n"
      "add z11.s, z11.s, z4.s\n"
      ".inst 0x44828873  // srshl z19.s, p2/M, z19.s, z3.s\n"
      "add z12.s, z12.s, z4.s\n"
      "add z13.s, z13.s, z4.s\n"
      "add z14.s, z14.s, z4.s\n"
      "add z15.s, z15.s, z4.s\n"
      "smin z8.s, p2/M, z8.s, z6.s\n"
      "smin z9.s, p2/M, z9.s, z6.s\n"
      "add z16.s, z16.s, z4.s\n"
      "add z17.s, z17.s, z4.s\n"
      "smin z10.s, p2/M, z10.s, z6.s\n"
      "smin z11.s, p2/M, z11.s, z6.s\n"
      "add z18.s, z18.s, z4.s\n"
      "add z19.s, z19.s, z4.s\n"
      "smin z12.s, p2/M, z12.s, z6.s\n"
      "smin z13.s, p2/M, z13.s, z6.s\n"
      "smin z14.s, p2/M, z14.s, z6.s\n"
      "smin z15.s, p2/M, z15.s, z6.s\n"
      "smin z16.s, p2/M, z16.s, z6.s\n"
      "smin z17.s, p2/M, z17.s, z6.s\n"
      "smin z18.s, p2/M, z18.s, z6.s\n"
      "smin z19.s, p2/M, z19.s, z6.s\n"
      "smax z8.s, p2/M, z8.s, z5.s\n"
      "smax z9.s, p2/M, z9.s, z5.s\n"
      "smax z10.s, p2/M, z10.s, z5.s\n"
      "smax z11.s, p2/M, z11.s, z5.s\n"
      "smax z12.s, p2/M, z12.s, z5.s\n"
      "smax z13.s, p2/M, z13.s, z5.s\n"
      "smax z14.s, p2/M, z14.s, z5.s\n"
      "smax z15.s, p2/M, z15.s, z5.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "smax z16.s, p2/M, z16.s, z5.s\n"
      "smax z17.s, p2/M, z17.s, z5.s\n"
      "uzp1 z9.h, z10.h, z11.h\n"
      "smax z18.s, p2/M, z18.s, z5.s\n"
      "smax z19.s, p2/M, z19.s, z5.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z13.h, z14.h, z15.h\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z8.b, z8.b, z9.b\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z12.b, z12.b, z13.b\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "st1b { z12.b }, p1, [x27]\n"
      "st1b { z16.b }, p1, [x26]\n"
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 26b\n"
      "b 74f\n"
      "37:"  // Height 4
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "38:"  // Height 4: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov x28, #0x0\n"
      "40:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 41f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 42f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 42f\n"
      "41:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "42:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "ble 44f\n"
      "43:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "sdot z16.s, z6.b, z2.b[0]\n"
      "sdot z20.s, z6.b, z3.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z7.b, z2.b[0]\n"
      "sdot z21.s, z7.b, z3.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z2.b[0]\n"
      "sdot z22.s, z6.b, z3.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z2.b[0]\n"
      "sdot z23.s, z7.b, z3.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "sdot z16.s, z6.b, z2.b[1]\n"
      "sdot z20.s, z6.b, z3.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "sdot z17.s, z7.b, z2.b[1]\n"
      "sdot z21.s, z7.b, z3.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z18.s, z6.b, z2.b[1]\n"
      "sdot z22.s, z6.b, z3.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "sdot z19.s, z7.b, z2.b[1]\n"
      "sdot z23.s, z7.b, z3.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "sdot z16.s, z6.b, z2.b[2]\n"
      "sdot z20.s, z6.b, z3.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "sdot z17.s, z7.b, z2.b[2]\n"
      "sdot z21.s, z7.b, z3.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z6.b, z2.b[2]\n"
      "sdot z22.s, z6.b, z3.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z7.b, z2.b[2]\n"
      "sdot z23.s, z7.b, z3.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "sdot z16.s, z6.b, z2.b[3]\n"
      "sdot z20.s, z6.b, z3.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "sdot z17.s, z7.b, z2.b[3]\n"
      "sdot z21.s, z7.b, z3.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z18.s, z6.b, z2.b[3]\n"
      "sdot z22.s, z6.b, z3.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "sdot z19.s, z7.b, z2.b[3]\n"
      "sdot z23.s, z7.b, z3.b[3]\n"
      "bgt 43b\n"
      "44:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "sdot z16.s, z6.b, z2.b[0]\n"
      "sdot z20.s, z6.b, z3.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z7.b, z2.b[0]\n"
      "sdot z21.s, z7.b, z3.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z2.b[0]\n"
      "sdot z22.s, z6.b, z3.b[0]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z2.b[0]\n"
      "sdot z23.s, z7.b, z3.b[0]\n"
      "ble 45f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "sdot z16.s, z6.b, z2.b[1]\n"
      "sdot z20.s, z6.b, z3.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "sdot z17.s, z7.b, z2.b[1]\n"
      "sdot z21.s, z7.b, z3.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z18.s, z6.b, z2.b[1]\n"
      "sdot z22.s, z6.b, z3.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "sdot z19.s, z7.b, z2.b[1]\n"
      "sdot z23.s, z7.b, z3.b[1]\n"
      "ble 45f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "sdot z16.s, z6.b, z2.b[2]\n"
      "sdot z20.s, z6.b, z3.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "sdot z17.s, z7.b, z2.b[2]\n"
      "sdot z21.s, z7.b, z3.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z6.b, z2.b[2]\n"
      "sdot z22.s, z6.b, z3.b[2]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z7.b, z2.b[2]\n"
      "sdot z23.s, z7.b, z3.b[2]\n"
      "ble 45f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "sdot z16.s, z6.b, z2.b[3]\n"
      "sdot z20.s, z6.b, z3.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "sdot z17.s, z7.b, z2.b[3]\n"
      "sdot z21.s, z7.b, z3.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z18.s, z6.b, z2.b[3]\n"
      "sdot z22.s, z6.b, z3.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "sdot z19.s, z7.b, z2.b[3]\n"
      "sdot z23.s, z7.b, z3.b[3]\n"
      "45:"  // Height 4: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 40b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "add z8.s, z8.s, z0.s\n"
      "add z12.s, z12.s, z0.s\n"
      "add x25, x26, x20\n"
      "add z9.s, z9.s, z1.s\n"
      "add z10.s, z10.s, z2.s\n"
      "add z11.s, z11.s, z3.s\n"
      "add z13.s, z13.s, z1.s\n"
      "add z14.s, z14.s, z2.s\n"
      "add z15.s, z15.s, z3.s\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z1.s\n"
      "add z18.s, z18.s, z2.s\n"
      "add z19.s, z19.s, z3.s\n"
      "add z20.s, z20.s, z0.s\n"
      "add z21.s, z21.s, z1.s\n"
      "add z22.s, z22.s, z2.s\n"
      "add z23.s, z23.s, z3.s\n"
      "tbz %x[flags], #4, 46f\n"
      "ld1w { z0.s }, p2/Z, [x13]\n"
      "ld1w { z4.s }, p2/Z, [x14]\n"
      "ld1w { z1.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x13, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "addvl x14, x14, #4\n"
      "b 47f\n"
      "46:"  // Height 4: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "47:"  // Height 4: parameters loaded
      ".inst 0x04a47108  // sqdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57129  // sqdmulh z9.s, z9.s, z5.s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      ".inst 0x04a6714a  // sqdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7716b  // sqdmulh z11.s, z11.s, z7.s\n"
      "add x20, %x[qp], %[minval]\n"
      ".inst 0x04a4718c  // sqdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a571ad  // sqdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a671ce  // sqdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a771ef  // sqdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x04a47210  // sqdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a57231  // sqdmulh z17.s, z17.s, z5.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x04a67252  // sqdmulh z18.s, z18.s, z6.s\n"
      ".inst 0x04a77273  // sqdmulh z19.s, z19.s, z7.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x04a47294  // sqdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a572b5  // sqdmulh z21.s, z21.s, z5.s\n"
      "ld1rw { z4.s }, p2/Z, [x22]\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      ".inst 0x04a672d6  // sqdmulh z22.s, z22.s, z6.s\n"
      ".inst 0x04a772f7  // sqdmulh z23.s, z23.s, z7.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      ".inst 0x44828831  // srshl z17.s, p2/M, z17.s, z1.s\n"
      ".inst 0x44828852  // srshl z18.s, p2/M, z18.s, z2.s\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      ".inst 0x44828873  // srshl z19.s, p2/M, z19.s, z3.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      "add z8.s, z8.s, z4.s\n"
      "add z9.s, z9.s, z4.s\n"
      ".inst 0x44828835  // srshl z21.s, p2/M, z21.s, z1.s\n"
      ".inst 0x44828856  // srshl z22.s, p2/M, z22.s, z2.s\n"
      "add z10.s, z10.s, z4.s\n"
      "add z11.s, z11.s, z4.s\n"
      ".inst 0x44828877  // srshl z23.s, p2/M, z23.s, z3.s\n"
      "add z12.s, z12.s, z4.s\n"
      "add z13.s, z13.s, z4.s\n"
      "add z14.s, z14.s, z4.s\n"
      "add z15.s, z15.s, z4.s\n"
      "smin z8.s, p2/M, z8.s, z6.s\n"
      "smin z9.s, p2/M, z9.s, z6.s\n"
      "add z16.s, z16.s, z4.s\n"
      "add z17.s, z17.s, z4.s\n"
      "smin z10.s, p2/M, z10.s, z6.s\n"
      "smin z11.s, p2/M, z11.s, z6.s\n"
      "add z18.s, z18.s, z4.s\n"
      "add z19.s, z19.s, z4.s\n"
      "smin z12.s, p2/M, z12.s, z6.s\n"
      "smin z13.s, p2/M, z13.s, z6.s\n"
      "add z20.s, z20.s, z4.s\n"
      "add z21.s, z21.s, z4.s\n"
      "smin z14.s, p2/M, z14.s, z6.s\n"
      "smin z15.s, p2/M, z15.s, z6.s\n"
      "add z22.s, z22.s, z4.s\n"
      "add z23.s, z23.s, z4.s\n"
      "smin z16.s, p2/M, z16.s, z6.s\n"
      "smin z17.s, p2/M, z17.s, z6.s\n"
      "smin z18.s, p2/M, z18.s, z6.s\n"
      "smin z19.s, p2/M, z19.s, z6.s\n"
      "smin z20.s, p2/M, z20.s, z6.s\n"
      "smin z21.s, p2/M, z21.s, z6.s\n"
      "smin z22.s, p2/M, z22.s, z6.s\n"
      "smin z23.s, p2/M, z23.s, z6.s\n"
      "smax z8.s, p2/M, z8.s, z5.s\n"
      "smax z9.s, p2/M, z9.s, z5.s\n"
      "smax z10.s, p2/M, z10.s, z5.s\n"
      "smax z11.s, p2/M, z11.s, z5.s\n"
      "smax z12.s, p2/M, z12.s, z5.s\n"
      "smax z13.s, p2/M, z13.s, z5.s\n"
      "smax z14.s, p2/M, z14.s, z5.s\n"
      "smax z15.s, p2/M, z15.s, z5.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "smax z16.s, p2/M, z16.s, z5.s\n"
      "smax z17.s, p2/M, z17.s, z5.s\n"
      "uzp1 z9.h, z10.h, z11.h\n"
      "smax z18.s, p2/M, z18.s, z5.s\n"
      "smax z19.s, p2/M, z19.s, z5.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "smax z20.s, p2/M, z20.s, z5.s\n"
      "smax z21.s, p2/M, z21.s, z5.s\n"
      "uzp1 z13.h, z14.h, z15.h\n"
      "smax z22.s, p2/M, z22.s, z5.s\n"
      "smax z23.s, p2/M, z23.s, z5.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z8.b, z8.b, z9.b\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z12.b, z12.b, z13.b\n"
      "uzp1 z21.h, z22.h, z23.h\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "uzp1 z20.b, z20.b, z21.b\n"
      "st1b { z12.b }, p1, [x27]\n"
      "st1b { z16.b }, p1, [x26]\n"
      "st1b { z20.b }, p1, [x25]\n"
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 38b\n"
      "b 74f\n"
      "49:"  // Height 5
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "50:"  // Height 5: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "mov x28, #0x0\n"
      "52:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 53f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 54f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 54f\n"
      "53:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "54:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "ble 56f\n"
      "55:"  // Height 5: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "ld1rqb { z4.b }, p0/Z, [x22]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z16.s, z6.b, z2.b[0]\n"
      "sdot z20.s, z6.b, z3.b[0]\n"
      "sdot z24.s, z6.b, z4.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z7.b, z2.b[0]\n"
      "sdot z21.s, z7.b, z3.b[0]\n"
      "sdot z25.s, z7.b, z4.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z2.b[0]\n"
      "sdot z22.s, z6.b, z3.b[0]\n"
      "sdot z26.s, z6.b, z4.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z2.b[0]\n"
      "sdot z23.s, z7.b, z3.b[0]\n"
      "sdot z27.s, z7.b, z4.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "sdot z16.s, z6.b, z2.b[1]\n"
      "sdot z20.s, z6.b, z3.b[1]\n"
      "sdot z24.s, z6.b, z4.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "sdot z17.s, z7.b, z2.b[1]\n"
      "sdot z21.s, z7.b, z3.b[1]\n"
      "sdot z25.s, z7.b, z4.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z18.s, z6.b, z2.b[1]\n"
      "sdot z22.s, z6.b, z3.b[1]\n"
      "sdot z26.s, z6.b, z4.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "sdot z19.s, z7.b, z2.b[1]\n"
      "sdot z23.s, z7.b, z3.b[1]\n"
      "sdot z27.s, z7.b, z4.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "sdot z16.s, z6.b, z2.b[2]\n"
      "sdot z20.s, z6.b, z3.b[2]\n"
      "sdot z24.s, z6.b, z4.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "sdot z17.s, z7.b, z2.b[2]\n"
      "sdot z21.s, z7.b, z3.b[2]\n"
      "sdot z25.s, z7.b, z4.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z6.b, z2.b[2]\n"
      "sdot z22.s, z6.b, z3.b[2]\n"
      "sdot z26.s, z6.b, z4.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z7.b, z2.b[2]\n"
      "sdot z23.s, z7.b, z3.b[2]\n"
      "sdot z27.s, z7.b, z4.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "sdot z16.s, z6.b, z2.b[3]\n"
      "sdot z20.s, z6.b, z3.b[3]\n"
      "sdot z24.s, z6.b, z4.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "sdot z17.s, z7.b, z2.b[3]\n"
      "sdot z21.s, z7.b, z3.b[3]\n"
      "sdot z25.s, z7.b, z4.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z18.s, z6.b, z2.b[3]\n"
      "sdot z22.s, z6.b, z3.b[3]\n"
      "sdot z26.s, z6.b, z4.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "sdot z19.s, z7.b, z2.b[3]\n"
      "sdot z23.s, z7.b, z3.b[3]\n"
      "sdot z27.s, z7.b, z4.b[3]\n"
      "bgt 55b\n"
      "56:"  // Height 5: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "ld1rqb { z4.b }, p0/Z, [x22]\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "sdot z16.s, z6.b, z2.b[0]\n"
      "sdot z20.s, z6.b, z3.b[0]\n"
      "sdot z24.s, z6.b, z4.b[0]\n"
      "sdot z17.s, z7.b, z2.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z21.s, z7.b, z3.b[0]\n"
      "sdot z25.s, z7.b, z4.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z2.b[0]\n"
      "sdot z22.s, z6.b, z3.b[0]\n"
      "sdot z26.s, z6.b, z4.b[0]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z2.b[0]\n"
      "sdot z23.s, z7.b, z3.b[0]\n"
      "sdot z27.s, z7.b, z4.b[0]\n"
      "ble 57f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "sdot z16.s, z6.b, z2.b[1]\n"
      "sdot z20.s, z6.b, z3.b[1]\n"
      "sdot z24.s, z6.b, z4.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "sdot z17.s, z7.b, z2.b[1]\n"
      "sdot z21.s, z7.b, z3.b[1]\n"
      "sdot z25.s, z7.b, z4.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "addvl x10, x10, #4\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z18.s, z6.b, z2.b[1]\n"
      "sdot z22.s, z6.b, z3.b[1]\n"
      "sdot z26.s, z6.b, z4.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "sdot z19.s, z7.b, z2.b[1]\n"
      "sdot z23.s, z7.b, z3.b[1]\n"
      "sdot z27.s, z7.b, z4.b[1]\n"
      "ble 57f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "sdot z16.s, z6.b, z2.b[2]\n"
      "sdot z20.s, z6.b, z3.b[2]\n"
      "sdot z24.s, z6.b, z4.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "sdot z17.s, z7.b, z2.b[2]\n"
      "sdot z21.s, z7.b, z3.b[2]\n"
      "sdot z25.s, z7.b, z4.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "addvl x10, x10, #4\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z6.b, z2.b[2]\n"
      "sdot z22.s, z6.b, z3.b[2]\n"
      "sdot z26.s, z6.b, z4.b[2]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z7.b, z2.b[2]\n"
      "sdot z23.s, z7.b, z3.b[2]\n"
      "sdot z27.s, z7.b, z4.b[2]\n"
      "ble 57f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "sdot z16.s, z6.b, z2.b[3]\n"
      "sdot z20.s, z6.b, z3.b[3]\n"
      "sdot z24.s, z6.b, z4.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "sdot z17.s, z7.b, z2.b[3]\n"
      "sdot z21.s, z7.b, z3.b[3]\n"
      "sdot z25.s, z7.b, z4.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z18.s, z6.b, z2.b[3]\n"
      "sdot z22.s, z6.b, z3.b[3]\n"
      "sdot z26.s, z6.b, z4.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "sdot z19.s, z7.b, z2.b[3]\n"
      "sdot z23.s, z7.b, z3.b[3]\n"
      "sdot z27.s, z7.b, z4.b[3]\n"
      "57:"  // Height 5: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 52b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "add z8.s, z8.s, z0.s\n"
      "add z12.s, z12.s, z0.s\n"
      "add x25, x26, x20\n"
      "add x24, x25, x20\n"
      "add z9.s, z9.s, z1.s\n"
      "add z10.s, z10.s, z2.s\n"
      "add z11.s, z11.s, z3.s\n"
      "add z13.s, z13.s, z1.s\n"
      "add z14.s, z14.s, z2.s\n"
      "add z15.s, z15.s, z3.s\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z1.s\n"
      "add z18.s, z18.s, z2.s\n"
      "add z19.s, z19.s, z3.s\n"
      "add z20.s, z20.s, z0.s\n"
      "add z21.s, z21.s, z1.s\n"
      "add z22.s, z22.s, z2.s\n"
      "add z23.s, z23.s, z3.s\n"
      "add z24.s, z24.s, z0.s\n"
      "add z25.s, z25.s, z1.s\n"
      "add z26.s, z26.s, z2.s\n"
      "add z27.s, z27.s, z3.s\n"
      "tbz %x[flags], #4, 58f\n"
      "ld1w { z0.s }, p2/Z, [x13]\n"
      "ld1w { z4.s }, p2/Z, [x14]\n"
      "ld1w { z1.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x13, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "addvl x14, x14, #4\n"
      "b 59f\n"
      "58:"  // Height 5: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "59:"  // Height 5: parameters loaded
      ".inst 0x04a47108  // sqdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57129  // sqdmulh z9.s, z9.s, z5.s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      ".inst 0x04a6714a  // sqdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7716b  // sqdmulh z11.s, z11.s, z7.s\n"
      "add x20, %x[qp], %[minval]\n"
      ".inst 0x04a4718c  // sqdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a571ad  // sqdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a671ce  // sqdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a771ef  // sqdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x04a47210  // sqdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a57231  // sqdmulh z17.s, z17.s, z5.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x04a67252  // sqdmulh z18.s, z18.s, z6.s\n"
      ".inst 0x04a77273  // sqdmulh z19.s, z19.s, z7.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x04a47294  // sqdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a572b5  // sqdmulh z21.s, z21.s, z5.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      ".inst 0x04a672d6  // sqdmulh z22.s, z22.s, z6.s\n"
      ".inst 0x04a772f7  // sqdmulh z23.s, z23.s, z7.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      ".inst 0x44828831  // srshl z17.s, p2/M, z17.s, z1.s\n"
      ".inst 0x04a47318  // sqdmulh z24.s, z24.s, z4.s\n"
      ".inst 0x04a57339  // sqdmulh z25.s, z25.s, z5.s\n"
      "ld1rw { z4.s }, p2/Z, [x22]\n"
      ".inst 0x44828852  // srshl z18.s, p2/M, z18.s, z2.s\n"
      ".inst 0x04a6735a  // sqdmulh z26.s, z26.s, z6.s\n"
      ".inst 0x04a7737b  // sqdmulh z27.s, z27.s, z7.s\n"
      ".inst 0x44828873  // srshl z19.s, p2/M, z19.s, z3.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      ".inst 0x44828835  // srshl z21.s, p2/M, z21.s, z1.s\n"
      ".inst 0x44828856  // srshl z22.s, p2/M, z22.s, z2.s\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      ".inst 0x44828877  // srshl z23.s, p2/M, z23.s, z3.s\n"
      ".inst 0x44828818  // srshl z24.s, p2/M, z24.s, z0.s\n"
      "add z8.s, z8.s, z4.s\n"
      "add z9.s, z9.s, z4.s\n"
      ".inst 0x44828839  // srshl z25.s, p2/M, z25.s, z1.s\n"
      ".inst 0x4482885a  // srshl z26.s, p2/M, z26.s, z2.s\n"
      "add z10.s, z10.s, z4.s\n"
      "add z11.s, z11.s, z4.s\n"
      ".inst 0x4482887b  // srshl z27.s, p2/M, z27.s, z3.s\n"
      "add z12.s, z12.s, z4.s\n"
      "add z13.s, z13.s, z4.s\n"
      "add z14.s, z14.s, z4.s\n"
      "add z15.s, z15.s, z4.s\n"
      "smin z8.s, p2/M, z8.s, z6.s\n"
      "smin z9.s, p2/M, z9.s, z6.s\n"
      "add z16.s, z16.s, z4.s\n"
      "add z17.s, z17.s, z4.s\n"
      "smin z10.s, p2/M, z10.s, z6.s\n"
      "smin z11.s, p2/M, z11.s, z6.s\n"
      "add z18.s, z18.s, z4.s\n"
      "add z19.s, z19.s, z4.s\n"
      "smin z12.s, p2/M, z12.s, z6.s\n"
      "smin z13.s, p2/M, z13.s, z6.s\n"
      "add z20.s, z20.s, z4.s\n"
      "add z21.s, z21.s, z4.s\n"
      "smin z14.s, p2/M, z14.s, z6.s\n"
      "smin z15.s, p2/M, z15.s, z6.s\n"
      "add z22.s, z22.s, z4.s\n"
      "add z23.s, z23.s, z4.s\n"
      "smin z16.s, p2/M, z16.s, z6.s\n"
      "smin z17.s, p2/M, z17.s, z6.s\n"
      "add z24.s, z24.s, z4.s\n"
      "add z25.s, z25.s, z4.s\n"
      "smin z18.s, p2/M, z18.s, z6.s\n"
      "smin z19.s, p2/M, z19.s, z6.s\n"
      "add z26.s, z26.s, z4.s\n"
      "add z27.s, z27.s, z4.s\n"
      "smin z20.s, p2/M, z20.s, z6.s\n"
      "smin z21.s, p2/M, z21.s, z6.s\n"
      "smin z22.s, p2/M, z22.s, z6.s\n"
      "smin z23.s, p2/M, z23.s, z6.s\n"
      "smin z24.s, p2/M, z24.s, z6.s\n"
      "smin z25.s, p2/M, z25.s, z6.s\n"
      "smin z26.s, p2/M, z26.s, z6.s\n"
      "smin z27.s, p2/M, z27.s, z6.s\n"
      "smax z8.s, p2/M, z8.s, z5.s\n"
      "smax z9.s, p2/M, z9.s, z5.s\n"
      "smax z10.s, p2/M, z10.s, z5.s\n"
      "smax z11.s, p2/M, z11.s, z5.s\n"
      "smax z12.s, p2/M, z12.s, z5.s\n"
      "smax z13.s, p2/M, z13.s, z5.s\n"
      "smax z14.s, p2/M, z14.s, z5.s\n"
      "smax z15.s, p2/M, z15.s, z5.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "smax z16.s, p2/M, z16.s, z5.s\n"
      "smax z17.s, p2/M, z17.s, z5.s\n"
      "uzp1 z9.h, z10.h, z11.h\n"
      "smax z18.s, p2/M, z18.s, z5.s\n"
      "smax z19.s, p2/M, z19.s, z5.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "smax z20.s, p2/M, z20.s, z5.s\n"
      "smax z21.s, p2/M, z21.s, z5.s\n"
      "uzp1 z13.h, z14.h, z15.h\n"
      "smax z22.s, p2/M, z22.s, z5.s\n"
      "smax z23.s, p2/M, z23.s, z5.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z8.b, z8.b, z9.b\n"
      "smax z24.s, p2/M, z24.s, z5.s\n"
      "smax z25.s, p2/M, z25.s, z5.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "smax z26.s, p2/M, z26.s, z5.s\n"
      "smax z27.s, p2/M, z27.s, z5.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z12.b, z12.b, z13.b\n"
      "uzp1 z21.h, z22.h, z23.h\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "uzp1 z25.h, z26.h, z27.h\n"
      "st1b { z12.b }, p1, [x27]\n"
      "uzp1 z20.b, z20.b, z21.b\n"
      "uzp1 z24.b, z24.b, z25.b\n"
      "st1b { z16.b }, p1, [x26]\n"
      "st1b { z20.b }, p1, [x25]\n"
      "st1b { z24.b }, p1, [x24]\n"
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 50b\n"
      "b 74f\n"
      "61:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x6\n"
      "mov x12, %x[col_bias]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "62:"  // Height 6: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "mov z28.s, #0x0\n"
      "mov z29.s, #0x0\n"
      "mov z30.s, #0x0\n"
      "mov z31.s, #0x0\n"
      "mov x28, #0x0\n"
      "64:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 65f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 66f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 66f\n"
      "65:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "66:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "ble 68f\n"
      "67:"  // Height 6: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "ld1rqb { z4.b }, p0/Z, [x22]\n"
      "ld1rqb { z5.b }, p0/Z, [x21]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "add x21, x21, #0x10\n"
      "sdot z16.s, z6.b, z2.b[0]\n"
      "sdot z20.s, z6.b, z3.b[0]\n"
      "sdot z24.s, z6.b, z4.b[0]\n"
      "sdot z28.s, z6.b, z5.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "sdot z17.s, z7.b, z2.b[0]\n"
      "sdot z21.s, z7.b, z3.b[0]\n"
      "sdot z25.s, z7.b, z4.b[0]\n"
      "sdot z29.s, z7.b, z5.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z2.b[0]\n"
      "sdot z22.s, z6.b, z3.b[0]\n"
      "sdot z26.s, z6.b, z4.b[0]\n"
      "sdot z30.s, z6.b, z5.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z2.b[0]\n"
      "sdot z23.s, z7.b, z3.b[0]\n"
      "sdot z27.s, z7.b, z4.b[0]\n"
      "sdot z31.s, z7.b, z5.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "sdot z16.s, z6.b, z2.b[1]\n"
      "sdot z20.s, z6.b, z3.b[1]\n"
      "sdot z24.s, z6.b, z4.b[1]\n"
      "sdot z28.s, z6.b, z5.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "sdot z17.s, z7.b, z2.b[1]\n"
      "sdot z21.s, z7.b, z3.b[1]\n"
      "sdot z25.s, z7.b, z4.b[1]\n"
      "sdot z29.s, z7.b, z5.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z18.s, z6.b, z2.b[1]\n"
      "sdot z22.s, z6.b, z3.b[1]\n"
      "sdot z26.s, z6.b, z4.b[1]\n"
      "sdot z30.s, z6.b, z5.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "sdot z19.s, z7.b, z2.b[1]\n"
      "sdot z23.s, z7.b, z3.b[1]\n"
      "sdot z27.s, z7.b, z4.b[1]\n"
      "sdot z31.s, z7.b, z5.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "sdot z16.s, z6.b, z2.b[2]\n"
      "sdot z20.s, z6.b, z3.b[2]\n"
      "sdot z24.s, z6.b, z4.b[2]\n"
      "sdot z28.s, z6.b, z5.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "sdot z17.s, z7.b, z2.b[2]\n"
      "sdot z21.s, z7.b, z3.b[2]\n"
      "sdot z25.s, z7.b, z4.b[2]\n"
      "sdot z29.s, z7.b, z5.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z6.b, z2.b[2]\n"
      "sdot z22.s, z6.b, z3.b[2]\n"
      "sdot z26.s, z6.b, z4.b[2]\n"
      "sdot z30.s, z6.b, z5.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z7.b, z2.b[2]\n"
      "sdot z23.s, z7.b, z3.b[2]\n"
      "sdot z27.s, z7.b, z4.b[2]\n"
      "sdot z31.s, z7.b, z5.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "sdot z16.s, z6.b, z2.b[3]\n"
      "sdot z20.s, z6.b, z3.b[3]\n"
      "sdot z24.s, z6.b, z4.b[3]\n"
      "sdot z28.s, z6.b, z5.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "sdot z17.s, z7.b, z2.b[3]\n"
      "sdot z21.s, z7.b, z3.b[3]\n"
      "sdot z25.s, z7.b, z4.b[3]\n"
      "sdot z29.s, z7.b, z5.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z18.s, z6.b, z2.b[3]\n"
      "sdot z22.s, z6.b, z3.b[3]\n"
      "sdot z26.s, z6.b, z4.b[3]\n"
      "sdot z30.s, z6.b, z5.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "sdot z19.s, z7.b, z2.b[3]\n"
      "sdot z23.s, z7.b, z3.b[3]\n"
      "sdot z27.s, z7.b, z4.b[3]\n"
      "sdot z31.s, z7.b, z5.b[3]\n"
      "bgt 67b\n"
      "68:"  // Height 6: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "ld1rqb { z4.b }, p0/Z, [x22]\n"
      "ld1rqb { z5.b }, p0/Z, [x21]\n"
      "sdot z8.s, z6.b, z0.b[0]\n"
      "sdot z12.s, z6.b, z1.b[0]\n"
      "sdot z9.s, z7.b, z0.b[0]\n"
      "sdot z13.s, z7.b, z1.b[0]\n"
      "sdot z16.s, z6.b, z2.b[0]\n"
      "sdot z20.s, z6.b, z3.b[0]\n"
      "sdot z24.s, z6.b, z4.b[0]\n"
      "sdot z28.s, z6.b, z5.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z7.b, z2.b[0]\n"
      "sdot z21.s, z7.b, z3.b[0]\n"
      "sdot z25.s, z7.b, z4.b[0]\n"
      "sdot z29.s, z7.b, z5.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[0]\n"
      "sdot z14.s, z6.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z2.b[0]\n"
      "sdot z22.s, z6.b, z3.b[0]\n"
      "sdot z26.s, z6.b, z4.b[0]\n"
      "sdot z30.s, z6.b, z5.b[0]\n"
      "sdot z11.s, z7.b, z0.b[0]\n"
      "sdot z15.s, z7.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z2.b[0]\n"
      "sdot z23.s, z7.b, z3.b[0]\n"
      "sdot z27.s, z7.b, z4.b[0]\n"
      "sdot z31.s, z7.b, z5.b[0]\n"
      "ble 69f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[1]\n"
      "sdot z12.s, z6.b, z1.b[1]\n"
      "sdot z16.s, z6.b, z2.b[1]\n"
      "sdot z20.s, z6.b, z3.b[1]\n"
      "sdot z24.s, z6.b, z4.b[1]\n"
      "sdot z28.s, z6.b, z5.b[1]\n"
      "sdot z9.s, z7.b, z0.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z13.s, z7.b, z1.b[1]\n"
      "sdot z17.s, z7.b, z2.b[1]\n"
      "sdot z21.s, z7.b, z3.b[1]\n"
      "sdot z25.s, z7.b, z4.b[1]\n"
      "sdot z29.s, z7.b, z5.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[1]\n"
      "addvl x10, x10, #4\n"
      "sdot z14.s, z6.b, z1.b[1]\n"
      "sdot z18.s, z6.b, z2.b[1]\n"
      "sdot z22.s, z6.b, z3.b[1]\n"
      "sdot z26.s, z6.b, z4.b[1]\n"
      "sdot z30.s, z6.b, z5.b[1]\n"
      "sdot z11.s, z7.b, z0.b[1]\n"
      "sdot z15.s, z7.b, z1.b[1]\n"
      "sdot z19.s, z7.b, z2.b[1]\n"
      "sdot z23.s, z7.b, z3.b[1]\n"
      "sdot z27.s, z7.b, z4.b[1]\n"
      "sdot z31.s, z7.b, z5.b[1]\n"
      "ble 69f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z6.b, z0.b[2]\n"
      "sdot z12.s, z6.b, z1.b[2]\n"
      "sdot z16.s, z6.b, z2.b[2]\n"
      "sdot z20.s, z6.b, z3.b[2]\n"
      "sdot z24.s, z6.b, z4.b[2]\n"
      "sdot z28.s, z6.b, z5.b[2]\n"
      "sdot z9.s, z7.b, z0.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z13.s, z7.b, z1.b[2]\n"
      "sdot z17.s, z7.b, z2.b[2]\n"
      "sdot z21.s, z7.b, z3.b[2]\n"
      "sdot z25.s, z7.b, z4.b[2]\n"
      "sdot z29.s, z7.b, z5.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z6.b, z0.b[2]\n"
      "addvl x10, x10, #4\n"
      "sdot z14.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z6.b, z2.b[2]\n"
      "sdot z22.s, z6.b, z3.b[2]\n"
      "sdot z26.s, z6.b, z4.b[2]\n"
      "sdot z30.s, z6.b, z5.b[2]\n"
      "sdot z11.s, z7.b, z0.b[2]\n"
      "sdot z15.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z7.b, z2.b[2]\n"
      "sdot z23.s, z7.b, z3.b[2]\n"
      "sdot z27.s, z7.b, z4.b[2]\n"
      "sdot z31.s, z7.b, z5.b[2]\n"
      "ble 69f\n"
      "ld1b { z6.b }, p2/Z, [x10]\n"
      "ld1b { z7.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z6.b, z0.b[3]\n"
      "sdot z12.s, z6.b, z1.b[3]\n"
      "sdot z16.s, z6.b, z2.b[3]\n"
      "sdot z20.s, z6.b, z3.b[3]\n"
      "sdot z24.s, z6.b, z4.b[3]\n"
      "sdot z28.s, z6.b, z5.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z7.b, z0.b[3]\n"
      "sdot z13.s, z7.b, z1.b[3]\n"
      "sdot z17.s, z7.b, z2.b[3]\n"
      "sdot z21.s, z7.b, z3.b[3]\n"
      "sdot z25.s, z7.b, z4.b[3]\n"
      "sdot z29.s, z7.b, z5.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z6.b, z0.b[3]\n"
      "sdot z14.s, z6.b, z1.b[3]\n"
      "sdot z18.s, z6.b, z2.b[3]\n"
      "sdot z22.s, z6.b, z3.b[3]\n"
      "sdot z26.s, z6.b, z4.b[3]\n"
      "sdot z30.s, z6.b, z5.b[3]\n"
      "sdot z11.s, z7.b, z0.b[3]\n"
      "sdot z15.s, z7.b, z1.b[3]\n"
      "sdot z19.s, z7.b, z2.b[3]\n"
      "sdot z23.s, z7.b, z3.b[3]\n"
      "sdot z27.s, z7.b, z4.b[3]\n"
      "sdot z31.s, z7.b, z5.b[3]\n"
      "69:"  // Height 6: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 64b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "add z8.s, z8.s, z0.s\n"
      "add z12.s, z12.s, z0.s\n"
      "add x25, x26, x20\n"
      "add x24, x25, x20\n"
      "add z9.s, z9.s, z1.s\n"
      "add z10.s, z10.s, z2.s\n"
      "add x23, x24, x20\n"
      "add z11.s, z11.s, z3.s\n"
      "add z13.s, z13.s, z1.s\n"
      "add z14.s, z14.s, z2.s\n"
      "add z15.s, z15.s, z3.s\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z1.s\n"
      "add z18.s, z18.s, z2.s\n"
      "add z19.s, z19.s, z3.s\n"
      "add z20.s, z20.s, z0.s\n"
      "add z21.s, z21.s, z1.s\n"
      "add z22.s, z22.s, z2.s\n"
      "add z23.s, z23.s, z3.s\n"
      "add z24.s, z24.s, z0.s\n"
      "add z25.s, z25.s, z1.s\n"
      "add z26.s, z26.s, z2.s\n"
      "add z27.s, z27.s, z3.s\n"
      "add z28.s, z28.s, z0.s\n"
      "add z29.s, z29.s, z1.s\n"
      "add z30.s, z30.s, z2.s\n"
      "add z31.s, z31.s, z3.s\n"
      "tbz %x[flags], #4, 70f\n"
      "ld1w { z0.s }, p2/Z, [x13]\n"
      "ld1w { z4.s }, p2/Z, [x14]\n"
      "ld1w { z1.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x13, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "addvl x14, x14, #4\n"
      "b 71f\n"
      "70:"  // Height 6: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "71:"  // Height 6: parameters loaded
      ".inst 0x04a47108  // sqdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57129  // sqdmulh z9.s, z9.s, z5.s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      ".inst 0x04a6714a  // sqdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7716b  // sqdmulh z11.s, z11.s, z7.s\n"
      "add x20, %x[qp], %[minval]\n"
      ".inst 0x04a4718c  // sqdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a571ad  // sqdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a671ce  // sqdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a771ef  // sqdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x04a47210  // sqdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a57231  // sqdmulh z17.s, z17.s, z5.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x04a67252  // sqdmulh z18.s, z18.s, z6.s\n"
      ".inst 0x04a77273  // sqdmulh z19.s, z19.s, z7.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x04a47294  // sqdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a572b5  // sqdmulh z21.s, z21.s, z5.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      ".inst 0x04a672d6  // sqdmulh z22.s, z22.s, z6.s\n"
      ".inst 0x04a772f7  // sqdmulh z23.s, z23.s, z7.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      ".inst 0x44828831  // srshl z17.s, p2/M, z17.s, z1.s\n"
      ".inst 0x04a47318  // sqdmulh z24.s, z24.s, z4.s\n"
      ".inst 0x04a57339  // sqdmulh z25.s, z25.s, z5.s\n"
      ".inst 0x44828852  // srshl z18.s, p2/M, z18.s, z2.s\n"
      ".inst 0x44828873  // srshl z19.s, p2/M, z19.s, z3.s\n"
      ".inst 0x04a6735a  // sqdmulh z26.s, z26.s, z6.s\n"
      ".inst 0x04a7737b  // sqdmulh z27.s, z27.s, z7.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      ".inst 0x44828835  // srshl z21.s, p2/M, z21.s, z1.s\n"
      ".inst 0x04a4739c  // sqdmulh z28.s, z28.s, z4.s\n"
      ".inst 0x04a573bd  // sqdmulh z29.s, z29.s, z5.s\n"
      "ld1rw { z4.s }, p2/Z, [x22]\n"
      ".inst 0x44828856  // srshl z22.s, p2/M, z22.s, z2.s\n"
      ".inst 0x04a673de  // sqdmulh z30.s, z30.s, z6.s\n"
      ".inst 0x04a773ff  // sqdmulh z31.s, z31.s, z7.s\n"
      ".inst 0x44828877  // srshl z23.s, p2/M, z23.s, z3.s\n"
      ".inst 0x44828818  // srshl z24.s, p2/M, z24.s, z0.s\n"
      ".inst 0x44828839  // srshl z25.s, p2/M, z25.s, z1.s\n"
      ".inst 0x4482885a  // srshl z26.s, p2/M, z26.s, z2.s\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      ".inst 0x4482887b  // srshl z27.s, p2/M, z27.s, z3.s\n"
      ".inst 0x4482881c  // srshl z28.s, p2/M, z28.s, z0.s\n"
      "add z8.s, z8.s, z4.s\n"
      "add z9.s, z9.s, z4.s\n"
      ".inst 0x4482883d  // srshl z29.s, p2/M, z29.s, z1.s\n"
      ".inst 0x4482885e  // srshl z30.s, p2/M, z30.s, z2.s\n"
      "add z10.s, z10.s, z4.s\n"
      "add z11.s, z11.s, z4.s\n"
      ".inst 0x4482887f  // srshl z31.s, p2/M, z31.s, z3.s\n"
      "add z12.s, z12.s, z4.s\n"
      "add z13.s, z13.s, z4.s\n"
      "add z14.s, z14.s, z4.s\n"
      "add z15.s, z15.s, z4.s\n"
      "smin z8.s, p2/M, z8.s, z6.s\n"
      "smin z9.s, p2/M, z9.s, z6.s\n"
      "add z16.s, z16.s, z4.s\n"
      "add z17.s, z17.s, z4.s\n"
      "smin z10.s, p2/M, z10.s, z6.s\n"
      "smin z11.s, p2/M, z11.s, z6.s\n"
      "add z18.s, z18.s, z4.s\n"
      "add z19.s, z19.s, z4.s\n"
      "smin z12.s, p2/M, z12.s, z6.s\n"
      "smin z13.s, p2/M, z13.s, z6.s\n"
      "add z20.s, z20.s, z4.s\n"
      "add z21.s, z21.s, z4.s\n"
      "smin z14.s, p2/M, z14.s, z6.s\n"
      "smin z15.s, p2/M, z15.s, z6.s\n"
      "add z22.s, z22.s, z4.s\n"
      "add z23.s, z23.s, z4.s\n"
      "smin z16.s, p2/M, z16.s, z6.s\n"
      "smin z17.s, p2/M, z17.s, z6.s\n"
      "add z24.s, z24.s, z4.s\n"
      "add z25.s, z25.s, z4.s\n"
      "smin z18.s, p2/M, z18.s, z6.s\n"
      "smin z19.s, p2/M, z19.s, z6.s\n"
      "add z26.s, z26.s, z4.s\n"
      "add z27.s, z27.s, z4.s\n"
      "smin z20.s, p2/M, z20.s, z6.s\n"
      "smin z21.s, p2/M, z21.s, z6.s\n"
      "add z28.s, z28.s, z4.s\n"
      "add z29.s, z29.s, z4.s\n"
      "smin z22.s, p2/M, z22.s, z6.s\n"
      "smin z23.s, p2/M, z23.s, z6.s\n"
      "add z30.s, z30.s, z4.s\n"
      "add z31.s, z31.s, z4.s\n"
      "smin z24.s, p2/M, z24.s, z6.s\n"
      "smin z25.s, p2/M, z25.s, z6.s\n"
      "smin z26.s, p2/M, z26.s, z6.s\n"
      "smin z27.s, p2/M, z27.s, z6.s\n"
      "smin z28.s, p2/M, z28.s, z6.s\n"
      "smin z29.s, p2/M, z29.s, z6.s\n"
      "smin z30.s, p2/M, z30.s, z6.s\n"
      "smin z31.s, p2/M, z31.s, z6.s\n"
      "smax z8.s, p2/M, z8.s, z5.s\n"
      "smax z9.s, p2/M, z9.s, z5.s\n"
      "smax z10.s, p2/M, z10.s, z5.s\n"
      "smax z11.s, p2/M, z11.s, z5.s\n"
      "smax z12.s, p2/M, z12.s, z5.s\n"
      "smax z13.s, p2/M, z13.s, z5.s\n"
      "smax z14.s, p2/M, z14.s, z5.s\n"
      "smax z15.s, p2/M, z15.s, z5.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "smax z16.s, p2/M, z16.s, z5.s\n"
      "smax z17.s, p2/M, z17.s, z5.s\n"
      "uzp1 z9.h, z10.h, z11.h\n"
      "smax z18.s, p2/M, z18.s, z5.s\n"
      "smax z19.s, p2/M, z19.s, z5.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "smax z20.s, p2/M, z20.s, z5.s\n"
      "smax z21.s, p2/M, z21.s, z5.s\n"
      "uzp1 z13.h, z14.h, z15.h\n"
      "smax z22.s, p2/M, z22.s, z5.s\n"
      "smax z23.s, p2/M, z23.s, z5.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z8.b, z8.b, z9.b\n"
      "smax z24.s, p2/M, z24.s, z5.s\n"
      "smax z25.s, p2/M, z25.s, z5.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "smax z26.s, p2/M, z26.s, z5.s\n"
      "smax z27.s, p2/M, z27.s, z5.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z12.b, z12.b, z13.b\n"
      "smax z28.s, p2/M, z28.s, z5.s\n"
      "smax z29.s, p2/M, z29.s, z5.s\n"
      "uzp1 z21.h, z22.h, z23.h\n"
      "st1b { z8.b }, p1, [x9]\n"
      "smax z30.s, p2/M, z30.s, z5.s\n"
      "smax z31.s, p2/M, z31.s, z5.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "uzp1 z25.h, z26.h, z27.h\n"
      "st1b { z12.b }, p1, [x27]\n"
      "addvl x9, x9, #1\n"
      "uzp1 z28.h, z28.h, z29.h\n"
      "uzp1 z20.b, z20.b, z21.b\n"
      "uzp1 z29.h, z30.h, z31.h\n"
      "st1b { z16.b }, p1, [x26]\n"
      "uzp1 z24.b, z24.b, z25.b\n"
      "uzp1 z28.b, z28.b, z29.b\n"
      "st1b { z20.b }, p1, [x25]\n"
      "st1b { z24.b }, p1, [x24]\n"
      "st1b { z28.b }, p1, [x23]\n"
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 62b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 74f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 73f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "73:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "74:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [flags] "r" (flags), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_multiplier_ptr] "I" (offsetof(KernelArgs, multiplier_ptr)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_shift_ptr] "I" (offsetof(KernelArgs, shift_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "p0", "p1", "p2", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
