/*
 * Copyright (c) 2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include <cassert>
#include <cstdint>

namespace arm_gemm {

void sve_hybrid_s8qa_dot_4x4VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    __asm__ __volatile__(
      "ptrue p2.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 40f\n"
      "cmp %x[M], #0x2\n"
      "bgt 27f\n"
      "beq 14f\n"
      "mov z11.s, #0x0\n"
      "mov z15.b, #0x1\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x28, %x[col_bias]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "whilelt p1.b, x20, x10\n"
      "mov x26, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "cbnz x26, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x24, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x25, #0x10\n"
      "ble 9f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x25\n"
      "ld1b { z4.b }, p2/Z, [x9]\n"
      "ld1b { z5.b }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1b { z6.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #3, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #4, MUL VL]\n"
      "ld1b { z9.b }, p2/Z, [x9, #5, MUL VL]\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1b { z10.b }, p2/Z, [x9, #6, MUL VL]\n"
      "add x24, x24, #0x10\n"
      "sdot z16.s, z4.b, z0.b[0]\n"
      "ld1b { z4.b }, p2/Z, [x9, #7, MUL VL]\n"
      "addvl x9, x9, #16\n"
      "sdot z17.s, z5.b, z0.b[0]\n"
      "sdot z18.s, z6.b, z0.b[0]\n"
      "sdot z19.s, z7.b, z0.b[0]\n"
      "sdot z16.s, z8.b, z0.b[1]\n"
      "ld1b { z5.b }, p2/Z, [x9, #-8, MUL VL]\n"
      "ld1b { z6.b }, p2/Z, [x9, #-7, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #-6, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #-5, MUL VL]\n"
      "sdot z17.s, z9.b, z0.b[1]\n"
      "ld1b { z9.b }, p2/Z, [x9, #-4, MUL VL]\n"
      "sdot z18.s, z10.b, z0.b[1]\n"
      "ld1b { z10.b }, p2/Z, [x9, #-3, MUL VL]\n"
      "sdot z19.s, z4.b, z0.b[1]\n"
      "ld1b { z4.b }, p2/Z, [x9, #-2, MUL VL]\n"
      "sdot z16.s, z5.b, z0.b[2]\n"
      "ld1b { z5.b }, p2/Z, [x9, #-1, MUL VL]\n"
      "sdot z17.s, z6.b, z0.b[2]\n"
      "sdot z18.s, z7.b, z0.b[2]\n"
      "sdot z19.s, z8.b, z0.b[2]\n"
      "sdot z16.s, z9.b, z0.b[3]\n"
      "sdot z17.s, z10.b, z0.b[3]\n"
      "sdot z18.s, z4.b, z0.b[3]\n"
      "sdot z19.s, z5.b, z0.b[3]\n"
      "tbnz %x[flags], #31, 8f\n"
      "sdot z11.s, z0.b, z15.b\n"
      "8:"  // Height 1: Multiply loop: unique 1: skip row sum
      "sub x25, x25, #0x10\n"
      "cmp x25, #0x10\n"
      "bgt 7b\n"
      "9:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x25\n"
      "ld1b { z4.b }, p2/Z, [x9]\n"
      "ld1b { z5.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z6.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "sdot z16.s, z4.b, z0.b[0]\n"
      "sdot z17.s, z5.b, z0.b[0]\n"
      "sdot z18.s, z6.b, z0.b[0]\n"
      "sdot z19.s, z7.b, z0.b[0]\n"
      "ble 10f\n"
      "ld1b { z8.b }, p2/Z, [x9]\n"
      "ld1b { z9.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z10.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z8.b, z0.b[1]\n"
      "sdot z17.s, z9.b, z0.b[1]\n"
      "sdot z18.s, z10.b, z0.b[1]\n"
      "sdot z19.s, z4.b, z0.b[1]\n"
      "ble 10f\n"
      "ld1b { z5.b }, p2/Z, [x9]\n"
      "ld1b { z6.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z7.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z5.b, z0.b[2]\n"
      "sdot z17.s, z6.b, z0.b[2]\n"
      "sdot z18.s, z7.b, z0.b[2]\n"
      "sdot z19.s, z8.b, z0.b[2]\n"
      "ble 10f\n"
      "ld1b { z9.b }, p2/Z, [x9]\n"
      "ld1b { z10.b }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z5.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z9.b, z0.b[3]\n"
      "sdot z17.s, z10.b, z0.b[3]\n"
      "sdot z18.s, z4.b, z0.b[3]\n"
      "sdot z19.s, z5.b, z0.b[3]\n"
      "10:"  // Height 1: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 11f\n"
      "sdot z11.s, z0.b, z15.b\n"
      "11:"  // Height 1: Multiply loop: unique 2: skip row sum
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 4b\n"
      "tbnz %x[flags], #31, 12f\n"
      "mov x21, #0x4\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1rw { z1.s }, p2/Z, [x20]\n"
      "whilelt p0.s, XZR, x21\n"
      "neg z1.s, p2/M, z1.s\n"
      "saddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "mul z11.s, p2/M, z11.s, z1.s\n"
      "12:"  // Height 1: skip row sum fixup
      "add z16.s, z16.s, z11.s\n"
      "add z17.s, z17.s, z11.s\n"
      "ld1w { z0.s }, p2/Z, [x28]\n"
      "ld1w { z1.s }, p2/Z, [x28, #1, MUL VL]\n"
      "add z18.s, z18.s, z11.s\n"
      "add z19.s, z19.s, z11.s\n"
      "ld1w { z2.s }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x28, #3, MUL VL]\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z1.s\n"
      "ld1rw { z4.s }, p2/Z, [x21]\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      "add x21, %x[qp], %[c_offset]\n"
      "add x20, %x[qp], %[maxval]\n"
      "add z18.s, z18.s, z2.s\n"
      "add z19.s, z19.s, z3.s\n"
      "ld1rw { z6.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "addvl x28, x28, #4\n"
      ".inst 0x04a47210  // sqdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a47231  // sqdmulh z17.s, z17.s, z4.s\n"
      ".inst 0x04a47252  // sqdmulh z18.s, z18.s, z4.s\n"
      ".inst 0x04a47273  // sqdmulh z19.s, z19.s, z4.s\n"
      "ld1rw { z4.s }, p2/Z, [x21]\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      ".inst 0x44828811  // srshl z17.s, p2/M, z17.s, z0.s\n"
      ".inst 0x44828812  // srshl z18.s, p2/M, z18.s, z0.s\n"
      ".inst 0x44828813  // srshl z19.s, p2/M, z19.s, z0.s\n"
      "add z16.s, z16.s, z4.s\n"
      "add z17.s, z17.s, z4.s\n"
      "add z18.s, z18.s, z4.s\n"
      "add z19.s, z19.s, z4.s\n"
      "smin z16.s, p2/M, z16.s, z6.s\n"
      "smin z17.s, p2/M, z17.s, z6.s\n"
      "smin z18.s, p2/M, z18.s, z6.s\n"
      "smin z19.s, p2/M, z19.s, z6.s\n"
      "smax z16.s, p2/M, z16.s, z5.s\n"
      "smax z17.s, p2/M, z17.s, z5.s\n"
      "smax z18.s, p2/M, z18.s, z5.s\n"
      "smax z19.s, p2/M, z19.s, z5.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "st1b { z16.b }, p1, [x27]\n"
      "addvl x27, x27, #1\n"
      "decw x10, ALL, MUL #4\n"
      "cmp x10, XZR\n"
      "bgt 2b\n"
      "b 54f\n"
      "14:"  // Height 2
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "mov z15.b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x28, %x[col_bias]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "15:"  // Height 2: Column loop
      "mov x20, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "whilelt p1.b, x20, x10\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov x26, #0x0\n"
      "17:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 18f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "cbnz x26, 19f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 19f\n"
      "18:"  // Height 2: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "19:"  // Height 2: input setup done
      "cmp x25, #0x10\n"
      "ble 22f\n"
      "20:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x25\n"
      "ld1b { z4.b }, p2/Z, [x9]\n"
      "ld1b { z5.b }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1b { z6.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #3, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #4, MUL VL]\n"
      "ld1b { z9.b }, p2/Z, [x9, #5, MUL VL]\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "ld1b { z10.b }, p2/Z, [x9, #6, MUL VL]\n"
      "sdot z16.s, z4.b, z0.b[0]\n"
      "sdot z20.s, z4.b, z1.b[0]\n"
      "ld1b { z4.b }, p2/Z, [x9, #7, MUL VL]\n"
      "addvl x9, x9, #16\n"
      "sdot z17.s, z5.b, z0.b[0]\n"
      "sdot z21.s, z5.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z0.b[0]\n"
      "sdot z22.s, z6.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z0.b[0]\n"
      "sdot z23.s, z7.b, z1.b[0]\n"
      "sdot z16.s, z8.b, z0.b[1]\n"
      "sdot z20.s, z8.b, z1.b[1]\n"
      "ld1b { z5.b }, p2/Z, [x9, #-8, MUL VL]\n"
      "ld1b { z6.b }, p2/Z, [x9, #-7, MUL VL]\n"
      "sdot z17.s, z9.b, z0.b[1]\n"
      "sdot z21.s, z9.b, z1.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x9, #-6, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #-5, MUL VL]\n"
      "sdot z18.s, z10.b, z0.b[1]\n"
      "sdot z22.s, z10.b, z1.b[1]\n"
      "ld1b { z9.b }, p2/Z, [x9, #-4, MUL VL]\n"
      "ld1b { z10.b }, p2/Z, [x9, #-3, MUL VL]\n"
      "sdot z19.s, z4.b, z0.b[1]\n"
      "sdot z23.s, z4.b, z1.b[1]\n"
      "ld1b { z4.b }, p2/Z, [x9, #-2, MUL VL]\n"
      "sdot z16.s, z5.b, z0.b[2]\n"
      "sdot z20.s, z5.b, z1.b[2]\n"
      "ld1b { z5.b }, p2/Z, [x9, #-1, MUL VL]\n"
      "sdot z17.s, z6.b, z0.b[2]\n"
      "sdot z21.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z7.b, z0.b[2]\n"
      "sdot z22.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z8.b, z0.b[2]\n"
      "sdot z23.s, z8.b, z1.b[2]\n"
      "sdot z16.s, z9.b, z0.b[3]\n"
      "sdot z20.s, z9.b, z1.b[3]\n"
      "sdot z17.s, z10.b, z0.b[3]\n"
      "sdot z21.s, z10.b, z1.b[3]\n"
      "sdot z18.s, z4.b, z0.b[3]\n"
      "sdot z22.s, z4.b, z1.b[3]\n"
      "sdot z19.s, z5.b, z0.b[3]\n"
      "sdot z23.s, z5.b, z1.b[3]\n"
      "tbnz %x[flags], #31, 21f\n"
      "sdot z11.s, z0.b, z15.b\n"
      "sdot z12.s, z1.b, z15.b\n"
      "21:"  // Height 2: Multiply loop: unique 3: skip row sum
      "sub x25, x25, #0x10\n"
      "cmp x25, #0x10\n"
      "bgt 20b\n"
      "22:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x25\n"
      "ld1b { z4.b }, p2/Z, [x9]\n"
      "ld1b { z5.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z6.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "sdot z16.s, z4.b, z0.b[0]\n"
      "sdot z20.s, z4.b, z1.b[0]\n"
      "sdot z17.s, z5.b, z0.b[0]\n"
      "sdot z21.s, z5.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z0.b[0]\n"
      "sdot z22.s, z6.b, z1.b[0]\n"
      "sdot z19.s, z7.b, z0.b[0]\n"
      "sdot z23.s, z7.b, z1.b[0]\n"
      "ble 23f\n"
      "ld1b { z8.b }, p2/Z, [x9]\n"
      "ld1b { z9.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z10.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z8.b, z0.b[1]\n"
      "sdot z20.s, z8.b, z1.b[1]\n"
      "sdot z17.s, z9.b, z0.b[1]\n"
      "sdot z21.s, z9.b, z1.b[1]\n"
      "sdot z18.s, z10.b, z0.b[1]\n"
      "sdot z22.s, z10.b, z1.b[1]\n"
      "sdot z19.s, z4.b, z0.b[1]\n"
      "sdot z23.s, z4.b, z1.b[1]\n"
      "ble 23f\n"
      "ld1b { z5.b }, p2/Z, [x9]\n"
      "ld1b { z6.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z7.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z5.b, z0.b[2]\n"
      "sdot z20.s, z5.b, z1.b[2]\n"
      "sdot z17.s, z6.b, z0.b[2]\n"
      "sdot z21.s, z6.b, z1.b[2]\n"
      "sdot z18.s, z7.b, z0.b[2]\n"
      "sdot z22.s, z7.b, z1.b[2]\n"
      "sdot z19.s, z8.b, z0.b[2]\n"
      "sdot z23.s, z8.b, z1.b[2]\n"
      "ble 23f\n"
      "ld1b { z9.b }, p2/Z, [x9]\n"
      "ld1b { z10.b }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z5.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z9.b, z0.b[3]\n"
      "sdot z20.s, z9.b, z1.b[3]\n"
      "sdot z17.s, z10.b, z0.b[3]\n"
      "sdot z21.s, z10.b, z1.b[3]\n"
      "sdot z18.s, z4.b, z0.b[3]\n"
      "sdot z22.s, z4.b, z1.b[3]\n"
      "sdot z19.s, z5.b, z0.b[3]\n"
      "sdot z23.s, z5.b, z1.b[3]\n"
      "23:"  // Height 2: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 24f\n"
      "sdot z11.s, z0.b, z15.b\n"
      "sdot z12.s, z1.b, z15.b\n"
      "24:"  // Height 2: Multiply loop: unique 4: skip row sum
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 17b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x27, x20\n"
      "tbnz %x[flags], #31, 25f\n"
      "mov x21, #0x4\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1rw { z2.s }, p2/Z, [x20]\n"
      "whilelt p0.s, XZR, x21\n"
      "neg z2.s, p2/M, z2.s\n"
      "saddv d11, p0, z11.s\n"
      "saddv d12, p0, z12.s\n"
      "mov z11.s, z11.s[0]\n"
      "mul z11.s, p2/M, z11.s, z2.s\n"
      "mov z12.s, z12.s[0]\n"
      "mul z12.s, p2/M, z12.s, z2.s\n"
      "25:"  // Height 2: skip row sum fixup
      "add z16.s, z16.s, z11.s\n"
      "add z17.s, z17.s, z11.s\n"
      "ld1w { z0.s }, p2/Z, [x28]\n"
      "ld1w { z1.s }, p2/Z, [x28, #1, MUL VL]\n"
      "add z18.s, z18.s, z11.s\n"
      "add z19.s, z19.s, z11.s\n"
      "ld1w { z2.s }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x28, #3, MUL VL]\n"
      "add z20.s, z20.s, z12.s\n"
      "add z21.s, z21.s, z12.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "add x23, %x[qp], %[per_layer_right_shift]\n"
      "add z22.s, z22.s, z12.s\n"
      "add z23.s, z23.s, z12.s\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z1.s\n"
      "add x21, %x[qp], %[maxval]\n"
      "add x20, %x[qp], %[minval]\n"
      "add z18.s, z18.s, z2.s\n"
      "add z19.s, z19.s, z3.s\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      "add z20.s, z20.s, z0.s\n"
      "add z21.s, z21.s, z1.s\n"
      "ld1rw { z0.s }, p2/Z, [x23]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "add z22.s, z22.s, z2.s\n"
      "add z23.s, z23.s, z3.s\n"
      "addvl x28, x28, #4\n"
      ".inst 0x04a47210  // sqdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a47231  // sqdmulh z17.s, z17.s, z4.s\n"
      ".inst 0x04a47252  // sqdmulh z18.s, z18.s, z4.s\n"
      ".inst 0x04a47273  // sqdmulh z19.s, z19.s, z4.s\n"
      ".inst 0x04a47294  // sqdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a472b5  // sqdmulh z21.s, z21.s, z4.s\n"
      ".inst 0x04a472d6  // sqdmulh z22.s, z22.s, z4.s\n"
      ".inst 0x04a472f7  // sqdmulh z23.s, z23.s, z4.s\n"
      "ld1rw { z4.s }, p2/Z, [x22]\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      ".inst 0x44828811  // srshl z17.s, p2/M, z17.s, z0.s\n"
      ".inst 0x44828812  // srshl z18.s, p2/M, z18.s, z0.s\n"
      ".inst 0x44828813  // srshl z19.s, p2/M, z19.s, z0.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      ".inst 0x44828815  // srshl z21.s, p2/M, z21.s, z0.s\n"
      ".inst 0x44828816  // srshl z22.s, p2/M, z22.s, z0.s\n"
      "add z16.s, z16.s, z4.s\n"
      ".inst 0x44828817  // srshl z23.s, p2/M, z23.s, z0.s\n"
      "add z17.s, z17.s, z4.s\n"
      "add z18.s, z18.s, z4.s\n"
      "add z19.s, z19.s, z4.s\n"
      "add z20.s, z20.s, z4.s\n"
      "add z21.s, z21.s, z4.s\n"
      "add z22.s, z22.s, z4.s\n"
      "smin z16.s, p2/M, z16.s, z6.s\n"
      "add z23.s, z23.s, z4.s\n"
      "smin z17.s, p2/M, z17.s, z6.s\n"
      "smin z18.s, p2/M, z18.s, z6.s\n"
      "smin z19.s, p2/M, z19.s, z6.s\n"
      "smin z20.s, p2/M, z20.s, z6.s\n"
      "smin z21.s, p2/M, z21.s, z6.s\n"
      "smin z22.s, p2/M, z22.s, z6.s\n"
      "smin z23.s, p2/M, z23.s, z6.s\n"
      "smax z16.s, p2/M, z16.s, z5.s\n"
      "smax z17.s, p2/M, z17.s, z5.s\n"
      "smax z18.s, p2/M, z18.s, z5.s\n"
      "smax z19.s, p2/M, z19.s, z5.s\n"
      "smax z20.s, p2/M, z20.s, z5.s\n"
      "smax z21.s, p2/M, z21.s, z5.s\n"
      "smax z22.s, p2/M, z22.s, z5.s\n"
      "smax z23.s, p2/M, z23.s, z5.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z21.h, z22.h, z23.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "uzp1 z20.b, z20.b, z21.b\n"
      "st1b { z16.b }, p1, [x27]\n"
      "addvl x27, x27, #1\n"
      "st1b { z20.b }, p1, [x26]\n"
      "decw x10, ALL, MUL #4\n"
      "cmp x10, XZR\n"
      "bgt 15b\n"
      "b 54f\n"
      "27:"  // Height 3
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "mov z13.s, #0x0\n"
      "mov z15.b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x28, %x[col_bias]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "28:"  // Height 3: Column loop
      "mov x20, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "whilelt p1.b, x20, x10\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "mov x26, #0x0\n"
      "30:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 31f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "cbnz x26, 32f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 32f\n"
      "31:"  // Height 3: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "32:"  // Height 3: input setup done
      "cmp x25, #0x10\n"
      "ble 35f\n"
      "33:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x25\n"
      "ld1b { z4.b }, p2/Z, [x9]\n"
      "ld1b { z5.b }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1b { z6.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #3, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #4, MUL VL]\n"
      "ld1b { z9.b }, p2/Z, [x9, #5, MUL VL]\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x22]\n"
      "ld1b { z10.b }, p2/Z, [x9, #6, MUL VL]\n"
      "add x22, x22, #0x10\n"
      "sdot z16.s, z4.b, z0.b[0]\n"
      "sdot z20.s, z4.b, z1.b[0]\n"
      "sdot z17.s, z5.b, z0.b[0]\n"
      "sdot z21.s, z5.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z0.b[0]\n"
      "sdot z24.s, z4.b, z2.b[0]\n"
      "sdot z25.s, z5.b, z2.b[0]\n"
      "ld1b { z4.b }, p2/Z, [x9, #7, MUL VL]\n"
      "addvl x9, x9, #16\n"
      "sdot z22.s, z6.b, z1.b[0]\n"
      "sdot z26.s, z6.b, z2.b[0]\n"
      "sdot z19.s, z7.b, z0.b[0]\n"
      "sdot z23.s, z7.b, z1.b[0]\n"
      "sdot z27.s, z7.b, z2.b[0]\n"
      "sdot z16.s, z8.b, z0.b[1]\n"
      "ld1b { z5.b }, p2/Z, [x9, #-8, MUL VL]\n"
      "ld1b { z6.b }, p2/Z, [x9, #-7, MUL VL]\n"
      "sdot z20.s, z8.b, z1.b[1]\n"
      "sdot z24.s, z8.b, z2.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x9, #-6, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #-5, MUL VL]\n"
      "sdot z17.s, z9.b, z0.b[1]\n"
      "sdot z21.s, z9.b, z1.b[1]\n"
      "sdot z25.s, z9.b, z2.b[1]\n"
      "sdot z18.s, z10.b, z0.b[1]\n"
      "ld1b { z9.b }, p2/Z, [x9, #-4, MUL VL]\n"
      "sdot z22.s, z10.b, z1.b[1]\n"
      "sdot z26.s, z10.b, z2.b[1]\n"
      "ld1b { z10.b }, p2/Z, [x9, #-3, MUL VL]\n"
      "sdot z19.s, z4.b, z0.b[1]\n"
      "sdot z23.s, z4.b, z1.b[1]\n"
      "sdot z27.s, z4.b, z2.b[1]\n"
      "sdot z16.s, z5.b, z0.b[2]\n"
      "ld1b { z4.b }, p2/Z, [x9, #-2, MUL VL]\n"
      "sdot z20.s, z5.b, z1.b[2]\n"
      "sdot z24.s, z5.b, z2.b[2]\n"
      "ld1b { z5.b }, p2/Z, [x9, #-1, MUL VL]\n"
      "sdot z17.s, z6.b, z0.b[2]\n"
      "sdot z21.s, z6.b, z1.b[2]\n"
      "sdot z25.s, z6.b, z2.b[2]\n"
      "sdot z18.s, z7.b, z0.b[2]\n"
      "sdot z22.s, z7.b, z1.b[2]\n"
      "sdot z26.s, z7.b, z2.b[2]\n"
      "sdot z19.s, z8.b, z0.b[2]\n"
      "sdot z23.s, z8.b, z1.b[2]\n"
      "sdot z27.s, z8.b, z2.b[2]\n"
      "sdot z16.s, z9.b, z0.b[3]\n"
      "sdot z20.s, z9.b, z1.b[3]\n"
      "sdot z24.s, z9.b, z2.b[3]\n"
      "sdot z17.s, z10.b, z0.b[3]\n"
      "sdot z21.s, z10.b, z1.b[3]\n"
      "sdot z25.s, z10.b, z2.b[3]\n"
      "sdot z18.s, z4.b, z0.b[3]\n"
      "sdot z22.s, z4.b, z1.b[3]\n"
      "sdot z26.s, z4.b, z2.b[3]\n"
      "sdot z19.s, z5.b, z0.b[3]\n"
      "sdot z23.s, z5.b, z1.b[3]\n"
      "sdot z27.s, z5.b, z2.b[3]\n"
      "tbnz %x[flags], #31, 34f\n"
      "sdot z11.s, z0.b, z15.b\n"
      "sdot z12.s, z1.b, z15.b\n"
      "sdot z13.s, z2.b, z15.b\n"
      "34:"  // Height 3: Multiply loop: unique 5: skip row sum
      "sub x25, x25, #0x10\n"
      "cmp x25, #0x10\n"
      "bgt 33b\n"
      "35:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x25\n"
      "ld1b { z4.b }, p2/Z, [x9]\n"
      "ld1b { z5.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z6.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "ld1rqb { z2.b }, p0/Z, [x22]\n"
      "sdot z16.s, z4.b, z0.b[0]\n"
      "sdot z20.s, z4.b, z1.b[0]\n"
      "sdot z17.s, z5.b, z0.b[0]\n"
      "sdot z21.s, z5.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z0.b[0]\n"
      "sdot z22.s, z6.b, z1.b[0]\n"
      "sdot z24.s, z4.b, z2.b[0]\n"
      "sdot z25.s, z5.b, z2.b[0]\n"
      "sdot z26.s, z6.b, z2.b[0]\n"
      "sdot z19.s, z7.b, z0.b[0]\n"
      "sdot z23.s, z7.b, z1.b[0]\n"
      "sdot z27.s, z7.b, z2.b[0]\n"
      "ble 36f\n"
      "ld1b { z8.b }, p2/Z, [x9]\n"
      "ld1b { z9.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z10.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z8.b, z0.b[1]\n"
      "sdot z20.s, z8.b, z1.b[1]\n"
      "sdot z24.s, z8.b, z2.b[1]\n"
      "sdot z17.s, z9.b, z0.b[1]\n"
      "sdot z21.s, z9.b, z1.b[1]\n"
      "sdot z25.s, z9.b, z2.b[1]\n"
      "sdot z18.s, z10.b, z0.b[1]\n"
      "sdot z22.s, z10.b, z1.b[1]\n"
      "sdot z26.s, z10.b, z2.b[1]\n"
      "sdot z19.s, z4.b, z0.b[1]\n"
      "sdot z23.s, z4.b, z1.b[1]\n"
      "sdot z27.s, z4.b, z2.b[1]\n"
      "ble 36f\n"
      "ld1b { z5.b }, p2/Z, [x9]\n"
      "ld1b { z6.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z7.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z5.b, z0.b[2]\n"
      "sdot z20.s, z5.b, z1.b[2]\n"
      "sdot z24.s, z5.b, z2.b[2]\n"
      "sdot z17.s, z6.b, z0.b[2]\n"
      "sdot z21.s, z6.b, z1.b[2]\n"
      "sdot z25.s, z6.b, z2.b[2]\n"
      "sdot z18.s, z7.b, z0.b[2]\n"
      "sdot z22.s, z7.b, z1.b[2]\n"
      "sdot z26.s, z7.b, z2.b[2]\n"
      "sdot z19.s, z8.b, z0.b[2]\n"
      "sdot z23.s, z8.b, z1.b[2]\n"
      "sdot z27.s, z8.b, z2.b[2]\n"
      "ble 36f\n"
      "ld1b { z9.b }, p2/Z, [x9]\n"
      "ld1b { z10.b }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z5.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z9.b, z0.b[3]\n"
      "sdot z20.s, z9.b, z1.b[3]\n"
      "sdot z24.s, z9.b, z2.b[3]\n"
      "sdot z17.s, z10.b, z0.b[3]\n"
      "sdot z21.s, z10.b, z1.b[3]\n"
      "sdot z25.s, z10.b, z2.b[3]\n"
      "sdot z18.s, z4.b, z0.b[3]\n"
      "sdot z22.s, z4.b, z1.b[3]\n"
      "sdot z26.s, z4.b, z2.b[3]\n"
      "sdot z19.s, z5.b, z0.b[3]\n"
      "sdot z23.s, z5.b, z1.b[3]\n"
      "sdot z27.s, z5.b, z2.b[3]\n"
      "36:"  // Height 3: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 37f\n"
      "sdot z11.s, z0.b, z15.b\n"
      "sdot z12.s, z1.b, z15.b\n"
      "sdot z13.s, z2.b, z15.b\n"
      "37:"  // Height 3: Multiply loop: unique 6: skip row sum
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 30b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x27, x20\n"
      "add x25, x26, x20\n"
      "tbnz %x[flags], #31, 38f\n"
      "mov x21, #0x4\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1rw { z3.s }, p2/Z, [x20]\n"
      "whilelt p0.s, XZR, x21\n"
      "neg z3.s, p2/M, z3.s\n"
      "saddv d11, p0, z11.s\n"
      "saddv d12, p0, z12.s\n"
      "saddv d13, p0, z13.s\n"
      "mov z11.s, z11.s[0]\n"
      "mov z12.s, z12.s[0]\n"
      "mul z11.s, p2/M, z11.s, z3.s\n"
      "mul z12.s, p2/M, z12.s, z3.s\n"
      "mov z13.s, z13.s[0]\n"
      "mul z13.s, p2/M, z13.s, z3.s\n"
      "38:"  // Height 3: skip row sum fixup
      "add z16.s, z16.s, z11.s\n"
      "add z17.s, z17.s, z11.s\n"
      "ld1w { z0.s }, p2/Z, [x28]\n"
      "ld1w { z1.s }, p2/Z, [x28, #1, MUL VL]\n"
      "add z18.s, z18.s, z11.s\n"
      "add z19.s, z19.s, z11.s\n"
      "ld1w { z2.s }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x28, #3, MUL VL]\n"
      "add z20.s, z20.s, z12.s\n"
      "add z21.s, z21.s, z12.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "add x23, %x[qp], %[per_layer_right_shift]\n"
      "add z22.s, z22.s, z12.s\n"
      "add z23.s, z23.s, z12.s\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add z24.s, z24.s, z13.s\n"
      "add z25.s, z25.s, z13.s\n"
      "add x21, %x[qp], %[maxval]\n"
      "add x20, %x[qp], %[minval]\n"
      "add z26.s, z26.s, z13.s\n"
      "add z27.s, z27.s, z13.s\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z1.s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "addvl x28, x28, #4\n"
      "add z18.s, z18.s, z2.s\n"
      "add z19.s, z19.s, z3.s\n"
      "add z20.s, z20.s, z0.s\n"
      "add z21.s, z21.s, z1.s\n"
      "add z22.s, z22.s, z2.s\n"
      "add z23.s, z23.s, z3.s\n"
      "add z24.s, z24.s, z0.s\n"
      "add z25.s, z25.s, z1.s\n"
      "ld1rw { z0.s }, p2/Z, [x23]\n"
      "add z26.s, z26.s, z2.s\n"
      "add z27.s, z27.s, z3.s\n"
      ".inst 0x04a47210  // sqdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a47231  // sqdmulh z17.s, z17.s, z4.s\n"
      ".inst 0x04a47252  // sqdmulh z18.s, z18.s, z4.s\n"
      ".inst 0x04a47273  // sqdmulh z19.s, z19.s, z4.s\n"
      ".inst 0x04a47294  // sqdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a472b5  // sqdmulh z21.s, z21.s, z4.s\n"
      ".inst 0x04a472d6  // sqdmulh z22.s, z22.s, z4.s\n"
      ".inst 0x04a472f7  // sqdmulh z23.s, z23.s, z4.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      ".inst 0x44828811  // srshl z17.s, p2/M, z17.s, z0.s\n"
      ".inst 0x04a47318  // sqdmulh z24.s, z24.s, z4.s\n"
      ".inst 0x04a47339  // sqdmulh z25.s, z25.s, z4.s\n"
      ".inst 0x44828812  // srshl z18.s, p2/M, z18.s, z0.s\n"
      ".inst 0x44828813  // srshl z19.s, p2/M, z19.s, z0.s\n"
      ".inst 0x04a4735a  // sqdmulh z26.s, z26.s, z4.s\n"
      ".inst 0x04a4737b  // sqdmulh z27.s, z27.s, z4.s\n"
      "ld1rw { z4.s }, p2/Z, [x22]\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      ".inst 0x44828815  // srshl z21.s, p2/M, z21.s, z0.s\n"
      ".inst 0x44828816  // srshl z22.s, p2/M, z22.s, z0.s\n"
      ".inst 0x44828817  // srshl z23.s, p2/M, z23.s, z0.s\n"
      ".inst 0x44828818  // srshl z24.s, p2/M, z24.s, z0.s\n"
      ".inst 0x44828819  // srshl z25.s, p2/M, z25.s, z0.s\n"
      ".inst 0x4482881a  // srshl z26.s, p2/M, z26.s, z0.s\n"
      "add z16.s, z16.s, z4.s\n"
      "add z17.s, z17.s, z4.s\n"
      ".inst 0x4482881b  // srshl z27.s, p2/M, z27.s, z0.s\n"
      "add z18.s, z18.s, z4.s\n"
      "add z19.s, z19.s, z4.s\n"
      "add z20.s, z20.s, z4.s\n"
      "add z21.s, z21.s, z4.s\n"
      "add z22.s, z22.s, z4.s\n"
      "add z23.s, z23.s, z4.s\n"
      "smin z16.s, p2/M, z16.s, z6.s\n"
      "smin z17.s, p2/M, z17.s, z6.s\n"
      "add z24.s, z24.s, z4.s\n"
      "add z25.s, z25.s, z4.s\n"
      "smin z18.s, p2/M, z18.s, z6.s\n"
      "smin z19.s, p2/M, z19.s, z6.s\n"
      "add z26.s, z26.s, z4.s\n"
      "add z27.s, z27.s, z4.s\n"
      "smin z20.s, p2/M, z20.s, z6.s\n"
      "smin z21.s, p2/M, z21.s, z6.s\n"
      "smin z22.s, p2/M, z22.s, z6.s\n"
      "smin z23.s, p2/M, z23.s, z6.s\n"
      "smin z24.s, p2/M, z24.s, z6.s\n"
      "smin z25.s, p2/M, z25.s, z6.s\n"
      "smin z26.s, p2/M, z26.s, z6.s\n"
      "smin z27.s, p2/M, z27.s, z6.s\n"
      "smax z16.s, p2/M, z16.s, z5.s\n"
      "smax z17.s, p2/M, z17.s, z5.s\n"
      "smax z18.s, p2/M, z18.s, z5.s\n"
      "smax z19.s, p2/M, z19.s, z5.s\n"
      "smax z20.s, p2/M, z20.s, z5.s\n"
      "smax z21.s, p2/M, z21.s, z5.s\n"
      "smax z22.s, p2/M, z22.s, z5.s\n"
      "smax z23.s, p2/M, z23.s, z5.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "smax z24.s, p2/M, z24.s, z5.s\n"
      "smax z25.s, p2/M, z25.s, z5.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "smax z26.s, p2/M, z26.s, z5.s\n"
      "smax z27.s, p2/M, z27.s, z5.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z21.h, z22.h, z23.h\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "uzp1 z25.h, z26.h, z27.h\n"
      "uzp1 z20.b, z20.b, z21.b\n"
      "st1b { z16.b }, p1, [x27]\n"
      "addvl x27, x27, #1\n"
      "uzp1 z24.b, z24.b, z25.b\n"
      "st1b { z20.b }, p1, [x26]\n"
      "st1b { z24.b }, p1, [x25]\n"
      "decw x10, ALL, MUL #4\n"
      "cmp x10, XZR\n"
      "bgt 28b\n"
      "b 54f\n"
      "40:"  // Height 4
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x4\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "mov z14.s, #0x0\n"
      "mov z15.b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x28, %x[col_bias]\n"
      "madd x20, x21, x20, x27\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "41:"  // Height 4: Column loop
      "mov x20, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "whilelt p1.b, x20, x10\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "mov z28.s, #0x0\n"
      "mov z29.s, #0x0\n"
      "mov z30.s, #0x0\n"
      "mov z31.s, #0x0\n"
      "mov x26, #0x0\n"
      "43:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 44f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "ldr x21, [x20, #0x18]\n"
      "cbnz x26, 45f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 45f\n"
      "44:"  // Height 4: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "45:"  // Height 4: input setup done
      "cmp x25, #0x10\n"
      "ble 48f\n"
      "46:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x25\n"
      "ld1b { z4.b }, p2/Z, [x9]\n"
      "ld1b { z5.b }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1b { z6.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #3, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #4, MUL VL]\n"
      "ld1b { z9.b }, p2/Z, [x9, #5, MUL VL]\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x22]\n"
      "ld1rqb { z3.b }, p0/Z, [x21]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      "ld1b { z10.b }, p2/Z, [x9, #6, MUL VL]\n"
      "sdot z16.s, z4.b, z0.b[0]\n"
      "sdot z20.s, z4.b, z1.b[0]\n"
      "sdot z17.s, z5.b, z0.b[0]\n"
      "sdot z21.s, z5.b, z1.b[0]\n"
      "sdot z24.s, z4.b, z2.b[0]\n"
      "sdot z28.s, z4.b, z3.b[0]\n"
      "ld1b { z4.b }, p2/Z, [x9, #7, MUL VL]\n"
      "addvl x9, x9, #16\n"
      "sdot z25.s, z5.b, z2.b[0]\n"
      "sdot z29.s, z5.b, z3.b[0]\n"
      "sdot z18.s, z6.b, z0.b[0]\n"
      "sdot z22.s, z6.b, z1.b[0]\n"
      "sdot z26.s, z6.b, z2.b[0]\n"
      "sdot z30.s, z6.b, z3.b[0]\n"
      "ld1b { z5.b }, p2/Z, [x9, #-8, MUL VL]\n"
      "ld1b { z6.b }, p2/Z, [x9, #-7, MUL VL]\n"
      "sdot z19.s, z7.b, z0.b[0]\n"
      "sdot z23.s, z7.b, z1.b[0]\n"
      "sdot z27.s, z7.b, z2.b[0]\n"
      "sdot z31.s, z7.b, z3.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x9, #-6, MUL VL]\n"
      "sdot z16.s, z8.b, z0.b[1]\n"
      "sdot z20.s, z8.b, z1.b[1]\n"
      "sdot z24.s, z8.b, z2.b[1]\n"
      "sdot z28.s, z8.b, z3.b[1]\n"
      "ld1b { z8.b }, p2/Z, [x9, #-5, MUL VL]\n"
      "sdot z17.s, z9.b, z0.b[1]\n"
      "sdot z21.s, z9.b, z1.b[1]\n"
      "sdot z25.s, z9.b, z2.b[1]\n"
      "sdot z29.s, z9.b, z3.b[1]\n"
      "ld1b { z9.b }, p2/Z, [x9, #-4, MUL VL]\n"
      "sdot z18.s, z10.b, z0.b[1]\n"
      "sdot z22.s, z10.b, z1.b[1]\n"
      "sdot z26.s, z10.b, z2.b[1]\n"
      "sdot z30.s, z10.b, z3.b[1]\n"
      "ld1b { z10.b }, p2/Z, [x9, #-3, MUL VL]\n"
      "sdot z19.s, z4.b, z0.b[1]\n"
      "sdot z23.s, z4.b, z1.b[1]\n"
      "sdot z27.s, z4.b, z2.b[1]\n"
      "sdot z31.s, z4.b, z3.b[1]\n"
      "ld1b { z4.b }, p2/Z, [x9, #-2, MUL VL]\n"
      "sdot z16.s, z5.b, z0.b[2]\n"
      "sdot z20.s, z5.b, z1.b[2]\n"
      "sdot z24.s, z5.b, z2.b[2]\n"
      "sdot z28.s, z5.b, z3.b[2]\n"
      "ld1b { z5.b }, p2/Z, [x9, #-1, MUL VL]\n"
      "sdot z17.s, z6.b, z0.b[2]\n"
      "sdot z21.s, z6.b, z1.b[2]\n"
      "sdot z25.s, z6.b, z2.b[2]\n"
      "sdot z29.s, z6.b, z3.b[2]\n"
      "sdot z18.s, z7.b, z0.b[2]\n"
      "sdot z22.s, z7.b, z1.b[2]\n"
      "sdot z26.s, z7.b, z2.b[2]\n"
      "sdot z30.s, z7.b, z3.b[2]\n"
      "sdot z19.s, z8.b, z0.b[2]\n"
      "sdot z23.s, z8.b, z1.b[2]\n"
      "sdot z27.s, z8.b, z2.b[2]\n"
      "sdot z31.s, z8.b, z3.b[2]\n"
      "sdot z16.s, z9.b, z0.b[3]\n"
      "sdot z20.s, z9.b, z1.b[3]\n"
      "sdot z24.s, z9.b, z2.b[3]\n"
      "sdot z28.s, z9.b, z3.b[3]\n"
      "sdot z17.s, z10.b, z0.b[3]\n"
      "sdot z21.s, z10.b, z1.b[3]\n"
      "sdot z25.s, z10.b, z2.b[3]\n"
      "sdot z29.s, z10.b, z3.b[3]\n"
      "sdot z18.s, z4.b, z0.b[3]\n"
      "sdot z22.s, z4.b, z1.b[3]\n"
      "sdot z26.s, z4.b, z2.b[3]\n"
      "sdot z30.s, z4.b, z3.b[3]\n"
      "sdot z19.s, z5.b, z0.b[3]\n"
      "sdot z23.s, z5.b, z1.b[3]\n"
      "sdot z27.s, z5.b, z2.b[3]\n"
      "sdot z31.s, z5.b, z3.b[3]\n"
      "tbnz %x[flags], #31, 47f\n"
      "sdot z11.s, z0.b, z15.b\n"
      "sdot z12.s, z1.b, z15.b\n"
      "sdot z13.s, z2.b, z15.b\n"
      "sdot z14.s, z3.b, z15.b\n"
      "47:"  // Height 4: Multiply loop: unique 7: skip row sum
      "sub x25, x25, #0x10\n"
      "cmp x25, #0x10\n"
      "bgt 46b\n"
      "48:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x25\n"
      "ld1b { z4.b }, p2/Z, [x9]\n"
      "ld1b { z5.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z6.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "ld1rqb { z2.b }, p0/Z, [x22]\n"
      "ld1rqb { z3.b }, p0/Z, [x21]\n"
      "sdot z16.s, z4.b, z0.b[0]\n"
      "sdot z20.s, z4.b, z1.b[0]\n"
      "sdot z17.s, z5.b, z0.b[0]\n"
      "sdot z21.s, z5.b, z1.b[0]\n"
      "sdot z18.s, z6.b, z0.b[0]\n"
      "sdot z22.s, z6.b, z1.b[0]\n"
      "sdot z24.s, z4.b, z2.b[0]\n"
      "sdot z28.s, z4.b, z3.b[0]\n"
      "sdot z25.s, z5.b, z2.b[0]\n"
      "sdot z29.s, z5.b, z3.b[0]\n"
      "sdot z26.s, z6.b, z2.b[0]\n"
      "sdot z30.s, z6.b, z3.b[0]\n"
      "sdot z19.s, z7.b, z0.b[0]\n"
      "sdot z23.s, z7.b, z1.b[0]\n"
      "sdot z27.s, z7.b, z2.b[0]\n"
      "sdot z31.s, z7.b, z3.b[0]\n"
      "ble 49f\n"
      "ld1b { z8.b }, p2/Z, [x9]\n"
      "ld1b { z9.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z10.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z8.b, z0.b[1]\n"
      "sdot z20.s, z8.b, z1.b[1]\n"
      "sdot z24.s, z8.b, z2.b[1]\n"
      "sdot z28.s, z8.b, z3.b[1]\n"
      "sdot z17.s, z9.b, z0.b[1]\n"
      "sdot z21.s, z9.b, z1.b[1]\n"
      "sdot z25.s, z9.b, z2.b[1]\n"
      "sdot z29.s, z9.b, z3.b[1]\n"
      "sdot z18.s, z10.b, z0.b[1]\n"
      "sdot z22.s, z10.b, z1.b[1]\n"
      "sdot z26.s, z10.b, z2.b[1]\n"
      "sdot z30.s, z10.b, z3.b[1]\n"
      "sdot z19.s, z4.b, z0.b[1]\n"
      "sdot z23.s, z4.b, z1.b[1]\n"
      "sdot z27.s, z4.b, z2.b[1]\n"
      "sdot z31.s, z4.b, z3.b[1]\n"
      "ble 49f\n"
      "ld1b { z5.b }, p2/Z, [x9]\n"
      "ld1b { z6.b }, p2/Z, [x9, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z7.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z8.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z5.b, z0.b[2]\n"
      "sdot z20.s, z5.b, z1.b[2]\n"
      "sdot z24.s, z5.b, z2.b[2]\n"
      "sdot z28.s, z5.b, z3.b[2]\n"
      "sdot z17.s, z6.b, z0.b[2]\n"
      "sdot z21.s, z6.b, z1.b[2]\n"
      "sdot z25.s, z6.b, z2.b[2]\n"
      "sdot z29.s, z6.b, z3.b[2]\n"
      "sdot z18.s, z7.b, z0.b[2]\n"
      "sdot z22.s, z7.b, z1.b[2]\n"
      "sdot z26.s, z7.b, z2.b[2]\n"
      "sdot z30.s, z7.b, z3.b[2]\n"
      "sdot z19.s, z8.b, z0.b[2]\n"
      "sdot z23.s, z8.b, z1.b[2]\n"
      "sdot z27.s, z8.b, z2.b[2]\n"
      "sdot z31.s, z8.b, z3.b[2]\n"
      "ble 49f\n"
      "ld1b { z9.b }, p2/Z, [x9]\n"
      "ld1b { z10.b }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1b { z5.b }, p2/Z, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "sdot z16.s, z9.b, z0.b[3]\n"
      "sdot z20.s, z9.b, z1.b[3]\n"
      "sdot z24.s, z9.b, z2.b[3]\n"
      "sdot z28.s, z9.b, z3.b[3]\n"
      "sdot z17.s, z10.b, z0.b[3]\n"
      "sdot z21.s, z10.b, z1.b[3]\n"
      "sdot z25.s, z10.b, z2.b[3]\n"
      "sdot z29.s, z10.b, z3.b[3]\n"
      "sdot z18.s, z4.b, z0.b[3]\n"
      "sdot z22.s, z4.b, z1.b[3]\n"
      "sdot z26.s, z4.b, z2.b[3]\n"
      "sdot z30.s, z4.b, z3.b[3]\n"
      "sdot z19.s, z5.b, z0.b[3]\n"
      "sdot z23.s, z5.b, z1.b[3]\n"
      "sdot z27.s, z5.b, z2.b[3]\n"
      "sdot z31.s, z5.b, z3.b[3]\n"
      "49:"  // Height 4: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 50f\n"
      "sdot z11.s, z0.b, z15.b\n"
      "sdot z12.s, z1.b, z15.b\n"
      "sdot z13.s, z2.b, z15.b\n"
      "sdot z14.s, z3.b, z15.b\n"
      "50:"  // Height 4: Multiply loop: unique 8: skip row sum
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 43b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x27, x20\n"
      "add x25, x26, x20\n"
      "add x24, x25, x20\n"
      "tbnz %x[flags], #31, 51f\n"
      "mov x21, #0x4\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "whilelt p0.s, XZR, x21\n"
      "neg z4.s, p2/M, z4.s\n"
      "saddv d11, p0, z11.s\n"
      "saddv d12, p0, z12.s\n"
      "saddv d13, p0, z13.s\n"
      "saddv d14, p0, z14.s\n"
      "mov z11.s, z11.s[0]\n"
      "mov z12.s, z12.s[0]\n"
      "mul z11.s, p2/M, z11.s, z4.s\n"
      "mul z12.s, p2/M, z12.s, z4.s\n"
      "mov z13.s, z13.s[0]\n"
      "mov z14.s, z14.s[0]\n"
      "mul z13.s, p2/M, z13.s, z4.s\n"
      "mul z14.s, p2/M, z14.s, z4.s\n"
      "51:"  // Height 4: skip row sum fixup
      "add z16.s, z16.s, z11.s\n"
      "add z17.s, z17.s, z11.s\n"
      "ld1w { z0.s }, p2/Z, [x28]\n"
      "ld1w { z1.s }, p2/Z, [x28, #1, MUL VL]\n"
      "add z18.s, z18.s, z11.s\n"
      "add z19.s, z19.s, z11.s\n"
      "ld1w { z2.s }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x28, #3, MUL VL]\n"
      "add z20.s, z20.s, z12.s\n"
      "add z21.s, z21.s, z12.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "add x23, %x[qp], %[per_layer_right_shift]\n"
      "add z22.s, z22.s, z12.s\n"
      "add z23.s, z23.s, z12.s\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add z24.s, z24.s, z13.s\n"
      "add z25.s, z25.s, z13.s\n"
      "add x21, %x[qp], %[maxval]\n"
      "add x20, %x[qp], %[minval]\n"
      "add z26.s, z26.s, z13.s\n"
      "add z27.s, z27.s, z13.s\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      "ld1rw { z5.s }, p2/Z, [x20]\n"
      "add z28.s, z28.s, z14.s\n"
      "add z29.s, z29.s, z14.s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "addvl x28, x28, #4\n"
      "add z30.s, z30.s, z14.s\n"
      "add z31.s, z31.s, z14.s\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z1.s\n"
      "add z18.s, z18.s, z2.s\n"
      "add z19.s, z19.s, z3.s\n"
      "add z20.s, z20.s, z0.s\n"
      "add z21.s, z21.s, z1.s\n"
      "add z22.s, z22.s, z2.s\n"
      "add z23.s, z23.s, z3.s\n"
      "add z24.s, z24.s, z0.s\n"
      "add z25.s, z25.s, z1.s\n"
      "add z26.s, z26.s, z2.s\n"
      "add z27.s, z27.s, z3.s\n"
      "add z28.s, z28.s, z0.s\n"
      "add z29.s, z29.s, z1.s\n"
      "ld1rw { z0.s }, p2/Z, [x23]\n"
      "add z30.s, z30.s, z2.s\n"
      "add z31.s, z31.s, z3.s\n"
      ".inst 0x04a47210  // sqdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a47231  // sqdmulh z17.s, z17.s, z4.s\n"
      ".inst 0x04a47252  // sqdmulh z18.s, z18.s, z4.s\n"
      ".inst 0x04a47273  // sqdmulh z19.s, z19.s, z4.s\n"
      ".inst 0x04a47294  // sqdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a472b5  // sqdmulh z21.s, z21.s, z4.s\n"
      ".inst 0x04a472d6  // sqdmulh z22.s, z22.s, z4.s\n"
      ".inst 0x04a472f7  // sqdmulh z23.s, z23.s, z4.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      ".inst 0x44828811  // srshl z17.s, p2/M, z17.s, z0.s\n"
      ".inst 0x04a47318  // sqdmulh z24.s, z24.s, z4.s\n"
      ".inst 0x04a47339  // sqdmulh z25.s, z25.s, z4.s\n"
      ".inst 0x44828812  // srshl z18.s, p2/M, z18.s, z0.s\n"
      ".inst 0x44828813  // srshl z19.s, p2/M, z19.s, z0.s\n"
      ".inst 0x04a4735a  // sqdmulh z26.s, z26.s, z4.s\n"
      ".inst 0x04a4737b  // sqdmulh z27.s, z27.s, z4.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      ".inst 0x44828815  // srshl z21.s, p2/M, z21.s, z0.s\n"
      ".inst 0x04a4739c  // sqdmulh z28.s, z28.s, z4.s\n"
      ".inst 0x04a473bd  // sqdmulh z29.s, z29.s, z4.s\n"
      ".inst 0x44828816  // srshl z22.s, p2/M, z22.s, z0.s\n"
      ".inst 0x44828817  // srshl z23.s, p2/M, z23.s, z0.s\n"
      ".inst 0x04a473de  // sqdmulh z30.s, z30.s, z4.s\n"
      ".inst 0x04a473ff  // sqdmulh z31.s, z31.s, z4.s\n"
      "ld1rw { z4.s }, p2/Z, [x22]\n"
      ".inst 0x44828818  // srshl z24.s, p2/M, z24.s, z0.s\n"
      ".inst 0x44828819  // srshl z25.s, p2/M, z25.s, z0.s\n"
      ".inst 0x4482881a  // srshl z26.s, p2/M, z26.s, z0.s\n"
      ".inst 0x4482881b  // srshl z27.s, p2/M, z27.s, z0.s\n"
      ".inst 0x4482881c  // srshl z28.s, p2/M, z28.s, z0.s\n"
      ".inst 0x4482881d  // srshl z29.s, p2/M, z29.s, z0.s\n"
      ".inst 0x4482881e  // srshl z30.s, p2/M, z30.s, z0.s\n"
      "add z16.s, z16.s, z4.s\n"
      "add z17.s, z17.s, z4.s\n"
      ".inst 0x4482881f  // srshl z31.s, p2/M, z31.s, z0.s\n"
      "add z18.s, z18.s, z4.s\n"
      "add z19.s, z19.s, z4.s\n"
      "add z20.s, z20.s, z4.s\n"
      "add z21.s, z21.s, z4.s\n"
      "add z22.s, z22.s, z4.s\n"
      "add z23.s, z23.s, z4.s\n"
      "smin z16.s, p2/M, z16.s, z6.s\n"
      "smin z17.s, p2/M, z17.s, z6.s\n"
      "add z24.s, z24.s, z4.s\n"
      "add z25.s, z25.s, z4.s\n"
      "smin z18.s, p2/M, z18.s, z6.s\n"
      "smin z19.s, p2/M, z19.s, z6.s\n"
      "add z26.s, z26.s, z4.s\n"
      "add z27.s, z27.s, z4.s\n"
      "smin z20.s, p2/M, z20.s, z6.s\n"
      "smin z21.s, p2/M, z21.s, z6.s\n"
      "add z28.s, z28.s, z4.s\n"
      "add z29.s, z29.s, z4.s\n"
      "smin z22.s, p2/M, z22.s, z6.s\n"
      "smin z23.s, p2/M, z23.s, z6.s\n"
      "add z30.s, z30.s, z4.s\n"
      "add z31.s, z31.s, z4.s\n"
      "smin z24.s, p2/M, z24.s, z6.s\n"
      "smin z25.s, p2/M, z25.s, z6.s\n"
      "smin z26.s, p2/M, z26.s, z6.s\n"
      "smin z27.s, p2/M, z27.s, z6.s\n"
      "smin z28.s, p2/M, z28.s, z6.s\n"
      "smin z29.s, p2/M, z29.s, z6.s\n"
      "smin z30.s, p2/M, z30.s, z6.s\n"
      "smin z31.s, p2/M, z31.s, z6.s\n"
      "smax z16.s, p2/M, z16.s, z5.s\n"
      "smax z17.s, p2/M, z17.s, z5.s\n"
      "smax z18.s, p2/M, z18.s, z5.s\n"
      "smax z19.s, p2/M, z19.s, z5.s\n"
      "smax z20.s, p2/M, z20.s, z5.s\n"
      "smax z21.s, p2/M, z21.s, z5.s\n"
      "smax z22.s, p2/M, z22.s, z5.s\n"
      "smax z23.s, p2/M, z23.s, z5.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "smax z24.s, p2/M, z24.s, z5.s\n"
      "smax z25.s, p2/M, z25.s, z5.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "smax z26.s, p2/M, z26.s, z5.s\n"
      "smax z27.s, p2/M, z27.s, z5.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "smax z28.s, p2/M, z28.s, z5.s\n"
      "smax z29.s, p2/M, z29.s, z5.s\n"
      "uzp1 z21.h, z22.h, z23.h\n"
      "smax z30.s, p2/M, z30.s, z5.s\n"
      "smax z31.s, p2/M, z31.s, z5.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "uzp1 z25.h, z26.h, z27.h\n"
      "uzp1 z28.h, z28.h, z29.h\n"
      "uzp1 z20.b, z20.b, z21.b\n"
      "uzp1 z29.h, z30.h, z31.h\n"
      "st1b { z16.b }, p1, [x27]\n"
      "addvl x27, x27, #1\n"
      "uzp1 z24.b, z24.b, z25.b\n"
      "uzp1 z28.b, z28.b, z29.b\n"
      "st1b { z20.b }, p1, [x26]\n"
      "st1b { z24.b }, p1, [x25]\n"
      "st1b { z28.b }, p1, [x24]\n"
      "decw x10, ALL, MUL #4\n"
      "cmp x10, XZR\n"
      "bgt 41b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 54f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 53f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "53:"  // Update direct input
      "mov x20, #0x4\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "54:"  // Exit
      : [M] "+&r" (M), [flags] "+&r" (flags), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [b_offset] "I" (offsetof(Requantize32, b_offset)), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "p0", "p1", "p2", "x9", "x10", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
