/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <cstdint>

namespace arm_gemm {

void sme2_gemv_u8qa_dot_16VL (
    const uint8_t *A_ptr, const uint8_t *B_ptr, uint8_t *output_ptr,
    size_t N, size_t K,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int
)
{
    struct KernelArgs {
        const uint8_t *B_ptr = {};
        size_t output_offset = {};
        unsigned int input_initial_col = {};
    } ka;

    unsigned long flags=0;
    ka.B_ptr = B_ptr;
    __asm__ __volatile__(
      "ptrue p8.b\n"
      ".inst 0xd503477f  // SMSTART ZA\n"
      "mov x9, #0x0\n"
      "cntw x28, ALL, MUL #4\n"
      "mov x27, %x[B_ptr]\n"
      "add x26, %x[N], x28\n"
      "mov x25, %x[output_ptr]\n"
      "sub x26, x26, #0x1\n"
      "ptrue p2.b\n"
      "udiv x26, x26, x28\n"
      ".inst 0x25207810  // ptrue pn8.b\n"
      "add x22, x26, #0x3\n"
      "mov x21, #0x1\n"
      "and x22, x22, #0xfffffffffffffffc\n"
      "mul x22, x22, x28\n"
      "mul x22, x22, %x[K]\n"
      "1:"  // RHS size check loop
      "cmp x22, #0x200000\n"
      "blt 2f\n"
      "tbnz x22, #0, 3f\n"
      "lsr x22, x22, #0x1\n"
      "lsl x21, x21, #0x1\n"
      "b 1b\n"
      "2:"  // RHS do prefetch
      "lsl x20, x22, #0x26\n"
      "sub x21, x21, #0x1\n"
      "lsl x21, x21, #0x16\n"
      "orr x22, x22, x20\n"
      "orr x22, x22, x21\n"
      ".inst 0xf8b64b7a  // rprfm pldonce, x22, [x27]\n"
      "3:"  // RHS prefetch exit
      "mov x24, %x[col_bias]\n"
      "mov z11.s, #0x0\n"
      "mov z8.b, #0x1\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "4:"  // Column loop
      "cmp x26, #0x4\n"
      "bge 34f\n"
      "cmp x26, #0x2\n"
      "bgt 24f\n"
      "beq 14f\n"
      "mov x23, %x[A_ptr]\n"
      "mov x21, %x[K]\n"
      "mov x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      "whilelt p1.b, XZR, x20\n"
      "cbz x24, 5f\n"
      ".inst 0xa040c30c  // ld1w { z12.s-z15.s }, pn8.b/Z, [x24]\n"
      ".inst 0xc0042d80  // mova za.d[x9, #0], { z12.d-z15.d }\n"
      "b 6f\n"
      "5:"  // Width 1: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "6:"  // Width 1: setup done
      "cmp x22, #0x10\n"
      "ble 9f\n"
      "7:"  // Width 1: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x22\n"
      ".inst 0xa0408361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      "ld1rqb { z9.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0408379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa0408375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b030  // udot za.s[x9, 0], { z0.b-z3.b }, z9.b[0]\n"
      ".inst 0xa040837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b730  // udot za.s[x9, 0], { z24.b-z27.b }, z9.b[1]\n"
      ".inst 0xc159bab0  // udot za.s[x9, 0], { z20.b-z23.b }, z9.b[2]\n"
      ".inst 0xc159bfb0  // udot za.s[x9, 0], { z28.b-z31.b }, z9.b[3]\n"
      "tbnz %x[flags], #31, 8f\n"
      "udot z11.s, z9.b, z8.b\n"
      "8:"  // Width 1: Multiply loop: unique 1: skip row sum
      "sub x22, x22, #0x10\n"
      "cmp x22, #0x10\n"
      "bgt 7b\n"
      "9:"  // Width 1: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x22\n"
      ".inst 0xa0408379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      "ld1rqb { z9.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b330  // udot za.s[x9, 0], { z24.b-z27.b }, z9.b[0]\n"
      "ble 10f\n"
      ".inst 0xa0408361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b430  // udot za.s[x9, 0], { z0.b-z3.b }, z9.b[1]\n"
      "ble 10f\n"
      ".inst 0xa040837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159bbb0  // udot za.s[x9, 0], { z28.b-z31.b }, z9.b[2]\n"
      "ble 10f\n"
      ".inst 0xa0408371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159be30  // udot za.s[x9, 0], { z16.b-z19.b }, z9.b[3]\n"
      "10:"  // Width 1: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 11f\n"
      "udot z11.s, z9.b, z8.b\n"
      "11:"  // Width 1: Multiply loop: unique 2: skip row sum
      "tbnz %x[flags], #31, 12f\n"
      "add x21, %x[qp], %[b_offset]\n"
      "mov x20, #0x4\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "whilelt p0.s, XZR, x20\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "uaddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "neg z17.s, p2/M, z17.s\n"
      "mul z11.s, p2/M, z11.s, z17.s\n"
      "12:"  // Width 1: skip row sum fixup
      ".inst 0xc0904960  // addha za0.s, p2/M, p2/M, z11.s\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      ".inst 0xc0904961  // addha za1.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z10.s }, p2/Z, [x21]\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[minval]\n"
      ".inst 0xc0904962  // addha za2.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z6.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0xc0904963  // addha za3.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z7.s }, p2/Z, [x22]\n"
      "ld1rw { z30.s }, p2/Z, [x21]\n"
      "ld1rw { z23.s }, p2/Z, [x20]\n"
      ".inst 0xc0062c0c  // mova { z12.d-z15.d }, za.d[x9, #0]\n"
      ".inst 0xc1aaac0c  // sqdmulh { z12.s-z15.s }, { z12.s-z15.s }, z10.s\n"
      ".inst 0xc1a6aa2c  // srshl { z12.s-z15.s }, { z12.s-z15.s }, z6.s\n"
      ".inst 0xc1a7ab0c  // add { z12.s-z15.s }, { z12.s-z15.s }, z7.s\n"
      ".inst 0xc1b7cfcc  // sclamp { z12.s-z15.s }, z30.s, z23.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z19.h, z14.h, z15.h\n"
      "uzp1 z12.b, z12.b, z19.b\n"
      "st1b { z12.b }, p1, [x25]\n"
      "addvl x25, x25, #1\n"
      "b 44f\n"
      "14:"  // Width 2
      "mov x23, %x[A_ptr]\n"
      "mov x21, %x[K]\n"
      "sub x20, %x[N], x28\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      "whilelt p1.b, XZR, x20\n"
      "cbz x24, 15f\n"
      ".inst 0xa040c304  // ld1w { z4.s-z7.s }, pn8.b/Z, [x24]\n"
      ".inst 0xa041c300  // ld1w { z0.s-z3.s }, pn8.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xc0042c80  // mova za.d[x9, #0], { z4.d-z7.d }\n"
      ".inst 0xc0042c01  // mova za.d[x9, #1], { z0.d-z3.d }\n"
      "b 16f\n"
      "15:"  // Width 2: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "16:"  // Width 2: setup done
      "cmp x22, #0x10\n"
      "ble 19f\n"
      "17:"  // Width 2: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x22\n"
      ".inst 0xa0408375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27]\n"
      "ld1rqb { z9.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0418379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa0408371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27]\n"
      ".inst 0xc159b2b0  // udot za.s[x9, 0], { z20.b-z23.b }, z9.b[0]\n"
      ".inst 0xa0418375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b331  // udot za.s[x9, 1], { z24.b-z27.b }, z9.b[0]\n"
      ".inst 0xa040837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27]\n"
      ".inst 0xa0418361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040836d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x27]\n"
      ".inst 0xc159b630  // udot za.s[x9, 0], { z16.b-z19.b }, z9.b[1]\n"
      ".inst 0xa0418371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b6b1  // udot za.s[x9, 1], { z20.b-z23.b }, z9.b[1]\n"
      ".inst 0xc159bbb0  // udot za.s[x9, 0], { z28.b-z31.b }, z9.b[2]\n"
      ".inst 0xc159b831  // udot za.s[x9, 1], { z0.b-z3.b }, z9.b[2]\n"
      ".inst 0xc159bdb0  // udot za.s[x9, 0], { z12.b-z15.b }, z9.b[3]\n"
      ".inst 0xc159be31  // udot za.s[x9, 1], { z16.b-z19.b }, z9.b[3]\n"
      "tbnz %x[flags], #31, 18f\n"
      "udot z11.s, z9.b, z8.b\n"
      "18:"  // Width 2: Multiply loop: unique 3: skip row sum
      "sub x22, x22, #0x10\n"
      "cmp x22, #0x10\n"
      "bgt 17b\n"
      "19:"  // Width 2: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x22\n"
      ".inst 0xa0408361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      "ld1rqb { z9.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0418379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b030  // udot za.s[x9, 0], { z0.b-z3.b }, z9.b[0]\n"
      ".inst 0xc159b331  // udot za.s[x9, 1], { z24.b-z27.b }, z9.b[0]\n"
      "ble 20f\n"
      ".inst 0xa0408371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa0418379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b630  // udot za.s[x9, 0], { z16.b-z19.b }, z9.b[1]\n"
      ".inst 0xc159b731  // udot za.s[x9, 1], { z24.b-z27.b }, z9.b[1]\n"
      "ble 20f\n"
      ".inst 0xa040837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa0418361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159bbb0  // udot za.s[x9, 0], { z28.b-z31.b }, z9.b[2]\n"
      ".inst 0xc159b831  // udot za.s[x9, 1], { z0.b-z3.b }, z9.b[2]\n"
      "ble 20f\n"
      ".inst 0xa0408375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27]\n"
      ".inst 0xa0418379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159beb0  // udot za.s[x9, 0], { z20.b-z23.b }, z9.b[3]\n"
      ".inst 0xc159bf31  // udot za.s[x9, 1], { z24.b-z27.b }, z9.b[3]\n"
      "20:"  // Width 2: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 21f\n"
      "udot z11.s, z9.b, z8.b\n"
      "21:"  // Width 2: Multiply loop: unique 4: skip row sum
      "tbnz %x[flags], #31, 22f\n"
      "add x21, %x[qp], %[b_offset]\n"
      "mov x20, #0x4\n"
      "ld1rw { z1.s }, p2/Z, [x21]\n"
      "whilelt p0.s, XZR, x20\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "uaddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "neg z1.s, p2/M, z1.s\n"
      "mul z11.s, p2/M, z11.s, z1.s\n"
      "22:"  // Width 2: skip row sum fixup
      ".inst 0xc0904960  // addha za0.s, p2/M, p2/M, z11.s\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      ".inst 0xc0904961  // addha za1.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z2.s }, p2/Z, [x21]\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[minval]\n"
      ".inst 0xc0904962  // addha za2.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z3.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0xc0904963  // addha za3.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z9.s }, p2/Z, [x22]\n"
      "ld1rw { z6.s }, p2/Z, [x21]\n"
      "ld1rw { z29.s }, p2/Z, [x20]\n"
      ".inst 0xc0062c14  // mova { z20.d-z23.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c2c  // mova { z12.d-z15.d }, za.d[x9, #1]\n"
      ".inst 0xc1a2ac14  // sqdmulh { z20.s-z23.s }, { z20.s-z23.s }, z2.s\n"
      ".inst 0xc1a2ac0c  // sqdmulh { z12.s-z15.s }, { z12.s-z15.s }, z2.s\n"
      ".inst 0xc1a3aa34  // srshl { z20.s-z23.s }, { z20.s-z23.s }, z3.s\n"
      ".inst 0xc1a3aa2c  // srshl { z12.s-z15.s }, { z12.s-z15.s }, z3.s\n"
      ".inst 0xc1a9ab14  // add { z20.s-z23.s }, { z20.s-z23.s }, z9.s\n"
      ".inst 0xc1a9ab0c  // add { z12.s-z15.s }, { z12.s-z15.s }, z9.s\n"
      ".inst 0xc1bdccd4  // sclamp { z20.s-z23.s }, z6.s, z29.s\n"
      ".inst 0xc1bdcccc  // sclamp { z12.s-z15.s }, z6.s, z29.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z16.h, z22.h, z23.h\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z24.h, z14.h, z15.h\n"
      "uzp1 z20.b, z20.b, z16.b\n"
      "uzp1 z12.b, z12.b, z24.b\n"
      "st1b { z20.b }, p2, [x25]\n"
      "st1b { z12.b }, p1, [x25, #1, MUL VL]\n"
      "addvl x25, x25, #2\n"
      "b 44f\n"
      "24:"  // Width 3
      "mov x20, #0x2\n"
      "mov x23, %x[A_ptr]\n"
      "mov x21, %x[K]\n"
      "msub x20, x28, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      "whilelt p1.b, XZR, x20\n"
      "cbz x24, 25f\n"
      ".inst 0xa040c314  // ld1w { z20.s-z23.s }, pn8.b/Z, [x24]\n"
      ".inst 0xa041c310  // ld1w { z16.s-z19.s }, pn8.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xa042c30c  // ld1w { z12.s-z15.s }, pn8.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xc0042e80  // mova za.d[x9, #0], { z20.d-z23.d }\n"
      ".inst 0xc0042e01  // mova za.d[x9, #1], { z16.d-z19.d }\n"
      ".inst 0xc0042d82  // mova za.d[x9, #2], { z12.d-z15.d }\n"
      "b 26f\n"
      "25:"  // Width 3: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "26:"  // Width 3: setup done
      "cmp x22, #0x10\n"
      "ble 29f\n"
      "27:"  // Width 3: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x22\n"
      ".inst 0xa040837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27]\n"
      "ld1rqb { z9.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0418371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042836d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b3b0  // udot za.s[x9, 0], { z28.b-z31.b }, z9.b[0]\n"
      ".inst 0xa0408379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27]\n"
      ".inst 0xc159b231  // udot za.s[x9, 1], { z16.b-z19.b }, z9.b[0]\n"
      ".inst 0xa041837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc159b1b2  // udot za.s[x9, 2], { z12.b-z15.b }, z9.b[0]\n"
      ".inst 0xa0428371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040836d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x27]\n"
      ".inst 0xc159b730  // udot za.s[x9, 0], { z24.b-z27.b }, z9.b[1]\n"
      ".inst 0xa0418361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc159b7b1  // udot za.s[x9, 1], { z28.b-z31.b }, z9.b[1]\n"
      ".inst 0xa042837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b632  // udot za.s[x9, 2], { z16.b-z19.b }, z9.b[1]\n"
      ".inst 0xa0408375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27]\n"
      ".inst 0xa0418379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc159b9b0  // udot za.s[x9, 0], { z12.b-z15.b }, z9.b[2]\n"
      ".inst 0xa0428371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b831  // udot za.s[x9, 1], { z0.b-z3.b }, z9.b[2]\n"
      ".inst 0xc159bbb2  // udot za.s[x9, 2], { z28.b-z31.b }, z9.b[2]\n"
      ".inst 0xc159beb0  // udot za.s[x9, 0], { z20.b-z23.b }, z9.b[3]\n"
      ".inst 0xc159bf31  // udot za.s[x9, 1], { z24.b-z27.b }, z9.b[3]\n"
      ".inst 0xc159be32  // udot za.s[x9, 2], { z16.b-z19.b }, z9.b[3]\n"
      "tbnz %x[flags], #31, 28f\n"
      "udot z11.s, z9.b, z8.b\n"
      "28:"  // Width 3: Multiply loop: unique 5: skip row sum
      "sub x22, x22, #0x10\n"
      "cmp x22, #0x10\n"
      "bgt 27b\n"
      "29:"  // Width 3: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x22\n"
      ".inst 0xa040836d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      "ld1rqb { z9.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0418361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b1b0  // udot za.s[x9, 0], { z12.b-z15.b }, z9.b[0]\n"
      ".inst 0xc159b031  // udot za.s[x9, 1], { z0.b-z3.b }, z9.b[0]\n"
      ".inst 0xc159b3b2  // udot za.s[x9, 2], { z28.b-z31.b }, z9.b[0]\n"
      "ble 30f\n"
      ".inst 0xa0408365  // ldnt1b { z4.b-z7.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa0418379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa0428371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b4b0  // udot za.s[x9, 0], { z4.b-z7.b }, z9.b[1]\n"
      ".inst 0xc159b731  // udot za.s[x9, 1], { z24.b-z27.b }, z9.b[1]\n"
      ".inst 0xc159b632  // udot za.s[x9, 2], { z16.b-z19.b }, z9.b[1]\n"
      "ble 30f\n"
      ".inst 0xa040837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa0418361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa0428379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159bbb0  // udot za.s[x9, 0], { z28.b-z31.b }, z9.b[2]\n"
      ".inst 0xc159b831  // udot za.s[x9, 1], { z0.b-z3.b }, z9.b[2]\n"
      ".inst 0xc159bb32  // udot za.s[x9, 2], { z24.b-z27.b }, z9.b[2]\n"
      "ble 30f\n"
      ".inst 0xa0408361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27]\n"
      ".inst 0xa0418375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa0428371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159bc30  // udot za.s[x9, 0], { z0.b-z3.b }, z9.b[3]\n"
      ".inst 0xc159beb1  // udot za.s[x9, 1], { z20.b-z23.b }, z9.b[3]\n"
      ".inst 0xc159be32  // udot za.s[x9, 2], { z16.b-z19.b }, z9.b[3]\n"
      "30:"  // Width 3: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 31f\n"
      "udot z11.s, z9.b, z8.b\n"
      "31:"  // Width 3: Multiply loop: unique 6: skip row sum
      "tbnz %x[flags], #31, 32f\n"
      "add x21, %x[qp], %[b_offset]\n"
      "mov x20, #0x4\n"
      "ld1rw { z16.s }, p2/Z, [x21]\n"
      "whilelt p0.s, XZR, x20\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "uaddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "neg z16.s, p2/M, z16.s\n"
      "mul z11.s, p2/M, z11.s, z16.s\n"
      "32:"  // Width 3: skip row sum fixup
      ".inst 0xc0904960  // addha za0.s, p2/M, p2/M, z11.s\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      ".inst 0xc0904961  // addha za1.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z2.s }, p2/Z, [x21]\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[minval]\n"
      ".inst 0xc0904962  // addha za2.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z1.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0xc0904963  // addha za3.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z0.s }, p2/Z, [x22]\n"
      "ld1rw { z21.s }, p2/Z, [x21]\n"
      "ld1rw { z20.s }, p2/Z, [x20]\n"
      ".inst 0xc0062c1c  // mova { z28.d-z31.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c2c  // mova { z12.d-z15.d }, za.d[x9, #1]\n"
      ".inst 0xc0062c50  // mova { z16.d-z19.d }, za.d[x9, #2]\n"
      ".inst 0xc1a2ac1c  // sqdmulh { z28.s-z31.s }, { z28.s-z31.s }, z2.s\n"
      ".inst 0xc1a2ac0c  // sqdmulh { z12.s-z15.s }, { z12.s-z15.s }, z2.s\n"
      ".inst 0xc1a2ac10  // sqdmulh { z16.s-z19.s }, { z16.s-z19.s }, z2.s\n"
      ".inst 0xc1a1aa3c  // srshl { z28.s-z31.s }, { z28.s-z31.s }, z1.s\n"
      ".inst 0xc1a1aa2c  // srshl { z12.s-z15.s }, { z12.s-z15.s }, z1.s\n"
      ".inst 0xc1a1aa30  // srshl { z16.s-z19.s }, { z16.s-z19.s }, z1.s\n"
      ".inst 0xc1a0ab1c  // add { z28.s-z31.s }, { z28.s-z31.s }, z0.s\n"
      ".inst 0xc1a0ab0c  // add { z12.s-z15.s }, { z12.s-z15.s }, z0.s\n"
      ".inst 0xc1a0ab10  // add { z16.s-z19.s }, { z16.s-z19.s }, z0.s\n"
      ".inst 0xc1b4cebc  // sclamp { z28.s-z31.s }, z21.s, z20.s\n"
      ".inst 0xc1b4ceac  // sclamp { z12.s-z15.s }, z21.s, z20.s\n"
      ".inst 0xc1b4ceb0  // sclamp { z16.s-z19.s }, z21.s, z20.s\n"
      "uzp1 z28.h, z28.h, z29.h\n"
      "uzp1 z21.h, z30.h, z31.h\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z20.h, z14.h, z15.h\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z28.b, z28.b, z21.b\n"
      "uzp1 z12.b, z12.b, z20.b\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "st1b { z28.b }, p2, [x25]\n"
      "st1b { z12.b }, p2, [x25, #1, MUL VL]\n"
      "st1b { z16.b }, p1, [x25, #2, MUL VL]\n"
      "addvl x25, x25, #3\n"
      "b 44f\n"
      "34:"  // Width 4
      "mov x20, #0x3\n"
      "mov x23, %x[A_ptr]\n"
      "mov x21, %x[K]\n"
      "msub x20, x28, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      "whilelt p1.b, XZR, x20\n"
      "cbz x24, 35f\n"
      ".inst 0xa040c300  // ld1w { z0.s-z3.s }, pn8.b/Z, [x24]\n"
      ".inst 0xa041c30c  // ld1w { z12.s-z15.s }, pn8.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xa042c310  // ld1w { z16.s-z19.s }, pn8.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xa043c314  // ld1w { z20.s-z23.s }, pn8.b/Z, [x24, #0xc, MUL VL]\n"
      ".inst 0xc0042c00  // mova za.d[x9, #0], { z0.d-z3.d }\n"
      "addvl x24, x24, #16\n"
      ".inst 0xc0042d81  // mova za.d[x9, #1], { z12.d-z15.d }\n"
      ".inst 0xc0042e02  // mova za.d[x9, #2], { z16.d-z19.d }\n"
      ".inst 0xc0042e83  // mova za.d[x9, #3], { z20.d-z23.d }\n"
      "b 36f\n"
      "35:"  // Width 4: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "36:"  // Width 4: setup done
      "cmp x22, #0x10\n"
      "ble 39f\n"
      "37:"  // Width 4: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x22\n"
      ".inst 0xa0408361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27]\n"
      "ld1rqb { z9.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0418375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa0438379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc159b030  // udot za.s[x9, 0], { z0.b-z3.b }, z9.b[0]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b2b1  // udot za.s[x9, 1], { z20.b-z23.b }, z9.b[0]\n"
      ".inst 0xa0408361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27]\n"
      ".inst 0xc159b3b2  // udot za.s[x9, 2], { z28.b-z31.b }, z9.b[0]\n"
      ".inst 0xa041837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc159b333  // udot za.s[x9, 3], { z24.b-z27.b }, z9.b[0]\n"
      ".inst 0xa0428371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043836d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc159b430  // udot za.s[x9, 0], { z0.b-z3.b }, z9.b[1]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b7b1  // udot za.s[x9, 1], { z28.b-z31.b }, z9.b[1]\n"
      ".inst 0xa0408375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27]\n"
      ".inst 0xc159b632  // udot za.s[x9, 2], { z16.b-z19.b }, z9.b[1]\n"
      ".inst 0xa041837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc159b5b3  // udot za.s[x9, 3], { z12.b-z15.b }, z9.b[1]\n"
      ".inst 0xa0428361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043836d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc159bab0  // udot za.s[x9, 0], { z20.b-z23.b }, z9.b[2]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159bbb1  // udot za.s[x9, 1], { z28.b-z31.b }, z9.b[2]\n"
      ".inst 0xa0408379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27]\n"
      ".inst 0xc159b832  // udot za.s[x9, 2], { z0.b-z3.b }, z9.b[2]\n"
      ".inst 0xa0418375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc159b9b3  // udot za.s[x9, 3], { z12.b-z15.b }, z9.b[2]\n"
      ".inst 0xa0428361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa0438371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc159bf30  // udot za.s[x9, 0], { z24.b-z27.b }, z9.b[3]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159beb1  // udot za.s[x9, 1], { z20.b-z23.b }, z9.b[3]\n"
      ".inst 0xc159bc32  // udot za.s[x9, 2], { z0.b-z3.b }, z9.b[3]\n"
      ".inst 0xc159be33  // udot za.s[x9, 3], { z16.b-z19.b }, z9.b[3]\n"
      "tbnz %x[flags], #31, 38f\n"
      "udot z11.s, z9.b, z8.b\n"
      "38:"  // Width 4: Multiply loop: unique 7: skip row sum
      "sub x22, x22, #0x10\n"
      "cmp x22, #0x10\n"
      "bgt 37b\n"
      "39:"  // Width 4: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x22\n"
      ".inst 0xa0408361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      "ld1rqb { z9.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa0418375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa0438371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc159b030  // udot za.s[x9, 0], { z0.b-z3.b }, z9.b[0]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b2b1  // udot za.s[x9, 1], { z20.b-z23.b }, z9.b[0]\n"
      ".inst 0xc159b3b2  // udot za.s[x9, 2], { z28.b-z31.b }, z9.b[0]\n"
      ".inst 0xc159b233  // udot za.s[x9, 3], { z16.b-z19.b }, z9.b[0]\n"
      "ble 40f\n"
      ".inst 0xa0408361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa0418379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa0438371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc159b430  // udot za.s[x9, 0], { z0.b-z3.b }, z9.b[1]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b731  // udot za.s[x9, 1], { z24.b-z27.b }, z9.b[1]\n"
      ".inst 0xc159b7b2  // udot za.s[x9, 2], { z28.b-z31.b }, z9.b[1]\n"
      ".inst 0xc159b633  // udot za.s[x9, 3], { z16.b-z19.b }, z9.b[1]\n"
      "ble 40f\n"
      ".inst 0xa0408379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27]\n"
      "subs x22, x22, #0x4\n"
      ".inst 0xa0418361  // ldnt1b { z0.b-z3.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa0428375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043836d  // ldnt1b { z12.b-z15.b }, pn8.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc159bb30  // udot za.s[x9, 0], { z24.b-z27.b }, z9.b[2]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159b831  // udot za.s[x9, 1], { z0.b-z3.b }, z9.b[2]\n"
      ".inst 0xc159bab2  // udot za.s[x9, 2], { z20.b-z23.b }, z9.b[2]\n"
      ".inst 0xc159b9b3  // udot za.s[x9, 3], { z12.b-z15.b }, z9.b[2]\n"
      "ble 40f\n"
      ".inst 0xa0408379  // ldnt1b { z24.b-z27.b }, pn8.b/Z, [x27]\n"
      ".inst 0xa041837d  // ldnt1b { z28.b-z31.b }, pn8.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa0428375  // ldnt1b { z20.b-z23.b }, pn8.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa0438371  // ldnt1b { z16.b-z19.b }, pn8.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc159bf30  // udot za.s[x9, 0], { z24.b-z27.b }, z9.b[3]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc159bfb1  // udot za.s[x9, 1], { z28.b-z31.b }, z9.b[3]\n"
      ".inst 0xc159beb2  // udot za.s[x9, 2], { z20.b-z23.b }, z9.b[3]\n"
      ".inst 0xc159be33  // udot za.s[x9, 3], { z16.b-z19.b }, z9.b[3]\n"
      "40:"  // Width 4: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 41f\n"
      "udot z11.s, z9.b, z8.b\n"
      "41:"  // Width 4: Multiply loop: unique 8: skip row sum
      "tbnz %x[flags], #31, 42f\n"
      "add x21, %x[qp], %[b_offset]\n"
      "mov x20, #0x4\n"
      "ld1rw { z16.s }, p2/Z, [x21]\n"
      "whilelt p0.s, XZR, x20\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "uaddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "neg z16.s, p2/M, z16.s\n"
      "mul z11.s, p2/M, z11.s, z16.s\n"
      "42:"  // Width 4: skip row sum fixup
      ".inst 0xc0904960  // addha za0.s, p2/M, p2/M, z11.s\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      ".inst 0xc0904961  // addha za1.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z10.s }, p2/Z, [x21]\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[minval]\n"
      ".inst 0xc0904962  // addha za2.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z12.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      ".inst 0xc0904963  // addha za3.s, p2/M, p2/M, z11.s\n"
      "ld1rw { z13.s }, p2/Z, [x22]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z18.s }, p2/Z, [x20]\n"
      ".inst 0xc0062c1c  // mova { z28.d-z31.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c20  // mova { z0.d-z3.d }, za.d[x9, #1]\n"
      ".inst 0xc0062c58  // mova { z24.d-z27.d }, za.d[x9, #2]\n"
      ".inst 0xc0062c74  // mova { z20.d-z23.d }, za.d[x9, #3]\n"
      ".inst 0xc1aaac1c  // sqdmulh { z28.s-z31.s }, { z28.s-z31.s }, z10.s\n"
      ".inst 0xc1aaac00  // sqdmulh { z0.s-z3.s }, { z0.s-z3.s }, z10.s\n"
      ".inst 0xc1aaac18  // sqdmulh { z24.s-z27.s }, { z24.s-z27.s }, z10.s\n"
      ".inst 0xc1aaac14  // sqdmulh { z20.s-z23.s }, { z20.s-z23.s }, z10.s\n"
      ".inst 0xc1acaa3c  // srshl { z28.s-z31.s }, { z28.s-z31.s }, z12.s\n"
      ".inst 0xc1acaa20  // srshl { z0.s-z3.s }, { z0.s-z3.s }, z12.s\n"
      ".inst 0xc1acaa38  // srshl { z24.s-z27.s }, { z24.s-z27.s }, z12.s\n"
      ".inst 0xc1acaa34  // srshl { z20.s-z23.s }, { z20.s-z23.s }, z12.s\n"
      ".inst 0xc1adab1c  // add { z28.s-z31.s }, { z28.s-z31.s }, z13.s\n"
      ".inst 0xc1adab00  // add { z0.s-z3.s }, { z0.s-z3.s }, z13.s\n"
      ".inst 0xc1adab18  // add { z24.s-z27.s }, { z24.s-z27.s }, z13.s\n"
      ".inst 0xc1adab14  // add { z20.s-z23.s }, { z20.s-z23.s }, z13.s\n"
      ".inst 0xc1b2ce3c  // sclamp { z28.s-z31.s }, z17.s, z18.s\n"
      ".inst 0xc1b2ce20  // sclamp { z0.s-z3.s }, z17.s, z18.s\n"
      ".inst 0xc1b2ce38  // sclamp { z24.s-z27.s }, z17.s, z18.s\n"
      ".inst 0xc1b2ce34  // sclamp { z20.s-z23.s }, z17.s, z18.s\n"
      "uzp1 z28.h, z28.h, z29.h\n"
      "uzp1 z14.h, z30.h, z31.h\n"
      "uzp1 z0.h, z0.h, z1.h\n"
      "uzp1 z18.h, z2.h, z3.h\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "uzp1 z17.h, z26.h, z27.h\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z16.h, z22.h, z23.h\n"
      "uzp1 z28.b, z28.b, z14.b\n"
      "uzp1 z0.b, z0.b, z18.b\n"
      "uzp1 z24.b, z24.b, z17.b\n"
      "uzp1 z20.b, z20.b, z16.b\n"
      "st1b { z28.b }, p2, [x25]\n"
      "st1b { z0.b }, p2, [x25, #1, MUL VL]\n"
      "st1b { z24.b }, p2, [x25, #2, MUL VL]\n"
      "st1b { z20.b }, p1, [x25, #3, MUL VL]\n"
      "addvl x25, x25, #4\n"
      "subs x26, x26, #0x4\n"
      "sub %x[N], %x[N], x28, LSL #2\n"
      "bgt 4b\n"
      "44:"  // Exit
      ".inst 0xd503467f  // SMSTOP\n"
      "ptrue p8.b\n"
      : [N] "+&r" (N), [flags] "+&r" (flags)
      : [A_ptr] "r" (A_ptr), [B_ptr] "r" (B_ptr), [K] "r" (K), [b_offset] "I" (offsetof(Requantize32, b_offset)), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [output_ptr] "r" (output_ptr), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x9", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
