/*
 * Copyright (c) 2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include <cassert>
#include <cstdint>

namespace arm_gemm {

void a64_hybrid_u8s8qa_mmla_4x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<uint8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<uint8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 94f\n"
      "cmp %x[M], #0x2\n"
      "bgt 63f\n"
      "beq 32f\n"
      "movi v11.4s, #0x0\n"
      "movi v15.16b, #0x1\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x28, %x[col_bias]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "mov x26, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "cbnz x26, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x24, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x25, #0x10\n"
      "blt 11f\n"
      "ldr q1, [x24, #0x0]\n"
      "ldr q5, [x9, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q6, [x9, #0x10]\n"
      "ldr q7, [x9, #0x20]\n"
      "ldr q8, [x9, #0x30]\n"
      "ldr q9, [x9, #0x40]\n"
      "ldr q10, [x9, #0x50]\n"
      "ldr q4, [x9, #0x60]\n"
      "blt 9f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      "add x24, x24, #0x10\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x4e85ac10  // usmmla v16.4s, v0.16b, v5.16b\n"
      "ldr q5, [x9, #0x70]\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e86ac14  // usmmla v20.4s, v0.16b, v6.16b\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4e87ac11  // usmmla v17.4s, v0.16b, v7.16b\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4e88ac15  // usmmla v21.4s, v0.16b, v8.16b\n"
      "ldr q8, [x9, #0xa0]\n"
      ".inst 0x4e89ac12  // usmmla v18.4s, v0.16b, v9.16b\n"
      "ldr q9, [x9, #0xb0]\n"
      ".inst 0x4e8aac16  // usmmla v22.4s, v0.16b, v10.16b\n"
      "ldr q10, [x9, #0xc0]\n"
      ".inst 0x4e84ac13  // usmmla v19.4s, v0.16b, v4.16b\n"
      "ldr q4, [x9, #0xd0]\n"
      ".inst 0x4e85ac17  // usmmla v23.4s, v0.16b, v5.16b\n"
      "ldr q5, [x9, #0xe0]\n"
      ".inst 0x4e86ac30  // usmmla v16.4s, v1.16b, v6.16b\n"
      "ldr q6, [x9, #0xf0]\n"
      ".inst 0x4e87ac34  // usmmla v20.4s, v1.16b, v7.16b\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e88ac31  // usmmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e89ac35  // usmmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e8aac32  // usmmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e84ac36  // usmmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e85ac33  // usmmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e86ac37  // usmmla v23.4s, v1.16b, v6.16b\n"
      "tbnz %x[flags], #31, 8f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f942b  // udot v11.4s, v1.16b, v15.16b\n"
      "8:"  // Height 1: Multiply loop: unique 1: skip row sum
      "ldr q1, [x24, #0x0]\n"
      "ldr q5, [x9, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q6, [x9, #0x10]\n"
      "ldr q7, [x9, #0x20]\n"
      "cmp x25, #0x20\n"
      "ldr q8, [x9, #0x30]\n"
      "ldr q9, [x9, #0x40]\n"
      "ldr q10, [x9, #0x50]\n"
      "ldr q4, [x9, #0x60]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "bge 7b\n"
      "9:"  // Height 1: Multiply loop: Single iteration only
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e85ac10  // usmmla v16.4s, v0.16b, v5.16b\n"
      "ldr q5, [x9, #0x70]\n"
      ".inst 0x4e86ac14  // usmmla v20.4s, v0.16b, v6.16b\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4e87ac11  // usmmla v17.4s, v0.16b, v7.16b\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4e88ac15  // usmmla v21.4s, v0.16b, v8.16b\n"
      "ldr q8, [x9, #0xa0]\n"
      ".inst 0x4e89ac12  // usmmla v18.4s, v0.16b, v9.16b\n"
      "ldr q9, [x9, #0xb0]\n"
      ".inst 0x4e8aac16  // usmmla v22.4s, v0.16b, v10.16b\n"
      "ldr q10, [x9, #0xc0]\n"
      ".inst 0x4e84ac13  // usmmla v19.4s, v0.16b, v4.16b\n"
      "ldr q4, [x9, #0xd0]\n"
      ".inst 0x4e85ac17  // usmmla v23.4s, v0.16b, v5.16b\n"
      "ldr q5, [x9, #0xe0]\n"
      ".inst 0x4e86ac30  // usmmla v16.4s, v1.16b, v6.16b\n"
      "ldr q6, [x9, #0xf0]\n"
      ".inst 0x4e87ac34  // usmmla v20.4s, v1.16b, v7.16b\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e88ac31  // usmmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e89ac35  // usmmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e8aac32  // usmmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e84ac36  // usmmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e85ac33  // usmmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e86ac37  // usmmla v23.4s, v1.16b, v6.16b\n"
      "tbnz %x[flags], #31, 10f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f942b  // udot v11.4s, v1.16b, v15.16b\n"
      "10:"  // Height 1: Multiply loop: unique 2: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "11:"  // Height 1: Multiply loop: Main loop skip
      "cbz x25, 20f\n"
      "cmp x25, #0x8\n"
      "blt 14f\n"
      "12:"  // Height 1: Multiply loop: Odd block loop
      "ldr d1, [x24], #0x8\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "tbnz %x[flags], #31, 13f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      "13:"  // Height 1: Multiply loop: unique 3: skip row sum
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "sub x25, x25, #0x8\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q4, [x9, #0x30]\n"
      "cmp x25, #0x8\n"
      "ldr q5, [x9, #0x40]\n"
      "ldr q6, [x9, #0x50]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4e88ac10  // usmmla v16.4s, v0.16b, v8.16b\n"
      "ldr q8, [x9, #0x70]\n"
      ".inst 0x4e89ac14  // usmmla v20.4s, v0.16b, v9.16b\n"
      ".inst 0x4e8aac11  // usmmla v17.4s, v0.16b, v10.16b\n"
      ".inst 0x4e84ac15  // usmmla v21.4s, v0.16b, v4.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e85ac12  // usmmla v18.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86ac16  // usmmla v22.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87ac13  // usmmla v19.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88ac17  // usmmla v23.4s, v0.16b, v8.16b\n"
      "bge 12b\n"
      "14:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x25, 20f\n"
      "tbz x25, #2, 16f\n"
      "ldr s1, [x24], #0x4\n"
      "tbz x25, #1, 15f\n"
      "ld1 { v1.h }[2], [x24], #0x2\n"
      "tbz x25, #0, 18f\n"
      "ld1 { v1.b }[6], [x24]\n"
      "b 18f\n"
      "15:"  // Height 1: Multiply loop: Ragged operand read: partial_1_4
      "tbz x25, #0, 18f\n"
      "ld1 { v1.b }[4], [x24]\n"
      "b 18f\n"
      "16:"  // Height 1: Multiply loop: Ragged operand read: partial_2_0
      "tbz x25, #1, 17f\n"
      "ldr h1, [x24], #0x2\n"
      "tbz x25, #0, 18f\n"
      "ld1 { v1.b }[2], [x24]\n"
      "b 18f\n"
      "17:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x24, #0x0]\n"
      "18:"  // Height 1: Multiply loop: Ragged operand read: Done
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "tbnz %x[flags], #31, 19f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      "19:"  // Height 1: Multiply loop: unique 4: skip row sum
      "ldr q10, [x9, #0x0]\n"
      "ldr q4, [x9, #0x10]\n"
      "ldr q5, [x9, #0x20]\n"
      "ldr q6, [x9, #0x30]\n"
      "ldr q7, [x9, #0x40]\n"
      "ldr q8, [x9, #0x50]\n"
      "ldr q9, [x9, #0x60]\n"
      ".inst 0x4e8aac10  // usmmla v16.4s, v0.16b, v10.16b\n"
      "ldr q10, [x9, #0x70]\n"
      ".inst 0x4e84ac14  // usmmla v20.4s, v0.16b, v4.16b\n"
      ".inst 0x4e85ac11  // usmmla v17.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86ac15  // usmmla v21.4s, v0.16b, v6.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e87ac12  // usmmla v18.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88ac16  // usmmla v22.4s, v0.16b, v8.16b\n"
      ".inst 0x4e89ac13  // usmmla v19.4s, v0.16b, v9.16b\n"
      ".inst 0x4e8aac17  // usmmla v23.4s, v0.16b, v10.16b\n"
      "20:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 4b\n"
      "uzp1 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v19.2d, v19.2d, v23.2d\n"
      "mov v23.16b, v16.16b\n"
      "tbnz %x[flags], #31, 21f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "ld1r { v1.4s }, [x20]\n"
      "neg v1.4s, v1.4s\n"
      "dup v11.4s, v11.s[0]\n"
      "mul v11.4s, v11.4s, v1.4s\n"
      "21:"  // Height 1: skip row sum fixup
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x28, #0x10]\n"
      "add v23.4s, v23.4s, v11.4s\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q2, [x28, #0x20]\n"
      "ldr q3, [x28, #0x30]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v4.4s }, [x21]\n"
      "add v23.4s, v23.4s, v0.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add x21, %x[qp], %[c_offset]\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v6.4s }, [x20]\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "cmp x10, #0x10\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "sqdmulh v17.4s, v17.4s, v4.4s\n"
      "add x28, x28, #0x40\n"
      "sqdmulh v18.4s, v18.4s, v4.4s\n"
      "sqdmulh v19.4s, v19.4s, v4.4s\n"
      "ld1r { v4.4s }, [x21]\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "uzp1 v23.8h, v23.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.16b, v23.16b, v17.16b\n"
      "bge 30f\n"
      "tbz x10, #3, 25f\n"
      "str d23, [x27], #0x8\n"
      "tbz x10, #2, 23f\n"
      "st1 { v23.s }[2], [x27], #0x4\n"
      "tbz x10, #1, 22f\n"
      "st1 { v23.h }[6], [x27], #0x2\n"
      "tbz x10, #0, 29f\n"
      "st1 { v23.b }[14], [x27]\n"
      "b 29f\n"
      "22:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x10, #0, 29f\n"
      "st1 { v23.b }[12], [x27]\n"
      "b 29f\n"
      "23:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x10, #1, 24f\n"
      "st1 { v23.h }[4], [x27], #0x2\n"
      "tbz x10, #0, 29f\n"
      "st1 { v23.b }[10], [x27]\n"
      "b 29f\n"
      "24:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x10, #0, 29f\n"
      "st1 { v23.b }[8], [x27]\n"
      "b 29f\n"
      "25:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x10, #2, 27f\n"
      "str s23, [x27], #0x4\n"
      "tbz x10, #1, 26f\n"
      "st1 { v23.h }[2], [x27], #0x2\n"
      "tbz x10, #0, 29f\n"
      "st1 { v23.b }[6], [x27]\n"
      "b 29f\n"
      "26:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x10, #0, 29f\n"
      "st1 { v23.b }[4], [x27]\n"
      "b 29f\n"
      "27:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x10, #1, 28f\n"
      "str h23, [x27], #0x2\n"
      "tbz x10, #0, 29f\n"
      "st1 { v23.b }[2], [x27]\n"
      "b 29f\n"
      "28:"  // Height 1: Partial direct writeback: partial_1_0
      "str b23, [x27, #0x0]\n"
      "29:"  // Height 1: Partial direct writeback: Done
      "b 31f\n"
      "30:"  // Height 1: Full writeback
      "str q23, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "31:"  // Height 1: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 2b\n"
      "b 126f\n"
      "32:"  // Height 2
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "movi v15.16b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x28, %x[col_bias]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "33:"  // Height 2: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "mov x26, #0x0\n"
      "35:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 36f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "cbnz x26, 37f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 37f\n"
      "36:"  // Height 2: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "37:"  // Height 2: input setup done
      "cmp x25, #0x10\n"
      "blt 42f\n"
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q5, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "ldr q7, [x9, #0x20]\n"
      "ldr q8, [x9, #0x30]\n"
      "ldr q9, [x9, #0x40]\n"
      "ldr q10, [x9, #0x50]\n"
      "ldr q4, [x9, #0x60]\n"
      "blt 40f\n"
      "38:"  // Height 2: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e85ac10  // usmmla v16.4s, v0.16b, v5.16b\n"
      "ldr q5, [x9, #0x70]\n"
      ".inst 0x4e86ac14  // usmmla v20.4s, v0.16b, v6.16b\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4e87ac11  // usmmla v17.4s, v0.16b, v7.16b\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4e88ac15  // usmmla v21.4s, v0.16b, v8.16b\n"
      "ldr q8, [x9, #0xa0]\n"
      ".inst 0x4e89ac12  // usmmla v18.4s, v0.16b, v9.16b\n"
      "ldr q9, [x9, #0xb0]\n"
      ".inst 0x4e8aac16  // usmmla v22.4s, v0.16b, v10.16b\n"
      "ldr q10, [x9, #0xc0]\n"
      ".inst 0x4e84ac13  // usmmla v19.4s, v0.16b, v4.16b\n"
      "ldr q4, [x9, #0xd0]\n"
      ".inst 0x4e85ac17  // usmmla v23.4s, v0.16b, v5.16b\n"
      "ldr q5, [x9, #0xe0]\n"
      ".inst 0x4e86ac30  // usmmla v16.4s, v1.16b, v6.16b\n"
      "ldr q6, [x9, #0xf0]\n"
      ".inst 0x4e87ac34  // usmmla v20.4s, v1.16b, v7.16b\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e88ac31  // usmmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e89ac35  // usmmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e8aac32  // usmmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e84ac36  // usmmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e85ac33  // usmmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e86ac37  // usmmla v23.4s, v1.16b, v6.16b\n"
      "tbnz %x[flags], #31, 39f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f942b  // udot v11.4s, v1.16b, v15.16b\n"
      "39:"  // Height 2: Multiply loop: unique 5: skip row sum
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q5, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "cmp x25, #0x20\n"
      "ldr q7, [x9, #0x20]\n"
      "ldr q8, [x9, #0x30]\n"
      "ldr q9, [x9, #0x40]\n"
      "ldr q10, [x9, #0x50]\n"
      "ldr q4, [x9, #0x60]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "bge 38b\n"
      "40:"  // Height 2: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e85ac10  // usmmla v16.4s, v0.16b, v5.16b\n"
      "ldr q5, [x9, #0x70]\n"
      ".inst 0x4e86ac14  // usmmla v20.4s, v0.16b, v6.16b\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4e87ac11  // usmmla v17.4s, v0.16b, v7.16b\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4e88ac15  // usmmla v21.4s, v0.16b, v8.16b\n"
      "ldr q8, [x9, #0xa0]\n"
      ".inst 0x4e89ac12  // usmmla v18.4s, v0.16b, v9.16b\n"
      "ldr q9, [x9, #0xb0]\n"
      ".inst 0x4e8aac16  // usmmla v22.4s, v0.16b, v10.16b\n"
      "ldr q10, [x9, #0xc0]\n"
      ".inst 0x4e84ac13  // usmmla v19.4s, v0.16b, v4.16b\n"
      "ldr q4, [x9, #0xd0]\n"
      ".inst 0x4e85ac17  // usmmla v23.4s, v0.16b, v5.16b\n"
      "ldr q5, [x9, #0xe0]\n"
      ".inst 0x4e86ac30  // usmmla v16.4s, v1.16b, v6.16b\n"
      "ldr q6, [x9, #0xf0]\n"
      ".inst 0x4e87ac34  // usmmla v20.4s, v1.16b, v7.16b\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e88ac31  // usmmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e89ac35  // usmmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e8aac32  // usmmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e84ac36  // usmmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e85ac33  // usmmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e86ac37  // usmmla v23.4s, v1.16b, v6.16b\n"
      "tbnz %x[flags], #31, 41f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f942b  // udot v11.4s, v1.16b, v15.16b\n"
      "41:"  // Height 2: Multiply loop: unique 6: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "42:"  // Height 2: Multiply loop: Main loop skip
      "cbz x25, 51f\n"
      "cmp x25, #0x8\n"
      "blt 45f\n"
      "43:"  // Height 2: Multiply loop: Odd block loop
      "ldr d1, [x24], #0x8\n"
      "ldr d2, [x23], #0x8\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "tbnz %x[flags], #31, 44f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      "44:"  // Height 2: Multiply loop: unique 7: skip row sum
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "sub x25, x25, #0x8\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q4, [x9, #0x30]\n"
      "cmp x25, #0x8\n"
      "ldr q5, [x9, #0x40]\n"
      "ldr q6, [x9, #0x50]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4e88ac10  // usmmla v16.4s, v0.16b, v8.16b\n"
      "ldr q8, [x9, #0x70]\n"
      ".inst 0x4e89ac14  // usmmla v20.4s, v0.16b, v9.16b\n"
      ".inst 0x4e8aac11  // usmmla v17.4s, v0.16b, v10.16b\n"
      ".inst 0x4e84ac15  // usmmla v21.4s, v0.16b, v4.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e85ac12  // usmmla v18.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86ac16  // usmmla v22.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87ac13  // usmmla v19.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88ac17  // usmmla v23.4s, v0.16b, v8.16b\n"
      "bge 43b\n"
      "45:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x25, 51f\n"
      "tbz x25, #2, 47f\n"
      "ldr s1, [x24], #0x4\n"
      "ldr s2, [x23], #0x4\n"
      "tbz x25, #1, 46f\n"
      "ld1 { v1.h }[2], [x24], #0x2\n"
      "ld1 { v2.h }[2], [x23], #0x2\n"
      "tbz x25, #0, 49f\n"
      "ld1 { v1.b }[6], [x24]\n"
      "ld1 { v2.b }[6], [x23]\n"
      "b 49f\n"
      "46:"  // Height 2: Multiply loop: Ragged operand read: partial_1_4
      "tbz x25, #0, 49f\n"
      "ld1 { v1.b }[4], [x24]\n"
      "ld1 { v2.b }[4], [x23]\n"
      "b 49f\n"
      "47:"  // Height 2: Multiply loop: Ragged operand read: partial_2_0
      "tbz x25, #1, 48f\n"
      "ldr h1, [x24], #0x2\n"
      "ldr h2, [x23], #0x2\n"
      "tbz x25, #0, 49f\n"
      "ld1 { v1.b }[2], [x24]\n"
      "ld1 { v2.b }[2], [x23]\n"
      "b 49f\n"
      "48:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x24, #0x0]\n"
      "ldr b2, [x23, #0x0]\n"
      "49:"  // Height 2: Multiply loop: Ragged operand read: Done
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "tbnz %x[flags], #31, 50f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      "50:"  // Height 2: Multiply loop: unique 8: skip row sum
      "ldr q10, [x9, #0x0]\n"
      "ldr q4, [x9, #0x10]\n"
      "ldr q5, [x9, #0x20]\n"
      "ldr q6, [x9, #0x30]\n"
      "ldr q7, [x9, #0x40]\n"
      "ldr q8, [x9, #0x50]\n"
      "ldr q9, [x9, #0x60]\n"
      ".inst 0x4e8aac10  // usmmla v16.4s, v0.16b, v10.16b\n"
      "ldr q10, [x9, #0x70]\n"
      ".inst 0x4e84ac14  // usmmla v20.4s, v0.16b, v4.16b\n"
      ".inst 0x4e85ac11  // usmmla v17.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86ac15  // usmmla v21.4s, v0.16b, v6.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e87ac12  // usmmla v18.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88ac16  // usmmla v22.4s, v0.16b, v8.16b\n"
      ".inst 0x4e89ac13  // usmmla v19.4s, v0.16b, v9.16b\n"
      ".inst 0x4e8aac17  // usmmla v23.4s, v0.16b, v10.16b\n"
      "51:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 35b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v4.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x26, x27, x20\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "mov v23.16b, v4.16b\n"
      "tbnz %x[flags], #31, 52f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "ld1r { v2.4s }, [x20]\n"
      "neg v2.4s, v2.4s\n"
      "dup v12.4s, v11.s[3]\n"
      "dup v11.4s, v11.s[0]\n"
      "mul v11.4s, v11.4s, v2.4s\n"
      "mul v12.4s, v12.4s, v2.4s\n"
      "52:"  // Height 2: skip row sum fixup
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x28, #0x10]\n"
      "add v23.4s, v23.4s, v11.4s\n"
      "add v20.4s, v20.4s, v11.4s\n"
      "ldr q2, [x28, #0x20]\n"
      "ldr q3, [x28, #0x30]\n"
      "add v21.4s, v21.4s, v11.4s\n"
      "add v22.4s, v22.4s, v11.4s\n"
      "add v16.4s, v16.4s, v12.4s\n"
      "add v17.4s, v17.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "add x23, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v4.4s }, [x20]\n"
      "add v18.4s, v18.4s, v12.4s\n"
      "add v19.4s, v19.4s, v12.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add v23.4s, v23.4s, v0.4s\n"
      "add v20.4s, v20.4s, v1.4s\n"
      "add x21, %x[qp], %[maxval]\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v6.4s }, [x21]\n"
      "ld1r { v5.4s }, [x20]\n"
      "add v21.4s, v21.4s, v2.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "ld1r { v0.4s }, [x23]\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "cmp x10, #0x10\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "add x28, x28, #0x40\n"
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "sqdmulh v21.4s, v21.4s, v4.4s\n"
      "sqdmulh v22.4s, v22.4s, v4.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v4.4s\n"
      "sqdmulh v18.4s, v18.4s, v4.4s\n"
      "sqdmulh v19.4s, v19.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "uzp1 v23.8h, v23.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.16b, v23.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 61f\n"
      "tbz x10, #3, 56f\n"
      "str d23, [x27], #0x8\n"
      "str d16, [x26], #0x8\n"
      "tbz x10, #2, 54f\n"
      "st1 { v23.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x26], #0x4\n"
      "tbz x10, #1, 53f\n"
      "st1 { v23.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x26], #0x2\n"
      "tbz x10, #0, 60f\n"
      "st1 { v23.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x26]\n"
      "b 60f\n"
      "53:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x10, #0, 60f\n"
      "st1 { v23.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x26]\n"
      "b 60f\n"
      "54:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x10, #1, 55f\n"
      "st1 { v23.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x26], #0x2\n"
      "tbz x10, #0, 60f\n"
      "st1 { v23.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x26]\n"
      "b 60f\n"
      "55:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x10, #0, 60f\n"
      "st1 { v23.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x26]\n"
      "b 60f\n"
      "56:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x10, #2, 58f\n"
      "str s23, [x27], #0x4\n"
      "str s16, [x26], #0x4\n"
      "tbz x10, #1, 57f\n"
      "st1 { v23.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x26], #0x2\n"
      "tbz x10, #0, 60f\n"
      "st1 { v23.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x26]\n"
      "b 60f\n"
      "57:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x10, #0, 60f\n"
      "st1 { v23.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x26]\n"
      "b 60f\n"
      "58:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x10, #1, 59f\n"
      "str h23, [x27], #0x2\n"
      "str h16, [x26], #0x2\n"
      "tbz x10, #0, 60f\n"
      "st1 { v23.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x26]\n"
      "b 60f\n"
      "59:"  // Height 2: Partial direct writeback: partial_1_0
      "str b23, [x27, #0x0]\n"
      "str b16, [x26, #0x0]\n"
      "60:"  // Height 2: Partial direct writeback: Done
      "b 62f\n"
      "61:"  // Height 2: Full writeback
      "str q23, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q16, [x26, #0x0]\n"
      "62:"  // Height 2: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 33b\n"
      "b 126f\n"
      "63:"  // Height 3
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "movi v13.4s, #0x0\n"
      "movi v15.16b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x28, %x[col_bias]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "64:"  // Height 3: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "mov x26, #0x0\n"
      "66:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 67f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "cbnz x26, 68f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 68f\n"
      "67:"  // Height 3: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "68:"  // Height 3: input setup done
      "cmp x25, #0x10\n"
      "blt 73f\n"
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q3, [x22, #0x0]\n"
      "ldr q5, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "ldr q7, [x9, #0x20]\n"
      "ldr q8, [x9, #0x30]\n"
      "ldr q9, [x9, #0x40]\n"
      "ldr q10, [x9, #0x50]\n"
      "blt 71f\n"
      "69:"  // Height 3: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "ldr q4, [x9, #0x60]\n"
      ".inst 0x4e85ac10  // usmmla v16.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86ac14  // usmmla v20.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87ac11  // usmmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88ac15  // usmmla v21.4s, v0.16b, v8.16b\n"
      ".inst 0x4e85ac58  // usmmla v24.4s, v2.16b, v5.16b\n"
      "ldr q5, [x9, #0x70]\n"
      ".inst 0x4e86ac5c  // usmmla v28.4s, v2.16b, v6.16b\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4e89ac12  // usmmla v18.4s, v0.16b, v9.16b\n"
      ".inst 0x4e87ac59  // usmmla v25.4s, v2.16b, v7.16b\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4e88ac5d  // usmmla v29.4s, v2.16b, v8.16b\n"
      "ldr q8, [x9, #0xa0]\n"
      ".inst 0x4e89ac5a  // usmmla v26.4s, v2.16b, v9.16b\n"
      "ldr q9, [x9, #0xb0]\n"
      ".inst 0x4e8aac16  // usmmla v22.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac5e  // usmmla v30.4s, v2.16b, v10.16b\n"
      "ldr q10, [x9, #0xc0]\n"
      ".inst 0x4e84ac13  // usmmla v19.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84ac5b  // usmmla v27.4s, v2.16b, v4.16b\n"
      "ldr q4, [x9, #0xd0]\n"
      ".inst 0x4e85ac17  // usmmla v23.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85ac5f  // usmmla v31.4s, v2.16b, v5.16b\n"
      "ldr q5, [x9, #0xe0]\n"
      ".inst 0x4e86ac30  // usmmla v16.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86ac78  // usmmla v24.4s, v3.16b, v6.16b\n"
      "ldr q6, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e87ac34  // usmmla v20.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87ac7c  // usmmla v28.4s, v3.16b, v7.16b\n"
      ".inst 0x4e88ac31  // usmmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e88ac79  // usmmla v25.4s, v3.16b, v8.16b\n"
      ".inst 0x4e89ac35  // usmmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e89ac7d  // usmmla v29.4s, v3.16b, v9.16b\n"
      ".inst 0x4e8aac32  // usmmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e8aac7a  // usmmla v26.4s, v3.16b, v10.16b\n"
      ".inst 0x4e84ac36  // usmmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84ac7e  // usmmla v30.4s, v3.16b, v4.16b\n"
      ".inst 0x4e85ac33  // usmmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e85ac7b  // usmmla v27.4s, v3.16b, v5.16b\n"
      ".inst 0x4e86ac37  // usmmla v23.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86ac7f  // usmmla v31.4s, v3.16b, v6.16b\n"
      "tbnz %x[flags], #31, 70f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f944d  // udot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x6e8f942b  // udot v11.4s, v1.16b, v15.16b\n"
      ".inst 0x6e8f946d  // udot v13.4s, v3.16b, v15.16b\n"
      "70:"  // Height 3: Multiply loop: unique 9: skip row sum
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q3, [x22, #0x0]\n"
      "ldr q5, [x9, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q6, [x9, #0x10]\n"
      "ldr q7, [x9, #0x20]\n"
      "ldr q8, [x9, #0x30]\n"
      "ldr q9, [x9, #0x40]\n"
      "ldr q10, [x9, #0x50]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "bge 69b\n"
      "71:"  // Height 3: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "ldr q4, [x9, #0x60]\n"
      ".inst 0x4e85ac10  // usmmla v16.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86ac14  // usmmla v20.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87ac11  // usmmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88ac15  // usmmla v21.4s, v0.16b, v8.16b\n"
      ".inst 0x4e85ac58  // usmmla v24.4s, v2.16b, v5.16b\n"
      "ldr q5, [x9, #0x70]\n"
      ".inst 0x4e86ac5c  // usmmla v28.4s, v2.16b, v6.16b\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4e89ac12  // usmmla v18.4s, v0.16b, v9.16b\n"
      ".inst 0x4e87ac59  // usmmla v25.4s, v2.16b, v7.16b\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4e88ac5d  // usmmla v29.4s, v2.16b, v8.16b\n"
      "ldr q8, [x9, #0xa0]\n"
      ".inst 0x4e89ac5a  // usmmla v26.4s, v2.16b, v9.16b\n"
      "ldr q9, [x9, #0xb0]\n"
      ".inst 0x4e8aac16  // usmmla v22.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac5e  // usmmla v30.4s, v2.16b, v10.16b\n"
      "ldr q10, [x9, #0xc0]\n"
      ".inst 0x4e84ac13  // usmmla v19.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84ac5b  // usmmla v27.4s, v2.16b, v4.16b\n"
      "ldr q4, [x9, #0xd0]\n"
      ".inst 0x4e85ac17  // usmmla v23.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85ac5f  // usmmla v31.4s, v2.16b, v5.16b\n"
      "ldr q5, [x9, #0xe0]\n"
      ".inst 0x4e86ac30  // usmmla v16.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86ac78  // usmmla v24.4s, v3.16b, v6.16b\n"
      "ldr q6, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e87ac34  // usmmla v20.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87ac7c  // usmmla v28.4s, v3.16b, v7.16b\n"
      ".inst 0x4e88ac31  // usmmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e88ac79  // usmmla v25.4s, v3.16b, v8.16b\n"
      ".inst 0x4e89ac35  // usmmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e89ac7d  // usmmla v29.4s, v3.16b, v9.16b\n"
      ".inst 0x4e8aac32  // usmmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e8aac7a  // usmmla v26.4s, v3.16b, v10.16b\n"
      ".inst 0x4e84ac36  // usmmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84ac7e  // usmmla v30.4s, v3.16b, v4.16b\n"
      ".inst 0x4e85ac33  // usmmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e85ac7b  // usmmla v27.4s, v3.16b, v5.16b\n"
      ".inst 0x4e86ac37  // usmmla v23.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86ac7f  // usmmla v31.4s, v3.16b, v6.16b\n"
      "tbnz %x[flags], #31, 72f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f944d  // udot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x6e8f942b  // udot v11.4s, v1.16b, v15.16b\n"
      ".inst 0x6e8f946d  // udot v13.4s, v3.16b, v15.16b\n"
      "72:"  // Height 3: Multiply loop: unique 10: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "73:"  // Height 3: Multiply loop: Main loop skip
      "cbz x25, 82f\n"
      "cmp x25, #0x8\n"
      "blt 76f\n"
      "74:"  // Height 3: Multiply loop: Odd block loop
      "ldr d1, [x24], #0x8\n"
      "ldr d2, [x23], #0x8\n"
      "ldr d3, [x22], #0x8\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v7.2d\n"
      "tbnz %x[flags], #31, 75f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f944d  // udot v13.4s, v2.16b, v15.16b\n"
      "75:"  // Height 3: Multiply loop: unique 11: skip row sum
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "sub x25, x25, #0x8\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q4, [x9, #0x30]\n"
      "cmp x25, #0x8\n"
      "ldr q5, [x9, #0x40]\n"
      "ldr q6, [x9, #0x50]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4e88ac10  // usmmla v16.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88ac58  // usmmla v24.4s, v2.16b, v8.16b\n"
      "ldr q8, [x9, #0x70]\n"
      ".inst 0x4e89ac14  // usmmla v20.4s, v0.16b, v9.16b\n"
      ".inst 0x4e89ac5c  // usmmla v28.4s, v2.16b, v9.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e8aac11  // usmmla v17.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac59  // usmmla v25.4s, v2.16b, v10.16b\n"
      ".inst 0x4e84ac15  // usmmla v21.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84ac5d  // usmmla v29.4s, v2.16b, v4.16b\n"
      ".inst 0x4e85ac12  // usmmla v18.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85ac5a  // usmmla v26.4s, v2.16b, v5.16b\n"
      ".inst 0x4e86ac16  // usmmla v22.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86ac5e  // usmmla v30.4s, v2.16b, v6.16b\n"
      ".inst 0x4e87ac13  // usmmla v19.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87ac5b  // usmmla v27.4s, v2.16b, v7.16b\n"
      ".inst 0x4e88ac17  // usmmla v23.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88ac5f  // usmmla v31.4s, v2.16b, v8.16b\n"
      "bge 74b\n"
      "76:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x25, 82f\n"
      "tbz x25, #2, 78f\n"
      "ldr s1, [x24], #0x4\n"
      "ldr s2, [x23], #0x4\n"
      "ldr s3, [x22], #0x4\n"
      "tbz x25, #1, 77f\n"
      "ld1 { v1.h }[2], [x24], #0x2\n"
      "ld1 { v2.h }[2], [x23], #0x2\n"
      "ld1 { v3.h }[2], [x22], #0x2\n"
      "tbz x25, #0, 80f\n"
      "ld1 { v1.b }[6], [x24]\n"
      "ld1 { v2.b }[6], [x23]\n"
      "ld1 { v3.b }[6], [x22]\n"
      "b 80f\n"
      "77:"  // Height 3: Multiply loop: Ragged operand read: partial_1_4
      "tbz x25, #0, 80f\n"
      "ld1 { v1.b }[4], [x24]\n"
      "ld1 { v2.b }[4], [x23]\n"
      "ld1 { v3.b }[4], [x22]\n"
      "b 80f\n"
      "78:"  // Height 3: Multiply loop: Ragged operand read: partial_2_0
      "tbz x25, #1, 79f\n"
      "ldr h1, [x24], #0x2\n"
      "ldr h2, [x23], #0x2\n"
      "ldr h3, [x22], #0x2\n"
      "tbz x25, #0, 80f\n"
      "ld1 { v1.b }[2], [x24]\n"
      "ld1 { v2.b }[2], [x23]\n"
      "ld1 { v3.b }[2], [x22]\n"
      "b 80f\n"
      "79:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x24, #0x0]\n"
      "ldr b2, [x23, #0x0]\n"
      "ldr b3, [x22, #0x0]\n"
      "80:"  // Height 3: Multiply loop: Ragged operand read: Done
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v9.2d\n"
      "tbnz %x[flags], #31, 81f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f944d  // udot v13.4s, v2.16b, v15.16b\n"
      "81:"  // Height 3: Multiply loop: unique 12: skip row sum
      "ldr q10, [x9, #0x0]\n"
      "ldr q4, [x9, #0x10]\n"
      "ldr q5, [x9, #0x20]\n"
      "ldr q6, [x9, #0x30]\n"
      "ldr q7, [x9, #0x40]\n"
      "ldr q8, [x9, #0x50]\n"
      "ldr q9, [x9, #0x60]\n"
      ".inst 0x4e8aac10  // usmmla v16.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac58  // usmmla v24.4s, v2.16b, v10.16b\n"
      "ldr q10, [x9, #0x70]\n"
      ".inst 0x4e84ac14  // usmmla v20.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84ac5c  // usmmla v28.4s, v2.16b, v4.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e85ac11  // usmmla v17.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85ac59  // usmmla v25.4s, v2.16b, v5.16b\n"
      ".inst 0x4e86ac15  // usmmla v21.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86ac5d  // usmmla v29.4s, v2.16b, v6.16b\n"
      ".inst 0x4e87ac12  // usmmla v18.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87ac5a  // usmmla v26.4s, v2.16b, v7.16b\n"
      ".inst 0x4e88ac16  // usmmla v22.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88ac5e  // usmmla v30.4s, v2.16b, v8.16b\n"
      ".inst 0x4e89ac13  // usmmla v19.4s, v0.16b, v9.16b\n"
      ".inst 0x4e89ac5b  // usmmla v27.4s, v2.16b, v9.16b\n"
      ".inst 0x4e8aac17  // usmmla v23.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac5f  // usmmla v31.4s, v2.16b, v10.16b\n"
      "82:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 66b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v4.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x26, x27, x20\n"
      "add x25, x26, x20\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "uzp1 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v27.2d, v27.2d, v31.2d\n"
      "mov v31.16b, v4.16b\n"
      "tbnz %x[flags], #31, 83f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "ld1r { v3.4s }, [x20]\n"
      "neg v3.4s, v3.4s\n"
      "dup v12.4s, v11.s[3]\n"
      "dup v11.4s, v11.s[0]\n"
      "dup v13.4s, v13.s[0]\n"
      "mul v11.4s, v11.4s, v3.4s\n"
      "mul v12.4s, v12.4s, v3.4s\n"
      "mul v13.4s, v13.4s, v3.4s\n"
      "83:"  // Height 3: skip row sum fixup
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x28, #0x10]\n"
      "add v31.4s, v31.4s, v11.4s\n"
      "add v20.4s, v20.4s, v11.4s\n"
      "ldr q2, [x28, #0x20]\n"
      "ldr q3, [x28, #0x30]\n"
      "add v21.4s, v21.4s, v11.4s\n"
      "add v22.4s, v22.4s, v11.4s\n"
      "add v16.4s, v16.4s, v12.4s\n"
      "add v17.4s, v17.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "add x23, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v4.4s }, [x20]\n"
      "add v18.4s, v18.4s, v12.4s\n"
      "add v19.4s, v19.4s, v12.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add v24.4s, v24.4s, v13.4s\n"
      "add v25.4s, v25.4s, v13.4s\n"
      "add x21, %x[qp], %[maxval]\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v6.4s }, [x21]\n"
      "ld1r { v5.4s }, [x20]\n"
      "add v26.4s, v26.4s, v13.4s\n"
      "add v27.4s, v27.4s, v13.4s\n"
      "add v31.4s, v31.4s, v0.4s\n"
      "add v20.4s, v20.4s, v1.4s\n"
      "cmp x10, #0x10\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "add v21.4s, v21.4s, v2.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add x28, x28, #0x40\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "ld1r { v0.4s }, [x23]\n"
      "add v25.4s, v25.4s, v1.4s\n"
      "add v26.4s, v26.4s, v2.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "sqdmulh v31.4s, v31.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "sqdmulh v21.4s, v21.4s, v4.4s\n"
      "sqdmulh v22.4s, v22.4s, v4.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v4.4s\n"
      "sqdmulh v18.4s, v18.4s, v4.4s\n"
      "sqdmulh v19.4s, v19.4s, v4.4s\n"
      "sqdmulh v24.4s, v24.4s, v4.4s\n"
      "sqdmulh v25.4s, v25.4s, v4.4s\n"
      "sqdmulh v26.4s, v26.4s, v4.4s\n"
      "sqdmulh v27.4s, v27.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v0.4s\n"
      "srshl v26.4s, v26.4s, v0.4s\n"
      "srshl v27.4s, v27.4s, v0.4s\n"
      "add v31.4s, v31.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v4.4s\n"
      "add v27.4s, v27.4s, v4.4s\n"
      "smin v31.4s, v31.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v24.4s, v24.4s, v6.4s\n"
      "smin v25.4s, v25.4s, v6.4s\n"
      "smin v26.4s, v26.4s, v6.4s\n"
      "smin v27.4s, v27.4s, v6.4s\n"
      "smax v31.4s, v31.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v24.4s, v24.4s, v5.4s\n"
      "smax v25.4s, v25.4s, v5.4s\n"
      "smax v26.4s, v26.4s, v5.4s\n"
      "smax v27.4s, v27.4s, v5.4s\n"
      "uzp1 v31.8h, v31.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v25.8h, v26.8h, v27.8h\n"
      "uzp1 v31.16b, v31.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v24.16b, v24.16b, v25.16b\n"
      "bge 92f\n"
      "tbz x10, #3, 87f\n"
      "str d31, [x27], #0x8\n"
      "str d16, [x26], #0x8\n"
      "str d24, [x25], #0x8\n"
      "tbz x10, #2, 85f\n"
      "st1 { v31.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x26], #0x4\n"
      "st1 { v24.s }[2], [x25], #0x4\n"
      "tbz x10, #1, 84f\n"
      "st1 { v31.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x26], #0x2\n"
      "st1 { v24.h }[6], [x25], #0x2\n"
      "tbz x10, #0, 91f\n"
      "st1 { v31.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x26]\n"
      "st1 { v24.b }[14], [x25]\n"
      "b 91f\n"
      "84:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x10, #0, 91f\n"
      "st1 { v31.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x26]\n"
      "st1 { v24.b }[12], [x25]\n"
      "b 91f\n"
      "85:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x10, #1, 86f\n"
      "st1 { v31.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x26], #0x2\n"
      "st1 { v24.h }[4], [x25], #0x2\n"
      "tbz x10, #0, 91f\n"
      "st1 { v31.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x26]\n"
      "st1 { v24.b }[10], [x25]\n"
      "b 91f\n"
      "86:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x10, #0, 91f\n"
      "st1 { v31.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x26]\n"
      "st1 { v24.b }[8], [x25]\n"
      "b 91f\n"
      "87:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x10, #2, 89f\n"
      "str s31, [x27], #0x4\n"
      "str s16, [x26], #0x4\n"
      "str s24, [x25], #0x4\n"
      "tbz x10, #1, 88f\n"
      "st1 { v31.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x26], #0x2\n"
      "st1 { v24.h }[2], [x25], #0x2\n"
      "tbz x10, #0, 91f\n"
      "st1 { v31.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x26]\n"
      "st1 { v24.b }[6], [x25]\n"
      "b 91f\n"
      "88:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x10, #0, 91f\n"
      "st1 { v31.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x26]\n"
      "st1 { v24.b }[4], [x25]\n"
      "b 91f\n"
      "89:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x10, #1, 90f\n"
      "str h31, [x27], #0x2\n"
      "str h16, [x26], #0x2\n"
      "str h24, [x25], #0x2\n"
      "tbz x10, #0, 91f\n"
      "st1 { v31.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x26]\n"
      "st1 { v24.b }[2], [x25]\n"
      "b 91f\n"
      "90:"  // Height 3: Partial direct writeback: partial_1_0
      "str b31, [x27, #0x0]\n"
      "str b16, [x26, #0x0]\n"
      "str b24, [x25, #0x0]\n"
      "91:"  // Height 3: Partial direct writeback: Done
      "b 93f\n"
      "92:"  // Height 3: Full writeback
      "str q31, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q16, [x26, #0x0]\n"
      "str q24, [x25, #0x0]\n"
      "93:"  // Height 3: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 64b\n"
      "b 126f\n"
      "94:"  // Height 4
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x4\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "movi v14.4s, #0x0\n"
      "movi v15.16b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x28, %x[col_bias]\n"
      "madd x20, x21, x20, x27\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "95:"  // Height 4: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "mov x26, #0x0\n"
      "97:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 98f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "ldr x21, [x20, #0x18]\n"
      "cbnz x26, 99f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 99f\n"
      "98:"  // Height 4: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "99:"  // Height 4: input setup done
      "cmp x25, #0x10\n"
      "blt 104f\n"
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q3, [x22, #0x0]\n"
      "ldr q4, [x21, #0x0]\n"
      "ldr q5, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "ldr q7, [x9, #0x20]\n"
      "ldr q8, [x9, #0x30]\n"
      "ldr q9, [x9, #0x40]\n"
      "ldr q10, [x9, #0x50]\n"
      "blt 102f\n"
      "100:"  // Height 4: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "ldr q4, [x9, #0x60]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4e85ac10  // usmmla v16.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86ac14  // usmmla v20.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87ac11  // usmmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88ac15  // usmmla v21.4s, v0.16b, v8.16b\n"
      ".inst 0x4e85ac58  // usmmla v24.4s, v2.16b, v5.16b\n"
      "ldr q5, [x9, #0x70]\n"
      ".inst 0x4e86ac5c  // usmmla v28.4s, v2.16b, v6.16b\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4e87ac59  // usmmla v25.4s, v2.16b, v7.16b\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4e88ac5d  // usmmla v29.4s, v2.16b, v8.16b\n"
      "ldr q8, [x9, #0xa0]\n"
      ".inst 0x4e89ac12  // usmmla v18.4s, v0.16b, v9.16b\n"
      ".inst 0x4e89ac5a  // usmmla v26.4s, v2.16b, v9.16b\n"
      "ldr q9, [x9, #0xb0]\n"
      ".inst 0x4e8aac16  // usmmla v22.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac5e  // usmmla v30.4s, v2.16b, v10.16b\n"
      "ldr q10, [x9, #0xc0]\n"
      ".inst 0x4e84ac13  // usmmla v19.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84ac5b  // usmmla v27.4s, v2.16b, v4.16b\n"
      "ldr q4, [x9, #0xd0]\n"
      ".inst 0x4e85ac17  // usmmla v23.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85ac5f  // usmmla v31.4s, v2.16b, v5.16b\n"
      "ldr q5, [x9, #0xe0]\n"
      ".inst 0x4e86ac30  // usmmla v16.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86ac78  // usmmla v24.4s, v3.16b, v6.16b\n"
      "ldr q6, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e87ac34  // usmmla v20.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87ac7c  // usmmla v28.4s, v3.16b, v7.16b\n"
      ".inst 0x4e88ac31  // usmmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e88ac79  // usmmla v25.4s, v3.16b, v8.16b\n"
      ".inst 0x4e89ac35  // usmmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e89ac7d  // usmmla v29.4s, v3.16b, v9.16b\n"
      ".inst 0x4e8aac32  // usmmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e8aac7a  // usmmla v26.4s, v3.16b, v10.16b\n"
      ".inst 0x4e84ac36  // usmmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84ac7e  // usmmla v30.4s, v3.16b, v4.16b\n"
      ".inst 0x4e85ac33  // usmmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e85ac7b  // usmmla v27.4s, v3.16b, v5.16b\n"
      ".inst 0x4e86ac37  // usmmla v23.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86ac7f  // usmmla v31.4s, v3.16b, v6.16b\n"
      "tbnz %x[flags], #31, 101f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f944d  // udot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x6e8f942b  // udot v11.4s, v1.16b, v15.16b\n"
      ".inst 0x6e8f946d  // udot v13.4s, v3.16b, v15.16b\n"
      "101:"  // Height 4: Multiply loop: unique 13: skip row sum
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q3, [x22, #0x0]\n"
      "ldr q4, [x21, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q5, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "ldr q7, [x9, #0x20]\n"
      "ldr q8, [x9, #0x30]\n"
      "ldr q9, [x9, #0x40]\n"
      "ldr q10, [x9, #0x50]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "bge 100b\n"
      "102:"  // Height 4: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "ldr q4, [x9, #0x60]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4e85ac10  // usmmla v16.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86ac14  // usmmla v20.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87ac11  // usmmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88ac15  // usmmla v21.4s, v0.16b, v8.16b\n"
      ".inst 0x4e85ac58  // usmmla v24.4s, v2.16b, v5.16b\n"
      "ldr q5, [x9, #0x70]\n"
      ".inst 0x4e86ac5c  // usmmla v28.4s, v2.16b, v6.16b\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4e87ac59  // usmmla v25.4s, v2.16b, v7.16b\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4e88ac5d  // usmmla v29.4s, v2.16b, v8.16b\n"
      "ldr q8, [x9, #0xa0]\n"
      ".inst 0x4e89ac12  // usmmla v18.4s, v0.16b, v9.16b\n"
      ".inst 0x4e89ac5a  // usmmla v26.4s, v2.16b, v9.16b\n"
      "ldr q9, [x9, #0xb0]\n"
      ".inst 0x4e8aac16  // usmmla v22.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac5e  // usmmla v30.4s, v2.16b, v10.16b\n"
      "ldr q10, [x9, #0xc0]\n"
      ".inst 0x4e84ac13  // usmmla v19.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84ac5b  // usmmla v27.4s, v2.16b, v4.16b\n"
      "ldr q4, [x9, #0xd0]\n"
      ".inst 0x4e85ac17  // usmmla v23.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85ac5f  // usmmla v31.4s, v2.16b, v5.16b\n"
      "ldr q5, [x9, #0xe0]\n"
      ".inst 0x4e86ac30  // usmmla v16.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86ac78  // usmmla v24.4s, v3.16b, v6.16b\n"
      "ldr q6, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e87ac34  // usmmla v20.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87ac7c  // usmmla v28.4s, v3.16b, v7.16b\n"
      ".inst 0x4e88ac31  // usmmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e88ac79  // usmmla v25.4s, v3.16b, v8.16b\n"
      ".inst 0x4e89ac35  // usmmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e89ac7d  // usmmla v29.4s, v3.16b, v9.16b\n"
      ".inst 0x4e8aac32  // usmmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e8aac7a  // usmmla v26.4s, v3.16b, v10.16b\n"
      ".inst 0x4e84ac36  // usmmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84ac7e  // usmmla v30.4s, v3.16b, v4.16b\n"
      ".inst 0x4e85ac33  // usmmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e85ac7b  // usmmla v27.4s, v3.16b, v5.16b\n"
      ".inst 0x4e86ac37  // usmmla v23.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86ac7f  // usmmla v31.4s, v3.16b, v6.16b\n"
      "tbnz %x[flags], #31, 103f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f944d  // udot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x6e8f942b  // udot v11.4s, v1.16b, v15.16b\n"
      ".inst 0x6e8f946d  // udot v13.4s, v3.16b, v15.16b\n"
      "103:"  // Height 4: Multiply loop: unique 14: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "104:"  // Height 4: Multiply loop: Main loop skip
      "cbz x25, 113f\n"
      "cmp x25, #0x8\n"
      "blt 107f\n"
      "105:"  // Height 4: Multiply loop: Odd block loop
      "ldr d1, [x24], #0x8\n"
      "ldr d2, [x23], #0x8\n"
      "ldr d3, [x22], #0x8\n"
      "ldr d7, [x21], #0x8\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v7.2d\n"
      "tbnz %x[flags], #31, 106f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f944d  // udot v13.4s, v2.16b, v15.16b\n"
      "106:"  // Height 4: Multiply loop: unique 15: skip row sum
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "sub x25, x25, #0x8\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q4, [x9, #0x30]\n"
      "cmp x25, #0x8\n"
      "ldr q5, [x9, #0x40]\n"
      "ldr q6, [x9, #0x50]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4e88ac10  // usmmla v16.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88ac58  // usmmla v24.4s, v2.16b, v8.16b\n"
      "ldr q8, [x9, #0x70]\n"
      ".inst 0x4e89ac14  // usmmla v20.4s, v0.16b, v9.16b\n"
      ".inst 0x4e89ac5c  // usmmla v28.4s, v2.16b, v9.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e8aac11  // usmmla v17.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac59  // usmmla v25.4s, v2.16b, v10.16b\n"
      ".inst 0x4e84ac15  // usmmla v21.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84ac5d  // usmmla v29.4s, v2.16b, v4.16b\n"
      ".inst 0x4e85ac12  // usmmla v18.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85ac5a  // usmmla v26.4s, v2.16b, v5.16b\n"
      ".inst 0x4e86ac16  // usmmla v22.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86ac5e  // usmmla v30.4s, v2.16b, v6.16b\n"
      ".inst 0x4e87ac13  // usmmla v19.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87ac5b  // usmmla v27.4s, v2.16b, v7.16b\n"
      ".inst 0x4e88ac17  // usmmla v23.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88ac5f  // usmmla v31.4s, v2.16b, v8.16b\n"
      "bge 105b\n"
      "107:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x25, 113f\n"
      "tbz x25, #2, 109f\n"
      "ldr s1, [x24], #0x4\n"
      "ldr s2, [x23], #0x4\n"
      "ldr s3, [x22], #0x4\n"
      "ldr s9, [x21], #0x4\n"
      "tbz x25, #1, 108f\n"
      "ld1 { v1.h }[2], [x24], #0x2\n"
      "ld1 { v2.h }[2], [x23], #0x2\n"
      "ld1 { v3.h }[2], [x22], #0x2\n"
      "ld1 { v9.h }[2], [x21], #0x2\n"
      "tbz x25, #0, 111f\n"
      "ld1 { v1.b }[6], [x24]\n"
      "ld1 { v2.b }[6], [x23]\n"
      "ld1 { v3.b }[6], [x22]\n"
      "ld1 { v9.b }[6], [x21]\n"
      "b 111f\n"
      "108:"  // Height 4: Multiply loop: Ragged operand read: partial_1_4
      "tbz x25, #0, 111f\n"
      "ld1 { v1.b }[4], [x24]\n"
      "ld1 { v2.b }[4], [x23]\n"
      "ld1 { v3.b }[4], [x22]\n"
      "ld1 { v9.b }[4], [x21]\n"
      "b 111f\n"
      "109:"  // Height 4: Multiply loop: Ragged operand read: partial_2_0
      "tbz x25, #1, 110f\n"
      "ldr h1, [x24], #0x2\n"
      "ldr h2, [x23], #0x2\n"
      "ldr h3, [x22], #0x2\n"
      "ldr h9, [x21], #0x2\n"
      "tbz x25, #0, 111f\n"
      "ld1 { v1.b }[2], [x24]\n"
      "ld1 { v2.b }[2], [x23]\n"
      "ld1 { v3.b }[2], [x22]\n"
      "ld1 { v9.b }[2], [x21]\n"
      "b 111f\n"
      "110:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x24, #0x0]\n"
      "ldr b2, [x23, #0x0]\n"
      "ldr b3, [x22, #0x0]\n"
      "ldr b9, [x21, #0x0]\n"
      "111:"  // Height 4: Multiply loop: Ragged operand read: Done
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v9.2d\n"
      "tbnz %x[flags], #31, 112f\n"
      ".inst 0x6e8f940b  // udot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x6e8f944d  // udot v13.4s, v2.16b, v15.16b\n"
      "112:"  // Height 4: Multiply loop: unique 16: skip row sum
      "ldr q10, [x9, #0x0]\n"
      "ldr q4, [x9, #0x10]\n"
      "ldr q5, [x9, #0x20]\n"
      "ldr q6, [x9, #0x30]\n"
      "ldr q7, [x9, #0x40]\n"
      "ldr q8, [x9, #0x50]\n"
      "ldr q9, [x9, #0x60]\n"
      ".inst 0x4e8aac10  // usmmla v16.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac58  // usmmla v24.4s, v2.16b, v10.16b\n"
      "ldr q10, [x9, #0x70]\n"
      ".inst 0x4e84ac14  // usmmla v20.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84ac5c  // usmmla v28.4s, v2.16b, v4.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e85ac11  // usmmla v17.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85ac59  // usmmla v25.4s, v2.16b, v5.16b\n"
      ".inst 0x4e86ac15  // usmmla v21.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86ac5d  // usmmla v29.4s, v2.16b, v6.16b\n"
      ".inst 0x4e87ac12  // usmmla v18.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87ac5a  // usmmla v26.4s, v2.16b, v7.16b\n"
      ".inst 0x4e88ac16  // usmmla v22.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88ac5e  // usmmla v30.4s, v2.16b, v8.16b\n"
      ".inst 0x4e89ac13  // usmmla v19.4s, v0.16b, v9.16b\n"
      ".inst 0x4e89ac5b  // usmmla v27.4s, v2.16b, v9.16b\n"
      ".inst 0x4e8aac17  // usmmla v23.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aac5f  // usmmla v31.4s, v2.16b, v10.16b\n"
      "113:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 97b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v4.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x26, x27, x20\n"
      "add x25, x26, x20\n"
      "add x24, x25, x20\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "uzp1 v23.2d, v24.2d, v28.2d\n"
      "uzp2 v24.2d, v24.2d, v28.2d\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp1 v28.2d, v25.2d, v29.2d\n"
      "uzp2 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v29.2d, v26.2d, v30.2d\n"
      "uzp2 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v30.2d, v27.2d, v31.2d\n"
      "uzp2 v27.2d, v27.2d, v31.2d\n"
      "mov v31.16b, v4.16b\n"
      "tbnz %x[flags], #31, 114f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "ld1r { v4.4s }, [x20]\n"
      "neg v4.4s, v4.4s\n"
      "dup v12.4s, v11.s[3]\n"
      "dup v11.4s, v11.s[0]\n"
      "dup v14.4s, v13.s[3]\n"
      "dup v13.4s, v13.s[0]\n"
      "mul v11.4s, v11.4s, v4.4s\n"
      "mul v12.4s, v12.4s, v4.4s\n"
      "mul v14.4s, v14.4s, v4.4s\n"
      "mul v13.4s, v13.4s, v4.4s\n"
      "114:"  // Height 4: skip row sum fixup
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x28, #0x10]\n"
      "add v31.4s, v31.4s, v11.4s\n"
      "add v20.4s, v20.4s, v11.4s\n"
      "ldr q2, [x28, #0x20]\n"
      "ldr q3, [x28, #0x30]\n"
      "add v21.4s, v21.4s, v11.4s\n"
      "add v22.4s, v22.4s, v11.4s\n"
      "add v16.4s, v16.4s, v12.4s\n"
      "add v17.4s, v17.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "add x23, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v4.4s }, [x20]\n"
      "add v18.4s, v18.4s, v12.4s\n"
      "add v19.4s, v19.4s, v12.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add v23.4s, v23.4s, v13.4s\n"
      "add v28.4s, v28.4s, v13.4s\n"
      "add x21, %x[qp], %[maxval]\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v6.4s }, [x21]\n"
      "ld1r { v5.4s }, [x20]\n"
      "add v29.4s, v29.4s, v13.4s\n"
      "add v30.4s, v30.4s, v13.4s\n"
      "add v24.4s, v24.4s, v14.4s\n"
      "add v25.4s, v25.4s, v14.4s\n"
      "cmp x10, #0x10\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "add v26.4s, v26.4s, v14.4s\n"
      "add v27.4s, v27.4s, v14.4s\n"
      "add x28, x28, #0x40\n"
      "add v31.4s, v31.4s, v0.4s\n"
      "add v20.4s, v20.4s, v1.4s\n"
      "add v21.4s, v21.4s, v2.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v23.4s, v23.4s, v0.4s\n"
      "add v28.4s, v28.4s, v1.4s\n"
      "add v29.4s, v29.4s, v2.4s\n"
      "add v30.4s, v30.4s, v3.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "ld1r { v0.4s }, [x23]\n"
      "add v25.4s, v25.4s, v1.4s\n"
      "add v26.4s, v26.4s, v2.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "sqdmulh v31.4s, v31.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "sqdmulh v21.4s, v21.4s, v4.4s\n"
      "sqdmulh v22.4s, v22.4s, v4.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v4.4s\n"
      "sqdmulh v18.4s, v18.4s, v4.4s\n"
      "sqdmulh v19.4s, v19.4s, v4.4s\n"
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "sqdmulh v28.4s, v28.4s, v4.4s\n"
      "sqdmulh v29.4s, v29.4s, v4.4s\n"
      "sqdmulh v30.4s, v30.4s, v4.4s\n"
      "sqdmulh v24.4s, v24.4s, v4.4s\n"
      "sqdmulh v25.4s, v25.4s, v4.4s\n"
      "sqdmulh v26.4s, v26.4s, v4.4s\n"
      "sqdmulh v27.4s, v27.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v28.4s, v28.4s, v0.4s\n"
      "srshl v29.4s, v29.4s, v0.4s\n"
      "srshl v30.4s, v30.4s, v0.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v0.4s\n"
      "srshl v26.4s, v26.4s, v0.4s\n"
      "srshl v27.4s, v27.4s, v0.4s\n"
      "add v31.4s, v31.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v28.4s, v28.4s, v4.4s\n"
      "add v29.4s, v29.4s, v4.4s\n"
      "add v30.4s, v30.4s, v4.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v4.4s\n"
      "add v27.4s, v27.4s, v4.4s\n"
      "smin v31.4s, v31.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v28.4s, v28.4s, v6.4s\n"
      "smin v29.4s, v29.4s, v6.4s\n"
      "smin v30.4s, v30.4s, v6.4s\n"
      "smin v24.4s, v24.4s, v6.4s\n"
      "smin v25.4s, v25.4s, v6.4s\n"
      "smin v26.4s, v26.4s, v6.4s\n"
      "smin v27.4s, v27.4s, v6.4s\n"
      "smax v31.4s, v31.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v28.4s, v28.4s, v5.4s\n"
      "smax v29.4s, v29.4s, v5.4s\n"
      "smax v30.4s, v30.4s, v5.4s\n"
      "smax v24.4s, v24.4s, v5.4s\n"
      "smax v25.4s, v25.4s, v5.4s\n"
      "smax v26.4s, v26.4s, v5.4s\n"
      "smax v27.4s, v27.4s, v5.4s\n"
      "uzp1 v31.8h, v31.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.8h, v23.8h, v28.8h\n"
      "uzp1 v28.8h, v29.8h, v30.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v25.8h, v26.8h, v27.8h\n"
      "uzp1 v31.16b, v31.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v23.16b, v23.16b, v28.16b\n"
      "uzp1 v24.16b, v24.16b, v25.16b\n"
      "bge 123f\n"
      "tbz x10, #3, 118f\n"
      "str d31, [x27], #0x8\n"
      "str d16, [x26], #0x8\n"
      "str d23, [x25], #0x8\n"
      "str d24, [x24], #0x8\n"
      "tbz x10, #2, 116f\n"
      "st1 { v31.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x26], #0x4\n"
      "st1 { v23.s }[2], [x25], #0x4\n"
      "st1 { v24.s }[2], [x24], #0x4\n"
      "tbz x10, #1, 115f\n"
      "st1 { v31.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x26], #0x2\n"
      "st1 { v23.h }[6], [x25], #0x2\n"
      "st1 { v24.h }[6], [x24], #0x2\n"
      "tbz x10, #0, 122f\n"
      "st1 { v31.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x26]\n"
      "st1 { v23.b }[14], [x25]\n"
      "st1 { v24.b }[14], [x24]\n"
      "b 122f\n"
      "115:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x10, #0, 122f\n"
      "st1 { v31.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x26]\n"
      "st1 { v23.b }[12], [x25]\n"
      "st1 { v24.b }[12], [x24]\n"
      "b 122f\n"
      "116:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x10, #1, 117f\n"
      "st1 { v31.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x26], #0x2\n"
      "st1 { v23.h }[4], [x25], #0x2\n"
      "st1 { v24.h }[4], [x24], #0x2\n"
      "tbz x10, #0, 122f\n"
      "st1 { v31.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x26]\n"
      "st1 { v23.b }[10], [x25]\n"
      "st1 { v24.b }[10], [x24]\n"
      "b 122f\n"
      "117:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x10, #0, 122f\n"
      "st1 { v31.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x26]\n"
      "st1 { v23.b }[8], [x25]\n"
      "st1 { v24.b }[8], [x24]\n"
      "b 122f\n"
      "118:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x10, #2, 120f\n"
      "str s31, [x27], #0x4\n"
      "str s16, [x26], #0x4\n"
      "str s23, [x25], #0x4\n"
      "str s24, [x24], #0x4\n"
      "tbz x10, #1, 119f\n"
      "st1 { v31.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x26], #0x2\n"
      "st1 { v23.h }[2], [x25], #0x2\n"
      "st1 { v24.h }[2], [x24], #0x2\n"
      "tbz x10, #0, 122f\n"
      "st1 { v31.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x26]\n"
      "st1 { v23.b }[6], [x25]\n"
      "st1 { v24.b }[6], [x24]\n"
      "b 122f\n"
      "119:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x10, #0, 122f\n"
      "st1 { v31.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x26]\n"
      "st1 { v23.b }[4], [x25]\n"
      "st1 { v24.b }[4], [x24]\n"
      "b 122f\n"
      "120:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x10, #1, 121f\n"
      "str h31, [x27], #0x2\n"
      "str h16, [x26], #0x2\n"
      "str h23, [x25], #0x2\n"
      "str h24, [x24], #0x2\n"
      "tbz x10, #0, 122f\n"
      "st1 { v31.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x26]\n"
      "st1 { v23.b }[2], [x25]\n"
      "st1 { v24.b }[2], [x24]\n"
      "b 122f\n"
      "121:"  // Height 4: Partial direct writeback: partial_1_0
      "str b31, [x27, #0x0]\n"
      "str b16, [x26, #0x0]\n"
      "str b23, [x25, #0x0]\n"
      "str b24, [x24, #0x0]\n"
      "122:"  // Height 4: Partial direct writeback: Done
      "b 124f\n"
      "123:"  // Height 4: Full writeback
      "str q31, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q16, [x26, #0x0]\n"
      "str q23, [x25, #0x0]\n"
      "str q24, [x24, #0x0]\n"
      "124:"  // Height 4: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 95b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 126f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 125f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "125:"  // Update direct input
      "mov x20, #0x4\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "126:"  // Exit
      : [M] "+&r" (M), [flags] "+&r" (flags), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [b_offset] "I" (offsetof(Requantize32, b_offset)), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
