/*
 * Copyright (c) 2019-2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include <cassert>
#include <cstdint>

namespace arm_gemm {

void a64_hybrid_s8qs_dot_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int col_base
)
{
    struct KernelArgs {
        const int32_t *multiplier_ptr = {};
        const int32_t *shift_ptr = {};
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->per_channel_requant) {
        flags |= 0x10;
        ka.multiplier_ptr=qp->per_channel_muls + col_base;
        ka.shift_ptr=qp->per_channel_right_shifts + col_base;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 131f\n"
      "cmp %x[M], #0x4\n"
      "bgt 105f\n"
      "beq 79f\n"
      "cmp %x[M], #0x2\n"
      "bgt 53f\n"
      "beq 27f\n"
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "mov x28, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "blt 9f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 8f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 7b\n"
      "8:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x26, x26, #0x10\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "9:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 14f\n"
      "cmp x27, #0x4\n"
      "blt 11f\n"
      "10:"  // Height 1: Multiply loop: Odd block loop
      "ldr s0, [x26], #0x4\n"
      "ldr q6, [x10, #0x0]\n"
      "sub x27, x27, #0x4\n"
      "ldr q7, [x10, #0x10]\n"
      "cmp x27, #0x4\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "bge 10b\n"
      "11:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x27, 14f\n"
      "tbz x27, #1, 12f\n"
      "ldr h0, [x26], #0x2\n"
      "tbz x27, #0, 13f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "b 13f\n"
      "12:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "13:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "14:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 4b\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "tbz %x[flags], #4, 15f\n"
      "ldr q0, [x13, #0x0]\n"
      "ldr q4, [x14, #0x0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 16f\n"
      "15:"  // Height 1: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "16:"  // Height 1: parameters loaded
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "add x21, %x[qp], %[c_offset]\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v4.4s }, [x21]\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "ld1r { v6.4s }, [x20]\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "ld1r { v5.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "bge 25f\n"
      "tbz x11, #3, 20f\n"
      "str d8, [x9], #0x8\n"
      "tbz x11, #2, 18f\n"
      "st1 { v8.s }[2], [x9], #0x4\n"
      "tbz x11, #1, 17f\n"
      "st1 { v8.h }[6], [x9], #0x2\n"
      "tbz x11, #0, 24f\n"
      "st1 { v8.b }[14], [x9]\n"
      "b 24f\n"
      "17:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x11, #0, 24f\n"
      "st1 { v8.b }[12], [x9]\n"
      "b 24f\n"
      "18:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x11, #1, 19f\n"
      "st1 { v8.h }[4], [x9], #0x2\n"
      "tbz x11, #0, 24f\n"
      "st1 { v8.b }[10], [x9]\n"
      "b 24f\n"
      "19:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x11, #0, 24f\n"
      "st1 { v8.b }[8], [x9]\n"
      "b 24f\n"
      "20:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x11, #2, 22f\n"
      "str s8, [x9], #0x4\n"
      "tbz x11, #1, 21f\n"
      "st1 { v8.h }[2], [x9], #0x2\n"
      "tbz x11, #0, 24f\n"
      "st1 { v8.b }[6], [x9]\n"
      "b 24f\n"
      "21:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x11, #0, 24f\n"
      "st1 { v8.b }[4], [x9]\n"
      "b 24f\n"
      "22:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x11, #1, 23f\n"
      "str h8, [x9], #0x2\n"
      "tbz x11, #0, 24f\n"
      "st1 { v8.b }[2], [x9]\n"
      "b 24f\n"
      "23:"  // Height 1: Partial direct writeback: partial_1_0
      "str b8, [x9, #0x0]\n"
      "24:"  // Height 1: Partial direct writeback: Done
      "b 26f\n"
      "25:"  // Height 1: Full writeback
      "str q8, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "26:"  // Height 1: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 2b\n"
      "b 158f\n"
      "27:"  // Height 2
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "28:"  // Height 2: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "mov x28, #0x0\n"
      "30:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 31f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 32f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 32f\n"
      "31:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "32:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "blt 35f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 34f\n"
      "33:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 33b\n"
      "34:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x25, x25, #0x10\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "35:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 40f\n"
      "cmp x27, #0x4\n"
      "blt 37f\n"
      "36:"  // Height 2: Multiply loop: Odd block loop
      "ldr s0, [x26], #0x4\n"
      "ldr s1, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "cmp x27, #0x4\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "bge 36b\n"
      "37:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x27, 40f\n"
      "tbz x27, #1, 38f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "tbz x27, #0, 39f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "b 39f\n"
      "38:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "39:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "40:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 30b\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v12.4s, v12.4s, v0.4s\n"
      "add v13.4s, v13.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add x27, x9, x20\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "add v14.4s, v14.4s, v2.4s\n"
      "add v15.4s, v15.4s, v3.4s\n"
      "tbz %x[flags], #4, 41f\n"
      "ldr q0, [x13, #0x0]\n"
      "ldr q4, [x14, #0x0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 42f\n"
      "41:"  // Height 2: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "42:"  // Height 2: parameters loaded
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v12.4s, v12.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v13.4s, v13.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v14.4s, v14.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v15.4s, v15.4s, v7.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v13.8h, v14.8h, v15.8h\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v12.16b, v12.16b, v13.16b\n"
      "bge 51f\n"
      "tbz x11, #3, 46f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x27], #0x8\n"
      "tbz x11, #2, 44f\n"
      "st1 { v8.s }[2], [x9], #0x4\n"
      "st1 { v12.s }[2], [x27], #0x4\n"
      "tbz x11, #1, 43f\n"
      "st1 { v8.h }[6], [x9], #0x2\n"
      "st1 { v12.h }[6], [x27], #0x2\n"
      "tbz x11, #0, 50f\n"
      "st1 { v8.b }[14], [x9]\n"
      "st1 { v12.b }[14], [x27]\n"
      "b 50f\n"
      "43:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x11, #0, 50f\n"
      "st1 { v8.b }[12], [x9]\n"
      "st1 { v12.b }[12], [x27]\n"
      "b 50f\n"
      "44:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x11, #1, 45f\n"
      "st1 { v8.h }[4], [x9], #0x2\n"
      "st1 { v12.h }[4], [x27], #0x2\n"
      "tbz x11, #0, 50f\n"
      "st1 { v8.b }[10], [x9]\n"
      "st1 { v12.b }[10], [x27]\n"
      "b 50f\n"
      "45:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x11, #0, 50f\n"
      "st1 { v8.b }[8], [x9]\n"
      "st1 { v12.b }[8], [x27]\n"
      "b 50f\n"
      "46:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x11, #2, 48f\n"
      "str s8, [x9], #0x4\n"
      "str s12, [x27], #0x4\n"
      "tbz x11, #1, 47f\n"
      "st1 { v8.h }[2], [x9], #0x2\n"
      "st1 { v12.h }[2], [x27], #0x2\n"
      "tbz x11, #0, 50f\n"
      "st1 { v8.b }[6], [x9]\n"
      "st1 { v12.b }[6], [x27]\n"
      "b 50f\n"
      "47:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x11, #0, 50f\n"
      "st1 { v8.b }[4], [x9]\n"
      "st1 { v12.b }[4], [x27]\n"
      "b 50f\n"
      "48:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x11, #1, 49f\n"
      "str h8, [x9], #0x2\n"
      "str h12, [x27], #0x2\n"
      "tbz x11, #0, 50f\n"
      "st1 { v8.b }[2], [x9]\n"
      "st1 { v12.b }[2], [x27]\n"
      "b 50f\n"
      "49:"  // Height 2: Partial direct writeback: partial_1_0
      "str b8, [x9, #0x0]\n"
      "str b12, [x27, #0x0]\n"
      "50:"  // Height 2: Partial direct writeback: Done
      "b 52f\n"
      "51:"  // Height 2: Full writeback
      "str q8, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "str q12, [x27, #0x0]\n"
      "52:"  // Height 2: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 28b\n"
      "b 158f\n"
      "53:"  // Height 3
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "54:"  // Height 3: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "mov x28, #0x0\n"
      "56:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 57f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 58f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 58f\n"
      "57:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "58:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "blt 61f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 60f\n"
      "59:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x24, x24, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 59b\n"
      "60:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "61:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 66f\n"
      "cmp x27, #0x4\n"
      "blt 63f\n"
      "62:"  // Height 3: Multiply loop: Odd block loop
      "ldr s0, [x26], #0x4\n"
      "ldr s1, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr s2, [x24], #0x4\n"
      "ldr q6, [x10, #0x0]\n"
      "cmp x27, #0x4\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "bge 62b\n"
      "63:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x27, 66f\n"
      "tbz x27, #1, 64f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "tbz x27, #0, 65f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "b 65f\n"
      "64:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "65:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "66:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 56b\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v12.4s, v12.4s, v0.4s\n"
      "add v13.4s, v13.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add v14.4s, v14.4s, v2.4s\n"
      "add v15.4s, v15.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "tbz %x[flags], #4, 67f\n"
      "ldr q0, [x13, #0x0]\n"
      "ldr q4, [x14, #0x0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 68f\n"
      "67:"  // Height 3: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "68:"  // Height 3: parameters loaded
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v12.4s, v12.4s, v4.4s\n"
      "sqdmulh v13.4s, v13.4s, v5.4s\n"
      "sqdmulh v14.4s, v14.4s, v6.4s\n"
      "sqdmulh v15.4s, v15.4s, v7.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v17.4s, v17.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v18.4s, v18.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v19.4s, v19.4s, v7.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v13.8h, v14.8h, v15.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v12.16b, v12.16b, v13.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 77f\n"
      "tbz x11, #3, 72f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x27], #0x8\n"
      "str d16, [x26], #0x8\n"
      "tbz x11, #2, 70f\n"
      "st1 { v8.s }[2], [x9], #0x4\n"
      "st1 { v12.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x26], #0x4\n"
      "tbz x11, #1, 69f\n"
      "st1 { v8.h }[6], [x9], #0x2\n"
      "st1 { v12.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x26], #0x2\n"
      "tbz x11, #0, 76f\n"
      "st1 { v8.b }[14], [x9]\n"
      "st1 { v12.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x26]\n"
      "b 76f\n"
      "69:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x11, #0, 76f\n"
      "st1 { v8.b }[12], [x9]\n"
      "st1 { v12.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x26]\n"
      "b 76f\n"
      "70:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x11, #1, 71f\n"
      "st1 { v8.h }[4], [x9], #0x2\n"
      "st1 { v12.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x26], #0x2\n"
      "tbz x11, #0, 76f\n"
      "st1 { v8.b }[10], [x9]\n"
      "st1 { v12.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x26]\n"
      "b 76f\n"
      "71:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x11, #0, 76f\n"
      "st1 { v8.b }[8], [x9]\n"
      "st1 { v12.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x26]\n"
      "b 76f\n"
      "72:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x11, #2, 74f\n"
      "str s8, [x9], #0x4\n"
      "str s12, [x27], #0x4\n"
      "str s16, [x26], #0x4\n"
      "tbz x11, #1, 73f\n"
      "st1 { v8.h }[2], [x9], #0x2\n"
      "st1 { v12.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x26], #0x2\n"
      "tbz x11, #0, 76f\n"
      "st1 { v8.b }[6], [x9]\n"
      "st1 { v12.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x26]\n"
      "b 76f\n"
      "73:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x11, #0, 76f\n"
      "st1 { v8.b }[4], [x9]\n"
      "st1 { v12.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x26]\n"
      "b 76f\n"
      "74:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x11, #1, 75f\n"
      "str h8, [x9], #0x2\n"
      "str h12, [x27], #0x2\n"
      "str h16, [x26], #0x2\n"
      "tbz x11, #0, 76f\n"
      "st1 { v8.b }[2], [x9]\n"
      "st1 { v12.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x26]\n"
      "b 76f\n"
      "75:"  // Height 3: Partial direct writeback: partial_1_0
      "str b8, [x9, #0x0]\n"
      "str b12, [x27, #0x0]\n"
      "str b16, [x26, #0x0]\n"
      "76:"  // Height 3: Partial direct writeback: Done
      "b 78f\n"
      "77:"  // Height 3: Full writeback
      "str q8, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "str q12, [x27, #0x0]\n"
      "str q16, [x26, #0x0]\n"
      "78:"  // Height 3: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 54b\n"
      "b 158f\n"
      "79:"  // Height 4
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "80:"  // Height 4: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "mov x28, #0x0\n"
      "82:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 83f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 84f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 84f\n"
      "83:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "84:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "blt 87f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 86f\n"
      "85:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "cmp x27, #0x20\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 85b\n"
      "86:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x23, x23, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "87:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 92f\n"
      "cmp x27, #0x4\n"
      "blt 89f\n"
      "88:"  // Height 4: Multiply loop: Odd block loop
      "ldr s0, [x26], #0x4\n"
      "ldr s1, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr s2, [x24], #0x4\n"
      "ldr s3, [x23], #0x4\n"
      "cmp x27, #0x4\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "bge 88b\n"
      "89:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x27, 92f\n"
      "tbz x27, #1, 90f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "tbz x27, #0, 91f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "b 91f\n"
      "90:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "91:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "92:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 82b\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v12.4s, v12.4s, v0.4s\n"
      "add v13.4s, v13.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "add x25, x26, x20\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add v14.4s, v14.4s, v2.4s\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "add v15.4s, v15.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v20.4s, v20.4s, v0.4s\n"
      "add v21.4s, v21.4s, v1.4s\n"
      "add v22.4s, v22.4s, v2.4s\n"
      "add v23.4s, v23.4s, v3.4s\n"
      "tbz %x[flags], #4, 93f\n"
      "ldr q0, [x13, #0x0]\n"
      "ldr q4, [x14, #0x0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 94f\n"
      "93:"  // Height 4: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "94:"  // Height 4: parameters loaded
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v12.4s, v12.4s, v4.4s\n"
      "sqdmulh v13.4s, v13.4s, v5.4s\n"
      "sqdmulh v14.4s, v14.4s, v6.4s\n"
      "sqdmulh v15.4s, v15.4s, v7.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v5.4s\n"
      "sqdmulh v18.4s, v18.4s, v6.4s\n"
      "sqdmulh v19.4s, v19.4s, v7.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v21.4s, v21.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v22.4s, v22.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v23.4s, v23.4s, v7.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v1.4s\n"
      "srshl v22.4s, v22.4s, v2.4s\n"
      "srshl v23.4s, v23.4s, v3.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v13.8h, v14.8h, v15.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v21.8h, v22.8h, v23.8h\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v12.16b, v12.16b, v13.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v20.16b, v20.16b, v21.16b\n"
      "bge 103f\n"
      "tbz x11, #3, 98f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x27], #0x8\n"
      "str d16, [x26], #0x8\n"
      "str d20, [x25], #0x8\n"
      "tbz x11, #2, 96f\n"
      "st1 { v8.s }[2], [x9], #0x4\n"
      "st1 { v12.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x26], #0x4\n"
      "st1 { v20.s }[2], [x25], #0x4\n"
      "tbz x11, #1, 95f\n"
      "st1 { v8.h }[6], [x9], #0x2\n"
      "st1 { v12.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x26], #0x2\n"
      "st1 { v20.h }[6], [x25], #0x2\n"
      "tbz x11, #0, 102f\n"
      "st1 { v8.b }[14], [x9]\n"
      "st1 { v12.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x26]\n"
      "st1 { v20.b }[14], [x25]\n"
      "b 102f\n"
      "95:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x11, #0, 102f\n"
      "st1 { v8.b }[12], [x9]\n"
      "st1 { v12.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x26]\n"
      "st1 { v20.b }[12], [x25]\n"
      "b 102f\n"
      "96:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x11, #1, 97f\n"
      "st1 { v8.h }[4], [x9], #0x2\n"
      "st1 { v12.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x26], #0x2\n"
      "st1 { v20.h }[4], [x25], #0x2\n"
      "tbz x11, #0, 102f\n"
      "st1 { v8.b }[10], [x9]\n"
      "st1 { v12.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x26]\n"
      "st1 { v20.b }[10], [x25]\n"
      "b 102f\n"
      "97:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x11, #0, 102f\n"
      "st1 { v8.b }[8], [x9]\n"
      "st1 { v12.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x26]\n"
      "st1 { v20.b }[8], [x25]\n"
      "b 102f\n"
      "98:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x11, #2, 100f\n"
      "str s8, [x9], #0x4\n"
      "str s12, [x27], #0x4\n"
      "str s16, [x26], #0x4\n"
      "str s20, [x25], #0x4\n"
      "tbz x11, #1, 99f\n"
      "st1 { v8.h }[2], [x9], #0x2\n"
      "st1 { v12.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x26], #0x2\n"
      "st1 { v20.h }[2], [x25], #0x2\n"
      "tbz x11, #0, 102f\n"
      "st1 { v8.b }[6], [x9]\n"
      "st1 { v12.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x26]\n"
      "st1 { v20.b }[6], [x25]\n"
      "b 102f\n"
      "99:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x11, #0, 102f\n"
      "st1 { v8.b }[4], [x9]\n"
      "st1 { v12.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x26]\n"
      "st1 { v20.b }[4], [x25]\n"
      "b 102f\n"
      "100:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x11, #1, 101f\n"
      "str h8, [x9], #0x2\n"
      "str h12, [x27], #0x2\n"
      "str h16, [x26], #0x2\n"
      "str h20, [x25], #0x2\n"
      "tbz x11, #0, 102f\n"
      "st1 { v8.b }[2], [x9]\n"
      "st1 { v12.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x26]\n"
      "st1 { v20.b }[2], [x25]\n"
      "b 102f\n"
      "101:"  // Height 4: Partial direct writeback: partial_1_0
      "str b8, [x9, #0x0]\n"
      "str b12, [x27, #0x0]\n"
      "str b16, [x26, #0x0]\n"
      "str b20, [x25, #0x0]\n"
      "102:"  // Height 4: Partial direct writeback: Done
      "b 104f\n"
      "103:"  // Height 4: Full writeback
      "str q8, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "str q12, [x27, #0x0]\n"
      "str q16, [x26, #0x0]\n"
      "str q20, [x25, #0x0]\n"
      "104:"  // Height 4: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 80b\n"
      "b 158f\n"
      "105:"  // Height 5
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "106:"  // Height 5: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "mov x28, #0x0\n"
      "108:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 109f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 110f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 110f\n"
      "109:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "110:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "blt 113f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 112f\n"
      "111:"  // Height 5: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "add x22, x22, #0x10\n"
      "cmp x27, #0x20\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 111b\n"
      "112:"  // Height 5: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      "113:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 118f\n"
      "cmp x27, #0x4\n"
      "blt 115f\n"
      "114:"  // Height 5: Multiply loop: Odd block loop
      "ldr s0, [x26], #0x4\n"
      "ldr s1, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr s2, [x24], #0x4\n"
      "ldr s3, [x23], #0x4\n"
      "cmp x27, #0x4\n"
      "ldr s4, [x22], #0x4\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "bge 114b\n"
      "115:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x27, 118f\n"
      "tbz x27, #1, 116f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "tbz x27, #0, 117f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "ld1 { v4.b }[2], [x22]\n"
      "b 117f\n"
      "116:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "ldr b4, [x22, #0x0]\n"
      "117:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "118:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 108b\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v12.4s, v12.4s, v0.4s\n"
      "add v13.4s, v13.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "add x25, x26, x20\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add v14.4s, v14.4s, v2.4s\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "add v15.4s, v15.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v20.4s, v20.4s, v0.4s\n"
      "add v21.4s, v21.4s, v1.4s\n"
      "add v22.4s, v22.4s, v2.4s\n"
      "add v23.4s, v23.4s, v3.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "add v25.4s, v25.4s, v1.4s\n"
      "add v26.4s, v26.4s, v2.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "tbz %x[flags], #4, 119f\n"
      "ldr q0, [x13, #0x0]\n"
      "ldr q4, [x14, #0x0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 120f\n"
      "119:"  // Height 5: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "120:"  // Height 5: parameters loaded
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v12.4s, v12.4s, v4.4s\n"
      "sqdmulh v13.4s, v13.4s, v5.4s\n"
      "sqdmulh v14.4s, v14.4s, v6.4s\n"
      "sqdmulh v15.4s, v15.4s, v7.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v5.4s\n"
      "sqdmulh v18.4s, v18.4s, v6.4s\n"
      "sqdmulh v19.4s, v19.4s, v7.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "sqdmulh v21.4s, v21.4s, v5.4s\n"
      "sqdmulh v22.4s, v22.4s, v6.4s\n"
      "sqdmulh v23.4s, v23.4s, v7.4s\n"
      "sqdmulh v24.4s, v24.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v25.4s, v25.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v26.4s, v26.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v27.4s, v27.4s, v7.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v1.4s\n"
      "srshl v22.4s, v22.4s, v2.4s\n"
      "srshl v23.4s, v23.4s, v3.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v1.4s\n"
      "srshl v26.4s, v26.4s, v2.4s\n"
      "srshl v27.4s, v27.4s, v3.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v4.4s\n"
      "add v27.4s, v27.4s, v4.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v24.4s, v24.4s, v6.4s\n"
      "smin v25.4s, v25.4s, v6.4s\n"
      "smin v26.4s, v26.4s, v6.4s\n"
      "smin v27.4s, v27.4s, v6.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v24.4s, v24.4s, v5.4s\n"
      "smax v25.4s, v25.4s, v5.4s\n"
      "smax v26.4s, v26.4s, v5.4s\n"
      "smax v27.4s, v27.4s, v5.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v13.8h, v14.8h, v15.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v21.8h, v22.8h, v23.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v25.8h, v26.8h, v27.8h\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v12.16b, v12.16b, v13.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v20.16b, v20.16b, v21.16b\n"
      "uzp1 v24.16b, v24.16b, v25.16b\n"
      "bge 129f\n"
      "tbz x11, #3, 124f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x27], #0x8\n"
      "str d16, [x26], #0x8\n"
      "str d20, [x25], #0x8\n"
      "str d24, [x24], #0x8\n"
      "tbz x11, #2, 122f\n"
      "st1 { v8.s }[2], [x9], #0x4\n"
      "st1 { v12.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x26], #0x4\n"
      "st1 { v20.s }[2], [x25], #0x4\n"
      "st1 { v24.s }[2], [x24], #0x4\n"
      "tbz x11, #1, 121f\n"
      "st1 { v8.h }[6], [x9], #0x2\n"
      "st1 { v12.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x26], #0x2\n"
      "st1 { v20.h }[6], [x25], #0x2\n"
      "st1 { v24.h }[6], [x24], #0x2\n"
      "tbz x11, #0, 128f\n"
      "st1 { v8.b }[14], [x9]\n"
      "st1 { v12.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x26]\n"
      "st1 { v20.b }[14], [x25]\n"
      "st1 { v24.b }[14], [x24]\n"
      "b 128f\n"
      "121:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x11, #0, 128f\n"
      "st1 { v8.b }[12], [x9]\n"
      "st1 { v12.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x26]\n"
      "st1 { v20.b }[12], [x25]\n"
      "st1 { v24.b }[12], [x24]\n"
      "b 128f\n"
      "122:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x11, #1, 123f\n"
      "st1 { v8.h }[4], [x9], #0x2\n"
      "st1 { v12.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x26], #0x2\n"
      "st1 { v20.h }[4], [x25], #0x2\n"
      "st1 { v24.h }[4], [x24], #0x2\n"
      "tbz x11, #0, 128f\n"
      "st1 { v8.b }[10], [x9]\n"
      "st1 { v12.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x26]\n"
      "st1 { v20.b }[10], [x25]\n"
      "st1 { v24.b }[10], [x24]\n"
      "b 128f\n"
      "123:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x11, #0, 128f\n"
      "st1 { v8.b }[8], [x9]\n"
      "st1 { v12.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x26]\n"
      "st1 { v20.b }[8], [x25]\n"
      "st1 { v24.b }[8], [x24]\n"
      "b 128f\n"
      "124:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x11, #2, 126f\n"
      "str s8, [x9], #0x4\n"
      "str s12, [x27], #0x4\n"
      "str s16, [x26], #0x4\n"
      "str s20, [x25], #0x4\n"
      "str s24, [x24], #0x4\n"
      "tbz x11, #1, 125f\n"
      "st1 { v8.h }[2], [x9], #0x2\n"
      "st1 { v12.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x26], #0x2\n"
      "st1 { v20.h }[2], [x25], #0x2\n"
      "st1 { v24.h }[2], [x24], #0x2\n"
      "tbz x11, #0, 128f\n"
      "st1 { v8.b }[6], [x9]\n"
      "st1 { v12.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x26]\n"
      "st1 { v20.b }[6], [x25]\n"
      "st1 { v24.b }[6], [x24]\n"
      "b 128f\n"
      "125:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x11, #0, 128f\n"
      "st1 { v8.b }[4], [x9]\n"
      "st1 { v12.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x26]\n"
      "st1 { v20.b }[4], [x25]\n"
      "st1 { v24.b }[4], [x24]\n"
      "b 128f\n"
      "126:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x11, #1, 127f\n"
      "str h8, [x9], #0x2\n"
      "str h12, [x27], #0x2\n"
      "str h16, [x26], #0x2\n"
      "str h20, [x25], #0x2\n"
      "str h24, [x24], #0x2\n"
      "tbz x11, #0, 128f\n"
      "st1 { v8.b }[2], [x9]\n"
      "st1 { v12.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x26]\n"
      "st1 { v20.b }[2], [x25]\n"
      "st1 { v24.b }[2], [x24]\n"
      "b 128f\n"
      "127:"  // Height 5: Partial direct writeback: partial_1_0
      "str b8, [x9, #0x0]\n"
      "str b12, [x27, #0x0]\n"
      "str b16, [x26, #0x0]\n"
      "str b20, [x25, #0x0]\n"
      "str b24, [x24, #0x0]\n"
      "128:"  // Height 5: Partial direct writeback: Done
      "b 130f\n"
      "129:"  // Height 5: Full writeback
      "str q8, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "str q12, [x27, #0x0]\n"
      "str q16, [x26, #0x0]\n"
      "str q20, [x25, #0x0]\n"
      "str q24, [x24, #0x0]\n"
      "130:"  // Height 5: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 106b\n"
      "b 158f\n"
      "131:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x6\n"
      "mov x12, %x[col_bias]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "132:"  // Height 6: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "mov x28, #0x0\n"
      "134:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 135f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 136f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 136f\n"
      "135:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "136:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "blt 139f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q5, [x21, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 138f\n"
      "137:"  // Height 6: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0dc  // sdot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0fd  // sdot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0de  // sdot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0ff  // sdot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8dc  // sdot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8fd  // sdot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8de  // sdot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8ff  // sdot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8dc  // sdot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8fd  // sdot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8de  // sdot v30.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      "ldr q4, [x22, #0x0]\n"
      ".inst 0x4fa5e8ff  // sdot v31.4s, v7.16b, v5.4b[3]\n"
      "ldr q5, [x21, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 137b\n"
      "138:"  // Height 6: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x21, x21, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0dc  // sdot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0fd  // sdot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0de  // sdot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0ff  // sdot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8dc  // sdot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8fd  // sdot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8de  // sdot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8ff  // sdot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8dc  // sdot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8fd  // sdot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8de  // sdot v30.4s, v6.16b, v5.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8ff  // sdot v31.4s, v7.16b, v5.4b[3]\n"
      "139:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 144f\n"
      "cmp x27, #0x4\n"
      "blt 141f\n"
      "140:"  // Height 6: Multiply loop: Odd block loop
      "ldr s0, [x26], #0x4\n"
      "ldr s1, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr s2, [x24], #0x4\n"
      "ldr s3, [x23], #0x4\n"
      "cmp x27, #0x4\n"
      "ldr s4, [x22], #0x4\n"
      "ldr s5, [x21], #0x4\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "bge 140b\n"
      "141:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x27, 144f\n"
      "tbz x27, #1, 142f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "ldr h5, [x21], #0x2\n"
      "tbz x27, #0, 143f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "ld1 { v4.b }[2], [x22]\n"
      "ld1 { v5.b }[2], [x21]\n"
      "b 143f\n"
      "142:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "ldr b4, [x22, #0x0]\n"
      "ldr b5, [x21, #0x0]\n"
      "143:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "144:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 134b\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v12.4s, v12.4s, v0.4s\n"
      "add v13.4s, v13.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "add x25, x26, x20\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add v14.4s, v14.4s, v2.4s\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "add v15.4s, v15.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add x23, x24, x20\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v20.4s, v20.4s, v0.4s\n"
      "add v21.4s, v21.4s, v1.4s\n"
      "add v22.4s, v22.4s, v2.4s\n"
      "add v23.4s, v23.4s, v3.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "add v25.4s, v25.4s, v1.4s\n"
      "add v26.4s, v26.4s, v2.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "add v28.4s, v28.4s, v0.4s\n"
      "add v29.4s, v29.4s, v1.4s\n"
      "add v30.4s, v30.4s, v2.4s\n"
      "add v31.4s, v31.4s, v3.4s\n"
      "tbz %x[flags], #4, 145f\n"
      "ldr q0, [x13, #0x0]\n"
      "ldr q4, [x14, #0x0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 146f\n"
      "145:"  // Height 6: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "146:"  // Height 6: parameters loaded
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v12.4s, v12.4s, v4.4s\n"
      "sqdmulh v13.4s, v13.4s, v5.4s\n"
      "sqdmulh v14.4s, v14.4s, v6.4s\n"
      "sqdmulh v15.4s, v15.4s, v7.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v5.4s\n"
      "sqdmulh v18.4s, v18.4s, v6.4s\n"
      "sqdmulh v19.4s, v19.4s, v7.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "sqdmulh v21.4s, v21.4s, v5.4s\n"
      "sqdmulh v22.4s, v22.4s, v6.4s\n"
      "sqdmulh v23.4s, v23.4s, v7.4s\n"
      "sqdmulh v24.4s, v24.4s, v4.4s\n"
      "sqdmulh v25.4s, v25.4s, v5.4s\n"
      "sqdmulh v26.4s, v26.4s, v6.4s\n"
      "sqdmulh v27.4s, v27.4s, v7.4s\n"
      "sqdmulh v28.4s, v28.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v29.4s, v29.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v30.4s, v30.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v31.4s, v31.4s, v7.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v1.4s\n"
      "srshl v22.4s, v22.4s, v2.4s\n"
      "srshl v23.4s, v23.4s, v3.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v1.4s\n"
      "srshl v26.4s, v26.4s, v2.4s\n"
      "srshl v27.4s, v27.4s, v3.4s\n"
      "srshl v28.4s, v28.4s, v0.4s\n"
      "srshl v29.4s, v29.4s, v1.4s\n"
      "srshl v30.4s, v30.4s, v2.4s\n"
      "srshl v31.4s, v31.4s, v3.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v4.4s\n"
      "add v27.4s, v27.4s, v4.4s\n"
      "add v28.4s, v28.4s, v4.4s\n"
      "add v29.4s, v29.4s, v4.4s\n"
      "add v30.4s, v30.4s, v4.4s\n"
      "add v31.4s, v31.4s, v4.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v24.4s, v24.4s, v6.4s\n"
      "smin v25.4s, v25.4s, v6.4s\n"
      "smin v26.4s, v26.4s, v6.4s\n"
      "smin v27.4s, v27.4s, v6.4s\n"
      "smin v28.4s, v28.4s, v6.4s\n"
      "smin v29.4s, v29.4s, v6.4s\n"
      "smin v30.4s, v30.4s, v6.4s\n"
      "smin v31.4s, v31.4s, v6.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v24.4s, v24.4s, v5.4s\n"
      "smax v25.4s, v25.4s, v5.4s\n"
      "smax v26.4s, v26.4s, v5.4s\n"
      "smax v27.4s, v27.4s, v5.4s\n"
      "smax v28.4s, v28.4s, v5.4s\n"
      "smax v29.4s, v29.4s, v5.4s\n"
      "smax v30.4s, v30.4s, v5.4s\n"
      "smax v31.4s, v31.4s, v5.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v13.8h, v14.8h, v15.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v21.8h, v22.8h, v23.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v25.8h, v26.8h, v27.8h\n"
      "uzp1 v28.8h, v28.8h, v29.8h\n"
      "uzp1 v29.8h, v30.8h, v31.8h\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v12.16b, v12.16b, v13.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v20.16b, v20.16b, v21.16b\n"
      "uzp1 v24.16b, v24.16b, v25.16b\n"
      "uzp1 v28.16b, v28.16b, v29.16b\n"
      "bge 155f\n"
      "tbz x11, #3, 150f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x27], #0x8\n"
      "str d16, [x26], #0x8\n"
      "str d20, [x25], #0x8\n"
      "str d24, [x24], #0x8\n"
      "str d28, [x23], #0x8\n"
      "tbz x11, #2, 148f\n"
      "st1 { v8.s }[2], [x9], #0x4\n"
      "st1 { v12.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x26], #0x4\n"
      "st1 { v20.s }[2], [x25], #0x4\n"
      "st1 { v24.s }[2], [x24], #0x4\n"
      "st1 { v28.s }[2], [x23], #0x4\n"
      "tbz x11, #1, 147f\n"
      "st1 { v8.h }[6], [x9], #0x2\n"
      "st1 { v12.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x26], #0x2\n"
      "st1 { v20.h }[6], [x25], #0x2\n"
      "st1 { v24.h }[6], [x24], #0x2\n"
      "st1 { v28.h }[6], [x23], #0x2\n"
      "tbz x11, #0, 154f\n"
      "st1 { v8.b }[14], [x9]\n"
      "st1 { v12.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x26]\n"
      "st1 { v20.b }[14], [x25]\n"
      "st1 { v24.b }[14], [x24]\n"
      "st1 { v28.b }[14], [x23]\n"
      "b 154f\n"
      "147:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x11, #0, 154f\n"
      "st1 { v8.b }[12], [x9]\n"
      "st1 { v12.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x26]\n"
      "st1 { v20.b }[12], [x25]\n"
      "st1 { v24.b }[12], [x24]\n"
      "st1 { v28.b }[12], [x23]\n"
      "b 154f\n"
      "148:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x11, #1, 149f\n"
      "st1 { v8.h }[4], [x9], #0x2\n"
      "st1 { v12.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x26], #0x2\n"
      "st1 { v20.h }[4], [x25], #0x2\n"
      "st1 { v24.h }[4], [x24], #0x2\n"
      "st1 { v28.h }[4], [x23], #0x2\n"
      "tbz x11, #0, 154f\n"
      "st1 { v8.b }[10], [x9]\n"
      "st1 { v12.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x26]\n"
      "st1 { v20.b }[10], [x25]\n"
      "st1 { v24.b }[10], [x24]\n"
      "st1 { v28.b }[10], [x23]\n"
      "b 154f\n"
      "149:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x11, #0, 154f\n"
      "st1 { v8.b }[8], [x9]\n"
      "st1 { v12.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x26]\n"
      "st1 { v20.b }[8], [x25]\n"
      "st1 { v24.b }[8], [x24]\n"
      "st1 { v28.b }[8], [x23]\n"
      "b 154f\n"
      "150:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x11, #2, 152f\n"
      "str s8, [x9], #0x4\n"
      "str s12, [x27], #0x4\n"
      "str s16, [x26], #0x4\n"
      "str s20, [x25], #0x4\n"
      "str s24, [x24], #0x4\n"
      "str s28, [x23], #0x4\n"
      "tbz x11, #1, 151f\n"
      "st1 { v8.h }[2], [x9], #0x2\n"
      "st1 { v12.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x26], #0x2\n"
      "st1 { v20.h }[2], [x25], #0x2\n"
      "st1 { v24.h }[2], [x24], #0x2\n"
      "st1 { v28.h }[2], [x23], #0x2\n"
      "tbz x11, #0, 154f\n"
      "st1 { v8.b }[6], [x9]\n"
      "st1 { v12.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x26]\n"
      "st1 { v20.b }[6], [x25]\n"
      "st1 { v24.b }[6], [x24]\n"
      "st1 { v28.b }[6], [x23]\n"
      "b 154f\n"
      "151:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x11, #0, 154f\n"
      "st1 { v8.b }[4], [x9]\n"
      "st1 { v12.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x26]\n"
      "st1 { v20.b }[4], [x25]\n"
      "st1 { v24.b }[4], [x24]\n"
      "st1 { v28.b }[4], [x23]\n"
      "b 154f\n"
      "152:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x11, #1, 153f\n"
      "str h8, [x9], #0x2\n"
      "str h12, [x27], #0x2\n"
      "str h16, [x26], #0x2\n"
      "str h20, [x25], #0x2\n"
      "str h24, [x24], #0x2\n"
      "str h28, [x23], #0x2\n"
      "tbz x11, #0, 154f\n"
      "st1 { v8.b }[2], [x9]\n"
      "st1 { v12.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x26]\n"
      "st1 { v20.b }[2], [x25]\n"
      "st1 { v24.b }[2], [x24]\n"
      "st1 { v28.b }[2], [x23]\n"
      "b 154f\n"
      "153:"  // Height 6: Partial direct writeback: partial_1_0
      "str b8, [x9, #0x0]\n"
      "str b12, [x27, #0x0]\n"
      "str b16, [x26, #0x0]\n"
      "str b20, [x25, #0x0]\n"
      "str b24, [x24, #0x0]\n"
      "str b28, [x23, #0x0]\n"
      "154:"  // Height 6: Partial direct writeback: Done
      "b 156f\n"
      "155:"  // Height 6: Full writeback
      "str q8, [x9, #0x0]\n"
      "add x9, x9, #0x10\n"
      "str q12, [x27, #0x0]\n"
      "str q16, [x26, #0x0]\n"
      "str q20, [x25, #0x0]\n"
      "str q24, [x24, #0x0]\n"
      "str q28, [x23, #0x0]\n"
      "156:"  // Height 6: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 132b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 158f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 157f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "157:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "158:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [flags] "r" (flags), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_multiplier_ptr] "I" (offsetof(KernelArgs, multiplier_ptr)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_shift_ptr] "I" (offsetof(KernelArgs, shift_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
