/*
 * Copyright (c) 2019-2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include <cassert>
#include <cstdint>

namespace arm_gemm {

void a64_hybrid_s8qa_dot_4x16_a55 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 88f\n"
      "cmp %x[M], #0x2\n"
      "bgt 59f\n"
      "beq 30f\n"
      "movi v11.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "movi v15.16b, #0x1\n"
      "ldr x16, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "mov x12, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w11, [x20, x12, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x12, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x10, [x20, #0x0]\n"
      "cbnz x12, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x10, x10, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x10, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x11, #0x10\n"
      "blt 11f\n"
      "ldr q0, [x10, #0x0]\n"
      "cmp x11, #0x20\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "ldr q8, [x15, #0x40]\n"
      "ldr q9, [x15, #0x50]\n"
      "ldr q10, [x15, #0x60]\n"
      "blt 9f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "ldr d4, [x15, #0x70]\n"
      "ldr x20, [x15, #0x78]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "ldr d5, [x15, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      "ldr d6, [x15, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      "ldr d7, [x15, #0xa0]\n"
      "mov v4.d[1], x20\n"
      "ldr x20, [x15, #0x88]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      "ldr d8, [x15, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      "ldr d9, [x15, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      "ldr d10, [x15, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      "ldr d4, [x15, #0xe0]\n"
      "mov v5.d[1], x20\n"
      "ldr x22, [x15, #0x98]\n"
      "add x10, x10, #0x10\n"
      "ldr x21, [x15, #0xa8]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      "ldr d5, [x15, #0xf0]\n"
      "ldr x20, [x15, #0xb8]\n"
      "mov v6.d[1], x22\n"
      "mov v7.d[1], x21\n"
      "ldr x23, [x15, #0xc8]\n"
      "ldr x22, [x15, #0xd8]\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      "mov v8.d[1], x20\n"
      "ldr x21, [x15, #0xe8]\n"
      "ldr x20, [x15, #0xf8]\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      "mov v9.d[1], x23\n"
      "mov v10.d[1], x22\n"
      "add x15, x15, #0x100\n"
      "mov v4.d[1], x21\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      "mov v5.d[1], x20\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      "tbnz %x[flags], #31, 8f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "8:"  // Height 1: Multiply loop: unique 1: skip row sum
      "ldr q0, [x10, #0x0]\n"
      "sub x11, x11, #0x10\n"
      "ldr q4, [x15, #0x0]\n"
      "cmp x11, #0x20\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "ldr q8, [x15, #0x40]\n"
      "ldr q9, [x15, #0x50]\n"
      "ldr q10, [x15, #0x60]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "bge 7b\n"
      "9:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "ldr q4, [x15, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "ldr q5, [x15, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x15, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x15, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      "ldr q8, [x15, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      "ldr q9, [x15, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      "ldr q10, [x15, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      "ldr q4, [x15, #0xe0]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      "ldr q5, [x15, #0xf0]\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      "sub x11, x11, #0x10\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      "add x15, x15, #0x100\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      "tbnz %x[flags], #31, 10f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "10:"  // Height 1: Multiply loop: unique 2: skip row sum
      "prfm pldl1keep, [x10, #0x80]\n"
      "11:"  // Height 1: Multiply loop: Main loop skip
      "cbz x11, 18f\n"
      "cmp x11, #0x4\n"
      "blt 14f\n"
      "12:"  // Height 1: Multiply loop: Odd block loop
      "ldr s0, [x10], #0x4\n"
      "tbnz %x[flags], #31, 13f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "13:"  // Height 1: Multiply loop: unique 3: skip row sum
      "ldr q6, [x15, #0x0]\n"
      "sub x11, x11, #0x4\n"
      "ldr q7, [x15, #0x10]\n"
      "cmp x11, #0x4\n"
      "ldr q8, [x15, #0x20]\n"
      "ldr q9, [x15, #0x30]\n"
      ".inst 0x4f80e0d0  // sdot v16.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f80e0f1  // sdot v17.4s, v7.16b, v0.4b[0]\n"
      "add x15, x15, #0x40\n"
      ".inst 0x4f80e112  // sdot v18.4s, v8.16b, v0.4b[0]\n"
      ".inst 0x4f80e133  // sdot v19.4s, v9.16b, v0.4b[0]\n"
      "bge 12b\n"
      "14:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x11, 18f\n"
      "tbz x11, #1, 15f\n"
      "ldr h0, [x10], #0x2\n"
      "tbz x11, #0, 16f\n"
      "ld1 { v0.b }[2], [x10]\n"
      "b 16f\n"
      "15:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x10, #0x0]\n"
      "16:"  // Height 1: Multiply loop: Ragged operand read: Done
      "tbnz %x[flags], #31, 17f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "17:"  // Height 1: Multiply loop: unique 4: skip row sum
      "ldr q10, [x15, #0x0]\n"
      "ldr q4, [x15, #0x10]\n"
      "ldr q5, [x15, #0x20]\n"
      "ldr q6, [x15, #0x30]\n"
      ".inst 0x4f80e150  // sdot v16.4s, v10.16b, v0.4b[0]\n"
      ".inst 0x4f80e091  // sdot v17.4s, v4.16b, v0.4b[0]\n"
      "add x15, x15, #0x40\n"
      ".inst 0x4f80e0b2  // sdot v18.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f80e0d3  // sdot v19.4s, v6.16b, v0.4b[0]\n"
      "18:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x12, x12, #0x1\n"
      "cmp x12, x20\n"
      "bne 4b\n"
      "prfm pstl1keep, [x13, #0x0]\n"
      "tbnz %x[flags], #31, 19f\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1r { v1.4s }, [x20]\n"
      "neg v1.4s, v1.4s\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "mul v11.4s, v11.4s, v1.4s\n"
      "19:"  // Height 1: skip row sum fixup
      "ldr q0, [x14, #0x0]\n"
      "add v16.4s, v16.4s, v11.4s\n"
      "ldr q1, [x14, #0x10]\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q2, [x14, #0x20]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "ldr q3, [x14, #0x30]\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v4.4s }, [x21]\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add x21, %x[qp], %[c_offset]\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v6.4s }, [x20]\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v4.4s\n"
      "cmp x16, #0x10\n"
      "sqdmulh v18.4s, v18.4s, v4.4s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "sqdmulh v19.4s, v19.4s, v4.4s\n"
      "ld1r { v4.4s }, [x21]\n"
      "add x14, x14, #0x40\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 28f\n"
      "tbz x16, #3, 23f\n"
      "str d16, [x13], #0x8\n"
      "tbz x16, #2, 21f\n"
      "st1 { v16.s }[2], [x13], #0x4\n"
      "tbz x16, #1, 20f\n"
      "st1 { v16.h }[6], [x13], #0x2\n"
      "tbz x16, #0, 27f\n"
      "st1 { v16.b }[14], [x13]\n"
      "b 27f\n"
      "20:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x16, #0, 27f\n"
      "st1 { v16.b }[12], [x13]\n"
      "b 27f\n"
      "21:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x16, #1, 22f\n"
      "st1 { v16.h }[4], [x13], #0x2\n"
      "tbz x16, #0, 27f\n"
      "st1 { v16.b }[10], [x13]\n"
      "b 27f\n"
      "22:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x16, #0, 27f\n"
      "st1 { v16.b }[8], [x13]\n"
      "b 27f\n"
      "23:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x16, #2, 25f\n"
      "str s16, [x13], #0x4\n"
      "tbz x16, #1, 24f\n"
      "st1 { v16.h }[2], [x13], #0x2\n"
      "tbz x16, #0, 27f\n"
      "st1 { v16.b }[6], [x13]\n"
      "b 27f\n"
      "24:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x16, #0, 27f\n"
      "st1 { v16.b }[4], [x13]\n"
      "b 27f\n"
      "25:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x16, #1, 26f\n"
      "str h16, [x13], #0x2\n"
      "tbz x16, #0, 27f\n"
      "st1 { v16.b }[2], [x13]\n"
      "b 27f\n"
      "26:"  // Height 1: Partial direct writeback: partial_1_0
      "str b16, [x13, #0x0]\n"
      "27:"  // Height 1: Partial direct writeback: Done
      "b 29f\n"
      "28:"  // Height 1: Full writeback
      "str q16, [x13, #0x0]\n"
      "add x13, x13, #0x10\n"
      "29:"  // Height 1: Writeback done
      "subs x16, x16, #0x10\n"
      "bgt 2b\n"
      "b 118f\n"
      "30:"  // Height 2
      "movi v11.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "movi v12.4s, #0x0\n"
      "ldr x16, [%x[args_ptr], %[offsetof_N]]\n"
      "movi v15.16b, #0x1\n"
      "ldr x15, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "31:"  // Height 2: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "mov x12, #0x0\n"
      "33:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w11, [x20, x12, LSL #0x2]\n"
      "tbz %x[flags], #3, 34f\n"
      "ldr x20, [%x[input_ptr], x12, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x10, [x20, #0x0]\n"
      "ldr x9, [x20, #0x8]\n"
      "cbnz x12, 35f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x10, x10, x20\n"
      "add x9, x9, x20\n"
      "b 35f\n"
      "34:"  // Height 2: setup direct input
      "mov x10, %x[input_ptr]\n"
      "add x9, x10, x21\n"
      "35:"  // Height 2: input setup done
      "cmp x11, #0x10\n"
      "blt 40f\n"
      "ldr q0, [x10, #0x0]\n"
      "cmp x11, #0x20\n"
      "ldr q1, [x9, #0x0]\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "ldr q8, [x15, #0x40]\n"
      "ldr q9, [x15, #0x50]\n"
      "ldr q10, [x15, #0x60]\n"
      "blt 38f\n"
      "36:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "ldr x20, [x15, #0x78]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "ldr d4, [x15, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "ldr x23, [x15, #0x88]\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "ldr d5, [x15, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      "mov v4.d[1], x20\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      "ldr d6, [x15, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      "ldr x22, [x15, #0x98]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      "ldr d7, [x15, #0xa0]\n"
      "ldr x21, [x15, #0xa8]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      "ldr d8, [x15, #0xb0]\n"
      "ldr x20, [x15, #0xb8]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      "ldr d9, [x15, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      "mov v5.d[1], x23\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      "ldr d10, [x15, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      "mov v6.d[1], x22\n"
      ".inst 0x4fa1e097  // sdot v23.4s, v4.16b, v1.4b[1]\n"
      "ldr d4, [x15, #0xe0]\n"
      "mov v7.d[1], x21\n"
      "ldr x23, [x15, #0xc8]\n"
      "mov v8.d[1], x20\n"
      "ldr x22, [x15, #0xd8]\n"
      "ldr x21, [x15, #0xe8]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      ".inst 0x4f81e8b4  // sdot v20.4s, v5.16b, v1.4b[2]\n"
      "ldr d5, [x15, #0xf0]\n"
      "ldr x20, [x15, #0xf8]\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8d5  // sdot v21.4s, v6.16b, v1.4b[2]\n"
      "mov v9.d[1], x23\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      "mov v10.d[1], x22\n"
      ".inst 0x4f81e8f6  // sdot v22.4s, v7.16b, v1.4b[2]\n"
      "mov v4.d[1], x21\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      "mov v5.d[1], x20\n"
      ".inst 0x4f81e917  // sdot v23.4s, v8.16b, v1.4b[2]\n"
      "add x10, x10, #0x10\n"
      "add x9, x9, #0x10\n"
      "add x15, x15, #0x100\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa1e934  // sdot v20.4s, v9.16b, v1.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa1e955  // sdot v21.4s, v10.16b, v1.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa1e896  // sdot v22.4s, v4.16b, v1.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8b7  // sdot v23.4s, v5.16b, v1.4b[3]\n"
      "tbnz %x[flags], #31, 37f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      "37:"  // Height 2: Multiply loop: unique 5: skip row sum
      "ldr q0, [x10, #0x0]\n"
      "sub x11, x11, #0x10\n"
      "ldr q1, [x9, #0x0]\n"
      "cmp x11, #0x20\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "ldr q8, [x15, #0x40]\n"
      "ldr q9, [x15, #0x50]\n"
      "ldr q10, [x15, #0x60]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "bge 36b\n"
      "38:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "sub x11, x11, #0x10\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "ldr q4, [x15, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "ldr q5, [x15, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x15, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x15, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      "ldr q8, [x15, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      "ldr q9, [x15, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      "ldr q10, [x15, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      ".inst 0x4fa1e097  // sdot v23.4s, v4.16b, v1.4b[1]\n"
      "ldr q4, [x15, #0xe0]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      ".inst 0x4f81e8b4  // sdot v20.4s, v5.16b, v1.4b[2]\n"
      "ldr q5, [x15, #0xf0]\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      "add x15, x15, #0x100\n"
      ".inst 0x4f81e8d5  // sdot v21.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8f6  // sdot v22.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      ".inst 0x4f81e917  // sdot v23.4s, v8.16b, v1.4b[2]\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa1e934  // sdot v20.4s, v9.16b, v1.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa1e955  // sdot v21.4s, v10.16b, v1.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa1e896  // sdot v22.4s, v4.16b, v1.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8b7  // sdot v23.4s, v5.16b, v1.4b[3]\n"
      "tbnz %x[flags], #31, 39f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      "39:"  // Height 2: Multiply loop: unique 6: skip row sum
      "prfm pldl1keep, [x10, #0x80]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "40:"  // Height 2: Multiply loop: Main loop skip
      "cbz x11, 47f\n"
      "cmp x11, #0x4\n"
      "blt 43f\n"
      "41:"  // Height 2: Multiply loop: Odd block loop
      "ldr s0, [x10], #0x4\n"
      "ldr s1, [x9], #0x4\n"
      "tbnz %x[flags], #31, 42f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      "42:"  // Height 2: Multiply loop: unique 7: skip row sum
      "ldr q6, [x15, #0x0]\n"
      "sub x11, x11, #0x4\n"
      "ldr q7, [x15, #0x10]\n"
      "cmp x11, #0x4\n"
      "ldr q8, [x15, #0x20]\n"
      "ldr q9, [x15, #0x30]\n"
      ".inst 0x4f80e0d0  // sdot v16.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d4  // sdot v20.4s, v6.16b, v1.4b[0]\n"
      "add x15, x15, #0x40\n"
      ".inst 0x4f80e0f1  // sdot v17.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f5  // sdot v21.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f80e112  // sdot v18.4s, v8.16b, v0.4b[0]\n"
      ".inst 0x4f81e116  // sdot v22.4s, v8.16b, v1.4b[0]\n"
      ".inst 0x4f80e133  // sdot v19.4s, v9.16b, v0.4b[0]\n"
      ".inst 0x4f81e137  // sdot v23.4s, v9.16b, v1.4b[0]\n"
      "bge 41b\n"
      "43:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x11, 47f\n"
      "tbz x11, #1, 44f\n"
      "ldr h0, [x10], #0x2\n"
      "ldr h1, [x9], #0x2\n"
      "tbz x11, #0, 45f\n"
      "ld1 { v0.b }[2], [x10]\n"
      "ld1 { v1.b }[2], [x9]\n"
      "b 45f\n"
      "44:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x10, #0x0]\n"
      "ldr b1, [x9, #0x0]\n"
      "45:"  // Height 2: Multiply loop: Ragged operand read: Done
      "tbnz %x[flags], #31, 46f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      "46:"  // Height 2: Multiply loop: unique 8: skip row sum
      "ldr q10, [x15, #0x0]\n"
      "ldr q4, [x15, #0x10]\n"
      "ldr q5, [x15, #0x20]\n"
      "ldr q6, [x15, #0x30]\n"
      ".inst 0x4f80e150  // sdot v16.4s, v10.16b, v0.4b[0]\n"
      ".inst 0x4f81e154  // sdot v20.4s, v10.16b, v1.4b[0]\n"
      "add x15, x15, #0x40\n"
      ".inst 0x4f80e091  // sdot v17.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e095  // sdot v21.4s, v4.16b, v1.4b[0]\n"
      ".inst 0x4f80e0b2  // sdot v18.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b6  // sdot v22.4s, v5.16b, v1.4b[0]\n"
      ".inst 0x4f80e0d3  // sdot v19.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d7  // sdot v23.4s, v6.16b, v1.4b[0]\n"
      "47:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x12, x12, #0x1\n"
      "cmp x12, x20\n"
      "bne 33b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x13, #0x0]\n"
      "add x24, x13, x20\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "tbnz %x[flags], #31, 48f\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1r { v2.4s }, [x20]\n"
      "neg v2.4s, v2.4s\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "mul v11.4s, v11.4s, v2.4s\n"
      "mul v12.4s, v12.4s, v2.4s\n"
      "48:"  // Height 2: skip row sum fixup
      "ldr q0, [x14, #0x0]\n"
      "add v16.4s, v16.4s, v11.4s\n"
      "ldr q1, [x14, #0x10]\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q2, [x14, #0x20]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "ldr q3, [x14, #0x30]\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add v20.4s, v20.4s, v12.4s\n"
      "add v21.4s, v21.4s, v12.4s\n"
      "add v22.4s, v22.4s, v12.4s\n"
      "add v23.4s, v23.4s, v12.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v20.4s, v20.4s, v0.4s\n"
      "add v21.4s, v21.4s, v1.4s\n"
      "add v22.4s, v22.4s, v2.4s\n"
      "add v23.4s, v23.4s, v3.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v4.4s\n"
      "sqdmulh v18.4s, v18.4s, v4.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "sqdmulh v19.4s, v19.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "sqdmulh v21.4s, v21.4s, v4.4s\n"
      "sqdmulh v22.4s, v22.4s, v4.4s\n"
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v4.4s }, [x20]\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v6.4s }, [x20]\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v5.4s }, [x20]\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "cmp x16, #0x10\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "add x14, x14, #0x40\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v21.8h, v22.8h, v23.8h\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v20.16b, v20.16b, v21.16b\n"
      "bge 57f\n"
      "tbz x16, #3, 52f\n"
      "str d16, [x13], #0x8\n"
      "str d20, [x24], #0x8\n"
      "tbz x16, #2, 50f\n"
      "st1 { v16.s }[2], [x13], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "tbz x16, #1, 49f\n"
      "st1 { v16.h }[6], [x13], #0x2\n"
      "st1 { v20.h }[6], [x24], #0x2\n"
      "tbz x16, #0, 56f\n"
      "st1 { v16.b }[14], [x13]\n"
      "st1 { v20.b }[14], [x24]\n"
      "b 56f\n"
      "49:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x16, #0, 56f\n"
      "st1 { v16.b }[12], [x13]\n"
      "st1 { v20.b }[12], [x24]\n"
      "b 56f\n"
      "50:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x16, #1, 51f\n"
      "st1 { v16.h }[4], [x13], #0x2\n"
      "st1 { v20.h }[4], [x24], #0x2\n"
      "tbz x16, #0, 56f\n"
      "st1 { v16.b }[10], [x13]\n"
      "st1 { v20.b }[10], [x24]\n"
      "b 56f\n"
      "51:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x16, #0, 56f\n"
      "st1 { v16.b }[8], [x13]\n"
      "st1 { v20.b }[8], [x24]\n"
      "b 56f\n"
      "52:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x16, #2, 54f\n"
      "str s16, [x13], #0x4\n"
      "str s20, [x24], #0x4\n"
      "tbz x16, #1, 53f\n"
      "st1 { v16.h }[2], [x13], #0x2\n"
      "st1 { v20.h }[2], [x24], #0x2\n"
      "tbz x16, #0, 56f\n"
      "st1 { v16.b }[6], [x13]\n"
      "st1 { v20.b }[6], [x24]\n"
      "b 56f\n"
      "53:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x16, #0, 56f\n"
      "st1 { v16.b }[4], [x13]\n"
      "st1 { v20.b }[4], [x24]\n"
      "b 56f\n"
      "54:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x16, #1, 55f\n"
      "str h16, [x13], #0x2\n"
      "str h20, [x24], #0x2\n"
      "tbz x16, #0, 56f\n"
      "st1 { v16.b }[2], [x13]\n"
      "st1 { v20.b }[2], [x24]\n"
      "b 56f\n"
      "55:"  // Height 2: Partial direct writeback: partial_1_0
      "str b16, [x13, #0x0]\n"
      "str b20, [x24, #0x0]\n"
      "56:"  // Height 2: Partial direct writeback: Done
      "b 58f\n"
      "57:"  // Height 2: Full writeback
      "str q16, [x13, #0x0]\n"
      "add x13, x13, #0x10\n"
      "str q20, [x24, #0x0]\n"
      "58:"  // Height 2: Writeback done
      "subs x16, x16, #0x10\n"
      "bgt 31b\n"
      "b 118f\n"
      "59:"  // Height 3
      "movi v11.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "movi v12.4s, #0x0\n"
      "ldr x16, [%x[args_ptr], %[offsetof_N]]\n"
      "movi v13.4s, #0x0\n"
      "ldr x15, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "movi v15.16b, #0x1\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "60:"  // Height 3: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "mov x12, #0x0\n"
      "62:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w11, [x20, x12, LSL #0x2]\n"
      "tbz %x[flags], #3, 63f\n"
      "ldr x20, [%x[input_ptr], x12, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x10, [x20, #0x0]\n"
      "ldr x9, [x20, #0x8]\n"
      "ldr x28, [x20, #0x10]\n"
      "cbnz x12, 64f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x10, x10, x20\n"
      "add x9, x9, x20\n"
      "add x28, x28, x20\n"
      "b 64f\n"
      "63:"  // Height 3: setup direct input
      "mov x10, %x[input_ptr]\n"
      "add x9, x10, x21\n"
      "add x28, x9, x21\n"
      "64:"  // Height 3: input setup done
      "cmp x11, #0x10\n"
      "blt 69f\n"
      "ldr q0, [x10, #0x0]\n"
      "cmp x11, #0x20\n"
      "ldr q1, [x9, #0x0]\n"
      "ldr q2, [x28, #0x0]\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "ldr q8, [x15, #0x40]\n"
      "ldr q9, [x15, #0x50]\n"
      "ldr q10, [x15, #0x60]\n"
      "blt 67f\n"
      "65:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "ldr x20, [x15, #0x78]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "ldr x23, [x15, #0x88]\n"
      ".inst 0x4f82e098  // sdot v24.4s, v4.16b, v2.4b[0]\n"
      "ldr d4, [x15, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "ldr x22, [x15, #0x98]\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "ldr x21, [x15, #0xa8]\n"
      ".inst 0x4f82e0b9  // sdot v25.4s, v5.16b, v2.4b[0]\n"
      "ldr d5, [x15, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      "mov v4.d[1], x20\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x15, #0xb8]\n"
      ".inst 0x4f82e0da  // sdot v26.4s, v6.16b, v2.4b[0]\n"
      "ldr d6, [x15, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      "mov v5.d[1], x23\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      "ldr x23, [x15, #0xc8]\n"
      ".inst 0x4f82e0fb  // sdot v27.4s, v7.16b, v2.4b[0]\n"
      "ldr d7, [x15, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      "mov v6.d[1], x22\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      "ldr x22, [x15, #0xd8]\n"
      ".inst 0x4fa2e118  // sdot v24.4s, v8.16b, v2.4b[1]\n"
      "ldr d8, [x15, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      "ldr x21, [x15, #0xe8]\n"
      ".inst 0x4fa2e139  // sdot v25.4s, v9.16b, v2.4b[1]\n"
      "ldr d9, [x15, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      "mov v8.d[1], x20\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      "ldr x20, [x15, #0xf8]\n"
      ".inst 0x4fa2e15a  // sdot v26.4s, v10.16b, v2.4b[1]\n"
      "ldr d10, [x15, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      "mov v9.d[1], x23\n"
      ".inst 0x4fa1e097  // sdot v23.4s, v4.16b, v1.4b[1]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4fa2e09b  // sdot v27.4s, v4.16b, v2.4b[1]\n"
      "ldr d4, [x15, #0xe0]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      "mov v10.d[1], x22\n"
      ".inst 0x4f81e8b4  // sdot v20.4s, v5.16b, v1.4b[2]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f82e8b8  // sdot v24.4s, v5.16b, v2.4b[2]\n"
      "ldr d5, [x15, #0xf0]\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      "mov v4.d[1], x21\n"
      ".inst 0x4f81e8d5  // sdot v21.4s, v6.16b, v1.4b[2]\n"
      "add x28, x28, #0x10\n"
      ".inst 0x4f82e8d9  // sdot v25.4s, v6.16b, v2.4b[2]\n"
      "mov v5.d[1], x20\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      "add x15, x15, #0x100\n"
      ".inst 0x4f81e8f6  // sdot v22.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8fa  // sdot v26.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      ".inst 0x4f81e917  // sdot v23.4s, v8.16b, v1.4b[2]\n"
      ".inst 0x4f82e91b  // sdot v27.4s, v8.16b, v2.4b[2]\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa1e934  // sdot v20.4s, v9.16b, v1.4b[3]\n"
      ".inst 0x4fa2e938  // sdot v24.4s, v9.16b, v2.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa1e955  // sdot v21.4s, v10.16b, v1.4b[3]\n"
      ".inst 0x4fa2e959  // sdot v25.4s, v10.16b, v2.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa1e896  // sdot v22.4s, v4.16b, v1.4b[3]\n"
      ".inst 0x4fa2e89a  // sdot v26.4s, v4.16b, v2.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8b7  // sdot v23.4s, v5.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8bb  // sdot v27.4s, v5.16b, v2.4b[3]\n"
      "tbnz %x[flags], #31, 66f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "66:"  // Height 3: Multiply loop: unique 9: skip row sum
      "ldr q0, [x10, #0x0]\n"
      "sub x11, x11, #0x10\n"
      "ldr q1, [x9, #0x0]\n"
      "cmp x11, #0x20\n"
      "ldr q2, [x28, #0x0]\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "ldr q8, [x15, #0x40]\n"
      "ldr q9, [x15, #0x50]\n"
      "ldr q10, [x15, #0x60]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "bge 65b\n"
      "67:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "sub x11, x11, #0x10\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f82e098  // sdot v24.4s, v4.16b, v2.4b[0]\n"
      "ldr q4, [x15, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "add x28, x28, #0x10\n"
      ".inst 0x4f82e0b9  // sdot v25.4s, v5.16b, v2.4b[0]\n"
      "ldr q5, [x15, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0da  // sdot v26.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x15, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0fb  // sdot v27.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x15, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      ".inst 0x4fa2e118  // sdot v24.4s, v8.16b, v2.4b[1]\n"
      "ldr q8, [x15, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      ".inst 0x4fa2e139  // sdot v25.4s, v9.16b, v2.4b[1]\n"
      "ldr q9, [x15, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      ".inst 0x4fa2e15a  // sdot v26.4s, v10.16b, v2.4b[1]\n"
      "ldr q10, [x15, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      ".inst 0x4fa1e097  // sdot v23.4s, v4.16b, v1.4b[1]\n"
      ".inst 0x4fa2e09b  // sdot v27.4s, v4.16b, v2.4b[1]\n"
      "ldr q4, [x15, #0xe0]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      ".inst 0x4f81e8b4  // sdot v20.4s, v5.16b, v1.4b[2]\n"
      ".inst 0x4f82e8b8  // sdot v24.4s, v5.16b, v2.4b[2]\n"
      "ldr q5, [x15, #0xf0]\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      "add x15, x15, #0x100\n"
      ".inst 0x4f81e8d5  // sdot v21.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d9  // sdot v25.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8f6  // sdot v22.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8fa  // sdot v26.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      ".inst 0x4f81e917  // sdot v23.4s, v8.16b, v1.4b[2]\n"
      ".inst 0x4f82e91b  // sdot v27.4s, v8.16b, v2.4b[2]\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa1e934  // sdot v20.4s, v9.16b, v1.4b[3]\n"
      ".inst 0x4fa2e938  // sdot v24.4s, v9.16b, v2.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa1e955  // sdot v21.4s, v10.16b, v1.4b[3]\n"
      ".inst 0x4fa2e959  // sdot v25.4s, v10.16b, v2.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa1e896  // sdot v22.4s, v4.16b, v1.4b[3]\n"
      ".inst 0x4fa2e89a  // sdot v26.4s, v4.16b, v2.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8b7  // sdot v23.4s, v5.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8bb  // sdot v27.4s, v5.16b, v2.4b[3]\n"
      "tbnz %x[flags], #31, 68f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "68:"  // Height 3: Multiply loop: unique 10: skip row sum
      "prfm pldl1keep, [x10, #0x80]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "69:"  // Height 3: Multiply loop: Main loop skip
      "cbz x11, 76f\n"
      "cmp x11, #0x4\n"
      "blt 72f\n"
      "70:"  // Height 3: Multiply loop: Odd block loop
      "ldr s0, [x10], #0x4\n"
      "ldr s1, [x9], #0x4\n"
      "ldr s2, [x28], #0x4\n"
      "tbnz %x[flags], #31, 71f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "71:"  // Height 3: Multiply loop: unique 11: skip row sum
      "ldr q6, [x15, #0x0]\n"
      "sub x11, x11, #0x4\n"
      "ldr q7, [x15, #0x10]\n"
      "cmp x11, #0x4\n"
      "ldr q8, [x15, #0x20]\n"
      "ldr q9, [x15, #0x30]\n"
      ".inst 0x4f80e0d0  // sdot v16.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d4  // sdot v20.4s, v6.16b, v1.4b[0]\n"
      "add x15, x15, #0x40\n"
      ".inst 0x4f82e0d8  // sdot v24.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f80e0f1  // sdot v17.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f5  // sdot v21.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f9  // sdot v25.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f80e112  // sdot v18.4s, v8.16b, v0.4b[0]\n"
      ".inst 0x4f81e116  // sdot v22.4s, v8.16b, v1.4b[0]\n"
      ".inst 0x4f82e11a  // sdot v26.4s, v8.16b, v2.4b[0]\n"
      ".inst 0x4f80e133  // sdot v19.4s, v9.16b, v0.4b[0]\n"
      ".inst 0x4f81e137  // sdot v23.4s, v9.16b, v1.4b[0]\n"
      ".inst 0x4f82e13b  // sdot v27.4s, v9.16b, v2.4b[0]\n"
      "bge 70b\n"
      "72:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x11, 76f\n"
      "tbz x11, #1, 73f\n"
      "ldr h0, [x10], #0x2\n"
      "ldr h1, [x9], #0x2\n"
      "ldr h2, [x28], #0x2\n"
      "tbz x11, #0, 74f\n"
      "ld1 { v0.b }[2], [x10]\n"
      "ld1 { v1.b }[2], [x9]\n"
      "ld1 { v2.b }[2], [x28]\n"
      "b 74f\n"
      "73:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x10, #0x0]\n"
      "ldr b1, [x9, #0x0]\n"
      "ldr b2, [x28, #0x0]\n"
      "74:"  // Height 3: Multiply loop: Ragged operand read: Done
      "tbnz %x[flags], #31, 75f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "75:"  // Height 3: Multiply loop: unique 12: skip row sum
      "ldr q10, [x15, #0x0]\n"
      "ldr q4, [x15, #0x10]\n"
      "ldr q5, [x15, #0x20]\n"
      "ldr q6, [x15, #0x30]\n"
      ".inst 0x4f80e150  // sdot v16.4s, v10.16b, v0.4b[0]\n"
      ".inst 0x4f81e154  // sdot v20.4s, v10.16b, v1.4b[0]\n"
      "add x15, x15, #0x40\n"
      ".inst 0x4f82e158  // sdot v24.4s, v10.16b, v2.4b[0]\n"
      ".inst 0x4f80e091  // sdot v17.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e095  // sdot v21.4s, v4.16b, v1.4b[0]\n"
      ".inst 0x4f82e099  // sdot v25.4s, v4.16b, v2.4b[0]\n"
      ".inst 0x4f80e0b2  // sdot v18.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b6  // sdot v22.4s, v5.16b, v1.4b[0]\n"
      ".inst 0x4f82e0ba  // sdot v26.4s, v5.16b, v2.4b[0]\n"
      ".inst 0x4f80e0d3  // sdot v19.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d7  // sdot v23.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0db  // sdot v27.4s, v6.16b, v2.4b[0]\n"
      "76:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x12, x12, #0x1\n"
      "cmp x12, x20\n"
      "bne 62b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x13, #0x0]\n"
      "add x24, x13, x20\n"
      "add x23, x24, x20\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "tbnz %x[flags], #31, 77f\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1r { v3.4s }, [x20]\n"
      "neg v3.4s, v3.4s\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "mul v11.4s, v11.4s, v3.4s\n"
      "mul v12.4s, v12.4s, v3.4s\n"
      "mul v13.4s, v13.4s, v3.4s\n"
      "77:"  // Height 3: skip row sum fixup
      "ldr q0, [x14, #0x0]\n"
      "add v16.4s, v16.4s, v11.4s\n"
      "ldr q1, [x14, #0x10]\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q2, [x14, #0x20]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "ldr q3, [x14, #0x30]\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add v20.4s, v20.4s, v12.4s\n"
      "add v21.4s, v21.4s, v12.4s\n"
      "add v22.4s, v22.4s, v12.4s\n"
      "add v23.4s, v23.4s, v12.4s\n"
      "add v24.4s, v24.4s, v13.4s\n"
      "add v25.4s, v25.4s, v13.4s\n"
      "add v26.4s, v26.4s, v13.4s\n"
      "add v27.4s, v27.4s, v13.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v20.4s, v20.4s, v0.4s\n"
      "add v21.4s, v21.4s, v1.4s\n"
      "add v22.4s, v22.4s, v2.4s\n"
      "add v23.4s, v23.4s, v3.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "add v25.4s, v25.4s, v1.4s\n"
      "add v26.4s, v26.4s, v2.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v4.4s }, [x21]\n"
      "add x21, %x[qp], %[c_offset]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v6.4s }, [x20]\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v4.4s\n"
      "sqdmulh v18.4s, v18.4s, v4.4s\n"
      "sqdmulh v19.4s, v19.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "sqdmulh v21.4s, v21.4s, v4.4s\n"
      "sqdmulh v22.4s, v22.4s, v4.4s\n"
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "sqdmulh v24.4s, v24.4s, v4.4s\n"
      "sqdmulh v25.4s, v25.4s, v4.4s\n"
      "sqdmulh v26.4s, v26.4s, v4.4s\n"
      "sqdmulh v27.4s, v27.4s, v4.4s\n"
      "ld1r { v4.4s }, [x21]\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v0.4s\n"
      "srshl v26.4s, v26.4s, v0.4s\n"
      "srshl v27.4s, v27.4s, v0.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v4.4s\n"
      "add v27.4s, v27.4s, v4.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v24.4s, v24.4s, v6.4s\n"
      "smin v25.4s, v25.4s, v6.4s\n"
      "smin v26.4s, v26.4s, v6.4s\n"
      "smin v27.4s, v27.4s, v6.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v24.4s, v24.4s, v5.4s\n"
      "smax v25.4s, v25.4s, v5.4s\n"
      "smax v26.4s, v26.4s, v5.4s\n"
      "smax v27.4s, v27.4s, v5.4s\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "cmp x16, #0x10\n"
      "uzp1 v21.8h, v22.8h, v23.8h\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "add x14, x14, #0x40\n"
      "uzp1 v25.8h, v26.8h, v27.8h\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v20.16b, v20.16b, v21.16b\n"
      "uzp1 v24.16b, v24.16b, v25.16b\n"
      "bge 86f\n"
      "tbz x16, #3, 81f\n"
      "str d16, [x13], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "tbz x16, #2, 79f\n"
      "st1 { v16.s }[2], [x13], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "tbz x16, #1, 78f\n"
      "st1 { v16.h }[6], [x13], #0x2\n"
      "st1 { v20.h }[6], [x24], #0x2\n"
      "st1 { v24.h }[6], [x23], #0x2\n"
      "tbz x16, #0, 85f\n"
      "st1 { v16.b }[14], [x13]\n"
      "st1 { v20.b }[14], [x24]\n"
      "st1 { v24.b }[14], [x23]\n"
      "b 85f\n"
      "78:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x16, #0, 85f\n"
      "st1 { v16.b }[12], [x13]\n"
      "st1 { v20.b }[12], [x24]\n"
      "st1 { v24.b }[12], [x23]\n"
      "b 85f\n"
      "79:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x16, #1, 80f\n"
      "st1 { v16.h }[4], [x13], #0x2\n"
      "st1 { v20.h }[4], [x24], #0x2\n"
      "st1 { v24.h }[4], [x23], #0x2\n"
      "tbz x16, #0, 85f\n"
      "st1 { v16.b }[10], [x13]\n"
      "st1 { v20.b }[10], [x24]\n"
      "st1 { v24.b }[10], [x23]\n"
      "b 85f\n"
      "80:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x16, #0, 85f\n"
      "st1 { v16.b }[8], [x13]\n"
      "st1 { v20.b }[8], [x24]\n"
      "st1 { v24.b }[8], [x23]\n"
      "b 85f\n"
      "81:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x16, #2, 83f\n"
      "str s16, [x13], #0x4\n"
      "str s20, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "tbz x16, #1, 82f\n"
      "st1 { v16.h }[2], [x13], #0x2\n"
      "st1 { v20.h }[2], [x24], #0x2\n"
      "st1 { v24.h }[2], [x23], #0x2\n"
      "tbz x16, #0, 85f\n"
      "st1 { v16.b }[6], [x13]\n"
      "st1 { v20.b }[6], [x24]\n"
      "st1 { v24.b }[6], [x23]\n"
      "b 85f\n"
      "82:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x16, #0, 85f\n"
      "st1 { v16.b }[4], [x13]\n"
      "st1 { v20.b }[4], [x24]\n"
      "st1 { v24.b }[4], [x23]\n"
      "b 85f\n"
      "83:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x16, #1, 84f\n"
      "str h16, [x13], #0x2\n"
      "str h20, [x24], #0x2\n"
      "str h24, [x23], #0x2\n"
      "tbz x16, #0, 85f\n"
      "st1 { v16.b }[2], [x13]\n"
      "st1 { v20.b }[2], [x24]\n"
      "st1 { v24.b }[2], [x23]\n"
      "b 85f\n"
      "84:"  // Height 3: Partial direct writeback: partial_1_0
      "str b16, [x13, #0x0]\n"
      "str b20, [x24, #0x0]\n"
      "str b24, [x23, #0x0]\n"
      "85:"  // Height 3: Partial direct writeback: Done
      "b 87f\n"
      "86:"  // Height 3: Full writeback
      "str q16, [x13, #0x0]\n"
      "add x13, x13, #0x10\n"
      "str q20, [x24, #0x0]\n"
      "str q24, [x23, #0x0]\n"
      "87:"  // Height 3: Writeback done
      "subs x16, x16, #0x10\n"
      "bgt 60b\n"
      "b 118f\n"
      "88:"  // Height 4
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x4\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "movi v13.4s, #0x0\n"
      "ldr x16, [%x[args_ptr], %[offsetof_N]]\n"
      "madd x20, x21, x20, x13\n"
      "movi v14.4s, #0x0\n"
      "movi v15.16b, #0x1\n"
      "ldr x15, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x14, %x[col_bias]\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "89:"  // Height 4: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "mov x12, #0x0\n"
      "91:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w11, [x20, x12, LSL #0x2]\n"
      "tbz %x[flags], #3, 92f\n"
      "ldr x20, [%x[input_ptr], x12, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x10, [x20, #0x0]\n"
      "ldr x9, [x20, #0x8]\n"
      "ldr x28, [x20, #0x10]\n"
      "ldr x27, [x20, #0x18]\n"
      "cbnz x12, 93f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x10, x10, x20\n"
      "add x9, x9, x20\n"
      "add x28, x28, x20\n"
      "add x27, x27, x20\n"
      "b 93f\n"
      "92:"  // Height 4: setup direct input
      "mov x10, %x[input_ptr]\n"
      "add x9, x10, x21\n"
      "add x28, x9, x21\n"
      "add x27, x28, x21\n"
      "93:"  // Height 4: input setup done
      "cmp x11, #0x10\n"
      "blt 98f\n"
      "ldr q0, [x10, #0x0]\n"
      "cmp x11, #0x20\n"
      "ldr q1, [x9, #0x0]\n"
      "ldr q2, [x28, #0x0]\n"
      "ldr q3, [x27, #0x0]\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "ldr q8, [x15, #0x40]\n"
      "ldr q9, [x15, #0x50]\n"
      "ldr q10, [x15, #0x60]\n"
      "blt 96f\n"
      "94:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "ldr x21, [x15, #0x78]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "ldr x20, [x15, #0x88]\n"
      ".inst 0x4f82e098  // sdot v24.4s, v4.16b, v2.4b[0]\n"
      "ldr x26, [x15, #0x98]\n"
      ".inst 0x4f83e09c  // sdot v28.4s, v4.16b, v3.4b[0]\n"
      "ldr d4, [x15, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "ldr x25, [x15, #0xa8]\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "ldr x24, [x15, #0xb8]\n"
      ".inst 0x4f82e0b9  // sdot v25.4s, v5.16b, v2.4b[0]\n"
      "mov v4.d[1], x21\n"
      ".inst 0x4f83e0bd  // sdot v29.4s, v5.16b, v3.4b[0]\n"
      "ldr d5, [x15, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      "ldr x23, [x15, #0xc8]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      "ldr x22, [x15, #0xd8]\n"
      ".inst 0x4f82e0da  // sdot v26.4s, v6.16b, v2.4b[0]\n"
      "mov v5.d[1], x20\n"
      ".inst 0x4f83e0de  // sdot v30.4s, v6.16b, v3.4b[0]\n"
      "ldr d6, [x15, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      "ldr x21, [x15, #0xe8]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      "ldr x20, [x15, #0xf8]\n"
      ".inst 0x4f82e0fb  // sdot v27.4s, v7.16b, v2.4b[0]\n"
      "mov v6.d[1], x26\n"
      ".inst 0x4f83e0ff  // sdot v31.4s, v7.16b, v3.4b[0]\n"
      "ldr d7, [x15, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4fa2e118  // sdot v24.4s, v8.16b, v2.4b[1]\n"
      "mov v7.d[1], x25\n"
      ".inst 0x4fa3e11c  // sdot v28.4s, v8.16b, v3.4b[1]\n"
      "ldr d8, [x15, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      "add x28, x28, #0x10\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      "add x27, x27, #0x10\n"
      ".inst 0x4fa2e139  // sdot v25.4s, v9.16b, v2.4b[1]\n"
      "mov v8.d[1], x24\n"
      ".inst 0x4fa3e13d  // sdot v29.4s, v9.16b, v3.4b[1]\n"
      "ldr d9, [x15, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      ".inst 0x4fa2e15a  // sdot v26.4s, v10.16b, v2.4b[1]\n"
      "mov v9.d[1], x23\n"
      ".inst 0x4fa3e15e  // sdot v30.4s, v10.16b, v3.4b[1]\n"
      "ldr d10, [x15, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      ".inst 0x4fa1e097  // sdot v23.4s, v4.16b, v1.4b[1]\n"
      ".inst 0x4fa2e09b  // sdot v27.4s, v4.16b, v2.4b[1]\n"
      "mov v10.d[1], x22\n"
      ".inst 0x4fa3e09f  // sdot v31.4s, v4.16b, v3.4b[1]\n"
      "ldr d4, [x15, #0xe0]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      ".inst 0x4f81e8b4  // sdot v20.4s, v5.16b, v1.4b[2]\n"
      ".inst 0x4f82e8b8  // sdot v24.4s, v5.16b, v2.4b[2]\n"
      "mov v4.d[1], x21\n"
      ".inst 0x4f83e8bc  // sdot v28.4s, v5.16b, v3.4b[2]\n"
      "ldr d5, [x15, #0xf0]\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      "add x15, x15, #0x100\n"
      ".inst 0x4f81e8d5  // sdot v21.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d9  // sdot v25.4s, v6.16b, v2.4b[2]\n"
      "mov v5.d[1], x20\n"
      ".inst 0x4f83e8dd  // sdot v29.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8f6  // sdot v22.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8fa  // sdot v26.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8fe  // sdot v30.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      ".inst 0x4f81e917  // sdot v23.4s, v8.16b, v1.4b[2]\n"
      ".inst 0x4f82e91b  // sdot v27.4s, v8.16b, v2.4b[2]\n"
      ".inst 0x4f83e91f  // sdot v31.4s, v8.16b, v3.4b[2]\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa1e934  // sdot v20.4s, v9.16b, v1.4b[3]\n"
      ".inst 0x4fa2e938  // sdot v24.4s, v9.16b, v2.4b[3]\n"
      ".inst 0x4fa3e93c  // sdot v28.4s, v9.16b, v3.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa1e955  // sdot v21.4s, v10.16b, v1.4b[3]\n"
      ".inst 0x4fa2e959  // sdot v25.4s, v10.16b, v2.4b[3]\n"
      ".inst 0x4fa3e95d  // sdot v29.4s, v10.16b, v3.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa1e896  // sdot v22.4s, v4.16b, v1.4b[3]\n"
      ".inst 0x4fa2e89a  // sdot v26.4s, v4.16b, v2.4b[3]\n"
      ".inst 0x4fa3e89e  // sdot v30.4s, v4.16b, v3.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8b7  // sdot v23.4s, v5.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8bb  // sdot v27.4s, v5.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8bf  // sdot v31.4s, v5.16b, v3.4b[3]\n"
      "tbnz %x[flags], #31, 95f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f946e  // sdot v14.4s, v3.16b, v15.16b\n"
      "95:"  // Height 4: Multiply loop: unique 13: skip row sum
      "ldr q0, [x10, #0x0]\n"
      "sub x11, x11, #0x10\n"
      "ldr q1, [x9, #0x0]\n"
      "cmp x11, #0x20\n"
      "ldr q2, [x28, #0x0]\n"
      "ldr q3, [x27, #0x0]\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "ldr q8, [x15, #0x40]\n"
      "ldr q9, [x15, #0x50]\n"
      "ldr q10, [x15, #0x60]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "bge 94b\n"
      "96:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "sub x11, x11, #0x10\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f82e098  // sdot v24.4s, v4.16b, v2.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f83e09c  // sdot v28.4s, v4.16b, v3.4b[0]\n"
      "ldr q4, [x15, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "add x28, x28, #0x10\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "add x27, x27, #0x10\n"
      ".inst 0x4f82e0b9  // sdot v25.4s, v5.16b, v2.4b[0]\n"
      ".inst 0x4f83e0bd  // sdot v29.4s, v5.16b, v3.4b[0]\n"
      "ldr q5, [x15, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0da  // sdot v26.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0de  // sdot v30.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x15, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0fb  // sdot v27.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0ff  // sdot v31.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x15, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      ".inst 0x4fa2e118  // sdot v24.4s, v8.16b, v2.4b[1]\n"
      ".inst 0x4fa3e11c  // sdot v28.4s, v8.16b, v3.4b[1]\n"
      "ldr q8, [x15, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      ".inst 0x4fa2e139  // sdot v25.4s, v9.16b, v2.4b[1]\n"
      ".inst 0x4fa3e13d  // sdot v29.4s, v9.16b, v3.4b[1]\n"
      "ldr q9, [x15, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      ".inst 0x4fa2e15a  // sdot v26.4s, v10.16b, v2.4b[1]\n"
      ".inst 0x4fa3e15e  // sdot v30.4s, v10.16b, v3.4b[1]\n"
      "ldr q10, [x15, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      ".inst 0x4fa1e097  // sdot v23.4s, v4.16b, v1.4b[1]\n"
      ".inst 0x4fa2e09b  // sdot v27.4s, v4.16b, v2.4b[1]\n"
      ".inst 0x4fa3e09f  // sdot v31.4s, v4.16b, v3.4b[1]\n"
      "ldr q4, [x15, #0xe0]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      ".inst 0x4f81e8b4  // sdot v20.4s, v5.16b, v1.4b[2]\n"
      ".inst 0x4f82e8b8  // sdot v24.4s, v5.16b, v2.4b[2]\n"
      ".inst 0x4f83e8bc  // sdot v28.4s, v5.16b, v3.4b[2]\n"
      "ldr q5, [x15, #0xf0]\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      "add x15, x15, #0x100\n"
      ".inst 0x4f81e8d5  // sdot v21.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d9  // sdot v25.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8dd  // sdot v29.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8f6  // sdot v22.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8fa  // sdot v26.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8fe  // sdot v30.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      ".inst 0x4f81e917  // sdot v23.4s, v8.16b, v1.4b[2]\n"
      ".inst 0x4f82e91b  // sdot v27.4s, v8.16b, v2.4b[2]\n"
      ".inst 0x4f83e91f  // sdot v31.4s, v8.16b, v3.4b[2]\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa1e934  // sdot v20.4s, v9.16b, v1.4b[3]\n"
      ".inst 0x4fa2e938  // sdot v24.4s, v9.16b, v2.4b[3]\n"
      ".inst 0x4fa3e93c  // sdot v28.4s, v9.16b, v3.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa1e955  // sdot v21.4s, v10.16b, v1.4b[3]\n"
      ".inst 0x4fa2e959  // sdot v25.4s, v10.16b, v2.4b[3]\n"
      ".inst 0x4fa3e95d  // sdot v29.4s, v10.16b, v3.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa1e896  // sdot v22.4s, v4.16b, v1.4b[3]\n"
      ".inst 0x4fa2e89a  // sdot v26.4s, v4.16b, v2.4b[3]\n"
      ".inst 0x4fa3e89e  // sdot v30.4s, v4.16b, v3.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8b7  // sdot v23.4s, v5.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8bb  // sdot v27.4s, v5.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8bf  // sdot v31.4s, v5.16b, v3.4b[3]\n"
      "tbnz %x[flags], #31, 97f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f946e  // sdot v14.4s, v3.16b, v15.16b\n"
      "97:"  // Height 4: Multiply loop: unique 14: skip row sum
      "prfm pldl1keep, [x10, #0x80]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "98:"  // Height 4: Multiply loop: Main loop skip
      "cbz x11, 105f\n"
      "cmp x11, #0x4\n"
      "blt 101f\n"
      "99:"  // Height 4: Multiply loop: Odd block loop
      "ldr s0, [x10], #0x4\n"
      "ldr s1, [x9], #0x4\n"
      "ldr s2, [x28], #0x4\n"
      "ldr s3, [x27], #0x4\n"
      "tbnz %x[flags], #31, 100f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f946e  // sdot v14.4s, v3.16b, v15.16b\n"
      "100:"  // Height 4: Multiply loop: unique 15: skip row sum
      "ldr q6, [x15, #0x0]\n"
      "sub x11, x11, #0x4\n"
      "ldr q7, [x15, #0x10]\n"
      "cmp x11, #0x4\n"
      "ldr q8, [x15, #0x20]\n"
      "ldr q9, [x15, #0x30]\n"
      ".inst 0x4f80e0d0  // sdot v16.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d4  // sdot v20.4s, v6.16b, v1.4b[0]\n"
      "add x15, x15, #0x40\n"
      ".inst 0x4f82e0d8  // sdot v24.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0dc  // sdot v28.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f80e0f1  // sdot v17.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f5  // sdot v21.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f9  // sdot v25.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0fd  // sdot v29.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f80e112  // sdot v18.4s, v8.16b, v0.4b[0]\n"
      ".inst 0x4f81e116  // sdot v22.4s, v8.16b, v1.4b[0]\n"
      ".inst 0x4f82e11a  // sdot v26.4s, v8.16b, v2.4b[0]\n"
      ".inst 0x4f83e11e  // sdot v30.4s, v8.16b, v3.4b[0]\n"
      ".inst 0x4f80e133  // sdot v19.4s, v9.16b, v0.4b[0]\n"
      ".inst 0x4f81e137  // sdot v23.4s, v9.16b, v1.4b[0]\n"
      ".inst 0x4f82e13b  // sdot v27.4s, v9.16b, v2.4b[0]\n"
      ".inst 0x4f83e13f  // sdot v31.4s, v9.16b, v3.4b[0]\n"
      "bge 99b\n"
      "101:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x11, 105f\n"
      "tbz x11, #1, 102f\n"
      "ldr h0, [x10], #0x2\n"
      "ldr h1, [x9], #0x2\n"
      "ldr h2, [x28], #0x2\n"
      "ldr h3, [x27], #0x2\n"
      "tbz x11, #0, 103f\n"
      "ld1 { v0.b }[2], [x10]\n"
      "ld1 { v1.b }[2], [x9]\n"
      "ld1 { v2.b }[2], [x28]\n"
      "ld1 { v3.b }[2], [x27]\n"
      "b 103f\n"
      "102:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x10, #0x0]\n"
      "ldr b1, [x9, #0x0]\n"
      "ldr b2, [x28, #0x0]\n"
      "ldr b3, [x27, #0x0]\n"
      "103:"  // Height 4: Multiply loop: Ragged operand read: Done
      "tbnz %x[flags], #31, 104f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f946e  // sdot v14.4s, v3.16b, v15.16b\n"
      "104:"  // Height 4: Multiply loop: unique 16: skip row sum
      "ldr q10, [x15, #0x0]\n"
      "ldr q4, [x15, #0x10]\n"
      "ldr q5, [x15, #0x20]\n"
      "ldr q6, [x15, #0x30]\n"
      ".inst 0x4f80e150  // sdot v16.4s, v10.16b, v0.4b[0]\n"
      ".inst 0x4f81e154  // sdot v20.4s, v10.16b, v1.4b[0]\n"
      "add x15, x15, #0x40\n"
      ".inst 0x4f82e158  // sdot v24.4s, v10.16b, v2.4b[0]\n"
      ".inst 0x4f83e15c  // sdot v28.4s, v10.16b, v3.4b[0]\n"
      ".inst 0x4f80e091  // sdot v17.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e095  // sdot v21.4s, v4.16b, v1.4b[0]\n"
      ".inst 0x4f82e099  // sdot v25.4s, v4.16b, v2.4b[0]\n"
      ".inst 0x4f83e09d  // sdot v29.4s, v4.16b, v3.4b[0]\n"
      ".inst 0x4f80e0b2  // sdot v18.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b6  // sdot v22.4s, v5.16b, v1.4b[0]\n"
      ".inst 0x4f82e0ba  // sdot v26.4s, v5.16b, v2.4b[0]\n"
      ".inst 0x4f83e0be  // sdot v30.4s, v5.16b, v3.4b[0]\n"
      ".inst 0x4f80e0d3  // sdot v19.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d7  // sdot v23.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0db  // sdot v27.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0df  // sdot v31.4s, v6.16b, v3.4b[0]\n"
      "105:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x12, x12, #0x1\n"
      "cmp x12, x20\n"
      "bne 91b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x13, #0x0]\n"
      "add x24, x13, x20\n"
      "add x23, x24, x20\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x22, x23, x20\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "tbnz %x[flags], #31, 106f\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "addp v14.4s, v14.4s, v14.4s\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1r { v4.4s }, [x20]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "neg v4.4s, v4.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "addp v14.4s, v14.4s, v14.4s\n"
      "mul v11.4s, v11.4s, v4.4s\n"
      "mul v12.4s, v12.4s, v4.4s\n"
      "mul v13.4s, v13.4s, v4.4s\n"
      "mul v14.4s, v14.4s, v4.4s\n"
      "106:"  // Height 4: skip row sum fixup
      "ldr q0, [x14, #0x0]\n"
      "add v16.4s, v16.4s, v11.4s\n"
      "ldr q1, [x14, #0x10]\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q2, [x14, #0x20]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "ldr q3, [x14, #0x30]\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add v20.4s, v20.4s, v12.4s\n"
      "add v21.4s, v21.4s, v12.4s\n"
      "add v22.4s, v22.4s, v12.4s\n"
      "add v23.4s, v23.4s, v12.4s\n"
      "add v24.4s, v24.4s, v13.4s\n"
      "add v25.4s, v25.4s, v13.4s\n"
      "add v26.4s, v26.4s, v13.4s\n"
      "add v27.4s, v27.4s, v13.4s\n"
      "add v28.4s, v28.4s, v14.4s\n"
      "add v29.4s, v29.4s, v14.4s\n"
      "add v30.4s, v30.4s, v14.4s\n"
      "add v31.4s, v31.4s, v14.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v20.4s, v20.4s, v0.4s\n"
      "add v21.4s, v21.4s, v1.4s\n"
      "add v22.4s, v22.4s, v2.4s\n"
      "add v23.4s, v23.4s, v3.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "add v25.4s, v25.4s, v1.4s\n"
      "add v26.4s, v26.4s, v2.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "add v28.4s, v28.4s, v0.4s\n"
      "add v29.4s, v29.4s, v1.4s\n"
      "add v30.4s, v30.4s, v2.4s\n"
      "add v31.4s, v31.4s, v3.4s\n"
      "add x21, %x[qp], %[per_layer_mul]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v4.4s }, [x21]\n"
      "add x21, %x[qp], %[c_offset]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v6.4s }, [x20]\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v4.4s\n"
      "sqdmulh v18.4s, v18.4s, v4.4s\n"
      "sqdmulh v19.4s, v19.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v4.4s\n"
      "sqdmulh v21.4s, v21.4s, v4.4s\n"
      "sqdmulh v22.4s, v22.4s, v4.4s\n"
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "sqdmulh v24.4s, v24.4s, v4.4s\n"
      "sqdmulh v25.4s, v25.4s, v4.4s\n"
      "sqdmulh v26.4s, v26.4s, v4.4s\n"
      "sqdmulh v27.4s, v27.4s, v4.4s\n"
      "sqdmulh v28.4s, v28.4s, v4.4s\n"
      "sqdmulh v29.4s, v29.4s, v4.4s\n"
      "sqdmulh v30.4s, v30.4s, v4.4s\n"
      "sqdmulh v31.4s, v31.4s, v4.4s\n"
      "ld1r { v4.4s }, [x21]\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v0.4s\n"
      "srshl v26.4s, v26.4s, v0.4s\n"
      "srshl v27.4s, v27.4s, v0.4s\n"
      "srshl v28.4s, v28.4s, v0.4s\n"
      "srshl v29.4s, v29.4s, v0.4s\n"
      "srshl v30.4s, v30.4s, v0.4s\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v4.4s\n"
      "add v27.4s, v27.4s, v4.4s\n"
      "add v28.4s, v28.4s, v4.4s\n"
      "add v29.4s, v29.4s, v4.4s\n"
      "add v30.4s, v30.4s, v4.4s\n"
      "add v31.4s, v31.4s, v4.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v24.4s, v24.4s, v6.4s\n"
      "smin v25.4s, v25.4s, v6.4s\n"
      "smin v26.4s, v26.4s, v6.4s\n"
      "smin v27.4s, v27.4s, v6.4s\n"
      "smin v28.4s, v28.4s, v6.4s\n"
      "smin v29.4s, v29.4s, v6.4s\n"
      "smin v30.4s, v30.4s, v6.4s\n"
      "smin v31.4s, v31.4s, v6.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v24.4s, v24.4s, v5.4s\n"
      "smax v25.4s, v25.4s, v5.4s\n"
      "smax v26.4s, v26.4s, v5.4s\n"
      "smax v27.4s, v27.4s, v5.4s\n"
      "smax v28.4s, v28.4s, v5.4s\n"
      "smax v29.4s, v29.4s, v5.4s\n"
      "smax v30.4s, v30.4s, v5.4s\n"
      "smax v31.4s, v31.4s, v5.4s\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v21.8h, v22.8h, v23.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v25.8h, v26.8h, v27.8h\n"
      "uzp1 v28.8h, v28.8h, v29.8h\n"
      "cmp x16, #0x10\n"
      "uzp1 v29.8h, v30.8h, v31.8h\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "add x14, x14, #0x40\n"
      "uzp1 v20.16b, v20.16b, v21.16b\n"
      "uzp1 v24.16b, v24.16b, v25.16b\n"
      "uzp1 v28.16b, v28.16b, v29.16b\n"
      "bge 115f\n"
      "tbz x16, #3, 110f\n"
      "str d16, [x13], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "str d28, [x22], #0x8\n"
      "tbz x16, #2, 108f\n"
      "st1 { v16.s }[2], [x13], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "st1 { v28.s }[2], [x22], #0x4\n"
      "tbz x16, #1, 107f\n"
      "st1 { v16.h }[6], [x13], #0x2\n"
      "st1 { v20.h }[6], [x24], #0x2\n"
      "st1 { v24.h }[6], [x23], #0x2\n"
      "st1 { v28.h }[6], [x22], #0x2\n"
      "tbz x16, #0, 114f\n"
      "st1 { v16.b }[14], [x13]\n"
      "st1 { v20.b }[14], [x24]\n"
      "st1 { v24.b }[14], [x23]\n"
      "st1 { v28.b }[14], [x22]\n"
      "b 114f\n"
      "107:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x16, #0, 114f\n"
      "st1 { v16.b }[12], [x13]\n"
      "st1 { v20.b }[12], [x24]\n"
      "st1 { v24.b }[12], [x23]\n"
      "st1 { v28.b }[12], [x22]\n"
      "b 114f\n"
      "108:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x16, #1, 109f\n"
      "st1 { v16.h }[4], [x13], #0x2\n"
      "st1 { v20.h }[4], [x24], #0x2\n"
      "st1 { v24.h }[4], [x23], #0x2\n"
      "st1 { v28.h }[4], [x22], #0x2\n"
      "tbz x16, #0, 114f\n"
      "st1 { v16.b }[10], [x13]\n"
      "st1 { v20.b }[10], [x24]\n"
      "st1 { v24.b }[10], [x23]\n"
      "st1 { v28.b }[10], [x22]\n"
      "b 114f\n"
      "109:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x16, #0, 114f\n"
      "st1 { v16.b }[8], [x13]\n"
      "st1 { v20.b }[8], [x24]\n"
      "st1 { v24.b }[8], [x23]\n"
      "st1 { v28.b }[8], [x22]\n"
      "b 114f\n"
      "110:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x16, #2, 112f\n"
      "str s16, [x13], #0x4\n"
      "str s20, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "str s28, [x22], #0x4\n"
      "tbz x16, #1, 111f\n"
      "st1 { v16.h }[2], [x13], #0x2\n"
      "st1 { v20.h }[2], [x24], #0x2\n"
      "st1 { v24.h }[2], [x23], #0x2\n"
      "st1 { v28.h }[2], [x22], #0x2\n"
      "tbz x16, #0, 114f\n"
      "st1 { v16.b }[6], [x13]\n"
      "st1 { v20.b }[6], [x24]\n"
      "st1 { v24.b }[6], [x23]\n"
      "st1 { v28.b }[6], [x22]\n"
      "b 114f\n"
      "111:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x16, #0, 114f\n"
      "st1 { v16.b }[4], [x13]\n"
      "st1 { v20.b }[4], [x24]\n"
      "st1 { v24.b }[4], [x23]\n"
      "st1 { v28.b }[4], [x22]\n"
      "b 114f\n"
      "112:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x16, #1, 113f\n"
      "str h16, [x13], #0x2\n"
      "str h20, [x24], #0x2\n"
      "str h24, [x23], #0x2\n"
      "str h28, [x22], #0x2\n"
      "tbz x16, #0, 114f\n"
      "st1 { v16.b }[2], [x13]\n"
      "st1 { v20.b }[2], [x24]\n"
      "st1 { v24.b }[2], [x23]\n"
      "st1 { v28.b }[2], [x22]\n"
      "b 114f\n"
      "113:"  // Height 4: Partial direct writeback: partial_1_0
      "str b16, [x13, #0x0]\n"
      "str b20, [x24, #0x0]\n"
      "str b24, [x23, #0x0]\n"
      "str b28, [x22, #0x0]\n"
      "114:"  // Height 4: Partial direct writeback: Done
      "b 116f\n"
      "115:"  // Height 4: Full writeback
      "str q16, [x13, #0x0]\n"
      "add x13, x13, #0x10\n"
      "str q20, [x24, #0x0]\n"
      "str q24, [x23, #0x0]\n"
      "str q28, [x22, #0x0]\n"
      "116:"  // Height 4: Writeback done
      "subs x16, x16, #0x10\n"
      "bgt 89b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 118f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 117f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "117:"  // Update direct input
      "mov x20, #0x4\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "118:"  // Exit
      : [M] "+&r" (M), [flags] "+&r" (flags), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [b_offset] "I" (offsetof(Requantize32, b_offset)), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
