# Configuration file for the Sphinx documentation builder.
#
# This file only contains a selection of the most common options. For a full
# list see the documentation:
# https://www.sphinx-doc.org/en/master/usage/configuration.html

# -- Path setup --------------------------------------------------------------

# If extensions (or modules to document with autodoc) are in another directory,
# add these directories to sys.path here. If the directory is relative to the
# documentation root, use os.path.abspath to make it absolute, like shown here.
#
# import os
# import sys
# sys.path.insert(0, os.path.abspath('.'))

# -- Project information -----------------------------------------------------

import os
import sys
from datetime import datetime
from subprocess import Popen

import where
from packaging import version as version_

# Get current location
_DOC_PATH = os.path.dirname(os.path.abspath(__file__))
_PROJ_PATH = os.path.abspath(os.path.join(_DOC_PATH, '..', '..'))
_LIBS_PATH = os.path.join(_DOC_PATH, '_libs')
_SHIMS_PATH = os.path.join(_DOC_PATH, '_shims')
os.chdir(_PROJ_PATH)

# Set environment, remove the pre-installed package
sys.path.insert(0, _PROJ_PATH)
modnames = [mname for mname in sys.modules if mname.startswith('ditk')]
for modname in modnames:
    del sys.modules[modname]

# Build dependencies if needed
if not os.environ.get("NO_CONTENTS_BUILD"):
    _env = dict(os.environ)
    _env.update(
        dict(
            SOURCEDIR=_DOC_PATH,
            BUILDDIR=os.path.abspath(os.path.join(_DOC_PATH, '..', 'build')),
            PYTHONPATH=':'.join([_PROJ_PATH, _LIBS_PATH]),
            PATH=':'.join([_SHIMS_PATH, os.environ.get('PATH', '')]),
        )
    )

    if os.path.exists(os.path.join(_PROJ_PATH, 'requirements-build.txt')):
        pip_build_cmd = (where.first('pip'), 'install', '-r', os.path.join(_PROJ_PATH, 'requirements-build.txt'))
        print("Install pip requirements {cmd}...".format(cmd=repr(pip_build_cmd)))
        pip_build = Popen(pip_build_cmd, stdout=sys.stdout, stderr=sys.stderr, env=_env, cwd=_PROJ_PATH)
        if pip_build.wait() != 0:
            raise ChildProcessError("Pip install failed with %d." % (pip_build.returncode, ))

        make_build_cmd = (where.first('make'), 'clean', 'build')
        print("Try building extensions {cmd}...".format(cmd=repr(make_build_cmd)))
        make_build = Popen(make_build_cmd, stdout=sys.stdout, stderr=sys.stderr, env=_env, cwd=_PROJ_PATH)
        if make_build.wait() != 0:
            raise ChildProcessError("Extension build failed with %d." % (make_build.returncode, ))

    if os.path.exists(os.path.join(_PROJ_PATH, 'requirements.txt')):
        pip_cmd = (where.first('pip'), 'install', '-r', os.path.join(_PROJ_PATH, 'requirements.txt'))
        print("Install pip requirements {cmd}...".format(cmd=repr(pip_cmd)))
        pip = Popen(pip_cmd, stdout=sys.stdout, stderr=sys.stderr, env=_env, cwd=_DOC_PATH)
        if pip.wait() != 0:
            raise ChildProcessError("Pip install failed with %d." % (pip.returncode, ))

    pip_docs_cmd = (where.first('pip'), 'install', '-r', os.path.join(_PROJ_PATH, 'requirements-doc.txt'))
    print("Install pip docs requirements {cmd}...".format(cmd=repr(pip_docs_cmd)))
    pip_docs = Popen(pip_docs_cmd, stdout=sys.stdout, stderr=sys.stderr, env=_env, cwd=_DOC_PATH)
    if pip_docs.wait() != 0:
        raise ChildProcessError("Pip docs install failed with %d." % (pip.returncode, ))

    all_cmd = (where.first('make'), '-f', "all.mk", "build")
    print("Building all {cmd} at {cp}...".format(cmd=repr(all_cmd), cp=repr(_DOC_PATH)))
    all_ = Popen(all_cmd, stdout=sys.stdout, stderr=sys.stderr, env=_env, cwd=_DOC_PATH)
    if all_.wait() != 0:
        raise ChildProcessError("Diagrams failed with %d." % (all_.returncode, ))

    print("Build of contents complete.")

from grl import __TITLE__, __AUTHOR__, __VERSION__

project = __TITLE__
copyright = '{year}, {author}'.format(year=datetime.now().year, author=__AUTHOR__)
author = __AUTHOR__

# The short X.Y version
version = version_.parse(__VERSION__).base_version
# The full version, including alpha/beta/rc tags
release = __VERSION__

# -- General configuration ---------------------------------------------------

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
extensions = [
    'sphinx.ext.autodoc',
    'sphinx.ext.doctest',
    'sphinx.ext.mathjax',
    'sphinx.ext.ifconfig',
    'sphinx.ext.viewcode',
    'sphinx.ext.githubpages',
    'sphinx.ext.todo',
    'sphinx.ext.graphviz',
    'enum_tools.autoenum',
    "sphinx_multiversion",
    "sphinx.ext.coverage",
    "sphinx.ext.intersphinx",
    'sphinx.ext.napoleon',
    "sphinx_autodoc_typehints",
    "nbsphinx",
    "m2r2",
]

#napoleon_google_docstring = True
#autodoc_typehints = 'description'

# Add any paths that contain templates here, relative to this directory.
templates_path = ['_templates']

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
language = None

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path.
exclude_patterns = []

# -- Options for HTML output -------------------------------------------------

# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of plugins themes.
#
import pytorch_sphinx_theme
html_theme = 'pytorch_sphinx_theme'

html_theme_path = [pytorch_sphinx_theme.get_html_theme_path()]
html_theme_options = {
    'logo_url':
    'https://opendilab.github.io/GenerativeRL/',
    'menu': [
        {
            'name': 'GitHub',
            'url': 'https://github.com/opendilab/GenerativeRL'
        },
    ],
    # Specify the language of shared menu
    'menu_lang': "en"
}
html_static_path = ['_static']
html_css_files = ['css/style.css']

htmlhelp_basename = 'GenerativeRL'

epub_title = project
epub_exclude_files = ['search.html']

# Whitelist pattern for tags (set to None to ignore all tags)
smv_tag_whitelist = r'^v.*$'  # Include all tags start with 'v'
smv_branch_whitelist = r'^.*$'  # Include all branches
smv_remote_whitelist = r'^.*$'  # Use branches from all remotes
smv_released_pattern = r'^tags/.*$'  # Tags only
smv_outputdir_format = '{ref.name}'  # Use the branch/tag name

if not os.environ.get("ENV_PROD"):
    todo_include_todos = True
    todo_emit_warnings = True

latex_engine = 'xelatex'
latex_use_xindy = False
latex_elements = {
    'fontpkg': r'''
\setmainfont{DejaVu Serif}
\setsansfont{DejaVu Sans}
\setmonofont{DejaVu Sans Mono}
''',
    'preamble': r'''
\usepackage{amsmath}
\usepackage{amssymb}
\usepackage[titles]{tocloft}
\cftsetpnumwidth {1.25cm}\cftsetrmarg{1.5cm}
\setlength{\cftchapnumwidth}{0.75cm}
\setlength{\cftsecindent}{\cftchapnumwidth}
\setlength{\cftsecnumwidth}{1.25cm}
''',
    'fncychap': r'\usepackage[Bjornstrup]{fncychap}',
    'printindex': r'\footnotesize\raggedright\printindex',
}
latex_show_urls = 'footnote'
