---
title: 指针和引用
author: ["comewei","jjd","枫长","LH"]
---



### 1. *a和&a有什么区别（引用和指针有什么区别）？

在 C++ 中，`*a` 和 `&a` 分别表示指针和引用，主要区别：

指针（`*a`）: 指针是一个变量，它存储了另一个变量的内存地址。通过指针，我们可以间接地访问和操作其所指向的变量。指针需要先被声明并初始化，然后才能使用。例如：

```cpp
int x = 10;
int *ptr = &x; // ptr 是一个指向 x 的指针
```

在这个例子中，`ptr` 是一个指向 `x` 的指针，`&x` 表示取变量 `x` 的内存地址。

引用（`&a`）: 引用是另一个变量的别名，它可以用来访问和操作同一个变量。引用在创建时必须被初始化，并且在整个生命周期中不能改变其引用的对象。例如：

```cpp
int x = 10;
int &ref = x; // ref 是 x 的引用
```

在这个例子中，`ref` 是 `x` 的引用。通过 `ref`可以直接访问和操作 `x`。与指针不同，引用本身不存储内存地址，而是提供了另一个访问变量的途径。

总结：指针和引用在 C++ 中都可以用来间接访问和操作变量，但它们的行为和使用方式有所不同。指针是一个变量，需要先声明并初始化，然后才能使用；而引用是一个变量的别名，它在创建时必须被初始化，并且在整个生命周期中不能改变其引用的对象。



### 2. 指针可以是 volatile 吗？

指针可以是 `volatile` 。`volatile` 关键字用于表示一个变量可能会在程序中的其他地方（例如，由外部硬件、中断服务例程或多线程代码）以不可预测的方式进行修改。当一个指针被声明为 `volatile` 时，编译器会确保不对该指针进行优化，以确保正确处理外部对该指针的修改。

以下是几个关于 `volatile` 指针的示例：

声明一个指向 `volatile` 类型的指针：

```cpp
int volatile *ptr; // 指针指向一个 volatile 类型的整数
```

在这个例子中，`ptr` 是一个指向 `volatile` 整数的指针。指针本身不是 `volatile`，而它所指向的整数是 `volatile` 的。

声明一个 `volatile` 指针：

```cpp
int * volatile ptr; // 指针本身是 volatile 的
```

在这个例子中，`ptr` 是一个 `volatile` 指针，它指向一个非 `volatile` 整数。这表示指针本身可能会被外部修改。

声明一个指向 `volatile` 类型且本身也是 `volatile` 的指针：

```cpp
int volatile * volatile ptr; // 指针本身和指向的整数都是 volatile 的
```

在这个例子中，`ptr` 是一个 `volatile` 指针，它指向一个 `volatile` 整数。这表示指针本身和它所指向的整数都可能会被外部修改。

`volatile` 关键字通常与硬件寄存器、中断服务例程或多线程编程等场景一起使用，以确保编译器不会对相关变量进行不必要的优化，从而导致潜在的问题。



### 3. 指针常量和常量指针有什么区别？

指针常量（constant pointer）和常量指针（pointer to constant）是两种不同的概念，它们之间的主要区别在于所限制的修改对象不同。**const先看左边，左边没有修饰右边。**

常量指针（pointer to constant）： 常量指针是指向常量对象的指针，即指针指向的对象的值不能被修改，但指针本身可以指向其他对象。例如：

```cpp
const int *ptr;
```

`ptr` 是一个指向常量整数的指针。通过这个指针，不能修改它所指向的整数的值。但可以让 `ptr` 指向其他整数对象。

指针常量（constant pointer）： 指针常量是一个不能改变指向的指针，即指针本身的值不能被修改，但它指向的对象的值可以被修改。例如：

```cpp
int *const ptr = &x;
```

`ptr` 是一个指向整数的指针常量。不能改变 `ptr` 的值，即它必须始终指向同一个整数对象（在这里是 `x`）。然而可以通过 `ptr` 修改它所指向的整数的值。

如果希望指针本身和它所指向的对象的值都不能被修改，可以同时使用 `const` 关键字，例如：

```cpp
const int *const ptr = &x;
```

在这个例子中，`ptr` 是一个指向常量整数的指针常量。既不能修改 `ptr` 的值，也不能通过 `ptr` 修改它所指向的整数的值。



### 4. 什么情况用指针当参数，什么时候用引用，为什么？

在 C++ 中，选择在函数参数中使用指针还是引用通常取决于多种因素，包括函数的设计意图、性能需求、可读性和易用性：

**使用引用**

- 当希望在函数内修改传递的参数时，使用引用是一种更直观和易读的方式。使用引用可以避免显式解引用和修改指针所指向的值。
- 当参数类型是类或结构时，使用引用可以避免复制成本。这对于大型对象或容器尤其重要，因为它们的复制可能会导致性能下降。
- 当你需要一个对象的别名，且这个别名在函数内部始终指向同一个对象时，使用引用。
- 当编写操作符重载函数时，通常使用引用，因为这可以提供更自然的语法和易用性。

**使用指针**

- 当参数是可选的，即可以传递 `nullptr` 作为有效值时，使用指针。引用不能为 `nullptr`，而指针可以。
- 当需要在函数内部改变指针本身时，例如在链表或树的操作中，使用指针。
- 当需要显式地表示所有权和生命周期时，使用指针。指针可与智能指针（如 `std::unique_ptr` 或 `std::shared_ptr`）结合使用，以更清晰地表示资源所有权和生命周期。



### 5. 从汇编层去解释一下引用？

在汇编层面，引用实际上是一个隐式指针。引用的底层实现依赖于编译器，但通常情况下，引用在汇编代码中会被转换为指向目标对象的指针。这意味着，在汇编层面，引用的行为与指针相似。

C++ 代码：

```cpp
#include <iostream>

void increment(int &x) {
    x++;
}

int main() {
    int a = 10;
    increment(a);
    std::cout << a << std::endl;
    return 0;
}
```

定义了一个 `increment` 函数，该函数接受一个整数引用作为参数，并将其递增。在 `main` 函数中创建了一个名为 `a` 的整数变量，并调用 `increment` 函数递增它。

在编译并查看汇编代码后，你可能会看到类似以下片段：

```R
# increment 函数的调用
mov eax, DWORD PTR [rbp-4]    # 将变量 a 的内存地址放入 eax 寄存器
lea rdi, [rax]                # 将 eax 寄存器的值（a 的内存地址）放入 rdi 寄存器，rdi 是 increment 函数的第一个参数
call increment                # 调用 increment 函数
...
```

在这个汇编代码片段中，变量 `a` 的内存地址首先被加载到寄存器 `eax` 中。然后，`eax` 寄存器的值（即 `a` 的内存地址）被传递给 `increment` 函数的第一个参数（在这里是 `rdi` 寄存器）。这表明，在汇编层面，引用被实现为指针。

这个例子说明了引用在汇编层面的实现，具体的汇编代码可能因编译器、优化级别和目标平台的不同而有所不同。从概念上讲，引用在汇编层面通常会被实现为指向目标对象的指针。



### 6. 数组名和指针有什么区别？

**类型**： 数组名实际上是一个指向数组首元素的常量指针。它的类型是数组元素的类型，例如 `int[]` 或 `char[]`。而指针是一个变量，可以指向任意类型的对象，其类型是所指向对象的类型，例如 `int*` 或 `char*`。

**可修改性**： 数组名是一个指针常量，也就是说，你不能改变数组名的值（即不能让它指向其他内存位置）。然而，指针是一个变量，可以修改它的值，让它指向其他对象或内存位置。

**内存分配**： 数组名表示的内存是在声明数组时静态分配的（在栈上或全局/静态存储区）。数组的大小在编译时确定，运行时不能改变。而指针可以指向静态分配的内存（例如栈上的局部变量）或动态分配的内存（例如通过 `new` 或 `malloc` 分配的内存）。

**数组大小**： 对于数组，编译器知道数组的大小，因此可以使用 `sizeof` 运算符计算数组的大小（字节数）。但对于指针，编译器并不知道指针所指向的内存块的大小。使用 `sizeof` 计算指针的大小只会得到指针本身的大小，而不是指针所指向的内存块的大小。

下面是一个简单的示例，说明数组名和指针之间的区别：

```cpp
int arr[5];     // arr 是一个整数数组，包含 5 个元素
int *ptr = arr; // ptr 是一个整数指针，指向 arr 的首元素

arr = ptr;      // 错误！不能修改数组名的值
ptr = &arr[2];  // 正确，可以修改指针的值
```



### 7. 野指针是什么？

野指针（dangling pointer 或 wild pointer）是指向无效内存区域的指针。这类指针通常是因为编程错误引入的，如指针没有被初始化、指针使用后没有被置空、或指针指向的内存区域已被释放。野指针的使用可能导致程序行为不可预测、数据损坏、程序崩溃等问题。

**未初始化的指针**

```cpp
int *ptr; // 未初始化的指针
int value = *ptr; // 未知行为，因为指针指向不确定的内存区域
```

在这个例子中，指针 `ptr` 没有被初始化，它的值是不确定的。试图通过这个指针访问或修改内存可能导致未知行为。

**已释放的内存**

```CPP
int *ptr = new int(10);
delete ptr; // 释放内存
int value = *ptr; // 未知行为，因为指针指向的内存已被释放
```

在这个例子中，`ptr` 指向通过 `new` 分配的内存，然后使用 `delete` 将其释放。在释放内存之后，`ptr` 变成了野指针，因为它现在指向无效的内存区域。通过这个指针访问或修改内存可能导致未知行为。

**指向局部变量的指针**

```CPP
int *ptr;

void func() {
    int x = 10;
    ptr = &x;
} // x 的生命周期在函数退出时结束

func();
int value = *ptr; // 未知行为，因为指针指向的局部变量已超出生命周期
```

在这个例子中，指针 `ptr` 指向一个局部变量 `x`。但是，在 `func` 函数退出时，局部变量 `x` 的生命周期结束，其内存可能会被其他数据覆盖。因此，试图通过 `ptr` 访问或修改内存可能导致未知行为。



### 8. 如何检测内存泄漏？

内存泄漏是当程序无法释放不再使用的内存时发生的。这可能导致程序耗尽可用内存，从而降低性能或导致崩溃。

- **使用智能指针**：C++11引入了智能指针（例如`std::unique_ptr`和`std::shared_ptr`），这些指针可以自动管理内存。当智能指针的生命周期结束时，它们会自动释放所指向的内存。

- **重载new和delete**：可以重载全局或类特定的`new`和`delete`操作符，以便跟踪分配和释放的内存。通过这种方式发现没有被正确释放的内存。
- **使用内存泄漏检测工具**：有许多现成的工具可以帮助你检测C++程序中的内存泄漏。这些工具有助于识别和定位内存泄漏，以便你可以修复它们。一些流行的内存泄漏检测工具包括：
  - Valgrind：这是一个功能强大的Linux下的内存泄漏检测工具。它可以检测出许多内存泄漏和其他内存相关的问题。
  - Visual Leak Detector：这是一个用于Microsoft Visual Studio的内存泄漏检测插件。它可以帮助你在Windows平台上检测内存泄漏。
  - AddressSanitizer：这是一个由Google开发的内存错误检测器，可以在编译时加入你的程序。它可以检测内存泄漏、越界访问等问题。
- **静态代码分析**：使用静态代码分析工具（例如Clang-Tidy、Cppcheck等）可以在编译时自动检测潜在的内存泄漏和其他问题。
- **代码审查和测试**：定期进行代码审查和编写测试用例可以识别和预防内存泄漏。



### 9. 如何避免“野指针”？

**初始化指针**：在定义指针时，将其初始化为`nullptr`。这样可以确保指针不会指向随机内存地址。例如：

```CPP
int* ptr = nullptr;
```

**使用智能指针**：智能指针，如`std::unique_ptr`和`std::shared_ptr`，可以自动管理内存。当智能指针的生命周期结束时，它们会自动释放所指向的内存。使用智能指针可以帮助避免野指针，因为它们在释放内存后将自动设置为`nullptr`。

**在释放内存后将指针设为nullptr**：当你使用`delete`释放内存后，将原始指针设置为`nullptr`。这样可以防止对已释放内存的意外访问。例如：

```CPP
delete ptr;
ptr = nullptr;
```

**避免多次释放**：确保在释放指针指向的内存后不再次释放该内存。多次释放可能导致未定义行为。可以通过在释放内存后将指针设置为`nullptr`来实现这一点。

**不要返回局部变量的地址**：避免从函数中返回局部变量的地址。局部变量在函数返回后可能被销毁，因此指向它们的指针将成为野指针。如果需要返回指针，请使用动态内存分配（例如`new`）或返回全局/静态变量的地址。

**检查指针有效性**：在使用指针之前，检查它们是否有效。例如，确保指针不是`nullptr`，并确保它们指向合法的内存地址。

**谨慎使用指针算术**：在进行指针运算时要特别小心，以避免意外访问无效内存地址。确保在数组边界内使用指针，避免越界访问。



### 10. 常引用有什么作用？

可以提高代码的安全性、效率和可读性。

**保护数据不被修改**：通过将引用声明为`const`，你可以确保在引用的作用域内不会意外地修改所引用的数据。这有助于保护数据的完整性并减少潜在的编程错误。

```CPP
const int& const_ref = some_int_variable;
```

**避免不必要的拷贝**：当将大型对象作为函数参数传递时，使用常引用可以避免不必要的拷贝。这提高了程序的性能，尤其是在处理大型数据结构（如容器、类或结构体）时。例如：

```CPP
void func(const std::vector<int>& vec) {
    // 使用vec进行操作，但不会修改它
}
```

**支持多态性**：常引用允许将const对象和非const对象作为参数传递给函数。这使得函数能够处理各种参数类型，增强了函数的灵活性。

**重载函数的区分**：在C++中，你可以使用const引用来区分重载函数。例如，你可以为const和非const对象创建不同版本的成员函数。这使得在需要时可以为const对象提供特定的实现。

```CPP
class MyClass {
public:
    void someFunction() {
        // 非const版本
    }

    void someFunction() const {
        // const版本
    }
};
```





### 11. C++中的指针参数传递和引用参数传递？

**指针参数传递**

- 指针传递意味着将指针的副本传递给函数。因此，函数接收到的是指向原始数据的指针。
- 指针传递允许你修改所指向的数据，但不允许修改指针本身。因为传递的是指针的副本，所以在函数内部修改指针不会影响到原始指针。
- 指针可以为`nullptr`，这可能导致运行时错误。在使用指针之前，需要检查其有效性。
- 语法较为繁琐，需要使用指针操作符`*`和`->`来访问和修改数据。

```CPP
void pointer_pass(int* ptr) {
    if (ptr) {
        *ptr = 10; // 修改所指向的数据
    }
}
```

**引用参数传递**

- 引用传递意味着将原始数据的别名（引用）传递给函数。因此，函数接收到的是原始数据的引用，而不是副本。
- 引用传递允许直接修改原始数据，无需额外的操作符。
- 引用必须在创建时初始化，且不能重新绑定到其他对象。因此，引用通常比指针更安全。
- 语法更简洁，无需使用特殊操作符。引用的使用与普通变量相似。

```CPP
void reference_pass(int& ref) {
    ref = 10; // 直接修改原始数据
}
```

**总结**

- 指针传递通过传递指向原始数据的指针副本来实现。它允许修改所指向的数据，但不能修改指针本身。指针传递需要额外的操作符，并且可能引入`nullptr`错误。
- 引用传递通过传递原始数据的别名（引用）来实现。它允许直接修改原始数据，并具有更简洁的语法。引用传递通常比指针传递更安全。



### 12. 悬空指针和野指针有什么区别？

**悬空指针（Dangling Pointer）**： 悬空指针是指向已被释放内存的指针。当一个指针指向的对象被释放（例如通过delete操作符或者超出作用域）后，该指针并没有被重置为nullptr，而是继续指向原来的内存地址。由于原来的内存地址已经被释放，如果再次通过这个悬空指针访问或操作内存，可能会导致不确定的行为或程序崩溃。

**野指针（Wild Pointer）**： 野指针是指向未知或无效内存区域的指针。通常情况下，这是因为指针变量没有被初始化，或者指针的值被错误地设置为一个无效的地址。如果访问或操作野指针指向的内存，可能会导致程序崩溃或数据损坏。

总结：悬空指针和野指针的区别在于它们的来源，悬空指针通常是由于对象被释放后，指针没有被重置引起的；而野指针通常是由于指针没有被初始化或者被错误地设置为无效地址引起的。



### 13. 指针和引用之间如何转换？

在C++中，指针和引用之间不能直接转换，因为它们在底层实现和使用方式上有本质区别。但是可以通过一些方法在它们之间进行间接转换。

从引用到指针的转换： 要将引用转换为指针，可以使用取址操作符（&）来获取引用所指对象的地址。例如：

```cpp
int x = 10;
int &ref = x;       // ref 是 x 的引用
int *ptr = &ref;    // ptr 是指向 x 的指针
```

从指针到引用的转换： 要将指针转换为引用，可以使用解引用操作符（*）来获取指针所指向的对象。例如：

```cpp
int y = 20;
int *ptr2 = &y;     // ptr2 是指向 y 的指针
int &ref2 = *ptr2;  // ref2 是 y 的引用
```



### 14. 智能指针有哪些以及有什么作用？

**`std::unique_ptr`**

``std::unique_ptr`是一种独占式智能指针，表示对动态分配内存的唯一所有权。`unique_ptr`在作用域结束时自动释放它所管理的内存。由于它不能被复制，但可以通过`std::move`进行转移，因此确保了一个对象在任何时刻最多只有一个`unique_ptr`指向它。

```cpp
#include <memory>

void func() {
    std::unique_ptr<int> uptr(new int(42)); // uptr 独占指向动态分配的整数
} // uptr 超出作用域，自动释放内存
```

**`std::shared_ptr`**

`std::shared_ptr`是一种共享式智能指针，允许多个`shared_ptr`共享对同一个对象的所有权。`shared_ptr`使用引用计数机制来跟踪有多少个`shared_ptr`指向同一个对象。当最后一个指向该对象的`shared_ptr`超出作用域或被销毁时，对象的内存将被自动释放。

```cpp
#include <memory>

void func() {
    std::shared_ptr<int> sptr1(new int(42));
    {
        std::shared_ptr<int> sptr2 = sptr1; // sptr1 和 sptr2 共享指向同一个整数
    } // sptr2 超出作用域，但由于 sptr1 仍然存在，所以不释放内存
} // sptr1 超出作用域，引用计数为0，自动释放内存
```

**`std::weak_ptr`**

`std::weak_ptr`是一种弱引用智能指针，它不对所指向的对象的内存进行管理。`weak_ptr`通常与`shared_ptr`一起使用，用于避免循环引用（导致内存泄漏）的问题。`weak_ptr`可以从`shared_ptr`或另一个`weak_ptr`创建。要访问`weak_ptr`指向的对象，需要先将其提升为`shared_ptr`。

```cpp
#include <memory>

std::shared_ptr<int> sptr(new int(42));
std::weak_ptr<int> wptr = sptr;

if (auto locked = wptr.lock()) { // 提升为 shared_ptr
    // 使用 locked 访问对象
} else {
    // 对象已被释放
}
```



### 15. shared_ptr出现循环引用怎么解决？

`shared_ptr` 是 C++ 中的一种智能指针，它能自动管理引用计数，当引用计数为零时，智能指针所指向的对象会被自动销毁。但是，当出现循环引用时，智能指针可能会导致内存泄漏。

解决 `shared_ptr` 循环引用的常用方法是使用 `weak_ptr`。`weak_ptr` 是一种弱引用智能指针，它不会增加引用计数，但可以在需要时获取 `shared_ptr` 实例。通过将循环引用中的某个 `shared_ptr` 替换为 `weak_ptr`，可以避免循环引用，从而解决内存泄漏问题。

```cpp
#include <iostream>
#include <memory>

class B; // 前向声明

class A {
public:
    std::shared_ptr<B> b_ptr;
    ~A() { std::cout << "A destroyed" << std::endl; }
};

class B {
public:
    std::weak_ptr<A> a_ptr; // 使用 weak_ptr 替换 shared_ptr
    ~B() { std::cout << "B destroyed" << std::endl; }
};

int main() {
    {
        std::shared_ptr<A> a = std::make_shared<A>();
        std::shared_ptr<B> b = std::make_shared<B>();
        a->b_ptr = b;
        b->a_ptr = a;
    }
    // 当离开作用域时，A 和 B 实例将被正确销毁，避免了内存泄漏
    return 0;
}
```

类 A 和类 B 相互引用。使用 `weak_ptr` 替换了类 B 中的 `shared_ptr`，这样可以避免循环引用问题。当离开作用域时，A 和 B 实例将被正确销毁，避免了内存泄漏。



### 16. 使用智能指针管理内存资源，RAII？

智能指针是一种实现资源获取即初始化（RAII，Resource Acquisition Is Initialization）的方式，用于自动管理内存资源。RAII 是通过将资源的生命周期与对象的生命周期绑定来确保资源的正确使用和释放的技术。

常见的智能指针，可以用来实现 RAII:

`std::unique_ptr`: 一种独占所有权的智能指针，同一时间只能有一个 `unique_ptr` 指向给定的对象。当 `unique_ptr` 离开作用域或被销毁时，它所指向的对象也会被自动销毁。这种智能指针适用于单一所有权的场景。

```cpp
#include <iostream>
#include <memory>

class MyClass {
public:
    MyClass() { std::cout << "MyClass constructor" << std::endl; }
    ~MyClass() { std::cout << "MyClass destructor" << std::endl; }
};

int main() {
    {
        std::unique_ptr<MyClass> ptr = std::make_unique<MyClass>();
        // 当离开作用域时，MyClass 实例将被自动销毁
    }
    std::cout << "MyClass instance destroyed" << std::endl;
    return 0;
}
```

`std::shared_ptr`: 允许多个 `shared_ptr` 实例共享同一个对象的所有权。引用计数器跟踪指向该对象的智能指针数量，当引用计数为零时，对象将被自动销毁。这种智能指针适用于多个所有者共享资源的场景。

`std::weak_ptr`: 一种弱引用智能指针，它不会增加引用计数。它通常与 `shared_ptr` 一起使用，以避免循环引用导致的内存泄漏。



### 17. 手写实现智能指针类？

智能指针类包括构造函数、拷贝构造函数、赋值操作符和析构函数。它还重载了解引用操作符和箭头操作符，使其更像原生指针。这个实现使用一个整数变量来存储引用计数，并在拷贝构造函数和赋值操作符中更新引用计数。在析构函数中，如果引用计数为零，则释放内存资源。

```cpp
#include <iostream>

template <typename T>
class SmartPointer {
public:
    // 构造函数
    explicit SmartPointer(T* ptr = nullptr) : _ptr(ptr), _count(ptr ? new int(1) : nullptr) {}

    // 拷贝构造函数
    SmartPointer(const SmartPointer<T>& other) : _ptr(other._ptr), _count(other._count) {
        if (_count) {
            ++(*_count);
        }
    }

    // 赋值操作符
    SmartPointer<T>& operator=(const SmartPointer<T>& other) {
        if (this != &other) {
            release();
            _ptr = other._ptr;
            _count = other._count;
            if (_count) {
                ++(*_count);
            }
        }
        return *this;
    }

    // 析构函数
    ~SmartPointer() {
        release();
    }

    // 重载解引用操作符
    T& operator*() const {
        return *_ptr;
    }

    // 重载箭头操作符
    T* operator->() const {
        return _ptr;
    }

    // 获取引用计数
    int use_count() const {
        return _count ? *_count : 0;
    }

private:
    // 释放资源
    void release() {
        if (_count && --(*_count) == 0) {
            delete _ptr;
            delete _count;
        }
    }

    T* _ptr;
    int* _count;
};

int main() {
    SmartPointer<int> sp1(new int(10));
    std::cout << "sp1 use_count: " << sp1.use_count() << std::endl;

    SmartPointer<int> sp2(sp1);
    std::cout << "sp1 use_count: " << sp1.use_count() << std::endl;
    std::cout << "sp2 use_count: " << sp2.use_count() << std::endl;

    return 0;
}

```