---
title: 函数和运算重载符
author: ["comewei","jjd","枫长","LH"]
---



### 1. main函数有没有返回值？

在 C++ 中，main 函数确实有返回值。main 函数的返回值类型为 int，表示程序的退出状态。根据 C++ 标准，main 函数应具有以下两种形式之一：

1. 不带参数的 main 函数：

   ```cpp
   int main()
   {
       // 程序代码
       return 0;
   }
   ```

2. 带参数的 main 函数：

   ```cpp
   int main(int argc, char* argv[])
   {
       // 程序代码
       return 0;
   }
   ```

在这两种形式中，main 函数都有一个整数类型的返回值。通常，返回值 0 表示程序正常退出，非零值表示程序异常或错误。操作系统会捕获这个返回值，用于诊断程序的退出原因。如果 main 函数没有显式地包含 return 语句，编译器会自动插入一个`return 0;`作为默认返回值。因此，即使没有写 return 语句，main 函数仍然会返回一个整数值。

### 2. C++怎么实现一个函数先于main函数运行？

**做法一：**

`__attribute__((constructor))` 和 `__attribute__((destructor))` 是GCC编译器提供的特殊属性，用于指定某个函数在程序启动之前（主函数main()执行之前）或退出之后（main()函数执行结束后）自动执行。这些属性主要用于执行一些全局的初始化和清理操作。

`__attribute__((constructor))`： 当一个函数被声明为 `__attribute__((constructor))` 时，这个函数将在程序启动之前（main()执行之前）自动执行。这对于执行一些全局的初始化操作非常有用。

例如：

```cpp
#include <iostream>

void __attribute__((constructor)) init_function() {
    std::cout << "Before main()" << std::endl;
}

int main() {
    std::cout << "Inside main()" << std::endl;
    return 0;
}
```

输出：

```cpp
Before main()
Inside main()
```

`__attribute__((destructor))`： 当一个函数被声明为 `__attribute__((destructor))` 时，这个函数将在程序退出之后（main()函数执行结束后）自动执行。这对于执行一些全局的清理操作非常有用。

例如：

```cpp
#include <iostream>

void __attribute__((destructor)) cleanup_function() {
    std::cout << "After main()" << std::endl;
}

int main() {
    std::cout << "Inside main()" << std::endl;
    return 0;
}
```

输出：

```cpp
Inside main()
After main()
```

这些属性仅适用于GCC编译器，因此在使用其他编译器时，可能需要查找类似的功能。同时，这些功能在程序开发中应谨慎使用，以避免增加程序的复杂性。

**做法二：**

在 C++ 中，可以使用全局对象的构造函数在 main 函数之前运行一些代码。全局对象的构造函数在 main 函数执行前调用，析构函数在 main 函数执行后调用。这里是一个简单的例子：

```cpp
#include <iostream>

class MyPreMain {
public:
    MyPreMain() {
        std::cout << "This is called before main()" << std::endl;
    }

    ~MyPreMain() {
        std::cout << "This is called after main()" << std::endl;
    }
};

// 定义一个全局对象
MyPreMain my_pre_main;

int main() {
    std::cout << "This is main()" << std::endl;
    return 0;
}
```

输出如下：

```cpp
This is called before main()
This is main()
This is called after main()
```

如上所示，MyPreMain 类的构造函数在 main 函数之前执行，析构函数在 main 函数之后执行。通过在全局对象的构造函数中执行所需的代码，可以在 main 函数之前完成一些操作。然而，这种方法应谨慎使用，因为全局对象的构造函数和析构函数的调用顺序可能受到编译器和链接器的影响。

### 3. 函数调用过程栈的变化，返回值和参数变量哪个先入栈？

在讲解 C++ 函数调用过程栈的变化之前，需要了解一下函数调用栈（Call Stack）的基本概念。函数调用栈是一种数据结构，用于存储函数调用的上下文信息，包括局部变量、参数、返回地址等。每次调用一个函数时，都会在栈上分配一个新的栈帧（Stack Frame），用于存储当前函数的上下文信息。函数执行完成后，栈帧会被销毁，控制权返回到调用者。

C++ 函数调用过程栈的变化依赖于编译器和操作系统。常见的调用约定包括 cdecl、stdcall 和 fastcall 等。不同的调用约定有不同的参数传递方式和栈平衡策略。以下是一个简化的、通用的 C++ 函数调用过程栈变化示例：

1. 函数调用者将参数按照从右至左的顺序压入栈。
2. 函数调用者将返回地址压入栈。
3. 控制权转移到被调用函数。被调用函数为局部变量分配空间，将它们压入栈。
4. 被调用函数执行。
5. 被调用函数将返回值放入寄存器（如 EAX）或栈中的指定位置。
6. 被调用函数清理局部变量，销毁当前栈帧。
7. 控制权返回到函数调用者，恢复调用者的栈帧。
8. 函数调用者从栈中获取返回值。
9. 函数调用者清理参数。

请注意，这个过程并非固定不变，具体实现可能因编译器、操作系统和硬件平台而异。根据具体的调用约定，参数和返回值的传递方式也可能有所不同。一些调用约定可能会将参数通过寄存器传递，而不是通过栈。在通用示例中，参数是先入栈的。然后是返回地址。这是一个典型的调用过程，但实际实现可能会有所不同。总之，在研究函数调用过程时，需要考虑到编译器、操作系统和硬件平台的特性。



### 4. 谈一谈运算符重载？

运算符重载（Operator Overloading）是 C++ 中的一种特性，它允许程序员为自定义类型（如类和结构体）定义运算符的行为。这使得可以使用自然的语法来操作自定义类型的对象，提高了代码的可读性和易用性。运算符重载通过实现特殊的成员函数或非成员函数来完成。以下是一些关于运算符重载的详细介绍：

1. 成员函数和非成员函数：运算符重载可以通过实现类的成员函数或者非成员函数（通常是友元函数）来完成。例如，可以通过实现类的成员函数 `operator+` 来重载 `+` 运算符。
2. 可重载运算符：大多数 C++ 运算符都可以重载，例如 `+`、`-`、`*`、`/`、`%`、`==`、`!=`、`<`、`>`、`+=`、`-=` 等。然而，有一些运算符不能重载，如条件运算符（?:）、作用域解析运算符（::）和成员选择运算符（. 和 .*）。
3. 重载运算符的规则和限制：
   - 重载运算符的参数至少要有一个是自定义类型。这是为了防止对内置类型的运算符进行重载。
   - 不能更改运算符的优先级和结合性。
   - 重载运算符的数量和顺序应与原始运算符相同。
   - 除了赋值运算符（operator=）之外，运算符重载不能有默认实现。赋值运算符在未显式重载时会自动生成一个默认实现，执行逐成员赋值操作。

以下是一个运算符重载的示例，定义了一个简单的 `Complex` 类，用于表示复数，并重载了 `+` 和 `<<` 运算符：

```cpp
#include <iostream>

class Complex {
public:
    Complex(double real, double imag) : real_(real), imag_(imag) {}

    // 重载 + 运算符（成员函数）
    Complex operator+(const Complex& other) const {
        return Complex(real_ + other.real_, imag_ + other.imag_);
    }

    // 重载 << 运算符（友元函数）
    friend std::ostream& operator<<(std::ostream& os, const Complex& c) {
        os << c.real_ << " + " << c.imag_ << "i";
        return os;
    }

private:
    double real_;
    double imag_;
};

int main() {
    Complex c1(1.0, 2.0);
    Complex c2(3.0, 4.0);
    Complex c3 = c1 + c2;

    std::cout << "c1: " << c1 << std::endl;
    std::cout << "c2: " << c2 << std::endl;
    std::cout << "c3: " << c3 << std::endl;

    return 0;
}
```

在使用运算符重载时，需要注意以下几点：

- **保持运算符的语义**：在重载运算符时，应确保新的行为与原始运算符的语义保持一致。例如，`+` 运算符通常表示两个对象相加的操作，而 `==` 表示两个对象是否相等。不要为运算符赋予违反直觉或不符合通用规则的行为。
- **谨慎使用运算符重载**：过度使用运算符重载可能导致代码难以阅读和理解。在某些情况下，使用普通的成员函数可能会更清晰地表达意图。只有当运算符重载能明显提高代码可读性和易用性时，才考虑使用它。
- **确保运算符重载的效率**：在实现运算符重载时，要确保代码的效率。例如，在重载 `+` 运算符时，避免创建不必要的临时对象。在适当的情况下，可以考虑使用右值引用和移动语义来提高性能。

### 5. C++模板是什么，模板推导的规则？

C++模板（Templates）是一种泛型编程机制，允许在编写代码时使用参数化类型，从而使得函数或类能够处理多种数据类型。这可以提高代码的可重用性和灵活性。模板分为两种：函数模板和类模板。

**函数模板**：用于创建通用函数，适用于多种类型的参数。函数模板的定义使用 `template` 关键字，并在尖括号 `< >` 中指定类型参数。

```cpp
template <typename T>
T add(const T& a, const T& b) {
    return a + b;
}
```

**类模板**：用于创建通用类，适用于多种类型的成员。类模板的定义也使用 `template` 关键字，并在尖括号 `< >` 中指定类型参数。

```cpp
template <typename T>
class Stack {
public:
    void push(const T& value);
    T pop();
    bool empty() const;

private:
    std::vector<T> data_;
};
```

模板推导是编译器根据实际参数类型自动推导模板参数类型的过程。当使用模板函数时，大多数情况下无需显式指定模板参数类型，编译器会根据实际参数类型自动推导出相应的模板参数类型。以下是模板推导的规则：

- **类型推导**：如果实际参数类型与模板参数类型相同，那么直接使用实际参数类型。例如，`add(1, 2)` 中的实际参数类型为 `int`，因此模板参数类型为 `int`。
- **引用折叠**：当实际参数类型是引用时，编译器会进行引用折叠。例如，`add<int&>(a, b)` 中的实际参数类型为 `int&`，因此模板参数类型为 `int`。
- **const 传递**：编译器会自动移除 `const` 限定符。例如，`add(1, 2)` 中的实际参数类型为 `const int`，因此模板参数类型为 `int`。
- **数组和函数到指针的转换**：如果实际参数类型是数组或函数，编译器会将其转换为指针。例如，`add(arr1, arr2)` 中的实际参数类型为 `int[]`，因此模板参数类型为 `int*`。
- **无法推导的情况**：有些情况下，编译器无法推导出模板参数类型。例如，当实际参数类型不一致时（如 `add(1, 2.0)`），或者当实际参数类型与模板参数类型之间存在多层间接关系时。在这些情况下，需要显式指定模板参数类型。

### 6. 模板类和模板函数的区别是什么？

**实现目标**

- 模板类：模板类主要用于创建具有通用数据成员和成员函数的类。通过参数化类型，模板类可以在多种类型之间复用相同的代码。常见的例子是容器类，如 `std::vector<T>` 和 `std::list<T>`。
- 模板函数：模板函数主要用于创建通用的、适用于多种数据类型的函数。它们通常用于实现独立于类型的算法，如排序、查找等。例如，`std::sort` 和 `std::find` 都是模板函数。

**定义方式**

- 模板类：模板类使用 `template` 关键字和尖括号 `< >` 来定义类型参数。类型参数紧跟在 `template` 关键字之后，然后是类定义。

  ```cpp
  template <typename T>
  class MyClass {
      // 类的成员定义
  };
  ```

- 模板函数：模板函数也使用 `template` 关键字和尖括号 `< >` 来定义类型参数。类型参数紧跟在 `template` 关键字之后，然后是函数定义。

  ```cpp
  template <typename T>
  T myFunction(T a, T b) {
      // 函数实现
  }
  ```

**使用方式**

- 模板类：在使用模板类时，需要为其指定类型参数。类型参数在类名之后的尖括号 `< >` 中提供。例如，`std::vector<int>` 表示一个存储 `int` 类型元素的向量。
- 模板函数：当调用模板函数时，通常无需显式指定类型参数。编译器会根据实际参数类型自动推导出相应的模板参数类型。例如，调用 `myFunction(1, 2)` 时，编译器会自动推导出模板参数类型为 `int`。

### 7. 函数模板实现机制？

**定义函数模板**

函数模板的定义以 `template` 关键字开始，后面跟尖括号 `< >`，其中包含一个或多个类型参数。类型参数通常使用 `typename` 或 `class` 关键字声明。接下来是函数的声明和实现。

```cpp
template <typename T>
T max(const T& a, const T& b) {
    return a > b ? a : b;
}
```

在这个例子中定义了一个名为 `max` 的函数模板，它接受两个类型为 `T` 的参数，并返回较大的那个。

**实例化函数模板**

当调用函数模板时，编译器会根据实际参数的类型自动推导出相应的模板参数类型。然后，编译器会为每个不同的模板参数类型生成一个具体的函数实例。这个过程称为模板实例化。

```cpp
int main() {
    int a = 1, b = 2;
    double x = 3.0, y = 4.0;

    int c = max(a, b); // 实例化为 int 类型的 max 函数
    double z = max(x, y); // 实例化为 double 类型的 max 函数
}
```

在这个例子中，分别调用了 `max` 函数模板的 `int` 版本和 `double` 版本。编译器会为每个版本生成相应的函数实例。

**模板代码生成**

在编译过程中，编译器会为每个不同的函数模板实例生成相应的目标代码。这些代码在链接阶段被合并到最终的可执行文件中。模板代码的生成可能会导致代码膨胀，因为对于每个不同的模板参数类型，编译器都会生成一个新的函数实例。为了减小这种影响，编译器通常会进行一定程度的优化，以减少生成的代码的大小。