<?php
/**
 * Helper classes for PrestaShop CMS.
 *
 * @author    Maksim T. <zapalm@yandex.com>
 * @copyright 2018 Maksim T.
 * @license   https://opensource.org/licenses/MIT MIT
 * @link      https://github.com/zapalm/prestashop-helpers GitHub
 * @link      https://prestashop.modulez.ru/en/tools-scripts/53-helper-classes-for-prestashop.html Homepage
 */

namespace zapalm\prestashopHelpers\helpers;

use DateTime;

/**
 * Date helper.
 *
 * @version 0.2.0
 *
 * @author Maksim T. <zapalm@yandex.com>
 */
class DateHelper {

    /** Empty Date/Time. */
    const EMPTY_DATETIME = '0000-00-00 00:00:00';

    /** Beginning of the UNIX epoch. */
    const EPOCH_UNIX = '1970-01-01 00:00:00';

    /** Date/Time format by default. */
    const FORMAT_DATETIME_DEFAULT = 'Y-m-d H:i:s';

    /** Seconds in a hour. */
    const SEC_IN_HOUR = 3600;

    /** Seconds in a day. */
    const SEC_IN_DAY = 86400;

    /** Seconds in a week. */
    const SEC_IN_WEEK = 604800;

    /** Seconds in a month. */
    const SEC_IN_MONTH = 2419200;

    /** Seconds in a year. */
    const SEC_IN_YEAR = 31536000;

    /**
     * Returns a title of a day according to the current language.
     *
     * @param DateTime|null $dateTime        The DateTime object or null to use the current day.
     * @param string|null   $languageIsoCode The language ISO code or null to get it automatically from the context.
     *
     * @return string
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    public static function getDayTitle(DateTime $dateTime = null, $languageIsoCode = null)
    {
        if (null === $dateTime) {
            $dateTime = new DateTime();
        }

        return TranslateHelper::translate($dateTime->format('l'), static::class, $languageIsoCode);
    }

    /**
     * Returns a title of a month according to the current language.
     *
     * @param DateTime|null $dateTime        The DateTime object or null to use the current month.
     * @param string|null   $languageIsoCode The language ISO code or null to get it automatically from the context.
     * @param string|null   $prefix          The prefix, for example: 'of', 'in' and etc. It is usable for a correct translation with a declination.
     *
     * @return string
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    public static function getMonthTitle(DateTime $dateTime = null, $languageIsoCode = null, $prefix = null)
    {
        if (null === $dateTime) {
            $dateTime = new DateTime();
        }

        $title = trim($prefix . ' ' . $dateTime->format('F'));

        return TranslateHelper::translate($title, static::class, $languageIsoCode);
    }
}