module.exports = function (grunt) {

    var path = require('path'),

        HANDLEBARS_LIB        = path.join(process.cwd(), '../../../', 'handlebars.js'),
        HANDLEBARS_BUILD_PATH = 'dist/cjs',
        PREPENDED_COMMENT     = '/* THIS FILE IS GENERATED BY A BUILD SCRIPT - DO NOT EDIT! */\n',
        REMOVE_EXPORTS_REGEX  = /^\/\/ BEGIN\(BROWSER\)$\s*([\s\S]*)^\/\/ END\(BROWSER\)$/m,
        REMOVE_REQUIRES_REGEX = /^.+=\s*require\([^\)]+\)[^;,]*[;,]\s*/gm,
        MODULE_TEMPLATE = '{{comment}}' +
                          '{{before}}' +
                          '(function ({{argumentNames}}) {\n' +
                          '{{module}}\n' +
                          '}({{arguments}}));\n' +
                          '{{after}}',

        ROOT_FILES = {
            'handlebars.js': {
                arguments: {
                    exports: 'Handlebars',
                    Handlebars: 'Handlebars.Runtime',
                    AST: 'Handlebars.AST',
                    Parser: 'Handlebars.Parser',
                    parse: 'Handlebars.parse',
                    Compiler: 'Handlebars.Compiler',
                    compile: 'Handlebars.compile',
                    precompile: 'Handlebars.precompile',
                    JavaScriptCompiler: 'Handlebars.JavaScriptCompiler'
                },
                'after': 'Y.Handlebars = Handlebars[\'default\'];'
            },
            'handlebars.runtime.js': {
                before: 'Handlebars.Runtime = {};',
                arguments: {
                    exports: 'Handlebars.Runtime',
                    base: 'Handlebars',
                    SafeString: 'Handlebars.SafeString',
                    Exception: 'Handlebars.Exception',
                    Utils: 'Handlebars.Utils',
                    runtime: 'Handlebars.VM'
                },
                after: 'Handlebars.Runtime = Handlebars.Runtime[\'default\'];'
            }
        },

        BASE_FILES = {
            'handlebars/base.js': {
                arguments: {
                    exports: 'Handlebars',
                    Utils: 'Handlebars.Utils',
                    Exception: 'Handlebars.Exception'
                }
            },
            'handlebars/utils.js': {
                before: 'Handlebars.Utils = {};',
                arguments: {
                    exports: 'Handlebars.Utils',
                    SafeString: 'Handlebars.SafeString'
                }
            },
            'handlebars/runtime.js': {
                before: 'Handlebars.VM = {};',
                arguments: {
                    exports: 'Handlebars.VM',
                    Utils: 'Handlebars.Utils',
                    Exception: 'Handlebars.Exception',
                    createFrame: 'Handlebars.createFrame',
                    COMPILER_REVISION: 'Handlebars.COMPILER_REVISION',
                    REVISION_CHANGES: 'Handlebars.REVISION_CHANGES'
                },
                after: 'Handlebars.template = Handlebars.VM.template;'
            },
            'handlebars/exception.js': {
                before: 'Handlebars.Exception = {};',
                arguments: {
                    exports: 'Handlebars.Exception'
                },
                after: 'Handlebars.Exception = Handlebars.Exception[\'default\'];'
            },
            'handlebars/safe-string.js': {
                before: 'Handlebars.SafeString = {};',
                arguments: {
                    exports: 'Handlebars.SafeString'
                },
                after: 'Handlebars.SafeString = Handlebars.SafeString[\'default\'];'
            }
        },

        COMPILER_FILES = {
            'handlebars/compiler/helpers.js': {
                before: 'Handlebars.Helpers = {};',
                arguments: {
                    exports: 'Handlebars.Helpers',
                    Exception: 'Handlebars.Exception'
                }
            },
            'handlebars/compiler/parser.js': {
                before: 'Handlebars.Parser = {};',
                arguments: {
                    exports: 'Handlebars.Parser'
                },
                after: 'Handlebars.Parser = Handlebars.Parser[\'default\'];'
            },
            'handlebars/compiler/base.js': {
                arguments: {
                    exports: 'Handlebars',
                    parser: 'Handlebars.Parser',
                    AST: 'Handlebars.AST',
                    extend: 'Handlebars.Utils.extend',
                    Helpers: 'Handlebars.Helpers'
                }
            },
            'handlebars/compiler/ast.js': {
                before: 'Handlebars.AST = {};',
                arguments: {
                    exports: 'Handlebars.AST',
                    Exception: 'Handlebars.Exception'
                },
                after: 'Handlebars.AST = Handlebars.AST[\'default\'];'
            },
            'handlebars/compiler/compiler.js': {
                arguments: {
                    exports: 'Handlebars',
                    isArray: 'Handlebars.Utils.isArray',
                    Exception: 'Handlebars.Exception'
                }
            },
            'handlebars/compiler/javascript-compiler.js': {
                before: 'Handlebars.JavaScriptCompiler = {};',
                arguments: {
                    exports: 'Handlebars.JavaScriptCompiler',
                    COMPILER_REVISION: 'Handlebars.COMPILER_REVISION',
                    REVISION_CHANGES: 'Handlebars.REVISION_CHANGES',
                    log: 'Handlebars.log',
                    Exception: 'Handlebars.Exception'
                },
                after: 'Handlebars.JavaScriptCompiler = Handlebars.JavaScriptCompiler[\'default\'];'
            }
        };

    function removeExports(contents) {
        var match = contents.match(REMOVE_EXPORTS_REGEX);
        return match ? match[1] : contents;
    }

    function removeRequires(contents) {
        return contents.replace(REMOVE_REQUIRES_REGEX, '');
    }

    function render(template, templateValues) {
        var stachs = template.match(/\{\{([^{}]+)\}\}/g) || [];

        stachs.forEach(function (stach) {
            var key = stach.substring(2, stach.length - 2).trim(),
                value = templateValues[key] !== undefined ? templateValues[key] : '';
            stach = stach.replace(/\{/g, '\\{').replace(/\}/g, '\\}');
            if (value instanceof Array) {
                value = value.join(', ');
            }
            template = template.replace(new RegExp(stach, 'g'),  value);
        });

        return template;
    }

    function processFile(file, options) {
        var module = grunt.file.read(file),
            templateValues = {};

        templateValues.comment = PREPENDED_COMMENT;
        templateValues.before = options.before && (options.before + '\n');
        templateValues.module = removeRequires(removeExports(module)).trim();
        templateValues.argumentNames = Object.keys(options.arguments || {});
        templateValues.arguments = [];
        templateValues.argumentNames.forEach(function (argumentName) {
            var argument = options.arguments[argumentName];
            templateValues.arguments.push(argument);
        });
        templateValues.after = options.after && (options.after + '\n');

        return render(MODULE_TEMPLATE, templateValues);
    }

    function importFiles(files, prefix) {
        var fileName;
        prefix || (prefix = '');

        for (fileName in files) {
            var src  = path.join(HANDLEBARS_LIB, HANDLEBARS_BUILD_PATH, fileName),
                dest = path.join('js', prefix + path.basename(fileName));

            if (!grunt.file.exists(src)) {
                grunt.fail.fatal('Did you build Handlebars.js yet? Make sure to run grunt inside handlebars.js.');
            }

            grunt.log.writeln('Importing: '.green + fileName.cyan + ' to ' + dest.cyan);
            grunt.file.write(dest, processFile(src, files[fileName]));
        };
    }

    // -- Tasks ----------------------------------------------------------------

    grunt.registerTask('find', 'Find Handlebars.js project for import.', function () {
        grunt.log.write('Looking for Handlebars.js...'.green);

        if (!grunt.file.exists(HANDLEBARS_LIB)) {
            grunt.log.writeln('');
            grunt.fail.fatal(
                'Could not locate Handlebars repo: ' + HANDLEBARS_LIB +
                '\nDid you clone it above the yui3 repo?');
        }

        grunt.log.writeln('found: '.white + HANDLEBARS_LIB.cyan);
    });

    grunt.registerTask('clean', 'Delete imported Handlebars source files.', function () {
        var files = grunt.file.expand('js/handlebars-*.js');

        files.forEach(function (file) {
            grunt.log.writeln('Deleting: '.red + file.cyan);
            grunt.file['delete'](file);
        });
    });

    grunt.registerTask('import-root', 'Import Handlebars root source files.', function () {
        importFiles(ROOT_FILES);
    });

    grunt.registerTask('import-base', 'Import Handlebars base source files.', function () {
        importFiles(BASE_FILES, 'handlebars-');
    });

    grunt.registerTask('import-compiler', 'Import Handlebars compiler source files.', function () {
        importFiles(COMPILER_FILES, 'handlebars-compiler-');
    });

    grunt.registerTask('import', [
        'import-root',
        'import-base',
        'import-compiler'
    ]);

    grunt.registerTask('all', [
        'find',
        'clean',
        'import'
    ]);

    grunt.registerTask('default', ['all']);
};
