---
title: Graphics
subtitle: Power of graphics in knitr
date: '2011-12-09'
slug: graphics
---

## The Graphics Manual

The graphics manual shows all cool bells and whistles about graphics in **knitr**.

- Source and output of the graphics manual
  - Rnw source: [knitr-graphics.Rnw](https://github.com/yihui/knitr/blob/master/inst/examples/knitr-graphics.Rnw)
  - LyX source: [knitr-graphics.lyx](https://github.com/yihui/knitr/blob/master/inst/examples/knitr-graphics.lyx)
  - PDF output: [knitr-graphics.pdf](https://github.com/yihui/knitr/releases/download/doc/knitr-graphics.pdf)

You will probably realize how much room there is for improvement of R graphics in publications. Don't accept whatever R gives you; it is time starting making your graphics beautiful and professional.

A few screenshots from the manual:

![tikz graphics in knitr](https://db.yihui.org/imgur/HCkka.png)

![ggplot2 in knitr](https://db.yihui.org/imgur/BTEiu.png)

I'd like to thank the authors of the [`tufte-handout`](http://code.google.com/p/tufte-latex/) class, based on which this document was written, and the **tikzDevice** package makes the font styles in plots consistent with the document class (using serif fonts).

## A note on custom graphical devices

The [chunk option](../../options/) `dev` accepts custom graphical devices which can be defined as an R function with three arguments. Here is an example of a PDF device using `pointsize` 10:

```r 
<<custom-dev>>=
my_pdf = function(file, width, height) {
  pdf(file, width = width, height = height, pointsize = 10)
}
@
```

Then we can use this device in chunk options, but one important thing to remember is to provide the `fig.ext` option at the same time, because **knitr** is unable to guess what should be a correct file extension for the plot file. Finally we will use the custom device like this:

```r 
<<dev-demo, dev='my_pdf', fig.ext='pdf'>>=
plot(rnorm(100))
@
```

Of course you can set them globally using `\SweaveOpts{}` if you want to use this device through out the document.

## Passing more arguments to a device

We can actually have even finer control over graphical devices through the `dev.args` option. Instead of hardcoding `pointsize = 10`, we can add an option `dev.args = list(pointsize = 10)` to the chunk. Here is an example:

```r 
<<pdf-pointsize, dev='pdf', dev.args=list(pointsize=10)>>=
plot(rnorm(100))
@
```

Since `dev.args` is a list, it can take any possible device arguments, e.g. `dev.args=list(pointsize=11, family='serif')` for the `pdf()` device. All elements of `dev.args` are passed to the graphical device of the chunk.

## Create hyperlinks in R graphics

With the help of the **tikzDevice** package, we can write almost any LaTeX commands in R graphics. Here is an example [links.Rnw](https://gist.github.com/1937313) of writing hyperlinks (courtesy of Jonathan Kennel).

An important note is you have to add `\usepackage{hyperref}` to the list of metrics packages used by **tikzDevice**, otherwise the command `\hyperlink` or `\hypertarget` will not be recognized.

## Encoding of multibyte characters

When your plots contain multibyte characters, you may need set the `encoding` option of the `pdf()` device; see [#172](https://github.com/yihui/knitr/issues/172) for a discussion. For a possible list of encodings, see

```r 
list.files(system.file('enc', package = 'grDevices'))
## e.g. you can set pdf.options(encoding = 'CP1250')
```

You probably need to set the encoding when you see a warning like this: `Warning: conversion failure on '<var>' in 'mbcsToSbcs': dot substituted for <var>`.

Another possibility is to use the `cairo_pdf` device instead of `pdf` (see [#436](https://github.com/yihui/knitr/issues/436)):

```r 
options(device = function(file, width = 7, height = 7, ...) {
  cairo_pdf(tempfile(), width = width, height = height, ...)
})
```

If that fails under Windows, you may also take a look at [#527](https://github.com/yihui/knitr/issues/527).

## The Dingbats font

According to the documentation of `pdf()`, the `useDingbats` argument can reduce the file size of PDF's which contain small circles. If you are using **knitr** in RStudio, this option is disabled by default. You may want to enable it by putting `pdf.options(useDingbats = TRUE)` in your source document if you have large scatter plots. See [#311](https://github.com/yihui/knitr/issues/311) for more comments and discussions.

## Animations

When the chunk option `fig.show='animate'` and there are multiple plots produced from a code chunk, all plots will be combined to an animation. For LaTeX output, the LaTeX package **animate** is used to create animations in PDF. For HTML/Markdown output, by default [FFmpeg](http://ffmpeg.org) is used to create a [WebM](http://www.webmproject.org) video. Note you have to enable the **libvpx** support when installing FFmpeg. Linux and Windows users can just follow the download links on the FFmpeg website (**libvpx** has been enabled in the binaries). For OS X users, you can install FFmpeg via [Homebrew](http://brew.sh)

```bash
brew install ffmpeg --with-libvpx
```
