---
title: Looking for Software Paths in Windows Registry
date: '2010-03-28'
slug: looking-for-software-paths-in-windows-registry
---

When we want to call external programs in R under Windows, we often need to know the paths of these programs. For instance, we may want to know where ImageMagick is installed, as we need the `convert` (`convert.exe`) utility to convert images to other formats, or where OpenBUGS is installed because we need this path to use the function `bugs()`. Usually this problem does not exist under Linux, because the executables (or their symbolic links) are often put in the directories which are in the environment variable `PATH` (e.g. `/usr/bin`, `/usr/local/bin`).

However, we may be able to find the paths through the registry if the installation will save the path info in the registry hive. The R function is `readRegistry()`:

```r 
## ImageMagick:
## I used this trick in the function saveMovie (the animation package)
> readRegistry("SOFTWARE\\ImageMagick\\Current")
$BinPath
[1] "C:\\Program Files\\ImageMagick"
$CoderModulesPath
[1] "C:\\Program Files\\ImageMagick\\modules\\coders"
$ConfigurePath
[1] "C:\\Program Files\\ImageMagick\\config"
$FilterModulesPath
[1] "C:\\Program Files\\ImageMagick\\modules\\filters"
$LibPath
[1] "C:\\Program Files\\ImageMagick"
$QuantumDepth
[1] 16
$Version
[1] "6.3.8"

## OpenBUGS
> r = names(readRegistry("Software\\Microsoft\\Windows\\ShellNoRoam\\MUICache",
+    "HCU"))
> dirname(r[grep("OpenBUGS\\.exe", r)])
[1] "C:/Program Files/OpenBUGS"
```

There is no guarantee for this approach to work on any Windows platforms, but I think this is better than explaining what is the `PATH` variable to some Windows users...

## Examples

Here I show two examples for ImageMagick and OpenBUGS respectively. Make sure you are using Windows and have already installed ImageMagick and OpenBUGS. The first example shows how to find ImageMagick and call it to convert a sequence of images (generated by R) to a GIF animation:

```r 
## find the bin path of ImageMagick
magick.path = readRegistry("SOFTWARE\\ImageMagick\\Current")$BinPath

if (nzchar(magick.path) && require(animation)) {
    convert = shQuote(normalizePath(file.path(magick.path, "convert.exe")))

    ## record the demo 'recur.tree' in png images
    png("magickEx%d.png")
    demo(recur.tree)
    dev.off()

    ## call convert.exe to convert png images to GIF
    cmd = paste(sprintf("%s -delay", convert), 100, "-loop 0 magickEx*.png magickEx.gif")

    ## try to open the GIF animation
    if (system(cmd) == 0)
        shell.exec(file.path(getwd(), "magickEx.gif"))
}
```

You should be able to see the output as this:

![ImageMagick Example](https://db.yihui.org/imgur/O8BJB.gif)

OpenBUGS example:

```r 
r = names(readRegistry(
  "Software\\Microsoft\\Windows\\ShellNoRoam\\MUICache", "HCU"
))
bugsDIR = dirname(r[grep("OpenBUGS\\.exe", r)])
if (length(bugsDIR) == 1) {
    library(R2WinBUGS)
    library(BRugs)
    ## this example comes from ?bugs in R2WinBUGS
    model.file = system.file(package = "R2WinBUGS", "model",
        "schools.txt")
    data(schools)
    J = nrow(schools)
    y = schools$estimate
    sigma.y = schools$sd
    data = list("J", "y", "sigma.y")
    inits = function() {
        list(theta = rnorm(J, 0, 100), mu.theta = rnorm(1, 0,
            100), sigma.theta = runif(1, 0, 100))
    }
    parameters = c("theta", "mu.theta", "sigma.theta")
    schools.sim = bugs(data, inits, parameters, model.file,
        n.chains = 3, n.iter = 5000, bugs.directory = bugsDIR,
        program = "OpenBUGS")
    print(schools.sim)
    plot(schools.sim)
}
```

You might now be able to run the above examples in your Windows version, and I'll appreciate feedbacks from Vista and Win7 users. If the above approaches do not work, you may run the command `regedit` (Start --> Run) and search for ImageMagick/OpenBUGS, then change my code accordingly, because the locations of the registry hive for ImageMagick/OpenBUGS may not be the same for different versions of Windows. My session info is as below:

```r 
> sessionInfo()
R version 2.10.1 (2009-12-14)
i386-pc-mingw32

locale:
[1] LC_COLLATE=Chinese_People's Republic of China.936  LC_CTYPE=Chinese_People's Republic of China.936
[3] LC_MONETARY=Chinese_People's Republic of China.936 LC_NUMERIC=C
[5] LC_TIME=Chinese_People's Republic of China.936    

attached base packages:
[1] stats     graphics  grDevices utils     datasets  methods   base     

other attached packages:
[1] BRugs_0.5-3          R2WinBUGS_2.1-16     coda_0.13-5          lattice_0.18-3
[5] formatR_0.1-1        gWidgetsRGtk2_0.0-64 gWidgets_0.0-40      animation_1.1-2
[9] MASS_7.3-5          

loaded via a namespace (and not attached):
[1] grid_2.10.1   RGtk2_2.12.18 tools_2.10.1

> Sys.info()[1:3]
 sysname                      release                      version
 "Windows"                         "XP" "build 2600, Service Pack 3"
```
