/****************************************************************************
 ** @license
 ** This demo file is part of yFiles for HTML 2.6.
 ** Copyright (c) 2000-2024 by yWorks GmbH, Vor dem Kreuzberg 28,
 ** 72070 Tuebingen, Germany. All rights reserved.
 **
 ** yFiles demo files exhibit yFiles for HTML functionalities. Any redistribution
 ** of demo files in source code or binary form, with or without
 ** modification, is not permitted.
 **
 ** Owners of a valid software license for a yFiles for HTML version that this
 ** demo is shipped with are allowed to use the demo source code as basis
 ** for their own yFiles for HTML powered applications. Use of such programs is
 ** governed by the rights and conditions as set out in the yFiles for HTML
 ** license agreement.
 **
 ** THIS SOFTWARE IS PROVIDED ''AS IS'' AND ANY EXPRESS OR IMPLIED
 ** WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 ** MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 ** NO EVENT SHALL yWorks BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 ** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 ** TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 ** PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 ** LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 ***************************************************************************/
import type { INode } from 'yfiles'

/**
 * Describes a point of the trekking trail.
 */
export type Waypoint = {
  /**
   * The x-coordinate of the waypoint.
   */
  x: number

  /**
   * The y-coordinate of the waypoint.
   */
  y: number

  /**
   * The longitude of the waypoint.
   */
  lon?: number

  /**
   * The latitude of the waypoint.
   */
  lat?: number

  /**
   * The name of the waypoint.
   */
  name?: string

  /**
   * The category of the waypoint.
   */
  category?: string

  /**
   * The type of the waypoint.
   * Used for distinguishing the waypoints from the label nodes.
   */
  type?: number

  /**
   * An icon used for the description of the waypoint during highlighting.
   */
  icon?: string

  /**
   * The id of the waypoint.
   * Used only for the points from which the nodes will be created.
   */
  id?: number
}

/**
 * Describes the type of nodes used in this demo.
 */
export enum NodeType {
  /**
   * The nodes that represent the waypoints.
   */
  WAYPOINT = 0,
  /**
   * The nodes that represent the label associated with the waypoints.
   */
  LABEL = 1
}

/**
 * Describes the dataset used in this demo which consists of a set of waypoints and a set of points
 * that form the trail.
 */
export type TrekkingData = {
  /**
   * Represent the nodes of the graph from which the label nodes should be created.
   */
  waypoints: Waypoint[]
  /**
   * Represents the trail points used for creating the trail curve.
   */
  trail: Waypoint[]
}

/**
 * Returns the data associated with each waypoint.
 */
export function getWayPoint(node: INode): Waypoint | null {
  return node.tag as Waypoint
}

/**
 * The dataset for this demo.
 */
export const nodeData: TrekkingData = {
  waypoints: [
    { id: 0, x: 0, y: 323.3, lon: 9.026896, lat: 48.502703, name: 'start', category: 'start' },
    {
      id: 1,
      x: 1019,
      y: 321.2,
      lon: 9.035927,
      lat: 48.509325,
      name: 'Camping Site',
      category: 'camping'
    },
    {
      id: 2,
      x: 1341,
      y: 323.3,
      lon: 9.039074,
      lat: 48.511317,
      name: 'Outdoor Pool',
      category: 'swimming-pool'
    },
    {
      id: 30,
      x: 1450,
      y: 323.3,
      lon: 9.04206,
      lat: 48.5105,
      name: 'Paul Horn Arena',
      category: 'stadium',
      icon: 'resources/icons/paul-horn-arena.png'
    },
    {
      id: 3,
      x: 2393,
      y: 321.4,
      lon: 9.04854,
      lat: 48.517157,
      name: 'Alleenbrücke',
      category: 'bridge'
    },
    {
      id: 4,
      x: 2774,
      y: 349.8,
      lon: 9.049259,
      lat: 48.520373,
      name: 'Tunnel',
      category: 'tunnel',
      icon: 'resources/icons/tunnel.jpg'
    },
    {
      id: 5,
      x: 2992,
      y: 337.7,
      lon: 9.051917,
      lat: 48.521185,
      name: 'Krummebrücke',
      category: 'bridge'
    },
    {
      id: 6,
      x: 5804,
      y: 350.3,
      lon: 9.024994,
      lat: 48.525352,
      name: 'yWorks',
      category: 'yWorks',
      icon: 'resources/icons/yWorks.jpg'
    },
    {
      id: 7,
      x: 7614,
      y: 393.5,
      lon: 9.019128,
      lat: 48.540385,
      name: 'Hagelloch',
      category: 'village',
      icon: 'resources/icons/hagelloch.jpg'
    },
    {
      id: 8,
      x: 9207,
      y: 455.5,
      lon: 9.034966,
      lat: 48.540423,
      name: 'Botanical Garden',
      category: 'attraction',
      icon: 'resources/icons/botanical-garden.jpg'
    },
    {
      x: 10372,
      y: 466.4,
      lon: 9.048972,
      lat: 48.543004,
      name: 'Tübingen Wanne',
      category: 'distinct',
      icon: 'resources/icons/wanne.jpg'
    },
    {
      id: 9,
      x: 13133,
      y: 356,
      lon: 9.06072,
      lat: 48.559192,
      name: 'Bebenhausen',
      category: 'village',
      icon: 'resources/icons/bebenhausen.jpg'
    },
    {
      id: 10,
      x: 14829,
      y: 341.2,
      lon: 9.074589,
      lat: 48.549555,
      name: 'Grill Place',
      category: 'grill'
    },
    {
      id: 11,
      x: 16612,
      y: 392.9,
      lon: 9.091633,
      lat: 48.56028,
      name: 'Picnic Area',
      category: 'picnic'
    },
    {
      id: 12,
      x: 17873,
      y: 410,
      lon: 9.094568,
      lat: 48.570602,
      name: 'Picnic Area',
      category: 'picnic'
    },
    {
      id: 13,
      x: 19716,
      y: 457.4,
      lon: 9.095602,
      lat: 48.58636,
      name: 'Observation Point',
      category: 'observation-point'
    },
    {
      id: 14,
      x: 20270,
      y: 496.5,
      lon: 9.101368,
      lat: 48.588836,
      name: 'Bärlochklinge',
      category: 'attraction'
    },
    {
      id: 15,
      x: 20437,
      y: 507.6,
      lon: 9.103438,
      lat: 48.588794,
      name: 'Kirnbach Falls',
      category: 'attraction'
    },
    {
      id: 16,
      x: 23734,
      y: 496.9,
      lon: 9.140792,
      lat: 48.593553,
      name: 'Drehscheibe',
      category: 'parking'
    },
    {
      id: 17,
      x: 24388,
      y: 473.9,
      lon: 9.1477,
      lat: 48.592905,
      name: 'Steinbrennershaus',
      category: 'attraction'
    },
    {
      id: 18,
      x: 28512,
      y: 405.9,
      lon: 9.180673,
      lat: 48.575245,
      name: 'Gniebel',
      category: 'village',
      icon: 'resources/icons/gniebel.jpg'
    },
    {
      id: 19,
      x: 29929,
      y: 369.9,
      lon: 9.174264,
      lat: 48.564336,
      name: 'Rübgarten',
      category: 'village'
    },
    {
      id: 20,
      x: 33657,
      y: 305.4,
      lon: 9.152787,
      lat: 48.544143,
      name: 'Lake',
      category: 'lake',
      icon: 'resources/icons/baggersee.jpg'
    },
    {
      id: 21,
      x: 37780,
      y: 407.5,
      lon: 9.119964,
      lat: 48.521858,
      name: 'Kusterdingen',
      category: 'village',
      icon: 'resources/icons/kusterdingen.jpg'
    },
    {
      id: 22,
      x: 44329,
      y: 324.9,
      lon: 9.059025,
      lat: 48.514822,
      name: 'Steinlachbrücke',
      category: 'bridge'
    },
    {
      id: 23,
      x: 46248,
      y: 332.3,
      lon: 9.06182,
      lat: 48.499754,
      name: 'Derendingen',
      category: 'village'
    },
    {
      id: 24,
      x: 48765,
      y: 401.7,
      lon: 9.063008,
      lat: 48.483616,
      name: 'Golf Court',
      category: 'golf'
    },
    {
      id: 25,
      x: 49501,
      y: 448.9,
      lon: 9.056095,
      lat: 48.479761,
      name: 'Kreßbach',
      category: 'village'
    },
    {
      id: 26,
      x: 51098,
      y: 401.8,
      lon: 9.035697,
      lat: 48.482549,
      name: 'Picnic Area',
      category: 'picnic'
    },
    {
      id: 27,
      x: 52358,
      y: 330.6,
      lon: 9.03394,
      lat: 48.491768,
      name: 'Weilheim',
      category: 'village'
    },
    {
      id: 28,
      x: 53784,
      y: 323.2,
      lon: 9.026843,
      lat: 48.502649,
      name: 'finish',
      category: 'finish'
    }
  ],
  trail: [
    { x: 0, y: 323.3, lon: 9.026896, lat: 48.502703 },
    { x: 14, y: 323.6, lon: 9.027047, lat: 48.502781 },
    { x: 18, y: 323.3, lon: 9.04206, lat: 48.5105 },
    { x: 46, y: 323.1, lon: 9.027445, lat: 48.502901 },
    { x: 80, y: 322.7, lon: 9.027845, lat: 48.503054 },
    { x: 114, y: 322.6, lon: 9.028242, lat: 48.503205 },
    { x: 154, y: 322.8, lon: 9.028705, lat: 48.503392 },
    { x: 183, y: 322.5, lon: 9.02905, lat: 48.503514 },
    { x: 217, y: 321.5, lon: 9.029439, lat: 48.503686 },
    { x: 257, y: 320.9, lon: 9.029852, lat: 48.503921 },
    { x: 286, y: 320.2, lon: 9.0302, lat: 48.504033 },
    { x: 322, y: 319.7, lon: 9.030575, lat: 48.504233 },
    { x: 357, y: 320.2, lon: 9.030892, lat: 48.50447 },
    { x: 393, y: 320.8, lon: 9.031165, lat: 48.504742 },
    { x: 427, y: 321.1, lon: 9.031364, lat: 48.505016 },
    { x: 464, y: 321.1, lon: 9.031523, lat: 48.505328 },
    { x: 497, y: 321, lon: 9.031666, lat: 48.505616 },
    { x: 534, y: 321, lon: 9.031902, lat: 48.505903 },
    { x: 585, y: 321, lon: 9.032304, lat: 48.506283 },
    { x: 606, y: 321.1, lon: 9.032464, lat: 48.506434 },
    { x: 648, y: 321.1, lon: 9.032754, lat: 48.506762 },
    { x: 675, y: 321, lon: 9.032981, lat: 48.506944 },
    { x: 710, y: 321.2, lon: 9.033316, lat: 48.507169 },
    { x: 745, y: 321.4, lon: 9.033664, lat: 48.507386 },
    { x: 801, y: 322, lon: 9.034195, lat: 48.507748 },
    { x: 813, y: 322, lon: 9.034304, lat: 48.507827 },
    { x: 854, y: 322, lon: 9.034663, lat: 48.508108 },
    { x: 880, y: 321.8, lon: 9.034889, lat: 48.508286 },
    { x: 929, y: 322.6, lon: 9.035272, lat: 48.508643 },
    { x: 951, y: 322.9, lon: 9.035471, lat: 48.508789 },
    { x: 1006, y: 321.8, lon: 9.035821, lat: 48.509225 },
    {
      x: 1019,
      y: 321.2,
      lon: 9.035927,
      lat: 48.509325,
      name: 'Camping Site',
      category: 'camping'
    },
    { x: 1061, y: 319.2, lon: 9.036261, lat: 48.509632 },
    { x: 1094, y: 320.1, lon: 9.036575, lat: 48.509845 },
    { x: 1104, y: 320.4, lon: 9.036669, lat: 48.509905 },
    { x: 1138, y: 321.4, lon: 9.037021, lat: 48.5101 },
    { x: 1183, y: 322.4, lon: 9.037464, lat: 48.510384 },
    { x: 1230, y: 321.7, lon: 9.037929, lat: 48.510666 },
    { x: 1241, y: 321.8, lon: 9.038045, lat: 48.510735 },
    { x: 1294, y: 323.1, lon: 9.0386, lat: 48.511029 },
    { x: 1307, y: 323.6, lon: 9.038727, lat: 48.511113 },
    {
      x: 1341,
      y: 323.3,
      lon: 9.039074,
      lat: 48.511317,
      name: 'Outdoor Pool',
      category: 'swimming'
    },
    { x: 1394, y: 323.8, lon: 9.039593, lat: 48.511637 },
    { x: 1414, y: 323.9, lon: 9.039789, lat: 48.511761 },
    { x: 1447, y: 324.2, lon: 9.040095, lat: 48.511985 },
    { x: 1481, y: 324.2, lon: 9.040387, lat: 48.512213 },
    { x: 1518, y: 323.5, lon: 9.04073, lat: 48.51246 },
    { x: 1550, y: 322.7, lon: 9.041038, lat: 48.51266 },
    { x: 1612, y: 322, lon: 9.04162, lat: 48.51306 },
    { x: 1626, y: 322.2, lon: 9.041755, lat: 48.513158 },
    { x: 1664, y: 323.2, lon: 9.042177, lat: 48.513345 },
    { x: 1689, y: 324.2, lon: 9.042499, lat: 48.513404 },
    { x: 1711, y: 325.8, lon: 9.042684, lat: 48.513567 },
    { x: 1753, y: 327.3, lon: 9.043002, lat: 48.513876 },
    { x: 1796, y: 324.6, lon: 9.043334, lat: 48.514192 },
    { x: 1802, y: 324.2, lon: 9.043372, lat: 48.514241 },
    { x: 1838, y: 321, lon: 9.043738, lat: 48.514449 },
    { x: 1873, y: 320.9, lon: 9.044038, lat: 48.514703 },
    { x: 1911, y: 321.3, lon: 9.044377, lat: 48.514955 },
    { x: 1929, y: 321.2, lon: 9.04456, lat: 48.515068 },
    { x: 1956, y: 321.1, lon: 9.044797, lat: 48.51525 },
    { x: 1991, y: 321.3, lon: 9.045141, lat: 48.515463 },
    { x: 2007, y: 321.6, lon: 9.045305, lat: 48.515559 },
    { x: 2027, y: 321.8, lon: 9.045513, lat: 48.515668 },
    { x: 2059, y: 321.7, lon: 9.045857, lat: 48.515847 },
    { x: 2091, y: 321.8, lon: 9.046271, lat: 48.515942 },
    { x: 2149, y: 322.3, lon: 9.046976, lat: 48.516163 },
    { x: 2193, y: 322.2, lon: 9.047478, lat: 48.51637 },
    { x: 2217, y: 322.4, lon: 9.047795, lat: 48.516442 },
    { x: 2240, y: 322.2, lon: 9.048079, lat: 48.516508 },
    { x: 2273, y: 321.8, lon: 9.048527, lat: 48.516548 },
    { x: 2281, y: 321.7, lon: 9.04864, lat: 48.51655 },
    { x: 2286, y: 321.7, lon: 9.048698, lat: 48.516535 },
    { x: 2292, y: 321.7, lon: 9.048725, lat: 48.516482 },
    { x: 2295, y: 321.8, lon: 9.048743, lat: 48.51646 },
    { x: 2305, y: 322.1, lon: 9.048863, lat: 48.516426 },
    { x: 2326, y: 322, lon: 9.048915, lat: 48.516616 },
    { x: 2338, y: 321.8, lon: 9.048875, lat: 48.516723 },
    { x: 2364, y: 321.2, lon: 9.048727, lat: 48.516931 },
    { x: 2381, y: 321, lon: 9.048616, lat: 48.517071 },
    {
      x: 2393,
      y: 321.4,
      lon: 9.04854,
      lat: 48.517157,
      name: 'Alleenbrücke',
      category: 'bridge'
    },
    { x: 2417, y: 322.8, lon: 9.04843, lat: 48.517367 },
    { x: 2439, y: 324.1, lon: 9.048354, lat: 48.517556 },
    { x: 2462, y: 325.6, lon: 9.048248, lat: 48.517747 },
    { x: 2481, y: 327.1, lon: 9.048126, lat: 48.517901 },
    { x: 2502, y: 332.4, lon: 9.048015, lat: 48.518074 },
    {
      x: 2774,
      y: 349.8,
      lon: 9.049259,
      lat: 48.520373,
      name: 'Tunnel',
      category: 'tunnel'
    },
    { x: 2791, y: 351, lon: 9.049464, lat: 48.520444 },
    { x: 2809, y: 351.7, lon: 9.049698, lat: 48.520501 },
    { x: 2825, y: 351.8, lon: 9.049905, lat: 48.520543 },
    { x: 2860, y: 347.2, lon: 9.050262, lat: 48.520753 },
    { x: 2875, y: 346.2, lon: 9.050453, lat: 48.520799 },
    { x: 2894, y: 344.6, lon: 9.050699, lat: 48.52085 },
    { x: 2940, y: 340.2, lon: 9.051264, lat: 48.521012 },
    { x: 2949, y: 338.9, lon: 9.051366, lat: 48.521066 },
    { x: 2982, y: 337.8, lon: 9.05178, lat: 48.521158 },
    {
      x: 2992,
      y: 337.7,
      lon: 9.051917,
      lat: 48.521185,
      name: 'Krummebrücke',
      category: 'bridge'
    },
    { x: 3004, y: 336.6, lon: 9.052023, lat: 48.521263 },
    { x: 3024, y: 335.4, lon: 9.052037, lat: 48.521443 },
    { x: 3059, y: 333.3, lon: 9.052089, lat: 48.521759 },
    { x: 3072, y: 332.5, lon: 9.052121, lat: 48.521872 },
    { x: 3096, y: 331, lon: 9.052074, lat: 48.522087 },
    { x: 3113, y: 332, lon: 9.052119, lat: 48.52194 },
    { x: 3115, y: 332, lon: 9.052143, lat: 48.521937 },
    { x: 3116, y: 332, lon: 9.052132, lat: 48.521946 },
    { x: 3117, y: 332, lon: 9.052126, lat: 48.52195 },
    { x: 3117, y: 332, lon: 9.052127, lat: 48.521949 },
    { x: 3117, y: 332, lon: 9.052125, lat: 48.52195 },
    { x: 3117, y: 331.9, lon: 9.052125, lat: 48.521951 },
    { x: 3117, y: 331.9, lon: 9.052125, lat: 48.521951 },
    { x: 3123, y: 331.6, lon: 9.052139, lat: 48.522001 },
    { x: 3128, y: 331.3, lon: 9.052151, lat: 48.522047 },
    { x: 3129, y: 331.2, lon: 9.052151, lat: 48.522058 },
    { x: 3136, y: 331, lon: 9.052182, lat: 48.522114 },
    { x: 3137, y: 331, lon: 9.052178, lat: 48.522124 },
    { x: 3145, y: 331, lon: 9.05213, lat: 48.522186 },
    { x: 3155, y: 331, lon: 9.052136, lat: 48.522283 },
    { x: 3160, y: 331, lon: 9.052127, lat: 48.52232 },
    { x: 3167, y: 331, lon: 9.052136, lat: 48.522389 },
    { x: 3172, y: 331.1, lon: 9.052146, lat: 48.522428 },
    { x: 3175, y: 331.1, lon: 9.052184, lat: 48.522428 },
    { x: 3180, y: 331.1, lon: 9.052193, lat: 48.522478 },
    { x: 3190, y: 331.2, lon: 9.052213, lat: 48.522567 },
    { x: 3197, y: 331.2, lon: 9.052223, lat: 48.522627 },
    { x: 3207, y: 331.1, lon: 9.052157, lat: 48.522706 },
    { x: 3210, y: 331.1, lon: 9.052135, lat: 48.522727 },
    { x: 3217, y: 331, lon: 9.052072, lat: 48.522781 },
    { x: 3221, y: 331, lon: 9.052056, lat: 48.522813 },
    { x: 3223, y: 331, lon: 9.052066, lat: 48.522829 },
    { x: 3224, y: 331, lon: 9.052071, lat: 48.522838 },
    { x: 3224, y: 331, lon: 9.052071, lat: 48.522839 },
    { x: 3224, y: 331, lon: 9.052073, lat: 48.522839 },
    { x: 3225, y: 331, lon: 9.052074, lat: 48.522837 },
    { x: 3227, y: 331, lon: 9.052094, lat: 48.522853 },
    { x: 3229, y: 331.1, lon: 9.052109, lat: 48.522865 },
    { x: 3230, y: 331.1, lon: 9.052119, lat: 48.522879 },
    { x: 3230, y: 331.1, lon: 9.052119, lat: 48.522879 },
    { x: 3230, y: 331.1, lon: 9.052119, lat: 48.522879 },
    { x: 3230, y: 331.1, lon: 9.052119, lat: 48.522879 },
    { x: 3231, y: 331.1, lon: 9.052126, lat: 48.522885 },
    { x: 3234, y: 331.1, lon: 9.052129, lat: 48.522906 },
    { x: 3234, y: 331.1, lon: 9.052137, lat: 48.522906 },
    { x: 3234, y: 331.1, lon: 9.052139, lat: 48.522906 },
    { x: 3238, y: 331.2, lon: 9.052143, lat: 48.522935 },
    { x: 3238, y: 331.2, lon: 9.052142, lat: 48.52294 },
    { x: 3238, y: 331.2, lon: 9.052145, lat: 48.522942 },
    { x: 3254, y: 331.1, lon: 9.051934, lat: 48.522949 },
    { x: 3312, y: 331.2, lon: 9.051147, lat: 48.522967 },
    { x: 3319, y: 331.1, lon: 9.051067, lat: 48.522936 },
    { x: 3364, y: 331, lon: 9.050456, lat: 48.522898 },
    { x: 3437, y: 331.8, lon: 9.049492, lat: 48.52279 },
    { x: 3470, y: 331.3, lon: 9.049043, lat: 48.522801 },
    { x: 3487, y: 331.1, lon: 9.048817, lat: 48.522765 },
    { x: 3537, y: 331.5, lon: 9.048138, lat: 48.522746 },
    { x: 3580, y: 330.7, lon: 9.047562, lat: 48.522769 },
    { x: 3616, y: 329.5, lon: 9.04707, lat: 48.522769 },
    { x: 3627, y: 329.3, lon: 9.046935, lat: 48.522725 },
    { x: 3651, y: 328.4, lon: 9.046609, lat: 48.522735 },
    { x: 3682, y: 327.5, lon: 9.046192, lat: 48.522719 },
    { x: 3715, y: 327.5, lon: 9.045843, lat: 48.522903 },
    { x: 3726, y: 328.1, lon: 9.045813, lat: 48.522996 },
    { x: 3749, y: 329.3, lon: 9.045573, lat: 48.523134 },
    { x: 3787, y: 330.2, lon: 9.045055, lat: 48.523119 },
    { x: 3800, y: 330.4, lon: 9.044887, lat: 48.523092 },
    { x: 3832, y: 330.8, lon: 9.044471, lat: 48.523031 },
    { x: 3862, y: 330.8, lon: 9.044066, lat: 48.522963 },
    { x: 3895, y: 331.2, lon: 9.043635, lat: 48.522922 },
    { x: 3946, y: 331.9, lon: 9.042953, lat: 48.522846 },
    { x: 3958, y: 331.9, lon: 9.042791, lat: 48.522821 },
    { x: 3998, y: 332.1, lon: 9.042268, lat: 48.522739 },
    { x: 4016, y: 332.2, lon: 9.042021, lat: 48.522709 },
    { x: 4032, y: 332.3, lon: 9.041823, lat: 48.522664 },
    { x: 4062, y: 332.4, lon: 9.04144, lat: 48.522572 },
    { x: 4125, y: 331.8, lon: 9.04062, lat: 48.5224 },
    { x: 4139, y: 331.4, lon: 9.04045, lat: 48.522363 },
    { x: 4162, y: 331.2, lon: 9.040157, lat: 48.522273 },
    { x: 4200, y: 330.9, lon: 9.039711, lat: 48.522104 },
    { x: 4237, y: 331.1, lon: 9.039307, lat: 48.52191 },
    { x: 4264, y: 331.5, lon: 9.039024, lat: 48.521761 },
    { x: 4274, y: 331.7, lon: 9.038909, lat: 48.521715 },
    { x: 4312, y: 332.9, lon: 9.038482, lat: 48.521525 },
    { x: 4320, y: 333.2, lon: 9.038382, lat: 48.521496 },
    { x: 4350, y: 332.8, lon: 9.038194, lat: 48.521742 },
    { x: 4371, y: 332.3, lon: 9.038147, lat: 48.521922 },
    { x: 4399, y: 332.1, lon: 9.037815, lat: 48.522045 },
    { x: 4437, y: 332.2, lon: 9.037323, lat: 48.521953 },
    { x: 4467, y: 331.9, lon: 9.036911, lat: 48.52193 },
    { x: 4494, y: 331.4, lon: 9.036553, lat: 48.521896 },
    { x: 4528, y: 331, lon: 9.03609, lat: 48.52185 },
    { x: 4563, y: 331, lon: 9.035635, lat: 48.521792 },
    { x: 4597, y: 331.5, lon: 9.035188, lat: 48.521713 },
    { x: 4630, y: 332, lon: 9.034764, lat: 48.521621 },
    { x: 4681, y: 331.6, lon: 9.034126, lat: 48.521438 },
    { x: 4693, y: 331.3, lon: 9.033972, lat: 48.521394 },
    { x: 4749, y: 331.1, lon: 9.033274, lat: 48.521205 },
    { x: 4774, y: 331.1, lon: 9.032949, lat: 48.521149 },
    { x: 4787, y: 331.1, lon: 9.032775, lat: 48.52112 },
    { x: 4819, y: 331.1, lon: 9.032356, lat: 48.52104 },
    { x: 4875, y: 332.1, lon: 9.031613, lat: 48.520954 },
    { x: 4900, y: 332.9, lon: 9.031286, lat: 48.5209 },
    { x: 4937, y: 334, lon: 9.030864, lat: 48.520728 },
    { x: 4943, y: 334.1, lon: 9.030793, lat: 48.520693 },
    { x: 4993, y: 334.4, lon: 9.030163, lat: 48.520531 },
    { x: 5007, y: 334.3, lon: 9.029978, lat: 48.520519 },
    { x: 5037, y: 334.2, lon: 9.029573, lat: 48.520496 },
    { x: 5073, y: 333.8, lon: 9.029086, lat: 48.520544 },
    { x: 5121, y: 334.4, lon: 9.028632, lat: 48.520855 },
    { x: 5135, y: 334.9, lon: 9.028611, lat: 48.520977 },
    { x: 5191, y: 335.8, lon: 9.028563, lat: 48.521481 },
    { x: 5225, y: 335.7, lon: 9.02852, lat: 48.521788 },
    { x: 5250, y: 335.7, lon: 9.028491, lat: 48.522004 },
    { x: 5275, y: 335.5, lon: 9.028467, lat: 48.522234 },
    { x: 5298, y: 335.4, lon: 9.02847, lat: 48.522439 },
    { x: 5340, y: 335, lon: 9.028448, lat: 48.522814 },
    { x: 5374, y: 335.5, lon: 9.028429, lat: 48.523122 },
    { x: 5379, y: 335.7, lon: 9.028431, lat: 48.523164 },
    { x: 5406, y: 336.6, lon: 9.028503, lat: 48.523408 },
    { x: 5434, y: 337.5, lon: 9.02859, lat: 48.523653 },
    { x: 5456, y: 338.1, lon: 9.0285, lat: 48.523841 },
    { x: 5479, y: 338.9, lon: 9.028325, lat: 48.524012 },
    { x: 5497, y: 339.1, lon: 9.028096, lat: 48.524051 },
    { x: 5528, y: 339.2, lon: 9.027681, lat: 48.524107 },
    { x: 5560, y: 339.5, lon: 9.027272, lat: 48.524199 },
    { x: 5573, y: 339.7, lon: 9.027101, lat: 48.524236 },
    { x: 5600, y: 340, lon: 9.026763, lat: 48.524323 },
    { x: 5624, y: 340.1, lon: 9.026454, lat: 48.524381 },
    { x: 5664, y: 341, lon: 9.025957, lat: 48.524527 },
    { x: 5677, y: 341.6, lon: 9.025815, lat: 48.524605 },
    { x: 5695, y: 342.2, lon: 9.025644, lat: 48.52471 },
    { x: 5717, y: 343.3, lon: 9.02544, lat: 48.524855 },
    { x: 5761, y: 347.7, lon: 9.025065, lat: 48.525169 },
    { x: 5775, y: 349.5, lon: 9.024958, lat: 48.525272 },
    { x: 5794, y: 351.4, lon: 9.024939, lat: 48.525437 },
    { x: 5800, y: 350.8, lon: 9.024976, lat: 48.525387 },
    {
      x: 5804,
      y: 350.3,
      lon: 9.024994,
      lat: 48.525352,
      name: 'yWorks',
      category: 'yWorks'
    },
    { x: 5828, y: 352.3, lon: 9.024771, lat: 48.525516 },
    { x: 5848, y: 353.2, lon: 9.024576, lat: 48.525632 },
    { x: 5866, y: 352.9, lon: 9.024414, lat: 48.52575 },
    { x: 5882, y: 352.8, lon: 9.024285, lat: 48.525871 },
    { x: 5910, y: 351.8, lon: 9.024532, lat: 48.526065 },
    { x: 5943, y: 350.7, lon: 9.024621, lat: 48.526353 },
    { x: 5962, y: 351.4, lon: 9.024429, lat: 48.526471 },
    { x: 5967, y: 351.6, lon: 9.02438, lat: 48.526495 },
    { x: 5981, y: 351.3, lon: 9.024414, lat: 48.526621 },
    { x: 6014, y: 352.7, lon: 9.024129, lat: 48.526845 },
    { x: 6041, y: 353.7, lon: 9.023961, lat: 48.527069 },
    { x: 6067, y: 354.5, lon: 9.023846, lat: 48.527281 },
    { x: 6087, y: 354.9, lon: 9.023775, lat: 48.527458 },
    { x: 6110, y: 355.2, lon: 9.023713, lat: 48.527661 },
    { x: 6135, y: 355.4, lon: 9.023667, lat: 48.52788 },
    { x: 6157, y: 355.5, lon: 9.023615, lat: 48.528077 },
    { x: 6181, y: 355.5, lon: 9.02357, lat: 48.528289 },
    { x: 6205, y: 355.7, lon: 9.023493, lat: 48.5285 },
    { x: 6229, y: 356, lon: 9.023417, lat: 48.528712 },
    { x: 6251, y: 357, lon: 9.023306, lat: 48.528898 },
    { x: 6273, y: 358.1, lon: 9.02317, lat: 48.52907 },
    { x: 6295, y: 358.9, lon: 9.023075, lat: 48.529258 },
    { x: 6317, y: 359.6, lon: 9.022965, lat: 48.529444 },
    { x: 6339, y: 360.3, lon: 9.022862, lat: 48.529629 },
    { x: 6360, y: 360.8, lon: 9.022812, lat: 48.529817 },
    { x: 6379, y: 361.4, lon: 9.022746, lat: 48.529976 },
    { x: 6396, y: 361.5, lon: 9.022761, lat: 48.530133 },
    { x: 6420, y: 362.1, lon: 9.022711, lat: 48.530344 },
    { x: 6436, y: 362.3, lon: 9.022688, lat: 48.530488 },
    { x: 6461, y: 362.2, lon: 9.022667, lat: 48.530711 },
    { x: 6485, y: 362.2, lon: 9.022596, lat: 48.530923 },
    { x: 6507, y: 362.2, lon: 9.022513, lat: 48.531116 },
    { x: 6528, y: 362.4, lon: 9.022411, lat: 48.531291 },
    { x: 6547, y: 363.1, lon: 9.022328, lat: 48.531448 },
    { x: 6564, y: 363.5, lon: 9.022267, lat: 48.531598 },
    { x: 6579, y: 363.8, lon: 9.022211, lat: 48.531725 },
    { x: 6595, y: 363.9, lon: 9.022153, lat: 48.531865 },
    { x: 6614, y: 364, lon: 9.022094, lat: 48.532031 },
    { x: 6631, y: 364.5, lon: 9.022021, lat: 48.532182 },
    { x: 6652, y: 365.3, lon: 9.021975, lat: 48.532362 },
    { x: 6673, y: 366.3, lon: 9.021969, lat: 48.532555 },
    { x: 6698, y: 367.3, lon: 9.021964, lat: 48.532776 },
    { x: 6721, y: 368.5, lon: 9.021904, lat: 48.532984 },
    { x: 6743, y: 369.7, lon: 9.021867, lat: 48.533175 },
    { x: 6767, y: 371, lon: 9.021788, lat: 48.533389 },
    { x: 6793, y: 372.3, lon: 9.021703, lat: 48.533617 },
    { x: 6818, y: 373.5, lon: 9.021611, lat: 48.533833 },
    { x: 6841, y: 374.4, lon: 9.021512, lat: 48.534022 },
    { x: 6862, y: 374.9, lon: 9.021414, lat: 48.534204 },
    { x: 6882, y: 375.1, lon: 9.021334, lat: 48.534379 },
    { x: 6900, y: 375, lon: 9.021267, lat: 48.534531 },
    { x: 6915, y: 375.6, lon: 9.02126, lat: 48.534663 },
    { x: 6956, y: 376, lon: 9.020826, lat: 48.534895 },
    { x: 6979, y: 376.5, lon: 9.020694, lat: 48.535081 },
    { x: 6995, y: 376.9, lon: 9.020618, lat: 48.535216 },
    { x: 7013, y: 377.3, lon: 9.02056, lat: 48.535375 },
    { x: 7022, y: 377.6, lon: 9.020534, lat: 48.535459 },
    { x: 7038, y: 377.5, lon: 9.020396, lat: 48.535569 },
    { x: 7057, y: 378.4, lon: 9.020249, lat: 48.535704 },
    { x: 7080, y: 379.1, lon: 9.020122, lat: 48.53589 },
    { x: 7116, y: 380.1, lon: 9.019947, lat: 48.536197 },
    { x: 7136, y: 380.9, lon: 9.019856, lat: 48.53637 },
    { x: 7153, y: 381.5, lon: 9.019839, lat: 48.536515 },
    { x: 7169, y: 382, lon: 9.019717, lat: 48.536636 },
    { x: 7197, y: 382.8, lon: 9.019722, lat: 48.53689 },
    { x: 7229, y: 384.1, lon: 9.019829, lat: 48.537168 },
    { x: 7232, y: 384.1, lon: 9.019834, lat: 48.537194 },
    { x: 7245, y: 384.5, lon: 9.019875, lat: 48.537311 },
    { x: 7260, y: 385, lon: 9.019945, lat: 48.537439 },
    { x: 7277, y: 385.4, lon: 9.019995, lat: 48.537584 },
    { x: 7291, y: 385.7, lon: 9.02002, lat: 48.537713 },
    { x: 7299, y: 385.8, lon: 9.020001, lat: 48.537778 },
    { x: 7310, y: 385.8, lon: 9.019955, lat: 48.537875 },
    { x: 7326, y: 386.8, lon: 9.019962, lat: 48.53802 },
    { x: 7344, y: 386.7, lon: 9.019745, lat: 48.538098 },
    { x: 7393, y: 389.4, lon: 9.019363, lat: 48.538452 },
    { x: 7409, y: 390.1, lon: 9.01927, lat: 48.538585 },
    { x: 7420, y: 390.3, lon: 9.019236, lat: 48.538676 },
    { x: 7481, y: 392, lon: 9.018977, lat: 48.539197 },
    { x: 7516, y: 393.7, lon: 9.019066, lat: 48.539511 },
    { x: 7536, y: 394.2, lon: 9.019095, lat: 48.539692 },
    { x: 7561, y: 394.1, lon: 9.01912, lat: 48.539914 },
    { x: 7576, y: 393.8, lon: 9.019094, lat: 48.540049 },
    { x: 7599, y: 393.7, lon: 9.019119, lat: 48.540249 },
    {
      x: 7614,
      y: 393.5,
      lon: 9.019128,
      lat: 48.540385,
      name: 'Hagelloch',
      category: 'village'
    },
    { x: 7632, y: 394, lon: 9.019321, lat: 48.540488 },
    { x: 7651, y: 394.2, lon: 9.019576, lat: 48.540487 },
    { x: 7666, y: 399.1, lon: 9.019785, lat: 48.540502 },
    { x: 7674, y: 401.5, lon: 9.019885, lat: 48.540518 },
    { x: 7687, y: 404.6, lon: 9.020021, lat: 48.540588 },
    { x: 7692, y: 406.1, lon: 9.02009, lat: 48.540611 },
    { x: 7699, y: 407.7, lon: 9.020155, lat: 48.540576 },
    { x: 7707, y: 410.3, lon: 9.020263, lat: 48.540564 },
    { x: 7717, y: 413.3, lon: 9.020396, lat: 48.54058 },
    { x: 7720, y: 414.2, lon: 9.020441, lat: 48.54059 },
    { x: 7729, y: 415.6, lon: 9.020544, lat: 48.540635 },
    { x: 7745, y: 419.3, lon: 9.020755, lat: 48.540641 },
    { x: 7747, y: 419.7, lon: 9.020781, lat: 48.540653 },
    { x: 7753, y: 420.2, lon: 9.020829, lat: 48.540697 },
    { x: 7755, y: 420.4, lon: 9.020841, lat: 48.540712 },
    { x: 7761, y: 421.4, lon: 9.020912, lat: 48.540745 },
    { x: 7771, y: 423.7, lon: 9.021025, lat: 48.540703 },
    { x: 7783, y: 426.1, lon: 9.021139, lat: 48.540622 },
    { x: 7795, y: 428.4, lon: 9.021256, lat: 48.540548 },
    { x: 7804, y: 429.6, lon: 9.021361, lat: 48.540523 },
    { x: 7810, y: 430.3, lon: 9.021441, lat: 48.540535 },
    { x: 7825, y: 432.2, lon: 9.021585, lat: 48.540431 },
    { x: 7830, y: 432.7, lon: 9.021642, lat: 48.540438 },
    { x: 7836, y: 433.5, lon: 9.021729, lat: 48.540451 },
    { x: 7846, y: 434.9, lon: 9.021864, lat: 48.540431 },
    { x: 7861, y: 436.2, lon: 9.022035, lat: 48.540365 },
    { x: 7870, y: 436.4, lon: 9.022162, lat: 48.54035 },
    { x: 7876, y: 436.3, lon: 9.022235, lat: 48.540341 },
    { x: 7880, y: 436.2, lon: 9.02229, lat: 48.540334 },
    { x: 7893, y: 436.7, lon: 9.02238, lat: 48.540435 },
    { x: 7904, y: 436.5, lon: 9.022524, lat: 48.540425 },
    { x: 7913, y: 436.1, lon: 9.022575, lat: 48.540348 },
    { x: 7923, y: 436, lon: 9.022705, lat: 48.540352 },
    { x: 7932, y: 435.7, lon: 9.022805, lat: 48.5403 },
    { x: 7938, y: 435.5, lon: 9.022866, lat: 48.54027 },
    { x: 7952, y: 435.8, lon: 9.023053, lat: 48.540273 },
    { x: 7967, y: 436.3, lon: 9.023254, lat: 48.540258 },
    { x: 7975, y: 436.6, lon: 9.023373, lat: 48.540262 },
    { x: 7980, y: 436.8, lon: 9.023437, lat: 48.540261 },
    { x: 7984, y: 436.9, lon: 9.023486, lat: 48.540267 },
    { x: 7997, y: 437.1, lon: 9.023643, lat: 48.540317 },
    { x: 7999, y: 437.2, lon: 9.02367, lat: 48.540323 },
    { x: 8009, y: 437.4, lon: 9.023807, lat: 48.540335 },
    { x: 8019, y: 437.9, lon: 9.023925, lat: 48.540283 },
    { x: 8024, y: 438.1, lon: 9.023966, lat: 48.540251 },
    { x: 8030, y: 438.3, lon: 9.024038, lat: 48.540235 },
    { x: 8040, y: 438.6, lon: 9.02417, lat: 48.540248 },
    { x: 8060, y: 438.9, lon: 9.024419, lat: 48.540339 },
    { x: 8069, y: 439.1, lon: 9.02453, lat: 48.540357 },
    { x: 8083, y: 440.1, lon: 9.024723, lat: 48.540341 },
    { x: 8099, y: 441.2, lon: 9.024883, lat: 48.540247 },
    { x: 8112, y: 442, lon: 9.025013, lat: 48.540164 },
    { x: 8125, y: 442.6, lon: 9.025109, lat: 48.540075 },
    { x: 8130, y: 442.9, lon: 9.025169, lat: 48.54004 },
    { x: 8149, y: 443.7, lon: 9.025337, lat: 48.53991 },
    { x: 8152, y: 443.8, lon: 9.025367, lat: 48.539892 },
    { x: 8174, y: 443.9, lon: 9.025605, lat: 48.539788 },
    { x: 8189, y: 443.7, lon: 9.02579, lat: 48.539721 },
    { x: 8203, y: 443.5, lon: 9.025976, lat: 48.53969 },
    { x: 8222, y: 443.1, lon: 9.026187, lat: 48.5396 },
    { x: 8238, y: 442.9, lon: 9.026388, lat: 48.539543 },
    { x: 8262, y: 443, lon: 9.026691, lat: 48.539473 },
    { x: 8303, y: 443.9, lon: 9.027206, lat: 48.539318 },
    { x: 8327, y: 444.8, lon: 9.027523, lat: 48.539277 },
    { x: 8362, y: 446, lon: 9.027966, lat: 48.539172 },
    { x: 8367, y: 446.1, lon: 9.028024, lat: 48.539155 },
    { x: 8387, y: 446.4, lon: 9.028291, lat: 48.539093 },
    { x: 8411, y: 446.6, lon: 9.028595, lat: 48.539019 },
    { x: 8432, y: 447.2, lon: 9.028837, lat: 48.538934 },
    { x: 8448, y: 448.1, lon: 9.029047, lat: 48.538878 },
    { x: 8461, y: 448.8, lon: 9.029217, lat: 48.538866 },
    { x: 8472, y: 449.1, lon: 9.029326, lat: 48.538795 },
    { x: 8491, y: 449.4, lon: 9.029448, lat: 48.538651 },
    { x: 8514, y: 449.3, lon: 9.029566, lat: 48.538456 },
    { x: 8543, y: 448.3, lon: 9.029763, lat: 48.538225 },
    { x: 8578, y: 446.1, lon: 9.030056, lat: 48.537983 },
    { x: 8605, y: 444.1, lon: 9.030364, lat: 48.53785 },
    { x: 8639, y: 444.3, lon: 9.030807, lat: 48.537772 },
    { x: 8675, y: 445.2, lon: 9.031296, lat: 48.537781 },
    { x: 8714, y: 448.7, lon: 9.031808, lat: 48.537843 },
    { x: 8751, y: 450.4, lon: 9.0323, lat: 48.537909 },
    { x: 8805, y: 451.4, lon: 9.032976, lat: 48.538106 },
    { x: 8810, y: 451.5, lon: 9.033039, lat: 48.53812 },
    { x: 8823, y: 451.9, lon: 9.03318, lat: 48.538186 },
    { x: 8843, y: 452.4, lon: 9.033438, lat: 48.538232 },
    { x: 8857, y: 452.8, lon: 9.033634, lat: 48.538263 },
    { x: 8872, y: 453.3, lon: 9.033828, lat: 48.538266 },
    { x: 8923, y: 456, lon: 9.034514, lat: 48.538199 },
    { x: 8933, y: 455.8, lon: 9.034588, lat: 48.538273 },
    { x: 8958, y: 454.3, lon: 9.034672, lat: 48.538492 },
    { x: 8969, y: 453.6, lon: 9.034691, lat: 48.538593 },
    { x: 8972, y: 453.5, lon: 9.03473, lat: 48.538603 },
    { x: 8976, y: 453.3, lon: 9.034778, lat: 48.538617 },
    { x: 8996, y: 452.2, lon: 9.034898, lat: 48.538781 },
    { x: 9012, y: 452.1, lon: 9.034921, lat: 48.538918 },
    { x: 9015, y: 452, lon: 9.03493, lat: 48.538943 },
    { x: 9029, y: 452.1, lon: 9.034883, lat: 48.539069 },
    { x: 9050, y: 452.8, lon: 9.034736, lat: 48.539234 },
    { x: 9083, y: 454.6, lon: 9.034496, lat: 48.539479 },
    { x: 9090, y: 455.1, lon: 9.034459, lat: 48.539541 },
    { x: 9105, y: 456.8, lon: 9.034356, lat: 48.539655 },
    { x: 9115, y: 457.6, lon: 9.034347, lat: 48.53974 },
    { x: 9133, y: 458.1, lon: 9.034423, lat: 48.539901 },
    { x: 9146, y: 458.6, lon: 9.034459, lat: 48.540016 },
    { x: 9148, y: 458.7, lon: 9.034466, lat: 48.540031 },
    { x: 9149, y: 458.7, lon: 9.034463, lat: 48.540036 },
    { x: 9150, y: 458.8, lon: 9.034459, lat: 48.540042 },
    { x: 9157, y: 458.8, lon: 9.034501, lat: 48.540108 },
    { x: 9179, y: 457.8, lon: 9.034678, lat: 48.540261 },
    { x: 9253, y: 453.2, lon: 9.035556, lat: 48.540568 },
    { x: 9286, y: 452.8, lon: 9.036003, lat: 48.540581 },
    { x: 9309, y: 452.4, lon: 9.03631, lat: 48.540589 },
    { x: 9342, y: 451.6, lon: 9.036751, lat: 48.540581 },
    { x: 9371, y: 451.3, lon: 9.037147, lat: 48.540583 },
    { x: 9398, y: 452, lon: 9.03752, lat: 48.540605 },
    { x: 9427, y: 452.3, lon: 9.037904, lat: 48.5406 },
    { x: 9454, y: 453.3, lon: 9.038274, lat: 48.540623 },
    { x: 9479, y: 454.6, lon: 9.038601, lat: 48.540679 },
    { x: 9503, y: 455.9, lon: 9.038892, lat: 48.540771 },
    { x: 9531, y: 457.3, lon: 9.039225, lat: 48.540892 },
    { x: 9547, y: 458.2, lon: 9.039403, lat: 48.540973 },
    { x: 9570, y: 459.8, lon: 9.039625, lat: 48.541115 },
    { x: 9599, y: 460.5, lon: 9.039929, lat: 48.541286 },
    { x: 9624, y: 461.6, lon: 9.040142, lat: 48.541454 },
    { x: 9669, y: 461.8, lon: 9.040595, lat: 48.541728 },
    { x: 9685, y: 461.6, lon: 9.04077, lat: 48.541821 },
    { x: 9702, y: 461.4, lon: 9.040943, lat: 48.541919 },
    { x: 9743, y: 461.1, lon: 9.041363, lat: 48.542157 },
    { x: 9771, y: 460.5, lon: 9.04167, lat: 48.542302 },
    { x: 9814, y: 459.1, lon: 9.042156, lat: 48.542512 },
    { x: 9866, y: 457.9, lon: 9.042742, lat: 48.54278 },
    { x: 9889, y: 457.1, lon: 9.043005, lat: 48.542886 },
    { x: 9932, y: 457.3, lon: 9.043523, lat: 48.543059 },
    { x: 9959, y: 458.2, lon: 9.04385, lat: 48.54318 },
    { x: 10010, y: 459.9, lon: 9.044472, lat: 48.543371 },
    { x: 10041, y: 461.7, lon: 9.044837, lat: 48.543517 },
    { x: 10095, y: 463.9, lon: 9.045537, lat: 48.543657 },
    { x: 10116, y: 464.9, lon: 9.045807, lat: 48.543702 },
    { x: 10147, y: 466.1, lon: 9.046221, lat: 48.543764 },
    { x: 10171, y: 466.1, lon: 9.046541, lat: 48.543766 },
    { x: 10206, y: 465.4, lon: 9.047007, lat: 48.543703 },
    { x: 10235, y: 464.7, lon: 9.047382, lat: 48.543616 },
    { x: 10240, y: 464.6, lon: 9.047445, lat: 48.543603 },
    { x: 10260, y: 464.4, lon: 9.047704, lat: 48.543546 },
    { x: 10281, y: 463.8, lon: 9.04794, lat: 48.543443 },
    { x: 10315, y: 464.2, lon: 9.048299, lat: 48.543256 },
    { x: 10329, y: 464.6, lon: 9.048458, lat: 48.543188 },
    { x: 10352, y: 465.7, lon: 9.048741, lat: 48.543101 },
    {
      x: 10372,
      y: 466.4,
      lon: 9.048972,
      lat: 48.543004,
      name: 'Tübingen Wanne',
      category: 'distinct'
    },
    { x: 10392, y: 466.8, lon: 9.049172, lat: 48.542884 },
    { x: 10413, y: 468, lon: 9.049418, lat: 48.542792 },
    { x: 10435, y: 469.2, lon: 9.049697, lat: 48.542714 },
    { x: 10470, y: 470.1, lon: 9.050121, lat: 48.54257 },
    { x: 10474, y: 470.2, lon: 9.050166, lat: 48.542558 },
    { x: 10491, y: 470.9, lon: 9.050388, lat: 48.542503 },
    { x: 10511, y: 472.1, lon: 9.050635, lat: 48.542446 },
    { x: 10533, y: 473.7, lon: 9.050925, lat: 48.542391 },
    { x: 10553, y: 475.3, lon: 9.051194, lat: 48.542362 },
    { x: 10575, y: 476.1, lon: 9.051493, lat: 48.542335 },
    { x: 10599, y: 476.7, lon: 9.051813, lat: 48.542293 },
    { x: 10623, y: 477.3, lon: 9.052128, lat: 48.54227 },
    { x: 10642, y: 477.8, lon: 9.052393, lat: 48.542271 },
    { x: 10654, y: 478.1, lon: 9.052546, lat: 48.542274 },
    { x: 10658, y: 478, lon: 9.052484, lat: 48.542283 },
    { x: 10659, y: 478, lon: 9.052495, lat: 48.542286 },
    { x: 10660, y: 478, lon: 9.052503, lat: 48.542287 },
    { x: 10667, y: 478, lon: 9.0525, lat: 48.542347 },
    { x: 10735, y: 478, lon: 9.052485, lat: 48.542957 },
    { x: 10741, y: 478.1, lon: 9.05248, lat: 48.543011 },
    { x: 10787, y: 478.3, lon: 9.052441, lat: 48.543425 },
    { x: 10810, y: 478.5, lon: 9.052425, lat: 48.543632 },
    { x: 10833, y: 478.8, lon: 9.052418, lat: 48.543837 },
    { x: 10856, y: 479.5, lon: 9.052394, lat: 48.544044 },
    { x: 10883, y: 480.2, lon: 9.052383, lat: 48.544294 },
    { x: 10913, y: 480.9, lon: 9.052379, lat: 48.544563 },
    { x: 10935, y: 481.7, lon: 9.052367, lat: 48.544755 },
    { x: 10960, y: 482.6, lon: 9.052345, lat: 48.544986 },
    { x: 10995, y: 483.8, lon: 9.052367, lat: 48.545295 },
    { x: 11011, y: 484.1, lon: 9.052456, lat: 48.545428 },
    { x: 11044, y: 483.7, lon: 9.052849, lat: 48.545576 },
    { x: 11058, y: 483.3, lon: 9.053035, lat: 48.545601 },
    { x: 11105, y: 481.2, lon: 9.05366, lat: 48.545594 },
    { x: 11111, y: 480.8, lon: 9.053742, lat: 48.545583 },
    { x: 11138, y: 480.7, lon: 9.054075, lat: 48.545485 },
    { x: 11181, y: 480.8, lon: 9.054553, lat: 48.545256 },
    { x: 11226, y: 479.1, lon: 9.055142, lat: 48.545156 },
    { x: 11228, y: 479.1, lon: 9.055169, lat: 48.545158 },
    { x: 11236, y: 478.8, lon: 9.055272, lat: 48.545163 },
    { x: 11246, y: 478.5, lon: 9.055401, lat: 48.545193 },
    { x: 11272, y: 479.1, lon: 9.055485, lat: 48.545423 },
    { x: 11300, y: 479.9, lon: 9.055544, lat: 48.54567 },
    { x: 11348, y: 481, lon: 9.055693, lat: 48.54609 },
    { x: 11359, y: 481.1, lon: 9.055727, lat: 48.546185 },
    { x: 11395, y: 481.1, lon: 9.05585, lat: 48.5465 },
    { x: 11431, y: 481.3, lon: 9.055983, lat: 48.546811 },
    { x: 11451, y: 481.6, lon: 9.056028, lat: 48.546988 },
    { x: 11480, y: 481.2, lon: 9.05606, lat: 48.547253 },
    { x: 11504, y: 480.4, lon: 9.056038, lat: 48.547462 },
    { x: 11536, y: 479.3, lon: 9.056043, lat: 48.547757 },
    { x: 11558, y: 478.9, lon: 9.056041, lat: 48.547949 },
    { x: 11584, y: 480.2, lon: 9.056032, lat: 48.548182 },
    { x: 11608, y: 481.5, lon: 9.056035, lat: 48.548403 },
    { x: 11632, y: 482.7, lon: 9.056008, lat: 48.548619 },
    { x: 11656, y: 483.3, lon: 9.055999, lat: 48.548832 },
    { x: 11689, y: 483, lon: 9.056018, lat: 48.549129 },
    { x: 11708, y: 482.8, lon: 9.056046, lat: 48.5493 },
    { x: 11745, y: 482.3, lon: 9.056071, lat: 48.549628 },
    { x: 11785, y: 481.3, lon: 9.056128, lat: 48.549988 },
    { x: 11855, y: 479.6, lon: 9.056219, lat: 48.550618 },
    { x: 11873, y: 479.2, lon: 9.056243, lat: 48.550774 },
    { x: 11932, y: 478.1, lon: 9.056301, lat: 48.551307 },
    { x: 11964, y: 478.4, lon: 9.056329, lat: 48.551587 },
    { x: 11971, y: 478.5, lon: 9.056372, lat: 48.551652 },
    { x: 11975, y: 478.5, lon: 9.056339, lat: 48.551672 },
    { x: 11976, y: 478.5, lon: 9.056356, lat: 48.551675 },
    { x: 11978, y: 478.5, lon: 9.05635, lat: 48.551689 },
    { x: 12009, y: 478.9, lon: 9.056393, lat: 48.551974 },
    { x: 12045, y: 478.9, lon: 9.056414, lat: 48.552293 },
    { x: 12072, y: 478.8, lon: 9.0564, lat: 48.552538 },
    { x: 12175, y: 477.3, lon: 9.056053, lat: 48.553435 },
    { x: 12197, y: 476.7, lon: 9.055985, lat: 48.553622 },
    { x: 12242, y: 470.7, lon: 9.055828, lat: 48.55402 },
    { x: 12297, y: 460.7, lon: 9.055965, lat: 48.554499 },
    { x: 12339, y: 455.6, lon: 9.056022, lat: 48.55488 },
    { x: 12381, y: 452.1, lon: 9.056155, lat: 48.555246 },
    { x: 12446, y: 443.5, lon: 9.056559, lat: 48.555763 },
    { x: 12508, y: 440.9, lon: 9.057385, lat: 48.555873 },
    { x: 12535, y: 440.3, lon: 9.057747, lat: 48.555886 },
    { x: 12603, y: 439, lon: 9.058675, lat: 48.555874 },
    { x: 12706, y: 427.7, lon: 9.060022, lat: 48.5561 },
    { x: 12729, y: 424.7, lon: 9.060329, lat: 48.556145 },
    { x: 12790, y: 411.4, lon: 9.06107, lat: 48.556382 },
    { x: 12830, y: 406, lon: 9.060802, lat: 48.5567 },
    { x: 12856, y: 403.5, lon: 9.060604, lat: 48.55689 },
    { x: 12934, y: 386.5, lon: 9.06006, lat: 48.55749 },
    { x: 12996, y: 370.4, lon: 9.060205, lat: 48.55804 },
    { x: 13006, y: 369.4, lon: 9.060235, lat: 48.558129 },
    { x: 13064, y: 364.1, lon: 9.060383, lat: 48.558637 },
    { x: 13077, y: 362.9, lon: 9.060394, lat: 48.558755 },
    { x: 13103, y: 359.9, lon: 9.060445, lat: 48.558991 },
    {
      x: 13133,
      y: 356,
      lon: 9.06072,
      lat: 48.559192,
      name: 'Bebenhausen',
      category: 'village'
    },
    { x: 13140, y: 355.3, lon: 9.060759, lat: 48.559248 },
    { x: 13144, y: 355.5, lon: 9.060706, lat: 48.559258 },
    { x: 13157, y: 356.2, lon: 9.060744, lat: 48.559148 },
    { x: 13197, y: 354.4, lon: 9.061156, lat: 48.55892 },
    { x: 13220, y: 353.3, lon: 9.06141, lat: 48.558793 },
    { x: 13255, y: 354.4, lon: 9.061734, lat: 48.558561 },
    { x: 13272, y: 355.5, lon: 9.06184, lat: 48.558425 },
    { x: 13298, y: 357.1, lon: 9.062009, lat: 48.558218 },
    { x: 13346, y: 358.9, lon: 9.062466, lat: 48.55791 },
    { x: 13359, y: 359.8, lon: 9.062625, lat: 48.557867 },
    { x: 13382, y: 362.2, lon: 9.062895, lat: 48.557769 },
    { x: 13422, y: 362.3, lon: 9.063418, lat: 48.557659 },
    { x: 13438, y: 361.9, lon: 9.063615, lat: 48.557618 },
    { x: 13459, y: 362, lon: 9.063878, lat: 48.557538 },
    { x: 13475, y: 361.6, lon: 9.064082, lat: 48.557493 },
    { x: 13534, y: 361.1, lon: 9.064786, lat: 48.557236 },
    { x: 13561, y: 359.9, lon: 9.065121, lat: 48.557138 },
    { x: 13572, y: 360.2, lon: 9.065213, lat: 48.557067 },
    { x: 13609, y: 359.3, lon: 9.065592, lat: 48.556837 },
    { x: 13625, y: 358.7, lon: 9.065749, lat: 48.556742 },
    { x: 13653, y: 357, lon: 9.066034, lat: 48.556572 },
    { x: 13700, y: 355.9, lon: 9.066404, lat: 48.556228 },
    { x: 13720, y: 355.9, lon: 9.06655, lat: 48.556077 },
    { x: 13758, y: 355.8, lon: 9.066796, lat: 48.555785 },
    { x: 13796, y: 355.3, lon: 9.067028, lat: 48.555481 },
    { x: 13844, y: 356.2, lon: 9.067391, lat: 48.555116 },
    { x: 13864, y: 356.2, lon: 9.067531, lat: 48.554966 },
    { x: 13891, y: 355.9, lon: 9.067706, lat: 48.554751 },
    { x: 13927, y: 355.7, lon: 9.067926, lat: 48.554462 },
    { x: 13961, y: 357.2, lon: 9.068203, lat: 48.554213 },
    { x: 14009, y: 359.4, lon: 9.068501, lat: 48.55383 },
    { x: 14031, y: 360.9, lon: 9.068642, lat: 48.553664 },
    { x: 14071, y: 365.4, lon: 9.068929, lat: 48.55335 },
    { x: 14095, y: 367.7, lon: 9.069101, lat: 48.553168 },
    { x: 14128, y: 370.5, lon: 9.069321, lat: 48.552914 },
    { x: 14170, y: 370.2, lon: 9.069625, lat: 48.552596 },
    { x: 14189, y: 369.5, lon: 9.06976, lat: 48.55245 },
    { x: 14218, y: 367.4, lon: 9.069958, lat: 48.55222 },
    { x: 14259, y: 365.6, lon: 9.070227, lat: 48.551897 },
    { x: 14296, y: 366.1, lon: 9.070476, lat: 48.551611 },
    { x: 14323, y: 365.3, lon: 9.070684, lat: 48.551408 },
    { x: 14360, y: 362.4, lon: 9.070976, lat: 48.551138 },
    { x: 14404, y: 361.8, lon: 9.071322, lat: 48.550818 },
    { x: 14428, y: 361.7, lon: 9.07153, lat: 48.550648 },
    { x: 14477, y: 359.7, lon: 9.071915, lat: 48.550299 },
    { x: 14505, y: 359.9, lon: 9.072113, lat: 48.55008 },
    { x: 14533, y: 359.6, lon: 9.072324, lat: 48.54987 },
    { x: 14568, y: 357.5, lon: 9.07261, lat: 48.54962 },
    { x: 14601, y: 360, lon: 9.072859, lat: 48.549369 },
    { x: 14640, y: 363.9, lon: 9.073172, lat: 48.549085 },
    { x: 14676, y: 365.3, lon: 9.073473, lat: 48.54883 },
    { x: 14691, y: 365.7, lon: 9.073572, lat: 48.548717 },
    { x: 14702, y: 364.3, lon: 9.073671, lat: 48.548641 },
    { x: 14724, y: 358.1, lon: 9.073914, lat: 48.548747 },
    { x: 14756, y: 350.7, lon: 9.074043, lat: 48.549022 },
    { x: 14760, y: 349.8, lon: 9.074063, lat: 48.549062 },
    { x: 14779, y: 346.3, lon: 9.074147, lat: 48.549219 },
    { x: 14801, y: 343.3, lon: 9.07435, lat: 48.54936 },
    {
      x: 14829,
      y: 341.2,
      lon: 9.074589,
      lat: 48.549555,
      name: 'Fire Place',
      category: 'attraction'
    },
    { x: 14845, y: 343, lon: 9.074704, lat: 48.54968 },
    { x: 14859, y: 345, lon: 9.074832, lat: 48.549772 },
    { x: 14877, y: 347.6, lon: 9.074963, lat: 48.549908 },
    { x: 14881, y: 348.2, lon: 9.075001, lat: 48.549935 },
    { x: 14887, y: 349.2, lon: 9.075038, lat: 48.549987 },
    { x: 14904, y: 351.6, lon: 9.07513, lat: 48.550122 },
    { x: 14918, y: 353.7, lon: 9.075193, lat: 48.550243 },
    { x: 14952, y: 358.8, lon: 9.07545, lat: 48.550502 },
    { x: 14966, y: 360.6, lon: 9.075558, lat: 48.550605 },
    { x: 14997, y: 364.3, lon: 9.075757, lat: 48.550843 },
    { x: 15023, y: 366.5, lon: 9.075948, lat: 48.55104 },
    { x: 15031, y: 367.1, lon: 9.076005, lat: 48.551105 },
    { x: 15047, y: 367.9, lon: 9.076122, lat: 48.551228 },
    { x: 15066, y: 369.1, lon: 9.076287, lat: 48.55136 },
    { x: 15081, y: 369.8, lon: 9.076429, lat: 48.551456 },
    { x: 15120, y: 366.9, lon: 9.076856, lat: 48.551659 },
    { x: 15130, y: 365.5, lon: 9.076955, lat: 48.551713 },
    { x: 15147, y: 363.6, lon: 9.077118, lat: 48.551823 },
    { x: 15158, y: 363.8, lon: 9.077181, lat: 48.551911 },
    { x: 15177, y: 363.7, lon: 9.077319, lat: 48.552056 },
    { x: 15193, y: 365.2, lon: 9.077469, lat: 48.552162 },
    { x: 15234, y: 366.6, lon: 9.077951, lat: 48.552351 },
    { x: 15239, y: 367.3, lon: 9.078017, lat: 48.552372 },
    { x: 15269, y: 370.1, lon: 9.078395, lat: 48.552468 },
    { x: 15271, y: 370.2, lon: 9.078409, lat: 48.552463 },
    { x: 15282, y: 370.7, lon: 9.078542, lat: 48.552508 },
    { x: 15325, y: 370.4, lon: 9.079032, lat: 48.552732 },
    { x: 15343, y: 368.6, lon: 9.07919, lat: 48.552854 },
    { x: 15393, y: 371, lon: 9.079598, lat: 48.553213 },
    { x: 15399, y: 371.7, lon: 9.079641, lat: 48.553252 },
    { x: 15406, y: 372.5, lon: 9.079697, lat: 48.553307 },
    { x: 15422, y: 373.8, lon: 9.079824, lat: 48.553419 },
    { x: 15453, y: 374.7, lon: 9.080149, lat: 48.553593 },
    { x: 15473, y: 373.6, lon: 9.080369, lat: 48.553709 },
    { x: 15481, y: 373.4, lon: 9.08043, lat: 48.553767 },
    { x: 15504, y: 377, lon: 9.080545, lat: 48.553962 },
    { x: 15515, y: 378.5, lon: 9.080626, lat: 48.554041 },
    { x: 15569, y: 381.7, lon: 9.081173, lat: 48.554366 },
    { x: 15585, y: 381, lon: 9.081275, lat: 48.554491 },
    { x: 15604, y: 381.9, lon: 9.081442, lat: 48.554618 },
    { x: 15614, y: 383.3, lon: 9.08154, lat: 48.554676 },
    { x: 15628, y: 384.9, lon: 9.081658, lat: 48.554782 },
    { x: 15646, y: 385.6, lon: 9.081879, lat: 48.554853 },
    { x: 15671, y: 386.6, lon: 9.082204, lat: 48.554915 },
    { x: 15688, y: 387.7, lon: 9.082401, lat: 48.554984 },
    { x: 15716, y: 389.4, lon: 9.082769, lat: 48.555047 },
    { x: 15721, y: 389.2, lon: 9.082819, lat: 48.55508 },
    { x: 15761, y: 386.6, lon: 9.083163, lat: 48.555356 },
    { x: 15782, y: 387.7, lon: 9.083373, lat: 48.555492 },
    { x: 15803, y: 389.8, lon: 9.083577, lat: 48.555615 },
    { x: 15812, y: 390.5, lon: 9.083669, lat: 48.555676 },
    { x: 15858, y: 394.1, lon: 9.084116, lat: 48.55596 },
    { x: 15868, y: 394.3, lon: 9.084187, lat: 48.556033 },
    { x: 15887, y: 394.4, lon: 9.084351, lat: 48.556164 },
    { x: 15910, y: 394.6, lon: 9.084536, lat: 48.556331 },
    { x: 15914, y: 394.9, lon: 9.084568, lat: 48.556367 },
    { x: 15936, y: 395.6, lon: 9.08476, lat: 48.556516 },
    { x: 15948, y: 395.6, lon: 9.084898, lat: 48.556565 },
    { x: 16005, y: 394.8, lon: 9.085569, lat: 48.556828 },
    { x: 16015, y: 394.9, lon: 9.085662, lat: 48.556887 },
    { x: 16034, y: 394.7, lon: 9.085837, lat: 48.55701 },
    { x: 16049, y: 394.3, lon: 9.086014, lat: 48.557087 },
    { x: 16067, y: 394, lon: 9.086255, lat: 48.557088 },
    { x: 16069, y: 394.2, lon: 9.086287, lat: 48.557087 },
    { x: 16111, y: 394.9, lon: 9.08675, lat: 48.557305 },
    { x: 16144, y: 395.3, lon: 9.087158, lat: 48.55741 },
    { x: 16174, y: 393.6, lon: 9.087373, lat: 48.55764 },
    { x: 16181, y: 393.2, lon: 9.087431, lat: 48.557696 },
    { x: 16206, y: 392, lon: 9.08765, lat: 48.557869 },
    { x: 16233, y: 392.6, lon: 9.08787, lat: 48.558056 },
    { x: 16257, y: 393.1, lon: 9.088094, lat: 48.558221 },
    { x: 16281, y: 393.2, lon: 9.088312, lat: 48.558381 },
    { x: 16293, y: 393.2, lon: 9.088417, lat: 48.558453 },
    { x: 16303, y: 393.1, lon: 9.088526, lat: 48.558519 },
    { x: 16321, y: 393.1, lon: 9.088657, lat: 48.558648 },
    { x: 16330, y: 393.1, lon: 9.088705, lat: 48.558722 },
    { x: 16357, y: 394.4, lon: 9.088897, lat: 48.558935 },
    { x: 16385, y: 395.5, lon: 9.089104, lat: 48.559138 },
    { x: 16395, y: 395.7, lon: 9.089178, lat: 48.559217 },
    { x: 16415, y: 395.4, lon: 9.089421, lat: 48.559291 },
    { x: 16434, y: 394.9, lon: 9.089654, lat: 48.559375 },
    { x: 16450, y: 394.6, lon: 9.089859, lat: 48.559422 },
    { x: 16521, y: 394.3, lon: 9.090648, lat: 48.559789 },
    { x: 16536, y: 395.1, lon: 9.090801, lat: 48.559871 },
    { x: 16545, y: 395.4, lon: 9.090907, lat: 48.559918 },
    { x: 16580, y: 395.1, lon: 9.091307, lat: 48.560088 },
    {
      x: 16612,
      y: 392.9,
      lon: 9.091633,
      lat: 48.56028,
      name: 'Picnic Area',
      category: 'picnic'
    },
    { x: 16628, y: 391.1, lon: 9.091774, lat: 48.560378 },
    { x: 16631, y: 390.6, lon: 9.091802, lat: 48.560404 },
    { x: 16666, y: 391.5, lon: 9.092071, lat: 48.560664 },
    { x: 16684, y: 393.1, lon: 9.092206, lat: 48.560796 },
    { x: 16697, y: 393.9, lon: 9.092299, lat: 48.560901 },
    { x: 16705, y: 394.2, lon: 9.092339, lat: 48.560959 },
    { x: 16715, y: 394.6, lon: 9.092428, lat: 48.561028 },
    { x: 16722, y: 394.7, lon: 9.092497, lat: 48.561072 },
    { x: 16730, y: 394.7, lon: 9.092572, lat: 48.561126 },
    { x: 16731, y: 394.7, lon: 9.092569, lat: 48.561134 },
    { x: 16731, y: 394.7, lon: 9.092569, lat: 48.561135 },
    { x: 16731, y: 394.7, lon: 9.09257, lat: 48.561135 },
    { x: 16735, y: 394.7, lon: 9.092614, lat: 48.561156 },
    { x: 16748, y: 394.3, lon: 9.092745, lat: 48.561234 },
    { x: 16770, y: 395.9, lon: 9.092935, lat: 48.561384 },
    { x: 16812, y: 399.8, lon: 9.093237, lat: 48.561708 },
    { x: 16821, y: 400.2, lon: 9.093266, lat: 48.561781 },
    { x: 16857, y: 402.1, lon: 9.093267, lat: 48.562107 },
    { x: 16886, y: 403.5, lon: 9.093208, lat: 48.56237 },
    { x: 16890, y: 403.6, lon: 9.093208, lat: 48.562403 },
    { x: 16920, y: 404.2, lon: 9.093236, lat: 48.562676 },
    { x: 16935, y: 404.8, lon: 9.09321, lat: 48.562808 },
    { x: 16975, y: 405.1, lon: 9.093237, lat: 48.563161 },
    { x: 16989, y: 404.9, lon: 9.093271, lat: 48.563286 },
    { x: 17024, y: 405.5, lon: 9.093227, lat: 48.563603 },
    { x: 17028, y: 405.5, lon: 9.093236, lat: 48.563636 },
    { x: 17055, y: 405.8, lon: 9.093251, lat: 48.563883 },
    { x: 17068, y: 406.2, lon: 9.093254, lat: 48.564 },
    { x: 17100, y: 407.3, lon: 9.093243, lat: 48.564282 },
    { x: 17111, y: 408.2, lon: 9.0932, lat: 48.564375 },
    { x: 17131, y: 408.8, lon: 9.093202, lat: 48.564562 },
    { x: 17147, y: 408.7, lon: 9.093204, lat: 48.5647 },
    { x: 17169, y: 408.3, lon: 9.09322, lat: 48.5649 },
    { x: 17206, y: 407.7, lon: 9.093236, lat: 48.565234 },
    { x: 17231, y: 407.3, lon: 9.09328, lat: 48.565452 },
    { x: 17261, y: 409, lon: 9.093253, lat: 48.565722 },
    { x: 17274, y: 409.3, lon: 9.093284, lat: 48.565836 },
    { x: 17287, y: 409.4, lon: 9.093329, lat: 48.565949 },
    { x: 17312, y: 409.2, lon: 9.093467, lat: 48.566161 },
    { x: 17333, y: 409.3, lon: 9.093596, lat: 48.566324 },
    { x: 17362, y: 409.9, lon: 9.093841, lat: 48.566537 },
    { x: 17381, y: 410.4, lon: 9.093994, lat: 48.566664 },
    { x: 17399, y: 410.8, lon: 9.09413, lat: 48.566808 },
    { x: 17412, y: 411.6, lon: 9.094159, lat: 48.566918 },
    { x: 17445, y: 411.4, lon: 9.094433, lat: 48.567149 },
    { x: 17459, y: 411.6, lon: 9.094555, lat: 48.567248 },
    { x: 17486, y: 413.8, lon: 9.094729, lat: 48.567467 },
    { x: 17504, y: 414.7, lon: 9.094854, lat: 48.5676 },
    { x: 17527, y: 414.8, lon: 9.095071, lat: 48.567757 },
    { x: 17554, y: 415.7, lon: 9.095201, lat: 48.567982 },
    { x: 17574, y: 416, lon: 9.095338, lat: 48.568135 },
    { x: 17581, y: 416.2, lon: 9.095389, lat: 48.568191 },
    { x: 17605, y: 417.6, lon: 9.095549, lat: 48.568374 },
    { x: 17627, y: 418.9, lon: 9.095634, lat: 48.568569 },
    { x: 17648, y: 420.6, lon: 9.095608, lat: 48.568752 },
    { x: 17666, y: 420.3, lon: 9.095599, lat: 48.568913 },
    { x: 17701, y: 419.3, lon: 9.095631, lat: 48.569228 },
    { x: 17718, y: 419.6, lon: 9.095546, lat: 48.569374 },
    { x: 17731, y: 419.8, lon: 9.095484, lat: 48.569485 },
    { x: 17747, y: 419.6, lon: 9.095373, lat: 48.569609 },
    { x: 17762, y: 418.2, lon: 9.095267, lat: 48.569716 },
    { x: 17795, y: 415.2, lon: 9.095064, lat: 48.569981 },
    { x: 17810, y: 414, lon: 9.094983, lat: 48.570104 },
    { x: 17848, y: 410.7, lon: 9.094695, lat: 48.570395 },
    {
      x: 17873,
      y: 410,
      lon: 9.094568,
      lat: 48.570602,
      name: 'Picnic Area',
      category: 'picnic'
    },
    { x: 17885, y: 410.2, lon: 9.094497, lat: 48.570696 },
    { x: 17921, y: 410.4, lon: 9.094336, lat: 48.571005 },
    { x: 17943, y: 410.4, lon: 9.094275, lat: 48.571195 },
    { x: 17964, y: 411.6, lon: 9.094176, lat: 48.571371 },
    { x: 17988, y: 413.8, lon: 9.09413, lat: 48.571585 },
    { x: 17996, y: 414.7, lon: 9.094106, lat: 48.57166 },
    { x: 18014, y: 416.4, lon: 9.09404, lat: 48.57181 },
    { x: 18040, y: 419.2, lon: 9.093988, lat: 48.572046 },
    { x: 18050, y: 419.6, lon: 9.093991, lat: 48.57213 },
    { x: 18078, y: 419.8, lon: 9.093949, lat: 48.572388 },
    { x: 18096, y: 420, lon: 9.093877, lat: 48.572538 },
    { x: 18110, y: 420.1, lon: 9.09383, lat: 48.572663 },
    { x: 18131, y: 420, lon: 9.093775, lat: 48.572847 },
    { x: 18151, y: 420.7, lon: 9.093731, lat: 48.573019 },
    { x: 18186, y: 423, lon: 9.093628, lat: 48.573332 },
    { x: 18202, y: 423.7, lon: 9.093594, lat: 48.573469 },
    { x: 18217, y: 424.6, lon: 9.093529, lat: 48.573598 },
    { x: 18227, y: 425.3, lon: 9.093466, lat: 48.573681 },
    { x: 18257, y: 426.7, lon: 9.093449, lat: 48.573946 },
    { x: 18277, y: 427.9, lon: 9.093427, lat: 48.574132 },
    { x: 18323, y: 429.8, lon: 9.093518, lat: 48.574535 },
    { x: 18331, y: 430.3, lon: 9.093495, lat: 48.57461 },
    { x: 18352, y: 431.3, lon: 9.093445, lat: 48.574796 },
    { x: 18368, y: 431.9, lon: 9.093434, lat: 48.574942 },
    { x: 18393, y: 432.7, lon: 9.093452, lat: 48.575162 },
    { x: 18412, y: 433.4, lon: 9.09344, lat: 48.575329 },
    { x: 18439, y: 433.3, lon: 9.093426, lat: 48.57558 },
    { x: 18464, y: 433.6, lon: 9.093297, lat: 48.575788 },
    { x: 18469, y: 433.6, lon: 9.093278, lat: 48.575829 },
    { x: 18519, y: 433.9, lon: 9.093194, lat: 48.57627 },
    { x: 18524, y: 433.9, lon: 9.093189, lat: 48.576317 },
    { x: 18548, y: 433.3, lon: 9.093182, lat: 48.576538 },
    { x: 18577, y: 432.7, lon: 9.09317, lat: 48.576791 },
    { x: 18588, y: 432.2, lon: 9.093195, lat: 48.576888 },
    { x: 18632, y: 432.7, lon: 9.093265, lat: 48.577284 },
    { x: 18636, y: 432.9, lon: 9.093267, lat: 48.577324 },
    { x: 18655, y: 433.9, lon: 9.093293, lat: 48.577488 },
    { x: 18677, y: 435.4, lon: 9.093262, lat: 48.577687 },
    { x: 18709, y: 437.7, lon: 9.0932, lat: 48.577975 },
    { x: 18720, y: 438.3, lon: 9.093187, lat: 48.578075 },
    { x: 18739, y: 439.4, lon: 9.093167, lat: 48.57824 },
    { x: 18773, y: 441.1, lon: 9.093186, lat: 48.578545 },
    { x: 18795, y: 442.1, lon: 9.093257, lat: 48.578738 },
    { x: 18804, y: 442.7, lon: 9.093294, lat: 48.578818 },
    { x: 18827, y: 444, lon: 9.093407, lat: 48.579007 },
    { x: 18860, y: 444.8, lon: 9.093641, lat: 48.579267 },
    { x: 18873, y: 445.1, lon: 9.093698, lat: 48.579377 },
    { x: 18905, y: 446.1, lon: 9.093897, lat: 48.579624 },
    { x: 18914, y: 446.7, lon: 9.093922, lat: 48.579706 },
    { x: 18929, y: 447.8, lon: 9.093916, lat: 48.579842 },
    { x: 18959, y: 448.2, lon: 9.09412, lat: 48.580071 },
    { x: 18962, y: 448.1, lon: 9.094144, lat: 48.580097 },
    { x: 18996, y: 445.9, lon: 9.094491, lat: 48.580303 },
    { x: 19011, y: 445.6, lon: 9.094534, lat: 48.580431 },
    { x: 19031, y: 446, lon: 9.094639, lat: 48.580599 },
    { x: 19045, y: 446.7, lon: 9.094691, lat: 48.580716 },
    { x: 19060, y: 447.2, lon: 9.094644, lat: 48.580846 },
    { x: 19074, y: 447.8, lon: 9.094635, lat: 48.580978 },
    { x: 19105, y: 449.4, lon: 9.094507, lat: 48.581245 },
    { x: 19112, y: 449.7, lon: 9.094472, lat: 48.581296 },
    { x: 19133, y: 450.9, lon: 9.094299, lat: 48.581446 },
    { x: 19154, y: 451.6, lon: 9.094144, lat: 48.581606 },
    { x: 19179, y: 451.6, lon: 9.094122, lat: 48.581831 },
    { x: 19201, y: 451.5, lon: 9.094155, lat: 48.582025 },
    { x: 19251, y: 452.2, lon: 9.094167, lat: 48.582478 },
    { x: 19268, y: 452.5, lon: 9.094133, lat: 48.582626 },
    { x: 19284, y: 452.8, lon: 9.094108, lat: 48.582775 },
    { x: 19317, y: 453.7, lon: 9.0941, lat: 48.583067 },
    { x: 19340, y: 454.4, lon: 9.094207, lat: 48.583264 },
    { x: 19374, y: 455.1, lon: 9.094366, lat: 48.583547 },
    { x: 19379, y: 455.2, lon: 9.094382, lat: 48.58359 },
    { x: 19405, y: 455.7, lon: 9.094448, lat: 48.583828 },
    { x: 19429, y: 456.2, lon: 9.09446, lat: 48.58404 },
    { x: 19437, y: 456.1, lon: 9.094486, lat: 48.584113 },
    { x: 19470, y: 456.2, lon: 9.094539, lat: 48.584402 },
    { x: 19474, y: 456.3, lon: 9.09454, lat: 48.584443 },
    { x: 19498, y: 457.4, lon: 9.094527, lat: 48.584655 },
    { x: 19521, y: 458.8, lon: 9.094566, lat: 48.584857 },
    { x: 19551, y: 461.6, lon: 9.094551, lat: 48.585134 },
    { x: 19563, y: 462.3, lon: 9.094575, lat: 48.585234 },
    { x: 19583, y: 463.5, lon: 9.094645, lat: 48.585408 },
    { x: 19598, y: 462.6, lon: 9.09472, lat: 48.585541 },
    { x: 19615, y: 461.4, lon: 9.09482, lat: 48.585674 },
    { x: 19630, y: 460.6, lon: 9.094871, lat: 48.585806 },
    { x: 19650, y: 459.5, lon: 9.094934, lat: 48.585981 },
    { x: 19687, y: 456.8, lon: 9.095264, lat: 48.586225 },
    { x: 19709, y: 456.9, lon: 9.095528, lat: 48.586323 },
    {
      x: 19716,
      y: 457.4,
      lon: 9.095602,
      lat: 48.58636,
      name: 'Beobachtungskanzel',
      category: 'attraction'
    },
    { x: 19726, y: 458.2, lon: 9.095714, lat: 48.586415 },
    { x: 19741, y: 459.1, lon: 9.095868, lat: 48.586496 },
    { x: 19775, y: 461.4, lon: 9.096279, lat: 48.586645 },
    { x: 19780, y: 462.1, lon: 9.096337, lat: 48.58667 },
    { x: 19802, y: 464.8, lon: 9.096575, lat: 48.586787 },
    { x: 19812, y: 465.5, lon: 9.096655, lat: 48.586856 },
    { x: 19832, y: 467.6, lon: 9.096907, lat: 48.586936 },
    { x: 19861, y: 470.7, lon: 9.097247, lat: 48.587054 },
    { x: 19875, y: 472.8, lon: 9.097411, lat: 48.587125 },
    { x: 19894, y: 475.5, lon: 9.097622, lat: 48.587215 },
    { x: 19922, y: 478.5, lon: 9.097954, lat: 48.587346 },
    { x: 19942, y: 480, lon: 9.098197, lat: 48.587432 },
    { x: 19959, y: 481, lon: 9.098401, lat: 48.587504 },
    { x: 19969, y: 481.9, lon: 9.098526, lat: 48.587522 },
    { x: 19991, y: 483.1, lon: 9.098792, lat: 48.587611 },
    { x: 19999, y: 483.4, lon: 9.098878, lat: 48.587658 },
    { x: 20012, y: 484, lon: 9.09901, lat: 48.58773 },
    { x: 20024, y: 484.4, lon: 9.099129, lat: 48.587806 },
    { x: 20057, y: 487, lon: 9.099493, lat: 48.587976 },
    { x: 20058, y: 487.1, lon: 9.099506, lat: 48.587973 },
    { x: 20070, y: 488.7, lon: 9.099674, lat: 48.587997 },
    { x: 20111, y: 492, lon: 9.100153, lat: 48.588177 },
    { x: 20117, y: 492.4, lon: 9.100231, lat: 48.588203 },
    { x: 20144, y: 494.4, lon: 9.100559, lat: 48.588292 },
    { x: 20151, y: 495.1, lon: 9.100655, lat: 48.588319 },
    { x: 20160, y: 495.7, lon: 9.10076, lat: 48.588359 },
    { x: 20194, y: 498.7, lon: 9.101187, lat: 48.588473 },
    { x: 20197, y: 499.1, lon: 9.101228, lat: 48.588477 },
    { x: 20211, y: 499.4, lon: 9.101389, lat: 48.588532 },
    { x: 20234, y: 499.4, lon: 9.101643, lat: 48.588654 },
    { x: 20242, y: 499.1, lon: 9.101542, lat: 48.588632 },
    { x: 20245, y: 498.8, lon: 9.101496, lat: 48.58863 },
    {
      x: 20270,
      y: 496.5,
      lon: 9.101368,
      lat: 48.588836,
      name: 'Bärlochklinge',
      category: 'attraction'
    },
    { x: 20288, y: 498, lon: 9.101605, lat: 48.588797 },
    { x: 20315, y: 500, lon: 9.101972, lat: 48.588801 },
    { x: 20326, y: 500.7, lon: 9.102119, lat: 48.588817 },
    { x: 20336, y: 501.8, lon: 9.102253, lat: 48.588787 },
    { x: 20348, y: 503, lon: 9.102375, lat: 48.588714 },
    { x: 20360, y: 504, lon: 9.102479, lat: 48.588641 },
    { x: 20387, y: 505.8, lon: 9.102849, lat: 48.588634 },
    { x: 20400, y: 506.5, lon: 9.103027, lat: 48.588658 },
    { x: 20430, y: 507.1, lon: 9.103344, lat: 48.588816 },
    {
      x: 20437,
      y: 507.6,
      lon: 9.103438,
      lat: 48.588794,
      name: 'Kirnbachfall',
      category: 'attraction'
    },
    { x: 20493, y: 507.8, lon: 9.104066, lat: 48.589079 },
    { x: 20518, y: 508.6, lon: 9.104394, lat: 48.589021 },
    { x: 20525, y: 508.8, lon: 9.104482, lat: 48.589016 },
    { x: 20554, y: 509.3, lon: 9.104873, lat: 48.588955 },
    { x: 20558, y: 509.4, lon: 9.104925, lat: 48.588954 },
    { x: 20576, y: 509.7, lon: 9.105136, lat: 48.589027 },
    { x: 20588, y: 509.8, lon: 9.105233, lat: 48.589113 },
    { x: 20605, y: 509.9, lon: 9.105374, lat: 48.589235 },
    { x: 20638, y: 510.5, lon: 9.105717, lat: 48.589429 },
    { x: 20650, y: 510.6, lon: 9.105853, lat: 48.589493 },
    { x: 20651, y: 510.7, lon: 9.105869, lat: 48.589494 },
    { x: 20652, y: 510.7, lon: 9.105872, lat: 48.589503 },
    { x: 20660, y: 510.7, lon: 9.105942, lat: 48.589551 },
    { x: 20665, y: 510.7, lon: 9.105952, lat: 48.589594 },
    { x: 20668, y: 510.8, lon: 9.105974, lat: 48.589619 },
    { x: 20694, y: 511.8, lon: 9.106122, lat: 48.589833 },
    { x: 20699, y: 512, lon: 9.106175, lat: 48.589864 },
    { x: 20725, y: 512, lon: 9.106461, lat: 48.589993 },
    { x: 20744, y: 512, lon: 9.106539, lat: 48.590159 },
    { x: 20752, y: 512, lon: 9.106556, lat: 48.590225 },
    { x: 20771, y: 511.9, lon: 9.106595, lat: 48.590394 },
    { x: 20788, y: 512.7, lon: 9.106617, lat: 48.590553 },
    { x: 20825, y: 514.4, lon: 9.106683, lat: 48.59088 },
    { x: 20839, y: 515.3, lon: 9.106681, lat: 48.591004 },
    { x: 20857, y: 516.5, lon: 9.106666, lat: 48.59117 },
    { x: 20877, y: 517.2, lon: 9.106636, lat: 48.59135 },
    { x: 20893, y: 517, lon: 9.106666, lat: 48.591491 },
    { x: 20909, y: 516.5, lon: 9.106752, lat: 48.591616 },
    { x: 20936, y: 515.2, lon: 9.10699, lat: 48.591809 },
    { x: 20940, y: 515, lon: 9.107036, lat: 48.591827 },
    { x: 20973, y: 515.7, lon: 9.107462, lat: 48.591896 },
    { x: 20998, y: 516.7, lon: 9.1078, lat: 48.591924 },
    { x: 21025, y: 517.4, lon: 9.108175, lat: 48.591938 },
    { x: 21066, y: 517.8, lon: 9.10872, lat: 48.591922 },
    { x: 21098, y: 515.6, lon: 9.109138, lat: 48.591824 },
    { x: 21112, y: 514.6, lon: 9.109313, lat: 48.591778 },
    { x: 21140, y: 513.3, lon: 9.10968, lat: 48.59171 },
    { x: 21182, y: 513.1, lon: 9.110211, lat: 48.591596 },
    { x: 21206, y: 513.1, lon: 9.110533, lat: 48.59153 },
    { x: 21220, y: 513.1, lon: 9.110709, lat: 48.591511 },
    { x: 21241, y: 513.6, lon: 9.110792, lat: 48.591695 },
    { x: 21254, y: 513.8, lon: 9.110858, lat: 48.591803 },
    { x: 21303, y: 514.6, lon: 9.111092, lat: 48.592215 },
    { x: 21308, y: 514.7, lon: 9.111136, lat: 48.592254 },
    { x: 21348, y: 515.7, lon: 9.111347, lat: 48.592585 },
    { x: 21368, y: 516.2, lon: 9.111467, lat: 48.592741 },
    { x: 21388, y: 516.6, lon: 9.111553, lat: 48.592912 },
    { x: 21414, y: 515.8, lon: 9.11167, lat: 48.593136 },
    { x: 21443, y: 514.8, lon: 9.11198, lat: 48.593288 },
    { x: 21471, y: 515.2, lon: 9.112362, lat: 48.593317 },
    { x: 21487, y: 515.7, lon: 9.112574, lat: 48.593332 },
    { x: 21537, y: 517.4, lon: 9.11326, lat: 48.59331 },
    { x: 21552, y: 517.8, lon: 9.113451, lat: 48.593329 },
    { x: 21592, y: 519.6, lon: 9.114005, lat: 48.593321 },
    { x: 21611, y: 520.7, lon: 9.114262, lat: 48.59334 },
    { x: 21662, y: 522.6, lon: 9.114946, lat: 48.593407 },
    { x: 21695, y: 523.2, lon: 9.115393, lat: 48.593413 },
    { x: 21712, y: 522.8, lon: 9.115614, lat: 48.593431 },
    { x: 21759, y: 521.2, lon: 9.116243, lat: 48.593502 },
    { x: 21782, y: 520.2, lon: 9.11656, lat: 48.59352 },
    { x: 21800, y: 519.4, lon: 9.116794, lat: 48.593522 },
    { x: 21831, y: 517.6, lon: 9.117204, lat: 48.593438 },
    { x: 21858, y: 514.1, lon: 9.117561, lat: 48.593468 },
    { x: 21920, y: 508.5, lon: 9.118404, lat: 48.593526 },
    { x: 21935, y: 507.6, lon: 9.118594, lat: 48.593547 },
    { x: 21958, y: 507.2, lon: 9.118912, lat: 48.593552 },
    { x: 22008, y: 508.4, lon: 9.119586, lat: 48.593586 },
    { x: 22029, y: 509.9, lon: 9.119868, lat: 48.593562 },
    { x: 22065, y: 512.4, lon: 9.120356, lat: 48.593533 },
    { x: 22100, y: 512.9, lon: 9.120825, lat: 48.593499 },
    { x: 22132, y: 513.4, lon: 9.121256, lat: 48.593457 },
    { x: 22182, y: 515.7, lon: 9.121929, lat: 48.593416 },
    { x: 22210, y: 515.7, lon: 9.122304, lat: 48.593411 },
    { x: 22251, y: 514.6, lon: 9.122866, lat: 48.593398 },
    { x: 22284, y: 513.9, lon: 9.12332, lat: 48.593392 },
    { x: 22308, y: 513.4, lon: 9.123632, lat: 48.593386 },
    { x: 22316, y: 513.2, lon: 9.123751, lat: 48.593378 },
    { x: 22333, y: 513.3, lon: 9.123938, lat: 48.593294 },
    { x: 22355, y: 513, lon: 9.124224, lat: 48.593234 },
    { x: 22386, y: 512.1, lon: 9.124642, lat: 48.593222 },
    { x: 22421, y: 511.7, lon: 9.125116, lat: 48.593228 },
    { x: 22448, y: 511.3, lon: 9.125481, lat: 48.59326 },
    { x: 22471, y: 512, lon: 9.125787, lat: 48.593248 },
    { x: 22488, y: 512.8, lon: 9.12601, lat: 48.593199 },
    { x: 22505, y: 513.4, lon: 9.126237, lat: 48.593179 },
    { x: 22522, y: 513, lon: 9.126448, lat: 48.593244 },
    { x: 22523, y: 513, lon: 9.126454, lat: 48.593246 },
    { x: 22527, y: 513, lon: 9.126507, lat: 48.593236 },
    { x: 22527, y: 513, lon: 9.126513, lat: 48.593235 },
    { x: 22527, y: 513, lon: 9.126514, lat: 48.593234 },
    { x: 22527, y: 513, lon: 9.126514, lat: 48.593234 },
    { x: 22527, y: 513, lon: 9.126515, lat: 48.593234 },
    { x: 22527, y: 513, lon: 9.126515, lat: 48.593234 },
    { x: 22527, y: 513, lon: 9.126516, lat: 48.593234 },
    { x: 22528, y: 513, lon: 9.126517, lat: 48.593233 },
    { x: 22528, y: 513, lon: 9.126517, lat: 48.593233 },
    { x: 22528, y: 513, lon: 9.126517, lat: 48.593233 },
    { x: 22529, y: 513, lon: 9.126543, lat: 48.59323 },
    { x: 22538, y: 513.3, lon: 9.126616, lat: 48.593176 },
    { x: 22548, y: 513.4, lon: 9.12674, lat: 48.593135 },
    { x: 22549, y: 513.4, lon: 9.126761, lat: 48.593132 },
    { x: 22551, y: 513.4, lon: 9.12674, lat: 48.59314 },
    { x: 22552, y: 513.4, lon: 9.12673, lat: 48.59314 },
    { x: 22552, y: 513.4, lon: 9.126729, lat: 48.59314 },
    { x: 22561, y: 513.4, lon: 9.12685, lat: 48.593127 },
    { x: 22564, y: 513.4, lon: 9.126892, lat: 48.59312 },
    { x: 22633, y: 510.6, lon: 9.127821, lat: 48.593094 },
    { x: 22642, y: 510.2, lon: 9.12794, lat: 48.593123 },
    { x: 22685, y: 509.3, lon: 9.128515, lat: 48.593105 },
    { x: 22702, y: 509, lon: 9.128752, lat: 48.59309 },
    { x: 22742, y: 508.9, lon: 9.129274, lat: 48.593012 },
    { x: 22787, y: 508.2, lon: 9.129815, lat: 48.592825 },
    { x: 22796, y: 507.8, lon: 9.129915, lat: 48.592776 },
    { x: 22830, y: 506.4, lon: 9.130364, lat: 48.592719 },
    { x: 22853, y: 505.8, lon: 9.130652, lat: 48.592624 },
    { x: 22864, y: 505.4, lon: 9.130712, lat: 48.592536 },
    { x: 22880, y: 505, lon: 9.130777, lat: 48.592396 },
    { x: 22893, y: 504.6, lon: 9.130856, lat: 48.592299 },
    { x: 22903, y: 504.7, lon: 9.130983, lat: 48.592338 },
    { x: 22907, y: 504.8, lon: 9.130997, lat: 48.592371 },
    { x: 22908, y: 504.8, lon: 9.13099, lat: 48.592385 },
    { x: 22922, y: 505.2, lon: 9.131, lat: 48.59251 },
    { x: 22926, y: 505.3, lon: 9.130993, lat: 48.592542 },
    { x: 22943, y: 505.7, lon: 9.130959, lat: 48.592698 },
    { x: 22990, y: 506.5, lon: 9.131011, lat: 48.593119 },
    { x: 23007, y: 506.5, lon: 9.131119, lat: 48.593249 },
    { x: 23009, y: 506.5, lon: 9.131148, lat: 48.593261 },
    { x: 23040, y: 507, lon: 9.131567, lat: 48.593269 },
    { x: 23051, y: 507.3, lon: 9.13171, lat: 48.593276 },
    { x: 23115, y: 507.8, lon: 9.132563, lat: 48.593396 },
    { x: 23154, y: 507.3, lon: 9.13306, lat: 48.593499 },
    { x: 23200, y: 507.8, lon: 9.133694, lat: 48.593503 },
    { x: 23204, y: 507.9, lon: 9.133749, lat: 48.5935 },
    { x: 23230, y: 507.4, lon: 9.134099, lat: 48.59349 },
    { x: 23255, y: 506.9, lon: 9.134429, lat: 48.593475 },
    { x: 23280, y: 506.6, lon: 9.134773, lat: 48.593455 },
    { x: 23309, y: 506.4, lon: 9.135163, lat: 48.593461 },
    { x: 23357, y: 507.3, lon: 9.135808, lat: 48.593515 },
    { x: 23385, y: 508.6, lon: 9.136175, lat: 48.593581 },
    { x: 23442, y: 509, lon: 9.136949, lat: 48.593626 },
    { x: 23448, y: 509.1, lon: 9.137018, lat: 48.593644 },
    { x: 23473, y: 507.8, lon: 9.137354, lat: 48.593619 },
    { x: 23498, y: 505.9, lon: 9.137687, lat: 48.593576 },
    { x: 23521, y: 504.4, lon: 9.13799, lat: 48.593533 },
    { x: 23544, y: 503.4, lon: 9.138292, lat: 48.593472 },
    { x: 23607, y: 502.2, lon: 9.139139, lat: 48.593423 },
    { x: 23628, y: 501.4, lon: 9.139421, lat: 48.593367 },
    { x: 23667, y: 499.5, lon: 9.13995, lat: 48.593338 },
    { x: 23671, y: 499.4, lon: 9.139993, lat: 48.593346 },
    { x: 23690, y: 499, lon: 9.140231, lat: 48.593413 },
    { x: 23718, y: 497.9, lon: 9.140586, lat: 48.593518 },
    { x: 23723, y: 497.7, lon: 9.140642, lat: 48.593534 },
    {
      x: 23734,
      y: 496.9,
      lon: 9.140792,
      lat: 48.593553,
      name: 'Drehscheibe',
      category: 'parking'
    },
    { x: 23761, y: 494.4, lon: 9.141148, lat: 48.593534 },
    { x: 23771, y: 493.6, lon: 9.141292, lat: 48.593531 },
    { x: 23791, y: 493, lon: 9.141565, lat: 48.593541 },
    { x: 23825, y: 492.5, lon: 9.142007, lat: 48.593597 },
    { x: 23914, y: 496.9, lon: 9.143129, lat: 48.593908 },
    { x: 23928, y: 496.6, lon: 9.143313, lat: 48.593928 },
    { x: 24001, y: 497.5, lon: 9.144243, lat: 48.594139 },
    { x: 24022, y: 497.7, lon: 9.144534, lat: 48.594174 },
    { x: 24072, y: 494.8, lon: 9.145194, lat: 48.594254 },
    { x: 24106, y: 492.9, lon: 9.14563, lat: 48.59436 },
    { x: 24130, y: 490.7, lon: 9.145947, lat: 48.594407 },
    { x: 24150, y: 488.7, lon: 9.146207, lat: 48.594451 },
    { x: 24171, y: 487.7, lon: 9.146483, lat: 48.594488 },
    { x: 24184, y: 487.1, lon: 9.146661, lat: 48.594501 },
    { x: 24232, y: 484.1, lon: 9.147142, lat: 48.594209 },
    { x: 24317, y: 478.5, lon: 9.147295, lat: 48.593448 },
    { x: 24326, y: 477.8, lon: 9.147309, lat: 48.593373 },
    { x: 24366, y: 474.8, lon: 9.147457, lat: 48.593021 },
    {
      x: 24388,
      y: 473.9,
      lon: 9.1477,
      lat: 48.592905,
      name: 'Steinbrennershaus',
      category: 'attraction'
    },
    { x: 24432, y: 473.2, lon: 9.148271, lat: 48.592818 },
    { x: 24468, y: 472.6, lon: 9.148751, lat: 48.592747 },
    { x: 24483, y: 471.7, lon: 9.148945, lat: 48.592716 },
    { x: 24516, y: 469.9, lon: 9.149391, lat: 48.592677 },
    { x: 24550, y: 468.8, lon: 9.149858, lat: 48.592671 },
    { x: 24585, y: 468.1, lon: 9.150326, lat: 48.592696 },
    { x: 24616, y: 468, lon: 9.15074, lat: 48.592732 },
    { x: 24642, y: 468.3, lon: 9.151089, lat: 48.592787 },
    { x: 24679, y: 468.6, lon: 9.151572, lat: 48.592853 },
    { x: 24701, y: 468.7, lon: 9.151877, lat: 48.592874 },
    { x: 24742, y: 468.7, lon: 9.152429, lat: 48.592878 },
    { x: 24784, y: 468.2, lon: 9.15299, lat: 48.592814 },
    { x: 24805, y: 467.5, lon: 9.153278, lat: 48.592785 },
    { x: 24850, y: 466.1, lon: 9.153882, lat: 48.592724 },
    { x: 24891, y: 464.6, lon: 9.154421, lat: 48.592668 },
    { x: 24956, y: 461.7, lon: 9.155306, lat: 48.592611 },
    { x: 24978, y: 460.2, lon: 9.1556, lat: 48.592597 },
    { x: 25035, y: 455.7, lon: 9.156375, lat: 48.592576 },
    { x: 25094, y: 453.2, lon: 9.157172, lat: 48.592571 },
    { x: 25148, y: 452.5, lon: 9.157907, lat: 48.592599 },
    { x: 25207, y: 450.8, lon: 9.158694, lat: 48.59269 },
    { x: 25278, y: 450.2, lon: 9.159554, lat: 48.592977 },
    { x: 25297, y: 450, lon: 9.159764, lat: 48.593068 },
    { x: 25329, y: 449.3, lon: 9.160181, lat: 48.593144 },
    { x: 25380, y: 446.7, lon: 9.160882, lat: 48.593153 },
    { x: 25427, y: 444, lon: 9.161499, lat: 48.593075 },
    { x: 25467, y: 442.2, lon: 9.161998, lat: 48.592923 },
    { x: 25528, y: 440.8, lon: 9.162719, lat: 48.592653 },
    { x: 25580, y: 439.8, lon: 9.163337, lat: 48.592443 },
    { x: 25639, y: 438.9, lon: 9.164043, lat: 48.592191 },
    { x: 25676, y: 437.6, lon: 9.164487, lat: 48.592038 },
    { x: 25719, y: 434, lon: 9.165006, lat: 48.591845 },
    { x: 25765, y: 430.5, lon: 9.165541, lat: 48.591638 },
    { x: 25824, y: 426.2, lon: 9.166223, lat: 48.591359 },
    { x: 25856, y: 423.3, lon: 9.166569, lat: 48.591185 },
    { x: 25888, y: 420.9, lon: 9.166857, lat: 48.590977 },
    { x: 25922, y: 418.7, lon: 9.166957, lat: 48.590675 },
    { x: 25970, y: 417.1, lon: 9.167492, lat: 48.590431 },
    { x: 25984, y: 417.1, lon: 9.167688, lat: 48.590432 },
    { x: 25986, y: 417.1, lon: 9.16771, lat: 48.590436 },
    { x: 26007, y: 417, lon: 9.167433, lat: 48.590425 },
    { x: 26029, y: 416.8, lon: 9.167159, lat: 48.590346 },
    { x: 26058, y: 416.2, lon: 9.167085, lat: 48.590083 },
    { x: 26107, y: 415.4, lon: 9.167016, lat: 48.589646 },
    { x: 26159, y: 415.9, lon: 9.166937, lat: 48.589181 },
    { x: 26184, y: 416.2, lon: 9.166908, lat: 48.588957 },
    { x: 26224, y: 417.3, lon: 9.166852, lat: 48.588602 },
    { x: 26298, y: 420.6, lon: 9.166764, lat: 48.587943 },
    { x: 26305, y: 421, lon: 9.166764, lat: 48.587878 },
    { x: 26345, y: 423.1, lon: 9.16671, lat: 48.587515 },
    { x: 26376, y: 424.7, lon: 9.166676, lat: 48.587238 },
    { x: 26394, y: 425.5, lon: 9.166667, lat: 48.587081 },
    { x: 26414, y: 426.4, lon: 9.166847, lat: 48.586948 },
    { x: 26428, y: 426.7, lon: 9.167034, lat: 48.586926 },
    { x: 26456, y: 426, lon: 9.167417, lat: 48.586939 },
    { x: 26477, y: 425, lon: 9.167697, lat: 48.586993 },
    { x: 26501, y: 425.7, lon: 9.167876, lat: 48.586813 },
    { x: 26530, y: 427.3, lon: 9.16784, lat: 48.586555 },
    { x: 26544, y: 428.1, lon: 9.167819, lat: 48.586427 },
    { x: 26555, y: 428.7, lon: 9.167805, lat: 48.586332 },
    { x: 26573, y: 429, lon: 9.16778, lat: 48.586171 },
    { x: 26591, y: 428.7, lon: 9.167756, lat: 48.58601 },
    { x: 26611, y: 428.4, lon: 9.167733, lat: 48.585831 },
    { x: 26633, y: 428, lon: 9.167755, lat: 48.585631 },
    { x: 26659, y: 427.3, lon: 9.167819, lat: 48.585401 },
    { x: 26688, y: 426.7, lon: 9.167891, lat: 48.585148 },
    { x: 26723, y: 426.4, lon: 9.167991, lat: 48.584841 },
    { x: 26746, y: 426.3, lon: 9.168043, lat: 48.584641 },
    { x: 26784, y: 427.1, lon: 9.16816, lat: 48.584306 },
    { x: 26819, y: 427.7, lon: 9.168274, lat: 48.583997 },
    { x: 26856, y: 428, lon: 9.168396, lat: 48.583677 },
    { x: 26887, y: 427.9, lon: 9.168499, lat: 48.583408 },
    { x: 26934, y: 427.9, lon: 9.168664, lat: 48.582993 },
    { x: 26967, y: 428.2, lon: 9.168778, lat: 48.582709 },
    { x: 26987, y: 428.2, lon: 9.168844, lat: 48.582537 },
    { x: 27026, y: 428.1, lon: 9.168977, lat: 48.582198 },
    { x: 27068, y: 426.4, lon: 9.169116, lat: 48.581826 },
    { x: 27090, y: 425.3, lon: 9.169189, lat: 48.58164 },
    { x: 27133, y: 423.4, lon: 9.169309, lat: 48.581258 },
    { x: 27162, y: 422.3, lon: 9.169409, lat: 48.581004 },
    { x: 27202, y: 420.9, lon: 9.16952, lat: 48.580654 },
    { x: 27265, y: 417.6, lon: 9.169616, lat: 48.580091 },
    { x: 27295, y: 415.8, lon: 9.169561, lat: 48.579829 },
    { x: 27349, y: 413.5, lon: 9.169439, lat: 48.579348 },
    { x: 27408, y: 411.7, lon: 9.169296, lat: 48.578827 },
    { x: 27495, y: 413.5, lon: 9.16921, lat: 48.578046 },
    { x: 27516, y: 413.1, lon: 9.169479, lat: 48.577969 },
    { x: 27550, y: 412.1, lon: 9.169933, lat: 48.577944 },
    { x: 27601, y: 410.8, lon: 9.170588, lat: 48.577804 },
    { x: 27672, y: 409.1, lon: 9.171495, lat: 48.577585 },
    { x: 27688, y: 409.1, lon: 9.171697, lat: 48.577522 },
    { x: 27739, y: 407.7, lon: 9.172377, lat: 48.57744 },
    { x: 27787, y: 405, lon: 9.173016, lat: 48.577486 },
    { x: 27817, y: 403.1, lon: 9.173417, lat: 48.577501 },
    { x: 27853, y: 401.3, lon: 9.173893, lat: 48.577583 },
    { x: 27903, y: 400.2, lon: 9.174456, lat: 48.577828 },
    { x: 27909, y: 400.1, lon: 9.174518, lat: 48.577868 },
    { x: 27939, y: 401, lon: 9.174829, lat: 48.578039 },
    { x: 27964, y: 402.2, lon: 9.175095, lat: 48.578189 },
    { x: 27981, y: 402.8, lon: 9.175285, lat: 48.578116 },
    { x: 27995, y: 402.9, lon: 9.17537, lat: 48.578 },
    { x: 28005, y: 403.1, lon: 9.175439, lat: 48.577927 },
    { x: 28011, y: 403.2, lon: 9.175494, lat: 48.577878 },
    { x: 28013, y: 403.2, lon: 9.175483, lat: 48.577886 },
    { x: 28022, y: 402.6, lon: 9.175472, lat: 48.5778 },
    { x: 28043, y: 401.7, lon: 9.175527, lat: 48.577615 },
    { x: 28127, y: 399.9, lon: 9.17627, lat: 48.577039 },
    { x: 28162, y: 399.8, lon: 9.176661, lat: 48.57687 },
    { x: 28177, y: 399.8, lon: 9.176856, lat: 48.576823 },
    { x: 28235, y: 400.1, lon: 9.177538, lat: 48.576567 },
    { x: 28263, y: 400.4, lon: 9.177865, lat: 48.576436 },
    { x: 28310, y: 400.7, lon: 9.178364, lat: 48.576173 },
    { x: 28323, y: 400.9, lon: 9.178507, lat: 48.576109 },
    { x: 28348, y: 401.1, lon: 9.17877, lat: 48.575973 },
    { x: 28384, y: 402.6, lon: 9.179207, lat: 48.57582 },
    { x: 28413, y: 404.4, lon: 9.179587, lat: 48.575743 },
    { x: 28429, y: 404.7, lon: 9.179766, lat: 48.575669 },
    { x: 28455, y: 405.6, lon: 9.180089, lat: 48.575575 },
    { x: 28492, y: 405.9, lon: 9.180477, lat: 48.575357 },
    {
      x: 28512,
      y: 405.9,
      lon: 9.180673,
      lat: 48.575245,
      name: 'Gniebel',
      category: 'village'
    },
    { x: 28529, y: 406, lon: 9.180853, lat: 48.575149 },
    { x: 28552, y: 406.2, lon: 9.181108, lat: 48.575027 },
    { x: 28569, y: 406.4, lon: 9.181313, lat: 48.574949 },
    { x: 28614, y: 403.8, lon: 9.181224, lat: 48.574551 },
    { x: 28651, y: 402.2, lon: 9.181121, lat: 48.574228 },
    { x: 28729, y: 398.6, lon: 9.180978, lat: 48.573527 },
    { x: 28752, y: 397.6, lon: 9.180963, lat: 48.573321 },
    { x: 28799, y: 395.2, lon: 9.180891, lat: 48.572905 },
    { x: 28862, y: 392.2, lon: 9.180826, lat: 48.57234 },
    { x: 28951, y: 388.8, lon: 9.180677, lat: 48.571546 },
    { x: 28991, y: 387.3, lon: 9.180458, lat: 48.571216 },
    { x: 29057, y: 386.9, lon: 9.179849, lat: 48.570783 },
    { x: 29141, y: 385.4, lon: 9.178974, lat: 48.570295 },
    { x: 29213, y: 385.4, lon: 9.178353, lat: 48.569793 },
    { x: 29290, y: 384, lon: 9.177736, lat: 48.569244 },
    { x: 29353, y: 381.1, lon: 9.177139, lat: 48.568836 },
    { x: 29463, y: 375.8, lon: 9.175976, lat: 48.568214 },
    { x: 29519, y: 373.4, lon: 9.175429, lat: 48.567862 },
    { x: 29572, y: 374, lon: 9.175034, lat: 48.567466 },
    { x: 29647, y: 372.4, lon: 9.174676, lat: 48.566838 },
    { x: 29691, y: 371.3, lon: 9.174586, lat: 48.566444 },
    { x: 29759, y: 369.9, lon: 9.174623, lat: 48.56583 },
    { x: 29824, y: 368.4, lon: 9.174565, lat: 48.565252 },
    { x: 29893, y: 369.2, lon: 9.174296, lat: 48.564655 },
    {
      x: 29929,
      y: 369.9,
      lon: 9.174264,
      lat: 48.564336,
      name: 'Rübgarten',
      category: 'village'
    },
    { x: 29985, y: 368.6, lon: 9.173992, lat: 48.563859 },
    { x: 30056, y: 367.1, lon: 9.17372, lat: 48.563245 },
    { x: 30093, y: 367.3, lon: 9.173611, lat: 48.562921 },
    { x: 30161, y: 367.7, lon: 9.173378, lat: 48.56233 },
    { x: 30220, y: 366.5, lon: 9.173197, lat: 48.561816 },
    { x: 30268, y: 365.1, lon: 9.173136, lat: 48.561383 },
    { x: 30339, y: 362.1, lon: 9.173079, lat: 48.560748 },
    { x: 30387, y: 359.9, lon: 9.173015, lat: 48.560317 },
    { x: 30474, y: 360.4, lon: 9.173069, lat: 48.559538 },
    { x: 30555, y: 362.8, lon: 9.172992, lat: 48.558807 },
    { x: 30567, y: 363.1, lon: 9.172979, lat: 48.558702 },
    { x: 30625, y: 364.4, lon: 9.1729, lat: 48.558183 },
    { x: 30685, y: 364.1, lon: 9.172809, lat: 48.557651 },
    { x: 30745, y: 361.8, lon: 9.172631, lat: 48.557122 },
    { x: 30849, y: 353.2, lon: 9.17218, lat: 48.556232 },
    { x: 30877, y: 350.2, lon: 9.172055, lat: 48.555998 },
    { x: 30950, y: 343, lon: 9.171836, lat: 48.555359 },
    { x: 31071, y: 334.9, lon: 9.171455, lat: 48.554296 },
    { x: 31103, y: 332.1, lon: 9.171296, lat: 48.554028 },
    { x: 31185, y: 332.4, lon: 9.171089, lat: 48.55331 },
    { x: 31276, y: 333.2, lon: 9.171856, lat: 48.552661 },
    { x: 31390, y: 325, lon: 9.173343, lat: 48.552407 },
    { x: 31422, y: 322.8, lon: 9.173766, lat: 48.552347 },
    { x: 31537, y: 321.9, lon: 9.175316, lat: 48.55224 },
    { x: 31587, y: 317.6, lon: 9.175974, lat: 48.552145 },
    { x: 31659, y: 308, lon: 9.176736, lat: 48.551739 },
    { x: 31692, y: 305.1, lon: 9.176726, lat: 48.551442 },
    { x: 31714, y: 305.5, lon: 9.176476, lat: 48.551335 },
    { x: 31740, y: 306.9, lon: 9.17617, lat: 48.551206 },
    { x: 31749, y: 308, lon: 9.176069, lat: 48.551161 },
    { x: 31760, y: 309.7, lon: 9.175931, lat: 48.55113 },
    { x: 31763, y: 310, lon: 9.175899, lat: 48.551124 },
    { x: 31766, y: 309.6, lon: 9.175905, lat: 48.551093 },
    { x: 31793, y: 310.4, lon: 9.175639, lat: 48.550923 },
    { x: 31817, y: 310.9, lon: 9.175362, lat: 48.550816 },
    { x: 31844, y: 309.8, lon: 9.175025, lat: 48.550711 },
    { x: 31886, y: 307.3, lon: 9.174523, lat: 48.550548 },
    { x: 31898, y: 306.5, lon: 9.174367, lat: 48.550499 },
    { x: 31935, y: 305.5, lon: 9.173917, lat: 48.550364 },
    { x: 31970, y: 307.5, lon: 9.173472, lat: 48.550259 },
    { x: 31993, y: 308.9, lon: 9.173168, lat: 48.550199 },
    { x: 32008, y: 309.4, lon: 9.172988, lat: 48.550148 },
    { x: 32047, y: 310.5, lon: 9.17251, lat: 48.549991 },
    { x: 32074, y: 310.1, lon: 9.172195, lat: 48.549873 },
    { x: 32090, y: 309.7, lon: 9.172006, lat: 48.549807 },
    { x: 32119, y: 309.6, lon: 9.171641, lat: 48.549708 },
    { x: 32144, y: 308.2, lon: 9.17134, lat: 48.549604 },
    { x: 32190, y: 309.5, lon: 9.17076, lat: 48.549442 },
    { x: 32202, y: 309.5, lon: 9.170615, lat: 48.549389 },
    { x: 32230, y: 309.4, lon: 9.170278, lat: 48.54928 },
    { x: 32280, y: 309.7, lon: 9.169644, lat: 48.549124 },
    { x: 32299, y: 309.4, lon: 9.169399, lat: 48.549058 },
    { x: 32333, y: 308.7, lon: 9.168979, lat: 48.548937 },
    { x: 32360, y: 308.4, lon: 9.168631, lat: 48.548857 },
    { x: 32389, y: 307.8, lon: 9.168266, lat: 48.548784 },
    { x: 32411, y: 307.5, lon: 9.167982, lat: 48.548709 },
    { x: 32452, y: 308.9, lon: 9.167464, lat: 48.548581 },
    { x: 32477, y: 309.6, lon: 9.167146, lat: 48.548509 },
    { x: 32515, y: 309.2, lon: 9.166661, lat: 48.548381 },
    { x: 32527, y: 308.9, lon: 9.166516, lat: 48.548335 },
    { x: 32551, y: 308.6, lon: 9.166213, lat: 48.548269 },
    { x: 32584, y: 307.9, lon: 9.165785, lat: 48.548175 },
    { x: 32610, y: 307.4, lon: 9.165449, lat: 48.548107 },
    { x: 32641, y: 306.9, lon: 9.165066, lat: 48.548001 },
    { x: 32671, y: 306.7, lon: 9.164682, lat: 48.547905 },
    { x: 32714, y: 309.1, lon: 9.164159, lat: 48.547738 },
    { x: 32739, y: 309.9, lon: 9.163844, lat: 48.547658 },
    { x: 32769, y: 309.5, lon: 9.163483, lat: 48.547533 },
    { x: 32799, y: 308.2, lon: 9.163109, lat: 48.547421 },
    { x: 32829, y: 306.5, lon: 9.162745, lat: 48.547293 },
    { x: 32860, y: 304.8, lon: 9.162386, lat: 48.54715 },
    { x: 32886, y: 303.9, lon: 9.162093, lat: 48.547032 },
    { x: 32921, y: 304.2, lon: 9.161679, lat: 48.546865 },
    { x: 32954, y: 303.3, lon: 9.161308, lat: 48.546704 },
    { x: 32981, y: 303.3, lon: 9.161003, lat: 48.546578 },
    { x: 33021, y: 303.3, lon: 9.16053, lat: 48.546396 },
    { x: 33071, y: 303.1, lon: 9.159912, lat: 48.546215 },
    { x: 33091, y: 303.2, lon: 9.159664, lat: 48.546149 },
    { x: 33109, y: 303.5, lon: 9.159425, lat: 48.546094 },
    { x: 33138, y: 304.1, lon: 9.159069, lat: 48.54599 },
    { x: 33188, y: 305.6, lon: 9.158407, lat: 48.545888 },
    { x: 33199, y: 305.9, lon: 9.15826, lat: 48.545866 },
    { x: 33249, y: 306.4, lon: 9.15762, lat: 48.545734 },
    { x: 33283, y: 305.4, lon: 9.15719, lat: 48.545624 },
    { x: 33294, y: 304.8, lon: 9.157047, lat: 48.545583 },
    { x: 33325, y: 304.4, lon: 9.156677, lat: 48.545464 },
    { x: 33348, y: 305.2, lon: 9.156372, lat: 48.545401 },
    { x: 33376, y: 306.2, lon: 9.156036, lat: 48.545296 },
    { x: 33399, y: 306.4, lon: 9.155772, lat: 48.545177 },
    { x: 33412, y: 306.1, lon: 9.155658, lat: 48.54509 },
    { x: 33455, y: 305.4, lon: 9.15524, lat: 48.544819 },
    { x: 33460, y: 305.4, lon: 9.155184, lat: 48.544792 },
    { x: 33502, y: 305.8, lon: 9.154693, lat: 48.544611 },
    { x: 33539, y: 306.2, lon: 9.154298, lat: 48.544406 },
    { x: 33565, y: 306.4, lon: 9.153978, lat: 48.544294 },
    { x: 33581, y: 306.4, lon: 9.153788, lat: 48.544227 },
    { x: 33600, y: 306.1, lon: 9.153537, lat: 48.544204 },
    { x: 33622, y: 305.6, lon: 9.153259, lat: 48.544143 },
    {
      x: 33657,
      y: 305.4,
      lon: 9.152787,
      lat: 48.544143,
      name: 'Lake',
      category: 'lake'
    },
    { x: 33663, y: 305.7, lon: 9.152723, lat: 48.544185 },
    { x: 33739, y: 308.7, lon: 9.151771, lat: 48.54445 },
    { x: 33760, y: 309.3, lon: 9.151489, lat: 48.544454 },
    { x: 33780, y: 309.8, lon: 9.151217, lat: 48.544434 },
    { x: 33802, y: 310.6, lon: 9.150961, lat: 48.544343 },
    { x: 33816, y: 311, lon: 9.150786, lat: 48.544286 },
    { x: 33846, y: 311.3, lon: 9.150432, lat: 48.544162 },
    { x: 33890, y: 313.9, lon: 9.149916, lat: 48.543958 },
    { x: 33918, y: 315.1, lon: 9.149576, lat: 48.54386 },
    { x: 33921, y: 315.8, lon: 9.149527, lat: 48.543857 },
    { x: 33938, y: 319.1, lon: 9.149298, lat: 48.543848 },
    { x: 33966, y: 320.3, lon: 9.148975, lat: 48.54372 },
    { x: 34006, y: 321.5, lon: 9.148578, lat: 48.543473 },
    { x: 34048, y: 321, lon: 9.148132, lat: 48.543232 },
    { x: 34070, y: 318.7, lon: 9.147923, lat: 48.543094 },
    { x: 34100, y: 316.6, lon: 9.147606, lat: 48.542934 },
    { x: 34141, y: 317.5, lon: 9.147142, lat: 48.542732 },
    { x: 34165, y: 319.1, lon: 9.146867, lat: 48.542604 },
    { x: 34197, y: 320.7, lon: 9.146502, lat: 48.542451 },
    { x: 34262, y: 323, lon: 9.145805, lat: 48.542097 },
    { x: 34269, y: 323.6, lon: 9.145737, lat: 48.54206 },
    { x: 34317, y: 326.7, lon: 9.145256, lat: 48.541762 },
    { x: 34349, y: 327.9, lon: 9.144954, lat: 48.541559 },
    { x: 34365, y: 326.9, lon: 9.144822, lat: 48.541441 },
    { x: 34428, y: 326.3, lon: 9.144292, lat: 48.540997 },
    { x: 34447, y: 325.3, lon: 9.144176, lat: 48.540844 },
    { x: 34479, y: 325.2, lon: 9.143915, lat: 48.540621 },
    { x: 34490, y: 325.3, lon: 9.143813, lat: 48.540546 },
    { x: 34546, y: 326.2, lon: 9.14338, lat: 48.540132 },
    { x: 34560, y: 326.4, lon: 9.143259, lat: 48.540036 },
    { x: 34601, y: 324.9, lon: 9.142907, lat: 48.539755 },
    { x: 34630, y: 323.1, lon: 9.142653, lat: 48.539548 },
    { x: 34670, y: 322.5, lon: 9.142314, lat: 48.539267 },
    { x: 34697, y: 320.1, lon: 9.142105, lat: 48.539066 },
    { x: 34733, y: 318.3, lon: 9.141831, lat: 48.538804 },
    { x: 34760, y: 320, lon: 9.141572, lat: 48.538636 },
    { x: 34823, y: 318, lon: 9.141025, lat: 48.538199 },
    { x: 34840, y: 315.8, lon: 9.140906, lat: 48.538068 },
    { x: 34897, y: 313.6, lon: 9.140359, lat: 48.537699 },
    { x: 34922, y: 314.4, lon: 9.140111, lat: 48.537548 },
    { x: 34969, y: 313.8, lon: 9.139629, lat: 48.537281 },
    { x: 35013, y: 312.5, lon: 9.139167, lat: 48.537023 },
    { x: 35047, y: 312.6, lon: 9.138817, lat: 48.536828 },
    { x: 35084, y: 312.5, lon: 9.138392, lat: 48.536651 },
    { x: 35135, y: 311, lon: 9.137853, lat: 48.536366 },
    { x: 35168, y: 311.2, lon: 9.13745, lat: 48.536239 },
    { x: 35192, y: 312.5, lon: 9.137125, lat: 48.53619 },
    { x: 35195, y: 312.7, lon: 9.137095, lat: 48.536203 },
    { x: 35208, y: 313, lon: 9.136916, lat: 48.536184 },
    { x: 35235, y: 313.4, lon: 9.136559, lat: 48.536135 },
    { x: 35253, y: 313.2, lon: 9.136344, lat: 48.536059 },
    { x: 35271, y: 312.1, lon: 9.136229, lat: 48.535915 },
    { x: 35294, y: 310.6, lon: 9.13636, lat: 48.535729 },
    { x: 35328, y: 309.2, lon: 9.136676, lat: 48.535516 },
    { x: 35337, y: 308.9, lon: 9.136772, lat: 48.535462 },
    { x: 35361, y: 308.5, lon: 9.13698, lat: 48.53529 },
    { x: 35401, y: 307.5, lon: 9.137272, lat: 48.534993 },
    { x: 35423, y: 307, lon: 9.137414, lat: 48.534818 },
    { x: 35457, y: 306.6, lon: 9.137588, lat: 48.534535 },
    { x: 35475, y: 306.6, lon: 9.137676, lat: 48.534388 },
    { x: 35504, y: 306.8, lon: 9.137825, lat: 48.534138 },
    { x: 35540, y: 307.1, lon: 9.138011, lat: 48.533843 },
    { x: 35565, y: 307.2, lon: 9.138117, lat: 48.53363 },
    { x: 35607, y: 307.4, lon: 9.13827, lat: 48.533268 },
    { x: 35645, y: 307.6, lon: 9.138379, lat: 48.532934 },
    { x: 35711, y: 310, lon: 9.138509, lat: 48.532341 },
    { x: 35720, y: 310.3, lon: 9.138521, lat: 48.532267 },
    { x: 35763, y: 311, lon: 9.1387, lat: 48.531896 },
    { x: 35765, y: 311, lon: 9.138704, lat: 48.531879 },
    { x: 35766, y: 311, lon: 9.138696, lat: 48.531882 },
    { x: 35767, y: 311, lon: 9.138717, lat: 48.531883 },
    { x: 35768, y: 311, lon: 9.138715, lat: 48.531871 },
    { x: 35771, y: 311, lon: 9.138692, lat: 48.531881 },
    { x: 35771, y: 311, lon: 9.138699, lat: 48.531879 },
    { x: 35772, y: 311, lon: 9.138715, lat: 48.531881 },
    { x: 35781, y: 311, lon: 9.138671, lat: 48.531811 },
    { x: 35784, y: 311, lon: 9.13864, lat: 48.531794 },
    { x: 35808, y: 310.8, lon: 9.138402, lat: 48.531644 },
    { x: 35860, y: 311.7, lon: 9.137694, lat: 48.53166 },
    { x: 35891, y: 313.2, lon: 9.137292, lat: 48.531739 },
    { x: 35899, y: 313.5, lon: 9.137195, lat: 48.531756 },
    { x: 35934, y: 314.5, lon: 9.136731, lat: 48.531847 },
    { x: 35978, y: 314.5, lon: 9.136158, lat: 48.531949 },
    { x: 35990, y: 314.7, lon: 9.136007, lat: 48.531977 },
    { x: 36027, y: 314.6, lon: 9.13552, lat: 48.532078 },
    { x: 36052, y: 315, lon: 9.135194, lat: 48.532141 },
    { x: 36064, y: 315.1, lon: 9.135047, lat: 48.532177 },
    { x: 36089, y: 315.5, lon: 9.13472, lat: 48.532224 },
    { x: 36103, y: 315.9, lon: 9.134527, lat: 48.53225 },
    { x: 36123, y: 317, lon: 9.134259, lat: 48.532263 },
    { x: 36138, y: 317.8, lon: 9.13406, lat: 48.532269 },
    { x: 36156, y: 319.2, lon: 9.133814, lat: 48.53224 },
    { x: 36170, y: 319.4, lon: 9.133635, lat: 48.532224 },
    { x: 36185, y: 319.2, lon: 9.133437, lat: 48.532184 },
    { x: 36201, y: 319.5, lon: 9.133244, lat: 48.532114 },
    { x: 36225, y: 319.8, lon: 9.132933, lat: 48.532057 },
    { x: 36243, y: 320.8, lon: 9.132691, lat: 48.532024 },
    { x: 36262, y: 323.4, lon: 9.132503, lat: 48.531911 },
    { x: 36287, y: 326.2, lon: 9.132268, lat: 48.531752 },
    { x: 36302, y: 327, lon: 9.132102, lat: 48.531672 },
    { x: 36321, y: 330.2, lon: 9.131925, lat: 48.531543 },
    { x: 36352, y: 335.4, lon: 9.131701, lat: 48.531309 },
    { x: 36372, y: 337.8, lon: 9.131564, lat: 48.531151 },
    { x: 36391, y: 339.1, lon: 9.131441, lat: 48.531001 },
    { x: 36419, y: 340.1, lon: 9.131294, lat: 48.530774 },
    { x: 36433, y: 341.4, lon: 9.131197, lat: 48.530662 },
    { x: 36449, y: 344, lon: 9.131079, lat: 48.530544 },
    { x: 36465, y: 345.6, lon: 9.131014, lat: 48.53041 },
    { x: 36493, y: 350.4, lon: 9.130808, lat: 48.530197 },
    { x: 36508, y: 352.3, lon: 9.1307, lat: 48.530083 },
    { x: 36511, y: 352.5, lon: 9.130686, lat: 48.530059 },
    { x: 36532, y: 353.6, lon: 9.130586, lat: 48.529883 },
    { x: 36543, y: 354, lon: 9.130536, lat: 48.529789 },
    { x: 36566, y: 354, lon: 9.130416, lat: 48.529592 },
    { x: 36580, y: 355.2, lon: 9.130369, lat: 48.52947 },
    { x: 36583, y: 355.6, lon: 9.130342, lat: 48.529448 },
    { x: 36600, y: 357.2, lon: 9.130242, lat: 48.529314 },
    { x: 36609, y: 358, lon: 9.130186, lat: 48.529244 },
    { x: 36618, y: 358.9, lon: 9.1301, lat: 48.529189 },
    { x: 36629, y: 359.8, lon: 9.130013, lat: 48.529107 },
    { x: 36652, y: 360.6, lon: 9.129925, lat: 48.528903 },
    { x: 36664, y: 361, lon: 9.129864, lat: 48.528811 },
    { x: 36664, y: 361, lon: 9.12986, lat: 48.528805 },
    { x: 36685, y: 361.9, lon: 9.129698, lat: 48.528659 },
    { x: 36697, y: 362.4, lon: 9.129646, lat: 48.528549 },
    { x: 36707, y: 362.7, lon: 9.129575, lat: 48.528476 },
    { x: 36715, y: 363.2, lon: 9.129508, lat: 48.528418 },
    { x: 36723, y: 363.6, lon: 9.129442, lat: 48.528365 },
    { x: 36735, y: 364.3, lon: 9.129323, lat: 48.528286 },
    { x: 36745, y: 364.6, lon: 9.12926, lat: 48.52821 },
    { x: 36764, y: 365, lon: 9.129154, lat: 48.528057 },
    { x: 36779, y: 365.3, lon: 9.129045, lat: 48.527936 },
    { x: 36800, y: 366.3, lon: 9.128918, lat: 48.527775 },
    { x: 36810, y: 366.8, lon: 9.128853, lat: 48.527694 },
    { x: 36817, y: 367, lon: 9.12879, lat: 48.527649 },
    { x: 36829, y: 367.4, lon: 9.128636, lat: 48.527603 },
    { x: 36842, y: 367.7, lon: 9.128577, lat: 48.527492 },
    { x: 36857, y: 367.9, lon: 9.128459, lat: 48.52738 },
    { x: 36877, y: 367.7, lon: 9.128321, lat: 48.527233 },
    { x: 36880, y: 367.6, lon: 9.128292, lat: 48.527206 },
    { x: 36912, y: 367.6, lon: 9.128091, lat: 48.526953 },
    { x: 36915, y: 367.7, lon: 9.128071, lat: 48.526932 },
    { x: 36929, y: 368.2, lon: 9.127947, lat: 48.526831 },
    { x: 36953, y: 370.2, lon: 9.127705, lat: 48.52669 },
    { x: 36967, y: 371.4, lon: 9.127582, lat: 48.526597 },
    { x: 36980, y: 372.4, lon: 9.127458, lat: 48.526514 },
    { x: 36988, y: 373, lon: 9.127377, lat: 48.526469 },
    { x: 37004, y: 374, lon: 9.127188, lat: 48.5264 },
    { x: 37022, y: 374.8, lon: 9.126965, lat: 48.526331 },
    { x: 37024, y: 374.9, lon: 9.126938, lat: 48.526321 },
    { x: 37035, y: 375.3, lon: 9.126797, lat: 48.526279 },
    { x: 37047, y: 375.7, lon: 9.126656, lat: 48.526229 },
    { x: 37060, y: 376.3, lon: 9.126495, lat: 48.526191 },
    { x: 37073, y: 376.8, lon: 9.126327, lat: 48.52616 },
    { x: 37087, y: 377.5, lon: 9.126154, lat: 48.526111 },
    { x: 37100, y: 378.1, lon: 9.125983, lat: 48.526065 },
    { x: 37115, y: 378.5, lon: 9.125793, lat: 48.526027 },
    { x: 37134, y: 379, lon: 9.125569, lat: 48.525949 },
    { x: 37142, y: 379.2, lon: 9.12548, lat: 48.525909 },
    { x: 37154, y: 380.2, lon: 9.125348, lat: 48.525846 },
    { x: 37166, y: 381.9, lon: 9.1252, lat: 48.52579 },
    { x: 37176, y: 383.1, lon: 9.125097, lat: 48.525734 },
    { x: 37194, y: 384.9, lon: 9.12493, lat: 48.52561 },
    { x: 37200, y: 385.4, lon: 9.124881, lat: 48.525576 },
    { x: 37213, y: 386.6, lon: 9.124747, lat: 48.525496 },
    { x: 37226, y: 387.6, lon: 9.124627, lat: 48.525417 },
    { x: 37239, y: 388.5, lon: 9.124531, lat: 48.525315 },
    { x: 37251, y: 389.3, lon: 9.124455, lat: 48.525219 },
    { x: 37264, y: 389.9, lon: 9.124393, lat: 48.525109 },
    { x: 37269, y: 390.2, lon: 9.124358, lat: 48.525067 },
    { x: 37272, y: 389.9, lon: 9.124384, lat: 48.525082 },
    { x: 37286, y: 391.1, lon: 9.124261, lat: 48.524986 },
    { x: 37302, y: 391.2, lon: 9.12422, lat: 48.524847 },
    { x: 37321, y: 391.8, lon: 9.124115, lat: 48.524689 },
    { x: 37342, y: 392.7, lon: 9.123997, lat: 48.524512 },
    { x: 37367, y: 393.8, lon: 9.123881, lat: 48.524308 },
    { x: 37400, y: 394.6, lon: 9.12371, lat: 48.524033 },
    { x: 37403, y: 394.7, lon: 9.123692, lat: 48.524004 },
    { x: 37430, y: 395, lon: 9.123549, lat: 48.523781 },
    { x: 37451, y: 396, lon: 9.123392, lat: 48.523622 },
    { x: 37458, y: 396.3, lon: 9.123342, lat: 48.523574 },
    { x: 37473, y: 396.8, lon: 9.123214, lat: 48.523463 },
    { x: 37494, y: 397.3, lon: 9.123028, lat: 48.523326 },
    { x: 37505, y: 397.3, lon: 9.122955, lat: 48.523241 },
    { x: 37530, y: 397.7, lon: 9.122748, lat: 48.523061 },
    { x: 37540, y: 397.8, lon: 9.122676, lat: 48.522987 },
    { x: 37552, y: 397.8, lon: 9.122586, lat: 48.522896 },
    { x: 37570, y: 398.2, lon: 9.122416, lat: 48.522784 },
    { x: 37587, y: 398.6, lon: 9.122208, lat: 48.52271 },
    { x: 37598, y: 398.7, lon: 9.122086, lat: 48.522657 },
    { x: 37614, y: 399.6, lon: 9.121906, lat: 48.52257 },
    { x: 37625, y: 400.2, lon: 9.121782, lat: 48.522512 },
    { x: 37645, y: 401.4, lon: 9.121525, lat: 48.522456 },
    { x: 37661, y: 401.9, lon: 9.121357, lat: 48.52237 },
    { x: 37697, y: 402.4, lon: 9.120982, lat: 48.522169 },
    { x: 37704, y: 402.5, lon: 9.1209, lat: 48.522134 },
    { x: 37722, y: 403.1, lon: 9.120684, lat: 48.52206 },
    { x: 37738, y: 404.2, lon: 9.120481, lat: 48.522 },
    { x: 37757, y: 405.6, lon: 9.120249, lat: 48.521937 },
    {
      x: 37780,
      y: 407.5,
      lon: 9.119964,
      lat: 48.521858,
      name: 'Kusterdingen',
      category: 'village'
    },
    { x: 37792, y: 408.1, lon: 9.119981, lat: 48.521749 },
    { x: 37815, y: 408.8, lon: 9.120067, lat: 48.521545 },
    { x: 37849, y: 409.2, lon: 9.120238, lat: 48.521261 },
    { x: 37863, y: 408.9, lon: 9.120332, lat: 48.52115 },
    { x: 37874, y: 408.6, lon: 9.120435, lat: 48.521078 },
    { x: 37886, y: 408.8, lon: 9.120469, lat: 48.520978 },
    { x: 37906, y: 409.1, lon: 9.120607, lat: 48.520822 },
    { x: 37916, y: 409.2, lon: 9.120701, lat: 48.520754 },
    { x: 37962, y: 410.3, lon: 9.121022, lat: 48.520402 },
    { x: 37979, y: 410.5, lon: 9.12106, lat: 48.520256 },
    { x: 37999, y: 410.6, lon: 9.121119, lat: 48.520073 },
    { x: 38031, y: 410.9, lon: 9.121198, lat: 48.519797 },
    { x: 38063, y: 411.3, lon: 9.12117, lat: 48.519502 },
    { x: 38075, y: 411.4, lon: 9.121132, lat: 48.519402 },
    { x: 38095, y: 411.7, lon: 9.121071, lat: 48.519227 },
    { x: 38116, y: 411.8, lon: 9.121032, lat: 48.519035 },
    { x: 38142, y: 412, lon: 9.121063, lat: 48.518803 },
    { x: 38156, y: 412.1, lon: 9.1211, lat: 48.518685 },
    { x: 38178, y: 412.6, lon: 9.121123, lat: 48.518482 },
    { x: 38206, y: 413.2, lon: 9.12115, lat: 48.518237 },
    { x: 38228, y: 413.7, lon: 9.121194, lat: 48.518035 },
    { x: 38251, y: 413.8, lon: 9.121246, lat: 48.51783 },
    { x: 38274, y: 413.3, lon: 9.121279, lat: 48.517626 },
    { x: 38278, y: 413.2, lon: 9.121281, lat: 48.517595 },
    { x: 38305, y: 412.8, lon: 9.121136, lat: 48.517366 },
    { x: 38323, y: 413, lon: 9.120916, lat: 48.517312 },
    { x: 38342, y: 413.4, lon: 9.120659, lat: 48.517264 },
    { x: 38365, y: 413.9, lon: 9.120383, lat: 48.517182 },
    { x: 38375, y: 414, lon: 9.120298, lat: 48.517113 },
    { x: 38389, y: 414, lon: 9.120158, lat: 48.517026 },
    { x: 38409, y: 413.9, lon: 9.11989, lat: 48.517014 },
    { x: 38434, y: 413.9, lon: 9.119554, lat: 48.516953 },
    { x: 38453, y: 413.9, lon: 9.119319, lat: 48.516899 },
    { x: 38475, y: 414.1, lon: 9.119035, lat: 48.516823 },
    { x: 38497, y: 414.4, lon: 9.118752, lat: 48.51676 },
    { x: 38520, y: 414.5, lon: 9.118466, lat: 48.516697 },
    { x: 38543, y: 414.5, lon: 9.118171, lat: 48.516626 },
    { x: 38572, y: 414.9, lon: 9.117793, lat: 48.516551 },
    { x: 38588, y: 415.3, lon: 9.117588, lat: 48.516498 },
    { x: 38609, y: 415.8, lon: 9.11731, lat: 48.516452 },
    { x: 38631, y: 416.2, lon: 9.117025, lat: 48.516402 },
    { x: 38663, y: 416.2, lon: 9.116599, lat: 48.51634 },
    { x: 38679, y: 416.2, lon: 9.116389, lat: 48.516307 },
    { x: 38692, y: 416.2, lon: 9.116222, lat: 48.516289 },
    { x: 38727, y: 417.2, lon: 9.115763, lat: 48.516221 },
    { x: 38735, y: 417.5, lon: 9.115649, lat: 48.516212 },
    { x: 38758, y: 418.3, lon: 9.115333, lat: 48.516202 },
    { x: 38777, y: 419.2, lon: 9.115084, lat: 48.516178 },
    { x: 38800, y: 420.1, lon: 9.114779, lat: 48.51615 },
    { x: 38827, y: 420.9, lon: 9.114418, lat: 48.516116 },
    { x: 38843, y: 421.2, lon: 9.114202, lat: 48.516089 },
    { x: 38870, y: 421.8, lon: 9.113844, lat: 48.51606 },
    { x: 38887, y: 422.4, lon: 9.113615, lat: 48.516035 },
    { x: 38914, y: 423.6, lon: 9.113259, lat: 48.515982 },
    { x: 38941, y: 424.6, lon: 9.112913, lat: 48.515897 },
    { x: 38956, y: 424.5, lon: 9.112727, lat: 48.515853 },
    { x: 38979, y: 424.5, lon: 9.112424, lat: 48.515784 },
    { x: 39012, y: 424.7, lon: 9.111999, lat: 48.515692 },
    { x: 39035, y: 424.9, lon: 9.111705, lat: 48.515633 },
    { x: 39051, y: 424.9, lon: 9.111498, lat: 48.515599 },
    { x: 39075, y: 424.9, lon: 9.111167, lat: 48.515567 },
    { x: 39098, y: 424.6, lon: 9.110867, lat: 48.515592 },
    { x: 39120, y: 424.5, lon: 9.110596, lat: 48.515678 },
    { x: 39141, y: 424.1, lon: 9.110345, lat: 48.515591 },
    { x: 39167, y: 423.7, lon: 9.110203, lat: 48.51538 },
    { x: 39204, y: 422.9, lon: 9.109871, lat: 48.515129 },
    { x: 39232, y: 422.5, lon: 9.109589, lat: 48.514961 },
    { x: 39256, y: 422.5, lon: 9.10935, lat: 48.514812 },
    { x: 39293, y: 423.4, lon: 9.109001, lat: 48.514578 },
    { x: 39329, y: 423.6, lon: 9.108661, lat: 48.514341 },
    { x: 39359, y: 423.2, lon: 9.108332, lat: 48.514175 },
    { x: 39387, y: 423.2, lon: 9.107993, lat: 48.514083 },
    { x: 39423, y: 422.7, lon: 9.107514, lat: 48.514024 },
    { x: 39461, y: 421.9, lon: 9.106998, lat: 48.513997 },
    { x: 39504, y: 421, lon: 9.106419, lat: 48.513963 },
    { x: 39545, y: 420.9, lon: 9.105861, lat: 48.513909 },
    { x: 39578, y: 421, lon: 9.105428, lat: 48.51387 },
    { x: 39605, y: 421.2, lon: 9.105056, lat: 48.513835 },
    { x: 39632, y: 421.3, lon: 9.104696, lat: 48.513827 },
    { x: 39654, y: 422, lon: 9.104406, lat: 48.513798 },
    { x: 39701, y: 423.9, lon: 9.103765, lat: 48.513759 },
    { x: 39717, y: 425.4, lon: 9.103558, lat: 48.513741 },
    { x: 39737, y: 427.4, lon: 9.103285, lat: 48.513718 },
    { x: 39759, y: 429.4, lon: 9.102988, lat: 48.513716 },
    { x: 39797, y: 431.4, lon: 9.102471, lat: 48.513697 },
    { x: 39808, y: 431.8, lon: 9.102322, lat: 48.513681 },
    { x: 39826, y: 432.5, lon: 9.102088, lat: 48.513645 },
    { x: 39854, y: 433.1, lon: 9.101724, lat: 48.513589 },
    { x: 39876, y: 433.4, lon: 9.101437, lat: 48.513519 },
    { x: 39892, y: 433.6, lon: 9.101248, lat: 48.513453 },
    { x: 39919, y: 433.4, lon: 9.100899, lat: 48.513376 },
    { x: 39933, y: 433.4, lon: 9.100718, lat: 48.513344 },
    { x: 39962, y: 433.1, lon: 9.100395, lat: 48.513203 },
    { x: 39992, y: 433.1, lon: 9.100024, lat: 48.513092 },
    { x: 40010, y: 432.7, lon: 9.09993, lat: 48.512937 },
    { x: 40026, y: 432.1, lon: 9.099837, lat: 48.512807 },
    { x: 40031, y: 431.9, lon: 9.099791, lat: 48.512775 },
    { x: 40038, y: 431.6, lon: 9.099751, lat: 48.512715 },
    { x: 40056, y: 430.7, lon: 9.099771, lat: 48.512558 },
    { x: 40103, y: 429.5, lon: 9.100032, lat: 48.512177 },
    { x: 40113, y: 429.7, lon: 9.100129, lat: 48.512113 },
    { x: 40180, y: 427.3, lon: 9.100492, lat: 48.511559 },
    { x: 40190, y: 426.9, lon: 9.100577, lat: 48.511485 },
    { x: 40235, y: 425.7, lon: 9.10089, lat: 48.511142 },
    { x: 40248, y: 425.2, lon: 9.100965, lat: 48.511035 },
    { x: 40329, y: 423, lon: 9.101239, lat: 48.510331 },
    { x: 40354, y: 421, lon: 9.101251, lat: 48.510101 },
    { x: 40387, y: 418, lon: 9.101342, lat: 48.509815 },
    { x: 40409, y: 416.2, lon: 9.101351, lat: 48.509619 },
    { x: 40444, y: 414.7, lon: 9.101475, lat: 48.509311 },
    { x: 40477, y: 413.7, lon: 9.101584, lat: 48.509021 },
    { x: 40519, y: 413.2, lon: 9.101723, lat: 48.508663 },
    { x: 40546, y: 413.2, lon: 9.101585, lat: 48.508431 },
    { x: 40578, y: 413.6, lon: 9.101611, lat: 48.508142 },
    { x: 40604, y: 413.5, lon: 9.101457, lat: 48.507934 },
    { x: 40663, y: 411.5, lon: 9.100667, lat: 48.50798 },
    { x: 40717, y: 409.1, lon: 9.099976, lat: 48.508146 },
    { x: 40725, y: 408.7, lon: 9.099871, lat: 48.508172 },
    { x: 40774, y: 406.6, lon: 9.099272, lat: 48.50836 },
    { x: 40861, y: 400.5, lon: 9.098343, lat: 48.508839 },
    { x: 40896, y: 398.8, lon: 9.097947, lat: 48.509016 },
    { x: 40963, y: 396.4, lon: 9.097261, lat: 48.509404 },
    { x: 41021, y: 394.4, lon: 9.096673, lat: 48.50975 },
    { x: 41098, y: 392.6, lon: 9.095828, lat: 48.510153 },
    { x: 41162, y: 390.4, lon: 9.095132, lat: 48.510497 },
    { x: 41222, y: 388.1, lon: 9.09436, lat: 48.510668 },
    { x: 41281, y: 384.5, lon: 9.093634, lat: 48.510887 },
    { x: 41335, y: 380.8, lon: 9.092976, lat: 48.511093 },
    { x: 41437, y: 373.1, lon: 9.091634, lat: 48.511306 },
    { x: 41483, y: 371.2, lon: 9.09102, lat: 48.51136 },
    { x: 41568, y: 372.2, lon: 9.089868, lat: 48.511317 },
    { x: 41604, y: 370.9, lon: 9.089399, lat: 48.511402 },
    { x: 41646, y: 366.6, lon: 9.088957, lat: 48.511639 },
    { x: 41688, y: 354.5, lon: 9.089069, lat: 48.512016 },
    { x: 41756, y: 346.8, lon: 9.08846, lat: 48.512468 },
    { x: 41800, y: 341.4, lon: 9.088128, lat: 48.512798 },
    { x: 41857, y: 341.1, lon: 9.087376, lat: 48.512894 },
    { x: 41862, y: 341, lon: 9.087303, lat: 48.512901 },
    { x: 41903, y: 336.4, lon: 9.086772, lat: 48.512993 },
    { x: 41935, y: 329.3, lon: 9.086502, lat: 48.513223 },
    { x: 41971, y: 322.7, lon: 9.086197, lat: 48.513478 },
    { x: 42012, y: 323.5, lon: 9.08573, lat: 48.513274 },
    { x: 42020, y: 323.4, lon: 9.085637, lat: 48.513241 },
    { x: 42083, y: 324, lon: 9.084966, lat: 48.512889 },
    { x: 42099, y: 324.9, lon: 9.084802, lat: 48.512803 },
    { x: 42152, y: 325.4, lon: 9.084167, lat: 48.512583 },
    { x: 42167, y: 324.9, lon: 9.083969, lat: 48.512535 },
    { x: 42207, y: 323.3, lon: 9.083452, lat: 48.512438 },
    { x: 42232, y: 321.9, lon: 9.08312, lat: 48.512412 },
    { x: 42257, y: 320.9, lon: 9.082778, lat: 48.51237 },
    { x: 42277, y: 321.1, lon: 9.082556, lat: 48.51228 },
    { x: 42328, y: 320.7, lon: 9.081892, lat: 48.512135 },
    { x: 42348, y: 320.4, lon: 9.081635, lat: 48.512096 },
    { x: 42352, y: 320.4, lon: 9.081574, lat: 48.512084 },
    { x: 42395, y: 321.1, lon: 9.081017, lat: 48.511981 },
    { x: 42425, y: 320.5, lon: 9.08062, lat: 48.511993 },
    { x: 42442, y: 320.2, lon: 9.080388, lat: 48.512005 },
    { x: 42458, y: 320, lon: 9.080206, lat: 48.51208 },
    { x: 42476, y: 320.1, lon: 9.079981, lat: 48.512149 },
    { x: 42481, y: 320.1, lon: 9.079949, lat: 48.512113 },
    { x: 42487, y: 320, lon: 9.079879, lat: 48.512076 },
    { x: 42489, y: 320, lon: 9.079862, lat: 48.512092 },
    { x: 42490, y: 320, lon: 9.079871, lat: 48.512093 },
    { x: 42491, y: 320, lon: 9.079853, lat: 48.512088 },
    { x: 42499, y: 320.1, lon: 9.079772, lat: 48.512048 },
    { x: 42507, y: 320.2, lon: 9.079666, lat: 48.512018 },
    { x: 42511, y: 320.2, lon: 9.07964, lat: 48.511995 },
    { x: 42513, y: 320.3, lon: 9.079645, lat: 48.51197 },
    { x: 42519, y: 320.2, lon: 9.079576, lat: 48.511997 },
    { x: 42572, y: 321.8, lon: 9.078893, lat: 48.512127 },
    { x: 42588, y: 322.2, lon: 9.078686, lat: 48.512186 },
    { x: 42608, y: 321.8, lon: 9.07843, lat: 48.512215 },
    { x: 42630, y: 321.4, lon: 9.078146, lat: 48.512286 },
    { x: 42661, y: 320.9, lon: 9.077756, lat: 48.512382 },
    { x: 42692, y: 320.4, lon: 9.077354, lat: 48.51247 },
    { x: 42723, y: 319.9, lon: 9.076957, lat: 48.512553 },
    { x: 42769, y: 319.8, lon: 9.076368, lat: 48.512681 },
    { x: 42799, y: 320.3, lon: 9.075967, lat: 48.512748 },
    { x: 42810, y: 320.6, lon: 9.075825, lat: 48.512755 },
    { x: 42857, y: 320.7, lon: 9.075257, lat: 48.512948 },
    { x: 42864, y: 320.4, lon: 9.075184, lat: 48.512987 },
    { x: 42885, y: 319.6, lon: 9.075012, lat: 48.513142 },
    { x: 42910, y: 319.2, lon: 9.074676, lat: 48.513158 },
    { x: 42944, y: 319.1, lon: 9.074222, lat: 48.513153 },
    { x: 42972, y: 319, lon: 9.073854, lat: 48.513217 },
    { x: 42992, y: 318.9, lon: 9.07359, lat: 48.51328 },
    { x: 43025, y: 318.7, lon: 9.073168, lat: 48.513366 },
    { x: 43052, y: 318.4, lon: 9.072821, lat: 48.51345 },
    { x: 43083, y: 317.6, lon: 9.072432, lat: 48.513549 },
    { x: 43114, y: 316.9, lon: 9.072043, lat: 48.51365 },
    { x: 43153, y: 317.6, lon: 9.071549, lat: 48.513769 },
    { x: 43166, y: 317.8, lon: 9.071395, lat: 48.513834 },
    { x: 43202, y: 318.6, lon: 9.070942, lat: 48.513947 },
    { x: 43209, y: 318.7, lon: 9.070851, lat: 48.513948 },
    { x: 43213, y: 318.8, lon: 9.070801, lat: 48.513961 },
    { x: 43214, y: 318.8, lon: 9.070812, lat: 48.513949 },
    { x: 43215, y: 318.8, lon: 9.070809, lat: 48.513944 },
    { x: 43218, y: 318.9, lon: 9.07077, lat: 48.513931 },
    { x: 43219, y: 318.9, lon: 9.070771, lat: 48.513927 },
    { x: 43220, y: 318.9, lon: 9.070781, lat: 48.513915 },
    { x: 43221, y: 318.9, lon: 9.070784, lat: 48.513911 },
    { x: 43252, y: 319.3, lon: 9.070394, lat: 48.514031 },
    { x: 43259, y: 319.3, lon: 9.070317, lat: 48.514057 },
    { x: 43296, y: 319, lon: 9.069837, lat: 48.514173 },
    { x: 43310, y: 318.9, lon: 9.069662, lat: 48.514203 },
    { x: 43340, y: 318.5, lon: 9.069274, lat: 48.514284 },
    { x: 43357, y: 318.2, lon: 9.069055, lat: 48.514323 },
    { x: 43388, y: 317.8, lon: 9.068636, lat: 48.514393 },
    { x: 43418, y: 317, lon: 9.068272, lat: 48.514499 },
    { x: 43446, y: 316.1, lon: 9.067906, lat: 48.514555 },
    { x: 43469, y: 316.4, lon: 9.067606, lat: 48.514628 },
    { x: 43499, y: 317, lon: 9.067225, lat: 48.514713 },
    { x: 43543, y: 318.2, lon: 9.066661, lat: 48.514841 },
    { x: 43547, y: 318.3, lon: 9.066618, lat: 48.514862 },
    { x: 43565, y: 318.5, lon: 9.066399, lat: 48.514928 },
    { x: 43579, y: 318.2, lon: 9.06635, lat: 48.515056 },
    { x: 43610, y: 317.4, lon: 9.0664, lat: 48.515326 },
    { x: 43643, y: 317.5, lon: 9.066019, lat: 48.515483 },
    { x: 43649, y: 317.6, lon: 9.06594, lat: 48.515495 },
    { x: 43678, y: 318.3, lon: 9.065554, lat: 48.515542 },
    { x: 43710, y: 319, lon: 9.065112, lat: 48.51552 },
    { x: 43736, y: 319.3, lon: 9.064766, lat: 48.515546 },
    { x: 43763, y: 319.5, lon: 9.064424, lat: 48.515624 },
    { x: 43800, y: 319.8, lon: 9.063975, lat: 48.515772 },
    { x: 43817, y: 319.9, lon: 9.06379, lat: 48.515857 },
    { x: 43830, y: 319.9, lon: 9.063639, lat: 48.515925 },
    { x: 43855, y: 320.2, lon: 9.063363, lat: 48.516057 },
    { x: 43877, y: 320.6, lon: 9.063105, lat: 48.516153 },
    { x: 43894, y: 321.1, lon: 9.062878, lat: 48.516191 },
    { x: 43946, y: 321.4, lon: 9.062645, lat: 48.515753 },
    { x: 43968, y: 321.2, lon: 9.062653, lat: 48.515553 },
    { x: 43979, y: 321.1, lon: 9.062621, lat: 48.515459 },
    { x: 43985, y: 321, lon: 9.062604, lat: 48.515404 },
    { x: 44000, y: 321, lon: 9.06256, lat: 48.515271 },
    { x: 44025, y: 321, lon: 9.062509, lat: 48.515052 },
    { x: 44049, y: 321, lon: 9.062457, lat: 48.514836 },
    { x: 44068, y: 321, lon: 9.062315, lat: 48.514694 },
    { x: 44091, y: 321.1, lon: 9.062116, lat: 48.514542 },
    { x: 44112, y: 321.7, lon: 9.061832, lat: 48.514532 },
    { x: 44150, y: 322.7, lon: 9.061349, lat: 48.514647 },
    { x: 44171, y: 322.9, lon: 9.061055, lat: 48.514673 },
    { x: 44192, y: 323, lon: 9.060791, lat: 48.514737 },
    { x: 44196, y: 323, lon: 9.060848, lat: 48.51473 },
    { x: 44203, y: 323, lon: 9.06076, lat: 48.514722 },
    { x: 44214, y: 323.1, lon: 9.060618, lat: 48.514728 },
    { x: 44243, y: 323.7, lon: 9.060217, lat: 48.514732 },
    { x: 44258, y: 324.3, lon: 9.060014, lat: 48.514745 },
    { x: 44295, y: 325.2, lon: 9.059526, lat: 48.514801 },
    { x: 44301, y: 325.1, lon: 9.059443, lat: 48.514803 },
    { x: 44312, y: 325.2, lon: 9.0593, lat: 48.514765 },
    { x: 44316, y: 325.1, lon: 9.059257, lat: 48.514783 },
    { x: 44318, y: 325, lon: 9.059236, lat: 48.51479 },
    { x: 44328, y: 324.9, lon: 9.059106, lat: 48.514817 },
    { x: 44329, y: 324.9, lon: 9.059092, lat: 48.514819 },
    {
      x: 44329,
      y: 324.9,
      lon: 9.059088,
      lat: 48.514822,
      name: 'Tübingen',
      category: 'city'
    },
    { x: 44343, y: 325.1, lon: 9.059077, lat: 48.514702 },
    { x: 44376, y: 324.9, lon: 9.058782, lat: 48.514476 },
    { x: 44381, y: 324.8, lon: 9.058757, lat: 48.514439 },
    { x: 44409, y: 324.6, lon: 9.058595, lat: 48.514205 },
    { x: 44436, y: 324.6, lon: 9.058424, lat: 48.513993 },
    { x: 44445, y: 324.6, lon: 9.058373, lat: 48.513921 },
    { x: 44469, y: 324.6, lon: 9.058251, lat: 48.513714 },
    { x: 44495, y: 324.7, lon: 9.058129, lat: 48.513501 },
    { x: 44518, y: 324.9, lon: 9.057993, lat: 48.513313 },
    { x: 44544, y: 325, lon: 9.057863, lat: 48.513098 },
    { x: 44575, y: 325.1, lon: 9.057735, lat: 48.512833 },
    { x: 44602, y: 325.2, lon: 9.057579, lat: 48.512612 },
    { x: 44628, y: 325.3, lon: 9.057433, lat: 48.512401 },
    { x: 44648, y: 325.2, lon: 9.057315, lat: 48.512236 },
    { x: 44681, y: 325.1, lon: 9.057148, lat: 48.511957 },
    { x: 44723, y: 325, lon: 9.056943, lat: 48.511605 },
    { x: 44736, y: 325, lon: 9.05687, lat: 48.511507 },
    { x: 44763, y: 325, lon: 9.056726, lat: 48.511282 },
    { x: 44797, y: 325.3, lon: 9.056503, lat: 48.511016 },
    { x: 44820, y: 325.5, lon: 9.056369, lat: 48.510827 },
    { x: 44832, y: 325.6, lon: 9.056291, lat: 48.510731 },
    { x: 44861, y: 325.8, lon: 9.056167, lat: 48.51048 },
    { x: 44911, y: 325.8, lon: 9.056078, lat: 48.510035 },
    { x: 44943, y: 325.7, lon: 9.056005, lat: 48.509757 },
    { x: 44948, y: 325.7, lon: 9.055997, lat: 48.509708 },
    { x: 44987, y: 325.8, lon: 9.056006, lat: 48.509355 },
    { x: 45000, y: 325.8, lon: 9.056013, lat: 48.509243 },
    { x: 45031, y: 326, lon: 9.056094, lat: 48.508971 },
    { x: 45049, y: 326, lon: 9.056158, lat: 48.508809 },
    { x: 45088, y: 326.4, lon: 9.056293, lat: 48.508474 },
    { x: 45107, y: 326.6, lon: 9.056369, lat: 48.508313 },
    { x: 45129, y: 326.8, lon: 9.056482, lat: 48.508129 },
    { x: 45149, y: 327, lon: 9.056564, lat: 48.507959 },
    { x: 45183, y: 326.8, lon: 9.056664, lat: 48.507661 },
    { x: 45195, y: 326.8, lon: 9.056709, lat: 48.507556 },
    { x: 45216, y: 326.8, lon: 9.056788, lat: 48.507367 },
    { x: 45244, y: 326.8, lon: 9.056929, lat: 48.507137 },
    { x: 45264, y: 327.2, lon: 9.057001, lat: 48.506968 },
    { x: 45292, y: 327.8, lon: 9.057106, lat: 48.506726 },
    { x: 45326, y: 328.5, lon: 9.057276, lat: 48.506435 },
    { x: 45357, y: 329.2, lon: 9.057387, lat: 48.506173 },
    { x: 45402, y: 330.1, lon: 9.057521, lat: 48.505777 },
    { x: 45404, y: 330.1, lon: 9.057542, lat: 48.50576 },
    { x: 45425, y: 330.4, lon: 9.057623, lat: 48.505582 },
    { x: 45429, y: 330.4, lon: 9.057641, lat: 48.505543 },
    { x: 45455, y: 330.3, lon: 9.05782, lat: 48.505343 },
    { x: 45460, y: 330.3, lon: 9.057841, lat: 48.505306 },
    { x: 45485, y: 330.4, lon: 9.057947, lat: 48.505088 },
    { x: 45530, y: 331.2, lon: 9.058104, lat: 48.504699 },
    { x: 45558, y: 331.5, lon: 9.058233, lat: 48.504466 },
    { x: 45566, y: 331.4, lon: 9.058267, lat: 48.50439 },
    { x: 45588, y: 331.2, lon: 9.058375, lat: 48.504214 },
    { x: 45608, y: 330.8, lon: 9.058482, lat: 48.504048 },
    { x: 45639, y: 330.1, lon: 9.058636, lat: 48.503781 },
    { x: 45672, y: 330.3, lon: 9.058777, lat: 48.503505 },
    { x: 45706, y: 330.6, lon: 9.058913, lat: 48.503211 },
    { x: 45718, y: 330.6, lon: 9.058972, lat: 48.503115 },
    { x: 45766, y: 331.1, lon: 9.059193, lat: 48.502709 },
    { x: 45784, y: 331.4, lon: 9.059278, lat: 48.502556 },
    { x: 45789, y: 331.4, lon: 9.059325, lat: 48.502523 },
    { x: 45815, y: 331.8, lon: 9.059419, lat: 48.502295 },
    { x: 45851, y: 331.8, lon: 9.059588, lat: 48.501988 },
    { x: 45872, y: 331.5, lon: 9.059751, lat: 48.50184 },
    { x: 45890, y: 331.5, lon: 9.059987, lat: 48.501792 },
    { x: 45920, y: 331.2, lon: 9.060178, lat: 48.501551 },
    { x: 45949, y: 331.1, lon: 9.060391, lat: 48.501338 },
    { x: 45970, y: 331.4, lon: 9.060507, lat: 48.50116 },
    { x: 45992, y: 332.1, lon: 9.060648, lat: 48.500992 },
    { x: 46025, y: 333, lon: 9.060807, lat: 48.500709 },
    { x: 46044, y: 333.3, lon: 9.060936, lat: 48.500564 },
    { x: 46054, y: 333.3, lon: 9.061027, lat: 48.5005 },
    { x: 46065, y: 333.2, lon: 9.060949, lat: 48.50059 },
    { x: 46083, y: 332.8, lon: 9.060853, lat: 48.500734 },
    { x: 46092, y: 332.7, lon: 9.06079, lat: 48.500803 },
    { x: 46092, y: 332.7, lon: 9.060785, lat: 48.500805 },
    { x: 46094, y: 332.7, lon: 9.060769, lat: 48.500813 },
    { x: 46097, y: 332.8, lon: 9.060746, lat: 48.500787 },
    { x: 46098, y: 332.8, lon: 9.060755, lat: 48.500778 },
    { x: 46101, y: 332.7, lon: 9.060781, lat: 48.500794 },
    { x: 46102, y: 332.7, lon: 9.060784, lat: 48.5008 },
    { x: 46102, y: 332.7, lon: 9.060786, lat: 48.500802 },
    { x: 46102, y: 332.7, lon: 9.060787, lat: 48.500802 },
    { x: 46102, y: 332.7, lon: 9.060786, lat: 48.500801 },
    { x: 46102, y: 332.7, lon: 9.060786, lat: 48.500801 },
    { x: 46102, y: 332.7, lon: 9.060786, lat: 48.500801 },
    { x: 46102, y: 332.7, lon: 9.060786, lat: 48.500801 },
    { x: 46102, y: 332.7, lon: 9.060786, lat: 48.500801 },
    { x: 46102, y: 332.7, lon: 9.060784, lat: 48.500801 },
    { x: 46106, y: 332.7, lon: 9.060825, lat: 48.50079 },
    { x: 46107, y: 332.7, lon: 9.060828, lat: 48.50078 },
    { x: 46111, y: 332.9, lon: 9.060808, lat: 48.50074 },
    { x: 46144, y: 333.5, lon: 9.06098, lat: 48.500472 },
    { x: 46172, y: 333, lon: 9.061176, lat: 48.500259 },
    { x: 46192, y: 332.7, lon: 9.061319, lat: 48.500102 },
    { x: 46209, y: 332.6, lon: 9.061475, lat: 48.499991 },
    { x: 46223, y: 332.5, lon: 9.061588, lat: 48.499884 },
    { x: 46236, y: 332.5, lon: 9.061732, lat: 48.499824 },
    { x: 46237, y: 332.5, lon: 9.061724, lat: 48.499833 },
    {
      x: 46248,
      y: 332.3,
      lon: 9.06182,
      lat: 48.499754,
      name: 'Derendingen',
      category: 'village'
    },
    { x: 46285, y: 332.1, lon: 9.062071, lat: 48.499465 },
    { x: 46312, y: 332.4, lon: 9.062201, lat: 48.499241 },
    { x: 46336, y: 332.6, lon: 9.062339, lat: 48.499046 },
    { x: 46377, y: 332.9, lon: 9.062568, lat: 48.498708 },
    { x: 46406, y: 332.5, lon: 9.062762, lat: 48.498485 },
    { x: 46425, y: 332, lon: 9.062894, lat: 48.498342 },
    { x: 46464, y: 331.6, lon: 9.063153, lat: 48.498028 },
    { x: 46485, y: 331.6, lon: 9.063291, lat: 48.497868 },
    { x: 46502, y: 332.1, lon: 9.063386, lat: 48.497731 },
    { x: 46542, y: 333.2, lon: 9.063531, lat: 48.497383 },
    { x: 46594, y: 334, lon: 9.063758, lat: 48.496937 },
    { x: 46625, y: 334.4, lon: 9.063898, lat: 48.496681 },
    { x: 46658, y: 334.7, lon: 9.064029, lat: 48.496394 },
    { x: 46675, y: 334.8, lon: 9.06409, lat: 48.496245 },
    { x: 46701, y: 335.4, lon: 9.064209, lat: 48.496028 },
    { x: 46719, y: 335.7, lon: 9.064282, lat: 48.495868 },
    { x: 46754, y: 336.5, lon: 9.064433, lat: 48.495571 },
    { x: 46774, y: 336.8, lon: 9.064516, lat: 48.495404 },
    { x: 46797, y: 337, lon: 9.064614, lat: 48.495201 },
    { x: 46829, y: 337.3, lon: 9.064758, lat: 48.494938 },
    { x: 46854, y: 337.6, lon: 9.064861, lat: 48.494716 },
    { x: 46886, y: 337.7, lon: 9.064988, lat: 48.494441 },
    { x: 46897, y: 337.7, lon: 9.065049, lat: 48.494353 },
    { x: 46926, y: 337.4, lon: 9.065139, lat: 48.494101 },
    { x: 46959, y: 337.2, lon: 9.065392, lat: 48.493861 },
    { x: 46963, y: 337.4, lon: 9.065452, lat: 48.493866 },
    { x: 46979, y: 337.9, lon: 9.065645, lat: 48.493804 },
    { x: 47002, y: 338.5, lon: 9.065732, lat: 48.493607 },
    { x: 47027, y: 339.6, lon: 9.065911, lat: 48.493409 },
    { x: 47058, y: 340.9, lon: 9.06611, lat: 48.493164 },
    { x: 47092, y: 342, lon: 9.066286, lat: 48.492885 },
    { x: 47109, y: 342, lon: 9.066362, lat: 48.492736 },
    { x: 47117, y: 342, lon: 9.066404, lat: 48.492679 },
    { x: 47136, y: 342.2, lon: 9.066531, lat: 48.49253 },
    { x: 47175, y: 342, lon: 9.066524, lat: 48.492178 },
    { x: 47193, y: 342, lon: 9.066553, lat: 48.492017 },
    { x: 47202, y: 342.2, lon: 9.066614, lat: 48.491945 },
    { x: 47238, y: 342.6, lon: 9.066905, lat: 48.491688 },
    { x: 47266, y: 342.6, lon: 9.067203, lat: 48.491531 },
    { x: 47285, y: 342.3, lon: 9.067325, lat: 48.491383 },
    { x: 47313, y: 341.9, lon: 9.067482, lat: 48.491152 },
    { x: 47346, y: 341.9, lon: 9.06766, lat: 48.49088 },
    { x: 47378, y: 341.9, lon: 9.067839, lat: 48.490616 },
    { x: 47394, y: 342, lon: 9.067905, lat: 48.490481 },
    { x: 47426, y: 342.6, lon: 9.068057, lat: 48.490212 },
    { x: 47447, y: 342.9, lon: 9.068156, lat: 48.490033 },
    { x: 47472, y: 343.1, lon: 9.06826, lat: 48.489816 },
    { x: 47496, y: 343, lon: 9.068341, lat: 48.489609 },
    { x: 47527, y: 343.4, lon: 9.068423, lat: 48.489336 },
    { x: 47540, y: 343.6, lon: 9.068463, lat: 48.489218 },
    { x: 47562, y: 343.8, lon: 9.068543, lat: 48.489032 },
    { x: 47586, y: 344, lon: 9.068572, lat: 48.488817 },
    { x: 47615, y: 343.8, lon: 9.068575, lat: 48.488558 },
    { x: 47635, y: 343.7, lon: 9.068563, lat: 48.488401 },
    { x: 47658, y: 343.5, lon: 9.068501, lat: 48.488176 },
    { x: 47681, y: 343.6, lon: 9.06831, lat: 48.488004 },
    { x: 47702, y: 343.8, lon: 9.068082, lat: 48.4879 },
    { x: 47725, y: 344.1, lon: 9.067789, lat: 48.487963 },
    { x: 47766, y: 344.8, lon: 9.067224, lat: 48.487983 },
    { x: 47785, y: 345.7, lon: 9.066979, lat: 48.488002 },
    { x: 47809, y: 347.7, lon: 9.066659, lat: 48.488051 },
    { x: 47832, y: 349.1, lon: 9.066366, lat: 48.488122 },
    { x: 47856, y: 352.5, lon: 9.066037, lat: 48.488108 },
    { x: 47862, y: 353.6, lon: 9.065965, lat: 48.488087 },
    { x: 47899, y: 360.9, lon: 9.065483, lat: 48.487993 },
    { x: 47913, y: 364, lon: 9.065312, lat: 48.487947 },
    { x: 47939, y: 370.4, lon: 9.064973, lat: 48.487878 },
    { x: 47943, y: 371.3, lon: 9.06493, lat: 48.487869 },
    { x: 47959, y: 375.5, lon: 9.064725, lat: 48.48782 },
    { x: 47980, y: 379.2, lon: 9.064445, lat: 48.487779 },
    { x: 47985, y: 379.4, lon: 9.064377, lat: 48.487786 },
    { x: 48002, y: 379.1, lon: 9.064165, lat: 48.487848 },
    { x: 48011, y: 380.1, lon: 9.064042, lat: 48.487834 },
    { x: 48026, y: 381.9, lon: 9.063851, lat: 48.487801 },
    { x: 48039, y: 383.5, lon: 9.063692, lat: 48.487758 },
    { x: 48048, y: 384.5, lon: 9.063568, lat: 48.487726 },
    { x: 48070, y: 386.7, lon: 9.063313, lat: 48.487638 },
    { x: 48078, y: 386.4, lon: 9.0632, lat: 48.487665 },
    { x: 48089, y: 386.6, lon: 9.063061, lat: 48.487667 },
    { x: 48111, y: 389.4, lon: 9.062796, lat: 48.487563 },
    { x: 48116, y: 390.3, lon: 9.062773, lat: 48.487523 },
    { x: 48125, y: 391.6, lon: 9.062799, lat: 48.487446 },
    { x: 48128, y: 392.1, lon: 9.062789, lat: 48.487417 },
    { x: 48149, y: 391.3, lon: 9.063068, lat: 48.487425 },
    { x: 48155, y: 391.5, lon: 9.063147, lat: 48.487409 },
    { x: 48158, y: 391.3, lon: 9.06318, lat: 48.487418 },
    { x: 48176, y: 392.6, lon: 9.063399, lat: 48.487345 },
    { x: 48182, y: 393.5, lon: 9.063455, lat: 48.487301 },
    { x: 48192, y: 394.4, lon: 9.063561, lat: 48.487255 },
    { x: 48200, y: 395, lon: 9.063671, lat: 48.487223 },
    { x: 48209, y: 395.5, lon: 9.063774, lat: 48.487197 },
    { x: 48220, y: 396.5, lon: 9.063894, lat: 48.487134 },
    { x: 48243, y: 395.6, lon: 9.064212, lat: 48.487138 },
    { x: 48260, y: 395.7, lon: 9.064435, lat: 48.487111 },
    { x: 48274, y: 394.9, lon: 9.064617, lat: 48.487115 },
    { x: 48293, y: 394.4, lon: 9.064783, lat: 48.486975 },
    { x: 48298, y: 394, lon: 9.06484, lat: 48.486965 },
    { x: 48302, y: 393.4, lon: 9.064901, lat: 48.486971 },
    { x: 48306, y: 393.9, lon: 9.064855, lat: 48.486956 },
    { x: 48339, y: 394.3, lon: 9.064981, lat: 48.486672 },
    { x: 48348, y: 394.6, lon: 9.065002, lat: 48.486591 },
    { x: 48365, y: 395.2, lon: 9.065066, lat: 48.486448 },
    { x: 48380, y: 395.8, lon: 9.065156, lat: 48.486329 },
    { x: 48381, y: 395.8, lon: 9.065158, lat: 48.486316 },
    { x: 48398, y: 395.7, lon: 9.065167, lat: 48.486162 },
    { x: 48401, y: 395.6, lon: 9.065169, lat: 48.486141 },
    { x: 48439, y: 392.6, lon: 9.06541, lat: 48.485837 },
    { x: 48444, y: 392.1, lon: 9.065419, lat: 48.485794 },
    { x: 48468, y: 390.2, lon: 9.065421, lat: 48.485572 },
    { x: 48475, y: 390, lon: 9.065392, lat: 48.485511 },
    { x: 48493, y: 389.7, lon: 9.065252, lat: 48.485378 },
    { x: 48518, y: 389.2, lon: 9.065058, lat: 48.485197 },
    { x: 48542, y: 390.6, lon: 9.064812, lat: 48.485059 },
    { x: 48554, y: 392.4, lon: 9.064655, lat: 48.485045 },
    { x: 48579, y: 392.5, lon: 9.06434, lat: 48.484952 },
    { x: 48586, y: 392.3, lon: 9.064254, lat: 48.484925 },
    { x: 48599, y: 392, lon: 9.064113, lat: 48.484863 },
    { x: 48614, y: 391.5, lon: 9.064003, lat: 48.484749 },
    { x: 48629, y: 391.2, lon: 9.063852, lat: 48.484659 },
    { x: 48645, y: 391.3, lon: 9.063708, lat: 48.484551 },
    { x: 48668, y: 393, lon: 9.063507, lat: 48.484396 },
    { x: 48675, y: 393.6, lon: 9.063466, lat: 48.484335 },
    { x: 48687, y: 394.8, lon: 9.063395, lat: 48.484236 },
    { x: 48691, y: 395.1, lon: 9.063349, lat: 48.484227 },
    { x: 48705, y: 396.4, lon: 9.06333, lat: 48.484103 },
    { x: 48718, y: 397.7, lon: 9.063309, lat: 48.483984 },
    { x: 48727, y: 398.8, lon: 9.063271, lat: 48.483902 },
    { x: 48741, y: 400.5, lon: 9.063205, lat: 48.483786 },
    { x: 48753, y: 401.4, lon: 9.063108, lat: 48.483705 },
    {
      x: 48765,
      y: 401.7,
      lon: 9.063008,
      lat: 48.483616,
      name: 'Golf Court',
      category: 'attraction'
    },
    { x: 48774, y: 402.1, lon: 9.062936, lat: 48.483547 },
    { x: 48783, y: 402.4, lon: 9.062871, lat: 48.483483 },
    { x: 48793, y: 402.9, lon: 9.062773, lat: 48.483417 },
    { x: 48807, y: 403.4, lon: 9.062667, lat: 48.483319 },
    { x: 48824, y: 403.9, lon: 9.062551, lat: 48.483189 },
    { x: 48830, y: 404, lon: 9.062517, lat: 48.48314 },
    { x: 48848, y: 406, lon: 9.062267, lat: 48.483125 },
    { x: 48852, y: 406.4, lon: 9.062217, lat: 48.483117 },
    { x: 48866, y: 406.7, lon: 9.062148, lat: 48.482997 },
    { x: 48870, y: 407, lon: 9.062099, lat: 48.482987 },
    { x: 48875, y: 407.1, lon: 9.062079, lat: 48.482947 },
    { x: 48886, y: 407.9, lon: 9.061962, lat: 48.482877 },
    { x: 48898, y: 409.3, lon: 9.061804, lat: 48.482847 },
    { x: 48916, y: 411.6, lon: 9.061569, lat: 48.482831 },
    { x: 48944, y: 415.4, lon: 9.061199, lat: 48.482883 },
    { x: 48955, y: 417, lon: 9.061053, lat: 48.482898 },
    { x: 48976, y: 420.2, lon: 9.060772, lat: 48.482936 },
    { x: 48978, y: 420.5, lon: 9.060744, lat: 48.482935 },
    { x: 48987, y: 421.8, lon: 9.060636, lat: 48.482959 },
    { x: 48996, y: 423.1, lon: 9.06051, lat: 48.482946 },
    { x: 49003, y: 423.9, lon: 9.060428, lat: 48.482919 },
    { x: 49006, y: 424.4, lon: 9.060384, lat: 48.482922 },
    { x: 49024, y: 426.9, lon: 9.060161, lat: 48.482861 },
    { x: 49040, y: 427.1, lon: 9.060053, lat: 48.482733 },
    { x: 49049, y: 427, lon: 9.060004, lat: 48.482663 },
    { x: 49052, y: 426.9, lon: 9.059987, lat: 48.482632 },
    { x: 49056, y: 427, lon: 9.059962, lat: 48.482604 },
    { x: 49073, y: 426.6, lon: 9.059876, lat: 48.482461 },
    { x: 49074, y: 426.5, lon: 9.059872, lat: 48.482449 },
    { x: 49080, y: 426.8, lon: 9.059823, lat: 48.482416 },
    { x: 49091, y: 427.2, lon: 9.059717, lat: 48.482341 },
    { x: 49106, y: 427.4, lon: 9.059606, lat: 48.482232 },
    { x: 49123, y: 427.3, lon: 9.059482, lat: 48.4821 },
    { x: 49139, y: 428.3, lon: 9.059339, lat: 48.48199 },
    { x: 49144, y: 428.6, lon: 9.059298, lat: 48.481953 },
    { x: 49156, y: 429.6, lon: 9.059171, lat: 48.481889 },
    { x: 49174, y: 431.2, lon: 9.05898, lat: 48.481792 },
    { x: 49184, y: 431.9, lon: 9.058888, lat: 48.481723 },
    { x: 49192, y: 432.1, lon: 9.058834, lat: 48.481666 },
    { x: 49204, y: 433.3, lon: 9.058693, lat: 48.481605 },
    { x: 49216, y: 433.6, lon: 9.058609, lat: 48.481511 },
    { x: 49228, y: 434.2, lon: 9.058509, lat: 48.481432 },
    { x: 49240, y: 434.7, lon: 9.058413, lat: 48.481344 },
    { x: 49252, y: 434.9, lon: 9.058343, lat: 48.481246 },
    { x: 49265, y: 435.1, lon: 9.058249, lat: 48.481142 },
    { x: 49284, y: 436.1, lon: 9.058081, lat: 48.481016 },
    { x: 49293, y: 436.6, lon: 9.058003, lat: 48.480951 },
    { x: 49302, y: 437.2, lon: 9.057925, lat: 48.480895 },
    { x: 49316, y: 438.4, lon: 9.057804, lat: 48.480795 },
    { x: 49325, y: 439.1, lon: 9.057733, lat: 48.480727 },
    { x: 49334, y: 439.9, lon: 9.057652, lat: 48.480675 },
    { x: 49340, y: 439.8, lon: 9.057641, lat: 48.480619 },
    { x: 49350, y: 438.9, lon: 9.057668, lat: 48.480532 },
    { x: 49358, y: 439, lon: 9.057632, lat: 48.480463 },
    { x: 49367, y: 439.1, lon: 9.057582, lat: 48.480388 },
    { x: 49394, y: 440.9, lon: 9.057314, lat: 48.480218 },
    { x: 49414, y: 443.6, lon: 9.057061, lat: 48.480176 },
    { x: 49418, y: 444.1, lon: 9.057007, lat: 48.480169 },
    { x: 49428, y: 444.4, lon: 9.056903, lat: 48.480102 },
    { x: 49453, y: 445.4, lon: 9.056655, lat: 48.47995 },
    { x: 49457, y: 445.9, lon: 9.056608, lat: 48.479956 },
    { x: 49482, y: 447.3, lon: 9.056343, lat: 48.479817 },
    { x: 49487, y: 447.9, lon: 9.056272, lat: 48.479804 },
    {
      x: 49501,
      y: 448.9,
      lon: 9.056095,
      lat: 48.479761,
      name: 'Kressbach',
      category: 'village'
    },
    { x: 49512, y: 449.6, lon: 9.055961, lat: 48.479729 },
    { x: 49522, y: 450.4, lon: 9.055825, lat: 48.479696 },
    { x: 49531, y: 450.7, lon: 9.055745, lat: 48.479639 },
    { x: 49547, y: 452.1, lon: 9.055542, lat: 48.479608 },
    { x: 49557, y: 453.3, lon: 9.055426, lat: 48.479654 },
    { x: 49568, y: 454.5, lon: 9.055275, lat: 48.479668 },
    { x: 49585, y: 456, lon: 9.055047, lat: 48.479673 },
    { x: 49591, y: 456.6, lon: 9.054974, lat: 48.479695 },
    { x: 49617, y: 458.7, lon: 9.054679, lat: 48.479833 },
    { x: 49629, y: 459.4, lon: 9.054534, lat: 48.479885 },
    { x: 49635, y: 459.5, lon: 9.054458, lat: 48.479884 },
    { x: 49644, y: 459.6, lon: 9.054337, lat: 48.479903 },
    { x: 49657, y: 459.8, lon: 9.05417, lat: 48.47993 },
    { x: 49672, y: 460, lon: 9.053967, lat: 48.479903 },
    { x: 49684, y: 460.1, lon: 9.053803, lat: 48.479902 },
    { x: 49689, y: 460.2, lon: 9.053739, lat: 48.479905 },
    { x: 49703, y: 460.6, lon: 9.053557, lat: 48.479914 },
    { x: 49708, y: 460.8, lon: 9.053486, lat: 48.47992 },
    { x: 49721, y: 461.1, lon: 9.053314, lat: 48.479906 },
    { x: 49722, y: 461.2, lon: 9.053294, lat: 48.479913 },
    { x: 49726, y: 461, lon: 9.053332, lat: 48.479899 },
    { x: 49727, y: 461.1, lon: 9.053323, lat: 48.479905 },
    { x: 49728, y: 461.1, lon: 9.053306, lat: 48.479909 },
    { x: 49731, y: 461.2, lon: 9.053269, lat: 48.479911 },
    { x: 49732, y: 461.2, lon: 9.053255, lat: 48.47991 },
    { x: 49744, y: 461.7, lon: 9.053084, lat: 48.479919 },
    { x: 49766, y: 462.5, lon: 9.052786, lat: 48.479931 },
    { x: 49776, y: 462.9, lon: 9.052657, lat: 48.479932 },
    { x: 49797, y: 463.9, lon: 9.052371, lat: 48.479944 },
    { x: 49817, y: 464.8, lon: 9.052107, lat: 48.479957 },
    { x: 49829, y: 466, lon: 9.051944, lat: 48.479965 },
    { x: 49843, y: 467.5, lon: 9.051754, lat: 48.479981 },
    { x: 49862, y: 469.6, lon: 9.051501, lat: 48.480012 },
    { x: 49883, y: 471.6, lon: 9.051219, lat: 48.480028 },
    { x: 49908, y: 472.8, lon: 9.050882, lat: 48.480011 },
    { x: 49929, y: 473.9, lon: 9.050595, lat: 48.480034 },
    { x: 49947, y: 474.5, lon: 9.050356, lat: 48.480044 },
    { x: 49972, y: 474.2, lon: 9.050023, lat: 48.480089 },
    { x: 49998, y: 473.8, lon: 9.049671, lat: 48.480119 },
    { x: 50046, y: 471.2, lon: 9.049022, lat: 48.4801 },
    { x: 50060, y: 470.4, lon: 9.048838, lat: 48.480117 },
    { x: 50134, y: 466.3, lon: 9.04784, lat: 48.480205 },
    { x: 50144, y: 464.9, lon: 9.047711, lat: 48.480221 },
    { x: 50223, y: 453.9, lon: 9.04664, lat: 48.480256 },
    { x: 50292, y: 445.1, lon: 9.045741, lat: 48.480411 },
    { x: 50426, y: 444.3, lon: 9.043948, lat: 48.480545 },
    { x: 50485, y: 443.3, lon: 9.043151, lat: 48.4805 },
    { x: 50513, y: 442.2, lon: 9.042764, lat: 48.480504 },
    { x: 50590, y: 438.2, lon: 9.041733, lat: 48.480526 },
    { x: 50712, y: 427.7, lon: 9.040144, lat: 48.480829 },
    { x: 50726, y: 427.7, lon: 9.039981, lat: 48.480898 },
    { x: 50800, y: 429.4, lon: 9.039141, lat: 48.481265 },
    { x: 50886, y: 425.8, lon: 9.038048, lat: 48.481514 },
    { x: 51045, y: 411.9, lon: 9.036135, lat: 48.482179 },
    {
      x: 51098,
      y: 401.8,
      lon: 9.035697,
      lat: 48.482549,
      name: 'Picnic Area',
      category: 'picnic'
    },
    { x: 51147, y: 393.2, lon: 9.0353, lat: 48.4829 },
    { x: 51281, y: 367.8, lon: 9.034322, lat: 48.483924 },
    { x: 51337, y: 357.9, lon: 9.033835, lat: 48.484301 },
    { x: 51366, y: 353.6, lon: 9.033512, lat: 48.484457 },
    { x: 51425, y: 347.4, lon: 9.032794, lat: 48.484688 },
    { x: 51467, y: 346.1, lon: 9.032456, lat: 48.484994 },
    { x: 51483, y: 345.8, lon: 9.032335, lat: 48.485107 },
    { x: 51534, y: 347, lon: 9.031742, lat: 48.48534 },
    { x: 51540, y: 347.1, lon: 9.031679, lat: 48.485382 },
    { x: 51561, y: 348.1, lon: 9.03142, lat: 48.485445 },
    { x: 51587, y: 348.9, lon: 9.031081, lat: 48.485522 },
    { x: 51618, y: 347.9, lon: 9.030822, lat: 48.485734 },
    { x: 51646, y: 346.1, lon: 9.030779, lat: 48.48599 },
    { x: 51691, y: 344.3, lon: 9.030923, lat: 48.486382 },
    { x: 51705, y: 343.9, lon: 9.030966, lat: 48.486502 },
    { x: 51747, y: 342.7, lon: 9.031122, lat: 48.48687 },
    { x: 51774, y: 341.8, lon: 9.031213, lat: 48.487106 },
    { x: 51810, y: 341.1, lon: 9.031394, lat: 48.487404 },
    { x: 51849, y: 340.5, lon: 9.031605, lat: 48.487722 },
    { x: 51902, y: 339.3, lon: 9.031822, lat: 48.488182 },
    { x: 51923, y: 338.9, lon: 9.031905, lat: 48.48836 },
    { x: 51958, y: 338.2, lon: 9.032049, lat: 48.488663 },
    { x: 51979, y: 337.8, lon: 9.03212, lat: 48.488842 },
    { x: 52019, y: 337.2, lon: 9.032276, lat: 48.48919 },
    { x: 52071, y: 336.3, lon: 9.032437, lat: 48.489638 },
    { x: 52106, y: 335.6, lon: 9.03251, lat: 48.489952 },
    { x: 52144, y: 335.1, lon: 9.032695, lat: 48.490274 },
    { x: 52152, y: 335, lon: 9.032754, lat: 48.490331 },
    { x: 52206, y: 333.9, lon: 9.033175, lat: 48.490726 },
    { x: 52211, y: 333.8, lon: 9.033228, lat: 48.490762 },
    { x: 52252, y: 332.9, lon: 9.033653, lat: 48.490992 },
    { x: 52279, y: 332.3, lon: 9.033934, lat: 48.491149 },
    { x: 52294, y: 332, lon: 9.034072, lat: 48.491244 },
    { x: 52327, y: 331.1, lon: 9.034175, lat: 48.491538 },
    {
      x: 52358,
      y: 330.6,
      lon: 9.03394,
      lat: 48.491768,
      name: 'Weilheim',
      category: 'village'
    },
    { x: 52371, y: 330.6, lon: 9.033802, lat: 48.491833 },
    { x: 52395, y: 329.9, lon: 9.033531, lat: 48.491959 },
    { x: 52432, y: 328.6, lon: 9.033153, lat: 48.492168 },
    { x: 52455, y: 328.3, lon: 9.03292, lat: 48.492306 },
    { x: 52481, y: 327.9, lon: 9.032736, lat: 48.492512 },
    { x: 52511, y: 327.3, lon: 9.032572, lat: 48.492761 },
    { x: 52541, y: 326.4, lon: 9.032358, lat: 48.492987 },
    { x: 52543, y: 326.3, lon: 9.032339, lat: 48.492987 },
    { x: 52544, y: 326.2, lon: 9.032324, lat: 48.492984 },
    { x: 52545, y: 326.2, lon: 9.032311, lat: 48.49299 },
    { x: 52546, y: 326.1, lon: 9.032302, lat: 48.49299 },
    { x: 52546, y: 326.1, lon: 9.032299, lat: 48.49299 },
    { x: 52546, y: 326.1, lon: 9.032298, lat: 48.49299 },
    { x: 52546, y: 326.1, lon: 9.032298, lat: 48.49299 },
    { x: 52546, y: 326.1, lon: 9.032299, lat: 48.492989 },
    { x: 52547, y: 326.1, lon: 9.032297, lat: 48.492995 },
    { x: 52548, y: 326.1, lon: 9.032294, lat: 48.493003 },
    { x: 52549, y: 326, lon: 9.032274, lat: 48.493009 },
    { x: 52550, y: 326, lon: 9.032274, lat: 48.493001 },
    { x: 52551, y: 326, lon: 9.032271, lat: 48.493005 },
    { x: 52555, y: 326.1, lon: 9.032279, lat: 48.493049 },
    { x: 52587, y: 325.6, lon: 9.032103, lat: 48.493311 },
    { x: 52614, y: 325.6, lon: 9.031983, lat: 48.493538 },
    { x: 52644, y: 325.8, lon: 9.031874, lat: 48.493794 },
    { x: 52670, y: 326.1, lon: 9.031823, lat: 48.49403 },
    { x: 52708, y: 326.4, lon: 9.031731, lat: 48.494362 },
    { x: 52721, y: 326.5, lon: 9.031696, lat: 48.494476 },
    { x: 52742, y: 326.5, lon: 9.031675, lat: 48.494671 },
    { x: 52761, y: 326.4, lon: 9.031625, lat: 48.49483 },
    { x: 52767, y: 326.4, lon: 9.031627, lat: 48.49489 },
    { x: 52798, y: 326.3, lon: 9.031552, lat: 48.49516 },
    { x: 52815, y: 326.3, lon: 9.031549, lat: 48.495319 },
    { x: 52853, y: 325.9, lon: 9.031606, lat: 48.495651 },
    { x: 52872, y: 325.5, lon: 9.031578, lat: 48.495822 },
    { x: 52876, y: 325.5, lon: 9.03154, lat: 48.495848 },
    { x: 52888, y: 325.3, lon: 9.031438, lat: 48.495936 },
    { x: 52915, y: 325, lon: 9.031341, lat: 48.49617 },
    { x: 52933, y: 325.1, lon: 9.03128, lat: 48.496322 },
    { x: 52936, y: 325.1, lon: 9.031282, lat: 48.496357 },
    { x: 52972, y: 325.5, lon: 9.031197, lat: 48.496674 },
    { x: 52979, y: 325.5, lon: 9.031155, lat: 48.496617 },
    { x: 53006, y: 325.7, lon: 9.031258, lat: 48.496849 },
    { x: 53017, y: 325.8, lon: 9.031253, lat: 48.496949 },
    { x: 53045, y: 325.7, lon: 9.031305, lat: 48.497197 },
    { x: 53071, y: 325.2, lon: 9.031339, lat: 48.497427 },
    { x: 53086, y: 324.9, lon: 9.031375, lat: 48.497565 },
    { x: 53114, y: 324.3, lon: 9.031316, lat: 48.497812 },
    { x: 53125, y: 324.1, lon: 9.03131, lat: 48.497909 },
    { x: 53162, y: 324.4, lon: 9.031199, lat: 48.498229 },
    { x: 53188, y: 324.5, lon: 9.031031, lat: 48.498442 },
    { x: 53228, y: 324.3, lon: 9.030683, lat: 48.498718 },
    { x: 53260, y: 323.3, lon: 9.030426, lat: 48.498949 },
    { x: 53301, y: 322.1, lon: 9.030084, lat: 48.499241 },
    { x: 53339, y: 321.2, lon: 9.029752, lat: 48.499501 },
    { x: 53376, y: 321.2, lon: 9.029454, lat: 48.499766 },
    { x: 53390, y: 321.2, lon: 9.029338, lat: 48.499863 },
    { x: 53437, y: 321.4, lon: 9.029137, lat: 48.500269 },
    { x: 53456, y: 321.5, lon: 9.029105, lat: 48.500438 },
    { x: 53489, y: 321.9, lon: 9.029033, lat: 48.500723 },
    { x: 53515, y: 322.4, lon: 9.02896, lat: 48.500952 },
    { x: 53548, y: 322.9, lon: 9.028707, lat: 48.501196 },
    { x: 53585, y: 322.8, lon: 9.028251, lat: 48.501343 },
    { x: 53623, y: 322.4, lon: 9.027798, lat: 48.501507 },
    { x: 53634, y: 322.4, lon: 9.027664, lat: 48.501557 },
    { x: 53683, y: 322.3, lon: 9.027103, lat: 48.501781 },
    { x: 53692, y: 322.3, lon: 9.027008, lat: 48.501836 },
    { x: 53712, y: 322.1, lon: 9.026975, lat: 48.502012 },
    { x: 53737, y: 322.4, lon: 9.027, lat: 48.502241 },
    { x: 53764, y: 322.9, lon: 9.026933, lat: 48.50248 },
    { x: 53784, y: 323.2, lon: 9.026843, lat: 48.502649 }
  ]
}
