/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.jdbc.datasource;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import javax.sql.DataSource;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jdbc.CannotGetJdbcConnectionException;
import org.springframework.jdbc.datasource.ConnectionHolder;
import org.springframework.jdbc.datasource.ConnectionProxy;
import org.springframework.jdbc.datasource.DelegatingDataSource;
import org.springframework.jdbc.datasource.SmartDataSource;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.Assert;

public abstract class DataSourceUtils {
    public static final int CONNECTION_SYNCHRONIZATION_ORDER = 1000;
    private static final Log logger = LogFactory.getLog((Class)(class$org$springframework$jdbc$datasource$DataSourceUtils == null ? (class$org$springframework$jdbc$datasource$DataSourceUtils = DataSourceUtils.class$("org.springframework.jdbc.datasource.DataSourceUtils")) : class$org$springframework$jdbc$datasource$DataSourceUtils));
    static /* synthetic */ Class class$org$springframework$jdbc$datasource$DataSourceUtils;

    public static Connection getConnection(DataSource dataSource) throws CannotGetJdbcConnectionException {
        try {
            return DataSourceUtils.doGetConnection(dataSource);
        }
        catch (SQLException sQLException) {
            throw new CannotGetJdbcConnectionException("Could not get JDBC Connection", sQLException);
        }
    }

    public static Connection doGetConnection(DataSource dataSource) throws SQLException {
        Assert.notNull((Object)dataSource, (String)"No DataSource specified");
        ConnectionHolder connectionHolder = (ConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)dataSource));
        if (connectionHolder != null && (connectionHolder.hasConnection() || connectionHolder.isSynchronizedWithTransaction())) {
            connectionHolder.requested();
            if (!connectionHolder.hasConnection()) {
                logger.debug((Object)"Fetching resumed JDBC Connection from DataSource");
                connectionHolder.setConnection(dataSource.getConnection());
            }
            return connectionHolder.getConnection();
        }
        logger.debug((Object)"Fetching JDBC Connection from DataSource");
        Connection connection = dataSource.getConnection();
        if (TransactionSynchronizationManager.isSynchronizationActive()) {
            logger.debug((Object)"Registering transaction synchronization for JDBC Connection");
            ConnectionHolder connectionHolder2 = connectionHolder;
            if (connectionHolder2 == null) {
                connectionHolder2 = new ConnectionHolder(connection);
            } else {
                connectionHolder2.setConnection(connection);
            }
            connectionHolder2.requested();
            TransactionSynchronizationManager.registerSynchronization((TransactionSynchronization)new ConnectionSynchronization(connectionHolder2, dataSource));
            connectionHolder2.setSynchronizedWithTransaction(true);
            if (connectionHolder2 != connectionHolder) {
                TransactionSynchronizationManager.bindResource((Object)dataSource, (Object)((Object)connectionHolder2));
            }
        }
        return connection;
    }

    public static Integer prepareConnectionForTransaction(Connection connection, TransactionDefinition transactionDefinition) throws SQLException {
        Assert.notNull((Object)connection, (String)"No Connection specified");
        if (transactionDefinition != null && transactionDefinition.isReadOnly()) {
            try {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Setting JDBC Connection [" + connection + "] read-only"));
                }
                connection.setReadOnly(true);
            }
            catch (Throwable throwable) {
                logger.debug((Object)"Could not set JDBC Connection read-only", throwable);
            }
        }
        Integer n = null;
        if (transactionDefinition != null && transactionDefinition.getIsolationLevel() != -1) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Changing isolation level of JDBC Connection [" + connection + "] to " + transactionDefinition.getIsolationLevel()));
            }
            n = new Integer(connection.getTransactionIsolation());
            connection.setTransactionIsolation(transactionDefinition.getIsolationLevel());
        }
        return n;
    }

    public static void resetConnectionAfterTransaction(Connection connection, Integer n) {
        Assert.notNull((Object)connection, (String)"No Connection specified");
        try {
            if (n != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Resetting isolation level of JDBC Connection [" + connection + "] to " + n));
                }
                connection.setTransactionIsolation(n);
            }
            if (connection.isReadOnly()) {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Resetting read-only flag of JDBC Connection [" + connection + "]"));
                }
                connection.setReadOnly(false);
            }
        }
        catch (Throwable throwable) {
            logger.debug((Object)"Could not reset JDBC Connection after transaction", throwable);
        }
    }

    public static boolean isConnectionTransactional(Connection connection, DataSource dataSource) {
        if (dataSource == null) {
            return false;
        }
        ConnectionHolder connectionHolder = (ConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)dataSource));
        return connectionHolder != null && DataSourceUtils.connectionEquals(connectionHolder, connection);
    }

    public static void applyTransactionTimeout(Statement statement, DataSource dataSource) throws SQLException {
        DataSourceUtils.applyTimeout(statement, dataSource, 0);
    }

    public static void applyTimeout(Statement statement, DataSource dataSource, int n) throws SQLException {
        Assert.notNull((Object)statement, (String)"No Statement specified");
        Assert.notNull((Object)dataSource, (String)"No DataSource specified");
        ConnectionHolder connectionHolder = (ConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)dataSource));
        if (connectionHolder != null && connectionHolder.hasTimeout()) {
            statement.setQueryTimeout(connectionHolder.getTimeToLiveInSeconds());
        } else if (n > 0) {
            statement.setQueryTimeout(n);
        }
    }

    public static void releaseConnection(Connection connection, DataSource dataSource) {
        try {
            DataSourceUtils.doReleaseConnection(connection, dataSource);
        }
        catch (SQLException sQLException) {
            logger.debug((Object)"Could not close JDBC Connection", (Throwable)sQLException);
        }
        catch (Throwable throwable) {
            logger.debug((Object)"Unexpected exception on closing JDBC Connection", throwable);
        }
    }

    public static void doReleaseConnection(Connection connection, DataSource dataSource) throws SQLException {
        ConnectionHolder connectionHolder;
        if (connection == null) {
            return;
        }
        if (dataSource != null && (connectionHolder = (ConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)dataSource))) != null && DataSourceUtils.connectionEquals(connectionHolder, connection)) {
            connectionHolder.released();
            return;
        }
        if (!(dataSource instanceof SmartDataSource) || ((SmartDataSource)dataSource).shouldClose(connection)) {
            logger.debug((Object)"Returning JDBC Connection to DataSource");
            connection.close();
        }
    }

    private static boolean connectionEquals(ConnectionHolder connectionHolder, Connection connection) {
        if (!connectionHolder.hasConnection()) {
            return false;
        }
        Connection connection2 = connectionHolder.getConnection();
        return connection2 == connection || connection2.equals(connection) || DataSourceUtils.getTargetConnection(connection2).equals(connection);
    }

    public static Connection getTargetConnection(Connection connection) {
        Connection connection2 = connection;
        while (connection2 instanceof ConnectionProxy) {
            connection2 = ((ConnectionProxy)connection2).getTargetConnection();
        }
        return connection2;
    }

    private static int getConnectionSynchronizationOrder(DataSource dataSource) {
        int n = 1000;
        DataSource dataSource2 = dataSource;
        while (dataSource2 instanceof DelegatingDataSource) {
            --n;
            dataSource2 = ((DelegatingDataSource)dataSource2).getTargetDataSource();
        }
        return n;
    }

    static /* synthetic */ Class class$(String string) {
        try {
            return Class.forName(string);
        }
        catch (ClassNotFoundException classNotFoundException) {
            throw new NoClassDefFoundError(classNotFoundException.getMessage());
        }
    }

    private static class ConnectionSynchronization
    extends TransactionSynchronizationAdapter {
        private final ConnectionHolder connectionHolder;
        private final DataSource dataSource;
        private int order;
        private boolean holderActive = true;

        public ConnectionSynchronization(ConnectionHolder connectionHolder, DataSource dataSource) {
            this.connectionHolder = connectionHolder;
            this.dataSource = dataSource;
            this.order = DataSourceUtils.getConnectionSynchronizationOrder(dataSource);
        }

        public int getOrder() {
            return this.order;
        }

        public void suspend() {
            if (this.holderActive) {
                TransactionSynchronizationManager.unbindResource((Object)this.dataSource);
                if (this.connectionHolder.hasConnection() && !this.connectionHolder.isOpen()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                    this.connectionHolder.setConnection(null);
                }
            }
        }

        public void resume() {
            if (this.holderActive) {
                TransactionSynchronizationManager.bindResource((Object)this.dataSource, (Object)((Object)this.connectionHolder));
            }
        }

        public void beforeCompletion() {
            if (!this.connectionHolder.isOpen()) {
                TransactionSynchronizationManager.unbindResource((Object)this.dataSource);
                this.holderActive = false;
                if (this.connectionHolder.hasConnection()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                }
            }
        }

        public void afterCompletion(int n) {
            if (this.holderActive) {
                if (TransactionSynchronizationManager.hasResource((Object)this.dataSource)) {
                    TransactionSynchronizationManager.unbindResource((Object)this.dataSource);
                }
                this.holderActive = false;
                if (this.connectionHolder.hasConnection()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                    this.connectionHolder.setConnection(null);
                }
                this.connectionHolder.reset();
            }
        }
    }
}

