--CUSTOM CLAIMS FUNCTIONS
--https://github.com/supabase-community/supabase-custom-claims/blob/main/install.sql
CREATE OR REPLACE FUNCTION is_claims_admin() RETURNS "bool"
  LANGUAGE "plpgsql" 
  AS $$
  BEGIN
    IF session_user = 'authenticator' THEN
      --------------------------------------------
      -- To disallow any authenticated app users
      -- from editing claims, delete the following
      -- block of code and replace it with:
      -- RETURN FALSE;
      --------------------------------------------
      IF extract(epoch from now()) > coalesce((current_setting('request.jwt.claims', true)::jsonb)->>'exp', '0')::numeric THEN
        return false; -- jwt expired
      END IF; 
      IF coalesce((current_setting('request.jwt.claims', true)::jsonb)->'app_metadata'->'claims_admin', 'false')::bool THEN
        return true; -- user has claims_admin set to true
      ELSE
        return false; -- user does NOT have claims_admin set to true
      END IF;
      --------------------------------------------
      -- End of block 
      --------------------------------------------
    ELSE -- not a user session, probably being called from a trigger or something
      return true;
    END IF;
  END;
$$;

CREATE OR REPLACE FUNCTION get_my_claims() RETURNS "jsonb"
    LANGUAGE "sql" STABLE
    AS $$
  select 
  	coalesce(nullif(current_setting('request.jwt.claims', true), '')::jsonb -> 'app_metadata', '{}'::jsonb)::jsonb
$$;

CREATE OR REPLACE FUNCTION get_my_claim(claim TEXT) RETURNS "jsonb"
    LANGUAGE "sql" STABLE
    AS $$
  select 
  	coalesce(nullif(current_setting('request.jwt.claims', true), '')::jsonb -> 'app_metadata' -> claim, null)
$$;

CREATE OR REPLACE FUNCTION get_claims(uid uuid) RETURNS "jsonb"
    LANGUAGE "plpgsql" SECURITY DEFINER SET search_path = public
    AS $$
    DECLARE retval jsonb;
    BEGIN
      IF NOT is_claims_admin() THEN
          RETURN '{"error":"access denied"}'::jsonb;
      ELSE
        select raw_app_meta_data from auth.users into retval where id = uid::uuid;
        return retval;
      END IF;
    END;
$$;

CREATE OR REPLACE FUNCTION get_claim(uid uuid, claim text) RETURNS "jsonb"
    LANGUAGE "plpgsql" SECURITY DEFINER SET search_path = public
    AS $$
    DECLARE retval jsonb;
    BEGIN
      IF NOT is_claims_admin() THEN
          RETURN '{"error":"access denied"}'::jsonb;
      ELSE
        select coalesce(raw_app_meta_data->claim, null) from auth.users into retval where id = uid::uuid;
        return retval;
      END IF;
    END;
$$;

CREATE OR REPLACE FUNCTION set_claim(uid uuid, claim text, value jsonb) RETURNS "text"
    LANGUAGE "plpgsql" SECURITY DEFINER SET search_path = public
    AS $$
    BEGIN
      IF NOT is_claims_admin() THEN
          RETURN 'error: access denied';
      ELSE        
        update auth.users set raw_app_meta_data = 
          raw_app_meta_data || 
            json_build_object(claim, value)::jsonb where id = uid;
        return 'OK';
      END IF;
    END;
$$;

CREATE OR REPLACE FUNCTION delete_claim(uid uuid, claim text) RETURNS "text"
    LANGUAGE "plpgsql" SECURITY DEFINER SET search_path = public
    AS $$
    BEGIN
      IF NOT is_claims_admin() THEN
          RETURN 'error: access denied';
      ELSE        
        update auth.users set raw_app_meta_data = 
          raw_app_meta_data - claim where id = uid;
        return 'OK';
      END IF;
    END;
$$;

CREATE OR REPLACE FUNCTION log(message text) RETURNS VOID
  LANGUAGE "plpgsql"
  AS $$ 
  BEGIN
    RAISE LOG '%', message;
  END;
$$ ;

-- CREATE TABLE FOR ORGANIZATIONS
DROP TABLE IF EXISTS orgs;

CREATE TABLE orgs (
  id bigint generated BY DEFAULT AS identity primary key,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT timezone('utc' :: text, now()) NOT NULL,
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT timezone('utc' :: text, now()) NOT NULL,
  name VARCHAR UNIQUE NOT NULL,
  logo VARCHAR
);

--ACTIVATE RLS AND CREATE POLICIES
ALTER TABLE orgs ENABLE ROW LEVEL SECURITY;

CREATE POLICY "ONLY SUPER CAN ACCESS ORGANIZATIONS" 
ON orgs FOR ALL 
USING (get_my_claim('role') = '"super"');

CREATE OR REPLACE FUNCTION auth.org() RETURNS TEXT 
  LANGUAGE SQL
  AS $$
    SELECT
    NULLIF(
      ((current_setting('request.jwt.claims', TRUE) :: jsonb 
      ->> 'app_metadata') :: jsonb ->> 'org'),''
    ) :: TEXT
$$ ;

CREATE OR REPLACE FUNCTION auth.app_role() RETURNS TEXT
  LANGUAGE SQL
  AS $$
    SELECT
      NULLIF(
        ((current_setting('request.jwt.claims', TRUE) :: jsonb
         ->> 'app_metadata') :: jsonb ->> 'role'),''
      ) :: TEXT
$$;

-- USER TRACKING TABLE
CREATE TABLE IF NOT EXISTS tracking (
  id bigint generated BY DEFAULT AS identity primary key,
  TIMESTAMP TIMESTAMP WITH TIME ZONE DEFAULT timezone('utc' :: text, now()) NOT NULL,
  user jsonb,
  event jsonb
);

-- STORAGE RULES
-- DROP POLICY IF EXISTS "SUPER ACCES TO OBJECTS" ON storage.objects;
CREATE POLICY "SUPER ACCES TO OBJECTS" ON storage.objects
AS PERMISSIVE FOR ALL
TO authenticated
USING (get_my_claim('role') = '"super"');

-- DROP POLICY IF EXISTS "SUPER ACCES TO BUCKETS" ON storage.objects;
CREATE POLICY "SUPER ACCES TO BUCKETS" ON storage.buckets
AS PERMISSIVE FOR ALL
TO authenticated
USING (get_my_claim('role') = '"super"');

NOTIFY pgrst, 'reload schema';