/*!
 * Jodit Editor (https://xdsoft.net/jodit/)
 * Released under MIT see LICENSE.txt in the project root for license information.
 * Copyright (c) 2013-2024 Valeriy Chupurnov. All rights reserved. https://xdsoft.net
 */

/**
 * @module modules/dialog
 */

import type { IDialog } from 'jodit/types';
import { isFunction } from 'jodit/core/helpers/checker/is-function';
import { attr } from 'jodit/core/helpers/utils/attr';
import { Button } from 'jodit/core/ui/button/button/button';

import { Dialog } from './dialog';

/**
 * Show `Prompt` dialog. Work without Jodit object
 *
 * @param msg - Dialog content
 * @param title - Title or callback
 * @param callback - callback. The first argument is the value entered
 * @param placeholder - Placeholder for input
 * @example
 * ```javascript
 * Jodit.Prompt("Enter your name", "Prompt Dialog", function (name) {
 *     if (name.length < 3) {
 *         Jodit.Alert("The name must be at least 3 letters");
 *         return false;
 *     }
 *     // do something
 * });
 * ```
 */
export function Prompt(
	this: IDialog | unknown,
	msg: string,
	title: string | (() => false | void) | undefined,
	callback: (value: string) => false | void,
	placeholder?: string,
	defaultValue?: string
): IDialog {
	const dialog =
			this instanceof Dialog
				? this
				: new Dialog({ closeOnClickOverlay: true }),
		cancelButton = Button(dialog, 'cancel', 'Cancel'),
		okButton = Button(dialog, 'ok', 'Ok'),
		form = dialog.c.element('form', {
			class: 'jodit-dialog_prompt'
		}),
		inputElement = dialog.c.element('input', {
			autofocus: true,
			class: 'jodit-input'
		}),
		labelElement = dialog.c.element('label');

	if (isFunction(title)) {
		callback = title;
		title = undefined;
	}

	if (placeholder) {
		attr(inputElement, 'placeholder', placeholder);
	}

	labelElement.appendChild(dialog.c.text(msg));

	form.appendChild(labelElement);
	form.appendChild(inputElement);

	cancelButton.onAction(dialog.close);

	const onclick = (): void => {
		if (
			!callback ||
			!isFunction(callback) ||
			callback(inputElement.value) !== false
		) {
			dialog.close();
		}
	};

	okButton.onAction(onclick);

	dialog.e.on(form, 'submit', () => {
		onclick();
		return false;
	});

	dialog.setFooter([okButton, cancelButton]);

	dialog.open(form, (title as string) || '&nbsp;', true, true);
	inputElement.focus();

	if (defaultValue !== undefined && defaultValue.length) {
		inputElement.value = defaultValue;
		inputElement.select();
	}

	return dialog;
}
