#pragma once

#include <ntdll/ntdll.h>

// https://github.com/x64dbg/ScyllaHide/blob/6817d32581b7a420322f34e36b1a1c8c3e4b434c/Scylla/Win32kSyscalls.h

/*
 * The tables below were generated for each OS version using a modified version of wscg64 (https://github.com/hfiref0x/SyscallTables).
 * They were then converted to the format below to save space. std::map and std::string are banned in this file, they add about 600KB.
 * The syscalls are sorted by name only, so lookup is O(N) and done with string comparisons. This is okay because lookups are rare.
 *
 * All OS versions have separate syscall tables for x86 and x64, except 2600 which only exists as x86. WOW64 processes use the x64 table.
 *
 * Versions 14393 and later are not included because they come with win32u.dll, which exports win32k syscalls like ntdll does for ntoskrnl.
 */

// Space saving hack. The longest name here is 58 chars, so UCHAR is OK

typedef struct _WIN32K_SYSCALL_INFO
{
    const char* Name;
    struct
    {
        // Index 0 is x86, index 1 is WOW64 and x64.
        // None of the syscall numbers exceed 64K, so store them as SHORTs
        SHORT Index2600;        // XP (x86 only)
        SHORT Index3790[2];     // Server 2003 / XP x64
        SHORT Index6000[2];     // Vista
        SHORT Index7601[2];     // 7
        SHORT Index9200[2];     // 8
        SHORT Index9600[2];     // 8.1
        SHORT Index10240[2];    // 10.0.10240.0
        SHORT Index10586[2];    // 10.0.10586.0
    } s;

    // A value of -1 means Win32k does not have the syscall on this OS/bitness combination.
    int GetSyscallIndex(USHORT osBuildNumber, bool nativeX86) const
    {
        return (int)(osBuildNumber == 2600
                     ? s.Index2600
                     : ((osBuildNumber >= 3790 && osBuildNumber <= 3800)
                        ? s.Index3790[nativeX86 ? 0 : 1]
                        : ((osBuildNumber >= 6000 && osBuildNumber <= 6002)
                           ? s.Index6000[nativeX86 ? 0 : 1]
                           : ((osBuildNumber == 7600 || osBuildNumber == 7601)
                              ? s.Index7601[nativeX86 ? 0 : 1]
                              : (osBuildNumber == 9200
                                 ? s.Index9200[nativeX86 ? 0 : 1]
                                 : (osBuildNumber == 9600
                                    ? s.Index9600[nativeX86 ? 0 : 1]
                                    : (osBuildNumber == 10240
                                       ? s.Index10240[nativeX86 ? 0 : 1]
                                       : (osBuildNumber == 10586
                                          ? s.Index10586[nativeX86 ? 0 : 1]
                                          : -1))))))));
    }
} WIN32K_SYSCALL_INFO, * PWIN32K_SYSCALL_INFO;

#define MIN_WIN32K_SYSCALL_NUM          4096
#define MAX_WIN32K_SYSCALL_NUM          5230

static const WIN32K_SYSCALL_INFO Win32kSyscalls[] =
{
    { "NtBindCompositionSurface", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4994, 4383 }, { 5012, 4384 }, { 5069, 4385 }, { 5071, 4385 } } },
    { "NtCompositionInputThread", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5030, 4385 }, { 5093, 4386 }, { 5095, 4386 } } },
    { "NtCompositionSetDropTarget", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 4387 }, { 5098, 4387 } } },
    { "NtCreateCompositionInputSink", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5025, 4386 }, { 5086, 4387 }, { 5088, 4388 } } },
    { "NtCreateCompositionSurfaceHandle", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4992, 4384 }, { 5010, 4387 }, { 5067, 4388 }, { 5069, 4389 } } },
    { "NtCreateImplicitCompositionInputSink", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5087, 4389 }, { 5089, 4390 } } },
    { "NtDCompositionAddCrossDeviceVisualChild",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5033, 4385 }, { 5066, 4388 }, { 5134, 4390 }, { 5137, 4391 } } },
    { "NtDCompositionAddVisualChild", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5034, 4386 }, { 5067, 4389 }, { 5135, 4391 }, { 5138, 4392 } } },
    { "NtDCompositionAttachMouseWheelToHwnd", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 4393 }, { 5168, 4393 } } },
    { "NtDCompositionBeginFrame", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5008, 4387 }, { 5037, 4390 }, { 5105, 4392 }, { 5108, 4394 } } },
    { "NtDCompositionCapturePointer", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5162, 4393 }, { 5164, 4395 } } },
    { "NtDCompositionCommitChannel",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5023, 4388 }, { 5052, 4391 }, { 5120, 4394 }, { 5123, 4396 } } },
    { "NtDCompositionConfirmFrame",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5009, 4389 }, { 5038, 4392 }, { 5106, 4395 }, { 5109, 4397 } } },
    { "NtDCompositionConnectPipe",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5050, 4390 }, { 5088, 4393 }, { 5155, 4396 }, { 5157, 4398 } } },
    { "NtDCompositionCreateAndBindSharedSection", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5085, 4394 }, { 5152, 4397 }, { 5154, 4399 } } },
    { "NtDCompositionCreateChannel",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5044, 4391 }, { 5080, 4395 }, { 5148, 4398 }, { 5151, 4400 } } },
    { "NtDCompositionCreateConnection",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5048, 4396 }, { 5116, 4399 }, { 5119, 4401 } } },
    { "NtDCompositionCreateConnectionContext",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5019, 4392 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtDCompositionCreateDwmChannel",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5043, 4393 }, { 5079, 4397 }, { 5147, 4400 }, { 5150, 4402 } } },
    { "NtDCompositionCreateResource", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5042, 4394 }, { 5076, 4398 }, { 5144, 4401 }, { 5147, 4403 } } },
    { "NtDCompositionCurrentBatchId", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5029, 4395 }, { 5062, 4399 }, { 5130, 4402 }, { 5133, 4404 } } },
    { "NtDCompositionDestroyChannel", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5020, 4396 }, { 5049, 4400 }, { 5117, 4403 }, { 5120, 4405 } } },
    { "NtDCompositionDestroyConnection",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5043, 4401 }, { 5111, 4404 }, { 5114, 4406 } } },
    { "NtDCompositionDestroyConnectionContext",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5014, 4397 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtDCompositionDiscardFrame",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5011, 4398 }, { 5040, 4402 }, { 5108, 4405 }, { 5111, 4407 } } },
    { "NtDCompositionDuplicateHandleToProcess",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5090, 4403 }, { 5158, 4406 }, { 5160, 4408 } } },
    { "NtDCompositionDuplicateSwapchainHandleToDwm",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5165, 4407 }, { 5167, 4409 } } },
    { "NtDCompositionDwmSyncFlush",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5046, 4399 }, { 5082, 4404 }, { -1, -1 }, { -1, -1 } } },
    { "NtDCompositionEnableDDASupport",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5085, 4408 }, { 5087, 4410 } } },
    { "NtDCompositionEnableMMCSS",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5083, 4409 }, { 5085, 4411 } } },
    { "NtDCompositionGetAnimationTime",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5160, 4410 }, { 5162, 4412 } } },
    { "NtDCompositionGetChannels",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5049, 4400 }, { 5087, 4405 }, { 5154, 4411 }, { 5156, 4413 } } },
    { "NtDCompositionGetConnectionBatch", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5044, 4406 }, { 5112, 4412 }, { 5115, 4414 } } },
    { "NtDCompositionGetConnectionContextBatch",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5015, 4401 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtDCompositionGetDeletedResources",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5017, 4402 }, { 5046, 4407 }, { 5114, 4413 }, { 5117, 4415 } } },
    { "NtDCompositionGetFrameLegacyTokens",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5013, 4403 }, { 5042, 4408 }, { 5110, 4414 }, { 5113, 4416 } } },
    { "NtDCompositionGetFrameStatistics", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5016, 4404 }, { 5045, 4409 }, { 5113, 4415 }, { 5116, 4417 } } },
    { "NtDCompositionGetFrameSurfaceUpdates", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5012, 4405 }, { 5041, 4410 }, { 5109, 4416 }, { 5112, 4418 } } },
    { "NtDCompositionOpenSharedResource", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5077, 4411 }, { 5145, 4417 }, { 5148, 4419 } } },
    { "NtDCompositionOpenSharedResourceHandle",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5078, 4412 }, { 5146, 4418 }, { 5149, 4420 } } },
    { "NtDCompositionReferenceSharedResourceOnDwmChannel",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5083, 4413 }, { 5150, 4419 }, { 5153, 4421 } } },
    { "NtDCompositionRegisterThumbnailVisual",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5089, 4414 }, { 5156, 4420 }, { 5158, 4422 } } },
    { "NtDCompositionRegisterVirtualDesktopVisual",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5157, 4421 }, { 5159, 4423 } } },
    { "NtDCompositionReleaseAllResources",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5021, 4406 }, { 5050, 4415 }, { 5118, 4422 }, { 5121, 4424 } } },
    { "NtDCompositionReleaseResource",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5030, 4407 }, { 5063, 4416 }, { 5131, 4423 }, { 5134, 4425 } } },
    { "NtDCompositionRemoveCrossDeviceVisualChild",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5031, 4408 }, { 5064, 4417 }, { 5132, 4424 }, { 5135, 4426 } } },
    { "NtDCompositionRemoveVisualChild",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5032, 4409 }, { 5065, 4418 }, { 5133, 4425 }, { 5136, 4427 } } },
    { "NtDCompositionReplaceVisualChildren",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5035, 4410 }, { 5068, 4419 }, { 5136, 4426 }, { 5139, 4428 } } },
    { "NtDCompositionRetireFrame",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5010, 4411 }, { 5039, 4420 }, { 5107, 4427 }, { 5110, 4429 } } },
    { "NtDCompositionSetChannelCallbackId",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5163, 4428 }, { 5165, 4430 } } },
    { "NtDCompositionSetChannelCommitCompletionEvent",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5025, 4412 }, { 5054, 4421 }, { 5122, 4429 }, { 5125, 4431 } } },
    { "NtDCompositionSetDebugCounter",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5086, 4422 }, { 5153, 4430 }, { 5155, 4432 } } },
    { "NtDCompositionSetResourceAnimationProperty",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5036, 4413 }, { 5069, 4423 }, { 5137, 4431 }, { 5140, 4433 } } },
    { "NtDCompositionSetResourceBufferProperty",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5039, 4414 }, { 5072, 4424 }, { 5140, 4432 }, { 5143, 4434 } } },
    { "NtDCompositionSetResourceCallbackId",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5164, 4433 }, { 5166, 4435 } } },
    { "NtDCompositionSetResourceDeletedNotificationTag",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5018, 4415 }, { 5047, 4425 }, { 5115, 4434 }, { 5118, 4436 } } },
    { "NtDCompositionSetResourceFloatProperty",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5041, 4416 }, { 5074, 4426 }, { 5142, 4435 }, { 5145, 4437 } } },
    { "NtDCompositionSetResourceHandleProperty",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5075, 4427 }, { 5143, 4436 }, { 5146, 4438 } } },
    { "NtDCompositionSetResourceIntegerProperty", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5040, 4417 }, { 5073, 4428 }, { 5141, 4437 }, { 5144, 4439 } } },
    { "NtDCompositionSetResourceReferenceArrayProperty",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5037, 4418 }, { 5070, 4429 }, { 5138, 4438 }, { 5141, 4440 } } },
    { "NtDCompositionSetResourceReferenceProperty",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5038, 4419 }, { 5071, 4430 }, { 5139, 4439 }, { 5142, 4441 } } },
    { "NtDCompositionSetVisualInputSink", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5159, 4440 }, { 5161, 4442 } } },
    { "NtDCompositionSignalGpuFence", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5048, 4420 }, { 5084, 4431 }, { 5151, 4441 }, { -1, -1 } } },
    { "NtDCompositionSubmitDWMBatch", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5022, 4421 }, { 5051, 4432 }, { 5119, 4442 }, { 5122, 4443 } } },
    { "NtDCompositionSynchronize",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5045, 4422 }, { 5081, 4433 }, { 5149, 4443 }, { 5152, 4444 } } },
    { "NtDCompositionTelemetryAnimationScenarioBegin",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5059, 4434 }, { 5127, 4444 }, { 5130, 4445 } } },
    { "NtDCompositionTelemetryAnimationScenarioReference",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5060, 4435 }, { 5128, 4445 }, { 5131, 4446 } } },
    { "NtDCompositionTelemetryAnimationScenarioUnreference",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5061, 4436 }, { 5129, 4446 }, { 5132, 4447 } } },
    { "NtDCompositionTelemetrySetApplicationId",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5058, 4437 }, { 5126, 4447 }, { 5129, 4448 } } },
    { "NtDCompositionTelemetryTouchInteractionBegin", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5026, 4423 }, { 5055, 4438 }, { 5123, 4448 }, { 5126, 4449 } } },
    { "NtDCompositionTelemetryTouchInteractionEnd",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5028, 4424 }, { 5057, 4439 }, { 5125, 4449 }, { 5128, 4450 } } },
    { "NtDCompositionTelemetryTouchInteractionUpdate",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5027, 4425 }, { 5056, 4440 }, { 5124, 4450 }, { 5127, 4451 } } },
    { "NtDCompositionUpdatePointerCapture",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5161, 4451 }, { 5163, 4452 } } },
    { "NtDCompositionValidateAndReferenceSystemVisualForHwndTarget",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5047, 4426 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtDCompositionWaitForChannel", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5024, 4427 }, { 5053, 4441 }, { 5121, 4452 }, { 5124, 4453 } } },
    { "NtDesktopCaptureBits", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5082, 4453 }, { 5084, 4454 } } },
    { "NtDuplicateCompositionInputSink",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5026, 4442 }, { 5088, 4454 }, { 5090, 4455 } } },
    { "NtGdiAbortDoc",    { 4096, { 4096, 4391 }, { 4096, 4392 }, { 4096, 4383 }, { 4412, 4428 }, { 4414, 4443 }, { 4415, 4455 }, { 4417, 4456 } } },
    { "NtGdiAbortPath",   { 4097, { 4097, -1 }, { 4097, 4393 }, { 4097, 4384 }, { 4411, 4429 }, { 4413, 4444 }, { 4414, 4456 }, { 4416, 4457 } } },
    { "NtGdiAddEmbFontToDC",  { 4310, { 4309, 4393 }, { 4318, 4394 }, { 4320, 4385 }, { 4186, 4430 }, { 4187, 4445 }, { 4188, 4457 }, { 4188, 4458 } } },
    { "NtGdiAddFontMemResourceEx",    { 4100, { 4100, -1 }, { 4100, 4375 }, { 4100, 4368 }, { 4408, 4368 }, { 4410, 4369 }, { 4411, 4370 }, { 4413, 4370 } } },
    { "NtGdiAddFontResourceW",    { 4098, { 4098, -1 }, { 4098, 4395 }, { 4098, 4386 }, { 4410, 4431 }, { 4412, 4446 }, { 4413, 4458 }, { 4415, 4459 } } },
    { "NtGdiAddRemoteFontToDC",   { 4099, { 4099, 4395 }, { 4099, 4396 }, { 4099, 4387 }, { 4409, 4432 }, { 4411, 4447 }, { 4412, 4459 }, { 4414, 4460 } } },
    { "NtGdiAddRemoteMMInstanceToDC", { 4102, { 4102, 4396 }, { 4102, 4397 }, { 4102, 4388 }, { 4406, 4433 }, { 4408, 4448 }, { 4409, 4460 }, { 4411, 4461 } } },
    { "NtGdiAlphaBlend",  { 4103, { 4103, -1 }, { 4103, 4222 }, { 4103, 4220 }, { 4405, 4220 }, { 4407, 4221 }, { 4408, 4222 }, { 4410, 4222 } } },
    { "NtGdiAngleArc",    { 4104, { 4104, -1 }, { 4104, 4398 }, { 4104, 4389 }, { 4404, 4434 }, { 4406, 4449 }, { 4407, 4461 }, { 4409, 4462 } } },
    { "NtGdiAnyLinkedFonts",  { 4105, { 4105, 4398 }, { 4105, 4399 }, { 4105, 4390 }, { 4403, 4435 }, { 4405, 4450 }, { 4406, 4462 }, { 4408, 4463 } } },
    { "NtGdiArcInternal", { 4107, { 4107, -1 }, { 4107, 4400 }, { 4107, 4391 }, { 4401, 4436 }, { 4403, 4451 }, { 4404, 4463 }, { 4406, 4464 } } },
    { "NtGdiBeginGdiRendering",   { -1, { -1, -1 }, { -1, -1 }, { 4108, 4397 }, { 4400, 4442 }, { 4402, 4457 }, { 4403, 4469 }, { 4405, 4470 } } },
    { "NtGdiBeginPath",   { 4108, { 4108, 4367 }, { 4108, 4368 }, { 4109, 4361 }, { 4399, 4361 }, { 4401, 4362 }, { 4402, 4363 }, { 4404, 4363 } } },
    { "NtGdiBitBlt",  { 4109, { 4109, -1 }, { 4109, 4104 }, { 4110, 4104 }, { 4398, 4105 }, { 4400, 4106 }, { 4401, 4107 }, { 4403, 4107 } } },
    { "NtGdiBRUSHOBJ_DeleteRbrush",   { 4760, { 4756, 4400 }, { 4793, 4401 }, { 4825, 4392 }, { 4865, 4437 }, { 4874, 4452 }, { 4886, 4464 }, { 4887, 4465 } } },
    { "NtGdiBRUSHOBJ_hGetColorTransform", { 4733, { 4729, 4401 }, { 4766, 4402 }, { 4798, 4393 }, { 4835, 4438 }, { 4844, 4453 }, { 4856, 4465 }, { 4857, 4466 } } },
    { "NtGdiBRUSHOBJ_pvAllocRbrush",  { 4731, { 4727, 4402 }, { 4764, 4403 }, { 4796, 4394 }, { 4837, 4439 }, { 4846, 4454 }, { 4858, 4466 }, { 4859, 4467 } } },
    { "NtGdiBRUSHOBJ_pvGetRbrush",    { 4732, { 4728, 4403 }, { 4765, 4404 }, { 4797, 4395 }, { 4836, 4440 }, { 4845, 4455 }, { 4857, 4467 }, { 4858, 4468 } } },
    { "NtGdiBRUSHOBJ_ulGetBrushColor",    { 4730, { 4726, -1 }, { 4763, 4405 }, { 4795, 4396 }, { 4838, 4441 }, { 4847, 4456 }, { 4859, 4468 }, { 4860, 4469 } } },
    { "NtGdiCancelDC",    { 4110, { 4110, 4408 }, { 4110, 4409 }, { 4111, 4401 }, { 4397, 4446 }, { 4399, 4461 }, { 4400, 4473 }, { 4402, 4474 } } },
    { "NtGdiChangeGhostFont", { 4309, { 4308, 4409 }, { 4317, 4410 }, { 4319, 4402 }, { 4187, 4447 }, { 4188, 4462 }, { 4189, 4474 }, { 4189, 4475 } } },
    { "NtGdiCheckBitmapBits", { 4111, { 4111, -1 }, { 4111, 4411 }, { 4112, 4403 }, { 4396, 4448 }, { 4398, 4463 }, { 4399, 4475 }, { 4401, 4476 } } },
    { "NtGdiClearBitmapAttributes",   { 4113, { 4113, 4411 }, { 4113, 4412 }, { 4114, 4404 }, { 4394, 4449 }, { 4396, 4464 }, { 4397, 4476 }, { 4399, 4477 } } },
    { "NtGdiClearBrushAttributes",    { 4114, { 4361, 4412 }, { 4114, 4413 }, { 4405, 4405 }, { 4393, 4450 }, { 4395, 4465 }, { 4396, 4477 }, { 4398, 4478 } } },
    { "NtGdiCLIPOBJ_bEnum",   { 4724, { 4720, 4405 }, { 4757, 4406 }, { 4789, 4398 }, { 4834, 4443 }, { 4843, 4458 }, { 4855, 4470 }, { 4856, 4471 } } },
    { "NtGdiCLIPOBJ_cEnumStart",  { 4725, { 4721, -1 }, { 4758, 4407 }, { 4790, 4399 }, { 4833, 4444 }, { 4842, 4459 }, { 4854, 4471 }, { 4855, 4472 } } },
    { "NtGdiCLIPOBJ_ppoGetPath",  { 4726, { 4722, 4407 }, { 4759, 4408 }, { 4791, 4400 }, { 4832, 4445 }, { 4841, 4460 }, { 4853, 4472 }, { 4854, 4473 } } },
    { "NtGdiCloseFigure", { 4112, { 4112, -1 }, { 4112, 4358 }, { 4113, 4352 }, { 4395, 4352 }, { 4397, 4353 }, { 4398, 4354 }, { 4400, 4354 } } },
    { "NtGdiColorCorrectPalette", { 4115, { 4115, -1 }, { 4115, 4414 }, { 4116, 4406 }, { 4392, 4451 }, { 4394, 4466 }, { 4395, 4478 }, { 4397, 4479 } } },
    { "NtGdiCombineRgn",  { 4116, { 4116, 4147 }, { 4116, 4148 }, { 4117, 4148 }, { 4391, 4149 }, { 4393, 4150 }, { 4394, 4151 }, { 4396, 4151 } } },
    { "NtGdiCombineTransform",    { 4117, { 4117, 4340 }, { 4117, 4341 }, { 4118, 4336 }, { 4390, 4336 }, { 4392, 4337 }, { 4393, 4338 }, { 4395, 4338 } } },
    { "NtGdiComputeXformCoefficients",    { 4118, { 4118, 4236 }, { 4118, 4237 }, { 4119, 4235 }, { 4389, 4235 }, { 4391, 4236 }, { 4392, 4237 }, { 4394, 4237 } } },
    { "NtGdiConfigureOPMProtectedOutput", { -1, { -1, -1 }, { 4119, 4415 }, { 4120, 4407 }, { 4388, 4452 }, { 4390, 4467 }, { 4391, 4479 }, { 4393, 4480 } } },
    { "NtGdiConsoleTextOut",  { 4119, { 4119, 4205 }, { 4120, 4206 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiConvertMetafileRect", { 4120, { 4120, 4414 }, { 4121, 4416 }, { 4121, 4408 }, { 4387, 4453 }, { 4389, 4468 }, { 4390, 4480 }, { 4392, 4481 } } },
    { "NtGdiCreateBitmap",    { 4121, { 4121, -1 }, { 4122, 4205 }, { 4122, 4205 }, { 4386, 4205 }, { 4388, 4206 }, { 4389, 4207 }, { 4391, 4207 } } },
    { "NtGdiCreateBitmapFromDxSurface",   { -1, { -1, -1 }, { -1, -1 }, { 4123, 4409 }, { 4385, 4454 }, { 4387, 4469 }, { 4388, 4481 }, { 4390, 4482 } } },
    { "NtGdiCreateBitmapFromDxSurface2",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4384, 4455 }, { 4386, 4470 }, { 4387, 4482 }, { 4389, 4483 } } },
    { "NtGdiCreateClientObj", { 4122, { 4122, 4286 }, { 4123, 4287 }, { 4124, 4282 }, { 4383, 4282 }, { 4385, 4283 }, { 4386, 4284 }, { 4388, 4284 } } },
    { "NtGdiCreateColorSpace",    { 4123, { 4123, 4388 }, { 4124, 4389 }, { 4125, 4380 }, { 4382, 4380 }, { 4384, 4381 }, { 4385, 4382 }, { 4387, 4382 } } },
    { "NtGdiCreateColorTransform",    { 4124, { 4124, -1 }, { 4125, 4417 }, { 4126, 4410 }, { 4381, 4456 }, { 4383, 4471 }, { 4384, 4483 }, { 4386, 4484 } } },
    { "NtGdiCreateCompatibleBitmap",  { 4125, { 4125, 4170 }, { 4126, 4171 }, { 4127, 4171 }, { 4380, 4172 }, { 4382, 4173 }, { 4383, 4174 }, { 4385, 4174 } } },
    { "NtGdiCreateCompatibleDC",  { 4126, { 4126, 4180 }, { 4127, 4181 }, { 4128, 4181 }, { 4379, 4181 }, { 4381, 4182 }, { 4382, 4183 }, { 4384, 4183 } } },
    { "NtGdiCreateDIBBrush",  { 4127, { 4127, -1 }, { 4128, 4214 }, { 4129, 4212 }, { 4378, 4212 }, { 4380, 4213 }, { 4381, 4214 }, { 4383, 4214 } } },
    { "NtGdiCreateDIBitmapInternal",  { 4128, { 4128, -1 }, { 4129, 4257 }, { 4130, 4255 }, { 4377, 4255 }, { 4379, 4256 }, { 4380, 4257 }, { 4382, 4257 } } },
    { "NtGdiCreateDIBSection",    { 4129, { 4129, -1 }, { 4130, 4252 }, { 4131, 4250 }, { 4376, 4250 }, { 4378, 4251 }, { 4379, 4252 }, { 4381, 4252 } } },
    { "NtGdiCreateEllipticRgn",   { 4130, { 4130, 4416 }, { 4131, 4418 }, { 4132, 4411 }, { 4374, 4457 }, { 4376, 4472 }, { 4377, 4484 }, { 4379, 4485 } } },
    { "NtGdiCreateHalftonePalette",   { 4131, { 4131, 4337 }, { 4132, 4338 }, { 4133, 4333 }, { 4373, 4333 }, { 4375, 4334 }, { 4376, 4335 }, { 4378, 4335 } } },
    { "NtGdiCreateHatchBrushInternal",    { 4132, { 4132, -1 }, { 4133, 4419 }, { 4134, 4412 }, { 4372, 4458 }, { 4374, 4473 }, { 4375, 4485 }, { 4377, 4486 } } },
    { "NtGdiCreateMetafileDC",    { 4133, { 4133, -1 }, { 4134, 4420 }, { 4135, 4413 }, { 4371, 4459 }, { 4373, 4474 }, { 4374, 4486 }, { 4376, 4487 } } },
    { "NtGdiCreateOPMProtectedOutputs",   { -1, { -1, -1 }, { 4135, 4421 }, { 4136, 4414 }, { 4370, 4460 }, { 4372, 4475 }, { 4373, 4487 }, { 4375, 4488 } } },
    { "NtGdiCreatePaletteInternal",   { 4134, { 4134, 4272 }, { 4136, 4273 }, { 4137, 4268 }, { 4369, 4268 }, { 4371, 4269 }, { 4372, 4270 }, { 4374, 4270 } } },
    { "NtGdiCreatePatternBrushInternal",  { 4135, { 4135, 4277 }, { 4137, 4278 }, { 4138, 4273 }, { 4368, 4273 }, { 4370, 4274 }, { 4371, 4275 }, { 4373, 4275 } } },
    { "NtGdiCreatePen",   { 4136, { 4136, 4182 }, { 4138, 4183 }, { 4139, 4183 }, { 4367, 4183 }, { 4369, 4184 }, { 4370, 4185 }, { 4372, 4185 } } },
    { "NtGdiCreateRectRgn",   { 4137, { 4137, 4229 }, { 4139, 4230 }, { 4140, 4228 }, { 4366, 4228 }, { 4368, 4229 }, { 4369, 4230 }, { 4371, 4230 } } },
    { "NtGdiCreateRoundRectRgn",  { 4138, { 4138, -1 }, { 4140, 4422 }, { 4141, 4415 }, { 4365, 4461 }, { 4367, 4476 }, { 4368, 4488 }, { 4370, 4489 } } },
    { "NtGdiCreateServerMetaFile",    { 4139, { 4139, -1 }, { 4141, 4423 }, { 4142, 4416 }, { 4364, 4462 }, { 4366, 4477 }, { 4367, 4489 }, { 4369, 4490 } } },
    { "NtGdiCreateSessionMappedDIBSection",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4375, 4463 }, { 4377, 4478 }, { 4378, 4490 }, { 4380, 4491 } } },
    { "NtGdiCreateSolidBrush",    { 4140, { 4140, 4284 }, { 4142, 4285 }, { 4143, 4280 }, { 4363, 4280 }, { 4365, 4281 }, { 4366, 4282 }, { 4368, 4282 } } },
    { "NtGdiD3dContextCreate",    { 4141, { 4141, 4421 }, { 4143, 4424 }, { 4144, 4417 }, { 4362, 4464 }, { 4364, 4479 }, { 4365, 4491 }, { 4367, 4492 } } },
    { "NtGdiD3dContextDestroy",   { 4142, { 4142, 4422 }, { 4144, 4425 }, { 4145, 4418 }, { 4339, 4465 }, { 4341, 4480 }, { 4364, 4492 }, { 4366, 4493 } } },
    { "NtGdiD3dContextDestroyAll",    { 4143, { 4143, 4423 }, { 4145, 4426 }, { 4146, 4419 }, { 4342, 4466 }, { 4344, 4481 }, { 4363, 4493 }, { 4365, 4494 } } },
    { "NtGdiD3dDrawPrimitives2",  { 4145, { 4145, -1 }, { 4147, 4197 }, { 4148, 4197 }, { 4358, 4197 }, { 4360, 4198 }, { 4361, 4199 }, { 4363, 4199 } } },
    { "NtGdiD3dValidateTextureStageState",    { 4144, { 4144, 4424 }, { 4146, 4427 }, { 4147, 4420 }, { 4357, 4467 }, { 4359, 4482 }, { 4362, 4494 }, { 4364, 4495 } } },
    { "NtGdiDdAddAttachedSurface",    { 4147, { 4147, 4425 }, { 4149, 4434 }, { 4150, 4427 }, { 4331, 4474 }, { 4333, 4489 }, { 4359, 4501 }, { 4361, 4502 } } },
    { "NtGdiDdAlphaBlt",  { 4148, { 4148, 4426 }, { 4150, 4435 }, { 4151, 4428 }, { 4312, 4475 }, { 4314, 4490 }, { 4358, 4502 }, { 4360, 4503 } } },
    { "NtGdiDdAttachSurface", { 4149, { 4149, 4427 }, { 4151, 4436 }, { 4152, 4429 }, { 4321, 4476 }, { 4323, 4491 }, { 4357, 4503 }, { 4359, 4504 } } },
    { "NtGdiDdBeginMoCompFrame",  { 4150, { 4150, 4428 }, { 4152, 4437 }, { 4153, 4430 }, { 4310, 4477 }, { 4312, 4492 }, { 4356, 4504 }, { 4358, 4505 } } },
    { "NtGdiDdBlt",   { 4151, { 4151, 4222 }, { 4153, 4223 }, { 4154, 4221 }, { 4314, 4221 }, { 4316, 4222 }, { 4355, 4223 }, { 4357, 4223 } } },
    { "NtGdiDdCanCreateD3DBuffer",    { 4153, { 4153, 4429 }, { 4155, 4438 }, { 4156, 4431 }, { 4334, 4478 }, { 4336, 4493 }, { 4353, 4505 }, { 4355, 4506 } } },
    { "NtGdiDdCanCreateSurface",  { 4152, { 4152, 4262 }, { 4154, 4263 }, { 4155, 4258 }, { 4353, 4258 }, { 4355, 4259 }, { 4354, 4260 }, { 4356, 4260 } } },
    { "NtGdiDDCCIGetCapabilitiesString",  { -1, { -1, -1 }, { 4864, 4428 }, { 4908, 4421 }, { 4975, 4468 }, { 4993, 4483 }, { 5050, 4495 }, { 5052, 4496 } } },
    { "NtGdiDDCCIGetCapabilitiesStringLength",    { -1, { -1, -1 }, { 4863, 4429 }, { 4907, 4422 }, { 4976, 4469 }, { 4994, 4484 }, { 5051, 4496 }, { 5053, 4497 } } },
    { "NtGdiDDCCIGetTimingReport",    { -1, { -1, -1 }, { 4865, 4430 }, { 4909, 4423 }, { 4974, 4470 }, { 4992, 4485 }, { 5049, 4497 }, { 5051, 4498 } } },
    { "NtGdiDDCCIGetVCPFeature",  { -1, { -1, -1 }, { 4860, 4431 }, { 4904, 4424 }, { 4979, 4471 }, { 4997, 4486 }, { 5054, 4498 }, { 5056, 4499 } } },
    { "NtGdiDDCCISaveCurrentSettings",    { -1, { -1, -1 }, { 4862, 4432 }, { 4906, 4425 }, { 4977, 4472 }, { 4995, 4487 }, { 5052, 4499 }, { 5054, 4500 } } },
    { "NtGdiDDCCISetVCPFeature",  { -1, { -1, -1 }, { 4861, 4433 }, { 4905, 4426 }, { 4978, 4473 }, { 4996, 4488 }, { 5053, 4500 }, { 5055, 4501 } } },
    { "NtGdiDdColorControl",  { 4154, { 4154, 4430 }, { 4156, 4439 }, { 4157, 4432 }, { 4296, 4479 }, { 4298, 4494 }, { 4352, 4506 }, { 4354, 4507 } } },
    { "NtGdiDdCreateD3DBuffer",   { 4157, { 4157, -1 }, { 4159, 4440 }, { 4160, 4433 }, { 4346, 4480 }, { 4348, 4495 }, { 4349, 4507 }, { 4351, 4508 } } },
    { "NtGdiDdCreateDirectDrawObject",    { 4155, { 4155, 4432 }, { 4157, 4441 }, { 4158, 4434 }, { 4361, 4481 }, { 4363, 4496 }, { 4351, 4508 }, { 4353, 4509 } } },
    { "NtGdiDdCreateFullscreenSprite",    { -1, { -1, -1 }, { -1, -1 }, { 4910, 4435 }, { 4983, 4482 }, { 5001, 4497 }, { 5058, 4509 }, { 5060, 4510 } } },
    { "NtGdiDdCreateMoComp",  { 4158, { 4158, 4433 }, { 4160, 4442 }, { 4161, 4436 }, { 4305, 4483 }, { 4307, 4498 }, { 4348, 4510 }, { 4350, 4511 } } },
    { "NtGdiDdCreateSurface", { 4156, { 4156, -1 }, { 4158, 4264 }, { 4159, 4259 }, { 4347, 4259 }, { 4349, 4260 }, { 4350, 4261 }, { 4352, 4261 } } },
    { "NtGdiDdCreateSurfaceEx",   { 4190, { 4190, 4294 }, { 4192, 4295 }, { 4193, 4290 }, { 4313, 4290 }, { 4315, 4291 }, { 4316, 4292 }, { 4318, 4292 } } },
    { "NtGdiDdCreateSurfaceObject",   { 4159, { 4159, -1 }, { 4161, 4296 }, { 4162, 4291 }, { 4344, 4291 }, { 4346, 4292 }, { 4347, 4293 }, { 4349, 4293 } } },
    { "NtGdiDdDDIAbandonSwapChain",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5032, 4511 }, { 5032, 4512 } } },
    { "NtGdiDdDDIAcquireKeyedMutex",  { -1, { -1, -1 }, { -1, -1 }, { 4892, 4437 }, { 4947, 4484 }, { 4956, 4499 }, { 4972, 4512 }, { 4973, 4513 } } },
    { "NtGdiDdDDIAcquireKeyedMutex2", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4932, 4485 }, { 4941, 4500 }, { 4957, 4513 }, { 4958, 4514 } } },
    { "NtGdiDdDDIAcquireSwapChain",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5030, 4514 }, { 5030, 4515 } } },
    { "NtGdiDdDDIAdjustFullscreenGamma",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5040, 4515 }, { 5040, 4516 } } },
    { "NtGdiDdDDICacheHybridQueryValue",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4980, 4501 }, { 5017, 4516 }, { 5018, 4517 } } },
    { "NtGdiDdDDIChangeVideoMemoryReservation",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5026, 4517 }, { 5026, 4518 } } },
    { "NtGdiDdDDICheckExclusiveOwnership",    { -1, { -1, -1 }, { 4850, 4443 }, { 4885, 4438 }, { 4954, 4486 }, { 4963, 4502 }, { 4979, 4518 }, { 4980, 4519 } } },
    { "NtGdiDdDDICheckMonitorPowerState", { -1, { -1, -1 }, { 4849, 4444 }, { 4884, 4439 }, { 4955, 4487 }, { 4964, 4503 }, { 4980, 4519 }, { 4981, 4520 } } },
    { "NtGdiDdDDICheckMultiPlaneOverlaySupport",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4978, 4504 }, { 4996, 4520 }, { 4997, 4521 } } },
    { "NtGdiDdDDICheckMultiPlaneOverlaySupport2", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5034, 4521 }, { 5034, 4522 } } },
    { "NtGdiDdDDICheckOcclusion", { -1, { -1, -1 }, { 4847, 4445 }, { 4882, 4440 }, { 4957, 4488 }, { 4966, 4505 }, { 4982, 4522 }, { 4983, 4523 } } },
    { "NtGdiDdDDICheckSharedResourceAccess",  { -1, { -1, -1 }, { -1, -1 }, { 4897, 4441 }, { 4942, 4489 }, { 4951, 4506 }, { 4967, 4523 }, { 4968, 4524 } } },
    { "NtGdiDdDDICheckVidPnExclusiveOwnership",   { -1, { -1, -1 }, { -1, -1 }, { 4896, 4442 }, { 4943, 4490 }, { 4952, 4507 }, { 4968, 4524 }, { 4969, 4525 } } },
    { "NtGdiDdDDICloseAdapter",   { -1, { -1, -1 }, { 4823, 4446 }, { 4857, 4443 }, { 4894, 4491 }, { 4903, 4508 }, { 4915, 4525 }, { 4916, 4526 } } },
    { "NtGdiDdDDIConfigureSharedResource",    { -1, { -1, -1 }, { -1, -1 }, { 4894, 4444 }, { 4945, 4492 }, { 4954, 4509 }, { 4970, 4526 }, { 4971, 4527 } } },
    { "NtGdiDdDDICreateAllocation",   { -1, { -1, -1 }, { 4798, 4447 }, { 4831, 4445 }, { 4927, 4493 }, { 4936, 4510 }, { 4952, 4527 }, { 4953, 4528 } } },
    { "NtGdiDdDDICreateContext",  { -1, { -1, -1 }, { 4806, 4448 }, { 4839, 4446 }, { 4914, 4494 }, { 4923, 4511 }, { 4936, 4528 }, { 4937, 4529 } } },
    { "NtGdiDdDDICreateContextVirtual",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5014, 4529 }, { 5015, 4530 } } },
    { "NtGdiDdDDICreateDCFromMemory", { -1, { -1, -1 }, { 4836, 4449 }, { 4871, 4447 }, { 4879, 4495 }, { 4888, 4512 }, { 4900, 4530 }, { 4901, 4531 } } },
    { "NtGdiDdDDICreateDevice",   { -1, { -1, -1 }, { 4804, 4450 }, { 4837, 4448 }, { 4916, 4496 }, { 4925, 4513 }, { 4938, 4531 }, { 4939, 4532 } } },
    { "NtGdiDdDDICreateKeyedMutex",   { -1, { -1, -1 }, { -1, -1 }, { 4889, 4449 }, { 4950, 4497 }, { 4959, 4514 }, { 4975, 4532 }, { 4976, 4533 } } },
    { "NtGdiDdDDICreateKeyedMutex2",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4934, 4498 }, { 4943, 4515 }, { 4959, 4533 }, { 4960, 4534 } } },
    { "NtGdiDdDDICreateOutputDupl",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4939, 4499 }, { 4948, 4516 }, { 4964, 4534 }, { 4965, 4535 } } },
    { "NtGdiDdDDICreateOverlay",  { -1, { -1, -1 }, { 4829, 4451 }, { 4864, 4450 }, { 4886, 4500 }, { 4895, 4517 }, { 4907, 4535 }, { 4908, 4536 } } },
    { "NtGdiDdDDICreatePagingQueue",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5004, 4536 }, { 5005, 4537 } } },
    { "NtGdiDdDDICreateSwapChain",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5027, 4537 }, { 5027, 4538 } } },
    { "NtGdiDdDDICreateSynchronizationObject",    { -1, { -1, -1 }, { 4808, 4452 }, { 4841, 4451 }, { 4912, 4501 }, { 4921, 4518 }, { 4934, 4538 }, { 4935, 4539 } } },
    { "NtGdiDdDDIDestroyAllocation",  { -1, { -1, -1 }, { 4801, 4453 }, { 4834, 4452 }, { 4919, 4502 }, { 4928, 4519 }, { 4941, 4539 }, { 4942, 4540 } } },
    { "NtGdiDdDDIDestroyAllocation2", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4942, 4540 }, { 4943, 4541 } } },
    { "NtGdiDdDDIDestroyContext", { -1, { -1, -1 }, { 4807, 4454 }, { 4840, 4453 }, { 4913, 4503 }, { 4922, 4520 }, { 4935, 4541 }, { 4936, 4542 } } },
    { "NtGdiDdDDIDestroyDCFromMemory",    { -1, { -1, -1 }, { 4837, 4455 }, { 4872, 4454 }, { 4878, 4504 }, { 4887, 4521 }, { 4899, 4542 }, { 4900, 4543 } } },
    { "NtGdiDdDDIDestroyDevice",  { -1, { -1, -1 }, { 4805, 4456 }, { 4838, 4455 }, { 4915, 4505 }, { 4924, 4522 }, { 4937, 4543 }, { 4938, 4544 } } },
    { "NtGdiDdDDIDestroyKeyedMutex",  { -1, { -1, -1 }, { -1, -1 }, { 4891, 4456 }, { 4948, 4506 }, { 4957, 4523 }, { 4973, 4544 }, { 4974, 4545 } } },
    { "NtGdiDdDDIDestroyOutputDupl",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4938, 4507 }, { 4947, 4524 }, { 4963, 4545 }, { 4964, 4546 } } },
    { "NtGdiDdDDIDestroyOverlay", { -1, { -1, -1 }, { 4832, 4457 }, { 4867, 4457 }, { 4883, 4508 }, { 4892, 4525 }, { 4904, 4546 }, { 4905, 4547 } } },
    { "NtGdiDdDDIDestroyPagingQueue", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5005, 4547 }, { 5006, 4548 } } },
    { "NtGdiDdDDIDestroySynchronizationObject",   { -1, { -1, -1 }, { 4809, 4458 }, { 4843, 4458 }, { 4910, 4509 }, { 4919, 4526 }, { 4932, 4548 }, { 4933, 4549 } } },
    { "NtGdiDdDDIEnumAdapters",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4896, 4510 }, { 4905, 4527 }, { 4917, 4549 }, { 4918, 4550 } } },
    { "NtGdiDdDDIEnumAdapters2",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4918, 4550 }, { 4919, 4551 } } },
    { "NtGdiDdDDIEscape", { -1, { -1, -1 }, { 4825, 4459 }, { 4859, 4459 }, { 4892, 4511 }, { 4901, 4528 }, { 4913, 4551 }, { 4914, 4552 } } },
    { "NtGdiDdDDIEvict",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4998, 4552 }, { 4999, 4553 } } },
    { "NtGdiDdDDIFlipOverlay",    { -1, { -1, -1 }, { 4831, 4460 }, { 4866, 4460 }, { 4884, 4512 }, { 4893, 4529 }, { 4905, 4553 }, { 4906, 4554 } } },
    { "NtGdiDdDDIFlushHeapTransitions",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 4555 }, { 5043, 4555 } } },
    { "NtGdiDdDDIFreeGpuVirtualAddress",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5012, 4554 }, { 5013, 4556 } } },
    { "NtGdiDdDDIGetCachedHybridQueryValue",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4979, 4530 }, { 5016, 4555 }, { 5017, 4557 } } },
    { "NtGdiDdDDIGetContextInProcessSchedulingPriority",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4965, 4513 }, { 4974, 4531 }, { 4992, 4556 }, { 4993, 4558 } } },
    { "NtGdiDdDDIGetContextSchedulingPriority",   { -1, { -1, -1 }, { 4839, 4461 }, { 4874, 4461 }, { 4876, 4514 }, { 4885, 4532 }, { 4897, 4557 }, { 4898, 4559 } } },
    { "NtGdiDdDDIGetDeviceState", { -1, { -1, -1 }, { 4835, 4462 }, { 4870, 4462 }, { 4880, 4515 }, { 4889, 4533 }, { 4901, 4559 }, { 4902, 4561 } } },
    { "NtGdiDdDDIGetDisplayModeList", { -1, { -1, -1 }, { 4816, 4463 }, { 4850, 4463 }, { 4903, 4516 }, { 4912, 4534 }, { 4925, 4560 }, { 4926, 4562 } } },
    { "NtGdiDdDDIGetDWMVerticalBlankEvent",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4989, 4558 }, { 4990, 4560 } } },
    { "NtGdiDdDDIGetMultisampleMethodList",   { -1, { -1, -1 }, { 4818, 4464 }, { 4852, 4464 }, { 4901, 4517 }, { 4910, 4535 }, { 4923, 4561 }, { 4924, 4563 } } },
    { "NtGdiDdDDIGetOverlayState",    { -1, { -1, -1 }, { -1, -1 }, { 4895, 4465 }, { 4944, 4518 }, { 4953, 4536 }, { 4969, 4562 }, { 4970, 4564 } } },
    { "NtGdiDdDDIGetPresentHistory",  { -1, { -1, -1 }, { 4828, 4465 }, { 4862, 4466 }, { 4888, 4519 }, { 4897, 4537 }, { 4909, 4563 }, { 4910, 4565 } } },
    { "NtGdiDdDDIGetPresentQueueEvent",   { -1, { -1, -1 }, { -1, -1 }, { 4863, 4467 }, { 4887, 4520 }, { 4896, 4538 }, { 4908, 4564 }, { 4909, 4566 } } },
    { "NtGdiDdDDIGetProcessSchedulingPriorityClass",  { -1, { -1, -1 }, { 4841, 4466 }, { 4876, 4468 }, { 4874, 4521 }, { 4883, 4539 }, { 4895, 4565 }, { 4896, 4567 } } },
    { "NtGdiDdDDIGetResourcePresentPrivateDriverData",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5009, 4566 }, { 5010, 4568 } } },
    { "NtGdiDdDDIGetRuntimeData", { -1, { -1, -1 }, { 4812, 4467 }, { 4846, 4469 }, { 4907, 4522 }, { 4916, 4540 }, { 4929, 4567 }, { 4930, 4569 } } },
    { "NtGdiDdDDIGetScanLine",    { -1, { -1, -1 }, { 4843, 4468 }, { 4878, 4470 }, { 4872, 4523 }, { 4881, 4541 }, { 4893, 4568 }, { 4894, 4570 } } },
    { "NtGdiDdDDIGetSetSwapChainMetadata",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5029, 4569 }, { 5029, 4571 } } },
    { "NtGdiDdDDIGetSharedPrimaryHandle", { -1, { -1, -1 }, { 4824, 4469 }, { 4858, 4471 }, { 4893, 4524 }, { 4902, 4542 }, { 4914, 4570 }, { 4915, 4572 } } },
    { "NtGdiDdDDIGetSharedResourceAdapterLuid",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4966, 4525 }, { 4975, 4543 }, { 4993, 4571 }, { 4994, 4573 } } },
    { "NtGdiDdDDIInvalidateActiveVidPn",  { -1, { -1, -1 }, { 4846, 4470 }, { 4881, 4472 }, { 4958, 4526 }, { 4967, 4544 }, { 4983, 4572 }, { 4984, 4574 } } },
    { "NtGdiDdDDIInvalidateCache",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5008, 4573 }, { 5009, 4575 } } },
    { "NtGdiDdDDILock",   { -1, { -1, -1 }, { 4814, 4471 }, { 4848, 4473 }, { 4905, 4527 }, { 4914, 4545 }, { 4927, 4574 }, { 4928, 4576 } } },
    { "NtGdiDdDDILock2",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5006, 4575 }, { 5007, 4577 } } },
    { "NtGdiDdDDIMakeResident",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4997, 4576 }, { 4998, 4578 } } },
    { "NtGdiDdDDIMapGpuVirtualAddress",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5010, 4577 }, { 5011, 4579 } } },
    { "NtGdiDdDDIMarkDeviceAsError",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5042, 4578 }, { 5042, 4580 } } },
    { "NtGdiDdDDINetDispGetNextChunkInfo",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4981, 4546 }, { 5018, 4579 }, { 5019, 4581 } } },
    { "NtGdiDdDDINetDispQueryMiracastDisplayDeviceStatus",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4985, 4547 }, { 5023, 4580 }, { 5023, 4582 } } },
    { "NtGdiDdDDINetDispQueryMiracastDisplayDeviceSupport",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4982, 4548 }, { 5019, 4581 }, { 5020, 4583 } } },
    { "NtGdiDdDDINetDispStartMiracastDisplayDevice",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4983, 4549 }, { 5020, 4582 }, { 5021, 4584 } } },
    { "NtGdiDdDDINetDispStartMiracastDisplayDeviceEx",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5021, 4583 }, { -1, -1 } } },
    { "NtGdiDdDDINetDispStopMiracastDisplayDevice",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4984, 4550 }, { 5022, 4584 }, { 5022, 4585 } } },
    { "NtGdiDdDDINetDispStopSessions",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5024, 4585 }, { 5024, 4586 } } },
    { "NtGdiDdDDIOfferAllocations",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4941, 4528 }, { 4950, 4551 }, { 4966, 4586 }, { 4967, 4587 } } },
    { "NtGdiDdDDIOpenAdapterFromDeviceName",  { -1, { -1, -1 }, { 4821, 4472 }, { 4855, 4474 }, { 4898, 4529 }, { 4907, 4552 }, { 4920, 4587 }, { 4921, 4588 } } },
    { "NtGdiDdDDIOpenAdapterFromHdc", { -1, { -1, -1 }, { 4822, 4473 }, { 4856, 4475 }, { 4897, 4530 }, { 4906, 4553 }, { 4919, 4588 }, { 4920, 4589 } } },
    { "NtGdiDdDDIOpenAdapterFromLuid",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4895, 4531 }, { 4904, 4554 }, { 4916, 4589 }, { 4917, 4590 } } },
    { "NtGdiDdDDIOpenKeyedMutex", { -1, { -1, -1 }, { -1, -1 }, { 4890, 4476 }, { 4949, 4532 }, { 4958, 4555 }, { 4974, 4590 }, { 4975, 4591 } } },
    { "NtGdiDdDDIOpenKeyedMutex2",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4933, 4533 }, { 4942, 4556 }, { 4958, 4591 }, { 4959, 4592 } } },
    { "NtGdiDdDDIOpenNtHandleFromName",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4923, 4534 }, { 4932, 4557 }, { 4947, 4592 }, { 4948, 4593 } } },
    { "NtGdiDdDDIOpenResource",   { -1, { -1, -1 }, { 4800, 4474 }, { 4833, 4477 }, { 4922, 4535 }, { 4931, 4558 }, { 4946, 4593 }, { 4947, 4594 } } },
    { "NtGdiDdDDIOpenResourceFromNtHandle",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4920, 4536 }, { 4929, 4559 }, { 4943, 4594 }, { 4944, 4595 } } },
    { "NtGdiDdDDIOpenSwapChain",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5028, 4595 }, { 5028, 4596 } } },
    { "NtGdiDdDDIOpenSynchronizationObject",  { -1, { -1, -1 }, { -1, -1 }, { 4842, 4478 }, { 4911, 4538 }, { 4920, 4561 }, { 4933, 4599 }, { 4934, 4600 } } },
    { "NtGdiDdDDIOpenSyncObjectFromNtHandle", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4921, 4537 }, { 4930, 4560 }, { 4944, 4596 }, { 4945, 4597 } } },
    { "NtGdiDdDDIOpenSyncObjectFromNtHandle2",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4945, 4597 }, { 4946, 4598 } } },
    { "NtGdiDdDDIOpenSyncObjectNtHandleFromName", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4948, 4598 }, { 4949, 4599 } } },
    { "NtGdiDdDDIOutputDuplGetFrameInfo", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4937, 4539 }, { 4946, 4562 }, { 4962, 4600 }, { 4963, 4601 } } },
    { "NtGdiDdDDIOutputDuplGetMetaData",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4936, 4540 }, { 4945, 4563 }, { 4961, 4601 }, { 4962, 4602 } } },
    { "NtGdiDdDDIOutputDuplGetPointerShapeData",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4935, 4541 }, { 4944, 4564 }, { 4960, 4602 }, { 4961, 4603 } } },
    { "NtGdiDdDDIOutputDuplPresent",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4930, 4542 }, { 4939, 4565 }, { 4955, 4603 }, { 4956, 4604 } } },
    { "NtGdiDdDDIOutputDuplReleaseFrame", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4928, 4543 }, { 4937, 4566 }, { 4953, 4604 }, { 4954, 4605 } } },
    { "NtGdiDdDDIPinDirectFlipResources", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4961, 4544 }, { 4970, 4567 }, { 4986, 4605 }, { 4987, 4606 } } },
    { "NtGdiDdDDIPollDisplayChildren",    { -1, { -1, -1 }, { 4845, 4475 }, { 4880, 4479 }, { 4959, 4545 }, { 4968, 4568 }, { 4984, 4606 }, { 4985, 4607 } } },
    { "NtGdiDdDDIPresent",    { -1, { -1, -1 }, { 4819, 4476 }, { 4853, 4480 }, { 4900, 4546 }, { 4909, 4569 }, { 4922, 4607 }, { 4923, 4608 } } },
    { "NtGdiDdDDIPresentMultiPlaneOverlay",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4977, 4570 }, { 4995, 4608 }, { 4996, 4609 } } },
    { "NtGdiDdDDIPresentMultiPlaneOverlay2",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5035, 4609 }, { 5035, 4610 } } },
    { "NtGdiDdDDIQueryAdapterInfo",   { -1, { -1, -1 }, { 4813, 4477 }, { 4847, 4481 }, { 4906, 4547 }, { 4915, 4571 }, { 4928, 4610 }, { 4929, 4611 } } },
    { "NtGdiDdDDIQueryAllocationResidency",   { -1, { -1, -1 }, { 4803, 4478 }, { 4836, 4482 }, { 4917, 4548 }, { 4926, 4572 }, { 4939, 4611 }, { 4940, 4612 } } },
    { "NtGdiDdDDIQueryClockCalibration",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5038, 4612 }, { 5038, 4613 } } },
    { "NtGdiDdDDIQueryRemoteVidPnSourceFromGdiDisplayName",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4929, 4549 }, { 4938, 4573 }, { 4954, 4613 }, { 4955, 4614 } } },
    { "NtGdiDdDDIQueryResourceInfo",  { -1, { -1, -1 }, { 4799, 4479 }, { 4832, 4483 }, { 4926, 4550 }, { 4935, 4574 }, { 4951, 4614 }, { 4952, 4615 } } },
    { "NtGdiDdDDIQueryResourceInfoFromNtHandle",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4925, 4551 }, { 4934, 4575 }, { 4950, 4615 }, { 4951, 4616 } } },
    { "NtGdiDdDDIQueryStatistics",    { -1, { -1, -1 }, { 4826, 4480 }, { 4860, 4484 }, { 4891, 4552 }, { 4900, 4576 }, { 4912, 4616 }, { 4913, 4617 } } },
    { "NtGdiDdDDIQueryVideoMemoryInfo",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5025, 4618 }, { 5025, 4619 } } },
    { "NtGdiDdDDIQueryVidPnExclusiveOwnership",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5039, 4617 }, { 5039, 4618 } } },
    { "NtGdiDdDDIReclaimAllocations", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4940, 4553 }, { 4949, 4577 }, { 4965, 4619 }, { 4966, 4620 } } },
    { "NtGdiDdDDIReclaimAllocations2",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5036, 4620 }, { 5036, 4621 } } },
    { "NtGdiDdDDIReleaseKeyedMutex",  { -1, { -1, -1 }, { -1, -1 }, { 4893, 4485 }, { 4946, 4554 }, { 4955, 4578 }, { 4971, 4621 }, { 4972, 4622 } } },
    { "NtGdiDdDDIReleaseKeyedMutex2", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4931, 4555 }, { 4940, 4579 }, { 4956, 4622 }, { 4957, 4623 } } },
    { "NtGdiDdDDIReleaseProcessVidPnSourceOwners",    { -1, { -1, -1 }, { 4842, 4481 }, { 4877, 4486 }, { 4873, 4556 }, { 4882, 4580 }, { 4894, 4623 }, { 4895, 4624 } } },
    { "NtGdiDdDDIReleaseSwapChain",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5031, 4624 }, { 5031, 4625 } } },
    { "NtGdiDdDDIRender", { -1, { -1, -1 }, { 4820, 4482 }, { 4854, 4487 }, { 4899, 4557 }, { 4908, 4581 }, { 4921, 4625 }, { 4922, 4626 } } },
    { "NtGdiDdDDIReserveGpuVirtualAddress",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5011, 4626 }, { 5012, 4627 } } },
    { "NtGdiDdDDISetAllocationPriority",  { -1, { -1, -1 }, { 4802, 4483 }, { 4835, 4488 }, { 4918, 4558 }, { 4927, 4582 }, { 4940, 4627 }, { 4941, 4628 } } },
    { "NtGdiDdDDISetContextInProcessSchedulingPriority",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4964, 4559 }, { 4973, 4583 }, { 4991, 4628 }, { 4992, 4629 } } },
    { "NtGdiDdDDISetContextSchedulingPriority",   { -1, { -1, -1 }, { 4838, 4484 }, { 4873, 4489 }, { 4877, 4560 }, { 4886, 4584 }, { 4898, 4629 }, { 4899, 4630 } } },
    { "NtGdiDdDDISetDisplayMode", { -1, { -1, -1 }, { 4817, 4485 }, { 4851, 4490 }, { 4902, 4561 }, { 4911, 4585 }, { 4924, 4630 }, { 4925, 4631 } } },
    { "NtGdiDdDDISetDisplayPrivateDriverFormat",  { -1, { -1, -1 }, { 4851, 4486 }, { 4886, 4491 }, { 4953, 4562 }, { 4962, 4586 }, { 4978, 4631 }, { 4979, 4632 } } },
    { "NtGdiDdDDISetDodIndirectSwapchain",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5033, 4632 }, { 5033, 4633 } } },
    { "NtGdiDdDDISetGammaRamp",   { -1, { -1, -1 }, { 4834, 4487 }, { 4869, 4492 }, { 4881, 4563 }, { 4890, 4587 }, { 4902, 4633 }, { 4903, 4634 } } },
    { "NtGdiDdDDISetHwProtectionTeardownRecovery",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 4635 }, { 5044, 4635 } } },
    { "NtGdiDdDDISetProcessSchedulingPriorityClass",  { -1, { -1, -1 }, { 4840, 4488 }, { 4875, 4493 }, { 4875, 4564 }, { 4884, 4588 }, { 4896, 4634 }, { 4897, 4636 } } },
    { "NtGdiDdDDISetQueuedLimit", { -1, { -1, -1 }, { 4844, 4489 }, { 4879, 4494 }, { 4960, 4565 }, { 4969, 4589 }, { 4985, 4635 }, { 4986, 4637 } } },
    { "NtGdiDdDDISetStablePowerState",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5037, 4636 }, { 5037, 4638 } } },
    { "NtGdiDdDDISetStereoEnabled",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4967, 4566 }, { 4976, 4590 }, { 4994, 4637 }, { 4995, 4639 } } },
    { "NtGdiDdDDISetSyncRefreshCountWaitTarget",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4990, 4638 }, { 4991, 4640 } } },
    { "NtGdiDdDDISetVidPnSourceHwProtection", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5041, 4639 }, { 5041, 4641 } } },
    { "NtGdiDdDDISetVidPnSourceOwner",    { -1, { -1, -1 }, { 4827, 4490 }, { 4861, 4495 }, { 4890, 4567 }, { 4899, 4591 }, { 4911, 4640 }, { 4912, 4642 } } },
    { "NtGdiDdDDISetVidPnSourceOwner1",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4889, 4568 }, { 4898, 4592 }, { 4910, 4641 }, { 4911, 4643 } } },
    { "NtGdiDdDDISharedPrimaryLockNotification",  { -1, { -1, -1 }, { 4852, 4491 }, { 4887, 4496 }, { 4952, 4570 }, { 4961, 4594 }, { 4977, 4643 }, { 4978, 4645 } } },
    { "NtGdiDdDDISharedPrimaryUnLockNotification",    { -1, { -1, -1 }, { 4853, 4492 }, { 4888, 4497 }, { 4951, 4571 }, { 4960, 4595 }, { 4976, 4644 }, { 4977, 4646 } } },
    { "NtGdiDdDDIShareObjects",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4924, 4569 }, { 4933, 4593 }, { 4949, 4642 }, { 4950, 4644 } } },
    { "NtGdiDdDDISignalSynchronizationObject",    { -1, { -1, -1 }, { 4811, 4493 }, { 4845, 4498 }, { 4908, 4572 }, { 4917, 4596 }, { 4930, 4645 }, { 4931, 4647 } } },
    { "NtGdiDdDDISignalSynchronizationObjectFromCpu", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5000, 4646 }, { 5001, 4648 } } },
    { "NtGdiDdDDISignalSynchronizationObjectFromGpu", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5002, 4647 }, { 5003, 4649 } } },
    { "NtGdiDdDDISignalSynchronizationObjectFromGpu2",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5003, 4648 }, { 5004, 4650 } } },
    { "NtGdiDdDDISubmitCommand",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5015, 4649 }, { 5016, 4651 } } },
    { "NtGdiDdDDIUnlock", { -1, { -1, -1 }, { 4815, 4494 }, { 4849, 4499 }, { 4904, 4573 }, { 4913, 4597 }, { 4926, 4650 }, { 4927, 4652 } } },
    { "NtGdiDdDDIUnlock2",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5007, 4651 }, { 5008, 4653 } } },
    { "NtGdiDdDDIUnpinDirectFlipResources",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4962, 4574 }, { 4971, 4598 }, { 4987, 4652 }, { 4988, 4654 } } },
    { "NtGdiDdDDIUpdateGpuVirtualAddress",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5013, 4653 }, { 5014, 4655 } } },
    { "NtGdiDdDDIUpdateOverlay",  { -1, { -1, -1 }, { 4830, 4495 }, { 4865, 4500 }, { 4885, 4575 }, { 4894, 4599 }, { 4906, 4654 }, { 4907, 4656 } } },
    { "NtGdiDdDDIWaitForIdle",    { -1, { -1, -1 }, { 4848, 4496 }, { 4883, 4501 }, { 4956, 4576 }, { 4965, 4600 }, { 4981, 4655 }, { 4982, 4657 } } },
    { "NtGdiDdDDIWaitForSynchronizationObject",   { -1, { -1, -1 }, { 4810, 4497 }, { 4844, 4502 }, { 4909, 4577 }, { 4918, 4601 }, { 4931, 4656 }, { 4932, 4658 } } },
    { "NtGdiDdDDIWaitForSynchronizationObjectFromCpu",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4999, 4657 }, { 5000, 4659 } } },
    { "NtGdiDdDDIWaitForSynchronizationObjectFromGpu",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5001, 4658 }, { 5002, 4660 } } },
    { "NtGdiDdDDIWaitForVerticalBlankEvent",  { -1, { -1, -1 }, { 4833, 4498 }, { 4868, 4503 }, { 4882, 4578 }, { 4891, 4602 }, { 4903, 4659 }, { 4904, 4661 } } },
    { "NtGdiDdDDIWaitForVerticalBlankEvent2", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4963, 4579 }, { 4972, 4603 }, { 4988, 4660 }, { 4989, 4662 } } },
    { "NtGdiDdDeleteDirectDrawObject",    { 4160, { 4160, 4434 }, { 4162, 4499 }, { 4163, 4504 }, { 4359, 4580 }, { 4361, 4604 }, { 4346, 4661 }, { 4348, 4663 } } },
    { "NtGdiDdDeleteSurfaceObject",   { 4161, { 4161, 4258 }, { 4163, 4259 }, { 4164, 4257 }, { 4343, 4257 }, { 4345, 4258 }, { 4345, 4259 }, { 4347, 4259 } } },
    { "NtGdiDdDestroyD3DBuffer",  { 4164, { 4164, 4435 }, { 4166, 4500 }, { 4167, 4505 }, { 4348, 4581 }, { 4350, 4605 }, { 4342, 4662 }, { 4344, 4664 } } },
    { "NtGdiDdDestroyFullscreenSprite",   { -1, { -1, -1 }, { -1, -1 }, { 4912, 4506 }, { 4981, 4582 }, { 4999, 4606 }, { 5056, 4663 }, { 5058, 4665 } } },
    { "NtGdiDdDestroyMoComp", { 4162, { 4162, 4436 }, { 4164, 4501 }, { 4165, 4507 }, { 4323, 4583 }, { 4325, 4607 }, { 4344, 4664 }, { 4346, 4666 } } },
    { "NtGdiDdDestroySurface",    { 4163, { 4163, 4265 }, { 4165, 4266 }, { 4166, 4261 }, { 4292, 4261 }, { 4294, 4262 }, { 4343, 4263 }, { 4345, 4263 } } },
    { "NtGdiDdEndMoCompFrame",    { 4165, { 4165, 4437 }, { 4167, 4502 }, { 4168, 4508 }, { 4316, 4584 }, { 4318, 4608 }, { 4341, 4665 }, { 4343, 4667 } } },
    { "NtGdiDdFlip",  { 4166, { 4166, -1 }, { 4168, 4503 }, { 4169, 4509 }, { 4337, 4585 }, { 4339, 4609 }, { 4340, 4666 }, { 4342, 4668 } } },
    { "NtGdiDdFlipToGDISurface",  { 4167, { 4167, 4439 }, { 4169, 4504 }, { 4170, 4510 }, { 4336, 4586 }, { 4338, 4610 }, { 4339, 4667 }, { 4341, 4669 } } },
    { "NtGdiDdGetAvailDriverMemory",  { 4168, { 4168, 4440 }, { 4170, 4505 }, { 4171, 4511 }, { 4350, 4587 }, { 4352, 4611 }, { 4338, 4668 }, { 4340, 4670 } } },
    { "NtGdiDdGetBltStatus",  { 4169, { 4169, 4441 }, { 4171, 4506 }, { 4172, 4512 }, { 4290, 4588 }, { 4292, 4612 }, { 4337, 4669 }, { 4339, 4671 } } },
    { "NtGdiDdGetDC", { 4170, { 4170, 4442 }, { 4172, 4507 }, { 4173, 4513 }, { 4326, 4589 }, { 4328, 4613 }, { 4336, 4670 }, { 4338, 4672 } } },
    { "NtGdiDdGetDriverInfo", { 4171, { 4171, 4443 }, { 4173, 4508 }, { 4174, 4514 }, { 4327, 4590 }, { 4329, 4614 }, { 4335, 4671 }, { 4337, 4673 } } },
    { "NtGdiDdGetDriverState",    { 4146, { 4146, 4444 }, { 4148, 4509 }, { 4149, 4515 }, { 4360, 4591 }, { 4362, 4615 }, { 4360, 4672 }, { 4362, 4674 } } },
    { "NtGdiDdGetDxHandle",   { 4172, { 4172, 4445 }, { 4174, 4510 }, { 4175, 4516 }, { 4308, 4592 }, { 4310, 4616 }, { 4334, 4673 }, { 4336, 4675 } } },
    { "NtGdiDdGetFlipStatus", { 4173, { 4173, 4446 }, { 4175, 4511 }, { 4176, 4517 }, { 4306, 4593 }, { 4308, 4617 }, { 4333, 4674 }, { 4335, 4676 } } },
    { "NtGdiDdGetInternalMoCompInfo", { 4174, { 4174, 4447 }, { 4176, 4512 }, { 4177, 4518 }, { 4349, 4594 }, { 4351, 4618 }, { 4332, 4675 }, { 4334, 4677 } } },
    { "NtGdiDdGetMoCompBuffInfo", { 4175, { 4175, 4448 }, { 4177, 4513 }, { 4178, 4519 }, { 4351, 4595 }, { 4353, 4619 }, { 4331, 4676 }, { 4333, 4678 } } },
    { "NtGdiDdGetMoCompFormats",  { 4177, { 4177, 4449 }, { 4179, 4514 }, { 4180, 4520 }, { 4330, 4596 }, { 4332, 4620 }, { 4329, 4677 }, { 4331, 4679 } } },
    { "NtGdiDdGetMoCompGuids",    { 4176, { 4176, 4450 }, { 4178, 4515 }, { 4179, 4521 }, { 4297, 4597 }, { 4299, 4621 }, { 4330, 4678 }, { 4332, 4680 } } },
    { "NtGdiDdGetScanLine",   { 4178, { 4178, 4451 }, { 4180, 4516 }, { 4181, 4522 }, { 4341, 4598 }, { 4343, 4622 }, { 4328, 4679 }, { 4330, 4681 } } },
    { "NtGdiDdLock",  { 4179, { 4179, 4452 }, { 4181, 4517 }, { 4182, 4523 }, { 4356, 4599 }, { 4358, 4623 }, { 4327, 4680 }, { 4329, 4682 } } },
    { "NtGdiDdLockD3D",   { 4180, { 4180, 4297 }, { 4182, 4298 }, { 4183, 4293 }, { 4315, 4293 }, { 4317, 4294 }, { 4326, 4295 }, { 4328, 4295 } } },
    { "NtGdiDdNotifyFullscreenSpriteUpdate",  { -1, { -1, -1 }, { -1, -1 }, { 4911, 4524 }, { 4982, 4600 }, { 5000, 4624 }, { 5057, 4681 }, { 5059, 4683 } } },
    { "NtGdiDdQueryDirectDrawObject", { 4181, { 4181, -1 }, { 4183, 4518 }, { 4184, 4525 }, { 4322, 4601 }, { 4324, 4625 }, { 4325, 4682 }, { 4327, 4684 } } },
    { "NtGdiDdQueryMoCompStatus", { 4182, { 4182, 4454 }, { 4184, 4519 }, { 4185, 4526 }, { 4295, 4602 }, { 4297, 4626 }, { 4324, 4683 }, { 4326, 4685 } } },
    { "NtGdiDdQueryVisRgnUniqueness", { -1, { -1, -1 }, { -1, -1 }, { 4913, 4527 }, { 4980, 4603 }, { 4998, 4627 }, { 5055, 4684 }, { 5057, 4686 } } },
    { "NtGdiDdReenableDirectDrawObject",  { 4183, { 4183, 4455 }, { 4185, 4520 }, { 4186, 4528 }, { 4328, 4604 }, { 4330, 4628 }, { 4323, 4685 }, { 4325, 4687 } } },
    { "NtGdiDdReleaseDC", { 4184, { 4184, 4456 }, { 4186, 4521 }, { 4187, 4529 }, { 4319, 4605 }, { 4321, 4629 }, { 4322, 4686 }, { 4324, 4688 } } },
    { "NtGdiDdRenderMoComp",  { 4185, { 4185, 4457 }, { 4187, 4522 }, { 4188, 4530 }, { 4294, 4606 }, { 4296, 4630 }, { 4321, 4687 }, { 4323, 4689 } } },
    { "NtGdiDdResetVisrgn",   { 4186, { 4186, 4270 }, { 4188, 4271 }, { 4189, 4266 }, { 4332, 4266 }, { 4334, 4267 }, { 4320, 4268 }, { 4322, 4268 } } },
    { "NtGdiDdSetColorKey",   { 4187, { 4187, 4458 }, { 4189, 4523 }, { 4190, 4531 }, { 4318, 4607 }, { 4320, 4631 }, { 4319, 4688 }, { 4321, 4690 } } },
    { "NtGdiDdSetExclusiveMode",  { 4188, { 4188, 4459 }, { 4190, 4524 }, { 4191, 4532 }, { 4335, 4608 }, { 4337, 4632 }, { 4318, 4689 }, { 4320, 4691 } } },
    { "NtGdiDdSetGammaRamp",  { 4189, { 4189, 4460 }, { 4191, 4525 }, { 4192, 4533 }, { 4324, 4609 }, { 4326, 4633 }, { 4317, 4690 }, { 4319, 4692 } } },
    { "NtGdiDdSetOverlayPosition",    { 4191, { 4191, 4461 }, { 4193, 4526 }, { 4194, 4534 }, { 4291, 4610 }, { 4293, 4634 }, { 4315, 4691 }, { 4317, 4693 } } },
    { "NtGdiDdUnattachSurface",   { 4192, { 4192, 4462 }, { 4194, 4527 }, { 4195, 4535 }, { 4311, 4611 }, { 4313, 4635 }, { 4314, 4692 }, { 4316, 4694 } } },
    { "NtGdiDdUnlock",    { 4193, { 4193, 4463 }, { 4195, 4528 }, { 4196, 4536 }, { 4298, 4612 }, { 4300, 4636 }, { 4313, 4693 }, { 4315, 4695 } } },
    { "NtGdiDdUnlockD3D", { 4194, { 4194, 4298 }, { 4196, 4299 }, { 4197, 4294 }, { 4329, 4294 }, { 4331, 4295 }, { 4312, 4296 }, { 4314, 4296 } } },
    { "NtGdiDdUpdateOverlay", { 4195, { 4195, 4464 }, { 4197, 4529 }, { 4198, 4537 }, { 4352, 4613 }, { 4354, 4637 }, { 4311, 4694 }, { 4313, 4696 } } },
    { "NtGdiDdWaitForVerticalBlank",  { 4196, { 4196, 4465 }, { 4198, 4530 }, { 4199, 4538 }, { 4345, 4614 }, { 4347, 4638 }, { 4310, 4695 }, { 4312, 4697 } } },
    { "NtGdiDeleteClientObj", { 4215, { 4215, 4232 }, { 4217, 4233 }, { 4218, 4231 }, { 4288, 4231 }, { 4290, 4232 }, { 4291, 4233 }, { 4293, 4233 } } },
    { "NtGdiDeleteColorSpace",    { 4216, { 4216, 4389 }, { 4218, 4390 }, { 4219, 4381 }, { 4287, 4381 }, { 4289, 4382 }, { 4290, 4383 }, { 4292, 4383 } } },
    { "NtGdiDeleteColorTransform",    { 4217, { 4217, 4466 }, { 4219, 4531 }, { 4220, 4539 }, { 4286, 4615 }, { 4288, 4639 }, { 4289, 4696 }, { 4291, 4698 } } },
    { "NtGdiDeleteObjectApp", { 4218, { 4218, 4130 }, { 4220, 4131 }, { 4221, 4131 }, { 4285, 4132 }, { 4287, 4133 }, { 4288, 4134 }, { 4290, 4134 } } },
    { "NtGdiDescribePixelFormat", { 4219, { 4219, 4467 }, { 4221, 4532 }, { 4222, 4540 }, { 4284, 4616 }, { 4286, 4640 }, { 4287, 4697 }, { 4289, 4699 } } },
    { "NtGdiDestroyOPMProtectedOutput",   { -1, { -1, -1 }, { 4222, 4533 }, { 4223, 4541 }, { 4283, 4617 }, { 4285, 4641 }, { 4286, 4698 }, { 4288, 4700 } } },
    { "NtGdiDestroyPhysicalMonitor",  { -1, { -1, -1 }, { 4859, 4534 }, { 4903, 4542 }, { 4970, 4618 }, { 4988, 4642 }, { 5045, 4699 }, { 5047, 4701 } } },
    { "NtGdiDoBanding",   { 4221, { 4221, 4468 }, { 4224, 4535 }, { 4225, 4543 }, { 4281, 4619 }, { 4283, 4643 }, { 4284, 4700 }, { 4286, 4702 } } },
    { "NtGdiDoPalette",   { 4222, { 4222, -1 }, { 4225, 4167 }, { 4226, 4167 }, { 4280, 4168 }, { 4282, 4169 }, { 4283, 4170 }, { 4285, 4170 } } },
    { "NtGdiDrawEscape",  { 4223, { 4223, 4469 }, { 4226, 4536 }, { 4227, 4544 }, { 4279, 4620 }, { 4281, 4644 }, { 4282, 4701 }, { 4284, 4703 } } },
    { "NtGdiDrawStream",  { 4762, { 4758, 4193 }, { 4795, 4194 }, { 4827, 4194 }, { 4868, 4194 }, { 4877, 4195 }, { 4889, 4196 }, { 4890, 4196 } } },
    { "NtGdiDvpAcquireNotification",  { 4212, { 4212, 4470 }, { 4214, 4537 }, { 4215, 4545 }, { 4355, 4621 }, { 4357, 4645 }, { 4294, 4702 }, { 4296, 4704 } } },
    { "NtGdiDvpCanCreateVideoPort",   { 4197, { 4197, 4471 }, { 4199, 4538 }, { 4200, 4546 }, { 4338, 4622 }, { 4340, 4646 }, { 4309, 4703 }, { 4311, 4705 } } },
    { "NtGdiDvpColorControl", { 4198, { 4198, 4472 }, { 4200, 4539 }, { 4201, 4547 }, { 4301, 4623 }, { 4303, 4647 }, { 4308, 4704 }, { 4310, 4706 } } },
    { "NtGdiDvpCreateVideoPort",  { 4199, { 4199, 4473 }, { 4201, 4540 }, { 4202, 4548 }, { 4340, 4624 }, { 4342, 4648 }, { 4307, 4705 }, { 4309, 4707 } } },
    { "NtGdiDvpDestroyVideoPort", { 4200, { 4200, 4474 }, { 4202, 4541 }, { 4203, 4549 }, { 4299, 4625 }, { 4301, 4649 }, { 4306, 4706 }, { 4308, 4708 } } },
    { "NtGdiDvpFlipVideoPort",    { 4201, { 4201, 4475 }, { 4203, 4542 }, { 4204, 4550 }, { 4302, 4626 }, { 4304, 4650 }, { 4305, 4707 }, { 4307, 4709 } } },
    { "NtGdiDvpGetVideoPortBandwidth",    { 4202, { 4202, 4476 }, { 4204, 4543 }, { 4205, 4551 }, { 4303, 4627 }, { 4305, 4651 }, { 4304, 4708 }, { 4306, 4710 } } },
    { "NtGdiDvpGetVideoPortConnectInfo",  { 4208, { 4208, 4477 }, { 4210, 4544 }, { 4211, 4552 }, { 4325, 4628 }, { 4327, 4652 }, { 4298, 4709 }, { 4300, 4711 } } },
    { "NtGdiDvpGetVideoPortField",    { 4203, { 4203, 4478 }, { 4205, 4545 }, { 4206, 4553 }, { 4333, 4629 }, { 4335, 4653 }, { 4303, 4710 }, { 4305, 4712 } } },
    { "NtGdiDvpGetVideoPortFlipStatus",   { 4204, { 4204, 4479 }, { 4206, 4546 }, { 4207, 4554 }, { 4309, 4630 }, { 4311, 4654 }, { 4302, 4711 }, { 4304, 4713 } } },
    { "NtGdiDvpGetVideoPortInputFormats", { 4205, { 4205, 4480 }, { 4207, 4547 }, { 4208, 4555 }, { 4317, 4631 }, { 4319, 4655 }, { 4301, 4712 }, { 4303, 4714 } } },
    { "NtGdiDvpGetVideoPortLine", { 4206, { 4206, 4481 }, { 4208, 4548 }, { 4209, 4556 }, { 4354, 4632 }, { 4356, 4656 }, { 4300, 4713 }, { 4302, 4715 } } },
    { "NtGdiDvpGetVideoPortOutputFormats",    { 4207, { 4207, 4482 }, { 4209, 4549 }, { 4210, 4557 }, { 4307, 4633 }, { 4309, 4657 }, { 4299, 4714 }, { 4301, 4716 } } },
    { "NtGdiDvpGetVideoSignalStatus", { 4209, { 4209, 4483 }, { 4211, 4550 }, { 4212, 4558 }, { 4320, 4634 }, { 4322, 4658 }, { 4297, 4715 }, { 4299, 4717 } } },
    { "NtGdiDvpReleaseNotification",  { 4213, { 4213, 4484 }, { 4215, 4551 }, { 4216, 4559 }, { 4300, 4635 }, { 4302, 4659 }, { 4293, 4716 }, { 4295, 4718 } } },
    { "NtGdiDvpUpdateVideoPort",  { 4210, { 4210, 4485 }, { 4212, 4552 }, { 4213, 4560 }, { 4293, 4636 }, { 4295, 4660 }, { 4296, 4717 }, { 4298, 4719 } } },
    { "NtGdiDvpWaitForVideoPortSync", { 4211, { 4211, 4486 }, { 4213, 4553 }, { 4214, 4561 }, { 4304, 4637 }, { 4306, 4661 }, { 4295, 4718 }, { 4297, 4720 } } },
    { "NtGdiDwmCreatedBitmapRemotingOutput",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4871, 4638 }, { 4880, 4662 }, { 4892, 4719 }, { 4893, 4721 } } },
    { "NtGdiDwmGetDirtyRgn",  { -1, { -1, -1 }, { 4796, 4554 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiDwmGetSurfaceData",   { -1, { -1, -1 }, { 4797, 4555 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiDxgGenericThunk", { 4214, { 4214, -1 }, { 4216, 4556 }, { 4217, 4562 }, { 4289, 4639 }, { 4291, 4663 }, { 4292, 4720 }, { 4294, 4722 } } },
    { "NtGdiEllipse", { 4224, { 4224, -1 }, { 4227, 4557 }, { 4228, 4563 }, { 4278, 4640 }, { 4280, 4664 }, { 4281, 4721 }, { 4283, 4723 } } },
    { "NtGdiEnableEudc",  { 4225, { 4225, -1 }, { 4228, 4558 }, { 4229, 4564 }, { 4277, 4641 }, { 4279, 4665 }, { 4280, 4722 }, { 4282, 4724 } } },
    { "NtGdiEndDoc",  { 4226, { 4226, 4490 }, { 4229, 4559 }, { 4230, 4565 }, { 4276, 4642 }, { 4278, 4666 }, { 4279, 4723 }, { 4281, 4725 } } },
    { "NtGdiEndGdiRendering", { -1, { -1, -1 }, { -1, -1 }, { 4231, 4566 }, { 4275, 4643 }, { 4277, 4667 }, { 4278, 4724 }, { 4280, 4726 } } },
    { "NtGdiEndPage", { 4227, { 4227, 4491 }, { 4230, 4560 }, { 4232, 4567 }, { 4274, 4644 }, { 4276, 4668 }, { 4277, 4725 }, { 4279, 4727 } } },
    { "NtGdiEndPath", { 4228, { 4228, -1 }, { 4231, 4369 }, { 4233, 4362 }, { 4273, 4362 }, { 4275, 4363 }, { 4276, 4364 }, { 4278, 4364 } } },
    { "NtGdiEngAlphaBlend",   { 4716, { 4712, -1 }, { 4749, 4561 }, { 4781, 4568 }, { 4775, 4645 }, { 4778, 4669 }, { 4788, 4726 }, { 4789, 4728 } } },
    { "NtGdiEngAssociateSurface", { 4695, { 4691, 4493 }, { 4728, 4562 }, { 4760, 4569 }, { 4796, 4646 }, { 4799, 4670 }, { 4809, 4727 }, { 4810, 4729 } } },
    { "NtGdiEngBitBlt",   { 4707, { 4703, -1 }, { 4740, 4563 }, { 4772, 4570 }, { 4784, 4647 }, { 4787, 4671 }, { 4797, 4728 }, { 4798, 4730 } } },
    { "NtGdiEngCheckAbort",   { 4755, { 4751, 4495 }, { 4788, 4564 }, { 4820, 4571 }, { 4859, 4648 }, { 4868, 4672 }, { 4880, 4729 }, { 4881, 4731 } } },
    { "NtGdiEngComputeGlyphSet",  { 4700, { 4696, 4496 }, { 4733, 4565 }, { 4765, 4572 }, { 4791, 4649 }, { 4794, 4673 }, { 4804, 4730 }, { 4805, 4732 } } },
    { "NtGdiEngCopyBits", { 4701, { 4697, -1 }, { 4734, 4566 }, { 4766, 4573 }, { 4790, 4650 }, { 4793, 4674 }, { 4803, 4731 }, { 4804, 4733 } } },
    { "NtGdiEngCreateBitmap", { 4696, { 4692, -1 }, { 4729, 4567 }, { 4761, 4574 }, { 4795, 4651 }, { 4798, 4675 }, { 4808, 4732 }, { 4809, 4734 } } },
    { "NtGdiEngCreateClip",   { 4728, { 4724, 4499 }, { 4761, 4568 }, { 4793, 4575 }, { 4830, 4652 }, { 4839, 4676 }, { 4851, 4733 }, { 4852, 4735 } } },
    { "NtGdiEngCreateDeviceBitmap",   { 4698, { 4694, 4500 }, { 4731, 4569 }, { 4763, 4576 }, { 4793, 4653 }, { 4796, 4677 }, { 4806, 4734 }, { 4807, 4736 } } },
    { "NtGdiEngCreateDeviceSurface",  { 4697, { 4693, 4501 }, { 4730, 4570 }, { 4762, 4577 }, { 4794, 4654 }, { 4797, 4678 }, { 4807, 4735 }, { 4808, 4737 } } },
    { "NtGdiEngCreatePalette",    { 4699, { 4695, -1 }, { 4732, 4571 }, { 4764, 4578 }, { 4792, 4655 }, { 4795, 4679 }, { 4805, 4736 }, { 4806, 4738 } } },
    { "NtGdiEngDeleteClip",   { 4729, { 4725, 4503 }, { 4762, 4572 }, { 4794, 4579 }, { 4829, 4656 }, { 4838, 4680 }, { 4850, 4737 }, { 4851, 4739 } } },
    { "NtGdiEngDeletePalette",    { 4702, { 4698, -1 }, { 4735, 4573 }, { 4767, 4580 }, { 4789, 4657 }, { 4792, 4681 }, { 4802, 4738 }, { 4803, 4740 } } },
    { "NtGdiEngDeletePath",   { 4727, { 4723, 4505 }, { 4760, 4574 }, { 4792, 4581 }, { 4831, 4658 }, { 4840, 4682 }, { 4852, 4739 }, { 4853, 4741 } } },
    { "NtGdiEngDeleteSurface",    { 4703, { 4699, 4506 }, { 4736, 4575 }, { 4768, 4582 }, { 4788, 4659 }, { 4791, 4683 }, { 4801, 4740 }, { 4802, 4742 } } },
    { "NtGdiEngEraseSurface", { 4704, { 4700, 4507 }, { 4737, 4576 }, { 4769, 4583 }, { 4787, 4660 }, { 4790, 4684 }, { 4800, 4741 }, { 4801, 4743 } } },
    { "NtGdiEngFillPath", { 4712, { 4708, -1 }, { 4745, 4577 }, { 4777, 4584 }, { 4779, 4661 }, { 4782, 4685 }, { 4792, 4742 }, { 4793, 4744 } } },
    { "NtGdiEngGradientFill", { 4717, { 4713, -1 }, { 4750, 4578 }, { 4782, 4585 }, { 4774, 4662 }, { 4777, 4686 }, { 4787, 4743 }, { 4788, 4745 } } },
    { "NtGdiEngLineTo",   { 4715, { 4711, -1 }, { 4748, 4579 }, { 4780, 4586 }, { 4776, 4663 }, { 4779, 4687 }, { 4789, 4744 }, { 4790, 4746 } } },
    { "NtGdiEngLockSurface",  { 4706, { 4702, 4511 }, { 4739, 4580 }, { 4771, 4587 }, { 4785, 4664 }, { 4788, 4688 }, { 4798, 4745 }, { 4799, 4747 } } },
    { "NtGdiEngMarkBandingSurface",   { 4710, { 4706, 4512 }, { 4743, 4581 }, { 4775, 4588 }, { 4781, 4665 }, { 4784, 4689 }, { 4794, 4746 }, { 4795, 4748 } } },
    { "NtGdiEngPaint",    { 4714, { 4710, -1 }, { 4747, 4582 }, { 4779, 4589 }, { 4777, 4666 }, { 4780, 4690 }, { 4790, 4747 }, { 4791, 4749 } } },
    { "NtGdiEngPlgBlt",   { 4709, { 4705, -1 }, { 4742, 4583 }, { 4774, 4590 }, { 4782, 4667 }, { 4785, 4691 }, { 4795, 4748 }, { 4796, 4750 } } },
    { "NtGdiEngStretchBlt",   { 4708, { 4704, -1 }, { 4741, 4584 }, { 4773, 4591 }, { 4783, 4668 }, { 4786, 4692 }, { 4796, 4749 }, { 4797, 4751 } } },
    { "NtGdiEngStretchBltROP",    { 4720, { 4716, -1 }, { 4753, 4585 }, { 4785, 4592 }, { 4771, 4669 }, { 4774, 4693 }, { 4784, 4750 }, { 4785, 4752 } } },
    { "NtGdiEngStrokeAndFillPath",    { 4713, { 4709, -1 }, { 4746, 4586 }, { 4778, 4593 }, { 4778, 4670 }, { 4781, 4694 }, { 4791, 4751 }, { 4792, 4753 } } },
    { "NtGdiEngStrokePath",   { 4711, { 4707, -1 }, { 4744, 4587 }, { 4776, 4594 }, { 4780, 4671 }, { 4783, 4695 }, { 4793, 4752 }, { 4794, 4754 } } },
    { "NtGdiEngTextOut",  { 4719, { 4715, -1 }, { 4752, 4588 }, { 4784, 4595 }, { 4772, 4672 }, { 4775, 4696 }, { 4785, 4753 }, { 4786, 4755 } } },
    { "NtGdiEngTransparentBlt",   { 4718, { 4714, -1 }, { 4751, 4589 }, { 4783, 4596 }, { 4773, 4673 }, { 4776, 4697 }, { 4786, 4754 }, { 4787, 4756 } } },
    { "NtGdiEngUnlockSurface",    { 4705, { 4701, 4521 }, { 4738, 4590 }, { 4770, 4597 }, { 4786, 4674 }, { 4789, 4698 }, { 4799, 4755 }, { 4800, 4757 } } },
    { "NtGdiEnumFontChunk",   { 4229, { 4229, -1 }, { 4232, 4262 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiEnumFontClose",   { 4230, { 4230, 4259 }, { 4233, 4260 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiEnumFontOpen",    { 4231, { 4231, -1 }, { 4234, 4261 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiEnumFonts",   { -1, { -1, -1 }, { -1, -1 }, { 4234, 4598 }, { 4272, 4675 }, { 4274, 4699 }, { 4275, 4756 }, { 4277, 4758 } } },
    { "NtGdiEnumObjects", { 4232, { 4232, 4522 }, { 4235, 4591 }, { 4235, 4599 }, { 4271, 4676 }, { 4273, 4700 }, { 4274, 4757 }, { 4276, 4759 } } },
    { "NtGdiEqualRgn",    { 4233, { 4233, 4375 }, { 4236, 4376 }, { 4236, 4369 }, { 4270, 4369 }, { 4272, 4370 }, { 4273, 4371 }, { 4275, 4371 } } },
    { "NtGdiEudcLoadUnloadLink",  { 4234, { 4234, -1 }, { 4237, 4592 }, { 4237, 4600 }, { 4269, 4677 }, { 4271, 4701 }, { 4272, 4758 }, { 4274, 4760 } } },
    { "NtGdiExcludeClipRect", { 4235, { 4235, -1 }, { 4238, 4251 }, { 4238, 4249 }, { 4268, 4249 }, { 4270, 4250 }, { 4271, 4251 }, { 4273, 4251 } } },
    { "NtGdiExtCreatePen",    { 4236, { 4236, -1 }, { 4239, 4272 }, { 4239, 4267 }, { 4267, 4267 }, { 4269, 4268 }, { 4270, 4269 }, { 4272, 4269 } } },
    { "NtGdiExtCreateRegion", { 4237, { 4237, 4235 }, { 4240, 4236 }, { 4240, 4234 }, { 4266, 4234 }, { 4268, 4235 }, { 4269, 4236 }, { 4271, 4236 } } },
    { "NtGdiExtEscape",   { 4238, { 4238, -1 }, { 4241, 4381 }, { 4241, 4374 }, { 4265, 4374 }, { 4267, 4375 }, { 4268, 4376 }, { 4270, 4376 } } },
    { "NtGdiExtFloodFill",    { 4239, { 4239, -1 }, { 4242, 4593 }, { 4242, 4601 }, { 4264, 4678 }, { 4266, 4702 }, { 4267, 4759 }, { 4269, 4761 } } },
    { "NtGdiExtGetObjectW",   { 4240, { 4240, 4177 }, { 4243, 4178 }, { 4243, 4178 }, { 4263, 4179 }, { 4265, 4180 }, { 4266, 4181 }, { 4268, 4181 } } },
    { "NtGdiExtSelectClipRgn",    { 4241, { 4241, 4141 }, { 4244, 4142 }, { 4244, 4142 }, { 4262, 4143 }, { 4264, 4144 }, { 4265, 4145 }, { 4267, 4145 } } },
    { "NtGdiExtTextOutW", { 4242, { 4242, -1 }, { 4245, 4152 }, { 4245, 4152 }, { 4261, 4153 }, { 4263, 4154 }, { 4264, 4155 }, { 4266, 4155 } } },
    { "NtGdiFillPath",    { 4243, { 4243, 4369 }, { 4246, 4370 }, { 4246, 4363 }, { 4260, 4363 }, { 4262, 4364 }, { 4263, 4365 }, { 4265, 4365 } } },
    { "NtGdiFillRgn", { 4244, { 4244, 4312 }, { 4247, 4313 }, { 4247, 4308 }, { 4259, 4308 }, { 4261, 4309 }, { 4262, 4310 }, { 4264, 4310 } } },
    { "NtGdiFlattenPath", { 4245, { 4245, -1 }, { 4248, 4602 }, { 4248, 4610 }, { 4258, 4687 }, { 4260, 4711 }, { 4261, 4768 }, { 4263, 4770 } } },
    { "NtGdiFlush",   { 4247, { 4246, 4113 }, { 4249, 4114 }, { 4249, 4114 }, { 4257, 4115 }, { 4259, 4116 }, { 4260, 4117 }, { 4262, 4117 } } },
    { "NtGdiFlushUserBatch",  { 4246, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiFontIsLinked",    { 4106, { 4106, 4534 }, { 4106, 4603 }, { 4106, 4611 }, { 4402, 4688 }, { 4404, 4712 }, { 4405, 4769 }, { 4407, 4771 } } },
    { "NtGdiFONTOBJ_cGetAllGlyphHandles", { 4743, { 4739, 4525 }, { 4776, 4594 }, { 4808, 4602 }, { 4847, 4679 }, { 4856, 4703 }, { 4868, 4760 }, { 4869, 4762 } } },
    { "NtGdiFONTOBJ_cGetGlyphs",  { 4738, { 4734, -1 }, { 4771, 4595 }, { 4803, 4603 }, { 4844, 4680 }, { 4853, 4704 }, { 4865, 4761 }, { 4866, 4763 } } },
    { "NtGdiFONTOBJ_pfdg",    { 4740, { 4736, 4528 }, { 4773, 4597 }, { 4805, 4605 }, { 4842, 4682 }, { 4851, 4706 }, { 4863, 4763 }, { 4864, 4765 } } },
    { "NtGdiFONTOBJ_pifi",    { 4739, { 4735, 4529 }, { 4772, 4598 }, { 4804, 4606 }, { 4843, 4683 }, { 4852, 4707 }, { 4864, 4764 }, { 4865, 4766 } } },
    { "NtGdiFONTOBJ_pQueryGlyphAttrs",    { 4741, { 4737, 4527 }, { 4774, 4596 }, { 4806, 4604 }, { 4841, 4681 }, { 4850, 4705 }, { 4862, 4762 }, { 4863, 4764 } } },
    { "NtGdiFONTOBJ_pvTrueTypeFontFile",  { 4742, { 4738, 4530 }, { 4775, 4599 }, { 4807, 4607 }, { 4848, 4684 }, { 4857, 4708 }, { 4869, 4765 }, { 4870, 4767 } } },
    { "NtGdiFONTOBJ_pxoGetXform", { 4737, { 4733, 4531 }, { 4770, 4600 }, { 4802, 4608 }, { 4845, 4685 }, { 4854, 4709 }, { 4866, 4766 }, { 4867, 4768 } } },
    { "NtGdiFONTOBJ_vGetInfo",    { 4736, { 4732, 4532 }, { 4769, 4601 }, { 4801, 4609 }, { 4846, 4686 }, { 4855, 4710 }, { 4867, 4767 }, { 4868, 4769 } } },
    { "NtGdiForceUFIMapping", { 4248, { 4247, 4535 }, { 4250, 4604 }, { 4250, 4612 }, { 4256, 4689 }, { 4258, 4713 }, { 4259, 4770 }, { 4261, 4772 } } },
    { "NtGdiFrameRgn",    { 4249, { 4248, -1 }, { 4251, 4605 }, { 4251, 4613 }, { 4255, 4690 }, { 4257, 4714 }, { 4258, 4771 }, { 4260, 4773 } } },
    { "NtGdiFullscreenControl",   { 4250, { 4249, -1 }, { 4252, 4606 }, { 4252, 4614 }, { 4254, 4691 }, { 4256, 4715 }, { 4257, 4772 }, { 4259, 4774 } } },
    { "NtGdiGetAndSetDCDword",    { 4251, { 4250, 4199 }, { 4253, 4200 }, { 4253, 4200 }, { 4253, 4200 }, { 4255, 4201 }, { 4256, 4202 }, { 4258, 4202 } } },
    { "NtGdiGetAppClipBox",   { 4252, { 4251, -1 }, { 4254, 4163 }, { 4254, 4163 }, { 4252, 4164 }, { 4254, 4165 }, { 4255, 4166 }, { 4257, 4166 } } },
    { "NtGdiGetBitmapBits",   { 4253, { 4252, 4322 }, { 4255, 4323 }, { 4255, 4318 }, { 4251, 4318 }, { 4253, 4319 }, { 4254, 4320 }, { 4256, 4320 } } },
    { "NtGdiGetBitmapDimension",  { 4254, { 4253, -1 }, { 4256, 4354 }, { 4256, 4348 }, { 4250, 4348 }, { 4252, 4349 }, { 4253, 4350 }, { 4255, 4350 } } },
    { "NtGdiGetBoundsRect",   { 4255, { 4254, -1 }, { 4257, 4607 }, { 4257, 4615 }, { 4249, 4692 }, { 4251, 4716 }, { 4252, 4773 }, { 4254, 4775 } } },
    { "NtGdiGetCertificate",  { -1, { -1, -1 }, { 4258, 4609 }, { 4258, 4617 }, { 4248, 4694 }, { 4250, 4718 }, { 4251, 4775 }, { 4251, 4777 } } },
    { "NtGdiGetCertificateByHandle",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 4778 }, { 4253, 4778 } } },
    { "NtGdiGetCertificateSize",  { -1, { -1, -1 }, { 4259, 4610 }, { 4259, 4618 }, { 4247, 4695 }, { 4249, 4719 }, { 4250, 4776 }, { 4250, 4779 } } },
    { "NtGdiGetCertificateSizeByHandle",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 4780 }, { 4252, 4780 } } },
    { "NtGdiGetCharABCWidthsW",   { 4256, { 4255, -1 }, { 4260, 4611 }, { 4260, 4619 }, { 4246, 4696 }, { 4248, 4720 }, { 4249, 4777 }, { 4249, 4781 } } },
    { "NtGdiGetCharacterPlacementW",  { 4257, { 4256, -1 }, { 4261, 4612 }, { 4261, 4620 }, { 4245, 4697 }, { 4247, 4721 }, { 4248, 4778 }, { 4248, 4782 } } },
    { "NtGdiGetCharSet",  { 4258, { 4257, 4105 }, { 4262, 4105 }, { 4262, 4105 }, { 4244, 4106 }, { 4246, 4107 }, { 4247, 4108 }, { 4247, 4108 } } },
    { "NtGdiGetCharWidthInfo",    { 4260, { 4259, -1 }, { 4264, 4305 }, { 4264, 4300 }, { 4242, 4300 }, { 4244, 4301 }, { 4245, 4302 }, { 4245, 4302 } } },
    { "NtGdiGetCharWidthW",   { 4259, { 4258, -1 }, { 4263, 4300 }, { 4263, 4295 }, { 4243, 4295 }, { 4245, 4296 }, { 4246, 4297 }, { 4246, 4297 } } },
    { "NtGdiGetColorAdjustment",  { 4261, { 4260, -1 }, { 4265, 4613 }, { 4265, 4621 }, { 4241, 4698 }, { 4243, 4722 }, { 4244, 4779 }, { 4244, 4783 } } },
    { "NtGdiGetColorSpaceforBitmap",  { 4262, { 4261, -1 }, { 4266, 4614 }, { 4266, 4622 }, { 4240, 4699 }, { 4242, 4723 }, { 4243, 4780 }, { 4243, 4784 } } },
    { "NtGdiGetCOPPCompatibleOPMInformation", { -1, { -1, -1 }, { 4267, 4608 }, { 4267, 4616 }, { 4239, 4693 }, { 4241, 4717 }, { 4242, 4774 }, { 4242, 4776 } } },
    { "NtGdiGetCurrentDpiInfo",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4240, 4724 }, { 4241, 4781 }, { 4241, 4785 } } },
    { "NtGdiGetDCDword",  { 4263, { 4262, 4158 }, { 4268, 4159 }, { 4268, 4159 }, { 4238, 4160 }, { 4239, 4161 }, { 4240, 4162 }, { 4240, 4162 } } },
    { "NtGdiGetDCforBitmap",  { 4264, { 4263, -1 }, { 4269, 4253 }, { 4269, 4251 }, { 4237, 4251 }, { 4238, 4252 }, { 4239, 4253 }, { 4239, 4253 } } },
    { "NtGdiGetDCObject", { 4265, { 4264, 4148 }, { 4270, 4149 }, { 4270, 4149 }, { 4236, 4150 }, { 4237, 4151 }, { 4238, 4152 }, { 4238, 4152 } } },
    { "NtGdiGetDCPoint",  { 4266, { 4265, -1 }, { 4271, 4212 }, { 4271, 4211 }, { 4235, 4211 }, { 4236, 4212 }, { 4237, 4213 }, { 4237, 4213 } } },
    { "NtGdiGetDeviceCaps",   { 4267, { 4266, 4543 }, { 4272, 4615 }, { 4272, 4623 }, { 4234, 4700 }, { 4235, 4725 }, { 4236, 4782 }, { 4236, 4786 } } },
    { "NtGdiGetDeviceCapsAll",    { 4269, { 4268, 4544 }, { 4274, 4616 }, { 4274, 4624 }, { 4232, 4701 }, { 4233, 4726 }, { 4234, 4783 }, { 4234, 4787 } } },
    { "NtGdiGetDeviceGammaRamp",  { 4268, { 4267, 4545 }, { 4273, 4617 }, { 4273, 4625 }, { 4233, 4702 }, { 4234, 4727 }, { 4235, 4784 }, { 4235, 4788 } } },
    { "NtGdiGetDeviceWidth",  { 4375, { 4374, -1 }, { 4383, 4618 }, { 4385, 4626 }, { 4121, 4703 }, { 4122, 4728 }, { 4123, 4785 }, { 4123, 4789 } } },
    { "NtGdiGetDhpdev",   { 4754, { 4750, 4547 }, { 4787, 4619 }, { 4819, 4627 }, { 4860, 4704 }, { 4869, 4729 }, { 4881, 4786 }, { 4882, 4790 } } },
    { "NtGdiGetDIBitsInternal",   { 4270, { 4269, -1 }, { 4275, 4231 }, { 4275, 4229 }, { 4231, 4229 }, { 4232, 4230 }, { 4233, 4231 }, { 4233, 4231 } } },
    { "NtGdiGetEmbedFonts",   { 4308, { 4307, 4550 }, { 4316, 4622 }, { 4318, 4630 }, { 4188, 4707 }, { 4189, 4732 }, { 4190, 4789 }, { 4190, 4793 } } },
    { "NtGdiGetEmbUFI",   { 4306, { 4305, -1 }, { 4314, 4621 }, { 4316, 4629 }, { 4190, 4706 }, { 4191, 4731 }, { 4192, 4788 }, { 4192, 4792 } } },
    { "NtGdiGetETM",  { 4271, { 4270, 4548 }, { 4276, 4620 }, { 4276, 4628 }, { 4230, 4705 }, { 4231, 4730 }, { 4232, 4787 }, { 4232, 4791 } } },
    { "NtGdiGetEudcTimeStampEx",  { 4272, { 4271, -1 }, { 4277, 4623 }, { 4277, 4631 }, { 4229, 4708 }, { 4230, 4733 }, { 4231, 4790 }, { 4231, 4794 } } },
    { "NtGdiGetFontData", { 4273, { 4272, -1 }, { 4278, 4316 }, { 4278, 4311 }, { 4228, 4311 }, { 4229, 4312 }, { 4230, 4313 }, { 4230, 4313 } } },
    { "NtGdiGetFontFileData", { -1, { -1, -1 }, { -1, -1 }, { 4279, 4632 }, { 4227, 4709 }, { 4228, 4734 }, { 4229, 4791 }, { 4229, 4795 } } },
    { "NtGdiGetFontFileInfo", { -1, { -1, -1 }, { -1, -1 }, { 4280, 4633 }, { 4226, 4710 }, { 4227, 4735 }, { 4228, 4792 }, { 4228, 4796 } } },
    { "NtGdiGetFontResourceInfoInternalW",    { 4274, { 4273, -1 }, { 4279, 4624 }, { 4281, 4634 }, { 4225, 4711 }, { 4226, 4736 }, { 4227, 4793 }, { 4227, 4797 } } },
    { "NtGdiGetFontUnicodeRanges",    { 4311, { 4310, 4553 }, { 4319, 4625 }, { 4321, 4635 }, { 4185, 4712 }, { 4186, 4737 }, { 4187, 4794 }, { 4187, 4798 } } },
    { "NtGdiGetGlyphIndicesW",    { 4275, { 4274, -1 }, { 4280, 4626 }, { 4282, 4636 }, { 4224, 4713 }, { 4225, 4738 }, { 4226, 4795 }, { 4226, 4799 } } },
    { "NtGdiGetGlyphIndicesWInternal",    { 4276, { 4275, -1 }, { 4281, 4627 }, { 4283, 4637 }, { 4223, 4714 }, { 4224, 4739 }, { 4225, 4796 }, { 4225, 4800 } } },
    { "NtGdiGetGlyphOutline", { 4277, { 4276, -1 }, { 4282, 4628 }, { 4284, 4638 }, { 4222, 4715 }, { 4223, 4740 }, { 4224, 4797 }, { 4224, 4801 } } },
    { "NtGdiGetKerningPairs", { 4278, { 4277, 4557 }, { 4284, 4629 }, { 4286, 4639 }, { 4220, 4716 }, { 4221, 4741 }, { 4222, 4798 }, { 4222, 4802 } } },
    { "NtGdiGetLinkedUFIs",   { 4279, { 4278, 4558 }, { 4285, 4630 }, { 4287, 4640 }, { 4219, 4717 }, { 4220, 4742 }, { 4221, 4799 }, { 4221, 4803 } } },
    { "NtGdiGetMiterLimit",   { 4280, { 4279, -1 }, { 4286, 4631 }, { 4288, 4641 }, { 4218, 4718 }, { 4219, 4743 }, { 4220, 4800 }, { 4220, 4804 } } },
    { "NtGdiGetMonitorID",    { 4281, { 4280, 4560 }, { 4287, 4632 }, { 4289, 4642 }, { 4217, 4719 }, { 4218, 4744 }, { 4219, 4801 }, { 4219, 4805 } } },
    { "NtGdiGetNearestColor", { 4282, { 4281, 4209 }, { 4288, 4210 }, { 4290, 4209 }, { 4216, 4209 }, { 4217, 4210 }, { 4218, 4211 }, { 4218, 4211 } } },
    { "NtGdiGetNearestPaletteIndex",  { 4283, { 4282, 4296 }, { 4289, 4297 }, { 4291, 4292 }, { 4215, 4292 }, { 4216, 4293 }, { 4217, 4294 }, { 4217, 4294 } } },
    { "NtGdiGetNumberOfPhysicalMonitors", { -1, { -1, -1 }, { 4856, 4633 }, { 4900, 4643 }, { 4973, 4720 }, { 4991, 4745 }, { 5048, 4802 }, { 5050, 4806 } } },
    { "NtGdiGetObjectBitmapHandle",   { 4284, { 4283, -1 }, { 4290, 4636 }, { 4292, 4646 }, { 4214, 4723 }, { 4215, 4748 }, { 4216, 4805 }, { 4216, 4809 } } },
    { "NtGdiGetOPMInformation",   { -1, { -1, -1 }, { 4283, 4634 }, { 4285, 4644 }, { 4221, 4721 }, { 4222, 4746 }, { 4223, 4803 }, { 4223, 4807 } } },
    { "NtGdiGetOPMRandomNumber",  { -1, { -1, -1 }, { 4291, 4635 }, { 4293, 4645 }, { 4213, 4722 }, { 4214, 4747 }, { 4215, 4804 }, { 4215, 4808 } } },
    { "NtGdiGetOutlineTextMetricsInternalW",  { 4285, { 4284, 4279 }, { 4292, 4280 }, { 4294, 4275 }, { 4212, 4275 }, { 4213, 4276 }, { 4214, 4277 }, { 4214, 4277 } } },
    { "NtGdiGetPath", { 4286, { 4285, 4562 }, { 4293, 4637 }, { 4295, 4647 }, { 4211, 4724 }, { 4212, 4749 }, { 4213, 4806 }, { 4213, 4810 } } },
    { "NtGdiGetPerBandInfo",  { 4220, { 4220, -1 }, { 4223, 4638 }, { 4224, 4648 }, { 4282, 4725 }, { 4284, 4750 }, { 4285, 4807 }, { 4287, 4811 } } },
    { "NtGdiGetPhysicalMonitorDescription",   { -1, { -1, -1 }, { 4858, 4639 }, { 4902, 4649 }, { 4971, 4726 }, { 4989, 4751 }, { 5046, 4808 }, { 5048, 4812 } } },
    { "NtGdiGetPhysicalMonitors", { -1, { -1, -1 }, { 4857, 4640 }, { 4901, 4650 }, { 4972, 4727 }, { 4990, 4752 }, { 5047, 4809 }, { 5049, 4813 } } },
    { "NtGdiGetPixel",    { 4287, { 4286, 4291 }, { 4294, 4292 }, { 4296, 4287 }, { 4210, 4287 }, { 4211, 4288 }, { 4212, 4289 }, { 4212, 4289 } } },
    { "NtGdiGetRandomRgn",    { 4288, { 4287, 4138 }, { 4295, 4139 }, { 4297, 4139 }, { 4209, 4140 }, { 4210, 4141 }, { 4211, 4142 }, { 4211, 4142 } } },
    { "NtGdiGetRasterizerCaps",   { 4289, { 4288, 4331 }, { 4296, 4332 }, { 4298, 4327 }, { 4208, 4327 }, { 4209, 4328 }, { 4210, 4329 }, { 4210, 4329 } } },
    { "NtGdiGetRealizationInfo",  { 4290, { 4289, 4564 }, { 4297, 4641 }, { 4299, 4651 }, { 4207, 4728 }, { 4208, 4753 }, { 4209, 4810 }, { 4209, 4814 } } },
    { "NtGdiGetRegionData",   { 4291, { 4290, 4159 }, { 4298, 4160 }, { 4300, 4160 }, { 4206, 4161 }, { 4207, 4162 }, { 4208, 4163 }, { 4208, 4163 } } },
    { "NtGdiGetRgnBox",   { 4292, { 4291, -1 }, { 4299, 4199 }, { 4301, 4199 }, { 4205, 4199 }, { 4206, 4200 }, { 4207, 4201 }, { 4207, 4201 } } },
    { "NtGdiGetServerMetaFileBits",   { 4293, { 4292, -1 }, { 4300, 4642 }, { 4302, 4652 }, { 4204, 4729 }, { 4205, 4754 }, { 4206, 4811 }, { 4206, 4815 } } },
    { "NtGdiGetSpoolMessage", { 4294, { 4293, 4566 }, { 4301, 4643 }, { 4303, 4653 }, { 4203, 4730 }, { 4204, 4755 }, { 4205, 4812 }, { 4205, 4816 } } },
    { "NtGdiGetStats",    { 4295, { 4294, -1 }, { 4302, 4644 }, { 4304, 4654 }, { 4202, 4731 }, { 4203, 4756 }, { 4204, 4813 }, { 4204, 4817 } } },
    { "NtGdiGetStockObject",  { 4296, { 4295, 4308 }, { 4303, 4309 }, { 4305, 4304 }, { 4201, 4304 }, { 4202, 4305 }, { 4203, 4306 }, { 4203, 4306 } } },
    { "NtGdiGetStringBitmapW",    { 4297, { 4296, -1 }, { 4304, 4645 }, { 4306, 4655 }, { 4200, 4732 }, { 4201, 4757 }, { 4202, 4814 }, { 4202, 4818 } } },
    { "NtGdiGetSuggestedOPMProtectedOutputArraySize", { -1, { -1, -1 }, { 4305, 4646 }, { 4307, 4656 }, { 4199, 4733 }, { 4200, 4758 }, { 4201, 4815 }, { 4201, 4819 } } },
    { "NtGdiGetSystemPaletteUse", { 4298, { 4297, 4376 }, { 4306, 4377 }, { 4308, 4370 }, { 4198, 4370 }, { 4199, 4371 }, { 4200, 4372 }, { 4200, 4372 } } },
    { "NtGdiGetTextCharsetInfo",  { 4299, { 4298, -1 }, { 4307, 4173 }, { 4309, 4173 }, { 4197, 4174 }, { 4198, 4175 }, { 4199, 4176 }, { 4199, 4176 } } },
    { "NtGdiGetTextExtent",   { 4300, { 4299, -1 }, { 4308, 4246 }, { 4310, 4244 }, { 4196, 4244 }, { 4197, 4245 }, { 4198, 4246 }, { 4198, 4246 } } },
    { "NtGdiGetTextExtentExW",    { 4301, { 4300, -1 }, { 4309, 4647 }, { 4311, 4657 }, { 4195, 4734 }, { 4196, 4759 }, { 4197, 4816 }, { 4197, 4820 } } },
    { "NtGdiGetTextFaceW",    { 4302, { 4301, 4225 }, { 4310, 4226 }, { 4312, 4224 }, { 4194, 4224 }, { 4195, 4225 }, { 4196, 4226 }, { 4196, 4226 } } },
    { "NtGdiGetTextMetricsW", { 4303, { 4302, 4214 }, { 4311, 4215 }, { 4313, 4213 }, { 4193, 4213 }, { 4194, 4214 }, { 4195, 4215 }, { 4195, 4215 } } },
    { "NtGdiGetTransform",    { 4304, { 4303, -1 }, { 4312, 4321 }, { 4314, 4316 }, { 4192, 4316 }, { 4193, 4317 }, { 4194, 4318 }, { 4194, 4318 } } },
    { "NtGdiGetUFI",  { 4305, { 4304, -1 }, { 4313, 4648 }, { 4315, 4658 }, { 4191, 4735 }, { 4192, 4760 }, { 4193, 4817 }, { 4193, 4821 } } },
    { "NtGdiGetUFIPathname",  { 4307, { 4306, -1 }, { 4315, 4649 }, { 4317, 4659 }, { 4189, 4736 }, { 4190, 4761 }, { 4191, 4818 }, { 4191, 4822 } } },
    { "NtGdiGetWidthTable",   { 4312, { 4311, -1 }, { 4320, 4202 }, { 4322, 4202 }, { 4184, 4202 }, { 4185, 4203 }, { 4186, 4204 }, { 4186, 4204 } } },
    { "NtGdiGradientFill",    { 4313, { 4312, -1 }, { 4321, 4650 }, { 4323, 4660 }, { 4183, 4737 }, { 4184, 4762 }, { 4185, 4819 }, { 4185, 4823 } } },
    { "NtGdiHfontCreate", { 4314, { 4313, -1 }, { 4322, 4189 }, { 4324, 4189 }, { 4182, 4189 }, { 4183, 4190 }, { 4184, 4191 }, { 4184, 4191 } } },
    { "NtGdiHLSurfGetInformation",    { -1, { -1, -1 }, { -1, -1 }, { 4829, 4661 }, { 4870, 4738 }, { 4879, 4763 }, { 4891, 4820 }, { 4892, 4824 } } },
    { "NtGdiHLSurfSetInformation",    { -1, { -1, -1 }, { -1, -1 }, { 4830, 4662 }, { 4869, 4739 }, { 4878, 4764 }, { 4890, 4821 }, { 4891, 4825 } } },
    { "NtGdiHT_Get8BPPFormatPalette", { 4756, { 4752, 4573 }, { 4789, 4651 }, { 4821, 4663 }, { 4862, 4740 }, { 4871, 4765 }, { 4883, 4822 }, { 4884, 4826 } } },
    { "NtGdiHT_Get8BPPMaskPalette",   { 4757, { 4753, -1 }, { 4790, 4652 }, { 4822, 4664 }, { 4861, 4741 }, { 4870, 4766 }, { 4882, 4823 }, { 4883, 4827 } } },
    { "NtGdiIcmBrushInfo",    { 4315, { 4314, -1 }, { 4323, 4653 }, { 4325, 4665 }, { 4181, 4742 }, { 4182, 4767 }, { 4183, 4824 }, { 4183, 4828 } } },
    { "NtGdiInit",    { 4316, { 4315, 4636 }, { 4324, 4654 }, { 4326, 4666 }, { 4180, 4743 }, { 4181, 4768 }, { 4182, 4825 }, { 4182, 4829 } } },
    { "NtGdiInitSpool",   { 4317, { 4316, 4577 }, { 4325, 4655 }, { 4327, 4667 }, { 4179, 4744 }, { 4180, 4769 }, { 4181, 4826 }, { 4181, 4830 } } },
    { "NtGdiIntersectClipRect",   { 4318, { 4317, -1 }, { 4326, 4128 }, { 4328, 4128 }, { 4178, 4129 }, { 4179, 4130 }, { 4180, 4131 }, { 4180, 4131 } } },
    { "NtGdiInvertRgn",   { 4319, { 4318, 4197 }, { 4327, 4198 }, { 4329, 4198 }, { 4177, 4198 }, { 4178, 4199 }, { 4179, 4200 }, { 4179, 4200 } } },
    { "NtGdiLineTo",  { 4320, { 4319, 4160 }, { 4328, 4161 }, { 4330, 4161 }, { 4176, 4162 }, { 4177, 4163 }, { 4178, 4164 }, { 4178, 4164 } } },
    { "NtGdiMakeFontDir", { 4321, { 4320, -1 }, { 4329, 4656 }, { 4331, 4668 }, { 4175, 4745 }, { 4176, 4770 }, { 4177, 4827 }, { 4177, 4831 } } },
    { "NtGdiMakeInfoDC",  { 4322, { 4321, -1 }, { 4330, 4657 }, { 4332, 4669 }, { 4174, 4746 }, { 4175, 4771 }, { 4176, 4828 }, { 4176, 4832 } } },
    { "NtGdiMakeObjectUnXferable",    { -1, { 4760, 4580 }, { 4855, 4658 }, { 4899, 4670 }, { 4968, 4747 }, { 4986, 4772 }, { 5043, 4829 }, { 5045, 4833 } } },
    { "NtGdiMakeObjectXferable",  { -1, { 4759, 4581 }, { 4854, 4659 }, { 4898, 4671 }, { 4969, 4748 }, { 4987, 4773 }, { 5044, 4830 }, { 5046, 4834 } } },
    { "NtGdiMaskBlt", { 4323, { 4322, -1 }, { 4331, 4201 }, { 4333, 4201 }, { 4173, 4201 }, { 4174, 4202 }, { 4175, 4203 }, { 4175, 4203 } } },
    { "NtGdiMirrorWindowOrg", { 4376, { 4375, 4582 }, { 4384, 4660 }, { 4386, 4672 }, { 4120, 4749 }, { 4121, 4774 }, { 4122, 4831 }, { 4122, 4835 } } },
    { "NtGdiModifyWorldTransform",    { 4324, { 4323, 4314 }, { 4332, 4315 }, { 4334, 4310 }, { 4172, 4310 }, { 4173, 4311 }, { 4174, 4312 }, { 4174, 4312 } } },
    { "NtGdiMonoBitmap",  { 4325, { 4324, -1 }, { 4333, 4661 }, { 4335, 4673 }, { 4171, 4750 }, { 4172, 4775 }, { 4173, 4832 }, { 4173, 4836 } } },
    { "NtGdiMoveTo",  { 4326, { 4325, -1 }, { 4334, 4662 }, { 4336, 4674 }, { 4170, 4751 }, { 4171, 4776 }, { 4172, 4833 }, { 4172, 4837 } } },
    { "NtGdiOffsetClipRgn",   { 4327, { 4326, 4585 }, { 4335, 4663 }, { 4337, 4675 }, { 4169, 4752 }, { 4170, 4777 }, { 4171, 4834 }, { 4171, 4838 } } },
    { "NtGdiOffsetRgn",   { 4328, { 4327, 4223 }, { 4336, 4224 }, { 4338, 4222 }, { 4168, 4222 }, { 4169, 4223 }, { 4170, 4224 }, { 4170, 4224 } } },
    { "NtGdiOpenDCW", { 4329, { 4328, -1 }, { 4337, 4319 }, { 4339, 4314 }, { 4167, 4314 }, { 4168, 4315 }, { 4169, 4316 }, { 4169, 4316 } } },
    { "NtGdiPatBlt",  { 4330, { 4329, -1 }, { 4338, 4186 }, { 4340, 4186 }, { 4166, 4186 }, { 4167, 4187 }, { 4168, 4188 }, { 4168, 4188 } } },
    { "NtGdiPATHOBJ_bEnum",   { 4750, { 4746, 4586 }, { 4783, 4664 }, { 4815, 4676 }, { 4857, 4753 }, { 4866, 4778 }, { 4878, 4835 }, { 4879, 4839 } } },
    { "NtGdiPATHOBJ_bEnumClipLines",  { 4753, { 4749, 4587 }, { 4786, 4665 }, { 4818, 4677 }, { 4854, 4754 }, { 4863, 4779 }, { 4875, 4836 }, { 4876, 4840 } } },
    { "NtGdiPATHOBJ_vEnumStart",  { 4751, { 4747, -1 }, { 4784, 4666 }, { 4816, 4678 }, { 4856, 4755 }, { 4865, 4780 }, { 4877, 4837 }, { 4878, 4841 } } },
    { "NtGdiPATHOBJ_vEnumStartClipLines", { 4752, { 4748, 4589 }, { 4785, 4667 }, { 4817, 4679 }, { 4855, 4756 }, { 4864, 4781 }, { 4876, 4838 }, { 4877, 4842 } } },
    { "NtGdiPATHOBJ_vGetBounds",  { 4749, { 4745, 4590 }, { 4782, 4668 }, { 4814, 4680 }, { 4858, 4757 }, { 4867, 4782 }, { 4879, 4839 }, { 4880, 4843 } } },
    { "NtGdiPathToRegion",    { 4332, { 4331, 4591 }, { 4340, 4669 }, { 4342, 4681 }, { 4164, 4758 }, { 4165, 4783 }, { 4166, 4840 }, { 4166, 4844 } } },
    { "NtGdiPlgBlt",  { 4333, { 4332, -1 }, { 4341, 4670 }, { 4343, 4682 }, { 4163, 4759 }, { 4164, 4784 }, { 4165, 4841 }, { 4165, 4845 } } },
    { "NtGdiPolyDraw",    { 4334, { 4333, 4593 }, { 4342, 4671 }, { 4344, 4683 }, { 4162, 4760 }, { 4163, 4785 }, { 4164, 4842 }, { 4164, 4846 } } },
    { "NtGdiPolyPatBlt",  { 4331, { 4330, -1 }, { 4339, 4208 }, { 4341, 4207 }, { 4165, 4207 }, { 4166, 4208 }, { 4167, 4209 }, { 4167, 4209 } } },
    { "NtGdiPolyPolyDraw",    { 4335, { 4334, -1 }, { 4343, 4168 }, { 4345, 4168 }, { 4161, 4169 }, { 4162, 4170 }, { 4163, 4171 }, { 4163, 4171 } } },
    { "NtGdiPolyTextOutW",    { 4336, { 4335, 4594 }, { 4344, 4672 }, { 4346, 4684 }, { 4160, 4761 }, { 4161, 4786 }, { 4162, 4843 }, { 4162, 4847 } } },
    { "NtGdiPtInRegion",  { 4337, { 4336, 4595 }, { 4345, 4673 }, { 4347, 4685 }, { 4159, 4762 }, { 4160, 4787 }, { 4161, 4844 }, { 4161, 4848 } } },
    { "NtGdiPtVisible",   { 4338, { 4337, 4596 }, { 4346, 4674 }, { 4348, 4686 }, { 4158, 4763 }, { 4159, 4788 }, { 4160, 4845 }, { 4160, 4849 } } },
    { "NtGdiQueryFontAssocInfo",  { 4340, { 4339, 4345 }, { 4348, 4346 }, { 4350, 4341 }, { 4156, 4341 }, { 4157, 4342 }, { 4158, 4343 }, { 4158, 4343 } } },
    { "NtGdiQueryFonts",  { 4339, { 4338, 4597 }, { 4347, 4675 }, { 4349, 4687 }, { 4157, 4764 }, { 4158, 4789 }, { 4159, 4846 }, { 4159, 4850 } } },
    { "NtGdiRectangle",   { 4341, { 4340, -1 }, { 4349, 4242 }, { 4351, 4240 }, { 4155, 4240 }, { 4156, 4241 }, { 4157, 4242 }, { 4157, 4242 } } },
    { "NtGdiRectInRegion",    { 4342, { 4341, 4289 }, { 4350, 4290 }, { 4352, 4285 }, { 4154, 4285 }, { 4155, 4286 }, { 4156, 4287 }, { 4156, 4287 } } },
    { "NtGdiRectVisible", { 4343, { 4342, 4146 }, { 4351, 4147 }, { 4353, 4147 }, { 4153, 4148 }, { 4154, 4149 }, { 4155, 4150 }, { 4155, 4150 } } },
    { "NtGdiRemoveFontMemResourceEx", { 4345, { 4344, 4377 }, { 4353, 4378 }, { 4355, 4371 }, { 4151, 4371 }, { 4152, 4372 }, { 4153, 4373 }, { 4153, 4373 } } },
    { "NtGdiRemoveFontResourceW", { 4344, { 4343, -1 }, { 4352, 4676 }, { 4354, 4688 }, { 4152, 4765 }, { 4153, 4790 }, { 4154, 4847 }, { 4154, 4851 } } },
    { "NtGdiRemoveMergeFont", { 4101, { 4101, 4599 }, { 4101, 4677 }, { 4101, 4689 }, { 4407, 4766 }, { 4409, 4791 }, { 4410, 4848 }, { 4412, 4852 } } },
    { "NtGdiResetDC", { 4346, { 4345, -1 }, { 4354, 4678 }, { 4356, 4690 }, { 4150, 4767 }, { 4151, 4792 }, { 4152, 4849 }, { 4152, 4853 } } },
    { "NtGdiResizePalette",   { 4347, { 4346, 4601 }, { 4355, 4679 }, { 4357, 4691 }, { 4149, 4768 }, { 4150, 4793 }, { 4151, 4850 }, { 4151, 4854 } } },
    { "NtGdiRestoreDC",   { 4348, { 4347, 4153 }, { 4356, 4154 }, { 4358, 4154 }, { 4148, 4155 }, { 4149, 4156 }, { 4150, 4157 }, { 4150, 4157 } } },
    { "NtGdiRoundRect",   { 4349, { 4348, -1 }, { 4357, 4680 }, { 4359, 4692 }, { 4147, 4769 }, { 4148, 4794 }, { 4149, 4851 }, { 4149, 4855 } } },
    { "NtGdiSaveDC",  { 4350, { 4349, 4154 }, { 4358, 4155 }, { 4360, 4155 }, { 4146, 4156 }, { 4147, 4157 }, { 4148, 4158 }, { 4148, 4158 } } },
    { "NtGdiScaleViewportExtEx",  { 4351, { 4350, -1 }, { 4359, 4686 }, { 4361, 4698 }, { 4145, 4775 }, { 4146, 4800 }, { 4147, 4857 }, { 4147, 4861 } } },
    { "NtGdiScaleWindowExtEx",    { 4352, { 4351, -1 }, { 4360, 4687 }, { 4362, 4699 }, { 4144, 4776 }, { 4145, 4801 }, { 4146, 4858 }, { 4146, 4862 } } },
    { "NtGdiSelectBitmap",    { 4353, { 4352, 4107 }, { 4361, 4107 }, { 4363, 4107 }, { 4143, 4108 }, { 4144, 4109 }, { 4145, 4110 }, { 4145, 4110 } } },
    { "NtGdiSelectBrush", { 4354, { 4353, -1 }, { 4362, 4688 }, { 4364, 4700 }, { 4142, 4777 }, { 4143, 4802 }, { 4144, 4859 }, { 4144, 4863 } } },
    { "NtGdiSelectClipPath",  { 4355, { 4354, 4611 }, { 4363, 4689 }, { 4365, 4701 }, { 4141, 4778 }, { 4142, 4803 }, { 4143, 4860 }, { 4143, 4864 } } },
    { "NtGdiSelectFont",  { 4356, { 4355, 4152 }, { 4364, 4153 }, { 4366, 4153 }, { 4140, 4154 }, { 4141, 4155 }, { 4142, 4156 }, { 4142, 4156 } } },
    { "NtGdiSelectPen",   { 4357, { 4356, -1 }, { 4365, 4690 }, { 4367, 4702 }, { 4139, 4779 }, { 4140, 4804 }, { 4141, 4861 }, { 4141, 4865 } } },
    { "NtGdiSetBitmapAttributes", { 4358, { 4357, 4613 }, { 4366, 4691 }, { 4368, 4703 }, { 4138, 4780 }, { 4139, 4805 }, { 4140, 4862 }, { 4140, 4866 } } },
    { "NtGdiSetBitmapBits",   { 4359, { 4358, 4280 }, { 4367, 4281 }, { 4369, 4276 }, { 4137, 4276 }, { 4138, 4277 }, { 4139, 4278 }, { 4139, 4278 } } },
    { "NtGdiSetBitmapDimension",  { 4360, { 4359, -1 }, { 4368, 4382 }, { 4370, 4375 }, { 4136, 4375 }, { 4137, 4376 }, { 4138, 4377 }, { 4138, 4377 } } },
    { "NtGdiSetBoundsRect",   { 4361, { 4360, 4351 }, { 4369, 4352 }, { 4371, 4347 }, { 4135, 4347 }, { 4136, 4348 }, { 4137, 4349 }, { 4137, 4349 } } },
    { "NtGdiSetBrushAttributes",  { 4362, { 4114, 4614 }, { 4370, 4692 }, { 4372, 4704 }, { 4134, 4781 }, { 4135, 4806 }, { 4136, 4863 }, { 4136, 4867 } } },
    { "NtGdiSetBrushOrg", { 4363, { 4362, -1 }, { 4371, 4274 }, { 4373, 4269 }, { 4133, 4269 }, { 4134, 4270 }, { 4135, 4271 }, { 4135, 4271 } } },
    { "NtGdiSetColorAdjustment",  { 4364, { 4363, 4615 }, { 4372, 4693 }, { 4374, 4705 }, { 4132, 4782 }, { 4133, 4807 }, { 4134, 4864 }, { 4134, 4868 } } },
    { "NtGdiSetColorSpace",   { 4365, { 4364, 4616 }, { 4373, 4694 }, { 4375, 4706 }, { 4131, 4783 }, { 4132, 4808 }, { 4133, 4865 }, { 4133, 4869 } } },
    { "NtGdiSetDeviceGammaRamp",  { 4366, { 4365, 4617 }, { 4374, 4695 }, { 4376, 4707 }, { 4130, 4784 }, { 4131, 4809 }, { 4132, 4866 }, { 4132, 4870 } } },
    { "NtGdiSetDIBitsToDeviceInternal",   { 4367, { 4366, -1 }, { 4375, 4137 }, { 4377, 4137 }, { 4129, 4138 }, { 4130, 4139 }, { 4131, 4140 }, { 4131, 4140 } } },
    { "NtGdiSetFontEnumeration",  { 4368, { 4367, 4382 }, { 4376, 4383 }, { 4378, 4376 }, { 4128, 4376 }, { 4129, 4377 }, { 4130, 4378 }, { 4130, 4378 } } },
    { "NtGdiSetFontXform",    { 4369, { 4368, 4618 }, { 4377, 4696 }, { 4379, 4708 }, { 4127, 4785 }, { 4128, 4810 }, { 4129, 4867 }, { 4129, 4871 } } },
    { "NtGdiSetIcmMode",  { 4370, { 4369, 4619 }, { 4378, 4697 }, { 4380, 4709 }, { 4126, 4786 }, { 4127, 4811 }, { 4128, 4868 }, { 4128, 4872 } } },
    { "NtGdiSetLayout",   { 4377, { 4376, 4247 }, { 4385, 4248 }, { 4387, 4246 }, { 4119, 4246 }, { 4120, 4247 }, { 4121, 4248 }, { 4121, 4248 } } },
    { "NtGdiSetLinkedUFIs",   { 4371, { 4370, 4620 }, { 4379, 4698 }, { 4381, 4710 }, { 4125, 4787 }, { 4126, 4812 }, { 4127, 4869 }, { 4127, 4873 } } },
    { "NtGdiSetMagicColors",  { 4372, { 4371, 4621 }, { 4380, 4699 }, { 4382, 4711 }, { 4124, 4788 }, { 4125, 4813 }, { 4126, 4870 }, { 4126, 4874 } } },
    { "NtGdiSetMetaRgn",  { 4373, { 4372, -1 }, { 4381, 4329 }, { 4383, 4324 }, { 4123, 4324 }, { 4124, 4325 }, { 4125, 4326 }, { 4125, 4326 } } },
    { "NtGdiSetMiterLimit",   { 4374, { 4373, 4329 }, { 4382, 4330 }, { 4384, 4325 }, { 4122, 4325 }, { 4123, 4326 }, { 4124, 4327 }, { 4124, 4327 } } },
    { "NtGdiSetOPMSigningKeyAndSequenceNumbers",  { -1, { -1, -1 }, { 4386, 4700 }, { 4388, 4712 }, { 4118, 4789 }, { 4119, 4814 }, { 4120, 4871 }, { 4120, 4875 } } },
    { "NtGdiSetPixel",    { 4378, { 4377, 4275 }, { 4387, 4276 }, { 4389, 4271 }, { 4117, 4271 }, { 4118, 4272 }, { 4119, 4273 }, { 4119, 4273 } } },
    { "NtGdiSetPixelFormat",  { 4379, { 4378, 4623 }, { 4388, 4702 }, { 4390, 4714 }, { 4116, 4791 }, { 4117, 4816 }, { 4118, 4873 }, { 4118, 4877 } } },
    { "NtGdiSetPUMPDOBJ", { 4759, { 4755, 4622 }, { 4792, 4701 }, { 4824, 4713 }, { 4866, 4790 }, { 4875, 4815 }, { 4887, 4872 }, { 4888, 4876 } } },
    { "NtGdiSetRectRgn",  { 4380, { 4379, -1 }, { 4389, 4703 }, { 4391, 4715 }, { 4115, 4792 }, { 4116, 4817 }, { 4117, 4874 }, { 4117, 4878 } } },
    { "NtGdiSetSizeDevice",   { 4385, { 4384, 4625 }, { 4394, 4704 }, { 4395, 4716 }, { 4111, 4793 }, { 4112, 4818 }, { 4113, 4875 }, { 4113, 4879 } } },
    { "NtGdiSetSystemPaletteUse", { 4381, { 4380, 4626 }, { 4390, 4705 }, { 4392, 4717 }, { 4114, 4794 }, { 4115, 4819 }, { 4116, 4876 }, { 4116, 4880 } } },
    { "NtGdiSetTextJustification",    { 4382, { 4381, 4627 }, { 4391, 4706 }, { 4393, 4718 }, { 4113, 4795 }, { 4114, 4820 }, { 4115, 4877 }, { 4115, 4881 } } },
    { "NtGdiSetUMPDSandboxState", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4867, 4796 }, { 4876, 4821 }, { 4888, 4878 }, { 4889, 4882 } } },
    { "NtGdiSetupPublicCFONT",    { 4383, { 4382, 4363 }, { 4392, 4364 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiSetVirtualResolution",    { 4384, { 4383, -1 }, { 4393, 4331 }, { 4394, 4326 }, { 4112, 4326 }, { 4113, 4327 }, { 4114, 4328 }, { 4114, 4328 } } },
    { "NtGdiSfmGetNotificationTokens",    { -1, { -1, -1 }, { -1, -1 }, { 4828, 4719 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtGdiStartDoc",    { 4386, { 4385, 4628 }, { 4395, 4707 }, { 4396, 4720 }, { 4110, 4797 }, { 4111, 4822 }, { 4112, 4879 }, { 4112, 4883 } } },
    { "NtGdiStartPage",   { 4387, { 4386, 4629 }, { 4396, 4708 }, { 4397, 4721 }, { 4109, 4798 }, { 4110, 4823 }, { 4111, 4880 }, { 4111, 4884 } } },
    { "NtGdiStretchBlt",  { 4388, { 4387, -1 }, { 4397, 4145 }, { 4398, 4145 }, { 4108, 4146 }, { 4109, 4147 }, { 4110, 4148 }, { 4110, 4148 } } },
    { "NtGdiStretchDIBitsInternal",   { 4389, { 4388, -1 }, { 4398, 4227 }, { 4399, 4225 }, { 4107, 4225 }, { 4108, 4226 }, { 4109, 4227 }, { 4109, 4227 } } },
    { "NtGdiSTROBJ_bEnum",    { 4744, { 4740, 4603 }, { 4777, 4681 }, { 4809, 4693 }, { 4853, 4770 }, { 4862, 4795 }, { 4874, 4852 }, { 4875, 4856 } } },
    { "NtGdiSTROBJ_bEnumPositionsOnly",   { 4745, { 4741, 4604 }, { 4778, 4682 }, { 4810, 4694 }, { 4852, 4771 }, { 4861, 4796 }, { 4873, 4853 }, { 4874, 4857 } } },
    { "NtGdiSTROBJ_bGetAdvanceWidths",    { 4746, { 4742, 4605 }, { 4779, 4683 }, { 4811, 4695 }, { 4851, 4772 }, { 4860, 4797 }, { 4872, 4854 }, { 4873, 4858 } } },
    { "NtGdiSTROBJ_dwGetCodePage",    { 4748, { 4744, -1 }, { 4781, 4684 }, { 4813, 4696 }, { 4849, 4773 }, { 4858, 4798 }, { 4870, 4855 }, { 4871, 4859 } } },
    { "NtGdiSTROBJ_vEnumStart",   { 4747, { 4743, -1 }, { 4780, 4685 }, { 4812, 4697 }, { 4850, 4774 }, { 4859, 4799 }, { 4871, 4856 }, { 4872, 4860 } } },
    { "NtGdiStrokeAndFillPath",   { 4390, { 4389, 4630 }, { 4399, 4709 }, { 4400, 4722 }, { 4106, 4799 }, { 4107, 4824 }, { 4108, 4881 }, { 4108, 4885 } } },
    { "NtGdiStrokePath",  { 4391, { 4390, 4631 }, { 4400, 4710 }, { 4401, 4723 }, { 4105, 4800 }, { 4106, 4825 }, { 4107, 4882 }, { 4107, 4886 } } },
    { "NtGdiSwapBuffers", { 4392, { 4391, 4632 }, { 4401, 4711 }, { 4402, 4724 }, { 4104, 4801 }, { 4105, 4826 }, { 4106, 4883 }, { 4106, 4887 } } },
    { "NtGdiTransformPoints", { 4393, { 4392, -1 }, { 4402, 4211 }, { 4403, 4210 }, { 4103, 4210 }, { 4104, 4211 }, { 4105, 4212 }, { 4105, 4212 } } },
    { "NtGdiTransparentBlt",  { 4394, { 4393, -1 }, { 4403, 4712 }, { 4404, 4725 }, { 4102, 4802 }, { 4103, 4827 }, { 4104, 4884 }, { 4104, 4888 } } },
    { "NtGdiUMPDEngFreeUserMem",  { 4396, { 4395, 4634 }, { 4405, 4713 }, { 4406, 4726 }, { 4864, 4803 }, { 4873, 4828 }, { 4885, 4885 }, { 4886, 4889 } } },
    { "NtGdiUnloadPrinterDriver", { 4395, { 4394, 4635 }, { 4404, 4714 }, { 4115, 4727 }, { 4101, 4804 }, { 4102, 4829 }, { 4103, 4886 }, { 4103, 4890 } } },
    { "NtGdiUnmapMemFont",    { 4761, { 4757, 4576 }, { 4794, 4715 }, { 4826, 4728 }, { 4100, 4805 }, { 4101, 4830 }, { 4102, 4887 }, { 4102, 4891 } } },
    { "NtGdiUnrealizeObject", { 4397, { 4396, 4239 }, { 4406, 4240 }, { 4407, 4238 }, { 4099, 4238 }, { 4100, 4239 }, { 4101, 4240 }, { 4101, 4240 } } },
    { "NtGdiUpdateColors",    { 4398, { 4397, 4637 }, { 4407, 4716 }, { 4408, 4729 }, { 4098, 4806 }, { 4099, 4831 }, { 4100, 4888 }, { 4100, 4892 } } },
    { "NtGdiUpdateTransform", { 4758, { 4754, 4638 }, { 4791, 4717 }, { 4823, 4730 }, { 4863, 4807 }, { 4872, 4832 }, { 4884, 4889 }, { 4885, 4893 } } },
    { "NtGdiWidenPath",   { 4399, { 4398, 4639 }, { 4408, 4718 }, { 4409, 4731 }, { 4097, 4808 }, { 4098, 4833 }, { 4099, 4890 }, { 4099, 4894 } } },
    { "NtGdiXFORMOBJ_bApplyXform",    { 4734, { 4730, -1 }, { 4767, 4719 }, { 4799, 4732 }, { 4840, 4809 }, { 4849, 4834 }, { 4861, 4891 }, { 4862, 4895 } } },
    { "NtGdiXFORMOBJ_iGetXform",  { 4735, { 4731, 4641 }, { 4768, 4720 }, { 4800, 4733 }, { 4839, 4810 }, { 4848, 4835 }, { 4860, 4892 }, { 4861, 4896 } } },
    { "NtGdiXLATEOBJ_cGetPalette",    { 4721, { 4717, 4642 }, { 4754, 4721 }, { 4786, 4734 }, { 4828, 4811 }, { 4837, 4836 }, { 4849, 4893 }, { 4850, 4897 } } },
    { "NtGdiXLATEOBJ_hGetColorTransform", { 4723, { 4719, 4643 }, { 4756, 4722 }, { 4788, 4735 }, { 4826, 4812 }, { 4835, 4837 }, { 4847, 4894 }, { 4848, 4898 } } },
    { "NtGdiXLATEOBJ_iXlate", { 4722, { 4718, 4644 }, { 4755, 4723 }, { 4787, 4736 }, { 4827, 4813 }, { 4836, 4838 }, { 4848, 4895 }, { 4849, 4899 } } },
    { "NtHWCursorUpdatePointer",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5223, 4896 }, { 5228, 4900 } } },
    { "NtNotifyPresentToCompositionSurface",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4997, 4814 }, { 5015, 4839 }, { 5072, 4897 }, { 5074, 4901 } } },
    { "NtOpenCompositionSurfaceDirtyRegion",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5002, 4815 }, { 5020, 4840 }, { 5077, 4898 }, { 5079, 4902 } } },
    { "NtOpenCompositionSurfaceSectionInfo",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4999, 4816 }, { 5017, 4841 }, { 5074, 4899 }, { 5076, 4903 } } },
    { "NtOpenCompositionSurfaceSwapChainHandleInfo",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5000, 4817 }, { 5018, 4842 }, { 5075, 4900 }, { 5077, 4904 } } },
    { "NtQueryCompositionInputIsImplicit",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5095, 4901 }, { 5097, 4905 } } },
    { "NtQueryCompositionInputQueueAndTransform", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5094, 4902 }, { 5096, 4906 } } },
    { "NtQueryCompositionInputSink",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5027, 4843 }, { 5089, 4903 }, { 5091, 4907 } } },
    { "NtQueryCompositionInputSinkLuid",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5028, 4844 }, { 5090, 4904 }, { 5092, 4908 } } },
    { "NtQueryCompositionInputSinkViewId",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5091, 4905 }, { 5093, 4909 } } },
    { "NtQueryCompositionSurfaceBinding", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4996, 4818 }, { 5014, 4845 }, { 5071, 4906 }, { 5073, 4910 } } },
    { "NtQueryCompositionSurfaceRenderingRealization",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5001, 4819 }, { 5019, 4846 }, { 5076, 4907 }, { 5078, 4911 } } },
    { "NtQueryCompositionSurfaceStatistics",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4998, 4820 }, { 5016, 4847 }, { 5073, 4908 }, { 5075, 4912 } } },
    { "NtRIMAddInputObserver",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5215, 4909 }, { 5220, 4913 } } },
    { "NtRIMGetDevicePreparsedDataLockfree",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5219, 4910 }, { 5224, 4914 } } },
    { "NtRIMObserveNextInput",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5218, 4911 }, { 5223, 4915 } } },
    { "NtRIMRemoveInputObserver", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5216, 4912 }, { 5221, 4916 } } },
    { "NtRIMUpdateInputObserverRegistration", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5217, 4913 }, { 5222, 4917 } } },
    { "NtSetCompositionSurfaceAnalogExclusive",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5104, 4914 }, { 5107, 4918 } } },
    { "NtSetCompositionSurfaceBufferCompositionMode", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5023, 4848 }, { 5080, 4915 }, { -1, -1 } } },
    { "NtSetCompositionSurfaceBufferCompositionModeAndOrientation",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 4919 }, { 5082, 4919 } } },
    { "NtSetCompositionSurfaceIndependentFlipInfo",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5024, 4849 }, { 5081, 4916 }, { 5083, 4920 } } },
    { "NtSetCompositionSurfaceOutOfFrameDirectFlipNotification",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5003, 4821 }, { 5021, 4850 }, { 5078, 4917 }, { 5080, 4921 } } },
    { "NtSetCompositionSurfaceStatistics",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5004, 4822 }, { 5022, 4851 }, { 5079, 4918 }, { 5081, 4922 } } },
    { "NtTokenManagerConfirmOutstandingAnalogToken",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5103, 4919 }, { 5106, 4923 } } },
    { "NtTokenManagerCreateCompositionTokenHandle",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5036, 4852 }, { 5100, 4920 }, { 5103, 4924 } } },
    { "NtTokenManagerDeleteOutstandingDirectFlipTokens",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5035, 4853 }, { 5099, 4921 }, { 5102, 4925 } } },
    { "NtTokenManagerGetAnalogExclusiveSurfaceUpdates",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5102, 4922 }, { 5105, 4926 } } },
    { "NtTokenManagerGetAnalogExclusiveTokenEvent",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5101, 4923 }, { 5104, 4927 } } },
    { "NtTokenManagerGetOutOfFrameDirectFlipSurfaceUpdates",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5007, 4823 }, { 5034, 4854 }, { 5098, 4924 }, { 5101, 4928 } } },
    { "NtTokenManagerOpenEvent",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5005, 4824 }, { 5032, 4855 }, { -1, -1 }, { -1, -1 } } },
    { "NtTokenManagerOpenSection",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5031, 4856 }, { -1, -1 }, { -1, -1 } } },
    { "NtTokenManagerOpenSectionAndEvents",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5096, 4925 }, { 5099, 4929 } } },
    { "NtTokenManagerThread", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5006, 4825 }, { 5033, 4857 }, { 5097, 4926 }, { 5100, 4930 } } },
    { "NtUnBindCompositionSurface",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4995, 4826 }, { 5013, 4858 }, { 5070, 4927 }, { 5072, 4931 } } },
    { "NtUpdateInputSinkTransforms",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5029, 4859 }, { 5092, 4928 }, { 5094, 4932 } } },
    { "NtUserAcquireIAMKey",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5063, 4827 }, { 5102, 4860 }, { 5177, 4929 }, { 5180, 4933 } } },
    { "NtUserActivateKeyboardLayout", { 4400, { 4399, 4390 }, { 4409, 4391 }, { 4410, 4382 }, { 4465, 4382 }, { 4467, 4383 }, { 4468, 4384 }, { 4470, 4384 } } },
    { "NtUserAddClipboardFormatListener", { -1, { -1, -1 }, { 4410, 4724 }, { 4411, 4737 }, { 4464, 4828 }, { 4466, 4861 }, { 4467, 4930 }, { 4469, 4934 } } },
    { "NtUserAlterWindowStyle",   { 4401, { 4400, 4311 }, { 4411, 4312 }, { 4412, 4307 }, { 4463, 4307 }, { 4465, 4308 }, { 4466, 4309 }, { 4468, 4309 } } },
    { "NtUserAssociateInputContext",  { 4402, { 4401, 4645 }, { 4412, 4725 }, { 4413, 4738 }, { 4462, 4829 }, { 4464, 4862 }, { 4465, 4931 }, { 4467, 4935 } } },
    { "NtUserAttachThreadInput",  { 4403, { 4402, 4336 }, { 4413, 4337 }, { 4414, 4332 }, { 4461, 4332 }, { 4463, 4333 }, { 4464, 4334 }, { 4466, 4334 } } },
    { "NtUserAutoPromoteMouseInPointer",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5073, 4830 }, { 5114, 4863 }, { 5191, 4932 }, { 5195, 4936 } } },
    { "NtUserAutoRotateScreen",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5061, 4831 }, { 5101, 4864 }, { 5176, 4933 }, { 5179, 4937 } } },
    { "NtUserBeginPaint", { 4404, { 4403, 4118 }, { 4414, 4119 }, { 4415, 4119 }, { 4460, 4120 }, { 4462, 4121 }, { 4463, 4122 }, { 4465, 4122 } } },
    { "NtUserBitBltSysBmp",   { 4405, { 4404, 4303 }, { 4415, 4304 }, { 4416, 4299 }, { 4459, 4299 }, { 4461, 4300 }, { 4462, 4301 }, { 4464, 4301 } } },
    { "NtUserBlockInput", { 4406, { 4405, 4646 }, { 4416, 4726 }, { 4417, 4739 }, { 4458, 4832 }, { 4460, 4865 }, { 4461, 4934 }, { 4463, 4938 } } },
    { "NtUserBuildHimcList",  { 4407, { 4406, 4647 }, { 4417, 4727 }, { 4418, 4740 }, { 4457, 4833 }, { 4459, 4866 }, { 4460, 4935 }, { 4462, 4939 } } },
    { "NtUserBuildHwndList",  { 4408, { 4407, 4123 }, { 4418, 4124 }, { 4419, 4124 }, { 4456, 4125 }, { 4458, 4126 }, { 4459, 4127 }, { 4461, 4127 } } },
    { "NtUserBuildNameList",  { 4409, { 4408, 4274 }, { 4419, 4275 }, { 4420, 4270 }, { 4455, 4270 }, { 4457, 4271 }, { 4458, 4272 }, { 4460, 4272 } } },
    { "NtUserBuildPropList",  { 4410, { 4409, 4648 }, { 4420, 4728 }, { 4421, 4741 }, { 4454, 4834 }, { 4456, 4867 }, { 4457, 4936 }, { 4459, 4940 } } },
    { "NtUserCalcMenuBar",    { 4662, { 4658, 4248 }, { 4686, 4249 }, { 4699, 4247 }, { 4633, 4247 }, { 4636, 4248 }, { 4639, 4249 }, { 4641, 4249 } } },
    { "NtUserCalculatePopupWindowPosition",   { -1, { -1, -1 }, { -1, -1 }, { 4698, 4742 }, { 4634, 4835 }, { 4637, 4868 }, { 4640, 4937 }, { 4642, 4941 } } },
    { "NtUserCallHwnd",   { 4411, { 4410, 4370 }, { 4421, 4371 }, { 4422, 4364 }, { 4453, 4364 }, { 4455, 4365 }, { 4456, 4366 }, { 4458, 4366 } } },
    { "NtUserCallHwndLock",   { 4412, { 4411, 4128 }, { 4422, 4129 }, { 4423, 4129 }, { 4452, 4130 }, { 4454, 4131 }, { 4455, 4132 }, { 4457, 4132 } } },
    { "NtUserCallHwndOpt",    { 4413, { 4412, 4649 }, { 4423, 4729 }, { 4424, 4743 }, { 4451, 4836 }, { 4453, 4869 }, { 4454, 4938 }, { 4456, 4942 } } },
    { "NtUserCallHwndParam",  { 4414, { 4413, 4255 }, { 4424, 4256 }, { 4425, 4254 }, { 4450, 4254 }, { 4452, 4255 }, { 4453, 4256 }, { 4455, 4256 } } },
    { "NtUserCallHwndParamLock",  { 4415, { 4414, 4134 }, { 4425, 4135 }, { 4426, 4135 }, { 4449, 4136 }, { 4451, 4137 }, { 4452, 4138 }, { 4454, 4138 } } },
    { "NtUserCallMsgFilter",  { 4416, { 4415, 4116 }, { 4426, 4117 }, { 4427, 4117 }, { 4448, 4118 }, { 4450, 4119 }, { 4451, 4120 }, { 4453, 4120 } } },
    { "NtUserCallNextHookEx", { 4417, { 4416, 4125 }, { 4427, 4126 }, { 4428, 4126 }, { 4447, 4127 }, { 4449, 4128 }, { 4450, 4129 }, { 4452, 4129 } } },
    { "NtUserCallNoParam",    { 4418, { 4417, 4101 }, { 4428, 4101 }, { 4429, 4101 }, { 4446, 4102 }, { 4448, 4103 }, { 4449, 4104 }, { 4451, 4104 } } },
    { "NtUserCallOneParam",   { 4419, { 4418, 4098 }, { 4429, 4098 }, { 4430, 4098 }, { 4445, 4099 }, { 4447, 4100 }, { 4448, 4101 }, { 4450, 4101 } } },
    { "NtUserCallTwoParam",   { 4420, { 4419, 4137 }, { 4430, 4138 }, { 4431, 4138 }, { 4444, 4139 }, { 4446, 4140 }, { 4447, 4141 }, { 4449, 4141 } } },
    { "NtUserCanBrokerForceForeground",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4416, 4837 }, { 4418, 4870 }, { 4419, 4939 }, { 4421, 4943 } } },
    { "NtUserChangeClipboardChain",   { 4421, { 4420, 4383 }, { 4431, 4384 }, { 4432, 4377 }, { 4443, 4377 }, { 4445, 4378 }, { 4446, 4379 }, { 4448, 4379 } } },
    { "NtUserChangeDisplaySettings",  { 4422, { 4421, 4650 }, { 4432, 4730 }, { 4433, 4744 }, { 4442, 4838 }, { 4444, 4871 }, { 4445, 4940 }, { 4447, 4944 } } },
    { "NtUserChangeWindowMessageFilterEx",    { -1, { -1, -1 }, { -1, -1 }, { 4753, 4745 }, { 4825, 4839 }, { 4834, 4872 }, { 4846, 4941 }, { 4847, 4945 } } },
    { "NtUserCheckAccessForIntegrityLevel",   { -1, { -1, -1 }, { 4433, 4731 }, { 4439, 4746 }, { 4436, 4840 }, { 4438, 4873 }, { 4439, 4942 }, { 4441, 4946 } } },
    { "NtUserCheckDesktopByThreadId", { -1, { -1, -1 }, { 4434, 4732 }, { 4440, 4747 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserCheckImeHotKey", { 4423, { 4422, 4212 }, { 4436, 4213 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserCheckMenuItem",  { 4424, { 4423, 4360 }, { 4437, 4361 }, { 4442, 4355 }, { 4433, 4355 }, { 4435, 4356 }, { 4436, 4357 }, { 4438, 4357 } } },
    { "NtUserCheckProcessForClipboardAccess", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5077, 4841 }, { 5118, 4874 }, { 5195, 4943 }, { 5199, 4947 } } },
    { "NtUserCheckProcessSession",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4736, 4842 }, { 4739, 4875 }, { 4749, 4944 }, { 4750, 4948 } } },
    { "NtUserCheckWindowThreadDesktop",   { -1, { -1, -1 }, { 4435, 4733 }, { 4441, 4748 }, { 4434, 4843 }, { 4436, 4876 }, { 4437, 4945 }, { 4439, 4949 } } },
    { "NtUserChildWindowFromPointEx", { 4425, { 4424, 4651 }, { 4438, 4734 }, { 4443, 4749 }, { 4432, 4844 }, { 4434, 4877 }, { 4435, 4946 }, { 4437, 4950 } } },
    { "NtUserClearForeground",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5107, 4878 }, { 5182, 4947 }, { 5185, 4951 } } },
    { "NtUserClipCursor", { 4426, { 4425, 4652 }, { 4439, 4735 }, { 4444, 4750 }, { 4431, 4845 }, { 4433, 4879 }, { 4434, 4948 }, { 4436, 4952 } } },
    { "NtUserCloseClipboard", { 4427, { 4426, 4306 }, { 4440, 4307 }, { 4445, 4302 }, { 4430, 4302 }, { 4432, 4303 }, { 4433, 4304 }, { 4435, 4304 } } },
    { "NtUserCloseDesktop",   { 4428, { 4427, 4266 }, { 4441, 4267 }, { 4446, 4262 }, { 4429, 4262 }, { 4431, 4263 }, { 4432, 4264 }, { 4434, 4264 } } },
    { "NtUserCloseWindowStation", { 4429, { 4428, 4281 }, { 4442, 4282 }, { 4447, 4277 }, { 4428, 4277 }, { 4430, 4278 }, { 4431, 4279 }, { 4433, 4279 } } },
    { "NtUserCompositionInputSinkLuidFromPoint",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5121, 4880 }, { 5198, 4949 }, { 5202, 4953 } } },
    { "NtUserConsoleControl", { 4430, { 4429, 4323 }, { 4443, 4324 }, { 4448, 4319 }, { 4427, 4319 }, { 4429, 4320 }, { 4430, 4321 }, { 4432, 4321 } } },
    { "NtUserConvertMemHandle",   { 4431, { 4430, 4354 }, { 4444, 4355 }, { 4449, 4349 }, { 4426, 4349 }, { 4428, 4350 }, { 4429, 4351 }, { 4431, 4351 } } },
    { "NtUserCopyAcceleratorTable",   { 4432, { 4431, 4139 }, { 4445, 4140 }, { 4450, 4140 }, { 4425, 4141 }, { 4427, 4142 }, { 4428, 4143 }, { 4430, 4143 } } },
    { "NtUserCountClipboardFormats",  { 4433, { 4432, 4373 }, { 4446, 4374 }, { 4451, 4367 }, { 4424, 4367 }, { 4426, 4368 }, { 4427, 4369 }, { 4429, 4369 } } },
    { "NtUserCreateAcceleratorTable", { 4434, { 4433, 4341 }, { 4447, 4342 }, { 4452, 4337 }, { 4423, 4337 }, { 4425, 4338 }, { 4426, 4339 }, { 4428, 4339 } } },
    { "NtUserCreateCaret",    { 4435, { 4434, 4145 }, { 4448, 4146 }, { 4453, 4146 }, { 4422, 4147 }, { 4424, 4148 }, { 4425, 4149 }, { 4427, 4149 } } },
    { "NtUserCreateDCompositionHwndTarget",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5052, 4846 }, { 5092, 4881 }, { 5167, 4950 }, { 5170, 4954 } } },
    { "NtUserCreateDesktop",  { 4436, { 4435, 4653 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserCreateDesktopEx",    { -1, { -1, -1 }, { 4449, 4736 }, { 4454, 4751 }, { 4421, 4847 }, { 4423, 4882 }, { 4424, 4951 }, { 4426, 4955 } } },
    { "NtUserCreateInputContext", { 4437, { 4436, -1 }, { 4450, 4737 }, { 4455, 4752 }, { 4420, 4848 }, { 4422, 4883 }, { 4423, 4952 }, { 4425, 4956 } } },
    { "NtUserCreateLocalMemHandle",   { 4438, { 4437, 4335 }, { 4451, 4336 }, { 4456, 4331 }, { 4419, 4331 }, { 4421, 4332 }, { 4422, 4333 }, { 4424, 4333 } } },
    { "NtUserCreateWindowEx", { 4439, { 4438, 4215 }, { 4452, 4216 }, { 4457, 4214 }, { 4418, 4214 }, { 4420, 4215 }, { 4421, 4216 }, { 4423, 4216 } } },
    { "NtUserCreateWindowStation",    { 4440, { 4439, 4655 }, { 4453, 4738 }, { 4458, 4753 }, { 4417, 4849 }, { 4419, 4884 }, { 4420, 4953 }, { 4422, 4957 } } },
    { "NtUserCtxDisplayIOCtl",    { 4694, { 4690, 4656 }, { 4719, 4739 }, { 4731, 4754 }, { 4739, 4850 }, { 4742, 4885 }, { 4752, 4954 }, { 4753, 4958 } } },
    { "NtUserDdeGetQualityOfService", { 4441, { 4440, 4657 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserDdeInitialize",  { 4442, { 4441, 4371 }, { 4454, 4372 }, { 4459, 4365 }, { 4415, 4365 }, { 4417, 4366 }, { 4418, 4367 }, { 4420, 4367 } } },
    { "NtUserDdeSetQualityOfService", { 4443, { 4442, 4658 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserDeferWindowPos", { 4444, { 4443, 4178 }, { 4455, 4179 }, { 4460, 4179 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserDeferWindowPosAndBand",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4414, 4851 }, { 4416, 4886 }, { 4417, 4955 }, { 4419, 4959 } } },
    { "NtUserDefSetText", { 4445, { 4444, 4224 }, { 4456, 4225 }, { 4461, 4223 }, { 4413, 4223 }, { 4415, 4224 }, { 4416, 4225 }, { 4418, 4225 } } },
    { "NtUserDelegateCapturePointers",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4466, 4852 }, { 4468, 4887 }, { 4469, 4956 }, { 4471, 4960 } } },
    { "NtUserDelegateInput",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4467, 4853 }, { 4469, 4888 }, { 4470, 4957 }, { 4472, 4961 } } },
    { "NtUserDeleteMenu", { 4446, { 4445, 4288 }, { 4457, 4289 }, { 4462, 4284 }, { 4476, 4284 }, { 4478, 4285 }, { 4479, 4286 }, { 4481, 4286 } } },
    { "NtUserDestroyAcceleratorTable",    { 4447, { 4446, 4355 }, { 4458, 4356 }, { 4463, 4350 }, { 4475, 4350 }, { 4477, 4351 }, { 4478, 4352 }, { 4480, 4352 } } },
    { "NtUserDestroyCursor",  { 4448, { 4447, 4253 }, { 4459, 4254 }, { 4464, 4252 }, { 4474, 4252 }, { 4476, 4253 }, { 4477, 4254 }, { 4479, 4254 } } },
    { "NtUserDestroyDCompositionHwndTarget",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5051, 4854 }, { 5091, 4889 }, { 5166, 4958 }, { 5169, 4962 } } },
    { "NtUserDestroyInputContext",    { 4449, { 4448, -1 }, { 4460, 4740 }, { 4465, 4755 }, { 4473, 4855 }, { 4475, 4890 }, { 4476, 4959 }, { 4478, 4963 } } },
    { "NtUserDestroyMenu",    { 4450, { 4449, 4321 }, { 4461, 4322 }, { 4466, 4317 }, { 4472, 4317 }, { 4474, 4318 }, { 4475, 4319 }, { 4477, 4319 } } },
    { "NtUserDestroyWindow",  { 4451, { 4450, 4254 }, { 4462, 4255 }, { 4467, 4253 }, { 4471, 4253 }, { 4473, 4254 }, { 4474, 4255 }, { 4476, 4255 } } },
    { "NtUserDisableImmersiveOwner",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5066, 4856 }, { 5106, 4891 }, { 5181, 4960 }, { 5184, 4964 } } },
    { "NtUserDisableProcessWindowFiltering",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4469, 4857 }, { 4471, 4892 }, { 4472, 4961 }, { 4474, 4965 } } },
    { "NtUserDisableThreadIme",   { 4452, { 4451, 4660 }, { 4463, 4741 }, { 4468, 4756 }, { 4470, 4858 }, { 4472, 4893 }, { 4473, 4962 }, { 4475, 4966 } } },
    { "NtUserDiscardPointerFrameMessages",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4815, 4859 }, { 4823, 4894 }, { 4834, 4963 }, { 4835, 4967 } } },
    { "NtUserDispatchMessage",    { 4453, { 4452, 4149 }, { 4464, 4150 }, { 4469, 4150 }, { 4468, 4151 }, { 4470, 4152 }, { 4471, 4153 }, { 4473, 4153 } } },
    { "NtUserDisplayConfigGetDeviceInfo", { -1, { -1, -1 }, { -1, -1 }, { 4437, 4757 }, { 4438, 4860 }, { 4440, 4895 }, { 4441, 4964 }, { 4443, 4968 } } },
    { "NtUserDisplayConfigSetDeviceInfo", { -1, { -1, -1 }, { -1, -1 }, { 4438, 4758 }, { 4437, 4861 }, { 4439, 4896 }, { 4440, 4965 }, { 4442, 4969 } } },
    { "NtUserDoSoundConnect", { -1, { -1, -1 }, { 4465, 4742 }, { 4470, 4759 }, { 4478, 4862 }, { 4480, 4897 }, { 4481, 4966 }, { 4483, 4970 } } },
    { "NtUserDoSoundDisconnect",  { -1, { -1, -1 }, { 4466, 4743 }, { 4471, 4760 }, { 4477, 4863 }, { 4479, 4898 }, { 4480, 4967 }, { 4482, 4971 } } },
    { "NtUserDragDetect", { 4454, { 4453, 4661 }, { 4467, 4744 }, { 4472, 4761 }, { 4575, 4864 }, { 4576, 4899 }, { 4578, 4968 }, { 4580, 4972 } } },
    { "NtUserDragObject", { 4455, { 4454, 4662 }, { 4468, 4745 }, { 4473, 4762 }, { 4574, 4865 }, { 4575, 4900 }, { 4577, 4969 }, { 4579, 4973 } } },
    { "NtUserDrawAnimatedRects",  { 4456, { 4455, 4663 }, { 4469, 4746 }, { 4474, 4763 }, { 4573, 4866 }, { 4574, 4901 }, { 4576, 4970 }, { 4578, 4974 } } },
    { "NtUserDrawCaption",    { 4457, { 4456, 4664 }, { 4470, 4747 }, { 4475, 4764 }, { 4572, 4867 }, { 4573, 4902 }, { 4575, 4971 }, { 4577, 4975 } } },
    { "NtUserDrawCaptionTemp",    { 4458, { 4457, 4665 }, { 4471, 4748 }, { 4476, 4765 }, { 4571, 4868 }, { 4572, 4903 }, { 4574, 4972 }, { 4576, 4976 } } },
    { "NtUserDrawIconEx", { 4459, { 4458, 4191 }, { 4472, 4192 }, { 4477, 4192 }, { 4570, 4192 }, { 4571, 4193 }, { 4573, 4194 }, { 4575, 4194 } } },
    { "NtUserDrawMenuBarTemp",    { 4460, { 4459, 4666 }, { 4473, 4749 }, { 4478, 4766 }, { 4569, 4869 }, { 4570, 4904 }, { 4572, 4973 }, { 4574, 4977 } } },
    { "NtUserDwmGetDxRgn",    { -1, { -1, -1 }, { 4726, 4750 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserDwmGetRemoteSessionOcclusionEvent",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4733, 4870 }, { 4736, 4905 }, { 4744, 4974 }, { 4745, 4978 } } },
    { "NtUserDwmGetRemoteSessionOcclusionState",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4734, 4871 }, { 4737, 4906 }, { 4745, 4975 }, { 4746, 4979 } } },
    { "NtUserDwmHintDxUpdate",    { -1, { -1, -1 }, { 4725, 4751 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserDwmKernelShutdown",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4746, 4976 }, { 4747, 4980 } } },
    { "NtUserDwmKernelStartup",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4747, 4977 }, { 4748, 4981 } } },
    { "NtUserDwmStartRedirection",    { -1, { -1, -1 }, { 4723, 4752 }, { 4735, 4767 }, { 4732, 4872 }, { 4735, 4907 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserDwmStopRedirection", { -1, { -1, -1 }, { 4724, 4753 }, { 4736, 4768 }, { 4731, 4873 }, { 4734, 4908 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserDwmValidateWindow",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4435, 4874 }, { 4437, 4909 }, { 4438, 4978 }, { 4440, 4982 } } },
    { "NtUserEmptyClipboard", { 4461, { 4460, 4348 }, { 4474, 4349 }, { 4479, 4344 }, { 4568, 4344 }, { 4569, 4345 }, { 4571, 4346 }, { 4573, 4346 } } },
    { "NtUserEnableChildWindowDpiMessage",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4677, 4979 }, { 4679, 4983 } } },
    { "NtUserEnableIAMAccess",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5067, 4875 }, { 5108, 4910 }, { 5183, 4980 }, { 5186, 4984 } } },
    { "NtUserEnableMenuItem", { 4462, { 4461, 4310 }, { 4475, 4311 }, { 4480, 4306 }, { 4567, 4306 }, { 4568, 4307 }, { 4570, 4308 }, { 4572, 4308 } } },
    { "NtUserEnableMouseInPointer",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5070, 4876 }, { 5111, 4911 }, { 5188, 4981 }, { 5192, 4985 } } },
    { "NtUserEnableMouseInputForCursorSuppression",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5074, 4877 }, { 5115, 4912 }, { 5192, 4982 }, { 5196, 4986 } } },
    { "NtUserEnableScrollBar",    { 4463, { 4462, 4283 }, { 4476, 4284 }, { 4481, 4279 }, { 4566, 4279 }, { 4567, 4280 }, { 4569, 4281 }, { 4571, 4281 } } },
    { "NtUserEnableTouchPad", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4819, 4913 }, { 4830, 4983 }, { 4831, 4987 } } },
    { "NtUserEndDeferWindowPosEx",    { 4464, { 4463, 4133 }, { 4477, 4134 }, { 4482, 4134 }, { 4565, 4135 }, { 4566, 4136 }, { 4568, 4137 }, { 4570, 4137 } } },
    { "NtUserEndMenu",    { 4465, { 4464, 4667 }, { 4478, 4754 }, { 4483, 4769 }, { 4564, 4878 }, { 4565, 4914 }, { 4567, 4984 }, { 4569, 4988 } } },
    { "NtUserEndPaint",   { 4466, { 4465, 4120 }, { 4479, 4121 }, { 4484, 4121 }, { 4563, 4122 }, { 4564, 4123 }, { 4566, 4124 }, { 4568, 4124 } } },
    { "NtUserEndTouchOperation",  { -1, { -1, -1 }, { -1, -1 }, { 4751, 4770 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserEnumDisplayDevices", { 4467, { 4466, 4347 }, { 4480, 4348 }, { 4485, 4343 }, { 4562, 4343 }, { 4563, 4344 }, { 4565, 4345 }, { 4567, 4345 } } },
    { "NtUserEnumDisplayMonitors",    { 4468, { 4467, 4169 }, { 4481, 4170 }, { 4486, 4170 }, { 4561, 4171 }, { 4562, 4172 }, { 4564, 4173 }, { 4566, 4173 } } },
    { "NtUserEnumDisplaySettings",    { 4469, { 4468, 4378 }, { 4482, 4379 }, { 4487, 4372 }, { 4560, 4372 }, { 4561, 4373 }, { 4563, 4374 }, { 4565, 4374 } } },
    { "NtUserEvent",  { 4470, { 4469, 4668 }, { 4483, 4755 }, { 4488, 4771 }, { 4559, 4879 }, { 4560, 4915 }, { 4562, 4985 }, { 4564, 4989 } } },
    { "NtUserExcludeUpdateRgn",   { 4471, { 4470, 4175 }, { 4484, 4176 }, { 4489, 4176 }, { 4558, 4177 }, { 4559, 4178 }, { 4561, 4179 }, { 4563, 4179 } } },
    { "NtUserFillWindow", { 4472, { 4471, 4234 }, { 4485, 4235 }, { 4490, 4233 }, { 4557, 4233 }, { 4558, 4234 }, { 4560, 4235 }, { 4562, 4235 } } },
    { "NtUserFindExistingCursorIcon", { 4473, { 4472, 4157 }, { 4486, 4158 }, { 4491, 4158 }, { 4556, 4159 }, { 4557, 4160 }, { 4559, 4161 }, { 4561, 4161 } } },
    { "NtUserFindWindowEx",   { 4474, { 4473, 4206 }, { 4487, 4207 }, { 4492, 4206 }, { 4555, 4206 }, { 4556, 4207 }, { 4558, 4208 }, { 4560, 4208 } } },
    { "NtUserFlashWindowEx",  { 4475, { 4474, 4669 }, { 4488, 4756 }, { 4493, 4772 }, { 4554, 4880 }, { 4555, 4916 }, { 4557, 4986 }, { 4559, 4990 } } },
    { "NtUserFrostCrashedWindow", { -1, { -1, -1 }, { 4489, 4757 }, { 4494, 4773 }, { 4553, 4881 }, { 4554, 4917 }, { 4556, 4987 }, { 4558, 4991 } } },
    { "NtUserGetAltTabInfo",  { 4476, { 4475, 4343 }, { 4490, 4344 }, { 4495, 4339 }, { 4552, 4339 }, { 4553, 4340 }, { 4555, 4341 }, { 4557, 4341 } } },
    { "NtUserGetAncestor",    { 4477, { 4476, 4278 }, { 4491, 4279 }, { 4496, 4274 }, { 4551, 4274 }, { 4552, 4275 }, { 4554, 4276 }, { 4556, 4276 } } },
    { "NtUserGetAppImeLevel", { 4478, { 4477, -1 }, { 4492, 4758 }, { 4497, 4774 }, { 4550, 4882 }, { 4551, 4918 }, { 4553, 4988 }, { 4555, 4992 } } },
    { "NtUserGetAsyncKeyState",   { 4479, { 4478, 4163 }, { 4493, 4164 }, { 4498, 4164 }, { 4549, 4165 }, { 4550, 4166 }, { 4552, 4167 }, { 4554, 4167 } } },
    { "NtUserGetAtomName",    { 4480, { 4479, 4269 }, { 4494, 4270 }, { 4499, 4265 }, { 4548, 4265 }, { 4549, 4266 }, { 4551, 4267 }, { 4553, 4267 } } },
    { "NtUserGetAutoRotationState",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5060, 4883 }, { 5100, 4919 }, { 5175, 4989 }, { 5178, 4993 } } },
    { "NtUserGetCaretBlinkTime",  { 4481, { 4480, 4344 }, { 4495, 4345 }, { 4500, 4340 }, { 4547, 4340 }, { 4548, 4341 }, { 4550, 4342 }, { 4552, 4342 } } },
    { "NtUserGetCaretPos",    { 4482, { 4481, 4671 }, { 4496, 4759 }, { 4501, 4775 }, { 4546, 4885 }, { 4547, 4921 }, { 4549, 4991 }, { 4551, 4995 } } },
    { "NtUserGetCIMSSM",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4533, 4884 }, { 4534, 4920 }, { 4536, 4990 }, { 4538, 4994 } } },
    { "NtUserGetClassInfo",   { 4483, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserGetClassInfoEx", { -1, { 4482, 4285 }, { 4497, 4286 }, { 4502, 4281 }, { 4545, 4281 }, { 4546, 4282 }, { 4548, 4283 }, { 4550, 4283 } } },
    { "NtUserGetClassName",   { 4484, { 4483, 4220 }, { 4498, 4221 }, { 4503, 4219 }, { 4544, 4219 }, { 4545, 4220 }, { 4547, 4221 }, { 4549, 4221 } } },
    { "NtUserGetClipboardAccessToken",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5078, 4887 }, { 5119, 4923 }, { 5196, 4993 }, { 5200, 4997 } } },
    { "NtUserGetClipboardData",   { 4485, { 4484, 4349 }, { 4499, 4350 }, { 4504, 4345 }, { 4543, 4345 }, { 4544, 4346 }, { 4546, 4347 }, { 4548, 4347 } } },
    { "NtUserGetClipboardFormatName", { 4486, { 4485, 4333 }, { 4500, 4334 }, { 4505, 4329 }, { 4542, 4329 }, { 4543, 4330 }, { 4545, 4331 }, { 4547, 4331 } } },
    { "NtUserGetClipboardOwner",  { 4487, { 4486, 4301 }, { 4501, 4302 }, { 4506, 4297 }, { 4541, 4297 }, { 4542, 4298 }, { 4544, 4299 }, { 4546, 4299 } } },
    { "NtUserGetClipboardSequenceNumber", { 4488, { 4487, 4181 }, { 4502, 4182 }, { 4507, 4182 }, { 4540, 4182 }, { 4541, 4183 }, { 4543, 4184 }, { 4545, 4184 } } },
    { "NtUserGetClipboardViewer", { 4489, { 4488, 4673 }, { 4503, 4761 }, { 4508, 4777 }, { 4539, 4888 }, { 4540, 4924 }, { 4542, 4994 }, { 4544, 4998 } } },
    { "NtUserGetClipCursor",  { 4490, { 4489, 4672 }, { 4504, 4760 }, { 4509, 4776 }, { 4538, 4886 }, { 4539, 4922 }, { 4541, 4992 }, { 4543, 4996 } } },
    { "NtUserGetComboBoxInfo",    { 4491, { 4490, 4674 }, { 4505, 4762 }, { 4510, 4778 }, { 4537, 4889 }, { 4538, 4925 }, { 4540, 4995 }, { 4542, 4999 } } },
    { "NtUserGetControlBrush",    { 4492, { 4491, 4219 }, { 4506, 4220 }, { 4511, 4218 }, { 4536, 4218 }, { 4537, 4219 }, { 4539, 4220 }, { 4541, 4220 } } },
    { "NtUserGetControlColor",    { 4493, { 4492, 4327 }, { 4507, 4328 }, { 4512, 4323 }, { 4535, 4323 }, { 4536, 4324 }, { 4538, 4325 }, { 4540, 4325 } } },
    { "NtUserGetCPD", { 4494, { 4493, 4164 }, { 4508, 4165 }, { 4513, 4165 }, { 4534, 4166 }, { 4535, 4167 }, { 4537, 4168 }, { 4539, 4168 } } },
    { "NtUserGetCurrentInputMessageSource",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4532, 4890 }, { 4533, 4926 }, { 4535, 4996 }, { 4537, 5000 } } },
    { "NtUserGetCursorDims",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5130, 4927 }, { 5207, 4997 }, { 5211, 5001 } } },
    { "NtUserGetCursorFrameInfo", { 4495, { 4494, 4342 }, { 4509, 4343 }, { 4514, 4338 }, { 4531, 4338 }, { 4532, 4339 }, { 4534, 4340 }, { 4536, 4340 } } },
    { "NtUserGetCursorInfo",  { 4496, { 4495, 4675 }, { 4510, 4763 }, { 4515, 4779 }, { 4530, 4891 }, { 4531, 4928 }, { 4533, 4998 }, { 4535, 5002 } } },
    { "NtUserGetDC",  { 4497, { 4496, 4106 }, { 4511, 4106 }, { 4516, 4106 }, { 4529, 4107 }, { 4530, 4108 }, { 4532, 4109 }, { 4534, 4109 } } },
    { "NtUserGetDCEx",    { 4498, { 4497, 4243 }, { 4512, 4244 }, { 4517, 4242 }, { 4528, 4242 }, { 4529, 4243 }, { 4531, 4244 }, { 4533, 4244 } } },
    { "NtUserGetDesktopID",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4527, 4892 }, { 4528, 4929 }, { 4530, 5000 }, { 4532, 5004 } } },
    { "NtUserGetDisplayAutoRotationPreferences",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5057, 4893 }, { 5097, 4930 }, { 5172, 5001 }, { 5175, 5005 } } },
    { "NtUserGetDisplayAutoRotationPreferencesByProcessId",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5056, 4894 }, { 5096, 4931 }, { 5171, 5002 }, { 5174, 5006 } } },
    { "NtUserGetDisplayConfigBufferSizes",    { -1, { -1, -1 }, { -1, -1 }, { 4434, 4780 }, { 4441, 4895 }, { 4443, 4932 }, { 4444, 5003 }, { 4446, 5007 } } },
    { "NtUserGetDManipHookInitFunction",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4734, 4999 }, { 4735, 5003 } } },
    { "NtUserGetDoubleClickTime", { 4499, { 4498, 4282 }, { 4513, 4283 }, { 4518, 4278 }, { 4526, 4278 }, { 4527, 4279 }, { 4529, 4280 }, { 4531, 4280 } } },
    { "NtUserGetDpiForMonitor",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5127, 4933 }, { 5204, 5004 }, { 5208, 5008 } } },
    { "NtUserGetDpiSystemMetrics",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4528, 5005 }, { 4530, 5009 } } },
    { "NtUserGetForegroundWindow",    { 4500, { 4499, 4155 }, { 4514, 4156 }, { 4519, 4156 }, { 4525, 4157 }, { 4526, 4158 }, { 4527, 4159 }, { 4529, 4159 } } },
    { "NtUserGetGestureConfig",   { -1, { -1, -1 }, { -1, -1 }, { 4759, 4781 }, { 4820, 4896 }, { 4829, 4934 }, { 4841, 5006 }, { 4842, 5010 } } },
    { "NtUserGetGestureExtArgs",  { -1, { -1, -1 }, { -1, -1 }, { 4756, 4782 }, { 4822, 4897 }, { 4831, 4935 }, { 4843, 5007 }, { 4844, 5011 } } },
    { "NtUserGetGestureInfo", { -1, { -1, -1 }, { -1, -1 }, { 4755, 4783 }, { 4823, 4898 }, { 4832, 4936 }, { 4844, 5008 }, { 4845, 5012 } } },
    { "NtUserGetGlobalIMEStatus", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4524, 4899 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserGetGuiResources",    { 4501, { 4500, 4676 }, { 4515, 4764 }, { 4520, 4784 }, { 4523, 4900 }, { 4525, 4937 }, { 4526, 5009 }, { 4528, 5013 } } },
    { "NtUserGetGUIThreadInfo",   { 4502, { 4501, 4356 }, { 4516, 4357 }, { 4521, 4351 }, { 4522, 4351 }, { 4524, 4352 }, { 4525, 4353 }, { 4527, 4353 } } },
    { "NtUserGetHimetricScaleFactorFromPixelLocation",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5125, 4938 }, { 5202, 5010 }, { 5206, 5014 } } },
    { "NtUserGetIconInfo",    { 4503, { 4502, 4174 }, { 4517, 4175 }, { 4522, 4175 }, { 4521, 4176 }, { 4523, 4177 }, { 4524, 4178 }, { 4526, 4178 } } },
    { "NtUserGetIconSize",    { 4504, { 4503, 4233 }, { 4518, 4234 }, { 4523, 4232 }, { 4520, 4232 }, { 4522, 4233 }, { 4523, 4234 }, { 4525, 4234 } } },
    { "NtUserGetImeHotKey",   { 4505, { 4504, 4677 }, { 4519, 4765 }, { 4524, 4785 }, { 4519, 4901 }, { 4521, 4939 }, { 4522, 5011 }, { 4524, 5015 } } },
    { "NtUserGetImeInfoEx",   { 4506, { 4505, 4678 }, { 4520, 4766 }, { 4525, 4786 }, { 4518, 4902 }, { 4520, 4940 }, { 4521, 5012 }, { 4523, 5016 } } },
    { "NtUserGetInputLocaleInfo", { -1, { -1, -1 }, { -1, -1 }, { 4526, 4787 }, { 4517, 4903 }, { 4519, 4941 }, { 4520, 5013 }, { 4522, 5017 } } },
    { "NtUserGetInternalWindowPos",   { 4507, { 4506, 4679 }, { 4521, 4767 }, { 4527, 4788 }, { 4516, 4904 }, { 4518, 4942 }, { 4519, 5014 }, { 4521, 5018 } } },
    { "NtUserGetKeyboardLayoutList",  { 4508, { 4507, 4184 }, { 4522, 4185 }, { 4528, 4185 }, { 4515, 4185 }, { 4517, 4186 }, { 4518, 4187 }, { 4520, 4187 } } },
    { "NtUserGetKeyboardLayoutName",  { 4509, { 4508, 4681 }, { 4523, 4769 }, { 4529, 4790 }, { 4514, 4906 }, { 4516, 4944 }, { 4517, 5016 }, { 4519, 5020 } } },
    { "NtUserGetKeyboardState",   { 4510, { 4509, 4217 }, { 4524, 4218 }, { 4530, 4216 }, { 4513, 4216 }, { 4515, 4217 }, { 4516, 4218 }, { 4518, 4218 } } },
    { "NtUserGetKeyNameText", { 4511, { 4510, 4680 }, { 4525, 4768 }, { 4531, 4789 }, { 4512, 4905 }, { 4514, 4943 }, { 4515, 5015 }, { 4517, 5019 } } },
    { "NtUserGetKeyState",    { 4512, { 4511, 4099 }, { 4526, 4099 }, { 4532, 4099 }, { 4511, 4100 }, { 4513, 4101 }, { 4514, 4102 }, { 4516, 4102 } } },
    { "NtUserGetLayeredWindowAttributes", { 4676, { 4672, 4682 }, { 4700, 4770 }, { 4713, 4791 }, { 4757, 4907 }, { 4760, 4945 }, { 4770, 5017 }, { 4771, 5021 } } },
    { "NtUserGetListBoxInfo", { 4513, { 4512, 4683 }, { 4527, 4771 }, { 4533, 4792 }, { 4510, 4908 }, { 4512, 4946 }, { 4513, 5018 }, { 4515, 5022 } } },
    { "NtUserGetMenuBarInfo", { 4514, { 4513, 4293 }, { 4528, 4294 }, { 4534, 4289 }, { 4509, 4289 }, { 4511, 4290 }, { 4512, 4291 }, { 4514, 4291 } } },
    { "NtUserGetMenuIndex",   { 4515, { 4514, 4684 }, { 4529, 4772 }, { 4535, 4793 }, { 4508, 4909 }, { 4510, 4947 }, { 4511, 5019 }, { 4513, 5023 } } },
    { "NtUserGetMenuItemRect",    { 4516, { 4515, 4685 }, { 4530, 4773 }, { 4536, 4794 }, { 4507, 4910 }, { 4509, 4948 }, { 4510, 5020 }, { 4512, 5024 } } },
    { "NtUserGetMessage", { 4517, { 4516, 4102 }, { 4531, 4102 }, { 4537, 4102 }, { 4506, 4103 }, { 4508, 4104 }, { 4509, 4105 }, { 4511, 4105 } } },
    { "NtUserGetMouseMovePointsEx",   { 4518, { 4517, 4686 }, { 4532, 4774 }, { 4538, 4795 }, { 4505, 4911 }, { 4507, 4949 }, { 4508, 5021 }, { 4510, 5025 } } },
    { "NtUserGetObjectInformation",   { 4519, { 4518, 4203 }, { 4533, 4204 }, { 4539, 4204 }, { 4504, 4204 }, { 4506, 4205 }, { 4507, 4206 }, { 4509, 4206 } } },
    { "NtUserGetOpenClipboardWindow", { 4520, { 4519, 4316 }, { 4534, 4317 }, { 4540, 4312 }, { 4503, 4312 }, { 4505, 4313 }, { 4506, 4314 }, { 4508, 4314 } } },
    { "NtUserGetOwnerTransformedMonitorRect", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5131, 4950 }, { 4096, 4096 }, { 4096, 4096 } } },
    { "NtUserGetPhysicalDeviceRect",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4810, 4951 }, { 4821, 5022 }, { 4822, 5026 } } },
    { "NtUserGetPointerCursorId", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4818, 4912 }, { 4827, 4952 }, { 4839, 5023 }, { 4840, 5027 } } },
    { "NtUserGetPointerDevice",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4812, 4913 }, { 4817, 4953 }, { 4828, 5024 }, { 4829, 5028 } } },
    { "NtUserGetPointerDeviceCursors",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4808, 4914 }, { 4813, 4954 }, { 4824, 5025 }, { 4825, 5029 } } },
    { "NtUserGetPointerDeviceProperties", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4811, 4915 }, { 4816, 4955 }, { 4827, 5026 }, { 4828, 5030 } } },
    { "NtUserGetPointerDeviceRects",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4809, 4916 }, { 4814, 4956 }, { 4825, 5027 }, { 4826, 5031 } } },
    { "NtUserGetPointerDevices",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4813, 4917 }, { 4818, 4957 }, { 4829, 5028 }, { 4830, 5032 } } },
    { "NtUserGetPointerFrameArrivalTimes",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4836, 5029 }, { 4837, 5033 } } },
    { "NtUserGetPointerInfoList", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4817, 4918 }, { 4826, 4958 }, { 4838, 5030 }, { 4839, 5034 } } },
    { "NtUserGetPointerInputTransform",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4825, 4959 }, { 4837, 5031 }, { 4838, 5035 } } },
    { "NtUserGetPointerType", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4819, 4919 }, { 4828, 4960 }, { 4840, 5032 }, { 4841, 5036 } } },
    { "NtUserGetPrecisionTouchPadConfiguration",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4820, 4961 }, { 4831, 5033 }, { 4832, 5037 } } },
    { "NtUserGetPriorityClipboardFormat", { 4521, { 4520, 4687 }, { 4535, 4775 }, { 4541, 4796 }, { 4502, 4920 }, { 4504, 4962 }, { 4505, 5034 }, { 4507, 5038 } } },
    { "NtUserGetProcessDpiAwareness", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5126, 4963 }, { 5203, 5035 }, { 5207, 5039 } } },
    { "NtUserGetProcessUIContextInformation", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5068, 4921 }, { 5109, 4964 }, { 5184, 5036 }, { 5187, 5040 } } },
    { "NtUserGetProcessWindowStation",    { 4522, { 4521, 4129 }, { 4536, 4130 }, { 4542, 4130 }, { 4501, 4131 }, { 4503, 4132 }, { 4504, 4133 }, { 4506, 4133 } } },
    { "NtUserGetProp",    { -1, { -1, -1 }, { 4648, 4110 }, { 4658, 4110 }, { 4684, 4111 }, { 4687, 4112 }, { 4693, 4113 }, { 4694, 4113 } } },
    { "NtUserGetQueueEventStatus",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5079, 4922 }, { 5120, 4965 }, { 5197, 5037 }, { 5201, 5041 } } },
    { "NtUserGetRawInputBuffer",  { 4523, { 4522, 4688 }, { 4537, 4776 }, { 4543, 4797 }, { 4500, 4923 }, { 4502, 4966 }, { 4503, 5038 }, { 4505, 5042 } } },
    { "NtUserGetRawInputData",    { 4524, { 4523, 4689 }, { 4538, 4777 }, { 4544, 4798 }, { 4499, 4924 }, { 4501, 4967 }, { 4502, 5039 }, { 4504, 5043 } } },
    { "NtUserGetRawInputDeviceInfo",  { 4525, { 4524, 4690 }, { 4539, 4778 }, { 4545, 4799 }, { 4498, 4925 }, { 4500, 4968 }, { 4501, 5040 }, { 4503, 5044 } } },
    { "NtUserGetRawInputDeviceList",  { 4526, { 4525, 4691 }, { 4540, 4779 }, { 4546, 4800 }, { 4497, 4926 }, { 4499, 4969 }, { 4500, 5041 }, { 4502, 5045 } } },
    { "NtUserGetRawPointerDeviceData",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4807, 4927 }, { 4812, 4970 }, { 4823, 5042 }, { 4824, 5046 } } },
    { "NtUserGetRegisteredRawInputDevices",   { 4527, { 4526, 4692 }, { 4541, 4780 }, { 4547, 4801 }, { 4496, 4928 }, { 4498, 4971 }, { 4499, 5043 }, { 4501, 5047 } } },
    { "NtUserGetScrollBarInfo",   { 4528, { 4527, 4244 }, { 4542, 4245 }, { 4548, 4243 }, { 4495, 4243 }, { 4497, 4244 }, { 4498, 4245 }, { 4500, 4245 } } },
    { "NtUserGetSystemMenu",  { 4529, { 4528, 4192 }, { 4543, 4193 }, { 4549, 4193 }, { 4494, 4193 }, { 4496, 4194 }, { 4497, 4195 }, { 4499, 4195 } } },
    { "NtUserGetThreadDesktop",   { 4530, { 4529, 4228 }, { 4544, 4229 }, { 4550, 4227 }, { 4493, 4227 }, { 4495, 4228 }, { 4496, 4229 }, { 4498, 4229 } } },
    { "NtUserGetThreadState", { 4531, { 4530, 4096 }, { 4545, 4096 }, { 4551, 4096 }, { 4492, 4097 }, { 4494, 4098 }, { 4495, 4099 }, { 4497, 4099 } } },
    { "NtUserGetTitleBarInfo",    { 4532, { 4531, 4240 }, { 4546, 4241 }, { 4552, 4239 }, { 4491, 4239 }, { 4493, 4240 }, { 4494, 4241 }, { 4496, 4241 } } },
    { "NtUserGetTopLevelWindow",  { -1, { -1, -1 }, { -1, -1 }, { 4553, 4802 }, { 4490, 4929 }, { 4492, 4972 }, { 4493, 5044 }, { 4495, 5048 } } },
    { "NtUserGetTouchInputInfo",  { -1, { -1, -1 }, { -1, -1 }, { 4752, 4803 }, { 4768, 4930 }, { 4771, 4973 }, { 4781, 5045 }, { 4782, 5049 } } },
    { "NtUserGetTouchValidationStatus",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4801, 4931 }, { 4804, 4974 }, { 4815, 5046 }, { 4816, 5050 } } },
    { "NtUserGetUpdatedClipboardFormats", { -1, { -1, -1 }, { 4547, 4781 }, { 4554, 4804 }, { 4489, 4932 }, { 4491, 4975 }, { 4492, 5047 }, { 4494, 5051 } } },
    { "NtUserGetUpdateRect",  { 4533, { 4532, 4179 }, { 4548, 4180 }, { 4555, 4180 }, { 4488, 4180 }, { 4490, 4181 }, { 4491, 4182 }, { 4493, 4182 } } },
    { "NtUserGetUpdateRgn",   { 4534, { 4533, 4231 }, { 4549, 4232 }, { 4556, 4230 }, { 4487, 4230 }, { 4489, 4231 }, { 4490, 4232 }, { 4492, 4232 } } },
    { "NtUserGetWindowBand",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4486, 4934 }, { 4488, 4977 }, { 4489, 5049 }, { 4491, 5053 } } },
    { "NtUserGetWindowCompositionAttribute",  { -1, { -1, -1 }, { -1, -1 }, { 4558, 4806 }, { 4484, 4935 }, { 4486, 4978 }, { 4487, 5050 }, { 4489, 5054 } } },
    { "NtUserGetWindowCompositionInfo",   { -1, { -1, -1 }, { -1, -1 }, { 4557, 4807 }, { 4485, 4936 }, { 4487, 4979 }, { 4488, 5051 }, { 4490, 5055 } } },
    { "NtUserGetWindowDC",    { 4535, { 4534, 4195 }, { 4550, 4196 }, { 4559, 4196 }, { 4483, 4196 }, { 4485, 4197 }, { 4486, 4198 }, { 4488, 4198 } } },
    { "NtUserGetWindowDisplayAffinity",   { -1, { -1, -1 }, { -1, -1 }, { 4560, 4808 }, { 4482, 4937 }, { 4484, 4980 }, { 4485, 5052 }, { 4487, 5056 } } },
    { "NtUserGetWindowFeedbackSetting",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4799, 4938 }, { 4802, 4981 }, { 4813, 5053 }, { 4814, 5057 } } },
    { "NtUserGetWindowMinimizeRect",  { -1, { -1, -1 }, { 4727, 4783 }, { 4737, 4809 }, { 4730, 4939 }, { 4733, 4982 }, { 4743, 5054 }, { 4744, 5058 } } },
    { "NtUserGetWindowPlacement", { 4536, { 4535, 4313 }, { 4551, 4314 }, { 4561, 4309 }, { 4481, 4309 }, { 4483, 4310 }, { 4484, 4311 }, { 4486, 4311 } } },
    { "NtUserGetWindowRgnEx", { -1, { -1, -1 }, { 4666, 4784 }, { 4678, 4810 }, { 4658, 4940 }, { 4661, 4983 }, { 4664, 5055 }, { 4666, 5059 } } },
    { "NtUserGetWOWClass",    { 4537, { 4536, 4693 }, { 4552, 4782 }, { 4562, 4805 }, { 4480, 4933 }, { 4482, 4976 }, { 4483, 5048 }, { 4485, 5052 } } },
    { "NtUserGhostWindowFromHungWindow",  { -1, { -1, -1 }, { 4553, 4785 }, { 4563, 4811 }, { 4479, 4941 }, { 4481, 4984 }, { 4482, 5056 }, { 4484, 5060 } } },
    { "NtUserHandleDelegatedInput",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4576, 4942 }, { 4577, 4985 }, { 4579, 5057 }, { 4581, 5061 } } },
    { "NtUserHardErrorControl",   { 4538, { 4537, 4694 }, { 4554, 4786 }, { 4564, 4812 }, { 4627, 4943 }, { 4630, 4986 }, { 4632, 5058 }, { 4634, 5062 } } },
    { "NtUserHideCaret",  { 4539, { 4538, 4126 }, { 4555, 4127 }, { 4565, 4127 }, { 4626, 4128 }, { 4629, 4129 }, { 4631, 4130 }, { 4633, 4130 } } },
    { "NtUserHidePointerContactVisualization",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4800, 4944 }, { 4803, 4987 }, { 4814, 5059 }, { 4815, 5063 } } },
    { "NtUserHiliteMenuItem", { 4540, { 4539, 4695 }, { 4556, 4787 }, { 4566, 4813 }, { 4625, 4945 }, { 4628, 4988 }, { 4630, 5060 }, { 4632, 5064 } } },
    { "NtUserHungWindowFromGhostWindow",  { -1, { -1, -1 }, { 4557, 4788 }, { 4567, 4814 }, { 4624, 4946 }, { 4627, 4989 }, { 4629, 5061 }, { 4631, 5065 } } },
    { "NtUserHwndQueryRedirectionInfo",   { -1, { -1, -1 }, { -1, -1 }, { 4919, 4815 }, { 4991, 4947 }, { 5009, 4990 }, { 5066, 5062 }, { 5068, 5066 } } },
    { "NtUserHwndSetRedirectionInfo", { -1, { -1, -1 }, { -1, -1 }, { 4920, 4816 }, { 4990, 4948 }, { 5008, 4991 }, { 5065, 5063 }, { 5067, 5067 } } },
    { "NtUserImpersonateDdeClientWindow", { 4541, { 4540, 4696 }, { 4558, 4789 }, { 4568, 4817 }, { 4623, 4949 }, { 4626, 4992 }, { 4628, 5064 }, { 4630, 5068 } } },
    { "NtUserInitialize", { 4542, { 4541, 4698 }, { 4559, 4791 }, { 4569, 4819 }, { 4622, 4951 }, { 4625, 4994 }, { 4627, 5066 }, { 4629, 5070 } } },
    { "NtUserInitializeClientPfnArrays",  { 4543, { 4542, 4699 }, { 4560, 4792 }, { 4570, 4820 }, { 4621, 4952 }, { 4624, 4995 }, { 4626, 5067 }, { 4628, 5071 } } },
    { "NtUserInitializeInputDeviceInjection", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5208, 5068 }, { 5212, 5072 } } },
    { "NtUserInitializePointerDeviceInjection",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5209, 5069 }, { 5213, 5073 } } },
    { "NtUserInitializeTouchInjection",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4802, 4953 }, { 4805, 4996 }, { 4816, 5070 }, { 4817, 5074 } } },
    { "NtUserInitTask",   { 4544, { 4543, 4697 }, { 4561, 4790 }, { 4571, 4818 }, { 4620, 4950 }, { 4623, 4993 }, { 4625, 5065 }, { 4627, 5069 } } },
    { "NtUserInjectDeviceInput",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5211, 5071 }, { 5216, 5075 } } },
    { "NtUserInjectGesture",  { -1, { -1, -1 }, { -1, -1 }, { 4754, 4821 }, { 4824, 4954 }, { 4833, 4997 }, { 4845, 5072 }, { 4846, 5076 } } },
    { "NtUserInjectKeyboardInput",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5213, 5073 }, { 5218, 5077 } } },
    { "NtUserInjectMouseInput",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5212, 5074 }, { 5217, 5078 } } },
    { "NtUserInjectPointerInput", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5214, 5075 }, { 5219, 5079 } } },
    { "NtUserInjectTouchInput",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4803, 4955 }, { 4806, 4998 }, { 4817, 5076 }, { 4818, 5080 } } },
    { "NtUserInternalClipCursor", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5076, 4956 }, { 5117, 4999 }, { 5194, 5077 }, { 5198, 5081 } } },
    { "NtUserInternalGetWindowIcon",  { -1, { -1, -1 }, { 4563, 4793 }, { 4573, 4822 }, { 4618, 4957 }, { 4621, 5000 }, { 4623, 5078 }, { 4625, 5082 } } },
    { "NtUserInternalGetWindowText",  { 4545, { 4544, 4194 }, { 4562, 4195 }, { 4572, 4195 }, { 4619, 4195 }, { 4622, 4196 }, { 4624, 4197 }, { 4626, 4197 } } },
    { "NtUserInvalidateRect", { 4546, { 4545, 4100 }, { 4564, 4100 }, { 4574, 4100 }, { 4617, 4101 }, { 4620, 4102 }, { 4622, 4103 }, { 4624, 4103 } } },
    { "NtUserInvalidateRgn",  { 4547, { 4546, 4300 }, { 4565, 4301 }, { 4575, 4296 }, { 4616, 4296 }, { 4619, 4297 }, { 4621, 4298 }, { 4623, 4298 } } },
    { "NtUserIsChildWindowDpiMessageEnabled", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4678, 5079 }, { 4680, 5083 } } },
    { "NtUserIsClipboardFormatAvailable", { 4548, { 4547, 4142 }, { 4566, 4143 }, { 4576, 4143 }, { 4615, 4144 }, { 4618, 4145 }, { 4620, 4146 }, { 4622, 4146 } } },
    { "NtUserIsMouseInPointerEnabled",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5071, 4958 }, { 5112, 5001 }, { 5189, 5080 }, { 5193, 5084 } } },
    { "NtUserIsMouseInputEnabled",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5075, 4959 }, { 5116, 5002 }, { 5193, 5081 }, { 5197, 5085 } } },
    { "NtUserIsTopLevelWindow",   { -1, { -1, -1 }, { -1, -1 }, { 4577, 4823 }, { 4614, 4960 }, { 4617, 5003 }, { 4619, 5082 }, { 4621, 5086 } } },
    { "NtUserIsTouchWindow",  { -1, { -1, -1 }, { -1, -1 }, { 4749, 4824 }, { 4769, 4961 }, { 4772, 5004 }, { 4782, 5083 }, { 4783, 5087 } } },
    { "NtUserIsWindowBroadcastingDpiToChildren",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4679, 5084 }, { 4681, 5088 } } },
    { "NtUserKillTimer",  { 4549, { 4548, 4122 }, { 4567, 4123 }, { 4578, 4123 }, { 4613, 4124 }, { 4616, 4125 }, { 4618, 4126 }, { 4620, 4126 } } },
    { "NtUserLayoutCompleted",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4607, 4962 }, { 4609, 5005 }, { 4611, 5085 }, { 4613, 5089 } } },
    { "NtUserLinkDpiCursor",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5129, 5006 }, { 5206, 5086 }, { 5210, 5090 } } },
    { "NtUserLoadKeyboardLayoutEx",   { 4550, { 4549, 4700 }, { 4568, 4794 }, { 4579, 4825 }, { 4612, 4963 }, { 4615, 5007 }, { 4617, 5087 }, { 4619, 5091 } } },
    { "NtUserLockWindowStation",  { 4551, { 4550, 4701 }, { 4569, 4795 }, { 4580, 4826 }, { 4611, 4964 }, { 4614, 5008 }, { 4616, 5088 }, { 4618, 5092 } } },
    { "NtUserLockWindowUpdate",   { 4552, { 4551, 4364 }, { 4570, 4365 }, { 4581, 4358 }, { 4610, 4358 }, { 4613, 4359 }, { 4615, 4360 }, { 4617, 4360 } } },
    { "NtUserLockWorkStation",    { 4553, { 4552, 4702 }, { 4571, 4796 }, { 4582, 4827 }, { 4609, 4965 }, { 4612, 5009 }, { 4614, 5089 }, { 4616, 5093 } } },
    { "NtUserLogicalToPerMonitorDPIPhysicalPoint",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4610, 5010 }, { 4612, 5090 }, { 4614, 5094 } } },
    { "NtUserLogicalToPhysicalPoint", { -1, { -1, -1 }, { 4572, 4797 }, { 4583, 4828 }, { 4608, 4966 }, { 4611, 5011 }, { 4613, 5091 }, { 4615, 5095 } } },
    { "NtUserMagControl", { -1, { -1, -1 }, { -1, -1 }, { 4916, 4831 }, { 4988, 4969 }, { 5006, 5014 }, { 5063, 5094 }, { 5065, 5098 } } },
    { "NtUserMagGetContextInformation",   { -1, { -1, -1 }, { -1, -1 }, { 4918, 4832 }, { 4986, 4970 }, { 5004, 5015 }, { 5061, 5095 }, { 5063, 5099 } } },
    { "NtUserMagSetContextInformation",   { -1, { -1, -1 }, { -1, -1 }, { 4917, 4833 }, { 4987, 4971 }, { 5005, 5016 }, { 5062, 5096 }, { 5064, 5100 } } },
    { "NtUserManageGestureHandlerWindow", { -1, { -1, -1 }, { -1, -1 }, { 4757, 4834 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserMapVirtualKeyEx",    { 4554, { 4553, 4186 }, { 4573, 4187 }, { 4584, 4187 }, { 4606, 4187 }, { 4608, 4188 }, { 4610, 4189 }, { 4612, 4189 } } },
    { "NtUserMenuItemFromPoint",  { 4555, { 4554, 4705 }, { 4574, 4800 }, { 4585, 4835 }, { 4605, 4972 }, { 4607, 5017 }, { 4609, 5097 }, { 4611, 5101 } } },
    { "NtUserMessageCall",    { 4556, { 4555, 4103 }, { 4575, 4103 }, { 4586, 4103 }, { 4604, 4104 }, { 4606, 4105 }, { 4608, 4106 }, { 4610, 4106 } } },
    { "NtUserMinMaximize",    { 4557, { 4556, 4706 }, { 4576, 4801 }, { 4587, 4836 }, { 4603, 4973 }, { 4605, 5018 }, { 4607, 5098 }, { 4609, 5102 } } },
    { "NtUserMNDragLeave",    { 4558, { 4557, 4703 }, { 4577, 4798 }, { 4588, 4829 }, { 4602, 4967 }, { 4604, 5012 }, { 4606, 5092 }, { 4608, 5096 } } },
    { "NtUserMNDragOver", { 4559, { 4558, 4704 }, { 4578, 4799 }, { 4589, 4830 }, { 4601, 4968 }, { 4603, 5013 }, { 4605, 5093 }, { 4607, 5097 } } },
    { "NtUserModifyUserStartupInfoFlags", { 4560, { 4559, 4372 }, { 4579, 4373 }, { 4590, 4366 }, { 4600, 4366 }, { 4602, 4367 }, { 4604, 4368 }, { 4606, 4368 } } },
    { "NtUserModifyWindowTouchCapability",    { -1, { -1, -1 }, { -1, -1 }, { 4748, 4837 }, { 4770, 4974 }, { 4773, 5019 }, { 4783, 5099 }, { 4784, 5103 } } },
    { "NtUserMoveWindow", { 4561, { 4560, 4189 }, { 4580, 4190 }, { 4591, 4190 }, { 4599, 4190 }, { 4601, 4191 }, { 4603, 4192 }, { 4605, 4192 } } },
    { "NtUserNavigateFocus",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5222, 5100 }, { 5227, 5104 } } },
    { "NtUserNotifyIMEStatus",    { 4562, { 4561, 4707 }, { 4581, 4802 }, { 4592, 4838 }, { 4598, 4975 }, { 4600, 5020 }, { 4602, 5101 }, { 4604, 5105 } } },
    { "NtUserNotifyProcessCreate",    { 4563, { 4562, 4238 }, { 4582, 4239 }, { 4593, 4237 }, { 4597, 4237 }, { 4599, 4238 }, { 4601, 4239 }, { 4603, 4239 } } },
    { "NtUserNotifyWinEvent", { 4564, { 4563, 4140 }, { 4583, 4141 }, { 4594, 4141 }, { 4596, 4142 }, { 4598, 4143 }, { 4600, 4144 }, { 4602, 4144 } } },
    { "NtUserOpenClipboard",  { 4565, { 4564, 4307 }, { 4584, 4308 }, { 4595, 4303 }, { 4595, 4303 }, { 4597, 4304 }, { 4599, 4305 }, { 4601, 4305 } } },
    { "NtUserOpenDesktop",    { 4566, { 4565, 4267 }, { 4585, 4268 }, { 4596, 4263 }, { 4594, 4263 }, { 4596, 4264 }, { 4598, 4265 }, { 4600, 4265 } } },
    { "NtUserOpenInputDesktop",   { 4567, { 4566, 4708 }, { 4586, 4803 }, { 4597, 4839 }, { 4593, 4976 }, { 4595, 5021 }, { 4597, 5102 }, { 4599, 5106 } } },
    { "NtUserOpenThreadDesktop",  { -1, { -1, -1 }, { 4587, 4804 }, { 4598, 4840 }, { 4592, 4977 }, { 4594, 5022 }, { 4596, 5103 }, { 4598, 5107 } } },
    { "NtUserOpenWindowStation",  { 4568, { 4567, 4257 }, { 4588, 4258 }, { 4599, 4256 }, { 4591, 4256 }, { 4593, 4257 }, { 4595, 4258 }, { 4597, 4258 } } },
    { "NtUserPaintDesktop",   { 4569, { 4568, 4379 }, { 4589, 4380 }, { 4600, 4373 }, { 4590, 4373 }, { 4592, 4374 }, { 4594, 4375 }, { 4596, 4375 } } },
    { "NtUserPaintMenuBar",   { 4663, { 4659, 4338 }, { 4687, 4339 }, { 4700, 4334 }, { 4632, 4334 }, { 4635, 4335 }, { 4638, 4336 }, { 4640, 4336 } } },
    { "NtUserPaintMonitor",   { -1, { -1, -1 }, { 4590, 4805 }, { 4601, 4841 }, { 4589, 4978 }, { 4591, 5023 }, { 4593, 5104 }, { 4595, 5108 } } },
    { "NtUserPeekMessage",    { 4570, { 4569, 4097 }, { 4591, 4097 }, { 4602, 4097 }, { 4588, 4098 }, { 4590, 4099 }, { 4592, 4100 }, { 4594, 4100 } } },
    { "NtUserPerMonitorDPIPhysicalToLogicalPoint",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4584, 5024 }, { 4586, 5105 }, { 4588, 5109 } } },
    { "NtUserPhysicalToLogicalPoint", { -1, { -1, -1 }, { 4592, 4806 }, { 4603, 4842 }, { 4587, 4979 }, { 4589, 5025 }, { 4591, 5106 }, { 4593, 5110 } } },
    { "NtUserPostMessage",    { 4571, { 4570, 4110 }, { 4593, 4111 }, { 4604, 4111 }, { 4586, 4112 }, { 4588, 4113 }, { 4590, 4114 }, { 4592, 4114 } } },
    { "NtUserPostThreadMessage",  { 4572, { 4571, 4190 }, { 4594, 4191 }, { 4605, 4191 }, { 4585, 4191 }, { 4587, 4192 }, { 4589, 4193 }, { 4591, 4193 } } },
    { "NtUserPrintWindow",    { 4573, { 4572, 4709 }, { 4595, 4807 }, { 4606, 4843 }, { 4584, 4980 }, { 4586, 5026 }, { 4588, 5107 }, { 4590, 5111 } } },
    { "NtUserProcessConnect", { 4574, { 4573, 4346 }, { 4596, 4347 }, { 4607, 4342 }, { 4583, 4342 }, { 4585, 4343 }, { 4587, 4344 }, { 4589, 4344 } } },
    { "NtUserPromoteMouseInPointer",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5072, 4981 }, { 5113, 5027 }, { 5190, 5108 }, { 5194, 5112 } } },
    { "NtUserPromotePointer", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4814, 4982 }, { 4822, 5028 }, { 4833, 5109 }, { 4834, 5113 } } },
    { "NtUserQueryBSDRWindow",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4582, 4983 }, { 4583, 5029 }, { 4585, 5110 }, { 4587, 5114 } } },
    { "NtUserQueryDisplayConfig", { -1, { -1, -1 }, { -1, -1 }, { 4436, 4844 }, { 4439, 4984 }, { 4441, 5030 }, { 4442, 5111 }, { 4444, 5115 } } },
    { "NtUserQueryInformationThread", { 4575, { 4574, 4710 }, { 4597, 4808 }, { 4608, 4845 }, { 4581, 4985 }, { 4582, 5031 }, { 4584, 5112 }, { 4586, 5116 } } },
    { "NtUserQueryInputContext",  { 4576, { 4575, 4711 }, { 4598, 4809 }, { 4609, 4846 }, { 4580, 4986 }, { 4581, 5032 }, { 4583, 5113 }, { 4585, 5117 } } },
    { "NtUserQuerySendMessage",   { 4577, { 4576, 4712 }, { 4599, 4810 }, { 4610, 4847 }, { 4579, 4987 }, { 4580, 5033 }, { 4582, 5114 }, { 4584, 5118 } } },
    { "NtUserQueryUserCounters",  { 4578, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserQueryWindow",    { 4579, { 4577, 4111 }, { 4600, 4112 }, { 4611, 4112 }, { 4578, 4113 }, { 4579, 4114 }, { 4581, 4115 }, { 4583, 4115 } } },
    { "NtUserRealChildWindowFromPoint",   { 4580, { 4578, 4713 }, { 4601, 4811 }, { 4612, 4848 }, { 4577, 4988 }, { 4578, 5034 }, { 4580, 5115 }, { 4582, 5119 } } },
    { "NtUserRealInternalGetMessage", { 4581, { 4579, 4334 }, { 4602, 4335 }, { 4613, 4330 }, { 4628, 4330 }, { 4631, 4331 }, { 4633, 4332 }, { 4635, 4332 } } },
    { "NtUserRealWaitMessageEx",  { 4582, { 4580, 4714 }, { 4603, 4812 }, { 4614, 4849 }, { 4629, 4989 }, { 4632, 5035 }, { 4634, 5116 }, { 4636, 5120 } } },
    { "NtUserRedrawWindow",   { 4583, { 4581, 4114 }, { 4604, 4115 }, { 4615, 4115 }, { 4728, 4116 }, { 4731, 4117 }, { 4741, 4118 }, { 4742, 4118 } } },
    { "NtUserRegisterBSDRWindow", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4727, 4990 }, { 4730, 5036 }, { 4740, 5117 }, { 4741, 5121 } } },
    { "NtUserRegisterClassExWOW", { 4584, { 4582, 4276 }, { 4605, 4277 }, { 4616, 4272 }, { 4726, 4272 }, { 4729, 4273 }, { 4739, 4274 }, { 4740, 4274 } } },
    { "NtUserRegisterDManipHook", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4733, 5118 }, { 4734, 5122 } } },
    { "NtUserRegisterEdgy",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4798, 4991 }, { 4801, 5037 }, { 4811, 5119 }, { 4812, 5123 } } },
    { "NtUserRegisterErrorReportingDialog",   { -1, { -1, -1 }, { 4606, 4813 }, { 4617, 4850 }, { 4725, 4992 }, { 4728, 5038 }, { 4738, 5120 }, { 4739, 5124 } } },
    { "NtUserRegisterHotKey", { 4586, { 4584, 4715 }, { 4608, 4814 }, { 4619, 4851 }, { 4723, 4993 }, { 4726, 5039 }, { 4732, 5121 }, { 4733, 5125 } } },
    { "NtUserRegisterManipulationThread", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4735, 5122 }, { 4736, 5126 } } },
    { "NtUserRegisterPointerDeviceNotifications", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4810, 4994 }, { 4815, 5040 }, { 4826, 5123 }, { 4827, 5127 } } },
    { "NtUserRegisterPointerInputTarget", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4816, 4995 }, { 4824, 5041 }, { 4835, 5124 }, { 4836, 5128 } } },
    { "NtUserRegisterRawInputDevices",    { 4587, { 4585, 4716 }, { 4609, 4815 }, { 4620, 4852 }, { 4722, 4996 }, { 4725, 5042 }, { 4731, 5125 }, { 4732, 5129 } } },
    { "NtUserRegisterServicesProcess",    { -1, { -1, -1 }, { -1, -1 }, { 4621, 4853 }, { 4721, 4997 }, { 4724, 5043 }, { 4730, 5126 }, { 4731, 5130 } } },
    { "NtUserRegisterSessionPort",    { -1, { -1, -1 }, { 4720, 4816 }, { 4732, 4854 }, { 4738, 4998 }, { 4741, 5044 }, { 4751, 5127 }, { 4752, 5131 } } },
    { "NtUserRegisterShellPTPListener",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4812, 5128 }, { 4813, 5132 } } },
    { "NtUserRegisterTasklist",   { 4588, { 4586, 4717 }, { 4610, 4817 }, { 4622, 4855 }, { 4720, 4999 }, { 4723, 5045 }, { 4729, 5129 }, { 4730, 5133 } } },
    { "NtUserRegisterTouchHitTestingWindow",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4804, 5000 }, { 4807, 5046 }, { 4818, 5130 }, { 4819, 5134 } } },
    { "NtUserRegisterTouchPadCapable",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4811, 5047 }, { 4822, 5131 }, { 4823, 5135 } } },
    { "NtUserRegisterUserApiHook",    { 4585, { 4583, 4718 }, { 4607, 4818 }, { 4618, 4856 }, { 4724, 5001 }, { 4727, 5048 }, { 4737, 5132 }, { 4738, 5136 } } },
    { "NtUserRegisterWindowMessage",  { 4589, { 4587, 4150 }, { 4611, 4151 }, { 4623, 4151 }, { 4719, 4152 }, { 4722, 4153 }, { 4728, 4154 }, { 4729, 4154 } } },
    { "NtUserReleaseDwmHitTestWaiters",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4635, 5133 }, { 4637, 5137 } } },
    { "NtUserRemoteConnect",  { 4690, { 4686, 4719 }, { 4715, 4819 }, { 4727, 4857 }, { 4743, 5002 }, { 4746, 5049 }, { 4756, 5134 }, { 4757, 5138 } } },
    { "NtUserRemoteRedrawRectangle",  { 4691, { 4687, 4720 }, { 4716, 4820 }, { 4728, 4858 }, { 4742, 5003 }, { 4745, 5050 }, { 4755, 5135 }, { 4756, 5139 } } },
    { "NtUserRemoteRedrawScreen", { 4692, { 4688, -1 }, { 4717, 4821 }, { 4729, 4859 }, { 4741, 5004 }, { 4744, 5051 }, { 4754, 5136 }, { 4755, 5140 } } },
    { "NtUserRemoteStopScreenUpdates",    { 4693, { 4689, -1 }, { 4718, 4822 }, { 4730, 4860 }, { 4740, 5005 }, { 4743, 5052 }, { 4753, 5137 }, { 4754, 5141 } } },
    { "NtUserRemoveClipboardFormatListener",  { -1, { -1, -1 }, { 4612, 4823 }, { 4624, 4861 }, { 4718, 5006 }, { 4721, 5053 }, { 4727, 5138 }, { 4728, 5142 } } },
    { "NtUserRemoveInjectionDevice",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 5143 }, { 5214, 5143 } } },
    { "NtUserRemoveMenu", { 4590, { 4588, 4350 }, { 4613, 4351 }, { 4625, 4346 }, { 4717, 4346 }, { 4720, 4347 }, { 4726, 4348 }, { 4727, 4348 } } },
    { "NtUserRemoveProp", { 4591, { 4589, 4165 }, { 4614, 4166 }, { 4626, 4166 }, { 4716, 4167 }, { 4719, 4168 }, { 4725, 4169 }, { 4726, 4169 } } },
    { "NtUserReportInertia",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5128, 5054 }, { 5205, 5139 }, { 5209, 5144 } } },
    { "NtUserResolveDesktop", { 4592, { 4590, 4384 }, { 4615, 4385 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserResolveDesktopForWOW",   { 4593, { 4591, 4723 }, { 4616, 4824 }, { 4627, 4862 }, { 4715, 5007 }, { 4718, 5055 }, { 4724, 5140 }, { 4725, 5145 } } },
    { "NtUserSBGetParms", { 4594, { 4592, 4173 }, { 4617, 4174 }, { 4628, 4174 }, { 4714, 4175 }, { 4717, 4176 }, { 4723, 4177 }, { 4724, 4177 } } },
    { "NtUserScrollDC",   { 4595, { 4593, 4202 }, { 4618, 4203 }, { 4629, 4203 }, { 4713, 4203 }, { 4716, 4204 }, { 4722, 4205 }, { 4723, 4205 } } },
    { "NtUserScrollWindowEx", { 4596, { 4594, 4290 }, { 4619, 4291 }, { 4630, 4286 }, { 4712, 4286 }, { 4715, 4287 }, { 4721, 4288 }, { 4722, 4288 } } },
    { "NtUserSelectPalette",  { 4597, { 4595, 4124 }, { 4620, 4125 }, { 4631, 4125 }, { 4711, 4126 }, { 4714, 4127 }, { 4720, 4128 }, { 4721, 4128 } } },
    { "NtUserSendEventMessage",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4641, 5008 }, { 4644, 5056 }, { 4647, 5141 }, { 4649, 5146 } } },
    { "NtUserSendInput",  { 4598, { 4596, 4227 }, { 4621, 4228 }, { 4632, 4226 }, { 4710, 4226 }, { 4713, 4227 }, { 4719, 4228 }, { 4720, 4228 } } },
    { "NtUserSendTouchInput", { -1, { -1, -1 }, { -1, -1 }, { 4750, 4863 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSetActivationFilter",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5103, 5057 }, { 5178, 5142 }, { 5181, 5147 } } },
    { "NtUserSetActiveProcess",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5055, 5009 }, { 5095, 5058 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSetActiveProcessForMonitor", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5170, 5143 }, { 5173, 5148 } } },
    { "NtUserSetActiveWindow",    { 4599, { 4597, 4324 }, { 4622, 4325 }, { 4633, 4320 }, { 4709, 4320 }, { 4712, 4321 }, { 4718, 4322 }, { 4719, 4322 } } },
    { "NtUserSetAppImeLevel", { 4600, { 4598, 4724 }, { 4623, 4825 }, { 4634, 4864 }, { 4708, 5010 }, { 4711, 5059 }, { 4717, 5144 }, { 4718, 5149 } } },
    { "NtUserSetAutoRotation",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5059, 5011 }, { 5099, 5060 }, { 5174, 5145 }, { 5177, 5150 } } },
    { "NtUserSetBrokeredForeground",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5065, 5012 }, { 5105, 5061 }, { 5180, 5146 }, { 5183, 5151 } } },
    { "NtUserSetCalibrationData", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4806, 5013 }, { 4809, 5062 }, { 4820, 5147 }, { 4821, 5152 } } },
    { "NtUserSetCapture", { 4601, { 4599, 4168 }, { 4624, 4169 }, { 4635, 4169 }, { 4707, 4170 }, { 4710, 4171 }, { 4716, 4172 }, { 4717, 4172 } } },
    { "NtUserSetChildWindowNoActivate",   { -1, { -1, -1 }, { -1, -1 }, { 4636, 4865 }, { 4706, 5014 }, { 4709, 5063 }, { 4715, 5148 }, { 4716, 5153 } } },
    { "NtUserSetClassLong",   { 4602, { 4600, 4292 }, { 4625, 4293 }, { 4637, 4288 }, { 4705, 4288 }, { 4708, 4289 }, { 4714, 4290 }, { 4715, 4290 } } },
    { "NtUserSetClassLongPtr",    { -1, { -1, 4759 }, { -1, 4868 }, { -1, 4921 }, { -1, 5080 }, { -1, 5132 }, { 5224, 5224 }, { 5229, 5229 } } },
    { "NtUserSetClassWord",   { 4603, { 4601, 4725 }, { 4626, 4826 }, { 4638, 4866 }, { 4704, 5015 }, { 4707, 5064 }, { 4713, 5149 }, { 4714, 5154 } } },
    { "NtUserSetClipboardData",   { 4604, { 4602, 4309 }, { 4627, 4310 }, { 4639, 4305 }, { 4703, 4305 }, { 4706, 4306 }, { 4712, 4307 }, { 4713, 4307 } } },
    { "NtUserSetClipboardViewer", { 4605, { 4603, 4385 }, { 4628, 4386 }, { 4640, 4378 }, { 4702, 4378 }, { 4705, 4379 }, { 4711, 4380 }, { 4712, 4380 } } },
    { "NtUserSetConsoleReserveKeys",  { 4606, { 4604, 4387 }, { 4629, 4388 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSetCoreWindow",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5220, 5150 }, { 5225, 5155 } } },
    { "NtUserSetCoreWindowPartner",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5221, 5151 }, { 5226, 5156 } } },
    { "NtUserSetCursor",  { 4607, { 4605, 4121 }, { 4630, 4122 }, { 4641, 4122 }, { 4701, 4123 }, { 4704, 4124 }, { 4710, 4125 }, { 4711, 4125 } } },
    { "NtUserSetCursorContents",  { 4608, { 4606, 4726 }, { 4631, 4827 }, { 4642, 4867 }, { 4700, 5016 }, { 4703, 5065 }, { 4709, 5152 }, { 4710, 5157 } } },
    { "NtUserSetCursorIconData",  { 4609, { 4607, 4264 }, { 4632, 4265 }, { 4643, 4260 }, { 4699, 4260 }, { 4702, 4261 }, { 4708, 4262 }, { 4709, 4262 } } },
    { "NtUserSetDbgTag",  { 4610, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSetDisplayAutoRotationPreferences",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5058, 5017 }, { 5098, 5066 }, { 5173, 5153 }, { 5176, 5158 } } },
    { "NtUserSetDisplayConfig",   { -1, { -1, -1 }, { -1, -1 }, { 4435, 4868 }, { 4440, 5018 }, { 4442, 5067 }, { 4443, 5154 }, { 4445, 5159 } } },
    { "NtUserSetDisplayMapping",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4805, 5019 }, { 4808, 5068 }, { 4819, 5155 }, { 4820, 5160 } } },
    { "NtUserSetFallbackForeground",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5064, 5020 }, { 5104, 5069 }, { 5179, 5156 }, { 5182, 5161 } } },
    { "NtUserSetFeatureReportResponse",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5210, 5157 }, { 5215, 5162 } } },
    { "NtUserSetFocus",   { 4611, { 4608, 4176 }, { 4633, 4177 }, { 4644, 4177 }, { 4698, 4178 }, { 4701, 4179 }, { 4707, 4180 }, { 4708, 4180 } } },
    { "NtUserSetGestureConfig",   { -1, { -1, -1 }, { -1, -1 }, { 4758, 4869 }, { 4821, 5021 }, { 4830, 5070 }, { 4842, 5158 }, { 4843, 5163 } } },
    { "NtUserSetImeHotKey",   { 4612, { 4609, 4727 }, { 4634, 4828 }, { 4645, 4870 }, { 4697, 5022 }, { 4700, 5071 }, { 4706, 5159 }, { 4707, 5164 } } },
    { "NtUserSetImeInfoEx",   { 4613, { 4610, -1 }, { 4635, 4829 }, { 4646, 4871 }, { 4696, 5023 }, { 4699, 5072 }, { 4705, 5160 }, { 4706, 5165 } } },
    { "NtUserSetImeOwnerWindow",  { 4614, { 4611, 4729 }, { 4636, 4830 }, { 4647, 4872 }, { 4695, 5024 }, { 4698, 5073 }, { 4704, 5161 }, { 4705, 5166 } } },
    { "NtUserSetImmersiveBackgroundWindow",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4681, 5025 }, { 4684, 5074 }, { 4690, 5162 }, { -1, -1 } } },
    { "NtUserSetInformationProcess",  { 4615, { 4612, 4352 }, { 4637, 4353 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSetInformationThread",   { 4616, { 4613, 4325 }, { 4638, 4326 }, { 4648, 4321 }, { 4694, 4321 }, { 4697, 4322 }, { 4703, 4323 }, { 4704, 4323 } } },
    { "NtUserSetInternalWindowPos",   { 4617, { 4614, 4730 }, { 4639, 4831 }, { 4649, 4873 }, { 4693, 5026 }, { 4696, 5075 }, { 4702, 5163 }, { 4703, 5167 } } },
    { "NtUserSetKeyboardState",   { 4618, { 4615, 4339 }, { 4640, 4340 }, { 4650, 4335 }, { 4692, 4335 }, { 4695, 4336 }, { 4701, 4337 }, { 4702, 4337 } } },
    { "NtUserSetLayeredWindowAttributes", { 4677, { 4673, 4731 }, { 4701, 4832 }, { 4714, 4874 }, { 4756, 5027 }, { 4759, 5076 }, { 4769, 5164 }, { 4770, 5168 } } },
    { "NtUserSetLogonNotifyWindow",   { 4619, { 4616, 4732 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSetManipulationInputTarget", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4736, 5165 }, { 4737, 5169 } } },
    { "NtUserSetMenu",    { 4620, { 4617, 4733 }, { 4641, 4833 }, { 4651, 4875 }, { 4691, 5028 }, { 4694, 5077 }, { 4700, 5166 }, { 4701, 5170 } } },
    { "NtUserSetMenuContextHelpId",   { 4621, { 4618, 4734 }, { 4642, 4834 }, { 4652, 4876 }, { 4690, 5029 }, { 4693, 5078 }, { 4699, 5167 }, { 4700, 5171 } } },
    { "NtUserSetMenuDefaultItem", { 4622, { 4619, 4359 }, { 4643, 4360 }, { 4653, 4354 }, { 4689, 4354 }, { 4692, 4355 }, { 4698, 4356 }, { 4699, 4356 } } },
    { "NtUserSetMenuFlagRtoL",    { 4623, { 4620, 4735 }, { 4644, 4835 }, { 4654, 4877 }, { 4688, 5030 }, { 4691, 5079 }, { 4697, 5168 }, { 4698, 5172 } } },
    { "NtUserSetMirrorRendering", { -1, { -1, -1 }, { 4866, 4836 }, { 4914, 4878 }, { 4985, 5031 }, { 5003, 5080 }, { 5060, 5169 }, { 5062, 5173 } } },
    { "NtUserSetObjectInformation",   { 4624, { 4621, 4736 }, { 4645, 4837 }, { 4655, 4879 }, { 4687, 5032 }, { 4690, 5081 }, { 4696, 5170 }, { 4697, 5174 } } },
    { "NtUserSetParent",  { 4625, { 4622, 4216 }, { 4646, 4217 }, { 4656, 4215 }, { 4686, 4215 }, { 4689, 4216 }, { 4695, 4217 }, { 4696, 4217 } } },
    { "NtUserSetPrecisionTouchPadConfiguration",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4821, 5082 }, { 4832, 5171 }, { 4833, 5175 } } },
    { "NtUserSetProcessDPIAware", { -1, { -1, -1 }, { 4660, 4838 }, { 4670, 4880 }, { 4670, 5033 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSetProcessDpiAwareness", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4673, 5083 }, { 4676, 5172 }, { 4678, 5176 } } },
    { "NtUserSetProcessRestrictionExemption", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5069, 5034 }, { 5110, 5084 }, { 5185, 5173 }, { 5188, 5177 } } },
    { "NtUserSetProcessUIAccessZorder",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4669, 5035 }, { 4672, 5085 }, { 4675, 5174 }, { 4677, 5178 } } },
    { "NtUserSetProcessWindowStation",    { 4626, { 4623, 4268 }, { 4647, 4269 }, { 4657, 4264 }, { 4685, 4264 }, { 4688, 4265 }, { 4694, 4266 }, { 4695, 4266 } } },
    { "NtUserSetProp",    { 4627, { 4624, 4171 }, { 4649, 4172 }, { 4659, 4172 }, { 4683, 4173 }, { 4686, 4174 }, { 4692, 4175 }, { 4693, 4175 } } },
    { "NtUserSetRipFlags",    { 4628, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSetScrollInfo",  { 4629, { 4625, 4143 }, { 4650, 4144 }, { 4660, 4144 }, { 4682, 4145 }, { 4685, 4146 }, { 4691, 4147 }, { 4692, 4147 } } },
    { "NtUserSetSensorPresence",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5062, 5036 }, { 4097, 4097 }, { 4098, 4098 }, { 4098, 4098 } } },
    { "NtUserSetShellWindowEx",   { 4630, { 4626, 4737 }, { 4651, 4839 }, { 4661, 4881 }, { 4680, 5037 }, { 4683, 5086 }, { 4689, 5175 }, { 4691, 5179 } } },
    { "NtUserSetSysColors",   { 4631, { 4627, 4738 }, { 4652, 4840 }, { 4662, 4882 }, { 4679, 5038 }, { 4682, 5087 }, { 4688, 5176 }, { 4690, 5180 } } },
    { "NtUserSetSystemCursor",    { 4632, { 4628, 4739 }, { 4653, 4841 }, { 4663, 4883 }, { 4678, 5039 }, { 4681, 5088 }, { 4687, 5177 }, { 4689, 5181 } } },
    { "NtUserSetSystemMenu",  { 4633, { 4629, 4365 }, { 4654, 4366 }, { 4664, 4359 }, { 4677, 4359 }, { 4680, 4360 }, { 4686, 4361 }, { 4688, 4361 } } },
    { "NtUserSetSystemTimer", { 4634, { 4630, 4740 }, { 4655, 4842 }, { 4665, 4884 }, { 4676, 5040 }, { 4679, 5089 }, { 4685, 5178 }, { 4687, 5182 } } },
    { "NtUserSetThreadDesktop",   { 4635, { 4631, 4242 }, { 4656, 4243 }, { 4666, 4241 }, { 4674, 4241 }, { 4677, 4242 }, { 4683, 4243 }, { 4685, 4243 } } },
    { "NtUserSetThreadInputBlocked",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4675, 5041 }, { 4678, 5090 }, { 4684, 5179 }, { 4686, 5183 } } },
    { "NtUserSetThreadLayoutHandles", { 4636, { 4632, 4741 }, { 4657, 4843 }, { 4667, 4885 }, { 4673, 5042 }, { 4676, 5091 }, { 4682, 5180 }, { 4684, 5184 } } },
    { "NtUserSetThreadState", { 4637, { 4633, 4317 }, { 4658, 4318 }, { 4668, 4313 }, { 4672, 4313 }, { 4675, 4314 }, { 4681, 4315 }, { 4683, 4315 } } },
    { "NtUserSetTimer",   { 4638, { 4634, 4119 }, { 4659, 4120 }, { 4669, 4120 }, { 4671, 4121 }, { 4674, 4122 }, { 4680, 4123 }, { 4682, 4123 } } },
    { "NtUserSetWindowArrangement",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5186, 5181 }, { 5189, 5185 } } },
    { "NtUserSetWindowBand",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4668, 5043 }, { 4671, 5092 }, { 4674, 5182 }, { 4676, 5186 } } },
    { "NtUserSetWindowCompositionAttribute",  { -1, { -1, -1 }, { -1, -1 }, { 4671, 4886 }, { 4667, 5044 }, { 4670, 5093 }, { 4673, 5183 }, { 4675, 5187 } } },
    { "NtUserSetWindowCompositionTransition", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4665, 5045 }, { 4668, 5094 }, { 4671, 5184 }, { 4673, 5188 } } },
    { "NtUserSetWindowDisplayAffinity",   { -1, { -1, -1 }, { -1, -1 }, { 4672, 4887 }, { 4664, 5046 }, { 4667, 5095 }, { 4670, 5185 }, { 4672, 5189 } } },
    { "NtUserSetWindowFeedbackSetting",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4797, 5047 }, { 4800, 5096 }, { 4810, 5186 }, { 4811, 5190 } } },
    { "NtUserSetWindowFNID",  { 4639, { 4635, 4246 }, { 4661, 4247 }, { 4673, 4245 }, { 4663, 4245 }, { 4666, 4246 }, { 4669, 4247 }, { 4671, 4247 } } },
    { "NtUserSetWindowLong",  { 4640, { 4636, 4187 }, { 4662, 4188 }, { 4674, 4188 }, { 4662, 4188 }, { 4665, 4189 }, { 4668, 4190 }, { 4670, 4190 } } },
    { "NtUserSetWindowLongPtr",   { -1, { -1, 4760 }, { -1, 4869 }, { -1, 4922 }, { -1, 5081 }, { -1, 5133 }, { 5225, 5225 }, { 5230, 5230 } } },
    { "NtUserSetWindowPlacement", { 4641, { 4637, 4326 }, { 4663, 4327 }, { 4675, 4322 }, { 4661, 4322 }, { 4664, 4323 }, { 4667, 4324 }, { 4669, 4324 } } },
    { "NtUserSetWindowPos",   { 4642, { 4638, 4131 }, { 4664, 4132 }, { 4676, 4132 }, { 4660, 4133 }, { 4663, 4134 }, { 4666, 4135 }, { 4668, 4135 } } },
    { "NtUserSetWindowRgn",   { 4643, { 4639, 4302 }, { 4665, 4303 }, { 4677, 4298 }, { 4659, 4298 }, { 4662, 4299 }, { 4665, 4300 }, { 4667, 4300 } } },
    { "NtUserSetWindowRgnEx", { -1, { -1, -1 }, { 4667, 4844 }, { 4679, 4888 }, { 4657, 5048 }, { 4660, 5097 }, { 4663, 5187 }, { 4665, 5191 } } },
    { "NtUserSetWindowsHookAW",   { 4644, { 4640, 4358 }, { 4668, 4359 }, { 4680, 4353 }, { 4656, 4353 }, { 4659, 4354 }, { 4662, 4355 }, { 4664, 4355 } } },
    { "NtUserSetWindowsHookEx",   { 4645, { 4641, 4237 }, { 4669, 4238 }, { 4681, 4236 }, { 4655, 4236 }, { 4658, 4237 }, { 4661, 4238 }, { 4663, 4238 } } },
    { "NtUserSetWindowShowState", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5187, 5188 }, { 5190, 5192 } } },
    { "NtUserSetWindowStationUser",   { 4646, { 4642, 4742 }, { 4670, 4845 }, { 4682, 4889 }, { 4654, 5049 }, { 4657, 5098 }, { 4660, 5189 }, { 4662, 5193 } } },
    { "NtUserSetWindowWord",  { 4647, { 4643, 4332 }, { 4671, 4333 }, { 4683, 4328 }, { 4653, 4328 }, { 4656, 4329 }, { 4659, 4330 }, { 4661, 4330 } } },
    { "NtUserSetWinEventHook",    { 4648, { 4644, 4361 }, { 4672, 4362 }, { 4684, 4356 }, { 4652, 4356 }, { 4655, 4357 }, { 4658, 4358 }, { 4660, 4358 } } },
    { "NtUserSfmDestroyLogicalSurfaceBinding",    { -1, { -1, -1 }, { -1, -1 }, { 4747, 4890 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmDxBindSwapChain", { -1, { -1, -1 }, { -1, -1 }, { 4738, 4891 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmDxGetSwapChainStats", { -1, { -1, -1 }, { -1, -1 }, { 4744, 4892 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmDxOpenSwapChain", { -1, { -1, -1 }, { -1, -1 }, { 4739, 4893 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmDxQuerySwapChainBindingStatus",   { -1, { -1, -1 }, { -1, -1 }, { 4742, 4894 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmDxReleaseSwapChain",  { -1, { -1, -1 }, { -1, -1 }, { 4740, 4895 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmDxReportPendingBindingsToDwm",    { -1, { -1, -1 }, { -1, -1 }, { 4743, 4896 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmDxSetSwapChainBindingStatus", { -1, { -1, -1 }, { -1, -1 }, { 4741, 4897 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmDxSetSwapChainStats", { -1, { -1, -1 }, { -1, -1 }, { 4745, 4898 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserSfmGetLogicalSurfaceBinding",    { -1, { -1, -1 }, { -1, -1 }, { 4746, 4899 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserShowCaret",  { 4649, { 4645, 4132 }, { 4673, 4133 }, { 4685, 4133 }, { 4651, 4134 }, { 4654, 4135 }, { 4657, 4136 }, { 4659, 4136 } } },
    { "NtUserShowScrollBar",  { 4650, { 4646, 4156 }, { 4674, 4157 }, { 4686, 4157 }, { 4650, 4158 }, { 4653, 4159 }, { 4656, 4160 }, { 4658, 4160 } } },
    { "NtUserShowSystemCursor",   { -1, { -1, -1 }, { 4867, 4846 }, { 4915, 4900 }, { 4984, 5050 }, { 5002, 5099 }, { 5059, 5190 }, { 5061, 5194 } } },
    { "NtUserShowWindow", { 4651, { 4647, 4183 }, { 4675, 4184 }, { 4687, 4184 }, { 4649, 4184 }, { 4652, 4185 }, { 4655, 4186 }, { 4657, 4186 } } },
    { "NtUserShowWindowAsync",    { 4652, { 4648, 4386 }, { 4676, 4387 }, { 4688, 4379 }, { 4648, 4379 }, { 4651, 4380 }, { 4654, 4381 }, { 4656, 4381 } } },
    { "NtUserShutdownBlockReasonCreate",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4647, 5051 }, { 4650, 5100 }, { 4653, 5191 }, { 4655, 5195 } } },
    { "NtUserShutdownBlockReasonQuery",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4646, 5052 }, { 4649, 5101 }, { 4652, 5192 }, { 4654, 5196 } } },
    { "NtUserShutdownReasonDestroy",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4645, 5053 }, { 4648, 5102 }, { 4651, 5193 }, { 4653, 5197 } } },
    { "NtUserSignalRedirectionStartComplete", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5054, 5054 }, { 5094, 5103 }, { 5169, 5194 }, { 5172, 5198 } } },
    { "NtUserSlicerControl",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4989, 5055 }, { 5007, 5104 }, { 5064, 5195 }, { 5066, 5199 } } },
    { "NtUserSoundSentry",    { 4653, { 4649, -1 }, { 4677, 4847 }, { 4689, 4901 }, { 4644, 5056 }, { 4647, 5105 }, { 4650, 5196 }, { 4652, 5200 } } },
    { "NtUserSwitchDesktop",  { 4654, { 4650, 4744 }, { 4678, 4848 }, { 4690, 4902 }, { 4643, 5057 }, { 4646, 5106 }, { 4649, 5197 }, { 4651, 5201 } } },
    { "NtUserSystemParametersInfo",   { 4655, { 4651, 4161 }, { 4679, 4162 }, { 4691, 4162 }, { 4642, 4163 }, { 4645, 4164 }, { 4648, 4165 }, { 4650, 4165 } } },
    { "NtUserTestForInteractiveUser", { 4656, { 4652, 4745 }, { 4680, 4849 }, { 4692, 4903 }, { 4640, 5058 }, { 4643, 5107 }, { 4646, 5198 }, { 4648, 5202 } } },
    { "NtUserThunkedMenuInfo",    { 4657, { 4653, 4366 }, { 4681, 4367 }, { 4693, 4360 }, { 4639, 4360 }, { 4642, 4361 }, { 4645, 4362 }, { 4647, 4362 } } },
    { "NtUserThunkedMenuItemInfo",    { 4658, { 4654, 4249 }, { 4682, 4250 }, { 4694, 4248 }, { 4638, 4248 }, { 4641, 4249 }, { 4644, 4250 }, { 4646, 4250 } } },
    { "NtUserToUnicodeEx",    { 4659, { 4655, 4218 }, { 4683, 4219 }, { 4695, 4217 }, { 4637, 4217 }, { 4640, 4218 }, { 4643, 4219 }, { 4645, 4219 } } },
    { "NtUserTrackMouseEvent",    { 4660, { 4656, 4319 }, { 4684, 4320 }, { 4696, 4315 }, { 4636, 4315 }, { 4639, 4316 }, { 4642, 4317 }, { 4644, 4317 } } },
    { "NtUserTrackPopupMenuEx",   { 4661, { 4657, 4746 }, { 4685, 4850 }, { 4697, 4904 }, { 4635, 5059 }, { 4638, 5108 }, { 4641, 5199 }, { 4643, 5203 } } },
    { "NtUserTransformPoint", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5123, 5109 }, { 5200, 5200 }, { 5204, 5204 } } },
    { "NtUserTransformRect",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5124, 5110 }, { 5201, 5201 }, { 5205, 5205 } } },
    { "NtUserTranslateAccelerator",   { 4664, { 4660, 4112 }, { 4688, 4113 }, { 4701, 4113 }, { 4631, 4114 }, { 4634, 4115 }, { 4637, 4116 }, { 4639, 4116 } } },
    { "NtUserTranslateMessage",   { 4665, { 4661, 4109 }, { 4689, 4109 }, { 4702, 4109 }, { 4630, 4110 }, { 4633, 4111 }, { 4636, 4112 }, { 4638, 4112 } } },
    { "NtUserUndelegateInput",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4729, 5060 }, { 4732, 5111 }, { 4742, 5202 }, { 4743, 5206 } } },
    { "NtUserUnhookWindowsHookEx",    { 4666, { 4662, 4208 }, { 4690, 4209 }, { 4703, 4208 }, { 4767, 4208 }, { 4770, 4209 }, { 4780, 4210 }, { 4781, 4210 } } },
    { "NtUserUnhookWinEvent", { 4667, { 4663, 4362 }, { 4691, 4363 }, { 4704, 4357 }, { 4766, 4357 }, { 4769, 4358 }, { 4779, 4359 }, { 4780, 4359 } } },
    { "NtUserUnloadKeyboardLayout",   { 4668, { 4664, 4747 }, { 4692, 4851 }, { 4705, 4905 }, { 4765, 5061 }, { 4768, 5112 }, { 4778, 5203 }, { 4779, 5207 } } },
    { "NtUserUnlockWindowStation",    { 4669, { 4665, 4748 }, { 4693, 4852 }, { 4706, 4906 }, { 4764, 5062 }, { 4767, 5113 }, { 4777, 5204 }, { 4778, 5208 } } },
    { "NtUserUnregisterClass",    { 4670, { 4666, 4287 }, { 4694, 4288 }, { 4707, 4283 }, { 4763, 4283 }, { 4766, 4284 }, { 4776, 4285 }, { 4777, 4285 } } },
    { "NtUserUnregisterHotKey",   { 4672, { 4668, 4749 }, { 4696, 4853 }, { 4709, 4907 }, { 4761, 5063 }, { 4764, 5114 }, { 4774, 5205 }, { 4775, 5209 } } },
    { "NtUserUnregisterSessionPort",  { -1, { -1, -1 }, { 4721, 4854 }, { 4733, 4908 }, { 4737, 5064 }, { 4740, 5115 }, { 4750, 5206 }, { 4751, 5210 } } },
    { "NtUserUnregisterUserApiHook",  { 4671, { 4667, 4750 }, { 4695, 4855 }, { 4708, 4909 }, { 4762, 5065 }, { 4765, 5116 }, { 4775, 5207 }, { 4776, 5211 } } },
    { "NtUserUpdateDefaultDesktopThumbnail",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4666, 5066 }, { 4669, 5117 }, { 4672, 5208 }, { 4674, 5212 } } },
    { "NtUserUpdateInputContext", { 4673, { 4669, 4751 }, { 4697, 4856 }, { 4710, 4910 }, { 4760, 5067 }, { 4763, 5118 }, { 4773, 5209 }, { 4774, 5213 } } },
    { "NtUserUpdateInstance", { 4674, { 4670, 4752 }, { 4698, 4857 }, { 4711, 4911 }, { 4759, 5068 }, { 4762, 5119 }, { 4772, 5210 }, { 4773, 5214 } } },
    { "NtUserUpdateLayeredWindow",    { 4675, { 4671, 4753 }, { 4699, 4858 }, { 4712, 4912 }, { 4758, 5069 }, { 4761, 5120 }, { 4771, 5211 }, { 4772, 5215 } } },
    { "NtUserUpdatePerUserSystemParameters",  { 4678, { 4674, 4754 }, { 4702, 4859 }, { 4715, 4913 }, { 4755, 5070 }, { 4758, 5121 }, { 4768, 5212 }, { 4769, 5216 } } },
    { "NtUserUpdateWindowInputSinkHints", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5122, 5122 }, { 5199, 5213 }, { 5203, 5217 } } },
    { "NtUserUpdateWindowTrackingInfo",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, 5218 }, { 5191, 5218 } } },
    { "NtUserUpdateWindowTransform",  { -1, { -1, -1 }, { 4722, 4860 }, { 4734, 4914 }, { 4735, 5071 }, { 4738, 5123 }, { 4748, 5214 }, { 4749, 5219 } } },
    { "NtUserUserHandleGrantAccess",  { 4679, { 4675, 4755 }, { 4703, 4861 }, { 4716, 4915 }, { 4754, 5072 }, { 4757, 5124 }, { 4767, 5215 }, { 4768, 5220 } } },
    { "NtUserValidateHandleSecure",   { 4680, { 4676, 4756 }, { 4704, 4862 }, { 4717, 4916 }, { 4753, 5073 }, { 4756, 5125 }, { 4766, 5216 }, { 4767, 5221 } } },
    { "NtUserValidateRect",   { 4681, { 4677, 4305 }, { 4705, 4306 }, { 4718, 4301 }, { 4752, 4301 }, { 4755, 4302 }, { 4765, 4303 }, { 4766, 4303 } } },
    { "NtUserValidateTimerCallback",  { 4682, { 4678, 4117 }, { 4706, 4118 }, { 4719, 4118 }, { 4751, 4119 }, { 4754, 4120 }, { 4764, 4121 }, { 4765, 4121 } } },
    { "NtUserVkKeyScanEx",    { 4683, { 4679, 4135 }, { 4707, 4136 }, { 4720, 4136 }, { 4750, 4137 }, { 4753, 4138 }, { 4763, 4139 }, { 4764, 4139 } } },
    { "NtUserWaitAvailableMessageEx", { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4744, 5074 }, { 4747, 5126 }, { 4757, 5217 }, { 4758, 5222 } } },
    { "NtUserWaitForInputIdle",   { 4684, { 4680, 4757 }, { 4708, 4863 }, { 4721, 4917 }, { 4749, 5075 }, { 4752, 5127 }, { 4762, 5218 }, { 4763, 5223 } } },
    { "NtUserWaitForMsgAndEvent", { 4685, { 4681, 4758 }, { 4709, 4864 }, { 4722, 4918 }, { 4748, 5076 }, { 4751, 5128 }, { 4761, 5219 }, { 4762, 5224 } } },
    { "NtUserWaitForRedirectionStartComplete",    { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5053, 5077 }, { 5093, 5129 }, { 5168, 5220 }, { 5171, 5225 } } },
    { "NtUserWaitMessage",    { 4686, { 4682, 4108 }, { 4710, 4108 }, { 4723, 4108 }, { 4747, 4109 }, { 4750, 4110 }, { 4760, 4111 }, { 4761, 4111 } } },
    { "NtUserWin32PoolAllocationStats",   { 4687, { 4683, 4761 }, { 4711, 4865 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 } } },
    { "NtUserWindowFromPhysicalPoint",    { -1, { -1, -1 }, { 4712, 4866 }, { 4724, 4919 }, { 4746, 5078 }, { 4749, 5130 }, { 4759, 5221 }, { 4760, 5226 } } },
    { "NtUserWindowFromPoint",    { 4688, { 4684, 4115 }, { 4713, 4116 }, { 4725, 4116 }, { 4745, 4117 }, { 4748, 4118 }, { 4758, 4119 }, { 4759, 4119 } } },
    { "NtUserYieldTask",  { 4689, { 4685, 4762 }, { 4714, 4867 }, { 4726, 4920 }, { 4096, 4096 }, { 4096, 4096 }, { 4097, 4097 }, { 4097, 4097 } } },
    { "NtValidateCompositionSurfaceHandle",   { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 4993, 5079 }, { 5011, 5131 }, { 5068, 5222 }, { 5070, 5227 } } },
    { "NtVisualCaptureBits",  { -1, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { -1, -1 }, { 5084, 5223 }, { 5086, 5228 } } }
};