# 纯前端下载

```js
export function downloadDirect(url) {
  const aTag = document.createElement("a");
  aTag.download = url.split("/").pop();
  aTag.href = url;
  aTag.click();
}

export function downloadByContent(content, filename, type) {
  const aTag = document.createElement("a");
  aTag.download = filename;
  const blob = new Blob([content], { type });
  const blobUrl = URL.createObjectURL(blob);
  aTag.href = blobUrl;
  aTag.click();
  URL.revokeObjectURL(blob);
}

export function downloadByDataURL(content, filename, type) {
  const aTag = document.createElement("a");
  aTag.download = filename;
  const dataUrl = `data:${type};base64,${window.btoa(
    unescape(encodeURIComponent(content))
  )}`;
  aTag.href = dataUrl;
  aTag.click();
}

export function downloadByBlob(blob, filename) {
  const aTag = document.createElement("a");
  aTag.download = filename;
  const blobUrl = URL.createObjectURL(blob);
  aTag.href = blobUrl;
  aTag.click();
  URL.revokeObjectURL(blob);
}
```

## json/text

下载 text 和 json 非常的简单，可以直接构造一个 Blob。

```js
Blob(blobParts[, options])
// 返回一个新创建的 Blob 对象，其内容由参数中给定的数组串联组成。

// html
<textarea name="" id="text" cols="30" rows="10"></textarea>
<button id="textBtn">下载文本</button>
<p></p>
<textarea name="" id="json" cols="30" rows="10" disabled>
{
    "name": "秋风的笔记"
}
</textarea>

<button id="jsonBtn">下载JSON</button>

//js
import { downloadByContent, downloadByDataURL } from "../js/utils.js";
textBtn.onclick = () => {
  const value = text.value;
  downloadByContent(value, "hello.txt", "text/plain");
  // downloadByDataURL(value, 'hello.txt', 'text/plain');
};
jsonBtn.onclick = () => {
  const value = json.value;
  downloadByContent(value, "hello.json", "application/json");
  // downloadByDataURL(value, 'hello.json', 'application/json');
};
```

# excel

excel 可以说是我们部分前端打交道很深的一个场景，什么数据中台，天天需要导出各种报表。以前都是前端请求后端，来获取一个 excel 文件地址。现在让我们来展示下纯前端是如何实现下载 excel。

```js
const template =
  '<html xmlns:o="urn:schemas-microsoft-com:office:office" ' +
  'xmlns:x="urn:schemas-microsoft-com:office:excel" ' +
  'xmlns="http://www.w3.org/TR/REC-html40">' +
  "<head>" +
  "</head>" +
  '<body><table border="1" style="width:60%; text-align: center;">{table}</table></body>' +
  "</html>";
const context = template.replace(
  "{table}",
  document.getElementById("excel").innerHTML
);
downloadByContent(context, "qiufengblue.xls", "application/vnd.ms-excel");
```

但是编写并不复杂，依旧是和我们之前一样，通过构造出 excel 的格式，转化成 blob 来进行下载。

# word

```js
exportWord.onclick = () => {
  const template =
    '<html xmlns:o="urn:schemas-microsoft-com:office:office" ' +
    'xmlns:x="urn:schemas-microsoft-com:office:word" ' +
    'xmlns="http://www.w3.org/TR/REC-html40">' +
    "<head>" +
    "</head>" +
    "<body>{table}</body>" +
    "</html>";
  const context = template.replace(
    "{table}",
    document.getElementById("word").innerHTML
  );
  downloadByContent(context, "qiufeng.blue.doc", "application/msword");
};
```

如果你想有更高级的用法，可以使用 docx.js 这个库。当然用上述方法也是可以高级定制的。

```js
<button type="button" onclick="generate()">下载word</button>

<script>
  async function generate() {
    const res = await axios({
      method: "get",
      url: "http://localhost:8888/static/1597375650384.jpg",
      responseType: "blob",
    });
    const doc = new docx.Document();
    const image1 = docx.Media.addImage(doc, res.data, 300, 400);
    doc.addSection({
      properties: {},
      children: [
        new docx.Paragraph({
          children: [
            new docx.TextRun("欢迎关注[秋风的笔记]公众号").break(),
            new docx.TextRun("").break(),
            new docx.TextRun("定期发送优质文章").break(),
            new docx.TextRun("").break(),
            new docx.TextRun("美团点评2020校招-内推").break(),
          ],
        }),
        new docx.Paragraph(image1),
      ],
    });

    docx.Packer.toBlob(doc).then((blob) => {
      console.log(blob);
      saveAs(blob, "qiufeng.blue.docx");
      console.log("Document created successfully");
    });
  }
</script>
```

# zip 下载

前端压缩还是非常有用的，在一定的场景下，可以节省流量。而这个场景比较使用于，例如前端打包图片下载、前端打包下载图标。

```js
download.onclick = () => {
  const zip = new JSZip();
  const svgList = [
    {
      id: "demo1",
    },
    {
      id: "demo2",
    },
  ];
  svgList.map((item) => {
    zip.file(item.id + ".svg", document.getElementById(item.id).outerHTML);
  });
  zip
    .generateAsync({
      type: "blob",
    })
    .then(function (content) {
      // 下载的文件名
      var filename = "svg" + ".zip";
      // 创建隐藏的可下载链接
      var eleLink = document.createElement("a");
      eleLink.download = filename;
      // 下载内容转变成blob地址
      eleLink.href = URL.createObjectURL(content);
      // 触发点击
      eleLink.click();
      // 然后移除
    });
};
```
